﻿// <summary>ソースコード：ライブラリ出力CUI　ライブラリ出力引数変換クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Security;

using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYMCO.CUI.HymcoCUI
{
    /// <summary><para>class outline:</para>
    /// <para>ライブラリ出力CUI　ライブラリ出力引数変換クラス</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>無し</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/12/09][新規作成]</para>
    /// </remarks>
    public class McLibraryExportArgsConverter
    {
        /// <summary> ワイルドカード文字列 </summary>
        static readonly string WILD_CARD_CHAR = "*";

        /// <summary><para>method outline:</para>
        /// <para>ライブラリ出力引数変換</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> int code = ConvertLibraryExportArgs(string[] args,McLibraryExportInfo exportInfo) </para>
        /// </example>
        /// <param name="args">コマンドライン引数</param>
        /// <param name="exportInfo">ライブラリ出力情報</param>
        /// <returns><para>0:正常 -1:オプション不正エラー -2:アーカイブファイルパス不正 </para>
        /// <para>-3:コマンドの構文不正 -4:実行形式ファイル無し</para>
        /// <para>-5モデルアイコンファイル無し -6:プロジェクトフォルダパス不正</para>
        /// <para>-7:プロパティ画面プロジェクトフォルダパス不正 -8：プロパティ画面プロジェクト実行形式ファイル無し</para>
        /// <para> -999:その他エラー</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public int ConvertLibraryExportArgs(
            string[] args,
            out McLibraryExportInfo exportInfo
            )
        {

            // コマンドライン引数解析
            HySString projectFolderFullpath;
            HySString exportFileFullpath;
            bool isExportSource;
            HySString propertyPrjFolderFullPath;

            int result_work = DecipherArg(
                                    args,
                                    out projectFolderFullpath,
                                    out exportFileFullpath,
                                    out isExportSource,
                                    out propertyPrjFolderFullPath
                                    );
            if (result_work != 0)
            {
                // 内部メソッドのエラーを判定して本メソッドの戻り値を決定する。
                int result = (-999);
                switch (result_work)
                {
                    case (-1):
                        result = (-1);
                        break;

                    case (-2):
                        result = (-2);
                        break;

                    case (-3):
                        result = (-3);
                        break;

                    case (-4):
                        result = (-6);
                        break;

                    case (-5):
                        result = (-7);
                        break;

                    default:

                        break;
                }
                exportInfo = null;
                return result;
            }

            // ライブラリ出力情報作成
            result_work = ConvertToLibraryExportInfo(
                        projectFolderFullpath,
                        exportFileFullpath,
                        isExportSource,
                        propertyPrjFolderFullPath,
                        out exportInfo
                        );
            if (result_work != 0)
            {
                // 内部メソッドのエラーを判定して本メソッドの戻り値を決定する。
                int result = (-999);
                switch (result_work)
                {
                    case (-1):
                        result = (-4);
                        break;

                    case (-2):
                        result = (-5);
                        break;

                    case (-3):
                        result = (-8);
                        break;

                    default:
                        break;

                }
                return result;
            }
            return 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>引数解読</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> int code = DecipherArg(string[] args, out projectFolderFullpath,
        /// out exportFullpath, out isOutputSource, out propertyPrjFolderFullPath) </para>
        /// </example>
        /// <param name="args">コマンドライン引数</param>
        /// <param name="projectFolderFullpath">プロジェクトフォルダフルパス</param>
        /// <param name="exportFullpath">アーカイブ出力先フルパス</param>
        /// <param name="isOutputSource">開発コード出力要否</param>
        /// <param name="propertyPrjFolderFullPath">プロパティ画面プロジェクトフォルダフルパス</param>
        /// <returns><para>0:正常 -1:オプション不正エラー -2:アーカイブファイルパス不正</para>
        /// <para>-3:コマンドの構文不正  -4:プロジェクトフォルダパス不正</para>
        /// <para>-5:プロパティ画面プロジェクトフォルダパス不正</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static private int DecipherArg(
            string[] args,
            out HySString projectFolderFullpath,
            out HySString exportFullpath,
            out bool isOutputSource,
            out HySString propertyPrjFolderFullPath
            )
        {
            projectFolderFullpath = null;
            exportFullpath = null;
            isOutputSource = false;
            propertyPrjFolderFullPath = null;

            HySString folderPath = new HySString();
            HySString exportPath = new HySString();
            HySString propertyExportOption = null;
            HySString propertyPrjFldFullPath = null;
            HySString isExportSource = null;

            //-----------------------------------------
            // コマンドライン引数の要素数で構文を解析
            //-----------------------------------------
            if (args.Length == 3)
            {   // プロパティ画面出力無,開発コード出力無
                folderPath = new HySString(args[1]);
                exportPath = new HySString(args[2]);
            }
            else if (args.Length == 4)
            {   // プロパティ画面出力無, 開発コード出力有
                folderPath = new HySString(args[1]);
                exportPath = new HySString(args[2]);
                isExportSource = new HySString(args[3]);
            }
            else if (args.Length == 5)
            {   // プロパティ画面出力有, 開発コード出力無
                folderPath = new HySString(args[1]);
                exportPath = new HySString(args[2]);
                propertyExportOption = new HySString(args[3]);
                propertyPrjFldFullPath = new HySString(args[4]);
            }
            else if (args.Length == 6)
            {   // プロパティ画面出力有, 開発コード出力有
                folderPath = new HySString(args[1]);
                exportPath = new HySString(args[2]);
                propertyExportOption = new HySString(args[3]);
                propertyPrjFldFullPath = new HySString(args[4]);
                isExportSource = new HySString(args[5]);
            }
            else
            {
                // 構文不正
                // ver1.5 エラートレース日本語対応
                LogOut(
                    "DecipherArg",
                    Properties.HymcoCUIResources.STATEMENT_COMMAND_ELEMNT_NG + args.Length.ToString()
                    );
               // LogOut(
               //     "DecipherArg",
               //     "コマンドラインの要素数不正 args.Length:" + args.Length.ToString()
               //     );
                return (-3);
            }

            //---------------------
            // オプションチェック
            //---------------------
            // ライブラリ出力オプション確認
            if (args[0] != "-a")
            {   // オプションが"-a"でない場合
                // ver1.5 エラートレース日本語対応
                LogOut("ConvertLibraryExportArgs", Properties.HymcoCUIResources.STATEMENT_OPTION_NG + args[0]);
               // LogOut("ConvertLibraryExportArgs", "オプション不正エラー args[0]:" + args[0]);
                return (-1);
            }

            // 開発コード出力要否オプションチェック
            if ((object)isExportSource != null)
            {
                if (isExportSource != "-os")
                {
                    // ver1.5 エラートレース日本語対応
                    LogOut(
                        "DecipherArg",
                        Properties.HymcoCUIResources.STATEMENT_DEVELOP_CODE_NG + isExportSource.ToString());
                   // LogOut(
                   //     "DecipherArg",
                   //     "開発コード出力要否オプション不正 isExportSource:" + isExportSource.ToString());
                    return (-1);
                }
                isOutputSource = true;
            }
            // プロパティ画面出力オプションチェック
            if ((object)propertyExportOption != null)
            {
                if (propertyExportOption != "--property-project")
                {
                    // ver1.5 エラートレース日本語対応
                    LogOut(
                        "DecipherArg",
                        Properties.HymcoCUIResources.STATEMENT_PROPERTIS_OPTION_NG + propertyExportOption.ToString());
                   // LogOut(
                   //     "DecipherArg",
                   //     "プロパティ画面出力オプション不正 propertyExportOption:" + propertyExportOption.ToString());
                    return (-1);
                }
            }

            //--------------------------------------
            // プロジェクトフォルダ名が
            // 相対パス指定の場合は絶対パスに変換
            //--------------------------------------
            try
            {
                // モデルプロジェクトのフォルダ名変換
                bool result = System.IO.Path.IsPathRooted(folderPath.ToString());
                if (result == false)
                {
                    folderPath = new HySString(System.IO.Path.GetFullPath(folderPath.ToString()));
                }
                // プロパティ画面プロジェクトのフォルダ名変換
                if ((object)propertyPrjFldFullPath != null)
                {
                    result = System.IO.Path.IsPathRooted(propertyPrjFldFullPath.ToString());
                    if (result == false)
                    {
                        System.Environment.CurrentDirectory = System.IO.Directory.GetCurrentDirectory();
                        propertyPrjFldFullPath = new HySString(System.IO.Path.GetFullPath(propertyPrjFldFullPath.ToString()));
                    }
                }
                // アーカイブ出力先ファイルパスのフォルダ
                result = System.IO.Path.IsPathRooted(exportPath.ToString());
                if (result == false)
                {
                    System.Environment.CurrentDirectory = System.IO.Directory.GetCurrentDirectory();

                    exportPath = new HySString(System.IO.Path.GetFullPath(exportPath.ToString()));
                }

            }
            catch (ArgumentException ex)
            {   // プロジェクトフォルダに無効なものが含まれている例外が発生した場合
               // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-4);
            }
            catch (PathTooLongException ex)
            {   // プロジェクトフォルダ名が最大長を超える例外が発生した場合
               // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-4);
            }
            catch (NotSupportedException ex)
            {   // 呼び出されたメソッドがサポートされていない例外が発生した場合
               // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-4);
            }
            catch (UnauthorizedAccessException ex)
            {   // セキュリティエラーのためアクセスを拒否される例外が発生した場合
               // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-4);
            }
            catch (SecurityException ex)
            {   // セキュリティエラーが発生した場合
               // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-4);
            }

            //--------------------------------------
            // アーカイブ出力先パスが
            // 相対パス指定の場合は絶対パスに変換
            //--------------------------------------
            try
            {
                bool result = System.IO.Path.IsPathRooted(exportPath.ToString());
                if (result == false)
                {
                    exportPath = new HySString(System.IO.Path.GetFullPath(exportPath.ToString()));
                }
            }
            catch (ArgumentException ex)
            {   // プロジェクトフォルダに無効なものが含まれている例外が発生した場合
                // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-2);
            }
            catch (PathTooLongException ex)
            {   // プロジェクトフォルダ名が最大長を超える例外が発生した場合
                // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-2);
            }
            catch (NotSupportedException ex)
            {   // 呼び出されたメソッドがサポートされていない例外が発生した場合
                // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-2);
            }
            catch (UnauthorizedAccessException ex)
            {   // セキュリティエラーのためアクセスを拒否される例外が発生した場合
                // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-2);
            }
            catch (SecurityException ex)
            {   // セキュリティエラーが発生した場合
                // ver1.5 エラートレース日本語対応
                LogOut("DecipherArg", Properties.HymcoCUIResources.STATEMENT_FOLDER_PATH + folderPath.ToString() + Environment.NewLine + ex.ToString());
               // LogOut("DecipherArg", "folderPath:" + folderPath.ToString() + Environment.NewLine + ex.ToString());
                return (-2);
            }
            
            //-------------------------------------
            // ファイル/フォルダパスの有無チェック
            //-------------------------------------
            // プロジェクトフォルダパスの有無をチェック
            bool isDirectory = HySFileUtility.ExistDirectory(folderPath);
            if (isDirectory == false)
            {
                // ver1.5 エラートレース日本語対応
                LogOut(
                    "DecipherArg",
                    Properties.HymcoCUIResources.STATEMENT_PROJ_FOLDER_NO + folderPath.ToString()
                    );
               // LogOut(
               //     "DecipherArg",
               //     "プロジェクトフォルダが存在しません。 folderPath:" + folderPath.ToString()
               //     );
                return (-4);
            }

            // アーカイブ出力先パスのファイルが存在するかチェック
            bool isFile = HySFileUtility.ExistFile(exportPath);
            if (isFile == true)
            {
                // ver1.5 エラートレース日本語対応
                LogOut(
                    "DecipherArg",
                    Properties.HymcoCUIResources.STATEMENT_ARCHIVE_ALREADY_EXISTS + exportPath.ToString());
               // LogOut(
               //     "DecipherArg",
               //     "アーカイブ出力先のファイルが既に存在します。 exportPath:" + exportPath.ToString());
                return (-2);
            }

            if ((object)propertyPrjFldFullPath != null)
            {
                // プロパティ画面プロジェクトフォルダの有無チェック
                bool isExist = HySFileUtility.ExistDirectory(propertyPrjFldFullPath);
                if (isExist == false)
                {
                    // ver1.5 エラートレース日本語対応
                    LogOut(
                        "DecipherArg",
                        Properties.HymcoCUIResources.STATEMENT_PROJ_PROPER_FOLDER_NO + exportPath.ToString());
                   // LogOut(
                   //     "DecipherArg",
                   //     "プロパティ画面プロジェクトフォルダが存在しません。 exportPath:" + exportPath.ToString());
                    return (-5);
                }
            }

            // アーカイブ出力先パスの有無をチェック
            isDirectory = HySFileUtility.EnableFileFullPath(exportPath);
            if (isDirectory == false)
            {
                // ver1.5 エラートレース日本語対応
                LogOut(
                    "DecipherArg",
                    Properties.HymcoCUIResources.STATEMENT_ARCHIVE_DESTINATION_NG + exportPath.ToString());
               // LogOut(
               //     "DecipherArg",
               //     "アーカイブ出力先が不正です。 exportPath:" + exportPath.ToString());
                return (-2);
            }
            
            projectFolderFullpath = folderPath;
            propertyPrjFolderFullPath = propertyPrjFldFullPath;
            exportFullpath = exportPath;
            
            return 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>ライブラリ出力情報変換</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> int code = ConvertToLibraryExportInfo(projectFolderFullpath, exportFullpath, isOutputSource, propertyPrjFldFullPath, out exportInfo) </para>
        /// </example>
        /// <param name="projectFolderFullpath">プロジェクトフォルダフルパス</param>
        /// <param name="exportFullpath">アーカイブ出力先フルパス</param>
        /// <param name="isOutputSource">開発コード出力要否</param>
        /// <param name="exportInfo">ライブラリ出力情報</param>
        /// <param name="propertyPrjFldFullPath">プロパティ画面プロジェクトフォルダフルパス</param>
        /// <returns><para>0:正常 -1:実行形式ファイル不足エラー -2:モデルアイコンファイルエラー</para>
        /// <para>-3:プロジェクト画面プロジェクト実行形式ファイル不足エラー</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static private int ConvertToLibraryExportInfo(
            HySString projectFolderFullpath,
            HySString exportFullpath,
            bool isOutputSource,
            HySString propertyPrjFldFullPath,
            out McLibraryExportInfo exportInfo
            )
        {
            exportInfo = null;
            
            //------------------------
            // ライブラリ出力情報作成
            //------------------------
            // モデルプロジェクト実行形式ファイルパス取得
            HySDataLinkedList dllFileFullPathList = GetDllFileFullpath(projectFolderFullpath);
            if (dllFileFullPathList.GetCount() == 0)
            {
                // ver1.5 エラートレース日本語対応
                LogOut("ConvertToLibraryExportInfo", Properties.HymcoCUIResources.STATEMENT_MODEL_PROJECT_NO );
               // LogOut("ConvertToLibraryExportInfo", "モデルプロジェクト実行形式ファイル無し");
                return (-1);
            }

            // モデルアイコンファイルパス取得
            HySDataLinkedList modelIconFullpathList = GetModelIconPathList(projectFolderFullpath);

            // モデル解説書ファイルパス取得
            HySDataLinkedList manualPathList = GetManualPathList(projectFolderFullpath);
            
            // 開発コード取得
            HySDataLinkedList devSrcFullPathList = null;
            HySDataLinkedList propertyDevSrcFullPathList = null;
            if (isOutputSource == true)
            {
                // モデルプロジェクト開発コード
                devSrcFullPathList = GetDevelopmentSourceFullPathList(projectFolderFullpath);

                // プロパティ画面プロジェクト開発コード
                if ((object)propertyPrjFldFullPath != null)
                {
                    propertyDevSrcFullPathList = GetDevelopmentSourceFullPathList(propertyPrjFldFullPath);
                }
            }

            //------------------------------------
            // ライブラリ出力情報オブジェクト生成
            //------------------------------------
            McLibraryExportInfo libraryExportInfo = new McLibraryExportInfo(
                projectFolderFullpath,
                exportFullpath,
                dllFileFullPathList,
                modelIconFullpathList
                );

            //------------------------
            // ライブラリ出力情報設定
            //------------------------
            // モデルプロジェクト開発コード出力要否
            libraryExportInfo.SetOutputSourceCode(isOutputSource);  
            // モデル解説書ファイルパス                  
            libraryExportInfo.SetManualFullpathList(manualPathList);
            // モデルプロジェクト開発コードファイルパスリスト
            libraryExportInfo.SetDevelopmentSourceFullPathList(devSrcFullPathList);
            // プロパティプロジェクト開発コードファイルパスリスト
            libraryExportInfo.SetPropertyProjectDevSourceFullPathList(propertyDevSrcFullPathList);


            // プロパティ画面プロジェクト情報作成/設定
            if ((object)propertyPrjFldFullPath != null)
            {   // プロパティ画面プロジェクト出力要の場合

                // プロパティ画面プロジェクトフォルダパス取得/設定
                libraryExportInfo.SetPropertyProjectFolderFullPath(propertyPrjFldFullPath);

                // 実行形式ファイルパスリスト取得/設定
                HySDataLinkedList propertyPrjDllFileFullPathList = GetDllFileFullpath(propertyPrjFldFullPath);
                if (propertyPrjDllFileFullPathList.GetCount() == 0)
                {
                    // ver1.5 エラートレース日本語対応
                    LogOut("ConvertToLibraryExportInfo", Properties.HymcoCUIResources.STATEMENT_PROPERTIES_PROJ_EXEC_NO );
                   // LogOut("ConvertToLibraryExportInfo", "プロパティ画面プロジェクト実行形式ファイル無し");
                    return (-3);
                }
                libraryExportInfo.SetPropertyProjectBinaryFullpathList(propertyPrjDllFileFullPathList);
            }
            
            exportInfo = libraryExportInfo;
            return 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>エラーメッセージ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString message = GetErrorMessage(int errorCode) </para>
        /// </example>
        /// <param name="errorCode">エラーコード</param>
        /// <returns><para>エラーメッセージ</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public HySString GetErrorMessage(
            int errorCode
            )
        {
            string errorComment = "";
            switch (errorCode)
            {
                case (-1):
                    errorComment = "コマンドラインに不正なオプションが指定されました。\r\nコマンドラインに規定されたオプションが設定されているか確認して下さい。";
                    break;
                case (-2):
                    errorComment = "指定されたアーカイブ出力先のファイルは既に存在します。\r\n出力ファイルの名称を変更して再度操作を行って下さい。";
                    break;
                case (-3):
                    errorComment = "コマンドラインの構文が不正です。\r\nコマンドラインが規定された形式であることを確認して下さい。";
                    break;
                case (-4):
                    errorComment = "指定されたプロジェクトフォルダに実行形式ファイルが存在しません。";
                    break;
                case (-5):
                    errorComment = "指定されたプロジェクトフォルダにモデルアイコンが存在しません。";
                    break;
                case (-6):
                    errorComment = "指定されたプロジェクトフォルダパスは存在しません。\r\nプロジェクトフォルダパスを確認して、再度操作を行って下さい。";
                    break;
                case (-7):
                    errorComment = "指定されたプロパティ画面プロジェクトフォルダパスは存在しません。\r\nプロパティ画面フォルダパスを確認して、再度操作を行って下さい。";
                    break;
                case (-8):
                    errorComment = "指定されたプロパティ画面プロジェクトフォルダには実行形式ファイルが存在しません。";
                    break;
                case (-999):
                    errorComment = "ライブラリ出力に失敗しました。";
                    break;
                default:
                    break;
            }
            HySString message = new HySString(errorComment);
            return message;
        }

        /// <summary><para>method outline:</para>
        /// <para>実行形式パス取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataLinkedList filePathList = GetExecuteFileFullpath(projectFolderFullpath) </para>
        /// </example>
        /// <param name="projectFolderFullpath">プロジェクトフォルダ名称</param>
        /// <returns><para>実行形式ファイルパスリスト</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static private HySDataLinkedList GetDllFileFullpath(
            HySString projectFolderFullpath
            )
        {
            HySDataLinkedList dllFileFullpath = new HySDataLinkedList();   
         
            // Ver1.5 追加（プラットフォーム取得）
            string sPlatform = "\\x86";

            if (System.Environment.Is64BitProcess == true)
            {
                sPlatform = "\\x64";
            }

            // 実行形式ファイルの格納フォルダ存在有無チェック
            HySString dllFolderFullpath = (projectFolderFullpath + "\\bin\\Release\\");
          //HySString dllFolderFullpath = (projectFolderFullpath + "\\bin" + sPlatform + "\\Release" + "\\");   // Ver1.5 追加（プラットフォーム取得）

            if (System.IO.Directory.Exists(dllFolderFullpath.ToString()))
            {

                // 実行形式ファイルの格納フォルダ以下のファイルを全て取得
                HySDataLinkedList pathList = HySFileUtility.GetFiles(dllFolderFullpath);

                // プロジェクトファイルリスト取得
                HySDataLinkedList projectFileList = 
                    GetProjectFileFullPathList(GetLowestFolder(projectFolderFullpath), pathList);

                // 実行形式ファイル(.dll)パスのみリストに追加
                for (int i = 0; i < projectFileList.GetCount(); i++)
                {
                    HySString fileFullPath = (HySString)(projectFileList.GetData(i));
                    HySString extension = HySFileUtility.GetExtension(fileFullPath);
                    if (((object)extension != null) && extension == ".dll" || extension == ".DLL")
                    {   // ファイル拡張子がdllの場合
                        dllFileFullpath.AddLast(fileFullPath);
                    }
                }
            }
            return dllFileFullpath;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルアイコンファイルパスリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataLinkedList modelIconPathList = GetModelIconPathList(HySString projectFolderFullpath) </para>
        /// </example>
        /// <param name="projectFolderFullpath">プロジェクトフォルダ名称</param>
        /// <returns><para>モデルアイコンファイルパスリスト</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static private HySDataLinkedList GetModelIconPathList(
            HySString projectFolderFullpath
            )
        {
            HySDataLinkedList modelIconFilePathList = new HySDataLinkedList();
            HySString modelIconFolderRelativePath = HySEnvInf.GetEnvInf("ICON_FILE_PATH");

            if (System.IO.Directory.Exists((projectFolderFullpath + modelIconFolderRelativePath).ToString()))
            {
                // モデルアイコンフォルダ以下のファイルパスのリストを取得
                HySDataLinkedList pathList = HySFileUtility.GetFiles(projectFolderFullpath + modelIconFolderRelativePath);

                for (int i = 0; i < pathList.GetCount(); i++)
                {
                    HySString extension = HySFileUtility.GetExtension((HySString)pathList.GetData(i));
                    if (((object)extension != null) && (extension.ToString() == ".ico" || extension.ToString() == ".ICO"))
                    {
                        modelIconFilePathList.AddLast(pathList.GetData(i));
                    }
                }
            }
            return modelIconFilePathList;
        }

        /// <summary><para>method outline:</para>
        /// <para>要素モデル詳細情報ファイルパスリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataLinkedList manualList = GetManualPathList(HySString projectFolderFullpath) </para>
        /// </example>
        /// <param name="projectFolderFullpath">プロジェクトフォルダ名称</param>
        /// <returns><para>要素モデル詳細情報ファイルパスリスト</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static private HySDataLinkedList GetManualPathList(
            HySString projectFolderFullpath
            )
        {
            HySDataLinkedList modelManualFilePathList = new HySDataLinkedList();
            HySString manualFolderRelativePath = HySEnvInf.GetEnvInf("MODEL_MANUAL_PATH");

            if (System.IO.Directory.Exists((projectFolderFullpath + manualFolderRelativePath).ToString()))
            {
                // モデル解説書フォルダ以下のファイルパスのリストを取得
                HySDataLinkedList pathList = HySFileUtility.GetFiles(projectFolderFullpath + manualFolderRelativePath);

                for (int i = 0; i < pathList.GetCount(); i++)
                {
                    HySString extension = HySFileUtility.GetExtension((HySString)pathList.GetData(i));
                    if (((object)extension != null) && (extension.ToString() == ".pdf" || extension.ToString() == ".PDF"))
                    {
                        modelManualFilePathList.AddLast(pathList.GetData(i));
                    }
                }
            }
            return modelManualFilePathList;
        }

        /// <summary><para>method outline:</para>
        /// <para>開発コードファイルパスリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataLinkedList devSrcFullPathList = GetDevelopmentSourceFullPathList(HySString projectFolderFullpath) </para>
        /// </example>
        /// <param name="projectFolderFullpath">プロジェクトフォルダ名称</param>
        /// <returns><para>開発コードファイルパスリスト</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>プロジェクトフォルダ以下のファイルをリストで返却する。</para>
        /// <para>ただし、bin,objフォルダ以下のファイルは含まないものとする。(フォルダパスのみリストに追加する。)</para>
        /// </remarks>
        static private HySDataLinkedList GetDevelopmentSourceFullPathList(
            HySString projectFolderFullpath
            )
        {
            HySDataLinkedList devSrcFullPathList = new HySDataLinkedList();

            string[] files = Directory.GetFiles(
                projectFolderFullpath.ToString(),
                WILD_CARD_CHAR,
                System.IO.SearchOption.AllDirectories
                );
            HySString extension = new HySString();
            bool existBin = false;
            bool existObj = false;
            bool binResult;
            bool objResult;

            // "bin","obj"フォルダ以外のファイルフルパスをリストに追加
            for (int i = 0; i < files.Length; i++)
            {
                // プロジェクトフォルダ以下のbin,objフォルダ以外のファイル名フルパスをリストに追加
                binResult = files[i].StartsWith(projectFolderFullpath.ToString() + "\\bin\\");
                objResult = files[i].StartsWith(projectFolderFullpath.ToString() + "\\obj\\");
                
                if (binResult == true)
                {
                    existBin = true;
                }
                else if (objResult == true)
                {
                    existObj = true;
                }
                else if (binResult == false && objResult == false)
                {
                    HySString data = new HySString(files[i]);
                    devSrcFullPathList.AddLast(data);
                }
            }

            // "bin","obj"フォルダが1つでも存在する場合は空フォルダをリストに追加
            if (existBin == true)
            {
                HySString binFolderPath = new HySString(projectFolderFullpath.ToString() + @"\bin");
                devSrcFullPathList.AddLast(binFolderPath);
            }
            if (existObj == true)
            {
                HySString objFolderPath = new HySString(projectFolderFullpath.ToString() + @"\obj");
                devSrcFullPathList.AddLast(objFolderPath);
            }

            return devSrcFullPathList;
        }

        /// <summary><para>method outline:</para>
        /// <para>プロジェクト名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string projectName = GetLowestFolder(HySString projectFolderFullPath)</para>
        /// </example>
        /// <param name="projectFolderFullPath">プロジェクトフォルダ名称</param>
        /// <returns><para>プロジェクト名称</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static private string GetLowestFolder(HySString projectFolderFullPath)
        {
            string path = projectFolderFullPath.ToString();
            string[] splitPath = path.Split('\\');

            string projectName = "";
            if (splitPath[splitPath.Length - 1] == "")
            {
                projectName = splitPath[splitPath.Length - 2];
            }
            else
            {
                projectName = splitPath[splitPath.Length - 1];
            }

            return projectName;
        }
        
        /// <summary><para>method outline:</para>
        /// <para>プロジェクトファイルリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string projectName = GetProjectFileFullPathList(projectName, fileList)</para>
        /// </example>
        /// <param name="projectName">プロジェクト名称</param>
        /// <param name="fileList">ファイルフルパスリスト</param>
        /// <returns><para>無し</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ファイル名にプロジェクト名を含むファイルパスのリストを返却する</para>
        /// </remarks>
        static private HySDataLinkedList GetProjectFileFullPathList(
            string projectName,
            HySDataLinkedList fileList
            )
        {
            HySDataLinkedList list = new HySDataLinkedList();
            for (int i = 0; i < fileList.GetCount(); i++)
            {
                HySString fileFullPath = (HySString)(fileList.GetData(i));
                HySString fileName = HySFileUtility.GetFileName(fileFullPath);
                if ((object)fileName != null && fileName.ToString().Contains(projectName) == true)
                {   // プロジェクト名称を含むファイルの場合
                    list.AddLast(fileFullPath);
                }
            }
            return list;
        }

        /// <summary><para>method outline:</para>
        /// <para>ログ出力処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> LogOut(methodName, message) </para>
        /// </example>
        /// <param name="methodName">メソッド名称</param>
        /// <param name="message">出力メッセージ</param>
        /// <returns><para>無し</para></returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public void LogOut(string methodName, string message)
        {
            HySLog.LogOut(HySLog.ONLINE, "McLibraryExportArgsConverter", methodName, message);
        }
    }
}
