﻿// <summary>ソースコード：ＨＹＭＣＯＧＲ化演算要素クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSData;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Controller;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.Interface.Model;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.StructInfo;
using CommonMP.HYMCO.CoreImpl.Model;
using CommonMP.HYMCO.CoreImpl.Tool;

namespace CommonMP.HYMCO.CoreImpl.Controller
{
    /// <summary><para>class outline:</para>
    /// <para>要素共通基底クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    public abstract class McCmnElement : McElement
    {
        /// <summary>演算インスタンス識別ＩＤ</summary>
        protected HySID m_csID;
        /// <summary>要素名称</summary>
        protected HySString m_csName;
        /// <summary>計算実行モデルクラス</summary>
        protected McCalModel m_csCalModel;

        /// <summary>受信端子リスト</summary>
        protected List<McReceivePort> m_csReceivePortList = new List<McReceivePort>();
        /// <summary>送信端子リスト</summary>
        protected List<McSendPort> m_csSendPortList = new List<McSendPort>(); 

        /// <summary>受信情報リスト</summary>
        protected HySDataLinkedList m_csReceiveInfoList = new HySDataLinkedList();
        /// <summary>送信情報リスト</summary>
        protected HySDataLinkedList m_csSendInfoList = new HySDataLinkedList();

        /// <summary>ユニーク識別子を作る為の一貫番号</summary>
        protected static ulong m_lElmNo = 0;

        /// <summary> Calculate()メソッドコールフラグ DataFusion()メソッドコール前 </summary>
        protected long m_lCalculate_Call_Before_DataFution_Call = 1; // ０でないことが重要

        /// <summary>モデル初期化情報 </summary>
        protected McPropertyInfoRoot m_csInitializeData=null;

        /// <summary><para>method outline:</para>
        /// <para>ユニーク識別子自動付与用の一貫番号設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McSendPort.SetPrimNo(lNo)</para>
        /// </example>
        /// <param name="ulNo">設定番号</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static void SetPrimNo(ulong ulNo)
        {
            m_lElmNo = ulNo;
        }

        /// <summary><para>method outline:</para>
        /// <para>コンストラクター</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McCmnElement csElm = new McCmnElement()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>コンストラクターから呼ばれる</para>
        /// </remarks>
        public McCmnElement()
        {
            this.Init();
            //HySID csID = new HySID(new HySString("Elm"), new HySString(), new HySString(m_lElmNo.ToString()));
            HySID csID = new HySID(new HySString("Elm") + new HySString() + new HySString(m_lElmNo.ToString()));
            this.SetID(csID);
        }

        /// <summary><para>method outline:</para>
        /// <para>コンストラクター</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McCmnElement csElm = new McCmnElement( csID ) </para>
        /// </example>
        /// <param name="csID">識別子</param>
        /// <returns>インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McCmnElement(HySIdentifier csID)
        {
            this.Init();
            this.SetID(csID);
        }

        /// <summary><para>method outline:</para>
        /// <para>初期化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McCmnElement.Init( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>コンストラクターから呼ばれる</para>
        /// </remarks>
        protected void Init()
        {
            m_lElmNo += 1;
        }


        /// <summary><para>method outline:</para>
        /// <para>識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetID( csID ) </para>
        /// </example>
        /// <param name="csID">識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public void SetID(HySIdentifier csID)
        {
            m_csID = (HySID)csID;
            if (m_csCalModel != null)
            {
                McCalInfo csCalInfo = m_csCalModel.GetCalInfo();
                if (csCalInfo != null)
                {
                    csCalInfo.SetID(csID);
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySIdentifier csIdentifier = GetID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public HySIdentifier GetID()
        {
            return m_csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>識別子による同一比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Equal( sID ) </para>
        /// </example>
        /// <param name="sID">比較する種別識別子</param>
        /// <returns>  true :＝ , false:≠</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public bool Equal(HySIdentifier sID)
        {
            return m_csID.Equals(sID);
        }

        /// <summary><para>method outline:</para>
        /// <para>要素名称設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetElementName( csName ) </para>
        /// </example>
        /// <param name="csName">要素名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public void SetElementName(HySString csName)
        {
            m_csName = csName;
        }
        /// <summary><para>method outline:</para>
        /// <para>要素名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csName = GetElementName( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>要素名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public HySString GetElementName()
        {
            return m_csName;
        }

        /// <summary><para>method outline:</para>
        /// <para>送信端子を取り付ける</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = AddSendPort( csSendPort ) </para>
        /// </example>
        /// <param name="csSendPort">接続端子</param>
        /// <returns>=true:正常、=-false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        internal virtual bool AddSendPort(McPort csSendPort)
        {
            bool bRtn =false;
            McSendPort csPort = csSendPort as McSendPort;
            if (csPort != null)
            {
                csPort.SetElement(this);
                m_csSendPortList.Add(csPort);
                bRtn = true;
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>受信端子を取り付ける</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = AddReceivePort( csReceivePort ) </para>
        /// </example>
        /// <param name="csReceivePort">受信接続端子</param>
        /// <returns>=true:正常、=-false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        internal virtual bool AddReceivePort(McPort csReceivePort)
        {
            bool bRtn = false;
            McReceivePort csPort = csReceivePort as McReceivePort;
            if (csPort != null)
            {
                csPort.SetElement(this); 
                m_csReceivePortList.Add(csPort);
                bRtn = true;
            }

            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>端子の取り外し</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = DeletePort( csID ) </para>
        /// </example>
        /// <param name="csID">取り外す接続端子のＩＤ</param>
        /// <returns>=true:正常、=-false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public virtual bool DeletePort(HySIdentifier csID)
        {
            int iCnt = m_csSendPortList.Count;
            for(int iLp=0;iLp<iCnt;iLp++)
            {    // ここで、リスト内を調べて同じIDのPortがあれば削除する
                McSendPort csSndPort = m_csSendPortList[iLp]; 
                if (csSndPort.Equal(csID) == true)
                {
                    m_csSendInfoList.RemoveData(csSndPort.GetTranInfo()); // 送信情報も削除する
                    m_csSendPortList.Remove(csSndPort); // 送信端子を削除する
                    break;
                }
            }
            iCnt = m_csReceivePortList.Count;
            for (int iLp = 0; iLp < iCnt; iLp++)
            {    // ここで、リスト内を調べて同じIDのPortがあれば削除する
                McReceivePort csRevPort = m_csReceivePortList[iLp]; 
                if (csRevPort.Equal(csID) == true)
                {
                    m_csReceiveInfoList.RemoveData(csRevPort.GetTranInfo()); // 送信情報も削除する
                    m_csReceivePortList.Remove(csRevPort); // 受信端子を削除する
                    break;
                }
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>端子の検索</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McPort csPort = GetPort( csPortID ) </para>
        /// </example>
        /// <param name="csPortID">接続端子のＩＤ</param>
        /// <returns>取得端子（=null の場合は該当無し）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        internal virtual McPort GetPort(HySIdentifier csPortID)
        {
            McPort csRtn = null;
            int iCnt = m_csSendPortList.Count;
            for (int iLp = 0; iLp < iCnt; iLp++)
            {   
                McSendPort csSndPort = m_csSendPortList[iLp];
                if (csSndPort.Equal(csPortID) == true)
                {
                    csRtn = csSndPort;
                    break;
                }
            }
            if (csRtn == null)
            {
                iCnt = m_csReceivePortList.Count;
                for (int iLp = 0; iLp < iCnt; iLp++)
                {   
                    McReceivePort csRevPort = m_csReceivePortList[iLp];
                    if (csRevPort.Equal(csPortID) == true)
                    {
                        csRtn = csRevPort;
                        break;
                    }
                }
            }

            return csRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>送信端子生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McSendPort csPort = CreateSendPort( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>接続端子インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        internal virtual McSendPort CreateSendPort()
        {
            McSendPort csRtn = new McSendPort();
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>受信端子生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McReceivePort csPort = CreateReceivePort( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>接続端子インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        internal virtual McReceivePort CreateReceivePort()
        {
            McReceivePort csRtn = new McReceivePort();
            return csRtn;
        }

        //================
        // 各種設定メソッド
        //================

        /// <summary><para>method outline:</para>
        /// <para>内部にモデルクラスを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCalModel( csCalModel ) </para>
        /// </example>
        /// <param name="csCalModel">演算モデル</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public virtual void SetCalModel(McCalModel csCalModel)
        {
            m_csCalModel = csCalModel;
            if (csCalModel != null)
            {
                m_csCalModel.SetElement(this);
            }
            else
            {
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SetCalModel", Properties.HymcoImplResources.STATEMENT_NO_CALMODEL );
               // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SetCalModel", "No CalModel");
            }

        }
        /// <summary><para>method outline:</para>
        /// <para>モデルクラスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McCalModel csMdl = GetCalModel( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McCalModel モデルクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public virtual McCalModel GetCalModel()
        {
            return m_csCalModel;
        }
        /// <summary><para>method outline:</para>
        /// <para>モデル演算中の情報格納クラスを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCalInfo(csCalInfo)</para>
        /// </example>
        /// <param name="csCalInfo">演算結果格納するクラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCalInfo(McCalInfo csCalInfo)
        {
            if (m_csCalModel != null)
            {
                m_csCalModel.SetCalInfo(csCalInfo);
                if (csCalInfo != null)
                {
                    csCalInfo.SetID(this.GetID());
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>演算中の情報格納クラスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McCalInfo csInfo = GetCalInfo( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McCalInfo 演算中の情報格納クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public virtual McCalInfo GetCalInfo()
        {
            McCalInfo csRtn = null;
            if (m_csCalModel != null)
            {
                csRtn = m_csCalModel.GetCalInfo();
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>解法を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSolutionType(elType)</para>
        /// </example>
        /// <param name="elType">解法（通常計算／収束計算）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSolutionType(McDefine.SolutionType elType)
        {
            if (m_csCalModel != null)
            {
                m_csCalModel.SetSolutionType(elType);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>解法を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetSolutionType()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>解法（McDefine.NORMAL_TYPE/CONVERGENCE_TYPE）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McDefine.SolutionType GetSolutionType()
        {
            if (m_csCalModel != null)
            {
                return m_csCalModel.GetSolutionType();
            }
            else
            {
                return McDefine.SolutionType.NORMAL_TYPE;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>DataFusion()発行のタイミング（演算フロー制御）設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDataFusionTiming(eDtFusionTiming)</para>
        /// </example>
        /// <param name="eDtFusionTiming"> 同期／非同期
        /// McDefine.DataFusionTiming.SYNCHRONOUS：全ての要素が演算終了後DataFutionを行う
        /// McDefine.DataFusionTiming.ASYNCHRONOUS：非同期：要素が演算終了後個別にDataFutionを行う）
        /// </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDataFusionTiming(McDefine.DataFusionTiming eDtFusionTiming)
        {
            if (m_csCalModel != null)
            {
                m_csCalModel.SetDataFusionTiming(eDtFusionTiming);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>DataFusion()発行のタイミング（演算フロー制御）取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McDefine.DataFusionTiming eTm = GetDataFusionTiming()</para>
        /// </example>
        /// <param name=""> 無し</param>
        /// <returns>同期／非同期</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McDefine.DataFusionTiming GetDataFusionTiming()
        {
            if (m_csCalModel != null)
            {
                return m_csCalModel.GetDataFusionTiming();
            }
            else
            {
                return McDefine.DataFusionTiming.SYNCHRONOUS;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>演算時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimulationTime(csCalTime)</para>
        /// </example>
        /// <param name="csCalTime">演算時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimulationTime(HySTime csCalTime)
        {
            m_csCalModel.SetSimulationTime(csCalTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>演算開始時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetStartTime(csStartTime)</para>
        /// </example>
        /// <param name="csStartTime">演算開始時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetStartTime(HySTime csStartTime)
        {
            m_csCalModel.SetStartTime(csStartTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>演算目標時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetTargetTime(csTgtTime)</para>
        /// </example>
        /// <param name="csTgtTime">目標時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetTargetTime(HySTime csTgtTime)
        {
            m_csCalModel.SetTargetTime(csTgtTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>演算中に次の演算目標時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetNextTargetTime(csCalTime,csDltTime)</para>
        /// </example>
        /// <param name="csCalTime">現在時刻</param>
        /// <param name="csDltTime">増加時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetNextTargetTime(ref HySTime csCalTime, ref HySTime csDltTime)
        {
            m_csCalModel.SetNextTargetTime(ref csCalTime, ref csDltTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>演算刻み時間を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDeltaTime(csDltTime)</para>
        /// </example>
        /// <param name="csDltTime">演算刻み時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDeltaTime(HySTime csDltTime)
        {
            m_csCalModel.SetDeltaTime(csDltTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>最終目標時間設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuGoalTime( csTm )</para>
        /// </example>
        /// <param name="csTm">最終目標時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuGoalTime(HySTime csTm)
        {
            this.m_csCalModel.SetSimuGoalTime(csTm);
        }

        /// <summary><para>method outline:</para>
        /// <para>初期化情報を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>PresetInitializeData(csInitializeData)</para>
        /// </example>
        /// <param name="csInitializeData">初期化情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void PresetInitializeData(McPropertyInfoRoot csInitializeData)
        {
            m_csInitializeData = csInitializeData;
        }
        /// <summary><para>method outline:</para>
        /// <para>初期化情報を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McPropertyInfoRoot csInitDt = GetInitializeData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>初期化情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McPropertyInfoRoot GetInitializeData( )
        {
            return m_csInitializeData;
        }

        //=========================
        // 演算実行前処理関連メソッド
        //=========================

        /// <summary><para>method outline:</para>
        /// <para>計算準備(接続チェック等)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = Prepare( ref csErrorInfo)</para>
        /// </example>
        /// <param name="csErrorInfo">接続エラー情報</param>
        /// <returns>=true:接続エラー無し　 =false:接続エラー有り</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool Prepare(ref McStructErrorInfo csErrorInfo)
        {
            bool bRtn = true;

            // 送信側情報を設定する
            m_csSendInfoList.Clear();   // クリア
            int iCnt = m_csSendPortList.Count;
            for (int iLp = 0; iLp < iCnt; iLp++)
            {   // 送信ポート数繰り返し
                McSendPort csSndPort = m_csSendPortList[iLp]; // 送信ポート取得
                McTranInfo csSndInfo = csSndPort.GetTranInfo(); // 送信伝送データ取得
                if (csSndInfo != null)
                {   // 送信伝送データ有り
                    m_csSendInfoList.AddLast(csSndInfo);
                    csSndInfo.GetCommonInf().SetUpperElement(this);
                    //csSndInfo.SetUpperElementID(this.m_csID);
                    //csSndInfo.SetUpperElementName(this.m_csName);
                    // 中継端子相手の場合を考える事（但し此処ではダメ：相手先がFixしていない場合がある）
                }
                else
                {   // 送信伝送データ無しならばエラー
                    //csErrorInfo.xxxx エラー設定
                    ((McStructCheckData)csErrorInfo).AddCheckData(this.m_csID,new HySObjectKind("McElement"),
                        McDefine.MessageLevel.HYM_MODELCHECK_ERROR, new HySString("McSendPort have not McTranInfo"));
                    bRtn = false;
               }
            }
            // 受信側情報を設定する
            m_csReceiveInfoList.Clear();    // クリア
            iCnt = m_csReceivePortList.Count;
            for (int iLp = 0; iLp < iCnt; iLp++)
            {   // 受信ポート数繰り返し
                McReceivePort csRevPort = m_csReceivePortList[iLp]; // 受信ポート取得
                McTranInfo csRevInfo = csRevPort.GetTranInfo();  // 受信伝送データ取得
                if( csRevInfo != null )
                {   // 受信伝送データ有り
                    m_csReceiveInfoList.AddLast(csRevInfo);
                    csRevInfo.GetCommonInf().SetLowerElement(this);
                    //csRevInfo.SetLowerElementID(this.m_csID);
                    //csRevInfo..SetUpperElementName(this.m_csName);
                    // 中継端子相手の場合を考える事（但し此処ではダメ：相手先がFixしていない場合がある）
                }
                else
                {   // 受信伝送データ無しならばエラー
                    //csErrorInfo.xxxx エラー設定
                    ((McStructCheckData)csErrorInfo).AddCheckData(this.m_csID, new HySObjectKind("McElement"),
                            McDefine.MessageLevel.HYM_MODELCHECK_ERROR, new HySString("McReceivePort have not McTranInfo"));
                    bRtn = false;
               }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SimInitialize()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>bool true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool SimInitialize()
        {
            bool bRtn = false;
            if (m_csCalModel != null)
            {
                try
                {
                    //checked <- cheked 宣言は　チェックの直前で行わなければ、作用しない
                    {
                        bRtn = m_csCalModel.Initialize(ref m_csInitializeData, ref m_csReceiveInfoList, ref m_csSendInfoList);
                    }
                }
                catch (OverflowException ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SimInitialize", 
                        Properties.HymcoImplResources.STATEMENT_OVER_EXEP_CATCH  + Properties.HymcoImplResources.STATEMENT_IN_CLASS
                        + m_csCalModel.ToString() + "Initialize() :: " + ex.Message); //ex.StackTrace);
                    // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SimInitialize", "OverflowException Catch" + " in " + m_csCalModel.ToString() + "Initialize() :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "OverflowException Catch", 
                        Properties.HymcoImplResources.STATEMENT_IN_CLASS + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "OverflowException Catch", " in " + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);

                    McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                    if (csErrorInfo != null)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                                Properties.HymcoImplResources.STATEMENT_OVER_EXEP_CATCH + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);
                       // csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                       //         "OverflowException Catch in " + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);
                    }
                    // 異常通知
                    bRtn = false;
                }
                catch (Exception ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SimInitialize", 
                        Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS
                        + m_csCalModel.ToString() + "Initialize() :: " + ex.Message); //ex.StackTrace);
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SimInitialize", "Exception Catch" + " in " + m_csCalModel.ToString() + "Initialize() :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch", 
                        Properties.HymcoImplResources.STATEMENT_IN_CLASS + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch", " in " + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);

                    McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID() ) as McStructErrorInfo;
                    if (csErrorInfo != null)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                                 Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);
                       // csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                       //         "Exception Catch in " + m_csCalModel.ToString() + ".Initialize() :: " + ex.Message);
                    }
                    // 異常通知
                    bRtn = false;
                }
                //m_csCalModel.DataFusion(ref m_csSendInfoList); // 初回に内部情報を外に出しておく必要がある
                // モデルによっては、初回情報を出さないものもあるため、初回のDataFusionのコールはモデルに任せる。
                if (bRtn == false)
                {
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "SimInitialize",
                        Properties.HymcoImplResources.STATEMENT_INIT + "_" + Properties.HymcoImplResources.STATEMENT_ERROR
                        +" :" + Properties.HymcoImplResources.STATEMENT_MODEL_NAME + " =" + m_csCalModel.ToString());
                   // HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "SimInitialize", "Initialize_ERROR :Model = " + m_csCalModel.ToString());
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " Initialize Error",
                        Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ")");
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " Initialize Error", " in (" + m_csCalModel.ToString() + ")");
                    McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                    if (csErrorInfo != null)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                                Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csCalModel.ToString() + ".Initialize()");
                       // csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                       //         "Error return from " + m_csCalModel.ToString() + ".Initialize()");
                    }
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ等情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SetProperty( csElmID, csMdlInfo, ref csErrorInfo)</para>
        /// </example>
        /// <param name="csElmID">変更対象要素識別子</param>
        /// <param name="csMdlInfo">変更情報</param>
        /// <param name="csErrorInfo">接続エラー情報</param>
        /// <returns>=true:変更対象であった、!=true:変更対象ではない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>モデル生成時にモデル内部情報を設定する</para>
        /// </remarks>
        internal abstract bool SetProperty(HySID csElmID, McModelInfo csMdlInfo, ref McStructErrorInfo csErrorInfo);

        /// <summary><para>method outline:</para>
        /// <para>演算実行中断中のプロパティ等情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SetOnlineProperty( csElmID, csMdlInfo, ref csErrorInfo)</para>
        /// </example>
        /// <param name="csElmID">変更対象要素識別子</param>
        /// <param name="csMdlInfo">変更情報</param>
        /// <param name="csErrorInfo">接続エラー情報</param>
        /// <returns>=true:変更対象であった、!=true:変更対象ではない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中にプロパティ、初期値等を変更して動作させ場合等に使用する</para>
        /// </remarks>
        internal abstract bool SetOnlineProperty(HySID csElmID, McModelInfo csMdlInfo, ref McStructErrorInfo csErrorInfo);

        //=======================
        // 演算実行処理関連メソッド
        //=======================

        /// <summary><para>method outline:</para>
        /// <para>モデル演算可能かをチェックする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsAbleToCalculate(ref csErrorInf)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <returns>=true:演算可能、=false:演算不能</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool IsAbleToCalculate(ref McStructErrorInfo csErrorInf)
        {
            bool bRtn = true;
            if (m_csCalModel == null)
            {
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddSimuErrorData(m_csID, McModelLibraryDefine.ELEMENT, 
                       Properties.HymcoImplResources.STATEMENT_MCCALMODEL_NO_SET );
               // csErrorInf.AddSimuErrorData(m_csID, McModelLibraryDefine.ELEMENT, "McCalModel is not set");
                bRtn = false;
            }
            else
            {
                try
                {
                    bRtn = m_csCalModel.IsAbleToCalculate(ref m_csReceiveInfoList, ref m_csSendInfoList, ref csErrorInf);
                }
                catch (Exception ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Calculate", 
                        Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS
                        + m_csCalModel.ToString() + ".IsAbleToCalculate() :: " + ex.Message); //ex.StackTrace);
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Calculate", "Exception Catch" + " in Class " + m_csCalModel.ToString() + ".IsAbleToCalculate() :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch",
                        Properties.HymcoImplResources.STATEMENT_IN_CLASS + m_csCalModel.ToString() + ".IsAbleToCalculate() :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch", " in " + m_csCalModel.ToString() + ".IsAbleToCalculate() :: " + ex.Message);
                   // ver1.5 エラートレース日本語対応
                    csErrorInf.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                        Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS
                        + m_csCalModel.ToString() + ".IsAbleToCalculate() :: " + ex.Message); //ex.StackTrace);
                   // csErrorInf.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                   //           "Exception Catch in " + m_csCalModel.ToString() + ".IsAbleToCalculate() :: " + ex.Message);
                    // 異常通知
                    bRtn = false;
                }
            }
            return bRtn;
        }
            

        /// <summary><para>method outline:</para>
        /// <para>モデル演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = Calculate()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long Calculate( )
        {
            long lRtn = -1;
            if (m_csCalModel != null)
            {   // 内部モデルに計算を行わせる
                try
                {
                    //checked //<- cheked 宣言は　チェックの直前で行わなければ、作用しない
                    {
                        lRtn = m_csCalModel.Calculate(ref m_csReceiveInfoList);
                    }

                }
                catch (OverflowException ex)
                {
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Calculate", Properties.HymcoImplResources.STATEMENT_OVER_EXEP_CATCH
                         + Properties.HymcoImplResources.STATEMENT_IN_CLASS + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message); //ex.StackTrace);
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Calculate", "OverflowException Catch in Class" + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "OverflowException Catch",
                        Properties.HymcoImplResources.STATEMENT_IN_CLASS + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "OverflowException Catch", " in " + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);

                    McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                    if (csErrorInfo != null)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                                Properties.HymcoImplResources.STATEMENT_OVER_EXEP_CATCH + Properties.HymcoImplResources.STATEMENT_IN_CLASS 
                                + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);
                       // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                       //         "OverflowException Catch in " + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);
                    }

                    // 異常通知
                    lRtn = -1;

                }
                catch (Exception ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Calculate",
                        Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS 
                        + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message); //ex.StackTrace);
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Calculate", "Exception Catch in Class" + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch",
                        Properties.HymcoImplResources.STATEMENT_IN_CLASS + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch", " in " + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);

                    McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                    if (csErrorInfo != null)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                                Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS 
                                + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);
                       // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                       //         "Exception Catch in " + m_csCalModel.ToString() + ".Calculate() :: " + ex.Message);
                    }

                    // 異常通知
                    lRtn = -1;
                }
                m_lCalculate_Call_Before_DataFution_Call += 1; // Calculate コール数カウントアップ
            }
            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsConverged( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=true:収束完了 =false:未収束</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool IsConverged()
        {
            bool bRtn = false;
            if (m_csCalModel != null)
            {   // 内部モデルに収束判定を行わせる
                try
                {
                    bRtn = m_csCalModel.IsConverged( );
                }
                catch (Exception ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    string DispStatement = "IsConverged" + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS
                        + "(" +  m_csCalModel.ToString() + ") :: " + ex.Message; 
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::IsConverged", DispStatement );
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::IsConverged", "IsConverged Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " IsConverged Exception Catch",
                       Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " IsConverged Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);
                    // 異常通知
                    bRtn = true;
                }
            }

            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>モデル演算結果を外部のエレメントに対して公開する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = DataFusion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long DataFusion( )
        {
            long lRtn = -1;
            if (m_csCalModel != null)
            {   // 内部モデルに情報書き出しを行わせる
                lRtn = 0;
                if (m_lCalculate_Call_Before_DataFution_Call > 0)
                {   // DataFusion() 前に　Calculateが行われていたならば
                    try
                    {
                        lRtn = m_csCalModel.DataFusion(ref m_csSendInfoList);
                    }
                    catch (Exception ex)
                    {
                        // ログ出力
                       // ver1.5 エラートレース日本語対応
                        string DispStatement = Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS
                            + m_csCalModel.ToString() + ".DataFusion() :: " + ex.Message; //ex.StackTrace);
                        HySLog.LogOut(HySLog.ONLINE, "McCmnElement::DataFusion", DispStatement );
                       // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::DataFusion", "Exception Catch in Class " + m_csCalModel.ToString() + ".DataFusion() :: " + ex.Message); //ex.StackTrace);
                       // ver1.5 エラートレース日本語対応
                        McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch",
                            Properties.HymcoImplResources.STATEMENT_IN_CLASS + m_csCalModel.ToString() + ".DataFusion() :: " + ex.Message);
                       // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "Exception Catch", " in " + m_csCalModel.ToString() + ".DataFusion() :: " + ex.Message);
                        
                        McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                        if (csErrorInfo != null)
                        {
                           // ver1.5 エラートレース日本語対応
                            csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                                    Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS 
                                    + m_csCalModel.ToString() + ".DataFusion() :: " + ex.Message);
                           // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                           //         "Exception Catch in " + m_csCalModel.ToString() + ".DataFusion() :: " + ex.Message);
                        }
                        // 異常通知
                        lRtn = -1;
                    }
                }
                // Calculate コール数リセット
                m_lCalculate_Call_Before_DataFution_Call = 0;
            }
            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>強制的にDataFusionを行わせる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = ForceDataFusion( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long ForceDataFusion()
        {
            m_lCalculate_Call_Before_DataFution_Call += 1; // Calculate コール数カウントアップ
            return DataFusion();
        }

        /// <summary><para>method outline:</para>
        /// <para>計算開始時に動作する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReadyCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool ReadyCalculation()
        {
            bool bRtn = false;
            try
            {
                bRtn = m_csCalModel.ReadyCalculation();
            }
            catch (Exception ex)
            {
                // ログ出力
               // ver1.5 エラートレース日本語対応
                string DispStatement = Properties.HymcoImplResources.STATEMENT_READY_CALC + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION
                    + Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message; //ex.StackTrace);
                HySLog.LogOut(HySLog.ONLINE, "McCmnElement::ReadyCalculation", DispStatement );
               // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::ReadyCalculation", "ReadyCalculation Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " ReadyCalculation Exception Catch", 
                    Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
               // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " ReadyCalculation Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);
                McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                if (csErrorInfo != null)
                {
                    //csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                    //        "Exception Catch in " + m_csCalModel.ToString() + ".ReadyCalculation() :: " + ex.Message);
                   // ver1.5 エラートレース日本語対応
                    csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                            Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS 
                            + m_csCalModel.ToString() + ".ReadyCalculation() :: " + ex.Message);
                   // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                   //         "Exception Catch in " + m_csCalModel.ToString() + ".ReadyCalculation() :: " + ex.Message);
                }
                // 異常通知
                bRtn = false;
            }
            if (bRtn != true)
            {
               // ver1.5 エラートレース日本語対応
                string DispStatement = Properties.HymcoImplResources.STATEMENT_READY_CALC + "＿" + Properties.HymcoImplResources.STATEMENT_ERROR
                    + " :Model =" +  m_csCalModel.ToString();
                HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "ReadyCalculation", DispStatement );
               // HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "ReadyCalculation", "ReadyCalculation_ERROR :Model = " + m_csCalModel.ToString());
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "ReadyCalculation",
                    Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csCalModel.ToString() + ".ReadyCalculation() :: ");
               // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "ReadyCalculation", " Error return from " + m_csCalModel.ToString() + ".ReadyCalculation() :: ");
                McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                if (csErrorInfo != null)
                {
                    //csErrorInfo.AddCheckErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                    //        "Error return from  " + m_csCalModel.ToString() + ".ReadyCalculation() ");
                   // ver1.5 エラートレース日本語対応
                    csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                            Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM  + m_csCalModel.ToString() + ".ReadyCalculation() ");
                   // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                   //         "Error return from  " + m_csCalModel.ToString() + ".ReadyCalculation() ");
                }
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>計算終了時に動作する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CompleteCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool CompleteCalculation()
        {
            bool bRtn = false;
            try
            {
                bRtn = m_csCalModel.CompleteCalculation();
            }
            catch (Exception ex)
            {
                // ログ出力
               // ver1.5 エラートレース日本語対応
                string DispStatement = Properties.HymcoImplResources.STATEMENT_COMPLETE_CALC + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION
                    + Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message; //ex.StackTrace);
                HySLog.LogOut(HySLog.ONLINE, "McCmnElement::CompleteCalculation", DispStatement );
               // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::CompleteCalculation", "CompleteCalculation Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " CompleteCalculation Exception Catch",
                    Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
               // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " CompleteCalculation Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);
                McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                if (csErrorInfo != null)
                {
                   // ver1.5 エラートレース日本語対応
                    csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                            Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS 
                            + m_csCalModel.ToString() + ".CompleteCalculation() :: " + ex.Message);
                   // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                   //         "Exception Catch in " + m_csCalModel.ToString() + ".CompleteCalculation() :: " + ex.Message);
                }
                // 異常通知
                bRtn = false;
            }
            if (bRtn != true)
            {
               // ver1.5 エラートレース日本語対応
                string DispStatement = Properties.HymcoImplResources.STATEMENT_COMPLETE_CALC + "＿" + Properties.HymcoImplResources.STATEMENT_ERROR
                    + " :Model =" +  m_csCalModel.ToString();
                HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "CompleteCalculation", DispStatement );
               // HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "CompleteCalculation", "CompleteCalculation_ERROR :Model = " + m_csCalModel.ToString());
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "CompleteCalculation", 
                    Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csCalModel.ToString() + ".CompleteCalculation() :: ");
               // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "CompleteCalculation", " Error return from " + m_csCalModel.ToString() + ".CompleteCalculation() :: ");
                McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                if (csErrorInfo != null)
                {
                   // ver1.5 エラートレース日本語対応
                    csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                             Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csCalModel.ToString() + ".CompleteCalculation() ");
                   // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                   //         "Error return from  " + m_csCalModel.ToString() + ".CompleteCalculation() ");
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算中断時に動作する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SuspendCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool SuspendCalculation()
        {
            bool bRtn = false;
            try
            {
                bRtn = m_csCalModel.SuspendCalculation();
            }
            catch (Exception ex)
            {
                // ログ出力
               // ver1.5 エラートレース日本語対応
                string DispStatement = Properties.HymcoImplResources.STATEMENT_SUSPEND_CALC + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION
                    + Properties.HymcoImplResources.STATEMENT_IN_CLASS + "("  + m_csCalModel.ToString() + ") :: " + ex.Message; //ex.StackTrace);
                HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SuspendCalculation", DispStatement );
               // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::SuspendCalculation", "SuspendCalculation Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " SuspendCalculation Exception Catch", 
                    Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
               // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " SuspendCalculation Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);
                McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                if (csErrorInfo != null)
                {
                   // ver1.5 エラートレース日本語対応
                    csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                             Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + Properties.HymcoImplResources.STATEMENT_IN_CLASS 
                             + m_csCalModel.ToString() + ".SuspendCalculation() :: " + ex.Message);
                   // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                   //         "Exception Catch in " + m_csCalModel.ToString() + ".SuspendCalculation() :: " + ex.Message);
                }
                // 異常通知
                bRtn = false;
            }
            if (bRtn != true)
            {
               // ver1.5 エラートレース日本語対応
                string DispStatement = Properties.HymcoImplResources.STATEMENT_SUSPEND_CALC + " :" + Properties.HymcoImplResources.STATEMENT_MODEL_NAME
                    +" =" + m_csCalModel.ToString(); 
                HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "SuspendCalculation", DispStatement ); 
               // HySLog.LogOut(HySLog.TRIAL_RUN, "McCmnElement", "SuspendCalculation", "SuspendCalculation :Model = " + m_csCalModel.ToString());
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "SuspendCalculation", 
                    Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csCalModel.ToString() + ".SuspendCalculation() :: ");
               // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, "SuspendCalculation", " Error return from " + m_csCalModel.ToString() + ".SuspendCalculation() :: ");
                McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.GetOwnerProjectID()) as McStructErrorInfo;
                if (csErrorInfo != null)
                {
                   // ver1.5 エラートレース日本語対応
                    csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                             Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csCalModel.ToString() + ".SuspendCalculation() ");
                   // csErrorInfo.AddSimuErrorData(this.GetID(), McModelLibraryDefine.ELEMENT,
                   //         "Error return from  " + m_csCalModel.ToString() + ".SuspendCalculation() ");
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル内部の時刻を進める</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GainSimulationTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void GainSimulationTime()
        {
            m_csCalModel.GainSimulationTime( );
        }

        /// <summary><para>method outline:</para>
        /// <para>演算用刻み時間を自動的に変更する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ChangeDeltaTimeAutomatically()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ChangeDeltaTimeAutomatically()
        {
            m_csCalModel.ChangeDeltaTimeAutomatically();
        }


        //=======================
        // 要素間結合状態等取得メソッド
        //=======================
        /// <summary><para>method outline:</para>
        /// <para>自要素の上流に位置する要素を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>
        /// List＜McElement＞ csUpperElmList = GetUpperElementList( ) 
        /// </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>  List＜McElement＞ 自要素の上流に位置する要素リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 自らが属するグループ外からの入力の場合には含めない
        /// </para>
        /// </remarks> 
        internal virtual List<McElement> GetUpperElementList()
        {
            List<McElement> csRtn = new List<McElement>();
            McSendPort csSndPort = null;
            //受信端子リスト
            long lNPort = m_csReceivePortList.Count;
            for( int lP=0;lP<lNPort;lP++)
            {
                csSndPort = m_csReceivePortList[lP].GetPartnerPort();
                if (csSndPort != null)
                {
                    //McElement csElm = csSndPort.GetElement();
                    if (csSndPort is McInnerSndPort)
                    {   // 送信先が中継端子内の内部向け送信端子ならば、グループ外からの入力
                        // Do nothing
                    }
                    else
                    {   // 送信先が中継端子でないならば　グループ内の要素からの入力
                        McElement csElm = csSndPort.GetElement();
                        if( csElm != null )
                        {
                            csRtn.Add(csElm);
                        }
                    }
                }
            }
            return csRtn;
        }

        // ========================================

        /// <summary><para>method outline:</para>
        /// <para>計算状態復元のためのデータクラスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McCmnElementOutData csElementOutData = CreateOutData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>データクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract McCmnElementOutData CreateOutData();

        /// <summary><para>method outline:</para>
        /// <para>シミュレーションデータコンテナの内容を設定する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimDataContainerInfo( csDataContainer )</para>
        /// </example>
        /// <param name="csDataContainer">シミュレーションデータコンテナ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimDataContainer(HySSimulationDataContainer csDataContainer)
        {
            m_csCalModel.SetSimDataContainer(csDataContainer);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーションIDを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimulationID(csSimulationID)</para>
        /// </example>
        /// <param name="csSimulationID">シミュレーションID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimulationID(HySIdentifier csSimulationID)
        {
            m_csCalModel.SetSimulationID(csSimulationID);
        }

        /// <summary><para>method outline:</para>
        /// <para>ファイルにモデル内情報を全て書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileOUT(csData)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool FileOUT(HySDataRoot csData)
        {
            bool bRtn = false;

            // 端子そのものをFileOUTしてはならない
            //csFile.SerializeAndSave(m_csReceivePortList);
            //csFile.SerializeAndSave(m_csSendPortList);

            // 自分のポートの伝送データに対して　ファイル保存する必要がある

            // リストに繋がっている伝送データをファイルアウトする
            // 内部の演算モデルクラスにファイルアウトを指示する

            // 自要素のメンバ保存
            if (csData != null && csData as McCmnElementOutData != null)
            {
                McCmnElementOutData csElementOutData = (McCmnElementOutData)(csData);
                csElementOutData.SetID(m_csID);
                csElementOutData.SetTranInfoData(McDefine.SEND_INFO, m_csSendInfoList);
                csElementOutData.SetTranInfoData(McDefine.RECEIVE_INFO, m_csReceiveInfoList);
                csElementOutData.SetElmNo(m_lElmNo);
                csElementOutData.SetCalculateCallNum(m_lCalculate_Call_Before_DataFution_Call);

                // モデル内メンバの保存
                try
                {
                    bRtn = m_csCalModel.FileOUT(csData);
                }
                catch (Exception ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    string DispStatement = Properties.HymcoImplResources.STATEMENT_FILE_OUT + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION
                        + Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message; //ex.StackTrace);
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::FileOUT", DispStatement );
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::FileOUT", "FileOUT Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " FileOUT Exception Catch",
                        Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " FileOUT Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);

                    // 異常通知
                    bRtn = false;
                }
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>ファイルからモデル情報を全て読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileIN(csData)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool FileIN(HySDataRoot csData)
        {
            bool bRtn = true;
            if (csData == null || csData as McCmnElementOutData == null)
            {
                return false;
            }

            // 端子そのものをFileINしてはならない
            //m_csReceivePortList = csFile.LoadAndDeserialize() as List<McReceivePort>;
            //m_csSendPortList = csFile.LoadAndDeserialize() as List<McSendPort>;

            // 自要素のメンバ復元
            McCmnElementOutData csElementOutData = (McCmnElementOutData)(csData);
            // 識別ID
            m_csID = csElementOutData.GetID();
            // 伝送データの復元(受信情報)
            long lReceiveInfoCnt = m_csReceiveInfoList.GetCount();
            if (lReceiveInfoCnt != 0)
            {
                m_csReceiveInfoList.SetCursorFirst();
                (csElementOutData.GetTranInfoData(McDefine.RECEIVE_INFO)).SetCursorFirst();
            }

            for (long lLp = 0; lLp < lReceiveInfoCnt; lLp++)
            {

                McTranInfo csReceiveInfo = (McTranInfo)(m_csReceiveInfoList.GetCursorData());
                McTranInfo csReceiveInData = (McTranInfo)(csElementOutData.GetTranInfoData(McDefine.RECEIVE_INFO)).GetCursorData();

                // 受信情報の復元
                csReceiveInfo.CopyInfo(csReceiveInData);

                m_csReceiveInfoList.MoveCursorNext();
                (csElementOutData.GetTranInfoData(McDefine.RECEIVE_INFO)).MoveCursorNext();
            }
            // 伝送データの復元(送信情報)
            long lSendInfoCnt = m_csSendInfoList.GetCount();
            if (lSendInfoCnt != 0)
            {
                m_csSendInfoList.SetCursorFirst();
                (csElementOutData.GetTranInfoData(McDefine.SEND_INFO)).SetCursorFirst();
            }

            for (long lLp = 0; lLp < lSendInfoCnt; lLp++)
            {

                McTranInfo csSendInfo = (McTranInfo)(m_csSendInfoList.GetCursorData());
                McTranInfo csSendInData = (McTranInfo)(csElementOutData.GetTranInfoData(McDefine.SEND_INFO)).GetCursorData();

                // 送信情報の復元
                csSendInfo.CopyInfo(csSendInData);

                m_csSendInfoList.MoveCursorNext();
                (csElementOutData.GetTranInfoData(McDefine.SEND_INFO)).MoveCursorNext();
            }
            // 一貫番号
            m_lElmNo = csElementOutData.GetElmNo();
            // Calculateコールフラグ
            m_lCalculate_Call_Before_DataFution_Call = csElementOutData.GetCalculateCallNum();
            // 復元後の整合性確保
            if (m_csID != null)
            {
                m_csID.PostDeserialize();
            }

            // モデル内メンバの復元
            try
            {
                bRtn = m_csCalModel.FileIN(csData);
            }
            catch (Exception ex)
            {
                // ログ出力
               // ver1.5 エラートレース日本語対応
                string DispStatement = Properties.HymcoImplResources.STATEMENT_FILE_IN + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION
                    + Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message; //ex.StackTrace);
                HySLog.LogOut(HySLog.ONLINE, "McCmnElement::FileIN", DispStatement );
               // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::FileIN", "FileIN Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " FileIN Exception Catch",
                    Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
               // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " FileIN Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);

                // 異常通知
                bRtn = false;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>伝送データのリセット</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ResetTranInfo() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ResetTranInfo()
        {
            //m_csReceiveInfoList, ref m_csSendInfoList
            McTranInfo csTrnDt = null;
            long lNumData = m_csReceiveInfoList.GetCount();
            m_csReceiveInfoList.SetCursorFirst();
            for (long lP = 0; lP < lNumData; lP++)
            {   // データ数分
                csTrnDt = (McTranInfo)m_csReceiveInfoList.GetCursorData();
                csTrnDt.ResetData();
                m_csReceiveInfoList.MoveCursorNext();
            }
            lNumData = m_csSendInfoList.GetCount();
            m_csSendInfoList.SetCursorFirst();
            for (long lP = 0; lP < lNumData; lP++)
            {   // データ数分
                csTrnDt = (McTranInfo)m_csSendInfoList.GetCursorData();
                csTrnDt.ResetData();
                m_csSendInfoList.MoveCursorNext();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>現在状態の一時記憶（但し計算途中状態は除く）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Memorize(bMemLvl) </para>
        /// </example>
        /// <param name="bMemLvl">
        /// 記憶レベル：McDefine.ElementMemoryLevel.ELEMENT_MEMORY_ALL／McDefine.ElementMemoryLevel.ELEMENT_MEMORY_SNDPORT_ONLY</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>Memorize()とRemember()は対で使用する</para>
        /// </remarks>
        public virtual void Memorize(McDefine.ElementMemoryLevel bMemLvl)
        {
            if (bMemLvl == McDefine.ElementMemoryLevel.ELEMENT_MEMORY_ALL)
            {
                try
                {
                    this.m_csCalModel.Memorize();
                }
                catch (Exception ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    string DispStatement = Properties.HymcoImplResources.STATEMENT_MEMORIZE + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION
                        + Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message; //ex.StackTrace);
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Memorize", DispStatement);
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Memorize", "Memorize Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " Memorize Exception Catch",
                       Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " Memorize Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);
                }
            }

            int lPortNum = m_csSendPortList.Count;
            for( int lP=0;lP<lPortNum;lP++)
            {
                m_csSendPortList[lP].Memorize();
            } 
        }
        /// <summary><para>method outline:</para>
        /// <para>Memorize()した情報の復元</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Remember(bMemLvl) </para>
        /// </example>
        /// <param name="bMemLvl">
        /// 記憶レベル：McDefine.ElementMemoryLevel.ELEMENT_MEMORY_ALL／McDefine.ElementMemoryLevel.ELEMENT_MEMORY_SNDPORT_ONLY</param>
        /// <returns>無し</returns>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>Memorize()とRemember()は対で使用する</para>
        /// </remarks>
        public virtual void Remember(McDefine.ElementMemoryLevel bMemLvl)
        {
            if (bMemLvl == McDefine.ElementMemoryLevel.ELEMENT_MEMORY_ALL)
            {
                try
                {
                    this.m_csCalModel.Remember();
                }
                catch (Exception ex)
                {
                    // ログ出力
                   // ver1.5 エラートレース日本語対応
                    string DispStatement = Properties.HymcoImplResources.STATEMENT_REMEMBER + Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION
                        + Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message; //ex.StackTrace);
                    HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Remember", DispStatement );
                   // HySLog.LogOut(HySLog.ONLINE, "McCmnElement::Remember", "Remember Exception Catch" + " in Class(" + m_csCalModel.ToString() + ") :: " + ex.Message); //ex.StackTrace);
                   // ver1.5 エラートレース日本語対応
                    McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " Remember Exception Catch",
                       Properties.HymcoImplResources.STATEMENT_IN_CLASS + "(" + m_csCalModel.ToString() + ") :: " + ex.Message);
                   // McLog.DebugOut(m_csCalModel.GetSimulationTime(), GetID(), McLog.SYSTEM, " Remember Exception Catch", " in (" + m_csCalModel.ToString() + ") :: " + ex.Message);
                }
            }
            int lPortNum = m_csSendPortList.Count;
            for (int lP = 0; lP < lPortNum; lP++)
            {
                m_csSendPortList[lP].Remember();
            }
        }

        // Ver1.1 でメソッド追加
        /// <summary>自インスタンスを保持しているプロジェクトのＩＤ値 </summary>
        protected string m_sOwnerProjectID = "";

        /// <summary><para>method outline:</para>
        /// <para>自インスタンスを保持しているプロジェクトのＩＤ値を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetOwnerProjectID(sOwnerProjectID) </para>
        /// </example>
        /// <param name="sOwnerProjectID">自インスタンスを保持しているプロジェクトのＩＤ値</param>
        /// <returns>無し</returns>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetOwnerProjectID(string sOwnerProjectID)
        {
            m_sOwnerProjectID = sOwnerProjectID;
        }
        /// <summary><para>method outline:</para>
        /// <para>自インスタンスを保持しているプロジェクトのＩＤ値を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sID = GetOwnerProjectID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>自インスタンスを保持しているプロジェクトのＩＤ値</returns>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual string GetOwnerProjectID()
        {
            return m_sOwnerProjectID;
        }


        // Ver1.3
        /// <summary><para>method outline:</para>
        /// <para>収束計算に必要な初期出力を強制的に行わせる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = InitialDataFusionForConvergentCalculation(dErrorVal)</para>
        /// </example>
        /// <param name="dErrorVal">出力の目安を示す値</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long InitialDataFusionForConvergentCalculation(double dErrorVal)
        {
            // 派生クラスにて必要に応じオーバーライドして実装のこと
            return m_csCalModel.InitialDataFusionForConvergentCalculation(dErrorVal);
        }
    }
}
