﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;

namespace CommonMP.HYMCO.CoreImpl.Data.LibraryCtl
{
    /// <summary><para>class outline:</para>
    /// <para>ライブラリインストール情報</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>インストール情報やモデル情報とDllとの対応情報等を格納</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.1.0][2010/12/07][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class McLibraryInstallInfoData : HySDataRoot, HySVersionManagement
    {
        /// <summary>モデル情報とDLL名称対応情報 </summary>
        [NonSerialized]
        protected HySDataHashTable m_csCalModelInfoLinkedDll = new HySDataHashTable();
        /// <summary>DLLに含まれるモデル情報(実行環境下にあるDLLの特定にも使用) </summary>
        [NonSerialized]
        protected HySDataHashTable m_csDllIncludeCalModelInfo = new HySDataHashTable();
        ///// <summary>実行環境下にあるDLL名称情報 </summary>
        //protected List<HySString> m_csDllNameInfoList = new List<HySString>();
        /// <summary>展開対象情報 </summary>
        protected HySDataHashTable m_csImportFileInfo = new HySDataHashTable();
        /// <summary>削除対象DLL名称情報 </summary>
        protected List<HySString> m_csTargetDllNameList = new List<HySString>();
        /// <summary>CommonMP標準DLL名称情報 </summary>
        protected HySDataHashTable m_csCMPStdDllNameInfo = new HySDataHashTable();
        /// <summary>モデルのDLLと固有プロパティ画面DLLとの対応情報(設定時どちらのDLLか区別がつかない為) </summary>
        protected HySDataHashTable m_csModelDllLinkedPrtyDll = new HySDataHashTable();
        /// <summary>インストール情報チェックフラグ </summary>
        protected bool m_bDataCheckFlg = true;

        // CommonMP標準ライブラリ定義
        /// <summary>HymcoBase.dll</summary>
        static private readonly String HYMCOBASE = "HymcoBase.dll";
        /// <summary>HymcoDotNetViewer.dll</summary>
        static private readonly String HYMCODOTNETVIEW = "HymcoDotNetViewer.dll";
        /// <summary>HymcoFileDBA.dll</summary>
        static private readonly String HYMCOFILEDBA = "HymcoFileDBA.dll";
        /// <summary>HymcoFileLibraryManager.dll</summary>
        static private readonly String HYMCOFILELIBMNG = "HymcoFileLibraryManager.dll";
        /// <summary>HymcoIF.dll</summary>
        static private readonly String HYMCOIF = "HymcoIF.dll";
        /// <summary>HymcoStandardModel.dll</summary>
        static private readonly String HYMCOSTDMODEL = "HymcoStandardModel.dll";
        /// <summary>HymcoStandardModelProperty.dll</summary>
        static private readonly String HYMCOSTDMODELPROP = "HymcoStandardModelProperty.dll";
        /// <summary>HysAddinDotNetJobSample.dll</summary>
        static private readonly String HYSADDINDOTNETJOBSAMPLE = "HysAddinDotNetJobSample.dll";
        /// <summary>HysAddinHydroHyetoGraph.dll</summary>
        static private readonly String HYSADDINHYDROHYETO = "HysAddinHydroHyetoGraph.dll";
        /// <summary>HysAddinLsor.dll</summary>
        static private readonly String HYSADDINLSOR = "HysAddinLsor.dll";
        /// <summary>HysAddinScreenSample.dll</summary>
        static private readonly String HYSADDINSCREENSAMPLE = "HysAddinScreenSample.dll";
        /// <summary>HysGISDioVista.dll</summary>
        static private readonly String HYSGISDIOVISTA = "HysGISDioVista.dll";
        /// <summary>HyssopBase.dll</summary>
        static private readonly String HYSSOPBASE = "HyssopBase.dll";
        /// <summary>HyssopDotNetIpcTool.dll</summary>
        static private readonly String HYSSOPDOTNETIPCTOOL = "HyssopDotNetIpcTool.dll";
        /// <summary>HyssopDotNetViewer.dll</summary>
        static private readonly String HYSSOPDOTNETVIEWER = "HyssopDotNetViewer.dll";
        /// <summary>HyssopIF.dll</summary>
        static private readonly String HYSSOPIF = "HyssopIF.dll";
        /// <summary>HyssopVSAddinAgent.dll</summary>
        static private readonly String HYSSOPVSADDINAGENT = "HyssopVSAddinAgent.dll";
        /// <summary>JPRiverInformationSystems.dll</summary>
        static private readonly String JPRRIVERINFSYS = "JPRiverInformationSystems.dll";
        /// <summary>McMdlGenCommonScreenLIB.dll</summary>
        static private readonly String MCMDIGENCOMSCREENLIB = "McMdlGenCommonScreenLIB.dll";
        ///// <summary>McModelSample.dll</summary>
        //static private readonly String MCMODELSAMPLE = "McModelSample.dll";
        ///// <summary>McModelSampleProperty.dll</summary>
        //static private readonly String MCMODELSAMPLEPROP = "McModelSampleProperty.dll";
        /// <summary>RiverSurveyDataManagementSystem.dll</summary>
        static private readonly String RIVSURDATAMNGSYS = "RiverSurveyDataManagementSystem.dll";
        /// <summary>RiverWatershedDBSystem.dll</summary>
        static private readonly String RWDBSYSTEM = "RiverWatershedDBSystem.dll";
        /// <summary>Ionic.Zip.dll</summary>
        static private readonly String IONICZIP = "Ionic.Zip.dll";
        /// <summary>McStructureXmlFileEditor.dll</summary>
        static private readonly String MCSTRXMLEDIT = "McStructureXmlFileEditor.dll";
        /// <summary>HysAddinExcelDrive.dll</summary>
        static private readonly String HYSADDINEXCELDRIVE = "HysAddinExcelDrive.dll";
        /// <summary>HysDotNetGISControllerJob.dll</summary>
        static private readonly String HYSDOTNETGISCONJOB = "HysDotNetGISControllerJob.dll";

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>無し</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McLibraryInstallInfoData()
        {
            // CommonMP標準DLL名称情報作成
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYMCOBASE), new HySString(HYMCOBASE));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYMCODOTNETVIEW), new HySString(HYMCODOTNETVIEW));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYMCOFILEDBA), new HySString(HYMCOFILEDBA));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYMCOFILELIBMNG), new HySString(HYMCOFILELIBMNG));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYMCOIF), new HySString(HYMCOIF));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYMCOSTDMODEL), new HySString(HYMCOSTDMODEL));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYMCOSTDMODELPROP), new HySString(HYMCOSTDMODELPROP));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSADDINDOTNETJOBSAMPLE), new HySString(HYSADDINDOTNETJOBSAMPLE));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSADDINHYDROHYETO), new HySString(HYSADDINHYDROHYETO));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSADDINLSOR), new HySString(HYSADDINLSOR));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSADDINSCREENSAMPLE), new HySString(HYSADDINSCREENSAMPLE));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSGISDIOVISTA), new HySString(HYSGISDIOVISTA));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSSOPBASE), new HySString(HYSSOPBASE));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSSOPDOTNETIPCTOOL), new HySString(HYSSOPDOTNETIPCTOOL));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSSOPDOTNETVIEWER), new HySString(HYSSOPDOTNETVIEWER));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSSOPIF), new HySString(HYSSOPIF));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSSOPVSADDINAGENT), new HySString(HYSSOPVSADDINAGENT));
            m_csCMPStdDllNameInfo.AddObject(new HySString(JPRRIVERINFSYS), new HySString(JPRRIVERINFSYS));
            m_csCMPStdDllNameInfo.AddObject(new HySString(MCMDIGENCOMSCREENLIB), new HySString(MCMDIGENCOMSCREENLIB));
            //m_csCMPStdDllNameInfo.AddObject(new HySString(MCMODELSAMPLE), new HySString(MCMODELSAMPLE));
            //m_csCMPStdDllNameInfo.AddObject(new HySString(MCMODELSAMPLEPROP), new HySString(MCMODELSAMPLEPROP));
            m_csCMPStdDllNameInfo.AddObject(new HySString(RIVSURDATAMNGSYS), new HySString(RIVSURDATAMNGSYS));
            m_csCMPStdDllNameInfo.AddObject(new HySString(RWDBSYSTEM), new HySString(RWDBSYSTEM));
            m_csCMPStdDllNameInfo.AddObject(new HySString(IONICZIP), new HySString(IONICZIP));
            m_csCMPStdDllNameInfo.AddObject(new HySString(MCSTRXMLEDIT), new HySString(MCSTRXMLEDIT));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSADDINEXCELDRIVE), new HySString(HYSADDINEXCELDRIVE));
            m_csCMPStdDllNameInfo.AddObject(new HySString(HYSDOTNETGISCONJOB), new HySString(HYSDOTNETGISCONJOB));
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル情報とDLL名称対応情報設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetCalModelInfoLinkedDll(csCalModelInfoLinkedDll)</para>
        /// </example>
        /// <param name="csCalModelInfoLinkedDll">モデル情報とDLL名称対応情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCalModelInfoLinkedDll(HySDataHashTable csCalModelInfoLinkedDll)
        {
            m_csCalModelInfoLinkedDll = csCalModelInfoLinkedDll;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル情報とDLL名称対応情報取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>GetCalModelInfoLinkedDll(csModelKind , ref csDllName)</para>
        /// </example>
        /// <param name="csModelKind">モデル識別子</param>
        /// <param name="csDllName">DLL名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>モデル識別子をKeyとしてDLL名称を取得する</para>
        /// </remarks>
        public void GetCalModelInfoLinkedDll(HySString csModelKind , ref HySString csDllName)
        {
            if ((object)csModelKind != null)
            {
                csDllName = (HySString)m_csCalModelInfoLinkedDll[csModelKind];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>DLLに含まれるモデル情報設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetDllIncludeCalModelInfo(csDllIncludeCalModelInfo)</para>
        /// </example>
        /// <param name="csDllIncludeCalModelInfo">DLLに含まれるモデル情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDllIncludeCalModelInfo(HySDataHashTable csDllIncludeCalModelInfo)
        {
            m_csDllIncludeCalModelInfo = csDllIncludeCalModelInfo;
        }

        /// <summary><para>method outline:</para>
        /// <para>DLLに含まれるモデル情報取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>GetDllIncludeCalModelInfo(csDllName, ref csModelInfoList)</para>
        /// </example>
        /// <param name="csDllName">DLL名称</param>
        /// <param name="csModelInfoList">モデル情報リスト</param>
        /// <returns>同一DLLに含まれるモデル情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>DLL名称をKeyとしてモデル情報(リスト)を取得する</para>
        /// </remarks>
        public void GetDllIncludeCalModelInfo(HySString csDllName, ref HySDataLinkedList csModelInfoList)
        {
            if ((object)csDllName != null)
            {
                csModelInfoList = (HySDataLinkedList)m_csDllIncludeCalModelInfo[csDllName];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>展開対象情報設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetImportFileInfo(csDllName, csImportFileList)</para>
        /// </example>
        /// <param name="csDllName">DLL名称</param>
        /// <param name="csImportFileList">展開対象ファイル情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetImportFileInfo(HySString csDllName, HySDataLinkedList csImportFileList)
        {
            if ((object)csDllName != null)
            {
                m_csImportFileInfo.AddObject(csDllName, csImportFileList);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>展開対象情報取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>GetImportFileInfo(csDllName, ref csImportFileList)</para>
        /// </example>
        /// <param name="csDllName">DLL名称</param>
        /// <param name="csImportFileList">展開対象ファイル情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void GetImportFileInfo(HySString csDllName, ref HySDataLinkedList csImportFileList)
        {
            if ((object)csDllName != null)
            {
                csImportFileList = (HySDataLinkedList)m_csImportFileInfo[csDllName];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>展開対象情報初期化</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>ClearImportFileInfo(csDllName)</para>
        /// </example>
        /// <param name="csDllName">DLL名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ClearImportFileInfo(HySString csDllName)
        {
            if ((object)csDllName != null)
            {
                m_csImportFileInfo.DeleteData(csDllName);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>削除対象DLL名称設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetTargetDllName(csDllName)</para>
        /// </example>
        /// <param name="csDllName">DLL名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetTargetDllName(HySString csDllName)
        {
            m_csTargetDllNameList.Add(csDllName);
        }

        /// <summary><para>method outline:</para>
        /// <para>削除対象DLL名称取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>GetTargetDllName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>削除対象DLL名称情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<HySString> GetTargetDllName()
        {
            return m_csTargetDllNameList;
        }

        /// <summary><para>method outline:</para>
        /// <para>削除対象DLL名称初期化   </para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>ClearTargetDllName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ClearTargetDllName()
        {
            m_csTargetDllNameList.Clear();
        }

        /// <summary><para>method outline:</para>
        /// <para>CommonMP標準DLL名称との重複チェック</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>CheckCMPStdDll()</para>
        /// </example>
        /// <param name="csDllName">DLL名称</param>
        /// <returns>bool true: CommonMP標準DLL名称と重複 false: 重複無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool CheckCMPStdDll(HySString csDllName)
        {
            bool bRtn = false;
            HySDataRoot csData = m_csCMPStdDllNameInfo[csDllName];
            if (csData != null)
            {
                bRtn = true;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルのDLLと固有プロパティ画面DLLとの対応情報設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetModelDllLinkedPrtyDll(csDllName)</para>
        /// </example>
        /// <param name="csDllName">DLL名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetModelDllLinkedPrtyDll(List<HySString> csDllName)
        {
            // アーカイブ内にモデルと画面のdllがある場合のみ対象
            if (csDllName.Count == 2)
            {
                // 設定時どちらのDLLか区別がつかない為、各dllをキーとして情報設定
                m_csModelDllLinkedPrtyDll.AddObject(csDllName[0], csDllName[csDllName.Count - 1]);
                m_csModelDllLinkedPrtyDll.AddObject(csDllName[csDllName.Count - 1], csDllName[0]);
            }
            // (制約)2つ以上のdllが含まれるケースは対象外。この場合、展開対象情報リストがクリアされない。
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルのDLLと固有プロパティ画面DLLとの対応情報取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>GetModelDllLinkedPrtyDll(csKeyDllName , ref csValDllName)</para>
        /// </example>
        /// <param name="csKeyDllName">DLL名称</param>
        /// <param name="csValDllName">対となるDLL名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>DLL名称を設定し対となるDLL名称を返す。(DLL名称は、モデルor固有プロパティ画面DLLどちらか。区別はつかない。)</para>
        /// </remarks>
        public void GetModelDllLinkedPrtyDll(HySString csKeyDllName, ref HySString csValDllName)
        {
            if ((object)csKeyDllName != null)
            {
                csValDllName = (HySString)m_csModelDllLinkedPrtyDll[csKeyDllName];
            }
            // 設定時にハッシュ値として設定したもう一方の情報(プロパティ画面DLLキー情報)は不要なので削除する。
            if ((object)csValDllName != null)
            {
                m_csModelDllLinkedPrtyDll.DeleteData(csValDllName);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルのDLLと固有プロパティ画面DLLとの対応情報初期化</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>ClearModelDllLinkedPrtyDll(csDllName)</para>
        /// </example>
        /// <param name="csDllName">対応情報のキーとなるDLL名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ClearModelDllLinkedPrtyDll(HySString csDllName)
        {
            if ((object)csDllName != null)
            {
                m_csModelDllLinkedPrtyDll.DeleteData(csDllName);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>インストール情報チェックフラグ設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetDataCheckFlg(bFlg)</para>
        /// </example>
        /// <param name="bFlg">データ状態フラグ（true: 正常データ false: 異常データ）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDataCheckFlg(bool bFlg)
        {
            m_bDataCheckFlg = bFlg;
        }

        /// <summary><para>method outline:</para>
        /// <para>インストール情報チェックフラグ取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>bool bRtn = GetDataCheckFlg()</para>
        /// </example>
        /// <returns>bool true: 正常データ false: 異常データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetDataCheckFlg()
        {
            return m_bDataCheckFlg;
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.10";
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                if (GetDecodedVersion() == null)
                {
                    // 旧バージョンからの移行処理を入れる
                }
                else
                {
                    //旧バージョンから順にバージョンを上げて行く
                    switch (GetDecodedVersion())
                    {
                        case "1.00":
                            {
                                // 旧バージョンからの移行処理を入れる
                                break;
                            }
                        //case "2.00": << 以後バージョンアップに伴って処理追加
                        //{
                        //    break;
                        //}
                    }
                }
                SetNowVersion();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
    }
}
