// <summary>ソースコード：マージモデルプロパティ情報クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Model;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.CoreImpl.Model;
using CommonMP.HYMCO.CoreImpl.Data;

namespace CommonMP.HYMCO.CoreImpl.Data
{
    /// <summary><para>class outline:</para>
    /// <para>マージモデルプロパティ情報データクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// <para>remarks</para>
    /// <para>演マージモデルのプロパティ情報を保持するデータクラス</para>
    /// </remarks>
    [Serializable]
    public class McDataMergePropertyInfo : McCellModelPropertyInfo, HySVersionManagement
    {
        /// <summary>出力セルの内部変数 </summary>
        protected HySDataCharacteristicInCell m_csOutCellChara = null;

        //// <summary>入力接続数 </summary>
        // 親クラスの　m_lInputNum　を使用

        /// <summary>入力接続側の接続要素ID </summary>
        protected HySID[] m_csInElementID = null;
        /// <summary>入力接続側の入力パターンID : Xml出力対象外</summary>
        protected HySID[] m_csInPatternID = null;
        /// <summary>入力接続側の入力セル内部の変数 : Xml出力対象外</summary>
        protected HySDataCharacteristicInCell[] m_csInCellChara = null;

        /// <summary>入力接続側の内挿処理区分 </summary>
        protected HySDefine.InterpolateType[] m_eInterpolateType = null;

        //接続情報
        /// <summary>接続情報</summary>
        protected McMergeConnection m_csConnectionData = null;

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McPropertyInfo csModelPropertyInfoRoot = new McDataMergePropertyInfo(csLibraryID, csModelKind)</para>
        /// </example>
        /// <param name="csLibraryID">ライブラリー識別子</param>
        /// <param name="csModelKind">モデル種別識別子</param>
        /// <returns>McModelPropertyInfo 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McDataMergePropertyInfo(HySID csLibraryID, HySObjectKind csModelKind)
            : base(csLibraryID, csModelKind)
        {
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>自分と同じクラスのインスタンスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McPropertyInfoRoot csModelPropertyInfoRoot = Clone() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McPropertyInfoRoot 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McPropertyInfoRoot Clone()
        {
            McDataMergePropertyInfo csRtn = new McDataMergePropertyInfo(m_csLibraryID, m_csModelKind);
            csRtn.CopyInf(this);
            return csRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>引数で与えられた情報を自分にコピーを行う</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CopyInf(csOrgObj) </para>
        /// </example>
        /// <param name="csOrgObj">コピー元情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override void CopyInf(McPropertyInfoRoot csOrgObj)
        {
            base.CopyInf(csOrgObj);
            McDataMergePropertyInfo csPrpOrgObj = csOrgObj as McDataMergePropertyInfo;
            if (csPrpOrgObj != null)
            {
                if (csPrpOrgObj.m_csOutCellChara != null) { m_csOutCellChara = csPrpOrgObj.m_csOutCellChara.Clone(); }
                if (csPrpOrgObj.m_lInputNum > 0)
                {
                    m_csInElementID = new HySID[csPrpOrgObj.m_lInputNum];
                    m_csInPatternID = new HySID[csPrpOrgObj.m_lInputNum];
                    m_csInCellChara = new HySDataCharacteristicInCell[csPrpOrgObj.m_lInputNum];
                    m_eInterpolateType = new HySDefine.InterpolateType[csPrpOrgObj.m_lInputNum];
                }
                for (long lLp = 0; lLp < csPrpOrgObj.m_lInputNum; lLp++)
                {
                    if (csPrpOrgObj.m_csInElementID[lLp] != null) { m_csInElementID[lLp] = csPrpOrgObj.m_csInElementID[lLp].Clone(); }
                    if (csPrpOrgObj.m_csInPatternID[lLp] != null) { m_csInPatternID[lLp] = csPrpOrgObj.m_csInPatternID[lLp].Clone(); }
                    if (csPrpOrgObj.m_csInCellChara[lLp] != null) { m_csInCellChara[lLp] = csPrpOrgObj.m_csInCellChara[lLp].Clone(); }
                    m_eInterpolateType[lLp] = csPrpOrgObj.m_eInterpolateType[lLp];
                }
                if (csPrpOrgObj.m_csConnectionData != null) { m_csConnectionData = new McMergeConnection(csPrpOrgObj.m_csConnectionData); }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>入力接続の情報リスト設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetInputConnectionInfoList( csList ) </para>
        /// </example>
        /// <param name="csList">入力接続の情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>接続変更の際には、取得値も変化する
        /// リスト内は　McModelInfoが入っており、プロパティは、.GetPropertyInfo();で取り出す必要があるが、モデル未設定等によりnull の場合もある
        /// </para>
        /// </remarks>
        public override void SetInputConnectionInfoList(HySDataLinkedList csList)
        {
            base.SetInputConnectionInfoList(csList);
            if (m_lInputNum > 0)
            {
                if (m_eInterpolateType == null)
                {
                    m_eInterpolateType = new HySDefine.InterpolateType[m_lInputNum];
                    for (long lLp = 0; lLp < m_lInputNum; lLp++)
                    {
                        m_eInterpolateType[lLp] = HySDefine.InterpolateType.LINEAR; // <-- 初期値
                    }
                }
                else if (m_eInterpolateType.Length != m_lInputNum)
                {
                    m_eInterpolateType = new HySDefine.InterpolateType[m_lInputNum];
                    for (long lLp = 0; lLp < m_lInputNum; lLp++)
                    {
                        m_eInterpolateType[lLp] = HySDefine.InterpolateType.LINEAR; // <-- 初期値
                    }
                }
                m_csInElementID = new HySID[m_lInputNum];
                m_csInPatternID = new HySID[m_lInputNum];
                m_csInCellChara = new HySDataCharacteristicInCell[m_lInputNum];

                McModelInfo csMdlInfo = null;
                McCellArrayTranInfoPropertyInfo csCnctPrpty = null;
                McTranInfoPattern csPtn = null;

                m_csInputList.SetCursorFirst();
                for (long lLp = 0; lLp < m_lInputNum; lLp++)
                {
                    csMdlInfo = m_csInputList.GetCursorData() as McModelInfo;
                    csCnctPrpty = csMdlInfo.GetPropertyInfo() as McCellArrayTranInfoPropertyInfo;
                    if (csCnctPrpty != null)
                    {
                        if (csCnctPrpty.GetUpperModelInfo() != null)// 接続線プロパティ情報に上流要素モデル情報が設定済か確認
                        {
                            m_csInElementID[lLp] = csCnctPrpty.GetUpperModelInfo().GetElementID();

                            csPtn = csCnctPrpty.GetSendTranInfoPattern() as McTranInfoPattern;
                            if (csPtn != null)
                            {
                                m_csInPatternID[lLp] = csPtn.GetPatternID();
                                m_csInCellChara[lLp] = csPtn.GetCellDataCharacteristic();
                                if (m_csInCellChara[lLp] != null)
                                {
                                    csCnctPrpty.SetCellDataGetter(new HySCellDataGetter(m_csInCellChara[lLp].GetDataNumber()));
                                }
                            }
                        }
                    }
                    m_csInputList.MoveCursorNext();
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>出力セル内データの設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetOutputCellCharacteristic( csOutCellChara ) </para>
        /// </example>
        /// <param name="csOutCellChara">出力セル内データ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>出力として　本モデルが出すセル内の情報の設定を行う </para>
        /// </remarks>
        public virtual void SetOutputCellCharacteristic(HySDataCharacteristicInCell csOutCellChara)
        {
            m_csOutCellChara = csOutCellChara;

            // 各パターンに全て設定する
            McTranInfoPattern csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_SINGLE_CELL_SERIAL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_D1_CELL_SERIAL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_D2_CELL_SERIAL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_D3_CELL_SERIAL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_GEOD1_CELL_SERIAL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_GEOD2_CELL_SERIAL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_GEOD3_CELL_SERIAL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_D1_CELL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_D2_CELL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_D3_CELL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_GEOD1_CELL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_GEOD2_CELL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }
            csTrnInfoPtn = this.GetSendPattern(McDataMergePropertyInfo.OUT_PTN_GEOD3_CELL) as McTranInfoPattern;
            if (csTrnInfoPtn != null) { csTrnInfoPtn.SetCellDataCharacteristic(m_csOutCellChara); }

            // 接続情報を生成する
            long lValNum = csOutCellChara.GetDataNumber();
            m_csConnectionData = new McMergeConnection(lValNum);

            // 出力側の要素配線に対して　パターン設定を解除する
            // 既に設定がされていても、　出力セルの内容を変更したならば、もう一度
            // オペレーターに配線させ直す必要がある。
            if (m_csOutputList != null)
            {
                McModelInfo csMdlInfo = null;
                McCellArrayTranInfoPropertyInfo csCnctPrpty = null;
                m_csOutputList.SetCursorFirst();
                for (long lLp = 0; lLp < m_lOutputNum; lLp++)
                {
                    csMdlInfo = m_csOutputList.GetCursorData() as McModelInfo;
                    csCnctPrpty = csMdlInfo.GetPropertyInfo() as McCellArrayTranInfoPropertyInfo;
                    if (csCnctPrpty != null)
                    {
                        //  csCnctPrpty.SetSendTranInfoPattern(null); // パターンの解除　＜＝＝此処は十分に検討
                        // この操作を行うと、下流接続への再設定「接続」時に　落ちる
                        //  なぜか？
                    }
                    m_csOutputList.MoveCursorNext();
                }
            }

        }

        /// <summary><para>method outline:</para>
        /// <para>出力セル内データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataCharacteristicInCell csOutputCellChara = GetOutputCellCharacteristic( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>出力セル内データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し </para>
        /// </remarks>
        public virtual HySDataCharacteristicInCell GetOutputCellCharacteristic()
        { return m_csOutCellChara; }

        /// <summary><para>method outline:</para>
        /// <para>入力側セル内情報設定値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataCharacteristicInCell[] csInputCellChara = GetInputCellCharacteristic( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>入力側セル内情報設定</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し </para>
        /// </remarks>
        public virtual HySDataCharacteristicInCell[] GetInputCellCharacteristic()
        {   return m_csInCellChara; }
        /// <summary><para>method outline:</para>
        /// <para>入力接続側の接続要素ＩＤ値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID[] csInputElmID = GetInputElementID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>入力接続側の接続要素ＩＤ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し </para>
        /// </remarks>
        public virtual  HySID[] GetInputElementID()
        {   return m_csInElementID; } 
        /// <summary><para>method outline:</para>
        /// <para>入力側内挿設定値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDefine.InterpolateType[] csItpltType = GetInputInterpolateType( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>入力側内挿設定</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し </para>
        /// </remarks>
        public virtual HySDefine.InterpolateType[] GetInputInterpolateType()
        { return m_eInterpolateType; }

        /// <summary><para>method outline:</para>
        /// <para>マージモデル内接続設定値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McMergeConnection csCnnt = GetMergeConnection( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>マージモデル内接続設定</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し </para>
        /// </remarks>
        public virtual McMergeConnection GetMergeConnection()
        { return m_csConnectionData; }


        /// <summary><para>method outline:</para>
        /// <para>モデル情報をXMLへ書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySXmlNode csXmlNode = XmlOUT(csXmlWriter) </para>
        /// </example>
        /// <param name="csXmlWriter">XML書き出しクラス</param>
        /// <returns>HySXmlNode  XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySXmlNode XmlOUT(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtn = base.XmlOUT(ref csXmlWriter);
            HySXmlNode csMargeInfo = HySXmlNode.CreateElement(McDataMergePropertyInfo.xml_MARGE_INFOR, csXmlWriter); 
            HySXmlNode csXml = null;

            // 出力セルの内部
            if (m_csOutCellChara != null)
            {
                csXml = m_csOutCellChara.XmlOUT(ref csXmlWriter);
                csMargeInfo.AppendChild(csXml);
            }
            // 接続関連情報
            for (long lLp = 0; lLp < m_lInputNum; lLp++)
            {
                csXml = HySXmlNode.CreateElement(McDataMergePropertyInfo.xml_MARGE_INFOR, csXmlWriter);

                if (m_csInElementID[lLp] != null)
                {// 2010.9.22 入力側要素ID取得済の場合
                    csXml.SetAttribute(McDataMergePropertyInfo.xml_INPUT_ELEMENTID, m_csInElementID[lLp].ToString());
                }
                else
                {// 2010.9.22 入力側要素IDがnullの場合(上流要素モデル未設定・接続線モデル未設定時)
                    csXml.SetAttribute(McDataMergePropertyInfo.xml_INPUT_ELEMENTID,"");
                }

                if (m_eInterpolateType[lLp] == HySDefine.InterpolateType.LINEAR)
                {
                    csXml.SetAttribute(McDefine.xml_INTERPOLATETYPE, HySDefine.sLINEAR_INTERPOLATE);
                }
                else if (m_eInterpolateType[lLp] == HySDefine.InterpolateType.GET_NEXT_DATA)
                {
                    csXml.SetAttribute(McDefine.xml_INTERPOLATETYPE, HySDefine.sGET_NEXT_DATA_INTERPOLATE);
                }
                //else if (m_eInterpolateType[lLp] == HySDefine.InterpolateType.POLYNOMIAL)
                //{
                //    csLclNode.SetAttribute(McDefine.xml_INTERPOLATETYPE, HySDefine.sPOLYNOMIAL_INTERPOLATE);
                //}
                else
                {
                    csXml.SetAttribute(McDefine.xml_INTERPOLATETYPE, HySDefine.sNO_INTERPOLATE);
                }

                csMargeInfo.AppendChild(csXml);
            }

            
            // 接続情報
            if (m_csConnectionData != null)
            {
                csXml = m_csConnectionData.XmlOUT(ref csXmlWriter);
                csMargeInfo.AppendChild(csXml);
            }
            csRtn.AppendChild(csMargeInfo);   // AppendChildは最後に行う
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>XMLからモデル情報を読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = XmlIN(csPrptyXml) </para>
        /// </example>
        /// <param name="csPrptyXml">入力XML</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool XmlIN(HySXmlNode csPrptyXml)
        {
            bool bRtn = base.XmlIN(csPrptyXml);

            HySXmlNode csXml = null;
            HySXmlNodeList csXmlVariableList = csPrptyXml.GetChildNodeListByTagName(McDataMergePropertyInfo.xml_MARGE_INFOR);
            HySXmlNode csMergeXml = csXmlVariableList.GetNode(0);
            if (csMergeXml != null)
            {
                // 出力セルの内部変数
                csXmlVariableList = csMergeXml.GetChildNodeListByTagName(HySDefine.xml_DATA_CHARACTERISTIC_IN_CELL);
                csXml = csXmlVariableList.GetNode(0);
                if (csXml != null)
                {
                    HySDataCharacteristicInCell csOutCellChara = new HySDataCharacteristicInCell(1);
                    csOutCellChara.XmlIN(csXml);
                    this.SetOutputCellCharacteristic(csOutCellChara);
                    //if (m_csOutCellChara == null) { m_csOutCellChara = new HySDataCharacteristicInCell(1); }
                    //m_csOutCellChara.XmlIN(csXml);
                }
                else
                {
                    bRtn = false;
                }
                // 接続関連の情報
                string sVal = "";
                csXmlVariableList = csMergeXml.GetChildNodeListByTagName(McDataMergePropertyInfo.xml_MARGE_INFOR);
                long lValCnt = csXmlVariableList.GetCount();
                m_csInElementID = new HySID[lValCnt];
                m_csInPatternID = new HySID[lValCnt];
                m_csInCellChara = new HySDataCharacteristicInCell[lValCnt];
                m_eInterpolateType = new HySDefine.InterpolateType[lValCnt];
                for (long lLp = 0; lLp < lValCnt; lLp++)
                {   // ノード数繰り返し
                    csXml = csXmlVariableList.GetNode(lLp);
                    sVal = csXml.GetAttribute(McDataMergePropertyInfo.xml_INPUT_ELEMENTID);
                    m_csInElementID[lLp] = new HySID(sVal);
                    sVal = csXml.GetAttribute(McDefine.xml_INTERPOLATETYPE);
                    if (sVal == HySDefine.sLINEAR_INTERPOLATE)
                    {   // 線形補間ならば
                        m_eInterpolateType[lLp] = HySDefine.InterpolateType.LINEAR;
                    }
                    else if (sVal == HySDefine.sGET_NEXT_DATA_INTERPOLATE)
                    {   // 直後値取得ならば
                        m_eInterpolateType[lLp] = HySDefine.InterpolateType.GET_NEXT_DATA;
                    }
                    //else if (sVal == HySDefine.sPOLYNOMIAL_INTERPOLATE)
                    //{
                    //    m_eInterpolateType[lLp] = HySDefine.InterpolateType.POLYNOMIAL;
                    //}
                    else
                    {   // 何もしない（直近値取得）
                        m_eInterpolateType[lLp] = HySDefine.InterpolateType.NO_INTERPOLATE;
                    }
                }

                // 接続情報
                csXmlVariableList = csMergeXml.GetChildNodeListByTagName(McMergeConnection.xml_MARGE_CONNECTION);
                csXml = csXmlVariableList.GetNode(0);
                if (csXml != null)
                {
                    m_csConnectionData = new McMergeConnection(1);
                    m_csConnectionData.XmlIN(csXml);
                }
            }
            else
            {
                bRtn = false;
            }
            return bRtn;
        }

        /// <summary> ポイント時系列出力</summary>
        [NonSerialized]
        public static String OUT_NAME_SINGLE_CELL_SERIAL = "ポイント時系列出力";
        /// <summary> ポイント時系列出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_SINGLE_CELL_SERIAL = new HySID("OUT_PTN_SINGLE_CELL_SERIAL");
        /// <summary> １次元時系列出力</summary>
        [NonSerialized]
        public static String OUT_NAME_D1_CELL_SERIAL = "１次元時系列出力";
        /// <summary> １次元時系列出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_D1_CELL_SERIAL = new HySID("OUT_PTN_D1_CELL_SERIAL");
        /// <summary> ２次元時系列出力</summary>
        [NonSerialized]
        public static String OUT_NAME_D2_CELL_SERIAL = "２次元時系列出力";
        /// <summary> ２次元時系列出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_D2_CELL_SERIAL = new HySID("OUT_PTN_D2_CELL_SERIAL");
        /// <summary> ３次元時系列出力</summary>
        [NonSerialized]
        public static String OUT_NAME_D3_CELL_SERIAL = "３次元時系列出力";
        /// <summary> ３次元時系列出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_D3_CELL_SERIAL = new HySID("OUT_PTN_D3_CELL_SERIAL");
        /// <summary> ＧＩＳ：１次元時系列出力</summary>
        [NonSerialized]
        public static String OUT_NAME_GEOD1_CELL_SERIAL = "ＧＩＳ：１次元時系列出力";
        /// <summary> ＧＩＳ：１次元時系列出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_GEOD1_CELL_SERIAL = new HySID("OUT_PTN_GEOD1_CELL_SERIAL");
        /// <summary> ＧＩＳ：２次元時系列出力</summary>
        [NonSerialized]
        public static String OUT_NAME_GEOD2_CELL_SERIAL = "ＧＩＳ：２次元時系列出力";
        /// <summary> ＧＩＳ：２次元時系列出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_GEOD2_CELL_SERIAL = new HySID("OUT_PTN_GEOD2_CELL_SERIAL");
        /// <summary> ＧＩＳ：３次元時系列出力</summary>
        [NonSerialized]
        public static String OUT_NAME_GEOD3_CELL_SERIAL = "ＧＩＳ：３次元時系列出力";
        /// <summary> ＧＩＳ：３次元時系列出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_GEOD3_CELL_SERIAL = new HySID("OUT_PTN_GEOD3_CELL_SERIAL");

        /// <summary> １次元定常出力</summary>
        [NonSerialized]
        public static String OUT_NAME_D1_CELL = "１次元定常出力";
        /// <summary> １次元定常出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_D1_CELL = new HySID("OUT_PTN_D1_CELL");
        /// <summary> ２次元定常出力</summary>
        [NonSerialized]
        public static String OUT_NAME_D2_CELL = "２次元定常出力";
        /// <summary> ２次元定常出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_D2_CELL = new HySID("OUT_PTN_D2_CELL");
        /// <summary> ３次元定常出力</summary>
        [NonSerialized]
        public static String OUT_NAME_D3_CELL = "３次元定常出力";
        /// <summary> ３次元定常出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_D3_CELL = new HySID("OUT_PTN_D3_CELL");
        /// <summary> ＧＩＳ：１次元定常出力</summary>
        [NonSerialized]
        public static String OUT_NAME_GEOD1_CELL = "ＧＩＳ：１次元定常出力";
        /// <summary> ＧＩＳ：１次元定常出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_GEOD1_CELL = new HySID("OUT_PTN_GEOD1_CELL");
        /// <summary> ＧＩＳ：２次元定常出力</summary>
        [NonSerialized]
        public static String OUT_NAME_GEOD2_CELL = "ＧＩＳ：２次元定常出力";
        /// <summary> ＧＩＳ：２次元定常出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_GEOD2_CELL = new HySID("OUT_PTN_GEOD2_CELL");
        /// <summary> ＧＩＳ：３次元定常出力</summary>
        [NonSerialized]
        public static String OUT_NAME_GEOD3_CELL = "ＧＩＳ：３次元定常出力";
        /// <summary> ＧＩＳ：３次元定常出力</summary>
        [NonSerialized]
        public static HySID OUT_PTN_GEOD3_CELL = new HySID("OUT_PTN_GEOD3_CELL");

        /// <summary>マージ情報タグ</summary>
        [NonSerialized]
        static public readonly string xml_MARGE_INFOR = "MargeInfo";
        /// <summary>入力情報タグ</summary>
        [NonSerialized]
        static public readonly string xml_INPUT_INFOR = "ReceivePortInfo";
        /// <summary>入力情報タグ</summary>
        [NonSerialized]
        static public readonly string xml_INPUT_ELEMENTID = "ReceiveElementID";

        /// <summary><para>class outline:</para>
        /// <para>入力線との接続情報管理データクラス</para>
        /// </summary>
        /// <remarks><para>history:</para>
        /// <para>[CommonMP][ver 1.0.0][2009/12/01][新規作成]</para>
        /// <para>remarks</para>
        /// <para>マージモデル処理時の接続情報を保持するデータクラス</para>
        /// </remarks>
        [Serializable]
        public class McMergeConnection : HySCellDataGetter, HySVersionManagement
        {
            /// <summary> 入力接続線を示すインデックス値  負の値 は　未接続を示す</summary>
            protected long[] m_InCnctIdx=null;
            // 親クラスの　long[] m_lIdxは、　入力したパターン内のセル内変数のインデックス値を示す。
            /// <summary>入力エレメントのＩＤ </summary>
            protected HySID[] m_csInElementID = null;

            /// <summary><para>method outline:</para>
            /// <para>デフォルトコンストラクタ</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> 使用禁止 </para>
            /// </example>
            /// <param name="">無し</param>
            /// <returns> HySDataCharacteristicInCell 生成されたクラスのインスタンス</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>使用禁止</para>
            /// </remarks>
            protected McMergeConnection() : base(1)
            {
                //バージョン情報設定
                SetNowVersion();
            } 
            /// <summary><para>method outline:</para>
            /// <para>コンストラクタ</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> HySCellDataGetter csData = new HySCellDataGetter(lDim); </para>
            /// </example>
            /// <param name="lDim">データ配列数</param>
            /// <returns> HySCellDataGetter 生成されたクラスのインスタンス</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>無し</para>
            /// </remarks>
            public McMergeConnection(long lDim)
                : base(lDim)
            {
                m_InCnctIdx = new long[lDim];
                m_csInElementID = new HySID[lDim];
                for (long lLp = 0; lLp < m_lIdxNum; lLp++)
                {
                    m_lIdx[lLp] = -1-lLp; 
                    m_InCnctIdx[lLp] = -1;
                    m_csInElementID[lLp] = new HySID("");
                }

                //バージョン情報設定
                SetNowVersion();
            }
 
            /// <summary><para>method outline:</para>
            /// <para>コピーコンストラクタ</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> McMergeConnection csData = new McMergeConnection(csOrgData); </para>
            /// </example>
            /// <param name="csOrgData">オリジナルインスタンス</param>
            /// <returns> McMergeConnection 生成されたクラスのインスタンス</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>無し</para>
            /// </remarks>
            public McMergeConnection(McMergeConnection csOrgData) : base(csOrgData)
            {
                m_InCnctIdx = new long[m_lIdxNum];
                m_csInElementID = new HySID[m_lIdxNum];
                for (long lLp = 0; lLp < m_lIdxNum; lLp++)
                {
                    m_InCnctIdx[lLp] = csOrgData.m_InCnctIdx[lLp];
                    m_csInElementID[lLp] = csOrgData.m_csInElementID[lLp].Clone();
                }
                //バージョン情報設定
                SetNowVersion();
            }

            /// <summary><para>method outline:</para>
            /// <para>セル内の変数を示すインデックス値設定</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> SetIdx(lIdx,lVal); </para>
            /// </example>
            /// <param name="lIdx">データ位置</param>
            /// <param name="lCnctIdxVal">接続位置インデックス値</param>
            /// <param name="csElmID">接続要素ＩＤ</param>
            /// <param name="lCellIdxVal">セル内インデックス値</param>
            /// <returns> なし</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>無し</para>
            /// </remarks>
            public virtual void SetIdx(long lIdx, long lCnctIdxVal, HySID csElmID,long lCellIdxVal)
            {
                if (lIdx >= 0 && lIdx < m_lIdxNum)
                {
                    m_lIdx[lIdx] = lCellIdxVal;
                    m_InCnctIdx[lIdx] = lCnctIdxVal;
                    m_csInElementID[lIdx] = csElmID.Clone();
                }
            }
            /// <summary><para>method outline:</para>
            /// <para>インデックス値取得</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> long[] lIdxVal = GetIdx(); </para>
            /// </example>
            /// <param name="">なし</param>
            /// <returns> long[] インデックス値配列</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>無し</para>
            /// </remarks>
            public virtual long[] GetConnectIdx()
            {
                return m_InCnctIdx;
            }
            /// <summary><para>method outline:</para>
            /// <para>入力エレメントＩＤ値取得</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> HySID[] csElmID = GetConnectElementID(); </para>
            /// </example>
            /// <param name="">なし</param>
            /// <returns> HySID[] 入力エレメントＩＤ配列</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>無し</para>
            /// </remarks>
            public virtual HySID[] GetConnectElementID()
            {
                return m_csInElementID;
            }

            /// <summary><para>method outline:</para>
            /// <para>セルと識別子の対応管理情報をXMLへ書き出す</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> HySXmlNode csXmlNode = XmlOUT(csXmlWriter) </para>
            /// </example>
            /// <param name="csXmlWriter">XML書き出しクラス</param>
            /// <returns>HySXmlNode  XMLノードクラス</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>無し</para>
            /// </remarks>
            public override HySXmlNode XmlOUT(ref HySXmlWriter csXmlWriter)
            {
                HySXmlNode csRtn = HySXmlNode.CreateElement(McMergeConnection.xml_MARGE_CONNECTION, csXmlWriter);

                HySXmlNode csLclNode;
                for (int iLp = 0; iLp < m_lIdxNum; iLp++)
                {
                    csLclNode = HySXmlNode.CreateElement(HySDefine.xml_INDEX, csXmlWriter);

                    csLclNode.SetAttribute(HySDefine.xml_INDEX_NO, iLp.ToString());
                    //csLclNode.SetAttribute(McMergeConnection.xml_INPORT_INDEX, m_InCnctIdx[iLp].ToString()); <-- これを出力してはならない
                    csLclNode.SetAttribute(McDataMergePropertyInfo.xml_INPUT_ELEMENTID, m_csInElementID[iLp].ToString());
                    csLclNode.SetAttribute(HySDefine.xml_INDEX_VALUE, m_lIdx[iLp].ToString());

                    csRtn.AppendChild(csLclNode);   // AppendChildは最後に行う
                }
                /*
                if (m_csUndefefValCell != null)
                {
                    csLclNode = m_csUndefefValCell.XmlOUT(ref csXmlWriter);
                    csRtn.AppendChild(csLclNode);   // AppendChildは最後に行う
                }
                */
                return csRtn;
            }
            /// <summary><para>method outline:</para>
            /// <para>XMLからセルと識別子の対応管理情報を読み出す</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> bool bRtn = XmlIN(csPrptyXml) </para>
            /// </example>
            /// <param name="csPrptyXml">入力XML</param>
            /// <returns>bool  =true:正常、=false:型式不適等</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>HySDefine.xml_CELL_INDEXタグの選択は外部で行っておく</para>
            /// </remarks>
            public override bool XmlIN(HySXmlNode csPrptyXml)
            {
                bool bRtn = true;
                if (csPrptyXml == null)
                { return false; }

                m_lIdx = null;

                HySXmlNode csXml = null;
                HySXmlNodeList csXmlVariableList = csPrptyXml.GetChildNodeListByTagName(HySDefine.xml_INDEX);
                long lValCnt = csXmlVariableList.GetCount();

                m_lIdxNum = lValCnt;
                m_lIdx = new long[m_lIdxNum];
                m_csInElementID = new HySID[m_lIdxNum];
                m_InCnctIdx = new long[m_lIdxNum];
                string sOrderAttr; long lOrder = 0;
                string sValueAttr; long lVal = 0;

                for (long lLoop = 0; lLoop < m_lIdxNum; lLoop++)
                {
                    csXml = csXmlVariableList.GetNode(lLoop);
                    // 順序
                    sOrderAttr = csXml.GetAttribute(HySDefine.xml_INDEX_NO);
                    lOrder = Convert.ToInt32(sOrderAttr);
                    // 接続端子インデックス（これは入力しない）＝＝モデル側で再構築する為＝＝
                    //sValueAttr = csXml.GetAttribute(McMergeConnection.xml_INPORT_INDEX);
                    //lVal = Convert.ToInt32(sValueAttr);
                    //m_InCnctIdx[lOrder] = lVal;
                    // 接続端子インデックス
                    sValueAttr = csXml.GetAttribute(McDataMergePropertyInfo.xml_INPUT_ELEMENTID);
                    m_csInElementID[lOrder] = new HySID(sValueAttr);

                    // セル接続インデックス値
                    sValueAttr = csXml.GetAttribute(HySDefine.xml_INDEX_VALUE);
                    lVal = Convert.ToInt32(sValueAttr);
                    m_lIdx[lOrder] = lVal;
                }

                csXmlVariableList = csPrptyXml.GetChildNodeListByTagName(HySDefine.xml_CELL_DATA);
                lValCnt = csXmlVariableList.GetCount();
                if (lValCnt > 0)
                {
                    m_csUndefefValCell = new HySCellData(1);
                    csXml = csXmlVariableList.GetNode(0);
                    bRtn = m_csUndefefValCell.XmlIN(csXml);
                }

                return bRtn;
            }

            // ================================================
            //   バージョンアップ時の下位互換性を保つ為の処理
            // ================================================
            /// <summary>保存バージョン情報</summary>
            private string m_sDecodedVersion;
            /// <summary><para>method outline:</para>
            /// <para>バージョン情報取得</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> string sDecodedVersion = GetDecodedVersion()</para>
            /// </example>
            /// <param name="">無し</param>
            /// <returns>string 保存バージョン情報</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>メンバー変数に設定されているバージョン情報を返す</para>
            /// </remarks>
            public new string GetDecodedVersion()
            {
                return m_sDecodedVersion;
            }
            /// <summary><para>method outline:</para>
            /// <para>バージョン情報取得</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> string sVersion = GetNowVersion()</para>
            /// </example>
            /// <param name="">無し</param>
            /// <returns>string 現在のバージョン情報</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>無し</para>
            /// </remarks>
            public new string GetNowVersion()
            {
                return "1.00";
            }
            /// <summary><para>method outline:</para>
            /// <para>バージョン情報設定</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> SetNowVersion()</para>
            /// </example>
            /// <param name="">無し</param>
            /// <returns>無し</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>現在のバージョン情報をメンバー変数に設定する</para>
            /// </remarks>
            public new void SetNowVersion()
            {
                m_sDecodedVersion = GetNowVersion();
            }
            /// <summary><para>method outline:</para>
            /// <para>復元後の整合性確保処理</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> PostDeserialize( ) </para>
            /// </example>
            /// <param name="">無し</param>
            /// <returns>無し</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
            /// バージョンによって変数等追加がない場合には、実装の必要がない
            /// </para>
            /// </remarks>
            public override void PostDeserialize()
            {
                if (GetDecodedVersion() != GetNowVersion())
                {
                    //旧バージョンから順にバージョンを上げて行く
                    switch (GetDecodedVersion())
                    {
                        case "1.00":
                            break;
                    }
                    SetNowVersion();
                }
            }
            /*
            /// <summary><para>method outline:</para>
            /// <para>逆シリアル化時追加処理</para>
            /// </summary>
            /// <example><para>usage:</para>
            /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
            /// </example>
            /// <param name="csObj">ダミー</param>
            /// <returns>無し</returns>
            /// <exception cref="">無し</exception>
            /// <remarks><para>remarks:</para>
            /// <para>予約処理</para>
            /// </remarks>
            void IDeserializationCallback.OnDeserialization(Object csObj)
            {
                PostDeserialize();
            }
            */
            /// <summary>マージ接続情報タグ</summary>
            [NonSerialized]
            static public readonly string xml_MARGE_CONNECTION = "MargeConnection";
            /// <summary>インデックス順序タグ</summary>
            [NonSerialized]
            static public readonly string xml_INPORT_INDEX = "InPortIdx";
        }

        // =============================================
        // === バージョンアップ時の互換性を保つ為の処理 ===
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public new string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public new void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public override void PostDeserialize()
        {
            base.PostDeserialize();

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
    }
}
