﻿// <summary>ソースコード：ＨＹＭＣＯモデルプロパティ表示／設定情報クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Data;

using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.StructInfo;

namespace CommonMP.HYMCO.CoreImpl.Data
{
    /// <summary>
    /// <para>class outline:</para>
    /// <para>プロパティ表示／設定情報クラス</para>
    /// </summary>
    /// <remarks>
    /// <para>remarks:</para>
    /// <para>プロパティ画面に表示すべき情報を格納</para>
    /// </remarks>
    /// <remarks>
    /// <para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/12/23][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class McPropertyInfo : McPropertyInfoBase //McPropertyInfoRoot
    {
        /// <summary>情報引き出しキーリスト</summary>
        protected List<HySString> m_cskeyList = new List<HySString>();
        /// <summary>情報名称リスト</summary>
        protected List<HySString> m_csPropatyNameList = new List<HySString>();
        /// <summary>情報型リスト</summary>
        protected List<McDefine.ValKind> m_csValList = new List<McDefine.ValKind>();
        /// <summary>データ情報(bool)</summary>
        protected List<bool> m_csBoolDataList = new List<bool>();
        /// <summary>データ情報(bool)</summary>
        protected List<long> m_csLongDataList = new List<long>();
        /// <summary>配列データ情報(bouble)</summary>
        protected List<double> m_csDoubleDataList = new List<double>();
        /// <summary>配列データ情報(string)</summary>
        protected List<string> m_csStringDataList = new List<string>();
        /// <summary>配列データ情報(McTranInfo)</summary>
        protected List<McTranInfo> m_csMcTranInfoDataList = new List<McTranInfo>();

  
        /// <summary> 自プロパティ情報を保持しているMcStructElement インスタンス</summary>
        [NonSerialized]
        protected McStructElement  m_csStructElm=null;

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>使用禁止</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McPropertyInfo 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private McPropertyInfo()
            : base(null,null)
        {
            this.Init();
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McPropertyInfoRoot csPropertyInfo = new McPropertyInfo(csLibraryID, csModelKind) </para>
        /// </example>
        /// <param name="csLibraryID">ライブラリー識別子</param>
        /// <param name="csModelKind">モデル種別識別子</param>
        /// <returns>McPropertyInfo 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McPropertyInfo(HySID csLibraryID, HySObjectKind csModelKind)
            : base(csLibraryID, csModelKind)
        {
            this.Init();
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>初期設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Init() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void Init()
        {
            this.AddInfoType(TIME_STEP, Properties.HymcoImplResources.MODELCTL_PROPERTY_INFO_TIME_STEP, McDefine.ValKind.DOUBLE);   //"タイムステップ（秒）"
            this.SetInfo(TIME_STEP, 60.0);  // 1 min
        }
        /// <summary><para>method outline:</para>
        /// <para>自分と同じクラスのインスタンスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McPropertyInfoRoot csModelPropertyInfoRoot = Clone() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McPropertyInfoRoot 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McPropertyInfoRoot Clone()
        {
            McPropertyInfo csRtn = new McPropertyInfo(m_csLibraryID, m_csModelKind);
            csRtn.CopyInf(this);
            return csRtn;
        }
        /*
        /// <summary><para>method outline:</para>
        /// <para>自分と同じクラスのインスタンスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McPropertyInfoRoot csNewObj = Clone( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McPropertyInfoRoot Clone();
        */

        /// <summary><para>method outline:</para>
        /// <para>引数で与えられた情報を自分にコピーを行う</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CopyInf(csOrgObj) </para>
        /// </example>
        /// <param name="csOrgObj">コピー元情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override void CopyInf(McPropertyInfoRoot csOrgObj)
        {
            base.CopyInf(csOrgObj);
            McPropertyInfo csPrpOrgObj = csOrgObj as McPropertyInfo;
            if( csPrpOrgObj != null )
            {

                m_cskeyList.Clear();
                m_csPropatyNameList.Clear();
                m_csValList.Clear();
                m_csBoolDataList.Clear();
                m_csLongDataList.Clear();
                m_csDoubleDataList.Clear();
                m_csStringDataList.Clear();
                m_csMcTranInfoDataList.Clear();

                if (csPrpOrgObj.m_cskeyList != null)
                {
                    int iDataNum = csPrpOrgObj.m_cskeyList.Count;
                    for (int iLp = 0; iLp < iDataNum; iLp++)
                    {
                        // デフォルトのデータの入り方は全エリアにデータが入っているので
                        // 全てにセット
                        this.m_cskeyList.Add(csPrpOrgObj.m_cskeyList[iLp]);
                        this.m_csPropatyNameList.Add(csPrpOrgObj.m_csPropatyNameList[iLp]);
                        McDefine.ValKind lKind = csPrpOrgObj.m_csValList[iLp];
                        this.m_csValList.Add(lKind);
                        this.m_csBoolDataList.Add(csPrpOrgObj.m_csBoolDataList[iLp]);
                        this.m_csLongDataList.Add(csPrpOrgObj.m_csLongDataList[iLp]);
                        this.m_csDoubleDataList.Add(csPrpOrgObj.m_csDoubleDataList[iLp]);
                        this.m_csStringDataList.Add(csPrpOrgObj.m_csStringDataList[iLp]);
                        this.m_csMcTranInfoDataList.Add(csPrpOrgObj.m_csMcTranInfoDataList[iLp]);
                    }  
                }
            }
        }

        // =================================
        // プロパティ情報型式設定／取得メソッド
        // =================================
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報型式の追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddInfoType(csKey, csPropatyName, lObjKind) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="csPropatyName">情報名称</param>
        /// <param name="lObjKind">情報データの型</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void AddInfoType(HySString csKey, HySString csPropatyName, McDefine.ValKind lObjKind )
        {
            // 情報引出し用キーリストから検索
            bool bFlg = false;
            int iKeyCnt = m_cskeyList.Count;
            for (int iLp = 0; iLp < iKeyCnt; iLp++)
            {
                // 情報引出し用キーは一致か？
                if (m_cskeyList[iLp].Equal(csKey) == true)
                {
                    // 一致ならば情報名称、情報データの型を入替え
                    bFlg = true;
                    m_csPropatyNameList[iLp] = csPropatyName; // 情報名称上書き
                    if (m_csValList[iLp] != lObjKind)
                    { // 型が異なる場合は前回値をリセット
                        m_csValList[iLp] = lObjKind;
                        m_csBoolDataList[iLp] = false;
                        m_csLongDataList[iLp] = 0;
                        m_csDoubleDataList[iLp] = 0.0D;
                        m_csMcTranInfoDataList[iLp] = null;
                        m_csStringDataList[iLp] = "";
                    }
                    break;
                }
            }

            if (!bFlg)
            { // 新規の場合
                // 情報引き出しキー設定
                m_cskeyList.Add(csKey);
                // プロパティ情報型式設定
                m_csPropatyNameList.Add(csPropatyName);
                // 情報データ型設定
                m_csValList.Add(lObjKind);
                // 配列データ設定（初期値）
                m_csBoolDataList.Add(false);
                m_csLongDataList.Add(0);
                m_csDoubleDataList.Add(0.0D);
                m_csMcTranInfoDataList.Add(null);
                m_csStringDataList.Add("");
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報型式の追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddInfoType(csKey, csPropatyName, lObjKind) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="csPropatyName">情報名称</param>
        /// <param name="lObjKind">情報データの型</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void AddInfoType(string csKey, string csPropatyName, McDefine.ValKind lObjKind)
        {
            this.AddInfoType(new HySString(csKey), new HySString(csPropatyName), lObjKind);
        }

        /// <summary><para>method outline:</para>
        /// <para>指定プロパティ情報型式の削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveInfoType(csKey) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void RemoveInfoType(HySString csKey)
        {      
            int iKeyCnt = m_cskeyList.Count;
            for (int iLp = 0; iLp < iKeyCnt; iLp++)
            {
                // 情報引出し用キーは一致か？
                if (m_cskeyList[iLp].Equal(csKey) == true)
                {
                    //m_cskeyList.Remove(m_cskeyList[iLp]);
                    //m_csPropatyNameList.Remove(m_csPropatyNameList[iLp]);
                    //m_csValList.Remove(m_csValList[iLp]);
                    //m_csBoolDataList.Remove(m_csBoolDataList[iLp]);
                    //m_csLongDataList.Remove(m_csLongDataList[iLp]);
                    //m_csDoubleDataList.Remove(m_csDoubleDataList[iLp]);
                    //m_csStringDataList.Remove(m_csStringDataList[iLp]);
                    //m_csMcTranInfoDataList.Remove(m_csMcTranInfoDataList[iLp]);
                    m_cskeyList.RemoveAt(iLp);
                    m_csPropatyNameList.RemoveAt(iLp);
                    m_csValList.RemoveAt(iLp);
                    m_csBoolDataList.RemoveAt(iLp);
                    m_csLongDataList.RemoveAt(iLp);
                    m_csDoubleDataList.RemoveAt(iLp);
                    m_csStringDataList.RemoveAt(iLp);
                    m_csMcTranInfoDataList.RemoveAt(iLp);
                    break;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>指定プロパティ情報型式の削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveInfoType(csKey) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void RemoveInfoType(string csKey)
        {
            this.RemoveInfoType(new HySString(csKey));
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報型式の数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lCount = GetInfoTypeNum() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long  情報型式の数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetInfoTypeNum()
        {
            return m_cskeyList.Count;
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報型式の取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfoType(csKey, ref csPropatyName, ref lObjKind) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="csPropatyName">情報名称</param>
        /// <param name="lObjKind">情報データの型</param>
        /// <returns>bool  =true:正常、=false:該当無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfoType(HySString csKey, ref HySString csPropatyName, ref McDefine.ValKind lObjKind)
        {
            if (m_cskeyList == null)
            {
                return false;
            }
            // 情報引出し用キーリストから検索
            int iKeyCnt = m_cskeyList.Count;
            for (int iLp = 0; iLp < iKeyCnt; iLp++)
            {
                // 情報引出し用キーは一致か？
                if (m_cskeyList[iLp].Equal(csKey) == true)
                {
                    // 一致ならば情報名称、情報データの型を取得
                    csPropatyName = m_csPropatyNameList[iLp];
                    lObjKind = m_csValList[iLp];
                    return true;
                }
            }
            return false;
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報型式の取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfoType(lOrderNo, ref csKey, ref csPropatyName, ref lObjKind) </para>
        /// </example>
        /// <param name="lOrderNo">アイテム位置（１相対）</param>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="csPropatyName">情報名称</param>
        /// <param name="lObjKind">情報データの型</param>
        /// <returns>bool  =true:正常、=false:該当無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfoType(long lOrderNo, ref HySString csKey, ref HySString csPropatyName, ref McDefine.ValKind lObjKind)
        {
            // 登録済み情報引出し用キー件数以内か判定
            if (lOrderNo > 0 && lOrderNo <= m_cskeyList.Count)
            {
                // 情報名称、情報データの型を取得
                csKey = m_cskeyList[(int)lOrderNo - 1];
                csPropatyName = m_csPropatyNameList[(int)lOrderNo - 1];
                lObjKind = m_csValList[(int)lOrderNo - 1];
                return true;
            }
            else
            {
                return false;
            }
        }

        // =================
        // 値のセットメソッド
        // =================
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(csKey, bVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="bVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(HySString csKey, bool bVal)
        {
            if (m_cskeyList != null)
            {
                // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        m_csBoolDataList[iLp] = bVal;
                        return true;
                    }
                }
             }
           return false;
        }
        /// <summary>
        /// <para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(sKey, bVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="bVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(string sKey, bool bVal)
        {
            return this.SetInfo(new HySString(sKey), bVal);
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(csKey, lVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="lVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(HySString csKey, long lVal)
        {
            if (m_cskeyList != null)
            {
                // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        m_csLongDataList[iLp] = lVal;
                        return true;
                    }
                }
            }
            return false;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(sKey, lVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="lVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(string sKey, long lVal)
        {
            return this.SetInfo(new HySString(sKey), lVal);
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(csKey, dVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="dVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(HySString csKey, double dVal)
        {
            if (m_cskeyList != null)
            {
                 // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        m_csDoubleDataList[iLp] = dVal;
                        return true;
                    }
                } 
            }
            return false;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(sKey, sdVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="dVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(string sKey, double dVal)
        {
            return this.SetInfo(new HySString(sKey), dVal);
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(csKey, sVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="sVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(HySString csKey, string sVal)
        {
            if (m_cskeyList != null)
            {
                // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        m_csStringDataList[iLp] = sVal;
                        return true;
                    }
                }
            }
            return false;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(sKey, dVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="sVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(string sKey, string sVal)
        {
            return this.SetInfo(new HySString(sKey), sVal);
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = SetInfo(csKey, csVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="csVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool SetInfo(HySString csKey, HySString csVal)
        {
            return this.SetInfo(csKey, csVal.ToString());
        }

        // =================
        // 値取得メソッド
        // =================
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(csKey, ref bVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="bVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(HySString csKey, ref bool bVal)
        {
            if (m_cskeyList != null)
            {
                // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        bVal = m_csBoolDataList[iLp];
                        return true;
                    }
                }
            }
            return false;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(sKey, ref bVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="bVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(string sKey, ref bool bVal)
        {
            return this.GetInfo(new HySString(sKey), ref bVal);
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(csKey, ref lVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="lVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(HySString csKey, ref long lVal)
        {
            if (m_cskeyList != null)
            {
                // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        lVal = m_csLongDataList[iLp];
                        return true;
                    }
                }
            }
            return false;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(sKey, ref lVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="lVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(string sKey, ref long lVal)
        {
            return this.GetInfo(new HySString(sKey), ref lVal);
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(csKey, ref dVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="dVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(HySString csKey, ref double dVal)
        {
            if (m_cskeyList != null)
            {
                // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        dVal = m_csDoubleDataList[iLp];
                        return true;
                    }
                }  
            }

            return false;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(sKey, ref dVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="dVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(string sKey, ref double dVal)
        {
            return this.GetInfo(new HySString(sKey), ref dVal);
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(csKey, ref dVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="sVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(HySString csKey, ref string sVal)
        {
            if (m_cskeyList != null)
            {
                // 情報引出し用キーリストから検索
                int iKeyCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < iKeyCnt; iLp++)
                {
                    // 情報引出し用キーは一致か？
                    if (m_cskeyList[iLp].Equal(csKey) == true)
                    {
                        // 一致ならば情報データを設定
                        sVal = m_csStringDataList[iLp];
                        return true;
                    }
                }
            }
            return false;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(sKey, ref dVal) </para>
        /// </example>
        /// <param name="sKey">情報引出し用キー</param>
        /// <param name="sVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(string sKey, ref string sVal)
        {
            return this.GetInfo(new HySString(sKey), ref sVal);
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetInfo(csKey, ref csVal) </para>
        /// </example>
        /// <param name="csKey">情報引出し用キー</param>
        /// <param name="csVal">設定値</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetInfo(HySString csKey, ref HySString csVal)
        {
            string sVal="";
            bool bRtn = this.GetInfo(csKey, ref sVal);
            csVal.SetChar(sVal);
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>内部情報クリア</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Clear() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public void Clear()
        {
            if (m_cskeyList != null)
            {
                m_cskeyList.Clear();
                m_csPropatyNameList.Clear();
                m_csValList.Clear();
                m_csBoolDataList.Clear();
                m_csLongDataList.Clear();
                m_csDoubleDataList.Clear();
                m_csStringDataList.Clear();
                m_csMcTranInfoDataList.Clear();
            }

        }


        /// <summary><para>method outline:</para>
        /// <para>保持できる最大レコード数を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetMaxRecordDataNum(lMaxRecord) </para>
        /// </example>
        /// <param name="lMaxRecord">最大レコード数</param>
        /// <returns>無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>時系列情報の場合に有効</para>
        /// </remarks>
        public virtual void SetMaxRecordDataNum(long lMaxRecord)
        {
            this.RemoveInfoType(McPropertyInfo.MAX_RECORD_DATA_NUM);
            this.AddInfoType(McPropertyInfo.MAX_RECORD_DATA_NUM, Properties.HymcoImplResources.TRANINFO_PROPERTY_INFO_CELL_ARRAY_MAX_RECORD, McDefine.ValKind.LONG);   //"最大レコード数"
            this.SetInfo(McPropertyInfo.MAX_RECORD_DATA_NUM, lMaxRecord);
        }
        /// <summary><para>method outline:</para>
        /// <para>保持できる最大レコード数を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lNum = GetMaxRecordDataNum() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 最大レコード数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>時系列情報の場合に有効</para>
        /// </remarks>
        public virtual long GetMaxRecordDataNum()
        {
            long lRtn = 0;
            this.GetInfo(McPropertyInfo.MAX_RECORD_DATA_NUM, ref lRtn);
            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算実行刻み時間（δT)設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetStepTime(dDltT) </para>
        /// </example>
        /// <param name="dDltT">演算実行刻み時間（δT)(sec)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetStepTime(double dDltT)
        {
            this.SetInfo(TIME_STEP, dDltT);
        }
        /// <summary><para>method outline:</para>
        /// <para>演算実行刻み時間（δT)取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double dDltT = GetStepTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double  演算実行刻み時間（δT) (sec)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual double GetStepTime()
        {
            double dRtn = 0.0;
            this.GetInfo(TIME_STEP, ref dRtn);
            return dRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>演算実行刻み時間設定解除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveStepTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void RemoveStepTime()
        {
            RemoveInfoType(TIME_STEP);
        }


        /// <summary><para>method outline:</para>
        /// <para>モデル情報をXMLへ書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySXmlNode csXmlNode = XmlOUT(csXmlWriter) </para>
        /// </example>
        /// <param name="csXmlWriter">XML書き出しクラス</param>
        /// <returns>HySXmlNode  XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySXmlNode XmlOUT(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtn = base.XmlOUT(ref csXmlWriter);

            //if (g_lXmlOutFlg == true)
            if (g_lXmlOutFlg != McDefine.HYMCO_MODEL_STRUCTINFO_WRITE_PART)
            {   // XML出力ONならば
                HySXmlNode csLclNode;
                long lDataCnt = m_cskeyList.Count;
                for (int iLp = 0; iLp < lDataCnt; iLp++)
                {
                    csLclNode = HySXmlNode.CreateElement(McDefine.xml_VARIABLE, csXmlWriter);


                    HySString csKey = m_cskeyList[iLp];
                    HySString csName = m_csPropatyNameList[iLp];
                    McDefine.ValKind lKnd = m_csValList[iLp];
                    bool bVal = m_csBoolDataList[iLp];
                    long lVal = m_csLongDataList[iLp];
                    double dVal = m_csDoubleDataList[iLp];
                    string sVal = m_csStringDataList[iLp];

                    csLclNode.SetAttribute(McDefine.xml_KEY, csKey.ToString());
                    csLclNode.SetAttribute(McDefine.xml_NAME, csName.ToString());
                    if (lKnd == McDefine.ValKind.BOOL)
                    {
                        csLclNode.SetAttribute(McDefine.xml_DATATYPE, "BOOL");
                        csLclNode.SetAttribute(McDefine.xml_VALUE, bVal.ToString());
                    }
                    else if (lKnd == McDefine.ValKind.LONG)
                    {
                        csLclNode.SetAttribute(McDefine.xml_DATATYPE, "LONG");
                        csLclNode.SetAttribute(McDefine.xml_VALUE, lVal.ToString());
                    }
                    else if (lKnd == McDefine.ValKind.DOUBLE)
                    {
                        csLclNode.SetAttribute(McDefine.xml_DATATYPE, "DOUBLE");
                        csLclNode.SetAttribute(McDefine.xml_VALUE, dVal.ToString());
                    }
                    else if (lKnd == McDefine.ValKind.STRING)
                    {
                        csLclNode.SetAttribute(McDefine.xml_DATATYPE, "STRING");
                        csLclNode.SetAttribute(McDefine.xml_VALUE, sVal);
                    }
                    csRtn.AppendChild(csLclNode);   // AppendChildは最後に行う
                }
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>XMLからモデル情報を読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = XmlIN(csPrptyXml) </para>
        /// </example>
        /// <param name="csPrptyXml">入力XML</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool XmlIN(HySXmlNode csPrptyXml)
        {
            this.Clear();
            bool bRtn = base.XmlIN(csPrptyXml);

            HySXmlNode csXml = null;
            HySXmlNodeList csXmlVariableList = csPrptyXml.GetChildNodeListByTagName(McDefine.xml_VARIABLE);
            long lValCnt = csXmlVariableList.GetCount();
            for (long lLoop = 0; lLoop < lValCnt; lLoop++)
            {
                csXml = csXmlVariableList.GetNode(lLoop);

                string sKeyAttr;
                string sNameAttr;
                string sDataTypeAttr;
                string sValueAttr;
                McDefine.ValKind eKind;
                bool bValue = false;
                long lValue = 0;
                double dValue = 0.0D;
                string sValue = "";
                // キー
                sKeyAttr = csXml.GetAttribute(McDefine.xml_KEY);
                // 名称
                sNameAttr = csXml.GetAttribute(McDefine.xml_NAME);
                // データタイプ
                sDataTypeAttr = csXml.GetAttribute(McDefine.xml_DATATYPE);
                // データ
                sValueAttr = csXml.GetAttribute(McDefine.xml_VALUE);
                if (sDataTypeAttr == "BOOL")
                {
                    eKind = McDefine.ValKind.BOOL;
                    bValue = Convert.ToBoolean(sValueAttr);
                }
                else if (sDataTypeAttr == "LONG")
                {
                    eKind = McDefine.ValKind.LONG;
                    lValue = Convert.ToInt32(sValueAttr);
                }
                else if (sDataTypeAttr == "DOUBLE")
                {
                    eKind = McDefine.ValKind.DOUBLE;
                    dValue = Convert.ToDouble(sValueAttr);
                }
                else
                {
                    eKind = McDefine.ValKind.STRING;
                    sValue = sValueAttr;
                }
                if (sKeyAttr != "" )
                {
                    this.AddInfoType(sKeyAttr, sNameAttr, eKind);
                    this.SetInfo(sKeyAttr, bValue);
                    this.SetInfo(sKeyAttr, lValue);
                    this.SetInfo(sKeyAttr, dValue);
                    this.SetInfo(sKeyAttr, sValue); 
                }
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>セルと識別子の対応管理データの生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McCellIDMngIF csDt = CreateCellIDMngData(csKind) </para>
        /// </example>
        /// <param name="csKind">識別子</param>
        /// <returns>McCellIDMngIF  セルと識別子の対応管理データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McCellIDMngIF CreateCellIDMngData(HySKind csKind)
        {
            McCellIDMngIF csRtn = McCellIDMngrFactory.CreateCellIDManageData(csKind) as McCellIDMngIF;
            if( csRtn == null )
            {
                csRtn = new McModelCellCorrespondenceInf();
            }

            return csRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>自分を保持しているStructElementインスタンスの設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetStructElement(csElm) </para>
        /// </example>
        /// <param name="csElm">自分を保持しているMcStructElementインスタンス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetStructElement(McStructElement csElm)
        {
            m_csStructElm = csElm;
        }
        /// <summary><para>method outline:</para>
        /// <para>自インスタンスを保持しているプロジェクトのＩＤ値を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sID = GetOwnerProjectID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>自インスタンスを保持しているプロジェクトのＩＤ値</returns>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual string GetOwnerProjectID()
        {
            string sRtn = "";
            if (m_csStructElm != null)
            {
                sRtn = m_csStructElm.GetOwnerProjectID();
            }
            return sRtn;
        }

        /*
        /// <summary><para>method outline:</para>
        /// <para>CommonMPデータホームディレクトリ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csDataHomeDirectory = GetDataHomeDirectory()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString  CommonMPデータホームディレクトリ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySString GetDataHomeDirectory()
        {
            return HySEnvInf.GetDataHomeDirectory();
        }
        */
        /// <summary><para>method outline:</para>
        /// <para>プロジェクトグループ名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csGrpName = GetProjectGroupName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString  プロジェクトグループ名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySString GetProjectGroupName()
        {
            string sPrjName = "";
            HySCommonInfoHash.GetCorrespond(McDefine.PROJECT_GROUP_NAME, this.GetOwnerProjectID(), ref sPrjName);
            return new HySString(sPrjName);
        }
        /// <summary><para>method outline:</para>
        /// <para>プロジェクトグループディレクトリ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csDirectory = GetProjectGroupDirectory()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString  プロジェクトグループディレクトリ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySString GetProjectGroupDirectory()
        {
            return (HySEnvInf.GetDataHomeDirectory() + new HySString("\\") + this.GetProjectGroupName());
        }
        /// <summary><para>method outline:</para>
        /// <para>計算の識別子（計算ロット）を取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csCalLot = GetCalculationLotName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString  計算の識別子（計算ロット）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>同じプロジェクトで計算を繰り返す場合、前回の計算と今回の計算を区別するための識別子を取得する</para>
        /// </remarks>
        public override HySString GetCalculationLotName()
        {
            string sLotName = "";
            HySCommonInfoHash.GetCorrespond(McDefine.CALCULATION_LOT_NAME, this.GetOwnerProjectID(), ref sLotName);
            return new HySString(sLotName);
        }
        /// <summary><para>method outline:</para>
        /// <para>データホームを基準とした相対パスを取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetRelativePathBasedOnDataHome(csAbsolutePath, refcsRelativePath)</para>
        /// </example>
        /// <param name="csAbsolutePath">絶対パス</param>
        /// <param name="csRelativePath">相対パス</param>
        /// <returns>相対化不能の場合には　false を返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>与えられた絶対パス値を　CommonMPデータホームを基準とした相対パスに変換して取得する</para>
        /// </remarks>
        public override bool GetRelativePathBasedOnDataHome(HySString csAbsolutePath, ref HySString csRelativePath)
        {
            string sRelativePath = "";
            bool bRtn = HySCommonInfoHash.DirectoryPathExchange(
                this.GetDataHomeDirectory().ToString(),
                csAbsolutePath.ToString(),
                ref sRelativePath);
            csRelativePath.SetChar(sRelativePath);
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロジェクトグループディレクトリを基準とした相対パスを取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetRelativePathBasedOnProjectGroup(csAbsolutePath, refcsRelativePath)</para>
        /// </example>
        /// <param name="csAbsolutePath">絶対パス</param>
        /// <param name="csRelativePath">相対パス</param>
        /// <returns>相対化不能の場合には　false を返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>与えられた絶対パス値を　プロジェクトグループディレクトリを基準とした相対パスに変換して取得する</para>
        /// </remarks>
        public override bool GetRelativePathBasedOnProjectGroup(HySString csAbsolutePath, ref HySString csRelativePath)
        {
            string sRelativePath = "";
            bool bRtn = HySCommonInfoHash.DirectoryPathExchange(
                this.GetDataHomeDirectory().ToString() + "\\" + this.GetProjectGroupName().ToString(),
                csAbsolutePath.ToString(),
                ref sRelativePath);
            csRelativePath.SetChar(sRelativePath);
            return bRtn;
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public new string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public new void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public override void PostDeserialize()
        {
            base.PostDeserialize();

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }

        // ================================================
        //   常数
        // ================================================
        /// <summary>  最大レコード数（指定された数までレコードを保持できる）</summary>
        static public string MAX_RECORD_DATA_NUM = "MaxRecordDataNum";
        /// <summary> 演算実行刻み時間（δT) </summary>
        static public string TIME_STEP = "TIME_STEP";
    }
}
