﻿// <summary>ソースコード：シミュレーション状態保持クラス シミュレーションに関する状態を保持する。</summary>
// <author>CommonMP</author>

using System;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;

namespace CommonMP.HYMCO.CoreImpl.Data.ProjectCtl
{
    /// <summary><para>class outline:</para>
    /// <para>シミュレーション状態保持クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/02/10][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class McSimulationStatus : HySDataRoot, HySVersionManagement
    {
        /// <summary>計算状態種別</summary>
        public enum CalculationStatus
        {
            /// <summary>未計算</summary>
            INITIAL,
            /// <summary>中断</summary>
            INTERRUPTED,
            /// <summary>計算中</summary>
            CALCULATING,
            /// <summary>計算終了</summary>
            FINISHED,
            /// <summary>計算強制終了</summary>
            FINISHED_STOP,
        }

        /// <summary>モデル構築状態種別</summary>
        public enum ModelConstructionStatus
        {
            /// <summary>モデル未構築</summary>
            NOT_CONSTRUCTED,
            /// <summary>モデル構築済</summary>
            CONSTRUCTED,
        }

        /// <summary>モデル変更状態種別</summary>
        public enum ModelModificationStatus
        {
            /// <summary>変更無</summary>
            UNMODIFIED,
            /// <summary>変更有(モデル構築不要)</summary>
            MODIFIED_CONSTRUCTION_NOT_REQUIRED,
            /// <summary>変更有(モデル構築要)</summary>
            MODIFIED_CONSTRUCTION_REQUIRED,
        }

        /// <summary>モデル保存状態種別</summary>
        public enum ModelSaveStatus
        {
            /// <summary>未保存</summary>
            UNSAVED,
            /// <summary>保存済</summary>
            SAVED,
        }

        /// <summary>計算状態</summary>
        private CalculationStatus m_eCalcStatus;
        /// <summary>モデル構築状態</summary>
        private ModelConstructionStatus m_eModelConStatus;
        /// <summary>モデル変更状態</summary>
        private ModelModificationStatus m_eModelModStatus;
        /// <summary>モデル保存状態</summary>
        private ModelSaveStatus m_eModelSaveStatus;

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McSimulationStatus status = new McSimulationStatus()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McSimulationStatus()
        {
            m_eCalcStatus = CalculationStatus.INITIAL;
            m_eModelConStatus = ModelConstructionStatus.NOT_CONSTRUCTED;
            m_eModelModStatus = ModelModificationStatus.UNMODIFIED;
            m_eModelSaveStatus = ModelSaveStatus.UNSAVED;
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>計算状態設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCalcStatus(status)</para>
        /// </example>
        /// <param name="status">計算状態</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCalcStatus(CalculationStatus status)
        {
            m_eCalcStatus = status;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算状態取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalculationStatus eCalStatus = GetCalcStatus()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>計算状態</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public CalculationStatus GetCalcStatus()
        {
            return m_eCalcStatus;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル構築状態設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetModelConStatus(status)</para>
        /// </example>
        /// <param name="status">モデル構築状態</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetModelConStatus(ModelConstructionStatus status)
        {
            m_eModelConStatus = status;
            if (status == ModelConstructionStatus.CONSTRUCTED)
            {
                m_eModelModStatus = ModelModificationStatus.UNMODIFIED;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル構築状態取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ModelConstructionStatus eModelConStatus = GetModelConStatus()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>モデル構築状態</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public ModelConstructionStatus GetModelConStatus()
        {
            return m_eModelConStatus;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル変更状態設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetModelModState(status)</para>
        /// </example>
        /// <param name="status">モデル変更状態</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetModelModStatus(ModelModificationStatus status)
        {
            m_eModelModStatus = status;
            if (status == ModelModificationStatus.MODIFIED_CONSTRUCTION_NOT_REQUIRED ||
                status == ModelModificationStatus.MODIFIED_CONSTRUCTION_REQUIRED)
            {
                m_eModelSaveStatus = ModelSaveStatus.UNSAVED;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル変更状態取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ModelModificationStatus eModelModStatus = GetModelModStatus()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>モデル変更状態</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public ModelModificationStatus GetModelModStatus()
        {
            return m_eModelModStatus;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル保存状態設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetModelSaveStatus(status)</para>
        /// </example>
        /// <param name="status">モデル保存状態</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetModelSaveStatus(ModelSaveStatus status)
        {
            m_eModelSaveStatus = status;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル保存状態取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ModelSaveStatus eModelSaveStatus = GetModelSaveStatus()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>モデル保存状態</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public ModelSaveStatus GetModelSaveStatus()
        {
            return m_eModelSaveStatus;
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.0":
                        break;
                }
                SetNowVersion();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
    }
}
