﻿// <summary>ソースコード：接続構造要素</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.Interface.Model;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Model;


namespace CommonMP.HYMCO.CoreImpl.Data.StructInfo
{
    /// <summary><para>class outline:</para>
    /// <para>接続構造要素クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>
    /// [CommonMP][ver 1.0.0][2008/11/17][新規作成]
    /// [CommonMP][ver 1.2.0][2011/11/10][レコード数最適化処理追加]
    /// </para>
    /// </remarks>
    [Serializable]
    public class McStructElement : HySDataRoot, HySVersionManagement //, IDeserializationCallback
    {
        /// <summary>識別子</summary>
        protected HySIdentifier m_csID;

        // ========================
        // 演算モデル生成に必要な情報
        // ========================
        /// <summary>モデル個別情報（必須）</summary>
        protected McModelInfo m_csModelInfo = new McModelInfo(new HySID(""), new HySObjectKind(""), new HySObjectKind(""), new HySString(""));
                      // 最初は仮設定しておく。　 モデルを設定していなくても，構造定義ファイルを吐き出したときに死なない為;

        ///// <summary>モデルプロパティ表示／設定情報（必須）</summary>
        //McModelPropertyInfo m_csModelProperty;  ← m_csModelInfo内にある為、此処での宣言は不要

        /// <summary>モデル生成用ファクトリーのセット </summary>
        [NonSerialized]
        protected static McModelFactorySet s_csMdlFactorySet = null;

        // ========================
        // 表示に必要な情報
        // ========================
        /// <summary>種別</summary>
        protected McDefine.SymbolDrawKind m_eDrawKind = McDefine.SymbolDrawKind.HYM_STD_ELEMENT;
        /// <summary>描画オブジェクト区分</summary>
        protected HySObjectKind m_csDrawSubKind = null;
        /// <summary>画面表示端子リスト</summary>
        protected HySDataLinkedList m_csPortList = new HySDataLinkedList();
        /// <summary>描画位置(X)</summary>
        protected long m_lX = 10;
        /// <summary>描画位置(Y)</summary>
        protected long m_lY = 10;
        /// <summary>描画サイズ（幅）</summary>
        protected long m_lWidth = 0;
        /// <summary>描画サイズ（高さ）</summary>
        protected long m_lHeight = 0;
        /// <summary>描画位置(回転)</summary>
        protected double m_lRotate = 0.0D;
        /// <summary>要素名称</summary>
        protected HySString m_sElementName = new HySString();

        // start of ＸＭＬ出力不用
        /// <summary>自身を所有するグループ要素</summary>
        protected McStructGroupElement m_csOwnerGrElm = null;
        // start of ＸＭＬ出力不用

        // 一貫番号生成
        /// <summary>画面表示要素カウンタ</summary>
        static protected ulong m_ulPrimElm = 0UL;
        /// <summary> 要素インスタンス数監視 </summary>
        static public ulong m_ulTotalElmInstanceNum = 0UL;

        //// <summary> 一時保管 </summary>
        //[NonSerialized]
        //McCalModel m_csCalModel = null;

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructElement csElm = new McStructElement(eKind)</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <returns>インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McStructElement(McDefine.SymbolDrawKind eKind)
        {
            m_eDrawKind = eKind;
            McStructElement.m_ulPrimElm += 1;
            McStructElement.m_ulTotalElmInstanceNum += 1;
            //string sStr = DateTime.Now.Minute.ToString() + DateTime.Now.Hour.ToString()
            //                + "." + DateTime.Now.Day.ToString() + DateTime.Now.Month.ToString() + DateTime.Now.Year.ToString();
            //m_csID = new HySID("E" + m_ulPrimElm.ToString() + DateTime.Now.Second.ToString() + "-",
            //                    ((long)eKind).ToString() + "-", sStr);
            string sStr = DateTime.Now.Minute.ToString() + DateTime.Now.Hour.ToString()
                            + "-" + DateTime.Now.Day.ToString() + DateTime.Now.Month.ToString() + DateTime.Now.Year.ToString();
            //m_csID = new HySID("E" + m_ulPrimElm.ToString() + DateTime.Now.Second.ToString() + "-",
            //                    ((long)eKind).ToString(), sStr);
            m_csID = new HySID("E" + m_ulPrimElm.ToString() + DateTime.Now.Second.ToString() + "-"
                                + ((long)eKind).ToString() + sStr);

            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>ID設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetID(csID)</para>
        /// </example>
        /// <param name="csID">ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetID(HySIdentifier csID)
        {
            m_csID = csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>ID取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySIdentifier cs Identifier = GetID()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySIdentifier GetID()
        {
            return m_csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>ID一致判定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = Equal( csID )</para>
        /// </example>
        /// <param name="csID">ID</param>
        /// <returns>bool true:一致，false:不一致</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool Equal(HySIdentifier csID)
        {
            if (csID == null || m_csID == null)
            {// パラメータ,メンバ変数異常
                return false;
            }

            return m_csID.Equals(csID);
        }

        /// <summary><para>method outline:</para>
        /// <para>要素名称設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetElementName(ElementName)</para>
        /// </example>
        /// <param name="ElementName">要素名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetElementName(HySString ElementName)
        {
            if ((object)ElementName != null)
            {
                //m_sElementName = ElementName;
                if ((object)m_sElementName != null)
                {
                    m_sElementName.SetString(ElementName);
                }
                else
                {
                    m_sElementName = ElementName;
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>要素名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySString csString = GetElementName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>要素名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySString GetElementName()
        {
            HySString sRtnElementName = new HySString();
            if ((object)m_sElementName != null)
            {
                sRtnElementName = m_sElementName;
            }

            return sRtnElementName;
        }

        /// <summary><para>method outline:</para>
        /// <para>シンボル描画種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>シンボル描画種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McDefine.SymbolDrawKind GetSymbolDrawKind()
        {
            return m_eDrawKind;
        }

        /// <summary><para>method outline:</para>
        /// <para>自分を所有するグループ要素設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetOwnerElement(csOwnerGrElm)</para>
        /// </example>
        /// <param name="csOwnerGrElm">自分を所有するグループ要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetOwnerElement(McStructGroupElement csOwnerGrElm)
        {
            m_csOwnerGrElm = csOwnerGrElm;
        }
        /// <summary><para>method outline:</para>
        /// <para>自分を所有するグループ要素取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructGroupElement csOwner = GetOwnerElement()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McStructGroupElement 自分を所有するグループ要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McStructGroupElement GetOwnerElement()
        {
            return m_csOwnerGrElm;
        }
        /// <summary><para>method outline:</para>
        /// <para>自インスタンスを保持しているプロジェクトのＩＤ値を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sID = GetOwnerProjectID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>自インスタンスを保持しているプロジェクトのＩＤ値</returns>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetOwnerProjectID()
        {
            string sRtn = this.m_csID.ToString();
            if (m_csOwnerGrElm != null)
            {
                sRtn = m_csOwnerGrElm.GetOwnerProjectID();
            }
            return sRtn;
        }
        //======================================
        // 端子関連メソッド
        //======================================
        /// <summary><para>method outline:</para>
        /// <para>画面表示端子数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = GetPortCount()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>画面表示端子数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetPortCount()
        {
            return m_csPortList.GetCount();
        }
        /// <summary><para>method outline:</para>
        /// <para>画面表示端子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructPort csStructPort = GetPort(lPos)</para>
        /// </example>
        /// <param name="lPos">位置(0相対)</param>
        /// <returns>端子インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McStructPort GetPort(long lPos)
        {
            McStructPort csPort = null;  // 画面表示端子

            csPort = (McStructPort)m_csPortList.GetData(lPos);

            return csPort;
        }
        /// <summary><para>method outline:</para>
        /// <para>画面表示端子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructPort csStructPort = GetPort(csPortID)</para>
        /// </example>
        /// <param name="csPortID">描画要素端子ID</param>
        /// <returns>端子インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McStructPort GetPort(HySIdentifier csPortID)
        {
            McStructPort csRtnPort = null;  // 端子
            McStructPort csPort = null;  // 端子

            long lPNum = m_csPortList.GetCount();
            m_csPortList.SetCursorFirst();
            for (int iLp = 0; iLp < lPNum; iLp++)
            {
                csPort = (McStructPort)m_csPortList.GetCursorData();
                if (csPort.GetID().Equals(csPortID) == true)
                {
                    csRtnPort = csPort;
                    break;
                }
                m_csPortList.MoveCursorNext();
            }
            return csRtnPort;
        }
        /// <summary><para>method outline:</para>
        /// <para>画面表示端子追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = AddPort( csPort )</para>
        /// </example>
        /// <param name="csPort">画面表示端子</param>
        /// <returns>long 0:成功，-1:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual long AddPort(McStructPort csPort)
        {
            if (csPort == null)
            {// パラメータ異常
                return -1L;
            }
            csPort.SetElmID(this.GetID());
            m_csPortList.AddLast(csPort);

            return 0L;
        }
        /// <summary><para>method outline:</para>
        /// <para>描画要素端子削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructPort csStructPort = DelPort(csPortID)</para>
        /// </example>
        /// <param name="csPortID">描画要素端子ID</param>
        /// <returns>McStructPort !=null:成功，null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McStructPort DelPort(HySIdentifier csPortID)
        {
            McStructPort csRtnPort = null;
            McStructPort csVPort = null;
            long lNum = m_csPortList.GetCount();

            m_csPortList.SetCursorFirst();
            for (int iLp = 0; iLp < lNum; iLp++)
            {
                csVPort = (McStructPort)m_csPortList.GetCursorData();
                if (csVPort.GetID().Equals(csPortID) == true)
                {
                    csRtnPort = csVPort;
                    m_csPortList.RemoveData(iLp);
                    break;
                }
                m_csPortList.MoveCursorNext();
            }
            return csRtnPort;
        }     
        /// <summary>
        /// <para>method outline:</para>
        /// <para>端子生成</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McStructPort csStructPort = CreateNewStructPort(eKind)</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <returns>McStructPort !=null:成功，null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual protected McStructPort CreateNewStructPort(McDefine.SymbolDrawKind eKind)
        {
            McStructPort csPort = null;    // 画面表示端子

            if (eKind == McDefine.SymbolDrawKind.HYM_TRANS_IN)
            {   // 入力中継端子
                csPort = new McStructTranINPort();
            }
            else if (eKind == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
            {   // 出力中継端子
                csPort = new McStructTranOUTPort();
            }
            //else if (eKind == McDefine.SymbolDrawKind.HYM_INPUT)
            //{   // 入力端子
            //    csPort = new McStructPort(eKind);
            //}
            //else if (eKind == McDefine.SymbolDrawKind.HYM_OUTPUT)
            //{   // 出力端子
            //    csPort = new McStructPort(eKind);
            //}
            else if (eKind == McDefine.SymbolDrawKind.HYM_RECEIVE)
            {   // 受信端子
                csPort = new McStructPort(eKind);
            }
            else if (eKind == McDefine.SymbolDrawKind.HYM_SEND)
            {   // 送信端子
                csPort = new McStructPort(eKind);
            }
            return csPort;
        }

        //======================================
        // モデル情報関連メソッド
        //======================================
        /// <summary>
        /// <para>method outline:</para>
        /// <para>伝送情報デフォルトパターン設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetDefaultPattern( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDefaultPattern()
        {
            McStructConnection csConnect = null;    // 接続情報
            HySIdentifier csFromID = null;          // 受信接続情報データ識別子
            HySIdentifier csToID = null;            // 送信接続情報データ識別子
            for (int iLoop = 0; iLoop < m_csPortList.GetCount(); iLoop++)
            {   // 接続ポート数分処理
                ((McStructPort)m_csPortList.GetData(iLoop)).GetPortID(ref csFromID,ref csToID);
                csConnect = m_csOwnerGrElm.QueryConnectionByPortID(csFromID);
                if (csConnect != null)
                {   // 接続情報有り
                    csConnect.SetDefaultPattern(null);
                }
                csConnect = m_csOwnerGrElm.QueryConnectionByPortID(csToID);
                if (csConnect != null)
                {   // 接続情報有り
                    csConnect.SetDefaultPattern(null);
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>伝送情報デフォルトパターン設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetDefaultPattern(ref csElmIDList, ref csModelList, ref csModelList)</para>
        /// </example>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csTranList">設定前モデル情報リスト</param>
        /// <param name="csModTranList">設定後モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDefaultPattern(ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csTranList, ref List<McModelInfo> csModTranList)
        {
            McStructConnection csConnect = null;    // 接続情報
            HySIdentifier csFromID = null;          // 受信接続情報データ識別子
            HySIdentifier csToID = null;            // 送信接続情報データ識別子
            for (int iLoop = 0; iLoop < m_csPortList.GetCount(); iLoop++)
            {   // 接続ポート数分処理
                ((McStructPort)m_csPortList.GetData(iLoop)).GetPortID(ref csFromID, ref csToID);
                csConnect = m_csOwnerGrElm.QueryConnectionByPortID(csFromID);
                if (csConnect != null)
                {   // 接続情報有り
                    csElmIDList.Add(csConnect.GetID());
                    csTranList.Add(csConnect.GetTranInfo());
                    csConnect.SetDefaultPattern(null);
                    csModTranList.Add(csConnect.GetTranInfo());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデルに関連するパターン情報のクリア</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>ClearPatternInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ClearPatternInfo()
        {
            // 接続元
            McModelInfo csTranInfo = null;
            McCellArrayTranInfoPropertyInfo csTranInfoPropertyInfo = null;
            HySDataLinkedList csUpperConectInfo = this.GetUpperConnectionInfoList();
            long lCnctNum = csUpperConectInfo.GetCount();
            csUpperConectInfo.SetCursorFirst();
            for (long lCn = 0; lCn < lCnctNum; lCn++)
            {   // 接続数繰り返す
                csTranInfo = (McModelInfo)csUpperConectInfo.GetCursorData();
                csTranInfoPropertyInfo = csTranInfo.GetPropertyInfo() as McCellArrayTranInfoPropertyInfo;
                if (csTranInfoPropertyInfo != null)
                {
                    // モデルに関連するパターン情報（下流）のクリア
                    csTranInfoPropertyInfo.SetReceiveTranInfoPattern(null);
                    csTranInfoPropertyInfo.SetLowerConnectionPatternID(new HySID(""));
                    // 下流側モデル情報の再設定
                    csTranInfoPropertyInfo.SetLowerElementModelInfor(this.m_csModelInfo);
                }
                csUpperConectInfo.MoveCursorNext();
            }
            // 接続先
            HySDataLinkedList csLowerConectInfo = this.GetLowerConnectionInfoList();
            lCnctNum = csLowerConectInfo.GetCount();
            csLowerConectInfo.SetCursorFirst();
            for (long lCn = 0; lCn < lCnctNum; lCn++)
            {   // 接続数繰り返す
                csTranInfo = (McModelInfo)csLowerConectInfo.GetCursorData();
                csTranInfoPropertyInfo = csTranInfo.GetPropertyInfo() as McCellArrayTranInfoPropertyInfo;
                if (csTranInfoPropertyInfo != null)
                {
                    // モデルに関連するパターン情報（上流）のクリア
                    csTranInfoPropertyInfo.SetSendTranInfoPattern(null);
                    csTranInfoPropertyInfo.SetUpperConnectionPatternID(new HySID(""));
                    // 上流側モデル情報の再設定
                    csTranInfoPropertyInfo.SetUpperElementModelInfor(this.m_csModelInfo);
                }
                csLowerConectInfo.MoveCursorNext();
            }
        }
        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデル個別情報設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetModelInfo(csModelInfo)</para>
        /// </example>
        /// <param name="csModelInfo">モデル個別情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetModelInfo(McModelInfo csModelInfo)
        {
            m_csModelInfo = csModelInfo;
            m_csModelInfo.SetElementID((HySID)this.m_csID);
            m_csModelInfo.SetUserName(m_sElementName);
            if (m_csModelInfo.GetPropertyInfo() != null)
            {
                m_csModelInfo.GetPropertyInfo().SetID((HySID)this.m_csID);    // プロパティ情報にも要素識別子を反映
            }
            if (m_csModelInfo.GetInitialInfo() != null)
            {
                m_csModelInfo.GetInitialInfo().SetID((HySID)this.m_csID);    // プロパティ情報にも要素識別子を反映
            }
            // モデルに接続されている接続線のパターン情報をクリアする
            if (this.m_eDrawKind != McDefine.SymbolDrawKind.HYM_CALC_GROUP && this.m_eDrawKind != McDefine.SymbolDrawKind.HYM_GROUP)
            {
                this.ClearPatternInfo();
            }
        }
        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデル個別情報解除</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>CancelModelInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void CancelModelInfo()
        {
            m_csModelInfo = new McModelInfo(new HySID(""), new HySObjectKind(""), new HySObjectKind(""), new HySString(""));
            // モデルに接続されている接続線のパターン情報をクリアする
            if (this.m_eDrawKind != McDefine.SymbolDrawKind.HYM_CALC_GROUP && this.m_eDrawKind != McDefine.SymbolDrawKind.HYM_GROUP)
            {
                this.ClearPatternInfo();
            }
        }
        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデル個別情報取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McModelInfo csModelInfo = GetModelInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McModelInfo　モデル個別情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McModelInfo GetModelInfo()
        {
            McPropertyInfo csMdlProperty = m_csModelInfo.GetPropertyInfo() as McPropertyInfo;
            if (csMdlProperty != null)
            {
                csMdlProperty.SetStructElement(this);
            }

            McCellModelPropertyInfo csMdlPrpty = m_csModelInfo.GetPropertyInfo() as McCellModelPropertyInfo;
            if (csMdlPrpty != null)
            {
                csMdlPrpty.SetInputConnectionInfoList(this.GetUpperConnectionInfoList());
                csMdlPrpty.SetOutputConnectionInfoList(this.GetLowerConnectionInfoList());
            }

            m_csModelInfo.SetUserName(m_sElementName);
            return m_csModelInfo;
        }
        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデル個別情報取得（内部の接続関連情報等は設定しないでそのまま返す）</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McModelInfo csModelInfo = GetModelInfo(bSW)</para>
        /// </example>
        /// <param name="bSW">(=必ずfalse）</param>
        /// <returns>McModelInfo　モデル個別情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual McModelInfo GetModelInfo(bool bSW)
        {
            return m_csModelInfo;
        }
        

        //======================================
        // 構造定義ファイル入出力関連メソッド
        //======================================

        /// <summary><para>method outline:</para>
        /// <para>構造定義ファイル入力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReadStructureConfiguration( csXml,  csMdlFctry )</para>
        /// </example>
        /// <param name="csXml">Xmlノード</param>
        /// <param name="csMdlFctry">モデルファクトリクラス</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool ReadStructureConfiguration(HySXmlNode csXml, McModelFactorySet csMdlFctry )
        {
            string sWorkAttr;
            // 識別子
            sWorkAttr = csXml.GetAttribute(McDefine.xml_ID);
            //m_csID = new HySID(sWorkAttr, "", "");
            m_csID = new HySID(sWorkAttr);
            // 種別
            sWorkAttr = csXml.GetAttribute(McDefine.xml_KIND);
            m_csDrawSubKind = new HySObjectKind(sWorkAttr);

            string sLibraryID = csXml.GetAttribute(McDefine.xml_FACTORYID);
            HySID csLibraryID = new HySID(sLibraryID); //new HySID(sLibraryID,"","");
            string sModel = csXml.GetAttribute(McDefine.xml_MODEL);
            HySObjectKind csMdlKind = new HySObjectKind(sModel);

            if (sLibraryID != string.Empty && sModel != string.Empty)
            {
                McModelInfo csModelInfo = null;
                HySObjectKind csClassification = new HySObjectKind("");
                HySString csModelName = new HySString("");
                HySString csIconName = new HySString("");
                HySObjectKind csModelType = new HySObjectKind("");
                bool bAttachedScreenKind = false;
                // モデル情報をモデルファクトリクラスより取得する
                HySDataLinkedList csModelInfoList = csMdlFctry.GetCalModelInfoList();
                for (long lLoop = 0; lLoop < csModelInfoList.GetCount(); lLoop++)
                {
                    csModelInfo = (McModelInfo)csModelInfoList.GetData(lLoop);
                    if (csLibraryID.Equals(csModelInfo.GetLibraryID()) == true &&
                       csMdlKind.Equals(csModelInfo.GetModelKind()) == true)
                    {
                        csClassification = (HySObjectKind)csModelInfo.GetClassification();
                        csModelName = (HySString)csModelInfo.GetModelName();
                        csIconName = csModelInfo.GetIconName();
                        csModelType = (HySObjectKind)csModelInfo.GetModelType();
                        bAttachedScreenKind = csModelInfo.GetAttachedScreenKind();
                        break;
                    }
                }
                // モデル個別情報生成
                //m_csModelInfo = new McModelInfo(
                //                        csLibraryID,
                //                        csClassification,
                //                        csMdlKind,
                //                        csModelName);
                m_csModelInfo = new McModelInfo(
                                        csLibraryID,
                                        csClassification,
                                        csMdlKind,
                                        csModelName,
                                        csModelType);

                m_csModelInfo.SetIconName(csIconName);  // アイコン名称設定
                m_csModelInfo.SetAttachedScreenKind(bAttachedScreenKind);  // 付属画面有無区分設定
                McPropertyInfoRoot csPropertyDt = csMdlFctry.CreateModelPropertyInfo(csLibraryID, csMdlKind);
                if (csPropertyDt != null)
                {
                    csPropertyDt.SetID((HySID)this.m_csID);                        // プロパティ情報に要素ＩＤをセット
                    m_csModelInfo.SetPropertyInfo(csPropertyDt);
                }
                McPropertyInfoRoot csInitialDt = csMdlFctry.CreateModelInitialInfo(csLibraryID, csMdlKind);
                if (csInitialDt != null)
                {
                    csInitialDt.SetID((HySID)this.m_csID);                        // 初期情報に要素ＩＤをセット
                    m_csModelInfo.SetInitialInfo(csInitialDt);
                }

                m_csModelInfo.SetElementID((HySID)this.m_csID);
            }

            string sX = csXml.GetAttribute(McDefine.xml_X);
            if (sX != "")
            {
                m_lX = long.Parse(sX);
            }
            string sY = csXml.GetAttribute(McDefine.xml_Y);
            if (sY != "")
            {
                m_lY = long.Parse(sY);
            }
            string sRot = csXml.GetAttribute(McDefine.xml_ROTATION);
            if (sRot != "")
            {
                m_lRotate = long.Parse(sRot);
            }
            string sWidth = csXml.GetAttribute(McDefine.xml_Width);
            if (sWidth != "")
            {
                m_lWidth = long.Parse(sWidth);
            }
            string sHeight = csXml.GetAttribute(McDefine.xml_Height);
            if (sHeight != "") 
            {
                m_lHeight = long.Parse(sHeight);
            }
            string sElementName = csXml.GetAttribute(McDefine.xml_ElementName);
            if (sElementName != "")
            {
                m_sElementName = new HySString(sElementName);
            }

            // プロパティ解読
            HySXmlNode csXmlPropertyNode;
            HySXmlNodeList csXmlPropertyList = csXml.GetChildNodeListByTagName(McDefine.xml_PROPERTY);
            long lPropertyNum = csXmlPropertyList.GetCount(); // 通常は１個
            for (long lLp = 0; lLp < lPropertyNum; lLp++)
            {
                csXmlPropertyNode = csXmlPropertyList.GetNode(lLp);
                // プロパティ情報ノード読み出し

                // プロパティタグ内容を読み出す
                McPropertyInfoRoot csPrpytInf = m_csModelInfo.GetPropertyInfo();
                if (csPrpytInf != null)
                {   // プロパティ情報有り
                    if (csXmlPropertyNode != null)
                    {   // XML情報あり
                        csPrpytInf.XmlIN(csXmlPropertyNode);
                    }
                }

                //this.ReadPropertyConfiguration(csXmlPropertyNode.GetChildNodeList(), csMdlFctry);
            }
            // 初期化情報解読
            HySXmlNode csXmlInitalNode;
            HySXmlNodeList csXmlInitialList = csXml.GetChildNodeListByTagName(McDefine.xml_INITIAL_SET);
            long lInitialNum = csXmlInitialList.GetCount(); // 通常は１個
            for (long lLp = 0; lLp < lInitialNum; lLp++)
            {
                csXmlInitalNode = csXmlInitialList.GetNode(lLp);
                // 初期情報ノード読み出し

                // プロパティタグ内容を読み出す
                McPropertyInfoRoot csInitInf = m_csModelInfo.GetInitialInfo();
                if (csInitInf != null)
                {   // プロパティ情報有り
                    if (csXmlInitalNode != null)
                    {   // XML情報あり
                        csInitInf.XmlIN(csXmlInitalNode);
                    }
                }

                //this.ReadPropertyConfiguration(csXmlPropertyNode.GetChildNodeList(), csMdlFctry);
            }

            // 端子解読
            HySXmlNode csXmlPortNode;
            HySXmlNodeList csXmlPortList = csXml.GetChildNodeListByTagName(McDefine.xml_PORT);
            long lPortNum = csXmlPortList.GetCount(); // 通常は１個
            for (long lLp = 0; lLp < lPortNum; lLp++)
            {
                csXmlPortNode = csXmlPortList.GetNode(lLp);
                this.ReadPortConfiguration(csXmlPortNode.GetChildNodeList(), csMdlFctry);
            }

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>端子構造定義ＸＭＬ入力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReadPortConfiguration( csXmlList,csMdlFctry )</para>
        /// </example>
        /// <param name="csXmlList">入力ＸＭＬ</param>
        /// <param name="csMdlFctry">モデルファクトリクラス</param>
        /// <returns>==true:正常、==false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual bool ReadPortConfiguration(HySXmlNodeList csXmlList , McModelFactorySet csMdlFctry)
        {
            bool bRtn = true;
            if (csXmlList == null)
            {
                return false;
            }

            bool bLocalRtn = false;
            string sWorkAttr;

            HySXmlNode csXml;
            McStructPort csStructPort;
            long lLstNum = csXmlList.GetCount();
            for (long lLp = 0; lLp < lLstNum; lLp++)
            {
                csXml = csXmlList.GetNode(lLp);

                csStructPort = null;
                //if( McDefine.xml_INPUT_PORT.Equals(csXml.GetName()) == true )
                //{
                //    csStructPort = this.CreateNewStructPort(McDefine.SymbolDrawKind.HYM_INPUT);
                //}
                //else if (McDefine.xml_OUTPUT_PORT.Equals(csXml.GetName()) == true)
                //{
                //    csStructPort = this.CreateNewStructPort(McDefine.SymbolDrawKind.HYM_OUTPUT);
                //}
                //else if (McDefine.xml_RECEIVE_PORT.Equals(csXml.GetName()) == true)
                if (McDefine.xml_RECEIVE_PORT.Equals(csXml.GetName()) == true)
                {
                    csStructPort = this.CreateNewStructPort(McDefine.SymbolDrawKind.HYM_RECEIVE);
                }
                else if (McDefine.xml_SEND_PORT.Equals(csXml.GetName()) == true)
                {
                    csStructPort = this.CreateNewStructPort(McDefine.SymbolDrawKind.HYM_SEND);
                }
                else if (McDefine.xml_RECEIVE_TERMINAL.Equals(csXml.GetName()) == true)
                {
                    csStructPort = this.CreateNewStructPort(McDefine.SymbolDrawKind.HYM_TRANS_IN);
                }
                else if (McDefine.xml_SEND_TERMINAL.Equals(csXml.GetName()) == true)
                {
                    csStructPort = this.CreateNewStructPort(McDefine.SymbolDrawKind.HYM_TRANS_OUT);
                }
                else
                {
                    // Do Nothing
                }

                if (csStructPort != null)
                {
                    sWorkAttr = csXml.GetAttribute(McDefine.xml_ID);
                    if (sWorkAttr != null)
                    {
                        csStructPort.SetID(new HySID(sWorkAttr));
                    }
                    else
                    {
                        csStructPort.SetID(new HySID("dmy"));
                    }

                    bLocalRtn = csStructPort.ReadStructureBody(csXml , csMdlFctry);

                    this.AddPort(csStructPort);
                    // 端子の積み込み
                }
                
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>構造定義ファイル出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WriteStructureConfiguration( ref csXmlWriter )</para>
        /// </example>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WriteStructureConfiguration(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtnNode = this.WriteStructureHeader(ref csXmlWriter);

            HySXmlNode csNode = null;
            // プロパティ情報ノード書き込み
            McPropertyInfoRoot csPrpytInf = m_csModelInfo.GetPropertyInfo();
            if (csPrpytInf != null)
            {   // プロパティ情報有り
                //if (McPropertyInfoRoot.GetXmlOutFlg() == true)
                //if (McPropertyInfoBase.GetXmlOutFlg() == true)
                if (McPropertyInfoBase.GetXmlOutFlg() == McDefine.HYMCO_MODEL_STRUCTINFO_WRITE_ALL)
                {
                    csNode =csPrpytInf.XmlOUT(ref csXmlWriter);
                    if (csNode != null)
                    {
                        csRtnNode.AppendChild(csNode);
                    }
                }
            }
            McPropertyInfoRoot csInitialInf = m_csModelInfo.GetInitialInfo();
            if (csInitialInf != null)
            {
                //if (McPropertyInfoRoot.GetXmlOutFlg() == true)
                //if (McPropertyInfoBase.GetXmlOutFlg() == true)
                if (McPropertyInfoBase.GetXmlOutFlg() == McDefine.HYMCO_MODEL_STRUCTINFO_WRITE_ALL)
                {
                    csNode = csInitialInf.XmlOUT(ref csXmlWriter);
                    if (csNode != null)
                    {
                        csRtnNode.AppendChild(csNode);
                    } 
                }
            }

            //csNode = HySXmlNode.CreateElement(McDefine.xml_PROPERTY, csXmlWriter);
            //if (csNode != null)
            //{
            //    csRtnNode.AppendChild(csNode);
            //}
            // 内部情報書き込み
            csNode = this.WriteStructureBody(csRtnNode, ref csXmlWriter);
            if (csNode != null)
            {
                csRtnNode.AppendChild(csNode);
            }
            return csRtnNode;
        }

        /// <summary><para>method outline:</para>
        /// <para>構造定義ファイルヘッダー部出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WriteStructureHeader( ref csXmlWriter )</para>
        /// </example>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WriteStructureHeader(ref HySXmlWriter csXmlWriter)
        {
            // CalElementタグ
            HySXmlNode csRtnNode = HySXmlNode.CreateElement(McDefine.xml_CALELEMENT, csXmlWriter);
            csRtnNode.SetAttribute(McDefine.xml_ID, (this.GetID().GetString()).ToString());
            csRtnNode.SetAttribute(McDefine.xml_MODEL, ((m_csModelInfo.GetModelKind()).GetString()).ToString());
            csRtnNode.SetAttribute(McDefine.xml_FACTORYID, ((m_csModelInfo.GetLibraryID()).GetString()).ToString());
            csRtnNode.SetAttribute(McDefine.xml_KIND, (m_csDrawSubKind.GetString()).ToString());
            csRtnNode.SetAttribute(McDefine.xml_X, m_lX.ToString());
            csRtnNode.SetAttribute(McDefine.xml_Y, m_lY.ToString());
            csRtnNode.SetAttribute(McDefine.xml_ROTATION, m_lRotate.ToString());
            csRtnNode.SetAttribute(McDefine.xml_Width, m_lWidth.ToString());
            csRtnNode.SetAttribute(McDefine.xml_Height, m_lHeight.ToString());
            if ((object)m_sElementName != null)
            {
                csRtnNode.SetAttribute(McDefine.xml_ElementName, m_sElementName.ToString());
            }
            else
            {
                csRtnNode.SetAttribute(McDefine.xml_ElementName, "");
            }
            return csRtnNode;
        }
        /// <summary><para>method outline:</para>
        /// <para>構造定義ファイル本体部出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WriteStructureBody( csInNode, ref csXmlWriter )</para>
        /// </example>
        /// <param name="csInNode">XMLノードクラス</param>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WriteStructureBody(HySXmlNode csInNode, ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtnNode = null;
            HySXmlNode csNode = null;

            long lPortNum = m_csPortList.GetCount();
            if (lPortNum > 0)
            {
                McStructPort csPort = null;
                // Portタグ
                csRtnNode = HySXmlNode.CreateElement(McDefine.xml_PORT, csXmlWriter);

                m_csPortList.SetCursorFirst();
                for (long lLp = 0; lLp < lPortNum; lLp++)
                {
                    csPort = (McStructPort)m_csPortList.GetCursorData();
                    csNode = csPort.WriteStructureBody(ref csXmlWriter);
                    csRtnNode.AppendChild(csNode);
                    m_csPortList.MoveCursorNext();
                }
            }

            return csRtnNode;
        }

                
        /// <summary><para>method outline:</para>
        /// <para>プロパティ定義ファイル出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WritePropertyConfiguration( ref csXmlWriter )</para>
        /// </example>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WritePropertyConfiguration(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtnNode = this.WritePropertyHeader(ref csXmlWriter);

            HySXmlNode csNode = null;
            // プロパティ情報ノード書き込み
            McPropertyInfoRoot csPrpytInf = m_csModelInfo.GetPropertyInfo();
            if (csPrpytInf != null)
            {   // プロパティ情報有り
                csNode = csPrpytInf.XmlOUT(ref csXmlWriter);
                if (csNode != null)
                {
                    csRtnNode.AppendChild(csNode);
                }
            }
            // 内部情報書き込み
            csNode = this.WritePropertyBody(csRtnNode, ref csXmlWriter);
            if (csNode != null)
            {
                csRtnNode.AppendChild(csNode);
            }
            return csRtnNode;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ定義ファイルヘッダー部出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WritePropertyHeader( ref csXmlWriter )</para>
        /// </example>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WritePropertyHeader(ref HySXmlWriter csXmlWriter)
        {
            // CalElementタグ
            HySXmlNode csRtnNode = HySXmlNode.CreateElement(McDefine.xml_CALELEMENT, csXmlWriter);
            csRtnNode.SetAttribute(McDefine.xml_ID, (this.GetID().GetString()).ToString());
            return csRtnNode;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ定義ファイル本体部出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WritePropertyBody( csInNode, ref csXmlWriter )</para>
        /// </example>
        /// <param name="csInNode">XMLノードクラス</param>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WritePropertyBody(HySXmlNode csInNode, ref HySXmlWriter csXmlWriter)
        {
            // Do Nothing
            return null;
        }
        /// <summary><para>method outline:</para>
        /// <para>プロパティ定義ファイル入力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReadPropertyConfiguration( csXml,  csMdlFctry )</para>
        /// </example>
        /// <param name="csXml">Xmlノード</param>
        /// <param name="csMdlFctry">モデルファクトリクラス</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool ReadPropertyConfiguration(HySXmlNode csXml, McModelFactorySet csMdlFctry)
        {
            bool bRtn = false;
            // プロパティ解読
            HySXmlNode csXmlPropertyNode;
            HySXmlNodeList csXmlPropertyList = csXml.GetChildNodeListByTagName(McDefine.xml_PROPERTY);
            long lPropertyNum = csXmlPropertyList.GetCount(); // 通常は１個
            for (long lLp = 0; lLp < lPropertyNum; lLp++)
            {
                csXmlPropertyNode = csXmlPropertyList.GetNode(lLp);
                // プロパティ情報ノード読み出し

                // プロパティタグ内容を読み出す
                McPropertyInfoRoot csPrpytInf = m_csModelInfo.GetPropertyInfo();
                if (csPrpytInf != null)
                {   // プロパティ情報有り
                    if (csXmlPropertyNode != null)
                    {   // XML情報あり
                        csPrpytInf.XmlIN(csXmlPropertyNode);
                        bRtn = true;
                    }
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>初期情報定義ファイル出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WriteInitialConfiguration( ref csXmlWriter )</para>
        /// </example>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WriteInitialConfiguration(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtnNode = this.WriteInitialHeader(ref csXmlWriter);

            HySXmlNode csNode = null;

            McPropertyInfoRoot csInitialInf = m_csModelInfo.GetInitialInfo();
            if (csInitialInf != null)
            {
                csNode = csInitialInf.XmlOUT(ref csXmlWriter);
                if (csNode != null)
                {
                    csRtnNode.AppendChild(csNode);
                }
            }
            // 内部情報書き込み
            csNode = this.WriteInitialBody(csRtnNode, ref csXmlWriter);
            if (csNode != null)
            {
                csRtnNode.AppendChild(csNode);
            }
            return csRtnNode;
        }
        /// <summary><para>method outline:</para>
        /// <para>初期情報定義ファイルヘッダー部出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WriteInitialHeader( ref csXmlWriter )</para>
        /// </example>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WriteInitialHeader(ref HySXmlWriter csXmlWriter)
        {
            // CalElementタグ
            HySXmlNode csRtnNode = HySXmlNode.CreateElement(McDefine.xml_CALELEMENT, csXmlWriter);
            csRtnNode.SetAttribute(McDefine.xml_ID, (this.GetID().GetString()).ToString());
            return csRtnNode;
        }
        /// <summary><para>method outline:</para>
        /// <para>初期情報定義ファイル本体部出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = WriteInitialBody( csInNode, ref csXmlWriter )</para>
        /// </example>
        /// <param name="csInNode">XMLノードクラス</param>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySXmlNode WriteInitialBody(HySXmlNode csInNode, ref HySXmlWriter csXmlWriter)
        {
            // Do Nothing
            return null;
        }
        /// <summary><para>method outline:</para>
        /// <para>初期情報定義ファイル入力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReadInitialConfiguration( csXml,  csMdlFctry )</para>
        /// </example>
        /// <param name="csXml">Xmlノード</param>
        /// <param name="csMdlFctry">モデルファクトリクラス</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool ReadInitialConfiguration(HySXmlNode csXml, McModelFactorySet csMdlFctry)
        {
            bool bRtn = true;
            // 初期化情報解読
            HySXmlNode csXmlInitalNode;
            HySXmlNodeList csXmlInitialList = csXml.GetChildNodeListByTagName(McDefine.xml_INITIAL_SET);
            long lInitialNum = csXmlInitialList.GetCount(); // 通常は１個
            for (long lLp = 0; lLp < lInitialNum; lLp++)
            {
                csXmlInitalNode = csXmlInitialList.GetNode(lLp);
                // 初期情報ノード読み出し

                // プロパティタグ内容を読み出す
                McPropertyInfoRoot csInitInf = m_csModelInfo.GetInitialInfo();
                if (csInitInf != null)
                {   // プロパティ情報有り
                    if (csXmlInitalNode != null)
                    {   // XML情報あり
                        csInitInf.XmlIN(csXmlInitalNode);
                        bRtn = true;
                    }
                }
            }
            return bRtn; // 必ずTrueを返すこと
        }

        /// <summary><para>method outline:</para>
        /// <para>要素新番号取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>ulong ulRtn = GetPrimNo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>要素新番号</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public ulong GetPrimNo()
        {
            return m_ulPrimElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>全削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = AllClear()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual bool AllClear()
        {
            McStructPort csPrt = null;
            long lPrt = m_csPortList.GetCount();
            m_csPortList.SetCursorFirst();
            for (long lLp = 0; lLp < lPrt; lLp++)
            {
                csPrt = (McStructPort)m_csPortList.GetCursorData();
                csPrt.Free();
                m_csPortList.MoveCursorNext();
            }

            m_csPortList.Clear();
            m_csModelInfo = new McModelInfo(new HySID(""), new HySObjectKind(""), new HySObjectKind(""), new HySString("")); // 仮設定 モデルを設定していなくても，構造定義ファイルを吐き出したときに死なない為;

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>情報のコピー</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>CopyInf(csOrgElm)</para>
        /// </example>
        /// <param name="csOrgElm">コピー元</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void CopyInf(McStructElement csOrgElm)
        {
            m_csPortList = csOrgElm.m_csPortList;
        }

        //======================================
        // 共通メソッド
        //======================================

        /// <summary><para>method outline:</para>
        /// <para>要素生成</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McStructElement newObj = CreateNewElement(eKind,csElementKind,csOwnerElm)</para>
        /// </example>
        /// <param name="eKind">要素種別</param>
        /// <param name="csElementKind">オブジェクト区分</param>
        /// <param name="csOwnerElm">生成した要素を所有するグループ要素</param>
        /// <returns>!=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public McStructElement CreateNewElement(McDefine.SymbolDrawKind eKind, HySObjectKind csElementKind,McStructGroupElement csOwnerElm )
        {

            McStructElement csNewElm = null;  // 画面表示要素
            //McModelInfo csGrModelInfo = null; // 要素のプロパティ
            if (eKind == McDefine.SymbolDrawKind.HYM_GROUP || eKind == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
            {   // グループ
                csNewElm = new McStructGroupElement();
                ((McStructGroupElement)csNewElm).SetSymbolDrawKind(eKind);

                // グルー化要素のプロパティを仮設定
                if (eKind == McDefine.SymbolDrawKind.HYM_GROUP)
                {
                    McModelInfo csMdlInf = s_csMdlFactorySet.GetModelInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, McModelLibraryDefine.HYM_MODEL_SUBSYSTEM);
                    csNewElm.SetModelInfo(csMdlInf.Clone());
                    /*
                    //csGrModelInfo = new McModelInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, new HySObjectKind(""),
                    //        McModelLibraryDefine.HYM_MODEL_SUBSYSTEM, McModelLibraryDefine.MODEL_NAME_SUBSYSTEM);
                    csGrModelInfo = new McModelInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, new HySObjectKind(""),
                            McModelLibraryDefine.HYM_MODEL_SUBSYSTEM, McModelLibraryDefine.MODEL_NAME_SUBSYSTEM,
                            McDefine.MODEL_TYPE_GROUP);
                    csGrModelInfo.SetPropertyInfo(
                        new McGroupPropertyInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, McModelLibraryDefine.HYM_MODEL_SUBSYSTEM)
                                                 );
                    csNewElm.SetModelInfo(csGrModelInfo);
                    */
                }
                //else if (eKind == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
                //{
                //    csGrModelInfo = new McModelInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, new HySObjectKind(""),
                //            McModelLibraryDefine.HYM_MODEL_CONVERGENCE_ITERATION, McModelLibraryDefine.MODEL_NAME_CONVERGENCE_ITERATION);
                //}
            }
            else
            {   // グループ以外
                csNewElm = new McStructElement(eKind);

                // 要素のプロパティを仮設定
                if (csElementKind.Equals(McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE) == true)
                {   // 合流シンボルならば
                    McModelInfo csMdlInf = s_csMdlFactorySet.GetModelInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, McModelLibraryDefine.HYM_MODEL_DATA_MERGE);
                    csNewElm.SetModelInfo(csMdlInf.Clone());
                    /*
                    csGrModelInfo = new McModelInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, new HySObjectKind(""),
                            McModelLibraryDefine.HYM_MODEL_DATA_MERGE, McModelLibraryDefine.MODEL_NAME_DATA_MERGE);
                    csGrModelInfo.SetPropertyInfo(
                        new McModelPropertyInfo(McModelLibraryDefine.SYSTEM_MODEL_FACTORY, McModelLibraryDefine.HYM_MODEL_DATA_MERGE)
                                                 );
                    csNewElm.SetModelInfo(csGrModelInfo);
                    */
                }
            }
            csNewElm.SetDrawSubKind(csElementKind);
            csNewElm.SetOwnerElement(csOwnerElm);
            return csNewElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>内部要素数</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HowManyElement() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 要素数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        public virtual long HowManyElement()
        {
            return 1;
        }

        /// <summary><para>method outline:</para>
        /// <para>上流接続線リスト作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetUpperConnectionInfoList(  )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataLinkedList 上流接続線のリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>リスト内には McModelInfo が入っている。プロパティは　csTrnInfo.GetPropertyInfo();で取り出す必要があるが、モデル未設定等によりnull の場合もある</para>
        /// </remarks>
        public virtual HySDataLinkedList GetUpperConnectionInfoList()
        {
            HySDataLinkedList csRtn = new HySDataLinkedList();
            McStructConnection csVCnct = null;
            McModelInfo csTrnInfo = null;
            McStructPort csPort = null;

            HySIdentifier csFromPortID = new HySID("");
            HySIdentifier csToPortID = new HySID("");

            long lNum = m_csPortList.GetCount();
            m_csPortList.SetCursorFirst();
            for (int iLp = 0; iLp < lNum; iLp++)
            {
                csPort = m_csPortList.GetCursorData() as McStructPort;
                csPort.GetPortID(ref csFromPortID, ref csToPortID);
                if (csToPortID.Equals(csPort.GetID()) == true)
                {   // 受信端子の場合
                    csVCnct = m_csOwnerGrElm.QueryConnectionByPortID(csToPortID);
                    if (csVCnct != null)
                    {
                        csTrnInfo = csVCnct.GetConnectionTranInfoOnly();
                        csRtn.AddLast(csTrnInfo);
                    }
                }
                m_csPortList.MoveCursorNext();
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>下流接続線リスト作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetLowerConnectionInfoList( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataLinkedList 下流接続線のリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>リスト内には McModelInfo が入っている。プロパティは　csTrnInfo.GetPropertyInfo();で取り出す必要があるが、モデル未設定等によりnull の場合もある</para>
        /// </remarks>
        public virtual HySDataLinkedList GetLowerConnectionInfoList()
        {
            HySDataLinkedList csRtn = new HySDataLinkedList();
            McStructConnection csVCnct = null;
            McModelInfo csTrnInfo = null;
            McStructPort csPort = null;

            HySIdentifier csFromPortID = new HySID("");
            HySIdentifier csToPortID = new HySID("");

            long lNum = m_csPortList.GetCount();
            m_csPortList.SetCursorFirst();
            for (int iLp = 0; iLp < lNum; iLp++)
            {
                csPort = m_csPortList.GetCursorData() as McStructPort;
                csPort.GetPortID(ref csFromPortID, ref csToPortID);
                if (csFromPortID.Equals(csPort.GetID()) == true)
                {   // 送信端子の場合
                    csVCnct = m_csOwnerGrElm.QueryConnectionByPortID(csFromPortID);
                    if (csVCnct != null)
                    {
                        csTrnInfo = csVCnct.GetConnectionTranInfoOnly();
                        csRtn.AddLast(csTrnInfo);
                    } 
                }
                m_csPortList.MoveCursorNext();
            }
            return csRtn;
        }

//------------
// start of レコード数自動調整の為の処理を追加
//  Ver1.2
//------------
        /// <summary><para>method outline:</para>
        /// <para>下位に接続されている要素のリストを取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetLowerTerminalElement(ref csElmList)</para>
        /// </example>
        /// <param name="csElmList">下位に接続されている要素のリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>接続先がGr化要素ならばその中まで探る</para>
        /// </remarks>
        internal virtual void GetLowerElementList(ref HySDataLinkedList csElmList)
        {
            McStructElement csNextElm = null;
            McStructConnection csVCnct = null;
            HySDataLinkedList csCnctionList = this.GetLowerConnectionList();
            long lCnctNo = csCnctionList.GetCount();
            csCnctionList.SetCursorFirst();
            for (long lLp = 0; lLp < lCnctNo; lLp++)
            {
                csVCnct = csCnctionList.GetCursorData() as McStructConnection;
                if (csVCnct != null)
                {
                    csNextElm = csVCnct.GetLowerTerminalElement();
                    if (csNextElm != null)
                    {
                        csElmList.AddLast(csNextElm);
                        //csNextElm.GetLowerElementList(ref csElmList);
                    }
                }
                csCnctionList.MoveCursorNext();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>自要素から下位へ接続されている接続線のリストを取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySDataLinkedList csCnctList = GetLowerConnectionList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataLinkedList　下位へ接続されている接続線のリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        internal virtual HySDataLinkedList GetLowerConnectionList()
        {
            HySDataLinkedList csRtn = new HySDataLinkedList();
            McStructConnection csVCnct = null;
            McStructPort csPort = null;

            HySIdentifier csFromPortID = new HySID("");
            HySIdentifier csToPortID = new HySID("");

            long lNum = m_csPortList.GetCount();// ポート数取得
            m_csPortList.SetCursorFirst();
            for (int iLp = 0; iLp < lNum; iLp++)
            {   // ポート数繰り返し
                csPort = m_csPortList.GetCursorData() as McStructPort;
                csPort.GetPortID(ref csFromPortID, ref csToPortID);
                if (csFromPortID.Equals(csPort.GetID()) == true)
                {   // ポートが送信端子の場合その接続線は　下位へ接続される
                    csVCnct = m_csOwnerGrElm.QueryConnectionByPortID(csFromPortID);
                    if (csVCnct != null)
                    {
                        csRtn.AddLast(csVCnct);
                    }
                }
                m_csPortList.MoveCursorNext();
            }
            return csRtn;
        }
//------------
// end  of レコード数自動調整の為の処理を追加
//  Ver1.2
//------------

        /// <summary><para>method outline:</para>
        /// <para>内部に含まれる部品のIDを変更する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ChangeID() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>接続線、ポートのID含む</para>
        /// </remarks>
        public virtual HySID ChangeID()
        {
            // 自身のID設定
            McStructElement.m_ulPrimElm += 1;
            McStructElement.m_ulTotalElmInstanceNum += 1;

            string sStr = DateTime.Now.Minute.ToString() + DateTime.Now.Hour.ToString()
                            + "-" + DateTime.Now.Day.ToString() + DateTime.Now.Month.ToString() + DateTime.Now.Year.ToString();
            //m_csID = new HySID("E" + m_ulPrimElm.ToString() + DateTime.Now.Second.ToString() + "-",
            //                    "Prj", sStr);

            //m_csID = new HySID("E" + m_ulPrimElm.ToString() + DateTime.Now.Second.ToString() + "-" + "Prj" + sStr);
            if (m_eDrawKind == McDefine.SymbolDrawKind.HYM_GROUP || m_eDrawKind == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
            {
                m_csID = new HySID("G" + m_ulPrimElm.ToString() + DateTime.Now.Second.ToString() + "-" + "Prj" + sStr);
            }
            else
            {
                m_csID = new HySID("E" + m_ulPrimElm.ToString() + DateTime.Now.Second.ToString() + "-" + "Prj" + sStr);
            }

            return m_csID as HySID;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルファクトリーの設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetModelFactorySet(csModelFactorySet)</para>
        /// </example>
        /// <param name="csModelFactorySet">モデルファクトリ集合体</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public void SetModelFactorySet(McModelFactorySet csModelFactorySet)
        {
            s_csMdlFactorySet = csModelFactorySet;
        }

 

        //======================================
        // 表示情報関連メソッド
        //======================================
        /// <summary><para>method outline:</para>
        /// <para>描画位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetPosition( lX, lY )</para>
        /// </example>
        /// <param name="lX">描画位置(X)</param>
        /// <param name="lY">描画位置(Y)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetPosition(long lX, long lY)
        {
            m_lX = lX;
            m_lY = lY;
        }
        /// <summary><para>method outline:</para>
        /// <para>描画位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetPosition( ref lX, ref lY )</para>
        /// </example>
        /// <param name="lX">描画位置(X)</param>
        /// <param name="lY">描画位置(Y)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void GetPosition(ref long lX, ref long lY)
        {
            lX = m_lX;
            lY = m_lY;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画位置設定(回転)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetRotate( dRotate )</para>
        /// </example>
        /// <param name="dRotate">描画位置(回転)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetRotate(double dRotate)
        {
            m_lRotate = dRotate;
        }
        /// <summary><para>method outline:</para>
        /// <para>描画位置取得(回転)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double dRtn = GetRotate()</para>
        /// </example>
        /// <returns>>描画位置(回転)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual double GetRotate()
        {
            return m_lRotate;
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントサイズ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetElementSize(lWidth, lHeight)</para>
        /// </example>
        /// <param name="lWidth">幅</param>
        /// <param name="lHeight">高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetElementSize(long lWidth, long lHeight)
        {
            m_lWidth = lWidth;
            m_lHeight = lHeight;
        }
        /// <summary><para>method outline:</para>
        /// <para>エレメントサイズ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetElementSize(ref lWidth, ref lHeight)</para>
        /// </example>
        /// <param name="lWidth">幅</param>
        /// <param name="lHeight">高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void GetElementSize(ref long lWidth, ref long lHeight)
        {
            lWidth = m_lWidth;
            lHeight = m_lHeight;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画オブジェクト区分設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDrawSubKind( csDrawSubKind )</para>
        /// </example>
        /// <param name="csDrawSubKind">描画オブジェクト区分</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDrawSubKind(HySObjectKind csDrawSubKind)
        {
            m_csDrawSubKind = csDrawSubKind;
        }
        /// <summary><para>method outline:</para>
        /// <para>描画オブジェクト区分取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySObjectKind csObjectKind = GetDrawSubKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySObjectKind　描画オブジェクト区分</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySObjectKind GetDrawSubKind()
        {
            return m_csDrawSubKind;
        }

        // ================================================
        //   Undo/Redo用処理(画面からの操作)
        // ================================================
        /// <summary><para>method outline:</para>
        /// <para>端子追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = AddPortUsedByUndoRedoOnly( csPort )</para>
        /// </example>
        /// <param name="csPort">端子要素</param>
        /// <returns>long 0:成功，-1:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>GUIでのUndo/Redoのみ使用、その他は使用禁止</para>
        /// </remarks>
        public long AddPortUsedByUndoRedoOnly(McStructPort csPort)
        {
            if (csPort == null)
            {// パラメータ異常
                return -1L;
            }

            csPort.SetElmID(m_csID);
            m_csPortList.AddLast(csPort);

            return 0L;
        }

        /// <summary><para>method outline:</para>
        /// <para>情報のコピー</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructElement csStructElement = CopyInfUsedByUndoRedoOnly(csCopyElement)</para>
        /// </example>
        /// <param name="csCopyElement">複製要素</param>
        /// <returns>McStructElement 複製要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>GUIでのUndo/Redoのみ使用、その他は使用禁止</para>
        /// </remarks>
        public virtual McStructElement CopyInfUsedByUndoRedoOnly(McStructElement csCopyElement)
        {
            if (csCopyElement == null)
            {
                csCopyElement = new McStructElement(m_eDrawKind);
            }

            McStructElement.m_ulPrimElm -= 1;
            McStructElement.m_ulTotalElmInstanceNum -= 1;

            csCopyElement.m_csID = m_csID;
            csCopyElement.m_csModelInfo = m_csModelInfo.Clone();
            csCopyElement.m_csDrawSubKind = m_csDrawSubKind;
            csCopyElement.m_lX = m_lX;
            csCopyElement.m_lY = m_lY;
            csCopyElement.m_lWidth = m_lWidth;
            csCopyElement.m_lHeight = m_lHeight;
            csCopyElement.m_lRotate = m_lRotate;
            csCopyElement.m_sElementName = m_sElementName;
            csCopyElement.m_csOwnerGrElm = m_csOwnerGrElm;

            for (long lLp = 0; lLp < m_csPortList.GetCount(); lLp++)
            {
                csCopyElement.m_csPortList.AddLast(((McStructPort)m_csPortList.GetData(lLp)).CopyInfUsedByUndoRedoOnly(null));
            }

            return csCopyElement;
        }


        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            // ﾒﾝﾊﾞｰ：McPropertyInfo内のシリアライズ対象外のインスタンスを設定する
            if (m_csModelInfo != null)
            {
                McPropertyInfo csMdlPrpty = m_csModelInfo.GetPropertyInfo() as McPropertyInfo;
                if (csMdlPrpty != null)
                {
                    csMdlPrpty.SetStructElement(this);
                }
            }


            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();//復元後の整合性確保処理
        }
    }
}
