// <summary>ソースコード：受信したセル型情報を管理するクラス</summary>
// <author>CommonMP</author>
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Data;


namespace CommonMP.HYMCO.CoreImpl.Data
{
    /// <summary><para>class outline:</para>
    /// <para>受信したセル型情報を取得するクラス</para>
    /// </summary>
    /// <remarks>
    /// <para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/08/01][新規作成]</para>
    /// <para>remarks:</para>
    /// <para>接続線毎に管理する</para>
    /// </remarks>
    [Serializable]
    public class McReceiveCellData : McReceiveCellDataIF, HySVersionManagement
    {
        /// <summary>インスタンス識別ＩＤ</summary>
        protected HySID m_csID=null;
        /// <summary>伝送データの種別 </summary>
        protected HySObjectKind m_csTranDataKind = null;

        /// <summary> 受信情報 </summary>
        [NonSerialized]
        protected McTranInfo m_csTranInfo = null;
        /// <summary> セル型キャスト：受信情報 </summary>
        [NonSerialized]
        protected McTranInfoIFCellType m_csCellTrnInfo = null;
        /// <summary> 時系列型キャスト：受信情報 </summary>
        [NonSerialized]
        protected McTranInfoIFTimeSeriesType m_csTimeTranInfo = null;
        /// <summary> ＧＩＳ非時系列キャスト：受信情報 </summary>
        [NonSerialized]
        protected HySGeoMesh m_csGISRecordInfo = null;
        /// <summary> ＧＩＳ時系列型キャスト：受信情報 </summary>
        [NonSerialized]
        protected HySGeoMeshSerial m_csGISTimeSeriesInfo = null;
        /// <summary> セル配列の次元 </summary>
        protected long m_lCellDim = 0;
        
        /// <summary>セル内データを取得するクラスを管理するクラス </summary>
        protected HySCellDataGetter m_csCellDataGetter = null;
        //// <summary>内挿方法式 </summary>
        //protected HySDefine.InterpolateType m_eInterpolateType = HySDefine.InterpolateType.LINEAR;
        /// <summary>内挿処理クラス </summary>
        protected HySInterpolatorIF m_csInterpolator = null;

        /// <summary>セル識別子とセルの配列管理 </summary>
        protected McCellIDMngData m_csCellIDMng = null;

        /// <summary>伝送情報パターンプロパティ </summary>
        protected McTranInfoPattern m_csTrnPattern = null;

        // 処理高速化のため、ローカル変数として　new されることを防ぐ為メンバーとする
        /// <summary>ワーク用変数：配列１ </summary>
        protected long work_lDim1 = 0;
        /// <summary>ワーク用変数：配列２ </summary>
        protected long work_lDim2 = 0;
        /// <summary>ワーク用変数：配列３ </summary>
        protected long work_lDim3 = 0;
        /// <summary>ワーク用変数：データ種別 </summary>
        protected HySObjectKind work_csKnd = new HySObjectKind("");

             
        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McReceiveCellData csCellData = new McReceiveCellData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McTranInfoPattern 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McReceiveCellData()
        {
            //バージョン情報設定
            SetNowVersion();
        } 

        /// <summary><para>method outline:</para>
        /// <para>識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetID( csID ) </para>
        /// </example>
        /// <param name="csID">識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public void SetID(HySID csID)
        {
            m_csID = csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySIdentifier csIdentifier = GetID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public HySID GetID()
        {
            return m_csID;
        }

        //==========================================
        //  セルから情報を取り出す処理を行う場合
        //==========================================
        /// <summary><para>method outline:</para>
        /// <para>セル内データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double Data(csCell, lDataIdx); </para>
        /// </example>
        /// <param name="csCell">対象セル</param>
        /// <param name="lDataIdx">データインデックス</param>
        /// <returns> double データ値</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// シミュレーション演算中に使用する
        /// 速度アップの為内部でのデータ整合性チェック等は行っていない。
        /// 使用者側の責任において使用すること。
        /// </para>
        /// </remarks>
        public virtual double Data(HySCellData csCell, long lDataIdx)
        {
            return m_csCellDataGetter.Data(csCell, lDataIdx);
        }

        //==========================================
        //  時系列情報の場合の処理用メソッド
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>セル配列の次元</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDim = GetCellDimention() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> long セル配列の次元（１、２，３次元）　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetCellDimention()
        {
            return m_lCellDim;
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCurrentTime(csTime) </para>
        /// </example>
        /// <param name="csTime">内挿すべき時刻</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>設定した時刻に内挿する</para>
        /// </remarks>
        public virtual void SetCurrentTime(HySTime csTime)
        {
            if (m_csInterpolator != null)
            {
                m_csInterpolator.SetCurrentTime(csTime);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>最終時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLastTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>保持しているデータの最終の時刻に設定する</para>
        /// </remarks>
        public virtual void SetLastTime()
        {
            this.SetCurrentTime(this.GetLastTime());
        }
        //==========================================
        //  個別セルについて内挿処理を行う場合
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellData csCell = GetInterpolatedCell(lIdx1) </para>
        /// </example>
        /// <param name="lCellIdx1">第１配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// 時系列以外の時には　内挿処理は行わない
        /// </para>
        /// </remarks>
        public virtual HySCellData GetInterpolatedCell(long lCellIdx1)
        {
            if (m_csInterpolator != null)
            {
                return m_csInterpolator.GetCell(lCellIdx1);
            }
            else
            {
                if (m_csCellTrnInfo != null)
                {
                    return m_csCellTrnInfo.GetLastCellArrayData().GetCell(lCellIdx1);
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellData csCell = GetInterpolatedCell(lIdx1,lIdx2) </para>
        /// </example>
        /// <param name="lCellIdx1">第１配列のセル位置</param>
        /// <param name="lCellIdx2">第２配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// 時系列以外の時には　内挿処理は行わない
        /// </para>
        /// </remarks>
        public virtual HySCellData GetInterpolatedCell(long lCellIdx1, long lCellIdx2)
        {
            if (m_csInterpolator != null)
            {
                return m_csInterpolator.GetCell(lCellIdx1, lCellIdx2);
            }
            else
            {
                if (m_csCellTrnInfo != null)
                {
                    return m_csCellTrnInfo.GetLastCellArrayData().GetCell(lCellIdx1, lCellIdx2);
                }
                else
                {
                    return null;
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellData csCell = GetInterpolatedCell(lIdx1,lIdx2,lIdx3) </para>
        /// </example>
        /// <param name="lCellIdx1">第１配列のセル位置</param>
        /// <param name="lCellIdx2">第２配列のセル位置</param>
        /// <param name="lCellIdx3">第３配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// 時系列以外の時には　内挿処理は行わない
        /// </para>
        /// </remarks>
        public virtual HySCellData GetInterpolatedCell(long lCellIdx1, long lCellIdx2, long lCellIdx3)
        {
            if (m_csInterpolator != null)
            {
                return m_csInterpolator.GetCell(lCellIdx1, lCellIdx2, lCellIdx3);
            }
            else
            {
                if (m_csCellTrnInfo != null)
                {
                    return m_csCellTrnInfo.GetLastCellArrayData().GetCell(lCellIdx1, lCellIdx2, lCellIdx3);
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>ＩＤで指定したセルの内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellData csCell = GetInterpolatedCell(csCellID) </para>
        /// </example>
        /// <param name="csCellID">セルＩＤ</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内部で検索処理が走る為処理速度が遅くなる
        /// 速度重視の際は使用を奨励しない
        /// 時系列以外の時には　内挿処理は行わない
        /// </para>
        /// </remarks>
        public virtual HySCellData GetInterpolatedCell(HySID csCellID)
        {
            HySCellData csRtn = null;
            if (m_csCellIDMng != null)
            {
                work_lDim1 = work_lDim2 = work_lDim3 = -1;
                m_csCellIDMng.GetCellIdx( csCellID, ref work_lDim1, ref work_lDim2, ref work_lDim3, ref work_csKnd);
                if( m_lCellDim == 1 )
                {
                    csRtn = GetInterpolatedCell(work_lDim1);
                }
                else if( m_lCellDim == 2 )
                {
                    csRtn = GetInterpolatedCell(work_lDim1,work_lDim2);
                }
                else if( m_lCellDim == 3 )
                {
                    csRtn = GetInterpolatedCell(work_lDim1,work_lDim2,work_lDim3);
                }
            }
            return csRtn;
        }

        //==========================================
        //  セルの配列全体について内挿処理を行う場合
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>一括内挿結果取得（１次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetInterpolatedCellD1() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[] 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// 時系列以外の時には　内挿処理は行わない
        /// </para>
        /// </remarks>
        public virtual HySCellData[] GetInterpolatedCellD1()
        {
            HySCellData[] csRtn = null;
            if (m_csInterpolator != null)
            {
                csRtn = m_csInterpolator.GetCellD1();
            }
            else
            {
                if (m_csCellTrnInfo != null)
                {
                    HySD1CellArrayData csD1CellArr = m_csCellTrnInfo.GetLastCellArrayData() as HySD1CellArrayData;
                    if (csD1CellArr != null)
                    {
                        csRtn = csD1CellArr.GetCellData();
                    }
                }
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>一括内挿結果取得（２次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetInterpolatedCellD2() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[,] 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD2CellArrayData,HySGeoDim2MeshSerialDataの時に使用
        /// 時系列以外の時には　内挿処理は行わない
        /// </para>
        /// </remarks>
        public virtual HySCellData[,] GetInterpolatedCellD2()
        {
            HySCellData[,] csRtn = null;

            if (m_csInterpolator != null)
            {
                csRtn = m_csInterpolator.GetCellD2();
            }
            else
            {
                if (m_csCellTrnInfo != null)
                {
                    HySD2CellArrayData csD2CellArr = m_csCellTrnInfo.GetLastCellArrayData() as HySD2CellArrayData;
                    if (csD2CellArr != null)
                    {
                        csRtn = csD2CellArr.GetCellData();
                    }
                }
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>一括内挿結果取得（３次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetInterpolatedCellD3() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[,,] 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD3CellArrayData,HySGeoDim3MeshSerialDataの時に使用
        /// 時系列以外の時には　内挿処理は行わない
        /// </para>
        /// </remarks>
        public virtual HySCellData[, ,] GetInterpolatedCellD3()
        {
            HySCellData[,,] csRtn = null;
            if (m_csInterpolator != null)
            {
                csRtn = m_csInterpolator.GetCellD3();
            }
            else
            {
                if (m_csCellTrnInfo != null)
                {
                    HySD3CellArrayData csD3CellArr = m_csCellTrnInfo.GetLastCellArrayData() as HySD3CellArrayData;
                    if (csD3CellArr != null)
                    {
                        csRtn = csD3CellArr.GetCellData();
                    }
                }
            }
            return csRtn;
        }

        //★★★★★★★★★★★★★★★★
        // Ver1.4で追加(一時ファイル作成)
        //★★★★★★★★★★★★★★★★
        /// <summary><para>method outline:</para>
        /// <para>一時ファイル取得（2次元）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetFileOutCellD2(csTime) </para>
        /// </example>
        /// <param name="csTime">時刻</param>
        /// <returns> HySCellData[,] 一時ファイル出力した2次元セル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySCellData[,] GetFileOutCellD2(HySTime csTime)
        {
            HySCellData[,] csRtn = null;

            if (m_csCellTrnInfo != null)
            {
                McTimeSeriesD2CellArrayTranInfo csD2CellArr = m_csTranInfo as McTimeSeriesD2CellArrayTranInfo;
                if (csD2CellArr != null)
                {
                    csRtn = csD2CellArr.GetFileOutCellD2(csTime);
                }
            }

            return csRtn;
        }

        //==========================================
        //  各種部品設定
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>伝送データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTranInfo(csTranInfo) </para>
        /// </example>
        /// <param name="csTranInfo">伝送データ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetTranInfo(McTranInfo csTranInfo)
        {
            m_csTranInfo = csTranInfo;
            m_csGISRecordInfo = csTranInfo as HySGeoMesh;
            m_csGISTimeSeriesInfo = csTranInfo as HySGeoMeshSerial;

            m_csCellTrnInfo = csTranInfo as McTranInfoIFCellType;
            m_csTimeTranInfo = csTranInfo as McTranInfoIFTimeSeriesType;
            if (m_csCellTrnInfo != null)
            {
                //m_csCellDataGetter = m_csCellTrnInfo.GetCellDataGetter();
                m_csCellIDMng = m_csCellTrnInfo.GetCellIDMngData() as McCellIDMngData;  // 

                long lD1 = 0; long lD2 = 0; long lD3 = 0;
                m_lCellDim = m_csCellTrnInfo.GetDimension(ref lD1, ref lD2, ref lD3);
            }
            if (m_csTimeTranInfo != null)
            {
                //HySDefine.InterpolateType eInterpolateType = m_csTimeTranInfo.GetInterpolateType();
                //m_csInterpolator = HySInterpolatorFactory.CreateInterpolator(eInterpolateType);
                if (m_csInterpolator != null)
                {
                    m_csInterpolator.SetTimeSeriesData(m_csTimeTranInfo as HySTimeSeriesBase);
                }
                //m_csInterpolator.SetDataIndex(m_csCellDataGetter);
            }
            else
            {
                if (m_csInterpolator != null)
                {
                    m_csInterpolator.SetTimeRecordData(m_csTranInfo as HySTimeRecordIF);
                }
            }
            m_csTranDataKind = csTranInfo.GetTranInfoKind();
        }
        /// <summary><para>method outline:</para>
        /// <para>伝送データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McTranInfo csTranInfo = GetTranInfo() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>csTranInfo 伝送データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McTranInfo GetTranInfo()
        {
            return m_csTranInfo;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿方式設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetInterpolateType(eInterpolateType) </para>
        /// </example>
        /// <param name="eInterpolateType">内挿方式</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetInterpolateType(HySDefine.InterpolateType eInterpolateType)
        {
            m_csInterpolator = HySInterpolatorFactory.CreateInterpolator(eInterpolateType);
            if (m_csCellDataGetter != null)
            {
                m_csInterpolator.SetDataIndex(m_csCellDataGetter);
            }
            if (m_csTimeTranInfo != null)
            {
                m_csInterpolator.SetTimeSeriesData(m_csTimeTranInfo as HySTimeSeriesBase);
            }
            else
            {
                HySTimeRecordIF csTmeRcrd = m_csTranInfo as HySTimeRecordIF;
                if (csTmeRcrd != null)
                {
                    m_csInterpolator.SetTimeRecordData(csTmeRcrd);
                }
            }

        }
        /// <summary><para>method outline:</para>
        /// <para>セル内データ取得ツール設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCellDataGetter(csCellDataGetter) </para>
        /// </example>
        /// <param name="csCellDataGetter">セル内データ取得ツール</param>
        /// <returns> 無し　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCellDataGetter(HySCellDataGetter csCellDataGetter)
        { 
            m_csCellDataGetter = csCellDataGetter;
            if (m_csInterpolator != null)
            {
                m_csInterpolator.SetDataIndex(m_csCellDataGetter);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>セル内データ取得クラスの取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellDataGetter csCellGetter = GetCellDataGetter(); </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySCellDataGetter セル内データ取得クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySCellDataGetter GetCellDataGetter()
        {
            return m_csCellDataGetter;
        }

        //==========================================
        //  各種情報取得
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>セルメッシュ分割数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDimension(ref lDim1, ref lDim2, ref lDim3) </para>
        /// </example>
        /// <param name="lDim1">第１分割数</param>
        /// <param name="lDim2">第２分割数</param>
        /// <param name="lDim3">第３分割数</param>
        /// <param name="lDataDimInCell">セル内変数の数</param>
        /// <returns>long 次元数　1次元＝１、２次元＝２、３次元＝３が　返る</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetDimension(ref long lDim1, ref long lDim2, ref long lDim3, ref long lDataDimInCell)
        {
            if (m_csCellTrnInfo != null)
            {
                m_csCellTrnInfo.GetDimension(ref lDim1, ref lDim2, ref lDim3);
                if (m_csCellDataGetter != null)
                {
                    lDataDimInCell = m_csCellDataGetter.GetIdxNum();
                }
            }
            return m_lCellDim;
        }

        /// <summary><para>method outline:</para>
        /// <para>伝送データ種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csKind = GetTranInfoKind( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySObjectKind 伝送データ種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySObjectKind GetTranInfoKind()
        {
            if (m_csTranDataKind != null)
            {
                return m_csTranInfo.GetTranInfoKind();
            }
            else
            {
                return new HySObjectKind("");
            }
        }
        
        /// <summary><para>method outline:</para>
        /// <para>セル識別子とセルの配列管理データ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellIDManageData csCellIDMng = GetCellIDManager(); </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySCellIDManageData セル識別子とセルの配列管理データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySCellIDManageData GetCellIDData()
        {
            return m_csCellIDMng;
        }

        /// <summary><para>method outline:</para>
        /// <para>受信伝送情報パターンプロパティ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetReceiveTranInfoPattern(csPattern) </para>
        /// </example>
        /// <param name="csPattern">受信伝送情報パターンプロパティ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetReceiveTranInfoPattern(McTranInfoPatternPropertyIF csPattern)
        {
            m_csTrnPattern = csPattern as McTranInfoPattern;
        }
        /// <summary><para>method outline:</para>
        /// <para>受信伝送情報パターンプロパティ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McTranInfoPatternPropertyIF csPattern = GetReceiveTranInfoPattern( ) </para>
        /// </example>
        /// <param name=""></param> 
        /// <returns>McTranInfoPatternPropertyIF 受信伝送情報パターンプロパティ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McTranInfoPatternPropertyIF GetReceiveTranInfoPattern()
        {
            return m_csTrnPattern;
        }
        /// <summary><para>method outline:</para>
        /// <para>受信パターン識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csID = GetReceivePatternID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>受信パターン識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetReceivePatternID()
        {
            if (m_csTrnPattern != null)
            {
                return m_csTrnPattern.GetPatternID();
            }
            else
            {
                return new HySID("");
            }
        }

        //==========================================
        //  ＧＩＳ関連：各種情報取得
        //==========================================
        /// <summary><para>method outline:</para>
        /// <para>距離配列の取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double[] dDistance = GetCellDistance()</para> 
        /// <para>例：1番目のセル始点から、1番目のセル終点までの距離情報設定　dDistance[0] = x;</para>
        /// <para>例：1番目のセル始点から、2番目のセル終点までの距離情報設定　dDistance[1] = x;</para>
        /// <para>例：1番目のセル始点から、1番目のセル終点までの距離情報取得　y = dDistance[0];</para>
        /// <para>例：1番目のセル始点から、2番目のセル終点までの距離情報取得　y = dDistance[1];</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] 最初のセル始点からの距離配列 GIS1次元情報でない場合配列1つを返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public double[] GetCellDistance()
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetCellDistance();
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetCellDistance();
            }
            else
            {
                return new double[1];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セル始点・終点緯度情報配列の取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double[] dCellLat = GetCellLatitude()</para> 
        /// <para>例：1番目のセル始点の緯度情報設定　dCellLat[0] = x;</para>
        /// <para>例：1番目のセル終点の緯度情報設定　dCellLat[1] = x;</para>
        /// <para>dCellLat[1]は同時に2番目のセル始点となる</para>
        /// <para>例：1番目のセル始点の緯度情報取得　y = dCellLat[0];</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] セルの始点／終点の緯度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public double[] GetCellLatitude()
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetCellLatitude();
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetCellLatitude();
            }
            else
            {
                return new double[1];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セル始点・終点経度情報配列の取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double[] dCellLon = GetCellLongitude()</para> 
        /// <para>例：1番目のセル始点の経度情報設定　dCellLon[0] = x;</para>
        /// <para>例：1番目のセル終点の経度情報設定　dCellLon[1] = x;</para>
        /// <para>dCellLon[1]は同時に2番目のセル始点となる</para>
        /// <para>例：1番目のセル始点の経度情報取得　y = dCellLon[0];</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] セルの始点／終点の経度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public double[] GetCellLongitude()
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetCellLongitude();
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetCellLongitude();
            }
            else
            {
                return new double[1];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>代表点の緯度情報配列の取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double[] dSpecificLat = GetSpecificLatitude()</para> 
        /// <para>例：1番目のセルデータの緯度情報設定　dSpecificLat[0] = x;</para>
        /// <para>例：1番目のセルデータの緯度情報取得　y = dSpecificLat[0];</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] 代表点の緯度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public double[] GetSpecificLatitude()
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetSpecificLatitude();
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetSpecificLatitude();
            }
            else
            {
                return new double[1];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>代表点の経度情報配列の取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double[] dSpecificLon = GetSpecificLongitude()</para> 
        /// <para>例：1番目のセルデータの経度情報設定　dSpecificLon[0] = x;</para>
        /// <para>例：1番目のセルデータの経度情報取得　y = dSpecificLon[0];</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] 代表点の経度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public double[] GetSpecificLongitude()
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetSpecificLongitude();
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetSpecificLongitude();
            }
            else
            {
                return new double[1];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>ＧＩＳメッシュ分割数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDimension(ref lKind, ref lDim1, ref lDim2, ref lDim3, ref lDataDimInCell) </para>
        /// </example>
        /// <param name="lKind">保持するデータの形</param>
        /// <param name="lLonDim">経度方向のメッシュ分割数</param>
        /// <param name="lLatDim">緯度方向のメッシュ分割数</param>
        /// <param name="lAltDim">高度方向のメッシュ分割数</param>
        /// <param name="lCellDataDim">１メッシュ内のデータ配列数</param>
        /// <returns>
        /// long 次元数　1次元＝１、２次元＝２、３次元＝３が　返る。
        /// 　　　　　　　ＧＩＳ情報でない時には０が　返る</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetGISMeshDimension(ref HySGeoMesh.MESH_DATA_KIND lKind, ref long lLonDim, ref long lLatDim, ref long lAltDim, ref long lCellDataDim)
        {
            long lDim = 0;
            lLonDim = 0;
            lLatDim = 0;
            lAltDim = 0;
            lCellDataDim = 0;

            if (m_csGISTimeSeriesInfo != null)
            {
                lKind = m_csGISTimeSeriesInfo.GetDataMeshKind();

                if (m_csGISTimeSeriesInfo is McTimeSeriesGeoD2MeshTranInfo)
                {
                    lDim = ((McTimeSeriesGeoD2MeshTranInfo)m_csGISTimeSeriesInfo).GetDimension(ref  lLonDim, ref lLatDim, ref lAltDim);
                    lCellDataDim = ((McTimeSeriesGeoD2MeshTranInfo)m_csGISTimeSeriesInfo).GetDataDimentionInCell();
                }
                else if (m_csGISTimeSeriesInfo is McTimeSeriesGeoD3MeshTranInfo)
                {
                    lDim = ((McTimeSeriesGeoD3MeshTranInfo)m_csGISTimeSeriesInfo).GetDimension(ref  lLonDim, ref lLatDim, ref lAltDim);
                    lCellDataDim = ((McTimeSeriesGeoD3MeshTranInfo)m_csGISTimeSeriesInfo).GetDataDimentionInCell();
                }
            }
            else if (m_csGISRecordInfo != null)
            {
                lKind = m_csGISRecordInfo.GetDataMeshKind();
                if (m_csGISRecordInfo is McGeoDim2MeshTranInfo)
                {
                    lDim = ((McGeoDim2MeshTranInfo)m_csGISRecordInfo).GetDimension(ref  lLonDim, ref lLatDim, ref lAltDim);
                    lCellDataDim = ((McGeoDim2MeshTranInfo)m_csGISRecordInfo).GetDataDimentionInCell();
                }
                else if (m_csGISRecordInfo is McGeoDim3MeshTranInfo)
                {
                    lDim = ((McGeoDim3MeshTranInfo)m_csGISRecordInfo).GetDimension(ref  lLonDim, ref lLatDim, ref lAltDim);
                    lCellDataDim = ((McGeoDim3MeshTranInfo)m_csGISRecordInfo).GetDataDimentionInCell();
                }
            }
            return lDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>地理的位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetLocation(ref dSWLat,ref dSWLon,ref dNELat,ref dNELon) </para>
        /// </example>
        /// <param name="dSWLat">南西頂点緯度（度単位）</param>
        /// <param name="dSWLon">南西頂点経度（度単位）</param>
        /// <param name="dNELat">北東頂点緯度（度単位）</param>
        /// <param name="dNELon">北東頂点経度（度単位）</param> 
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool GetLocation(ref double dSWLat, ref double dSWLon, ref double dNELat, ref double dNELon)
        {
            bool bRtn = false;
            if (m_csGISTimeSeriesInfo != null)
            {
                m_csGISTimeSeriesInfo.GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
                bRtn = true;
            }
            else if (m_csGISRecordInfo != null)
            {
                m_csGISRecordInfo.GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
                bRtn = true;
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>高度位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HGetAltitude(ref dUpAlt, ref dLwAlt) </para>
        /// </example>
        /// <param name="dUpAlt">上限高度（ｍ単位）</param>
        /// <param name="dLwAlt">下限高度（ｍ単位）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool GetAltitude(ref double dUpAlt, ref double dLwAlt)
        {
            bool bRtn = false;
            if (m_csGISTimeSeriesInfo != null)
            {
                m_csGISTimeSeriesInfo.GetAltitude(ref dUpAlt, ref dLwAlt);
                bRtn = true;
            }
            else if (m_csGISRecordInfo != null)
            {
                m_csGISRecordInfo.GetAltitude(ref dUpAlt, ref dLwAlt);
                bRtn = true;
            }
            return bRtn;
        }


        // ========================================
        // その他
        // ========================================
        /// <summary><para>method outline:</para>
        /// <para>接続線名取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csName = GetConnectionName( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString 接続線名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>要素を接続している接続線の名称を取得する</para>
        /// </remarks>
        public virtual HySString GetConnectionName()
        {
            HySString csRtn = null;
            if (m_csTranInfo != null)
            {
                csRtn = m_csTranInfo.GetConnectionName();
            }
            else
            {
                csRtn = new HySString("");
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>上流要素識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySIdentifier csElmID = GetUpperElementID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySIdentifier 識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>取得した要素が自分を更新している</para>
        /// </remarks>
        public virtual HySID GetUpperElementID()
        {
            if (m_csTranInfo != null)
            {
                return m_csTranInfo.GetUpperElementID();
            }
            else
            {
                return new HySID("");
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>上位要素の送信パターン識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csID = GetUpperElmSendPatternID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID 上位要素の送信パターン識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetUpperElmSendPatternID()
        {
            HySID csRtn = null;
            if (m_csCellTrnInfo != null)
            {
                McSendCellDataIF csSndCell = m_csCellTrnInfo.GetSendCellData();
                if (csSndCell != null)
                {
                    McTranInfoPattern csSndPtn = csSndCell.GetTranInfoPattern() as McTranInfoPattern;
                    if (csSndPtn != null)
                    {
                        csRtn = csSndPtn.GetPatternID();
                    }
                }
            }

            if (csRtn == null)
            {
                csRtn = new HySID("");
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>最終設定時刻取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = GetTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>保持している情報の最新時刻を取得する</para>
        /// </remarks>
        public virtual HySTime GetLastTime()
        {
            if (m_csTranInfo != null)
            {
                return m_csTranInfo.GetLastTime();
            }
            else
            {
                return HySTime.DEFAULT_TIME.Clone();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>自分と同じクラスのインスタンスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McReceiveCellData csDt = Clone() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McReceiveCellData  生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McReceiveCellData Clone()
        {
            McReceiveCellData csRtn = new McReceiveCellData();
            if (m_csID != null)
            {
                csRtn.m_csID = m_csID.Clone();
            }
            csRtn.m_csTranInfo = m_csTranInfo; // クローン化の必要なし
            csRtn.m_csCellTrnInfo = m_csCellTrnInfo; // クローン化の必要なし
            csRtn.m_csTimeTranInfo = m_csTimeTranInfo; // クローン化の必要なし
            if (m_csTranDataKind != null)
            {
                csRtn.m_csTranDataKind = m_csTranDataKind.Clone();
            }
            csRtn.m_lCellDim = m_lCellDim;
            csRtn.m_csCellDataGetter = new HySCellDataGetter(m_csCellDataGetter);
            csRtn.m_csInterpolator = m_csInterpolator;

            if (m_csCellIDMng != null)
            {
                csRtn.m_csCellIDMng = m_csCellIDMng.Clone() as McCellIDMngData;
            }

            if (m_csTrnPattern != null)
            {
                csRtn.m_csTrnPattern = (McTranInfoPattern)m_csTrnPattern.Clone();
            }

            return csRtn;
        }

        /*
        /// <summary><para>method outline:</para>
        /// <para>管理情報をXMLへ書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySXmlNode csXmlNode = XmlOUT(csXmlWriter) </para>
        /// </example>
        /// <param name="csXmlWriter">XML書き出しクラス</param>
        /// <returns>HySXmlNode  XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySXmlNode XmlOUT(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtn = HySXmlNode.CreateElement(McDefine.xml_MODEL_CELL_GIS_TABLE, csXmlWriter);
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>XMLからセルと管理情報を読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = XmlIN(csXml) </para>
        /// </example>
        /// <param name="csXml">入力XML</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool XmlIN(HySXmlNode csXml)
        {
            return true;
        }
        */


        // =============================================
        // === バージョンアップ時の互換性を保つ為の処理 ===
        // =============================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
            //m_csCellTrnInfo = m_csTranInfo as McTranInfoIFCellType;
            //m_csTimeTranInfo = m_csTranInfo as McTranInfoIFTimeSeriesType; 
        }
    }
}
