// <summary>ソースコード：伝送データに送信する情報を設定するクラス</summary>
// <author>CommonMP</author>
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Data;


namespace CommonMP.HYMCO.CoreImpl.Data
{
    /// <summary><para>class outline:</para>
    /// <para>伝送データに送信する情報を設定するクラス</para>
    /// </summary>
    /// <remarks>
    /// <para>history:</para>
    /// <para>
    /// [CommonMP][ver 1.0.0][2009/08/01][新規作成]
    /// [CommonMP][ver 1.4.0][2013/12/06][修正]
    /// </para>
    /// <para>remarks:</para>
    /// <para>接続線毎に管理する</para>
    /// </remarks>
    [Serializable]
    public class McSendCellData : McSendCellDataIF, HySVersionManagement
    {
        // start of XML化不用
        
        /// <summary>インスタンス識別ＩＤ</summary>
        protected HySID m_csID=null;
        
        /// <summary>セル識別子とセルの配列管理 </summary>
        protected McCellIDMngData m_csCellIDMngData = new McCellIDMngData();

        /// <summary>伝送データの種別 </summary>
        protected HySObjectKind m_csTranDataKind = null;

        /// <summary>伝送情報パターンプロパティ </summary>
        protected McTranInfoPattern m_csTrnPattern = null;

        /// <summary>第１分割数</summary>
        protected long m_lDim1 = 1;
        /// <summary>第２分割数</summary>
        protected long m_lDim2 = 0;
        /// <summary>第３分割数</summary>
        protected long m_lDim3 = 0;

        /// <summary> 送信情報 </summary>
        protected McTranInfo m_csTranInfo = null;
        /// <summary> セル型キャスト：送信情報 </summary>
        protected McTranInfoIFCellType m_csCellTrnInfo = null;
        /// <summary> 時系列型キャスト：送信情報 </summary>
        protected McTranInfoIFTimeSeriesType m_csTimeTranInfo = null;
        /// <summary> セル配列の次元 </summary>
        protected long m_lCellDim = 0;

        /// <summary>ワーク用　：現在時刻設定 </summary>
        protected HySTime m_csCurrentTime = HySTime.DEFAULT_TIME.Clone();

        /// <summary>ワーク用　： 新規作成した送信レコード</summary>
        protected HySTimeRecordIF m_csRecored = null;
        /// <summary>ワーク用　： 送信レコードの新規作成フラグ＝true:新規作成あり</summary>
        protected bool bNewRecoreFlg = false;
        /// <summary>ワーク用　： 送信レコードのセル配列</summary>
        protected HySCellArray m_csCellArray = null;
        /// <summary>ワーク用　： m_csCellArray内に具体的セル配列（１次元）</summary>
        protected HySCellData[] m_CellD1 = null;
        /// <summary>ワーク用　： m_csCellArray内に具体的セル配列（２次元）</summary>
        protected HySCellData[,] m_CellD2 = null;
        /// <summary>ワーク用　： m_csCellArray内に具体的セル配列（３次元）</summary>
        protected HySCellData[, ,] m_CellD3 = null;
        //// <summary>ワーク用　： GISパターン配列（１次元）</summary>
        //protected long[] m_lGISPttnD1 = null;
        /// <summary>ワーク用　： GISパターン配列（２次元）</summary>
        protected long[,] m_lGISPttnD2 = null;
        /// <summary>ワーク用　： GISパターン配列（３次元）</summary>
        protected long[,,] m_lGISPttnD3 = null;

        // end of XML化不用


             
        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McSendCellData csCellData = new McSendCellData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McTranInfoPattern 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McSendCellData()
        {
            //バージョン情報設定
            SetNowVersion();
        } 

        /// <summary><para>method outline:</para>
        /// <para>識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetID( csID ) </para>
        /// </example>
        /// <param name="csID">識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public void SetID(HySID csID)
        {
            m_csID = csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySIdentifier csIdentifier = GetID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public HySID GetID()
        {
            return m_csID;
        }

        //==========================================
        //  送信時に必要な処理
        //==========================================
        /// <summary><para>method outline:</para>
        /// <para>送信時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCurrentTime(csTime) </para>
        /// </example>
        /// <param name="csTime">送信時刻</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>設定した時刻の情報を設定する</para>
        /// </remarks>
        public virtual void SetCurrentTime(HySTime csTime)
        {
            m_csCurrentTime.SetTime(csTime);
            bNewRecoreFlg= false;
            if (m_csTimeTranInfo != null)
            {   // 時系列データならば
                if (m_csTimeTranInfo.Before(m_csCurrentTime) == true)
                {
                    m_csRecored = m_csTimeTranInfo.CreateNewRecord();
                    McTranInfoIFCellType csCellTrn = m_csRecored as McTranInfoIFCellType;
                    if (csCellTrn != null)
                    {   // 新規にレコードが生成されたならば
                        m_csCellArray = csCellTrn.GetLastCellArrayData();
                        m_csRecored.SetTime(m_csCurrentTime); // 現在時刻設定
                        bNewRecoreFlg = true;
                    }
                }
                else
                {   // 同一時刻データがあればそれを採用
                    m_csCellArray = m_csCellTrnInfo.GetLastCellArrayData();
                }
                if (m_csCellArray != null)
                {
                    if (m_lCellDim == 1)
                    {
                        m_CellD1 = ((HySD1CellArrayData)m_csCellArray).GetCellData();
                    }
                    else if (m_lCellDim == 2)
                    {
                        m_CellD2 = ((HySD2CellArrayData)m_csCellArray).GetCellData();
                        if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo && m_csRecored != null )
                        {
                            m_lGISPttnD2=((McGeoDim2MeshTranInfo)m_csRecored).GetPatternMeshData();
                        }
                    }
                    else if (m_lCellDim == 3)
                    {
                        m_CellD3 = ((HySD3CellArrayData)m_csCellArray).GetCellData();
                        if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo && m_csRecored != null)
                        {
                            m_lGISPttnD3 = ((McGeoDim3MeshTranInfo)m_csRecored).GetPatternMeshData();
                        }
                    }
                }
            }
            else
            {   // 時系列でなければ
                if (m_csCellArray != null)
                {
                    if (m_lCellDim == 1)
                    {
                        m_CellD1 = ((HySD1CellArrayData)m_csCellArray).GetCellData();
                        if (m_csCellTrnInfo is McGeoDim1TranInfo)
                        {
                            ((McGeoDim1TranInfo)m_csCellTrnInfo).SetTime(csTime);
                        }
                    }
                    else if (m_lCellDim == 2)
                    {
                        m_CellD2 = ((HySD2CellArrayData)m_csCellArray).GetCellData();
                        if (m_csCellTrnInfo is McGeoDim2MeshTranInfo)
                        {
                            ((McGeoDim2MeshTranInfo)m_csCellTrnInfo).SetTime(csTime);
                            m_lGISPttnD2 = ((McGeoDim2MeshTranInfo)m_csCellTrnInfo).GetPatternMeshData();
                        }
                    }
                    else if (m_lCellDim == 3)
                    {
                        m_CellD3 = ((HySD3CellArrayData)m_csCellArray).GetCellData();
                        if (m_csCellTrnInfo is McGeoDim3MeshTranInfo)
                        {
                            ((McGeoDim3MeshTranInfo)m_csCellTrnInfo).SetTime(csTime);
                            m_lGISPttnD3 = ((McGeoDim3MeshTranInfo)m_csCellTrnInfo).GetPatternMeshData();
                        }
                    }
                }
                //m_csCellTrnInfo.set
            }

            if (m_csCellArray != null)
            {
                m_csCellArray.SetTime(csTime);
            }
        }

        //★★★★★★★★★★★★★★★★★★
        // Ver1.4で追加(インスタンスの共有化)
        //★★★★★★★★★★★★★★★★★★
        /// <summary><para>method outline:</para>
        /// <para>送信時刻取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCurrentTime() </para>
        /// </example>
        /// <returns>送信時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>設定した時刻の情報を取得する</para>
        /// </remarks>
        public virtual HySTime GetCurrentTime()
        {
            return m_csCurrentTime;
        }

        //==========================================
        //  
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>送信すべきセルを準備（１次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PrepareSendCellD1() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[] これから送信するセル配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>　此処で準備されたセルに送信すべき情報をセットする </para>
        /// </remarks>
        public virtual HySCellData[] PrepareSendCellD1()
        {
            return m_CellD1;
        }
        /// <summary><para>method outline:</para>
        /// <para>送信すべきセルを準備（２次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PrepareSendCellD2() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[,] これから送信するセル配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>　此処で準備されたセルに送信すべき情報をセットする </para>
        /// </remarks>
        public virtual HySCellData[,] PrepareSendCellD2()
        {
            return m_CellD2;
        }
        /// <summary><para>method outline:</para>
        /// <para>送信すべきセルを準備（３次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PrepareSendCellD3() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[,,] これから送信するセル配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>　此処で準備されたセルに送信すべき情報をセットする </para>
        /// </remarks>
        public virtual HySCellData[, ,] PrepareSendCellD3()
        {
            return m_CellD3;
        }
        /// <summary><para>method outline:</para>
        /// <para>設定した情報を実際に送信する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> 
        /// SetCurrentTime( SendTime ); 送信時刻設定
        /// HySCellData[ .. ]  csDt = PrepareSendCellDx()
        /// csDt[ .. ].m_dData[x] = 送信情報設定
        /// SendData()  実際に送信する
        /// </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>時系列データの場合にはその値が設定されるが、定常データの場合にはデータを設定した時点で他のモデルからは参照出来る</para>
        /// </remarks>
        public virtual void SendData()
        {
            if (bNewRecoreFlg == true)
            {
                m_csTimeTranInfo.AddData(m_csRecored);
            }
            bNewRecoreFlg = false;
        }

        //==========================================
        //  プロパティ画面にて各種設定する情報
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>伝送情報パターンプロパティ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTranInfo(csPattern) </para>
        /// </example>
        /// <param name="csPattern">伝送情報パターンプロパティ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetTranInfoPattern(McTranInfoPatternPropertyIF csPattern)
        {
            m_csTrnPattern = csPattern as McTranInfoPattern;
            // パターンの値を入れてはならない
            //long lTD1 = 0; long lTD2 = 0; long lTD3 = 0; long lDataDimInCell = 0;
            //long lCellDim = m_csTrnPattern.GetDimension(ref lTD1, ref lTD2, ref lTD3, ref lDataDimInCell);
            //m_lDim1 = lTD1;
            //m_lDim2 = lTD2;
            //m_lDim3 = lTD3;
            // 
            //if (m_csID == null)
            //{
            //    m_csID = csPattern.GetPatternID().Clone();
            //}
        }

        /// <summary><para>method outline:</para>
        /// <para>セル識別管理データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCellIDMngData(csCellIDMngData) </para>
        /// </example>
        /// <param name="csCellIDMngData">セル識別管理データ</param> 
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCellIDMngData(HySCellIDManageData csCellIDMngData)
        {
            m_csCellIDMngData = csCellIDMngData as McCellIDMngData;
        }

        /// <summary><para>method outline:</para>
        /// <para>第１配列の数を設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDim(lDim1) </para>
        /// </example>
        /// <param name="lDim1">第１配列数</param>
        /// <returns>無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>システム側で使用。モデルからの使用は禁止</para>
        /// </remarks>
        public virtual void SetDim(long lDim1)
        {
            m_lDim1 = lDim1;
            m_lCellDim = 1;
        }
        /// <summary><para>method outline:</para>
        /// <para>第２配列の数を設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDim(lDim1,lDim2) </para>
        /// </example>
        /// <param name="lDim1">第１配列数</param>
        /// <param name="lDim2">第２配列数</param>
        /// <returns>無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>システム側で使用。モデルからの使用は禁止</para>
        /// </remarks>
        public virtual void SetDim(long lDim1, long lDim2)
        {
            m_lDim1 = lDim1;
            m_lDim2 = lDim2;
            m_lCellDim = 2;
        }
        /// <summary><para>method outline:</para>
        /// <para>第３配列の数を設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDim(lDim1,lDim2,lDim3) </para>
        /// </example>
        /// <param name="lDim1">第１配列数</param>
        /// <param name="lDim2">第２配列数</param>
        /// <param name="lDim3">第３配列数</param>
        /// <returns>無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>システム側で使用。モデルからの使用は禁止</para>
        /// </remarks>
        public virtual void SetDim(long lDim1, long lDim2, long lDim3)
        {
            m_lDim1 = lDim1;
            m_lDim2 = lDim2;
            m_lDim3 = lDim3;
            m_lCellDim = 3;
        }

        //==========================================
        //  モデル構築時に使用するメソッド
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>セル識別管理データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellIDManageData csCellIDMngDt = GetCellIDMngData( ) </para>
        /// </example>
        /// <param name=""></param> 
        /// <returns>HySCellIDManageData セル識別管理データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySCellIDManageData GetCellIDMngData()
        {
            return m_csCellIDMngData;
        }

        /// <summary><para>method outline:</para>
        /// <para>伝送情報パターンプロパティ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McTranInfoPatternPropertyIF csPattern = GetTranInfoPattern( ) </para>
        /// </example>
        /// <param name=""></param> 
        /// <returns>McTranInfoPatternPropertyIF 伝送情報パターンプロパティ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McTranInfoPatternPropertyIF GetTranInfoPattern()
        {
            return m_csTrnPattern;
        }
        /// <summary><para>method outline:</para>
        /// <para>送信パターン識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csID = GetSendPatternID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>送信パターン識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetSendPatternID()
        {
            if (m_csTrnPattern != null)
            {
                return m_csTrnPattern.GetPatternID();
            }
            else
            {
                return new HySID("");
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>伝送データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTranInfo(csTranInfo) </para>
        /// </example>
        /// <param name="csTranInfo">伝送データ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetTranInfo(McTranInfo csTranInfo)
        {
            m_csTranInfo = csTranInfo;
            m_csCellTrnInfo = csTranInfo as McTranInfoIFCellType;
            m_csTimeTranInfo = csTranInfo as McTranInfoIFTimeSeriesType;
            if (m_csCellTrnInfo != null)
            {   // セル型ならば
                
                //m_csCellTrnInfo.SetCellDataCharacteristic(this.m_csTrnPattern.GetCellDataCharacteristic().Clone()); // 伝送データへ１セル内の各データの種別設定

                long lD1 = 0; long lD2 = 0; long lD3 = 0;
                m_lCellDim = m_csCellTrnInfo.GetDimension(ref lD1, ref lD2, ref lD3);
                m_lDim1 = lD1;
                m_lDim2 = lD2;
                m_lDim3 = lD3;
                m_csCellTrnInfo.SetCellIDMngData(m_csCellIDMngData); // セル内識別データ設定
                if (m_csTrnPattern != null)
                {   // パターン設定ありならば
                    //m_csCellTrnInfo.GetCellDataCharacteristic().CopyData(m_csTrnPattern.GetCellDataCharacteristic());
                    m_csCellTrnInfo.SetCellDataCharacteristic(this.m_csTrnPattern.GetCellDataCharacteristic().Clone()); // 伝送データへ１セル内の各データの種別設定

                }
                if (m_csTimeTranInfo == null)
                {   // 時系列でなければ
                    m_csCellArray = m_csCellTrnInfo.GetLastCellArrayData();
                    if (m_lCellDim == 1)
                    {
                        m_CellD1 = ((HySD1CellArrayData)m_csCellArray).GetCellData();
                    }
                    else if (m_lCellDim == 2)
                    {
                        m_CellD2 = ((HySD2CellArrayData)m_csCellArray).GetCellData();
                    }
                    else if (m_lCellDim == 3)
                    {
                        m_CellD3 = ((HySD3CellArrayData)m_csCellArray).GetCellData();
                    }
                }
            }
            m_csTranDataKind = csTranInfo.GetTranInfoKind();
        }
        /// <summary><para>method outline:</para>
        /// <para>伝送データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McTranInfo csTranInfo = GetTranInfo() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>csTranInfo 伝送データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McTranInfo GetTranInfo()
        {
            return m_csTranInfo;
        }
        /// <summary><para>method outline:</para>
        /// <para>セルメッシュ分割数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDimension(ref lDim1, ref lDim2, ref lDim3) </para>
        /// </example>
        /// <param name="lDim1">第１分割数</param>
        /// <param name="lDim2">第２分割数</param>
        /// <param name="lDim3">第３分割数</param>
        /// <param name="lDataDimInCell">セル内変数の数</param>
        /// <returns>long 次元数　1次元＝１、２次元＝２、３次元＝３が　返る</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetDimension(ref long lDim1, ref long lDim2, ref long lDim3, ref long lDataDimInCell)
        {
            lDim1 = m_lDim1;
            lDim2 = m_lDim2;
            lDim3 = m_lDim3;
            if (m_csCellTrnInfo != null)
            {
                lDataDimInCell = m_csCellTrnInfo.GetCellDataCharacteristic().GetDataNumber();
            }
            return m_lCellDim;
        }

        /// <summary><para>method outline:</para>
        /// <para>出力側の接続情報チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SendConnectionCheck(ref csErrorInf, csElmID, csElmKind )</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <param name="csElmID">モデルＩＤ</param>
        /// <param name="csElmKind">モデル種別</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>送信端子に設定されている伝送データが自モデルが期待している情報か否かをチェックする</para>
        /// </remarks>
        public virtual bool SendConnectionCheck(ref McStructErrorInfo csErrorInf, HySID csElmID, HySObjectKind csElmKind)
        {
            bool bRtn = true;
            if (m_csTranInfo == null)
            {
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(csElmID, csElmKind, Properties.HymcoImplResources.STATEMENT_S_TRANINFO_NO_SET );
               // csErrorInf.AddCheckErrorData(csElmID, csElmKind, "Send TranInfo is not set.");
                return false;
            }
            if (m_csCellTrnInfo == null)
            {
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(csElmID, csElmKind, Properties.HymcoImplResources.STATEMENT_S_TRANINFO_NO_CELLTYPE );
               // csErrorInf.AddCheckErrorData(csElmID, csElmKind, "Send TranInfo is not CellType.");
                return false;
            }
            if (m_csTrnPattern == null)
            {
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(csElmID, csElmKind, Properties.HymcoImplResources.STATEMENT_TRANINFO_NO_PATTERN );
               // csErrorInf.AddCheckErrorData(csElmID, csElmKind, "Traninfor pattern is not set.");
                return false;
            }
            if (m_csTranDataKind.Equals(m_csTrnPattern.GetTranDataKind()) == false)
            {
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(csElmID, csElmKind, Properties.HymcoImplResources.STATEMENT_TRANINFO_NO_MATCH );
               // csErrorInf.AddCheckErrorData(csElmID, csElmKind, "TranInfo type is not match.");
                bRtn = false;
            }

            // 設定された伝送パターン
            long lTD1 = 0; long lTD2 = 0; long lTD3 = 0; long lDataDimInCell = 0; // パターン側の配列
            long lCellDim = m_csTrnPattern.GetDimension(ref lTD1, ref lTD2, ref lTD3, ref lDataDimInCell);
            // 実際の伝送データ
            long lD1 = 0; long lD2 = 0; long lD3 = 0; // 実際の伝送データの配列
            lCellDim = m_csCellTrnInfo.GetDimension(ref lD1, ref lD2, ref lD3);
            if (m_lCellDim != lCellDim )
            {   // セルの配列の次元が異なっている場合
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(csElmID, csElmKind, Properties.HymcoImplResources.STATEMENT_SELLDIM_NO_MATCH );
               // csErrorInf.AddCheckErrorData(csElmID, csElmKind, "Cell Dimention not match.");
                bRtn = false;
            }
            //if ( lD1 < lTD1 || lD2 < lTD2 || lD3 < lTD3)
            //{   // 伝送側のセル配列数が　モデルが送りたいセル配列数より小さい場合　＜＝これはモデルが決めること　従って此処で判断してはならない
            //    csErrorInf.AddCheckErrorData(csElmID, csElmKind,
            //        "Cell dimention is too short. (Send to " + m_csTranInfo.GetLowerElementID().ToString() + ")");
            //    bRtn = false;
            //}

            if (lDataDimInCell != m_csCellTrnInfo.GetDataDimentionInCell())
            {   // 想定された伝送パターンのセル内変数と実際のセル内の変数の数が異なる
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(csElmID, csElmKind, Properties.HymcoImplResources.STATEMENT_VAL_NO_MATCH );
               // csErrorInf.AddCheckErrorData(csElmID, csElmKind, "Valiable number in the Cell not match.");
                bRtn = false;
            }

            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>伝送データ種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csKind = GetTranInfoKind( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySObjectKind 伝送データ種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySObjectKind GetTranInfoKind()
        {
            if (m_csTranDataKind != null)
            {
                return m_csTranInfo.GetTranInfoKind();
            }
            else
            {
                return new HySObjectKind("");
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>自分と同じクラスのインスタンスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McSendCellDataIF csDt = Clone() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McSendCellDataIF  生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McSendCellDataIF Clone()
        {
            McSendCellData csRtn = new McSendCellData();
            csRtn.m_csID = m_csID.Clone();
            //csRtn.m_csCellIDMngData = new McCellIDMngData();
            //csRtn.m_csCellIDMngData.CopyData(m_csCellIDMngData);
            if (m_csCellIDMngData != null)
            {
                csRtn.m_csCellIDMngData = m_csCellIDMngData.Clone() as McCellIDMngData;
            }

            if (m_csTrnPattern != null)
            {
                csRtn.m_csTrnPattern = (McTranInfoPattern)m_csTrnPattern.Clone();
            }

            csRtn.m_csTranDataKind = m_csTranDataKind.Clone();
            
            csRtn.m_csTranInfo = m_csTranInfo; // クローン化の必要なし
            csRtn.m_csCellTrnInfo = m_csCellTrnInfo; // クローン化の必要なし
            csRtn.m_csTimeTranInfo = m_csTimeTranInfo; // クローン化の必要なし
            csRtn.m_lCellDim = m_lCellDim;
            csRtn.m_csCurrentTime = m_csCurrentTime.Clone();

            return csRtn;
        }

        /*
        /// <summary><para>method outline:</para>
        /// <para>管理情報をXMLへ書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySXmlNode csXmlNode = XmlOUT(csXmlWriter) </para>
        /// </example>
        /// <param name="csXmlWriter">XML書き出しクラス</param>
        /// <returns>HySXmlNode  XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySXmlNode XmlOUT(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtn = HySXmlNode.CreateElement(McDefine.xml_MODEL_CELL_GIS_TABLE, csXmlWriter);
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>XMLからセルと管理情報を読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = XmlIN(csXml) </para>
        /// </example>
        /// <param name="csXml">入力XML</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool XmlIN(HySXmlNode csXml)
        {
            return true;
        }
        */

        //===========================================
        // その他
        //===========================================

        /// <summary><para>method outline:</para>
        /// <para>接続線名取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csName = GetConnectionName( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString 接続線名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>要素を接続している接続線の名称を取得する</para>
        /// </remarks>
        public virtual HySString GetConnectionName()
        {
            HySString csRtn = null;
            if (m_csTranInfo != null)
            {
                csRtn = m_csTranInfo.GetConnectionName();
            }
            else
            {
                csRtn = new HySString("");
            }
            return csRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>下流要素識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySIdentifier csElmID = GetUpperElementID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySIdentifier 識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>接続先要素に対して情報を伝送する</para>
        /// </remarks>
        public virtual HySID GetLowerElementID()
        {
            if (m_csTranInfo != null)
            {
                return m_csTranInfo.GetLowerElementID();
            }
            else
            {
                return new HySID("");
            }
        }

        //==========================================
        //  ＧＩＳ関連：各種情報設定／取得
        //==========================================
        /*
        /// <summary><para>method outline:</para>
        /// <para>送信すべきＧＩＳパターンを準備（１次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PrepareSendGISPatternD2() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> long[] これから送信するＧＩＳパターン配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>　此処で準備されたＧＩＳパターンに送信すべき情報をセットする </para>
        /// </remarks>
        public virtual long[] PrepareSendGISPatternD1()
        { return m_lGISPttnD1; }
        */
        /// <summary><para>method outline:</para>
        /// <para>送信すべきＧＩＳパターンを準備（２次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PrepareSendGISPatternD2() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> long[,] これから送信するＧＩＳパターン配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>　此処で準備されたＧＩＳパターンに送信すべき情報をセットする （将来用：特殊用途）</para>
        /// </remarks>
        public virtual long[,] PrepareSendGISPatternD2()
        { return m_lGISPttnD2; }
        /// <summary><para>method outline:</para>
        /// <para>送信すべきＧＩＳパターンを準備（３次元セル）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PrepareSendGISPatternD3() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> long[,,] これから送信するＧＩＳパターン配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>　此処で準備されたＧＩＳパターンに送信すべき情報をセットする （将来用：特殊用途）</para>
        /// </remarks>
        public virtual long[, ,] PrepareSendGISPatternD3()
        { return m_lGISPttnD3; }

        /// <summary><para>method outline:</para>
        /// <para>距離配列の設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCellDistance(dDistance)</para> 
        /// </example>
        /// <param name="dDistance">double[] 最初のセル始点からの距離配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCellDistance(double[] dDistance)
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                ((McGeoDim1TranInfo)m_csCellTrnInfo).SetCellDistance(dDistance);
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).SetCellDistance(dDistance);
            }
            else
            {
                // Do nothing
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セル始点・終点緯度情報配列の設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCellLatitude(dCellLat)</para> 
        /// </example>
        /// <param name="dCellLat">double[] セルの始点／終点の緯度情報配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCellLatitude(double[] dCellLat)
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                ((McGeoDim1TranInfo)m_csCellTrnInfo).SetCellLatitude(dCellLat);
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).SetCellLatitude(dCellLat);
            }
            else
            {
                // Do nothing
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セル始点・終点経度情報配列の設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCellLongitude(dCellLon)</para> 
        /// </example>
        /// <param name="dCellLon">double[] セルの始点／終点の経度情報配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCellLongitude(double[] dCellLon)
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                ((McGeoDim1TranInfo)m_csCellTrnInfo).SetCellLongitude(dCellLon);
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).SetCellLongitude(dCellLon);
            }
            else
            {
                // Do nothing
            }
        }


        /// <summary><para>method outline:</para>
        /// <para>代表点の緯度情報配列の設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSpecificLatitude(dSpecificLat)</para> 
        /// </example>
        /// <param name="dSpecificLat">double[] 代表点の緯度情報配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetSpecificLatitude(double[] dSpecificLat)
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                ((McGeoDim1TranInfo)m_csCellTrnInfo).SetSpecificLatitude(dSpecificLat);
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).SetSpecificLatitude(dSpecificLat);
            }
            else
            {
                // Do nothing
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>代表点の経度情報配列の設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSpecificLongitude(dSpecificLon)</para> 
        /// </example>
        /// <param name="dSpecificLon">double[] 代表点の経度情報配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetSpecificLongitude(double[] dSpecificLon)
        {
            if (m_csCellTrnInfo is McGeoDim1TranInfo)
            {
                ((McGeoDim1TranInfo)m_csCellTrnInfo).SetSpecificLongitude(dSpecificLon);
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
            {
                ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).SetSpecificLongitude(dSpecificLon);
            }
            else
            {
                // Do nothing
            }
        }

        ///// <summary><para>method outline:</para>
        ///// <para>距離配列の取得</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para>double[] dDistance = GetCellDistance()</para> 
        ///// <para>例：1番目のセル始点から、1番目のセル終点までの距離情報設定　dDistance[0] = x;</para>
        ///// <para>例：1番目のセル始点から、2番目のセル終点までの距離情報設定　dDistance[1] = x;</para>
        ///// <para>例：1番目のセル始点から、1番目のセル終点までの距離情報取得　y = dDistance[0];</para>
        ///// <para>例：1番目のセル始点から、2番目のセル終点までの距離情報取得　y = dDistance[1];</para>
        ///// </example>
        ///// <param name="">無し</param>
        ///// <returns>double[] 最初のセル始点からの距離配列 GIS1次元情報でない場合配列1つを返す</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public double[] GetCellDistance()
        //{
        //    if (m_csCellTrnInfo is McGeoDim1TranInfo)
        //    {
        //        return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetCellDistance();
        //    }
        //    else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
        //    {
        //        return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetCellDistance();
        //    }
        //    else
        //    {
        //        return new double[1];
        //    }
        //}

        ///// <summary><para>method outline:</para>
        ///// <para>セル始点・終点緯度情報配列の取得</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para>double[] dCellLat = GetCellLatitude()</para> 
        ///// <para>例：1番目のセル始点の緯度情報設定　dCellLat[0] = x;</para>
        ///// <para>例：1番目のセル終点の緯度情報設定　dCellLat[1] = x;</para>
        ///// <para>dCellLat[1]は同時に2番目のセル始点となる</para>
        ///// <para>例：1番目のセル始点の緯度情報取得　y = dCellLat[0];</para>
        ///// </example>
        ///// <param name="">無し</param>
        ///// <returns>double[] セルの始点／終点の緯度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public double[] GetCellLatitude()
        //{
        //    if (m_csCellTrnInfo is McGeoDim1TranInfo)
        //    {
        //        return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetCellLatitude();
        //    }
        //    else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
        //    {
        //        return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetCellLatitude();
        //    }
        //    else
        //    {
        //        return new double[1];
        //    }
        //}

        ///// <summary><para>method outline:</para>
        ///// <para>セル始点・終点経度情報配列の取得</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para>double[] dCellLon = GetCellLongitude()</para> 
        ///// <para>例：1番目のセル始点の経度情報設定　dCellLon[0] = x;</para>
        ///// <para>例：1番目のセル終点の経度情報設定　dCellLon[1] = x;</para>
        ///// <para>dCellLon[1]は同時に2番目のセル始点となる</para>
        ///// <para>例：1番目のセル始点の経度情報取得　y = dCellLon[0];</para>
        ///// </example>
        ///// <param name="">無し</param>
        ///// <returns>double[] セルの始点／終点の経度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public double[] GetCellLongitude()
        //{
        //    if (m_csCellTrnInfo is McGeoDim1TranInfo)
        //    {
        //        return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetCellLongitude();
        //    }
        //    else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
        //    {
        //        return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetCellLongitude();
        //    }
        //    else
        //    {
        //        return new double[1];
        //    }
        //}

        ///// <summary><para>method outline:</para>
        ///// <para>代表点の緯度情報配列の取得</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para>double[] dSpecificLat = GetSpecificLatitude()</para> 
        ///// <para>例：1番目のセルデータの緯度情報設定　dSpecificLat[0] = x;</para>
        ///// <para>例：1番目のセルデータの緯度情報取得　y = dSpecificLat[0];</para>
        ///// </example>
        ///// <param name="">無し</param>
        ///// <returns>double[] 代表点の緯度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public double[] GetSpecificLatitude()
        //{
        //    if (m_csCellTrnInfo is McGeoDim1TranInfo)
        //    {
        //        return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetSpecificLatitude();
        //    }
        //    else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
        //    {
        //        return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetSpecificLatitude();
        //    }
        //    else
        //    {
        //        return new double[1];
        //    }
        //}

        ///// <summary><para>method outline:</para>
        ///// <para>代表点の経度情報配列の取得</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para>double[] dSpecificLon = GetSpecificLongitude()</para> 
        ///// <para>例：1番目のセルデータの経度情報設定　dSpecificLon[0] = x;</para>
        ///// <para>例：1番目のセルデータの経度情報取得　y = dSpecificLon[0];</para>
        ///// </example>
        ///// <param name="">無し</param>
        ///// <returns>double[] 代表点の経度情報配列 GIS1次元情報でない場合配列1つを返す</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public double[] GetSpecificLongitude()
        //{
        //    if (m_csCellTrnInfo is McGeoDim1TranInfo)
        //    {
        //        return ((McGeoDim1TranInfo)m_csCellTrnInfo).GetSpecificLongitude();
        //    }
        //    else if (m_csCellTrnInfo is McTimeSeriesGeoD1TranInfo)
        //    {
        //        return ((McTimeSeriesGeoD1TranInfo)m_csCellTrnInfo).GetSpecificLongitude();
        //    }
        //    else
        //    {
        //        return new double[1];
        //    }
        //}

        //==========================================
        //  ＧＩＳ関連：各種情報取得
        //==========================================
        /// <summary><para>method outline:</para>
        /// <para>ＧＩＳメッシュ分割数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDimension(ref lKind, ref lDim1, ref lDim2, ref lDim3, ref lDataDimInCell) </para>
        /// </example>
        /// <param name="lKind">保持するデータの形</param>
        /// <param name="lLonDim">経度方向のメッシュ分割数</param>
        /// <param name="lLatDim">緯度方向のメッシュ分割数</param>
        /// <param name="lAltDim">高度方向のメッシュ分割数</param>
        /// <param name="lCellDataDim">１メッシュ内のデータ配列数</param>
        /// <returns>
        /// long 次元数　1次元＝１、２次元＝２、３次元＝３が　返る。
        /// 　　　　　　　ＧＩＳ情報でない時には０が　返る</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetGISMeshDimension(ref HySGeoMesh.MESH_DATA_KIND lKind, ref long lLonDim, ref long lLatDim, ref long lAltDim, ref long lCellDataDim)
        {
            long lDim = 0;
            lLonDim = 0;
            lLatDim = 0;
            lAltDim = 0;
            lCellDataDim = 0;
            if (m_csCellTrnInfo != null)
            {
                lDim = m_csCellTrnInfo.GetDimension(ref  lLonDim, ref lLatDim, ref lAltDim);
                lCellDataDim = m_csCellTrnInfo.GetDataDimentionInCell();
            }


            if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo)
            {
                lKind = ((McTimeSeriesGeoD2MeshTranInfo)m_csCellTrnInfo).GetDataMeshKind();
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo)
            {
                lKind = ((McTimeSeriesGeoD3MeshTranInfo)m_csCellTrnInfo).GetDataMeshKind();
            }
            else if (m_csCellTrnInfo is McGeoDim2MeshTranInfo)
            {
                lKind = ((McGeoDim2MeshTranInfo)m_csCellTrnInfo).GetDataMeshKind();
            }
            else if (m_csCellTrnInfo is McGeoDim3MeshTranInfo)
            {
                lKind = ((McGeoDim3MeshTranInfo)m_csCellTrnInfo).GetDataMeshKind();
            }
            else
            {
                // Do nothing
            }
            return lDim;
        }

        /// <summary><para>method outline:</para>
        /// <para>地理的位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLocation(dSWLat, dSWLon, dNELat, dNELon) </para>
        /// </example>
        /// <param name="dSWLat">南西頂点緯度（度単位）</param>
        /// <param name="dSWLon">南西頂点経度（度単位）</param>
        /// <param name="dNELat">北東頂点緯度（度単位）</param>
        /// <param name="dNELon">北東頂点経度（度単位）</param> 
        /// <returns>=true:伝送データがＧＩＳデータの場合、=false:伝送データがＧＩＳ以外のデータの場合</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool SetLocation(double dSWLat, double dSWLon, double dNELat, double dNELon)
        {
            bool bRtn = true;
            //if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo && m_csRecored != null)
            if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo )
            {
                //((McGeoDim2MeshTranInfo)m_csRecored).SetLocation(dSWLat, dSWLon, dNELat, dNELon);
                ((McTimeSeriesGeoD2MeshTranInfo)m_csCellTrnInfo).SetLocation(dSWLat, dSWLon, dNELat, dNELon);
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo )
            //else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo && m_csRecored != null)
            {
                //((McGeoDim3MeshTranInfo)m_csRecored).SetLocation(dSWLat, dSWLon, dNELat, dNELon);
                ((McTimeSeriesGeoD3MeshTranInfo)m_csCellTrnInfo).SetLocation(dSWLat, dSWLon, dNELat, dNELon);
            }
            else if (m_csCellTrnInfo is McGeoDim2MeshTranInfo)
            {
                ((McGeoDim2MeshTranInfo)m_csCellTrnInfo).SetLocation(dSWLat, dSWLon, dNELat, dNELon);
            }
            else if (m_csCellTrnInfo is McGeoDim3MeshTranInfo)
            {
                ((McGeoDim3MeshTranInfo)m_csCellTrnInfo).SetLocation(dSWLat, dSWLon, dNELat, dNELon);
            }
            else
            {
                bRtn = false;
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>地理的位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetLocation(ref dSWLat,ref dSWLon,ref dNELat,ref dNELon) </para>
        /// </example>
        /// <param name="dSWLat">南西頂点緯度（度単位）</param>
        /// <param name="dSWLon">南西頂点経度（度単位）</param>
        /// <param name="dNELat">北東頂点緯度（度単位）</param>
        /// <param name="dNELon">北東頂点経度（度単位）</param> 
        /// <returns>=true:伝送データがＧＩＳデータの場合、=false:伝送データがＧＩＳ以外のデータの場合</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool GetLocation(ref double dSWLat, ref double dSWLon, ref double dNELat, ref double dNELon)
        {
            bool bRtn = true;
            if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo )
            //if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo && m_csRecored != null)
            {
                ((McTimeSeriesGeoD2MeshTranInfo)m_csCellTrnInfo).GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
                //((McGeoDim2MeshTranInfo)m_csRecored).GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
            }
            else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo )
            //else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo && m_csRecored != null)
            {
                ((McTimeSeriesGeoD3MeshTranInfo)m_csCellTrnInfo).GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
                //((McGeoDim3MeshTranInfo)m_csRecored).GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
            }
            else if (m_csCellTrnInfo is McGeoDim2MeshTranInfo)
            {
                ((McGeoDim2MeshTranInfo)m_csCellTrnInfo).GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
            }
            else if (m_csCellTrnInfo is McGeoDim3MeshTranInfo)
            {
                ((McGeoDim3MeshTranInfo)m_csCellTrnInfo).GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);
            }
            else
            {
                bRtn = false;
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>高度位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetAltitude(dUpAlt, dLwAlt) </para>
        /// </example>
        /// <param name="dUpAlt">上限高度（ｍ単位）</param>
        /// <param name="dLwAlt">下限高度（ｍ単位）</param>
        /// <returns>=true:伝送データがＧＩＳデータの場合、=false:伝送データがＧＩＳ以外のデータの場合</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool SetAltitude(double dUpAlt, double dLwAlt)
        {
            bool bRtn = true;
            //if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo && m_csRecored != null)
            if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo )
            {
                ((McTimeSeriesGeoD2MeshTranInfo)m_csCellTrnInfo).SetAltitude(dUpAlt, dLwAlt);
                //((McGeoDim2MeshTranInfo)m_csRecored).SetAltitude(dUpAlt, dLwAlt);
            }
            //else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo && m_csRecored != null)
            else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo )
            {
                ((McTimeSeriesGeoD3MeshTranInfo)m_csCellTrnInfo).SetAltitude(dUpAlt, dLwAlt);
                //((McGeoDim3MeshTranInfo)m_csRecored).SetAltitude(dUpAlt, dLwAlt);
            }
            else if (m_csCellTrnInfo is McGeoDim2MeshTranInfo)
            {
                ((McGeoDim2MeshTranInfo)m_csCellTrnInfo).SetAltitude(dUpAlt, dLwAlt);
            }
            else if (m_csCellTrnInfo is McGeoDim3MeshTranInfo)
            {
                ((McGeoDim3MeshTranInfo)m_csCellTrnInfo).SetAltitude(dUpAlt, dLwAlt);
            }
            else
            {
                bRtn = false;
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>高度位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HGetAltitude(ref dUpAlt, ref dLwAlt) </para>
        /// </example>
        /// <param name="dUpAlt">上限高度（ｍ単位）</param>
        /// <param name="dLwAlt">下限高度（ｍ単位）</param>
        /// <returns>=true:伝送データがＧＩＳデータの場合、=false:伝送データがＧＩＳ以外のデータの場合</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool GetAltitude(ref double dUpAlt, ref double dLwAlt)
        {
            bool bRtn = true;
            
            //if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo && m_csRecored != null)
            if (m_csCellTrnInfo is McTimeSeriesGeoD2MeshTranInfo )
            {
                ((McTimeSeriesGeoD2MeshTranInfo)m_csCellTrnInfo).GetAltitude(ref dUpAlt, ref dLwAlt);
                //((McGeoDim2MeshTranInfo)m_csRecored).GetAltitude(ref dUpAlt, ref dLwAlt);
            }
            //else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo && m_csRecored != null)
            else if (m_csCellTrnInfo is McTimeSeriesGeoD3MeshTranInfo )
            {
                ((McTimeSeriesGeoD3MeshTranInfo)m_csCellTrnInfo).GetAltitude(ref dUpAlt, ref dLwAlt);
                //((McGeoDim3MeshTranInfo)m_csRecored).GetAltitude(ref dUpAlt, ref dLwAlt);
            }
            else if (m_csCellTrnInfo is McGeoDim2MeshTranInfo)
            {
                ((McGeoDim2MeshTranInfo)m_csCellTrnInfo).GetAltitude(ref dUpAlt, ref dLwAlt);
            }
            else if (m_csCellTrnInfo is McGeoDim3MeshTranInfo)
            {
                ((McGeoDim3MeshTranInfo)m_csCellTrnInfo).GetAltitude(ref dUpAlt, ref dLwAlt);
            }
            else
            {
                bRtn = false;
            }
            return bRtn;
        }

        // =============================================
        // === バージョンアップ時の互換性を保つ為の処理 ===
        // =============================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
    }
}
