﻿// <summary>ソースコード：ＨＹＭＣＯ演算の制御を行うクラス</summary>
// <author>CommonMP</author>

using System;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSSimulator;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.Interface.HSController;
using CommonMP.HYSSOP.Interface.HSTools;

using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSController;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSSimulator;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Controller;
using CommonMP.HYMCO.Interface.Data;

using CommonMP.HYMCO.CoreImpl.Model;
using CommonMP.HYMCO.CoreImpl.BusiProc;
using CommonMP.HYMCO.CoreImpl.Controller;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.FileIO;
using CommonMP.HYMCO.CoreImpl.Data.StructInfo;
using CommonMP.HYMCO.CoreImpl.Data.ProjectCtl;
using CommonMP.HYMCO.CoreImpl.Tool;


namespace CommonMP.HYMCO.CoreImpl.FigureUnity
{
    /// <summary><para>class outline:</para>
    /// <para>ＨＹＭＣＯ演算制御を行う</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>
    /// [CommonMP][ver 1.0.0][2008/10/01][新規作成]
    /// [CommonMP][ver 1.4.0][2013/12/06][修正]
    /// </para>
    /// </remarks>
    public class McFigureUnity : McGrElement, HySSimulatorRoot 
    {
        /// <summary>シミュレーター種別識別子</summary>
        protected HySObjectKind m_csKindID; // シミュレーター種別識別子

        // /// <summary>演算インスタンス識別ＩＤ</summary>
        //protected HySID m_csID;  // 演算インスタンス識別ＩＤ  <-- McGrElement内に定義済み

        /// <summary>シミュレーション初期時刻</summary>
        protected HySTime m_csStartTime;

        /// <summary>演算スレッド制御クラス</summary>
        protected HySSimThreadController m_csThreadCtl;


        /// <summary>シミュレーター（演算単位系）ファクトリクラス</summary>
        protected HySSimulatorFactory m_csSimulatorFactory;

        /// <summary>データコンテナクラス</summary>
        protected HySSimulationDataContainer m_csDataContainer;

        /// <summary>要素間接続情報クラス</summary>
        protected McStructInfo m_csStructInfo;


        /// <summary>計算状態</summary>
        protected long m_lCalStatus = McGrElementCtl.DEFAULT_STATE;

        /// <summary>モデル生成ファクトリ</summary>
        protected McModelFactorySet m_csModelFactory;

        /// <summary>計算開始コール済みフラグ</summary>
        protected long m_bCalSatrtNum = 0;

        /// <summary>応答メッセージ </summary>
        protected HySString m_csRtnMessage = new HySString("");

        //======================================
        // 固有メソッド
        //======================================

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McFigureUnity csObj = new McFigureUnity( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McFigureUnity()
        {
            this.SetCalModel(new McSubSystemCtl());
        }

        /// <summary><para>method outline:</para>
        /// <para>デストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> 外部からの使用不可</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        ~McFigureUnity()
        {
            try
            {
                if (m_csID != null)
                {
                    HySCommonInfoHash.RemoveInfo(McDefine.PROJECT_GROUP_NAME, this.m_csID.ToString()); // 対応登録した情報を削除する
                    HySCommonInfoHash.RemoveInfo(McDefine.CALCULATION_LOT_NAME, this.m_csID.ToString());
                }
            }
            catch
            {
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ化要素への変換（自身が新しいグループ化要素になる）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McGrElement csGrElm = ChangeGroupElement( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成されたグループ化要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在の演算系が部分系として取り込まれる場合に使用する</para>
        /// </remarks>
        public virtual McGrElement ChangeGroupElement()
        {
            McGrElement csGrElm = this.CreateExchangeGrElement();

            // 演算モデル設定
            csGrElm.SetCalModel(new McSubSystemCtl());

            return csGrElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル生成ファクトリ集合体設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetModelFactory(csFctrySet)</para>
        /// </example>
        /// <param name="csFctrySet">モデル生成ファクトリ集合体</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetModelFactory(McModelFactorySet csFctrySet)
        {
            m_csModelFactory = csFctrySet;
        }

        //======================================
        // オーバーライド、実態化メソッド
        //======================================

        // ====================================
        // 各種設定
        // ====================================
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター（演算単位系）ファクトリクラス設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimulatorFactory(csSimulatorFactory)</para>
        /// </example>
        /// <param name="csSimulatorFactory">シミュレーター（演算単位系）ファクトリクラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimulatorFactory(HySSimulatorFactory csSimulatorFactory)
        {
            m_csSimulatorFactory = csSimulatorFactory;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimKind(csSimKindID)</para>
        /// </example>
        /// <param name="csSimKindID">シミュレーター種別識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimKind(HySKind csSimKindID)
        {
            m_csKindID = (HySObjectKind)csSimKindID;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySKind csKind = GetSimKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>シミュレーター種別識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySKind GetSimKind()
        {
            return m_csKindID;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bVar = EqualSimKind(csSimKindID)</para>
        /// </example>
        /// <param name="csSimKindID">シミュレーター種別識別子</param>
        /// <returns>
        /// true  : 同じ
        /// false : 異なる
        /// </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool EqualSimKind(HySKind csSimKindID)
        {
            return m_csKindID.Equals(csSimKindID);
        }

        // ＝＝　シミュレーション時間関係管理　＝＝
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算初期時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuStartTime( csTm )</para>
        /// </example>
        /// <param name="csTm">計算初期時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuStartTime(HySDate csTm)
        {
            m_csStartTime = (HySTime)csTm;
            base.SetStartTime((HySTime)csTm);
        }

        /// <summary><para>method outline:</para>
        /// <para>目標時間設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuGoalTime( csTm )</para>
        /// </example>
        /// <param name="csTm">目標時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuGoalTime(HySDate csTm)
        {
            this.SetTargetTime((HySTime)csTm);
            this.m_csCalModel.SetSimuGoalTime((HySTime)csTm);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuTime( csTm )</para>
        /// </example>
        /// <param name="csTm">計算時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuTime(HySDate csTm)
        {
            this.SetSimulationTime((HySTime)csTm);
        }
        /// <summary><para>method outline:</para>
        /// <para>演算時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySDate csSimuTime = GetSimuTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>演算時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySDate GetSimuTime()
        {
            if (m_csCalModel != null)
            {
                return this.m_csCalModel.GetSimulationTime();
            }
            else
            {
                return null;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>演算刻み時間を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuDeltaTime(csDltTime)</para>
        /// </example>
        /// <param name="csDltTime">演算刻み時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuDeltaTime(HySDate csDltTime)
        {
            this.SetDeltaTime((HySTime)csDltTime);
        }

        /// <summary><para>method outline:</para>
        /// <para>コンテナ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetContainer( csContainer ) </para>
        /// </example>
        /// <param name="csContainer">コンテナクラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetContainer(HySSimulationContainer csContainer)
        {
            m_csDataContainer = (HySSimulationDataContainer)csContainer;
            this.SetID(m_csDataContainer.GetID());
            m_csStructInfo = (McStructInfo)m_csDataContainer.GetData(McDefine.HYM_DATA_STRUCT_INFO);
            // プロジェクト情報
            McProjectInfData csProjectInfData = (McProjectInfData)m_csDataContainer.GetData(McDefine.HYM_DATA_PROJECT_INFO);
            csProjectInfData.SetProjectID((HySID)m_csDataContainer.GetID());

            //// ログ表示の画面設定（アクティブな画面に設定）
            //McModelLogData csModelLog = (McModelLogData)m_csDataContainer.GetData(McDefine.HYM_DATA_LOG_INFO);
            //McLog.SetActiveModelLogData(csModelLog);
        }

        /// <summary><para>method outline:</para>
        /// <para>コンテナ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySContainer csContainer = GetContainer() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>コンテナインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySSimulationContainer GetContainer()
        {
            return m_csDataContainer;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算スレッド制御クラス設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimThreadCtl( csThrdCtl ) </para>
        /// </example>
        /// <param name="csThrdCtl">演算スレッド制御クラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimThreadCtl(HySSimThreadCtl csThrdCtl)
        {
            m_csThreadCtl = (HySSimThreadController)csThrdCtl;
        }

        /// <summary><para>method outline:</para>
        /// <para>立ち上がり処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Initialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool Initialize()
        {
            //base.SimInitialize(); <-- これは良くない
            //m_csProjectInfData = (McProjectInfData)m_csSimulatorFactory.CreateData(m_csKindID,new HySObjectKind(McDefine.HYM_DATA_PROJECT_INFO),m_csID);
            //m_csDataContainer.SetData(McDefine.HYM_DATA_PROJECT_INFO, m_csProjectInfData);

            this.m_bCalSatrtNum = 0;   // 計算開始フラグリセット

            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>終了処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>  bool bRtn = Terminate( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool Terminate()
        {
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算実行の終了</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>  bool bRtn = TerminateCalculation( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>具体的には計算を行っていたスレッドが終了する</para>
        /// </remarks>
        public virtual bool TerminateCalculation()
        {
            McLog.RemoveModelLogData();
            if (m_lCalStatus == McGrElementCtl.STOP_STATE)
            {
                this.CompleteCalculation();
            }

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>内部演算モデル作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = ModelConstruction(csEvent) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool ModelConstruction(HySSysEvent csEvent)
        {
            //★サーバー／クライアント化で追加
            // ★　本来ならば、頻繁にコンテナインスタンスを修正すべきではない。
            //     変更のあった部分のみを　反映させて、インスタンス本体は入れ替えない。（将来の検討課題）
            HySSimulationDataContainer csRcvContainer = ((HySEventObject)csEvent).GetData() as HySSimulationDataContainer;
            if (csRcvContainer != null)
            {
                this.SetContainer(csRcvContainer); // データコンテナ設定
            }
            //★サーバー／クライアント化で追加

            bool bRtn = false;
            this.m_csRtnMessage.SetChar("");
            McStructInfo csStructInfo = m_csDataContainer.GetData(McDefine.HYM_DATA_STRUCT_INFO) as McStructInfo;
            McStructCheckData csCheckInf = m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO) as McStructCheckData;
            if (csCheckInf == null)
            {
                csCheckInf = new McStructCheckData();
                m_csDataContainer.SetData(McDefine.HYM_DATA_CHECK_INFO, csCheckInf);
            }
            csCheckInf.ClearCheckData();
            if (csStructInfo != null)
            {
                bRtn = this.ModelConstruction(csStructInfo, m_csModelFactory, ref csCheckInf);
                if (bRtn == false)
                {
                   // ver1.5 エラートレース日本語対応
                    csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), Properties.HymcoImplResources.STATEMENT_MODEL_CONST_NG );
                   // csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), "ModelConstruction fail.");
                }

                // コンテナの情報をシミュレータ各要素のモデルに設定(入出力モデルに関係)
                this.SetSimDataContainer(m_csDataContainer);
            }
            else
            {
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity","ModelConstruction", 
                    Properties.HymcoImplResources.STATEMENT_MCSTRUCTINFO_NOT_EXIST );
               // HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "ModelConstruction", "McStructInfo is not exist in the Container");
            }
            m_lCalStatus = McGrElementCtl.DEFAULT_STATE;
            return bRtn; 
        }

        /// <summary><para>method outline:</para>
        /// <para>内部演算モデル作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = ModelConstruction( csStructInfo,csModelFactory,csCheckInf ) </para>
        /// </example>
        /// <param name="csStructInfo">モデル構造データ</param>
        /// <param name="csModelFactory">モデルファクトリ</param>
        /// <param name="csCheckInf">接続エラー情報</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool ModelConstruction(McStructGroupElement csStructInfo, McModelFactorySet csModelFactory, ref McStructCheckData csCheckInf)
        {
            bool bRtn = true;
            try
            {
                long lMxElm = 500;
                HySString csString = HySEnvInf.GetEnvInf("HYMCO_MAX_ELEMENT_NUMBER");
                if ((object)csString != null)
                {
                    lMxElm = Convert.ToInt32(csString.ToString());
                }
                if (csStructInfo.HowManyElement() > lMxElm)
                {
                   // ver1.5 エラートレース日本語対応
                    csCheckInf.AddCheckErrorData(this.GetID(), new HySObjectKind("McFigureUnity"), 
                         Properties.HymcoImplResources.STATEMENT_MANY_MODELS + lMxElm.ToString() + ")");
                   // csCheckInf.AddCheckErrorData(this.GetID(), new HySObjectKind("McFigureUnity"), "Too many models :(Max Models number =" + lMxElm.ToString() + ")");
                    bRtn = false;
                }

                // 共通情報の登録　インスタンス識別子とプロジェクトグループの登録
                McProjectInfData csProjectInfData = m_csDataContainer.GetData(McDefine.HYM_DATA_PROJECT_INFO) as McProjectInfData;
                HySString csProjectGroupName = csProjectInfData.GetProjectGroupName(); //csProjectInfData から　プロジェクトグループを取得する
                HySCommonInfoHash.SetCorrespond(McDefine.PROJECT_GROUP_NAME, this.m_csID.ToString(), csProjectGroupName.ToString()); // 対応登録を行う
                HySCommonInfoHash.SetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.m_csID.ToString(), csCheckInf); // 対応登録を行う

                McModelInfo csMdlInf = csStructInfo.GetModelInfo();
                McFigUnityPropertyInfo csPrpyt = csMdlInf.GetPropertyInfo() as McFigUnityPropertyInfo;
                HySTime csTimeStep = csPrpyt.GetTimeStep();
                m_csDataContainer.SetDeltaTime(csTimeStep);
                // モデル構築実行
                if (bRtn == true)
                {   // モデル数がＯＫならば、
                    this.SetOwnerProjectID(this.m_csID.ToString());
                    //★★★★★★★★★★★★★★★★★
                    // Ver1.4で追加(伝送情報の事前作成)
                    //★★★★★★★★★★★★★★★★★
                    // 計算期間及び全体系タイムステップの情報抽出
                    HySTime csSimStart = (HySTime)m_csDataContainer.GetStartTime();// 開始時刻
                    HySTime csSimEnd = (HySTime)m_csDataContainer.GetGoalTime();// 終了時刻
                    HySTime csSimTimeStep = (HySTime)m_csDataContainer.GetDeltaTime();// 全体系タイムステップ
                    string sPrjID = this.GetOwnerProjectID();// プロジェクトID
                    // 設定
                    CommonMP.HYMCO.CoreImpl.Data.TranInfo.McTranInfoDataPool.SetSimInfo(sPrjID, csSimStart, csSimEnd, csSimTimeStep);

                    bRtn = base.ModelConstruction(csStructInfo, csModelFactory, ref csCheckInf);

                    // 此処で、内部の各モデルへ　プロジェクトインスタンス識別子の登録を行う
                    if (bRtn == true)
                    {
                        this.SetOwnerProjectID(m_csID.ToString());
                    }
                }

                if (bRtn == true)
                {   // モデル構築が成功したならば、
                    // 自シミュレータに属している要素モデルに対してシミュレータIDを通知する
                    this.SetSimulationID(m_csDataContainer.GetID());

                    McStructErrorInfo csErrorInfo = (McStructErrorInfo)csCheckInf;
                    bRtn = base.Prepare(ref csErrorInfo);   // 演算準備設定
                    if (bRtn == true)
                    {
                        // 演算準備設定でエラー無しならば
                        // 時刻仮設定
                        this.SetSimuStartTime(m_csDataContainer.GetStartTime()); m_csCalModel.SetStartTime((HySTime)m_csDataContainer.GetStartTime());
                        this.SetSimuGoalTime(m_csDataContainer.GetGoalTime()); m_csCalModel.SetTargetTime((HySTime)m_csDataContainer.GetGoalTime());
                        this.SetSimuTime(m_csDataContainer.GetSimuTime()); m_csCalModel.SetSimulationTime((HySTime)m_csDataContainer.GetSimuTime());
                        this.SetSimuDeltaTime(m_csDataContainer.GetDeltaTime()); m_csCalModel.SetDeltaTime((HySTime)m_csDataContainer.GetDeltaTime());
                        // 演算チェック
                        McStructErrorInfo csErrorInfData = (McStructErrorInfo)m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO);
                        // 接続順序チェック
                        long lLclRtn = this.CheckCalculationOrder();
                        if (lLclRtn == McDefine.CALCULATION_NORMAL_RETURN)
                        {   // 接続に異常なし
                            this.MakeContinuity(); // 中継端子の導通設定を行う

                            // パラメーター設定等チェック
                            bRtn = m_csCalModel.IsAbleToCalculate(ref m_csReceiveInfoList, ref m_csSendInfoList, ref csErrorInfData);
                            if (bRtn == true)
                            {   // 正常終了の場合
                               // ver1.5 エラートレース日本語対応
                                csErrorInfData.AddCheckInformationData(new HySID(""), new HySObjectKind("McFigureUnity"), 
                                    Properties.HymcoImplResources.STATEMENT_CHECK_COMPLETE_NO_ERR );
                               // csErrorInfData.AddCheckInformationData(new HySID(""), new HySObjectKind("McFigureUnity"), "  Model check is completed.   No error.");
                            }
                            else
                            {   // 異常の場合
                                //csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"), "");
                                base.ClearElement();
                                base.ClearConnection();
                            }
                        }
                        else
                        {
                           // ver1.5 エラートレース日本語対応
                            csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"),
                                 Properties.HymcoImplResources.STATEMENT_WRONG_MODEL_STRUCT );
                           // csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"), "  Wrong model structure. For example \"Loop connection exist.\"");
                            bRtn = false;
                            base.ClearElement();
                            base.ClearConnection();
                        }
                        // パラメーター設定等チェック
                        //bRtn = m_csCalModel.IsAbleToCalculate(ref m_csReceiveInfoList, ref m_csSendInfoList, ref csErrorInfData);
                        //if (bRtn == true)
                        //{   // 正常終了の場合
                        //    long lLclRtn = this.CheckCalculationOrder();
                        //    if (lLclRtn == McDefine.CALCULATION_NORMAL_RETURN)
                        //    {
                        //        bRtn = true;
                        //        csErrorInfData.AddCheckInformationData(new HySID(""), new HySObjectKind("McFigureUnity"), "  Model check is completed.   No error.");
                        //    }
                        //    else
                        //    {
                        //        csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"), "  Wrong model structure. For example \"Loop connection exist.\"");
                        //        bRtn = false;
                        //        base.ClearElement();
                        //        base.ClearConnection();
                        //    }
                        //}
                        //else
                        //{
                        //    base.ClearElement();
                        //    base.ClearConnection();
                        //}
                    }
                    else
                    {
                        base.ClearElement();
                        base.ClearConnection();
                    }
                }
                else
                {
                    base.ClearElement();
                    base.ClearConnection();
                }
            }
            catch
            {
               // ver1.5 エラートレース日本語対応
                csCheckInf.AddCheckErrorData(this.GetID(), new HySObjectKind("McFigureUnity"), Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION );
               // csCheckInf.AddCheckErrorData(this.GetID(), new HySObjectKind("McFigureUnity"), "catch exception");
                bRtn = false;
            }
            return bRtn;
        }

        // ====================================
        // 演算実行
        // ====================================
        /// <summary><para>method outline:</para>
        /// <para>シミュレーションの初期化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdInitializeSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean CmdInitializeSimulation(HySSysEvent csEvent)
        {
            //★サーバー／クライアント化で追加
            // ★　本来ならば、頻繁にコンテナインスタンスを修正すべきではない。
            //     変更のあった部分のみを　反映させて、インスタンス本体は入れ替えない。（将来の検討課題）
            HySSimulationDataContainer csRcvContainer = ((HySEventObject)csEvent).GetData() as HySSimulationDataContainer;
            if (csRcvContainer != null)
            {
                this.SetContainer(csRcvContainer); // データコンテナ設定
            }
            //★サーバー／クライアント化で追加

            bool bRtn = false;

            this.m_csRtnMessage.SetChar("");
            this.CompleteCalculation();
            //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity","CmdInitializeSimulation", "start");
            
            //this.Prepare();  //（接続作成時で既に行っている。毎回行う必要は無い）
            base.ResetTranInfo(); // 系内の伝送データをリセットする
            //McLog.Reset(); // ログファイルのリセット ←  ログのリセットをここにいれてよいか？ 業務側で行うべきではないか？

            // ログ情報の初期化
            McModelLogData csModelLog = (McModelLogData)m_csDataContainer.GetData(McDefine.HYM_DATA_LOG_INFO);
            csModelLog.ClearLogDataList();      // ログデータリスト
            McLog.SetActiveModelLogData(csModelLog);// 画面表示用　アクティブ化

            // 共通情報の登録　インスタンス識別子と計算ロットの登録
            //HySTime csNowTm = HySCalendar.CreateNowTime();
            // 計算ロットの生成
            DateTime csDT = DateTime.Now;
            int iUp = csDT.Year * 10000 + csDT.Month * 100 + csDT.Day;
            int iMd = csDT.Hour * 100 + csDT.Minute;
            int iLw = csDT.Minute * 1000 + csDT.Millisecond;
            string sLotName = iUp.ToString() + "_" + iMd.ToString() + "_" + iLw.ToString(); 
                //= csDT.Year.ToString() + csDT.Month.ToString() + csDT.Day.ToString() + "_" + csDT.Hour.ToString() + csDT.Minute.ToString() +"_"+ csDT.Second.ToString()+csDT.Millisecond.ToString();
            //HySString csCalLot = new HySString(sLotName);
            HySCommonInfoHash.SetCorrespond(McDefine.CALCULATION_LOT_NAME, this.m_csID.ToString(), sLotName); // 対応登録を行う
            //HySCommonInfoHash.SetCorrespond(McDefine.CALCULATION_LOT_NAME, this.m_csID.ToString(), csCalLot.ToString()); // 対応登録を行う

            // モデルチェック情報のクリア
            McStructCheckData csErrDt = m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO) as McStructCheckData;
            if (csErrDt != null)
            {
                csErrDt.ClearCheckData();
            }
            // 演算順序の設定
            this.CheckCalculationOrder();

            // 内部モデルへの初期化指示
            if (m_csCalModel != null)
            {
                try
                {
                    //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
                    // Ver1.3追加（収束計算に必要な出力を強制） 
                    //  収束演算が必要ないモデルは InitialDataFusionForConvergentCalculation() メソッド内では何もしない。
                    //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
                    m_csCalModel.InitialDataFusionForConvergentCalculation(0.0);

                    McPropertyInfoRoot csInitData = new McInitialInfo(new HySID(""), new HySObjectKind(""));
                    bRtn = m_csCalModel.Initialize(ref csInitData, ref m_csReceiveInfoList, ref m_csSendInfoList);
                    if (bRtn == true)
                    {
                        m_lCalStatus = McGrElementCtl.INITIAL_STATE;
                    }
                    else
                    {
                        McStructErrorInfo csErrorInfData = (McStructErrorInfo)m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO);
                       // ver1.5 エラートレース日本語対応
                        csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"), 
                             Properties.HymcoImplResources.STATEMENT_MODEL_INIT_NG );
                       // csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"), "  Model Inisialize fail.");
                       // ver1.5 エラートレース日本語対応
                        HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdInitializeSimulation", 
                            Properties.HymcoImplResources.STATEMENT_INIT_NG_ELEMENT );
                       // HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdInitializeSimulation", "Initialize fail.(Caused Inner Elements)");
                    }
                }
                catch (Exception ex)
                {
                    McStructErrorInfo csErrorInfData = (McStructErrorInfo)m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO);
                   // ver1.5 エラートレース日本語対応
                    csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"), 
                        Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION_MES +ex.Message+")");
                   // csErrorInfData.AddCheckErrorData(new HySID(""), new HySObjectKind("McFigureUnity"), "  catch exception(Message=" + ex.Message + ")");
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdInitializeSimulation", 
                        Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + ex.Message);
                   // HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdInitializeSimulation", "catch Exception =" + ex.Message);

                    bRtn= false;
                }

            }
            else
            {
                bRtn= false;
            }
            HySEventObject csRtnObj = null;
            csRtnObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_INITIALIZE_SIMULATION);
            csRtnObj.SetToSimKind(this.GetSimKind());
            if (bRtn == true)
            {   csRtnObj.SetSubEventNo(McDefine.HYM_OK_SUBEVENT);   }
            else
            {
                csRtnObj.SetSubEventNo(McDefine.HYM_NG_SUBEVENT);
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdInitializeSimulation", 
                    Properties.HymcoImplResources.STATEMENT_INIT_NG );
               // HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdInitializeSimulation", "Initialize fail.");
            }
            m_csThreadCtl.PutEvent(csRtnObj);
            McLog.RemoveModelLogData();// 画面表示用　アクティブ化解除
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の開始</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdStartSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean CmdStartSimulation(HySSysEvent csEvent)
        {
            //★サーバー／クライアント化で追加
            // ★　本来ならば、頻繁にコンテナインスタンスを修正すべきではない。
            //     変更のあった部分のみを　反映させて、インスタンス本体は入れ替えない。（将来の検討課題）
            HySSimulationDataContainer csRcvContainer = ((HySEventObject)csEvent).GetData() as HySSimulationDataContainer;
            if (csRcvContainer != null)
            {
                this.SetContainer(csRcvContainer); // データコンテナ設定
            }
            //★サーバー／クライアント化で追加
            HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity", "CmdStartSimulation", "start");
            bool bRtn = false;
            this.m_csRtnMessage.SetChar("");

            McStructErrorInfo csErrorInfData = (McStructErrorInfo)m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO);
            McStructCheckData csErrDt = csErrorInfData as McStructCheckData;

            if (csErrDt != null)
            {
                csErrDt.ClearCheckData();
            }

            if (m_lCalStatus == McGrElementCtl.DEFAULT_STATE)
            {
                // 初期化開始
                bRtn = this.CmdInitializeSimulation(csEvent);
            }

            if (m_csDataContainer.GetSimuTime() != null && 
                m_csDataContainer.GetStartTime() != null && 
                m_csDataContainer.GetGoalTime() != null &&
                m_csDataContainer.GetDeltaTime() != null )
            {
                // モデルに対してシミュレーション終了時刻を設定する
                //McGrElementCtl.SetFinalTargetTime((HySTime)m_csDataContainer.GetGoalTime());
                ((McGrElementCtl)m_csCalModel).SetFinalTargetTime((HySTime)m_csDataContainer.GetGoalTime());
                
                HySTime csSimuTime = (HySTime)m_csDataContainer.GetSimuTime();
                // 非同期演算の場合のタイムステップ設定
                HySTime csDltTime = null;
                McModelInfo csModelInfo = m_csStructInfo.GetModelInfo();
                if (csModelInfo != null)
                {
                    McFigUnityPropertyInfo csInfo = (McFigUnityPropertyInfo)csModelInfo.GetPropertyInfo();
                    if (csInfo.GetDataFusionTiming() == McDefine.DataFusionTiming.ASYNCHRONOUS)
                    {
                        // 非同期の場合、プロパティ画面より設定されたタイムステップ値を設定
                        csDltTime = ((McFigUnityPropertyInfo)csInfo).GetTimeStep();
                    }
                    else
                    {
                        // 同期の場合、奨励δT値を設定する
                        csDltTime = (HySTime)m_csDataContainer.GetDeltaTime();
                    }
                }
                m_csCalModel.SetNextTargetTime(ref csSimuTime, ref csDltTime);
                m_csCalModel.SetTargetTime((HySTime)m_csDataContainer.GetGoalTime());  // <-- m_csCalModel.SetNextTargetTime()にて自身のゴールが書き換えられている為再設定する（重要）
                m_csCalModel.ChangeDeltaTimeAutomatically();
                // 演算チェック(此処では必要なし）
                bRtn = true;
                //bRtn = m_csCalModel.IsAbleToCalculate(ref m_csReceiveInfoList, ref m_csSendInfoList,ref csErrorInfData);
                HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity", "CmdStartSimulation", "演算チェック(此処では必要なし）m_csCalModel.IsAbleToCalculate()");
            }
            else
            {
                bRtn = false;
               // ver1.5 エラートレース日本語対応
                csErrorInfData.AddSimuErrorData(m_csDataContainer.GetID(), McDefine.SIMULATOR_KIND, 
                     Properties.HymcoImplResources.STATEMENT_DATACONTAINER_NO_TIME );
               // csErrorInfData.AddSimuErrorData(m_csDataContainer.GetID(), McDefine.SIMULATOR_KIND, "DataContainer has no time");
                HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity", "CmdStartSimulation", "DataContainer has no time");
            }

            if (bRtn == false)
            {   // 演算開始不可を通知
                this.NoticeUnableStartSimulation();
            }
            else
            {   // 演算開始
                m_lCalStatus = McGrElementCtl.CALCULATION_STATE;
                bRtn = CmdRestartSimulation(csEvent);
                if (bRtn != true)
                {   // 異常発生ならば
                    //this.NoticeCalculationAbend();
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の中断</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdPauseSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean CmdPauseSimulation(HySSysEvent csEvent)
        {
            //★サーバー／クライアント化で追加
            // ★　本来ならば、頻繁にコンテナインスタンスを修正すべきではない。
            //     変更のあった部分のみを　反映させて、インスタンス本体は入れ替えない。（将来の検討課題）
            HySSimulationDataContainer csRcvContainer = ((HySEventObject)csEvent).GetData() as HySSimulationDataContainer;
            if (csRcvContainer != null)
            {
                this.SetContainer(csRcvContainer); // データコンテナ設定
            }
            //★サーバー／クライアント化で追加

            HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity","CmdPauseSimulation", "start");
            this.m_csRtnMessage.SetChar("");

            if (m_lCalStatus == McGrElementCtl.CALCULATION_STATE || m_lCalStatus == McGrElementCtl.RESTART_STATE || m_lCalStatus == McGrElementCtl.END_STATE)
            {
                this.PauseCalculation();   // 内部演算中断処理

                m_lCalStatus = McGrElementCtl.PAUSE_STATE;
            }

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の再開</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdRestartSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean CmdRestartSimulation(HySSysEvent csEvent)
        {
            //★サーバー／クライアント化で追加
            // ★　本来ならば、頻繁にコンテナインスタンスを修正すべきではない。
            //     変更のあった部分のみを　反映させて、インスタンス本体は入れ替えない。（将来の検討課題）
            HySSimulationDataContainer csRcvContainer = ((HySEventObject)csEvent).GetData() as HySSimulationDataContainer;
            if (csRcvContainer != null)
            {
                this.SetContainer(csRcvContainer); // データコンテナ設定
            }
            //★サーバー／クライアント化で追加

            HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity","CmdRestartSimulation", "start");
            long lLclRtn = 0;
            bool bLclRtn = true;
            bool bRtn = true;
            this.m_csRtnMessage.SetChar("");
            //McConvergenceInfo csConvergeCondInfo=null; // ダミー

            // ログ表示の画面設定（アクティブな画面に設定）
            McModelLogData csModelLog = (McModelLogData)m_csDataContainer.GetData(McDefine.HYM_DATA_LOG_INFO);
            McLog.SetActiveModelLogData(csModelLog);// 画面表示用　アクティブ化

            // モデルに対してシミュレーション終了時刻を設定する
            this.SetTargetTime((HySTime)m_csDataContainer.GetGoalTime());
            //McGrElementCtl.SetFinalTargetTime((HySTime)m_csDataContainer.GetGoalTime());
            ((McGrElementCtl)m_csCalModel).SetFinalTargetTime((HySTime)m_csDataContainer.GetGoalTime());

            // 設定
            if (m_lCalStatus == McGrElementCtl.PAUSE_STATE)
            {   // 中断中ならば
                this.RestartCalculation(); // 演算再開をモデルに設定
            }
            else // else if (m_lCalStatus == McGrElementCtl.INITIAL_STATE)
            {
                bLclRtn = this.ReadyCalculation();// 
                if (bLclRtn == false)
                {
                    bRtn = false;
                    this.CompleteCalculation();
                    this.m_csThreadCtl.SetSuppMessage(this.m_csRtnMessage);
                    this.NoticeCalculationAbend();
                    return bRtn;//     構造化プログラムの要件からは、リターンがこの様なところにあるのは望ましくない
                }
            }

            this.NoticeRestartSimulation();

            //bool bFirstCalcFlg = false; // 初回特別計算必要か？
            //if (this.GetDataFusionTiming() == McDefine.DataFusionTiming.ASYNCHRONOUS)
            //{   // 非同期型の場合
            //    bFirstCalcFlg = true;  // 初回特別計算必要
            //}
            for (;;)
            {   // ∞計算

                // ●目標時刻に達したならば終了
                if (this.IsConverged( ) == true)
                {   // 目標時刻まで到達したならば
                    bLclRtn = this.CompleteCalculation();
                    if (bLclRtn == false)
                    {
                        bRtn = false;
                        this.m_csThreadCtl.SetSuppMessage(this.m_csRtnMessage);
                        this.NoticeCalculationAbend();
                    }
                    else
                    {
                        this.NoticeCompleteCalculation();
                    }
                    m_lCalStatus = McGrElementCtl.END_STATE;
                    break;
                }

                // ●計算を行わせる
                lLclRtn = this.Calculate();  // 計算を行う
                //if (bFirstCalcFlg == true)
                //{   // 初回特別計算必要ならば
                //    lLclRtn = this.FirstCalculate(); // 初回特別計算を行う
                //    bFirstCalcFlg = false;
                //}
                //else
                //{   // 初回特別でない（通常の場合)
                //    lLclRtn=this.Calculate();  // 通常計算を行う
                //}


                if (lLclRtn != McDefine.CALCULATION_NORMAL_RETURN)
                {   // 演算に異常発生の場合
                    bRtn = false;
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdRestartSimulation", 
                        Properties.HymcoImplResources.STATEMENT_CALC_ERR + lLclRtn.ToString() );
                   // HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "CmdRestartSimulation", "Calculation_ERROR Error_No = " + lLclRtn.ToString());
                    //if (lLclRtn == McDefine.CALCULATION_ABEND_RETURN_BECAUSE_OF_STRUCT_ERROR)
                    //{
                    //    this.m_csRtnMessage.SetChar(" because of wrong model structure. For example \"Loop connection exist.\"");
                    //}
                    this.m_csThreadCtl.SetSuppMessage(this.m_csRtnMessage);
                    this.NoticeCalculationAbend();
                    break;
                }

                // ●時刻を進める
                this.GainSimulationTime();
                m_csDataContainer.SetSimuTime(this.m_csCalModel.GetSimulationTime()); // コンテナに対してシミュレーション時刻を更新する

                //System.Threading.Thread.Sleep(0);   // スレッドの隙間を空けるため（スレッド関係の処理は非常に時間を消費する為、入れるない方が良い）

                if (m_lCalStatus == McGrElementCtl.PAUSE_STATE)
                {
                    //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity","CmdRestartSimulation", "m_lCalStatus=PAUSE_STATE");
                    bLclRtn = this.SuspendCalculation();
                    if (bLclRtn == false)
                    {
                        bRtn = false;
                        this.m_csThreadCtl.SetSuppMessage(this.m_csRtnMessage);
                        this.NoticeCalculationAbend();
                    }
                    else
                    {
                        this.NoticePauseSimulation();
                    }
                    break;
                }
                else if (m_lCalStatus == McGrElementCtl.STOP_STATE)
                {
                    //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity","CmdRestartSimulation", "m_lCalStatus=STOP_STATE");
                    this.NoticeStopSimulation();
                    break;
                }

                // ●δＴを自動変更する場合がある
                this.ChangeDeltaTimeAutomatically();

                // 中間報告  ＜－－－スケジューラーで計算する時（複数のスケジューラーが動作するとき）には、この処理を入れておくと良くない
                // ↓↓　　＝＝＝自主報告有り無し設定が出来る様に　SimulatorRoot にメソッドを追加しておく
                //{
                //    long lPrgress = this.GetSimulationProgress(csEvent);
                //    HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity::CmdRestartSimulation", "    Progress = " + lPrgress.ToString() + "(%)");
                //    HySEventObject csRtnEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_REPORT_SIMU_PROGRESS, lPrgress);
                //    m_csThreadCtl.PutEvent(csRtnEvent);
                //}
                // ↑↑

            }
            HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity","CmdRestartSimulation", "end");
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の終了</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdStopSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean CmdStopSimulation(HySSysEvent csEvent)
        {
            HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McFigureUnity","CmdStopSimulation", "start");
            this.m_csRtnMessage.SetChar("");

            this.StopCalculation();   // 内部演算終了処理

            m_lCalStatus = McGrElementCtl.STOP_STATE;
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の進捗を取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lPrg = GetSimulationProgress( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>進捗（０～１００％）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetSimulationProgress(HySSysEvent csEvent)
        {
            long lRtn = 0;
            McGrElementCtl csMdl = m_csCalModel as McGrElementCtl;
            if (csMdl != null)
            {
                lRtn = csMdl.GetSimulationProgress();
            }
            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>任意イベント</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdArbitraryEvent( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean CmdArbitraryEvent(HySSysEvent csEvent)
        {
            //★サーバー／クライアント化で追加
            // ★　本来ならば、頻繁にコンテナインスタンスを修正すべきではない。
            //     変更のあった部分のみを　反映させて、インスタンス本体は入れ替えない。（将来の検討課題）
            HySSimulationDataContainer csRcvContainer = ((HySEventObject)csEvent).GetData() as HySSimulationDataContainer;
            if (csRcvContainer != null)
            {
                this.SetContainer(csRcvContainer); // データコンテナ設定
            }
            //★サーバー／クライアント化で追加

            Boolean bRtn = false;
            //long lEventNo = ((HySEventObject)csEvent).GetEventNo();
            long lSubEventNo = ((HySEventObject)csEvent).GetSubEventNo();
            //HySID csID = ((HySEventObject)csEvent).GetSuppID();
            //HySObjectKind csSimKind = (HySObjectKind)((HySEventObject)csEvent).GetSimKind();
            if (lSubEventNo == McDefine.MODEL_PROPERTY_SET)
            {   // モデル/伝送情報プロパティ設定の場合
                HySID csElmID = (HySID)((HySEventObject)csEvent).GetSubData();  // 要素ID取得
                McModelInfo csMdlInfo = (McModelInfo)((HySEventObject)csEvent).GetData(); // 設定情報取得
                McStructErrorInfo csCheckInf = m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO) as McStructErrorInfo;
                //bRtn = this.SetProperty(csElmID, csMdlInfo, ref csCheckInf);
                bRtn = this.SetOnlineProperty(csElmID, csMdlInfo, ref csCheckInf); // 実行中断中のモデルプロパティ等設定
            }
            else if (lSubEventNo == McDefine.HYM_STRUCT_FILE_WRITE_REQUEST)
            { // 構造定義ファイル書き出しならば
                HySString sSCFNameWithPath = ((HySEventObject)csEvent).GetData() as HySString;
                if ((object)sSCFNameWithPath != null)
                {
                    HySEventObject csRtnObj = null;

                    // ファイルパス取得
                    /*
                    string sFileNamePath = sSCFNameWithPath.ToString();
                    int lLstPathIdx = -1; int lTmpLstPathIdx = -1;
                    for (long lStr = 0; lStr < sSCFNameWithPath.GetLength(); lStr++)
                    {
                        lTmpLstPathIdx = sFileNamePath.IndexOf("\\", lLstPathIdx + 1);
                        if (lTmpLstPathIdx >= 0)
                        { lLstPathIdx = lTmpLstPathIdx; }
                        else
                        { break; }
                    }
                    string sFileName = sFileNamePath.Substring(lLstPathIdx + 1);
                    string sFileNameWithoutSfx = sFileName.Substring(0, sFileName.Length - 4);
                    string sPath = sFileNamePath.Substring(0, lLstPathIdx);
                    */
                    /*
                    string sFileName = "";
                    string sFileNameWithoutSfx = "";
                    string sPath = "";
                    GetFileNameString(sSCFNameWithPath, ref sFileName, ref sFileNameWithoutSfx, ref sPath);
                    // プロジェクトファイル名
                    HySString csPrjctFileName = new HySString(sPath + "\\" + sFileNameWithoutSfx + "ProjectFile.xml");
                    // イニシャルファイル名
                    HySString csInitialFileName = new HySString(sPath + "\\" + sFileNameWithoutSfx + "Initial.xml");
                    // プロパティファイル名
                    HySString csPrptyFileName = new HySString(sPath + "\\" + sFileNameWithoutSfx + "Property.xml");

                    // プロパティファイル名称等設定
                    m_csStructInfo.SetPropertyXmlFileName(new HySString("" + sFileNameWithoutSfx + "Property.xml"));
                    m_csStructInfo.SetInitialXmlFileName(new HySString("" + sFileNameWithoutSfx + "Initial.xml"));
                    */

                    HySString csPrjctFileName = new HySString(""); // プロジェクトファイル名（パス有り）
                    HySString csInitialFileName = new HySString(""); // 初期化ファイル名（パス有り）
                    HySString csPrptyFileName = new HySString(""); // プロパティファイル名（パス有り）

                    HySString csPath=new HySString(""); // ファイルへのパス
                    HySString csSCFName = new HySString(""); // 構造定義ファイル名（パス無し）
                    HySString csPrptyName = new HySString(""); // プロジェクトファイル名（パス無し）
                    HySString csInitialName = new HySString(""); // 初期化ファイル名（パス無し）
                    HySString csPrjctName = new HySString(""); // プロパティファイル名（パス無し）
                    try
                    {
                        // 各種ファイル名生成
                        m_csStructInfo.MakeCSFName(sSCFNameWithPath, ref csPath, ref csSCFName, ref csPrptyName, ref csInitialName, ref csPrjctName);
                        // 初期化及びプロパティファイル名設定
                        m_csStructInfo.SetPropertyXmlFileName(csPrptyName);
                        m_csStructInfo.SetInitialXmlFileName(csInitialName);

                        // 初期化、プロパティ、プロジェクト　のパス付きファイル名を作成する
                        HySString csSprtStr = new HySString("\\");
                        csPrjctFileName = csPath + csSprtStr + csPrjctName;
                        csInitialFileName = csPath + csSprtStr + csInitialName;
                        csPrptyFileName = csPath + csSprtStr + csPrptyName;

                        // 構造定義ファイル本体　.XML
                        HySXmlWriter csOutFile = new HySXmlWriter(sSCFNameWithPath);

                        // XmlWriterによる処理 -->
                        HySXmlNode csNode = m_csStructInfo.WriteStructureConfiguration(ref csOutFile);
                        csOutFile.Save();

                        if (McPropertyInfoBase.GetXmlOutFlg() == McDefine.HYMCO_MODEL_STRUCTINFO_WRITE_SEPARATE)
                        {
                            // プロパティファイル　.XML 生成
                            HySXmlWriter csPropertyXml = new HySXmlWriter(csPrptyFileName);
                            csNode = m_csStructInfo.WritePropertyConfiguration(ref csPropertyXml);
                            csPropertyXml.Save();

                            // イニシャルファイル　.XML 生成
                            HySXmlWriter csInitialXml = new HySXmlWriter(csInitialFileName);
                            csNode = m_csStructInfo.WriteInitialConfiguration(ref csInitialXml);
                            csInitialXml.Save();
                        }

                        // プロジェクトファイル  .XML 生成
                        HySXmlWriter csPrjctXml = new HySXmlWriter(csPrjctFileName);
                        //this.ProjectXML(ref csPrjctXml, csSCFName);
                        m_csStructInfo.ProjectXML(ref csPrjctXml, csSCFName, (HySTime)m_csDataContainer.GetStartTime(), (HySTime)m_csDataContainer.GetGoalTime(), (HySTime)m_csDataContainer.GetDeltaTime());
                        csPrjctXml.Save();
                    }
                    catch(Exception ex )
                    {
                       // ver1.5 エラートレース日本語対応
                        HySLog.LogOut(HySLog.ONLINE, "McFigureUnity", "lSubEventNo==McDefine.HYM_STRUCT_FILE_WRITE_REQUEST", 
                            Properties.HymcoImplResources.STATEMENT_CATCH_EXCEPTION + ex.Message);
                       // HySLog.LogOut(HySLog.ONLINE, "McFigureUnity", "lSubEventNo==McDefine.HYM_STRUCT_FILE_WRITE_REQUEST", "catch exception : " + ex.Message);
                    }
                    // XmlWriterによる処理 <--

                    csRtnObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_SIMULATION_OK, this.m_csID);
                    csRtnObj.SetSubEventNo(McDefine.HYM_NOTICE_STRUCT_FILE_WRITE_REQUEST);
                    csRtnObj.SetToSimKind(this.GetSimKind());
                    m_csThreadCtl.PutEvent(csRtnObj);
                }
            }
            else if (lSubEventNo == McDefine.HYM_STRUCT_FILE_READ_REQUEST)
            { // 構造定義ファイル読み込みならば
                HySString sSCFNameWithPath = ((HySEventObject)csEvent).GetData() as HySString;
                if ((object)sSCFNameWithPath != null)
                {
                    HySFile csFile = new HySFile(sSCFNameWithPath);
                    if (csFile.Exist() == true)
                    {
                        HySID csOrgID = m_csStructInfo.GetID() as HySID;// 構造定義ファイル読み込みにより、IDを変更したく無い為、最初のIDを保持しておく
                        HySXmlReader csReader = new HySXmlReader(sSCFNameWithPath);
                        HySID csWorkID;
                        {   // m_csStructInfo をXML復元中に必要なプロジェクトグループ名を　HySCommonInfoHash　へ登録する
                            McStructInfo csWorkStruct = new McStructInfo();
                            csWorkID = csWorkStruct.GetXmlStructID(csReader, m_csModelFactory);
                            McProjectInfData csProjectInfData = m_csDataContainer.GetData(McDefine.HYM_DATA_PROJECT_INFO) as McProjectInfData;
                            HySString csProjectGroupName = csProjectInfData.GetProjectGroupName(); //csProjectInfData から　プロジェクトグループを取得する
                            HySCommonInfoHash.SetCorrespond(McDefine.PROJECT_GROUP_NAME, csWorkID.ToString(), csProjectGroupName.ToString()); // 対応登録を行う

                            csReader = new HySXmlReader(sSCFNameWithPath);
                        }
                        bRtn = m_csStructInfo.ReadStructureConfiguration(csReader, m_csModelFactory);
                        m_csStructInfo.SetID(csOrgID);
                        {   // m_csStructInfo をXML復元中に必要なプロジェクトグループ名の　HySCommonInfoHash　登録を解除する（後始末）
                            HySCommonInfoHash.RemoveInfo(McDefine.PROJECT_GROUP_NAME, csWorkID.ToString());
                        }



                        HySString csPrptyFileName = m_csStructInfo.GetPropertyXmlFileName();
                        HySString csInitialFileName = m_csStructInfo.GetInitialXmlFileName();

                        string sFileName = "";
                        string sFileNameWithoutSfx = "";
                        string sPath = "";
                        //GetFileNameString(sSCFNameWithPath, ref sFileName, ref sFileNameWithoutSfx, ref sPath);
                        m_csStructInfo.GetFileNameString(sSCFNameWithPath, ref sFileName, ref sFileNameWithoutSfx, ref sPath);

                        if ((object)csPrptyFileName != null)
                        {   // プロパティ定義ファイル有りならば
                            csPrptyFileName.SetChar(sPath + "\\" + csPrptyFileName.ToString());
                            csReader = new HySXmlReader(csPrptyFileName);
                            m_csStructInfo.ReadPropertyConfiguration(csReader, m_csModelFactory);
                        }
                        if ((object)csInitialFileName != null)
                        {   // 初期設定ファイルありならば
                            csInitialFileName.SetChar(sPath + "\\" + csInitialFileName.ToString());
                            csReader = new HySXmlReader(csInitialFileName);
                            m_csStructInfo.ReadInitialConfiguration(csReader, m_csModelFactory);
                        }

                    }
                    HySEventObject csRtnObj = null;
                    if (bRtn == true)
                    {
                        csRtnObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_SIMULATION_OK, this.m_csID);
                    }
                    else
                    {
                        csRtnObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_SIMULATION_NG, this.m_csID);
                    }
                    csRtnObj.SetSubEventNo(McDefine.HYM_NOTICE_STRUCT_FILE_READ_REQUEST);
                    csRtnObj.SetToSimKind(this.GetSimKind());
                    m_csThreadCtl.PutEvent(csRtnObj);
                }
            }
            else if (lSubEventNo == McDefine.REQ_OPTIMIZE_RECORD_NUMBER)
            { // 要素モデル接続線レコード数の自動調整開始指示ならば
                bRtn = OptimizeTranInfoRecord((HySEventObject)csEvent);
            }
            else
            {
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算完了したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeCompleteCalculation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        /// 
        protected virtual void NoticeCompleteCalculation()
        {
            //this.CompleteCalculation();
            m_csThreadCtl.NoticeCompleteCalculation();
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算が異常終了したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeCalculationAbend(csEvent) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticeCalculationAbend()
        {
            m_csThreadCtl.NoticeCalculationAbend();
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算を開始不可という報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeUnableStartSimulation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticeUnableStartSimulation()
        {
            HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_CANNOT_START_SIMULATION);
            this.NoticeArbitraryEvent(csEvent);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算を中断したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticePauseSimulation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticePauseSimulation()
        {
            HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_PAUSE_SIMULATION);
            this.NoticeArbitraryEvent(csEvent);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算を開始したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeRestartSimulation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticeRestartSimulation()
        {
            HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_START_SIMULATION);
            this.NoticeArbitraryEvent(csEvent);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算を終了モードにしたという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeStopSimulation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticeStopSimulation()
        {
            HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_STOP_SIMULATION);
            this.NoticeArbitraryEvent(csEvent);
        }

        /// <summary><para>method outline:</para>
        /// <para>任意イベントを送る</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeArbitraryEvent( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送るイベント</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>自分からイベントを送信する時に使用する</para>
        /// </remarks>
        protected virtual void NoticeArbitraryEvent(HySSysEvent csEvent)
        {
            m_csThreadCtl.PutEvent(csEvent);
        }
        /// <summary><para>method outline:</para>
        /// <para>計算開始時に動作する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReadyCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool ReadyCalculation()
        {
            bool bRtn = true;
            if (m_bCalSatrtNum < 1)
            {   // 計算開始していなければ
                bRtn = m_csCalModel.ReadyCalculation();
                m_bCalSatrtNum += 1;
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>計算終了時に動作する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CompleteCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool CompleteCalculation()
        {
            bool bRtn = true;
            if (m_bCalSatrtNum >0 )
            {
                bRtn =  m_csCalModel.CompleteCalculation();
                m_bCalSatrtNum = 0; // 計算開始コール済みフラグリセット
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算中断時に動作する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SuspendCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SuspendCalculation()
        {
            bool bRtn = true;
            if (m_bCalSatrtNum > 0)
            {
                bRtn = m_csCalModel.SuspendCalculation();
                //m_bCalSatrtNum = 0; // 計算開始コール済みフラグリセット
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター間データ交換用コンテナに演算結果を出力する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = OutputMediateData( csDataContainer ) </para>
        /// </example>
        /// <param name="csDataContainer">シミュレーター間データ交換用コンテナ</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean OutputMediateData(HySContainer csDataContainer)
        {
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター間データ交換用コンテナから他のシミュレーターの演算結果を取り入れる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = InputMediateData( csDataContainer ) </para>
        /// </example>
        /// <param name="csDataContainer">シミュレーター間データ交換用コンテナ</param>
        /// <returns>true</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean InputMediateData(HySContainer csDataContainer)
        {
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算内部情報を全て出力する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = SerializeAllInfo( csData ) </para>
        /// </example>
        /// <param name="csData">保存データ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>計算中断した状態で内容を全て保存しておく必要がある場合等に使用する</para>
        /// </remarks>
        public virtual Boolean SerializeAllInfo(HySDataRoot csData)
        {
            Boolean Brtn = true;
            if (csData != null && csData as HySFileOutData != null)
            {
                // データコンテナの積み込み
                ((HySFileOutData)(csData)).SetContainer(m_csDataContainer);
                // 計算状態の保存
                McCmnElementOutData csFigureUnityOutData = this.CreateOutData();
                Brtn = this.FileOUT(csFigureUnityOutData);
                // 演算データの積み込み
                ((HySFileOutData)(csData)).SetSimulatorData(csFigureUnityOutData);
            }
            else
            {
                // データクラスがnullまたは正しくない
                return false;
            }
            return Brtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算内部情報を全て入力・復元する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = StructureAllInfo( csData ) </para>
        /// </example>
        /// <param name="csData">保存データ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>計算中断した状態で内容を全て保存しておく必要がある場合等に使用する</para>
        /// </remarks>
        public virtual Boolean StructureAllInfo(HySDataRoot csData)
        {
            Boolean Brtn = true;
            if (csData != null && csData as HySFileOutData != null)
            {
                // データコンテナの復元・設定
                this.SetContainer(((HySFileOutData)(csData)).GetContainer());
                // 復元対象プロジェクトのモデル構築状態を確認
                McSimulationStatus csSimuStatus = (McSimulationStatus)m_csDataContainer.GetData(McDefine.HYM_DATA_SIMULATION_INFO);
                if (csSimuStatus != null)
                {
                    if (csSimuStatus.GetModelConStatus() == McSimulationStatus.ModelConstructionStatus.CONSTRUCTED)
                    {
                        McStructInfo csStructInfo = m_csDataContainer.GetData(McDefine.HYM_DATA_STRUCT_INFO) as McStructInfo;
                        McStructCheckData csCheckInf = m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO) as McStructCheckData;
                        // この段階で、保存する際に表示されていたモデルチェック情報があるため、この先モデル構築する際に
                        // チェック情報が増えてしまう。従って、ここでクリアする。
                        csCheckInf.ClearCheckData();
                        McStructErrorInfo csErrorInfo = csCheckInf as McStructErrorInfo;
                        // モデル構築されていたならば、モデル構築し、初期設定等全て、通常の計算と同じ処理を行っておく必要がある。
                        //this.ModelConstruction(null);
                        this.ModelConstruction(csStructInfo, m_csModelFactory, ref csCheckInf);
                        this.SetSimDataContainer(m_csDataContainer);
                        // モデル構築処理にて行っているため不要（モデルチェック情報も増えてしまう）
                        //base.Prepare(ref csErrorInfo);   // 演算準備設定
                        //this.SetSimuStartTime(m_csDataContainer.GetStartTime()); m_csCalModel.SetStartTime((HySTime)m_csDataContainer.GetStartTime());
                        //this.SetSimuGoalTime(m_csDataContainer.GetGoalTime()); m_csCalModel.SetTargetTime((HySTime)m_csDataContainer.GetGoalTime());
                        //this.SetSimuTime(m_csDataContainer.GetSimuTime()); m_csCalModel.SetSimulationTime((HySTime)m_csDataContainer.GetSimuTime());
                        //this.SetSimuDeltaTime(m_csDataContainer.GetDeltaTime()); m_csCalModel.SetDeltaTime((HySTime)m_csDataContainer.GetDeltaTime());
                        //this.CheckCalculationOrder();
                        //this.MakeContinuity(); // 中継端子の導通設定を行う
                        //m_csCalModel.IsAbleToCalculate(ref m_csReceiveInfoList, ref m_csSendInfoList, ref csErrorInfo);
                    }
                }
                else
                {
                    // シミュレーション状態が取得できない
                    return false;
                }
                // 演算データの復元・計算状態の復元
                Brtn = this.FileIN(((HySFileOutData)(csData)).GetSimulatorData());
            }
            else
            {
                // データクラスがnullまたは正しくない
                return false;
            }

            return Brtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算状態復元のためのデータクラスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McCmnElementOutData csElementOutData = CreateOutData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>データクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McCmnElementOutData CreateOutData()
        {
            return new McFigureUnityOutData();
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーションデータコンテナの内容を設定する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimDataContainerInfo( csDataContainer )</para>
        /// </example>
        /// <param name="csDataContainer">シミュレーションデータコンテナ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetSimDataContainer(HySSimulationDataContainer csDataContainer)
        {
            base.SetSimDataContainer(csDataContainer);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーションIDを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimulationID(csSimulationID)</para>
        /// </example>
        /// <param name="csSimulationID">シミュレーションID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetSimulationID(HySIdentifier csSimulationID)
        {
            base.SetSimulationID(csSimulationID);
        }

        /// <summary><para>method outline:</para>
        /// <para>計算状態の保存処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileOUT(csData) </para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileOUT(HySDataRoot csData)
        {
            bool bRtn = base.FileOUT(csData);
            if (csData != null && csData as McFigureUnityOutData != null)
            {
                // 自メンバの保存処理
                ((McFigureUnityOutData)(csData)).SetStartTime(m_csStartTime);
                ((McFigureUnityOutData)(csData)).SetCalStatus(m_lCalStatus);

                string sLotName = "";
                if (HySCommonInfoHash.GetCorrespond(McDefine.CALCULATION_LOT_NAME, this.m_csID.ToString(), ref sLotName) == true)
                {// ロット情報有り
                    ((McFigureUnityOutData)(csData)).SetLotName(sLotName);
                }
            }
            else
            {
                return false;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算状態の復元処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileIN(csData)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileIN(HySDataRoot csData)
        {
            bool bRtn = base.FileIN(csData);
            if (csData != null && csData as McFigureUnityOutData != null)
            {
                // 自メンバの復元処理
                m_csStartTime = ((McFigureUnityOutData)(csData)).GetStartTime();
                m_lCalStatus = ((McFigureUnityOutData)(csData)).GetCalStatus();

                string sLotName = ((McFigureUnityOutData)(csData)).GetLotName();
                if (sLotName.Equals("") != true)
                {// ロット情報有り
                    HySCommonInfoHash.SetCorrespond(McDefine.CALCULATION_LOT_NAME, this.m_csID.ToString(), sLotName); // 対応登録を行う
                }
                // 復元後の整合性確保
                if (m_csStartTime != null)
                {
                    m_csStartTime.PostDeserialize();
                }
            }
            else
            {
                return false;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算強制終了</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>StopCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void StopCalculation()
        {
            // 初期化状態及び中断の場合に終了通知を発行する
            if (m_lCalStatus == McGrElementCtl.INITIAL_STATE ||
                m_lCalStatus == McGrElementCtl.PAUSE_STATE)
                //m_lCalStatus == McGrElementCtl.PAUSE_STATE ||
                //m_lCalStatus == McGrElementCtl.END_STATE)
            {   // 初期化状態及び中断の場合
                this.NoticeStopSimulation();
            }

            m_lCalStatus = McGrElementCtl.STOP_STATE;
            base.StopCalculation();
        }

        /// <summary><para>method outline:</para>
        /// <para>計算中断</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>PauseCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void PauseCalculation()
        {
            m_lCalStatus = McGrElementCtl.PAUSE_STATE;
            base.PauseCalculation();
        }

        /// <summary><para>method outline:</para>
        /// <para>計算再開</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>RestartCalculation( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void RestartCalculation()
        {
            this.ReadyCalculation();
            m_lCalStatus = McGrElementCtl.RESTART_STATE;
            base.RestartCalculation();
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ等情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SetProperty( csElmID, csMdlInfo, ref csErrorInfo)</para>
        /// </example>
        /// <param name="csElmID">変更対象要素識別子</param>
        /// <param name="csMdlInfo">変更情報</param>
        /// <param name="csErrorInfo">接続エラー情報</param>
        /// <returns>=true:変更対象であった、!=true:変更対象ではない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中にプロパティ、初期値等を変更して動作させ場合等に使用する</para>
        /// </remarks>
        internal override bool SetProperty(HySID csElmID, McModelInfo csMdlInfo, ref McStructErrorInfo csErrorInfo)
        {
            bool bRtn = false;
            if (csElmID != null && csMdlInfo !=null && csErrorInfo != null)
            {
                bRtn = base.SetProperty(csElmID, csMdlInfo, ref  csErrorInfo);
            }
            return bRtn;
        }
        /*
        /// <summary><para>method outline:</para>
        /// <para>構造定義ファイル出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySXmlNode csNode = ProjectXML( ref csXmlWriter,csSCFName )</para>
        /// </example>
        /// <param name="csXmlWriter">XMLファイル書き出しクラス</param>
        /// <param name="csSCFName">構造定義ファイル名</param>
        /// <returns>XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected HySXmlNode ProjectXML(ref HySXmlWriter csXmlWriter, HySString csSCFName)
        {
            HySXmlNode csRtnNode;
            // プロジェクトルート
            csRtnNode = csXmlWriter.CreateRoot(McDefine.xml_PROJECT);
            
            HySXmlNode csNode = csXmlWriter.CreateElement(McDefine.xml_PRJCT_SCF);
            csNode.SetAttribute(McDefine.xml_PRJCT_SCF_NAME, csSCFName.ToString() );
            csRtnNode.AppendChild(csNode);

            csNode = csXmlWriter.CreateElement(McDefine.xml_PRJCT_SIMULATION);
            HySXmlNode csSubNode = csXmlWriter.CreateElement(McDefine.xml_PRJCT_TIME);
            csSubNode.SetAttribute(McDefine.xml_PRJCT_START_TIME, (HySCalendar.ToString((HySTime)m_csDataContainer.GetStartTime(), HySCalendar.FORMAT.lSW_YEAR)));
            //csNode.SetAttribute(McDefine.xml_PRJCT_PAUSE_TIME, HySCalendar.ToString(m_csDataContainer.GetStartTime(), HySCalendar.FORMAT.lSW_YEAR));
            csSubNode.SetAttribute(McDefine.xml_PRJCT_GOAL_TIME, HySCalendar.ToString((HySTime)m_csDataContainer.GetGoalTime(), HySCalendar.FORMAT.lSW_YEAR));
            csSubNode.SetAttribute(McDefine.xml_PRJCT_DELTA_TIME, ((HySTime)m_csDataContainer.GetDeltaTime()).GetTime().ToString());
            csSubNode.SetAttribute(McDefine.xml_PRJCT_TIME_UNIT, "sec");
            csNode.AppendChild(csSubNode);
            csRtnNode.AppendChild(csNode);
            return csRtnNode;
        }
        */
        /*
        /// <summary><para>method outline:</para>
        /// <para>構造定義ファイル本体部出力</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = GetFileNameString(sSCFNameWithPath, ref sFileName, ref sFileNameWithoutSfx, ref sPath)</para>
        /// </example>
        /// <param name="sSCFNameWithPath">パス付きファイル名称</param>
        /// <param name="sFileName">パス無しファイル名</param>
        /// <param name="sFileNameWithoutSfx">X添え字無しファイル名</param>
        /// <param name="sPath">ファイルへのパス</param>
        /// <returns>bool =true:正常、=false:異常 </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected bool GetFileNameString(HySString sSCFNameWithPath, ref string sFileName, ref string sFileNameWithoutSfx, ref string sPath)
        {
            string sFileNamePath = sSCFNameWithPath.ToString();
            int lLstPathIdx = -1; int lTmpLstPathIdx = -1;
            for (long lStr = 0; lStr < sSCFNameWithPath.GetLength(); lStr++)
            {
                lTmpLstPathIdx = sFileNamePath.IndexOf("\\", lLstPathIdx + 1);
                if (lTmpLstPathIdx >= 0)
                { lLstPathIdx = lTmpLstPathIdx; }
                else
                { break; }
            }
            sFileName = sFileNamePath.Substring(lLstPathIdx + 1);
            sFileNameWithoutSfx = sFileName.Substring(0, sFileName.Length - 4);
            if (lLstPathIdx > -1)
            {
                sPath = sFileNamePath.Substring(0, lLstPathIdx);
            }
            return true;
        }
        */
         
        /// <summary><para>method outline:</para>
        /// <para>伝送データレコード数の最適化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>OptimizeTranInfoRecord(csEvent) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>常にtrueで返す</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool OptimizeTranInfoRecord(HySEventObject csEvent)
        {
            bool bRtn = false;
            //HySEventObject csRtnObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.CMND_SIMULATION_COMMON, this.m_csID);
            HySEventObject csRtnObj = csEvent;
            //csRtnObj.SetTo(HySSysEvent.OBJID_BUSIPROCEDURE);
            csRtnObj.SetTo(csEvent.From());
            csRtnObj.SetSuppID(this.m_csID);

            long lRtn = 0;
            this.m_csRtnMessage.SetChar("");
            McStructInfo csStructInfo = m_csDataContainer.GetData(McDefine.HYM_DATA_STRUCT_INFO) as McStructInfo;
            McStructCheckData csCheckInf = m_csDataContainer.GetData(McDefine.HYM_DATA_CHECK_INFO) as McStructCheckData;
            if (csCheckInf == null)
            {
                csCheckInf = new McStructCheckData();
                m_csDataContainer.SetData(McDefine.HYM_DATA_CHECK_INFO, csCheckInf);
            }
            csCheckInf.ClearCheckData();
            if (csStructInfo != null)
            {
                //最初にモデル生成を行い、エラーを抽出する
                bRtn = this.ModelConstruction(csStructInfo, m_csModelFactory, ref csCheckInf);
                if (bRtn == true)
                {   // モデル生成でOKならば、最適化を行う
                    csCheckInf.ClearCheckData();
                    lRtn = csStructInfo.OptimizeTranInfoRecord(m_csModelFactory);
                    if (lRtn == McDefine.CALCULATION_NORMAL_RETURN)
                    {   // 最適化OK
                       // ver1.5 エラートレース日本語対応
                        csCheckInf.AddCheckInformationData(new HySID(""), new HySObjectKind(""), 
                             Properties.HymcoImplResources.STATEMENT_OPTIMIZE_COMPLETE );
                       // csCheckInf.AddCheckInformationData(new HySID(""), new HySObjectKind(""), "Optimize TranInfo Record number complete.");
                        csRtnObj.SetSubEventNo(McDefine.ACK_OPTIMIZE_RECORD_NUMBER);
                    }
                    else if (lRtn == McDefine.CALCULATION_CANNOT_OPTIMIZE_RETURN)
                    {   // 最適化NG（収束演算系などが存在して、最適化が出来ないとき）
                       // ver1.5 エラートレース日本語対応
                        csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), 
                             Properties.HymcoImplResources.STATEMENT_NO_OPTIMIZE_RECORD );
                       // csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), "Cannot Optimize TranInfo Record number, because of Non-System-SubGroup existence.");
                        csRtnObj.SetSubEventNo(McDefine.NAK_OPTIMIZE_RECORD_NUMBER);
                    }
                    else
                    {   // 最適化時の異常
                       // ver1.5 エラートレース日本語対応
                        csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), 
                             Properties.HymcoImplResources.STATEMENT_NO_OPTIMIZE_BAD_MODEL );
                       // csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), "Cannot Optimize TranInfo Record number, because of bad ModelConstruction.");
                        csRtnObj.SetSubEventNo(McDefine.NAK_OPTIMIZE_RECORD_NUMBER);
                    }
                }
                else
                {   // モデル生成時にエラーが発生し、最適化処理に移行できなかった場合
                   // ver1.5 エラートレース日本語対応
                    csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), 
                         Properties.HymcoImplResources.STATEMENT_NO_OPTIMIZE_BAD_MODEL );
                   // csCheckInf.AddCheckErrorData(new HySID(""), new HySObjectKind(""), "Cannot Optimize TranInfo Record number, because of bad ModelConstruction.");
                    csRtnObj.SetSubEventNo(McDefine.NAK_OPTIMIZE_RECORD_NUMBER);
                }
            }
            else
            {
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "OptimizeTranInfoRecord", 
                    Properties.HymcoImplResources.STATEMENT_MCSTRUCTINFO_NOT_EXIST );
               // HySLog.LogOut(HySLog.TRIAL_RUN, "McFigureUnity", "OptimizeTranInfoRecord", "McStructInfo is not exist in the Container");
            }
            csRtnObj.SetToSimKind(this.GetSimKind());
            m_csThreadCtl.PutEvent(csRtnObj);

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算実行／中断中のプロパティ等情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CmndSetOnlineProperty( csElmID, csMdlInfo, ref csErrorInfo)</para>
        /// </example>
        /// <param name="csElmID">変更対象要素識別子</param>
        /// <param name="csMdlInfo">変更情報</param>
        /// <param name="csErrorInfo">接続エラー情報</param>
        /// <returns>=true:変更対象が有った、!=true:変更対象が無かった</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中にプロパティ、初期値等を変更して動作させ場合等に使用する</para>
        /// </remarks>
        public virtual bool CmndSetOnlineProperty(HySID csElmID, McModelInfo csMdlInfo, ref McStructErrorInfo csErrorInfo)
        {
            return this.SetOnlineProperty(csElmID, csMdlInfo, ref csErrorInfo);
        }
    }
}
