﻿// <summary>ソースコード：ＨＹＭＣＯ ライブラリ管理基底クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.Interface.HSController;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.LibraryMng;
using CommonMP.HYMCO.CoreImpl.Model;
using CommonMP.HYMCO.CoreImpl.FigureUnity;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.LibraryCtl;

namespace CommonMP.HYMCO.CoreImpl.LibraryMng
{
    /// <summary><para>class outline:</para>
    /// <para>ＨＹＭＣＯライブラリ管理基底クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/7/1][新規作成]</para>
    /// </remarks>
    public abstract class McBaseLibraryManager : McLibraryManager
    {
        #region private/protectedフィールド

        /// <summary>ファクトリクラス</summary>
        private McModelFactorySet m_csModelFactory = null;
        /// <summary>ライブラリ管理表示情報データ</summary>
        private McDispLibTreeData m_csDispLibTreeData = null;
        /// <summary>ライブラリインストール情報</summary>
        private McLibraryInstallInfoData m_csLibraryInstallInfoData = null;

        #endregion

        #region publicメソッド

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McBaseLibraryManager csObj = new McBaseLibraryManager( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McBaseLibraryManager()
            : base()
        {
        }

        /// <summary><para>method outline:</para>
        /// <para>初期化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Initialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override Boolean Initialize()
        {
            m_bWorkFlg = false;

            Boolean bRtn = false;

            // ライブラリインストール情報生成
            bRtn = this.LoadLibraryInstallInfoData();
            // モデル情報とDLLとの対応情報設定
            m_csLibraryInstallInfoData.SetCalModelInfoLinkedDll(m_csModelFactory.GetCalModelInfoLinkedDll());
            m_csLibraryInstallInfoData.SetDllIncludeCalModelInfo(m_csModelFactory.GetDllIncludeCalModelInfo());

            // 環境ファイルより取得
            HySString csFileName = HySEnvInf.GetEnvInf(new HySString("MCLIBRARY_CATEGORY_FILENAME"));
            if (IComparer.Equals(csFileName, null))
            {
                return bRtn;
            }

            //ライブラリ分類定義ファイル読み込み
            McLibConfigDataReader csLibConfigDataReader =
                McLibConfigDataReaderFactory.CreateInstanse(csFileName);

            //ライブラリ分類定義取得
            McLibCategoryData csLibCategoryData = csLibConfigDataReader.LibConfigDataLoad();
            if (csLibCategoryData != null)
            {
                //DLLから取得したすべてのModelInfoListを取得
                HySDataLinkedList csModelList = m_csModelFactory.GetCalModelInfoList();

                //モデル管理ライブラリデータ
                McModelLibraryData csModelLibraryData = new McModelLibraryData();

                //ModelInfoListを区分けし、McModelLibraryDataに追加する
                SortOutModelList(csLibCategoryData, csModelList, csModelLibraryData);

                //ライブラリ管理表示情報データ作成
                m_csDispLibTreeData = new McDispLibTreeData();

                //ライブラリ管理表示情報データに取得したデータを追加する
                m_csDispLibTreeData.SetModelLibraryData(csModelLibraryData);
                m_csDispLibTreeData.SetLibCategoryData(csLibCategoryData);
                // ライブラリ管理表示情報データにインストール情報を設定する
                m_csDispLibTreeData.SetLibraryInstallInfoData(m_csLibraryInstallInfoData);

                SetData(m_csDispLibTreeData);
                bRtn = true;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>終了処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Terminate( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Terminate()
        {
            bool bRtn = true;
            // ライブラリインストール情報と削除情報の保存を行う
            bRtn = this.SaveLibraryInstallInfoData();
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>ファクトリ管理クラス設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HyStViewer.SetFactoryManager( csFactoryManager ) </para>
        /// </example>
        /// <param name="csFactoryManager">ファクトリ管理クラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetFactoryManager(HySFactoryMng csFactoryManager)
        {
            McFigureUnityFactory csFigUnityFact = null; // (McFigureUnityFactory)csFactoryManager.GetFactory(McDefine.FACTORY_ID_SIMULATION);
            long lFctNo = csFactoryManager.GetFactoryNum();
            for (long lLp = 0; lLp < lFctNo; lLp++)
            {
                csFigUnityFact = csFactoryManager.GetFactory(lLp) as McFigureUnityFactory;
                if (csFigUnityFact != null)
                {
                    break;
                }
            }
            if (csFigUnityFact != null)
            {
                m_csModelFactory = csFigUnityFact.GetModelFactorySet();
            }
            else
            {
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.TRIAL_RUN, "McFileLibraryManager.SetFactoryManager", 
                   Properties.HymcoImplResources.STATEMENT_MCFIGUREUNITYFACTORI_NO );
               // HySLog.LogOut(HySLog.TRIAL_RUN, "McFileLibraryManager.SetFactoryManager", "McFigureUnityFactory is not found");
            }
        }

        #endregion

        #region privateメソッド
        /// <summary><para>method outline:</para>
        /// <para>モデルリスト分類処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SortOutModelList( csLibCategoryData, csModelList, csModelLibraryData ) </para>
        /// </example>
        /// <param name="csLibCategoryData">ライブラリ分類データ</param>
        /// <param name="csModelList">モデルリスト</param>
        /// <param name="csModelLibraryData">モデル管理ライブラリデータ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SortOutModelList(McLibCategoryData csLibCategoryData, HySDataLinkedList csModelList, McModelLibraryData csModelLibraryData)
        {
            //DLLから取得したすべてのModelInfo数分ループ（カーソルをまわす）
            McModelInfo csModelInfo = null;

            long lModelctNum = csModelList.GetCount(); // モデル数
            csModelList.SetCursorFirst();// カーソル位置最初
            for (long lLp = 0; lLp < lModelctNum; lLp++)
            {
                csModelInfo = (McModelInfo)csModelList.GetCursorData();

                // モデル大区分の取得
                HySObjectKind csDivision = csModelInfo.GetDivision();

                // モデル大区分が未設定の場合はカテゴリにて区分を判断する
                if (csDivision == null)
                {
                    SetModelLibraryData(csLibCategoryData, csModelInfo, csModelLibraryData);
                }
                // モデル大区分が設定済の場合はそれを利用して各モデルの区分を判断
                else if (csDivision.Equals(McModelLibraryDefine.DIVISION_CALCULATION_MODEL))
                {
                    csModelLibraryData.AddCalModelInfo( csModelInfo);
                }
                else if (csDivision.Equals(McModelLibraryDefine.DIVISION_INPUT_MODEL))
                {
                    csModelLibraryData.AddInputModelnfo(csModelInfo);
                }
                else if (csDivision.Equals(McModelLibraryDefine.DIVISION_OUTPUT_MODEL))
                {
                    csModelLibraryData.AddOutputModelInfo(csModelInfo);
                }
                else if (csDivision.Equals(McModelLibraryDefine.DIVISION_TRANSINFORMATION))
                {
                    csModelLibraryData.AddTranInfo(csModelInfo);
                }

                csModelList.MoveCursorNext();// カーソル位置次
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル管理ライブラリデータ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetModelLibraryData( csLibCategoryData, csModelInfo, csModelLibraryData ) </para>
        /// </example>
        /// <param name="csLibCategoryData">ライブラリ分類データ</param>
        /// <param name="csModelInfo">モデル個別情報</param>
        /// <param name="csModelLibraryData">モデル管理ライブラリデータ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetModelLibraryData(McLibCategoryData csLibCategoryData, McModelInfo csModelInfo, McModelLibraryData csModelLibraryData)
        {
            //モデル区分を取得
            HySObjectKind csModelClass = csModelInfo.GetClassification();

            //モデル分類定義に含まれるモデル分類リストを取得
            IList idList = csLibCategoryData.GetID();

            //モデル分類リスト数分ループ
            foreach (object id in idList)
            {
                //演算要素
                if (((HySString)id).Equal((HySString)McModelLibraryDefine.DIVISION_CALCULATION_MODEL.GetString()))    
                {
                    if (CheckNodeInfoList((HySString)id, csLibCategoryData, csModelClass))
                    {
                        csModelLibraryData.AddCalModelInfo(csModelInfo);
                    }
                }
                else if (((HySString)id).Equal((HySString)McModelLibraryDefine.DIVISION_INPUT_MODEL.GetString())) //入力方式
                {
                    if (CheckNodeInfoList((HySString)id, csLibCategoryData, csModelClass))
                    {
                        csModelLibraryData.AddInputModelnfo(csModelInfo);
                    }
                }
                else if (((HySString)id).Equal((HySString)McModelLibraryDefine.DIVISION_OUTPUT_MODEL.GetString()))  //出力方式
                {
                    if (CheckNodeInfoList((HySString)id, csLibCategoryData, csModelClass))
                    {
                        csModelLibraryData.AddOutputModelInfo(csModelInfo);
                    }
                }
                else if (((HySString)id).Equal((HySString)McModelLibraryDefine.DIVISION_TRANSINFORMATION.GetString())) //伝送情報型
                {
                    if (CheckNodeInfoList((HySString)id, csLibCategoryData, csModelClass))
                    {
                        csModelLibraryData.AddTranInfo(csModelInfo);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>分類定義ノードリストチェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CheckNodeInfoList( id, csLibCategoryData, csModelClass ) </para>
        /// </example>
        /// <param name="id">ライブラリ分類ID</param>
        /// <param name="csLibCategoryData">ライブラリ分類データ</param>
        /// <param name="csModelClass">モデル区分</param>
        /// <returns>true:在り、false:無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CheckNodeInfoList(HySString id, McLibCategoryData csLibCategoryData, HySObjectKind csModelClass)
        {
            bool bRtn = false;

            List<McLibInfoNode> csNodeInfoList = csLibCategoryData.GetChildNodeList(id);

            //ノードリストに存在するか
            for (int icnt = 0; icnt < csNodeInfoList.Count; icnt++)
            {
                if (csNodeInfoList[icnt].ID.Equal((HySString)csModelClass.GetString()))
                {
                    bRtn = true;
                    break;
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>ライブラリインストール情報生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> LoadLibraryInstallInfoData() </para>
        /// </example>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool LoadLibraryInstallInfoData()
        {
            bool bRtn = true;
            try
            {
                HySString csErrorFilePath = HySEnvInf.GetHomeDirectory() + "\\SystemData\\ErrorInfo.txt";
                HySFile csErrorFile = new HySFile(csErrorFilePath);
                // インストール情報をデシリアライズ
                HySString csFilePath = HySEnvInf.GetHomeDirectory() + "\\SystemData\\ModelInstallInfo.dat";
                HySFile csFile = new HySFile(csFilePath);
                if (csFile.Exist() == true)
                {
                    if (csFile.Open(HySFile.OPEN_MODE.OPEN, HySFile.READ_WRITE_MODE.READ, HySFile.DIRECTORY_MODE.NOT_MK_DIR) == 0)
                    {
                        m_csLibraryInstallInfoData = (McLibraryInstallInfoData)csFile.DataRead();
                        // 前回終了時、保存に失敗していないかチェックする
                        // エラー情報ファイルが出力されているか確認
                        if (csErrorFile.Exist() == true)
                        {
                            // 異常データフラグ設定
                            m_csLibraryInstallInfoData.SetDataCheckFlg(false);
                            csErrorFile.Delete();
                        }

                    }
                    else
                    {
                        // オープン失敗
                       // ver1.5 エラートレース日本語対応
                        HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "LoadLibraryInstallInfoData", 
                            Properties.HymcoImplResources.STATEMENT_LIBOPENINF_NG );
                       // HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "LoadLibraryInstallInfoData", "ライブラリインストール情報オープン失敗");
                        // インストール情報生成
                        m_csLibraryInstallInfoData = new McLibraryInstallInfoData();
                        // 異常データフラグ設定
                        m_csLibraryInstallInfoData.SetDataCheckFlg(false);
                        bRtn = false;
                    }
                    csFile.Close();
                    csFile = null;
                }
                else
                {
                    // インストール情報ファイル未作成の場合
                    m_csLibraryInstallInfoData = new McLibraryInstallInfoData();
                }
            }
            catch (Exception ex)
            {
                // ログ出力/メッセージ表示
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "LoadLibraryInstallInfoData",
                    Properties.HymcoImplResources.STATEMENT_EXCEP_GENERATION_PROC + Environment.NewLine + ex.ToString());
               // HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "LoadLibraryInstallInfoData", "ライブラリインストール情報生成処理で例外発生:" + Environment.NewLine + ex.ToString());
                // インストール情報生成
                if (m_csLibraryInstallInfoData == null)
                {
                    m_csLibraryInstallInfoData = new McLibraryInstallInfoData();
                    // 異常データフラグ設定
                    m_csLibraryInstallInfoData.SetDataCheckFlg(false);
                }
                bRtn = false;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>ライブラリインストール情報保存</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SaveLibraryInstallInfoData() </para>
        /// </example>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool SaveLibraryInstallInfoData()
        {
            bool bRtn = true;
            try
            {
                HySString csErrorFilePath = HySEnvInf.GetHomeDirectory() + "\\SystemData\\ErrorInfo.txt";
                HySFile csErrorFile = new HySFile(csErrorFilePath);
                // 削除対象情報が設定されているか確認
                List<HySString> csDelList = m_csLibraryInstallInfoData.GetTargetDllName();
                if (csDelList.Count > 0)
                {
                    // 削除情報保存
                    bRtn = this.WriteLibDeleteInfo();

                    // 削除対象情報あり
                    this.StartDeleteProcess();
                }
                // インストール情報をシリアライズ
                HySString csFilePath = HySEnvInf.GetHomeDirectory() + "\\SystemData\\ModelInstallInfo.dat";
                HySFile csFile = new HySFile(csFilePath);
                if (csFile.Open(HySFile.OPEN_MODE.CREATE_NEW, HySFile.READ_WRITE_MODE.WRITE, HySFile.DIRECTORY_MODE.NOT_MK_DIR) == 0)
                {
                    csFile.DataWrite(m_csLibraryInstallInfoData);
                }
                else
                {
                    // オープン失敗
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "SaveLibraryInstallInfoData",
                         Properties.HymcoImplResources.STATEMENT_LIBOPENINF_NG );
                   // HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "SaveLibraryInstallInfoData", "ライブラリインストール情報オープン失敗");
                    // 保存に失敗した為、エラー情報ファイルを出力
                    csErrorFile.Open(HySFile.OPEN_MODE.CREATE_NEW, HySFile.READ_WRITE_MODE.WRITE, HySFile.DIRECTORY_MODE.NOT_MK_DIR);
                    csErrorFile.Close();
                    csErrorFile = null;
                    bRtn = false;
                }
                csFile.Close();
                csFile = null;
            }
            catch (Exception ex)
            {
                HySString csErrorFilePath = HySEnvInf.GetHomeDirectory() + "\\SystemData\\ErrorInfo.txt";
                HySFile csErrorFile = new HySFile(csErrorFilePath);
                // ログ出力/メッセージ表示
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "SaveLibraryInstallInfoData", 
                    Properties.HymcoImplResources.STATEMENT_EXCEP_LIB_SAVE_PROC + Environment.NewLine + ex.ToString());
               // HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "SaveLibraryInstallInfoData", "ライブラリインストール情報保存処理で例外発生:" + Environment.NewLine + ex.ToString());
                // 保存に失敗した為、エラー情報ファイルを出力
                csErrorFile.Open(HySFile.OPEN_MODE.CREATE_NEW, HySFile.READ_WRITE_MODE.WRITE, HySFile.DIRECTORY_MODE.NOT_MK_DIR);
                csErrorFile.Close();
                csErrorFile = null;
                bRtn = false;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>削除情報保存</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> WriteLibDeleteInfo() </para>
        /// </example>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool WriteLibDeleteInfo()
        {
            bool bRtn = true;
            HySString csLinkedDllName = null;
            List<HySString> csDelList = null;
            HySDataLinkedList csTargetPathInfo = null;
            HySString csFilePath = null;
            HySFile csFile = null;
            try
            {
                csFilePath = HySEnvInf.GetHomeDirectory() + "\\temp\\LibDeleteInfo.txt";
                csFile = new HySFile(csFilePath);
                // 削除情報ファイル吐き出し
                if (csFile.Open(HySFile.OPEN_MODE.CREATE_NEW, HySFile.READ_WRITE_MODE.WRITE, HySFile.DIRECTORY_MODE.NOT_MK_DIR) == 0)
                {
                    // 削除対象情報抽出
                    csDelList = m_csLibraryInstallInfoData.GetTargetDllName();
                    for (int iLp = 0; iLp < csDelList.Count; iLp++)
                    {
                        m_csLibraryInstallInfoData.GetImportFileInfo(csDelList[iLp], ref csTargetPathInfo);
                        for (long lLp = 0; lLp < csTargetPathInfo.GetCount(); lLp++)
                        {
                            // ファイルへ書き込み
                            HySString csPath = (HySString)csTargetPathInfo.GetData(lLp);
                            csFile.WriteText(csPath);
                        }
                    }
                    // 展開対象情報クリア
                    for (int iLp = 0; iLp < csDelList.Count; iLp++)
                    {
                        m_csLibraryInstallInfoData.ClearImportFileInfo(csDelList[iLp]);
                        // モデルと固有プロパティ画面のDLL2つ取得された場合、展開対象情報は2つ生成されている為
                        // ここで対応情報を元に展開対象情報をすべてクリアしておく
                        m_csLibraryInstallInfoData.GetModelDllLinkedPrtyDll(csDelList[iLp], ref csLinkedDllName);
                        m_csLibraryInstallInfoData.ClearImportFileInfo(csLinkedDllName);
                        // 対応情報初期化
                        m_csLibraryInstallInfoData.ClearModelDllLinkedPrtyDll(csDelList[iLp]);
                    }
                    // 削除対象情報初期化
                    m_csLibraryInstallInfoData.ClearTargetDllName();
                }
                else
                {
                    // オープン失敗
                   // ver1.5 エラートレース日本語対応
                    HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "WriteLibDeleteInfo",
                        Properties.HymcoImplResources.STATEMENT_DELLINF_OPEN_NG );
                   // HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "WriteLibDeleteInfo", "削除情報オープン失敗");
                    bRtn = false;
                }
                csFile.Close();
                csFile = null;
                # region オープン・書き込み処理が正常終了した場合に初期化処理を行う。オープンできなかった場合はインストール情報は初期化しない。
                //// 展開対象情報クリア
                //for (int iLp = 0; iLp < csDelList.Count; iLp++)
                //{
                //    m_csLibraryInstallInfoData.ClearImportFileInfo(csDelList[iLp]);
                //    // モデルと固有プロパティ画面のDLL2つ取得された場合、展開対象情報は2つ生成されている為
                //    // ここで対応情報を元に展開対象情報をすべてクリアしておく
                //    m_csLibraryInstallInfoData.GetModelDllLinkedPrtyDll(csDelList[iLp], ref csLinkedDllName);
                //    m_csLibraryInstallInfoData.ClearImportFileInfo(csLinkedDllName);
                //    // 対応情報初期化
                //    m_csLibraryInstallInfoData.ClearModelDllLinkedPrtyDll(csDelList[iLp]);
                //}
                //// 削除対象情報初期化
                //m_csLibraryInstallInfoData.ClearTargetDllName();
                # endregion
            }
            catch (Exception ex)
            {
                // 本処理の過程における異常はインストール情報としては不完全状態となる為、例外として捕捉し、エラー情報ファイルを出力する。
                HySString csErrorFilePath = HySEnvInf.GetHomeDirectory() + "\\SystemData\\ErrorInfo.txt";
                HySFile csErrorFile = new HySFile(csErrorFilePath);
                // ログ出力/メッセージ表示
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "WriteLibDeleteInfo",
                    Properties.HymcoImplResources.STATEMENT_EXCEP_DEL_INF_SAVE_PROC + Environment.NewLine + ex.ToString());
               // HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "WriteLibDeleteInfo", "削除情報保存処理で例外発生:" + Environment.NewLine + ex.ToString());
                // 保存に失敗した為、エラー情報ファイルを出力
                csErrorFile.Open(HySFile.OPEN_MODE.CREATE_NEW, HySFile.READ_WRITE_MODE.WRITE, HySFile.DIRECTORY_MODE.NOT_MK_DIR);
                csErrorFile.Close();
                csErrorFile = null;
                csFile.Close();
                csFile = null;
                bRtn = false;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>削除プロセス起動</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> StartDeleteProcess() </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void StartDeleteProcess()
        {
            try
            {
                // 削除プロセス開始情報設定
                ProcessStartInfo processInfo = new ProcessStartInfo();
                // ファイル名
                processInfo.FileName = "HymcoLibDelete.exe";
                // コンソールウィンドウを開かない
                processInfo.CreateNoWindow = true;
                // シェル機能を使用しない
                processInfo.UseShellExecute = false;
                // 標準入力をリダイレクトしない
                processInfo.RedirectStandardInput = false;
                // 標準出力をリダイレクトする
                processInfo.RedirectStandardOutput = true;
                // エラー出力をリダイレクトする
                processInfo.RedirectStandardError = true;
                // 削除プロセス起動
                Process process = Process.Start(processInfo);
            }
            catch (Exception ex)
            {
                // ログ出力/メッセージ表示
               // ver1.5 エラートレース日本語対応
                HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "StartDeleteProcess", 
                    Properties.HymcoImplResources.STATEMENT_EXCEP_DEL_PROC_SAVE_PROC  + Environment.NewLine + ex.ToString());
               // HySLog.LogOut(HySLog.ONLINE, "McBaseLibraryManager", "StartDeleteProcess", "削除プロセス起動処理で例外発生:" + Environment.NewLine + ex.ToString());
            }
        }

        #endregion

    }
}
