﻿// <summary>ソースコード：収束演算制御基底クラス(水利収束演算制御)</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;



using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Controller;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.Interface.Model;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.FileIO;
using CommonMP.HYMCO.CoreImpl.Data.TranInfo;
using CommonMP.HYMCO.CoreImpl.Controller;
using CommonMP.HYMCO.CoreImpl.Tool;



namespace CommonMP.HYMCO.CoreImpl.Model
{
    /// <summary><para>class outline:</para>
    /// <para>収束演算制御クラス(水利収束演算制御)</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>
    /// [CommonMP][ver 1.0.0][2008/10/01][新規作成]
    /// [CommonMP][ver 1.3.0][2012/12/01][修正]
    /// [CommonMP][ver 1.4.0][2013/12/10][メソッド追加]
    /// </para>
    /// <para>Remarks</para>
    /// <para>無し</para>
    /// </remarks>
    public abstract class McCnvrgncIterationCtlBase : McGrElementCtl, McConvergenceCtl
    {
        /// <summary> 要素数 </summary>
        protected long m_lInnerElementNum = 0;
        /// <summary> グループ内の要素 </summary>
        protected McElement[] m_csInnerElement;

        /// <summary> 外部からの受信がある要素数 </summary>
        protected long m_lRecvElmNum = 0;
        /// <summary> 外部からの受信がある要素 </summary>
        protected McCalModel[] m_csRcvCalMdl;
        /// <summary> 外部からの受信接続 </summary>
        protected McTranInfo[] m_csRcvTranInfo;

        /// <summary> 最大繰り返し計算回数 </summary>
        protected long m_lMaxIterationNum = 10000;
        //// <summary>収束判定条件</summary>
        //protected McConvergenceInfo m_csConvergeCondInfo = null;
        /// <summary> 収束条件データ </summary>
        protected McCnvrgncIterationInfoBase m_csCnvItrInfo = null;

        /// <summary> 収束計算時の要素の記憶範囲 =ELEMENT_MEMORY_ALL:全て記憶、=ELEMENT_MEMORY_SNDPORT_ONLY:出力情報のみ記憶</summary>
        protected McDefine.ElementMemoryLevel m_eInitializeLevel = McDefine.ElementMemoryLevel.ELEMENT_MEMORY_ALL; //McDefine.ElementMemoryLevel.ELEMENT_MEMORY_SNDPORT_ONLY;

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> McCnvrgncIterationCtlBase csObj = new McCnvrgncIterationCtlBase( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McCnvrgncIterationCtlBase()
        {
            m_eDtFusionTiming = McDefine.DataFusionTiming.SYNCHRONOUS; // 同期をデフォルト化
        }

        /// <summary><para>method outline:</para>
        /// <para>DataFusion()発行のタイミング（演算フロー制御）設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDataFusionTiming(eDtFusionTiming)</para>
        /// </example>
        /// <param name="eDtFusionTiming"> 同期／非同期
        /// McDefine.DataFusionTiming.SYNCHRONOUS：全ての要素が演算終了後DataFutionを行う
        /// McDefine.DataFusionTiming.ASYNCHRONOUS：非同期：要素が演算終了後個別にDataFutionを行う）
        /// </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetDataFusionTiming(McDefine.DataFusionTiming eDtFusionTiming)
        {
            // 収束演算は　McDefine.DataFusionTiming.SYNCHRONOUS のみであるため何もしない
            // Do Nothing
        }

        //=========================
        // 演算実行前処理関連メソッド
        //=========================

        /// <summary><para>method outline:</para>
        /// <para>モデル演算可能かをチェックする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsAbleToCalculate(ref csInputDataList , ref csOutputDataList , ref csErrorInf)</para>
        /// </example>
        /// <param name="csInputDataList">前段接続要素からの伝送情報リスト</param>
        /// <param name="csOutputDataList">前段接続要素への伝送情報リスト</param>
        /// <param name="csErrorInf">エラー出力</param>
        /// <returns>=true:演算可能、=false:演算不能</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool IsAbleToCalculate(ref HySDataLinkedList csInputDataList, ref HySDataLinkedList csOutputDataList, ref McStructErrorInfo csErrorInf)
        {
            // 内部要素のチェック
            bool bRtn = true;
            //bRtn = InnerElementCheck(ref csErrorInf);

            // 親クラスの演算可能性チェック
            if (base.IsAbleToCalculate(ref csInputDataList, ref csOutputDataList, ref csErrorInf) == false)
            {
                bRtn = false;
            }

            LinkedListNode<McElement> csElmNode = null;

            // 収束演算情報の正常性チェック
            if (m_csCnvItrInfo == null)
            {   // 異常の場合
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), new HySObjectKind("McCnvrgncIterationCtlBase"), 
                                         Properties.HymcoImplResources.STATEMENT_BAD_MCCONVERGENEINFO );
               // csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), new HySObjectKind("McCnvrgncIterationCtlBase"), " bad McConvergenceInfo.");
                bRtn = false;
            }
            else
            {   // 正常の場合には，収束演算情報を設定する
                // 内部の収束情報の設定
                //   要素編
                long lElmNum = m_csElementList.Count;  // 要素数
                m_csCnvItrInfo.SetElementNum(lElmNum);

                McCmnElement csElementWork = null;
                csElmNode = m_csElementList.First;
                for (long lLp = 0; lLp < lElmNum; lLp++)
                {
                    csElementWork = (McCmnElement)csElmNode.Value;
                    m_csCnvItrInfo.SetElementCalInfo(lLp, csElementWork.GetCalInfo());
                    m_csCnvItrInfo.SetElementInitialInfo(lLp, (McInitialInfo)csElementWork.GetInitializeData());
                    m_csCnvItrInfo.SetElementName(lLp, csElementWork.GetElementName());
                    m_csCnvItrInfo.SetElementID(lLp, (HySID)csElementWork.GetID());
                    csElmNode = csElmNode.Next;
                }
                //   接続編
                long lCnnctNum = m_csConnectionList.Count;  // 接続数
                //   先ずGr外からの流入以外の接続がいくつあるかを調べる
                long lCnnctNumWithoutInput = 0;
                m_lRecvElmNum = 0;
                McConnection csCnnctWork = null;
                McInnerSndPort csRrySndPort = null;
                McGrElement csGrSendElm = null;
                LinkedListNode<McConnection> csCnnctNode = m_csConnectionList.First;
                for (long lLp = 0; lLp < lCnnctNum; lLp++)
                {
                    csCnnctWork = csCnnctNode.Value;
                    csRrySndPort = csCnnctWork.GetSendPort() as McInnerSndPort;
                    if (csRrySndPort != null)
                    {   // 中継送信端子
                        csGrSendElm = csRrySndPort.GetElement() as McGrElement;
                        if (csGrSendElm != null)
                        {   // Gr要素に接続
                            if (m_csElement.GetID().Equals(csGrSendElm.GetID()))
                            {   // 中継端子が自Gr(収束演算Gr)ならば
                                csCnnctNode = csCnnctNode.Next;
                                m_lRecvElmNum += 1;
                                continue; // 何もしない
                            }
                        }
                    }
                    lCnnctNumWithoutInput += 1;
          
                    // ↓以下の論理では　外に出て行く接続が抜けてしまう
                    //if (csCnnctWork.GetReceivePort().GetElement() != null)
                    //{
                    //    if (csCnnctWork.GetReceivePort().GetElement().GetID() != this.m_csElement.GetID())
                    //    {
                    //        lCnnctNumWithoutInput += 1;
                    //    }
                    //}

                    csCnnctNode = csCnnctNode.Next;
                }
                // 外部からの受信がある要素 </summary>
                m_csRcvCalMdl = new McCalModel[m_lRecvElmNum];
                m_csRcvTranInfo = new McTranInfo[m_lRecvElmNum];
                long lRcvElmNo = 0;
                // Gr外からの流入以外の接続のみ抽出して条件データに入れる
                m_csCnvItrInfo.SetConnectionNum(lCnnctNumWithoutInput); // 最初にエリア確保

                long lCnctNoIn = 0; // Gr外からの流入以外の接続累積番号
                csCnnctNode = m_csConnectionList.First;
                for (long lLp = 0; lLp < lCnnctNum; lLp++)
                {
                    csCnnctWork = csCnnctNode.Value;

                    csRrySndPort = csCnnctWork.GetSendPort() as McInnerSndPort;
                    if (csRrySndPort != null)
                    {   // 中継送信端子
                        csGrSendElm = csRrySndPort.GetElement() as McGrElement;
                        if (csGrSendElm != null)
                        {   // Gr要素に接続
                            if (m_csElement.GetID().Equals(csGrSendElm.GetID()))
                            {   // 中継端子が自Gr(収束演算Gr)ならば
                                csCnnctNode = csCnnctNode.Next;
                                m_csRcvCalMdl[lRcvElmNo] = csCnnctWork.GetReceivePort().GetElement().GetCalModel();
                                m_csRcvTranInfo[lRcvElmNo] = csCnnctWork.GetTranInfo();
                                lRcvElmNo += 1;
                                continue; // 何もしない
                            }
                        }
                    }

                    // ↓以下の論理では　外に出て行く接続が抜けてしまう
                    //if (csCnnctWork.GetReceivePort().GetElement() != null)
                    //{
                    //    if (csCnnctWork.GetReceivePort().GetElement().GetID() != this.m_csElement.GetID())
                    //    {
                    //        m_csCnvItrInfo.SetConnectionTranInfo(lLp, (McTranInfo)csCnnctWork.GetTranInfo());
                    //    }
                    //}

                    m_csCnvItrInfo.SetConnectionTranInfo(lCnctNoIn, (McTranInfo)csCnnctWork.GetTranInfo());
                    lCnctNoIn += 1;

                    csCnnctNode = csCnnctNode.Next;
                }

                // グループ内の要素に対して同期／非同期を設定する
                this.SetDataFusionTimingToInnerElement();
            }

            // 要素配列生成
            m_lInnerElementNum = m_csElementList.Count;  // 要素数
            if (m_lInnerElementNum == 0)
            {
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckWarningData(this.m_csElement.GetID(), new HySObjectKind("McCnvrgncIterationCtlBase"), 
                     Properties.HymcoImplResources.STATEMENT_HAVE_NO_ELEMENT );
               // csErrorInf.AddCheckWarningData(this.m_csElement.GetID(), new HySObjectKind("McCnvrgncIterationCtlBase"), " Have no element.");
            }
            m_csInnerElement = new McElement[m_lInnerElementNum];
            csElmNode = m_csElementList.First;
            for (long lLp = 0; lLp < m_lInnerElementNum; lLp++)
            {
                m_csInnerElement[lLp] = csElmNode.Value;
                csElmNode = csElmNode.Next;
            }

            if (InnerElementCheck(ref csErrorInf) == false)
            {
                bRtn = false;
            }

            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ内の要素に対して同期／非同期を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDataFusionTimingToInnerElement( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void SetDataFusionTimingToInnerElement()
        {
            LinkedListNode<McElement> csElmNode = null;
            McCmnElement csElementWork = null;
            long lElmNum = m_csElementList.Count;  // 要素数

            csElmNode = m_csElementList.First;
            for (long lLp = 0; lLp < lElmNum; lLp++)
            {
                csElementWork = (McCmnElement)csElmNode.Value;
                csElementWork.SetDataFusionTiming(this.m_eDtFusionTiming);
                csElmNode = csElmNode.Next;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>グループ内の要素が自分が期待している要素か否かをチェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = InnerElementCheck(ref csErrorInf)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ループ内の要素に指定がある場合、収束演算を行う処理は本チェックにおいて、
        /// 予定外要素が　グループ内に存在しないことを確認する。</para>
        /// </remarks>
        public abstract bool InnerElementCheck(ref McStructErrorInfo csErrorInf);

        /// <summary><para>method outline:</para>
        /// <para>演算順序のチェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = CheckCalculationOrder( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> long
        /// 正常時：McDefine.CALCULATION_NORMAL_RETURN
        /// 異常時：McDefine.CALCULATION_ABEND_RETURN_BECAUSE_OF_STRUCT_ERROR
        /// </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 非同期演算の時有効
        /// 演算順序を上流から下流に並び替える。
        /// もし、ループ等があればエラーを返す
        /// </para>
        /// </remarks>
        public override long CheckCalculationOrder()
        {
            return McDefine.CALCULATION_NORMAL_RETURN;
        }

        //=======================
        // 演算実行処理関連メソッド 
        //=======================
        
        /// <summary><para>method outline:</para>
        /// <para>モデルを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize( ref csInitialData,  ref csInputDataList, ref csOutputDataList)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <param name="csInputDataList">前段接続要素からの伝送情報リスト</param>
        /// <param name="csOutputDataList">前段接続要素への伝送情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Initialize(ref McPropertyInfoRoot csInitialData, ref HySDataLinkedList csInputDataList, ref HySDataLinkedList csOutputDataList)
        {
            ////**/
            if (m_eDtFusionTiming == McDefine.DataFusionTiming.ASYNCHRONOUS)
            {   // 非同期通信ならば
                // Do noting
            }
            else
            {
                int iElmCnt = m_csElementList.Count;  // グループ内のエレメント数取得

                m_csElmNode = m_csElementList.First;
                for (int iLp = 0; iLp < iElmCnt; iLp++)
                {   // 個数分繰り返す
                    m_csElementWork = m_csElmNode.Value;    // エレメント取得
                    // 現状計算型の同期演算対応） 
                    if (m_csElementWork.GetCalModel() as McStateCalModelBase != null)
                    {
                        m_csElementWork.GetCalModel().SetDeltaTime(this.m_csDltTime);
                        // 現状計算型のδTはサブシステムのδTに一致化させる
                    }
                    m_csElmNode = m_csElmNode.Next;
                }

            }
            // ↑　base.Initialize(。。。) の前に行うこと
            ////**/

            this.m_csCnvItrInfo.SetCtlInitInfo(csInitialData as McInitialInfo); // <-- 初期化情報を覚えておく

            bool bRtn = this.Initialize(ref csInitialData);
            if (bRtn == true)
            {
                bRtn = base.Initialize(ref csInitialData, ref csInputDataList, ref csOutputDataList);
            }

            return bRtn;

            /*
            long lElmNum =m_csElementList.Count;  // 要素数
            m_csCnvItrInfo.SetElementNum(lElmNum );

            McCmnElement csElementWork = null;
            LinkedListNode<McElement> csElmNode = m_csElementList.First;
            for (long lLp = 0; lLp < lElmNum; lLp++)
            {
                csElementWork = (McCmnElement)csElmNode.Value;
                m_csCnvItrInfo.SetElementCalInfo(lLp, csElementWork.GetCalInfo());
                m_csCnvItrInfo.SetElementInitialInfo(lLp,(McInitialInfo)csElementWork.GetInitializeData());
                csElmNode = csElmNode.Next;
            }

            long lCnnctNum = m_csConnectionList.Count;  // 接続数
            m_csCnvItrInfo.SetConnectionNum(lCnnctNum);

            McConnection csCnnctWork = null;
            LinkedListNode<McConnection> csCnnctNode = m_csConnectionList.First;
            for (long lLp = 0; lLp < lCnnctNum; lLp++)
            {
                csCnnctWork = csCnnctNode.Value;
                m_csCnvItrInfo.SetConnectionTranInfo(lLp, (McTranInfo)csCnnctWork.GetTranInfo());
                csCnnctNode = csCnnctNode.Next;
            }
            */
        }
        /// <summary><para>method outline:</para>
        /// <para>モデルを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(ref csInitialData)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected abstract bool Initialize(ref McPropertyInfoRoot csInitialData);
        /// <summary><para>method outline:</para>
        /// <para>演算実行中断中のモデルを初期化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(ref csInitialData, ref csInputDataList, ref csOutputDataList)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <param name="csInputDataList">前段接続要素からの伝送情報リスト</param>
        /// <param name="csOutputDataList">前段接続要素への伝送情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中にプロパティ、初期値等を変更して動作させ場合等に使用する</para>
        /// </remarks>
        public override bool OnlineInitialize(ref McPropertyInfoRoot csInitialData, ref HySDataLinkedList csInputDataList, ref HySDataLinkedList csOutputDataList)
        {
            return OnlineInitialize(ref csInitialData);
        }       
        /// <summary><para>method outline:</para>
        /// <para>演算実行中断中のモデルを初期化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(ref csInitialData)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば派生側でオーバーライドする</para>
        /// </remarks>
        protected virtual bool OnlineInitialize(ref McPropertyInfoRoot csInitialData)
        {
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算可能判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = Calculable()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=true:計算可 =false:計算不可</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Calculable()
        {
            bool bRtn = true;          
 
            // 内部の要素モデルが計算するためには、外部からの入力が全てそろわないと計算できない。
            for (long lElmNo = 0; lElmNo < m_lRecvElmNum; lElmNo++)
            {
                //if (m_csRcvCalMdl[lElmNo].Calculable() == false)
                if( m_csRcvCalMdl[lElmNo].GetSimulationTime().After( m_csRcvTranInfo[lElmNo].GetLastTime())== true )
                {
                    bRtn = false;
                    break;
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsConverged( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=true:収束完了 =false:未収束</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>収束演算要素の外から呼ばれるメソッド（Element.IsConverged() からコールされる）</para>
        /// </remarks>
        public override bool IsConverged()
        {
            bool bRtn = base.IsConverged();
            if(bRtn == false)
            {   // 内部に　要素モデルが存在し、そのモデルに直接外部から入力がある場合には、
                if (this.Calculable() == false)
                {
                    bRtn = true;
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = Calculate(ref csInputDataList)</para>
        /// </example>
        /// <param name="csInputDataList">演算に必要な入力情報リスト</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override long Calculate(ref HySDataLinkedList csInputDataList)
        {
            long lRtn = McDefine.CALCULATION_NORMAL_RETURN; // リターンコード
            bool bConvergedFlg = false;
            HySTime csMemorySimTime = m_csSimTime.Clone();

            long lLclRtn = 0;// ローカル変数

            long iElmCnt = m_csElementList.Count;  // グループ内のエレメント数取得

            //１　this.初期状態設定のメソッド
            this.SetUpConvergenceInf(ref m_csCnvItrInfo);

            // 2 初期状態の記憶、
            // 2.1 状態記憶の前に、今よりも未来の時間が伝送データ中に存在していればそれを削除しておく
            AdjustTranInfoTime(ref m_csCnvItrInfo);
            // 2.2 状態記憶
            m_csElmNode = m_csElementList.First;
            for (long lLp = 0; lLp < iElmCnt; lLp++)
            {   // 個数分繰り返す
                m_csElementWork = m_csElmNode.Value;    // エレメント取得
                m_csElementWork.Memorize(m_eInitializeLevel); // 初期状態の記憶
                m_csElmNode = m_csElmNode.Next;
            } // end of for(要素数)

            // 3 繰り返し計算
            for (long lItrtnLp = 0; lItrtnLp < m_lMaxIterationNum; lItrtnLp++)
            {   // 最大繰り返し回数

                // 収束計算回数設定
                m_csCnvItrInfo.SetCnvrgncLoopCount(lItrtnLp);

                // 最初に、初期出力を行わせる
                this.InitialDataFusion(iElmCnt);

                // 3.2 各要素計算
                for (long lLp=0;lLp<10000;lLp++) // 実質無限ループ
                {
                    if (m_csSimTime.Before(m_csTgtTime) != true)
                    {   // 計算時刻が、目標時刻を超えたならば
                        break;
                    }

                    //
                    lLclRtn = this.CalculateIteration();
                    if (lLclRtn != McDefine.CALCULATION_NORMAL_RETURN)
                    {   // エラー有りならば
                        // エラーログ等吐き出し
                        if (lLclRtn == McDefine.CALCULATION_FORCE_STOP_RETURN)
                        {   // 強制終了ならば
                           // ver1.5 エラートレース日本語対応
                            McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), "McCnvrgncIterationCtlBase", "CalculateIteration",
                                Properties.HymcoImplResources.STATEMENT_FORCE_STOP_CALC );
                           // McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), "McCnvrgncIterationCtlBase", "CalculateIteration", "Force Stop Calculation");
                        }
                        else
                        {
                            McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.m_csElement.GetOwnerProjectID()) as McStructErrorInfo;
                            if (csErrorInfo != null)
                            {
                               // ver1.5 エラートレース日本語対応
                                csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                                        Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString() + 
                                        ".CalculateIteration() :: " + Properties.HymcoImplResources.STATEMENT_RET_CODE + "(" + lLclRtn.ToString() + ")");
                               // csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                               //         "Error return from " + m_csElementWork.GetCalModel().ToString() + ".CalculateIteration() :: return_code(" + lLclRtn.ToString() + ")");
                            }
                           // ver1.5 エラートレース日本語対応
                            McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "CalculateIteration",
                                Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString() + ".CalculateIteration() :: return_code(" + lLclRtn.ToString() + ")");
                           // McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "CalculateIteration",
                           //     "Error return from " + m_csElementWork.GetCalModel().ToString() + ".CalculateIteration() :: return_code(" + lLclRtn.ToString() + ")");
                        }
                        lLclRtn = McDefine.CALCULATION_ABEND_RETURN;  // 全体の戻り値もエラー
                        break;
                    }

                    //
                    m_csSimTime.Add(m_csDltTime);//  += m_csDltTime; 演算子を使用すると遅くなる為、
                    m_csTotalPassingTime.SetTime(m_csSimTime.GetTime() - m_csStartTime.GetTime());
                }


                // 3.3 収束判別
                bConvergedFlg = this.IsConverged(ref m_csCnvItrInfo);
                if (bConvergedFlg == true)
                {   // 収束していた
                    // 収束した状態をメモリーしておく
                    //m_csElmNode = m_csElementList.First;
                    //for (long lLp = 0; lLp < iElmCnt; lLp++)
                    //{   // 個数分繰り返す
                    //    m_csElementWork = m_csElmNode.Value;    // エレメント取得
                    //    m_csElementWork.Memorize(m_eInitializeLevel); // 初期状態の記憶
                    //    m_csElmNode = m_csElmNode.Next;
                    //} // end of for(要素数)
                    break;
                }

                // 3.4 初期値再設定
                this.ModifyConvergenceInf(ref m_csCnvItrInfo);

                if (m_eInitializeLevel == McDefine.ElementMemoryLevel.ELEMENT_MEMORY_ALL)
                {   // 完全に元に戻して再収束計算を行う場合

                    // 初期状態に戻す
                    m_csElmNode = m_csElementList.First;
                    for (long lLp = 0; lLp < iElmCnt; lLp++)
                    {   // 個数分繰り返す
                        m_csElementWork = m_csElmNode.Value;    // エレメント取得

                        m_csElementWork.Remember(m_eInitializeLevel); // 初期状態の復活
                        m_csCnvItrInfo.SetElementCalInfo( lLp, m_csElementWork.GetCalInfo()); // インスタンス設定

                        m_csElementWork.SetSimulationTime(csMemorySimTime); // 時刻の復活

                        m_csElmNode = m_csElmNode.Next;

                    } // end of for(要素数)

                    this.PostRemenber(ref m_csCnvItrInfo);
                }
                else
                {   // これまでの収束結果の上に何らかの処理を行う場合
                    // 時間だけ初期状態に戻す
                    m_csElmNode = m_csElementList.First;
                    for (long lLp = 0; lLp < iElmCnt; lLp++)
                    {   // 個数分繰り返す
                        m_csElementWork = m_csElmNode.Value;    // エレメント取得

                        //m_csElementWork.Remember(m_eInitializeLevel); // 初期状態の復活 <-- これが無い

                        m_csElementWork.SetSimulationTime(csMemorySimTime); // 時刻の復活

                        m_csElmNode = m_csElmNode.Next;

                    } // end of for(要素数)                    // 初期化を行う
                }

                // Gr要素内の時間を元に戻す
                m_csSimTime.SetTime(csMemorySimTime);

                // 今よりも未来の時間が伝送データ中に存在していればそれを削除しておく
                AdjustTranInfoTime(ref m_csCnvItrInfo);
            
            } // end of for(lIterationLp)

            if (bConvergedFlg == false)
            {   // 結局収束しなかった
                lRtn = McDefine.CALCULATION_NOT_CONVERGED_RETURN;
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), "McCnvrgncIterationCtlBase", "Calculate",
                    Properties.HymcoImplResources.STATEMENT_NOT_CONVERGED );
               // ver1.5 エラートレース日本語対応
                McLog.DebugOut(new HySString(McLog.SYSTEM), new HySString(this.ToString()),
                    new HySString(" === " + Properties.HymcoImplResources.STATEMENT_NOT_CONVERGED + " === "));
               // McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), "McCnvrgncIterationCtlBase", "Calculate", "Not Converged.");
               // McLog.DebugOut(new HySString(McLog.SYSTEM), new HySString(this.ToString()), new HySString(" === Not Converged. === "));
                //McLog.DebugOut(new HySString(McLog.SYSTEM), new HySString(">  Not Converged int the "), new HySString(this.ToString() + " Model."));
                // 規定回数トライしたが結局収束しなかった場合の処理
                lRtn = this.NotConvergedOperation(ref m_csCnvItrInfo);
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>収束演算（１回分）を行う際の初期情報DataFusion</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = InitialDataFusion( lElmCnt)</para>
        /// </example>
        /// <param name="lElmCnt">エレメント数</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void InitialDataFusion(long lElmCnt)
        {
            HySTime csWorkTime = new HySTime(60.0); // ワーク変数（速度アップの為、ループ内での変数の new を控える
            // 最初に、初期出力を行わせる
            m_csElmNode = m_csElementList.First;
            for (long lLp = 0; lLp < lElmCnt; lLp++)
            {   // 個数分繰り返す
                m_csElementWork = m_csElmNode.Value;    // エレメント取得
                if (m_csElementWork.GetCalModel() as McStateCalModelBase != null)
                {   // 現状計算型ならば
                    //  通常現状計算型は、DataFusion後、時計を進めるが、此の位置におけるDataFusionでは時計を進めない（既に時計は進んでいるため）
                    csWorkTime.SetTime(m_csElementWork.GetCalModel().GetSimulationTime().GetTime()); // ｼﾐｭﾚｰﾀｰの現在時刻を一時保管
                    m_csElementWork.DataFusion();
                    m_csElementWork.GetCalModel().SetSimulationTime(csWorkTime);
                }
                else
                {   // 現状計算型ででなければ
                    m_csElementWork.ForceDataFusion();
                    //m_csElementWork.InitialDataFusionForConvergentCalculation(m_csCnvItrInfo.GetInitErrorVal(lLp));
                }
                m_csElmNode = m_csElmNode.Next;
            } // end of for(要素数)
        }

        /*
        /// <summary><para>method outline:</para>
        /// <para>初回モデル演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = Calculate( ref csInputDataList)</para>
        /// </example>
        /// <param name="csInputDataList">演算に必要な入力情報リスト</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 初回の演算時のみコールされる
        /// 本メソッド内では、中断は受け付けない
        /// </para>
        /// </remarks>
        public override long FirstCalculate(ref HySDataLinkedList csInputDataList)
        {
            return this.Calculate(ref csInputDataList);
        }
        **/
         
        /// <summary><para>method outline:</para>
        /// <para>内部モデルの演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = CalculateIteration()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual long CalculateIteration()
        {
            McCalculateModelBase csCalModel;
            McStateCalModelBase csStateCalMdl;
            long lLclRtn = McDefine.CALCULATION_FORCE_STOP_RETURN;
            for (long llLp = 0; llLp < m_lInnerElementNum; llLp++)
            {
                lLclRtn = this.SubSystemCalculate();
                if (lLclRtn != McDefine.CALCULATION_NORMAL_RETURN)
                {   // エラー有りならば
                    lLclRtn = McDefine.CALCULATION_ABEND_RETURN;  // 全体の戻り値もエラー
                    break;
                }
                bool bBreakFlg = true;
                for (long lElmNo = 0; lElmNo < m_lInnerElementNum; lElmNo++)
                {
                    csCalModel = m_csInnerElement[lElmNo].GetCalModel() as McCalculateModelBase;
                    if (csCalModel != null)
                    {
                        csStateCalMdl = csCalModel as McStateCalModelBase;
                        if (csStateCalMdl != null)
                        {   // 現状計算型
                            if (csStateCalMdl.GetSimulationTime().After(csStateCalMdl.GetTargetTime()) != true)
                            {   // 要素モデル目標時刻より後まで達していない
                                // 一つでも達していないものがあれば繰り返す。
                                bBreakFlg = false;
                            }
                        }
                        else
                        {
                            if (csCalModel.GetSimulationTime().Before(csCalModel.GetTargetTime()) == true)
                            {   // 要素モデル目標時刻達していない
                                // 一つでも達していないものがあれば繰り返す。
                                bBreakFlg = false;
                            }
                        }
                    }
                }
                if (bBreakFlg == true)
                {
                    break;
                }
            }
            return lLclRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>内部モデルの演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = SubSystemCalculate()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual long SubSystemCalculate( )
        {
            long lRtn = McDefine.CALCULATION_NORMAL_RETURN; // リターンコード
            long lLclRtn = 0;// ローカル変数
            HySTime csWorkTime = new HySTime(60.0); // ワーク変数（速度アップの為、ループ内での変数の new を控える
            int iElmCnt = m_csElementList.Count;  // グループ内のエレメント数取得
            int iLoop = 0;  // loop 内のローカル変数new を防ぐ

            if (McGrElementCtl.RESTART_STATE != this.m_lStatus)
            {   // 再開段階でなければ
                this.m_lStatus = McGrElementCtl.CALCULATION_STATE; // 計算モード
            }

            // 2) グループ内部エレメントに計算実行
            // 2-1 再開／中断等のステータスにより　計算の取り掛かり値を設定
            if (McGrElementCtl.RESTART_STATE == this.m_lStatus)
            {   // 再開段階ならば
                //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McSubSystemCtl::Calculation", "m_lStatus = RESTART_STATE");
                // 再開時の演算要素を求める
                m_csElmNode = m_csElementList.First;
                for (int iLp = 0; iLp < m_iNodeItemNo; iLp++)
                {   // 個数分繰り返す
                    m_csElmNode = m_csElmNode.Next;
                }
                this.m_lStatus = McGrElementCtl.CALCULATION_STATE; // 計算モードへ移行
            }
            else
            {   // 再開段階ではなければ
                // 最初の要素から計算を進める
                m_csElmNode = m_csElementList.First;
                m_iNodeItemNo = 0;
            }
            //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McSubSystemCtl::Calculation", " loop {Element.Calculate()} start");

            McGrElement csGrElementWork = null;
            HySTime csNextTargetTime = new HySTime(0); //  HySTime.DEFAULT_TIME.Clone(); <-- DEFAULT_TIME にする必要なし

            // 2-2 要素数分内部エレメントに計算を行わせる
            for (int iLp = m_iNodeItemNo; iLp < iElmCnt; iLp++)
            {   // 個数分繰り返す

                // 2-2-1 演算中断／終了時の処理
                if (McGrElementCtl.PAUSE_STATE == this.m_lStatus)
                {   // 計算中断モードならば　計算したところまでのノードを覚える
                    //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McSubSystemCtl::Calculation", "m_lStatus = PAUSE_STATE  break");
                    m_iNodeItemNo = iLp;
                    break;
                }
                else if (McGrElementCtl.STOP_STATE == this.m_lStatus)
                {   // 計算終了モードならば
                    //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McSubSystemCtl::Calculation", "m_lStatus = STOP_STATE  break");
                    break;
                }

                // 2-2-2 演算エレメント本体
                m_csElementWork = m_csElmNode.Value;    // エレメント取得
                csGrElementWork = m_csElementWork as McGrElement;

                // 2-2-3 各要素に演算時刻を設定する
                //m_csElementWork.SetTargetTime(m_csSimTime + m_csDltTime);
                // 次の演算目標時刻がシミュレーション終了時刻を越えないかどうか確認
                //csNextTargetTime = (m_csSimTime + m_csDltTime); // ＋オペレーターの使用は処理が遅くなる元
                csNextTargetTime.SetTime(m_csSimTime.GetTime() + m_csDltTime.GetTime());
                if (csGrElementWork == null && csNextTargetTime.Before(m_csFinalTgtTime) != true)
                {
                    // 単体要素で次の演算目標時刻がシミュレーション終了時刻を越えていれば
                    m_csElementWork.SetTargetTime(m_csFinalTgtTime);
                }
                else
                {
                    // Gr要素の場合は、終了時刻超過の確認はMcGrElementCtlで行われる
                    m_csElementWork.SetNextTargetTime(ref m_csSimTime, ref m_csDltTime);
                }

                // モノによっては、δTを変更する場合がある。
                //m_csElementWork.SetUpConvergenceInf(ref m_csConvergeCondInfo);

                // 2-2-4 一個の要素について演算を行う
                for (iLoop = 0; iLoop < 1000000; iLoop++)
                {   // 実質的∞：永遠に応答無しを防ぐ為、有限回に設定

                    // 2-2-4-1 個別要素の収束判別
                    //if (m_csElementWork.IsConverged() == true)
                    if ( this.EachElmIsConverged(ref m_csElementWork) == true)
                    {   // 収束していれば
                        //if (m_eDtFusionTiming == McDefine.DataFusionTiming.ASYNCHRONOUS)
                        //{   // 非同期通信ならば
                        //    m_csElementWork.DataFusion();   // 演算結果公開（此処でDataFusionすると蓄積データ数はあまり大きくはならない）
                        //}
                        break;
                        // 次の要素の演算へ移行
                    }

                    // 2-2-4-2 個別要素の演算計算
                    lLclRtn = m_csElementWork.Calculate(); // 内部の此処の要素に対して計算を行わせる
                    if (lLclRtn != McDefine.CALCULATION_NORMAL_RETURN)
                    {   // エラー有りならば
                        lRtn = McDefine.CALCULATION_ABEND_RETURN;  // 全体の戻り値もエラー
                        // エラーログ等吐き出し
                        if (lLclRtn == McDefine.CALCULATION_FORCE_STOP_RETURN)
                        {   // 強制終了ならば
                           // ver1.5 エラートレース日本語対応
                            McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), "McCnvrgncIterationCtlBase", "SubSystemCalculate", 
                               Properties.HymcoImplResources.STATEMENT_FORCE_STOP_CALC );
                           // McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), "McCnvrgncIterationCtlBase", "SubSystemCalculate", "Force Stop Calculation");
                            lRtn = lLclRtn;
                            break;
                        }
                        else
                        {
                            McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.m_csElement.GetOwnerProjectID()) as McStructErrorInfo;
                            if (csErrorInfo != null)
                            {
                               // ver1.5 エラートレース日本語対応
                                csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                                        Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString() 
                                        + ".Calculate() :: " + Properties.HymcoImplResources.STATEMENT_RET_CODE +"(" + lRtn.ToString() + ")");
                               // csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                               //         "Error return from " + m_csElementWork.GetCalModel().ToString() + ".Calculate() :: return_code(" + lRtn.ToString() + ")");
                            }
                           // ver1.5 エラートレース日本語対応
                            McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "Calculate",
                                Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString() + ".Calculate() :: return_code(" + lRtn.ToString() + ")");
                           // McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "Calculate",
                           //     "Error return from " + m_csElementWork.GetCalModel().ToString() + ".Calculate() :: return_code(" + lRtn.ToString() + ")");

                            break;
                        }
                    }
                    // 2-2-4-3 個別要素の時刻を進める
                    m_csElementWork.GainSimulationTime(); // 要素内の時刻を進める

                    if (m_eDtFusionTiming == McDefine.DataFusionTiming.ASYNCHRONOUS)
                    {   // 非同期通信ならば
                        lLclRtn = m_csElementWork.DataFusion();   // 演算結果公開(此処でDataFusionするとelementのδT が小さいときには非常に多くの情報が蓄積される)
                    }
                    else
                    {
                        // Ver1.3では下記コードを追加する（現状計算型の同期演算対応） 
                        if (m_csElementWork.GetCalModel() as McStateCalModelBase != null)
                        {
                            // 現状計算はGainSimulationTimeで時が進まないのでDataFusionを行い、要素自身の時刻を進める
                            lLclRtn = m_csElementWork.DataFusion();
                            //m_csElementWork.SetSimulationTime(m_csElementWork.GetCalModel().GetSimulationTime()+this.m_csDltTime);
                            //   ↑  現状計算型の時刻はサブシステムのシミュレーション時刻＋δTに一致化させる
                            //             ↑
                            //        既に現状型のδTをサブシステムのδTに一致化させているためこの処理は不要
                        }
                    }
                    if (lLclRtn != 0)
                    {
                        McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.m_csElement.GetOwnerProjectID()) as McStructErrorInfo;
                        if (csErrorInfo != null)
                        {
                           // ver1.5 エラートレース日本語対応
                            csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                                    Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString() 
                                    + ".DataFusion() :: "+ Properties.HymcoImplResources.STATEMENT_RET_CODE + "(" + lLclRtn.ToString() + ")");
                           // csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                           //         "Error return from " + m_csElementWork.GetCalModel().ToString() + ".DataFusion() :: return_code(" + lLclRtn.ToString() + ")");
                        }
                       // ver1.5 エラートレース日本語対応
                        McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "DataFusion",
                            Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString() + ".DataFusion() :: return_code(" + lLclRtn.ToString() + ")");
                       // McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "DataFusion",
                       //     "Error return from " + m_csElementWork.GetCalModel().ToString() + ".DataFusion() :: return_code(" + lLclRtn.ToString() + ")");
                        lRtn = -1;
                        break;
                    }

                    //System.Threading.Thread.Sleep(0);   // スレッドの隙間を空けるため（スレッド関係の処理は非常に時間を消費する為、此処に入れるべきではない）
                }   // end of for(∞)

                if (lRtn == -1)
                {
                    break;
                }
                // 2-2-5 次の演算要素取得
                m_csElmNode = m_csElmNode.Next;

                //System.Threading.Thread.Sleep(0);   // スレッドの隙間を空けるため（スレッド関係の処理は非常に時間を消費する為、此処に入れるべきではない）
            } // end of for(要素数)


            //if (McGrElementCtl.PAUSE_STATE != this.m_lStatus)
            //{   // 計算中断モードではないならば
            //    m_iNodeItemNo = 0;
            //}

            //HySLog.LogOut(HySLog.SYSTEM_DEBUG, "McSubSystemCtl::Calculation", " Element.DataFusion() start");
            if (m_eDtFusionTiming == McDefine.DataFusionTiming.SYNCHRONOUS)
            {   // 同期通信ならば
                // 2-3 内部エレメント同士ののデータ交換
                if (McGrElementCtl.CALCULATION_STATE == this.m_lStatus)
                {   // 計算モード中ならば
                    m_csElmNode = m_csElementList.First;
                    for (int iLp = m_iNodeItemNo; iLp < iElmCnt; iLp++)
                    {   // 個数分繰り返す
                        m_csElementWork = m_csElmNode.Value;    // エレメント取得

                        // Ver1.3（現状計算型の同期演算対応）  
                        if (m_csElementWork.GetCalModel() as McStateCalTypeModel != null)
                        {   // 現状計算型ならば
                            //  通常現状計算型は、DataFusion後、時計を進めるが、此の位置におけるDataFusionでは時計を進めない（既に時計は進んでいるため）
                            csWorkTime.SetTime(m_csElementWork.GetCalModel().GetSimulationTime().GetTime()); // ｼﾐｭﾚｰﾀｰの現在時刻を一時保管
                            lLclRtn = m_csElementWork.DataFusion();
                            m_csElementWork.GetCalModel().SetSimulationTime(csWorkTime);
                            // 現状計算型ならば何もしない（上記個別要素のループ内で既にDataFusionを行う為） <-- ver1.3 で修正
                        }
                        else
                        {
                            // 現状計算型以外ならば
                            // 演算結果公開
                            lLclRtn = m_csElementWork.DataFusion();
                        }
                        if (lLclRtn != 0)
                        {   // 異常発生ならば
                            McStructErrorInfo csErrorInfo = HySCommonInfoHash.GetCorrespondData(McDefine.CALCULATION_ERROR_INF, this.m_csElement.GetOwnerProjectID()) as McStructErrorInfo;
                            if (csErrorInfo != null)
                            {
                               // ver1.5 エラートレース日本語対応
                                csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                                        Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString()
                                        + ".DataFusion() ::" + Properties.HymcoImplResources.STATEMENT_RET_CODE + "(" + lLclRtn.ToString() + ")");
                               // csErrorInfo.AddSimuErrorData(m_csElementWork.GetID(), McModelLibraryDefine.HYM_MODEL_SUBSYSTEM,
                               //         "Error return from " + m_csElementWork.GetCalModel().ToString() + ".DataFusion() :: return_code(" + lLclRtn.ToString() + ")");
                            }
                           // ver1.5 エラートレース日本語対応
                            McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "DataFusion",
                                Properties.HymcoImplResources.STATEMENT_ERR_RETURN_FORM + m_csElementWork.GetCalModel().ToString() 
                                + ".DataFusion() ::" + Properties.HymcoImplResources.STATEMENT_RET_CODE + "(" + lLclRtn.ToString() + ")");
                           // McLog.DebugOut(this.m_csSimTime, this.m_csElement.GetID(), McLog.SYSTEM, "DataFusion",
                           //     "Error return from " + m_csElementWork.GetCalModel().ToString() + ".DataFusion() :: return_code(" + lLclRtn.ToString() + ")");
                            lRtn = -1;
                            break;
                        }
                        m_csElmNode = m_csElmNode.Next;
                    } // end of for(要素数)
                }
            }

            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>要素モデルの収束判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = EachElmIsConverged(ref csElement)</para>
        /// </example>
        /// <param name="csElement">収束判定要素</param>
        /// <returns>=true:収束完了 =false:未収束</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual bool EachElmIsConverged(ref McElement csElement)
        {
            return csElement.IsConverged();
        }
         
        /// <summary><para>method outline:</para>
        /// <para>内部モデルの演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = AdjustTranInfoTime(csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual long AdjustTranInfoTime(ref McCnvrgncIterationInfoBase csConvergeCondInfo)
        {
            HySTimeSeriesBase csSeriesDt = null;
            long lCnctNum = csConvergeCondInfo.GetConnectionNum();
            for (long lLp = 0; lLp < lCnctNum; lLp++)
            {
                csSeriesDt = csConvergeCondInfo.GetConnectionTranInfo(lLp) as HySTimeSeriesBase;
                if (csSeriesDt != null)
                {
                    csSeriesDt.RemoveAfterData(this.GetSimulationTime());
                }
            }
            return 0;
        }
        
        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsConverged(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>=true:収束完了 =false:未収束</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>McConvergenceCtl.IsConverged() メソッドの実装</para>
        /// </remarks>
        public virtual bool IsConverged(ref McConvergenceInfo csConvergeCondInfo)
        {
            return IsConverged(ref m_csCnvItrInfo);
        }
        
        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別初期設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetUpConvergenceInf(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>系全体で次の計算を行う場合に呼ばれる</para>
        /// </remarks>
        public virtual void SetUpConvergenceInf(ref McConvergenceInfo csConvergeCondInfo)
        {
            this.SetUpConvergenceInf(ref m_csCnvItrInfo);
        }
        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束値変更</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ModifyConvergenceInf(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>系全体で次の計算を行う場合に呼ばれる</para>
        /// </remarks>
        public virtual void ModifyConvergenceInf(ref McConvergenceInfo csConvergeCondInfo)
        {
            this.ModifyConvergenceInf(ref m_csCnvItrInfo);
        }
        /// <summary><para>method outline:</para>
        /// <para>計算前状態復帰直後処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>PostRemenber(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>収束演算中に計算前状態へ戻された直後に呼ばれる</para>
        /// </remarks>
        public virtual void PostRemenber(ref McConvergenceInfo csConvergeCondInfo)
        {
            this.PostRemenber(ref m_csCnvItrInfo);
        }
        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsConverged(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>=true:収束完了 =false:未収束</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>収束演算内で呼ばれる収束判定（this.Calculate() からコールされる）</para>
        /// </remarks>
        protected abstract bool IsConverged(ref McCnvrgncIterationInfoBase csConvergeCondInfo);
        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetUpConvergenceInf(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>系全体で次の計算を行う場合に呼ばれる</para>
        /// </remarks>
        protected abstract void SetUpConvergenceInf(ref McCnvrgncIterationInfoBase csConvergeCondInfo);
        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別変更</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ModifyConvergenceInf(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>系全体で次の計算を行う場合に呼ばれる</para>
        /// </remarks>
        protected abstract void ModifyConvergenceInf(ref McCnvrgncIterationInfoBase csConvergeCondInfo);


        /// <summary><para>method outline:</para>
        /// <para>計算前状態復帰直後処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>PostRemenber(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>収束演算中に計算前状態へ戻された直後に呼ばれる</para>
        /// </remarks>
        //protected abstract void PostRemenber(ref McCnvrgncIterationInfoBase csConvergeCondInfo); 
        protected virtual void PostRemenber(ref McCnvrgncIterationInfoBase csConvergeCondInfo)
        {
            // Do Nothing
        }
        

        /// <summary><para>method outline:</para>
        /// <para>非収束時の処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = NotConvergedOperation(ref csConvergeCondInfo)</para>
        /// </example>
        /// <param name="csConvergeCondInfo">収束条件情報</param>
        /// <returns>long </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>規定回数トライしたものの結局収束できなかった場合に　行うベキ処理を記述</para>
        /// </remarks>
        protected abstract long NotConvergedOperation(ref McCnvrgncIterationInfoBase csConvergeCondInfo);

        /// <summary><para>method outline:</para>
        /// <para>モデル内部の時刻を進める</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GainSimulationTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void GainSimulationTime()
        {
            // Do Nothing
            //m_csSimTime += m_csDltTime;
        }

        //================
        // 各種設定メソッド
        //================
        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetProperty(csPropertyInfo)</para>
        /// </example>
        /// <param name="csPropertyInfo">プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SetProperty(McPropertyInfoRoot csPropertyInfo)
        {
            bool bRtn = false;
            if (this.m_csCnvItrInfo != null)
            {
                this.m_csCnvItrInfo.SetCtlPropertyInfo(csPropertyInfo as McPropertyInfo); // <-- プロパティ情報を覚えておく
                bRtn = this.SetCnvrgncProperty(csPropertyInfo);

            }
            return true;
        }

        //★★★★★★★★★★★★★★★★★★★★★★★★★★★
        // Ver1.4で追加(プロパティ情報設定時のメッセージスロー)
        //★★★★★★★★★★★★★★★★★★★★★★★★★★★
        /// <summary><para>method outline:</para>
        /// <para>モデル構築時にプロパティ情報を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetProperty(csPropertyInfo, ref csErrorInfo)</para>
        /// </example>
        /// <param name="csPropertyInfo">プロパティ情報</param>
        /// <param name="csErrorInfo">エラー情報</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SetProperty(McPropertyInfoRoot csPropertyInfo, ref McStructErrorInfo csErrorInfo)
        {
            bool bRtn = false;
            if (this.m_csCnvItrInfo != null)
            {
                this.m_csCnvItrInfo.SetCtlPropertyInfo(csPropertyInfo as McPropertyInfo); // <-- プロパティ情報を覚えておく
                bRtn = this.SetCnvrgncProperty(csPropertyInfo);

            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算実行中断中のプロパティ等情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SetOnlineProperty(csPropertyInfo)</para>
        /// </example>
        /// <param name="csPropertyInfo">プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中にプロパティ、初期値等を変更して動作させ場合等に使用する</para>
        /// </remarks>
        public override bool SetOnlineProperty(McPropertyInfoRoot csPropertyInfo)
        {
            return true;
        }


        /// <summary><para>method outline:</para>
        /// <para>収束演算用：プロパティ情報を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCnvrgncProperty(csPropertyInfo)</para>
        /// </example>
        /// <param name="csPropertyInfo">プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected abstract bool SetCnvrgncProperty(McPropertyInfoRoot csPropertyInfo);

        /// <summary><para>method outline:</para>
        /// <para>DataFusion()発行のタイミング（演算フロー制御）取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McDefine.DataFusionTiming eTm = GetDataFusionTiming()</para>
        /// </example>
        /// <param name=""> 無し</param>
        /// <returns>必ず：McDefine.DataFusionTiming GetDataFusionTiming</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McDefine.DataFusionTiming GetDataFusionTiming()
        {
            return McDefine.DataFusionTiming.ASYNCHRONOUS;
        }
        /// <summary><para>method outline:</para>
        /// <para>モデル演算中の情報格納クラスを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCalInfo(csCalInfo)</para>
        /// </example>
        /// <param name="csCalInfo">演算結果格納するクラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetCalInfo(McCalInfo csCalInfo)
        {
            base.SetCalInfo(csCalInfo);
            //m_csConvergeCondInfo = (McConvergenceInfo)csCalInfo;
            m_csCnvItrInfo = csCalInfo as McCnvrgncIterationInfoBase;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算中の情報格納クラスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McCalInfo csCalInfo = GetCalInfo()</para>
        /// </example>
        /// <param name=""></param>
        /// <returns>モデル演算中の情報格納クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McCalInfo GetCalInfo()
        {
            return m_csCnvItrInfo;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルのユニーク種別を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySkind cskind = GetModelKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>モデルの種別を取得する</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySKind GetModelKind()
        {
            return McModelLibraryDefine.HYM_MODEL_CONVERGENCE_ITERATION;
        }

        /// <summary><para>method outline:</para>
        /// <para>ファイルにモデル内情報を全て書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileOUT(csData)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileOUT(HySDataRoot csData)
        {
            if (csData == null || m_csElementList == null)
            {
                return false;
            }
            // 内部要素ごとに、計算状態保存処理を実行
            int iElmCnt = m_csElementList.Count;
            m_csElmNode = m_csElementList.First;
            bool bRtn = true;
            McCmnElementOutData csOutData = null;
            for (int iLp = 0; iLp < iElmCnt; iLp++)
            {
                m_csElementWork = m_csElmNode.Value;
                // データクラスの生成
                csOutData = m_csElementWork.CreateOutData();
                // 内部要素のFileOUT→要素演算モデルのFileOUT
                bRtn = m_csElementWork.FileOUT(csOutData);
                if (bRtn == false)
                {
                    return false;
                }
                //リストに追加
                ((McGrElementOutData)(csData)).AddCalElementOutData(csOutData);
                // インスタンスクリア
                csOutData = null;
                m_csElmNode = m_csElmNode.Next;
            }

            // 自メンバ変数の保存処理
            ((McGrElementOutData)(csData)).SetStatus(m_lStatus);
            ((McGrElementOutData)(csData)).SetNodeItemNo(m_iNodeItemNo);
            ((McGrElementOutData)(csData)).SetDltTSetFlg(m_bOwnDltTSetFlg);
            ((McGrElementOutData)(csData)).SetTimeData(McDefine.SIMULATION_TIME, m_csSimTime);
            ((McGrElementOutData)(csData)).SetTimeData(McDefine.START_TIME, m_csStartTime);
            ((McGrElementOutData)(csData)).SetTimeData(McDefine.PASSING_TIME, m_csTotalPassingTime);
            ((McGrElementOutData)(csData)).SetTimeData(McDefine.TARGET_TIME, m_csTgtTime);
            ((McGrElementOutData)(csData)).SetTimeData(McDefine.DELTA_TIME, m_csDltTime);

            // 演算モデルの保存
            ((McGrElementOutData)(csData)).SetData(m_csCnvItrInfo);

            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>ファイルからモデル情報を全て読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileIN(csData)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true :正常 , false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileIN(HySDataRoot csData)
        {
            if (csData == null || m_csElementList == null)
            {
                return false;
            }
            // 内部要素ごとに、計算状態復元処理を実行
            // 復元データ数取得
            int iElmDataCnt = (int)((McGrElementOutData)(csData)).GetDataCount();
            // 要素・データリストのカーソルを先頭にセット
            m_csElmNode = m_csElementList.First;
            ((McGrElementOutData)(csData)).SetCursorFirst();
            bool bRtn = true;
            for (int iLp = 0; iLp < iElmDataCnt; iLp++)
            {
                if (m_csElmNode == null)
                {
                    //return false;
                    // ビルド後、全要素削除が行われ保存された場合正常とする
                    // モデル変更状態となるために再ビルドが走るため
                    break;
                }
                m_csElementWork = m_csElmNode.Value;

                // リストから取得
                McCmnElementOutData csCalElementOutData = ((McGrElementOutData)(csData)).GetCalElementOutData();
                HySID csElemID = (HySID)m_csElementWork.GetID();
                // ビルド後、要素の削除が行われ保存された場合、この段階で要素数と
                // 演算データ数の不一致が発生するため、要素IDで比較し整合をとる
                if (csElemID.Equals(csCalElementOutData.GetID()) == true)
                {
                    // 演算データが演算要素に対応したデータである
                    bRtn = m_csElementWork.FileIN(csCalElementOutData);
                    if (bRtn == false)
                    {
                        return false;
                    }
                    // インスタンスクリア
                    csCalElementOutData = null;
                    ((McGrElementOutData)(csData)).MoveCursorNext();
                    m_csElmNode = m_csElmNode.Next;

                    if (m_csElmNode == null)
                    {
                        // 内部要素に対し、計算状態復元処理を完了した
                        break;
                    }
                }
                else
                {
                    // 演算要素に対応したデータではない
                    ((McGrElementOutData)(csData)).MoveCursorNext();
                }

            }

            // 自メンバ変数の復元処理
            m_lStatus = ((McGrElementOutData)(csData)).GetStatus();
            m_iNodeItemNo = ((McGrElementOutData)(csData)).GetNodeItemNo();
            m_bOwnDltTSetFlg = ((McGrElementOutData)(csData)).GetDltTSetFlg();
            m_csSimTime = (HySTime)((McGrElementOutData)(csData)).GetTimeData(McDefine.SIMULATION_TIME);
            m_csStartTime = (HySTime)((McGrElementOutData)(csData)).GetTimeData(McDefine.START_TIME);
            m_csTotalPassingTime = (HySTime)((McGrElementOutData)(csData)).GetTimeData(McDefine.PASSING_TIME);
            m_csTgtTime = (HySTime)((McGrElementOutData)(csData)).GetTimeData(McDefine.TARGET_TIME);
            m_csDltTime = (HySTime)((McGrElementOutData)(csData)).GetTimeData(McDefine.DELTA_TIME);

            // 復元後の整合性確保
            if (m_csSimTime != null)
            {
                m_csSimTime.PostDeserialize();
            }
            if (m_csStartTime != null)
            {
                m_csStartTime.PostDeserialize();
            }
            if (m_csTotalPassingTime != null)
            {
                m_csTotalPassingTime.PostDeserialize();
            }
            if (m_csTgtTime != null)
            {
                m_csTgtTime.PostDeserialize();
            }
            if (m_csDltTime != null)
            {
                m_csDltTime.PostDeserialize();
            }

            // 演算モデルの保存
            McCnvrgncIterationInfoBase csCnvItrInfo = ((McGrElementOutData)(csData)).GetData() as McCnvrgncIterationInfoBase;
            if (m_csCnvItrInfo != null)
            {
                if (csCnvItrInfo != null)
                {
                    m_csCnvItrInfo.CopyInfo(csCnvItrInfo);
                }
            }

            return bRtn;
        }

        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        // Ver1.3で追加（収束計算に必要な出力を強制） 
        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        /// <summary><para>method outline:</para>
        /// <para>収束計算に必要な初期出力を強制的に行わせる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = InitialDataFusionForConvergentCalculation(dErrorVal)</para>
        /// </example>
        /// <param name="dErrorVal">出力の目安を示す値</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override long InitialDataFusionForConvergentCalculation(double dErrorVal)
        {
            int iElmCnt = m_csElementList.Count;  // グループ内のエレメント数取得
            m_csElmNode = m_csElementList.First;
            for (int iLp = m_iNodeItemNo; iLp < iElmCnt; iLp++)
            {                  // 2-2-2 演算エレメント本体
                m_csElementWork = m_csElmNode.Value;    // エレメント取得
                m_csElementWork.InitialDataFusionForConvergentCalculation(this.m_csCnvItrInfo.GetInitErrorVal(iLp));
                m_csElmNode = m_csElmNode.Next;
            }
            return 0;
        }

    }
}


