﻿// <summary>ソースコード：ＨＹＭＣＯ折れ線グラフ画面出力制御クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;
using System.Threading;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.Interface.HSViewer;
using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSBusiProc;
using CommonMP.HYSSOP.CoreImpl.HSViewer;
using CommonMP.HYSSOP.Interface.HSSimulator;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Model;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.Interface.Controller;
using CommonMP.HYMCO.CoreImpl.Model;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.FileIO;
using CommonMP.HYMCO.CoreImpl.Data.LibraryCtl;
using CommonMP.HYMCO.CoreImpl.Data.StructInfo;

namespace CommonMP.HYMCO.CoreImpl.Model
{
    /// <summary><para>class outline:</para>
    /// <para>折れ線グラフ画面出力制御クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>
    /// [CommonMP][ver 1.0.0][2008/10/01][新規作成]
    /// </para>
    /// </remarks>
    public class McLineGraphScreen : McScreenCtl
    {
        //--------------------------------
        // モデル設定定数
        //--------------------------------
        /// <summary> 演算結果出力刻み時間キー </summary>
        public static readonly string MODEL_TIME_STEP_KEY = "TIME_STEP";
        /// <summary> 演算結果出力刻み時間 </summary>
        public static readonly double MODEL_TIME_STEP_VALUE = 600.0;
        /// <summary> 演算結果出力要否フラグキー </summary>
        public static readonly string MODEL_DISP_FLAG_KEY = "m_bDispFlg";
        /// <summary> 演算結果出力要否フラグ名称 </summary>
        public static readonly string MODEL_DISP_FLAG_NAME = Properties.HymcoImplResources.MODEL_DISPFLG;
        /// <summary> 演算結果出力要否 </summary>
        public static readonly bool MODEL_DISP_FLAG_VALUE = true;
        /// <summary> 描画周期時間(ms)キー </summary>
        public static readonly string MODEL_DISP_CYCLE_KEY = "m_lDispCycle";
        /// <summary> 描画周期時間(ms)名称 </summary>
        public static readonly string MODEL_DISP_CYCLE_NAME = Properties.HymcoImplResources.MODEL_DISPCYCLE;
        /// <summary> 描画周期時間(ms) </summary>
        public static readonly long MODEL_DISP_CYCLE_VALUE = 200;
        /// <summary> 任意入力キー </summary>
        public static readonly string ANY_VALUE_KEY = "Arbitrary Unit";
        /// <summary> 任意入力名称 </summary>
        public static readonly string ANY_VALUE_NAME = Properties.HymcoImplResources.MODEL_ARBITRARY_INPUT_PATTERN;
        /// <summary> 縦断データ入力名称 </summary>
        public static readonly string RIVERBED_SHAPE_NAME = Properties.HymcoImplResources.MODEL_VERTICAL_SECTION_INPUT;
        /// <summary> 雨量入力名称 </summary>
        public static readonly string HYETO_NAME =  Properties.HymcoImplResources.MODEL_RAIN_FALL_INPUT;
        /// <summary> 雨量キー </summary>
        public static readonly string SCALOR_RAIN_FALLE_KEY = "mm/Hour";
        /// <summary> 雨量名称 </summary>
        public static readonly string SCALOR_RAIN_FALLE_NAME = Properties.HymcoImplResources.MODEL_SCALOR_RAIN_FALL;

        //---------------------------------------
        // 演算中対象メンバ定義
        //---------------------------------------
        /// <summary>前回計算結果出力時刻</summary>
        protected HySTime m_csNxTimeStep = null;
        /// <summary>前回計算結果出力単位</summary>
        protected double m_dNxTimeStep = 0;
        /// <summary>計算結果データ件数</summary>
        protected double m_dBaseCnt = 0;
        /// <summary>モニタ画面表示有無フラグ</summary>
        protected bool m_bDispOnFlg = false;
        /// <summary>画面表示設定環境変数名称</summary>
        protected static readonly string HYMCO_MONITOR_DISPLAY = "HYMCO_MONITOR_DISPLAY";
        /// <summary>クラス名称</summary>
        protected static readonly HySObjectKind csKindName = new HySObjectKind("McLineGraphScreen");
        /// <summary>結線デフォルト値メッセージ</summary>
        protected static readonly HySString DEFAULT_VALUE_MSG = new HySString(Properties.HymcoImplResources.MODEL_DEFAULT_VALUE_INPUT);
        /// <summary>描画最低周期</summary>
        protected static readonly long m_lCycleMin = 100;

        //=========================
        // 演算実行前処理関連メソッド
        //=========================
        /// <summary><para>method outline:</para>
        /// <para>モデルを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(csInitialData,lInputDataNum,csInputCellData)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override bool Initialize(ref McPropertyInfoRoot csInitialData, long lInputDataNum, ref McReceiveCellDataIF[] csInputCellData)
        {
            bool bRtn = false;

            long lNumData = lInputDataNum;
            if (lNumData > 0)
            {  // データがある場合
                m_csDispData = null; // エリア解放
                m_csNxTimeStep = null; // エリア解放
                m_csNxTimeStep = HySTime.DEFAULT_TIME.Clone(); // 計算結果出力時刻初期化
                m_dNxTimeStep = 0;
                m_dBaseCnt = 0;
                // リストデータを１件取得
                McTranInfo csInDt = (McTranInfo)m_csInputData[0];
                if (csInDt is McTimeSeriesSingleCellTranInfo ||
                    csInDt is McTimeSeriesD1CellArrayTranInfo)
                {  // ポイント時系列情報または１次元配列時系列情報の場合
                    HySString csKeyName = HySEnvInf.GetEnvInf(HYMCO_MONITOR_DISPLAY) as HySString;
                    if (csKeyName is HySString)
                    {  // 設定が有る場合
                        if (csKeyName.Equal(new HySString("ON")) == true ||
                            csKeyName.Equal(new HySString("on")) == true)
                        {  // 表示が必要な場合
                            m_bDispOnFlg = true;
                        }
                    }
                    if (m_bDispOnFlg == true)
                    { // モニタ画面の表示が必要な場合
                        m_csDispData = new HySDispInstanceCommon(); // インスタンス生成
                        // リンクドリスト配列・ハッシュテーブル配列・次回計算結果出力時刻の生成
                        HySDataLinkedList[] csListArray = new HySDataLinkedList[lNumData];
                        HySDataHashTable[] csHashTableArray = new HySDataHashTable[lNumData];
                        HySTime[] csTimeArray = new HySTime[lNumData];
                        for (long lP = 0; lP < lNumData; lP++)
                        {
                            // エリア生成
                            csListArray[lP] = new HySDataLinkedList();
                            csHashTableArray[lP] = new HySDataHashTable();
                            csTimeArray[lP] = HySTime.DEFAULT_TIME.Clone(); // new HySTime(0);
                            GetParamItem(lP, csListArray[lP]); // 伝送プロパティの項目情報取得
                        }
                        ((HySDispInstanceCommon)m_csDispData).SetDataLinkedListArray(csListArray);
                        ((HySDispInstanceCommon)m_csDispData).SetDataHashTableArray(csHashTableArray);
                        ((HySDispInstanceCommon)m_csDispData).SetTimeArray(csTimeArray);
                        // 描画周期・表示要否の設定
                        SetCycleData();
                        // 表示業務にデータインスタンスを通知
                        this.DataInstanceNotificationEvent();
                    }
                    bRtn = true;
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>出力側の接続情報チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SendConnectionCheck(ref csErrorInf)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <param name="lOutputDataNum">出力情報数</param>
        /// <param name="csOutputCellData">出力情報配列</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>送信端子に設定されている伝送データが自モデルが期待している情報か否かをチェックする</para>
        /// </remarks>
        protected override bool SendConnectionCheck(ref McStructErrorInfo csErrorInf, long lOutputDataNum, McSendCellDataIF[] csOutputCellData)
        {
            bool bRtn = true;
            if (lOutputDataNum > 0)
            {
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName,
                     Properties.HymcoImplResources.STATEMENT_MUST_NOT_HAVE_ELEMENT );
               // csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName, "McScreenCtl Element must not have Send Ports.");
                bRtn = false;
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算実行中断中のモデルを初期化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(ref csInitialData, ref lInputDataNum, ref csOutputDataList)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <param name="lInputDataNum">入力データ数</param>
        /// <param name="csInputCellData">入力セル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中に 初期値情報を変更した場合にコールされる</para>
        /// </remarks>
        protected override bool OnlineInitialize(ref McPropertyInfoRoot csInitialData, long lInputDataNum, ref McReceiveCellDataIF[] csInputCellData)
        {
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>入力側の接続情報チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReceiveConnectionCheck(ref csErrorInf, lInputDataNum, csInputCellData)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>受信するデータが自モデルが期待している情報か否かをチェックする</para>
        /// </remarks>
        protected override bool ReceiveConnectionCheck(ref McStructErrorInfo csErrorInf, long lInputDataNum, McReceiveCellDataIF[] csInputCellData)
        {
            bool bRtn = true;

            if (lInputDataNum == 0)
            {  // 要素間伝送データが無い場合
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName, 
                                             Properties.HymcoImplResources.STATEMENT_NO_REC_PORT );
               // csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName, "No Receive Port.");
                bRtn = false;
            }
            else if (lInputDataNum > 2)
            {  // 要素間伝送データが複数接続されている場合
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName,
                                             Properties.HymcoImplResources.STATEMENT_MANY_REC_PORT );
               // csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName,
               //                              "Too many Receive Ports. (Not support Receive Ports of two or more)");
                bRtn = false;
            }
            else if (lInputDataNum <= 2)
            {  // 要素間伝送データが２件以内の場合
                HySDataLinkedList csPatternID = new HySDataLinkedList();
                for (int iLp = 0; iLp < lInputDataNum; iLp++)
                {  // 要素間伝送データ数分繰り返す
                    long lD1 = 0; long lD2 = 0; long lD3 = 0; long lCellDataNum = 0;
                    long lCellDim = csInputCellData[iLp].GetDimension(ref lD1, ref lD2, ref lD3, ref lCellDataNum);
                    if (lCellDim == 1 && lD1 >= 1)
                    {   // １次元の場合
                        McTranInfoPattern csTranInfoPatternProperty = csInputCellData[iLp].GetReceiveTranInfoPattern() as McTranInfoPattern;
                        HySID csReceiveID = csTranInfoPatternProperty.GetPatternID(); // 受信パターンＩＤ取得
                        csPatternID.AddLast(csReceiveID.Clone()); // 受信パターンＩＤ登録
                        if (m_csInputData[iLp] is McTimeSeriesSingleCellTranInfo)
                        {  // ポイント時系列情報の場合
                            if (csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_RAIN_SERIAL) == true)
                            {  // ハイエトデータ入力の場合
                                McTimeSeriesSingleCellTranInfo csSerialTran = m_csInputData[iLp] as McTimeSeriesSingleCellTranInfo;
                                HySDataCharacteristicInCell csSDataCharacteristicInCell = csTranInfoPatternProperty.GetCellDataCharacteristic();
                                HySCellDataGetter csCellDataGetter = csInputCellData[iLp].GetCellDataGetter();
                                for (long lP = 0; lP < csSDataCharacteristicInCell.m_lDim; lP++)
                                {
                                    long lIdx = csCellDataGetter.GetIdx(lP);
                                    if (lIdx >= 0)
                                    {  // 入力情報設定の場合
                                        if (csSDataCharacteristicInCell.m_csDataKind[lP].Equals(((HySDataCharacteristicInCell)csSerialTran.GetCellDataCharacteristic()).m_csDataKind[lIdx]) == false)
                                        {  // データ種別に不一致が有る場合
                                           // ver1.5 エラートレース日本語対応
                                            csErrorInf.AddCheckWarningData(this.m_csElement.GetID(), csKindName,
                                                                           Properties.HymcoImplResources.STATEMENT_NO_DATA_KIND_SISAGREEMENT );
                                           // csErrorInf.AddCheckWarningData(this.m_csElement.GetID(), csKindName,
                                           //                                "Connected data kind is a disagreement.");
                                            break;
                                        }
                                    }
                                }
                            }
                        }
                        else if (m_csInputData[iLp] is McTimeSeriesD1CellArrayTranInfo)
                        {  // 1次元配列時系列情報の場合
                        }
                    }
                    else
                    {  // １次元以外の場合
                        bRtn = false;
                       // ver1.5 エラートレース日本語対応
                        csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName ,
                            Properties.HymcoImplResources.STATEMENT_UNEXPECT_REC_DATA_TYPE + m_csInputData[iLp].GetUpperElementID().ToString() + ")");
                       // csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName
                       //     , "Unexpected receive data type. (Received from " + m_csInputData[iLp].GetUpperElementID().ToString() + ")");
                    }
                }
                if (bRtn == true)
                {  // 接続要素配列が妥当な場合
                    HySID csVierFormID = HySMoniterScreen.GetViewFormID(csPatternID);
                    if (csVierFormID == null)
                    {  // 接続要素の組合せが不当な場合
                        bRtn = false;
                       // ver1.5 エラートレース日本語対応
                        csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName,
                                                     Properties.HymcoImplResources.STATEMENT_REC_DATA_TYPE_UNJUSTIFIED );
                       // csErrorInf.AddCheckErrorData(this.m_csElement.GetID(), csKindName,
                       //                            "The combination of the receive data type is unjustified. ");
                    }
                }
                csPatternID = null; // スタック解放
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>伝送プロパティの項目情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetParamItem( )</para>
        /// </example>
        /// <param name="lInputData">伝送プロパティの配列番号（０相対）</param>
        /// <param name="csListArray">リンクドリストインスタンス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void GetParamItem(long lInputData, HySDataLinkedList csListArray)
        {
            // 初期設定
            long lP = 0;
            long lIdx = 0;
            long lDim = 0;
            HySDataCharacteristicInCell csInputKind = null;

            // リストデータをｎ件取得
            if (lInputData >= 0 && lInputData < m_lInputDataNum)
            {
                // 伝送データの基本クラスのキャスト
                McReceiveCellDataIF csInputCellData = m_InputCellData[lInputData];
                HySCellDataGetter csCellDataGetter = csInputCellData.GetCellDataGetter();
                // 伝送データの受信パターンＩＤ取得
                McTranInfoPattern csTranInfoPatternProperty = csInputCellData.GetReceiveTranInfoPattern() as McTranInfoPattern;
                HySID csReceiveID = csTranInfoPatternProperty.GetPatternID();
                // 伝送データの配列数取得
                HySDataCharacteristicInCell csSDataCharacteristicInCell = csTranInfoPatternProperty.GetCellDataCharacteristic();
                if (m_csInputData[lInputData] is McTimeSeriesSingleCellTranInfo)
                {  // ポイント時系列情報の場合
                    if (csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_RAIN_SERIAL) == true ||
                        csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_SINGLE_CELL_SERIAL) == true ||
                        csReceiveID.Equals(HySMoniterScreen.LINE_GRAPH_SINGLE_CELL_SERIAL) == true ||
                        csReceiveID.Equals(HySMoniterScreen.LINE_GRAPH_SINGLE_CELL_ANY) == true)
                    {  // 任意入力またはハイエト入力の場合
                        McTimeSeriesSingleCellTranInfo csSerialTran = m_csInputData[lInputData] as McTimeSeriesSingleCellTranInfo;
                        lDim = csSDataCharacteristicInCell.m_lDim; // 入力配列数設定
                        csListArray.AddLast(csReceiveID);          // 受信パターンＩＤ設定
                        csListArray.AddLast(new HySString(lDim.ToString())); // セル配列数設定
                        // データ識別取得
                        csCellDataGetter = csInputCellData.GetCellDataGetter();
                        csInputKind = new HySDataCharacteristicInCell(lDim);
                        for (lP = 0; lP < lDim; lP++)
                        {
                            lIdx = csCellDataGetter.GetIdx(lP);
                            if (lIdx < 0)
                            {  // ディフォルト設定の場合
                                csInputKind.m_csLabel[lP] = DEFAULT_VALUE_MSG.ToString();
                                csInputKind.m_csDataKind[lP] = csSDataCharacteristicInCell.m_csDataKind[lP].Clone();
                            }
                            else
                            {  // 入力情報設定の場合
                                if (csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_RAIN_SERIAL) == true)
                                {  // ハイエト入力の場合
                                    csInputKind.m_csLabel[lP] = (string)csSDataCharacteristicInCell.m_csLabel[lP].Clone();
                                    csInputKind.m_csDataKind[lP] = csSDataCharacteristicInCell.m_csDataKind[lP].Clone();
                                }
                                else
                                {  // 任意入力の場合
                                    csInputKind.m_csLabel[lP] = (string)((HySDataCharacteristicInCell)csSerialTran.GetCellDataCharacteristic()).m_csLabel[lIdx].Clone();
                                    csInputKind.m_csDataKind[lP] = ((HySDataCharacteristicInCell)csSerialTran.GetCellDataCharacteristic()).m_csDataKind[lIdx].Clone();
                                }
                            }
                        }
                        csListArray.AddLast(csInputKind); // 伝送プロパティの項目情報設定
                    }
                }
                else if (m_csInputData[lInputData] is McTimeSeriesD1CellArrayTranInfo)
                {  // １次元配列時系列情報の場合
                    if (csReceiveID.Equals(HySMoniterScreen.LINE_GRAPH_D1_CELL_SERIAL) == true ||
                        csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_D1_CELL_SERIAL) == true ||
                        csReceiveID.Equals(HySMoniterScreen.RIVERBED_SHAPE_D1_CELL_SERIAL) == true)
                    {  // 任意入力または縦断データ入力の場合
                        McTimeSeriesD1CellArrayTranInfo csCellD1Trn = m_csInputData[lInputData] as McTimeSeriesD1CellArrayTranInfo;
                        lDim = csSDataCharacteristicInCell.m_lDim; // 入力配列数設定
                        csListArray.AddLast(csReceiveID);          // 受信パターンＩＤ設定
                        csListArray.AddLast(new HySString(csCellD1Trn.GetDimension().ToString())); // セル配列数設定
                        // データ識別取得
                        csCellDataGetter = csInputCellData.GetCellDataGetter();
                        csInputKind = new HySDataCharacteristicInCell(lDim);
                        for (lP = 0; lP < lDim; lP++)
                        {
                            lIdx = csCellDataGetter.GetIdx(lP);
                            if (lIdx < 0)
                            {  // ディフォルト設定の場合
                                csInputKind.m_csLabel[lP] = DEFAULT_VALUE_MSG.ToString();
                                csInputKind.m_csDataKind[lP] = csSDataCharacteristicInCell.m_csDataKind[lP].Clone();
                            }
                            else
                            {  // 任意入力情報設定の場合
                                csInputKind.m_csLabel[lP] = (string)((HySDataCharacteristicInCell)csCellD1Trn.GetCellDataCharacteristic()).m_csLabel[lIdx].Clone();
                                csInputKind.m_csDataKind[lP] = ((HySDataCharacteristicInCell)csCellD1Trn.GetCellDataCharacteristic()).m_csDataKind[lIdx].Clone();
                            }
                        }
                        csListArray.AddLast(csInputKind); // 伝送プロパティの項目情報設定
                    }
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>データ周期情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCycleData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void SetCycleData()
        {
            if (m_csDispData is HySDispInstanceCommon)
            {  // グラフデータ有効の場合
                long[] lList = ((HySDispInstanceCommon)m_csDispData).GetLongArray();
                if (lList == null)
                {  // エリアが無効な場合
                    ((HySDispInstanceCommon)m_csDispData).SetLongArray(new long[2]);
                    lList = ((HySDispInstanceCommon)m_csDispData).GetLongArray();
                }
                lList[0] = (m_lDispCycle >= m_lCycleMin) ? m_lDispCycle : m_lCycleMin; // 描画周期設定
                lList[1] = (m_bDispFlg == true) ? 1 : 0; // 表示要否設定
            }
        }

        //=======================
        // 演算実行処理関連メソッド
        //=======================

        /// <summary><para>method outline:</para>
        /// <para>モデル演算結果を外部のエレメントに対して公開する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = DataFusion( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override long DataFusion()
        {
            // 初期設定
            HySTime csTim = null;
            HySTime csLastTim = null;
            double[] dDt;
            int iLp = 0;
            long lSerialNum = 0;
            long lDataNum = 0;
            long lP = 0;
            long lP1 = 0;
            long lP2 = 0;
            long lDim = 0;
            long lCellDim = 0;
            long lInputData = 0;
            bool bDataFlg = false;
            long lTblCnt = 0;
            double dXUnderMin = 0.0;
            double dXUnderMax = 0.0;
            double dYLeftMin = 0.0;
            double dYLeftMax = 0.0;
            double dYRightMin = 0.0;
            double dYRightMax = 0.0;
            bool bFastFlg = false;
            McReceiveCellDataIF csInputCellData = null;
            McTranInfoPattern csTranInfoPatternProperty = null;
            HySDataCharacteristicInCell csSDataCharacteristicInCell = null;
            McTimeSeriesSingleCellTranInfo csSerialTran = null;
            McTimeSeriesD1CellArrayTranInfo csCellD1Trn = null;
            HySCellDataGetter csCellDataGetter = null;
            HySID csReceiveID = null;
            HySCellData[] csCell_New = null;
            HySDataLinkedList[] csListArray = null;
            HySDataHashTable[] csHashTableArray = null;
            HySTime[] csTimeArray = null;
            HySDataHashTable csDataList = null;
            HySD1CellArrayData csCellData = null;
            HySCellData csDt = null;

            // グラフクラスキャスト
            if (m_csDispData is HySDispInstanceCommon)
            {  // グラフデータ有効の場合
                csListArray = ((HySDispInstanceCommon)m_csDispData).GetDataLinkedListArray();     // リストインスタンス取得
                csHashTableArray = ((HySDispInstanceCommon)m_csDispData).GetDataHashTableArray(); // ハッシュインスタンス取得
                csTimeArray = ((HySDispInstanceCommon)m_csDispData).GetTimeArray(); // 次回計算結果出力時刻インスタンス取得
                if (((HySDispInstanceCommon)m_csDispData).GetDoubleArray() == null)
                {  // 初回の場合
                    SetScaleData(dXUnderMin, dXUnderMax, dYLeftMin, dYLeftMax, dYRightMin, dYRightMax);
                    bFastFlg = true;
                }
                else
                {  // データ継続の場合
                    GetScaleData(ref dXUnderMin, ref dXUnderMax, ref dYLeftMin ,
                                 ref dYLeftMax , ref dYRightMin, ref dYRightMax);
                }
                // リストデータをｎ件取得
                for (lInputData = 0; lInputData < m_lInputDataNum; lInputData++)
                {  // 伝送データ件数分繰り返す
                    m_csNxTimeStep.SetTime(csTimeArray[lInputData]); // 次回計算結果出力時刻設定
                    csDataList = csHashTableArray[lInputData];       // セルデータインスタンス取得
                    lTblCnt = csDataList.GetCount();                 // 前回のデータ件数取得
                    // 伝送データの基本クラスのキャスト
                    csInputCellData = m_InputCellData[lInputData];
                    csCellDataGetter = csInputCellData.GetCellDataGetter();
                    // 伝送データの受信パターンＩＤ取得
                    csTranInfoPatternProperty = csInputCellData.GetReceiveTranInfoPattern() as McTranInfoPattern;
                    csReceiveID = csTranInfoPatternProperty.GetPatternID();
                    // 伝送データの配列数取得
                    csSDataCharacteristicInCell = csTranInfoPatternProperty.GetCellDataCharacteristic();
                    if (m_csInputData[lInputData] is McTimeSeriesSingleCellTranInfo)
                    {  // ポイント時系列情報の場合
                        if (csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_RAIN_SERIAL) == true ||
                            csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_SINGLE_CELL_SERIAL) == true ||
                            csReceiveID.Equals(HySMoniterScreen.LINE_GRAPH_SINGLE_CELL_SERIAL) == true ||
                            csReceiveID.Equals(HySMoniterScreen.LINE_GRAPH_SINGLE_CELL_ANY) == true)
                        {  // 任意入力またはハイエト入力の場合
                            csSerialTran = m_csInputData[lInputData] as McTimeSeriesSingleCellTranInfo;
                            lCellDim = csSDataCharacteristicInCell.m_lDim; // 伝送元セル配列数設定
                            lDim = csSDataCharacteristicInCell.m_lDim; // 入力配列数設定
                            // データ識別取得
                            csCellDataGetter = csInputCellData.GetCellDataGetter();
                            // セルデータ取得
                            if (lTblCnt <= 0)
                            {  // 最終データが無効な場合
                                csLastTim = HySTime.DEFAULT_TIME.Clone();
                            }
                            else
                            {
                                csCellData = csDataList[new HySString((lTblCnt - 1).ToString())] as HySD1CellArrayData;
                                if (csCellData is HySD1CellArrayData)
                                {  // 最終データが有効な場合
                                    csLastTim = csCellData.GetLastTime();
                                }
                                else
                                {  // 最終データが無効な場合
                                    csLastTim = HySTime.DEFAULT_TIME.Clone();
                                }
                            }
                            // 演算データ取得準備
                            HySCellData csCell = null;
                            lSerialNum = csSerialTran.GetDataDimentionInCell();
                            dDt = new double[lSerialNum];
                            if (csLastTim is HySTime)
                            {  // 時刻データ有効の場合
                                if (csLastTim.GetTime() <= HySTime.DEFAULT_TIME.GetTime())
                                {  // 初回データの場合
                                    csSerialTran.SetCursorFirst();
                                    lDataNum = csSerialTran.GetCount();
                                }
                                else
                                {  // データ継続の場合
                                    lDataNum = csSerialTran.SetCursorPlus(csLastTim);
                                }
                            }
                            else
                            {  // 初回データの場合
                                csSerialTran.SetCursorFirst();
                                lDataNum = csSerialTran.GetCount();
                            }
                            if (m_csNxTimeStep.GetTime() <= HySTime.DEFAULT_TIME.GetTime())
                            {  // 演算初回の場合
                                if (lDataNum > 0)
                                {  // データ件数が有る場合
                                    m_csNxTimeStep.SetTime(csSerialTran.GetCursorTime().GetTime()); // 出力時刻初期設定
                                }
                                //else
                                //{
                                //    // Do Nothing
                                //}
                            }
                            for (lP = 0; lP < lDataNum; lP++)
                            {
                                // カーソル位置データ取得
                                csTim = csSerialTran.GetCursorTime(); // 時刻
                                if (csTim.Before(m_csNxTimeStep) == false)
                                {  // 出力時刻データの場合
                                    csCell = csSerialTran.GetCursorData(); // 時系列データ取得
                                    if (csCell != null)
                                    {
                                        // データ追加・更新
                                        csCellData = new HySD1CellArrayData(1, lCellDim);
                                        if (csCellData is HySD1CellArrayData)
                                        {  // 配列フィールドは有効な場合
                                            if (bFastFlg == true)
                                            {  // 時間設定初回の場合
                                                dXUnderMin = csTim.GetTime();
                                                dXUnderMax = csTim.GetTime();
                                            }
                                            else
                                            {  // データ継続の場合
                                                if (csTim.GetTime() > dXUnderMax)
                                                {  // 最大値更新が要の場合
                                                    dXUnderMax = csTim.GetTime();
                                                }
                                                if (csTim.GetTime() < dXUnderMin)
                                                {  // 最小値更新が要の場合
                                                    dXUnderMin = csTim.GetTime();
                                                }
                                            }
                                            csCellData.SetTime(csTim.Clone());
                                            csDt = new HySCellData(lCellDim);
                                            for (lP1 = 0; lP1 < lCellDim; lP1++)
                                            {
                                                csDt.m_dData[lP1] = csInputCellData.Data(csCell, lP1);
                                            }
                                            csCell_New = new HySCellData[1];
                                            csCell_New[0] = csDt;
                                            csCellData.SetCellData(csCell_New);
                                            if (csCell_New[0].m_lDim > 0)
                                            {  // データが有る場合
                                                for (iLp = 0; iLp < csCell_New[0].m_lDim; iLp++)
                                                {
                                                    if (csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_RAIN_SERIAL) == true)
                                                    {  // ハイエト入力の場合
                                                        if (bFastFlg == true)
                                                        {  // 時間設定初回の場合
                                                            dYRightMin = csCell_New[0].m_dData[0];
                                                            dYRightMax = csCell_New[0].m_dData[0];
                                                        }
                                                        if (csCell_New[0].m_dData[iLp] > dYRightMax)
                                                        {  // 最大値更新が要の場合
                                                            dYRightMax = csCell_New[0].m_dData[iLp];
                                                        }
                                                        if (csCell_New[0].m_dData[iLp] < dYRightMin)
                                                        {  // 最小値更新が要の場合
                                                            dYRightMin = csCell_New[0].m_dData[iLp];
                                                        }
                                                    }
                                                    else
                                                    {  // 任意入力の場合
                                                        if (bFastFlg == true)
                                                        {  // 時間設定初回の場合
                                                            dYLeftMin = csCell_New[0].m_dData[0];
                                                            dYLeftMax = csCell_New[0].m_dData[0];
                                                        }
                                                        if (csCell_New[0].m_dData[iLp] > dYLeftMax)
                                                        {  // 最大値更新が要の場合
                                                            dYLeftMax = csCell_New[0].m_dData[iLp];
                                                        }
                                                        if (csCell_New[0].m_dData[iLp] < dYLeftMin)
                                                        {  // 最小値更新が要の場合
                                                            dYLeftMin = csCell_New[0].m_dData[iLp];
                                                        }
                                                    }
                                                }
                                            }
                                            csDataList.AddObject(new HySString(lTblCnt.ToString()), csCellData);
                                            lTblCnt++;
                                            SetScaleData(dXUnderMin, dXUnderMax, dYLeftMin, dYLeftMax, dYRightMin, dYRightMax);
                                            bFastFlg = false;
                                        }
                                    }
                                    m_csNxTimeStep.SetTime(m_csNxTimeStep.GetTime() + m_dTimeStep); // 出力時刻更新
                                    csTimeArray[lInputData].SetTime(m_csNxTimeStep); // 次回計算結果出力時刻設定
                                    bDataFlg = true; // データ更新記憶
                                }
                                if (lP < (lDataNum - 1))
                                {  // 最終レコード以外の場合
                                    csSerialTran.MoveCursorNext();
                                }
                            }
                        }
                    }
                    else if (m_csInputData[lInputData] is McTimeSeriesD1CellArrayTranInfo)
                    {  // １次元配列時系列情報の場合
                        if (csReceiveID.Equals(HySMoniterScreen.LINE_GRAPH_D1_CELL_SERIAL) == true ||
                            csReceiveID.Equals(HySMoniterScreen.HYETO_GRAPH_D1_CELL_SERIAL) == true ||
                            csReceiveID.Equals(HySMoniterScreen.RIVERBED_SHAPE_D1_CELL_SERIAL) == true)
                        {  // 任意入力または縦断データ入力の場合
                            csCellD1Trn = m_csInputData[lInputData] as McTimeSeriesD1CellArrayTranInfo;
                            lCellDim = csCellD1Trn.GetDimension(); // 伝送元セル配列数設定
                            lDim = csSDataCharacteristicInCell.m_lDim; // 入力配列数設定
                            // データ識別取得
                            csCellDataGetter = csInputCellData.GetCellDataGetter();
                            // セルデータ取得
                            if (lTblCnt <= 0)
                            {  // 最終データが無効な場合
                                csLastTim = HySTime.DEFAULT_TIME.Clone(); // new HySTime(0);
                            }
                            else
                            {
                                csCellData = csDataList[new HySString((lTblCnt - 1).ToString())] as HySD1CellArrayData;
                                if (csCellData is HySD1CellArrayData)
                                {  // 最終データが有効な場合
                                    csLastTim = csCellData.GetLastTime();
                                }
                                else
                                {  // 最終データが無効な場合
                                    csLastTim = HySTime.DEFAULT_TIME.Clone();
                                }
                            }
                            // 演算データ取得準備
                            HySCellData[] csCell = null;
                            lSerialNum = csCellD1Trn.GetDataDimentionInCell();
                            dDt = new double[lSerialNum];
                            if (csLastTim is HySTime)
                            {  // 時刻データ有効の場合
                                if (csLastTim.GetTime() <= HySTime.DEFAULT_TIME.GetTime())
                                {  // 初回データの場合
                                    csCellD1Trn.SetCursorFirst();
                                    lDataNum = csCellD1Trn.GetCount();
                                }
                                else
                                {  // データ継続の場合
                                    lDataNum = csCellD1Trn.SetCursorPlus(csLastTim);
                                }
                            }
                            else
                            {  // 初回データの場合
                                csCellD1Trn.SetCursorFirst();
                                lDataNum = csCellD1Trn.GetCount();
                            }
                            if (m_csNxTimeStep.GetTime() <= HySTime.DEFAULT_TIME.GetTime())
                            {  // 演算初回の場合
                                if (lDataNum > 0)
                                {  // データ件数が有る場合
                                    m_csNxTimeStep.SetTime(csCellD1Trn.GetCursorTime().GetTime()); // 出力時刻初期設定
                                }
                                //else
                                //{
                                //    // Do Nothing
                                //}
                            }
                            for (lP = 0; lP < lDataNum; lP++)
                            {
                                // カーソル位置データ取得
                                csTim = csCellD1Trn.GetCursorTime(); // 時刻
                                if (csTim.Before(m_csNxTimeStep) == false)
                                {  // 出力時刻データの場合
                                    csCell = ((McD1CellArrayTranInfo)csCellD1Trn.GetCursorData()).GetCellData(); // 時系列データ取得
                                    if (csCell != null)
                                    {
// 海老沢問題　このの部分の処理は間違っている　new HySD1CellArrayData(lCellDim, 1); となるべき　何故一つのセル無いのデータが
//            lCellDim = csCellD1Trn.GetDimension();　で取得した値でなければならないか？
                                        // データ追加・更新
                                        csCellData = new HySD1CellArrayData(1, lCellDim);
                                        if (csCellData is HySD1CellArrayData)
                                        {  // 配列フィールドが有効な場合
                                            if (bFastFlg == true)
                                            {  // 時間設定初回の場合
                                                dXUnderMin = csTim.GetTime();
                                                dXUnderMax = csTim.GetTime();
                                            }
                                            else
                                            {  // データ継続の場合
                                                if (csTim.GetTime() > dXUnderMax)
                                                {  // 最大値更新が要の場合
                                                    dXUnderMax = csTim.GetTime();
                                                }
                                                if (csTim.GetTime() < dXUnderMin)
                                                {  // 最小値更新が要の場合
                                                    dXUnderMin = csTim.GetTime();
                                                }
                                            }
                                            csCellData.SetTime(csTim.Clone());
                                            csCell_New = new HySCellData[lDim];
                                            for (lP2 = 0; lP2 < lDim; lP2++)
                                            {
                                                csDt = new HySCellData(lCellDim);
                                                for (lP1 = 0; lP1 < lCellDim; lP1++)
                                                {
                                                    csDt.m_dData[lP1] = csInputCellData.Data(csCell[lP1], lP2);
                                                }
                                                csCell_New[lP2] = csDt;
                                                if (csCell_New[lP2].m_lDim > 0)
                                                {  // データが有る場合
                                                    int iCount = 0;
                                                    if (bFastFlg == true)
                                                    {  // 時間設定初回の場合
                                                        dYLeftMin = csCell_New[lP2].m_dData[0];
                                                        dYLeftMax = csCell_New[lP2].m_dData[0];
                                                        iCount = 1;
                                                    }
                                                    for (iLp = iCount; iLp < csCell_New[lP2].m_lDim; iLp++)
                                                    {
                                                        if (bFastFlg == true)
                                                        {  // 時間設定初回の場合
                                                            dYLeftMin = csCell_New[lP2].m_dData[0];
                                                            dYLeftMax = csCell_New[lP2].m_dData[0];
                                                        }
                                                        if (csCell_New[lP2].m_dData[iLp] > dYLeftMax)
                                                        {  // 最大値更新が要の場合
                                                            dYLeftMax = csCell_New[lP2].m_dData[iLp];
                                                        }
                                                        if (csCell_New[lP2].m_dData[iLp] < dYLeftMin)
                                                        {  // 最小値更新が要の場合
                                                            dYLeftMin = csCell_New[lP2].m_dData[iLp];
                                                        }
                                                    }
                                                }
                                            }
                                            csCellData.SetCellData(csCell_New);
                                            csDataList.AddObject(new HySString(lTblCnt.ToString()), csCellData);
                                            lTblCnt++;
                                            SetScaleData(dXUnderMin, dXUnderMax, dYLeftMin, dYLeftMax, dYRightMin, dYRightMax);
                                            bFastFlg = false;
                                        }
                                    }
                                    m_csNxTimeStep.SetTime(m_csNxTimeStep.GetTime() + m_dTimeStep); // 出力時刻更新
                                    csTimeArray[lInputData].SetTime(m_csNxTimeStep); // 次回計算結果出力時刻設定
                                    bDataFlg = true; // データ更新記憶
                                }
                                if (lP < (lDataNum - 1))
                                {  // 最終レコード以外の場合
                                    csCellD1Trn.MoveCursorNext();
                                }
                            }
                        }
                    }
                }
                if (bDataFlg == true)
                {  // データ更新があった場合
                    m_lDispEventNum += 1; // 伝送データ件数更新
                }
                // 境界値設定
                SetScaleData(dXUnderMin, dXUnderMax, dYLeftMin, dYLeftMax, dYRightMin, dYRightMax);
            }
            return 0;
        }
        /// <summary><para>method outline:</para>
        /// <para>モデル演算結果を外部のエレメントに対して公開する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = DataFusion( lOutputDataNum, ref csOutputCellData)</para>
        /// </example>
        /// <param name="lOutputDataNum">出力情報数</param>
        /// <param name="csOutputCellData">出力情報配列</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override long DataFusion(long lOutputDataNum, ref McSendCellDataIF[] csOutputCellData)
        {
            if (m_lCalculate_Call_Before_DataFution_Call > 0)
            {
                if (m_bDispOnFlg == true)
                { // モニタ画面の表示が必要な場合
                    this.DataFusion();
                }
                m_lCalculate_Call_Before_DataFution_Call = 0;
            }
            return 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>データスケール情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetScaleData(dXUnderMin, dXUnderMax, dYLeftMin, dYLeftMax, dYRightMin, dYRightMax )</para>
        /// </example>
        /// <param name="dXUnderMin">水平下段軸最小値</param>
        /// <param name="dXUnderMax">水平下段軸最大値</param>
        /// <param name="dYLeftMin">垂直左端軸最小値</param>
        /// <param name="dYLeftMax">垂直左端軸最大値</param>
        /// <param name="dYRightMin">垂直右端軸最小値</param>
        /// <param name="dYRightMax">垂直右端軸最大値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void SetScaleData(double dXUnderMin, double dXUnderMax,
                                    double dYLeftMin , double dYLeftMax ,
                                    double dYRightMin, double dYRightMax)
        {
            if (m_csDispData is HySDispInstanceCommon)
            {  // グラフデータ有効の場合
                double[] dList = ((HySDispInstanceCommon)m_csDispData).GetDoubleArray();
                if (dList == null)
                {  // エリアが無効な場合
                    dList = new double[6];
                    dList[0] = dXUnderMin; // X軸下段最小値設定
                    dList[1] = dXUnderMax; // X軸下段最大値設定
                    dList[2] = dYLeftMin;  // Y軸左端最小値設定
                    dList[3] = dYLeftMax;  // Y軸左端最大値設定
                    dList[4] = dYRightMin; // Y軸右端最小値設定
                    dList[5] = dYRightMax; // Y軸右端最大値設定
                    ((HySDispInstanceCommon)m_csDispData).SetDoubleArray(dList); // スケール設定
                }
                else
                {  // エリアが有効な場合
                    dList[0] = dXUnderMin; // X軸下段最小値設定
                    dList[1] = dXUnderMax; // X軸下段最大値設定
                    dList[2] = dYLeftMin;  // Y軸左端最小値設定
                    dList[3] = dYLeftMax;  // Y軸左端最大値設定
                    dList[4] = dYRightMin; // Y軸右端最小値設定
                    dList[5] = dYRightMax; // Y軸右端最大値設定
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>データスケール情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetScaleData(ref dXUnderMin, ref dXUnderMax, ref dYLeftMin, ref dYLeftMax, ref dYRightMin, ref dYRightMax )</para>
        /// </example>
        /// <param name="dXUnderMin">水平下段軸最小値</param>
        /// <param name="dXUnderMax">水平下段軸最大値</param>
        /// <param name="dYLeftMin">垂直左端軸最小値</param>
        /// <param name="dYLeftMax">垂直左端軸最大値</param>
        /// <param name="dYRightMin">垂直右端軸最小値</param>
        /// <param name="dYRightMax">垂直右端軸最大値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void GetScaleData(ref double dXUnderMin, ref double dXUnderMax,
                                    ref double dYLeftMin, ref double dYLeftMax,
                                    ref double dYRightMin, ref double dYRightMax)
        {
            if (m_csDispData is HySDispInstanceCommon)
            {  // グラフデータ有効の場合
                double[] dList = ((HySDispInstanceCommon)m_csDispData).GetDoubleArray();
                if (dList == null)
                {  // エリアが無効な場合
                    // 何もしない
                }
                else
                {  // エリアが有効な場合
                    dXUnderMin = dList[0]; // X軸下段最小値取得
                    dXUnderMax = dList[1]; // X軸下段最大値取得
                    dYLeftMin = dList[2]; // Y軸左端最小値取得
                    dYLeftMax = dList[3]; // Y軸左端最大値取得
                    dYRightMin = dList[4]; // Y軸右端最小値取得
                    dYRightMax = dList[5]; // Y軸右端最大値取得
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>データインスタンス通知イベント発行</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DataInstanceNotificationEvent( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void DataInstanceNotificationEvent()
        {
            // データインスタンスを表示業務に通知
            if (m_csDispData != null)
            {  // データが有効な場合
                HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.CMND_DISP_SCREEN);
                csEvent.SetFrom(HySSysEvent.OBJID_SIMULATOR);
                csEvent.SetFromSimKind(McDefine.SIMULATOR_KIND);
                csEvent.SetToSimKind(HySDefine.COMMON_DISP_BUSIPROCEDURE);
                csEvent.SetData(HySDefine.DISP_FORM_MONITER);
                csEvent.SetSuppID((HySID)m_csSimulationID); // 演算シミュレータＩＤ設定
                csEvent.SetResponseInfo((HySID)m_csElement.GetID()); // 要素ＩＤ設定
                csEvent.SetSubEventNo(HySDispDefine.NOTICE_RESULT_AREA_CREATE);
                csEvent.SetSubData(m_csDispData); // 演算結果データインスタンス設定
                m_csSimuCtl.PutEvent(csEvent);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetProperty(csCellMdlPropertyInfo)</para>
        /// </example>
        /// <param name="csCellMdlPropertyInfo">セル型プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SetProperty(McCellModelPropertyIF csCellMdlPropertyInfo)
        {
            McModelPropertyInfo csPropertyInfo = csCellMdlPropertyInfo as McModelPropertyInfo;
            if (csPropertyInfo is McPropertyInfo)
            {  // プロパティ情報が有効な場合
                ((McPropertyInfo)csPropertyInfo).GetInfo(MODEL_DISP_FLAG_KEY, ref m_bDispFlg);
                ((McPropertyInfo)csPropertyInfo).GetInfo(MODEL_TIME_STEP_KEY, ref m_dTimeStep);
                ((McPropertyInfo)csPropertyInfo).GetInfo(MODEL_DISP_CYCLE_KEY, ref m_lDispCycle);
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算実行中断中のプロパティ等情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SetOnlineProperty(csCellMdlPropertyInfo)</para>
        /// </example>
        /// <param name="csCellMdlPropertyInfo">プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中にプロパティ、初期値等を変更して動作させ場合等に使用する</para>
        /// </remarks>
        public override bool SetOnlineProperty(McCellModelPropertyIF csCellMdlPropertyInfo)
        {
            McModelPropertyInfo csPropertyInfo = csCellMdlPropertyInfo as McModelPropertyInfo;
            if (csPropertyInfo is McPropertyInfo)
            {  // プロパティ情報が有効な場合
                ((McPropertyInfo)csPropertyInfo).GetInfo(MODEL_DISP_FLAG_KEY, ref m_bDispFlg);
                ((McPropertyInfo)csPropertyInfo).GetInfo(MODEL_TIME_STEP_KEY, ref m_dTimeStep);
                ((McPropertyInfo)csPropertyInfo).GetInfo(MODEL_DISP_CYCLE_KEY, ref m_lDispCycle);
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>計算状態復元のためのデータクラスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McCmnElementOutData csElementOutData = CreateOutData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>データクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override McCmnElementOutData CreateOutData()
        {
            return new McLineGraphScreenOutData();
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーションデータコンテナの内容を設定する処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimDataContainerInfo( csDataContainer )</para>
        /// </example>
        /// <param name="csDataContainer">シミュレーションデータコンテナ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetSimDataContainer(HySSimulationDataContainer csDataContainer)
        {
            // 何もしない
        }

        /// <summary><para>method outline:</para>
        /// <para>ファイルにモデル内情報を全て書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileOUT(csFile)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileOUT(HySDataRoot csData)
        {
            base.FileOUT(csData);
            ((McLineGraphScreenOutData)(csData)).SetNextTimeStep(m_csNxTimeStep);
            ((McLineGraphScreenOutData)(csData)).SetNextStep(m_dNxTimeStep);
            ((McLineGraphScreenOutData)(csData)).SetBaseCount(m_dBaseCnt);
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>ファイルからモデル情報を全て読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileIN(csFile)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>bool true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileIN(HySDataRoot csData)
        {
            base.FileIN(csData);
            m_csNxTimeStep = ((McLineGraphScreenOutData)(csData)).GetNextTimeStep();
            m_dNxTimeStep = ((McLineGraphScreenOutData)(csData)).GetNextStep();
            m_dBaseCnt = ((McLineGraphScreenOutData)(csData)).GetBaseCount();

            // 復元後の整合性確保
            if (m_csNxTimeStep != null)
            {
                m_csNxTimeStep.PostDeserialize();
            }

            this.DataInstanceNotificationEvent(); // 表示業務にデータインスタンスを通知
            // 画面表示の再設定（現行情報を優先）
            HySString csKeyName = HySEnvInf.GetEnvInf(HYMCO_MONITOR_DISPLAY) as HySString;
            if (csKeyName is HySString)
            {  // 設定が有る場合
                if (csKeyName.Equal(new HySString("ON")) == true ||
                    csKeyName.Equal(new HySString("on")) == true)
                {  // 表示が必要な場合
                    m_bDispOnFlg = true;
                }
            }

            return true;
        }
    }
}
