﻿// <summary>ソースコード：ＨＹＭＣＯ ファイルベースのプロジェクトDBAクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Text;
using System.IO;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.CoreImpl.DBA;
using CommonMP.HYMCO.CoreImpl.Data.LibraryCtl;
using CommonMP.HYMCO.CoreImpl.Data.ProjectCtl;
using CommonMP.HYMCO.CoreImpl.Data.Tool;
using CommonMP.HYSSOP.CoreImpl.HSDBA.FileBase;
using CommonMP.HYSSOP.CoreImpl.HSDBA;
using System.Collections.Generic;

namespace CommonMP.HYMCO.CoreOptions.McDBA.FileDBA
{
    /// <summary><para>class outline:</para>
    /// <para>ＨＹＭＣＯファイルベースDBAクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/06/22][新規作成]</para>
    /// </remarks>
    public class McFileProjectDataDBA : HySCommonFileDBA
    {

        /// <summary>
        /// 自クラス名(ログ出力用)
        /// </summary>
        private const string m_csMyClassName = "McFileProjectDataDBA";

        /// <summary>
        /// DBフォルダパス(ホームディレクトリからの相対パス)
        /// </summary>
        private string m_csDBPath = null;

        /// <summary>
        /// DBフォルダパス名を環境設定から取得するためのキー
        /// </summary>
        public const string ENVKEY_DBPATH = "MCDBA_PROJECTDATA_PATH";

        /// <summary>
        /// デフォルトＤＢパス
        /// </summary>
        public const string DEFAULT_DBPATH = @"\SystemData\db\project";

        /// <summary>
        /// データファイルプレフィックス
        /// </summary>
        public const string DATAFILE_PREFFIX = "";

        /// <summary>
        /// データファイル拡張子
        /// </summary>
        public const string DATAFILE_EXTENSION = ".modeldat";

        /// <summary>
        /// メタデータリストファイル名
        /// </summary>
        public const string METADATALIST_FILENAME = "ProjectMaster.mng";

        /// <summary>
        /// メタデータリスト一時ファイル名
        /// </summary>
        public const string TMP_METADATALIST_FILENAME = "TmpProjectMaster.mng";

        /// <summary>
        /// メタデータリストダンプファイル名
        /// </summary>
        public const string METADATALIST_DUMP_FILENAME = "ProjectMasterDump.txt";


        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McFileProjectDataDBA csManager = new McFileProjectDataDBA()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McFileProjectDataDBA 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McFileProjectDataDBA()
        {

        }

        #region publicメソッドのオーバーライド

        /// <summary><para>method outline:</para>
        /// <para>1件新規登録</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = RegisterNew(csStockData)</para>
        /// </example>
        /// <param name="csStockData">DB登録データ</param>
        /// <returns>true:成功, false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool RegisterNew(HySStockData csStockData)
        {
            const string csMyMethodName = "RegisterNew(HySStockData)";

            // 複数一括バージョンで実行する
            HySStockDataList csStockDataList = new HySStockDataList();
            csStockDataList.AddLast(csStockData);
            if (!RegisterNew(csStockDataList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "RegisterNew(HySStockDataList) returned false");
                return false;
            }
            return true;

        }

        /// <summary><para>method outline:</para>
        /// <para>複数件一括新規登録</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = RegisterNew(csStockDataList)</para>
        /// </example>
        /// <param name="csStockDataList">DB登録データ</param>
        /// <returns>true:成功, false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>前処理として、HySStockDataの追加検索キー(KeyInfo)にプロジェクトデータの一部を設定する</para>
        /// </remarks>
        public override bool RegisterNew(HySStockDataList csStockDataList)
        {
            const string csMyMethodName = "RegisterNew(HySStockDataList)";

            csStockDataList.SetCursorFirst();
            for (long i = 0; i < csStockDataList.GetCount(); i++)
            {
                HySStockData csStockData = (HySStockData)csStockDataList.GetCursorData();
                McProjectInfData csProjInf = GetProjectInfDataFromStockData(csStockData);

                // 保存データのIDをプロジェクトIDに差し替える
                csStockData.SetID(csProjInf.GetProjectID());

                // 登録日と更新日をセットする
                HySTime csToday = GetToday();
                csProjInf.SetRegistDate(csToday);
                csProjInf.SetUpdateDate(csToday);

                // 追加検索キーをセットする
                if (!SetKeyInfoOfStockData(csStockData))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "SetKeyInfoOfStockData() returned false");
                    return false;
                }
                csStockDataList.MoveCursorNext();
            }
            // 登録処理実行
            return base.RegisterNew(csStockDataList);
        }

        /// <summary><para>method outline:</para>
        /// <para>1件更新</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = Update(csStockData)</para>
        /// </example>
        /// <param name="csStockData">DB登録データ</param>
        /// <returns>true:成功, false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>前処理として、HySStockDataの追加検索キー(KeyInfo)にプロジェクトデータの一部を設定する</para>
        /// </remarks>
        public override bool Update(HySStockData csStockData)
        {
            const string csMyMethodName = "Update";

            McProjectInfData csProjInf = GetProjectInfDataFromStockData(csStockData);

            // 保存データのIDをプロジェクトIDに差し替える
            csStockData.SetID(csProjInf.GetProjectID());

            // 更新日を設定する
            HySTime csToday = GetToday();
            csProjInf.SetUpdateDate(csToday);

            // 追加検索キーを設定する
            if (!SetKeyInfoOfStockData(csStockData))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "SetKeyInfoOfStockData() returned false");
                return false;
            }
            return base.Update(csStockData);
        }

        /// <summary><para>method outline:</para>
        /// <para>複数件データ検索</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SearchList(csQueryCtlData,csStockDataList)</para>
        /// </example>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csStockDataList">DB登録データ</param>
        /// <returns>true:成功, false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SearchList(HySQueryCtlData csQueryCtlData, out HySStockDataList csStockDataList)
        {
            // 基底クラスで処理を実行
            bool bResult = base.SearchList(csQueryCtlData, out csStockDataList);

            if (bResult && csQueryCtlData.GetIsMetaDataOnly())
            {
                // 「メタデータのみ」が指定された場合、プロジェクト情報オブジェクトを付加する
                long lDataCount = csStockDataList.GetCount();
                csStockDataList.SetCursorFirst();
                for (long lIdx = 0; lIdx < lDataCount; ++lIdx)
                {
                    HySStockData csStockData = (HySStockData)csStockDataList.GetCursorData();
                    csStockDataList.MoveCursorNext();
                    McProjectInfData csProjectInf = McFileProjectDataDBAUtility.ConvertKeyInfoToProjectInfData(csStockData);
                    csStockData.SetData(csProjectInf);
                }
            }
            return bResult;
        }

        #endregion

        #region テンプレートメソッド

        /// <summary><para>method outline:</para>
        /// <para>（オーバーライド）データファイルの拡張子を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string csExtension = GetDataFileExtension()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>プロジェクトデータデータファイルの拡張子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override string GetDataFileExtension()
        {
            return DATAFILE_EXTENSION;
        }

        /// <summary><para>method outline:</para>
        /// <para>（オーバーライド）データファイルのプレフィックスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string csPreffix = GetDataFilePreffix()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>プロジェクトデータデータファイルのプレフィックス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override string GetDataFilePreffix()
        {
            return DATAFILE_PREFFIX;
        }

        /// <summary><para>method outline:</para>
        /// <para>（オーバーライド）メタデータリストファイル名を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string csFileName = GetMetaDataListFileName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>メタデータリストファイル名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override string GetMetaDataListFileName()
        {
            return METADATALIST_FILENAME;
        }

        /// <summary><para>method outline:</para>
        /// <para>（オーバーライド）メタデータリスト一時ファイル名を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string csFileName = GetTmpMetaDataListFileName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>メタデータリスト一時ファイル名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override string GetTmpMetaDataListFileName()
        {
            return TMP_METADATALIST_FILENAME;
        }

        /// <summary><para>method outline:</para>
        /// <para>（オーバーライド）メタデータリストダンプファイル名を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string csFileName = GetMetaDataListDumpFileName()</para>
        /// </example>m
        /// <param name="">無し</param>
        /// <returns>メタデータリストダンプファイル名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override string GetMetaDataListDumpFileName()
        {
            return METADATALIST_DUMP_FILENAME;
        }

        /// <summary><para>method outline:</para>
        /// <para>（オーバーライド）DBパスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string csPath = GetDBBasePath()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>DBパス名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override string GetDBBasePath()
        {
            if (m_csDBPath == null)
            {
                // 環境設定からDBフォルダを取得する
                string csDBPath = null;
                HySString csHySStringDBPath = HySEnvInf.GetEnvInf(ENVKEY_DBPATH);
                if (((object)csHySStringDBPath) != null)
                {
                    csDBPath = csHySStringDBPath.ToString();
                }
                else
                {
                    csDBPath = DEFAULT_DBPATH;  // fallback value
                }
                m_csDBPath = csDBPath;
            }
            return m_csDBPath;
        }

        /// <summary><para>method outline:</para>
        /// <para>（オーバーライド）検索条件準備</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>PrepareQuery(csQueryCtlData,csComparerList)</para>
        /// </example>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csComparerList">フィールド値比較デリゲートリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override void PrepareQuery(HySQueryCtlData csQueryCtlData, ref List<HySDBAFieldComparer> csComparerList)
        {
            // 基本部分の検索条件を初期化する
            base.PrepareQuery(csQueryCtlData, ref csComparerList);

            // プロジェクトデータ固有の検索条件を初期化する
            HySQueryEntryData csQueryEntryData;

            // MANAGER_NAME
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MANAGER_NAME)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.MANAGER_NAME, csStockData);
                }));
            }
            // PROJECT_NAME
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.PROJECT_NAME)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.PROJECT_NAME, csStockData);
                }));
            }
            // REGISTRATION_DATE
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.REGISTRATION_DATE)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.REGISTRATION_DATE, csStockData);
                }));
            }
            // UPDATE_DATE
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.UPDATE_DATE)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.UPDATE_DATE, csStockData);
                }));
            }
            // MODEL_TYPE
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MODEL_TYPE)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)

                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.MODEL_TYPE, csStockData);
                }));
            }
            // MODEL_TARGET
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MODEL_TARGET)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.MODEL_TARGET, csStockData);
                }));
            }
            // MODEL_CALC_TYPE
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MODEL_CALC_TYPE)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.MODEL_CALC_TYPE, csStockData);
                }));
            }
            // MODEL_SUBJECT
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MODEL_SUBJECT)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.MODEL_SUBJECT, csStockData);
                }));
            }
            // MODEL_SITUATION
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MODEL_SITUATION)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.MODEL_SITUATION, csStockData);
                }));
            }

            // MODEL_PERIOD_TYPE, MODEL_PERIOD
            // この2項目は関連があるのでカスタム比較オブジェクトを使う
            csComparerList.Add(new McProjectModelPeriodComparer(
                csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MODEL_PERIOD_TYPE),
                csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.MODEL_PERIOD)));

            // THRESHOLD_CONDITION1
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.THRESHOLD_CONDITION1)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.THRESHOLD_CONDITION1, csStockData);
                }));
            }
            // THRESHOLD_CONDITION2
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.THRESHOLD_CONDITION2)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.THRESHOLD_CONDITION2, csStockData);
                }));
            }
            // THRESHOLD_CONDITION3
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.THRESHOLD_CONDITION3)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.THRESHOLD_CONDITION3, csStockData);
                }));
            }
            // THRESHOLD_CONDITION4
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.THRESHOLD_CONDITION4)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.THRESHOLD_CONDITION4, csStockData);
                }));
            }
            // THRESHOLD_CONDITION5
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.THRESHOLD_CONDITION5)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.THRESHOLD_CONDITION5, csStockData);
                }));
            }
            // CONDITION1
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.CONDITION1)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.CONDITION1, csStockData);
                }));
            }
            // CONDITION2
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.CONDITION2)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.CONDITION2, csStockData);
                }));
            }
            // CONDITION3
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.CONDITION3)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.CONDITION3, csStockData);
                }));
            }
            // CONDITION4
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.CONDITION4)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.CONDITION4, csStockData);
                }));
            }
            // CONDITION5
            if ((csQueryEntryData = csQueryCtlData.GetQueryEntryData(McProjectQueryFieldNames.CONDITION5)) != null)
            {
                csComparerList.Add(new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)
                {
                    return GetKeyInfoFieldValue(McProjectQueryFieldNames.CONDITION5, csStockData);
                }));
            }
        }

        #endregion

        #region privateメソッド

        /// <summary><para>method outline:</para>
        /// <para>追加検索キー取り出し</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>PrepareQueryGetKeyInfoFieldValue(csFieldName, csStockData)</para>
        /// </example>
        /// <param name="csFieldName">キーとなるフィールド名</param>
        /// <param name="csStockData">取り出し元のDB保存データ</param>
        /// <returns>無しキーに対応する値</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private static HySDataRoot GetKeyInfoFieldValue(HySString csFieldName, HySStockData csStockData)
        {
            long lKeyNumber = 0;
            HySDataHashTable csHash = csStockData.GetKeyInfo(ref lKeyNumber);
            if (lKeyNumber > 0)
            {
                return csHash[csFieldName];
            }
            else
            {
                return null;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>追加検索キー設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetKeyInfoOfStockData(csStockData)</para>
        /// </example>
        /// <param name="csStockData">追加検索キーが設定されるDB保存データ</param>
        /// <returns>true:成功、false:失敗(必須キーが未設定)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>追加する情報の一覧は<see cref="McProjectMetaKeyDefines"/>を参照のこと。</para>
        /// </remarks>
        private bool SetKeyInfoOfStockData(HySStockData csStockData)
        {
            //const string csMyMethodName = "SetKeyInfoOfStockData";

            McProjectInfData csProjectInf = GetProjectInfDataFromStockData(csStockData);

            HySString[] csKeys = new HySString[] {
                McProjectMetaKeyDefines.PROJECT_ID,    //シミュレーターID
                McProjectMetaKeyDefines.DISP_PROJECT_NAME,    //表示プロジェクト名
                McProjectMetaKeyDefines.DB_NAME,    //データベース名
                McProjectMetaKeyDefines.FILE_NAME,    //ファイルパス
                McProjectMetaKeyDefines.PROJECT_NAME,    //プロジェクト名
                McProjectMetaKeyDefines.MANAGER_NAME,    //管理者名
                McProjectMetaKeyDefines.CREATE_DATE,    //作成日
                McProjectMetaKeyDefines.REGISTRATION_DATE,    //登録日
                McProjectMetaKeyDefines.UPDATE_DATE,    //更新日
                McProjectMetaKeyDefines.START_SIM_TIME,    //計算開始時間
                McProjectMetaKeyDefines.END_SIM_TIME,    //計算終了時間
                McProjectMetaKeyDefines.SUSPEND_TIME,    //中断時刻
                McProjectMetaKeyDefines.COMMENT,    //プロジェクトコメント
                McProjectMetaKeyDefines.MODEL_TYPE,    //種別
                McProjectMetaKeyDefines.MODEL_TARGET,    //対象
                McProjectMetaKeyDefines.MODEL_CALC_TYPE,    //モデル
                McProjectMetaKeyDefines.MODEL_SUBJECT,    //件名
                McProjectMetaKeyDefines.MODEL_SITUATION,    //ケース名
                McProjectMetaKeyDefines.MODEL_PERIOD_TYPE,    //期間種別
                McProjectMetaKeyDefines.MODEL_PERIOD,    //期間
                McProjectMetaKeyDefines.THRESHOLD_CONDITION1,    //境界条件１
                McProjectMetaKeyDefines.THRESHOLD_CONDITION2,    //境界条件２
                McProjectMetaKeyDefines.THRESHOLD_CONDITION3,    //境界条件３
                McProjectMetaKeyDefines.THRESHOLD_CONDITION4,    //境界条件４
                McProjectMetaKeyDefines.THRESHOLD_CONDITION5,    //境界条件５
                McProjectMetaKeyDefines.CONDITION1,    //条件１
                McProjectMetaKeyDefines.CONDITION2,    //条件２
                McProjectMetaKeyDefines.CONDITION3,    //条件３
                McProjectMetaKeyDefines.CONDITION4,    //条件４
                McProjectMetaKeyDefines.CONDITION5,    //条件５
            };

            // 追加検索キーを一旦クリアする
            long lKeyNumber = 0;
            HySDataHashTable csKeyInfo = csStockData.GetKeyInfo(ref lKeyNumber);
            foreach (HySString csKey in csKeys)
            {
                if (csKeyInfo.GetObject(csKey) != null)
                {
                    csKeyInfo.DeleteData(csKey);
                }
            }

            // 追加検索キーに最新データをセットする
            csKeyInfo.AddObject(McProjectMetaKeyDefines.PROJECT_ID, csProjectInf.GetProjectID());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.DISP_PROJECT_NAME, csProjectInf.GetDispProjectName());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.DB_NAME, csProjectInf.GetDbName());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.FILE_NAME, csProjectInf.GetFileName());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.PROJECT_NAME, csProjectInf.GetProjectName());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MANAGER_NAME, csProjectInf.GetAdminName());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.CREATE_DATE, csProjectInf.GetCreateDate());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.REGISTRATION_DATE, csProjectInf.GetRegistDate());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.UPDATE_DATE, csProjectInf.GetUpdateDate());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.START_SIM_TIME, csProjectInf.GetStartSimuTime());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.END_SIM_TIME, csProjectInf.GetEndSimuTime());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.SUSPEND_TIME, csProjectInf.GetSuspendTime());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.COMMENT, csProjectInf.GetComment());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MODEL_TYPE, csProjectInf.GetModelType());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MODEL_TARGET, csProjectInf.GetModelTarget());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MODEL_CALC_TYPE, csProjectInf.GetModelCalcType());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MODEL_SUBJECT, csProjectInf.GetModelSubject());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MODEL_SITUATION, csProjectInf.GetModelSituation());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MODEL_PERIOD_TYPE, new HySLong((long)csProjectInf.GetModelPeriodType()));
            csKeyInfo.AddObject(McProjectMetaKeyDefines.MODEL_PERIOD, csProjectInf.GetModelPeriod());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.THRESHOLD_CONDITION1, csProjectInf.GetThresholdCondition1());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.THRESHOLD_CONDITION2, csProjectInf.GetThresholdCondition2());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.THRESHOLD_CONDITION3, csProjectInf.GetThresholdCondition3());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.THRESHOLD_CONDITION4, csProjectInf.GetThresholdCondition4());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.THRESHOLD_CONDITION5, csProjectInf.GetThresholdCondition5());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.CONDITION1, csProjectInf.GetCondition1());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.CONDITION2, csProjectInf.GetCondition2());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.CONDITION3, csProjectInf.GetCondition3());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.CONDITION4, csProjectInf.GetCondition4());
            csKeyInfo.AddObject(McProjectMetaKeyDefines.CONDITION5, csProjectInf.GetCondition5());
            //csKeyInfo.AddObject(McProjectQueryFieldNames.KEY_ONE, csProjectInf.GetKeyOne());
            //csKeyInfo.AddObject(McProjectQueryFieldNames.KEY_TWO, csProjectInf.GetKeyTwo());

            csStockData.SetKeyInfo(csKeyInfo);

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>DB保存データからプロジェクト情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetProjectInfDataFromStockData(csStockData)</para>
        /// </example>
        /// <param name="csStockData">DB保存データ</param>
        /// <returns>プロジェクト情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private static McProjectInfData GetProjectInfDataFromStockData(HySStockData csStockData)
        {
            HySSimulationContainer csContainer = (HySSimulationContainer)csStockData.GetData();
            McProjectInfData csProjectInf = (McProjectInfData)csContainer.GetData(McDefine.HYM_DATA_PROJECT_INFO);
            return csProjectInf;
        }

        #endregion


    }
}
