// <summary>ソースコード：.net 画面フォーム</summary>
// <author>CommonMP</author>
using System;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Drawing.Printing;
using System.Text;
using System.Windows.Forms;
using System.Xml;

using CommonMP.HYSSOP.Interface.HSViewer;
using CommonMP.HYSSOP.Interface.HSController;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSController;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.OptionImpl.HSViewer.DotNetViewer;


namespace CommonMP.HYMCO.OptionImpl.McStructureXmlFileEditorForDevelop
{
    /// <summary><para>class outline:</para>
    /// <para>構造定義ファイル編集画面フォーム</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2010/10/28][新規作成]</para>
    /// </remarks>
    public partial class McStructureXmlFileEditorForm : HySDotNetForm // : Form
    {
        ///// <summary>制御するクリーン </summary>
        //protected McStructureXmlFileEditorScreen m_csMyScreen = null;

        /// <summary> 編集モード </summary>
        public enum XmlEditMode
        {
            /// <summary> 入力データセット作成 </summary>
            Parameter,
            /// <summary> 構造定義ファイル作成 </summary>
            Connection
        }

        /// <summary> 編集モード[入力データセット作成/構造定義ファイル作成] </summary>
        public XmlEditMode m_enmEditMode = XmlEditMode.Parameter;

        /// <summary> Excelテンプレートファイルパス </summary>
        private string m_sTemplatePath = "";
        /// <summary> Excelテンプレートファイル名 </summary>
        private string m_sTemplateFile = "";


        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructureXmlFileEditorForm csNewProjectForm = new McStructureXmlFileEditorForm();</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>使用禁止</para>
        /// </remarks>
        protected McStructureXmlFileEditorForm()
        {
            InitializeComponent();
        }
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructureXmlFileEditorForm csNewProjectForm = new McStructureXmlFileEditorForm(csScreen, csEditMode);</para>
        /// </example>
        /// <param name="csScreen">スクリーンクラス</param>
        /// <param name="csEditMode"></param>
        /// <returns>McDotNetNewProjectForm 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McStructureXmlFileEditorForm(HySScreen csScreen, XmlEditMode csEditMode)
        {
            InitializeComponent();
            m_csScreen = csScreen;
            //m_csMyScreen = csScreen as McStructureXmlFileEditorScreen; // 便利の為

            //フォームの外見を設定
            //this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
            //this.MaximizeBox = false;
            //this.MinimizeBox = false;

            m_sTemplatePath = HySEnvInf.GetHomeDirectory().ToString() + McStructureXmlFileEditorDefine.SYSTEMDATA_PATH;
            m_sTemplateFile = McStructureXmlFileEditorDefine.EXCEL_TEMPLATE_FILE;

            m_enmEditMode = csEditMode;
            if (m_enmEditMode == XmlEditMode.Parameter)
            {   //入力データセット作成
                lblMsg0.Visible = false;
                //パラメータ編集用のExcelファイルを出力します。
                lblMsg1.Text = Properties.McStructureXmlFileEditorResources.MSG_INFO_OUTPUT_EXCEL_FOR_EDIT_PARAM;
                //Excelファイルを読み込んで、構造定義ファイルのパラメータを更新します。
                lblMsg2.Text = Properties.McStructureXmlFileEditorResources.MSG_INFO_UPDATE_PARAM;
                //構造定義ファイルのパラメータの編集を行います。
                lblTitle.Text = Properties.McStructureXmlFileEditorResources.TITLE_EDIT_PARAM;
                lblTitle.BackColor = Color.FromArgb(255, 255, 192);
            }
            else if (m_enmEditMode == XmlEditMode.Connection)
            {   //構造定義ファイル作成
                //要素モデル配置・接続作成用のExcelファイルを出力します。
                lblMsg1.Text = Properties.McStructureXmlFileEditorResources.MSG_INFO_CREATE_FOR_PLACEMENT_AND_CONNECTIVITY;
                //Excelファイルを読み込んで、構造定義ファイルのモデル配置と接続情報を出力します。
                lblMsg2.Text = Properties.McStructureXmlFileEditorResources.MSG_INFO_OUTPUT_FOR_PLACEMENT_AND_CONNECTIVITY;
                //構造定義ファイルの要素モデル配置と接続情報の作成・編集を行います。
                lblTitle.Text = Properties.McStructureXmlFileEditorResources.TITLE_EDIT_FOR_PLACEMENT_AND_CONNECTIVITY;
                lblTitle.BackColor = Color.FromArgb(192, 255, 192);
            }
            ImportBtn.BackColor = lblTitle.BackColor;
            ExportBtn.BackColor = lblTitle.BackColor;

            //エラー一覧初期設定
            InitList(dgvCheckError);
        }

        /// <summary><para>method outline:</para>
        /// <para>フォームを閉じた時の処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphViewHARForm_FormClosed(sender, e)：フォームクローズイベント</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">Event</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void McGraphViewHARForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            m_csScreen.CloseScreen();
        }

        /// <summary><para>method outline:</para>
        /// <para>ファイル選択ボタンクリック時処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>FileSelectBtn_Click(sender, e)：ファイル選択ボタンクリックイベント</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">EventArgs</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void FileSelectBtn_Click(object sender , EventArgs e)
        {
            TextBox txt = null;
            bool bOpen = true;
            string sExt = "xml";
            string sFileName = "";

            if (sender == FileOpenBtn)
            {   //構造定義ファイル読込み
                txt = txtInFilePath;
            }
            else if (sender == FileSaveBtn)
            {   //構造定義ファイル保存先
                txt = txtOutFilePath;
                bOpen = false;
            }
            else if (sender == ExcelOpenBtn)
            {   //Excelファイル読込み
                txt = txtInExcelPath;
                sExt = "xls";
            }
            else if (sender == ExcelSaveBtn)
            {   //Excelファイル保存先
                txt = txtOutExcelPath;
                sExt = "xls";
                bOpen = false;
            }
            

            //ダイアログを表示する
            if(bOpen)
                sFileName = ShowSelectFileDialog(sExt, txt.Text);
            else
                sFileName = ShowSaveFileDialog(sExt, txt.Text);

            if (!string.IsNullOrEmpty(sFileName))
            {
                txt.Text = sFileName;

                if (sender == FileOpenBtn && string.IsNullOrEmpty(txtOutFilePath.Text))
                {   //入力用構造定義ファイル選択した場合、出力用にも反映
                    txtOutFilePath.Text = sFileName;
                }
                if (sender == FileOpenBtn && string.IsNullOrEmpty(txtOutExcelPath.Text))
                {   //入力用構造定義ファイル選択した場合、出力用Excelファイルにも反映
                    string sPath = Path.GetDirectoryName(sFileName) + "\\"
                                + Path.GetFileNameWithoutExtension(sFileName) 
                                + (m_enmEditMode == XmlEditMode.Parameter ? "_Param" : "_Conn") + ".xls";
                    txtOutExcelPath.Text = sPath;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>Excel作成ボタンクリック時処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ExportBtn_Click(sender, e)：Excel作成ボタンクリックイベント</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">EventArgs</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ExportBtn_Click(object sender, EventArgs e)
        {
            //チェックエラークリア
            dgvCheckError.RowCount = 0;

            //--------------------------------------------------
            // 入力チェック
            //--------------------------------------------------
            bool bNotEmptyInXml = (m_enmEditMode == XmlEditMode.Parameter ? true : false);
            
            //編集対象XML構造定義ファイル
            if (!InputCheckFile(txtInFilePath, lblInFilePath.Text, bNotEmptyInXml, true))
                return;
            string sXmlFileNamePath = txtInFilePath.Text;
            
            //保存Excelファイル名
            if (!InputCheckFile(txtOutExcelPath, lblOutExcelPath.Text, true, false))
                return;
            string sExcelFileNamePath = txtOutExcelPath.Text;


            //カーソルを砂時計にする
            this.Cursor = Cursors.WaitCursor;

            //--------------------------------------------------
            //構造定義ファイルを読んでExcel作成
            //--------------------------------------------------
            StructFileController csStruct = null;
            try
            {
                if (m_enmEditMode == XmlEditMode.Parameter)
                {   //入力データセット作成
                    csStruct = new StructFileParamEditor(m_sTemplatePath, m_sTemplateFile);
                }
                else if (m_enmEditMode == XmlEditMode.Connection)
                {   //構造定義ファイル作成
                    csStruct = new StructFileCreater(m_sTemplatePath, m_sTemplateFile);
                }

                //Excel出力
                csStruct.OutputExcel(sXmlFileNamePath, sExcelFileNamePath);

                //保存したファイル名を②のExcelファイル名にセット
                txtInExcelPath.Text = sExcelFileNamePath;
            }
            catch (Exception ex)
            {   //エラーメッセージ
                MessageBox.Show(ex.Message, this.Text, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
            finally
            {
                //カーソルを戻す
                this.Cursor = Cursors.Default;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>構造定義ファイル更新ボタンクリック時処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ImportBtn_Click(sender, e)：構造定義ファイル更新ボタンクリックイベント</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">EventArgs</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ImportBtn_Click(object sender, EventArgs e)
        {
            //チェックエラークリア
            dgvCheckError.RowCount = 0;

            //--------------------------------------------------
            // 入力チェック
            //--------------------------------------------------
            bool bNotEmptyInXml = (m_enmEditMode == XmlEditMode.Parameter ? true : false);

            //編集対象XML構造定義ファイル
            if (!InputCheckFile(txtInFilePath, lblInFilePath.Text, bNotEmptyInXml, true))
                return;
            string sXmlFileNamePath = txtInFilePath.Text;

            //読み込むExcelファイル
            if (!InputCheckFile(txtInExcelPath, lblInExcelPath.Text, true, true))
                return;
            string sExcelFileNamePath = txtInExcelPath.Text;
            
            //出力XML構造定義ファイル
            if (!InputCheckFile(txtOutFilePath, lblOutFilePath.Text, true, false))
                return;
            string sXmlOutFileNamePath = txtOutFilePath.Text;


            //カーソルを砂時計にする
            this.Cursor = Cursors.WaitCursor;

            //--------------------------------------------------
            // Excelファイルを読んで構造定義ファイル更新
            //--------------------------------------------------
            StructFileController csStruct = null;
            try
            {
                if (m_enmEditMode == XmlEditMode.Parameter)
                {   //入力データセット作成
                    csStruct = new StructFileParamEditor(m_sTemplatePath, m_sTemplateFile);
                }
                else if (m_enmEditMode == XmlEditMode.Connection)
                {   //構造定義ファイル作成
                    csStruct = new StructFileCreater(m_sTemplatePath, m_sTemplateFile);
                }

                //構造定義ファイル出力
                bool bRtn = csStruct.OutputStructXml(sXmlFileNamePath, sExcelFileNamePath, sXmlOutFileNamePath);
                if (bRtn)
                {   //完了メッセージ表示
                    //構造定義ファイルを出力しました。
                    MessageBox.Show(Properties.McStructureXmlFileEditorResources.MSG_INFO_OUTPUT_STRUCTURE_DEFINITION, this.Text,
                        MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
                else
                {   //チェックエラー表示
                    int i = 0;
                    dgvCheckError.RowCount = csStruct.m_csErrorList.Count;
                    foreach(ErrorItem err in csStruct.m_csErrorList)
                    {
                        string sRowCol = "";
                        if (err.Row != 0 || err.Col != 0)
                        {
                            sRowCol = string.Format(Properties.McStructureXmlFileEditorResources.ROW_AND_COLUMN, err.Row, err.Col);
                        }
                        dgvCheckError["No", i].Value = i + 1;
                        dgvCheckError["Sheet", i].Value = err.SheetName + sRowCol;
                        dgvCheckError["Message", i].Value = err.Message;
                        i++;
                    }
                    
                    //エラーメッセージ
                    //エラーがあるため、構造定義ファイルは出力されませんでした。
                    MessageBox.Show(Properties.McStructureXmlFileEditorResources.MSG_ERROR_FAILED_TO_STRUCTURE_DEFINITION_OUTPUT, this.Text,
                        MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                }
            }
            catch (Exception ex)
            {   //エラーメッセージ
                MessageBox.Show(ex.Message, this.Text, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
            finally
            {
                //カーソルを戻す
                this.Cursor = Cursors.Default;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>ファイル入力チェック処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = InputCheckFile(txt, sCaption, bNotEmpty, bFileExists);</para>
        /// </example>
        /// <param name="txt">TextBox</param>
        /// <param name="sCaption">項目名</param>
        /// <param name="bNotEmpty">true=必須入力/false=任意入力</param>
        /// <param name="bFileExists">true=ファイル存在チェックする/false=ディレクトリチェックのみ</param>
        /// <returns>true=OK/false=チェックエラー</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool InputCheckFile(TextBox txt, string sCaption, bool bNotEmpty, bool bFileExists)
        {
            if (string.IsNullOrEmpty(txt.Text))
            {   //未入力
                if (bNotEmpty)
                {
                    MessageBox.Show(string.Format(Properties.McStructureXmlFileEditorResources.MSG_INFO_INPUT_CAPTION, sCaption),
                        Properties.McStructureXmlFileEditorResources.CAPTION_INPUT_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                    return false;
                }
                return true;
            }

            string sPath = GetFullPath(txt.Text);
            string sDir = System.IO.Path.GetDirectoryName(sPath);

            if (!System.IO.Directory.Exists(sDir))
            {   //パスが無効
                MessageBox.Show(string.Format(Properties.McStructureXmlFileEditorResources.MSG_ERROR_PATH_NOT_EXIST, sCaption),
                    Properties.McStructureXmlFileEditorResources.CAPTION_INPUT_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                return false;
            }
            if (bFileExists && !File.Exists(sPath))
            {   //ファイルが存在しない
                MessageBox.Show(string.Format(Properties.McStructureXmlFileEditorResources.MSG_ERROR_FILE_NOT_EXIST, sCaption),
                    Properties.McStructureXmlFileEditorResources.CAPTION_INPUT_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                return false;
            }

            return true;
        }

        #region ファイルダイアログ表示

        /// <summary><para>method outline:</para>
        /// <para>ファイル開くダイアログ表示</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string sFileNamePath = ShowSelectFileDialog(sExt, sDefaultFile);</para>
        /// </example>
        /// <param name="sExt">拡張子</param>
        /// <param name="sDefaultFile">初期値として表示するファイル名パス</param>
        /// <returns>ファイル名パス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private string ShowSelectFileDialog(string sExt, string sDefaultFile)
        {
            string sFileNamePath = "";

            //OpenFileDialogクラスのインスタンスを作成
            OpenFileDialog ofd = new OpenFileDialog();

            //はじめのファイル名を指定する
            //はじめに「ファイル名」で表示される文字列を指定する
            ofd.FileName = "default." + sExt;

            //はじめに表示されるフォルダを指定する
            ofd.InitialDirectory = HySEnvInf.GetHomeDirectory().ToString();

            //[ファイルの種類]に表示される選択肢を指定する
            ofd.Filter = string.Format(Properties.McStructureXmlFileEditorResources.FILE_DLG_FILTER_ANY_ALL, sExt);

            //[ファイルの種類]ではじめに
            ofd.FilterIndex = 1;

            //ダイアログボックスを閉じる前に現在のディレクトリを復元するようにする
            ofd.RestoreDirectory = true;

            //カレントフォルダを入力値に設定
            if (!string.IsNullOrEmpty(sDefaultFile))
            {
                string sPath = GetFullPath(sDefaultFile);
                string sDir = System.IO.Path.GetDirectoryName(sPath);
                string sFile = System.IO.Path.GetFileName(sPath);

                if (System.IO.Directory.Exists(sDir))
                {   //パスが有効ならばセット
                    ofd.InitialDirectory = sDir;
                }
                ofd.FileName = sFile;
            }

            //ダイアログを表示する
            if (ofd.ShowDialog() == DialogResult.OK)
            {
                sFileNamePath = ofd.FileName;
            }
            return sFileNamePath;
        }

        /// <summary><para>method outline:</para>
        /// <para>ファイル保存ダイアログ表示</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string sFileNamePath = ShowSaveFileDialog(sExt, sDefaultFile);</para>
        /// </example>
        /// <param name="sExt">拡張子</param>
        /// <param name="sDefaultFile">初期値として表示するファイル名パス</param>
        /// <returns>ファイル名パス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private string ShowSaveFileDialog(string sExt, string sDefaultFile)
        {
            string sFileNamePath = "";
            SaveFileDialog sfd = new SaveFileDialog();

            //はじめのファイル名を指定する
            //はじめに「ファイル名」で表示される文字列を指定する
            sfd.FileName = "default." + sExt;

            //はじめに表示されるフォルダを指定する
            sfd.InitialDirectory = HySEnvInf.GetHomeDirectory().ToString();

            //[ファイルの種類]に表示される選択肢を指定する
            sfd.Filter = string.Format(Properties.McStructureXmlFileEditorResources.FILE_DLG_FILTER_ANY_ALL, sExt);

            //[ファイルの種類]ではじめに
            sfd.FilterIndex = 1;

            //ダイアログボックスを閉じる前に現在のディレクトリを復元するようにする
            sfd.RestoreDirectory = true;

            //カレントフォルダを入力値に設定
            if (!string.IsNullOrEmpty(sDefaultFile))
            {
                string sPath = GetFullPath(sDefaultFile);
                string sDir = System.IO.Path.GetDirectoryName(sPath);
                string sFile = System.IO.Path.GetFileName(sPath);

                if (System.IO.Directory.Exists(sDir))
                {   //パスが有効ならばセット
                    sfd.InitialDirectory = sDir;
                }
                sfd.FileName = sFile;
            }

            //ダイアログを表示する
            if (sfd.ShowDialog() == DialogResult.OK)
            {   //OKボタンがクリックされたとき
                sFileNamePath = sfd.FileName;
            }
            return sFileNamePath;
        }

        /// <summary><para>method outline:</para>
        /// <para>Exeがある場所を基準に相対パスを絶対パスに変換する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string sPath = GetFullPath(sPath);</para>
        /// </example>
        /// <param name="sPath">パス</param>
        /// <returns>絶対パス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private string GetFullPath(string sPath)
        {
            //相対パスを絶対パスに変換
            string sRootPath = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().CodeBase);
            Uri uriPath = new Uri(new Uri(sRootPath + "\\"), sPath);
            return uriPath.LocalPath;
        }

        #endregion

        #region 一覧初期設定

        /// <summary><para>method outline:</para>
        /// <para>一覧初期設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>InitList(dgv);</para>
        /// </example>
        /// <param name="dgv">DataGridView</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void InitList(DataGridView dgv)
        {
            //----------------------------------------
            // 一覧の設定
            //----------------------------------------
            dgv.AllowUserToResizeColumns = true;    //ﾕｰｻﾞｰが列幅変更できるにｾｯﾄ
            dgv.AllowUserToResizeRows = true;       //ﾕｰｻﾞｰが行高変更できるにｾｯﾄ
            //列ヘッダー設定
            dgv.ColumnHeadersDefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleCenter;    //中央寄せ
            //dgv.ColumnHeadersHeightSizeMode = DataGridViewColumnHeadersHeightSizeMode.DisableResizing;  //列ヘッダー高さ固定
            //dgv.ColumnHeadersHeight = 24;
            //行ヘッダー設定
            dgv.RowHeadersVisible = false;           //行ヘッダー表示・非表示
            
            //No
            DataGridViewTextBoxColumn col1 = new DataGridViewTextBoxColumn();
            col1.SortMode = DataGridViewColumnSortMode.Automatic;     //Sort可(Default)
            col1.Name = "No";
            col1.HeaderText = "No";
            col1.ValueType = Type.GetType("System.Int32");
            col1.Width = 40;
            col1.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
            col1.ReadOnly = true;   //読み取り専用
            dgv.Columns.Add(col1);
            
            //シート名
            DataGridViewTextBoxColumn col2 = new DataGridViewTextBoxColumn();
            col2.SortMode = DataGridViewColumnSortMode.Automatic;     //Sort可(Default)
            col2.Name = "Sheet";
            col2.HeaderText = Properties.McStructureXmlFileEditorResources.SHEET;
            col2.ValueType = Type.GetType("System.String");
            col2.Width = 140;
            col2.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleLeft;
            col2.ReadOnly = true;   //読み取り専用
            dgv.Columns.Add(col2);

            //メッセージ内容
            DataGridViewTextBoxColumn col3 = new DataGridViewTextBoxColumn();
            col3.SortMode = DataGridViewColumnSortMode.Automatic;     //Sort可(Default)
            col3.Name = "Message";
            col3.HeaderText = Properties.McStructureXmlFileEditorResources.CONTENT;
            col3.ValueType = Type.GetType("System.String");
            col3.Width = 410;
            col3.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleLeft;
            col3.DefaultCellStyle.ForeColor = Color.Red;
            col3.DefaultCellStyle.WrapMode = DataGridViewTriState.True;     //折り返し
            col3.ReadOnly = true;   //読み取り専用
            dgv.Columns.Add(col3);
        }

        #endregion

    }
}