﻿// <summary>ソースコード：モデル編集画面カスタマイズ情報格納クラス</summary>
// <author>CommonMP</author>
using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Drawing;

using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYMCO.Interface;

namespace CommonMP.HYMCO.CoreOptions.McViewer.DotNetViewer
{
    /// <summary>
    /// <para>class outline:</para>
    /// <para>モデル編集画面カスタマイズ情報格納クラス</para>
    /// </summary>
    /// <remarks>
    /// <para>remarks:</para>
    /// <para>モデル編集画面のカスタマイズ情報を格納</para>
    /// <para>本クラスはバージョンアップ互換性の対象外</para>
    /// </remarks>
    /// <remarks>
    /// <para>history:</para>
    /// <para>
    /// [CommonMP][ver 1.0.0][2009/04/20][新規作成]
    /// [CommonMP][ver 1.4.0][2012/12/05][修正]
    /// </para>
    /// </remarks>
    public class McCustomInfo
    {
        /// <summary>カラー項目リスト</summary>
        private List<HySString> m_sColorItemList;

        /// <summary>カラーリスト</summary>
        private List<int> m_iColorList;

        /// <summary>移動量１</summary>
        private int m_iMovementRate1;

        /// <summary>移動量２</summary>
        private int m_iMovementRate2;

        /// <summary>接続線描画太さ</summary>
        private int m_iDrawingThickness;

        /// <summary>要素リサイズポイントサイズ</summary>
        private int m_iResizePointSize;

        /// <summary>ツールバー名称リスト</summary>
        private List<HySString> m_sToolStripNameList;

        /// <summary>ツールバー説明リスト</summary>
        private List<HySString> m_sToolStripExplanationList;

        /// <summary>ツールバー位置リスト</summary>
        private List<Point> m_pToolStripLocationList;

        /// <summary>ツールバー表示／非表示リスト</summary>
        private List<bool> m_bToolStripVisibleList;

        /// <summary>ツールバーコレクションリスト</summary>
        private List<List<HySString>> m_sToolStripCollectionList;

        /// <summary>ルートノード名称</summary>
        static private readonly String NODE_ROOT = "CustomItem";

        /// <summary>色ノード名称</summary>
        static private readonly String NODE_COLOR = "Color";

        /// <summary>その他ノード名称</summary>
        static private readonly String NODE_OTHERS = "Others";

        /// <summary>ツールバーノード名称</summary>
        static private readonly String NODE_TOOLBAR = "ToolBar";

        /// <summary>項目ノード名称</summary>
        static private readonly String NODE_ITEM = "Item";

        /// <summary>コレクションノード名称</summary>
        static private readonly String NODE_COLLECTION = "Collection";

        /// <summary>項目名</summary>
        static private readonly String KEY_NAME = "Name";

        /// <summary>色コード</summary>
        static private readonly String KEY_COLORCODE = "ColorCode";

        /// <summary>移動量１</summary>
        static private readonly String KEY_MOVEMENTRATE1 = Properties.McDotNetViewerResources.CUSTOMIZE_OTHERS_MOVEMENTRATE1;

        /// <summary>移動量２</summary>
        static private readonly String KEY_MOVEMENTRATE2 = Properties.McDotNetViewerResources.CUSTOMIZE_OTHERS_MOVEMENTRATE2;

        /// <summary>接続線</summary>
        static private readonly String KEY_LINE = Properties.McDotNetViewerResources.CUSTOMIZE_OTHERS_LINE;

        /// <summary>リサイズポイント</summary>
        static private readonly String KEY_RESIZEPOINT = Properties.McDotNetViewerResources.CUSTOMIZE_OTHERS_RESIZEPOINT;

        /// <summary>移動量</summary>
        static private readonly String KEY_MOVEMENTRATE = "MovementRate";

        /// <summary>描画太さ</summary>
        static private readonly String KEY_DRAWINGTHICKNESS = "DrawingThickness";

        /// <summary>リサイズポイントサイズ</summary>
        static private readonly String KEY_RESIZEPOINTSIZE = "ResizePointSize";

        /// <summary>説明</summary>
        static private readonly String KEY_EXPLANATION = "Explanation";

        /// <summary>位置X</summary>
        static private readonly String KEY_LOCATION_X = "Location_X";

        /// <summary>位置Y</summary>
        static private readonly String KEY_LOCATION_Y = "Location_Y";

        /// <summary>表示／非表示モード</summary>
        static private readonly String KEY_VISIBLE_MODE = "Visible_Mode";

        /// <summary>ファイルパス</summary>
        private static string DATA_PATH = "\\conf\\" + HySEnvInf.GetCultureDirectory().ToString();

        /// <summary>カスタマイズ情報状態種別</summary>
        public enum CustomInfoModificationStatus
        {
            /// <summary>変更無</summary>
            UNMODIFIED = 1,
            /// <summary>変更有(色情報)</summary>
            MODIFIED_COLOR = 2,
            /// <summary>変更有(設定値)</summary>
            MODIFIED_VALUE = 4,
            /// <summary>変更有(ツールバー)</summary>
            MODIFIED_TOOLBAR = 8,
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McCustomInfo csCustomInfo = new McCustomInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McCustomInfo()
        {
            m_sColorItemList = new List<HySString>();
            m_iColorList = new List<int>();
            m_sToolStripNameList = new List<HySString>();
            m_sToolStripExplanationList = new List<HySString>();
            m_pToolStripLocationList = new List<Point>();
            m_bToolStripVisibleList = new List<bool>();
            m_sToolStripCollectionList = new List<List<HySString>>();

            //デフォルト設定
            SetDefaultColor();
            SetDefaultValue();
            SetDefaultLocation();

            //定義情報取り込み
            LoadCustomInfo();
        }

        /// <summary><para>method outline:</para>
        /// <para>カラー項目リスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜HySString＞ sColorItemList = GetColorItemList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜HySString＞ カラー項目リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<HySString> GetColorItemList()
        {
            return m_sColorItemList;
        }

        /// <summary><para>method outline:</para>
        /// <para>カラーリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜int＞ iColorList = GetColorList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜int＞ カラーリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<int> GetColorList()
        {
            return m_iColorList;
        }

        /// <summary><para>method outline:</para>
        /// <para>カラーリスト設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetColorList(iColorList)</para>
        /// </example>
        /// <param name="iColorList">カラーリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetColorList(List<int> iColorList)
        {
            m_iColorList = iColorList;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画カラー取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iColor = GetDrawingColor(eDrawKind, csDrawSubKind, bModelFlg, bInnerFlg)</para>
        /// </example>
        /// <param name="eDrawKind">種別</param>
        /// <param name="csDrawSubKind">描画オブジェクト区分</param>
        /// <param name="bModelFlg">演算モデル設定有無</param>
        /// <param name="bInnerFlg">グループ内中継端子指定　true:グループ内　false:グループ外</param>
        /// <returns>int 描画カラー</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetDrawingColor(McDefine.SymbolDrawKind eDrawKind, HySObjectKind csDrawSubKind, bool bModelFlg, bool bInnerFlg)
        {
            int iRet = -1; // 復帰情報

            switch (eDrawKind)
            {
                case McDefine.SymbolDrawKind.HYM_GROUP:
                    iRet = m_iColorList[0];
                    break;
                case McDefine.SymbolDrawKind.HYM_CALC_GROUP:
                    if (bModelFlg == true)
                    {// モデル設定ありの場合
                        iRet = m_iColorList[1];
                    }
                    else
                    {// モデル設定なしの場合
                        iRet = m_iColorList[2];
                    }
                    break;
                case McDefine.SymbolDrawKind.HYM_STD_ELEMENT:
                    if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_BASIN)
                    {// 流域の場合
                        if (bModelFlg == true)
                        {// モデル設定ありの場合
                            iRet = m_iColorList[3];
                        }
                        else
                        {// モデル設定なしの場合
                            iRet = m_iColorList[4];
                        }
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_LANE)
                    {// 河道の場合
                        if (bModelFlg == true)
                        {// モデル設定ありの場合
                            iRet = m_iColorList[5];
                        }
                        else
                        {// モデル設定なしの場合
                            iRet = m_iColorList[6];
                        }
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_USER)
                    {// ユーザの場合
                        if (bModelFlg == true)
                        {// モデル設定ありの場合
                            iRet = m_iColorList[7];
                        }
                        else
                        {// モデル設定なしの場合
                            iRet = m_iColorList[8];
                        }
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_INPUT)
                    {// 入力の場合
                        if (bModelFlg == true)
                        {// モデル設定ありの場合
                            iRet = m_iColorList[9];
                        }
                        else
                        {// モデル設定なしの場合
                            iRet = m_iColorList[10];
                        }
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_OUTPUT)
                    {// 出力の場合
                        if (bModelFlg == true)
                        {// モデル設定ありの場合
                            iRet = m_iColorList[11];
                        }
                        else
                        {// モデル設定なしの場合
                            iRet = m_iColorList[12];
                        }
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE)
                    {// マージの場合
                        iRet = m_iColorList[13];
                    }
                    else
                    {// その他の場合
                        if (bModelFlg == true)
                        {// モデル設定ありの場合
                            iRet = m_iColorList[7];
                        }
                        else
                        {// モデル設定なしの場合
                            iRet = m_iColorList[8];
                        }
                    }
                    break;
                case McDefine.SymbolDrawKind.HYM_RECEIVE:
                    iRet = m_iColorList[14];
                    break;
                case McDefine.SymbolDrawKind.HYM_SEND:
                    iRet = m_iColorList[15];
                    break;
                case McDefine.SymbolDrawKind.HYM_TRANS_IN:
                    if (bInnerFlg == true)
                    {// グループ内部の場合
                        iRet = m_iColorList[16];
                    }
                    else
                    {// グループ外部の場合
                        iRet = m_iColorList[14];
                    }
                    break;
                case McDefine.SymbolDrawKind.HYM_TRANS_OUT:
                    if (bInnerFlg == true)
                    {// グループ内部の場合
                        iRet = m_iColorList[17];
                    }
                    else
                    {// グループ外部の場合
                        iRet = m_iColorList[15];
                    }
                    break;
                case McDefine.SymbolDrawKind.HYM_LINE:
                    if (bModelFlg == true)
                    {// モデル設定ありの場合
                        iRet = m_iColorList[18];
                    }
                    else
                    {// モデル設定なしの場合
                        iRet = m_iColorList[19];
                    }
                    break;
            }

            return iRet;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択描画カラー取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iColor = GetSelectDrawingColor(bModelFlg)</para>
        /// </example>
        /// <param name="bModelFlg">演算モデル設定有無</param>
        /// <returns>int 選択描画カラー</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetSelectDrawingColor(bool bModelFlg)
        {
            int iColor = -1;
            if (bModelFlg == true)
            {// モデル設定ありの場合
                iColor = m_iColorList[20];
            }
            else
            {// モデル設定なしの場合
                iColor = m_iColorList[21];
            }

            return iColor;
        }

        /// <summary><para>method outline:</para>
        /// <para>ID描画カラー取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iColor = GetIDDrawingColor()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>int ID描画カラー</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetIDDrawingColor()
        {
            return m_iColorList[22];
        }

        /// <summary><para>method outline:</para>
        /// <para>名称描画カラー取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iColor = GetNameDrawingColor()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>int 名称描画カラー</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetNameDrawingColor()
        {
            return m_iColorList[23];
        }

        /// <summary><para>method outline:</para>
        /// <para>移動量１取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iMovementRate = GetMovementRate1()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>int 移動量１</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetMovementRate1()
        {
            return m_iMovementRate1;
        }

        /// <summary><para>method outline:</para>
        /// <para>移動量１設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetMovementRate1(iMovementRate)</para>
        /// </example>
        /// <param name="iMovementRate">移動量１</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetMovementRate1(int iMovementRate)
        {
            m_iMovementRate1 = iMovementRate;
        }

        /// <summary><para>method outline:</para>
        /// <para>移動量２取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iMovementRate = GetMovementRate2()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>int 移動量２</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetMovementRate2()
        {
            return m_iMovementRate2;
        }

        /// <summary><para>method outline:</para>
        /// <para>移動量２設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetMovementRate2(iMovementRate)</para>
        /// </example>
        /// <param name="iMovementRate">移動量２</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetMovementRate2(int iMovementRate)
        {
            m_iMovementRate2 = iMovementRate;
        }

        /// <summary><para>method outline:</para>
        /// <para>接続線描画太さ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iDrawingThickness = GetDrawingThickness()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>int 接続線描画太さ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetDrawingThickness()
        {
            return m_iDrawingThickness;
        }

        /// <summary><para>method outline:</para>
        /// <para>接続線描画太さ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDrawingThickness(iDrawingThickness)</para>
        /// </example>
        /// <param name="iDrawingThickness">接続線描画太さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDrawingThickness(int iDrawingThickness)
        {
            m_iDrawingThickness = iDrawingThickness;
        }

        /// <summary><para>method outline:</para>
        /// <para>リサイズポイントサイズ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iResizePointSize = GetResizePointSize()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>int リサイズポイントサイズ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetResizePointSize()
        {
            return m_iResizePointSize;
        }

        /// <summary><para>method outline:</para>
        /// <para>リサイズポイントサイズ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetResizePointSize(iResizePointSize)</para>
        /// </example>
        /// <param name="iResizePointSize">リサイズポイントサイズ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetResizePointSize(int iResizePointSize)
        {
            m_iResizePointSize = iResizePointSize;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバー説明リスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜HySString＞ sToolStripExplanationList = GetToolStripExplanationList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜HySString＞ ツールバー説明リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<HySString> GetToolStripExplanationList()
        {
            return m_sToolStripExplanationList;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバー表示／非表示リスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜bool＞ bToolStripVisibleList = GetToolStripVisibleList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜bool＞ ツールバー表示／非表示リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<bool> GetToolStripVisibleList()
        {
            return m_bToolStripVisibleList;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバー表示／非表示リスト設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetToolStripVisibleList(bToolStripVisibleList)</para>
        /// </example>
        /// <param name="bToolStripVisibleList">ツールバー表示／非表示リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetToolStripVisibleList(List<bool> bToolStripVisibleList)
        {
            m_bToolStripVisibleList = bToolStripVisibleList;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバー位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Point pLocation = GetToolStripLocation(sToolStripName)</para>
        /// </example>
        /// <param name="sToolStripName">ツールバー名称</param>
        /// <returns>Point ツールバー位置</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public Point GetToolStripLocation(HySString sToolStripName)
        {
            Point pLocation = new Point(-1, -1);

            for (int iLp =0;iLp<m_sToolStripNameList.Count;iLp++)
            {// ツールバー名称数ループ
                if (sToolStripName == m_sToolStripNameList[iLp])
                {// 指定ツールバーの場合
                    pLocation = m_pToolStripLocationList[iLp];
                    break;
                }
            }

            return pLocation;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバー位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetToolStripLocation(sToolStripName, pLocation)</para>
        /// </example>
        /// <param name="sToolStripName">ツールバー名称</param>
        /// <param name="pLocation">ツールバー位置</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetToolStripLocation(HySString sToolStripName, Point pLocation)
        {
            for (int iLp = 0; iLp < m_sToolStripNameList.Count; iLp++)
            {// ツールバー名称数ループ
                if (sToolStripName == m_sToolStripNameList[iLp])
                {// 指定ツールバーの場合
                    m_pToolStripLocationList[iLp] = pLocation;
                    break;
                }
            }

            // ツールバーの位置にてリストをソート
            for (int iLp1 = 0; iLp1 < m_pToolStripLocationList.Count - 1; iLp1++)
            {// ツールバー位置数ループ
                for (int iLp2 = m_pToolStripLocationList.Count - 1; iLp2 > iLp1; iLp2--)
                {// 比較ツールバー数ループ
                    if (m_pToolStripLocationList[iLp2].Y < m_pToolStripLocationList[iLp2 - 1].Y)
                    {// リスト内の前のツールバーより横位置が小さい場合
                        m_pToolStripLocationList.Reverse(iLp2 - 1, 2);
                        m_sToolStripNameList.Reverse(iLp2 - 1, 2);
                        m_bToolStripVisibleList.Reverse(iLp2 - 1, 2);
                        m_sToolStripCollectionList.Reverse(iLp2 - 1, 2);
                        m_sToolStripExplanationList.Reverse(iLp2 - 1, 2);
                    }
                    else if ((m_pToolStripLocationList[iLp2].Y == m_pToolStripLocationList[iLp2 - 1].Y) &&
                             (m_pToolStripLocationList[iLp2].X < m_pToolStripLocationList[iLp2 - 1].X))
                    {// リスト内の前のツールバーより横位置が同じで縦位置が小さい場合
                        m_pToolStripLocationList.Reverse(iLp2 - 1, 2);
                        m_sToolStripNameList.Reverse(iLp2 - 1, 2);
                        m_bToolStripVisibleList.Reverse(iLp2 - 1, 2);
                        m_sToolStripCollectionList.Reverse(iLp2 - 1, 2);
                        m_sToolStripExplanationList.Reverse(iLp2 - 1, 2);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバー表示／非表示取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bVisibleMode = GetToolStripVisible(sToolStripName)</para>
        /// </example>
        /// <param name="sToolStripName">ツールバー名称</param>
        /// <returns>bool ture:表示 false:非表示</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetToolStripVisible(HySString sToolStripName)
        {
            bool bVisibleMode = false;

            for (int iLp = 0; iLp < m_sToolStripNameList.Count; iLp++)
            {// ツールバー名称数ループ
                if (sToolStripName == m_sToolStripNameList[iLp])
                {// 指定ツールバーの場合
                    bVisibleMode = m_bToolStripVisibleList[iLp];
                    break;
                }
            }

            return bVisibleMode;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバーコレクションリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜List＜HySString＞＞ sToolStripCollectionList = GetToolStripCollectionList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜List＜HySString＞＞ ツールバーコレクションリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<List<HySString>> GetToolStripCollectionList()
        {
            return m_sToolStripCollectionList;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバーコレクションリスト設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetToolStripCollectionList(sToolStripCollectionList)</para>
        /// </example>
        /// <param name="sToolStripCollectionList">ツールバーコレクションリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetToolStripCollectionList(List<List<HySString>> sToolStripCollectionList)
        {
            m_sToolStripCollectionList = sToolStripCollectionList;
        }

        /// <summary><para>method outline:</para>
        /// <para>ツールバー名称リスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜HySString＞ sToolStripNameList = GetToolStripNameList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜HySString＞ ツールバー名称リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<HySString> GetToolStripNameList()
        {
            return m_sToolStripNameList;
        }

        /// <summary><para>method outline:</para>
        /// <para>カスタマイズ情報保存</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SaveCustomInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SaveCustomInfo()
        {
            string sInfoName = HySEnvInf.GetHomeDirectory().ToString() + DATA_PATH + HySEnvInf.GetEnvInf(new HySString("MCMODEL_CUSTOM_INFO")).ToString();
            HySXmlWriter csXmlWriter = new HySXmlWriter(sInfoName);

            HySXmlNode csRootNode = csXmlWriter.CreateRoot(NODE_ROOT);

            //色ノード
            HySXmlNode csParentNode = HySXmlNode.CreateElement(NODE_COLOR, csXmlWriter);
            csRootNode.AppendChild(csParentNode);

            HySXmlNode csNode;
            HySXmlNode csChildNode;
            for (int iLp = 0; iLp < m_sColorItemList.Count; iLp++)
            {// カラー項目数ループ
                csNode = HySXmlNode.CreateElement(NODE_ITEM, csXmlWriter);
                csNode.SetAttribute(KEY_NAME, m_sColorItemList[iLp].ToString());
                csNode.SetAttribute(KEY_COLORCODE, m_iColorList[iLp].ToString());
                csParentNode.AppendChild(csNode);
            }

            //その他ノード
            csParentNode = HySXmlNode.CreateElement(NODE_OTHERS, csXmlWriter);
            csRootNode.AppendChild(csParentNode);

            csNode = HySXmlNode.CreateElement(NODE_ITEM, csXmlWriter);
            csNode.SetAttribute(KEY_NAME, KEY_MOVEMENTRATE1);
            csNode.SetAttribute(KEY_MOVEMENTRATE, m_iMovementRate1.ToString());
            csParentNode.AppendChild(csNode);

            csNode = HySXmlNode.CreateElement(NODE_ITEM, csXmlWriter);
            csNode.SetAttribute(KEY_NAME, KEY_MOVEMENTRATE2);
            csNode.SetAttribute(KEY_MOVEMENTRATE, m_iMovementRate2.ToString());
            csParentNode.AppendChild(csNode);

            csNode = HySXmlNode.CreateElement(NODE_ITEM, csXmlWriter);
            csNode.SetAttribute(KEY_NAME, KEY_LINE);
            csNode.SetAttribute(KEY_DRAWINGTHICKNESS, m_iDrawingThickness.ToString());
            csParentNode.AppendChild(csNode);

            csNode = HySXmlNode.CreateElement(NODE_ITEM, csXmlWriter);
            csNode.SetAttribute(KEY_NAME, KEY_RESIZEPOINT);
            csNode.SetAttribute(KEY_RESIZEPOINTSIZE, m_iResizePointSize.ToString());
            csParentNode.AppendChild(csNode);

            //ツールバーノード
            csParentNode = HySXmlNode.CreateElement(NODE_TOOLBAR, csXmlWriter);
            csRootNode.AppendChild(csParentNode);
            for (int iLp = 0; iLp < m_sToolStripNameList.Count; iLp++)
            {// ツールバー名称数ループ
                csNode = HySXmlNode.CreateElement(NODE_ITEM, csXmlWriter);
                csNode.SetAttribute(KEY_NAME, m_sToolStripNameList[iLp].ToString());
                csNode.SetAttribute(KEY_EXPLANATION, m_sToolStripExplanationList[iLp].ToString());
                csNode.SetAttribute(KEY_LOCATION_X, m_pToolStripLocationList[iLp].X.ToString());
                csNode.SetAttribute(KEY_LOCATION_Y, m_pToolStripLocationList[iLp].Y.ToString());
                csNode.SetAttribute(KEY_VISIBLE_MODE, m_bToolStripVisibleList[iLp].ToString());
                csParentNode.AppendChild(csNode);

                for (int iLp2 = 0; iLp2 < m_sToolStripCollectionList[iLp].Count; iLp2++)
                {// ツールバーコレクション数ループ
                    csChildNode = HySXmlNode.CreateElement(NODE_COLLECTION, csXmlWriter);
                    csChildNode.SetAttribute(KEY_NAME, m_sToolStripCollectionList[iLp][iLp2].ToString());
                    csNode.AppendChild(csChildNode);
                }
            }

            csXmlWriter.Save();
        }

        /// <summary><para>method outline:</para>
        /// <para>カスタマイズ情報読み込み</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>LoadCustomInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void LoadCustomInfo()
        {
            string sInfoName = HySEnvInf.GetHomeDirectory().ToString() + DATA_PATH + HySEnvInf.GetEnvInf(new HySString("MCMODEL_CUSTOM_INFO")).ToString();
            HySFile csFile = new HySFile(sInfoName);
            if (csFile.Exist() == true)
            {// ファイルがある場合
                m_sColorItemList.Clear();
                m_iColorList.Clear();
                m_sToolStripNameList.Clear();
                m_sToolStripExplanationList.Clear();
                m_pToolStripLocationList.Clear();
                m_bToolStripVisibleList.Clear();
                m_sToolStripCollectionList.Clear();

                HySXmlReader csXmlReader = new HySXmlReader(sInfoName);
                HySXmlNode csXmlRootNode = csXmlReader.GetRootNode();
                HySXmlNode csXmlChildNode;
                HySXmlNode csXmlGrandChildNode;
                HySXmlNode csXmlGreatGrandChildNode;
                HySXmlNodeList csXmlChildNodeList = csXmlRootNode.GetChildNodeList();
                HySXmlNodeList csXmlGrandChildNodeList;
                HySXmlNodeList csXmlGreatGrandChildNodeList;

                HySString csKeyName = null;
                HySString csKeyColorCode = null;
                HySString csKeyExplanation = null;
                int iLocationX = 0;
                int iLocationY = 0;
                bool bVisibleMode = true;
                for (int iLp1 = 0; iLp1 < csXmlChildNodeList.GetCount(); iLp1++)
                {// ノード数ループ
                    csXmlChildNode = csXmlChildNodeList.GetNode(iLp1);

                    if (csXmlChildNode.GetName().Equals(NODE_COLOR))
                    {// 色ノードの場合
                        csXmlGrandChildNodeList = csXmlChildNode.GetChildNodeList();
                        for (int iLp2 = 0; iLp2 < csXmlGrandChildNodeList.GetCount(); iLp2++)
                        {// ノード数ループ
                            csXmlGrandChildNode = csXmlGrandChildNodeList.GetNode(iLp2);

                            if (csXmlGrandChildNode.GetName().Equals(NODE_ITEM))
                            {// 項目ノードの場合
                                // 属性「name」を取得
                                csKeyName = new HySString(csXmlGrandChildNode.GetAttribute(KEY_NAME));
                                // 属性「ColorCode」を取得
                                csKeyColorCode = new HySString(csXmlGrandChildNode.GetAttribute(KEY_COLORCODE));

                                //項目名のデータを追加
                                m_sColorItemList.Add(csKeyName);
                                //色コードのデータを追加
                                m_iColorList.Add(Convert.ToInt32(csKeyColorCode.ToString()));
                            }
                        }
                    }

                    if (csXmlChildNode.GetName().Equals(NODE_OTHERS))
                    {// その他ノードの場合
                        csXmlGrandChildNodeList = csXmlChildNode.GetChildNodeList();
                        for (int iLp2 = 0; iLp2 < csXmlGrandChildNodeList.GetCount(); iLp2++)
                        {// ノード数ループ
                            csXmlGrandChildNode = csXmlGrandChildNodeList.GetNode(iLp2);

                            if (csXmlGrandChildNode.GetName().Equals(NODE_ITEM))
                            {// 項目ノードの場合
                                // 属性「name」を取得
                                csKeyName = new HySString(csXmlGrandChildNode.GetAttribute(KEY_NAME));

                                if (csKeyName == KEY_MOVEMENTRATE1)
                                {// 移動量１の場合
                                    // 属性「MovementRate」を取得
                                    m_iMovementRate1 = Convert.ToInt32(csXmlGrandChildNode.GetAttribute(KEY_MOVEMENTRATE));
                                }

                                if (csKeyName == KEY_MOVEMENTRATE2)
                                {// 移動量２の場合
                                    // 属性「MovementRate」を取得
                                    m_iMovementRate2 = Convert.ToInt32(csXmlGrandChildNode.GetAttribute(KEY_MOVEMENTRATE));
                                }

                                if (csKeyName == KEY_LINE)
                                {// 接続線の場合
                                    // 属性「DrawingThickness」を取得
                                    m_iDrawingThickness = Convert.ToInt32(csXmlGrandChildNode.GetAttribute(KEY_DRAWINGTHICKNESS));
                                }

                                if (csKeyName == KEY_RESIZEPOINT)
                                {// リサイズポイントの場合
                                    // 属性「ResizePointSize」を取得
                                    m_iResizePointSize = Convert.ToInt32(csXmlGrandChildNode.GetAttribute(KEY_RESIZEPOINTSIZE));
                                }
                            }
                        }
                    }

                    if (csXmlChildNode.GetName().Equals(NODE_TOOLBAR))
                    {// ツールバーノードの場合
                        csXmlGrandChildNodeList = csXmlChildNode.GetChildNodeList();
                        for (int iLp2 = 0; iLp2 < csXmlGrandChildNodeList.GetCount(); iLp2++)
                        {// ノード数ループ
                            csXmlGrandChildNode = csXmlGrandChildNodeList.GetNode(iLp2);

                            if (csXmlGrandChildNode.GetName().Equals(NODE_ITEM))
                            {// 項目ノードの場合
                                // 属性「name」を取得
                                csKeyName = new HySString(csXmlGrandChildNode.GetAttribute(KEY_NAME));
                                // 属性「Explanation」を取得
                                csKeyExplanation = new HySString(csXmlGrandChildNode.GetAttribute(KEY_EXPLANATION));
                                // 属性「Location_X」を取得
                                iLocationX = Convert.ToInt32(csXmlGrandChildNode.GetAttribute(KEY_LOCATION_X));
                                // 属性「Location_Y」を取得
                                iLocationY = Convert.ToInt32(csXmlGrandChildNode.GetAttribute(KEY_LOCATION_Y));
                                // 属性「Visible_Mode」を取得
                                bVisibleMode = Convert.ToBoolean(csXmlGrandChildNode.GetAttribute(KEY_VISIBLE_MODE));

                                //ツールバー名称データを追加
                                m_sToolStripNameList.Add(csKeyName);
                                //ツールバー説明データを追加
                                m_sToolStripExplanationList.Add(csKeyExplanation);
                                //ツールバー位置データを追加
                                m_pToolStripLocationList.Add(new Point(iLocationX, iLocationY));
                                //ツールバー表示／非表示モードデータを追加
                                m_bToolStripVisibleList.Add(bVisibleMode);

                                m_sToolStripCollectionList.Add(new List<HySString>());

                                csXmlGreatGrandChildNodeList = csXmlGrandChildNode.GetChildNodeList();
                                for (int iLp3 = 0; iLp3 < csXmlGreatGrandChildNodeList.GetCount(); iLp3++)
                                {// ノード数ループ
                                    csXmlGreatGrandChildNode = csXmlGreatGrandChildNodeList.GetNode(iLp3);

                                    if (csXmlGreatGrandChildNode.GetName().Equals(NODE_COLLECTION))
                                    {// コレクションノードの場合
                                        // 属性「name」を取得
                                        csKeyName = new HySString(csXmlGreatGrandChildNode.GetAttribute(KEY_NAME));

                                        //ツールバーコレクションデータを追加
                                        m_sToolStripCollectionList[iLp2].Add(csKeyName);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルトカラー設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDefaultColor()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDefaultColor()
        {
            // リストクリア
            m_sColorItemList.Clear();
            m_iColorList.Clear();

            // デフォルトカラー設定
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_PART));           //"部分系グループ要素"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_CONVERGENCE));    //"収束演算系グループ要素（モデル設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_CONVERGENCE_NON));//"収束演算系グループ要素（モデル未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_BASIN));          //"流域要素（モデル設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_BASIN_NON));      //"流域要素（モデル未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_RIVER));          //"河道要素（モデル設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_RIVER_NON));      //"河道要素（モデル未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_USER));           //"ユーザ要素（モデル設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_USER_NON));       //"ユーザ要素（モデル未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_INPUT));          //"入力要素（設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_INPUT_NON));      //"入力要素（未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_OUTPUT));         //"出力要素（設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_OUTPUT_NON));     //"出力要素（未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_MERGE));          //"データマージ要素"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_RECEPTION));      //"受信端子"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_TRANSMISSION));   //"送信端子"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_INBROADCAST));    //"入力中継端子"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_OUTBROADCAST));   //"出力中継端子"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_CONNECT));        //"接続線（伝送情報設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_CONNECT_NON));    //"接続線（伝送情報未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_SELECT));         //"選択（モデル等設定済）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_SELECT_NON));     //"選択（モデル等未設定）"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_ID));             //"要素ID"
            m_sColorItemList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_COLORITEM_NAME));           //"要素名称"
            m_iColorList.Add(4678655);
            m_iColorList.Add(3329330);
            m_iColorList.Add(64636);
            m_iColorList.Add(32768);
            m_iColorList.Add(9498256);
            m_iColorList.Add(16748574);
            m_iColorList.Add(16776960);
            m_iColorList.Add(16748574);
            m_iColorList.Add(16776960);
            m_iColorList.Add(1993170);
            m_iColorList.Add(42495);
            m_iColorList.Add(1993170);
            m_iColorList.Add(42495);
            m_iColorList.Add(16748574);
            m_iColorList.Add(32896);
            m_iColorList.Add(32896);
            m_iColorList.Add(15631086);
            m_iColorList.Add(15631086);
            m_iColorList.Add(16711680);
            m_iColorList.Add(11119017);
            m_iColorList.Add(16711935);
            m_iColorList.Add(12695295);
            m_iColorList.Add(8388736);
            m_iColorList.Add(16711680);
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルト設定値設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDefaultValue()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDefaultValue()
        {
            // デフォルト設定値設定
            m_iMovementRate1 = 1;
            m_iMovementRate2 = 1;
            m_iDrawingThickness = 2;
            m_iResizePointSize = 4;
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルト位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDefaultLocation()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDefaultLocation()
        {
            // リストクリア
            m_sToolStripNameList.Clear();
            m_sToolStripExplanationList.Clear();
            m_pToolStripLocationList.Clear();
            m_bToolStripVisibleList.Clear();
            m_sToolStripCollectionList.Clear();

            // デフォルト位置設定
            m_sToolStripNameList.Add(new HySString("ToolStrip1"));
            m_sToolStripNameList.Add(new HySString("ToolStrip3"));
            m_sToolStripNameList.Add(new HySString("ToolStrip5"));
            m_sToolStripNameList.Add(new HySString("ToolStrip2"));
            m_sToolStripNameList.Add(new HySString("ToolStrip4"));
            m_sToolStripNameList.Add(new HySString("ToolStrip6"));
            m_sToolStripExplanationList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_EDIT));        //"編集ツールバー"
            m_sToolStripExplanationList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_CONTROL));     //"コントロールツールバー"
            m_sToolStripExplanationList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_SIMULATION));  //"シミュレーションツールバー"
            m_sToolStripExplanationList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_PLACEMENT));   //"要素配置ツールバー"
            m_sToolStripExplanationList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_GROUP));       //"グループ化／解除ツールバー"
            m_sToolStripExplanationList.Add(new HySString(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_LININGUP));    //"要素整列ツールバー"
            m_pToolStripLocationList.Add(new Point(Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_EDIT_X), Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_EDIT_Y)));            //(3, 0)
            m_pToolStripLocationList.Add(new Point(Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_CONTROL_X), Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_CONTROL_Y)));      //(263, 0)
            m_pToolStripLocationList.Add(new Point(Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_SIMULATION_X), Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_SIMULATION_Y)));//(538, 0)
            m_pToolStripLocationList.Add(new Point(Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_PLACEMENT_X), Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_PLACEMENT_Y)));  //(3, 25)
            m_pToolStripLocationList.Add(new Point(Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_GROUP_X), Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_GROUP_Y)));          //(419, 25)
            m_pToolStripLocationList.Add(new Point(Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_LININGUP_X), Convert.ToInt32(Properties.McDotNetViewerResources.CUSTOMIZE_TOOLSTRIP_LININGUP_Y)));    //(500, 25)
            m_bToolStripVisibleList.Add(true);
            m_bToolStripVisibleList.Add(true);
            m_bToolStripVisibleList.Add(true);
            m_bToolStripVisibleList.Add(true);
            m_bToolStripVisibleList.Add(true);
            m_bToolStripVisibleList.Add(true);
            m_sToolStripCollectionList.Add(new List<HySString>());
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("CutButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("CopyButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("PasteButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("DeleteButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator11"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("UndoButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("RedoButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator12"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("TopButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("BottomButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator13"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("Left90degRotationButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("Right90degRotationButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("AllLeft90degRotationButton"));//[2010/01/06]
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("AllRight90degRotationButton"));//[2010/01/06]
            //★★★★★★★★★★★★★★★★★★★★★
            // Ver1.4で追加(GIS起動及び表示操作の簡略化
            //★★★★★★★★★★★★★★★★★★★★★
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("GISStartButton"));
            m_sToolStripCollectionList.Add(new List<HySString>());
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("SimpleMapButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator35"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("MoveButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator33"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ZoomInButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ZoomOutButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ZoomSizeComboBox"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator31"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("IDTypeButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator34"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ConnectCheckButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("HelpJumpButton"));
            m_sToolStripCollectionList.Add(new List<HySString>());
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("SimStartButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("SimPauseButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("SimStopButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("SimResetButton"));
            m_sToolStripCollectionList.Add(new List<HySString>());
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("SelectButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("PartGrElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("CalcGrElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("BasinElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("LaneElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("UserElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("JunctionElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("InputElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("OutputElementButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ToolSeparator21"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementDiagramMenu"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementConnectButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("SelectObjectText"));
            m_sToolStripCollectionList.Add(new List<HySString>());
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("PartGrButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("CalcGrButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("GrCancelButton"));
            m_sToolStripCollectionList.Add(new List<HySString>());
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementLineUpUpButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementLineUpBottomButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementLineUpLeftButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementLineUpRightButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementEqualDistanceLineUpVButton"));
            m_sToolStripCollectionList[m_sToolStripCollectionList.Count - 1].Add(new HySString("ElementEqualDistanceLineUpHButton"));
        }

        /// <summary><para>method outline:</para>
        /// <para>カスタマイズ情報変更チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ModificationCheck()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>CustomInfoModificationStatus 変更状態</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public CustomInfoModificationStatus ModificationCheck()
        {
            List<HySString> sTempColorItemList = new List<HySString>();
            List<int> iTempColorList = new List<int>();
            int iTempMovementRate1 = 0;
            int iTempMovementRate2 = 0;
            int iTempDrawingThickness = 0;
            int iTempResizePointSize = 0;
            List<HySString> sTempToolStripNameList = new List<HySString>();
            List<HySString> sTempToolStripExplanationList = new List<HySString>();
            List<Point> pTempToolStripLocationList = new List<Point>();
            List<bool> bTempToolStripVisibleList = new List<bool>();
            List<List<HySString>> sTempToolStripCollectionList = new List<List<HySString>>();
            CustomInfoModificationStatus eRtnStatus = CustomInfoModificationStatus.UNMODIFIED;
            bool bModifyFlg1 = false;
            bool bModifyFlg2 = false;
            bool bModifyFlg3 = false;

            // 現在値一時保存
            for (int iLp = 0; iLp < m_sColorItemList.Count; iLp++)
            {// カラー項目数ループ
                sTempColorItemList.Add(m_sColorItemList[iLp]);
            }
            for (int iLp = 0; iLp < m_iColorList.Count; iLp++)
            {// カラー数ループ
                iTempColorList.Add(m_iColorList[iLp]);
            }
            iTempMovementRate1 = m_iMovementRate1;
            iTempMovementRate2 = m_iMovementRate2;
            iTempDrawingThickness = m_iDrawingThickness;
            iTempResizePointSize = m_iResizePointSize;
            for (int iLp = 0; iLp < m_sToolStripNameList.Count; iLp++)
            {// ツールバー名称数ループ
                sTempToolStripNameList.Add(m_sToolStripNameList[iLp]);
            }
            for (int iLp = 0; iLp < m_sToolStripExplanationList.Count; iLp++)
            {// ツールバー説明数ループ
                sTempToolStripExplanationList.Add(m_sToolStripExplanationList[iLp]);
            }
            for (int iLp = 0; iLp < m_pToolStripLocationList.Count; iLp++)
            {// ツールバー位置数ループ
                pTempToolStripLocationList.Add(m_pToolStripLocationList[iLp]);
            }
            for (int iLp = 0; iLp < m_bToolStripVisibleList.Count; iLp++)
            {// ツールバー表示／非表示数ループ
                bTempToolStripVisibleList.Add(m_bToolStripVisibleList[iLp]);
            }
            for (int iLp = 0; iLp < m_sToolStripCollectionList.Count; iLp++)
            {// ツールバーコレクション数ループ
                sTempToolStripCollectionList.Add(new List<HySString>());
                for (int iLp2 = 0; iLp2 < m_sToolStripCollectionList[iLp].Count; iLp2++)
                {// ツールバーコレクション数ループ
                    sTempToolStripCollectionList[iLp].Add(m_sToolStripCollectionList[iLp][iLp2]);
                }
            }

            // カスタマイズ情報読み込み
            LoadCustomInfo();

            // カラー項目変更チェック
            if (m_sColorItemList.Count != sTempColorItemList.Count)
            {// カラー項目数が違う場合
                bModifyFlg1 = true;
            }
            else
            {// カラー項目数が同じ場合
                for (int iLp = 0; iLp < m_sColorItemList.Count; iLp++)
                {// カラー項目数ループ
                    if (m_sColorItemList[iLp] != sTempColorItemList[iLp])
                    {// カラー項目が違う場合
                        bModifyFlg1 = true;
                    }
                }
            }

            // カラー変更チェック
            if (m_iColorList.Count != iTempColorList.Count)
            {// カラー数が違う場合
                bModifyFlg1 = true;
            }
            else
            {// カラー数が同じ場合
                for (int iLp = 0; iLp < m_iColorList.Count; iLp++)
                {// カラー数ループ
                    if (m_iColorList[iLp] != iTempColorList[iLp])
                    {// カラーが違う場合
                        bModifyFlg1 = true;
                    }
                }
            }

            // 移動量１変更チェック
            if (m_iMovementRate1 != iTempMovementRate1)
            {// 移動量１が違う場合
                bModifyFlg2 = true;
            }

            // 移動量２変更チェック
            if (m_iMovementRate2 != iTempMovementRate2)
            {// 移動量２が違う場合
                bModifyFlg2 = true;
            }

            // 接続線描画太さ変更チェック
            if (m_iDrawingThickness != iTempDrawingThickness)
            {// 接続線描画太さが違う場合
                bModifyFlg2 = true;
            }

            // リサイズポイントサイズ変更チェック
            if (m_iResizePointSize != iTempResizePointSize)
            {// リサイズポイントサイズが違う場合
                bModifyFlg2 = true;
            }

            // ツーバー名称変更チェック
            if (m_sToolStripNameList.Count != sTempToolStripNameList.Count)
            {// ツーバー名称数が違う場合
                bModifyFlg3 = true;
            }
            else
            {// ツーバー名称数が同じ場合
                for (int iLp = 0; iLp < m_sToolStripNameList.Count; iLp++)
                {// ツーバー名称数ループ
                    if (m_sToolStripNameList[iLp] != sTempToolStripNameList[iLp])
                    {// ツーバー名称が違う場合
                        bModifyFlg3 = true;
                    }
                }
            }

            // ツールバー説明変更チェック
            if (m_sToolStripExplanationList.Count != sTempToolStripExplanationList.Count)
            {// ツーバー説明数が違う場合
                bModifyFlg3 = true;
            }
            else
            {// ツーバー説明数が同じ場合
                for (int iLp = 0; iLp < m_sToolStripExplanationList.Count; iLp++)
                {// ツーバー説明数ループ
                    if (m_sToolStripExplanationList[iLp] != sTempToolStripExplanationList[iLp])
                    {// ツーバー説明が違う場合
                        bModifyFlg3 = true;
                    }
                }
            }

            // ツールバー位置変更チェック
            if (m_pToolStripLocationList.Count != pTempToolStripLocationList.Count)
            {// ツーバー位置数が違う場合
                bModifyFlg3 = true;
            }
            else
            {// ツーバー位置数が同じ場合
                for (int iLp = 0; iLp < m_pToolStripLocationList.Count; iLp++)
                {// ツーバー位置数ループ
                    if (m_pToolStripLocationList[iLp] != pTempToolStripLocationList[iLp])
                    {// ツーバー位置が違う場合
                        bModifyFlg3 = true;
                    }
                }
            }

            // ツールバー表示／非表示変更チェック
            if (m_bToolStripVisibleList.Count != bTempToolStripVisibleList.Count)
            {// ツーバー表示／非表示数が違う場合
                bModifyFlg3 = true;
            }
            else
            {// ツーバー表示／非表示数が同じ場合
                for (int iLp = 0; iLp < m_bToolStripVisibleList.Count; iLp++)
                {// ツーバー表示／非表示数ループ
                    if (m_bToolStripVisibleList[iLp] != bTempToolStripVisibleList[iLp])
                    {// ツーバー表示／非表示が違う場合
                        bModifyFlg3 = true;
                    }
                }
            }

            // ツールバーコレクション変更チェック
            if (m_sToolStripCollectionList.Count != sTempToolStripCollectionList.Count)
            {// ツーバーコレクション数が違う場合
                bModifyFlg3 = true;
            }
            else
            {// ツーバーコレクション数が同じ場合
                for (int iLp = 0; iLp < m_sToolStripCollectionList.Count; iLp++)
                {// ツーバーコレクション数ループ
                    if (m_sToolStripCollectionList[iLp].Count != sTempToolStripCollectionList[iLp].Count)
                    {// ツーバーコレクション数が違う場合
                        bModifyFlg3 = true;
                    }
                    else
                    {// ツーバーコレクション数が同じ場合
                        for (int iLp2 = 0; iLp2 < m_sToolStripCollectionList[iLp].Count; iLp2++)
                        {// ツーバーコレクション数ループ
                            if (m_sToolStripCollectionList[iLp][iLp2] != sTempToolStripCollectionList[iLp][iLp2])
                            {// ツーバーコレクションが違う場合
                                bModifyFlg3 = true;
                            }
                        }
                    }
                }
            }

            // リターン値編集
            if (bModifyFlg1 == true)
            {// 変更フラグ１がONの場合
                eRtnStatus = CustomInfoModificationStatus.MODIFIED_COLOR;
            }
            if (bModifyFlg2 == true)
            {// 変更フラグ２がONの場合
                if (eRtnStatus == CustomInfoModificationStatus.UNMODIFIED)
                {// 変更状態が変更なしの場合
                    eRtnStatus = CustomInfoModificationStatus.MODIFIED_VALUE;
                }
                else
                {// 変更状態が変更なし以外の場合
                    eRtnStatus |= CustomInfoModificationStatus.MODIFIED_VALUE;
                }
            }
            if (bModifyFlg3 == true)
            {// 変更フラグ３がONの場合
                if (eRtnStatus == CustomInfoModificationStatus.UNMODIFIED)
                {// 変更状態が変更なしの場合
                    eRtnStatus = CustomInfoModificationStatus.MODIFIED_TOOLBAR;
                }
                else
                {// 変更状態が変更なし以外の場合
                    eRtnStatus |= CustomInfoModificationStatus.MODIFIED_TOOLBAR;
                }
            }

            return eRtnStatus;
        }
    }
}
