﻿// <summary>ソースコード：ＨＹＭＣＯ描画要素</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Drawing;

using CommonMP.HYSSOP.Interface;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.StructInfo;

namespace CommonMP.HYMCO.CoreOptions.McViewer.DotNetViewer
{
    /// <summary><para>class outline:</para>
    /// <para>描画要素クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/11/17][新規作成]</para>
    /// </remarks>
    /// <remarks><para>remarks:</para>
    /// <para>本クラスはバージョンアップ互換性の対象外</para>
    /// </remarks>
    [Serializable]
    public class McGraphicsElement : McGraphicsRoot
    {
        /// <summary> 演算要素 </summary>
        protected McStructElement m_csViewerElm;
        /// <summary>演算要素端子 </summary>
        protected McStructPort m_csViewerPort;
        /// <summary>　接続線 </summary>
        protected McStructConnection m_csViewerConnect;

        /// <summary> 自身を持つ親要素 </summary>
        protected McGraphicsElement m_csParentElm = null;

        /// <summary> 描画要素端子リスト </summary>
        protected List<McGraphicsElement> m_csPortList;

        /// <summary> シンボル描画 </summary>
        protected McSymbolDraw m_csDrawObject;

        /// <summary> 要素名称（中継端子のみ使用） </summary>
        protected HySString m_csElementName = new HySString();
        /// <summary> 画面表示設定 </summary>
        protected bool m_bDisplayMode = false;

        /// <summary> ダミーモデル個別情報 </summary>
        static readonly protected McModelInfo m_csDmyInfo = new McModelInfo(new HySID(""), new HySObjectKind(""), new HySObjectKind(""), new HySString(""));  //McStructElementに合わせて

        /// <summary>ユーザー設定IDのショートタイプ表示桁数</summary>
        static protected readonly int USER_MAKE_ID_SHORT_LENGTH = 4;

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = new McGraphicsElement( eKind, csDrawSubKind )</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <param name="csDrawSubKind">シンボル描画サブ種別</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement(McDefine.SymbolDrawKind eKind, HySObjectKind csDrawSubKind)
        {
            m_csPortList = new List<McGraphicsElement>();

            // シンボル描画生成
            CreateDrawObject(eKind, csDrawSubKind);
        }

        //======================================
        // 各種セッター、ゲッター関連メソッド
        //======================================
        /// <summary><para>method outline:</para>
        /// <para>ID設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetID(csID)</para>
        /// </example>
        /// <param name="csID">ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetID(HySIdentifier csID)
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.SetID(csID);
            }
            else if (m_csViewerPort != null)
            {// 端子の場合
                m_csViewerPort.SetID(csID);
            }
            else if (m_csViewerConnect != null)
            {// 接続の場合
                m_csViewerConnect.SetID(csID);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>ID取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySIdentifier csID = GetID()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySIdentifier : ID</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySIdentifier GetID()
        {
            HySIdentifier csID = new HySID("");
            if (m_csViewerElm != null)
            {// 要素の場合
                csID = m_csViewerElm.GetID();
            }
            else if (m_csViewerPort != null)
            {// 端子の場合
                csID = m_csViewerPort.GetID();
            }
            else if (m_csViewerConnect != null)
            {// 接続の場合
                csID = m_csViewerConnect.GetID();
            }
            return csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>ID一致判定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bFlg = Equal( csID )</para>
        /// </example>
        /// <param name="csID">ID</param>
        /// <returns>bool true:一致，false:不一致</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool Equal(HySIdentifier csID)
        {
            if (csID == null || GetID() == null)
            {// パラメータ,メンバ変数異常
                return false;
            }

            return GetID().Equals(csID);
        }


        /// <summary><para>method outline:</para>
        /// <para>画面表示要素設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetViewerElm(csElm)</para>
        /// </example>
        /// <param name="csElm">画面表示要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetViewerElm(McStructElement csElm)
        {
            m_csViewerElm = csElm;

            long lWidth = 0;
            long lHeight = 0;
            m_csViewerElm.GetElementSize(ref lWidth, ref lHeight);
            if (lWidth == 0)
            {// 幅が０の場合
                lWidth = m_csDrawObject.DefaultWidth / 2 * 2;
                m_csViewerElm.SetElementSize(lWidth, lHeight);
            }
            if (lHeight == 0)
            {// 高さが０の場合
                lHeight = m_csDrawObject.DefaultHeight / 2 * 2;
                m_csViewerElm.SetElementSize(lWidth, lHeight);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示要素取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructElement csStructElement = GetViewerElm()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McStructElement　画面表示要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McStructElement GetViewerElm()
        {
            return m_csViewerElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示端子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetViewerPort(csPort)</para>
        /// </example>
        /// <param name="csPort">画面表示端子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetViewerPort(McStructPort csPort)
        {
            m_csViewerPort = csPort;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示端子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructPort csPort = GetViewerPort()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McStructPort　画面表示端子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McStructPort GetViewerPort()
        {
            return m_csViewerPort;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示接続要素設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetViewerConnect(csConnect)</para>
        /// </example>
        /// <param name="csConnect">画面表示接続要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetViewerConnect(McStructConnection csConnect)
        {
            m_csViewerConnect = csConnect;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示接続要素取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructConnection csConnect = GetViewerConnect()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McStructConnection　画面表示接続要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McStructConnection GetViewerConnect()
        {
            return m_csViewerConnect;
        }

        /// <summary><para>method outline:</para>
        /// <para>自身を持つ親要素設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetParentElm( csParentElm )</para>
        /// </example>
        /// <param name="csParentElm">自身を持つ親要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetParentElm(McGraphicsElement csParentElm)
        {
            m_csParentElm = csParentElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>自身を持つ親要素取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csParentElm = GetParentElm()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McGraphicsElement 自身を持つ親要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement GetParentElm()
        {
            return m_csParentElm;
        }
        
        /// <summary><para>method outline:</para>
        /// <para>シンボル描画種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McDefine.SymbolDrawKind eKind = GetSymbolDrawKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McDefine.SymbolDrawKind  シンボル描画種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McDefine.SymbolDrawKind GetSymbolDrawKind()
        {
            McDefine.SymbolDrawKind eDrawKind = McDefine.SymbolDrawKind.HYM_NonSymbol;
            if (m_csViewerElm != null)
            {// 要素の場合
                eDrawKind = m_csViewerElm.GetSymbolDrawKind();
            }
            else if (m_csViewerPort != null)
            {// 端子の場合
                eDrawKind = m_csViewerPort.GetSymbolDrawKind();
            }
            else if (m_csViewerConnect != null)
            {// 接続の場合
                eDrawKind = McDefine.SymbolDrawKind.HYM_LINE;
            }
            return eDrawKind;
        }


        //======================================
        //表示関連メソッド
        //======================================

        /// <summary><para>method outline:</para>
        /// <para>シンボル描画生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRet = CreateDrawObject( eKind, csDrawSubKind )</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <param name="csDrawSubKind">シンボル描画サブ種別</param>
        /// <returns>>long  シンボル描画生成結果  0 : 成功、-1 : 失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected long CreateDrawObject(McDefine.SymbolDrawKind eKind, HySObjectKind csDrawSubKind)
        {
            long lRet = 0L; // 復帰情報

            switch (eKind)
            {
                case McDefine.SymbolDrawKind.HYM_STD_ELEMENT:
                    if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_BASIN)
                    {// 流域の場合
                        m_csDrawObject = new McBasinSymbolDraw();
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_LANE)
                    {// 河道の場合
                        m_csDrawObject = new McLaneSymbolDraw();
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_USER)
                    {// ユーザの場合
                        m_csDrawObject = new McFreeSymbolDraw();
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_INPUT)
                    {// 入力の場合
                        m_csDrawObject = new McInputSymbolDraw();
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_OUTPUT)
                    {// 出力の場合
                        m_csDrawObject = new McOutputSymbolDraw();
                    }
                    else if (csDrawSubKind == McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE)
                    {// マージの場合
                        m_csDrawObject = new McJunctionSymbolDraw();
                    }
                    else
                    {// その他の場合
                        m_csDrawObject = new McFreeSymbolDraw();
                    }
                    break;
                case McDefine.SymbolDrawKind.HYM_RECEIVE:
                    m_csDrawObject = new McReceiveSymbolDraw();
                    break;
                case McDefine.SymbolDrawKind.HYM_SEND:
                    m_csDrawObject = new McSendSymbolDraw();
                    break;
                case McDefine.SymbolDrawKind.HYM_TRANS_IN:
                    m_csDrawObject = new McTransSymbolDraw();
                    break;
                case McDefine.SymbolDrawKind.HYM_TRANS_OUT:
                    m_csDrawObject = new McTransSymbolDraw();
                    break;
                case McDefine.SymbolDrawKind.HYM_LINE:
                    m_csDrawObject = new McLineSymbolDraw();
                    break;
                case McDefine.SymbolDrawKind.HYM_GROUP:
                    m_csDrawObject = new McGroupSymbolDraw();
                    break;
                case McDefine.SymbolDrawKind.HYM_CALC_GROUP:
                    m_csDrawObject = new McGroupSymbolDraw();
                    break;
                default:
                    lRet = -1L;
                    break;
            }

            return lRet;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetPosition( lX, lY )</para>
        /// </example>
        /// <param name="lX">描画位置(X)</param>
        /// <param name="lY">描画位置(Y)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetPosition( long lX, long lY )
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.SetPosition(lX, lY);
            }
            else if (m_csViewerPort != null)
            {// 端子の場合
                m_csViewerPort.SetPosition(lX, lY);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>描画位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetPosition(ref lX, ref lY)</para>
        /// </example>
        /// <param name="lX">描画位置(X)</param>
        /// <param name="lY">描画位置(Y)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void GetPosition(ref long lX, ref long lY)
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.GetPosition(ref lX, ref lY);
            }
            else if (m_csViewerPort != null)
            {// 端子の場合
                m_csViewerPort.GetPosition(ref lX, ref lY);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>描画位置設定(回転)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetRotate( dRotate )</para>
        /// </example>
        /// <param name="dRotate">描画位置(回転)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetRotate(double dRotate)
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.SetRotate(dRotate);
            }
            else if (m_csViewerPort != null)
            {// 端子の場合
                m_csViewerPort.SetConnectDirection((long)dRotate);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>描画位置取得(回転)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double dRotate = GetRotate()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double  描画位置(回転)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public double GetRotate()
        {
            double dRotate = 0;
            if (m_csViewerElm != null)
            {// 要素の場合
                dRotate = m_csViewerElm.GetRotate();
            }
            else if (m_csViewerPort != null)
            {// 端子の場合
                dRotate = (double)m_csViewerPort.GetConnectDirection();
            }
            return dRotate;
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントサイズ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetElementSize(lWidth, lHeight)</para>
        /// </example>
        /// <param name="lWidth">幅</param>
        /// <param name="lHeight">高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetElementSize(long lWidth, long lHeight)
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.SetElementSize(lWidth, lHeight);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントサイズ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetElementSize(ref lWidth, ref lHeight)</para>
        /// </example>
        /// <param name="lWidth">幅</param>
        /// <param name="lHeight">高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void GetElementSize(ref long lWidth, ref long lHeight)
        {
            lWidth = m_csDrawObject.DefaultWidth / 2 * 2;
            lHeight = m_csDrawObject.DefaultHeight / 2 * 2;
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.GetElementSize(ref lWidth, ref lHeight);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>要素名称設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetElementName(ElementName)</para>
        /// </example>
        /// <param name="ElementName">要素名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetElementName(HySString ElementName)
        {
            if ((object)ElementName != null)
            {// パラメータ有効の場合
                if (m_csViewerElm != null)
                {// 要素の場合
                    m_csViewerElm.SetElementName(ElementName);
                }
                if (m_csViewerConnect != null)
                {// 接続線の場合
                    m_csViewerConnect.SetConnectionName(ElementName);

                    for (int iLp = 0; iLp < m_csPortList.Count; iLp++)
                    {// 端子を検索
                        if ((m_csPortList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                            (m_csPortList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                        {// 中継端子の場合
                            m_csPortList[iLp].SetElementName(ElementName);
                        }
                    }
                }
                if (m_csViewerPort != null)
                {// 端子の場合
                    if ((object)m_csElementName != null)
                    {// 設定がある場合
                        m_csElementName.SetString(ElementName);
                    }
                    else
                    {// 設定がない場合
                        m_csElementName = ElementName;
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>要素名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySString csName = GetElementName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString  要素名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySString GetElementName()
        {
            HySString sRtnElementName = new HySString();
            if (m_csViewerElm != null)
            {// 要素の場合
                sRtnElementName = m_csViewerElm.GetElementName();
            }
            if (m_csViewerConnect != null)
            {// 接続線の場合
                sRtnElementName = m_csViewerConnect.GetConnectionName();
            }
            if (m_csViewerPort != null)
            {// 端子の場合
                if ((object)m_csElementName != null)
                {// 設定がある場合
                    sRtnElementName = m_csElementName;
                }
            }

            return sRtnElementName;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetElementDisplayMode(DisplayMode)</para>
        /// </example>
        /// <param name="DisplayMode">要素名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetElementDisplayMode(bool DisplayMode)
        {
            m_bDisplayMode = DisplayMode;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示設定取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bMode = GetElementDisplayMode()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>bool  画面表示設定</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetElementDisplayMode()
        {
            return m_bDisplayMode;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定位置エレメント有無チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bFlg = ElementPositionCheck(Position_x, Position_y, lDispScale)</para>
        /// </example>
        /// <param name="Position_x">チェック位置X</param>
        /// <param name="Position_y">チェック位置Y</param>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <returns>bool　true：指定位置該当　false：指定位置非該当</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool ElementPositionCheck(int Position_x, int Position_y, long lDispScale)
        {
            bool bRtnFlg = false;
            int iWidth = 0;
            int iHeight = 0;
            long lDispX = 0;
            long lDispY = 0;

            double dRotate = GetRotate();
            long lWidth = 0;
            long lHeight = 0;
            GetElementSize(ref lWidth, ref lHeight);
            if ((dRotate % McDefine.HYM_DRAW_ANGLE_180) == 0.0)
            {// 向きが縦の場合
                iWidth = (int)lWidth;
                iHeight = (int)lHeight;
            }
            else
            {// 向きが横の場合
                iWidth = (int)lHeight;
                iHeight = (int)lWidth;
            }

            McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind();
            long lX = 0;
            long lY = 0;
            GetPosition(ref lX, ref lY);
            if ((eDrawKind != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_SEND) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 端子以外の場合
                McViewTransFormer.ViewTrans(lDispScale, lX, lY, ref lDispX, ref lDispY);

                McViewTransFormer.SizeTrans(lDispScale, eDrawKind, ref iWidth, ref iHeight);
            }
            else
            {// 端子の場合
                McViewTransFormer.ViewTrans(eDrawKind, lHeight, dRotate, lDispScale, lX, lY, ref lDispX, ref lDispY);
            }

            if ((Position_x >= lDispX - iWidth / 2) && (Position_y >= lDispY - iHeight / 2) &&
                (Position_x <= lDispX + iWidth / 2) && (Position_y <= lDispY + iHeight / 2))
            {// 指定位置該当の場合
                bRtnFlg = true;
            }

            return bRtnFlg;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定範囲該当チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bFlg = SelectedRangeCheck(iFromX, iFromY, iToX, iToY, lDispScale)</para>
        /// </example>
        /// <param name="iFromX">指定範囲FromX</param>
        /// <param name="iFromY">指定範囲FromY</param>
        /// <param name="iToX">指定範囲ToX</param>
        /// <param name="iToY">指定範囲ToY</param>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <returns>bool　true：指定範囲該当　false：指定範囲非該当</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool SelectedRangeCheck(int iFromX, int iFromY, int iToX, int iToY, long lDispScale)
        {
            bool bRtnFlg = false;
            int iWidth = 0;
            int iHeight = 0;
            long lDispX = 0;
            long lDispY = 0;

            double dRotate = GetRotate();
            long lWidth = 0;
            long lHeight = 0;
            GetElementSize(ref lWidth, ref lHeight);
            if ((dRotate % McDefine.HYM_DRAW_ANGLE_180) == 0.0)
            {// 向きが縦の場合
                iWidth = (int)lWidth;
                iHeight = (int)lHeight;
            }
            else
            {// 向きが横の場合
                iWidth = (int)lHeight;
                iHeight = (int)lWidth;
            }

            McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind();
            long lX = 0;
            long lY = 0;
            GetPosition(ref lX, ref lY);
            if ((eDrawKind != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_SEND) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 端子以外の場合
                McViewTransFormer.ViewTrans(lDispScale, lX, lY, ref lDispX, ref lDispY);

                McViewTransFormer.SizeTrans(lDispScale, eDrawKind, ref iWidth, ref iHeight);
            }
            else
            {// 端子の場合
                McViewTransFormer.ViewTrans(eDrawKind, lHeight, dRotate, lDispScale, lX, lY, ref lDispX, ref lDispY);
            }

            if ((iFromX <= lDispX - iWidth / 2) && (iFromY <= lDispY - iHeight / 2) &&
                (iToX >= lDispX + iWidth / 2) && (iToY >= lDispY + iHeight / 2))
            {// 指定範囲内の場合
                bRtnFlg = true;
            }

            return bRtnFlg;
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントリサイズポイントクリックチェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ElementResizePointCheck(Position_x, Position_y, lDispScale, iResizePointSize)</para>
        /// </example>
        /// <param name="Position_x">チェック位置X</param>
        /// <param name="Position_y">チェック位置Y</param>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <param name="iResizePointSize">リサイズポイントサイズ</param>
        /// <returns>bool　true：指定位置該当　false：指定位置非該当</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool ElementResizePointCheck(int Position_x, int Position_y, long lDispScale, int iResizePointSize)
        {
            bool bRtnFlg = false;
            int iWidth = 0;
            int iHeight = 0;
            long lDispX = 0;
            long lDispY = 0;

            double dRotate = GetRotate();
            long lWidth = 0;
            long lHeight = 0;
            GetElementSize(ref lWidth, ref lHeight);
            if ((dRotate % McDefine.HYM_DRAW_ANGLE_180) == 0.0)
            {// 向きが縦の場合
                iWidth = (int)lWidth;
                iHeight = (int)lHeight;
            }
            else
            {// 向きが横の場合
                iWidth = (int)lHeight;
                iHeight = (int)lWidth;
            }

            McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind();
            long lX = 0;
            long lY = 0;
            GetPosition(ref lX, ref lY);
            McViewTransFormer.ViewTrans(lDispScale, lX, lY, ref lDispX, ref lDispY);
            McViewTransFormer.SizeTrans(lDispScale, eDrawKind, ref iWidth, ref iHeight);

            if ((lDispX - iWidth / 2 - iResizePointSize / 2 > Position_x) ||
                (lDispX + iWidth / 2 + iResizePointSize / 2 < Position_x) ||
                (lDispY - iHeight / 2 - iResizePointSize / 2 > Position_y) ||
                (lDispY + iHeight / 2 + iResizePointSize / 2 < Position_y))
            {// 外接範囲外の場合
                return bRtnFlg;
            }

            Point[] pCheck = new Point[4];
            pCheck[0].X = (int)Math.Abs(Position_x - (lDispX - iWidth / 2));
            pCheck[0].Y = (int)Math.Abs(Position_y - (lDispY - iHeight / 2));
            pCheck[1].X = (int)Math.Abs(Position_x - (lDispX - iWidth / 2));
            pCheck[1].Y = (int)Math.Abs(Position_y - (lDispY + iHeight / 2));
            pCheck[2].X = (int)Math.Abs(Position_x - (lDispX + iWidth / 2));
            pCheck[2].Y = (int)Math.Abs(Position_y - (lDispY - iHeight / 2));
            pCheck[3].X = (int)Math.Abs(Position_x - (lDispX + iWidth / 2));
            pCheck[3].Y = (int)Math.Abs(Position_y - (lDispY + iHeight / 2));

            for (int iLp = 0; iLp < 4; iLp++)
            {// 四隅分ループ
                if ((pCheck[iLp].X <= iResizePointSize / 2) && (pCheck[iLp].Y <= iResizePointSize / 2))
                {// ポイント内の場合
                    bRtnFlg = true;
                    break;
                }
            }

            return bRtnFlg;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画範囲該当チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bFlg = DrawCheck(iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">編集領域（幅）</param>
        /// <param name="iHeight">編集領域（高さ）</param>
        /// <returns>bool　true：描画範囲該当　false：描画範囲非該当</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool DrawCheck(int iX, int iY, int iWidth, int iHeight)
        {
            bool bRtnFlg = false;
            int iElementWidth = 0;
            int iElementHeight = 0;

            double dRotate = GetRotate();
            long lWidth = 0;
            long lHeight = 0;
            GetElementSize(ref lWidth, ref lHeight);
            if ((dRotate % McDefine.HYM_DRAW_ANGLE_180) == 0.0)
            {// 向きが縦の場合
                iElementWidth = (int)lWidth;
                iElementHeight = (int)lHeight;
            }
            else
            {// 向きが横の場合
                iElementWidth = (int)lHeight;
                iElementHeight = (int)lWidth;
            }

            long lDispX = 0;
            long lDispY = 0;
            GetPosition(ref lDispX, ref lDispY);

            if ((iX < lDispX + iElementWidth / 2) && (iY < lDispY + iElementHeight / 2) &&
                (iX + iWidth > lDispX - iElementWidth / 2) && (iY + iHeight > lDispY - iElementHeight / 2))
            {// 描画範囲該当の場合
                bRtnFlg = true;
            }

            return bRtnFlg;
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObject(csGraphics, csCustomInfo, lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="csCustomInfo">モデル編集画面カスタマイズ情報</param>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">編集領域（幅）</param>
        /// <param name="iHeight">編集領域（高さ）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void DrawObject(Graphics csGraphics, McCustomInfo csCustomInfo, long lDispScale, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            if (false == DrawCheck(iX, iY, iWidth, iHeight))
            {// 描画範囲外の場合
                return;
            }

            long lDispX = 0;
            long lDispY = 0;
            McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind();
            long lX = 0;
            long lY = 0;
            long lWidth = 0;
            long lHeight = 0;
            GetPosition(ref lX, ref lY);
            GetElementSize(ref lWidth, ref lHeight);
            double dRotate = GetRotate();
            if ((eDrawKind != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_SEND) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 端子以外の場合
                McViewTransFormer.ViewTrans(lDispScale, lX, lY, ref lDispX, ref lDispY);
            }
            else
            {// 端子の場合
                McViewTransFormer.ViewTrans(eDrawKind, lHeight, dRotate, lDispScale, lX, lY, ref lDispX, ref lDispY);
            }

            int iElementWidth = (int)lWidth;
            int iElementHeight = (int)lHeight;
            McViewTransFormer.SizeTrans(lDispScale, eDrawKind, ref iElementWidth, ref iElementHeight);

            bool bModelFlg = false;
            if (this.GetModelInfo().GetLibraryID().ToString() != "")
            {// モデル設定ありの場合
                bModelFlg = true;
            }

            string sIconName = null;
            HySString csHomeDir = HySEnvInf.GetHomeDirectory();
            HySString csIconPath = HySEnvInf.GetEnvInf("ICON_FILE_PATH");
            HySString csIconName = this.GetModelInfo().GetIconName();
            if (((object)csIconName) == null)
            {// アイコン指定なしの場合
                sIconName = "Default.ico";
            }
            else
            {// アイコン指定ありの場合
                sIconName = csIconName.ToString();
                if (sIconName == null || sIconName == string.Empty)
                {// アイコン名称がなしの場合
                    sIconName = "Default.ico";
                }
                else
                {// アイコン名称ありの場合
                    if (sIconName.LastIndexOf(".ico", StringComparison.CurrentCultureIgnoreCase) < 0)
                    {// 拡張子なしの場合
                        sIconName = sIconName + ".ico";
                    }
                }
            }
            string sIconPath = csHomeDir.ToString() + csIconPath.ToString() + sIconName;
            HySFile csFile = new HySFile(sIconPath);
            if (csFile.Exist() != true)
            {// アイコンファイルがない場合
                sIconPath = null;
                sIconPath = csHomeDir.ToString() + csIconPath.ToString() + "Unknown.ico";
            }
            else
            {// アイコンファイルがある場合
                try
                {
                    Icon csIcon = new Icon(sIconPath);
                    csIcon.Dispose();
                }
                catch
                {
                    sIconPath = csHomeDir.ToString() + csIconPath.ToString() + "Unknown.ico";
                }
            }

            HySObjectKind csDrawSubKind = GetDrawSubKind();
            int iColor = csCustomInfo.GetDrawingColor(eDrawKind, csDrawSubKind, bModelFlg, false);
            Brush csDrawBrush = new SolidBrush(ColorTranslator.FromWin32(iColor));

            // 自身の描画
            m_csDrawObject.DrawObject(csGraphics, lDispX - iImageX, lDispY - iImageY, iElementWidth, iElementHeight, dRotate, csDrawBrush, new HySString(sIconPath));

            csDrawBrush.Dispose();
        }

        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        // Ver1.3では下記コードを追加する（Gr内表示） 
        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        /// <summary><para>method outline:</para>
        /// <para>Gr要素内部のイメージ画像設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetGrImage(csGraphics)</para>
        /// </example>
        /// <param name="csGraphics">イメージ画像</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetGrImage(Image csGraphics)
        {
            if (m_csDrawObject as McGroupSymbolDraw != null)
            {
                ((McGroupSymbolDraw)m_csDrawObject).SetGrImg(csGraphics);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトID描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObjectID(csGraphics, csCustomInfo, lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, bIDMode)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="csCustomInfo">モデル編集画面カスタマイズ情報</param>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">編集領域（幅）</param>
        /// <param name="iHeight">編集領域（高さ）</param>
        /// <param name="bIDMode">IDモード</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void DrawObjectID(Graphics csGraphics, McCustomInfo csCustomInfo, long lDispScale, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight, bool bIDMode)
        {
            if (false == DrawCheck(iX, iY, iWidth, iHeight))
            {// 描画範囲外の場合
                return;
            }

            HySString csString = (HySString)GetID().GetString();
            if (bIDMode == false)
            {// IDモードがショートモードの場合
                if (csString.ToString().IndexOf('-') != -1)
                {// 分割出来る場合
                    csString = new HySString(csString.ToString().Substring(0, csString.ToString().IndexOf('-')));
                }
                else
                {// 分割出来ない場合
                    if (csString.ToString().Length > USER_MAKE_ID_SHORT_LENGTH)
                    {// IDの長さが既定値以上の場合
                        csString = new HySString(csString.ToString().Substring(0, USER_MAKE_ID_SHORT_LENGTH));
                    }
                }
            }

            Font csFont = new Font("ＭＳ ゴシック", 9F);
            int iOffSetX = 0;
            int iOffSetY = 0;
            long lDispX = 0;
            long lDispY = 0;

            McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind();
            long lX = 0;
            long lY = 0;
            long lWidth = 0;
            long lHeight = 0;
            GetPosition(ref lX, ref lY);
            GetElementSize(ref lWidth, ref lHeight);
            double dRotate = GetRotate();
            if ((eDrawKind != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_SEND) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 端子以外の場合
                McViewTransFormer.ViewTrans(lDispScale, lX, lY, ref lDispX, ref lDispY);
            }
            else
            {// 端子の場合
                McViewTransFormer.ViewTrans(eDrawKind, lHeight, dRotate, lDispScale, lX, lY, ref lDispX, ref lDispY);
            }

            lDispX -= iImageX;
            lDispY -= iImageY;

            if (eDrawKind == McDefine.SymbolDrawKind.HYM_RECEIVE)
            {// 受信端子の場合
                if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                    (dRotate == McDefine.HYM_DRAW_ANGLE_90))
                {// 向きが上または右の場合
                    iOffSetX = 5;
                    iOffSetY = -15;
                }
                else if (dRotate == McDefine.HYM_DRAW_ANGLE_180)
                {// 向きが下の場合
                    iOffSetX = 5;
                    iOffSetY = 5;
                }
                else if (dRotate == McDefine.HYM_DRAW_ANGLE_270)
                {// 向きが左の場合
                    iOffSetX = -15;
                    iOffSetY = -15;
                }
            }
            else if (eDrawKind == McDefine.SymbolDrawKind.HYM_SEND)
            {// 送信端子の場合
                if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                    (dRotate == McDefine.HYM_DRAW_ANGLE_90))
                {// 向きが上または右の場合
                    iOffSetX = 5;
                    iOffSetY = -15;
                }
                else if (dRotate == McDefine.HYM_DRAW_ANGLE_180)
                {// 向きが下の場合
                    iOffSetX = 5;
                    iOffSetY = 5;
                }
                else if (dRotate == McDefine.HYM_DRAW_ANGLE_270)
                {// 向きが左の場合
                    iOffSetX = -15;
                    iOffSetY = -15;
                }
            }
            else if (eDrawKind == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
            {// 通常要素の場合
                iOffSetX = -csString.ToString().Length / 2 * 4 - 4;
                iOffSetY = -10;
            }
            else if ((eDrawKind == McDefine.SymbolDrawKind.HYM_GROUP) || (eDrawKind == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
            {// グループ要素の場合
                iOffSetX = -csString.ToString().Length / 2 * 4 - 4;
                iOffSetY = -10;
            }

            int iColor = csCustomInfo.GetIDDrawingColor();
            Brush csDrawBrush = new SolidBrush(ColorTranslator.FromWin32(iColor));

            csGraphics.DrawString(csString.ToString(), csFont, csDrawBrush, lDispX + iOffSetX, lDispY + iOffSetY);

            csDrawBrush.Dispose();
            csFont.Dispose();
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト名称描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObjectName(csGraphics, csCustomInfo, lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="csCustomInfo">モデル編集画面カスタマイズ情報</param>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">編集領域（幅）</param>
        /// <param name="iHeight">編集領域（高さ）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void DrawObjectName(Graphics csGraphics, McCustomInfo csCustomInfo, long lDispScale, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            if (false == DrawCheck(iX, iY, iWidth, iHeight))
            {// 描画範囲外の場合
                return;
            }

            HySString csString = new HySString();
            Font csFont = new Font("ＭＳ ゴシック", 9F);
            int iOffSetX = 0;
            int iOffSetY = 0;
            long lDispX = 0;
            long lDispY = 0;

            long lX = 0;
            long lY = 0;
            GetPosition(ref lX, ref lY);
            McViewTransFormer.ViewTrans(lDispScale, lX, lY, ref lDispX, ref lDispY);

            lDispX -= iImageX;
            lDispY -= iImageY;

            McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind();
            if (eDrawKind == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
            {// 通常要素の場合
                iOffSetX = -20;
                iOffSetY = 0;
            }
            else if (eDrawKind == McDefine.SymbolDrawKind.HYM_GROUP)
            {// 部分系グループ要素の場合
                iOffSetX = -20;
                iOffSetY = 0;
            }
            else if (eDrawKind == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
            {// 収束演算系グループ要素の場合
                iOffSetX = -20;
                iOffSetY = 0;
            }

            if (((object)GetElementName() != null) &&
                (GetElementName().GetLength() != 0))
            {// 名称設定がある場合
                csString = GetElementName();
            }

            int iColor = csCustomInfo.GetNameDrawingColor();
            Brush csDrawBrush = new SolidBrush(ColorTranslator.FromWin32(iColor));

            csGraphics.DrawString(csString.ToString(), csFont, csDrawBrush, lDispX + iOffSetX, lDispY + iOffSetY);

            csDrawBrush.Dispose();
            csFont.Dispose();
        }

        /// <summary><para>method outline:</para>
        /// <para>セレクトマーク描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SelectDraw(csGraphics, csCustomInfo, lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="csCustomInfo">モデル編集画面カスタマイズ情報</param>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">編集領域（幅）</param>
        /// <param name="iHeight">編集領域（高さ）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SelectDraw(Graphics csGraphics, McCustomInfo csCustomInfo, long lDispScale, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            if (false == DrawCheck(iX, iY, iWidth, iHeight))
            {// 描画範囲外の場合
                return;
            }

            int iElementWidth = 0;
            int iElementHeight = 0;

            double dRotate = GetRotate();
            long lWidth = 0;
            long lHeight = 0;
            GetElementSize(ref lWidth, ref lHeight);
            if ((dRotate % McDefine.HYM_DRAW_ANGLE_180) == 0.0)
            {// 向きが縦の場合
                iElementWidth = (int)lWidth;
                iElementHeight = (int)lHeight;
            }
            else
            {// 向きが横の場合
                iElementWidth = (int)lHeight;
                iElementHeight = (int)lWidth;
            }
            McDefine.SymbolDrawKind eDrawKind = GetSymbolDrawKind();
            McViewTransFormer.SizeTrans(lDispScale, eDrawKind, ref iElementWidth, ref iElementHeight);

            iElementWidth += 2;
            iElementHeight += 2;

            bool bModelFlg = false;
            if (this.GetModelInfo().GetLibraryID().ToString() != "")
            {// モデル設定ありの場合
                bModelFlg = true;
            }

            if (eDrawKind != McDefine.SymbolDrawKind.HYM_LINE)
            {// 接続線以外の場合
                long lDispX = 0;
                long lDispY = 0;

                long lX = 0;
                long lY = 0;
                GetPosition(ref lX, ref lY);
                if ((eDrawKind == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                    (eDrawKind == McDefine.SymbolDrawKind.HYM_SEND) ||
                    (eDrawKind == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (eDrawKind == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 端子の場合
                    McViewTransFormer.ViewTrans(eDrawKind, lHeight, dRotate, lDispScale, lX, lY, ref lDispX, ref lDispY);

                    lDispX -= iImageX;
                    lDispY -= iImageY;

                    int iColor = csCustomInfo.GetSelectDrawingColor(true);
                    Pen csDrawPen = new Pen(ColorTranslator.FromWin32(iColor), 1.0F);

                    csGraphics.DrawRectangle(csDrawPen, lDispX - iElementWidth / 2, lDispY - iElementHeight / 2, iElementWidth / 2 * 2, iElementHeight / 2 * 2);

                    csDrawPen.Dispose();
                }
                else
                {// 端子以外の場合
                    McViewTransFormer.ViewTrans(lDispScale, lX, lY, ref lDispX, ref lDispY);

                    lDispX -= iImageX;
                    lDispY -= iImageY;

                    int iColor = csCustomInfo.GetSelectDrawingColor(bModelFlg);
                    Brush csDrawBrush = new SolidBrush(ColorTranslator.FromWin32(iColor));
                    Pen csDrawPen = new Pen(ColorTranslator.FromWin32(iColor), 1.0F);

                    csGraphics.DrawRectangle(csDrawPen, lDispX - iElementWidth / 2, lDispY - iElementHeight / 2, iElementWidth / 2 * 2, iElementHeight / 2 * 2);

                    int ResizePointSize = csCustomInfo.GetResizePointSize();
                    HySObjectKind csDrawSubKind = GetDrawSubKind();
                    if ((eDrawKind != McDefine.SymbolDrawKind.HYM_STD_ELEMENT) ||
                        (csDrawSubKind != McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE))
                    {// 通常要素以外またはマージ要素以外の場合
                        csGraphics.FillRectangle(csDrawBrush,
                                                 lDispX - iElementWidth / 2 - ResizePointSize / 2,
                                                 lDispY - iElementHeight / 2 - ResizePointSize / 2,
                                                 ResizePointSize,
                                                 ResizePointSize);
                        csGraphics.FillRectangle(csDrawBrush,
                                                 lDispX - iElementWidth / 2 - ResizePointSize / 2,
                                                 lDispY + iElementHeight / 2 - ResizePointSize / 2,
                                                 ResizePointSize,
                                                 ResizePointSize);
                        csGraphics.FillRectangle(csDrawBrush,
                                                 lDispX + iElementWidth / 2 - ResizePointSize / 2,
                                                 lDispY - iElementHeight / 2 - ResizePointSize / 2,
                                                 ResizePointSize,
                                                 ResizePointSize);
                        csGraphics.FillRectangle(csDrawBrush,
                                                 lDispX + iElementWidth / 2 - ResizePointSize / 2,
                                                 lDispY + iElementHeight / 2 - ResizePointSize / 2,
                                                 ResizePointSize,
                                                 ResizePointSize);
                    }

                    csDrawBrush.Dispose();
                    csDrawPen.Dispose();
                }
            }
        }

        //======================================
        // 端子関連メソッド
        //======================================

        /// <summary><para>method outline:</para>
        /// <para>描画要素端子数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lCount = GetPortCount()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long  描画要素端子数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetPortCount()
        {
            return m_csPortList.Count;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素端子追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRet = AddPort( csPort )</para>
        /// </example>
        /// <param name="csPort">描画要素端子</param>
        /// <returns>long  0:成功，-1:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long AddPort(McGraphicsElement csPort)
        {
            if (csPort == null)
            {// パラメータ異常
                return -1L;
            }

            csPort.SetParentElm(this);

            m_csPortList.Add(csPort);

            return 0L;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素端子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csPort = GetPort(lPos)</para>
        /// </example>
        /// <param name="lPos">位置(0相対)</param>
        /// <returns>McGraphicsElement  描画要素端子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement GetPort(long lPos)
        {
            McGraphicsElement csPort = null;  // 画面表示端子

            if (lPos >= 0 && lPos < GetPortCount())
            {// 範囲内
                csPort = m_csPortList[(int)lPos];
            }

            return csPort;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素端子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csPort = GetPort(csPortID)</para>
        /// </example>
        /// <param name="csPortID">描画要素端子ID</param>
        /// <returns>McGraphicsElement  描画要素端子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement GetPort(HySIdentifier csPortID)
        {
            McGraphicsElement csPort = null;  // 画面表示端子

            for (int iLP = 0; iLP < m_csPortList.Count; iLP++)
            {// 端子数ループ
                if (m_csPortList[iLP].GetID().Equals(csPortID) == true)
                {// 取得指定端子の場合
                    csPort = m_csPortList[iLP];
                    break;
                }
            }

            return csPort;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素端子削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DelPort(csPortID)</para>
        /// </example>
        /// <param name="csPortID">描画要素端子ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DelPort(HySIdentifier csPortID)
        {
            for (int iLP = 0; iLP < m_csPortList.Count; iLP++)
            {// 端子数ループ
                if (m_csPortList[iLP].GetID().Equals(csPortID) == true)
                {// 削除指定端子の場合
                    m_csPortList.RemoveAt(iLP);
                    break;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素端子削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DelPort(lPos)</para>
        /// </example>
        /// <param name="lPos">位置(0相対)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DelPort(long lPos)
        {
            if (lPos >= 0 && lPos < GetPortCount())
            {// 範囲内
                m_csPortList.RemoveAt((int)lPos);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>端子接続ID設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetPortID( csFromID, csToID )</para>
        /// </example>
        /// <param name="csFromID">接続元ID</param>
        /// <param name="csToID">接続先ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetPortID(HySIdentifier csFromID, HySIdentifier csToID)
        {
            if (m_csViewerPort != null)
            {// 端子の場合
                m_csViewerPort.SetPortID(csFromID, csToID);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>端子接続ID設定取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetPortID(ref csFromID, ref csToID)</para>
        /// </example>
        /// <param name="csFromID">接続元ID</param>
        /// <param name="csToID">接続先ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void GetPortID(ref HySIdentifier csFromID, ref HySIdentifier csToID)
        {
            csFromID = new HySID("");
            csToID = new HySID("");
            if (m_csViewerPort != null)
            {// 端子の場合
                m_csViewerPort.GetPortID(ref csFromID, ref csToID);
            }
            if (m_csViewerConnect != null)
            {// 要素の場合
                m_csViewerConnect.GetConnectionPort(ref csFromID, ref csToID);
            }
        }

        //======================================
        // 内部エレメント関連メソッド
        //======================================
        /// <summary>
        /// <para>method outline:</para>
        /// <para>伝送情報デフォルトパターン設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetDefaultPattern( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDefaultPattern( )
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.SetDefaultPattern( );
            }
            if (m_csViewerConnect != null)
            {// 要素の場合
                m_csViewerConnect.SetDefaultPattern(null);
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>伝送情報デフォルトパターン設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetDefaultPattern(ref csElmIDList, ref csTranList, ref csModTranList)</para>
        /// </example>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csTranList">設定前モデル情報リスト</param>
        /// <param name="csModTranList">設定後モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDefaultPattern(ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csTranList, ref List<McModelInfo> csModTranList)
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.SetDefaultPattern(ref csElmIDList, ref csTranList, ref csModTranList);
            }
            if (m_csViewerConnect != null)
            {// 要素の場合
                m_csViewerConnect.SetDefaultPattern(null);
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデル個別情報設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetModelInfo(csModelInfo)</para>
        /// </example>
        /// <param name="csModelInfo">モデル個別情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetModelInfo(McModelInfo csModelInfo)
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.SetModelInfo(csModelInfo);
            }
            if (m_csViewerConnect != null)
            {// 要素の場合
                m_csViewerConnect.SetConnectionTranInfo(csModelInfo, null);
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデル個別情報解除</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>CancelModelInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void CancelModelInfo()
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                m_csViewerElm.CancelModelInfo();
            }
            if (m_csViewerConnect != null)
            {// 要素の場合
                m_csViewerConnect.CancelTranInfo();
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>モデル個別情報取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McModelInfo csModelInfo = GetModelInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McModelInfo　モデル個別情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McModelInfo GetModelInfo()
        {
            if (m_csViewerElm != null)
            {// 要素の場合
                return m_csViewerElm.GetModelInfo();
            }
            else if (m_csViewerConnect != null)
            {// 接続線の場合
                return m_csViewerConnect.GetTranInfo();
            }
            else
            {// その他の場合
                return m_csDmyInfo;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>描画オブジェクト区分取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySObjectKind csObjectKind = GetDrawSubKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySObjectKind　描画オブジェクト区分</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySObjectKind GetDrawSubKind()
        {
            HySObjectKind csDrawSubKind = new HySObjectKind();
            if (m_csViewerElm != null)
            {// 要素の場合
                csDrawSubKind = m_csViewerElm.GetDrawSubKind();
            }
            return csDrawSubKind;
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルト名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySString csDefaultName = GetDefaultName()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString　デフォルト名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySString GetDefaultName()
        {
            return m_csDrawObject.DefaultName;
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルト名称2取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySString csDefaultName = GetDefaultName2()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString　デフォルト名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySString GetDefaultName2()
        {
            return ((McGroupSymbolDraw)m_csDrawObject).DefaultName2;
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>要素情報検索</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McGraphicsElement csElement = SearchElement(csElementID)</para>
        /// </example>
        /// <param name="csElementID">検索要素ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McGraphicsElement SearchElement(HySIdentifier csElementID)
        {
            McGraphicsElement csElement = null;

            if (GetID().Equals(csElementID) == true)
            {// 自身が検索要素の場合
                csElement = this;
            }

            if (csElement == null)
            {// 自身が検索要素ではない場合
                for (int iLp = 0; iLp < m_csPortList.Count; iLp++)
                {// 端子数ループ
                    if (m_csPortList[iLp].GetID().Equals(csElementID) == true)
                    {// 検索要素の場合
                        csElement = m_csPortList[iLp];
                        break;
                    }
                }
            }

            return csElement;
        }


        // ================================================
        //   Undo/Redo用処理
        // ================================================
        /// <summary><para>method outline:</para>
        /// <para>情報のコピー</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = CopyInfUsedByUndoRedoOnly(csCopyElement)</para>
        /// </example>
        /// <param name="csCopyElement">複製要素</param>
        /// <returns>McGraphicsElement 複製要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual McGraphicsElement CopyInfUsedByUndoRedoOnly(McGraphicsElement csCopyElement)
        {
            if (csCopyElement == null)
            {// 複製要素がない場合
                csCopyElement = new McGraphicsElement(GetSymbolDrawKind(), GetDrawSubKind());
            }

            if (m_csViewerElm != null)
            {// 要素の場合
                csCopyElement.m_csViewerElm = m_csViewerElm.CopyInfUsedByUndoRedoOnly(null);
            }
            if (m_csViewerPort != null)
            {// 端子の場合
                csCopyElement.m_csViewerPort = m_csViewerPort.CopyInfUsedByUndoRedoOnly(null);
            }
            if (m_csViewerConnect != null)
            {// 接続線の場合
                csCopyElement.m_csViewerConnect = m_csViewerConnect.CopyInfUsedByUndoRedoOnly(null);
            }

            csCopyElement.m_csParentElm = m_csParentElm;
            csCopyElement.m_csElementName = m_csElementName;

            for (int iLp = 0; iLp < m_csPortList.Count; iLp++)
            {// 端子数ループ
                csCopyElement.m_csPortList.Add(m_csPortList[iLp].CopyInfUsedByUndoRedoOnly(null));
            }

            return csCopyElement;
        }
    }
}
