﻿// <summary>ソースコード：ＨＹＭＣＯグラフックオブジェクト管理</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;

using CommonMP.HYSSOP.Interface;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Data.StructInfo;
using CommonMP.HYMCO.CoreImpl.Model;

namespace CommonMP.HYMCO.CoreOptions.McViewer.DotNetViewer
{
    /// <summary><para>class outline:</para>
    /// <para>グラフックオブジェクト管理クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/11/17][新規作成]</para>
    /// </remarks>
    /// <remarks><para>remarks:</para>
    /// <para>本クラスはバージョンアップ互換性の対象外</para>
    /// </remarks>
    [Serializable]
    public class McGraphicsManager : McGraphicsRoot
    {
        /// <summary>要素間接続情報</summary>
        private McStructInfo m_csMcStructInfo;

        /// <summary>描画要素リスト</summary>
        protected List<McGraphicsElement> m_csElmList;

        /// <summary>描画要素(接続線)リスト</summary>
        protected List<McGraphicsElement> m_csLineList;

        /// <summary>選択中要素リスト</summary>
        protected List<McGraphicsElement> m_csSelectionList;

        /// <summary>選択中要素別カウンター</summary>
        private int[] m_iElementCounter ={ 0, 0, 0, 0, 0, 0 };

        /// <summary>画面表示拡大率</summary>
        protected long m_lDispScale;

        /// <summary>画面描画親要素</summary>
        protected McGraphicsElement m_csPossessionElm;

        /// <summary>ID表示モード</summary>
        private bool m_bIDMode = false;

        /// <summary>モデル編集画面カスタマイズ情報格納クラス</summary>
        private McCustomInfo m_csCustomInfo;

        /// <summary>出力要素名称用カウンター</summary>
        private int m_iOutputCounter = 1;

        /// <summary><para>method outline:</para>
        /// <para>描画要素生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = CreateGraphic(eKind, csDrawSubKind)</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <param name="csDrawSubKind">シンボル描画サブ種別</param>
        /// <returns>McGraphicsElement !=null:成功，null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private McGraphicsElement CreateGraphic(McDefine.SymbolDrawKind eKind, HySObjectKind csDrawSubKind)
        {
            McGraphicsElement csNewElm = null; // 描画要素

            if (eKind == McDefine.SymbolDrawKind.HYM_LINE)
            {// 接続線
                csNewElm = new McGraphicsLineElement();
            }
            else if (eKind == McDefine.SymbolDrawKind.HYM_TRANS_IN)
            {// 入力中継端子
                csNewElm = new McGraphicsTransElement(eKind);
            }
            else if (eKind == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
            {// 出力中継端子
                csNewElm = new McGraphicsTransElement(eKind);
            }
            else if (eKind == McDefine.SymbolDrawKind.HYM_GROUP)
            {// グループ
                csNewElm = new McGraphicsGroupElement(eKind);
            }
            else if (eKind == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
            {// グループ
                csNewElm = new McGraphicsGroupElement(eKind);
            }
            else
            {// 上記以外
                csNewElm = new McGraphicsElement(eKind, csDrawSubKind);
            }

            return csNewElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>要素復元</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = DecodeGraphic(csViewElm,ref csPortList)</para>
        /// </example>
        /// <param name="csViewElm">画面表示要素</param>
        /// <param name="csPortList">描画要素端子リスト</param>
        /// <returns>McGraphicsElement !=null:成功，null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private McGraphicsElement DecodeGraphic(McStructElement csViewElm, ref List<McGraphicsElement> csPortList)
        {
            McGraphicsElement csNewElm;    // 画面表示要素
            McGraphicsElement csNewPort;
            long lLp;
            McStructPort csViewPort;
            HySIdentifier csFromID = null;
            HySIdentifier csToID = null;
            McDefine.SymbolDrawKind eKind;
            McStructElement csElm;
            McStructConnection csViewConnect;
            HySIdentifier csNullID = new HySID("");
            McGraphicsElement csNewLine;
            HySObjectKind csDrawSubKind;
            bool bFind = false;

            eKind = csViewElm.GetSymbolDrawKind();
            csDrawSubKind = csViewElm.GetDrawSubKind();

            csNewElm = CreateGraphic(eKind, csDrawSubKind);
            csNewElm.SetViewerElm(csViewElm);

            // 描画要素が保持しているグループ要素の復元
            if ((csNewElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                (csNewElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
            {// グループ要素の場合
                for (lLp = 0L; lLp < ((McStructGroupElement)csViewElm).GetElmCount(); lLp++)
                {// グループ内要素数ループ
                    csElm = ((McStructGroupElement)csViewElm).GetElement(lLp);
                    ((McGraphicsGroupElement)csNewElm).AddElement(DecodeGraphic(csElm, ref csPortList));
                }
            }

            // 描画要素が保持している描画端子の復元
            for (lLp = 0L; lLp < csViewElm.GetPortCount(); lLp++)
            {// 端子数ループ
                csViewPort = csViewElm.GetPort(lLp);
                eKind = csViewPort.GetSymbolDrawKind();

                csNewPort = CreateGraphic(eKind, new HySObjectKind());
                csNewPort.SetViewerPort(csViewPort);
                csNewPort.SetParentElm(csNewElm);

                csNewElm.AddPort(csNewPort);

                // 描画要素(接続線)の復元の為のワーク
                csPortList.Add(csNewPort);
            }

            // 描画要素が保持しているグループ要素の復元
            if ((csNewElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                (csNewElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
            {// グループ要素の場合
                // 描画要素(接続線)の復元
                for (lLp = 0L; lLp < ((McStructGroupElement)csViewElm).GetConnectionCount(); lLp++)
                {// 接続線数ループ
                    bFind = false;
                    csViewConnect = ((McStructGroupElement)csViewElm).GetConnection(lLp);
                    csViewConnect.GetConnectionPort(ref csFromID, ref csToID);

                    csNewLine = CreateGraphic(McDefine.SymbolDrawKind.HYM_LINE, new HySObjectKind());
                    csNewLine.SetViewerConnect(csViewConnect);

                    for (int iLp = 0; iLp < csPortList.Count; iLp++)
                    {// ワークリスト内を検索

                        if (bFind == false && csPortList[iLp].Equal(csFromID) != false)
                        {// 接続元IDが一致
                            csNewLine.AddPort(csPortList[iLp]);
                            iLp = -1;
                            bFind = true;
                            continue;
                        }
                        if (bFind == true && csPortList[iLp].Equal(csToID) != false)
                        {// 接続先IDが一致
                            csNewLine.AddPort(csPortList[iLp]);
                            break;
                        }
                    }
                    csNewLine.SetElementName(csViewConnect.GetConnectionName());
                    ((McGraphicsGroupElement)csNewElm).AddLineElement(csNewLine);
                }
            }

            return csNewElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>要素復元</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DecodeViewerToGraphic(csStruct, csElmList, csLineList)</para>
        /// </example>
        /// <param name="csStruct">要素間接続情報</param>
        /// <param name="csElmList">要素リスト</param>
        /// <param name="csLineList">接続線リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void DecodeViewerToGraphic(McStructInfo csStruct, List<McGraphicsElement> csElmList, List<McGraphicsElement> csLineList)
        {
            long lLp;
            int iLp;
            bool bFind = false;
            McStructElement csViewElm;
            McStructConnection csViewConnect;
            HySIdentifier csFromID = null;
            HySIdentifier csToID = null;
            McGraphicsElement csNewElm;
            McGraphicsElement csNewLine;
            List<McGraphicsElement> csPortList;

            csPortList = new List<McGraphicsElement>();

            // 描画要素の復元
            for (lLp = 0L; lLp < csStruct.GetElmCount(); lLp++)
            {// 要素数ループ
                csViewElm = csStruct.GetElement(lLp);

                csNewElm = DecodeGraphic(csViewElm, ref csPortList);

                csNewElm.SetParentElm(null);

                csElmList.Add(csNewElm);
            }

            // 描画要素(接続線)の復元
            for (lLp = 0L; lLp < csStruct.GetConnectionCount(); lLp++)
            {// 接続線数ループ
                bFind = false;
                csViewConnect = csStruct.GetConnection(lLp);
                csViewConnect.GetConnectionPort(ref csFromID, ref csToID);

                csNewLine = CreateGraphic(McDefine.SymbolDrawKind.HYM_LINE, new HySObjectKind());
                csNewLine.SetViewerConnect(csViewConnect);

                for (iLp = 0; iLp < csPortList.Count; iLp++)
                {// ワークリスト内を検索

                    if (bFind == false && csPortList[iLp].Equal(csFromID) != false)
                    {// 接続元IDが一致
                        csNewLine.AddPort(csPortList[iLp]);
                        iLp = -1;
                        bFind = true;
                        continue;
                    }
                    if (bFind == true && csPortList[iLp].Equal(csToID) != false)
                    {// 接続先IDが一致
                        csNewLine.AddPort(csPortList[iLp]);
                        break;
                    }
                }
                csNewLine.SetElementName(csViewConnect.GetConnectionName());
                csLineList.Add(csNewLine);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>要素間接続情報セット</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetStructInfo( csStruct )</para>
        /// </example>
        /// <param name="csStruct">要素間接続情報</param>
        /// <returns>long 0:成功，-1:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetStructInfo(McStructInfo csStruct)
        {
            m_csElmList.Clear();
            m_csLineList.Clear();

            m_csMcStructInfo = csStruct;

            DecodeViewerToGraphic(csStruct, m_csElmList, m_csLineList);

            SetAllDecodePortPosition(m_csElmList, m_csLineList);
        }

        /// <summary><para>method outline:</para>
        /// <para>全端子配置位置設定（要素復元）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetAllDecodePortPosition(csElmList, csLineList)</para>
        /// </example>
        /// <param name="csElmList">要素リスト</param>
        /// <param name="csLineList">接続線リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetAllDecodePortPosition(List<McGraphicsElement> csElmList, List<McGraphicsElement> csLineList)
        {
            long lX = 0L;
            long lY = 0L;
            HySIdentifier csFromID = null;
            HySIdentifier csToID = null;
            McGraphicsElement csFromElement = null;
            McGraphicsElement csToElement = null;
            McGraphicsElement csFromPort = null;
            McGraphicsElement csToPort = null;

            for (int iLp = 0; iLp < csLineList.Count; iLp++)
            {// 接続線数ループ
                csFromElement = null;
                csToElement = null;
                csLineList[iLp].GetPortID(ref csFromID, ref csToID);
                csFromPort = csLineList[iLp].GetPort(csFromID);
                csToPort = csLineList[iLp].GetPort(csToID);
                csFromElement = csFromPort.GetParentElm();
                csToElement = csToPort.GetParentElm();

                csFromPort.GetPosition(ref lX, ref lY);
                if ((lX == 0) || (lY == 0))
                {// 座標が０の場合
                    if (csFromPort.GetRotate() != McDefine.HYM_DRAW_ANGLE_M90)
                    {// 向きが－９０の場合
                        ReSetPortPosition(csFromElement, csFromPort);
                    }
                    else
                    {// 向きが－９０以外の場合
                        SetPortPosition(csFromElement, csFromPort, csToElement, csToPort);
                    }
                }
                csToPort.GetPosition(ref lX, ref lY);
                if ((lX == 0) || (lY == 0))
                {// 座標が０の場合
                    if (csToPort.GetRotate() != McDefine.HYM_DRAW_ANGLE_M90)
                    {// 向きが－９０の場合
                        ReSetPortPosition(csToElement, csToPort);
                    }
                    else
                    {// 向きが－９０以外の場合
                        SetPortPosition(csFromElement, csFromPort, csToElement, csToPort);
                    }
                }
            }

            for (int iLp = 0; iLp < csElmList.Count; iLp++)
            {// 要素数ループ
                if ((csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                    (csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                {// グループ要素の場合
                    SetDecodePortPosition((McGraphicsGroupElement)csElmList[iLp]);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>端子配置位置設定（要素復元）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDecodePortPosition(csElement)</para>
        /// </example>
        /// <param name="csElement">端子配置位置設定親要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetDecodePortPosition(McGraphicsGroupElement csElement)
        {
            long lX = 0L;
            long lY = 0L;
            long lInnerX = 0L;
            long lInnerY = 0L;
            HySIdentifier csFromID = null;
            HySIdentifier csToID = null;
            McGraphicsElement csFromElement = null;
            McGraphicsElement csToElement = null;
            McGraphicsElement csFromPort = null;
            McGraphicsElement csToPort = null;

            for (int iLp = 0; iLp < csElement.GetLineElmCount(); iLp++)
            {// 接続線数ループ
                csFromElement = null;
                csToElement = null;
                csElement.GetLineElement(iLp).GetPortID(ref csFromID, ref csToID);
                csFromPort = csElement.GetLineElement(iLp).GetPort(csFromID);
                csToPort = csElement.GetLineElement(iLp).GetPort(csToID);
                csFromElement = csFromPort.GetParentElm();
                csToElement = csToPort.GetParentElm();

                if (csFromPort.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN)
                {// 流入中継端子以外の場合
                    csFromPort.GetPosition(ref lX, ref lY);
                    if ((lX == 0) || (lY == 0))
                    {// 座標が０の場合
                        if (csFromPort.GetRotate() != McDefine.HYM_DRAW_ANGLE_M90)
                        {// 向きが－９０の場合
                            ReSetPortPosition(csFromElement, csFromPort);
                        }
                        else
                        {// 向きが－９０以外の場合
                            SetPortPosition(csFromElement, csFromPort, csToElement, csToPort);
                        }
                    }
                }
                else
                {
                    ((McGraphicsTransElement)csFromPort).GetInnerPosition(ref lInnerX, ref lInnerY);
                    if ((lInnerX == 0) || (lInnerY == 0))
                    {// 座標が０の場合
                        //中継端子位置設定
                        ReSetGroupMemberPosition(csElement, csFromPort);
                    }
                }
                if (csToPort.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT)
                {
                    csToPort.GetPosition(ref lX, ref lY);
                    if ((lX == 0) || (lY == 0))
                    {// 座標が０の場合
                        if (csToPort.GetRotate() != McDefine.HYM_DRAW_ANGLE_M90)
                        {// 向きが－９０の場合
                            ReSetPortPosition(csToElement, csToPort);
                        }
                        else
                        {// 向きが－９０以外の場合
                            SetPortPosition(csFromElement, csFromPort, csToElement, csToPort);
                        }
                    }
                }
                else
                {
                    ((McGraphicsTransElement)csToPort).GetInnerPosition(ref lInnerX, ref lInnerY);
                    if ((lInnerX == 0) || (lInnerY == 0))
                    {// 座標が０の場合
                        //中継端子位置設定
                        ReSetGroupMemberPosition(csElement, csToPort);
                    }
                }
            }

            for (int iLp = 0; iLp < csElement.GetElmCount(); iLp++)
            {// グループ内要素数ループ
                if ((csElement.GetElement(iLp).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                    (csElement.GetElement(iLp).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                {// グループ要素の場合
                    SetDecodePortPosition((McGraphicsGroupElement)csElement.GetElement(iLp));
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>要素間接続情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McStructInfo csInfo = GetStructInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McStructInfo　要素間接続情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McStructInfo GetStructInfo()
        {
            return m_csMcStructInfo;
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsManager csManager = new McGraphicsManager()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsManager()
        {
            m_csElmList = new List<McGraphicsElement>();
            m_csLineList = new List<McGraphicsElement>();
            m_csSelectionList = new List<McGraphicsElement>();
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = AddElement(eKind, csDrawSubKind)</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <param name="csDrawSubKind">シンボル描画サブ種別</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement AddElement(McDefine.SymbolDrawKind eKind, HySObjectKind csDrawSubKind)
        {
            McGraphicsElement csNewElm;     // 描画要素
            McStructElement csElm = null;   // 画面要素

            csNewElm = CreateGraphic(eKind, csDrawSubKind);

            if (m_csPossessionElm == null)
            {// 全体系の場合
                csElm = m_csMcStructInfo.AddElement(eKind, csDrawSubKind);
            }
            else
            {// グループ内の場合
                csElm = ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).AddElement(eKind, csDrawSubKind);
            }

            if (csElm != null)
            {// 追加成功
                csNewElm.SetViewerElm(csElm);
                csNewElm.SetParentElm(m_csPossessionElm);
                m_csElmList.Add(csNewElm);
            }
            else
            {// 追加失敗
                csNewElm = null;
            }

            return csNewElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = AddElement(csGroupElement, eKind, csDrawSubKind)</para>
        /// </example>
        /// <param name="csGroupElement">要素追加グループ</param>
        /// <param name="eKind">シンボル描画種別</param>
        /// <param name="csDrawSubKind">シンボル描画サブ種別</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement AddElement(McGraphicsGroupElement csGroupElement, McDefine.SymbolDrawKind eKind, HySObjectKind csDrawSubKind)
        {
            McGraphicsElement csNewElm;     // 描画要素
            McStructElement csElm = null;   // 画面要素

            // 要素生成
            csNewElm = CreateGraphic(eKind, csDrawSubKind);

            // 生成要素追加
            csElm = ((McStructGroupElement)csGroupElement.GetViewerElm()).AddElement(eKind, csDrawSubKind);

            if (csElm != null)
            {// 追加成功
                csNewElm.SetViewerElm(csElm);
                csGroupElement.AddElement(csNewElm);
            }
            else
            {// 追加失敗
                csNewElm = null;
            }

            return csNewElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素中継端子追加(内側)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = AddRelayPort( eKind, csElement )</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <param name="csElement">端子追加要素</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement AddRelayPort(McDefine.SymbolDrawKind eKind, McGraphicsElement csElement)
        {
            McGraphicsElement csNewRelayPort = null;         // 描画要素端子

            if ((eKind == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                (eKind == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 中継端子の場合
                // 中継端子追加
                McStructTransPort csNewStructTransPort = ((McStructGroupElement)csElement.GetViewerElm()).AddRelayPort(eKind, null);

                if (csNewStructTransPort != null)
                {// 追加成功
                    csNewRelayPort = AddPort(eKind, csElement);
                    csNewRelayPort.SetViewerPort(csNewStructTransPort);
                    csNewRelayPort.SetRotate(McDefine.HYM_DRAW_ANGLE_0);
                    ((McGraphicsTransElement)csNewRelayPort).SetInnerPosition(50, 50);
                }
                else
                {// 追加失敗
                    csNewRelayPort = null;
                }
            }

            return csNewRelayPort;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素中継端子追加(内側)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = AddRelayPort( eKind )</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement AddRelayPort(McDefine.SymbolDrawKind eKind)
        {
            McGraphicsElement csNewRelayPort = null;         // 描画要素端子

            if (m_csSelectionList.Count > 0)
            {// 選択要素が０より多い場合
                if ((m_csSelectionList[0].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE) ||
                    (m_csSelectionList[0].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                    (m_csSelectionList[0].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_SEND) ||
                    (m_csSelectionList[0].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[0].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 接続線・端子・中継端子の場合
                    return csNewRelayPort;
                }

                if (m_csPossessionElm != null)
                {// グループ内の場合
                    if ((eKind == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                        (eKind == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 中継端子の場合
                        // 中継端子追加
                        McStructTransPort csNewStructTransPort = ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).AddRelayPort(eKind, null);

                        if (csNewStructTransPort != null)
                        {// 追加成功
                            csNewRelayPort = AddPort(eKind, m_csPossessionElm);
                            csNewRelayPort.SetViewerPort(csNewStructTransPort);
                            csNewRelayPort.SetRotate(McDefine.HYM_DRAW_ANGLE_0);
                            ReSetGroupMemberPosition((McGraphicsGroupElement)m_csPossessionElm, csNewRelayPort);
                            // 接続線生成
                            if (eKind == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                            {// 流入中継端子の場合
                                CreateNewConnection(csNewRelayPort.GetID(), m_csSelectionList[0].GetID());
                            }
                            else
                            {// 流入中継端子以外の場合
                                CreateNewConnection(m_csSelectionList[0].GetID(), csNewRelayPort.GetID());
                            }
                        }
                        else
                        {// 追加失敗
                            csNewRelayPort = null;
                        }
                    }
                }
            }

            return csNewRelayPort;
        }

        /// <summary><para>method outline:</para>
        /// <para>描画要素端子追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = AddPort( eKind, csElement )</para>
        /// </example>
        /// <param name="eKind">シンボル描画種別</param>
        /// <param name="csElement">端子追加要素</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private McGraphicsElement AddPort(McDefine.SymbolDrawKind eKind, McGraphicsElement csElement)
        {
            McGraphicsElement csNewPort = null;    // 描画要素端子
            long lRet = -1L;                        // 復帰情報

            if (csElement == null)
            {// パラメータ異常
                return csNewPort;
            }

            // 端子生成
            csNewPort = CreateGraphic(eKind, new HySObjectKind());
            csNewPort.SetParentElm(csElement);
            lRet = csElement.AddPort(csNewPort);

            if (lRet != 0L)
            {// 追加失敗
                csNewPort = null;
            }

            return csNewPort;
        }

        /// <summary><para>method outline:</para>
        /// <para>接続生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = CreateNewConnection( csFromID, csToID )</para>
        /// </example>
        /// <param name="csFromID">接続元端子ID</param>
        /// <param name="csToID">接続先端子ID</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>新規に接続生成する場合のみ使用可能</para>
        /// </remarks>
        public McGraphicsElement CreateNewConnection(HySIdentifier csFromID, HySIdentifier csToID)
        {
            McGraphicsElement csFromPort = null;    // 接続元端子
            McGraphicsElement csToPort = null;      // 接続先端子
            McGraphicsElement csFromElm = null;    // 接続元親要素
            McGraphicsElement csToElm = null;      // 接続先親要素
            McGraphicsElement csNewLine = null;     // 接続線

            // 接続チェックおよび各種インスタンス取得
            if (ConnectionElementCheck(csFromID, csToID, ref csFromPort, ref csToPort, ref csFromElm, ref csToElm) != true)
            {// 接続不可の場合
                return csNewLine;
            }

            // 接続線生成
            csNewLine = CreateNewConnection(csFromPort, csToPort);

            //ポート配置位置決定
            SetPortPosition(csFromElm, csFromPort, csToElm, csToPort);

            if ((csFromPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                (csFromPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 中継端子の場合
                //グループ内要素座標再セット
                ReSetGroupMemberPosition((McGraphicsGroupElement)csFromElm, csFromPort);
            }
            if ((csToPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                (csToPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 中継端子の場合
                //グループ内要素座標再セット
                ReSetGroupMemberPosition((McGraphicsGroupElement)csToElm, csToPort);
            }

            return csNewLine;
        }

        /// <summary><para>method outline:</para>
        /// <para>接続生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = CreateNewConnection( csFromPort, csToPort )</para>
        /// </example>
        /// <param name="csFromPort">接続元端子</param>
        /// <param name="csToPort">接続先端子</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>新規に接続生成する場合のみ使用可能</para>
        /// </remarks>
        private McGraphicsElement CreateNewConnection(McGraphicsElement csFromPort, McGraphicsElement csToPort)
        {
            McGraphicsElement csNewLine = null;     // 描画要素(接続線)
            HySIdentifier csFromID;                 // 接続元ID
            HySIdentifier csToID;                   // 接続先ID
            HySIdentifier csStructFromID = null;    // 接続元ID
            HySIdentifier csStructToID = null;      // 接続先ID
            McStructConnection csConnect = null;    // 画面要素(接続線)

            if (csFromPort == null || csToPort == null)
            {// パラメータ異常
                return null;
            }

            csFromID = csFromPort.GetID();
            csToID = csToPort.GetID();

            if (m_csPossessionElm == null)
            {// 全体系の場合
                csConnect = m_csMcStructInfo.AddInternalConnection(new HySID(""), csFromPort.GetParentElm().GetID(), csFromID, csToPort.GetParentElm().GetID(), csToID);
            }
            else
            {// グループ内の場合
                if ((csFromPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)&&
                    (csToPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 両方が中継端子の場合
                    //グループ内部中継端子直結　未作成
                }
                else
                {// どちらかまたは両方が中継端子ではない場合
                    if ((csFromPort.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                        (csToPort.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 両方が中継端子ではない場合
                        csConnect = ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).AddInternalConnection(new HySID(""), csFromPort.GetParentElm().GetID(), csFromID, csToPort.GetParentElm().GetID(), csToID);
                    }
                    else
                    {// どちらかが中継端子ではない場合
                        if (csFromPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                        {// 流入中継端子の場合
                            csConnect = ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).AddExternalINConnection(new HySID(""), csFromID, csToPort.GetParentElm().GetID(), csToID);
                        }
                        else if (csToPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
                        {// 流出中継端子の場合
                            csConnect = ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).AddExternalOUTConnection(new HySID(""), csFromPort.GetParentElm().GetID(), csFromID, csToID);
                        }
                    }
                }
            }

            if (csConnect != null)
            {// 成功
                csConnect.GetConnectionPort(ref csStructFromID, ref csStructToID);

                csFromPort.SetViewerPort(csFromPort.GetParentElm().GetViewerElm().GetPort(csStructFromID));
                csToPort.SetViewerPort(csToPort.GetParentElm().GetViewerElm().GetPort(csStructToID));

                // 接続線の生成
                csNewLine = CreateGraphic(McDefine.SymbolDrawKind.HYM_LINE, new HySObjectKind());
                csNewLine.SetViewerConnect(csConnect);
                csNewLine.AddPort(csFromPort);
                csNewLine.AddPort(csToPort);

                csNewLine.SetParentElm(m_csPossessionElm);

                m_csLineList.Add(csNewLine);
            }
            else
            {// 失敗
                csNewLine = null;
            }

            return csNewLine;
        }

        /// <summary><para>method outline:</para>
        /// <para>接続生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = CreateNewConnectionGroup( csGroupElement, csFromPort, csToPort )</para>
        /// </example>
        /// <param name="csGroupElement">グループ要素</param>
        /// <param name="csFromPort">接続元端子</param>
        /// <param name="csToPort">接続先端子</param>
        /// <returns>McGraphicsElement !=null:成功，==null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>新規に接続生成する場合のみ使用可能</para>
        /// </remarks>
        private McGraphicsElement CreateNewConnectionGroup(McGraphicsGroupElement csGroupElement, McGraphicsElement csFromPort, McGraphicsElement csToPort)
        {
            McGraphicsElement csNewLine = null;     // 描画要素(接続線)
            HySIdentifier csFromID;                 // 接続元ID
            HySIdentifier csToID;                   // 接続先ID
            HySIdentifier csStructFromID = null;    // 接続元ID
            HySIdentifier csStructToID = null;      // 接続先ID
            McStructConnection csConnect = null;    // 画面要素(接続線)

            if (csGroupElement == null || csFromPort == null || csToPort == null)
            {// パラメータ異常
                return null;
            }

            csFromID = csFromPort.GetID();
            csToID = csToPort.GetID();

            if ((csFromPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                (csToPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 両方が中継端子の場合
                //グループ内部中継端子直結　未作成
            }
            else
            {// どちらかまたは両方が中継端子ではない場合
                if ((csFromPort.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                    (csToPort.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 両方が中継端子ではない場合
                    csConnect = ((McStructGroupElement)csGroupElement.GetViewerElm()).AddInternalConnection(new HySID(""), csFromPort.GetParentElm().GetID(), csFromID, csToPort.GetParentElm().GetID(), csToID);
                }
                else
                {// どちらかが中継端子ではない場合
                    if (csFromPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                    {// 流入中継端子の場合
                        csConnect = ((McStructGroupElement)csGroupElement.GetViewerElm()).AddExternalINConnection(new HySID(""), csFromID, csToPort.GetParentElm().GetID(), csToID);
                    }
                    else if (csToPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
                    {// 流出中継端子の場合
                        csConnect = ((McStructGroupElement)csGroupElement.GetViewerElm()).AddExternalOUTConnection(new HySID(""), csFromPort.GetParentElm().GetID(), csFromID, csToID);
                    }
                }
            }

            if (csConnect != null)
            {// 成功
                csConnect.GetConnectionPort(ref csStructFromID, ref csStructToID);

                csFromPort.SetViewerPort(csFromPort.GetParentElm().GetViewerElm().GetPort(csStructFromID));
                csToPort.SetViewerPort(csToPort.GetParentElm().GetViewerElm().GetPort(csStructToID));

                // 接続線の生成
                csNewLine = CreateGraphic(McDefine.SymbolDrawKind.HYM_LINE, new HySObjectKind());
                csNewLine.SetViewerConnect(csConnect);
                csNewLine.AddPort(csFromPort);
                csNewLine.AddPort(csToPort);

                csGroupElement.AddLineElement(csNewLine);
            }
            else
            {// 失敗
                csNewLine = null;
            }

            return csNewLine;
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトグループ化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = GroupObject(eDrawKind)</para>
        /// </example>
        /// <param name="eDrawKind">シンボル描画種別</param>
        /// <returns>McGraphicsElement !=null:成功，null:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement GroupObject(McDefine.SymbolDrawKind eDrawKind)
        {
            McGraphicsGroupElement csNewGrElm = null; // グループ化要素
            McGraphicsElement csElm;            // 要素
            HySIdentifier csFromID = null;      // 接続元ID
            HySIdentifier csToID = null;        // 接続先ID
            HySDataLinkedList csGroupMemberElmList = new HySDataLinkedList();
            int iLp;                            // ループカウンタ
            int iLp2;                           // ループカウンタ
            long lX1 = 0L;                      // 描画位置X
            long lY1 = 0L;                      // 描画位置Y
            long lWidth = 0L;
            long lHeight = 0L;
            bool bCheckFlg = false;

            if ((eDrawKind != McDefine.SymbolDrawKind.HYM_GROUP) &&
                (eDrawKind != McDefine.SymbolDrawKind.HYM_CALC_GROUP))
            {// グループ要素ではない場合
                return csNewGrElm;
            }

            // 選択要素チェック
            for (iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_LINE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_SEND) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 接続線・端子・中継端子以外の場合
                    bCheckFlg = true;
                    break;
                }
            }
            if (bCheckFlg == false)
            {// 対象要素なしの場合
                return csNewGrElm;
            }

            List<McGraphicsElement> csWorkElmList = new List<McGraphicsElement>();

            for (iLp = m_csElmList.Count - 1; iLp >= 0; iLp--)
            {// すべての要素を検索
                csElm = null;
                for (iLp2 = 0; iLp2 < m_csSelectionList.Count; iLp2++)
                {// IDが一致するか検索
                    if (m_csElmList[iLp].GetID().Equals(m_csSelectionList[iLp2].GetID()) == true)
                    {// IDが一致する
                        csElm = m_csElmList[iLp];
                        csWorkElmList.Add(csElm);
                        csGroupMemberElmList.AddLast(csElm.GetViewerElm());

                        m_csElmList.RemoveAt(iLp);
                        break;
                    }
                }
            }

            //グループに入った要素間の接続線の移動
            bool bFromFlg = false;
            bool bToFlg = false;
            for (iLp = m_csLineList.Count - 1; iLp >= 0; iLp--)
            {// 接続線数ループ
                bFromFlg = false;
                bToFlg = false;
                m_csLineList[iLp].GetPortID(ref csFromID, ref csToID);
                csFromID = m_csLineList[iLp].GetPort(csFromID).GetParentElm().GetID();
                csToID = m_csLineList[iLp].GetPort(csToID).GetParentElm().GetID();
                for (iLp2 = 0; iLp2 < csWorkElmList.Count; iLp2++)
                {// グループ内要素数ループ
                    if (csWorkElmList[iLp2].Equal(csFromID) == true)
                    {// IDが一致する場合
                        bFromFlg = true;
                    }
                    if (csWorkElmList[iLp2].Equal(csToID) == true)
                    {// IDが一致する場合
                        bToFlg = true;
                    }
                    if ((bFromFlg == true) && (bToFlg == true))
                    {// 両方一致の場合
                        m_csLineList.RemoveAt(iLp);
                        break;
                    }
                }
            }

            // グループ化
            McStructGroupElement csNewViewGrElm = null;
            if (m_csPossessionElm == null)
            {// 全体系の場合
                csNewViewGrElm = (McStructGroupElement)m_csMcStructInfo.MakeGroup(new HySID(""), csGroupMemberElmList, eDrawKind, new HySObjectKind());
            }
            else
            {// グループ内の場合
                csNewViewGrElm = (McStructGroupElement)((McStructGroupElement)m_csPossessionElm.GetViewerElm()).MakeGroup(new HySID(""), csGroupMemberElmList, eDrawKind, new HySObjectKind());
            }

            // 描画要素の復元
            List<McGraphicsElement> csPortList = new List<McGraphicsElement>();
            csNewGrElm = (McGraphicsGroupElement)DecodeGraphic(csNewViewGrElm, ref csPortList);

            csNewGrElm.SetViewerElm(csNewViewGrElm);

            csNewGrElm.SetParentElm(m_csPossessionElm);

            //グループシンボル座標設定
            McViewTransFormer.GroupPositionCalc(csWorkElmList, ref lX1, ref lY1);
            csNewGrElm.GetElementSize(ref lWidth, ref lHeight);
            csNewGrElm.SetPosition(lX1 + lWidth / 2, lY1 + lHeight / 2);

            m_csElmList.Add(csNewGrElm);

            // 作成グループ要素接続線ポートリスト積み直し
            HySIdentifier csFromID1 = null;
            HySIdentifier csToID1 = null;
            HySIdentifier csFromID2 = null;
            HySIdentifier csToID2 = null;
            McGraphicsElement csFromPort = null;
            McGraphicsElement csToPort = null;
            for (iLp = 0; iLp < csNewGrElm.GetPortCount(); iLp++)
            {// 端子数ループ
                csNewGrElm.GetPort(iLp).GetPortID(ref csFromID1, ref csToID1);
                if ((csFromID1.ToString() != "") && (csToID1.ToString() != ""))
                {// 接続情報ありの場合
                    for (iLp2 = 0; iLp2 < m_csLineList.Count; iLp2++)
                    {// 接続線数ループ
                        m_csLineList[iLp2].GetPortID(ref csFromID2, ref csToID2);
                        if ((csFromID1.Equals(csFromID2) == true) && (csToID1.Equals(csToID2) == true))
                        {// IDが両方一致する場合
                            if (csNewGrElm.GetPort(iLp).GetID().Equals(csFromID2) == true)
                            {// IDが一致する場合
                                csToPort = m_csLineList[iLp2].GetPort(csToID2);
                                m_csLineList[iLp2].DelPort(1);
                                m_csLineList[iLp2].DelPort(0);
                                m_csLineList[iLp2].AddPort(csNewGrElm.GetPort(iLp));
                                m_csLineList[iLp2].AddPort(csToPort);
                            }
                            else
                            {// IDが一致する場合
                                csFromPort = m_csLineList[iLp2].GetPort(csFromID2);
                                m_csLineList[iLp2].DelPort(1);
                                m_csLineList[iLp2].DelPort(0);
                                m_csLineList[iLp2].AddPort(csFromPort);
                                m_csLineList[iLp2].AddPort(csNewGrElm.GetPort(iLp));
                            }
                        }
                    }
                }
            }

            // 端子位置設定
            if (m_csPossessionElm == null)
            {// 全体系の場合
                SetAllDecodePortPosition(m_csElmList, m_csLineList);
            }
            else
            {// グループ内の場合
                SetAllDecodePortPosition(((McGraphicsGroupElement)m_csPossessionElm).GetGroupElementList(), ((McGraphicsGroupElement)m_csPossessionElm).GetGroupLineElementList());
            }

            //グループ内要素座標再セット
            ReSetGroupMemberPosition(csNewGrElm);

            //中継端子配置座標設定
            ReSetPortPosition(csNewGrElm);

            return csNewGrElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトグループ解除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = DisbandObject(csID, ref csMoveElmIDList, ref csBeforePointList, ref csAfterPointList, ref csDelOuterLineList)</para>
        /// </example>
        /// <param name="csID">ID</param>
        /// <param name="csMoveElmIDList">移動要素ID</param>
        /// <param name="csBeforePointList">移動前座標</param>
        /// <param name="csAfterPointList">移動後座標</param>
        /// <param name="csDelOuterLineList">削除外部接続線リスト</param>
        /// <returns>long 0:成功，-1:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long DisbandObject(HySIdentifier csID, ref List<HySIdentifier> csMoveElmIDList, ref List<Point> csBeforePointList, ref List<Point> csAfterPointList, ref List<McGraphicsElement> csDelOuterLineList)
        {
            McGraphicsGroupElement csGrElm = null;     // グループオブジェクト
            McGraphicsElement csRelayPort;             // 端子
            McGraphicsElement csPort1;                 // 端子
            McGraphicsElement csPort2;                 // 端子
            McGraphicsElement csTempPort;              // 端子
            McGraphicsElement csElm;                   // グループ要素
            int iLp;                                   // ループカウンタ
            long lLp;                                  // ループカウンタ
            HySIdentifier csFID1 = null;               // 接続元ID
            HySIdentifier csTID1 = null;               // 接続先ID
            HySIdentifier csFID2 = null;               // 接続元ID
            HySIdentifier csTID2 = null;               // 接続先ID
            HySIdentifier csFID3 = null;               // 接続元ID
            HySIdentifier csTID3 = null;               // 接続先ID
            List<McGraphicsElement> csElmList;         // 要素リスト

            if ((object)csID == null)
            {// パラメータ異常
                return -1L;
            }

            csElmList = new List<McGraphicsElement>();

            for (iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// すべての要素を検索
                if ((m_csElmList[iLp].GetID().Equals(csID)) == true)
                {// ID一致
                    if ((m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                        (m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループオブジェクトの場合
                        csGrElm = (McGraphicsGroupElement)m_csElmList[iLp];
                    }
                }
                else
                {// ID不一致
                    csElmList.Add(m_csElmList[iLp]);
                }
            }

            if (csGrElm == null)
            {// グループオブジェクトではない
                return -1L;
            }

            //グループ内エレメント表示座標設定
            ReSetElementPosition(csGrElm, ref csMoveElmIDList, ref csBeforePointList, ref csAfterPointList);

            for (lLp = 0L; lLp < csGrElm.GetPortCount(); lLp++)
            {// ポートの接続情報取得
                csRelayPort = csGrElm.GetPort(lLp);
                csRelayPort.GetPortID(ref csFID1, ref csTID1);
                ((McGraphicsTransElement)csRelayPort).GetTransID(ref csFID2, ref csTID2);

                if (((csFID1.ToString() != "") && (csTID1.ToString() != "")) &&
                    ((csFID2.ToString() == "") || (csTID2.ToString() == "")))
                {// グループ外部のみ接続情報がある場合
                    // グループ外部のみ接続の情報削除
                    for (iLp = 0; iLp < m_csLineList.Count; iLp++)
                    {// 接続線数ループ
                        m_csLineList[iLp].GetPortID(ref csFID3, ref csTID3);

                        if ((csRelayPort.GetID().Equals(csFID3) == true) || (csRelayPort.GetID().Equals(csTID3) == true))
                        {// IDが一致
                            csPort1 = m_csLineList[iLp].GetPort(csFID3);
                            csPort2 = m_csLineList[iLp].GetPort(csTID3);

                            if (csRelayPort.GetID().Equals(csFID3) == true)
                            {// IDが一致する場合
                                if ((csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN)&&
                                    (csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort2.GetParentElm().DelPort(csPort2.GetID());
                                    ReSetPortPosition(csPort2.GetParentElm());
                                }
                            }
                            if (csRelayPort.GetID().Equals(csTID3) == true)
                            {// IDが一致する場合
                                if ((csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort1.GetParentElm().DelPort(csPort1.GetID());
                                    ReSetPortPosition(csPort1.GetParentElm());
                                }
                            }

                            // 削除接続線記憶
                            csDelOuterLineList.Add(m_csLineList[iLp]);

                            m_csLineList.RemoveAt(iLp);
                            break;
                        }
                    }
                }
                if (((csFID1.ToString() == "") || (csTID1.ToString() == "")) &&
                    ((csFID2.ToString() != "") && (csTID2.ToString() != "")))
                {// グループ内部のみ接続情報がある場合
                    // グループ内部のみ接続の情報削除
                    for (iLp = 0; iLp < (int)csGrElm.GetLineElmCount(); iLp++)
                    {// グループ内接続線数ループ
                        csGrElm.GetLineElement(iLp).GetPortID(ref csFID3, ref csTID3);

                        if ((csRelayPort.GetID().Equals(csFID3) == true) || (csRelayPort.GetID().Equals(csTID3) == true))
                        {// IDが一致
                            csPort1 = csGrElm.GetLineElement(iLp).GetPort(csFID3);
                            csPort2 = csGrElm.GetLineElement(iLp).GetPort(csTID3);

                            if (csRelayPort.GetID().Equals(csFID3) == true)
                            {// IDが一致する場合
                                if ((csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort2.GetParentElm().DelPort(csPort2.GetID());
                                    ReSetPortPosition(csPort2.GetParentElm());
                                }
                            }
                            if (csRelayPort.GetID().Equals(csTID3) == true)
                            {// IDが一致する場合
                                if ((csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort1.GetParentElm().DelPort(csPort1.GetID());
                                    ReSetPortPosition(csPort1.GetParentElm());
                                }
                            }
                            break;
                        }
                    }
                }
            }

            // グループ解除
            if (m_csPossessionElm == null)
            {// 全体系の場合
                m_csMcStructInfo.Disband(csGrElm.GetID());
            }
            else
            {// グループ内の場合
                ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).Disband(csGrElm.GetID());
            }

            // 接続線ポートリスト積み直し
            for (lLp = 0L; lLp < csGrElm.GetPortCount(); lLp++)
            {// ポートの接続情報取得

                csRelayPort = csGrElm.GetPort(lLp);
                csRelayPort.GetPortID(ref csFID1, ref csTID1);

                if ((csFID1.ToString() != "") && (csTID1.ToString() != ""))
                {// グループ外部の接続情報がある場合
                    for (iLp = m_csLineList.Count - 1; iLp >= 0; iLp--)
                    {// 接続線数ループ
                        csPort1 = m_csLineList[iLp].GetPort(0);
                        csPort2 = m_csLineList[iLp].GetPort(1);

                        if ((csRelayPort.GetID().Equals(csPort1.GetID()) == true) || (csRelayPort.GetID().Equals(csPort2.GetID()) == true))
                        {// IDが一致
                            ((McGraphicsTransElement)csRelayPort).GetTransID(ref csFID1, ref csTID1);

                            if ((csFID1.ToString() != "") && (csTID1.ToString() != ""))
                            {// グループ内部の接続情報がある場合
                                m_csLineList[iLp].GetPortID(ref csFID2, ref csTID2);
                                if (csRelayPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                                {// 流入中継端子の場合
                                    for (int iLp2 = 0; iLp2 < csGrElm.GetElmCount(); iLp2++)
                                    {// グループ内要素数ループ
                                        csTempPort = csGrElm.GetElement(iLp2).GetPort(csTID2);
                                        if (csTempPort != null)
                                        {// 端子がある場合
                                            m_csLineList[iLp].DelPort(1);
                                            m_csLineList[iLp].DelPort(0);
                                            m_csLineList[iLp].AddPort(csPort1);
                                            m_csLineList[iLp].AddPort(csTempPort);
                                            break;
                                        }
                                    }
                                }
                                else
                                {// 流入中継端子以外の場合
                                    for (int iLp2 = 0; iLp2 < csGrElm.GetElmCount(); iLp2++)
                                    {// グループ内要素数ループ
                                        csTempPort = csGrElm.GetElement(iLp2).GetPort(csFID2);
                                        if (csTempPort != null)
                                        {// 端子がある場合
                                            m_csLineList[iLp].DelPort(1);
                                            m_csLineList[iLp].DelPort(0);
                                            m_csLineList[iLp].AddPort(csTempPort);
                                            m_csLineList[iLp].AddPort(csPort2);
                                            break;
                                        }
                                    }
                                }
                            }
                            break;
                        }
                    }
                }
            }

            // グループ内部要素接続情報追加
            for (lLp = 0L; lLp < csGrElm.GetLineElmCount(); lLp++)
            {// グループ内接続線数ループ
                csElm = csGrElm.GetLineElement(lLp);
                csElm.GetPortID(ref csFID3, ref csTID3);
                if (csElm.GetPort(csFID3).GetParentElm().GetID().Equals(csGrElm.GetID()) == true)
                {// IDが一致する場合
                    continue;
                }
                if (csElm.GetPort(csTID3).GetParentElm().GetID().Equals(csGrElm.GetID()) == true)
                {// IDが一致する場合
                    continue;
                }
                csElm.SetParentElm(m_csPossessionElm);
                m_csLineList.Add(csElm);
            }
            
            ClearSelectionList();

            // グループ要素追加
            for (lLp = 0L; lLp < csGrElm.GetElmCount(); lLp++)
            {// グループ要素数ループ
                csElm = csGrElm.GetElement(lLp);
                csElm.SetParentElm(m_csPossessionElm);
                csElmList.Add(csElm);
                AddSelectionList(csElm);
            }

            m_csElmList.Clear();
            for (iLp = 0; iLp < csElmList.Count; iLp++)
            {// 要素数ループ
                m_csElmList.Add(csElmList[iLp]);
            }

            return 0L;
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト描画（要素）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObjectElement(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawObjectElement(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            //要素描画
            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                m_csElmList[iLp].DrawObject(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト描画（接続線）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObjectLine(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawObjectLine(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            //接続線描画
            for (int iLp = 0; iLp < m_csLineList.Count; iLp++)
            {// 接続線数ループ
                m_csLineList[iLp].DrawObject(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト描画（端子）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObjectPort(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawObjectPort(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            //端子描画
            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList[iLp].GetPortCount(); iLp2++)
                {// 端子数ループ
                    if ((m_csElmList[iLp].GetPort(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                        (m_csElmList[iLp].GetPort(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 中継端子の場合
                        ((McGraphicsTransElement)m_csElmList[iLp].GetPort(iLp2)).DrawObject(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, false);
                    }
                    else
                    {// 中継端子以外の場合
                        m_csElmList[iLp].GetPort(iLp2).DrawObject(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
                    }
                }
            }

            //中継端子描画
            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp)).DrawObject(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, true);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セレクトマーク描画（要素）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawSelectElement(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawSelectElement(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            //要素セレクトマーク描画
            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_LINE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_SEND) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 接続線・端子・中継端子以外の場合
                    m_csSelectionList[iLp].SelectDraw(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セレクトマーク描画（接続線）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawSelectLine(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawSelectLine(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            //接続線セレクトマーク描画
            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE)
                {// 接続線の場合
                    m_csSelectionList[iLp].SelectDraw(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セレクトマーク描画（端子）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawSelectPort(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawSelectPort(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                //中継端子セレクトマーク描画
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    if (m_csPossessionElm != null)
                    {// グループ内の場合
                        if (m_csPossessionElm.GetID().Equals(m_csSelectionList[iLp].GetParentElm().GetID()) == true)
                        {// IDが一致する場合
                            ((McGraphicsTransElement)m_csSelectionList[iLp]).SelectDraw(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, true);
                        }
                        else
                        {// IDが一致しない場合
                            ((McGraphicsTransElement)m_csSelectionList[iLp]).SelectDraw(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, false);
                        }
                    }
                    else
                    {// 全体系の場合
                        ((McGraphicsTransElement)m_csSelectionList[iLp]).SelectDraw(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, false);
                    }
                }
                //端子セレクトマーク描画
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                    (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_SEND))
                {// 端子の場合
                    m_csSelectionList[iLp].SelectDraw(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトID描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObjectID(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawObjectID(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            //要素ID描画
            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                m_csElmList[iLp].DrawObjectID(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, m_bIDMode);
            }

            //接続線ID描画
            for (int iLp = 0; iLp < m_csLineList.Count; iLp++)
            {// 接続線数ループ
                m_csLineList[iLp].DrawObjectID(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, m_bIDMode);
            }

            //中継端子（外側）ID描画
            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList[iLp].GetPortCount(); iLp2++)
                {// 端子数ループ
                    if ((m_csElmList[iLp].GetPort(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                        (m_csElmList[iLp].GetPort(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 中継端子の場合
                        ((McGraphicsTransElement)m_csElmList[iLp].GetPort(iLp2)).DrawObjectID(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, m_bIDMode, false);
                    }
                }
            }

            //中継端子（内側）ID描画
            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp)).DrawObjectID(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, m_bIDMode, true);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト名称描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawObjectName(csGraphics, iImageX, iImageY, iX, iY, iWidth, iHeight)</para>
        /// </example>
        /// <param name="csGraphics">グラフィックス</param>
        /// <param name="iImageX">画面表示座標X</param>
        /// <param name="iImageY">画面表示座標Y</param>
        /// <param name="iX">描画始点座標X</param>
        /// <param name="iY">描画始点座標Y</param>
        /// <param name="iWidth">描画範囲幅</param>
        /// <param name="iHeight">描画範囲高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawObjectName(Graphics csGraphics, int iImageX, int iImageY, int iX, int iY, int iWidth, int iHeight)
        {
            //要素名称描画
            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if (((object)m_csElmList[iLp].GetElementName() != null) &&
                    (m_csElmList[iLp].GetElementName().GetLength() != 0))
                {// 名称設定がある場合
                    m_csElmList[iLp].DrawObjectName(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
                }

                // 中継端子接続線名称描画
                if ((m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                    (m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                {// グループ要素の場合
                    for (int iLp2 = 0; iLp2 < m_csElmList[iLp].GetPortCount(); iLp2++)
                    {// 端子数ループ
                        if (((object)m_csElmList[iLp].GetPort(iLp2).GetElementName() != null) &&
                            (m_csElmList[iLp].GetPort(iLp2).GetElementName().GetLength() != 0))
                        {// 名称設定がある場合
                            ((McGraphicsTransElement)m_csElmList[iLp].GetPort(iLp2)).DrawObjectName(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, false);
                        }
                    }
                }
            }

            //接続線名称描画
            for (int iLp = 0; iLp < m_csLineList.Count; iLp++)
            {// 接続線数ループ
                if (((object)m_csLineList[iLp].GetElementName() != null) &&
                    (m_csLineList[iLp].GetElementName().GetLength() != 0))
                {// 名称設定がある場合
                    m_csLineList[iLp].DrawObjectName(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight);
                }
            }

            //中継端子（内側）接続線名称描画
            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    if (((object)m_csPossessionElm.GetPort(iLp).GetElementName() != null) &&
                        (m_csPossessionElm.GetPort(iLp).GetElementName().GetLength() != 0))
                    {// 名称設定がある場合
                        ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp)).DrawObjectName(csGraphics, m_csCustomInfo, m_lDispScale, iImageX, iImageY, iX, iY, iWidth, iHeight, true);
                    }
                }
            }
        }

        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        // Ver1.3では下記コードを追加する（Gr内表示） 
        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        /// <summary><para>method outline:</para>
        /// <para>Gr要素内部のイメージ画像設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetGrImage(csImgTbl)</para>
        /// </example>
        /// <param name="csImgTbl">Gr内部のイメージハッシュ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetGrImage(Hashtable csImgTbl)
        {
            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if (m_csElmList[iLp].GetViewerElm() as McStructGroupElement != null)
                {
                    // グループ要素
                    HySID csGrElmId = (HySID)m_csElmList[iLp].GetViewerElm().GetID();
                    // グループ要素IDごとに画像を識別
                    Image csImg = (Image)csImgTbl[csGrElmId.ToString()];
                    m_csElmList[iLp].SetGrImage(csImg);

                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜McGraphicsElement＞ csElementList = GetElementList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜McGraphicsElement＞　エレメントリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<McGraphicsElement> GetElementList()
        {
            return m_csElmList;
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントリスト設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetElementList(csElementList)</para>
        /// </example>
        /// <param name="csElementList">エレメントリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetElementList(List<McGraphicsElement> csElementList)
        {
            m_csElmList = csElementList;
        }

        /// <summary><para>method outline:</para>
        /// <para>ラインリスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜McGraphicsElement＞ csLineList = GetLineList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜McGraphicsElement＞　ラインリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<McGraphicsElement> GetLineList()
        {
            return m_csLineList;
        }

        /// <summary><para>method outline:</para>
        /// <para>ラインリスト設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetLineList(csLineList)</para>
        /// </example>
        /// <param name="csLineList">ラインリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetLineList(List<McGraphicsElement> csLineList)
        {
            m_csLineList = csLineList;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面描画要素保有要素設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetPossessionElement(csPossessionElement)</para>
        /// </example>
        /// <param name="csPossessionElement">画面描画要素保有要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetPossessionElement(McGraphicsElement csPossessionElement)
        {
            m_csPossessionElm = csPossessionElement;
        }

        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        // Ver1.3では下記コードを追加する（Gr内表示） 
        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        /// <summary><para>method outline:</para>
        /// <para>画面描画親要素取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetPossessionElement()</para>
        /// </example>
        /// <returns>画面描画親要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement GetPossessionElement()
        {
            return m_csPossessionElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素リスト取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜McGraphicsElement＞ csSelectList = GetSelectionList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>List＜McGraphicsElement＞　選択中要素リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<McGraphicsElement> GetSelectionList()
        {
            return m_csSelectionList;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素別カウンター取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int[] iElementCounter = GetSelectionCount()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>int[]　選択中要素別カウンター</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int[] GetSelectionCount()
        {
            return m_iElementCounter;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素リスト追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>AddSelectionList(addID)</para>
        /// </example>
        /// <param name="addID">追加要素ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void AddSelectionList(HySString addID)
        {
            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if ((HySString)m_csElmList[iLp].GetID().GetString() == addID)
                {// IDが一致する場合
                    m_csSelectionList.Add(m_csElmList[iLp]);

                    if (m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                    {// 通常要素の場合
                        // 要素数カウントアップ
                        m_iElementCounter[0]++;
                    }
                    if ((m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                        (m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループ要素の場合
                        // グループ数カウントアップ
                        m_iElementCounter[1]++;
                    }
                    return;
                }
                for (int iLp2 = 0; iLp2 < m_csElmList[iLp].GetPortCount(); iLp2++)
                {// 端子数ループ
                    if ((HySString)m_csElmList[iLp].GetPort(iLp2).GetID().GetString() == addID)
                    {// IDが一致する場合
                        m_csSelectionList.Add(m_csElmList[iLp].GetPort(iLp2));

                        if ((m_csElmList[iLp].GetPort(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                            (m_csElmList[iLp].GetPort(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                        {// 中継端子の場合
                            // 中継端子数カウントアップ
                            m_iElementCounter[3]++;
                        }
                        else
                        {// 中継端子以外の場合
                            // 端子数カウントアップ
                            m_iElementCounter[2]++;
                        }
                        return;
                    }
                }
            }

            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    if ((HySString)m_csPossessionElm.GetPort(iLp).GetID().GetString() == addID)
                    {// IDが一致する場合
                        m_csSelectionList.Add(m_csPossessionElm.GetPort(iLp));

                        // 中継端子数カウントアップ
                        m_iElementCounter[3]++;
                        // 中継端子（グループ内部）数カウントアップ
                        m_iElementCounter[5]++;
                        return;
                    }
                }
            }

            for (int iLp = 0; iLp < m_csLineList.Count; iLp++)
            {// 接続線数ループ
                if ((HySString)m_csLineList[iLp].GetID().GetString() == addID)
                {// IDが一致する場合
                    m_csSelectionList.Add(m_csLineList[iLp]);

                    // 接続線数カウントアップ
                    m_iElementCounter[4]++;
                    return;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素リスト追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>AddSelectionList(addElement)</para>
        /// </example>
        /// <param name="addElement">追加要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void AddSelectionList(McGraphicsElement addElement)
        {
            m_csSelectionList.Add(addElement);

            if (addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
            {// 通常要素の場合
                // 要素数カウントアップ
                m_iElementCounter[0]++;
            }
            else if ((addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                     (addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
            {// グループ要素の場合
                // グループ数カウントアップ
                m_iElementCounter[1]++;
            }
            else if ((addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                     (addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_SEND))
            {// 端子の場合
                // 端子数カウントアップ
                m_iElementCounter[2]++;
            }
            else if ((addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                     (addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 中継端子の場合
                // 中継端子数カウントアップ
                m_iElementCounter[3]++;

                if (m_csPossessionElm != null)
                {// グループ内の場合
                    for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                    {// 端子数ループ
                        if ((HySString)m_csPossessionElm.GetPort(iLp).GetID().GetString() == (HySString)addElement.GetID().GetString())
                        {// IDが一致する場合
                            // 中継端子（グループ内部）数カウントアップ
                            m_iElementCounter[5]++;
                            return;
                        }
                    }
                }
            }
            else if (addElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE)
            {// 接続線の場合
                // 接続線数カウントアップ
                m_iElementCounter[4]++;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素リストID指定解除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DelSelectionList(delID)</para>
        /// </example>
        /// <param name="delID">解除要素ID</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DelSelectionList(HySString delID)
        {
            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((HySString)m_csSelectionList[iLp].GetID().GetString() == delID)
                {
                    if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                    {// 通常要素の場合
                        // 要素数カウントダウン
                        m_iElementCounter[0]--;
                    }
                    else if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                             (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループ要素の場合
                        // グループ数カウントダウン
                        m_iElementCounter[1]--;
                    }
                    else if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                             (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_SEND))
                    {// 端子の場合
                        // 端子数カウントダウン
                        m_iElementCounter[2]--;
                    }
                    else if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                             (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 中継端子の場合
                        // 中継端子数カウントダウン
                        m_iElementCounter[3]--;

                        if (m_csPossessionElm != null)
                        {// グループ内の場合
                            for (int iLp2 = 0; iLp2 < m_csPossessionElm.GetPortCount(); iLp2++)
                            {// 端子数ループ
                                if ((HySString)m_csPossessionElm.GetPort(iLp2).GetID().GetString() == delID)
                                {// IDが一致する場合
                                    // 中継端子（グループ内部）数カウントダウン
                                    m_iElementCounter[5]--;
                                    return;
                                }
                            }
                        }
                    }
                    else if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE)
                    {// 接続線の場合
                        // 接続線数カウントダウン
                        m_iElementCounter[4]--;
                    }

                    m_csSelectionList.RemoveAt(iLp);

                    break;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素リストID指定検索</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iRtn = SearchSelectionList(SearchID)</para>
        /// </example>
        /// <param name="SearchID">選択中要素要素ID</param>
        /// <returns>int　該当リストNo（該当無ければ -1）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int SearchSelectionList(HySString SearchID)
        {
            int iRtnNo = -1;

            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((HySString)m_csSelectionList[iLp].GetID().GetString() == SearchID)
                {// IDが一致する場合
                    iRtnNo = iLp;
                    break;
                }
            }

            return iRtnNo;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素リストクリア</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ClearSelectionList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ClearSelectionList()
        {
            m_csSelectionList.Clear();

            m_iElementCounter[0] = 0;
            m_iElementCounter[1] = 0;
            m_iElementCounter[2] = 0;
            m_iElementCounter[3] = 0;
            m_iElementCounter[4] = 0;
            m_iElementCounter[5] = 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>要素名称設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = SetElementName(ElementName, ref ElementNameOld, csID)</para>
        /// </example>
        /// <param name="ElementName">要素名称</param>
        /// <param name="ElementNameOld">設定前要素名称</param>
        /// <param name="csID">要素ID</param>
        /// <returns>McGraphicsElement　設定要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement SetElementName(HySString ElementName, ref HySString ElementNameOld, HySString csID)
        {
            int iLp;    // ループカウンタ
            McGraphicsElement csElement = null;

            for (iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素を検索
                if ((HySString)m_csElmList[iLp].GetID().GetString() == csID)
                {// IDが一致した場合
                    ElementNameOld = m_csElmList[iLp].GetElementName().Clone();
                    m_csElmList[iLp].SetElementName(ElementName);
                    csElement = m_csElmList[iLp];
                    break;
                }
            }
            if (csElement == null)
            {// 要素に対象がない場合
                for (iLp = 0; iLp < m_csLineList.Count; iLp++)
                {// 接続線を検索
                    if ((HySString)m_csLineList[iLp].GetID().GetString() == csID)
                    {// IDが一致した場合
                        ElementNameOld = m_csLineList[iLp].GetElementName().Clone();
                        m_csLineList[iLp].SetElementName(ElementName);
                        csElement = m_csLineList[iLp];
                        break;
                    }
                }
            }

            return csElement;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySIdentifier csElementID = SetElementDisplayModeName(DisplayMode, csID)</para>
        /// </example>
        /// <param name="DisplayMode">画面表示設定</param>
        /// <param name="csID">要素ID</param>
        /// <returns>HySIdentifier　画面表示設定要素ID</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySIdentifier SetElementDisplayMode(bool DisplayMode, HySString csID)
        {
            int iLp;    // ループカウンタ
            HySIdentifier csElementID = null;

            for (iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素を検索
                if ((HySString)m_csElmList[iLp].GetID().GetString() == csID)
                {// IDが一致した場合
                    m_csElmList[iLp].SetElementDisplayMode(DisplayMode);
                    csElementID = m_csElmList[iLp].GetID();
                    break;
                }
            }
            return csElementID;
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示設定取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GetElementDisplayMode(csElementID)</para>
        /// </example>
        /// <param name="csElementID">要素ID</param>
        /// <returns>bool : 画面表示設定</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool GetElementDisplayMode(HySString csElementID)
        {
            bool bDisplayMode = false;

            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if (m_csElmList[iLp].GetID().GetString().Equals(csElementID) == true)
                {// IDが一致する場合
                    bDisplayMode = m_csElmList[iLp].GetElementDisplayMode();
                    break;
                }
            }
            return bDisplayMode;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定位置エレメント有無チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = ElementPositionCheck(Position_x, Position_y, bPortFlg)</para>
        /// </example>
        /// <param name="Position_x">チェック位置X</param>
        /// <param name="Position_y">チェック位置Y</param>
        /// <param name="bPortFlg">true:端子選択　false:端子以外選択</param>
        /// <returns>McGraphicsElement　指定位置該当エレメント（該当無ければ null）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement ElementPositionCheck(int Position_x, int Position_y, ref bool bPortFlg)
        {
            int iLp;    // ループカウンタ
            int iLp2;    // ループカウンタ
            McGraphicsElement csRtnElm = null;

            bPortFlg = false;

            for (iLp = m_csElmList.Count - 1; iLp >= 0; iLp--)
            {// 要素数ループ
                // 指定位置エレメント有無チェック
                if (m_csElmList[iLp].ElementPositionCheck(Position_x, Position_y, m_lDispScale) == true)
                {// 指定位置に要素があった場合
                    csRtnElm = m_csElmList[iLp];
                    return csRtnElm;
                }
            }
            for (iLp = m_csElmList.Count - 1; iLp >= 0; iLp--)
            {// 要素数ループ
                for (iLp2 = 0; iLp2 < m_csElmList[iLp].GetPortCount(); iLp2++)
                {// 端子数ループ
                    if ((m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                        (m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループ要素の場合
                        // 指定位置エレメント有無チェック
                        if (((McGraphicsTransElement)m_csElmList[iLp].GetPort(iLp2)).ElementPositionCheck(Position_x, Position_y, m_lDispScale, false) == true)
                        {// 指定位置に要素があった場合
                            csRtnElm = m_csElmList[iLp].GetPort(iLp2);
                            bPortFlg = true;
                            return csRtnElm;
                        }
                    }
                    else
                    {// グループ要素以外の場合
                        // 指定位置エレメント有無チェック
                        if (m_csElmList[iLp].GetPort(iLp2).ElementPositionCheck(Position_x, Position_y, m_lDispScale) == true)
                        {// 指定位置に要素があった場合
                            csRtnElm = m_csElmList[iLp].GetPort(iLp2);
                            bPortFlg = true;
                            return csRtnElm;
                        }
                    }
                }
            }
            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    // 指定位置エレメント有無チェック
                    if (((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp)).ElementPositionCheck(Position_x, Position_y, m_lDispScale, true) == true)
                    {// 指定位置に要素があった場合
                        csRtnElm = m_csPossessionElm.GetPort(iLp);
                        return csRtnElm;
                    }
                }
            }
            for (iLp = m_csLineList.Count - 1; iLp >= 0; iLp--)
            {// 接続線数ループ
                // 指定位置エレメント有無チェック
                if (((McGraphicsLineElement)m_csLineList[iLp]).ElementPositionCheck(Position_x, Position_y, m_lDispScale, m_csCustomInfo) == true)
                {// 指定位置に要素があった場合
                    csRtnElm = m_csLineList[iLp];
                    return csRtnElm;
                }
            }

            return csRtnElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>ポート配置位置決定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetPortPosition(FromElement, FromPort, ToElement, ToPort)</para>
        /// </example>
        /// <param name="FromElement">要素（From）</param>
        /// <param name="FromPort">ポート（From）</param>
        /// <param name="ToElement">要素（To）</param>
        /// <param name="ToPort">ポート（To）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetPortPosition(McGraphicsElement FromElement, McGraphicsElement FromPort, McGraphicsElement ToElement, McGraphicsElement ToPort)
        {
            long lFromX = 0;
            long lFromY = 0;
            long lToX = 0;
            long lToY = 0;
            bool bFromInnerFlg = false;
            bool bToInnerFlg = false;

            FromElement.GetPosition(ref lFromX, ref lFromY);
            ToElement.GetPosition(ref lToX, ref lToY);

            if (FromPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
            {// 流入中継端子の場合
                ((McGraphicsTransElement)FromPort).GetInnerPosition(ref lFromX, ref lFromY);
                bFromInnerFlg = true;
            }
            if (ToPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
            {// 流出中継端子の場合
                ((McGraphicsTransElement)ToPort).GetInnerPosition(ref lToX, ref lToY);
                bToInnerFlg = true;
            }

            // 端子の位置関係にて端子の向きを設定
            if (Math.Abs(lFromX - lToX) > Math.Abs(lFromY - lToY))
            {// 横方向より縦方向が近い場合
                if (lFromX > lToX)
                {// From側が右に位置する場合
                    if (bFromInnerFlg == false)
                    {// 流入中継端子以外の場合
                        FromPort.SetRotate(McDefine.HYM_DRAW_ANGLE_270);
                    }

                    if (bToInnerFlg == false)
                    {// 流出中継端子以外の場合
                        ToPort.SetRotate(McDefine.HYM_DRAW_ANGLE_90);
                    }
                }
                else
                {// From側が右以外に位置する場合
                    if (bFromInnerFlg == false)
                    {// 流入中継端子以外の場合
                        FromPort.SetRotate(McDefine.HYM_DRAW_ANGLE_90);
                    }

                    if (bToInnerFlg == false)
                    {// 流出中継端子以外の場合
                        ToPort.SetRotate(McDefine.HYM_DRAW_ANGLE_270);
                    }
                }
            }
            else
            {// 横方向より縦方向が近い以外場合
                if (lFromY > lToY)
                {// From側が下に位置する場合
                    if (bFromInnerFlg == false)
                    {// 流入中継端子以外の場合
                        FromPort.SetRotate(McDefine.HYM_DRAW_ANGLE_0);
                    }

                    if (bToInnerFlg == false)
                    {// 流出中継端子以外の場合
                        ToPort.SetRotate(McDefine.HYM_DRAW_ANGLE_180);
                    }
                }
                else
                {// From側が下以外に位置する場合
                    if (bFromInnerFlg == false)
                    {// 流入中継端子以外の場合
                        FromPort.SetRotate(McDefine.HYM_DRAW_ANGLE_180);
                    }

                    if (bToInnerFlg == false)
                    {// 流出中継端子以外の場合
                        ToPort.SetRotate(McDefine.HYM_DRAW_ANGLE_0);
                    }
                }
            }

            ReSetPortPosition(FromElement, FromPort);
            ReSetPortPosition(ToElement, ToPort);
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>要素種別取得</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McDefine.SymbolDrawKind eKind = GetSymbolDrawKind(csID)</para>
        /// </example>
        /// <param name="csID">要素ID</param>
        /// <returns>McDefine.SymbolDrawKind  シンボル描画種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McDefine.SymbolDrawKind GetSymbolDrawKind(HySIdentifier csID)
        {
            int iLp;    // ループカウンタ

            for (iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if (m_csElmList[iLp].Equal(csID) == true)
                {// IDが一致する場合
                    return m_csElmList[iLp].GetSymbolDrawKind();
                }
            }
            for (iLp = 0; iLp < m_csLineList.Count; iLp++)
            {// 接続線数ループ
                if (m_csLineList[iLp].Equal(csID) == true)
                {// IDが一致する場合
                    return m_csLineList[iLp].GetSymbolDrawKind();
                }
            }

            return McDefine.SymbolDrawKind.HYM_NonSymbol;
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>要素モデル設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetElementModel(csElementModelData, ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetElementModel(McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList)
        {
            bool bSetFlg = false;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                if (csElementModelData.GetModelType() == McDefine.MODEL_TYPE_STANDARD)
                {// 標準モデルの場合
                    if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                    {// 通常要素の場合
                        // 設定可能かチェック
                        if ((m_csSelectionList[iLp].GetDrawSubKind() != McDefine.HYM_DRAW_SUB_KIND_INPUT) &&
                            (m_csSelectionList[iLp].GetDrawSubKind() != McDefine.HYM_DRAW_SUB_KIND_OUTPUT) &&
                            (m_csSelectionList[iLp].GetDrawSubKind() != McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE))
                        {// 入力・出力・マージ以外の場合
                            csElmIDList.Add(m_csSelectionList[iLp].GetID());
                            csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                            m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                            bSetFlg = true;
                        }
                    }
                }
                if (csElementModelData.GetModelType() == McDefine.MODEL_TYPE_GROUP)
                {// グループモデルの場合
                    // 設定可能かチェック
                    if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
                    {// 収束演算系グループ要素の場合
                        csElmIDList.Add(m_csSelectionList[iLp].GetID());
                        csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                        m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                        bSetFlg = true;
                    }
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>要素モデル設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetElementModel(csElementModelData, ref csElmIDList, ref csModelList, ref csElmIDListList,ref csTranList, ref csModTranList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <param name="csElmIDListList">モデル設定要素IDリストリスト</param>
        /// <param name="csTranListList">設定前モデル情報リスト</param>
        /// <param name="csModTranListList">設定後モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetElementModel(McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList, ref List<List<HySIdentifier>> csElmIDListList, ref List<List<McModelInfo>> csTranListList, ref List<List<McModelInfo>> csModTranListList)
        {
            bool bSetFlg = false;
            List<HySIdentifier> csLineElmIDList = null;
            List<McModelInfo> csTranList = null;
            List<McModelInfo> csModTranList = null;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                if (csElementModelData.GetModelType() == McDefine.MODEL_TYPE_STANDARD)
                {// 標準モデルの場合
                    if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                    {// 通常要素の場合
                        // 設定可能かチェック
                        if ((m_csSelectionList[iLp].GetDrawSubKind() != McDefine.HYM_DRAW_SUB_KIND_INPUT) &&
                            (m_csSelectionList[iLp].GetDrawSubKind() != McDefine.HYM_DRAW_SUB_KIND_OUTPUT) &&
                            (m_csSelectionList[iLp].GetDrawSubKind() != McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE))
                        {// 入力・出力・マージ以外の場合
                            csElmIDList.Add(m_csSelectionList[iLp].GetID());
                            csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                            m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                            csLineElmIDList = new List<HySIdentifier>();
                            csTranList = new List<McModelInfo>();
                            csModTranList = new List<McModelInfo>();
                            m_csSelectionList[iLp].SetDefaultPattern(ref csLineElmIDList, ref csTranList, ref csModTranList);
                            csElmIDListList.Add(csLineElmIDList);
                            csTranListList.Add(csTranList);
                            csModTranListList.Add(csModTranList);
                            bSetFlg = true;
                        }
                    }
                }
                if (csElementModelData.GetModelType() == McDefine.MODEL_TYPE_GROUP)
                {// グループモデルの場合
                    // 設定可能かチェック
                    if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
                    {// 収束演算系グループ要素の場合
                        csElmIDList.Add(m_csSelectionList[iLp].GetID());
                        csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                        m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                        csLineElmIDList = new List<HySIdentifier>();
                        csTranList = new List<McModelInfo>();
                        csModTranList = new List<McModelInfo>();
                        m_csSelectionList[iLp].SetDefaultPattern(ref csLineElmIDList, ref csTranList, ref csModTranList);
                        csElmIDListList.Add(csLineElmIDList);
                        csTranListList.Add(csTranList);
                        csModTranListList.Add(csModTranList);
                        bSetFlg = true;
                    }
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>入力データ設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetInputData(csElementModelData, ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetInputData(McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList)
        {
            bool bSetFlg = false;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                {// 通常要素の場合
                    // 設定可能かチェック
                    if (m_csSelectionList[iLp].GetDrawSubKind() == McDefine.HYM_DRAW_SUB_KIND_INPUT)
                    {// 入力要素の場合
                        csElmIDList.Add(m_csSelectionList[iLp].GetID());
                        csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                        m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                        bSetFlg = true;
                    }
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>入力データ設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetInputData(csElementModelData, ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <param name="csElmIDListList">モデル設定要素IDリストリスト</param>
        /// <param name="csTranListList">設定前モデル情報リスト</param>
        /// <param name="csModTranListList">設定後モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetInputData(McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList, ref List<List<HySIdentifier>> csElmIDListList, ref List<List<McModelInfo>> csTranListList, ref List<List<McModelInfo>> csModTranListList)
        {
            bool bSetFlg = false;
            List<HySIdentifier> csLineElmIDList = null;
            List<McModelInfo> csTranList = null;
            List<McModelInfo> csModTranList = null;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                {// 通常要素の場合
                    // 設定可能かチェック
                    if (m_csSelectionList[iLp].GetDrawSubKind() == McDefine.HYM_DRAW_SUB_KIND_INPUT)
                    {// 入力要素の場合
                        csElmIDList.Add(m_csSelectionList[iLp].GetID());
                        csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                        m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                        csLineElmIDList = new List<HySIdentifier>();
                        csTranList = new List<McModelInfo>();
                        csModTranList = new List<McModelInfo>();
                        m_csSelectionList[iLp].SetDefaultPattern(ref csLineElmIDList, ref csTranList, ref csModTranList);
                        csTranListList.Add(csTranList);
                        csModTranListList.Add(csModTranList);
                        bSetFlg = true;
                    }
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>出力データ設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetOutputData(csElementModelData, ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetOutputData(McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList)
        {
            bool bSetFlg = false;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                {// 通常要素の場合
                    // 設定可能かチェック
                    if (m_csSelectionList[iLp].GetDrawSubKind() == McDefine.HYM_DRAW_SUB_KIND_OUTPUT)
                    {// 出力要素の場合
                        csElmIDList.Add(m_csSelectionList[iLp].GetID());
                        csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                        m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                        if (m_csSelectionList[iLp].GetElementName().GetLength() == 0)
                        {// 名称設定がない場合
                            m_csSelectionList[iLp].SetElementName(new HySString(Properties.McDotNetViewerResources.PROPERTY_DATA_OUTPUT + m_iOutputCounter.ToString()));
                            m_iOutputCounter++;
                        }
                        bSetFlg = true;
                    }
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>出力データ設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetOutputData(csElementModelData, ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <param name="csElmIDListList">モデル設定要素IDリストリスト</param>
        /// <param name="csTranListList">設定前モデル情報リスト</param>
        /// <param name="csModTranListList">設定後モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetOutputData(McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList, ref List<List<HySIdentifier>> csElmIDListList, ref List<List<McModelInfo>> csTranListList, ref List<List<McModelInfo>> csModTranListList)
        {
            bool bSetFlg = false;
            List<HySIdentifier> csLineElmIDList = null;
            List<McModelInfo> csTranList = null;
            List<McModelInfo> csModTranList = null;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT)
                {// 通常要素の場合
                    // 設定可能かチェック
                    if (m_csSelectionList[iLp].GetDrawSubKind() == McDefine.HYM_DRAW_SUB_KIND_OUTPUT)
                    {// 出力要素の場合
                        csElmIDList.Add(m_csSelectionList[iLp].GetID());
                        csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                        m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                        csLineElmIDList = new List<HySIdentifier>();
                        csTranList = new List<McModelInfo>();
                        csModTranList = new List<McModelInfo>();
                        m_csSelectionList[iLp].SetDefaultPattern(ref csLineElmIDList, ref csTranList, ref csModTranList);
                        csTranListList.Add(csTranList);
                        csModTranListList.Add(csModTranList);
                        if (m_csSelectionList[iLp].GetElementName().GetLength() == 0)
                        {// 名称設定がない場合
                            m_csSelectionList[iLp].SetElementName(new HySString(Properties.McDotNetViewerResources.PROPERTY_DATA_OUTPUT + m_iOutputCounter.ToString()));
                            m_iOutputCounter++;
                        }
                        bSetFlg = true;
                    }
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>伝送情報型設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetTransInfoType(csElementModelData, ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetTransInfoType(ref McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList)
        {
            bool bSetFlg = false;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                // 設定可能かチェック
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE)
                {// 接続線の場合
                    csElmIDList.Add(m_csSelectionList[iLp].GetID());
                    csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                    m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                    m_csSelectionList[iLp].SetDefaultPattern();
                    csElementModelData = m_csSelectionList[iLp].GetModelInfo();
                    bSetFlg = true;
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>伝送情報型設定</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>SetTransInfoType(csElementModelData, ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElementModelData">要素モデル情報</param>
        /// <param name="csElmIDList">モデル設定要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <param name="csModModelList">設定後モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetTransInfoType(McModelInfo csElementModelData, ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList, ref List<McModelInfo> csModModelList)
        {
            bool bSetFlg = false;

            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                bSetFlg = false;
                // 設定可能かチェック
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE)
                {// 接続線の場合
                    csElmIDList.Add(m_csSelectionList[iLp].GetID());
                    csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                    m_csSelectionList[iLp].SetModelInfo(csElementModelData.Clone());
                    m_csSelectionList[iLp].SetDefaultPattern();
                    csModModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                    bSetFlg = true;
                }
                if (bSetFlg == false)
                {// 設定要素なしの場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>ライブラリ設定情報解除</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>CancelLibraryData(ref csElmIDList, ref csModelList)</para>
        /// </example>
        /// <param name="csElmIDList">モデル解除要素IDリスト</param>
        /// <param name="csModelList">設定前モデル情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void CancelLibraryData(ref List<HySIdentifier> csElmIDList, ref List<McModelInfo> csModelList)
        {
            for (int iLp = m_csSelectionList.Count - 1; iLp >= 0; iLp--)
            {// 選択要素数ループ
                // 解除可能かチェック
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP)
                {// 部分系グループ要素の場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                    continue;
                }
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT) &&
                  (m_csSelectionList[iLp].GetDrawSubKind() == McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE))
                {// マージ要素の場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                    continue;
                }
                if (m_csSelectionList[iLp].GetModelInfo().GetLibraryID().ToString() == "")
                {// モデル未設定の場合
                    DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                    continue;
                }

                csElmIDList.Add(m_csSelectionList[iLp].GetID());
                csModelList.Add(m_csSelectionList[iLp].GetModelInfo());
                m_csSelectionList[iLp].CancelModelInfo();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜HySIdentifier＞ csDelElmIDList = DelElement(ref csDelElmList, ref csTransPortList, ref csConnectionIDList)</para>
        /// </example>
        /// <param name="csDelElmList">削除情報リスト</param>
        /// <param name="csTransPortList">設定変更中継端子リスト</param>
        /// <param name="csConnectionIDList">設定変更前接続情報リスト</param>
        /// <returns>List＜HySIdentifier＞　削除情報IDリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<HySIdentifier> DelElement(ref List<McGraphicsElement> csDelElmList, ref List<McGraphicsElement> csTransPortList, ref List<List<HySIdentifier>> csConnectionIDList)
        {
            McGraphicsElement csElementWork1 = null;
            McGraphicsElement csElementWork2 = null;
            HySIdentifier csFromID = null;
            HySIdentifier csToID = null;
            HySIdentifier csNullID = new HySID("");
            List<HySIdentifier> csDelElmIDList = new List<HySIdentifier>();
            List<HySIdentifier> csConnectIDList = null;

            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE)
                {// 接続線の場合
                    // 接続線の削除および接続端子の削除
                    for (int iLp2 = 0; iLp2 < m_csLineList.Count; iLp2++)
                    {// 接続線数ループ
                        if (m_csSelectionList[iLp].GetID().Equals(m_csLineList[iLp2].GetID()) == true)
                        {// IDが一致する場合
                            m_csLineList[iLp2].GetPortID(ref csFromID, ref csToID);
                            csElementWork1 = m_csLineList[iLp2].GetPort(csFromID);
                            csElementWork2 = m_csLineList[iLp2].GetPort(csToID);

                            if ((csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                (csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                            {// 中継端子以外の場合
                                csDelElmList.Add(csElementWork1.CopyInfUsedByUndoRedoOnly(null));
                                DelSelectionList((HySString)csElementWork1.GetID().GetString());
                                csElementWork1.GetParentElm().DelPort(csElementWork1.GetID());
                                ReSetPortPosition(csElementWork1.GetParentElm());
                            }
                            else
                            {// 中継端子の場合
                                csConnectIDList = new List<HySIdentifier>();
                                csElementWork1.GetPortID(ref csFromID, ref csToID);
                                csConnectIDList.Add(csFromID);
                                csConnectIDList.Add(csToID);
                                ((McGraphicsTransElement)csElementWork1).GetTransID(ref csFromID, ref csToID);
                                csConnectIDList.Add(csFromID);
                                csConnectIDList.Add(csToID);
                                csConnectionIDList.Add(csConnectIDList);
                                csTransPortList.Add(csElementWork1);
                            }

                            if ((csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                (csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                            {// 中継端子以外の場合
                                csDelElmList.Add(csElementWork2.CopyInfUsedByUndoRedoOnly(null));
                                DelSelectionList((HySString)csElementWork2.GetID().GetString());
                                csElementWork2.GetParentElm().DelPort(csElementWork2.GetID());
                                ReSetPortPosition(csElementWork2.GetParentElm());
                            }
                            else
                            {// 中継端子の場合
                                csConnectIDList = new List<HySIdentifier>();
                                csElementWork2.GetPortID(ref csFromID, ref csToID);
                                csConnectIDList.Add(csFromID);
                                csConnectIDList.Add(csToID);
                                ((McGraphicsTransElement)csElementWork2).GetTransID(ref csFromID, ref csToID);
                                csConnectIDList.Add(csFromID);
                                csConnectIDList.Add(csToID);
                                csConnectionIDList.Add(csConnectIDList);
                                csTransPortList.Add(csElementWork2);
                            }

                            if (m_csPossessionElm == null)
                            {// 全体系の場合
                                m_csMcStructInfo.DeleteConnection(m_csLineList[iLp2].GetID());
                            }
                            else
                            {// グループ内の場合
                                ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).DeleteConnection(m_csLineList[iLp2].GetID());
                            }
                            csDelElmIDList.Add(m_csLineList[iLp2].GetID());
                            csDelElmList.Add(m_csLineList[iLp2].CopyInfUsedByUndoRedoOnly(null));
                            DelSelectionList((HySString)m_csLineList[iLp2].GetID().GetString());
                            m_csLineList.RemoveAt(iLp2);
                            iLp = -1;
                            break;
                        }
                    }

                    continue;
                }
                for (int iLp2 = 0; iLp2 < m_csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp].GetID().Equals(m_csElmList[iLp2].GetID()) == true)
                    {// IDが一致する場合
                        // 要素削除および削除要素の接続線の削除・接続端子の削除
                        for (int iLp3 = 0; iLp3 < m_csElmList[iLp2].GetPortCount(); iLp3++)
                        {// 端子数ループ
                            for (int iLp4 = 0; iLp4 < m_csLineList.Count; iLp4++)
                            {// 接続線数ループ
                                m_csLineList[iLp4].GetPortID(ref csFromID, ref csToID);
                                csElementWork1 = m_csLineList[iLp4].GetPort(csFromID);
                                csElementWork2 = m_csLineList[iLp4].GetPort(csToID);

                                if (m_csElmList[iLp2].GetPort(iLp3).Equal(csElementWork1.GetID()))
                                {// IDが一致する場合
                                    if ((csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                        (csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子以外の場合
                                        csDelElmList.Add(csElementWork2.CopyInfUsedByUndoRedoOnly(null));
                                        DelSelectionList((HySString)csElementWork2.GetID().GetString());
                                        csElementWork2.GetParentElm().DelPort(csElementWork2.GetID());
                                        ReSetPortPosition(csElementWork2.GetParentElm());
                                    }
                                    else
                                    {// 中継端子の場合
                                        csConnectIDList = new List<HySIdentifier>();
                                        csElementWork2.GetPortID(ref csFromID, ref csToID);
                                        csConnectIDList.Add(csFromID);
                                        csConnectIDList.Add(csToID);
                                        ((McGraphicsTransElement)csElementWork2).GetTransID(ref csFromID, ref csToID);
                                        csConnectIDList.Add(csFromID);
                                        csConnectIDList.Add(csToID);
                                        csConnectionIDList.Add(csConnectIDList);
                                        csTransPortList.Add(csElementWork2);
                                    }
                                    csDelElmList.Add(csElementWork1.CopyInfUsedByUndoRedoOnly(null));

                                    if (m_csPossessionElm == null)
                                    {// 全体系の場合
                                        m_csMcStructInfo.DeleteConnection(m_csLineList[iLp4].GetID());
                                    }
                                    else
                                    {// グループ内の場合
                                        ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).DeleteConnection(m_csLineList[iLp4].GetID());
                                    }
                                    csDelElmIDList.Add(m_csLineList[iLp4].GetID());
                                    csDelElmList.Add(m_csLineList[iLp4].CopyInfUsedByUndoRedoOnly(null));
                                    DelSelectionList((HySString)m_csLineList[iLp4].GetID().GetString());
                                    m_csLineList.RemoveAt(iLp4);
                                    break;
                                }
                                if (m_csElmList[iLp2].GetPort(iLp3).Equal(csElementWork2.GetID()))
                                {// IDが一致する場合
                                    if ((csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                        (csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子以外の場合
                                        csDelElmList.Add(csElementWork1.CopyInfUsedByUndoRedoOnly(null));
                                        DelSelectionList((HySString)csElementWork1.GetID().GetString());
                                        csElementWork1.GetParentElm().DelPort(csElementWork1.GetID());
                                        ReSetPortPosition(csElementWork1.GetParentElm());
                                    }
                                    else
                                    {// 中継端子の場合
                                        csConnectIDList = new List<HySIdentifier>();
                                        csElementWork1.GetPortID(ref csFromID, ref csToID);
                                        csConnectIDList.Add(csFromID);
                                        csConnectIDList.Add(csToID);
                                        ((McGraphicsTransElement)csElementWork1).GetTransID(ref csFromID, ref csToID);
                                        csConnectIDList.Add(csFromID);
                                        csConnectIDList.Add(csToID);
                                        csConnectionIDList.Add(csConnectIDList);
                                        csTransPortList.Add(csElementWork1);
                                    }
                                    csDelElmList.Add(csElementWork2.CopyInfUsedByUndoRedoOnly(null));

                                    if (m_csPossessionElm == null)
                                    {// 全体系の場合
                                        m_csMcStructInfo.DeleteConnection(m_csLineList[iLp4].GetID());
                                    }
                                    else
                                    {// グループ内の場合
                                        ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).DeleteConnection(m_csLineList[iLp4].GetID());
                                    }
                                    csDelElmIDList.Add(m_csLineList[iLp4].GetID());
                                    csDelElmList.Add(m_csLineList[iLp4].CopyInfUsedByUndoRedoOnly(null));
                                    DelSelectionList((HySString)m_csLineList[iLp4].GetID().GetString());
                                    m_csLineList.RemoveAt(iLp4);
                                    break;
                                }
                            }
                            DelSelectionList((HySString)m_csElmList[iLp2].GetPort(iLp3).GetID().GetString());
                        }
                        if (m_csPossessionElm == null)
                        {// 全体系の場合
                            m_csMcStructInfo.RemoveElement(m_csElmList[iLp2].GetID());
                        }
                        else
                        {// グループ内の場合
                            ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).RemoveElement(m_csElmList[iLp2].GetID());
                        }
                        csDelElmIDList.Add(m_csElmList[iLp2].GetID());
                        csDelElmList.Add(m_csElmList[iLp2].CopyInfUsedByUndoRedoOnly(null));
                        if ((m_csElmList[iLp2].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP) ||
                            (m_csElmList[iLp2].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP))
                        {// グループ要素の場合
                            for (int iLp3 = 0; iLp3 < ((McGraphicsGroupElement)m_csElmList[iLp2]).GetLineElmCount(); iLp3++)
                            {// グループ内接続線数ループ
                                csDelElmIDList.Add(((McGraphicsGroupElement)m_csElmList[iLp2]).GetLineElement(iLp3).GetID());
                            }
                            DelGroupElementCheck((McGraphicsGroupElement)m_csElmList[iLp2], ref csDelElmIDList);
                        }
                        DelSelectionList((HySString)m_csElmList[iLp2].GetID().GetString());
                        m_csElmList.RemoveAt(iLp2);
                        iLp = -1;
                        break;
                    }
                }
            }

            return csDelElmIDList;
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ内削除要素確認</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DelGroupElementCheck(csGroupElement, ref csDelElmIDList)</para>
        /// </example>
        /// <param name="csGroupElement">グループ要素</param>
        /// <param name="csDelElmIDList">削除情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DelGroupElementCheck(McGraphicsGroupElement csGroupElement, ref List<HySIdentifier> csDelElmIDList)
        {
            for (int iLp = 0; iLp < csGroupElement.GetElmCount(); iLp++)
            {// グループ内要素数ループ
                csDelElmIDList.Add(csGroupElement.GetElement(iLp).GetID());
                if ((csGroupElement.GetElement(iLp).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP) ||
                    (csGroupElement.GetElement(iLp).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP))
                {// グループ要素の場合
                    for (int iLp2 = 0; iLp2 < ((McGraphicsGroupElement)csGroupElement.GetElement(iLp)).GetLineElmCount(); iLp2++)
                    {// グループ内接続線数ループ
                        csDelElmIDList.Add(((McGraphicsGroupElement)csGroupElement.GetElement(iLp)).GetLineElement(iLp2).GetID());
                    }
                    DelGroupElementCheck((McGraphicsGroupElement)csGroupElement.GetElement(iLp), ref csDelElmIDList);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素削除(中継端子用)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜HySIdentifier＞ csDelElmIDList = DelTransPort(csTopLineList, ref csDelElmList, ref csTransPortList, ref csConnectionIDList)</para>
        /// </example>
        /// <param name="csTopLineList">最上位接続線リスト</param>
        /// <param name="csDelElmList">削除情報リスト</param>
        /// <param name="csTransPortList">設定変更中継端子リスト</param>
        /// <param name="csConnectionIDList">設定変更前接続情報リスト</param>
        /// <returns>List＜HySIdentifier＞　削除情報IDリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public List<HySIdentifier> DelTransPort(List<McGraphicsElement> csTopLineList, ref List<McGraphicsElement> csDelElmList, ref List<McGraphicsElement> csTransPortList, ref List<List<HySIdentifier>> csConnectionIDList)
        {
            McGraphicsElement csElementWork1 = null;
            McGraphicsElement csElementWork2 = null;
            McGraphicsElement csTempPort2 = null;
            McGraphicsElement csParentElm = null;
            HySIdentifier csFromPortID = null;
            HySIdentifier csToPortID = null;
            HySIdentifier csNullID = new HySID("");
            List<HySIdentifier> csDelElmIDList = new List<HySIdentifier>();
            List<HySIdentifier> csConnectIDList = null;

            //グループ中継端子削除処理（グループ中継端子削除で中継端子・外接続線・外相手端子・内接続線・内相手端子を削除）
            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    if ((m_csPossessionElm == null) || (m_csPossessionElm.GetID().Equals(m_csSelectionList[iLp].GetParentElm().GetID()) != true))
                    {// グループ外部の場合
                        //中継端子削除
                        csDelElmList.Add(m_csSelectionList[iLp].CopyInfUsedByUndoRedoOnly(null));
                        m_csSelectionList[iLp].GetParentElm().DelPort(m_csSelectionList[iLp].GetID());
                        ReSetPortPosition(m_csSelectionList[iLp].GetParentElm());

                        ((McGraphicsTransElement)m_csSelectionList[iLp]).GetTransID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合
                            for (int iLp3 = 0; iLp3 < ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElmCount(); iLp3++)
                            {// グループ内接続線数ループ
                                ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).GetPortID(ref csFromPortID, ref csToPortID);
                                csElementWork1 = ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).GetPort(csFromPortID);
                                csElementWork2 = ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).GetPort(csToPortID);
                                if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                    (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                {// 両方IDが一致しない場合
                                    continue;
                                }
                                if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork2;
                                }
                                if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork1;
                                }

                                if ((csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                                    (csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子の場合
                                    //グループ内側接続相手中継端子クリア
                                    csConnectIDList = new List<HySIdentifier>();
                                    csTempPort2.GetPortID(ref csFromPortID, ref csToPortID);
                                    csConnectIDList.Add(csFromPortID);
                                    csConnectIDList.Add(csToPortID);
                                    ((McGraphicsTransElement)csTempPort2).GetTransID(ref csFromPortID, ref csToPortID);
                                    csConnectIDList.Add(csFromPortID);
                                    csConnectIDList.Add(csToPortID);
                                    csConnectionIDList.Add(csConnectIDList);
                                    csTransPortList.Add(csTempPort2);
                                }
                                else
                                {// 中継端子以外の場合
                                    //グループ内側接続相手端子削除
                                    csDelElmList.Add(csTempPort2.CopyInfUsedByUndoRedoOnly(null));
                                    csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                    ReSetPortPosition(csTempPort2.GetParentElm());
                                }

                                //グループ内側接続線削除
                                csDelElmIDList.Add(((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).GetID());
                                csDelElmList.Add(((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).CopyInfUsedByUndoRedoOnly(null));
                                ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).DelLineElement(iLp3);
                                break;
                            }
                        }

                        m_csSelectionList[iLp].GetPortID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合
                            for (int iLp3 = 0; iLp3 < m_csLineList.Count; iLp3++)
                            {// 接続線数ループ
                                m_csLineList[iLp3].GetPortID(ref csFromPortID, ref csToPortID);
                                csElementWork1 = m_csLineList[iLp3].GetPort(csFromPortID);
                                csElementWork2 = m_csLineList[iLp3].GetPort(csToPortID);
                                if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                    (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                {// 両方IDが一致しない場合
                                    continue;
                                }
                                if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork2;
                                }
                                if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork1;
                                }

                                if ((csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                                    (csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子の場合
                                    csConnectIDList = new List<HySIdentifier>();
                                    csTempPort2.GetPortID(ref csFromPortID, ref csToPortID);
                                    csConnectIDList.Add(csFromPortID);
                                    csConnectIDList.Add(csToPortID);
                                    ((McGraphicsTransElement)csTempPort2).GetTransID(ref csFromPortID, ref csToPortID);
                                    csConnectIDList.Add(csFromPortID);
                                    csConnectIDList.Add(csToPortID);
                                    csConnectionIDList.Add(csConnectIDList);
                                    csTransPortList.Add(csTempPort2);
                                }
                                else
                                {// 中継端子以外の場合
                                    //グループ外側接続相手端子削除
                                    csDelElmList.Add(csTempPort2.CopyInfUsedByUndoRedoOnly(null));
                                    csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                    ReSetPortPosition(csTempPort2.GetParentElm());
                                }

                                //グループ外側接続線削除
                                csDelElmIDList.Add(m_csLineList[iLp3].GetID());
                                csDelElmList.Add(m_csLineList[iLp3].CopyInfUsedByUndoRedoOnly(null));
                                m_csLineList.RemoveAt(iLp3);
                                break;
                            }
                        }

                        if (m_csPossessionElm == null)
                        {// 全体系の場合
                            ((McStructGroupElement)m_csSelectionList[iLp].GetParentElm().GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), m_csMcStructInfo);
                        }
                        else
                        {// グループ内の場合
                            ((McStructGroupElement)m_csSelectionList[iLp].GetParentElm().GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), (McStructGroupElement)m_csPossessionElm.GetViewerElm());
                        }
                        ((McStructGroupElement)m_csSelectionList[iLp].GetParentElm().GetViewerElm()).DelPort(m_csSelectionList[iLp].GetID());

                        DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                        iLp = -1;
                    }
                    else
                    {// グループ内部の場合
                        //中継端子削除
                        csDelElmList.Add(m_csSelectionList[iLp].CopyInfUsedByUndoRedoOnly(null));
                        m_csPossessionElm.DelPort(m_csSelectionList[iLp].GetID());
                        ReSetPortPosition(m_csPossessionElm);

                        ((McGraphicsTransElement)m_csSelectionList[iLp]).GetTransID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合
                            for (int iLp2 = 0; iLp2 < ((McGraphicsGroupElement)m_csPossessionElm).GetLineElmCount(); iLp2++)
                            {// グループ内接続線数ループ
                                ((McGraphicsGroupElement)m_csPossessionElm).GetLineElement(iLp2).GetPortID(ref csFromPortID, ref csToPortID);
                                csElementWork1 = ((McGraphicsGroupElement)m_csPossessionElm).GetLineElement(iLp2).GetPort(csFromPortID);
                                csElementWork2 = ((McGraphicsGroupElement)m_csPossessionElm).GetLineElement(iLp2).GetPort(csToPortID);
                                if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                    (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                {// 両方IDが一致しない場合
                                    continue;
                                }
                                if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork2;
                                }
                                if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork1;
                                }

                                if ((csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                                    (csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子の場合
                                    //グループ内側接続相手中継端子クリア
                                    csConnectIDList = new List<HySIdentifier>();
                                    csTempPort2.GetPortID(ref csFromPortID, ref csToPortID);
                                    csConnectIDList.Add(csFromPortID);
                                    csConnectIDList.Add(csToPortID);
                                    ((McGraphicsTransElement)csTempPort2).GetTransID(ref csFromPortID, ref csToPortID);
                                    csConnectIDList.Add(csFromPortID);
                                    csConnectIDList.Add(csToPortID);
                                    csConnectionIDList.Add(csConnectIDList);
                                    csTransPortList.Add(csTempPort2);
                                }
                                else
                                {// 中継端子以外の場合
                                    //グループ内側接続相手端子削除
                                    csDelElmList.Add(csTempPort2.CopyInfUsedByUndoRedoOnly(null));
                                    csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                    ReSetPortPosition(csTempPort2.GetParentElm());
                                }

                                //グループ内側接続線削除
                                csDelElmIDList.Add(((McGraphicsGroupElement)m_csPossessionElm).GetLineElement(iLp2).GetID());
                                csDelElmList.Add(((McGraphicsGroupElement)m_csPossessionElm).GetLineElement(iLp2).CopyInfUsedByUndoRedoOnly(null));
                                ((McGraphicsGroupElement)m_csPossessionElm).DelLineElement(iLp2);
                                break;
                            }
                        }

                        m_csSelectionList[iLp].GetPortID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合
                            csParentElm = m_csPossessionElm.GetParentElm();
                            if (csParentElm != null)
                            {// 親要素がある場合
                                for (int iLp2 = 0; iLp2 < ((McGraphicsGroupElement)csParentElm).GetLineElmCount(); iLp2++)
                                {// グループ内接続線数ループ
                                    ((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).GetPortID(ref csFromPortID, ref csToPortID);
                                    csElementWork1 = ((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).GetPort(csFromPortID);
                                    csElementWork2 = ((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).GetPort(csToPortID);
                                    if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                        (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                    {// 両方IDが一致しない場合
                                        continue;
                                    }
                                    if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork2;
                                    }
                                    if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork1;
                                    }

                                    if ((csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                                        (csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子の場合
                                        csConnectIDList = new List<HySIdentifier>();
                                        csTempPort2.GetPortID(ref csFromPortID, ref csToPortID);
                                        csConnectIDList.Add(csFromPortID);
                                        csConnectIDList.Add(csToPortID);
                                        ((McGraphicsTransElement)csTempPort2).GetTransID(ref csFromPortID, ref csToPortID);
                                        csConnectIDList.Add(csFromPortID);
                                        csConnectIDList.Add(csToPortID);
                                        csConnectionIDList.Add(csConnectIDList);
                                        csTransPortList.Add(csTempPort2);
                                    }
                                    else
                                    {// 中継端子以外の場合
                                        //グループ外側接続相手端子削除
                                        csDelElmList.Add(csTempPort2.CopyInfUsedByUndoRedoOnly(null));
                                        csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                        ReSetPortPosition(csTempPort2.GetParentElm());
                                    }

                                    //グループ外側接続線削除
                                    csDelElmIDList.Add(((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).GetID());
                                    csDelElmList.Add(((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).CopyInfUsedByUndoRedoOnly(null));
                                    ((McGraphicsGroupElement)csParentElm).DelLineElement(iLp2);
                                    break;
                                }
                            }
                            else
                            {// 親要素がない場合
                                for (int iLp2 = 0; iLp2 < csTopLineList.Count; iLp2++)
                                {// 全体系の接続線数ループ
                                    csTopLineList[iLp2].GetPortID(ref csFromPortID, ref csToPortID);
                                    csElementWork1 = csTopLineList[iLp2].GetPort(csFromPortID);
                                    csElementWork2 = csTopLineList[iLp2].GetPort(csToPortID);
                                    if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                        (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                    {// 両方IDが一致しない場合
                                        continue;
                                    }
                                    if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork2;
                                    }
                                    if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork1;
                                    }

                                    if ((csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                                        (csTempPort2.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子の場合
                                        //グループ外側接続相手中継端子クリア
                                        csConnectIDList = new List<HySIdentifier>();
                                        csTempPort2.GetPortID(ref csFromPortID, ref csToPortID);
                                        csConnectIDList.Add(csFromPortID);
                                        csConnectIDList.Add(csToPortID);
                                        ((McGraphicsTransElement)csTempPort2).GetTransID(ref csFromPortID, ref csToPortID);
                                        csConnectIDList.Add(csFromPortID);
                                        csConnectIDList.Add(csToPortID);
                                        csConnectionIDList.Add(csConnectIDList);
                                        csTransPortList.Add(csTempPort2);
                                    }
                                    else
                                    {// 中継端子以外の場合
                                        //グループ外側接続相手端子削除
                                        csDelElmList.Add(csTempPort2.CopyInfUsedByUndoRedoOnly(null));
                                        csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                        ReSetPortPosition(csTempPort2.GetParentElm());
                                    }

                                    //グループ外側接続線削除
                                    csDelElmIDList.Add(csTopLineList[iLp2].GetID());
                                    csDelElmList.Add(csTopLineList[iLp2].CopyInfUsedByUndoRedoOnly(null));
                                    csTopLineList.RemoveAt(iLp2);
                                    break;
                                }
                            }
                        }

                        if (m_csPossessionElm.GetParentElm() == null)
                        {// 全体系の場合
                            ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), m_csMcStructInfo);
                        }
                        else
                        {// グループ内の場合
                            ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), (McStructGroupElement)m_csPossessionElm.GetParentElm().GetViewerElm());
                        }
                        ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).DelPort(m_csSelectionList[iLp].GetID());

                        DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                        iLp = -1;
                    }
                }
            }

            return csDelElmIDList;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中中継端子接続確認</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = TransPortConnectionCheck()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>bool　true:接続情報無し　false:接続情報あり</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool TransPortConnectionCheck()
        {
            HySIdentifier csFromID=null;
            HySIdentifier csToID=null;
            HySIdentifier csNullID = new HySID("");

            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子以外の場合
                    continue;
                }

                m_csSelectionList[iLp].GetPortID(ref csFromID, ref csToID);
                if (m_csSelectionList[iLp].GetID().Equals(csFromID) == false)
                {// IDが一致しない場合
                    if (csFromID.Equals(csNullID) != true)
                    {// 接続情報がある場合
                        return false;
                    }
                }
                if (m_csSelectionList[iLp].GetID().Equals(csToID) == false)
                {// IDが一致しない場合
                    if (csToID.Equals(csNullID) != true)
                    {// 接続情報がある場合
                        return false;
                    }
                }

                ((McGraphicsTransElement)m_csSelectionList[iLp]).GetTransID(ref csFromID, ref csToID);
                if (m_csSelectionList[iLp].GetID().Equals(csFromID) == false)
                {// IDが一致しない場合
                    if (csFromID.Equals(csNullID) != true)
                    {// 接続情報がある場合
                        return false;
                    }
                }
                if (m_csSelectionList[iLp].GetID().Equals(csToID) == false)
                {// IDが一致しない場合
                    if (csToID.Equals(csNullID) != true)
                    {// 接続情報がある場合
                        return false;
                    }
                }
            }

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素回転</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ElementRotate(dRotate, ref csElmIDList)</para>
        /// </example>
        /// <param name="dRotate">回転角度</param>
        /// <param name="csElmIDList">回転要素IDリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ElementRotate(double dRotate, ref List<HySIdentifier> csElmIDList)
        {
            double dElmRotate;
            long lPortCount = 0;
            long lParentX = 0;
            long lParentY = 0;
            long lPortX = 0;
            long lPortY = 0;

            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_LINE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_SEND) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 接続線・端子・中継端子以外の場合
                    dElmRotate = m_csSelectionList[iLp].GetRotate();
                    csElmIDList.Add(m_csSelectionList[iLp].GetID());
                    dElmRotate = dElmRotate + dRotate;
                    if (dElmRotate < McDefine.HYM_DRAW_ANGLE_0)
                    {// 向きが０より小さい場合
                        dElmRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmRotate;
                    }
                    else if (dElmRotate >= McDefine.HYM_DRAW_ANGLE_360)
                    {// 向きが３６０以上の場合
                        dElmRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmRotate;
                    }
                    m_csSelectionList[iLp].SetRotate(dElmRotate);

                    lPortCount = m_csSelectionList[iLp].GetPortCount();

                    m_csSelectionList[iLp].GetPosition(ref lParentX, ref lParentY);

                    for (int iLp2 = 0; iLp2 < lPortCount; iLp2++)
                    {// 端子数ループ
                        m_csSelectionList[iLp].GetPort(iLp2).GetPosition(ref lPortX, ref lPortY);
                        McViewTransFormer.ViewRotateTrans(dRotate, lParentX, lParentY, ref lPortX, ref lPortY);
                        m_csSelectionList[iLp].GetPort(iLp2).SetPosition(lPortX, lPortY);
                        dElmRotate = m_csSelectionList[iLp].GetPort(iLp2).GetRotate() + dRotate;
                        if (dElmRotate < McDefine.HYM_DRAW_ANGLE_0)
                        {// 向きが０より小さい場合
                            dElmRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmRotate;
                        }
                        else if (dElmRotate >= McDefine.HYM_DRAW_ANGLE_360)
                        {// 向きが３６０以上の場合
                            dElmRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmRotate;
                        }
                        m_csSelectionList[iLp].GetPort(iLp2).SetRotate(dElmRotate);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>画面表示拡大率設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetDispScale(lDispScale)</para>
        /// </example>
        /// <param name="lDispScale">画面表示拡大率</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDispScale(long lDispScale)
        {
            m_lDispScale = lDispScale;
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中要素移動</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>MoveObject( MoveWidth, MoveHeight )</para>
        /// </example>
        /// <param name="MoveWidth">移動量(幅)</param>
        /// <param name="MoveHeight">移動量(高さ)</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void MoveObject(int MoveWidth, int MoveHeight)
        {
            long lPortCount = 0;
            long lElmX = 0;
            long lElmY = 0;
            long lPortX = 0;
            long lPortY = 0;

            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_LINE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_SEND) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 接続線・端子・中継端子以外の場合
                    m_csSelectionList[iLp].GetPosition(ref lElmX, ref lElmY);
                    lElmX = lElmX + MoveWidth;
                    lElmY = lElmY + MoveHeight;
                    m_csSelectionList[iLp].SetPosition(lElmX, lElmY);

                    lPortCount = m_csSelectionList[iLp].GetPortCount();
                    for (int iLp2 = 0; iLp2 < lPortCount; iLp2++)
                    {// 端子数ループ
                        m_csSelectionList[iLp].GetPort(iLp2).GetPosition(ref lPortX, ref lPortY);
                        lPortX = lPortX + MoveWidth;
                        lPortY = lPortY + MoveHeight;
                        m_csSelectionList[iLp].GetPort(iLp2).SetPosition(lPortX, lPortY);
                    }
                }
                if (m_csPossessionElm != null)
                {// グループ内の場合
                    if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                        (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 中継端子の場合
                        if (m_csPossessionElm.GetID().Equals(m_csSelectionList[iLp].GetParentElm().GetID()) == true)
                        {// IDが一致した場合
                            ((McGraphicsTransElement)m_csSelectionList[iLp]).GetInnerPosition(ref lPortX, ref lPortY);
                            lPortX = lPortX + MoveWidth;
                            lPortY = lPortY + MoveHeight;
                            ((McGraphicsTransElement)m_csSelectionList[iLp]).SetInnerPosition(lPortX, lPortY);
                        }
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>移動要素座標取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>MoveObjectPositionGet( csElementIDList, csPointList )</para>
        /// </example>
        /// <param name="csElementIDList">要素</param>
        /// <param name="csPointList">座標</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void MoveObjectPositionGet(ref List<HySIdentifier> csElementIDList, ref List<Point> csPointList)
        {
            long lPortCount = 0;
            long lElmX = 0;
            long lElmY = 0;
            long lPortX = 0;
            long lPortY = 0;

            csElementIDList.Clear();
            csPointList.Clear();

            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_LINE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_RECEIVE) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_SEND) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                    (m_csSelectionList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 接続線・端子・中継端子以外の場合
                    csElementIDList.Add(m_csSelectionList[iLp].GetID());
                    m_csSelectionList[iLp].GetPosition(ref lElmX, ref lElmY);
                    csPointList.Add(new Point((int)lElmX, (int)lElmY));

                    lPortCount = m_csSelectionList[iLp].GetPortCount();
                    for (int iLp2 = 0; iLp2 < lPortCount; iLp2++)
                    {// 端子数ループ
                        csElementIDList.Add(m_csSelectionList[iLp].GetPort(iLp2).GetID());
                        m_csSelectionList[iLp].GetPort(iLp2).GetPosition(ref lPortX, ref lPortY);
                        csPointList.Add(new Point((int)lPortX, (int)lPortY));
                    }
                }
                if (m_csPossessionElm != null)
                {// グループ内の場合
                    if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                        (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 中継端子の場合
                        if (m_csPossessionElm.GetID().Equals(m_csSelectionList[iLp].GetParentElm().GetID()) == true)
                        {// IDが一致した場合
                            csElementIDList.Add(m_csSelectionList[iLp].GetID());
                            ((McGraphicsTransElement)m_csSelectionList[iLp]).GetInnerPosition(ref lPortX, ref lPortY);
                            csPointList.Add(new Point((int)lPortX, (int)lPortY));
                        }
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>選択中端子移動</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>MovePortObject( iPortId, iPointX, iPointY )</para>
        /// </example>
        /// <param name="iPortId">選択端子ID</param>
        /// <param name="iPointX">移動位置X</param>
        /// <param name="iPointY">移動位置Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void MovePortObject(HySIdentifier iPortId, int iPointX, int iPointY)
        {
            long lPortCount = 0;
            long lParentX = 0;
            long lParentY = 0;
            long lParentWidth = 0;
            long lParentHeight = 0;
            long lPortX = 0;
            long lPortY = 0;
            long lPortWidth = 0;
            long lPortHeight = 0;
            double dParentRotete = 0;
            double dRotete = 0;

            for (int iLp1 = 0; iLp1 < m_csElmList.Count; iLp1++)
            {// 要素数ループ
                lPortCount = m_csElmList[iLp1].GetPortCount();
                for (int iLp2 = 0; iLp2 < lPortCount; iLp2++)
                {// 端子数ループ
                    if (m_csElmList[iLp1].GetPort(iLp2).Equal(iPortId) == true)
                    {// IDが一致した場合
                        m_csElmList[iLp1].GetPosition(ref lParentX, ref lParentY);
                        dParentRotete = m_csElmList[iLp1].GetRotate();
                        if ((dParentRotete == McDefine.HYM_DRAW_ANGLE_0) ||
                            (dParentRotete == McDefine.HYM_DRAW_ANGLE_180))
                        {// 向きが上下の場合
                            m_csElmList[iLp1].GetElementSize(ref lParentWidth, ref lParentHeight);
                        }
                        else
                        {// 向きが左右の場合
                            m_csElmList[iLp1].GetElementSize(ref lParentHeight, ref lParentWidth);
                        }
                        m_csElmList[iLp1].GetPort(iLp2).GetPosition(ref lPortX, ref lPortY);
                        m_csElmList[iLp1].GetPort(iLp2).GetElementSize(ref lPortWidth, ref lPortHeight);
                        dRotete = m_csElmList[iLp1].GetPort(iLp2).GetRotate();
                        McViewTransFormer.PortPositionMove(iPointX, iPointY, lParentX, lParentY, lParentWidth, lParentHeight, lPortHeight, ref lPortX, ref lPortY, ref dRotete);
                        m_csElmList[iLp1].GetPort(iLp2).SetPosition(lPortX, lPortY);
                        m_csElmList[iLp1].GetPort(iLp2).SetRotate(dRotete);
                        ReSetPortPosition(m_csElmList[iLp1]);
                        return;
                    }
                }
            }
            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp1 = 0; iLp1 < m_csPossessionElm.GetPortCount(); iLp1++)
                {// 端子数ループ
                    if (m_csPossessionElm.GetPort(iLp1).Equal(iPortId) == true)
                    {// IDが一致した場合
                        ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp1)).GetInnerPosition(ref lPortX, ref lPortY);
                        MoveObject(iPointX - (int)lPortX, iPointY - (int)lPortY);
                        break;
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントリサイズポイントクリックチェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ElementResizePointCheck(csElement, Position_x, Position_y)</para>
        /// </example>
        /// <param name="csElement">チェック要素</param>
        /// <param name="Position_x">チェック位置X</param>
        /// <param name="Position_y">チェック位置Y</param>
        /// <returns>bool　true：リサイズポイント　false：リサイズポイント外</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool ElementResizePointCheck(McGraphicsElement csElement, int Position_x, int Position_y)
        {
            if ((csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE) ||
                (csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                (csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_SEND) ||
                (csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                (csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
            {// 接続線・端子・中継端子の場合
                return false;
            }
            if ((csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT) &&
                (csElement.GetDrawSubKind() == McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE))
            {// マージ要素の場合
                return false;
            }

            int iResizePointSize = m_csCustomInfo.GetResizePointSize();

            return csElement.ElementResizePointCheck(Position_x, Position_y, m_lDispScale, iResizePointSize);
        }

        /// <summary><para>method outline:</para>
        /// <para>エレメントリサイズポイントクリックチェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphicsElement csElement = ElementResizePointCheck(Position_x, Position_y)</para>
        /// </example>
        /// <param name="Position_x">チェック位置X</param>
        /// <param name="Position_y">チェック位置Y</param>
        /// <returns>McGraphicsElement　指定位置該当エレメントID（該当無ければ null）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement ElementResizePointCheck(int Position_x, int Position_y)
        {
            McGraphicsElement csElement = null;
            bool bRtn = false;

            int iResizePointSize = m_csCustomInfo.GetResizePointSize();

            for (int iLp = m_csElmList.Count - 1; iLp >= 0; iLp--)
            {// 要素数ループ
                if ((m_csElmList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT) &&
                    (m_csElmList[iLp].GetDrawSubKind() == McDefine.HYM_DRAW_SUB_KIND_DATA_MERGE))
                {// マージ要素の場合
                    continue;
                }

                bRtn = m_csElmList[iLp].ElementResizePointCheck(Position_x, Position_y, m_lDispScale, iResizePointSize);
                if (bRtn == true)
                {// エレメントリサイズポイント該当の場合
                    csElement = m_csElmList[iLp];
                    break;
                }
            }

            return csElement;
        }

        /// <summary><para>method outline:</para>
        /// <para>要素リサイズ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ResizeObject(csID, iBeforeX, iBeforeY, iAfterX, iAfterY)</para>
        /// </example>
        /// <param name="csID">要素ID</param>
        /// <param name="iBeforeX">位置前X</param>
        /// <param name="iBeforeY">位置前Y</param>
        /// <param name="iAfterX">位置後X</param>
        /// <param name="iAfterY">位置後Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ResizeObject(HySIdentifier csID, int iBeforeX, int iBeforeY, int iAfterX, int iAfterY)
        {
            long lElementX = 0;
            long lElementY = 0;
            int iMoveX = iBeforeX - iAfterX;
            int iMoveY = iBeforeY - iAfterY;
            long lWidth = 0;
            long lHeight = 0;
            double dRotate = 0;
            long lTemp = 0;
            bool ReSizeFlg = false;
            McGraphicsElement csElement = null;

            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if (m_csElmList[iLp].Equal(csID) == true)
                {// IDが一致した場合
                    csElement = m_csElmList[iLp];
                    csElement.GetPosition(ref lElementX, ref lElementY);
                    dRotate = csElement.GetRotate();
                    if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                        (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                    {// 向きが上下の場合
                        csElement.GetElementSize(ref lWidth, ref lHeight);
                    }
                    else
                    {// 向きが左右の場合
                        csElement.GetElementSize(ref lHeight, ref lWidth);
                    }

                    if (lElementX > iBeforeX)
                    {// 左側操作の場合
                        if (iMoveX > 0)
                        {// 正方向移動の場合
                            //左方向X拡大
                            lTemp = lElementX + lWidth / 2;
                            if (lTemp != lElementX + (lWidth + iMoveX) / 2)
                            {
                                csElement.SetPosition(lTemp - (lWidth + iMoveX) / 2, lElementY);
                            }
                            if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                            {// 向きが上下の場合
                                csElement.SetElementSize(lWidth + iMoveX, lHeight);
                            }
                            else
                            {// 向きが左右の場合
                                csElement.SetElementSize(lHeight, lWidth + iMoveX);
                            }
                            ReSizeFlg = true;
                        }
                        else if (iMoveX < 0)
                        {// 負方向移動の場合
                            //左方向X縮小
                            if (1 < lWidth + iMoveX)
                            {
                                lTemp = lElementX + lWidth / 2;
                                if (lTemp != lElementX + (lWidth + iMoveX) / 2)
                                {
                                    csElement.SetPosition(lTemp - (lWidth + iMoveX) / 2, lElementY);
                                }
                                if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                    (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                                {// 向きが上下の場合
                                    csElement.SetElementSize(lWidth + iMoveX, lHeight);
                                }
                                else
                                {// 向きが左右の場合
                                    csElement.SetElementSize(lHeight, lWidth + iMoveX);
                                }
                                ReSizeFlg = true;
                            }
                        }
                    }
                    else
                    {// 右側操作の場合
                        if (iMoveX > 0)
                        {// 正方向移動の場合
                            //右方向X縮小
                            if (1 < lWidth - iMoveX)
                            {
                                lTemp = lElementX - lWidth / 2;
                                if (lTemp != lElementX - (lWidth - iMoveX) / 2)
                                {
                                    csElement.SetPosition(lTemp + (lWidth - iMoveX) / 2, lElementY);
                                }
                                if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                    (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                                {// 向きが上下の場合
                                    csElement.SetElementSize(lWidth - iMoveX, lHeight);
                                }
                                else
                                {// 向きが左右の場合
                                    csElement.SetElementSize(lHeight, lWidth - iMoveX);
                                }
                                ReSizeFlg = true;
                            }
                        }
                        else if (iMoveX < 0)
                        {// 負方向移動の場合
                            //右方向X拡大
                            lTemp = lElementX - lWidth / 2;
                            if (lTemp != lElementX - (lWidth - iMoveX) / 2)
                            {
                                csElement.SetPosition(lTemp + (lWidth - iMoveX) / 2, lElementY);
                            }
                            if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                            {// 向きが上下の場合
                                csElement.SetElementSize(lWidth - iMoveX, lHeight);
                            }
                            else
                            {// 向きが左右の場合
                                csElement.SetElementSize(lHeight, lWidth - iMoveX);
                            }
                            ReSizeFlg = true;
                        }
                    }

                    csElement.GetPosition(ref lElementX, ref lElementY);

                    if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                        (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                    {// 向きが上下の場合
                        csElement.GetElementSize(ref lWidth, ref lHeight);
                    }
                    else
                    {// 向きが左右の場合
                        csElement.GetElementSize(ref lHeight, ref lWidth);
                    }

                    if (lElementY > iBeforeY)
                    {// 上側操作の場合
                        if (iMoveY > 0)
                        {// 正方向移動の場合
                            //上方向Y拡大
                            lTemp = lElementY + lHeight / 2;
                            if (lTemp != lElementY + (lHeight + iMoveY) / 2)
                            {
                                csElement.SetPosition(lElementX, lTemp - (lHeight + iMoveY) / 2);
                            }
                            if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                            {// 向きが上下の場合
                                csElement.SetElementSize(lWidth, lHeight + iMoveY);
                            }
                            else
                            {// 向きが左右の場合
                                csElement.SetElementSize(lHeight + iMoveY, lWidth);
                            }
                            ReSizeFlg = true;
                        }
                        else if (iMoveY < 0)
                        {// 負方向移動の場合
                            //上方向Y縮小
                            if (1 < lHeight + iMoveY)
                            {
                                lTemp = lElementY + lHeight / 2;
                                if (lTemp != lElementY + (lHeight + iMoveY) / 2)
                                {
                                    csElement.SetPosition(lElementX, lTemp - (lHeight + iMoveY) / 2);
                                }
                                if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                    (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                                {// 向きが上下の場合
                                    csElement.SetElementSize(lWidth, lHeight + iMoveY);
                                }
                                else
                                {// 向きが左右の場合
                                    csElement.SetElementSize(lHeight + iMoveY, lWidth);
                                }
                                ReSizeFlg = true;
                            }
                        }
                    }
                    else
                    {// 下側操作の場合
                        if (iMoveY > 0)
                        {// 正方向移動の場合
                            //下方向Y縮小
                            if (1 < lHeight - iMoveY)
                            {
                                lTemp = lElementY - lHeight / 2;
                                if (lTemp != lElementY - (lHeight - iMoveY) / 2)
                                {
                                    csElement.SetPosition(lElementX, lTemp + (lHeight - iMoveY) / 2);
                                }
                                if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                    (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                                {// 向きが上下の場合
                                    csElement.SetElementSize(lWidth, lHeight - iMoveY);
                                }
                                else
                                {// 向きが左右の場合
                                    csElement.SetElementSize(lHeight - iMoveY, lWidth);
                                }
                                ReSizeFlg = true;
                            }
                        }
                        else if (iMoveY < 0)
                        {// 負方向移動の場合
                            //下方向Y拡大
                            lTemp = lElementY - lHeight / 2;
                            if (lTemp != lElementY - (lHeight - iMoveY) / 2)
                            {
                                csElement.SetPosition(lElementX, lTemp + (lHeight - iMoveY) / 2);
                            }
                            if ((dRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                                (dRotate == McDefine.HYM_DRAW_ANGLE_180))
                            {// 向きが上下の場合
                                csElement.SetElementSize(lWidth, lHeight - iMoveY);
                            }
                            else
                            {// 向きが左右の場合
                                csElement.SetElementSize(lHeight - iMoveY, lWidth);
                            }
                            ReSizeFlg = true;
                        }
                    }
                    break;
                }
            }
            if (ReSizeFlg == true)
            {// リサイズありの場合
                //端子位置移動
                ReSetPortPosition(csElement);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>端子位置再設定（間隔調整）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ReSetPortPosition(csElement)</para>
        /// </example>
        /// <param name="csElement">親要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ReSetPortPosition(McGraphicsElement csElement)
        {
            long lParentX = 0;
            long lParentY = 0;
            long lParentWidth = 0;
            long lParentHeight = 0;
            long lPortWidth = 0;
            long lPortHeight = 0;
            long lPortCount = 0;
            long lTempPortCount = 0;
            long lPortX = 0;
            long lPortY = 0;
            int lInsertRecNo = 0;
            double dParentRotate = 0;
            double dPortRotate = 0;

            McGraphicsElement csPort = null;
            List<McGraphicsElement> csTopList = new List<McGraphicsElement>();
            List<McGraphicsElement> csRightList = new List<McGraphicsElement>();
            List<McGraphicsElement> csLeftList = new List<McGraphicsElement>();
            List<McGraphicsElement> csBottomList = new List<McGraphicsElement>();
            List<long> lTopPartnerXList = new List<long>();
            List<long> lRightPartnerYList = new List<long>();
            List<long> lLeftPartnerYList = new List<long>();
            List<long> lBottomPartnerXList = new List<long>();

            csElement.GetPosition(ref lParentX, ref lParentY);
            dParentRotate = csElement.GetRotate();

            if ((dParentRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                (dParentRotate == McDefine.HYM_DRAW_ANGLE_180))
            {// 向きが上下の場合
                csElement.GetElementSize(ref lParentWidth, ref lParentHeight);
            }
            else
            {// 向きが左右の場合
                csElement.GetElementSize(ref lParentHeight, ref lParentWidth);
            }
            lPortCount = csElement.GetPortCount();

            for (int iLp1 = 0; iLp1 < lPortCount; iLp1++)
            {// 端子数ループ
                csPort = csElement.GetPort(iLp1);
                dPortRotate = csPort.GetRotate();
                csPort.GetElementSize(ref lPortWidth, ref lPortHeight);

                lPortX = 0;
                lPortY = 0;

                csPort.GetPosition(ref lPortX, ref lPortY);

                if (dPortRotate == McDefine.HYM_DRAW_ANGLE_0)
                {// 向きが上の場合
                    // 要素上部設置端子
                    lInsertRecNo = 0;
                    for (lInsertRecNo = 0; lInsertRecNo < lTopPartnerXList.Count; lInsertRecNo++)
                    {// 要素上部設置端子数ループ
                        if (lPortX < lTopPartnerXList[lInsertRecNo])
                        {
                            break;
                        }
                    }
                    lTopPartnerXList.Insert(lInsertRecNo, lPortX);
                    csTopList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csTopList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csTopList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
                else if (dPortRotate == McDefine.HYM_DRAW_ANGLE_90)
                {// 向きが右の場合
                    // 要素右側設置端子
                    lInsertRecNo = 0;
                    for (lInsertRecNo = 0; lInsertRecNo < lRightPartnerYList.Count; lInsertRecNo++)
                    {// 要素右側設置端子数ループ
                        if (lPortY < lRightPartnerYList[lInsertRecNo])
                        {
                            break;
                        }
                    }
                    lRightPartnerYList.Insert(lInsertRecNo, lPortY);
                    csRightList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csRightList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csRightList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
                else if (dPortRotate == McDefine.HYM_DRAW_ANGLE_180)
                {// 向きが下の場合
                    // 要素下部設置端子
                    lInsertRecNo = 0;
                    for (lInsertRecNo = 0; lInsertRecNo < lBottomPartnerXList.Count; lInsertRecNo++)
                    {// 要素下部設置端子数ループ
                        if (lPortX < lBottomPartnerXList[lInsertRecNo])
                        {
                            break;
                        }
                    }
                    lBottomPartnerXList.Insert(lInsertRecNo, lPortX);
                    csBottomList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csBottomList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csBottomList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
                else if (dPortRotate == McDefine.HYM_DRAW_ANGLE_270)
                {// 向きが左の場合
                    // 要素左側設置端子
                    lInsertRecNo = 0;
                    for (lInsertRecNo = 0; lInsertRecNo < lLeftPartnerYList.Count; lInsertRecNo++)
                    {// 要素左側設置端子数ループ
                        if (lPortY < lLeftPartnerYList[lInsertRecNo])
                        {
                            break;
                        }
                    }
                    lLeftPartnerYList.Insert(lInsertRecNo, lPortY);
                    csLeftList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csLeftList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csLeftList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>端子位置再設定（端子追加間隔調整）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ReSetPortPosition(csElement, csAddPort)</para>
        /// </example>
        /// <param name="csElement">親要素</param>
        /// <param name="csAddPort">追加ポート</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReSetPortPosition(McGraphicsElement csElement, McGraphicsElement csAddPort)
        {
            long lParentX = 0;
            long lParentY = 0;
            long lParentWidth = 0;
            long lParentHeight = 0;
            long lPortWidth = 0;
            long lPortHeight = 0;
            long lPortCount = 0;
            long lTempPortCount = 0;
            long lTempPartnerX = 0;
            long lTempPartnerY = 0;
            long lPortX = 0;
            long lPortY = 0;
            int lInsertRecNo = 0;
            double dParentRotate = 0;
            double dPortRotate = 0;

            McGraphicsElement csPort = null;
            List<McGraphicsElement> csPortList = new List<McGraphicsElement>();
            List<long> lPortXYList = new List<long>();
            HySIdentifier csFromPortID = null;
            HySIdentifier csToPortID = null;
            HySIdentifier csPartnerPortID = null;

            csElement.GetPosition(ref lParentX, ref lParentY);
            dParentRotate = csElement.GetRotate();

            if ((dParentRotate == McDefine.HYM_DRAW_ANGLE_0) ||
                (dParentRotate == McDefine.HYM_DRAW_ANGLE_180))
            {// 向きが上下の場合
                csElement.GetElementSize(ref lParentWidth, ref lParentHeight);
            }
            else
            {// 向きが左右の場合
                csElement.GetElementSize(ref lParentHeight, ref lParentWidth);
            }
            dPortRotate = csAddPort.GetRotate();
            csAddPort.GetPortID(ref csFromPortID, ref csToPortID);

            if (csFromPortID.Equals(csAddPort.GetID()) == true)
            {// IDが一致した場合
                csPartnerPortID = csToPortID;
            }
            else
            {// IDが一致しない場合
                csPartnerPortID = csFromPortID;
            }

            lTempPartnerX = 0;
            lTempPartnerY = 0;

            for (int iLp1 = 0; iLp1 < m_csElmList.Count; iLp1++)
            {// 要素数ループ
                if (m_csElmList[iLp1].GetPort(csPartnerPortID) != null)
                {// 相手端子があった場合
                    m_csElmList[iLp1].GetPosition(ref lTempPartnerX, ref lTempPartnerY);
                    break;
                }
            }

            lPortCount = csElement.GetPortCount();

            for (int iLp1 = 0; iLp1 < lPortCount; iLp1++)
            {// 端子数ループ
                csPort = csElement.GetPort(iLp1);
                if (dPortRotate != csPort.GetRotate())
                {// 追加端子向きと違う場合
                    continue;
                }
                csPort.GetElementSize(ref lPortWidth, ref lPortHeight);

                lPortX = 0;
                lPortY = 0;

                csPort.GetPosition(ref lPortX, ref lPortY);

                if (dPortRotate == McDefine.HYM_DRAW_ANGLE_0)
                {// 向きが上の場合
                    // 要素上部設置端子
                    lInsertRecNo = 0;
                    if (csPort.Equal(csAddPort.GetID()) == true)
                    {// IDが一致した場合
                        if (lTempPartnerX > lParentX)
                        {
                            lInsertRecNo = lPortXYList.Count;
                        }
                    }
                    else
                    {// IDが一致しない場合
                        for (lInsertRecNo = 0; lInsertRecNo < lPortXYList.Count; lInsertRecNo++)
                        {
                            if (lPortX < lPortXYList[lInsertRecNo])
                            {
                                break;
                            }
                        }
                    }
                    lPortXYList.Insert(lInsertRecNo, lPortX);
                    csPortList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csPortList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csPortList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
                else if (dPortRotate == McDefine.HYM_DRAW_ANGLE_90)
                {// 向きが右の場合
                    // 要素右側設置端子
                    lInsertRecNo = 0;
                    if (csPort.Equal(csAddPort.GetID()) == true)
                    {// IDが一致した場合
                        if (lTempPartnerY > lParentY)
                        {
                            lInsertRecNo = lPortXYList.Count;
                        }
                    }
                    else
                    {// IDが一致しない場合
                        for (lInsertRecNo = 0; lInsertRecNo < lPortXYList.Count; lInsertRecNo++)
                        {
                            if (lPortY < lPortXYList[lInsertRecNo])
                            {
                                break;
                            }
                        }
                    }
                    lPortXYList.Insert(lInsertRecNo, lPortY);
                    csPortList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csPortList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csPortList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
                else if (dPortRotate == McDefine.HYM_DRAW_ANGLE_180)
                {// 向きが下の場合
                    // 要素下部設置端子
                    lInsertRecNo = 0;
                    if (csPort.Equal(csAddPort.GetID()) == true)
                    {// IDが一致した場合
                        if (lTempPartnerX > lParentX)
                        {
                            lInsertRecNo = lPortXYList.Count;
                        }
                    }
                    else
                    {// IDが一致しない場合
                        for (lInsertRecNo = 0; lInsertRecNo < lPortXYList.Count; lInsertRecNo++)
                        {
                            if (lPortX < lPortXYList[lInsertRecNo])
                            {
                                break;
                            }
                        }
                    }
                    lPortXYList.Insert(lInsertRecNo, lPortX);
                    csPortList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csPortList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csPortList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
                else if (dPortRotate == McDefine.HYM_DRAW_ANGLE_270)
                {// 向きが左の場合
                    // 要素左側設置端子
                    lInsertRecNo = 0;
                    if (csPort.Equal(csAddPort.GetID()) == true)
                    {// IDが一致した場合
                        if (lTempPartnerY > lParentY)
                        {
                            lInsertRecNo = lPortXYList.Count;
                        }
                    }
                    else
                    {// IDが一致しない場合
                        for (lInsertRecNo = 0; lInsertRecNo < lPortXYList.Count; lInsertRecNo++)
                        {
                            if (lPortY < lPortXYList[lInsertRecNo])
                            {
                                break;
                            }
                        }
                    }
                    lPortXYList.Insert(lInsertRecNo, lPortY);
                    csPortList.Insert(lInsertRecNo, csPort);
                    lTempPortCount = csPortList.Count;
                    for (int iLp2 = 0; iLp2 < lTempPortCount; iLp2++)
                    {
                        McViewTransFormer.PortPotisionShift(lParentX, lParentY, lParentWidth, lParentHeight, dPortRotate, lPortHeight, lTempPortCount, iLp2, ref lPortX, ref lPortY);
                        csPortList[iLp2].SetPosition(lPortX, lPortY);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>表示順最前面設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetPiledUpTop(ref csIDList, ref csBeforePositionList, ref csAfterPositionList)</para>
        /// </example>
        /// <param name="csIDList">要素ID</param>
        /// <param name="csBeforePositionList">移動前位置</param>
        /// <param name="csAfterPositionList">移動後位置</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetPiledUpTop(ref List<HySIdentifier> csIDList, ref List<int> csBeforePositionList, ref List<int> csAfterPositionList)
        {
            List<McGraphicsElement> csTempElmList = new List<McGraphicsElement>();
            for (int iLp1 = 0; iLp1 < m_csSelectionList.Count; iLp1++)
            {// 選択要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp1].GetID().Equals(m_csElmList[iLp2].GetID()) == true)
                    {// IDが一致した場合
                        csIDList.Add(m_csElmList[iLp2].GetID());
                        csBeforePositionList.Add(iLp2);
                    }
                }
            }
            for (int iLp1 = 0; iLp1 < m_csSelectionList.Count; iLp1++)
            {// 選択要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp1].GetID().Equals(m_csElmList[iLp2].GetID()) == true)
                    {// IDが一致した場合
                        csTempElmList.Add(m_csElmList[iLp2]);
                        m_csElmList.RemoveAt(iLp2);
                    }
                }
            }
            if (csTempElmList.Count > 0)
            {// テンポラリ移動要素ありの場合
                for (int iLp1 = 0; iLp1 < csTempElmList.Count; iLp1++)
                {// テンポラリ移動要素数ループ
                    m_csElmList.Add(csTempElmList[iLp1]);
                }
            }
            for (int iLp1 = 0; iLp1 < m_csSelectionList.Count; iLp1++)
            {// 選択要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp1].GetID().Equals(m_csElmList[iLp2].GetID()) == true)
                    {// IDが一致した場合
                        csAfterPositionList.Add(iLp2);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>表示順最背面設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetPiledUpBottom(ref csIDList, ref csBeforePositionList, ref csAfterPositionList)</para>
        /// </example>
        /// <param name="csIDList">要素ID</param>
        /// <param name="csBeforePositionList">移動前位置</param>
        /// <param name="csAfterPositionList">移動後位置</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetPiledUpBottom(ref List<HySIdentifier> csIDList, ref List<int> csBeforePositionList, ref List<int> csAfterPositionList)
        {
            List<McGraphicsElement> csTempElmList = new List<McGraphicsElement>();
            for (int iLp1 = m_csSelectionList.Count - 1; iLp1 >= 0; iLp1--)
            {// 選択要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp1].GetID().Equals(m_csElmList[iLp2].GetID()) == true)
                    {// IDが一致した場合
                        csIDList.Add(m_csElmList[iLp2].GetID());
                        csBeforePositionList.Add(iLp2);
                    }
                }
            }
            for (int iLp1 = m_csSelectionList.Count - 1; iLp1 >= 0; iLp1--)
            {// 選択要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp1].GetID().Equals(m_csElmList[iLp2].GetID()) == true)
                    {// IDが一致した場合
                        csTempElmList.Add(m_csElmList[iLp2]);
                        m_csElmList.RemoveAt(iLp2);
                    }
                }
            }
            if (csTempElmList.Count > 0)
            {// テンポラリ移動要素ありの場合
                for (int iLp1 = 0; iLp1 < csTempElmList.Count; iLp1++)
                {// テンポラリ移動要素数ループ
                    m_csElmList.Insert(0, csTempElmList[iLp1]);
                }
            }
            for (int iLp1 = m_csSelectionList.Count - 1; iLp1 >= 0; iLp1--)
            {// 選択要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp1].GetID().Equals(m_csElmList[iLp2].GetID()) == true)
                    {// IDが一致した場合
                        csAfterPositionList.Add(iLp2);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>範囲指定要素選択</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>RangeSelect(iFromX, iFromY, iToX, iToY)</para>
        /// </example>
        /// <param name="iFromX">指定範囲（左上X）</param>
        /// <param name="iFromY">指定範囲（左上Y）</param>
        /// <param name="iToX">指定範囲（右下X）</param>
        /// <param name="iToY">指定範囲（右下Y）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void RangeSelect(int iFromX, int iFromY, int iToX, int iToY)
        {
            for (int iLp1 = 0; iLp1 < m_csElmList.Count; iLp1++)
            {// 要素数ループ
                if (SearchSelectionList((HySString)m_csElmList[iLp1].GetID().GetString()) != -1)
                {// 選択されている場合
                    continue;
                }

                //描画サイズが選択範囲に入っているもののみ選択対象とする。
                if (m_csElmList[iLp1].SelectedRangeCheck(iFromX, iFromY, iToX, iToY, m_lDispScale) == true)
                {// 選択範囲該当の場合
                    AddSelectionList(m_csElmList[iLp1]);
                }
            }

            for (int iLp1 = 0; iLp1 < m_csLineList.Count; iLp1++)
            {// 接続線数ループ
                if (SearchSelectionList((HySString)m_csLineList[iLp1].GetID().GetString()) != -1)
                {// 選択されている場合
                    continue;
                }

                //描画サイズが選択範囲に入っているもののみ選択対象とする。
                if (m_csLineList[iLp1].SelectedRangeCheck(iFromX, iFromY, iToX, iToY, m_lDispScale) == true)
                {// 選択範囲該当の場合
                    AddSelectionList(m_csLineList[iLp1]);
                }
            }

            for (int iLp1 = 0; iLp1 < m_csElmList.Count; iLp1++)
            {// 要素数ループ
                for (int iLp2 = 0; iLp2 < m_csElmList[iLp1].GetPortCount(); iLp2++)
                {// 端子数ループ
                    if (SearchSelectionList((HySString)m_csElmList[iLp1].GetPort(iLp2).GetID().GetString()) != -1)
                    {// 選択されている場合
                        continue;
                    }

                    if ((m_csElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                        (m_csElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループ要素の場合
                        //描画サイズが選択範囲に入っているもののみ選択対象とする。
                        if (((McGraphicsTransElement)m_csElmList[iLp1].GetPort(iLp2)).SelectedRangeCheck(iFromX, iFromY, iToX, iToY, m_lDispScale, false) == true)
                        {// 選択範囲該当の場合
                            AddSelectionList(m_csElmList[iLp1].GetPort(iLp2));
                        }
                    }
                    else
                    {// グループ要素以外の場合
                        //描画サイズが選択範囲に入っているもののみ選択対象とする。
                        if (m_csElmList[iLp1].GetPort(iLp2).SelectedRangeCheck(iFromX, iFromY, iToX, iToY, m_lDispScale) == true)
                        {// 選択範囲該当の場合
                            AddSelectionList(m_csElmList[iLp1].GetPort(iLp2));
                        }
                    }
                }
            }

            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    if (SearchSelectionList((HySString)m_csPossessionElm.GetPort(iLp).GetID().GetString()) != -1)
                    {// 選択されている場合
                        continue;
                    }

                    //描画サイズが選択範囲に入っているもののみ選択対象とする。
                    if (((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp)).SelectedRangeCheck(iFromX, iFromY, iToX, iToY, m_lDispScale, true) == true)
                    {// 選択範囲該当の場合
                        AddSelectionList(m_csPossessionElm.GetPort(iLp));
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ解除時のグループ内要素座標設定及び他要素の座標シフト</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ReSetElementPosition(csGroupElement, ref csMoveElmIDList, ref csBeforePointList, ref csAfterPointList)</para>
        /// </example>
        /// <param name="csGroupElement">グループ要素</param>
        /// <param name="csMoveElmIDList">移動要素ID</param>
        /// <param name="csBeforePointList">移動前座標</param>
        /// <param name="csAfterPointList">移動後座標</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReSetElementPosition(McGraphicsGroupElement csGroupElement, ref List<HySIdentifier> csMoveElmIDList, ref List<Point> csBeforePointList, ref List<Point> csAfterPointList)
        {
            long lMemberLeftX = 0;
            long lMemberTopY = 0;
            long lMemberRightX = 0;
            long lMemberBottomY = 0;
            long lGroupLeftX = 0;
            long lGroupTopY = 0;
            long lGroupRightX = 0;
            long lGroupBottomY = 0;
            long lGroupWidth = 0;
            long lGroupHeight = 0;
            long lGroupElmX = 0;
            long lGroupElmY = 0;
            long lElmX = 0;
            long lElmY = 0;
            long lElmWidth = 0;
            long lElmHeight = 0;
            long lShiftX = 0;
            long lShiftY = 0;
            long lPortX = 0;
            long lPortY = 0;
            bool bPiledUpFlg = false;
            
            if (csGroupElement.GetElmCount() == 0)
            {// グループ内要素数が０の場合
                return;
            }

            McViewTransFormer.GroupMemberPosiCheck(csGroupElement, ref lMemberLeftX, ref lMemberTopY, ref lMemberRightX, ref lMemberBottomY);

            csGroupElement.GetPosition(ref lGroupElmX, ref lGroupElmY);

            if ((csGroupElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_0) ||
                (csGroupElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_180))
            {// 向きが上下の場合
                csGroupElement.GetElementSize(ref lGroupWidth, ref lGroupHeight);
            }
            else
            {// 向きが左右の場合
                csGroupElement.GetElementSize(ref lGroupHeight, ref lGroupWidth);
            }

            lGroupLeftX = lGroupElmX - lGroupWidth / 2;
            lGroupTopY = lGroupElmY - lGroupHeight / 2;
            lGroupRightX = lGroupElmX + lGroupWidth / 2;
            lGroupBottomY = lGroupElmY + lGroupHeight / 2;

            // グループ内要素位置設定
            for (int iLp1 = 0; iLp1 < csGroupElement.GetElmCount(); iLp1++)
            {// グループ内要素数ループ
                csGroupElement.GetElement(iLp1).GetPosition(ref lElmX, ref lElmY);
                csGroupElement.GetElement(iLp1).SetPosition(lGroupLeftX + lElmX - lMemberLeftX, lGroupTopY + lElmY - lMemberTopY);

                for (int iLp2 = 0; iLp2 < csGroupElement.GetElement(iLp1).GetPortCount(); iLp2++)
                {// 端子数ループ
                    csGroupElement.GetElement(iLp1).GetPort(iLp2).GetPosition(ref lElmX, ref lElmY);
                    csGroupElement.GetElement(iLp1).GetPort(iLp2).SetPosition(lGroupLeftX + lElmX - lMemberLeftX, lGroupTopY + lElmY - lMemberTopY);
                }
            }

            lShiftX = lMemberRightX - lMemberLeftX - lGroupWidth;
            lShiftY = lMemberBottomY - lMemberTopY - lGroupHeight;

            //グループ解除にて他要素に重なる要素がある場合は解除するグループ要素より右または下方向にある他要素を右または下方向にずらす
            for (int iLp1 = 0; iLp1 < m_csElmList.Count; iLp1++)
            {// 要素数ループ
                if (m_csElmList[iLp1].Equal(csGroupElement.GetID()) == true)
                {// IDが一致した場合
                    continue;
                }

                m_csElmList[iLp1].GetPosition(ref lElmX, ref lElmY);

                if ((m_csElmList[iLp1].GetRotate() == McDefine.HYM_DRAW_ANGLE_0) ||
                    (m_csElmList[iLp1].GetRotate() == McDefine.HYM_DRAW_ANGLE_180))
                {// 向きが上下の場合
                    m_csElmList[iLp1].GetElementSize(ref lElmWidth, ref lElmHeight);
                }
                else
                {// 向きが左右の場合
                    m_csElmList[iLp1].GetElementSize(ref lElmHeight, ref lElmWidth);
                }

                if (((lGroupLeftX <= lElmX - lElmWidth / 2) && (lGroupLeftX + lMemberRightX - lMemberLeftX >= lElmX - lElmWidth / 2))&&
                    ((lGroupTopY <= lElmY - lElmHeight / 2) && (lGroupTopY + lMemberBottomY - lMemberTopY >= lElmY - lElmHeight / 2)))
                {// シフト対象範囲該当の場合
                    bPiledUpFlg = true;
                    break;
                }
            }
            if ((m_csPossessionElm != null) && (bPiledUpFlg == false))
            {// グループ内かつシフト対象無しの場合
                for (int iLp1 = 0; iLp1 < m_csPossessionElm.GetPortCount(); iLp1++)
                {// 端子数ループ
                    ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp1)).GetInnerPosition(ref lElmX, ref lElmY);

                    m_csPossessionElm.GetPort(iLp1).GetElementSize(ref lElmWidth, ref lElmHeight);

                    if (((lGroupLeftX <= lElmX - lElmWidth / 2) && (lGroupLeftX + lMemberRightX - lMemberLeftX >= lElmX - lElmWidth / 2)) &&
                        ((lGroupTopY <= lElmY - lElmHeight / 2) && (lGroupTopY + lMemberBottomY - lMemberTopY >= lElmY - lElmHeight / 2)))
                    {// シフト対象範囲該当の場合
                        bPiledUpFlg = true;
                        break;
                    }
                }
            }

            if (bPiledUpFlg == true)
            {// シフト対象ありの場合
                //解除するグループ要素より右または下方向にある要素をずらす
                for (int iLp1 = 0; iLp1 < m_csElmList.Count; iLp1++)
                {// 要素数ループ
                    if (m_csElmList[iLp1].Equal(csGroupElement.GetID()) == true)
                    {// IDが一致した場合
                        continue;
                    }

                    m_csElmList[iLp1].GetPosition(ref lElmX, ref lElmY);

                    if ((m_csElmList[iLp1].GetRotate() == McDefine.HYM_DRAW_ANGLE_0) ||
                        (m_csElmList[iLp1].GetRotate() == McDefine.HYM_DRAW_ANGLE_180))
                    {// 向きが上下の場合
                        m_csElmList[iLp1].GetElementSize(ref lElmWidth, ref lElmHeight);
                    }
                    else
                    {// 向きが左右の場合
                        m_csElmList[iLp1].GetElementSize(ref lElmHeight, ref lElmWidth);
                    }

                    //右方向要素シフト
                    if (lGroupLeftX <= lElmX - lElmWidth / 2)
                    {// 位置が右方向の場合
                        csMoveElmIDList.Add(m_csElmList[iLp1].GetID());
                        csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                        csAfterPointList.Add(new Point((int)lElmX + (int)lShiftX, (int)lElmY));
                        m_csElmList[iLp1].SetPosition(lElmX + lShiftX, lElmY);
                        lElmX = lElmX + lShiftX;

                        for (int iLp2 = 0; iLp2 < m_csElmList[iLp1].GetPortCount(); iLp2++)
                        {// 端子数ループ
                            m_csElmList[iLp1].GetPort(iLp2).GetPosition(ref lPortX, ref lPortY);
                            csMoveElmIDList.Add(m_csElmList[iLp1].GetPort(iLp2).GetID());
                            csBeforePointList.Add(new Point((int)lPortX, (int)lPortY));
                            csAfterPointList.Add(new Point((int)lPortX + (int)lShiftX, (int)lPortY));
                            m_csElmList[iLp1].GetPort(iLp2).SetPosition(lPortX + lShiftX, lPortY);
                        }
                    }

                    m_csElmList[iLp1].GetPosition(ref lElmX, ref lElmY);

                    //下方向要素シフト
                    if (lGroupTopY <= lElmY - lElmHeight / 2)
                    {// 位置が下方向の場合
                        csMoveElmIDList.Add(m_csElmList[iLp1].GetID());
                        csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                        csAfterPointList.Add(new Point((int)lElmX, (int)lElmY + (int)lShiftY));
                        m_csElmList[iLp1].SetPosition(lElmX, lElmY + lShiftY);
                        lElmY = lElmY + lShiftY;

                        for (int iLp2 = 0; iLp2 < m_csElmList[iLp1].GetPortCount(); iLp2++)
                        {// 端子数ループ
                            m_csElmList[iLp1].GetPort(iLp2).GetPosition(ref lPortX, ref lPortY);
                            csMoveElmIDList.Add(m_csElmList[iLp1].GetPort(iLp2).GetID());
                            csBeforePointList.Add(new Point((int)lPortX, (int)lPortY));
                            csAfterPointList.Add(new Point((int)lPortX, (int)lPortY + (int)lShiftY));
                            m_csElmList[iLp1].GetPort(iLp2).SetPosition(lPortX, lPortY + lShiftY);
                        }
                    }
                }
                if (m_csPossessionElm != null)
                {// グループ内の場合
                    for (int iLp1 = 0; iLp1 < m_csPossessionElm.GetPortCount(); iLp1++)
                    {// 端子数ループ
                        ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp1)).GetInnerPosition(ref lElmX, ref lElmY);

                        m_csPossessionElm.GetPort(iLp1).GetElementSize(ref lElmWidth, ref lElmHeight);

                        //右方向要素シフト
                        if (lGroupLeftX <= lElmX - lElmWidth / 2)
                        {// 位置が右方向の場合
                            csMoveElmIDList.Add(m_csPossessionElm.GetPort(iLp1).GetID());
                            csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                            csAfterPointList.Add(new Point((int)lElmX + (int)lShiftX, (int)lElmY));
                            ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp1)).SetInnerPosition(lElmX + lShiftX, lElmY);
                        }

                        ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp1)).GetInnerPosition(ref lElmX, ref lElmY);

                        //下方向要素シフト
                        if (lGroupTopY <= lElmY - lElmHeight / 2)
                        {// 位置が下方向の場合
                            csMoveElmIDList.Add(m_csPossessionElm.GetPort(iLp1).GetID());
                            csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                            csAfterPointList.Add(new Point((int)lElmX, (int)lElmY + (int)lShiftY));
                            ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp1)).SetInnerPosition(lElmX, lElmY + lShiftY);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>選択情報内グループ要素取り出し</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McGraphicsElement csGraphicsElement = GetGroupElement()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>McGraphicsElement　グループ要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement GetGroupElement()
        {
            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                    (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                {// グループ要素の場合
                    return m_csSelectionList[iLp];
                }
            }

            return null;
        }

        /// <summary><para>method outline:</para>
        /// <para>要素表示範囲取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = GetDispRect(pPointLeftTop,pPointRightBottom)</para>
        /// </example>
        /// <param name="pPointLeftTop">表示座標左上</param>
        /// <param name="pPointRightBottom">表示座標右下</param>
        /// <returns>long  0:正常、-1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetDispRect(ref Point pPointLeftTop, ref Point pPointRightBottom)
        {
            long lRtn = -1;                     // リターンコード
            long lTempWidth = 0;                // 要素幅
            long lTempHeight = 0;               // 要素高さ
            long lTempX = 0;                    // 要素座標X
            long lTempY = 0;                    // 要素座標Y

            long lMemberLeftX = 0;              // 左端座標X
            long lMemberTopY = 0;               // 上端座標Y
            long lMemberRightX = 0;             // 右端座標X
            long lMemberBottomY = 0;            // 下端座標Y

            McGraphicsElement csTmpElement=null;// 描画要素クラス

            if (m_csElmList.Count > 0)
            {   // 要素有り
                csTmpElement = m_csElmList[0];
                csTmpElement.GetPosition(ref lTempX, ref lTempY);
                if ((csTmpElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_0) ||
                    (csTmpElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_180))
                {// 向きが上下の場合
                    csTmpElement.GetElementSize(ref lTempWidth, ref lTempHeight);
                }
                else
                {// 向きが左右の場合
                    csTmpElement.GetElementSize(ref lTempHeight, ref lTempWidth);
                }
                lMemberLeftX = lTempX - lTempWidth / 2;
                lMemberTopY = lTempY - lTempHeight / 2;
                lMemberRightX = lTempX + lTempWidth / 2;
                lMemberBottomY = lTempY + lTempHeight / 2;

                for (int iLoop = 1; iLoop < m_csElmList.Count; iLoop++)
                {// プロジェクト要素メンバー数ループ
                    csTmpElement = m_csElmList[iLoop];
                    csTmpElement.GetPosition(ref lTempX, ref lTempY);
                    if ((csTmpElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_0) ||
                        (csTmpElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_180))
                    {// 向きが上下の場合
                        csTmpElement.GetElementSize(ref lTempWidth, ref lTempHeight);
                    }
                    else
                    {// 向きが左右の場合
                        csTmpElement.GetElementSize(ref lTempHeight, ref lTempWidth);
                    }

                    if (lMemberLeftX > lTempX - lTempWidth / 2)
                    {// 最も左に位置する場合
                        lMemberLeftX = lTempX - lTempWidth / 2;
                    }
                    if (lMemberTopY > lTempY - lTempHeight / 2)
                    {// 最も上に位置する場合
                        lMemberTopY = lTempY - lTempHeight / 2;
                    }
                    if (lMemberRightX < lTempX + lTempWidth / 2)
                    {// 最も右に位置する場合
                        lMemberRightX = lTempX + lTempWidth / 2;
                    }
                    if (lMemberBottomY < lTempY + lTempHeight / 2)
                    {// 最も下に位置する場合
                        lMemberBottomY = lTempY + lTempHeight / 2;
                    }
                }
                pPointLeftTop.X = (int)lMemberLeftX;
                pPointLeftTop.Y = (int)lMemberTopY;
                pPointRightBottom.X = (int)lMemberRightX;
                pPointRightBottom.Y = (int)lMemberBottomY;
                lRtn = 0;
            }
            return lRtn;
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>接続要素チェック</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>bool bCheckFlg = ConnectionElementCheck(csFromChkID, csToChkID, csFromPort, csToPort)</para>
        /// </example>
        /// <param name="csFromChkID">接続元要素ID</param>
        /// <param name="csToChkID">接続先要素ID</param>
        /// <param name="csFromPort">接続元端子</param>
        /// <param name="csToPort">接続先端子</param>
        /// <param name="csFromElm">接続元親要素</param>
        /// <param name="csToElm">接続先親要素</param>
        /// <returns>bool　true:接続OK　false:接続NG</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool ConnectionElementCheck(HySIdentifier csFromChkID, HySIdentifier csToChkID, ref McGraphicsElement csFromPort, ref McGraphicsElement csToPort, ref McGraphicsElement csFromElm, ref McGraphicsElement csToElm)
        {
            bool bFromFlg = false;
            bool bToFlg = false;
            HySIdentifier csFromID=null;
            HySIdentifier csToID=null;
            HySIdentifier csNullID = new HySID("");  //NULLID

            McGraphicsElement csTempPort = null;
            bool bFromTransFlg = false;
            bool bToTransFlg = false;
            csFromPort = null;
            csToPort = null;
            csFromElm = null;
            csToElm = null;

            if (csFromChkID.Equals(csToChkID) == true)
            {// IDが一致した場合
                return false;
            }

            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if ((bFromFlg == false) && (m_csElmList[iLp].Equal(csFromChkID) == true))
                {// IDが一致した場合
                    bFromFlg = true;
                    csFromElm = m_csElmList[iLp];

                    if ((csFromElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                        (csFromElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループ要素の場合
                        bFromTransFlg = true;
                    }
                }
                if ((bToFlg == false) && (m_csElmList[iLp].Equal(csToChkID) == true))
                {// IDが一致した場合
                    bToFlg = true;
                    csToElm = m_csElmList[iLp];

                    if ((csToElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                        (csToElm.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループ要素の場合
                        bToTransFlg = true;
                    }
                }
                if ((bFromFlg == true) && (bToFlg == true))
                {// 両方一致した場合
                    if (bFromTransFlg == true)
                    {// 中継端子の場合
                        csFromPort = AddPort(McDefine.SymbolDrawKind.HYM_TRANS_OUT, csFromElm);
                    }
                    else
                    {// 中継端子以外の場合
                        csFromPort = AddPort(McDefine.SymbolDrawKind.HYM_SEND, csFromElm);
                    }

                    if (bToTransFlg == true)
                    {// 中継端子の場合
                        csToPort = AddPort(McDefine.SymbolDrawKind.HYM_TRANS_IN, csToElm);
                    }
                    else
                    {// 中継端子以外の場合
                        csToPort = AddPort(McDefine.SymbolDrawKind.HYM_RECEIVE, csToElm);
                    }

                    return true;
                }
            }

            for (int iLp = 0; iLp < m_csElmList.Count; iLp++)
            {// 要素数ループ
                if ((m_csElmList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_GROUP) &&
                    (m_csElmList[iLp].GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                {// グループ要素以外の場合
                    continue;
                }
                for (int iLp2 = 0; iLp2 < m_csElmList[iLp].GetPortCount(); iLp2++)
                {// 端子数ループ
                    csTempPort = m_csElmList[iLp].GetPort(iLp2);
                    if ((bFromFlg == false) && (csTempPort.Equal(csFromChkID) == true))
                    {// IDが一致した場合
                        if (csTempPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                        {// 流入中継端子の場合
                            return false;
                        }
                        csTempPort.GetPortID(ref csFromID, ref csToID);
                        if (csToID.ToString() != "")
                        {// 接続情報がある場合
                            return false;
                        }
                        if (m_csElmList[iLp].Equal(csToChkID) == true)
                        {// IDが一致した場合
                            return false;
                        }
                        bFromFlg = true;
                        bFromTransFlg = true;
                        csFromPort = csTempPort;
                        csFromElm = m_csElmList[iLp];
                    }
                    if ((bToFlg == false) && (csTempPort.Equal(csToChkID) == true))
                    {// IDが一致した場合
                        if (csTempPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
                        {// 流出中継端子の場合
                            return false;
                        }
                        csTempPort.GetPortID(ref csFromID, ref csToID);
                        if (csFromID.ToString() != "")
                        {// 接続情報がある場合
                            return false;
                        }
                        if (m_csElmList[iLp].Equal(csFromChkID) == true)
                        {// IDが一致した場合
                            return false;
                        }
                        bToFlg = true;
                        bToTransFlg = true;
                        csToPort = csTempPort;
                        csToElm = m_csElmList[iLp];
                    }
                    if ((bFromFlg == true) && (bToFlg == true))
                    {// 両方一致した場合
                        if (csFromPort == null)
                        {// 端子未生成の場合
                            if (bFromTransFlg == true)
                            {// 中継端子の場合
                                csFromPort = AddPort(McDefine.SymbolDrawKind.HYM_TRANS_OUT, csFromElm);
                            }
                            else
                            {// 中継端子以外の場合
                                csFromPort = AddPort(McDefine.SymbolDrawKind.HYM_SEND, csFromElm);
                            }
                        }

                        if (csToPort == null)
                        {// 端子未生成の場合
                            if (bToTransFlg == true)
                            {// 中継端子の場合
                                csToPort = AddPort(McDefine.SymbolDrawKind.HYM_TRANS_IN, csToElm);
                            }
                            else
                            {// 中継端子以外の場合
                                csToPort = AddPort(McDefine.SymbolDrawKind.HYM_RECEIVE, csToElm);
                            }
                        }

                        return true;
                    }
                }
            }

            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    csTempPort = m_csPossessionElm.GetPort(iLp);
                    if ((bFromFlg == false) && (csTempPort.Equal(csFromChkID) == true))
                    {// IDが一致した場合
                        if (csTempPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
                        {// 流出中継端子の場合
                            return false;
                        }
                        ((McGraphicsTransElement)csTempPort).GetTransID(ref csFromID, ref csToID);
                        if (csToID.ToString() != "")
                        {// 接続情報がある場合
                            return false;
                        }
                        bFromFlg = true;
                        bFromTransFlg = true;
                        csFromPort = csTempPort;
                        csFromElm = m_csPossessionElm;
                    }
                    if ((bToFlg == false) && (csTempPort.Equal(csToChkID) == true))
                    {// IDが一致した場合
                        if (csTempPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                        {// 流入中継端子の場合
                            return false;
                        }
                        ((McGraphicsTransElement)csTempPort).GetTransID(ref csFromID, ref csToID);
                        if (csFromID.ToString() != "")
                        {// 接続情報がある場合
                            return false;
                        }
                        bToFlg = true;
                        bToTransFlg = true;
                        csToPort = csTempPort;
                        csToElm = m_csPossessionElm;
                    }
                    if ((bFromFlg == true) && (bToFlg == true))
                    {// 両方一致した場合
                        if (csFromPort == null)
                        {// 端子未生成の場合
                            if (bFromTransFlg == true)
                            {// 中継端子の場合
                                csFromPort = AddPort(McDefine.SymbolDrawKind.HYM_TRANS_OUT, csFromElm);
                            }
                            else
                            {// 中継端子以外の場合
                                csFromPort = AddPort(McDefine.SymbolDrawKind.HYM_SEND, csFromElm);
                            }
                        }

                        if (csToPort == null)
                        {// 端子未生成の場合
                            if (bToTransFlg == true)
                            {// 中継端子の場合
                                csToPort = AddPort(McDefine.SymbolDrawKind.HYM_TRANS_IN, csToElm);
                            }
                            else
                            {// 中継端子以外の場合
                                csToPort = AddPort(McDefine.SymbolDrawKind.HYM_RECEIVE, csToElm);
                            }
                        }

                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ内要素座標再セット</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ReSetGroupMemberPosition(csGroupElement)</para>
        /// </example>
        /// <param name="csGroupElement">グループ要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReSetGroupMemberPosition(McGraphicsGroupElement csGroupElement)
        {
            long lTempX = 0;
            long lTempY = 0;
            long lMemberLeftX = 0;
            long lMemberTopY = 0;
            long lMemberRightX = 0;
            long lMemberBottomY = 0;
            HySIdentifier csFromID = null;
            HySIdentifier csToID = null;
            McGraphicsElement csPartnerPort = null;
            bool bAgreeFlg = false;

            if (csGroupElement.GetElmCount() == 0)
            {// グループ内要素数が０の場合
                return;
            }

            McViewTransFormer.GroupMemberPosiCheck(csGroupElement, ref lMemberLeftX, ref lMemberTopY, ref lMemberRightX, ref lMemberBottomY);

            for (int iLp1 = 0; iLp1 < csGroupElement.GetElmCount(); iLp1++)
            {// グループ内要素数ループ
                csGroupElement.GetElement(iLp1).GetPosition(ref lTempX, ref lTempY);
                csGroupElement.GetElement(iLp1).SetPosition(lTempX - (lMemberLeftX + (lMemberRightX - lMemberLeftX) / 2), lTempY - (lMemberTopY + (lMemberBottomY - lMemberTopY) / 2));
                for (int iLp2 = 0; iLp2 < csGroupElement.GetElement(iLp1).GetPortCount(); iLp2++)
                {// 端子数ループ
                    csGroupElement.GetElement(iLp1).GetPort(iLp2).GetPosition(ref lTempX, ref lTempY);
                    csGroupElement.GetElement(iLp1).GetPort(iLp2).SetPosition(lTempX - (lMemberLeftX + (lMemberRightX - lMemberLeftX) / 2), lTempY - (lMemberTopY + (lMemberBottomY - lMemberTopY) / 2));
                }
            }
            for (int iLp1 = 0; iLp1 < csGroupElement.GetPortCount(); iLp1++)
            {// 端子数ループ
                bAgreeFlg = false;
                ((McGraphicsTransElement)csGroupElement.GetPort(iLp1)).GetTransID(ref csFromID, ref csToID);
                for (int iLp2 = 0; iLp2 < csGroupElement.GetElmCount(); iLp2++)
                {// グループ内要素数ループ
                    for (int iLp3 = 0; iLp3 < csGroupElement.GetElement(iLp2).GetPortCount(); iLp3++)
                    {// 端子数ループ
                        csPartnerPort = csGroupElement.GetElement(iLp2).GetPort(iLp3);
                        if (csPartnerPort.Equal(csFromID) == true)
                        {// IDが一致した場合
                            bAgreeFlg = true;
                            break;
                        }
                        if (csPartnerPort.Equal(csToID) == true)
                        {// IDが一致した場合
                            bAgreeFlg = true;
                            break;
                        }
                    }
                    if (bAgreeFlg == true)
                    {// 中継端子の場合
                        csPartnerPort.GetPosition(ref lTempX, ref lTempY);
                        if (csPartnerPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_0)
                        {// 向きが上の場合
                            ((McGraphicsTransElement)csGroupElement.GetPort(iLp1)).SetInnerPosition(lTempX, lTempY - 50);
                        }
                        else if (csPartnerPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_90)
                        {// 向きが右の場合
                            ((McGraphicsTransElement)csGroupElement.GetPort(iLp1)).SetInnerPosition(lTempX + 50, lTempY);
                        }
                        else if (csPartnerPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_180)
                        {// 向きが下の場合
                            ((McGraphicsTransElement)csGroupElement.GetPort(iLp1)).SetInnerPosition(lTempX, lTempY + 50);
                        }
                        else if (csPartnerPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_270)
                        {// 向きが左の場合
                            ((McGraphicsTransElement)csGroupElement.GetPort(iLp1)).SetInnerPosition(lTempX - 50, lTempY);
                        }
                        break;
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ内追加中継端子座標セット</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ReSetGroupMemberPosition(csGroupElement, csRelayPort)</para>
        /// </example>
        /// <param name="csGroupElement">グループ要素</param>
        /// <param name="csRelayPort">追加中継端子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReSetGroupMemberPosition(McGraphicsGroupElement csGroupElement, McGraphicsElement csRelayPort)
        {
            long lTempX = 0;
            long lTempY = 0;
            long lTemp2X = 0;
            long lTemp2Y = 0;
            long lMemberLeftX = 0;
            long lMemberTopY = 0;
            long lMemberRightX = 0;
            long lMemberBottomY = 0;
            long lWidth = 0;
            long lHeight = 0;

            McViewTransFormer.GroupMemberPosiCheck(csGroupElement, ref lMemberLeftX, ref lMemberTopY, ref lMemberRightX, ref lMemberBottomY);
            lWidth = lMemberRightX - lMemberLeftX;
            lHeight = lMemberBottomY - lMemberTopY;

            ((McGraphicsTransElement)csRelayPort).GetInnerPosition(ref lTempX, ref lTempY);
            if ((lTempX != 0) || (lTempY != 0))
            {// 座標が０ではない場合
                return;
            }
            if (csRelayPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_0)
            {// 向きが上の場合
                ((McGraphicsTransElement)csRelayPort).SetInnerPosition(lMemberLeftX + lWidth / 2, lMemberTopY - 50);
            }
            else if (csRelayPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_90)
            {// 向きが右の場合
                ((McGraphicsTransElement)csRelayPort).SetInnerPosition(lMemberRightX + 50, lMemberTopY + lHeight / 2);
            }
            else if (csRelayPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_180)
            {// 向きが下の場合
                ((McGraphicsTransElement)csRelayPort).SetInnerPosition(lMemberLeftX + lWidth / 2, lMemberBottomY + 50);
            }
            else if (csRelayPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_270)
            {// 向きが左の場合
                ((McGraphicsTransElement)csRelayPort).SetInnerPosition(lMemberLeftX - 50, lMemberTopY + lHeight / 2);
            }

            for (int iLp = 0; iLp < csGroupElement.GetPortCount(); iLp++)
            {// 端子数ループ
                ((McGraphicsTransElement)csRelayPort).GetInnerPosition(ref lTempX, ref lTempY);
                ((McGraphicsTransElement)csGroupElement.GetPort(iLp)).GetInnerPosition(ref lTemp2X, ref lTemp2Y);
                if ((csGroupElement.GetPort(iLp).Equal(csRelayPort.GetID()) == false) &&
                    (lTempX == lTemp2X) && (lTempY == lTemp2Y))
                {// 他の端子と重なる場合
                    if ((csRelayPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_0) ||
                        (csRelayPort.GetRotate() == McDefine.HYM_DRAW_ANGLE_180))
                    {// 向きが上下の場合
                        ((McGraphicsTransElement)csRelayPort).SetInnerPosition(lTempX + 20, lTempY);
                    }
                    else
                    {// 向きが左右の場合
                        ((McGraphicsTransElement)csRelayPort).SetInnerPosition(lTempX, lTempY + 20);
                    }
                    iLp = 0;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>要素貼り付け</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ElementPaste(csClipBoardList, iPasteX, iPasteY)</para>
        /// </example>
        /// <param name="csClipBoardList">貼り付け要素</param>
        /// <param name="iPasteX">貼り付け座標X</param>
        /// <param name="iPasteY">貼り付け座標Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ElementPaste(HySDataLinkedList csClipBoardList, int iPasteX, int iPasteY)
        {
            long lX = 0;
            long lY = 0;
            long lMinimumX = 0;
            long lMinimumY = 0;
            long lBaseX = 0;
            long lBaseY = 0;
            long lWidth = 0;
            long lHeight = 0;
            McGraphicsElement csBaseElement = null;
            McGraphicsElement csNewElement = null;
            List<McGraphicsElement> csNewElementList = new List<McGraphicsElement>();
            McGraphicsElement csFromElement = null;
            McGraphicsElement csToElement = null;
            McGraphicsElement csNewPort = null;
            McGraphicsElement csFromPort = null;
            McGraphicsElement csToPort = null;
            McGraphicsElement csBaseFromPort = null;
            McGraphicsElement csBaseToPort = null;
            HySIdentifier csBaseFromPortID = null;
            HySIdentifier csBaseToPortID = null;

            //貼り付けシンボル最小X・Y座標チェック
            McViewTransFormer.PasteMinimumXYCheck(csClipBoardList, ref lMinimumX, ref lMinimumY);

            for (int iLp = 0; iLp < csClipBoardList.GetCount(); iLp++)
            {// 貼り付け要素数ループ
                csNewElement = null;
                csBaseElement = (McGraphicsElement)csClipBoardList.GetData(iLp);
                if (csBaseElement.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_LINE)
                {// 接続線以外の場合
                    csNewElement = AddElement(csBaseElement.GetSymbolDrawKind(), csBaseElement.GetDrawSubKind());
                    lX = iPasteX;
                    lY = iPasteY;
                    csBaseElement.GetPosition(ref lBaseX, ref lBaseY);
                    csNewElement.SetPosition(lX + lBaseX - lMinimumX, lY + lBaseY - lMinimumY);
                    ElementDataCopy(null, csBaseElement, csNewElement);
                    if ((csBaseElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                        (csBaseElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                    {// グループ要素の場合
                        for (int iLp2 = 0; iLp2 < csBaseElement.GetPortCount(); iLp2++)
                        {// 端子数ループ
                            csNewPort = AddRelayPort(csBaseElement.GetPort(iLp2).GetSymbolDrawKind(), csNewElement);
                            csBaseElement.GetPort(iLp2).GetElementSize(ref lWidth, ref lHeight);
                            csNewPort.SetElementSize(lWidth, lHeight);
                            csNewPort.SetRotate(csBaseElement.GetPort(iLp2).GetRotate());
                            csBaseElement.GetPort(iLp2).GetPosition(ref lBaseX, ref lBaseY);
                            csNewPort.SetPosition(lX + lBaseX - lMinimumX, lY + lBaseY - lMinimumY);
                            ((McGraphicsTransElement)csBaseElement.GetPort(iLp2)).GetInnerPosition(ref lBaseX, ref lBaseY);
                            ((McGraphicsTransElement)csNewPort).SetInnerPosition(lBaseX , lBaseY);
                            AddSelectionList(csNewPort);
                        }
                        //グループ内部コピー
                        ElementPasteGroup((McGraphicsGroupElement)csBaseElement, (McGraphicsGroupElement)csNewElement);
                    }
                    csNewElementList.Add(csNewElement);
                }
                else
                {// 接続線の場合
                    csFromElement = null;
                    csToElement = null;
                    csBaseFromPort = null;
                    csBaseToPort = null;
                    csBaseElement.GetPortID(ref csBaseFromPortID, ref csBaseToPortID);
                    for (int iLp2 = 0; iLp2 < csClipBoardList.GetCount(); iLp2++)
                    {// 貼り付け要素数ループ
                        if (csBaseElement.GetPort(csBaseFromPortID).GetParentElm().GetID().Equals(((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetID()) == true)
                        {// IDが一致した場合
                            csFromElement = csNewElementList[iLp2];
                            if ((((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                                (((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                            {// グループ要素の場合
                                for (int iLp3 = 0; iLp3 < ((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetPortCount(); iLp3++)
                                {// 端子数ループ
                                    if (csBaseElement.GetPort(csBaseFromPortID).GetID().Equals(((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetPort(iLp3).GetID()) == true)
                                    {// IDが一致した場合
                                        csFromPort = csFromElement.GetPort(iLp3);
                                        break;
                                    }
                                }
                            }
                            else
                            {// グループ要素以外の場合
                                csFromPort = AddPort(csBaseElement.GetPort(csBaseFromPortID).GetSymbolDrawKind(), csFromElement);
                                csBaseFromPort = csBaseElement.GetPort(csBaseFromPortID);
                                AddSelectionList(csFromPort);
                            }
                        }
                        if (csBaseElement.GetPort(csBaseToPortID).GetParentElm().GetID().Equals(((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetID()) == true)
                        {// IDが一致した場合
                            csToElement = csNewElementList[iLp2];
                            if ((((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                                (((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                            {// グループ要素の場合
                                for (int iLp3 = 0; iLp3 < ((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetPortCount(); iLp3++)
                                {// 端子数ループ
                                    if (csBaseElement.GetPort(csBaseToPortID).GetID().Equals(((McGraphicsElement)csClipBoardList.GetData(iLp2)).GetPort(iLp3).GetID()) == true)
                                    {// IDが一致した場合
                                        csToPort = csToElement.GetPort(iLp3);
                                        break;
                                    }
                                }
                            }
                            else
                            {// グループ要素以外の場合
                                csToPort = AddPort(csBaseElement.GetPort(csBaseToPortID).GetSymbolDrawKind(), csToElement);
                                csBaseToPort = csBaseElement.GetPort(csBaseToPortID);
                                AddSelectionList(csToPort);
                            }
                        }
                        if ((csFromElement != null) && (csToElement != null))
                        {// 接続要素が両方あった場合
                            break;
                        }
                    }
                    if ((csFromElement != null) && (csToElement != null))
                    {// 接続要素が両方あった場合
                        //接続線設定
                        csNewElement = CreateNewConnection(csFromPort, csToPort);
                        csNewElement.SetModelInfo(csBaseElement.GetModelInfo().Clone());
                        if (csBaseFromPort != null)
                        {// 端子があった場合
                            csBaseFromPort.GetElementSize(ref lWidth, ref lHeight);
                            csFromPort.SetElementSize(lWidth, lHeight);
                            csFromPort.SetRotate(csBaseFromPort.GetRotate());
                            csBaseFromPort.GetPosition(ref lBaseX, ref lBaseY);
                            csFromPort.SetPosition(lX + lBaseX - lMinimumX, lY + lBaseY - lMinimumY);
                        }
                        if (csBaseToPort != null)
                        {// 端子があった場合
                            csBaseToPort.GetElementSize(ref lWidth, ref lHeight);
                            csToPort.SetElementSize(lWidth, lHeight);
                            csToPort.SetRotate(csBaseToPort.GetRotate());
                            csBaseToPort.GetPosition(ref lBaseX, ref lBaseY);
                            csToPort.SetPosition(lX + lBaseX - lMinimumX, lY + lBaseY - lMinimumY);
                        }
                    }
                }
                if (csNewElement != null)
                {// 追加正常の場合
                    AddSelectionList(csNewElement);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>要素貼り付け（グループ内部）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ElementPasteGroup(csBaseGroupElement, csNewGroupElement)</para>
        /// </example>
        /// <param name="csBaseGroupElement">コピー元要素</param>
        /// <param name="csNewGroupElement">コピー先要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ElementPasteGroup(McGraphicsGroupElement csBaseGroupElement, McGraphicsGroupElement csNewGroupElement)
        {
            long lBaseX = 0;
            long lBaseY = 0;
            long lWidth = 0;
            long lHeight = 0;
            McGraphicsElement csBaseElement = null;
            McGraphicsElement csNewElement = null;
            McGraphicsElement csFromElement = null;
            McGraphicsElement csToElement = null;
            McGraphicsElement csNewPort = null;
            McGraphicsElement csFromPort = null;
            McGraphicsElement csToPort = null;
            McGraphicsElement csBaseFromPort = null;
            McGraphicsElement csBaseToPort = null;
            HySIdentifier csBaseFromPortID = null;
            HySIdentifier csBaseToPortID = null;

            for (int iLp = 0; iLp < csBaseGroupElement.GetElmCount(); iLp++)
            {// グループ内要素数ループ
                csBaseElement = csBaseGroupElement.GetElement(iLp);
                csNewElement = AddElement(csNewGroupElement, csBaseElement.GetSymbolDrawKind(), csBaseElement.GetDrawSubKind());
                csBaseElement.GetPosition(ref lBaseX, ref lBaseY);
                csNewElement.SetPosition(lBaseX, lBaseY);
                ElementDataCopy(csNewGroupElement, csBaseElement, csNewElement);
                if ((csBaseElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                    (csBaseElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                {// グループ要素の場合
                    for (int iLp2 = 0; iLp2 < csBaseElement.GetPortCount(); iLp2++)
                    {// 端子数ループ
                        csNewPort = AddRelayPort(csBaseElement.GetPort(iLp2).GetSymbolDrawKind(), csNewElement);
                        csBaseElement.GetPort(iLp2).GetElementSize(ref lWidth, ref lHeight);
                        csNewPort.SetElementSize(lWidth, lHeight);
                        csNewPort.SetRotate(csBaseElement.GetPort(iLp2).GetRotate());
                        csBaseElement.GetPort(iLp2).GetPosition(ref lBaseX, ref lBaseY);
                        csNewPort.SetPosition(lBaseX, lBaseY);
                        ((McGraphicsTransElement)csBaseElement.GetPort(iLp2)).GetInnerPosition(ref lBaseX, ref lBaseY);
                        ((McGraphicsTransElement)csNewPort).SetInnerPosition(lBaseX, lBaseY);
                    }
                    //グループ内部コピー
                    ElementPasteGroup((McGraphicsGroupElement)csBaseElement, (McGraphicsGroupElement)csNewElement);
                }
            }

            for (int iLp = 0; iLp < csBaseGroupElement.GetLineElmCount(); iLp++)
            {// グループ内接続線数ループ
                csBaseElement = csBaseGroupElement.GetLineElement(iLp);
                csBaseElement.GetPortID(ref csBaseFromPortID, ref csBaseToPortID);
                csFromElement = null;
                csToElement = null;
                csBaseFromPort = null;
                csBaseToPort = null;
                for (int iLp2 = 0; iLp2 < csBaseGroupElement.GetPortCount(); iLp2++)
                {// 端子数ループ
                    if (csBaseElement.GetPort(csBaseFromPortID).GetID().Equals(csBaseGroupElement.GetPort(iLp2).GetID()) == true)
                    {// IDが一致した場合
                        csFromElement = csNewGroupElement;
                        csFromPort = csNewGroupElement.GetPort(iLp2);
                    }
                    if (csBaseElement.GetPort(csBaseToPortID).GetID().Equals(csBaseGroupElement.GetPort(iLp2).GetID()) == true)
                    {// IDが一致した場合
                        csToElement = csNewGroupElement;
                        csToPort = csNewGroupElement.GetPort(iLp2);
                    }
                    if ((csFromElement != null) && (csToElement != null))
                    {// 両方が一致した場合
                        break;
                    }
                }

                for (int iLp2 = 0; iLp2 < csBaseGroupElement.GetElmCount(); iLp2++)
                {// グループ内要素数ループ
                    if (csBaseElement.GetPort(csBaseFromPortID).GetParentElm().GetID().Equals(csBaseGroupElement.GetElement(iLp2).GetID()) == true)
                    {// IDが一致した場合
                        csFromElement = csNewGroupElement.GetElement(iLp2);
                        if ((csBaseGroupElement.GetElement(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                            (csBaseGroupElement.GetElement(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                        {// グループ要素の場合
                            for (int iLp3 = 0; iLp3 < csBaseGroupElement.GetElement(iLp2).GetPortCount(); iLp3++)
                            {// 端子数ループ
                                if (csBaseElement.GetPort(csBaseFromPortID).GetID().Equals(csBaseGroupElement.GetElement(iLp2).GetPort(iLp3).GetID()) == true)
                                {// IDが一致した場合
                                    csFromPort = csFromElement.GetPort(iLp3);
                                    break;
                                }
                            }
                        }
                        else
                        {// グループ要素以外の場合
                            csFromPort = AddPort(csBaseElement.GetPort(csBaseFromPortID).GetSymbolDrawKind(), csFromElement);
                            csBaseFromPort = csBaseElement.GetPort(csBaseFromPortID);
                        }
                    }
                    if (csBaseElement.GetPort(csBaseToPortID).GetParentElm().GetID().Equals(csBaseGroupElement.GetElement(iLp2).GetID()) == true)
                    {// IDが一致した場合
                        csToElement = csNewGroupElement.GetElement(iLp2);
                        if ((csBaseGroupElement.GetElement(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                            (csBaseGroupElement.GetElement(iLp2).GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                        {// グループ要素の場合
                            for (int iLp3 = 0; iLp3 < csBaseGroupElement.GetElement(iLp2).GetPortCount(); iLp3++)
                            {// 端子数ループ
                                if (csBaseElement.GetPort(csBaseToPortID).GetID().Equals(csBaseGroupElement.GetElement(iLp2).GetPort(iLp3).GetID()) == true)
                                {// IDが一致した場合
                                    csToPort = csToElement.GetPort(iLp3);
                                    break;
                                }
                            }
                        }
                        else
                        {// グループ要素以外の場合
                            csToPort = AddPort(csBaseElement.GetPort(csBaseToPortID).GetSymbolDrawKind(), csToElement);
                            csBaseToPort = csBaseElement.GetPort(csBaseToPortID);
                        }
                    }
                    if ((csFromElement != null) && (csToElement != null))
                    {// 接続要素が両方あった場合
                        break;
                    }
                }
                if ((csFromElement != null) && (csToElement != null))
                {// 接続要素が両方あった場合
                    //接続線設定
                    csNewElement = CreateNewConnectionGroup(csNewGroupElement, csFromPort, csToPort);
                    csNewElement.SetModelInfo(csBaseElement.GetModelInfo().Clone());
                    if (csBaseFromPort != null)
                    {// 端子があった場合
                        csBaseFromPort.GetElementSize(ref lWidth, ref lHeight);
                        csFromPort.SetElementSize(lWidth, lHeight);
                        csFromPort.SetRotate(csBaseFromPort.GetRotate());
                        csBaseFromPort.GetPosition(ref lBaseX, ref lBaseY);
                        csFromPort.SetPosition(lBaseX, lBaseY);
                    }
                    if (csBaseToPort != null)
                    {// 端子があった場合
                        csBaseToPort.GetElementSize(ref lWidth, ref lHeight);
                        csToPort.SetElementSize(lWidth, lHeight);
                        csToPort.SetRotate(csBaseToPort.GetRotate());
                        csBaseToPort.GetPosition(ref lBaseX, ref lBaseY);
                        csToPort.SetPosition(lBaseX, lBaseY);
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>要素間データコピー</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ElementDataCopy(csPossessionElm, csFromElement, csToElement)</para>
        /// </example>
        /// <param name="csPossessionElm">親グループ要素</param>
        /// <param name="csFromElement">コピー元要素</param>
        /// <param name="csToElement">コピー先要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ElementDataCopy(McGraphicsElement csPossessionElm, McGraphicsElement csFromElement, McGraphicsElement csToElement)
        {
            long lWidth = 0;
            long lHeight = 0;

            csFromElement.GetElementSize(ref lWidth, ref lHeight);
            csToElement.SetElementSize(lWidth, lHeight);
            csToElement.SetRotate(csFromElement.GetRotate());
            csToElement.SetElementName(csFromElement.GetElementName());
            csToElement.SetModelInfo(csFromElement.GetModelInfo().Clone());
            csToElement.SetElementDisplayMode(csFromElement.GetElementDisplayMode());
            csToElement.SetParentElm(csPossessionElm);
        }

        /// <summary><para>method outline:</para>
        /// <para>選択要素整列</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ElementLineUp(iLineUPMode, ref csMoveIDList, ref csBeforePointList, ref csAfterPointList)</para>
        /// </example>
        /// <param name="iLineUPMode">整列モード　0：選択要素上辺整列　1：選択要素右辺整列　2：選択要素下辺整列　3：選択要素左辺整列</param>
        /// <param name="csMoveIDList">移動要素ID</param>
        /// <param name="csBeforePointList">移動前座標</param>
        /// <param name="csAfterPointList">移動後座標</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ElementLineUp(int iLineUPMode, ref List<HySIdentifier> csMoveIDList, ref List<Point> csBeforePointList, ref List<Point> csAfterPointList)
        {
            if ((iLineUPMode < 0) || (iLineUPMode > 3))
            {// パラメータ異常の場合
                return;
            }

            long[] lElememtPosition = new long[4] { -1, -1, -1, -1 };
            long lElmX = 0;
            long lElmY = 0;
            List<McGraphicsElement> csSelectElmList = new List<McGraphicsElement>();

            for (int iLp1 = 0; iLp1 < m_csSelectionList.Count; iLp1++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_SEND))
                {// 接続線・端子の場合
                    continue;
                }
                if ((m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    if (m_csPossessionElm == null)
                    {// 全体系の場合
                        continue;
                    }
                    if (m_csSelectionList[iLp1].GetParentElm().GetID().Equals(m_csPossessionElm.GetID()) == false)
                    {// IDが一致した場合
                        continue;
                    }
                }
                csSelectElmList.Add(m_csSelectionList[iLp1]);
                if ((m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    ((McGraphicsTransElement)m_csSelectionList[iLp1]).GetInnerPosition(ref lElmX, ref lElmY);
                }
                else
                {// 中継端子以外の場合
                    m_csSelectionList[iLp1].GetPosition(ref lElmX, ref lElmY);
                }
                if ((lElmY < lElememtPosition[0]) || (lElememtPosition[0] == -1))
                {// 最も上に位置する場合
                    lElememtPosition[0] = lElmY;
                }
                if ((lElmX > lElememtPosition[1]) || (lElememtPosition[1] == -1))
                {// 最も右に位置する場合
                    lElememtPosition[1] = lElmX;
                }
                if ((lElmY > lElememtPosition[2]) || (lElememtPosition[2] == -1))
                {// 最も下に位置する場合
                    lElememtPosition[2] = lElmY;
                }
                if ((lElmX < lElememtPosition[3]) || (lElememtPosition[3] == -1))
                {// 最も左に位置する場合
                    lElememtPosition[3] = lElmX;
                }
            }
            for (int iLp1 = 0; iLp1 < csSelectElmList.Count; iLp1++)
            {// 対象要素数ループ
                if ((csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    ((McGraphicsTransElement)csSelectElmList[iLp1]).GetInnerPosition(ref lElmX, ref lElmY);
                }
                else
                {// 中継端子以外の場合
                    csSelectElmList[iLp1].GetPosition(ref lElmX, ref lElmY);
                }
                csMoveIDList.Add(csSelectElmList[iLp1].GetID());
                csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                if (iLineUPMode == 0)
                {// 選択要素上辺整列の場合
                    lElmY = lElememtPosition[0];
                }
                else if (iLineUPMode == 1)
                {// 選択要素右辺整列の場合
                    lElmX = lElememtPosition[1];
                }
                else if (iLineUPMode == 2)
                {// 選択要素下辺整列の場合
                    lElmY = lElememtPosition[2];
                }
                else if (iLineUPMode == 3)
                {// 選択要素左辺整列の場合
                    lElmX = lElememtPosition[3];
                }
                csAfterPointList.Add(new Point((int)lElmX, (int)lElmY));
                if ((csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    ((McGraphicsTransElement)csSelectElmList[iLp1]).SetInnerPosition(lElmX, lElmY);
                }
                else
                {// 中継端子以外の場合
                    csSelectElmList[iLp1].SetPosition(lElmX, lElmY);
                    for (long lLp = 0; lLp < csSelectElmList[iLp1].GetPortCount(); lLp++)
                    {// 端子数ループ
                        csMoveIDList.Add(csSelectElmList[iLp1].GetPort(lLp).GetID());
                        csSelectElmList[iLp1].GetPort(lLp).GetPosition(ref lElmX, ref lElmY);
                        csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                    }
                    ReSetPortPosition(csSelectElmList[iLp1]);
                    for (long lLp = 0; lLp < csSelectElmList[iLp1].GetPortCount(); lLp++)
                    {// 端子数ループ
                        csSelectElmList[iLp1].GetPort(lLp).GetPosition(ref lElmX, ref lElmY);
                        csAfterPointList.Add(new Point((int)lElmX, (int)lElmY));
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>選択要素等間隔整列</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ElementEqualDistanceLineUp(iLineUPMode, ref csMoveIDList, ref csBeforePointList, ref csAfterPointList)</para>
        /// </example>
        /// <param name="iLineUPMode">整列モード　0：選択要素上下等間隔整列　1：選択要素左右等間隔整列</param>
        /// <param name="csMoveIDList">移動要素ID</param>
        /// <param name="csBeforePointList">移動前座標</param>
        /// <param name="csAfterPointList">移動後座標</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void ElementEqualDistanceLineUp(int iLineUPMode, ref List<HySIdentifier> csMoveIDList, ref List<Point> csBeforePointList, ref List<Point> csAfterPointList)
        {
            if ((iLineUPMode < 0) || (iLineUPMode > 1))
            {// パラメータ異常の場合
                return;
            }

            long[] lElememtPosition = new long[4] { -1, -1, -1, -1 };
            long lElmX = 0;
            long lElmY = 0;
            long lTempElmX = 0;
            long lTempElmY = 0;
            int iLoopPosi = 0;
            List<McGraphicsElement> csSelectElmList = new List<McGraphicsElement>();

            for (int iLp1 = 0; iLp1 < m_csSelectionList.Count; iLp1++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_RECEIVE) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_SEND))
                {// 接続線・端子の場合
                    continue;
                }
                if ((m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    if (m_csPossessionElm == null)
                    {// 全体系の場合
                        continue;
                    }
                    if (m_csSelectionList[iLp1].GetParentElm().GetID().Equals(m_csPossessionElm.GetID()) == false)
                    {// IDが一致した場合
                        continue;
                    }
                }
                if ((m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    ((McGraphicsTransElement)m_csSelectionList[iLp1]).GetInnerPosition(ref lElmX, ref lElmY);
                }
                else
                {// 中継端子以外の場合
                    m_csSelectionList[iLp1].GetPosition(ref lElmX, ref lElmY);
                }
                if ((lElmY < lElememtPosition[0]) || (lElememtPosition[0] == -1))
                {// 最も上に位置する場合
                    lElememtPosition[0] = lElmY;
                }
                if ((lElmY > lElememtPosition[1]) || (lElememtPosition[1] == -1))
                {// 最も下に位置する場合
                    lElememtPosition[1] = lElmY;
                }
                if ((lElmX < lElememtPosition[2]) || (lElememtPosition[2] == -1))
                {// 最も右に位置する場合
                    lElememtPosition[2] = lElmX;
                }
                if ((lElmX > lElememtPosition[3]) || (lElememtPosition[3] == -1))
                {// 最も右に位置する場合
                    lElememtPosition[3] = lElmX;
                }
                for (iLoopPosi = 0; iLoopPosi < csSelectElmList.Count; iLoopPosi++)
                {// 対象要素数ループ
                    if ((csSelectElmList[iLoopPosi].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                        (csSelectElmList[iLoopPosi].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                    {// 中継端子の場合
                        ((McGraphicsTransElement)csSelectElmList[iLoopPosi]).GetInnerPosition(ref lTempElmX, ref lTempElmY);
                    }
                    else
                    {// 中継端子以外の場合
                        csSelectElmList[iLoopPosi].GetPosition(ref lTempElmX, ref lTempElmY);
                    }
                    if ((iLineUPMode == 0) && (lTempElmY > lElmY))
                    {// 選択要素上下等間隔整列で上に位置する場合
                        break;
                    }
                    if ((iLineUPMode == 1) && (lTempElmX > lElmX))
                    {// 選択要素左右等間隔整列で左に位置する場合
                        break;
                    }
                }
                csSelectElmList.Insert(iLoopPosi, m_csSelectionList[iLp1]);
            }
            for (int iLp1 = 1; iLp1 < csSelectElmList.Count - 1; iLp1++)
            {// 対象要素数ループ
                if ((csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    ((McGraphicsTransElement)csSelectElmList[iLp1]).GetInnerPosition(ref lElmX, ref lElmY);
                }
                else
                {// 中継端子以外の場合
                    csSelectElmList[iLp1].GetPosition(ref lElmX, ref lElmY);
                }
                csMoveIDList.Add(csSelectElmList[iLp1].GetID());
                csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                if (iLineUPMode == 0)
                {// 選択要素上下等間隔整列の場合
                    lElmY = (lElememtPosition[1] - lElememtPosition[0]) / (csSelectElmList.Count - 1) * iLp1 + lElememtPosition[0];
                }
                else if (iLineUPMode == 1)
                {// 選択要素左右等間隔整列の場合
                    lElmX = (lElememtPosition[3] - lElememtPosition[2]) / (csSelectElmList.Count - 1) * iLp1 + lElememtPosition[2];
                }
                csAfterPointList.Add(new Point((int)lElmX, (int)lElmY));
                if ((csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (csSelectElmList[iLp1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    ((McGraphicsTransElement)csSelectElmList[iLp1]).SetInnerPosition(lElmX, lElmY);
                }
                else
                {// 中継端子以外の場合
                    csSelectElmList[iLp1].SetPosition(lElmX, lElmY);
                    for (long lLp = 0; lLp < csSelectElmList[iLp1].GetPortCount(); lLp++)
                    {// 端子数ループ
                        csMoveIDList.Add(csSelectElmList[iLp1].GetPort(lLp).GetID());
                        csSelectElmList[iLp1].GetPort(lLp).GetPosition(ref lElmX, ref lElmY);
                        csBeforePointList.Add(new Point((int)lElmX, (int)lElmY));
                    }
                    ReSetPortPosition(csSelectElmList[iLp1]);
                    for (long lLp = 0; lLp < csSelectElmList[iLp1].GetPortCount(); lLp++)
                    {// 端子数ループ
                        csSelectElmList[iLp1].GetPort(lLp).GetPosition(ref lElmX, ref lElmY);
                        csAfterPointList.Add(new Point((int)lElmX, (int)lElmY));
                    }
                }
            }
        }

        /// <summary>
        /// <para>method outline:</para>
        /// <para>グループ要素検索</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>McGraphicsElement csGraphicsElement = SearchGroupElement(csElementList, csID)</para>
        /// </example>
        /// <param name="csElementList">要素リスト</param>
        /// <param name="csID">検索グループID</param>
        /// <returns>McGraphicsElement　グループ要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement SearchGroupElement(List<McGraphicsElement> csElementList, HySIdentifier csID)
        {
            McGraphicsElement csSearchGroup = null;

            for (int iLp = 0; iLp < csElementList.Count; iLp++)
            {// 要素数ループ
                if ((csElementList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP) ||
                    (csElementList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP))
                {// グループ要素の場合
                    if (csElementList[iLp].Equal(csID) == true)
                    {// IDが一致した場合
                        csSearchGroup = csElementList[iLp];
                        break;
                    }
                    csSearchGroup = ((McGraphicsGroupElement)csElementList[iLp]).SearchGroupElement(csID);
                    if (csSearchGroup != null)
                    {// 検索要素があった場合
                        break;
                    }
                }
            }

            return csSearchGroup;
        }

        /// <summary><para>method outline:</para>
        /// <para>ID表示モード設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetIDDispMode(bMode)</para>
        /// </example>
        /// <param name="bMode">ID表示モード</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetIDDispMode(bool bMode)
        {
            m_bIDMode = bMode;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル編集画面カスタマイズ情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetCustomInfo(csCustomInfo)</para>
        /// </example>
        /// <param name="csCustomInfo">モデル編集画面カスタマイズ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCustomInfo(McCustomInfo csCustomInfo)
        {
            m_csCustomInfo = csCustomInfo;
        }

        /// <summary><para>method outline:</para>
        /// <para>プロジェクト取り込み</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ProjectIncorporate(csProjectName, csStructInfo, csModelFactory, lPastePointX, lPastePointY)</para>
        /// </example>
        /// <param name="csProjectName">取り込みプロジェクト名</param>
        /// <param name="csStructInfo">取り込みプロジェクト</param>
        /// <param name="csModelFactory">モデルファクトリー</param>
        /// <param name="lPastePointX">配置座標X</param>
        /// <param name="lPastePointY">配置座標X</param>
        /// <returns>McGraphicsElement グループ化した要素</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphicsElement ProjectIncorporate(HySString csProjectName, McStructInfo csStructInfo, McModelFactorySet csModelFactory, long lPastePointX, long lPastePointY)
        {
            McGraphicsElement csNewElm;
            List<McGraphicsElement> csPortList = new List<McGraphicsElement>();

            McStructGroupElement csNewStructGrElm = csStructInfo.ChangeGroupElement(csModelFactory);

            if (m_csPossessionElm == null)
            {// 全体系の場合
                csNewStructGrElm.SetOwnerElement(m_csMcStructInfo);
                m_csMcStructInfo.AddElement(csNewStructGrElm);
            }
            else
            {// グループ内の場合
                csNewStructGrElm.SetOwnerElement((McStructGroupElement)m_csPossessionElm.GetViewerElm());
                ((McStructGroupElement)m_csPossessionElm.GetViewerElm()).AddElement(csNewStructGrElm);
            }

            // 描画要素の復元
            csNewElm = DecodeGraphic(csNewStructGrElm, ref csPortList);

            SetAllDecodePortPosition(((McGraphicsGroupElement)csNewElm).GetGroupElementList(), ((McGraphicsGroupElement)csNewElm).GetGroupLineElementList());

            csNewElm.SetElementName(csProjectName);
            csNewElm.SetPosition(lPastePointX, lPastePointY);
            csNewElm.SetParentElm(m_csPossessionElm);

            m_csElmList.Add(csNewElm);

            ClearSelectionList();
            AddSelectionList(csNewElm);

            return csNewElm;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定要素削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DelElement(csTopElmList, csTopLineList, csParent)</para>
        /// </example>
        /// <param name="csTopElmList">最上位要素リスト</param>
        /// <param name="csTopLineList">最上位接続線リスト</param>
        /// <param name="csParent">対象親要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DelElement(List<McGraphicsElement> csTopElmList, List<McGraphicsElement> csTopLineList, McGraphicsGroupElement csParent)
        {
            McGraphicsElement csElementWork1 = null;
            McGraphicsElement csElementWork2 = null;
            HySIdentifier csFromID = null;
            HySIdentifier csToID = null;
            HySIdentifier csNullID = new HySID("");
            List<McGraphicsElement> csElmList = null;
            List<McGraphicsElement> csLineList = null;

            if (csParent == null)
            {// 対象親要素がない場合
                csElmList = csTopElmList;
                csLineList = csTopLineList;
            }
            else
            {// 対象親要素がある場合
                csElmList = csParent.GetGroupElementList();
                csLineList = csParent.GetGroupLineElementList();
            }

            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_LINE)
                {// 接続線の場合
                    // 接続線の削除および接続端子の削除
                    for (int iLp2 = 0; iLp2 < csLineList.Count; iLp2++)
                    {// 接続線数ループ
                        if (m_csSelectionList[iLp].GetID().Equals(csLineList[iLp2].GetID()) == true)
                        {// IDが一致する場合
                            csLineList[iLp2].GetPortID(ref csFromID, ref csToID);
                            csElementWork1 = csLineList[iLp2].GetPort(csFromID);
                            csElementWork2 = csLineList[iLp2].GetPort(csToID);

                            if ((csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                (csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                            {// 中継端子以外の場合
                                DelSelectionList((HySString)csElementWork1.GetID().GetString());
                                csElementWork1.GetParentElm().DelPort(csElementWork1.GetID());
                                ReSetPortPosition(csElementWork1.GetParentElm());
                            }

                            if ((csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                (csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                            {// 中継端子以外の場合
                                DelSelectionList((HySString)csElementWork2.GetID().GetString());
                                csElementWork2.GetParentElm().DelPort(csElementWork2.GetID());
                                ReSetPortPosition(csElementWork2.GetParentElm());
                            }

                            if (csParent == null)
                            {// 対象親要素がない場合
                                m_csMcStructInfo.DeleteConnection(csLineList[iLp2].GetID());
                            }
                            else
                            {// 対象親要素がある場合
                                ((McStructGroupElement)csParent.GetViewerElm()).DeleteConnection(csLineList[iLp2].GetID());
                            }
                            DelSelectionList((HySString)csLineList[iLp2].GetID().GetString());
                            csLineList.RemoveAt(iLp2);
                            iLp = -1;
                            break;
                        }
                    }

                    continue;
                }
                for (int iLp2 = 0; iLp2 < csElmList.Count; iLp2++)
                {// 要素数ループ
                    if (m_csSelectionList[iLp].GetID().Equals(csElmList[iLp2].GetID()) == true)
                    {// IDが一致する場合
                        // 要素削除および削除要素の接続線の削除・接続端子の削除
                        for (int iLp3 = 0; iLp3 < csElmList[iLp2].GetPortCount(); iLp3++)
                        {// 端子数ループ
                            for (int iLp4 = 0; iLp4 < csLineList.Count; iLp4++)
                            {// 接続線数ループ
                                csLineList[iLp4].GetPortID(ref csFromID, ref csToID);
                                csElementWork1 = csLineList[iLp4].GetPort(csFromID);
                                csElementWork2 = csLineList[iLp4].GetPort(csToID);

                                if (csElmList[iLp2].GetPort(iLp3).Equal(csElementWork1.GetID()))
                                {// IDが一致する場合
                                    if ((csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                        (csElementWork2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子以外の場合
                                        DelSelectionList((HySString)csElementWork2.GetID().GetString());
                                        csElementWork2.GetParentElm().DelPort(csElementWork2.GetID());
                                        ReSetPortPosition(csElementWork2.GetParentElm());
                                    }

                                    if (csParent == null)
                                    {// 対象親要素がない場合
                                        m_csMcStructInfo.DeleteConnection(csLineList[iLp4].GetID());
                                    }
                                    else
                                    {// 対象親要素がある場合
                                        ((McStructGroupElement)csParent.GetViewerElm()).DeleteConnection(csLineList[iLp4].GetID());
                                    }
                                    DelSelectionList((HySString)csLineList[iLp4].GetID().GetString());
                                    csLineList.RemoveAt(iLp4);
                                    break;
                                }
                                if (csElmList[iLp2].GetPort(iLp3).Equal(csElementWork2.GetID()))
                                {// IDが一致する場合
                                    if ((csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                        (csElementWork1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子以外の場合
                                        DelSelectionList((HySString)csElementWork1.GetID().GetString());
                                        csElementWork1.GetParentElm().DelPort(csElementWork1.GetID());
                                        ReSetPortPosition(csElementWork1.GetParentElm());
                                    }

                                    if (csParent == null)
                                    {// 対象親要素がない場合
                                        m_csMcStructInfo.DeleteConnection(csLineList[iLp4].GetID());
                                    }
                                    else
                                    {// 対象親要素がある場合
                                        ((McStructGroupElement)csParent.GetViewerElm()).DeleteConnection(csLineList[iLp4].GetID());
                                    }
                                    DelSelectionList((HySString)csLineList[iLp4].GetID().GetString());
                                    csLineList.RemoveAt(iLp4);
                                    break;
                                }
                            }
                            DelSelectionList((HySString)csElmList[iLp2].GetPort(iLp3).GetID().GetString());
                        }
                        if (csParent == null)
                        {// 対象親要素がない場合
                            m_csMcStructInfo.RemoveElement(csElmList[iLp2].GetID());
                        }
                        else
                        {// 対象親要素がある場合
                            ((McStructGroupElement)csParent.GetViewerElm()).RemoveElement(csElmList[iLp2].GetID());
                        }
                        csElmList.RemoveAt(iLp2);
                        iLp = -1;
                        break;
                    }
                }
            }

            McGraphicsElement csTempPort2 = null;
            McGraphicsElement csParentElm = null;
            HySIdentifier csFromPortID = null;
            HySIdentifier csToPortID = null;

            //グループ中継端子削除処理（グループ中継端子削除で中継端子・外接続線・外相手端子・内接続線・内相手端子を削除）
            for (int iLp = 0; iLp < m_csSelectionList.Count; iLp++)
            {// 選択要素数ループ
                if ((m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN) ||
                    (m_csSelectionList[iLp].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                {// 中継端子の場合
                    if ((csParent == null) || (csParent.GetID().Equals(m_csSelectionList[iLp].GetParentElm().GetID()) != true))
                    {// グループ外部の場合
                        //中継端子削除
                        m_csSelectionList[iLp].GetParentElm().DelPort(m_csSelectionList[iLp].GetID());
                        ReSetPortPosition(m_csSelectionList[iLp].GetParentElm());

                        ((McGraphicsTransElement)m_csSelectionList[iLp]).GetTransID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合

                            for (int iLp3 = 0; iLp3 < ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElmCount(); iLp3++)
                            {// グループ内接続線数ループ
                                ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).GetPortID(ref csFromPortID, ref csToPortID);
                                csElementWork1 = ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).GetPort(csFromPortID);
                                csElementWork2 = ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).GetLineElement(iLp3).GetPort(csToPortID);
                                if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                    (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                {// 両方IDが一致しない場合
                                    continue;
                                }
                                if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork2;
                                }
                                if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork1;
                                }

                                if ((csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    //グループ内側接続相手端子削除
                                    csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                    ReSetPortPosition(csTempPort2.GetParentElm());
                                }

                                //グループ内側接続線削除
                                ((McGraphicsGroupElement)m_csSelectionList[iLp].GetParentElm()).DelLineElement(iLp3);
                                break;
                            }
                        }

                        m_csSelectionList[iLp].GetPortID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合
                            for (int iLp3 = 0; iLp3 < csLineList.Count; iLp3++)
                            {// 接続線数ループ
                                csLineList[iLp3].GetPortID(ref csFromPortID, ref csToPortID);
                                csElementWork1 = csLineList[iLp3].GetPort(csFromPortID);
                                csElementWork2 = csLineList[iLp3].GetPort(csToPortID);
                                if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                    (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                {// 両方IDが一致しない場合
                                    continue;
                                }
                                if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork2;
                                }
                                if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork1;
                                }

                                if ((csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    //グループ外側接続相手端子削除
                                    csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                    ReSetPortPosition(csTempPort2.GetParentElm());
                                }

                                //グループ外側接続線削除
                                csLineList.RemoveAt(iLp3);
                                break;
                            }
                        }

                        if (csParent == null)
                        {// 対象親要素がない場合
                            ((McStructGroupElement)m_csSelectionList[iLp].GetParentElm().GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), m_csMcStructInfo);
                        }
                        else
                        {// 対象親要素がある場合
                            ((McStructGroupElement)m_csSelectionList[iLp].GetParentElm().GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), (McStructGroupElement)csParent.GetViewerElm());
                        }
                        ((McStructGroupElement)m_csSelectionList[iLp].GetParentElm().GetViewerElm()).DelPort(m_csSelectionList[iLp].GetID());

                        DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                        iLp = -1;
                    }
                    else
                    {// グループ内部の場合
                        //中継端子削除
                        csParent.DelPort(m_csSelectionList[iLp].GetID());
                        ReSetPortPosition(csParent);

                        ((McGraphicsTransElement)m_csSelectionList[iLp]).GetTransID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合
                            for (int iLp2 = 0; iLp2 < ((McGraphicsGroupElement)csParent).GetLineElmCount(); iLp2++)
                            {// グループ内接続線数ループ
                                ((McGraphicsGroupElement)csParent).GetLineElement(iLp2).GetPortID(ref csFromPortID, ref csToPortID);
                                csElementWork1 = ((McGraphicsGroupElement)csParent).GetLineElement(iLp2).GetPort(csFromPortID);
                                csElementWork2 = ((McGraphicsGroupElement)csParent).GetLineElement(iLp2).GetPort(csToPortID);
                                if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                    (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                {// 両方IDが一致しない場合
                                    continue;
                                }
                                if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork2;
                                }
                                if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                {// IDが一致する場合
                                    csTempPort2 = csElementWork1;
                                }

                                if ((csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    //グループ内側接続相手端子削除
                                    csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                    ReSetPortPosition(csTempPort2.GetParentElm());
                                }

                                //グループ内側接続線削除
                                ((McGraphicsGroupElement)csParent).DelLineElement(iLp2);
                                break;
                            }
                        }

                        m_csSelectionList[iLp].GetPortID(ref csFromPortID, ref csToPortID);
                        if ((csFromPortID.Equals(csNullID) != true) && (csToPortID.Equals(csNullID) != true))
                        {// 接続情報ありの場合
                            csParentElm = csParent.GetParentElm();
                            if (csParentElm != null)
                            {// 親要素がある場合
                                for (int iLp2 = 0; iLp2 < ((McGraphicsGroupElement)csParentElm).GetLineElmCount(); iLp2++)
                                {// グループ内接続線数ループ
                                    ((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).GetPortID(ref csFromPortID, ref csToPortID);
                                    csElementWork1 = ((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).GetPort(csFromPortID);
                                    csElementWork2 = ((McGraphicsGroupElement)csParentElm).GetLineElement(iLp2).GetPort(csToPortID);
                                    if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                        (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                    {// 両方IDが一致しない場合
                                        continue;
                                    }
                                    if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork2;
                                    }
                                    if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork1;
                                    }

                                    if ((csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                        (csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子以外の場合
                                        //グループ外側接続相手端子削除
                                        csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                        ReSetPortPosition(csTempPort2.GetParentElm());
                                    }

                                    //グループ外側接続線削除
                                    ((McGraphicsGroupElement)csParentElm).DelLineElement(iLp2);
                                    break;
                                }
                            }
                            else
                            {// 親要素がない場合
                                for (int iLp2 = 0; iLp2 < csTopLineList.Count; iLp2++)
                                {// 全体系の接続線数ループ
                                    csTopLineList[iLp2].GetPortID(ref csFromPortID, ref csToPortID);
                                    csElementWork1 = csTopLineList[iLp2].GetPort(csFromPortID);
                                    csElementWork2 = csTopLineList[iLp2].GetPort(csToPortID);
                                    if ((csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) != true) &&
                                        (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) != true))
                                    {// 両方IDが一致しない場合
                                        continue;
                                    }
                                    if (csElementWork1.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork2;
                                    }
                                    if (csElementWork2.GetID().Equals(m_csSelectionList[iLp].GetID()) == true)
                                    {// IDが一致する場合
                                        csTempPort2 = csElementWork1;
                                    }

                                    if ((csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                        (csTempPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                    {// 中継端子以外の場合
                                        //グループ外側接続相手端子削除
                                        csTempPort2.GetParentElm().DelPort(csTempPort2.GetID());
                                        ReSetPortPosition(csTempPort2.GetParentElm());
                                    }

                                    //グループ外側接続線削除
                                    csTopLineList.RemoveAt(iLp2);
                                    break;
                                }
                            }
                        }

                        if (csParent.GetParentElm() == null)
                        {// 親要素がない場合
                            ((McStructGroupElement)csParent.GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), m_csMcStructInfo);
                        }
                        else
                        {// 親要素がある場合
                            ((McStructGroupElement)csParent.GetViewerElm()).DeleteRelayPort(m_csSelectionList[iLp].GetID(), (McStructGroupElement)csParent.GetParentElm().GetViewerElm());
                        }
                        ((McStructGroupElement)csParent.GetViewerElm()).DelPort(m_csSelectionList[iLp].GetID());

                        DelSelectionList((HySString)m_csSelectionList[iLp].GetID().GetString());
                        iLp = -1;
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトグループ解除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DisbandObject(csDisbandElm, csTopElmList, csTopLineList)</para>
        /// </example>
        /// <param name="csDisbandElm">解除グループ要素</param>
        /// <param name="csTopElmList">最上位要素リスト</param>
        /// <param name="csTopLineList">最上位接続線リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DisbandObject(McGraphicsGroupElement csDisbandElm, List<McGraphicsElement> csTopElmList, List<McGraphicsElement> csTopLineList)
        {
            McGraphicsElement csRelayPort;
            McGraphicsElement csPort1;
            McGraphicsElement csPort2;
            McGraphicsElement csTempPort;
            McGraphicsElement csElm;
            int iLp;
            long lLp;
            HySIdentifier csFID1 = null;
            HySIdentifier csTID1 = null;
            HySIdentifier csFID2 = null;
            HySIdentifier csTID2 = null;
            HySIdentifier csFID3 = null;
            HySIdentifier csTID3 = null;
            List<McGraphicsElement> csTempElmList = new List<McGraphicsElement>();
            List<McGraphicsElement> csElmList = null;
            List<McGraphicsElement> csLineList = null;

            if (csDisbandElm.GetParentElm() == null)
            {// 親要素がない場合
                csElmList = csTopElmList;
                csLineList = csTopLineList;
            }
            else
            {// 親要素がある場合
                csElmList = ((McGraphicsGroupElement)csDisbandElm.GetParentElm()).GetGroupElementList();
                csLineList = ((McGraphicsGroupElement)csDisbandElm.GetParentElm()).GetGroupLineElementList();
            }

            for (iLp = 0; iLp < csElmList.Count; iLp++)
            {// すべての要素を検索
                if ((csElmList[iLp].GetID().Equals(csDisbandElm.GetID())) != true)
                {// ID不一致
                    csTempElmList.Add(csElmList[iLp]);
                }
            }

            //グループ内エレメント表示座標設定
            ReSetElementPositionInGroup(csDisbandElm);

            for (lLp = 0L; lLp < csDisbandElm.GetPortCount(); lLp++)
            {// ポートの接続情報取得
                csRelayPort = csDisbandElm.GetPort(lLp);
                csRelayPort.GetPortID(ref csFID1, ref csTID1);
                ((McGraphicsTransElement)csRelayPort).GetTransID(ref csFID2, ref csTID2);

                if (((csFID1.ToString() != "") && (csTID1.ToString() != "")) &&
                    ((csFID2.ToString() == "") || (csTID2.ToString() == "")))
                {// グループ外部のみ接続情報がある場合
                    // グループ外部のみ接続情報削除
                    for (iLp = csLineList.Count - 1; iLp >= 0; iLp--)
                    {// 接続線数ループ
                        csLineList[iLp].GetPortID(ref csFID3, ref csTID3);

                        if ((csRelayPort.GetID().Equals(csFID3) == true) || (csRelayPort.GetID().Equals(csTID3) == true))
                        {// IDが一致
                            csPort1 = csLineList[iLp].GetPort(csFID3);
                            csPort2 = csLineList[iLp].GetPort(csTID3);

                            if (csRelayPort.GetID().Equals(csFID3) == true)
                            {// IDが一致する場合
                                if ((csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort2.GetParentElm().DelPort(csPort2.GetID());
                                    ReSetPortPosition(csPort2.GetParentElm());
                                }
                            }
                            if (csRelayPort.GetID().Equals(csTID3) == true)
                            {// IDが一致する場合
                                if ((csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort1.GetParentElm().DelPort(csPort1.GetID());
                                    ReSetPortPosition(csPort1.GetParentElm());
                                }
                            }

                            csLineList.RemoveAt(iLp);
                        }
                    }
                }
                if (((csFID1.ToString() == "") || (csTID1.ToString() == "")) &&
                    ((csFID2.ToString() != "") && (csTID2.ToString() != "")))
                {// グループ内部のみ接続情報がある場合
                    // グループ外部のみ接続情報削除
                    for (iLp = (int)csDisbandElm.GetLineElmCount() - 1; iLp >= 0; iLp--)
                    {// グループ内接続線数ループ
                        csDisbandElm.GetLineElement(iLp).GetPortID(ref csFID3, ref csTID3);

                        if ((csRelayPort.GetID().Equals(csFID3) == true) || (csRelayPort.GetID().Equals(csTID3) == true))
                        {// IDが一致
                            csPort1 = csDisbandElm.GetLineElement(iLp).GetPort(csFID3);
                            csPort2 = csDisbandElm.GetLineElement(iLp).GetPort(csTID3);

                            if (csRelayPort.GetID().Equals(csFID3) == true)
                            {// IDが一致する場合
                                if ((csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csPort2.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort2.GetParentElm().DelPort(csPort2.GetID());
                                    ReSetPortPosition(csPort2.GetParentElm());
                                }
                            }
                            if (csRelayPort.GetID().Equals(csTID3) == true)
                            {// IDが一致する場合
                                if ((csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_IN) &&
                                    (csPort1.GetSymbolDrawKind() != McDefine.SymbolDrawKind.HYM_TRANS_OUT))
                                {// 中継端子以外の場合
                                    csPort1.GetParentElm().DelPort(csPort1.GetID());
                                    ReSetPortPosition(csPort1.GetParentElm());
                                }
                            }
                        }
                    }
                }
            }

            if (csDisbandElm.GetParentElm() == null)
            {// 親要素がない場合
                m_csMcStructInfo.Disband(csDisbandElm.GetID());
            }
            else
            {// 親要素がある場合
                ((McStructGroupElement)csDisbandElm.GetParentElm().GetViewerElm()).Disband(csDisbandElm.GetID());
            }

            // 接続線ポートリスト積み直し
            for (lLp = 0L; lLp < csDisbandElm.GetPortCount(); lLp++)
            {// ポートの接続情報取得

                csRelayPort = csDisbandElm.GetPort(lLp);
                csRelayPort.GetPortID(ref csFID1, ref csTID1);

                if ((csFID1.ToString() != "") && (csTID1.ToString() != ""))
                {// グループ外部の接続情報がある場合
                    for (iLp = csLineList.Count - 1; iLp >= 0; iLp--)
                    {// 接続線数ループ
                        csPort1 = csLineList[iLp].GetPort(0);
                        csPort2 = csLineList[iLp].GetPort(1);

                        if ((csRelayPort.GetID().Equals(csPort1.GetID()) == true) || (csRelayPort.GetID().Equals(csPort2.GetID()) == true))
                        {// IDが一致
                            ((McGraphicsTransElement)csRelayPort).GetTransID(ref csFID1, ref csTID1);

                            if ((csFID1.ToString() != "") && (csTID1.ToString() != ""))
                            {// グループ内部の接続情報がある場合
                                csLineList[iLp].GetPortID(ref csFID2, ref csTID2);
                                if (csRelayPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                                {// 流入中継端子の場合
                                    for (int iLp2 = 0; iLp2 < csDisbandElm.GetElmCount(); iLp2++)
                                    {// グループ内要素数ループ
                                        csTempPort = csDisbandElm.GetElement(iLp2).GetPort(csTID2);
                                        if (csTempPort != null)
                                        {// 端子がある場合
                                            csLineList[iLp].DelPort(1);
                                            csLineList[iLp].DelPort(0);
                                            csLineList[iLp].AddPort(csPort1);
                                            csLineList[iLp].AddPort(csTempPort);
                                            break;
                                        }
                                    }
                                }
                                else
                                {// 流入中継端子以外の場合
                                    for (int iLp2 = 0; iLp2 < csDisbandElm.GetElmCount(); iLp2++)
                                    {// グループ内要素数ループ
                                        csTempPort = csDisbandElm.GetElement(iLp2).GetPort(csFID2);
                                        if (csTempPort != null)
                                        {// 端子がある場合
                                            csLineList[iLp].DelPort(1);
                                            csLineList[iLp].DelPort(0);
                                            csLineList[iLp].AddPort(csTempPort);
                                            csLineList[iLp].AddPort(csPort2);
                                            break;
                                        }
                                    }
                                }
                            }
                            else
                            {// グループ内部の接続情報がない場合
                                csLineList.RemoveAt(iLp);
                            }
                            break;
                        }
                    }
                }
            }

            // グループ内部要素接続情報追加
            for (lLp = 0L; lLp < csDisbandElm.GetLineElmCount(); lLp++)
            {// グループ内接続線数ループ
                csElm = csDisbandElm.GetLineElement(lLp);
                csElm.GetPortID(ref csFID3, ref csTID3);
                if (csElm.GetPort(csFID3).GetParentElm().GetID().Equals(csDisbandElm.GetID()) == true)
                {// IDが一致する場合
                    continue;
                }
                if (csElm.GetPort(csTID3).GetParentElm().GetID().Equals(csDisbandElm.GetID()) == true)
                {// IDが一致する場合
                    continue;
                }
                csElm.SetParentElm(csDisbandElm.GetParentElm());
                csLineList.Add(csElm);
            }

            for (lLp = 0L; lLp < csDisbandElm.GetElmCount(); lLp++)
            {// グループ要素追加
                csElm = csDisbandElm.GetElement(lLp);
                csElm.SetParentElm(csDisbandElm.GetParentElm());
                csTempElmList.Add(csElm);
            }

            csElmList.Clear();
            for (iLp = 0; iLp < csTempElmList.Count; iLp++)
            {// 要素数ループ
                csElmList.Add(csTempElmList[iLp]);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ解除時のグループ内要素座標設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ReSetElementPositionInGroup(csGroupElement)</para>
        /// </example>
        /// <param name="csGroupElement">グループ要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReSetElementPositionInGroup(McGraphicsGroupElement csGroupElement)
        {
            long lMemberLeftX = 0;
            long lMemberTopY = 0;
            long lMemberRightX = 0;
            long lMemberBottomY = 0;
            long lGroupLeftX = 0;
            long lGroupTopY = 0;
            long lGroupWidth = 0;
            long lGroupHeight = 0;
            long lGroupElmX = 0;
            long lGroupElmY = 0;
            long lElmX = 0;
            long lElmY = 0;

            if (csGroupElement.GetElmCount() == 0)
            {// グループ内要素数が０の場合
                return;
            }

            McViewTransFormer.GroupMemberPosiCheck(csGroupElement, ref lMemberLeftX, ref lMemberTopY, ref lMemberRightX, ref lMemberBottomY);

            csGroupElement.GetPosition(ref lGroupElmX, ref lGroupElmY);

            if ((csGroupElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_0) ||
                (csGroupElement.GetRotate() == McDefine.HYM_DRAW_ANGLE_180))
            {// 向きが上下の場合
                csGroupElement.GetElementSize(ref lGroupWidth, ref lGroupHeight);
            }
            else
            {// 向きが左右の場合
                csGroupElement.GetElementSize(ref lGroupHeight, ref lGroupWidth);
            }

            lGroupLeftX = lGroupElmX - lGroupWidth / 2;
            lGroupTopY = lGroupElmY - lGroupHeight / 2;

            // グループ内要素位置設定
            for (int iLp1 = 0; iLp1 < csGroupElement.GetElmCount(); iLp1++)
            {// グループ内要素数ループ
                csGroupElement.GetElement(iLp1).GetPosition(ref lElmX, ref lElmY);
                csGroupElement.GetElement(iLp1).SetPosition(lGroupLeftX + lElmX - lMemberLeftX, lGroupTopY + lElmY - lMemberTopY);

                for (int iLp2 = 0; iLp2 < csGroupElement.GetElement(iLp1).GetPortCount(); iLp2++)
                {// 端子数ループ
                    csGroupElement.GetElement(iLp1).GetPort(iLp2).GetPosition(ref lElmX, ref lElmY);
                    csGroupElement.GetElement(iLp1).GetPort(iLp2).SetPosition(lGroupLeftX + lElmX - lMemberLeftX, lGroupTopY + lElmY - lMemberTopY);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトグループ化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GroupObject(csOrgGroupElm, csDelOuterLineList, csTopElmList, csTopLineList, csParent)</para>
        /// </example>
        /// <param name="csOrgGroupElm">解除グループ要素</param>
        /// <param name="csDelOuterLineList">削除外部接続線リスト</param>
        /// <param name="csTopElmList">最上位要素リスト</param>
        /// <param name="csTopLineList">最上位接続線リスト</param>
        /// <param name="csParent">対象親要素</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void GroupObject(McGraphicsGroupElement csOrgGroupElm, List<McGraphicsElement> csDelOuterLineList, List<McGraphicsElement> csTopElmList, List<McGraphicsElement> csTopLineList, McGraphicsGroupElement csParent)
        {
            McGraphicsGroupElement csNewGrElm = null; // グループ化要素
            McGraphicsElement csElm;            // 要素
            HySIdentifier csFromID = null;      // 接続元ID
            HySIdentifier csToID = null;        // 接続先ID
            HySDataLinkedList csGroupMemberElmList = new HySDataLinkedList();
            int iLp;                            // ループカウンタ
            int iLp2;                           // ループカウンタ
            long lX1 = 0L;                      // 描画位置X
            long lY1 = 0L;                      // 描画位置Y
            long lWidth = 0L;
            long lHeight = 0L;

            List<McGraphicsElement> csWorkElmList = new List<McGraphicsElement>();

            List<McGraphicsElement> csElmList = null;
            List<McGraphicsElement> csLineList = null;

            if (csParent == null)
            {// 対象親要素がない場合
                csElmList = csTopElmList;
                csLineList = csTopLineList;
            }
            else
            {// 対象親要素がある場合
                csElmList = csParent.GetGroupElementList();
                csLineList = csParent.GetGroupLineElementList();
            }

            for (iLp = 0; iLp < csOrgGroupElm.GetElmCount(); iLp++)
            {// IDが一致するか検索
                for (iLp2 = csElmList.Count - 1; iLp2 >= 0; iLp2--)
                {// すべての要素を検索
                    if (csElmList[iLp2].GetID().Equals(csOrgGroupElm.GetElement((long)iLp).GetID()) == true)
                    {// IDが一致する
                        csElm = csElmList[iLp2];
                        csWorkElmList.Add(csElm);
                        csGroupMemberElmList.AddLast(csElm.GetViewerElm());

                        csElmList.RemoveAt(iLp2);
                        break;
                    }
                }
            }

            //グループに入った要素間の接続線の移動
            bool bFromFlg = false;
            bool bToFlg = false;
            for (iLp = csLineList.Count - 1; iLp >= 0; iLp--)
            {// 接続線数ループ
                bFromFlg = false;
                bToFlg = false;
                csLineList[iLp].GetPortID(ref csFromID, ref csToID);
                csFromID = csLineList[iLp].GetPort(csFromID).GetParentElm().GetID();
                csToID = csLineList[iLp].GetPort(csToID).GetParentElm().GetID();
                for (iLp2 = 0; iLp2 < csWorkElmList.Count; iLp2++)
                {// グループ内要素数ループ
                    if (csWorkElmList[iLp2].Equal(csFromID) == true)
                    {// IDが一致する場合
                        bFromFlg = true;
                    }
                    if (csWorkElmList[iLp2].Equal(csToID) == true)
                    {// IDが一致する場合
                        bToFlg = true;
                    }
                    if ((bFromFlg == true) && (bToFlg == true))
                    {// 両方一致の場合
                        csLineList.RemoveAt(iLp);
                        break;
                    }
                }
            }

            McStructGroupElement csNewViewGrElm = null;
            if (csParent == null)
            {// 対象親要素がない場合
                csNewViewGrElm = (McStructGroupElement)m_csMcStructInfo.MakeGroup(new HySID(""), csGroupMemberElmList, csOrgGroupElm.GetSymbolDrawKind(), new HySObjectKind());
            }
            else
            {// 対象親要素がある場合
                csNewViewGrElm = (McStructGroupElement)((McStructGroupElement)csParent.GetViewerElm()).MakeGroup(new HySID(""), csGroupMemberElmList, csOrgGroupElm.GetSymbolDrawKind(), new HySObjectKind());
            }

            // 描画要素の復元
            List<McGraphicsElement> csPortList = new List<McGraphicsElement>();
            csNewGrElm = (McGraphicsGroupElement)DecodeGraphic(csNewViewGrElm, ref csPortList);

            csNewGrElm.SetViewerElm(csNewViewGrElm);

            csNewGrElm.SetParentElm(csParent);

            //グループシンボル座標設定
            csOrgGroupElm.GetElementSize(ref lWidth, ref lHeight);
            csNewGrElm.SetElementSize(lWidth, lHeight);
            csOrgGroupElm.GetPosition(ref lX1, ref lY1);
            csNewGrElm.SetPosition(lX1, lY1);

            //グループ要素情報復元
            ReconstructionGroupData(csOrgGroupElm, csNewGrElm, csDelOuterLineList, csElmList, csLineList);

            csElmList.Add(csNewGrElm);

            SetAllDecodePortPosition(csElmList, csLineList);

            //グループ内要素座標再セット
            ReSetGroupMemberPosition(csNewGrElm);

            //中継端子配置座標設定
            ReSetPortPosition(csNewGrElm);
        }

        /// <summary><para>method outline:</para>
        /// <para>グループ要素情報復元</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ReconstructionGroupData(csOrgGroupElm, csNewGroupElm, csDelOuterLineList, csElmList, csLineList)</para>
        /// </example>
        /// <param name="csOrgGroupElm">オリジナルグループ要素</param>
        /// <param name="csNewGroupElm">新規グループ要素</param>
        /// <param name="csDelOuterLineList">削除外部接続線リスト</param>
        /// <param name="csElmList">要素リスト</param>
        /// <param name="csLineList">接続線リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReconstructionGroupData(McGraphicsGroupElement csOrgGroupElm, McGraphicsGroupElement csNewGroupElm, List<McGraphicsElement> csDelOuterLineList, List<McGraphicsElement> csElmList, List<McGraphicsElement> csLineList)
        {
            // グループ要素のID復元
            csNewGroupElm.SetIDUsedByUndoRedoOnly(csOrgGroupElm.GetID());

            // 中継端子のID復元
            McGraphicsElement csOrgPort = null;
            McGraphicsElement csNewPort = null;
            HySIdentifier csFromID1 = null;
            HySIdentifier csToID1 = null;
            HySIdentifier csFromID2 = null;
            HySIdentifier csToID2 = null;
            for (int iLp = 0; iLp < csOrgGroupElm.GetPortCount(); iLp++)
            {// 端子数ループ
                csOrgPort = csOrgGroupElm.GetPort(iLp);
                csOrgPort.GetPortID(ref csFromID1, ref csToID1);
                for (int iLp2 = 0; iLp2 < csNewGroupElm.GetPortCount(); iLp2++)
                {// 端子数ループ
                    csNewPort = csNewGroupElm.GetPort(iLp2);
                    csNewPort.GetPortID(ref csFromID2, ref csToID2);
                    if (((csFromID1.Equals(csFromID2) == true) || (csToID1.Equals(csToID2) == true))&&
                        ((csFromID1.ToString() != "") && (csToID1.ToString() != "")))
                    {// IDが一致した場合
                        csNewGroupElm.SetPortIDUsedByUndoRedoOnly(csNewPort.GetID(), csOrgPort.GetID());
                        break;
                    }
                }
            }

            // 内部接続線のID復元
            for (int iLp = 0; iLp < csOrgGroupElm.GetLineElmCount(); iLp++)
            {// グループ内接続線数ループ
                csOrgGroupElm.GetLineElement(iLp).GetPortID(ref csFromID1, ref csToID1);
                for (int iLp2 = 0; iLp2 < csNewGroupElm.GetLineElmCount(); iLp2++)
                {// グループ内接続線数ループ
                    csNewGroupElm.GetLineElement(iLp2).GetPortID(ref csFromID2, ref csToID2);
                    if ((csFromID1.Equals(csFromID2) == true) && (csToID1.Equals(csToID2) == true))
                    {// IDが一致した場合
                        csNewGroupElm.GetLineElement(iLp2).SetID(csOrgGroupElm.GetLineElement(iLp).GetID());
                        break;
                    }
                }
            }

            // 外部接続線のポートリスト積み直し
            McGraphicsElement csFromPort = null;
            McGraphicsElement csToPort = null;
            for (int iLp = 0; iLp < csNewGroupElm.GetPortCount(); iLp++)
            {// 端子数ループ
                csNewGroupElm.GetPort(iLp).GetPortID(ref csFromID1, ref csToID1);
                for (int iLp2 = 0; iLp2 < csLineList.Count; iLp2++)
                {// 接続線数ループ
                    csLineList[iLp2].GetPortID(ref csFromID2, ref csToID2);
                    if ((csFromID1.Equals(csFromID2) == true) && (csToID1.Equals(csToID2) == true))
                    {// IDが一致した場合
                        if (csNewGroupElm.GetPort(iLp).GetID().Equals(csFromID2) == true)
                        {// IDが一致した場合
                            csToPort = csLineList[iLp2].GetPort(csToID2);
                            csLineList[iLp2].DelPort(1);
                            csLineList[iLp2].DelPort(0);
                            csLineList[iLp2].AddPort(csNewGroupElm.GetPort(iLp));
                            csLineList[iLp2].AddPort(csToPort);
                        }
                        else
                        {// IDが一致しない場合
                            csFromPort = csLineList[iLp2].GetPort(csFromID2);
                            csLineList[iLp2].DelPort(1);
                            csLineList[iLp2].DelPort(0);
                            csLineList[iLp2].AddPort(csFromPort);
                            csLineList[iLp2].AddPort(csNewGroupElm.GetPort(iLp));
                        }
                        break;
                    }
                }
            }

            // 内部側または外部側のみ接続されている接続線・端子およびの未接続中継端子の復元
            McGraphicsElement csOrgPort2 = null;
            McGraphicsElement csOrgLine = null;
            HySIdentifier csFromID3 = null;
            HySIdentifier csToID3 = null;
            for (int iLp = 0; iLp < csOrgGroupElm.GetPortCount(); iLp++)
            {// 端子数ループ
                csOrgPort = csOrgGroupElm.GetPort(iLp);
                csOrgPort.GetPortID(ref csFromID1, ref csToID1);
                ((McGraphicsTransElement)csOrgPort).GetTransID(ref csFromID2, ref csToID2);
                if (((csFromID1.ToString() == "") || (csToID1.ToString() == "")) &&
                    ((csFromID2.ToString() != "") && (csToID2.ToString() != "")))
                {// 内部のみ接続
                    for (int iLp2 = 0; iLp2 < csOrgGroupElm.GetLineElmCount(); iLp2++)
                    {// グループ内接続線数ループ
                        csOrgLine = csOrgGroupElm.GetLineElement(iLp2);
                        csOrgLine.GetPortID(ref csFromID3, ref csToID3);
                        if ((csFromID2.Equals(csFromID3) == true) && (csToID2.Equals(csToID3) == true))
                        {// IDが一致した場合
                            csNewGroupElm.AddPort(csOrgPort);
                            csNewGroupElm.GetViewerElm().AddPortUsedByUndoRedoOnly(csOrgPort.GetViewerPort());

                            if (csOrgPort.GetID().Equals(csFromID2) == true)
                            {// IDが一致した場合
                                csOrgPort2 = csOrgLine.GetPort(csToID2);

                                for (int iLp3 = 0; iLp3 < csNewGroupElm.GetElmCount(); iLp3++)
                                {// グループ内要素数ループ
                                    if (csNewGroupElm.GetElement(iLp3).GetID().Equals(csOrgPort2.GetParentElm().GetID()) == true)
                                    {// IDが一致した場合
                                        csNewPort = csNewGroupElm.GetElement(iLp3).GetPort(csToID2);
                                        if (csNewPort != null)
                                        {// 端子があった場合
                                            csNewPort.SetPortID(csFromID2, csToID2);
                                            csOrgPort2 = csNewPort;
                                        }
                                        else
                                        {// 端子がない場合
                                            csNewGroupElm.GetElement(iLp3).AddPort(csOrgPort2);
                                            csNewGroupElm.GetElement(iLp3).GetViewerElm().AddPortUsedByUndoRedoOnly(csOrgPort2.GetViewerPort());
                                            ReSetPortPosition(csNewGroupElm.GetElement(iLp3));
                                        }
                                        break;
                                    }
                                }
                                csOrgLine.DelPort(1);
                                csOrgLine.DelPort(0);
                                csOrgLine.AddPort(csOrgPort);
                                csOrgLine.AddPort(csOrgPort2);
                            }
                            else
                            {// IDが一致しない場合
                                csOrgPort2 = csOrgLine.GetPort(csFromID2);

                                for (int iLp3 = 0; iLp3 < csNewGroupElm.GetElmCount(); iLp3++)
                                {// グループ内要素数ループ
                                    if (csNewGroupElm.GetElement(iLp3).GetID().Equals(csOrgPort2.GetParentElm().GetID()) == true)
                                    {// IDが一致した場合
                                        csNewPort = csNewGroupElm.GetElement(iLp3).GetPort(csFromID2);
                                        if (csNewPort != null)
                                        {// 端子があった場合
                                            csNewPort.SetPortID(csFromID2, csToID2);
                                            csOrgPort2 = csNewPort;
                                        }
                                        else
                                        {// 端子がない場合
                                            csNewGroupElm.GetElement(iLp3).AddPort(csOrgPort2);
                                            csNewGroupElm.GetElement(iLp3).GetViewerElm().AddPortUsedByUndoRedoOnly(csOrgPort2.GetViewerPort());
                                            ReSetPortPosition(csNewGroupElm.GetElement(iLp3));
                                        }
                                        break;
                                    }
                                }
                                csOrgLine.DelPort(1);
                                csOrgLine.DelPort(0);
                                csOrgLine.AddPort(csOrgPort2);
                                csOrgLine.AddPort(csOrgPort);
                            }
                            csNewGroupElm.AddLineElement(csOrgLine);
                            ((McStructGroupElement)csNewGroupElm.GetViewerElm()).AddConnectionUsedByUndoRedoOnly(csOrgLine.GetViewerConnect());
                            break;
                        }
                    }
                }
                else if (((csFromID1.ToString() != "") && (csToID1.ToString() != "")) &&
                         ((csFromID2.ToString() == "") || (csToID2.ToString() == "")))
                {// 外部のみ接続
                    csNewGroupElm.AddPort(csOrgPort);
                    csNewGroupElm.GetViewerElm().AddPortUsedByUndoRedoOnly(csOrgPort.GetViewerPort());
                    for (int iLp2 = 0; iLp2 < csDelOuterLineList.Count; iLp2++)
                    {// 接続線数ループ
                        csOrgLine = csDelOuterLineList[iLp2];
                        csOrgLine.GetPortID(ref csFromID3, ref csToID3);
                        if ((csFromID1.Equals(csFromID3) == true) && (csToID1.Equals(csToID3) == true))
                        {// IDが一致した場合
                            if (csOrgPort.GetID().Equals(csFromID1) == true)
                            {// IDが一致した場合
                                csOrgPort2 = csOrgLine.GetPort(csToID1);
                                for (int iLp3 = 0; iLp3 < csElmList.Count; iLp3++)
                                {// 要素数ループ
                                    if (csElmList[iLp3].GetID().Equals(csOrgPort2.GetParentElm().GetID()) == true)
                                    {// IDが一致した場合
                                        csNewPort = csElmList[iLp3].GetPort(csToID1);
                                        if (csNewPort != null)
                                        {// 端子があった場合
                                            if (csNewPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                                            {// 流入中継端子の場合
                                                csNewPort.SetPortID(csFromID1, csToID1);
                                            }
                                            else
                                            {// 流入中継端子以外の場合
                                                ((McGraphicsTransElement)csNewPort).SetTransID(csFromID1, csToID1);
                                            }
                                            csOrgPort2 = csNewPort;
                                        }
                                        else
                                        {// 端子がない場合
                                            csElmList[iLp3].AddPort(csOrgPort2);
                                            csElmList[iLp3].GetViewerElm().AddPortUsedByUndoRedoOnly(csOrgPort2.GetViewerPort());
                                            ReSetPortPosition(csElmList[iLp3]);
                                        }
                                        break;
                                    }
                                }
                                csOrgLine.DelPort(1);
                                csOrgLine.DelPort(0);
                                csOrgLine.AddPort(csOrgPort);
                                csOrgLine.AddPort(csOrgPort2);
                            }
                            else
                            {// IDが一致しない場合
                                csOrgPort2 = csOrgLine.GetPort(csFromID1);
                                for (int iLp3 = 0; iLp3 < csElmList.Count; iLp3++)
                                {// 要素数ループ
                                    if (csElmList[iLp3].GetID().Equals(csOrgPort2.GetParentElm().GetID()) == true)
                                    {// IDが一致した場合
                                        csNewPort = csElmList[iLp3].GetPort(csFromID1);
                                        if (csNewPort != null)
                                        {// 端子があった場合
                                            if (csNewPort.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN)
                                            {// 流入中継端子の場合
                                                ((McGraphicsTransElement)csNewPort).SetTransID(csFromID1, csToID1);
                                            }
                                            else
                                            {// 流入中継端子以外の場合
                                                csNewPort.SetPortID(csFromID1, csToID1);
                                            }
                                            csOrgPort2 = csNewPort;
                                        }
                                        else
                                        {// 端子がない場合
                                            csElmList[iLp3].AddPort(csOrgPort2);
                                            csElmList[iLp3].GetViewerElm().AddPortUsedByUndoRedoOnly(csOrgPort2.GetViewerPort());
                                            ReSetPortPosition(csElmList[iLp3]);
                                        }
                                        break;
                                    }
                                }
                                csOrgLine.DelPort(1);
                                csOrgLine.DelPort(0);
                                csOrgLine.AddPort(csOrgPort2);
                                csOrgLine.AddPort(csOrgPort);
                            }
                            if (csNewGroupElm.GetParentElm() != null)
                            {// 親要素がある場合
                                ((McGraphicsGroupElement)csNewGroupElm.GetParentElm()).AddLineElement(csOrgLine);
                                ((McStructGroupElement)csNewGroupElm.GetParentElm().GetViewerElm()).AddConnectionUsedByUndoRedoOnly(csOrgLine.GetViewerConnect());
                            }
                            else
                            {// 親要素がない場合
                                csLineList.Add(csOrgLine);
                                m_csMcStructInfo.AddConnectionUsedByUndoRedoOnly(csOrgLine.GetViewerConnect());
                            }
                            break;
                        }
                    }
                }
                else if (((csFromID1.ToString() == "") || (csToID1.ToString() == "")) &&
                         ((csFromID2.ToString() == "") || (csToID2.ToString() == "")))
                {// 未接続（中継端子のみ存在）
                    csNewGroupElm.AddPort(csOrgPort);
                    csNewGroupElm.GetViewerElm().AddPortUsedByUndoRedoOnly(csOrgPort.GetViewerPort());
                }
            }
        }


        /// <summary><para>method outline:</para>
        /// <para>選択要素全回転</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ElementRotate(dRotate, ref csElmIDList)</para>
        /// </example>
        /// <param name="dRotate">回転角度</param>
        /// <param name="csElmIDList">回転要素IDリスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void AllElementRotate(double dRotate,  ref List<HySIdentifier> csElmIDList)
        {
            //選択中の要素のみを取得
            List<McGraphicsElement> csSelectElementList = new List<McGraphicsElement>();
            List<McGraphicsElement> csSelectRlyElementList = new List<McGraphicsElement>();
            foreach (McGraphicsElement csElement in m_csSelectionList)
            {
                if (csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT ||
                    csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP ||
                    csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP
                    )
                {   // 演算要素
                    csSelectElementList.Add(csElement);
                }
            }

            // 中継端子が選択されているかチェックしリストに積む
            if (m_csPossessionElm != null)
            {// グループ内の場合
                for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
                {// 端子数ループ
                    foreach (McGraphicsElement csElement in m_csSelectionList)
                    {
                        if (csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN ||
                            csElement.GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT
                            )
                        {   // 中継端子
                            if (csElement.GetID().GetString().Equals(m_csPossessionElm.GetPort(iLp).GetID().GetString()) == true)
                            {   // データ一致
                                csSelectRlyElementList.Add(m_csPossessionElm.GetPort(iLp));
                            }
                        }
                    }
                }
            }

            //中心位置の取得
            long lCentralX = 0;
            long lCentralY = 0;
            McViewTransFormer.ProjectCenterPositionCalc(csSelectElementList, ref lCentralX, ref lCentralY);


            //要素数分回転させる
            long lElmX = 0;
            long lElmY = 0;
            //要素
            for (int iCnt = 0; iCnt < csSelectElementList.Count; iCnt++)
            {
                if (csSelectElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT ||
                    csSelectElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP ||
                    csSelectElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
                {
                    csSelectElementList[iCnt].GetPosition(ref lElmX, ref lElmY);
                    if (dRotate == McDefine.HYM_DRAW_ANGLE_M90)
                    {
                        McViewTransFormer.AllLeft90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
                    }
                    else if (dRotate == McDefine.HYM_DRAW_ANGLE_90)
                    {
                        McViewTransFormer.AllRight90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
                    }
                    csSelectElementList[iCnt].SetPosition(lElmX, lElmY);
                }
            }
            //要素の端子
            for (int iCnt1 = 0; iCnt1 < csSelectElementList.Count; iCnt1++)
            {
                if (csSelectElementList[iCnt1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT ||
                    csSelectElementList[iCnt1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP ||
                    csSelectElementList[iCnt1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
                {
                    for (int iCnt2 = 0; iCnt2 < csSelectElementList[iCnt1].GetPortCount(); iCnt2++)
                    {
                        csSelectElementList[iCnt1].GetPort(iCnt2).GetPosition(ref lElmX, ref lElmY);
                        if (dRotate == McDefine.HYM_DRAW_ANGLE_M90)
                        {
                            McViewTransFormer.AllLeft90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
                        }
                        else if (dRotate == McDefine.HYM_DRAW_ANGLE_90)
                        {
                            McViewTransFormer.AllRight90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
                        }
                        csSelectElementList[iCnt1].GetPort(iCnt2).SetPosition(lElmX, lElmY);
                    }
                }
            }

            //if (m_csPossessionElm != null)
            //{// グループ内の場合
            //    for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
            //    {// 端子数ループ
            //        ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp)).GetInnerPosition(ref lElmX, ref lElmY);
            //        if (dRotate == McDefine.HYM_DRAW_ANGLE_M90)
            //        {
            //            McViewTransFormer.AllLeft90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
            //        }
            //        else if (dRotate == McDefine.HYM_DRAW_ANGLE_90)
            //        {
            //            McViewTransFormer.AllRight90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
            //        }
            //        ((McGraphicsTransElement)m_csPossessionElm.GetPort(iLp)).SetInnerPosition(lElmX, lElmY);
            //    }
            //}

            // 中継端子
            for (int iCnt = 0; iCnt < csSelectRlyElementList.Count; iCnt++)
            {
                if (csSelectRlyElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN ||
                    csSelectRlyElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
                {   // 中継端子
                    ((McGraphicsTransElement)csSelectRlyElementList[iCnt]).GetInnerPosition(ref lElmX, ref lElmY);
                    if (dRotate == McDefine.HYM_DRAW_ANGLE_M90)
                    {
                        McViewTransFormer.AllLeft90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
                    }
                    else if (dRotate == McDefine.HYM_DRAW_ANGLE_90)
                    {
                        McViewTransFormer.AllRight90degRotation(lCentralX, lCentralY, ref lElmX, ref lElmY);
                    }
                    ((McGraphicsTransElement)csSelectRlyElementList[iCnt]).SetInnerPosition(lElmX, lElmY);
                }
            }

            // 要素
            for (int iCnt = 0; iCnt < csSelectElementList.Count; iCnt++)
            {
                if (csSelectElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT ||
                    csSelectElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP ||
                    csSelectElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
                {   // 要素
                    double dElmRotate = csSelectElementList[iCnt].GetRotate();
                    csElmIDList.Add(csSelectElementList[iCnt].GetID());
                    dElmRotate = dElmRotate + dRotate;
                    if (dElmRotate < McDefine.HYM_DRAW_ANGLE_0)
                    {// 向きが０より小さい場合
                        dElmRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmRotate;
                    }
                    else if (dElmRotate >= McDefine.HYM_DRAW_ANGLE_360)
                    {// 向きが３６０以上の場合
                        dElmRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmRotate;
                    }
                    csSelectElementList[iCnt].SetRotate(dElmRotate);
                }
            }

            //要素端子
            for (int iCnt1 = 0; iCnt1 < csSelectElementList.Count; iCnt1++)
            {   // 要素
                if (csSelectElementList[iCnt1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_STD_ELEMENT ||
                   csSelectElementList[iCnt1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_GROUP ||
                   csSelectElementList[iCnt1].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_CALC_GROUP)
                {
                    for (int iCnt2 = 0; iCnt2 < csSelectElementList[iCnt1].GetPortCount(); iCnt2++)
                    {
                        double dElmPortRotate = csSelectElementList[iCnt1].GetPort(iCnt2).GetRotate();

                        dElmPortRotate = dElmPortRotate + dRotate;
                        if (dElmPortRotate < McDefine.HYM_DRAW_ANGLE_0)
                        {// 向きが０より小さい場合
                            dElmPortRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmPortRotate;
                        }
                        else if (dElmPortRotate >= McDefine.HYM_DRAW_ANGLE_360)
                        {// 向きが３６０以上の場合
                            dElmPortRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmPortRotate;
                        }
                        csSelectElementList[iCnt1].GetPort(iCnt2).SetRotate(dElmPortRotate);
                    }
                }
            }

            //if (m_csPossessionElm != null)
            //{// グループ内の場合
            //    for (int iLp = 0; iLp < m_csPossessionElm.GetPortCount(); iLp++)
            //    {// 端子数ループ
            //        double dElmPortRotate = m_csPossessionElm.GetPort(iLp).GetRotate();

            //        dElmPortRotate = dElmPortRotate + dRotate;
            //        if (dElmPortRotate < McDefine.HYM_DRAW_ANGLE_0)
            //        {// 向きが０より小さい場合
            //            dElmPortRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmPortRotate;
            //        }
            //        else if (dElmPortRotate >= McDefine.HYM_DRAW_ANGLE_360)
            //        {// 向きが３６０以上の場合
            //            dElmPortRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmPortRotate;
            //        }
            //        m_csPossessionElm.GetPort(iLp).SetRotate(dElmPortRotate);
            //        double dElmParentRotate = m_csPossessionElm.GetPort(0).GetParentElm().GetRotate();
            //        dElmParentRotate = dElmParentRotate + dRotate;
            //        if (dElmParentRotate < McDefine.HYM_DRAW_ANGLE_0)
            //        {// 向きが０より小さい場合
            //            dElmParentRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmParentRotate;
            //        }
            //        else if (dElmParentRotate >= McDefine.HYM_DRAW_ANGLE_360)
            //        {// 向きが３６０以上の場合
            //            dElmParentRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmParentRotate;
            //        }
            //        m_csPossessionElm.GetPort(0).GetParentElm().SetRotate(dElmParentRotate);
            //    }
            //}

            // 中継端子
            for (int iCnt = 0; iCnt < csSelectRlyElementList.Count; iCnt++)
            {
                if (csSelectRlyElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_IN ||
                    csSelectRlyElementList[iCnt].GetSymbolDrawKind() == McDefine.SymbolDrawKind.HYM_TRANS_OUT)
                {   // 中継端子
                    double dElmRotate = csSelectRlyElementList[iCnt].GetRotate();
                    dElmRotate = dElmRotate + dRotate;
                    if (dElmRotate < McDefine.HYM_DRAW_ANGLE_0)
                    {// 向きが０より小さい場合
                        dElmRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmRotate;
                    }
                    else if (dElmRotate >= McDefine.HYM_DRAW_ANGLE_360)
                    {// 向きが３６０以上の場合
                        dElmRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmRotate;
                    }
                    csSelectRlyElementList[iCnt].SetRotate(dElmRotate);
                    double dElmParentRotate = csSelectRlyElementList[iCnt].GetParentElm().GetRotate();
                    dElmParentRotate = dElmParentRotate + dRotate;
                    if (dElmParentRotate < McDefine.HYM_DRAW_ANGLE_0)
                    {// 向きが０より小さい場合
                        dElmParentRotate = McDefine.HYM_DRAW_ANGLE_360 + dElmParentRotate;
                    }
                    else if (dElmParentRotate >= McDefine.HYM_DRAW_ANGLE_360)
                    {// 向きが３６０以上の場合
                        dElmParentRotate = McDefine.HYM_DRAW_ANGLE_360 - dElmParentRotate;
                    }
                    csSelectRlyElementList[iCnt].GetParentElm().SetRotate(dElmParentRotate);
                }
            }
        }
    }
}
