// <summary>ソースコード：現状計算型演算モデルクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.Interface.Controller;


namespace CommonMP.HYMCO.Interface.Model
{
    /// <summary><para>class outline:</para>
    /// <para>現在状態計算型演算モデル基底クラス（通常　演算処理は本クラスを派生して作成する）</para>
    /// </summary>
    /// <remarks>
    /// <para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/08/01][新規作成]</para>
    /// <para>remarks:</para>
    /// <para> 補間処理付き </para>
    /// <para>モデル特性：現在の入力値から現在のモデル内状態を計算し出力する（時間は進まない）</para>
    /// </remarks>
    public abstract class McStateCalModelBase : McBasicCalculateModelBase, McStateCalTypeModel
    {
        /// <summary>演算が完了したシミュレーション時刻</summary>
        protected HySTime m_csCalDoneSimulationTime = HySTime.DEFAULT_TIME.Clone();
        /// <summary>最終目標時刻</summary>
        protected HySTime m_csFinalGoalTime = HySTime.DEFAULT_TIME.Clone();
        /// <summary>刻み時間</summary>
        protected HySTime m_csOrgDltTime = new HySTime(999999999,0,0,0);
        /// <summary>演算時刻保存</summary>
        protected HySTime m_csMemoryFinalGoalTime = HySTime.DEFAULT_TIME.Clone();
        /// <summary>演算経過時刻保存</summary>
        protected HySTime m_csMemoryOrgDltTime = HySTime.DEFAULT_TIME.Clone();

        /// <summary><para>method outline:</para>
        /// <para>モデルを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(ref csInitialData, ref csInputDataList, ref csOutputDataList)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <param name="csInputDataList">前段接続要素からの伝送情報リスト</param>
        /// <param name="csOutputDataList">前段接続要素への伝送情報リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Initialize(ref McPropertyInfoRoot csInitialData, ref HySDataLinkedList csInputDataList, ref HySDataLinkedList csOutputDataList)
        {
            m_csOrgDltTime.SetTime(m_csDltTime); // δTの設定
            bool bRtn = base.Initialize(ref csInitialData, ref csInputDataList, ref csOutputDataList);
            // 内部で　DataFusion() を呼ばないことに注意
            if (this.m_csElement.GetDataFusionTiming() == McDefine.DataFusionTiming.SYNCHRONOUS)
            {
                base.DataFusion(ref csOutputDataList);
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算中に次の演算目標時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetNextTargetTime(ref csCalTime,ref csDltTime)</para>
        /// </example>
        /// <param name="csCalTime">現在時刻</param>
        /// <param name="csDltTime">増加時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetNextTargetTime(ref HySTime csCalTime, ref HySTime csDltTime)
        {
            //m_csTgtTime = csCalTime + csDltTime; // 速度アップの為には、＋演算子を私用しないほうが良い
            m_csTgtTime.SetTime(csCalTime.GetTime() + csDltTime.GetTime());

            if (m_csTgtTime.After(m_csFinalGoalTime))   //(最終演算ならば)
            {   // 強制的に終了時刻にあわせる
                m_csTgtTime.SetTime(m_csFinalGoalTime);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算における収束判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsConverged( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=true:収束完了 =false:未収束</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool IsConverged()
        {
            bool bRtn = false;

            if (m_csSimTime.After(m_csTgtTime) == true)
            {   // 計算時刻が、目標時刻を超えたならば
                if (m_csSimTime.Before(m_csFinalGoalTime) != true &&
                    m_csFinalGoalTime.GetTime() > (m_csSimTime.GetTime() - m_csDltTime.GetTime()))     //m_csFinalGoalTime.After(m_csSimTime - m_csDltTime) == true)  <-- 演算スピードが遅くなる為オペレーター演算子の使用は控える。
                {   // 計算時刻が、目標時刻を超えていても、最終目標時刻を計算していないならば、最終計算時刻の計算を行わせるために、収束していないと判断する
                    m_csOrgDltTime.SetTime(m_csDltTime);
                    m_csDltTime.SetTime(m_csFinalGoalTime.GetTime() - (m_csSimTime.GetTime() - m_csDltTime.GetTime()));
                    m_csSimTime.SetTime(m_csFinalGoalTime);
                }
                else
                {
                    bRtn = true;
                }
            }
            if (bRtn == false && m_eDtFusionTiming == McDefine.DataFusionTiming.ASYNCHRONOUS)
            {   // 先の条件で収束していないとき(但し非同期計算の場合)
                // 上流の全ての要素が必要時刻まで計算されていなければ計算を開始しない
                // 即ち　IsConverged=true で返す。（計算が行われない為、シミュレーション時刻も進行しない）
                for (long lP = 0; lP < m_lInputDataNum; lP++)
                {   // 入力データ数分
                    if (m_csSimTime.After(m_InputCellData[lP].GetLastTime()) == true)
                    //if (m_csSimTime.After(m_csUpperElm[lP].GetCalModel().GetCalDoneTime()) == true)
                    {
                        //McLog.DebugOut(GetSimulationTime(), GetID(), "MyHydrologicalModel", "IsConverged", "UpperElement NotCalDone" + HySCalendar.ToString(m_csUpperElm[lP].GetCalModel().GetCalDoneTime(), HySCalendar.FORMAT.lSW_HOUR));
                        //McLog.DebugOut(GetSimulationTime(), GetID(), "MyHydrologicalModel", "IsConverged", " (Received Time=)" + HySCalendar.ToString(m_InputCellData[lP].GetLastTime(), HySCalendar.FORMAT.lSW_HOUR));
                        bRtn = true;
                    }
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算結果を外部のエレメントに対して公開する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = DataFusion(ref csOutputDataList)</para>
        /// </example>
        /// <param name="csOutputDataList">演算結果を公開する出力情報リスト</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override long DataFusion(ref HySDataLinkedList csOutputDataList)
        {
            long lRtn = base.DataFusion(ref csOutputDataList);

            // 此処まで計算が終了し、外部公開した。従って、演算完了シミュレーション時刻を設定できる
            m_csCalDoneSimulationTime.SetTime(m_csSimTime);


            // データフュージョンの後シミュレーション時刻を進める
            m_csSimTime.Add(m_csDltTime);
            m_csTotalPassingTime.SetTime(m_csSimTime.GetTime() - m_csStartTime.GetTime());
             
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>モデル内部の時刻を進める</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GainSimulationTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要に応じて派生クラスでオーバーライドする。 (この場合 overrideを忘れないこと)</para>
        /// </remarks>
        public override void GainSimulationTime()
        {
            if (m_csDltTime.Equals(m_csOrgDltTime) != true)
            {
                m_csDltTime.SetTime(m_csOrgDltTime);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>演算済みのシミュレーション時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySTime csSimuTime = GetCalDoneTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime 演算済みのシミュレーション時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算済みのシミュレーション時刻と内部のシミュレーション時刻が一致しないモデルの場合に必要</para>
        /// </remarks>
        public override HySTime GetCalDoneTime()
        {
            return m_csCalDoneSimulationTime;
        }
        /// <summary><para>method outline:</para>
        /// <para>最終目標時間設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuGoalTime( csTm )</para>
        /// </example>
        /// <param name="csTm">最終目標時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetSimuGoalTime(HySTime csTm)
        {
            m_csFinalGoalTime.SetTime(csTm);
        }

        /// <summary><para>method outline:</para>
        /// <para>現在状態の一時記憶（但し計算途中状態は除く）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Memorize() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>Memorize()とRemember()は対で使用する</para>
        /// </remarks>
        public override void Memorize()
        {
            base.Memorize();
            m_csMemoryFinalGoalTime=m_csFinalGoalTime.Clone();
            m_csMemoryOrgDltTime = m_csOrgDltTime.Clone();

        }
        /// <summary><para>method outline:</para>
        /// <para>Memorize()した情報の復元</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Remember() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>Memorize()とRemember()は対で使用する</para>
        /// </remarks>
        public override void Remember()
        {
            base.Remember();
            m_csFinalGoalTime = m_csMemoryFinalGoalTime.Clone();
            m_csOrgDltTime = m_csMemoryOrgDltTime.Clone();
        }

        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        // Ver1.3で下記コードを無効とする（現状計算型の同期演算対応） 
        //★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★★
        # region 下記overrideして現状計算型の対応する演算制御を"非同期"に限定している処理を無効にする
        ///// <summary><para>method outline:</para>
        ///// <para>DataFusion()発行のタイミング（演算フロー制御）設定</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para>SetDataFusionTiming(eDtFusionTiming)</para>
        ///// </example>
        ///// <param name="eDtFusionTiming"> 同期／非同期
        ///// McDefine.DataFusionTiming.SYNCHRONOUS：全ての要素が演算終了後DataFutionを行う
        ///// McDefine.DataFusionTiming.ASYNCHRONOUS：非同期：要素が演算終了後個別にDataFutionを行う）
        ///// </param>
        ///// <returns>無し</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public override void SetDataFusionTiming(McDefine.DataFusionTiming eDtFusionTiming)
        //{
        //    // Do Nothing  <-- 重要
        //}
        ///// <summary><para>method outline:</para>
        ///// <para>DataFusion()発行のタイミング（演算フロー制御）取得</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para>McDefine.DataFusionTiming eTm = GetDataFusionTiming()</para>
        ///// </example>
        ///// <param name=""> 無し</param>
        ///// <returns>同期／非同期</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public override McDefine.DataFusionTiming GetDataFusionTiming()
        //{
        //    return McDefine.DataFusionTiming.ASYNCHRONOUS; // <- 重要
        //}
        # endregion

        /// <summary><para>method outline:</para>
        /// <para>ファイルにモデル内情報を全て書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileOUT(csData)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileOUT(HySDataRoot csData)
        {
            bool bRtn = base.FileOUT(csData);
            if (bRtn == true)
            {
                ((McCmnElementOutData)csData).SetTimeData(McStateCalModelBase.CAL_DONE_TIME, m_csCalDoneSimulationTime);
                ((McCmnElementOutData)csData).SetTimeData(McStateCalModelBase.FINAL_GOAL_TIME, m_csFinalGoalTime);
                ((McCmnElementOutData)csData).SetTimeData(McStateCalModelBase.ORG_DLT_TIME, m_csOrgDltTime);
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>ファイルからモデル情報を全て読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = FileIN(csData)</para>
        /// </example>
        /// <param name="csData">演算要素データ</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool FileIN(HySDataRoot csData)
        {
            bool bRtn = base.FileIN(csData);
            if (bRtn == true)
            {
                m_csCalDoneSimulationTime = (HySTime)((McCmnElementOutData)csData).GetTimeData(McStateCalModelBase.CAL_DONE_TIME);
                m_csFinalGoalTime = (HySTime)((McCmnElementOutData)csData).GetTimeData(McStateCalModelBase.FINAL_GOAL_TIME);
                m_csOrgDltTime = (HySTime)((McCmnElementOutData)csData).GetTimeData(McStateCalModelBase.ORG_DLT_TIME);
            }
            return bRtn;
        }


        /// <summary>最終演算目標時刻データ</summary>
        static public readonly HySString CAL_DONE_TIME = new HySString("m_csCalDoneSimulationTime");
        /// <summary>最終演算目標時刻データ</summary>
        static public readonly HySString FINAL_GOAL_TIME = new HySString("m_csFinalGoalTime");
        /// <summary>初期刻み時間データ</summary>
        static public readonly HySString ORG_DLT_TIME = new HySString("m_csOrgDltTime");
    }
}
