﻿// <summary>ソースコード：サンプル演算モデルクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Model;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Tool;



namespace CommonMP.HYMCO.OptionImpl.McSampleModelForDeveloper
{
    /// <summary><para>class outline:</para>
    /// <para>サンプル河道（KinematicWave)モデルクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// <para>remarks</para>
    /// <para>演算速度を考慮しないで、プログラムの判り易さを重視したプログラム
    /// 速度を考慮する場合には、繰り返し演算中にコールされるメソッド
    /// (Calculate、DataFusion)　を改善する必要がある
    /// →　McSampleKinematicWave　クラス参照
    /// </para>
    /// </remarks>
    public class McPrimitiveKinematicWave : McForecastModelBase
    {
        /// <summary> 演算データ（キャスト用） </summary>
        McPrimitiveKinematicWaveCalInfo m_KMdlInf = null; // 便利の為、キャスト用に定義しておく

        //=========================
        // 演算実行前処理関連メソッド
        //=========================


        /// <summary><para>method outline:</para>
        /// <para>入力側の接続情報チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReceiveConnectionCheck(ref csErrorInf)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>受信するデータが自モデルが期待している情報か否かをチェックする</para>
        /// </remarks>
        protected override bool ReceiveConnectionCheck(ref McStructErrorInfo csErrorInf, long lInputDataNum, McReceiveCellDataIF[] csInputCellData)
        {
            bool bRtn = true;

            if (lInputDataNum == 0)
            {
                // 入力が無い場合
                // エラー
                csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, "No receive data.");
                bRtn = false;
            }
            for (long lP = 0; lP < lInputDataNum; lP++)
            {   // 入力データ数分
                if (McModelSampleDefine.IN_PATTERN_TOP.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                {
                    // Do Nothing
                }
                else if (McModelSampleDefine.IN_PATTERN_LATERAL.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                {
                    // Do Nothing
                }
                else
                {
                    csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                        "Unexpected receive data type. (Received from " + csInputCellData[lP].GetUpperElementID().ToString() + ")");
                    bRtn = false;
                }
                long lD1 = 0; long lD2 = 0; long lD3 = 0; long lCellDataNum = 0;
                long lCellDim = csInputCellData[lP].GetDimension(ref lD1, ref lD2, ref lD3, ref lCellDataNum);

                if (lCellDim == 1 && lD1 == 1)
                {   // 1次元の１セルでならば
                    if (lCellDataNum < 1)
                    {
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                            "Variable number in the Cells is too short. (Received from " + csInputCellData[lP].GetUpperElementID().ToString() + ")");
                        bRtn = false;
                    }
                }
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>出力側の接続情報チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SendConnectionCheck(ref csErrorInf)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <param name="lOutputDataNum">出力情報数</param>
        /// <param name="csOutputCellData">出力情報配列</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>送信端子に設定されている伝送データが自モデルが期待している情報か否かをチェックする</para>
        /// </remarks>
        protected override bool SendConnectionCheck(ref McStructErrorInfo csErrorInf, long lOutputDataNum, McSendCellDataIF[] csOutputCellData)
        {
            bool bRtn = true;

            if (lOutputDataNum == 0)
            {
                // 警告（エラーではない）
                csErrorInf.AddCheckWarningData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, "No send port.");
            }
            for (long lP = 0; lP < lOutputDataNum; lP++)
            {   // 出力側接続線数分チェックを行う
                if (McModelSampleDefine.OUT_PATTERN_BOTTOM.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                {   // 最終セルだけの出力ならば
                    // Do Nothing
                }
                else if (McModelSampleDefine.OUT_PATTERN_ALL.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                {   // 全体出力ならば
                    // Do Nothing
                }
                else
                {
                    // エラー
                    csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                        "Unexpected send data type.");
                    bRtn = false;
                }

                long lD1 = 0; long lD2 = 0; long lD3 = 0; long lCellDataNum = 0;
                long lCellDim = csOutputCellData[lP].GetDimension(ref lD1, ref lD2, ref lD3, ref lCellDataNum);

                if (lCellDim == 1 && lD1 == 1)
                {   // １次元配列の１セルならば
                    // セル配列数が１ならば、最終セルのみ出力するものとする
                    // セル内の変数の数をチェック
                    if (lCellDataNum < 3)
                    {
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                            "Variable number in the Cells is too short. (Send To " + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                        bRtn = false;
                    }
                }
                else if (lCellDim == 1 && lD1 > 1)
                {   // １次元配列で複数セルならば
                    // セル配列数が複数ならば、全セルの出力とする
                    // 伝送セルの配列数が　モデル内のセルの配列数よりも小さいならば
                    if (m_KMdlInf.m_lCelNumber > lD1)
                    {
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                                                "Cell dimention(" + csOutputCellData[lP].GetTranInfoKind().ToString() + ") is too short. (send to " + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                        bRtn = false;
                    }
                    // セル内の変数の数をチェック
                    if (lCellDataNum < 3)
                    {
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                            "Variable number in the Cells is too short. (Send To " + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                        bRtn = false;
                    }
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(csInitialData)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override bool Initialize(ref McPropertyInfoRoot csInitialData, long lInputDataNum, ref McReceiveCellDataIF[] csInputCellData)
        {
            McInitialInfo csInDt = csInitialData as McInitialInfo;
            if (csInDt != null)
            {   // 初期化情報有り
                double dDtH = 0.0;
                csInDt.GetInfo("H_Initial", ref dDtH);

                double dDtQ = 0.0;
                dDtQ = (200.0 + (2000.0 - 200.0) * (Math.Pow((0.0 * Math.Exp(1.0 - 0.0)), 10)));  // 仮 <-- 読み込まれると上書きされる
                csInDt.GetInfo("Q_Initial", ref dDtQ);

                double dDtV = 0.0;
                csInDt.GetInfo("V_Initial", ref dDtV);


                for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                {
                    dDtH = Math.Pow(
                               (m_KMdlInf.m_dRough[lLp] * m_KMdlInf.m_dRough[lLp] * dDtQ * dDtQ) / m_KMdlInf.m_dSlope[lLp]
                                 , 0.3);
                    dDtV = dDtQ / dDtH;

                    m_KMdlInf.m_dH[lLp] = dDtH;
                    m_KMdlInf.m_dV[lLp] = dDtV;
                    m_KMdlInf.m_dQ[lLp] = dDtQ;
                }
            }
            return true;
        }

        //=======================
        // 演算実行処理関連メソッド
        //=======================

        /// <summary><para>method outline:</para>
        /// <para>モデル演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = Calculate(ref csInputCellData)</para>
        /// </example>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override long Calculate(long lInputDataNum, ref McReceiveCellDataIF[] csInputCellData)
        {
            McModelCellCorrespondenceInf csCellMng = null; // 横流入位置（０相対セル番号）

            double dElmIn = 0.0; // 個別流入量

            double dQTopTotalIn = 0.0; // 上流総合流入量リセット
            long lQIdx = 0;  // 流量は０番目

            // 横流入量リセット
            for (long lLp = 1; lLp < m_KMdlInf.m_lCelNumber; lLp++)
            {
                m_KMdlInf.m_dInQ[lLp] = 0.0;
            }

            HySCellData csCell = null;  // 取得するセル
            for (long lP = 0; lP < m_lInputDataNum; lP++)
            {   // 入力データ数分
                csCell = csInputCellData[lP].GetInterpolatedCell(0); // セル配列の０番目のセルを取得
                dElmIn = csInputCellData[lP].Data(csCell, lQIdx);   // セル内の０番目の変数値を取得

                if (McModelSampleDefine.IN_PATTERN_TOP.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                {   // 上流端入力ならば
                    dQTopTotalIn += dElmIn;
                }
                else if (McModelSampleDefine.IN_PATTERN_LATERAL.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                {   // 横流入力ならば
                    csCellMng = csInputCellData[lP].GetCellIDData() as McModelCellCorrespondenceInf;
                    if (csCellMng != null)
                    {
                        long lIdxInfNumber = 0;
                        McModelCellCorrespondenceInf.McCellIdxInf[] csCellIdx = csCellMng.GetCellIdxInf(ref lIdxInfNumber);
                        if (lIdxInfNumber > 0)
                        {
                            m_KMdlInf.m_dInQ[csCellIdx[0].m_lLowerModelCellIndexDim1] += dElmIn / m_KMdlInf.m_dWidth; // 単位幅当たり横流入量
                        }
                        else
                        {
                            m_KMdlInf.m_dInQ[m_KMdlInf.m_lDefaultInLQ] += dElmIn; // デフォルト位置流入
                        }

                    }
                    else
                    {
                        m_KMdlInf.m_dInQ[m_KMdlInf.m_lDefaultInLQ] += dElmIn;   // デフォルト位置流入
                    }
                }
            }


            double dQin = dQTopTotalIn / m_KMdlInf.m_dWidth;  // 上流端総合入力
            //McLog.DebugOut(this.m_csSimTime, GetID(), "McPrimitiveKinematicWave", "Calculate", " dQin=" + ((double)dQin).ToString());

            double dTimeLag = m_csDltTime.GetTime();
            double[] dH = new double[m_KMdlInf.m_lCelNumber];

            // 始点
            dH[0] = Math.Pow(
                           (m_KMdlInf.m_dRough[0] * m_KMdlInf.m_dRough[0] * dQin * dQin) / m_KMdlInf.m_dSlope[0]
                             , 0.3);

            // 河道内の断面ごとの水位、流速、流量の算定
            double dFV = 0.0;
            for (long lLp = 1; lLp < m_KMdlInf.m_lCelNumber; lLp++)
            {
                dFV = 5.0 / 3.0 * m_KMdlInf.m_dV[lLp]
                    * (m_KMdlInf.m_dH[lLp] - m_KMdlInf.m_dH[lLp - 1])
                    / m_KMdlInf.m_dCelLength
                    - m_KMdlInf.m_dInQ[lLp - 1]
                    / m_KMdlInf.m_dCelLength;   // lateral flow
                dH[lLp] = m_KMdlInf.m_dH[lLp] - dFV * dTimeLag;
            }

            for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
            {
                m_KMdlInf.m_dH[lLp] = dH[lLp];
                m_KMdlInf.m_dV[lLp] = (1.0 / m_KMdlInf.m_dRough[lLp])
                                      * Math.Pow(m_KMdlInf.m_dH[lLp], 2.0 / 3.0)
                                      * Math.Pow(m_KMdlInf.m_dSlope[lLp], 0.5);
                m_KMdlInf.m_dQ[lLp] = m_KMdlInf.m_dV[lLp] * m_KMdlInf.m_dH[lLp] * m_KMdlInf.m_dWidth;
            }
            //                McLog.DebugOut(this.m_csSimTime, GetID(), "McPrimitiveKinematicWave", "Calculate", " m_dH[20]=" + ((double)m_KMdlInf.m_dH[20]).ToString());
            //                McLog.DebugOut(this.m_csSimTime, GetID(), "McPrimitiveKinematicWave", "Calculate", " m_dV[20]=" + ((double)m_KMdlInf.m_dV[20]).ToString());
            //                McLog.DebugOut(this.m_csSimTime, GetID(), "McPrimitiveKinematicWave", "Calculate", " m_dQ[20]=" + ((double)m_KMdlInf.m_dQ[20]).ToString());

            return 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算結果を外部のエレメントに対して公開する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = DataFusion( lOutputDataNum, ref csOutputCellData)</para>
        /// </example>
        /// <param name="lOutputDataNum">出力情報数</param>
        /// <param name="csOutputCellData">出力情報配列</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override long DataFusion(long lOutputDataNum, ref McSendCellDataIF[] csOutputCellData)
        {
            HySCellData[] csSndCellData = null; // 送信セル
            for (long lP = 0; lP < m_lOutputDataNum; lP++)
            {   // 送信データ数繰り返し
                csSndCellData = csOutputCellData[lP].PrepareSendCellD1();

                if (McModelSampleDefine.OUT_PATTERN_BOTTOM.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                {   // 最終セルだけの出力ならば
                    csSndCellData[0].m_dData[0] = m_KMdlInf.m_dQ[m_KMdlInf.m_lCelNumber - 1];
                    csSndCellData[0].m_dData[1] = m_KMdlInf.m_dH[m_KMdlInf.m_lCelNumber - 1];
                    csSndCellData[0].m_dData[2] = m_KMdlInf.m_dV[m_KMdlInf.m_lCelNumber - 1];
                }
                else if (McModelSampleDefine.OUT_PATTERN_ALL.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                {   // 全体出力ならば
                    for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                    {
                        csSndCellData[lLp].m_dData[0] = m_KMdlInf.m_dQ[lLp];
                        csSndCellData[lLp].m_dData[1] = m_KMdlInf.m_dH[lLp];
                        csSndCellData[lLp].m_dData[2] = m_KMdlInf.m_dV[lLp];
                    }
                }
            }
            McLog.DebugOut(GetSimulationTime(), m_csElement.GetID(), "McPrimitiveKinematicWave", "DataFusion", " m_KMdlInf.m_dQ[Last]=" + m_KMdlInf.m_dQ[m_KMdlInf.m_lCelNumber - 1].ToString());
            return 0;
        }

        //====================
        // その他必要なメソッド
        //====================

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetProperty(csCellMdlPropertyInfo)</para>
        /// </example>
        /// <param name="csCellMdlPropertyInfo">セル型プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SetProperty(McCellModelPropertyIF csCellMdlPropertyInfo)
        {
            // 使用しやすいようにキャストしておく
            m_KMdlInf = (McPrimitiveKinematicWaveCalInfo)m_csCalInfo;

            // プロパティ設定
            McCellModelPropertyInfo csPrptyInfo = csCellMdlPropertyInfo as McCellModelPropertyInfo;
            if (csPrptyInfo != null)
            {
                // 演算ステップ時刻設定
                this.m_csDltTime = new HySTime(csPrptyInfo.GetStepTime());

                // 各種係数設定
                csPrptyInfo.GetInfo("m_lCelNumber", ref m_KMdlInf.m_lCelNumber);
                double dRough = 0.03;
                csPrptyInfo.GetInfo("m_dRough", ref dRough);
                double dSloop = 2000.0;
                csPrptyInfo.GetInfo("m_dSlope", ref dSloop);

                m_KMdlInf.m_dH = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dV = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dQ = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dRough = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dSlope = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dInQ = new double[m_KMdlInf.m_lCelNumber];
                for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                {
                    m_KMdlInf.m_dH[lLp] = 1.0;
                    m_KMdlInf.m_dV[lLp] = 0.0;
                    m_KMdlInf.m_dQ[lLp] = 0.0;
                    m_KMdlInf.m_dRough[lLp] = dRough;
                    m_KMdlInf.m_dSlope[lLp] = dSloop;
                    m_KMdlInf.m_dInQ[lLp] = 0.0;
                }
                csPrptyInfo.GetInfo("m_dLength", ref m_KMdlInf.m_dLength);
                m_KMdlInf.m_dCelLength = m_KMdlInf.m_dLength / (double)m_KMdlInf.m_lCelNumber;

                csPrptyInfo.GetInfo("m_dWidth", ref m_KMdlInf.m_dWidth);

                csPrptyInfo.GetInfo("m_inLQ", ref m_KMdlInf.m_lDefaultInLQ);

            }
            else
            {
                this.m_csDltTime = new HySTime(48.0 * 3600.0 / 2400.0); // 

                m_KMdlInf.m_lCelNumber = 50;
                m_KMdlInf.m_dH = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dV = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dQ = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dRough = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dSlope = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dInQ = new double[m_KMdlInf.m_lCelNumber];
                for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                {
                    m_KMdlInf.m_dH[lLp] = 1.0;
                    m_KMdlInf.m_dV[lLp] = 0.0;
                    m_KMdlInf.m_dQ[lLp] = 0.0;
                    m_KMdlInf.m_dRough[lLp] = 0.03;
                    m_KMdlInf.m_dSlope[lLp] = 1.0 / 2000.0;
                    m_KMdlInf.m_dInQ[lLp] = 0.0;
                }

                m_KMdlInf.m_dLength = 100000.0;
                m_KMdlInf.m_dCelLength = 100000.0 / (double)m_KMdlInf.m_lCelNumber;
                m_KMdlInf.m_dWidth = 200.0; // m

                m_KMdlInf.m_lDefaultInLQ = 20;

            }
            return true;
        }
    }
}
