﻿// <summary>ソースコード：ＨＹＭＣＯサンプル演算モデルクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

using CommonMP.HYMCO.Interface;
using CommonMP.HYMCO.Interface.Model;
using CommonMP.HYMCO.Interface.Data;
using CommonMP.HYMCO.CoreImpl.Data;
using CommonMP.HYMCO.CoreImpl.Tool;



namespace CommonMP.HYMCO.OptionImpl.McSampleModelForDeveloper
{
    /// <summary><para>class outline:</para>
    /// <para>サンプル河道（KinematicWave)モデルクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    public class McSampleKinematicWave : McForecastModelBase
    {
        /// <summary> 演算データ（キャスト用） </summary>
        McSampleKinematicWaveCalInfo m_KMdlInf = null; // 便利の為、キャスト用に定義しておく

        //=========================
        // 演算実行前処理関連メソッド
        //=========================


        /// <summary><para>method outline:</para>
        /// <para>入力側の接続情報チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ReceiveConnectionCheck(ref csErrorInf)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>受信するデータが自モデルが期待している情報か否かをチェックする</para>
        /// </remarks>
        protected override bool ReceiveConnectionCheck(ref McStructErrorInfo csErrorInf, long lInputDataNum, McReceiveCellDataIF[] csInputCellData)
        {
            bool bRtn = true;

            if (lInputDataNum == 0)
            {
                // 入力が無い場合
                // エラー
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, 
                    Properties.Resources.STATEMENT_NO_REC_DATA );
               // csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, "No receive data.");
                bRtn = false;
            }
            m_KMdlInf.m_lTopInDataNum = 0;
            m_KMdlInf.m_lSideInDataNum = 0;

            for (long lP = 0; lP < lInputDataNum; lP++)
            {   // 入力データ数分
                if (McModelSampleDefine.IN_PATTERN_TOP.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                {
                    m_KMdlInf.m_lTopInDataNum += 1;
                }
                else if (McModelSampleDefine.IN_PATTERN_LATERAL.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                {
                    m_KMdlInf.m_lSideInDataNum += 1;
                }
                else
                {
                   // ver1.5 エラートレース日本語対応
                    csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                        Properties.Resources.STATEMENT_UNEXPECT_REC_DATA_TYPE_R + csInputCellData[lP].GetUpperElementID().ToString() + ")");
                   // csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                   //     "Unexpected receive data type. (Received from " + csInputCellData[lP].GetUpperElementID().ToString() + ")");
                    bRtn = false;
                }
                long lD1 = 0; long lD2 = 0; long lD3 = 0; long lCellDataNum = 0;
                long lCellDim = csInputCellData[lP].GetDimension(ref lD1, ref lD2, ref lD3, ref lCellDataNum);
                if (lCellDim == 1 && lD1 == 1)
                {   // 1次元の１セルでならば
                    if (lCellDataNum < 1)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                            Properties.Resources.STATEMENT_VAL_NUM_SHORT_R + csInputCellData[lP].GetUpperElementID().ToString() + ")");
                       // csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                       //     "Variable number in the Cells is too short. (Received from " + csInputCellData[lP].GetUpperElementID().ToString() + ")");
                        bRtn = false;
                    }
                }
            }
            if (bRtn == true)
            {
                McModelCellCorrespondenceInf csCellMng = null;
                // 入力指標作成
                m_KMdlInf.m_lTopInIdx = new long[m_KMdlInf.m_lTopInDataNum];
                m_KMdlInf.m_lSideInIdx = new long[m_KMdlInf.m_lSideInDataNum];
                m_KMdlInf.m_lInQIdx = new long[m_KMdlInf.m_lSideInDataNum];

                long lTopInDataNum = 0;
                //long lLefInDataNum = 0;
                //long lRightInDataNum = 0;
                long lSideInDataNum = 0;
                for (long lP = 0; lP < lInputDataNum; lP++)
                {   // 入力データ数分
                    csCellMng = csInputCellData[lP].GetCellIDData() as McModelCellCorrespondenceInf;

                    if (McModelSampleDefine.IN_PATTERN_TOP.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                    {
                        m_KMdlInf.m_lTopInIdx[lTopInDataNum] = lP;
                        lTopInDataNum += 1;
                    }
                    else if (McModelSampleDefine.IN_PATTERN_LATERAL.Equals(csInputCellData[lP].GetReceivePatternID()) == true)
                    {
                        m_KMdlInf.m_lSideInIdx[lSideInDataNum] = lP;
                        if (csCellMng != null)
                        {
                            long lIdxInfNumber=0;
                            McModelCellCorrespondenceInf.McCellIdxInf[] csCellIdx = csCellMng.GetCellIdxInf(ref lIdxInfNumber);
                            if (lIdxInfNumber > 0)
                            {
                                m_KMdlInf.m_lInQIdx[lSideInDataNum] = csCellIdx[0].m_lLowerModelCellIndexDim1;
                            }
                            else
                            {
                                m_KMdlInf.m_lInQIdx[lSideInDataNum] = m_KMdlInf.m_lDefaultInLQ; // 流入位置
                            }
                        }
                        else
                        {
                            m_KMdlInf.m_lInQIdx[lSideInDataNum] = m_KMdlInf.m_lDefaultInLQ; // 流入位置
                        }
                        lSideInDataNum += 1;
                    }
                }
            }

            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>出力側の接続情報チェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SendConnectionCheck(ref csErrorInf)</para>
        /// </example>
        /// <param name="csErrorInf">エラー出力</param>
        /// <param name="lOutputDataNum">出力情報数</param>
        /// <param name="csOutputCellData">出力情報配列</param>
        /// <returns>=true:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>送信端子に設定されている伝送データが自モデルが期待している情報か否かをチェックする</para>
        /// </remarks>
        protected override bool SendConnectionCheck(ref McStructErrorInfo csErrorInf, long lOutputDataNum, McSendCellDataIF[] csOutputCellData)
        {
            bool bRtn = true;

            if (lOutputDataNum == 0)
            {
                // 警告（エラーではない）
               // ver1.5 エラートレース日本語対応
                csErrorInf.AddCheckWarningData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, 
                    Properties.Resources.STATEMENT_NO_SND_PORT );
               // csErrorInf.AddCheckWarningData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, "No send port.");
            }

            m_KMdlInf.m_lLastOutDataNum = 0;
            m_KMdlInf.m_lAllOutDataNum = 0;

            for (long lP = 0; lP < lOutputDataNum; lP++)
            {   // 出力側接続線数分チェックを行う
                if (McModelSampleDefine.OUT_PATTERN_BOTTOM.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                {   // 最終セルだけの出力ならば
                    m_KMdlInf.m_lLastOutDataNum += 1;
                }
                else if (McModelSampleDefine.OUT_PATTERN_ALL.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                {   // 全体出力ならば
                    m_KMdlInf.m_lAllOutDataNum += 1;
                }
                else
                {
                    // エラー
                   // ver1.5 エラートレース日本語対応
                    csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, 
                        Properties.Resources.STATEMENT_UNEXPECT_DATA_TYPE );
                   // csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND, "Unexpected send data type.");
                    bRtn = false;
                }

                long lD1 = 0; long lD2 = 0; long lD3 = 0; long lCellDataNum = 0;
                long lCellDim = csOutputCellData[lP].GetDimension(ref lD1, ref lD2, ref lD3, ref lCellDataNum);
                if (lCellDim == 1 && lD1 == 1)
                {   // １次元配列の１セルならば
                    // セル配列数が１ならば、最終セルのみ出力するものとする
                    // セル内の変数の数をチェック
                    if (lCellDataNum < 3)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                            Properties.Resources.STATEMENT_VAL_NUM_SHORT_S  + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                       // csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                       //     "Variable number in the Cells is too short. (Send To " + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                        bRtn = false;
                    }
                }
                else if (lCellDim == 1 && lD1 > 1)
                {   // １次元配列で複数セルならば
                    // セル配列数が複数ならば、全セルの出力とする
                    // 伝送セルの配列数が　モデル内のセルの配列数よりも小さいならば
                    if (m_KMdlInf.m_lCelNumber > lD1)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                                                Properties.Resources.STATEMENT_SELLDIM + "(" + csOutputCellData[lP].GetTranInfoKind().ToString() + ")"
                                                + Properties.Resources.STATEMENT_IS_TOO_SHORT_S + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                       // csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                       //                         "Cell dimention(" + csOutputCellData[lP].GetTranInfoKind().ToString() + ") is too short. (send to " + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                        bRtn = false;
                    }
                    // セル内の変数の数をチェック
                    if (lCellDataNum < 3)
                    {
                       // ver1.5 エラートレース日本語対応
                        csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                            Properties.Resources.STATEMENT_VAL_NUM_SHORT_S + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                       // csErrorInf.AddCheckErrorData(this.GetID(), McModelSampleDefine.KINEMATIC_WAVE_KIND,
                       //     "Variable number in the Cells is too short. (Send To " + csOutputCellData[lP].GetLowerElementID().ToString() + ")");
                        bRtn = false;
                    }
                }
            }
            if (bRtn == true)
            {
                // 入力指標作成
                m_KMdlInf.m_lLastOutIdx = new long[m_KMdlInf.m_lLastOutDataNum];
                m_KMdlInf.m_lAllOutIdx = new long[m_KMdlInf.m_lAllOutDataNum];

                long lLastOutDataNum = 0;
                long lAllOutDataNum = 0;

                for (long lP = 0; lP < lOutputDataNum; lP++)
                {   // 出力側接続線数分チェックを行う
                    if (McModelSampleDefine.OUT_PATTERN_BOTTOM.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                    {   // 最終セルだけの出力ならば
                        m_KMdlInf.m_lLastOutIdx[lLastOutDataNum] = lP;
                        lLastOutDataNum += 1;
                    }
                    else if (McModelSampleDefine.OUT_PATTERN_ALL.Equals(csOutputCellData[lP].GetSendPatternID()) == true)
                    {   // 全体出力ならば
                        m_KMdlInf.m_lAllOutIdx[lAllOutDataNum] = lP;
                        lAllOutDataNum += 1;
                    }
                }
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデルを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Initialize(csInitialData)</para>
        /// </example>
        /// <param name="csInitialData">初期化設定情報</param>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override bool Initialize(ref McPropertyInfoRoot csInitialData, long lInputDataNum, ref McReceiveCellDataIF[] csInputCellData)
        {
            McInitialInfo csInDt = csInitialData as McInitialInfo;
            if (csInDt != null)
            {   // 初期化情報有り
                double dDtH = 0.0;
                csInDt.GetInfo("H_Initial", ref dDtH);

                double dDtQ = 0.0;
                dDtQ = (200.0 + (2000.0 - 200.0) * (Math.Pow((0.0 * Math.Exp(1.0 - 0.0)), 10)));  // 仮 <-- 読み込まれると上書きされる
                csInDt.GetInfo("Q_Initial", ref dDtQ);

                double dDtV = 0.0;
                csInDt.GetInfo("V_Initial", ref dDtV);


                for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                {
                    dDtH = Math.Pow(
                               (m_KMdlInf.m_dRough[lLp] * m_KMdlInf.m_dRough[lLp] * dDtQ * dDtQ) / m_KMdlInf.m_dSlope[lLp]
                                 , 0.3);
                    dDtV = dDtQ / dDtH;

                    m_KMdlInf.m_dH[lLp] = dDtH;
                    m_KMdlInf.m_dV[lLp] = dDtV;
                    m_KMdlInf.m_dQ[lLp] = dDtQ;
                }
            }
            return true;
        }

        //=======================
        // 演算実行処理関連メソッド
        //=======================

        /// <summary><para>method outline:</para>
        /// <para>モデル演算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = Calculate(ref csInputCellData)</para>
        /// </example>
        /// <param name="lInputDataNum">入力情報数</param>
        /// <param name="csInputCellData">演算に必要な入力情報配列</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected override long Calculate(long lInputDataNum, ref McReceiveCellDataIF[] csInputCellData)
        {
            double dElmIn = 0.0; // 個別流入量

            double dQTopTotalIn = 0.0; // 総合流入量リセット
            long lQIdx = 0;  // 流量は０番目

            HySCellData csCell = null;

            // 上流端入力の合計を求める
            for (long lP = 0; lP < m_KMdlInf.m_lTopInDataNum; lP++)
            {   // 入力データ数分
                csCell = csInputCellData[m_KMdlInf.m_lTopInIdx[lP]].GetInterpolatedCell(0); // セル配列の０番目のセルを取得
                dQTopTotalIn += csInputCellData[m_KMdlInf.m_lTopInIdx[lP]].Data(csCell, lQIdx);   // 流量を取得
            }
            // 各セルへの横流入量を求める
            for (long lLp = 1; lLp < m_KMdlInf.m_lCelNumber; lLp++)
            {
                m_KMdlInf.m_dInQ[lLp] = 0.0;
            }

            for (long lP = 0; lP < m_KMdlInf.m_lSideInDataNum; lP++)
            {   // 入力データ数分
                csCell = csInputCellData[m_KMdlInf.m_lSideInIdx[lP]].GetInterpolatedCell(0); // セル配列の０番目のセルを取得
                dElmIn = csInputCellData[m_KMdlInf.m_lSideInIdx[lP]].Data(csCell, lQIdx);   // 流量を取得
                m_KMdlInf.m_dInQ[m_KMdlInf.m_lInQIdx[lP]] += dElmIn / m_KMdlInf.m_dWidth; // 単位幅当たり横流入量を追加
            }


            double dQin = dQTopTotalIn / m_KMdlInf.m_dWidth;  // 単位幅当たり上流端流入量
            //McLog.DebugOut(this.m_csSimTime, GetID(), "McSampleKinematicWave", "Calculate", " dQin=" + ((double)dQin).ToString());
            
            double dTimeLag = m_csDltTime.GetTime();
            double[] dH = new double[m_KMdlInf.m_lCelNumber];

            // 始点
            dH[0] = Math.Pow(
                           (m_KMdlInf.m_dRough[0] * m_KMdlInf.m_dRough[0] * dQin * dQin) / m_KMdlInf.m_dSlope[0]
                             , 0.3);

            // 河道内の断面ごとの水位、流速、流量の算定
            double dFV = 0.0;
            for (long lLp = 1; lLp < m_KMdlInf.m_lCelNumber; lLp++)
            {
                dFV = 5.0 / 3.0 * m_KMdlInf.m_dV[lLp]
                    * (m_KMdlInf.m_dH[lLp] - m_KMdlInf.m_dH[lLp - 1])
                    / m_KMdlInf.m_dCelLength
                    - m_KMdlInf.m_dInQ[ lLp - 1 ]
                    / m_KMdlInf.m_dCelLength;   // lateral flow
                dH[lLp] = m_KMdlInf.m_dH[lLp] - dFV * dTimeLag;
            }
            for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
            {
                m_KMdlInf.m_dH[lLp] = dH[lLp];
                m_KMdlInf.m_dV[lLp] = (1.0 / m_KMdlInf.m_dRough[lLp])
                                      * Math.Pow(m_KMdlInf.m_dH[lLp], 2.0 / 3.0)
                                      * Math.Pow(m_KMdlInf.m_dSlope[lLp], 0.5);
                m_KMdlInf.m_dQ[lLp] = m_KMdlInf.m_dV[lLp] * m_KMdlInf.m_dH[lLp] * m_KMdlInf.m_dWidth;
            }
            //McLog.DebugOut(this.m_csSimTime, GetID(), "McSampleKinematicWave", "Calculate", " m_dH[20]=" + ((double)m_KMdlInf.m_dH[20]).ToString());
            //McLog.DebugOut(this.m_csSimTime, GetID(), "McSampleKinematicWave", "Calculate", " m_dV[20]=" + ((double)m_KMdlInf.m_dV[20]).ToString());
            //McLog.DebugOut(this.m_csSimTime, GetID(), "McSampleKinematicWave", "Calculate", " m_dQ[20]=" + ((double)m_KMdlInf.m_dQ[20]).ToString());

            return 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>モデル演算結果を外部のエレメントに対して公開する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lRtn = DataFusion( lOutputDataNum, ref csOutputCellData)</para>
        /// </example>
        /// <param name="lOutputDataNum">出力情報数</param>
        /// <param name="csOutputCellData">出力情報配列</param>
        /// <returns>=0:正常 -1:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>複数接続があった場合、全て同じ情報が出ることに注意（流量を分配して出しているわけではない）</para>
        /// </remarks>
        protected override long DataFusion(long lOutputDataNum, ref McSendCellDataIF[] csOutputCellData)
        {
            HySCellData[] csSndCellData = null;

            // 最終セルだけの出力
            for (long lP = 0; lP < m_KMdlInf.m_lLastOutDataNum; lP++)
            {
                csSndCellData = csOutputCellData[m_KMdlInf.m_lLastOutIdx[lP]].PrepareSendCellD1();

                csSndCellData[0].m_dData[0] = m_KMdlInf.m_dQ[m_KMdlInf.m_lCelNumber - 1];
                csSndCellData[0].m_dData[1] = m_KMdlInf.m_dH[m_KMdlInf.m_lCelNumber - 1];
                csSndCellData[0].m_dData[2] = m_KMdlInf.m_dV[m_KMdlInf.m_lCelNumber - 1];
            }
            // 全体出力
            for (long lP = 0; lP < m_KMdlInf.m_lAllOutDataNum; lP++)
            {
                csSndCellData = csOutputCellData[m_KMdlInf.m_lAllOutIdx[lP]].PrepareSendCellD1();
                for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                {
                    csSndCellData[lLp].m_dData[0] = m_KMdlInf.m_dQ[lLp];
                    csSndCellData[lLp].m_dData[1] = m_KMdlInf.m_dH[lLp];
                    csSndCellData[lLp].m_dData[2] = m_KMdlInf.m_dV[lLp];
                }
            }
            McLog.DebugOut(GetSimulationTime(), m_csElement.GetID(), "McSampleKinematicWave", "DataFusion", " m_KMdlInf.m_dQ[0]=" + m_KMdlInf.m_dQ[0].ToString());
            return 0;
        }

        //====================
        // その他必要なメソッド
        //====================

        /// <summary><para>method outline:</para>
        /// <para>プロパティ情報を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetProperty(csCellMdlPropertyInfo)</para>
        /// </example>
        /// <param name="csCellMdlPropertyInfo">セル型プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SetProperty(McCellModelPropertyIF csCellMdlPropertyInfo)
        {
            // 使用しやすいようにキャストしておく
            m_KMdlInf = (McSampleKinematicWaveCalInfo)m_csCalInfo;

            // プロパティ設定
            McCellModelPropertyInfo csPrptyInfo = csCellMdlPropertyInfo as McCellModelPropertyInfo;
            if (csPrptyInfo != null)
            {
                // 演算ステップ時刻設定
                this.m_csDltTime = new HySTime(csPrptyInfo.GetStepTime());

                // 各種係数設定
                csPrptyInfo.GetInfo("m_lCelNumber", ref m_KMdlInf.m_lCelNumber);
                double dRough = 0.03;
                csPrptyInfo.GetInfo("m_dRough", ref dRough);
                double dSloop = 2000.0;
                csPrptyInfo.GetInfo("m_dSlope", ref dSloop);

                m_KMdlInf.m_dH = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dV = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dQ = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dRough = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dSlope = new double[m_KMdlInf.m_lCelNumber];

                m_KMdlInf.m_dInQ = new double[m_KMdlInf.m_lCelNumber];

                for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                {
                    m_KMdlInf.m_dH[lLp] = 1.0;
                    m_KMdlInf.m_dV[lLp] = 0.0;
                    m_KMdlInf.m_dQ[lLp] = 0.0;
                    m_KMdlInf.m_dRough[lLp] = dRough;
                    m_KMdlInf.m_dSlope[lLp] = dSloop;
                    m_KMdlInf.m_dInQ[lLp] = 0.0;
                }
                csPrptyInfo.GetInfo("m_dLength", ref m_KMdlInf.m_dLength);
                m_KMdlInf.m_dCelLength = m_KMdlInf.m_dLength / (double)m_KMdlInf.m_lCelNumber;

                csPrptyInfo.GetInfo("m_dWidth", ref m_KMdlInf.m_dWidth);


                csPrptyInfo.GetInfo("m_inLQ", ref m_KMdlInf.m_lDefaultInLQ);

            }
            else
            {
                this.m_csDltTime = new HySTime(48.0 * 3600.0 / 2400.0); // 

                m_KMdlInf.m_lCelNumber = 50;
                m_KMdlInf.m_dH = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dV = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dQ = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dRough = new double[m_KMdlInf.m_lCelNumber];
                m_KMdlInf.m_dSlope = new double[m_KMdlInf.m_lCelNumber];

                for (long lLp = 0; lLp < m_KMdlInf.m_lCelNumber; lLp++)
                {
                    m_KMdlInf.m_dH[lLp] = 1.0;
                    m_KMdlInf.m_dV[lLp] = 0.0;
                    m_KMdlInf.m_dQ[lLp] = 0.0;
                    m_KMdlInf.m_dRough[lLp] = 0.03;
                    m_KMdlInf.m_dSlope[lLp] = 1.0 / 2000.0;
                }

                m_KMdlInf.m_dLength = 100000.0;
                m_KMdlInf.m_dCelLength = 100000.0 / (double)m_KMdlInf.m_lCelNumber;
                m_KMdlInf.m_dWidth = 200.0; // m


                m_KMdlInf.m_lDefaultInLQ = 20;

            }
            return true;
        }
    
        /// <summary><para>method outline:</para>
        /// <para>演算実行中断中のプロパティ等情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SetOnlineProperty(csCellMdlPropertyInfo)</para>
        /// </example>
        /// <param name="csCellMdlPropertyInfo">プロパティ情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>演算中断中にプロパティ、初期値等を変更して動作させ場合等に使用する(将来拡張用)</para>
        /// </remarks>
        public override bool SetOnlineProperty(McCellModelPropertyIF csCellMdlPropertyInfo)
        {
            // プロパティ設定
            McCellModelPropertyInfo csPrptyInfo = csCellMdlPropertyInfo as McCellModelPropertyInfo;
            if (csPrptyInfo != null)
            {
                // 演算ステップ時刻設定
                this.m_csDltTime = new HySTime(csPrptyInfo.GetStepTime());

                // 各種係数設定
                csPrptyInfo.GetInfo("m_lCelNumber", ref m_KMdlInf.m_lCelNumber);
                double dRough = 0.03;
                csPrptyInfo.GetInfo("m_dRough", ref dRough);
                double dSloop = 2000.0;
                csPrptyInfo.GetInfo("m_dSlope", ref dSloop);

                csPrptyInfo.GetInfo("m_dLength", ref m_KMdlInf.m_dLength);
                m_KMdlInf.m_dCelLength = m_KMdlInf.m_dLength / (double)m_KMdlInf.m_lCelNumber;

                csPrptyInfo.GetInfo("m_dWidth", ref m_KMdlInf.m_dWidth);

                csPrptyInfo.GetInfo("m_inLQ", ref m_KMdlInf.m_lDefaultInLQ);
            }

            return true;
        }
    }
}
