﻿// <summary>ソースコード：表示業務処理親クラス</summary>
// <author>CommonMP</author>

using System;
using System.Threading;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.IO;

using CommonMP.HYSSOP.Interface;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.Interface.HSController;
using CommonMP.HYSSOP.Interface.HSBusiProc;

using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSController;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSSimulator;


namespace CommonMP.HYSSOP.CoreImpl.HSBusiProc
{
    /// <summary><para>class outline:</para>
    /// <para>シミュレーション制御業務の基底クラス</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>
    /// 本クラスは 計算エンジン（HySSimulatorBase） と同じプロセス内に配置される事を前提としている
    /// もし異なるプロセスに配置されるならば、データコンテナの管理を見直す必要がある。
    /// </para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.2.0][2011/10/01][新規作成]</para>
    /// </remarks>
    public abstract class HySSimuCtlBusiProcedureBase : HySBusiProcedureBase
    {
        /// <summary>データコンテナ管理テーブル</summary>
        protected HySDataHashTable m_csDataContainerTbl = new HySDataHashTable();
        /// <summary>計算状態テーブル</summary>
        protected Hashtable m_csSimuStatusTbl = new Hashtable();
        /// <summary>計算進捗状況テーブル</summary>
        protected Hashtable m_csSimuProgressTbl = new Hashtable();

        /// <summary><para>method outline:</para>
        /// <para>初期化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Initialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        override public Boolean Initialize()
        {
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>終了処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Terminate( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        override public Boolean Terminate()
        {
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>イベントを受け取った時に動作するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = EventCallback( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        override public Boolean EventCallback(HySSysEvent csEvent)
        {
            Boolean bRtn = false;
            HySEventObject csEventObj = csEvent as HySEventObject;
            if (csEventObj != null)
            {
                HySKind csSimKind = csEventObj.GetToSimKind();
                // if (m_csSimKindID.Equals(csSimKind) == true) <<-- この判断は不要（本処理前のHySBusiController.EventCallback()内で行っているため
                //{
                long lFromObjNo = csEvent.From();
                long lEventNo = csEventObj.GetEventNo();
                long lSubEventNo = csEventObj.GetSubEventNo();
                HySID csID = csEventObj.GetSuppID();
                HySDataRoot csMainData = csEventObj.GetData();
                HySDataRoot csSuppData = csEventObj.GetSubData();

                if (lEventNo == HySEventObject.CMND_NEW_SIMULATOR_CREATE)
                {   // シミュレーター作成指示
                    bRtn = this.CmdCreateSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_SIMULATOR_CREATE_SUCCESS || lEventNo == HySEventObject.NOTICE_SIMULATOR_CREATE_FAIL)
                {   // シミュレーター作成成功／失敗応答ならば
                    if (lEventNo == HySEventObject.NOTICE_SIMULATOR_CREATE_SUCCESS)
                    {
                        bRtn = this.NoticeSimulatorCreation(csEventObj, true);
                    }
                    else
                    {
                        bRtn = this.NoticeSimulatorCreation(csEventObj, false);
                    }
                }
                else if (lEventNo == HySEventObject.CMND_SIMULATOR_DELETE)
                {   // シミュレーター消去指示ならば
                    bRtn = this.CmdDeleteSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_END_SIMULATOR_DELETE)
                {   // シミュレーター消去応答ならば
                    bRtn = this.NoticeSimulatorDelete(csEventObj);
                }
                else if (lEventNo == HySEventObject.CMND_MODEL_CONSTRUCTION)
                {   // シミュレーター内にシミュレーションモデルを構築指示
                    bRtn = this.CmdModelConstruction(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_OK_MODEL_CONSTRUCTION || lEventNo == HySEventObject.NOTICE_NG_MODEL_CONSTRUCTION)
                {   // シミュレーションモデル作成成功／失敗応答ならば
                    if (lEventNo == HySEventObject.NOTICE_OK_MODEL_CONSTRUCTION)
                    {
                        bRtn = this.NoticeModelConstruction(csEventObj, true);
                    }
                    else
                    {
                        bRtn = this.NoticeModelConstruction(csEventObj, false);
                    }
                }
                else if (lEventNo == HySEventObject.CMND_INITIALIZE_SIMULATION)
                {   // シミュレーション初期化指示
                    bRtn = this.CmdInitializeSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_INITIALIZE_SIMULATION)
                {   // シミュレーション初期化応答ならば
                    bRtn = this.NoticeInitializeSimulation(csEventObj,true);
                }
                else if (lEventNo == HySEventObject.NOTICE_CANNOT_INITIALIZE_SIMULATION)
                {   // シミュレーション初期化失敗応答ならば
                    bRtn = this.NoticeInitializeSimulation(csEventObj,false);
                }
                else if (lEventNo == HySEventObject.CMND_START_SIMULATION)
                {   // シミュレーション開始指示
                    bRtn = this.CmdStartSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.CMND_RESTART_SIMULATION)
                {   // シミュレーション再開指示
                    bRtn = this.CmdRestartSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_START_SIMULATION || lEventNo == HySEventObject.NOTICE_CANNOT_START_SIMULATION)
                {   // 計算開始／計算開始不可応答ならば
                    if (lEventNo == HySEventObject.NOTICE_START_SIMULATION)
                    {
                        bRtn = this.NoticeStartSimulation(csEventObj, true);
                    }
                    else
                    {
                        bRtn = this.NoticeStartSimulation(csEventObj, false);
                    }
                }
                else if (lEventNo == HySEventObject.NOTICE_COMPLETE_SIMULATION || lEventNo == HySEventObject.NOTICE_SIMULATION_ABEND || lEventNo == HySEventObject.NOTICE_SIMULATION_NG)
                {   // 計算完了／異常終了応答ならば
                    if (lEventNo == HySEventObject.NOTICE_COMPLETE_SIMULATION)
                    {
                        bRtn = this.NoticeCompleteSimulation(csEventObj, true);
                    }
                    else
                    {
                        bRtn = this.NoticeCompleteSimulation(csEventObj, false);
                    }
                }
                else if (lEventNo == HySEventObject.CMND_PAUSE_SIMULATION)
                {   // シミュレーション中断指示
                    bRtn = this.CmdPauseSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_PAUSE_SIMULATION)
                {   // シミュレーション中断応答ならば
                    bRtn = this.NoticePauseSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.CMND_STOP_SIMULATION)
                {   // シミュレーション強制終了指示
                    bRtn = this.CmdStopSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_STOP_SIMULATION)
                {   // シミュレーション中止応答ならば
                    bRtn = this.NoticeStopSimulation(csEventObj);
                }
                else if (lEventNo == HySEventObject.CMND_REPORT_SIMU_PROGRESS)
                {   // シミュレーション計算進捗状況報告指示
                    bRtn = this.CmdReportSimuProgress(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_REPORT_SIMU_PROGRESS)
                {   // シミュレーション進捗報告応答ならば
                    bRtn = this.NoticeReportSimuProgress(csEventObj);
                }
                else if (lEventNo == HySEventObject.NOTICE_SIMULATION_OK)
                {   // 任意のＯＫイベントを受け取った
                    bRtn = this.NoticeSimulationOKEvent(csEventObj, csEventObj.GetSubEventNo());
                }
                //else if (lEventNo == HySEventObject.NOTICE_SIMULATION_NG || lEventNo == HySEventObject.NOTICE_SIMULATION_ABEND)
                //{   // 任意のＮＧイベントを受け取った）
                //    bRtn = this.NoticeSimulationNGEvent(csEventObj, csEventObj.GetSubEventNo());
                //}
                else if (lEventNo == HySEventObject.NOTICE_MAINVIWER_RUN_COMPLETE)
                {  // メイン画面起動完了
                    bRtn = this.NoticeInitViewerComplete(csEventObj);
                }
                else
                {
                    bRtn = this.ArbitraryEvent(csEventObj);
                    //bRtn = EventReceived
                    //    (
                    //     lFromObjNo,
                    //     lEventNo,
                    //     lSubEventNo,
                    //     csID,
                    //     csMainData,
                    //     csSuppData
                    //    );
                }
                //}
            }
            return bRtn;
        }


        // =============================================================================
        // start of シミュレーター（計算エンジン）からのイベントを受け取った時に行う処理
        // =============================================================================
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター生成応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeSimulatorCreation( csEventObj,bSuccessFlg ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeSimulatorCreation(HySEventObject csEventObj,bool bSuccessFlg)
        {
            Boolean bRtn = true;
            if( bSuccessFlg == true )
            {
                HySSimulationDataContainer csSimuContainer = csEventObj.GetData() as HySSimulationDataContainer;
                if (csSimuContainer != null)
                {
                    HySID csID = csSimuContainer.GetID() as HySID;
                    if (csID != null)
                    {
                        this.AddSimulationDataContainer(csID, csSimuContainer);
                    }
                    else
                    {
                        bRtn = false;
                    }
                }
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター消去応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeSimulatorDelete( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeSimulatorDelete(HySEventObject csEventObj)
        {
            Boolean bRtn = true;
            try
            {
                HySID csID = csEventObj.GetSuppID();
                this.DeleteSimulationDataContainer(csID);
            }
            catch
            {
                bRtn = false;
            }
            return bRtn;
        }
        
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター内にシミュレーションモデルを構築応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeModelConstruction( csEventObj,bSuccessFlg ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeModelConstruction(HySEventObject csEventObj, bool bSuccessFlg)
        {
            if (bSuccessFlg == true)
            {
                HySID csID = csEventObj.GetSuppID();
                this.SetSimulatorStatus(csID,HySSimulatorBase.STATUS.MODEL_CONSTRUCTION_STATE);
                HySSimulationDataContainer csDtCnt = csEventObj.GetData() as HySSimulationDataContainer;
                if (csDtCnt != null)
                {
                    m_csDataContainerTbl.SetObject((HySString)csID.GetString(), csDtCnt);
                }
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション初期化応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeInitializeSimulation( csEventObj, bool bSuccessFlg ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeInitializeSimulation(HySEventObject csEventObj, bool bSuccessFlg)
        {
            HySID csID = csEventObj.GetSuppID();
            if (bSuccessFlg == true)
            {
                this.SetSimulatorStatus(csID, HySSimulatorBase.STATUS.INITIAL_STATE);
                HySSimulationDataContainer csDtCnt = csEventObj.GetData() as HySSimulationDataContainer;
                if (csDtCnt != null)
                {
                    m_csDataContainerTbl.SetObject((HySString)csID.GetString(), csDtCnt);
                }
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター開始応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeStartSimulation( csEventObj,bSuccessFlg ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeStartSimulation(HySEventObject csEventObj, bool bSuccessFlg)
        {
            if (bSuccessFlg == true)
            {
                HySID csID = csEventObj.GetSuppID();
                this.SetSimulatorStatus(csID, HySSimulatorBase.STATUS.CALCULATION_STATE);
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター終了応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeCompleteSimulation( csEventObj,bSuccessFlg ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeCompleteSimulation(HySEventObject csEventObj, bool bSuccessFlg)
        {
            HySID csID = csEventObj.GetSuppID();
            if (bSuccessFlg == true)
            {
                this.SetSimulatorStatus(csID, HySSimulatorBase.STATUS.END_STATE);
                HySSimulationDataContainer csDtCnt = csEventObj.GetData() as HySSimulationDataContainer;
                if (csDtCnt != null)
                {
                    m_csDataContainerTbl.SetObject((HySString)csID.GetString(), csDtCnt);
                }
            }
            else
            {
                this.SetSimulatorStatus(csID, HySSimulatorBase.STATUS.ABEND_STATE);
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算中断応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticePauseSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticePauseSimulation(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            this.SetSimulatorStatus(csID, HySSimulatorBase.STATUS.PAUSE_STATE);
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算中断応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeStopSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeStopSimulation(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            this.SetSimulatorStatus(csID, HySSimulatorBase.STATUS.END_STATE);
            return true;
        } 
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算状況応答メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeReportSimuProgress( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean NoticeReportSimuProgress(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            this.SetProgress(csID, csEventObj.GetSubEventNo() ); // SubEventNo に進捗値が入っている
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション動作正常イベント(HySEventObject.NOTICE_SIMULATION_OK)を受け取ったときに動作するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeSimulationOKEvent( csEventObj,lOrgEventNo ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <param name="lOrgEventNo">業務から送ったイベント番号</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual Boolean NoticeSimulationOKEvent(HySEventObject csEventObj, long lOrgEventNo)
        { return true; }
        /*
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション動作異常イベントを受け取った時に動作するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeSimulationNGEvent( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected abstract Boolean NoticeSimulationNGEvent(HySEventObject csEventObj);
        */
 
        // =============================================================================
        // end   of シミュレーター（計算エンジン）からのイベントを受け取った時に行う処理
        // =============================================================================


        // ===================================================
        // start of シミュレーター（計算エンジン）に対する指示
        // ===================================================
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター生成指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdCreateSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdCreateSimulation(HySEventObject csEventObj)
        {
            HySID csID = this.CreateUniqueID();
            HySEventObject csCmnd = CreateSimulatorCommand(HySEventObject.CMND_NEW_SIMULATOR_CREATE, csID);
            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター削除指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdDeleteSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdDeleteSimulation(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            HySEventObject csCmnd = CreateSimulatorCommand(HySEventObject.CMND_SIMULATOR_DELETE, csID);
            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター内にシミュレーションモデルを構築指示するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdModelConstruction( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdModelConstruction(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            HySEventObject csCmnd = csEventObj;
            HySSimulationDataContainer csDtCnt = csEventObj.GetData() as HySSimulationDataContainer;
            if (csDtCnt == null)
            {
                csCmnd = CreateSimulatorCommand(HySEventObject.CMND_MODEL_CONSTRUCTION, csID);
                csDtCnt = m_csDataContainerTbl[(HySString)csID.GetString()] as HySSimulationDataContainer;
                csCmnd.SetData(csDtCnt);
            }
            else
            {
                m_csDataContainerTbl.SetObject((HySString)csID.GetString(),csDtCnt);
            }
            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション初期化指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdInitializeSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdInitializeSimulation(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            HySEventObject csCmnd = csEventObj;
            HySSimulationDataContainer csDtCnt = csEventObj.GetData() as HySSimulationDataContainer;
            if (csDtCnt == null)
            {
                csCmnd = CreateSimulatorCommand(HySEventObject.CMND_INITIALIZE_SIMULATION, csID);
                csDtCnt = m_csDataContainerTbl[(HySString)csID.GetString()] as HySSimulationDataContainer;
                csCmnd.SetData(csDtCnt);
            }
            else
            {
                m_csDataContainerTbl.SetObject((HySString)csID.GetString(), csDtCnt);
            }

            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算開始指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdStartSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdStartSimulation(HySEventObject csEventObj)
        {
            HySEventObject csCmnd = csEventObj;
            HySID csID = csEventObj.GetSuppID();
            if (GetSimulatorStatus(csID) != HySSimulatorBase.STATUS.PAUSE_STATE)
            {
                csCmnd = CreateSimulatorCommand(HySEventObject.CMND_START_SIMULATION, csID);
            }
            else
            {
                csCmnd = CreateSimulatorCommand(HySEventObject.CMND_RESTART_SIMULATION, csID);
            }
            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算中断指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdPauseSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdPauseSimulation(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            HySEventObject csCmnd = CreateSimulatorCommand(HySEventObject.CMND_PAUSE_SIMULATION, csID);
            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>中断中のシミュレーション計算再開指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdRestartSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdRestartSimulation(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            HySEventObject csCmnd = CreateSimulatorCommand(HySEventObject.CMND_RESTART_SIMULATION, csID);
            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算強制終了指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdStopSimulation( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdStopSimulation(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            HySEventObject csCmnd = CreateSimulatorCommand(HySEventObject.CMND_STOP_SIMULATION, csID);
            return SendCommandToSimulator(csCmnd);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算中状況報告指示メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CmdReportSimuProgress( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント(必要ならば使用する)</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual Boolean CmdReportSimuProgress(HySEventObject csEventObj)
        {
            HySID csID = csEventObj.GetSuppID();
            HySEventObject csCmnd = CreateSimulatorCommand(HySEventObject.CMND_REPORT_SIMU_PROGRESS, csID);
            return SendCommandToSimulator(csCmnd);
        }

        // ===================================================
        // end of シミュレーター（計算エンジン）に対する指示
        // ===================================================

        // ===================================================
        // start of ビューア関連イベント
        // ===================================================
        /// <summary><para>method outline:</para>
        /// <para>画面立上がり完了通知処理メソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeInitViewerComplete( csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected abstract Boolean NoticeInitViewerComplete(HySEventObject csEventObj);

        // ===================================================
        // end  of ビューア関連イベント
        // ===================================================

        // ===================================================
        // start of 任意イベント
        // ===================================================
        /// <summary><para>method outline:</para>
        /// <para>ユーザー側で定義した任意のイベントを受け取った時に動作するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = ArbitraryEvent( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected abstract Boolean ArbitraryEvent(HySEventObject csEvent);
        // ===================================================
        // end  of 任意イベント
        // ===================================================

        // ===================================================
        // start of 各種ツールメソッド
        // ===================================================
        /// <summary><para>method outline:</para>
        /// <para>シミュレーターへ送るコマンドオブジェクトを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>
        /// HySEventObject csCmnd = CreateSimulatorCommand(  lEventNo, csID )
        ///   生成されたオブジェクトに対して　色々な設定を行う
        ///   csCmnd.SetData(  ..... 等
        /// </para>
        /// </example>
        /// <param name="lEventNo">
        /// コマンド種別番号（HySEventObjectに定義されている）
        /// 　　例：HySEventObject.CMND_START_SIMULATION　等
        /// </param>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <returns>HySEventObject :シミュレーターへ送るコマンドオブジェクト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual HySEventObject CreateSimulatorCommand(long lEventNo,HySID csID)
        {
            HySEventObject csEventObj = new HySEventObject(HySSysEvent.OBJID_SIMULATOR, lEventNo);
            csEventObj.SetSuppID(csID);
            csEventObj.SetFromSimKind(this.GetSimKind());
            csEventObj.SetToSimKind(this.GetSimKind());
            csEventObj.SetFrom(HySSysEvent.OBJID_BUSIPROCEDURE);
            if (lEventNo == HySEventObject.CMND_NEW_SIMULATOR_CREATE)
            {   // シミュレーター作成指示
            }
            else if (lEventNo == HySEventObject.CMND_SIMULATOR_DELETE)
            {   // シミュレーター消去指示ならば
            }
            else if (lEventNo == HySEventObject.CMND_MODEL_CONSTRUCTION)
            {   // シミュレーションモデル作成開始指示
            }
            else if (lEventNo == HySEventObject.CMND_INITIALIZE_SIMULATION)
            {   // シミュレーション初期化指示
            }
            else if (lEventNo == HySEventObject.CMND_START_SIMULATION)
            {   // シミュレーション開始指示
            }
            else if (lEventNo == HySEventObject.CMND_RESTART_SIMULATION)
            {   // シミュレーション再開指示
                //bRtn = this.RestartSimulation(csEvent);
            }
            else if (lEventNo == HySEventObject.CMND_PAUSE_SIMULATION)
            {   // シミュレーション中断指示
            }
            else if (lEventNo == HySEventObject.CMND_STOP_SIMULATION)
            {   // シミュレーション強制終了指示
            }
            else if (lEventNo == HySEventObject.CMND_REPORT_SIMU_PROGRESS)
            {   // シミュレーション計算進捗状況報告指示
            }
            return csEventObj;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーターへコマンドを送信する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = SendCommandToSimulator(  csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">コマンドオブジェクト</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual Boolean SendCommandToSimulator(HySEventObject csEventObj)
        {
            csEventObj.SetTo(HySSysEvent.OBJID_SIMULATOR);
            return this.PutEvent(csEventObj);
        }
        /// <summary><para>method outline:</para>
        /// <para>ビューアへ送るコマンドオブジェクトを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySEventObject csCmnd = CreateViewerCommand(  lEventNo, csID ) </para>
        /// </example>
        /// <param name="lEventNo">
        /// コマンド種別番号（HySEventObjectに定義されている）
        /// 　　例：HySEventObject.CMND_START_SIMULATION　等
        /// </param>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <returns>HySEventObject :シミュレーターへ送るコマンドオブジェクト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual HySEventObject CreateViewerCommand(long lEventNo, HySID csID)
        {
            HySEventObject csEventObj = new HySEventObject(HySSysEvent.OBJID_VIEWER, lEventNo);
            csEventObj.SetSuppID(csID);
            csEventObj.SetFromSimKind(this.GetSimKind());
            csEventObj.SetToSimKind(this.GetSimKind());
            csEventObj.SetFrom(HySSysEvent.OBJID_BUSIPROCEDURE);

            return csEventObj;
        }
        /// <summary><para>method outline:</para>
        /// <para>ビューアへコマンドを送信する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = SendCommandToSimulator(  csEventObj ) </para>
        /// </example>
        /// <param name="csEventObj">コマンドオブジェクト</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual Boolean SendCommandToViewer(HySEventObject csEventObj)
        {
            csEventObj.SetTo(HySSysEvent.OBJID_VIEWER);
            csEventObj.SetFrom(HySSysEvent.OBJID_BUSIPROCEDURE);
            return this.PutEvent(csEventObj);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション用情報格納コンテナを管理登録する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddSimulationDataContainer(  csID,csDt ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <param name="csDt">シミュレーション用情報格納コンテナ </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual void AddSimulationDataContainer(HySID csID, HySSimulationDataContainer csDt)
        {
            m_csDataContainerTbl.AddObject((HySString)(csID.GetString()), csDt);
            m_csSimuStatusTbl.Add(csID.ToString(), HySSimulatorBase.STATUS.DEFAULT_STATE);
            m_csSimuProgressTbl.Add(csID.ToString(), 0);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション用情報格納コンテナを管理から外す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DeleteSimulationDataContainer(  csID ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual void DeleteSimulationDataContainer(HySID csID)
        {
            m_csDataContainerTbl.DeleteData((HySString)(csID.GetString()));
            m_csSimuStatusTbl.Remove(csID.ToString());
            m_csSimuProgressTbl.Remove(csID.ToString());
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション用情報格納コンテナを更新する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimulationDataContainer(  csID,csDt ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <param name="csDt">シミュレーション用情報格納コンテナ </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual void SetSimulationDataContainer(HySID csID, HySSimulationDataContainer csDt)
        {
            m_csDataContainerTbl.AddObject((HySString)(csID.GetString()),csDt);
        }
        /// <summary><para>method outline:</para>
        /// <para>管理しているシミュレーション用情報格納コンテナを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySSimulationDataContainer csDtCnt = GetSimulationDataContainer(  csID ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <returns>HySSimulationDataContainer :シミュレーション用情報格納コンテナ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual HySSimulationDataContainer GetSimulationDataContainer(HySID csID)
        {
            return m_csDataContainerTbl[(HySString)(csID.GetString())] as HySSimulationDataContainer;
        }

        /// <summary><para>method outline:</para>
        /// <para>管理しているシミュレーターの状態を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimulatorStatus( csID, eStatus ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <param name="eStatus">ステータス </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual void SetSimulatorStatus(HySID csID, HySSimulatorBase.STATUS eStatus)
        {
            m_csSimuStatusTbl[csID.ToString()]=eStatus;
        }
        /// <summary><para>method outline:</para>
        /// <para>管理しているシミュレーション状態を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySSimuCtlBusiProcedureBase.STATUS eStatus = GetSimulatorStatus(  csID ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <returns>HySSimuCtlBusiProcedureBase.STATUS :シミュレーション状態</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual HySSimulatorBase.STATUS GetSimulatorStatus(HySID csID)
        {
            return ((HySSimulatorBase.STATUS)m_csSimuStatusTbl[csID.ToString()]);
        }

        /// <summary><para>method outline:</para>
        /// <para>管理しているシミュレーターの計算進捗状況を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetProgress( csID, lProgress ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <param name="lProgress">進捗状況（％） </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual void SetProgress(HySID csID, long lProgress)
        {
            m_csSimuProgressTbl[csID.ToString()] = lProgress;
        }
        /// <summary><para>method outline:</para>
        /// <para>管理しているシミュレーション計算進捗状況を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lProgress = GetProgress(  csID ) </para>
        /// </example>
        /// <param name="csID">シミュレーターを区別するID </param>
        /// <returns>long :進捗状況（％）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        protected virtual long GetProgress(HySID csID)
        {
            return ((long)m_csSimuProgressTbl[csID.ToString()]);
        }
        /// <summary><para>method outline:</para>
        /// <para>ビューアに画面表示を指示する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DispScreen(  csScreenKind,csID,csDispData ) </para>
        /// </example>
        /// <param name="csScreenKind">画面種別 </param>
        /// <param name="csID">画面識別ID </param>
        /// <param name="csDispData">表示データ </param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド
        /// 画面種別、画面IDで　表示する画面をユニークに特定する
        /// </para>
        /// </remarks>
        protected virtual void DispScreen(HySObjectKind csScreenKind, HySID csID, HySDataRoot csDispData)
        {
            HySEventObject csEventObj = new HySEventObject(HySSysEvent.OBJID_VIEWER, HySEventObject.CMND_DISP_SCREEN);
            csEventObj.SetSuppID(csID);
            csEventObj.SetFromSimKind(this.GetSimKind());
            csEventObj.SetToSimKind(this.GetSimKind());
            csEventObj.SetFrom(HySSysEvent.OBJID_BUSIPROCEDURE);
            csEventObj.SetData(csDispData);
            csEventObj.SetSubData(csScreenKind);

            this.SendCommandToViewer(csEventObj);
        }

        /// <summary>シミュレータカウンタ(ユニークIDを作成するために使用する)</summary>
        protected ulong m_ulSimCounter = 1;
        /// <summary><para>method outline:</para>
        /// <para>ユニークID作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csID = CreateUniqueID()</para>
        /// </example>
        /// <returns>ユニークID</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>派生クラス側で使用するツールメソッド</para>
        /// </remarks>
        virtual protected HySID CreateUniqueID()
        {
            HySString csKind = this.GetSimKind().GetString() as HySString;
            HySString csGroup = new HySString((m_ulSimCounter++).ToString()+ ":" + DateTime.Now.Minute.ToString() + DateTime.Now.Hour.ToString() + DateTime.Now.Second.ToString() + ":");
            HySString csKey = new HySString((HySCalendar.CreateNowTime().GetTime().ToString()).GetHashCode().ToString());
            return new HySID(csKind + csGroup + csKey);
        }

        // ===================================================
        // end  of 各種ツールメソッド
        // ===================================================

        //
        /// <summary><para>method outline:</para>
        /// <para>イベントを受け取った時に動作するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = EventReceived(
        ///        long lFromObjNo,
        ///        long lEventNo,
        ///        long lSubEventNo,
        ///        HySID csID,
        ///        HySDataRoot csMainData,
        ///        HySDataRoot csSuppData
        ///                                 );
        /// </para>
        /// </example>
        /// <param name="lFromObjNo">
        /// イベントの送り元番号
        /// HySSysEvent.OBJID_VIEWER 等が設定されている
        /// </param>
        /// <param name="lEventNo">
        /// 送られたイベント番号
        /// HySEventObject.CMND_START_SIMULATION 等が設定されている
        /// </param>
        /// <param name="lSubEventNo">
        /// 送られたイベントの補足番号
        /// ユーザー側で定義できる
        /// </param>
        /// <param name="csID">
        /// 送られたイベントID
        /// ユーザー側で定義できる　null の事も有り得る
        /// </param>
        /// <param name="csMainData">
        /// 送られたイベント付属データ
        /// ユーザー側で定義できる　null の事も有り得る
        /// </param>
        /// <param name="csSuppData">
        /// 送られたイベント付属データ（補足）
        /// ユーザー側で定義できる　null の事も有り得る
        /// </param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 親クラスのオーバーライド：何もしない
        /// 本クラスの派生処理では、使用不可。
        /// </para>
        /// </remarks>
        override protected Boolean EventReceived
            (
                long lFromObjNo,
                long lEventNo,
                long lSubEventNo,
                HySID csID,
                HySDataRoot csMainData,
                HySDataRoot csSuppData
            )
        {
            return false;
        }
    }
}
