﻿// <summary>ソースコード：演算スケジュール制御を行うクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSController;
using CommonMP.HYSSOP.Interface.HSSimulator;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSController
{
    /// <summary><para>class outline:</para>
    /// <para>演算スケジュール制御を行う</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// <para>Remarks</para>
    /// <para>本クラスは、フレームワークの将来拡張用</para>
    /// </remarks>
    public class HySSimSchedule : HySSysUnit
    {
        /// <summary>識別子</summary>
        protected HySID m_csID;

        /// <summary>シミュレーション制御</summary>
        protected HySControllerRoot m_csController = null;

        /// <summary>シミュレーターファクトリクラス</summary>
        protected HySSimulatorFactory m_csSimulatorFactory = null;

        /// <summary>演算目標時刻</summary>
        protected HySTime m_csGoalTime;
        /// <summary>演算時刻</summary>
        protected HySTime m_csSimuTime;

        /// <summary>シミュレーション間情報交換データ 
        /// 計算順序として並べられたスケジュールユニット間で情報交換を行うために使用する
        /// </summary>
        protected HySSimuMediateContainer m_csMediationContainer;

        /// <summary>スケジュールユニットリスト 
        /// 　計算スケジュール単位（スケジュールユニット）を計算順序として　リスト状に並べたもの
        /// </summary>
        protected LinkedList<HySSimScheduleUnit> m_csScheduleUnitList;
        /// <summary>実効中のスケジュール単位のリスト上のノード /// </summary>
        protected LinkedListNode<HySSimScheduleUnit> m_ActiveSchUnitNode;
        /// <summary>実効中のスケジュール単位 /// </summary>
        protected HySSimScheduleUnit m_csActiveSchUnit;

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySSimSchedule csSchedule = new HySSimSchedule()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySSimSchedule()
        {
            m_csScheduleUnitList = new LinkedList<HySSimScheduleUnit>();
        }

        /// <summary><para>method outline:</para>
        /// <para>演算目標時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetGoalTime( csGoalTIme ) </para>
        /// </example>
        /// <param name="csGoalTime">演算目標時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetGoalTime(HySTime csGoalTime)
        {
            m_csGoalTime = csGoalTime;
        }
        /// <summary><para>method outline:</para>
        /// <para>演算目標時刻取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csGoalTime　GetGoalTime( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>演算目標時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySTime GetGoalTime()
        {
            return m_csGoalTime;
        }
        /// <summary><para>method outline:</para>
        /// <para>演算時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimuTime( csSimuTime ) </para>
        /// </example>
        /// <param name="csSimuTime">演算時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetSimuTime(HySTime csSimuTime)
        {
            m_csSimuTime = csSimuTime;
        }
        /// <summary><para>method outline:</para>
        /// <para>演算時刻取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csDltTime GetSimuTime( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>演算時刻取得幅</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySTime GetSimuTime()
        {
            return m_csSimuTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetID( csID ) </para>
        /// </example>
        /// <param name="csID">識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetID(HySIdentifier csID)
        {
            m_csID = (CommonMP.HYSSOP.CoreImpl.HSData.HySID)csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySIdentifier csID = GetID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySIdentifier GetID()
        {
            return m_csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>識別子による同一比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = Equal( sID ) </para>
        /// </example>
        /// <param name="sID">比較する種別識別子</param>
        /// <returns>  true :＝ , false:≠</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public Boolean Equal(HySIdentifier sID)
        {
            return m_csID.Equals(sID);
        }

        /// <summary><para>method outline:</para>
        /// <para>制御クラスを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetController( csCtl ) </para>
        /// </example>
        /// <param name="csCtl">制御クラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetController(HySControllerRoot csCtl)
        {
            m_csController = csCtl;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーターファクトリを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimulatorFactor( csSimulatorFactory ) </para>
        /// </example>
        /// <param name="csSimulatorFactory">シミュレーターファクトリクラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetSimulatorFactor(HySSimulatorFactory csSimulatorFactory)
        {
            m_csSimulatorFactory = csSimulatorFactory;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション間情報交換データコンテナ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimuMediateContainer( csMediateContainer ) </para>
        /// </example>
        /// <param name="csMediateContainer">シミュレーション間情報交換データコンテナ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetSimuMediateContainer(HySSimuMediateContainer csMediateContainer)
        {
            m_csMediationContainer = csMediateContainer;
        }


        // ====================================
        // 演算実行
        // ====================================

        /// <summary><para>method outline:</para>
        /// <para>イベントを受け取った時に動作するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = EventCallback( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>各実装クラスは受け取ったイベント毎に処理を行う</para>
        /// </remarks>
        public virtual Boolean EventCallback(HySSysEvent csEvent)
        {
            Boolean bRtn = false;
            long lSubEventNo = ((HySEventObject)csEvent).GetSubEventNo();

            
            if (lSubEventNo == HySEventObject.CMND_MODEL_CONSTRUCTION)
            {   // シミュレーションモデル作成開始指示
                HySSimulationScheduleData csSchduleData = ((HySEventObject)csEvent).GetData() as HySSimulationScheduleData;
                if (csSchduleData != null)
                {
                    bRtn = this.MakeSimuSchedule(csSchduleData);
                }
            }
            else if (lSubEventNo == HySEventObject.CMND_INITIALIZE_SIMULATION)
            {   // シミュレーション初期化指示
                bRtn = this.SimuInitialize();
            }
            else if (lSubEventNo == HySEventObject.CMND_START_SIMULATION)
            {   // シミュレーション開始指示
                bRtn = this.SimuStart();
            }
            else if (lSubEventNo == HySEventObject.CMND_PAUSE_SIMULATION)
            {   // シミュレーション中断指示
                bRtn = this.SimuPause();
            }
            else if (lSubEventNo == HySEventObject.CMND_RESTART_SIMULATION)
            {   // シミュレーション再開指示
                bRtn = this.SimuContinue();
            }
            else if (lSubEventNo == HySEventObject.CMND_STOP_SIMULATION)
            {   // シミュレーション強制終了指示
                bRtn = this.SimuMakeStop();
            }
            else
            {   // その他のイベント
                //bRtn = m_csSimulator.CmdArbitraryEvent(csEvent);
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>イベントを送る</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = PutEvent( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送るイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>自分からイベントを送信する時に使用する</para>
        /// </remarks>
        public virtual Boolean PutEvent(HySSysEvent csEvent)
        {
            return m_csController.PutEvent(csEvent);
        }
        /// <summary><para>method outline:</para>
        /// <para>システム終了準備完了判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = ExitOK( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true : システムExit準備OK　、false : システムExit準備NG</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean ExitOK()
        {
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>スケジュール単位クリア</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>ClearSimuScheduleUnit( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void ClearSimuScheduleUnit()
        {
            m_csScheduleUnitList.Clear();
            m_ActiveSchUnitNode = null;
            m_csActiveSchUnit = null;
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュール単位登録（最後に追加される）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>AddSimuScheduleUnit(csSimuSchdUnit) </para>
        /// </example>
        /// <param name="csSimuSchdUnit">スケジュール単位</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void AddSimuScheduleUnit(HySSimScheduleUnit csSimuSchdUnit)
        {
            m_csScheduleUnitList.AddLast(csSimuSchdUnit);
        }

        /// <summary><para>method outline:</para>
        /// <para>スケジュール作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = MakeSimuSchedule(csSchduleData) </para>
        /// </example>
        /// <param name="csSchduleData">スケジュールデータ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual Boolean MakeSimuSchedule(HySSimulationScheduleData csSchduleData)
        {
            this.ClearSimuScheduleUnit();

            /*
            int UnitNum = csSchduleData.GetUnitNum();
            HySSimuScheduleUnitData unitDt = null;
            for (int lp = 0; lp < UnitNum; lp++)
            {
                HySEngineSimuScheduleUnit SchUnit = new HySEngineSimuScheduleUnit(enginCtl);
                this.AddSimuScheduleUnit(SchUnit);

                unitDt = schduleData.GetUnit(lp);
                int engineNum = unitDt.EngineNum();
                for (int lp2 = 0; lp2 < engineNum; lp2++)
                {
                    String engineName = unitDt.GetEngineName(lp2);

                    HySEngineThreadController engThread = enginCtl.GetEngineThreadCtl(new HySSysObjectID(engineName));
                    SchUnit.AddEngine(engThread);
                }
            }
            */
            m_ActiveSchUnitNode = null;
            m_csActiveSchUnit = null;
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>スケジュール計算初期化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = SimuInitialize() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual Boolean SimuInitialize()
        {
            Boolean bRtn = false;
            m_csActiveSchUnit = null;

            m_ActiveSchUnitNode = m_csScheduleUnitList.First;
            if( m_ActiveSchUnitNode != null )
            {
                m_csActiveSchUnit = m_ActiveSchUnitNode.Value;

                bRtn = m_csActiveSchUnit.CmdSimulationInitialize(m_csMediationContainer);
                //bRtn = m_csActiveSchUnit.SimuInitial(m_MediateData);
                //bRtn = m_csActiveSchUnit.SimuDataFusion(m_MediateData);
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュール計算開始</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = SimuStart() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual Boolean SimuStart()
        {
            Boolean bRtn = false;
            if (m_csActiveSchUnit != null)
            {
                bRtn = m_csActiveSchUnit.CmdSimulationStart();
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュール計算中断</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = SimuPause() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual Boolean SimuPause()
        {
            Boolean bRtn = false;
            if (m_csActiveSchUnit != null)
            {
                bRtn = m_csActiveSchUnit.CmdSimulationPause();
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュール計算再開</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = SimuContinue() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual Boolean SimuContinue()
        {
            Boolean bRtn = false;
            if (m_csActiveSchUnit != null)
            {
                bRtn = m_csActiveSchUnit.CmdSimulationContinue();
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュール計算強制終了</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = SimuMakeStop() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual Boolean SimuMakeStop()
        {
            bool bRtn = false;
            if (m_csActiveSchUnit != null)
            {
                bRtn = m_csActiveSchUnit.CmdSimulationStop();
                // bRtn =m_csActiveSchUnit.SimuDataFusion( m_MediateData );
            }
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>スケジュールシミュレーション計算開始通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeSimulationStart() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeSimulationStart()
        {
            HySEventObject csEvent = new HySEventObject(
                    HySSysEvent.OBJID_BUSIPROCEDURE, 
                    HySEventObject.NOTICE_SIMU_SCHEDULE_EVENT,
                    HySEventObject.NOTICE_START_SIMULATION); // 演算開始イベント
            this.PutEvent(csEvent);

            /*
            boolean rtn = false;
            if( m_csActiveSchUnit != null )
            {
                rtn = m_csActiveSchUnit.NoticeSimulationStart(engineID);
                if( rtn == true )
                {
                    m_csEngineCtl.NoticeWorldSimulationStart();
                }
            }
            return rtn;
             */
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュールシミュレーション計算中断通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeSimulationPause() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeSimulationPause()
        {
            HySEventObject csEvent = new HySEventObject(
                    HySSysEvent.OBJID_BUSIPROCEDURE,
                    HySEventObject.NOTICE_SIMU_SCHEDULE_EVENT,
                    HySEventObject.NOTICE_PAUSE_SIMULATION); // 演算中断イベント
            this.PutEvent(csEvent);
            /*
            boolean rtn = false;
            if( m_csActiveSchUnit != null )
            {
                rtn = m_csActiveSchUnit.NoticeSimulationPause(engineID);
                if( rtn == true )
                {
                    m_csEngineCtl.NoticeWorldSimulationPause();
                }
            }
            return rtn;
             * */
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュールシミュレーション計算再開通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeSimulationRestart() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeSimulationRestart()
        {
            //HySEventObject csEvent = new HySEventObject(
            //        HySSysEvent.OBJID_BUSIPROCEDURE,
            //        HySEventObject.NOTICE_SIMU_SCHEDULE_EVENT,
            //        HySEventObject.NOTICE_RESTART_SIMULATION); // 演算再開イベント
            HySEventObject csEvent = new HySEventObject(
                    HySSysEvent.OBJID_BUSIPROCEDURE,
                    HySEventObject.NOTICE_SIMU_SCHEDULE_EVENT,
                    HySEventObject.NOTICE_START_SIMULATION); // 演算開始イベント
            this.PutEvent(csEvent);
            /*
            boolean rtn = false;
            if( m_csActiveSchUnit != null )
            {
                rtn = m_csActiveSchUnit.NoticeSimulationPause(engineID);
                if( rtn == true )
                {
                    m_csEngineCtl.NoticeWorldSimulationPause();
                }
            }
            return rtn;
             * */
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュールシミュレーション計算結果更新通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeSimulationResultChange() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シミュレーション計算中に結果表示を更新していく場合等に使用</para>
        /// </remarks>
        public virtual void NoticeSimulationResultChange()
        {
            HySEventObject csEvent = new HySEventObject(
                HySSysEvent.OBJID_BUSIPROCEDURE,
                HySEventObject.NOTICE_SIMU_SCHEDULE_EVENT,
                HySEventObject.NOTICE_RENEW_SIMULATION); // 演算結果更新イベント
            this.PutEvent(csEvent);
            /*
            boolean rtn = false;
            if( m_csActiveSchUnit != null )
            {
                rtn = m_csActiveSchUnit.NoticeSimulationResultChange(engineID);
                if( rtn == true )
                {
                    m_csEngineCtl.NoticeWorldSimulationResultChange();
                }
            }
            return rtn;
            */
        }
        /// <summary><para>method outline:</para>
        /// <para>スケジュールシミュレーション計算終了通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeSimulationStop() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeSimulationStop()
        {
            HySEventObject csEvent = new HySEventObject(
                    HySSysEvent.OBJID_BUSIPROCEDURE,
                    HySEventObject.NOTICE_SIMU_SCHEDULE_EVENT,
                    HySEventObject.NOTICE_STOP_SIMULATION); // 演算中止イベント
            this.PutEvent(csEvent);
            /*
            boolean rtn = false;
            if( m_csActiveSchUnit != null )
            {
                rtn = m_csActiveSchUnit.NoticeSimulationStart(engineID);
                if( rtn == true )
                {
                    m_csEngineCtl.NoticeWorldSimulationStart();
                }
            }
            return rtn;
             */
        }

        /// <summary><para>method outline:</para>
        /// <para>スケジュールシミュレーション計算完了通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>NoticeSimulationComplete() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>=ture:正常、=false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeSimulationComplete()
        {
            if (m_csActiveSchUnit == null)
            {
                return;
            }

            // 情報交換
            m_csActiveSchUnit.CmdSimulationDataFusion(m_csMediationContainer);

            // シミュレーション時間判断
            if (m_csGoalTime.Before(m_csActiveSchUnit.GetSimuTime()) == true)
            {   // スケジュール全体の　シミュレーション計算が終了していないならば　次のスケジュール単位へ切り替える
                m_csActiveSchUnit = null;
                m_ActiveSchUnitNode = m_ActiveSchUnitNode.Next;
                if (m_ActiveSchUnitNode != null)
                {   // 次のスケジュール単位有りの場合
                    m_csActiveSchUnit = m_ActiveSchUnitNode.Value;
                }
            }
            else
            {   // 予定時間まで計算終了したならば
                m_csActiveSchUnit = null; // 強制的に次は無し
            }

            if (m_csActiveSchUnit == null)
            {   // 次の計算する必要が無い場合
                HySEventObject csEvent = new HySEventObject(
                        HySSysEvent.OBJID_BUSIPROCEDURE,
                        HySEventObject.NOTICE_SIMU_SCHEDULE_EVENT,
                        HySEventObject.NOTICE_COMPLETE_SIMULATION); // 演算完了イベント
                this.PutEvent(csEvent);
            }
            else
            {
                m_csActiveSchUnit.CmdSimulationDataFusion(m_csMediationContainer);  // 次のスケジュール単位に対して先ず情報交換

                this.SimuContinue(); // シミュレーションを継続する
            }
            /*
            boolean rtn = false;
            if( m_csActiveSchUnit == null )
            {   return false;   }
            
            rtn = m_csActiveSchUnit.NoticeSimulationComplete(engineID);
            if( rtn == true )
            {   // 全エンジンがシミュレーション終了
            
                // シミュレーションデータフュージョン
                m_csActiveSchUnit.SimuDataFusion(m_MediateData);
                
                // シミュレーション時間判断
                if( m_GoatTime.Before( m_csActiveSchUnit.GetSimuTime() ) == true )
                {   // シミュレーション計算完了していないならば
                
                    // アクティブを変更
                    m_ActiveSchIndex += 1;
                    if( m_csActiveSchUnit == (HySEngineSimuScheduleUnit)m_SimuSchedlUnitList.getLast() )
                    {   // 既に最後まで計算が終わっているならば
                        m_csActiveSchUnit=null; // アクティブは無くなった
                    }
                    else
                    {
                        // 次のユニットに変更する
                        m_csActiveSchUnit=(HySEngineSimuScheduleUnit)m_SimuSchedlUnitList.get(m_ActiveSchIndex);    // 次のユニット
                    }
                }
                else
                {   // シミュレーション計算完了したならば
                    m_csActiveSchUnit = null; // 強制的に次は無し
                }
                
                if( m_csActiveSchUnit == null  )
                {   // 計算が全て終了したら
                    m_csEngineCtl.NoticeWorldSimulationComplete();
                }
                else
                {
                    // 計算再開準備
                    rtn = m_csActiveSchUnit.SimuPrepare(m_MediateData);
                    // 計算再開
                    rtn = m_csActiveSchUnit.SimuContinue();
                }
            } // end of 全エンジンシミュレーション完了
            
            return rtn;
             */
        }
    
    }
}
