﻿// <summary>ソースコード：>河道横断面DBAクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSDBA;
using CommonMP.HYSSOP.CoreImpl.HSDBA.FileBase;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSDBA.FileBase
{
    /// <summary><para>class outline:</para>
    /// <para>河道横断面DBAクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/09/24][新規作成]</para>
    /// </remarks>
    public class HySGeoRiverCrossSectionDBA : HySCommonDBA
    {

        #region 【定数】内部処理用

        /// <summary>DBフォルダパス名を環境設定から取得するためのキー</summary>
        public const string ENVKEY_DBPATH = "HSDBA_RIVERCROSSSECTIONDATA_PATH";

        /// <summary>DBフォルダパス名のデフォルト値</summary>
        public const string DEFAULT_DBPATH = @"\SystemData\db\rivercrosssectiondata";

        /// <summary>自クラス名（ログ出力用）</summary>
        private const string m_csMyClassName = "HySGeoRiverCrossSectionDBA";

        ///<summary>断面IDエンコーディング</summary>
        private static readonly Encoding m_csIDEncoding = Encoding.UTF8;

        ///<summary>断面ID区切り文字</summary>
        private const char ID_DELIMITER = '_';

        ///<summary>断面IDエスケープシーケンス開始文字</summary>
        private const char ID_ESCAPE_START_CHAR = '%';

        #endregion

        #region 【定数】XMLノード・属性名

        /// <summary>ノード名: crossSections</summary>
        private const string NODE_NAME_CROSSSECTIONS = "crossSections";
        /// <summary>ノード名: crossSection</summary>
        private const string NODE_NAME_CROSSSECTION = "crossSection";
        /// <summary>ノード名: distancePost</summary>
        private const string NODE_NAME_DISTANCEPOST = "distancePost";
        /// <summary>ノード名: shape</summary>
        private const string NODE_NAME_SHAPE = "shape";
        /// <summary>ノード名: point</summary>
        private const string NODE_NAME_POINT = "point";
        /// <summary>ノード名: attributes</summary>
        private const string NODE_NAME_ATTRIBUTES = "attributes";
        /// <summary>ノード名: sectionAttribute</summary>
        private const string NODE_NAME_SECTIONATTRIBUTE = "sectionAttribute";
        /// <summary>ノード名: pointAttribute</summary>
        private const string NODE_NAME_POINTATTRIBUTES = "pointAttributes";
        /// <summary>ノード名: lineAttribute</summary>
        private const string NODE_NAME_LINEATTRIBUTES = "lineAttributes";
        /// <summary>ノード名: leftDistancePost</summary>
        private const string NODE_NAME_LEFTDISTANCEPOST = "leftDistancePost";
        /// <summary>ノード名: rightDistancePost</summary>
        private const string NODE_NAME_RIGHTDISTANCEPOST = "rightDistancePost";
        /// <summary>ノード名: geographicCoordinates</summary>
        private const string NODE_NAME_GEOGRAPHICCOORDINATES = "geographicCoordinates";
        /// <summary>ノード名: attribute</summary>
        private const string NODE_NAME_ATTRIBUTE = "attribute";
        /// <summary>ノード名: item</summary>
        private const string NODE_NAME_ITEM = "item";
        /// <summary>属性名: id</summary>
        private const string ATTR_NAME_ID = "id";
        /// <summary>属性名: riverCode</summary>
        private const string ATTR_NAME_RIVERCODE = "riverCode";
        /// <summary>属性名: auxRiverCode</summary>
        private const string ATTR_NAME_AUXRIVERCODE = "auxRiverCode";
        /// <summary>属性名: distancePost</summary>
        private const string ATTR_NAME_DISTANCEPOST = "distancePost";
        /// <summary>属性名: distance</summary>
        private const string ATTR_NAME_DISTANCE = "distance";
        /// <summary>属性名: minXCoordinate</summary>
        private const string ATTR_NAME_MINXCOORDINATE = "minXCoordinate";
        /// <summary>属性名: maxXCoordinate</summary>
        private const string ATTR_NAME_MAXXCOORDINATE = "maxXCoordinate";
        /// <summary>属性名: pointId</summary>
        private const string ATTR_NAME_POINTID = "pointId";
        /// <summary>属性名: h</summary>
        private const string ATTR_NAME_H = "h";
        /// <summary>属性名: x</summary>
        private const string ATTR_NAME_X = "x";
        /// <summary>属性名: value</summary>
        private const string ATTR_NAME_VALUE = "value";
        /// <summary>属性名: latitude</summary>
        private const string ATTR_NAME_LATITUDE = "latitude";
        /// <summary>属性名: longitude</summary>
        private const string ATTR_NAME_LONGITUDE = "longitude";

        #endregion

        /// <summary>DBフォルダのパス名（フルパス）</summary>
        /// <remarks>末尾にディレクトリ区切り文字を付加すること。</remarks>
        private string m_csDBPath;

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySGeoRiverCrossSectionDBA csDBA = new HySGeoRiverCrossSectionDBA();</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySGeoRiverCrossSectionDBA 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySGeoRiverCrossSectionDBA()
            : base()
        {
            // 環境周りの初期化処理
            InitialEnvInfo();
        }

        /// <summary><para>method outline:</para>
        /// <para>1件新規登録</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = RegisterNew(csStockData)</para>
        /// </example>
        /// <param name="csStockData">DB登録データ</param>
        /// <returns>true:成功, false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool RegisterNew(HySStockData csStockData)
        {
            const string csMyMethodName = "RegisterNew(HySStockData)";

            // 一括登録バージョンで実行する
            HySStockDataList csStockDataList = new HySStockDataList();
            csStockDataList.AddLast(csStockData);
            if (!RegisterNew(csStockDataList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "RegisterNew(HySStockDataList) error");
                return false;
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>複数件一括新規登録</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = RegisterNew(csStockDataList) </para>
        /// </example>
        /// <param name="csStockDataList">登録対象のデータリスト</param>
        /// <returns>true:成功、false:失敗(データベースは更新されていない)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool RegisterNew(HySStockDataList csStockDataList)
        {
            const string csMyMethodName = "RegisterNew(HySStockDataList)";

            // DB保存データをデータファイルパスに展開する
            Dictionary<string, string> csDataFilePathList = new Dictionary<string, string>();
            Dictionary<string, string> csIndexFilePathList = new Dictionary<string, string>();
            csStockDataList.SetCursorFirst();
            long lDataCnt = csStockDataList.GetCount();
            for (long lIdx = 0; lIdx < lDataCnt; ++lIdx)
            {
                HySStockData csStockData = (HySStockData)csStockDataList.GetCursorData();
                csStockDataList.MoveCursorNext();

                string csIDString;
                string csDataFilePath;
                string csIndexFilePath;
                if (!CreateDataFilePathFromStockData(csStockData, out csIDString, out csDataFilePath, out csIndexFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateDataFilePathFromStockData() error");
                    return false;
                }
                if (csDataFilePathList.ContainsKey(csIDString))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "duplicate data specified",
                        "RiverCode", ((HySGeoRiverCrossSectionData)csStockData.GetData()).GetRiverCode(),
                        "DistancePost", ((HySGeoRiverCrossSectionData)csStockData.GetData()).GetDistancePost(),
                        "AuxRiverCode", ((HySGeoRiverCrossSectionData)csStockData.GetData()).GetAuxRiverCode());
                    return false;
                }
                csDataFilePathList.Add(csIDString, csDataFilePath);
                if (!csIndexFilePathList.ContainsKey(csIDString))
                {
                    csIndexFilePathList.Add(csIDString, csIndexFilePath);
                }
            }

            List<HySXMLDataFile> csDataFileList = new List<HySXMLDataFile>();
            Dictionary<string, HySXMLDataFile> csIndexFileList = new Dictionary<string, HySXMLDataFile>();
            try
            {
                // データ登録処理実行
                csStockDataList.SetCursorFirst();   // カーソルをリセット
                foreach (string csIDString in csDataFilePathList.Keys)
                {
                    HySStockData csStockData = (HySStockData)csStockDataList.GetCursorData();
                    csStockDataList.MoveCursorNext();
                    HySGeoRiverCrossSectionData csCrossSectionData = (HySGeoRiverCrossSectionData)csStockData.GetData();

                    // データファイルをロック
                    string csDataFilePath = csDataFilePathList[csIDString];
                    HySXMLDataFile csDataFile = new HySXMLDataFile(csDataFilePath);
                    if (!csDataFile.Lock())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(data) error",
                            "DataFilePath", csDataFilePath);
                        return false;
                    }
                    csDataFileList.Add(csDataFile);

                    // 重複チェック
                    if (csDataFile.GetRootNode() != null)
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "data already exists",
                            "RiverCode", csCrossSectionData.GetRiverCode(),
                            "DistancePost", csCrossSectionData.GetDistancePost(),
                            "AuxRiverCode", csCrossSectionData.GetAuxRiverCode(),
                            "DataFilePath", csDataFilePath);
                        return false;
                    }

                    // XMLデータを作成する
                    if (!CreateXMLDataFileFromObject(csIDString, csCrossSectionData, csDataFile))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateXMLDataFileFromObject() error",
                            "RiverCode", csCrossSectionData.GetRiverCode(),
                            "DistancePost", csCrossSectionData.GetDistancePost(),
                            "AuxRiverCode", csCrossSectionData.GetAuxRiverCode());
                        return false;
                    }
                    // データファイルを保存する
                    if (!csDataFile.Save())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Save(data) error",
                            "DataFilePath", csDataFilePath);
                        return false;
                    }

                    // インデックスファイルをロック
                    string csIndexFilePath = csIndexFilePathList[csIDString];
                    HySXMLDataFile csIndexFile;
                    if (!csIndexFileList.ContainsKey(csIndexFilePath))
                    {
                        csIndexFile = new HySXMLDataFile(csIndexFilePath);
                        if (!csIndexFile.Lock())
                        {
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(index) error",
                                "IndexFilePath", csIndexFilePath);
                            return false;
                        }
                        csIndexFileList.Add(csIndexFilePath, csIndexFile);
                    }
                    else
                    {
                        csIndexFile = csIndexFileList[csIndexFilePath];
                    }
                    // インデックスファイルに追加
                    AppendNewCrossSectionToIndex(csIDString, csCrossSectionData, csIndexFile);
                    // インデックスファイルを保存
                    if (!csIndexFile.Save())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Save(index) error",
                            "IndexFilePath", csIndexFilePath);
                        return false;
                    }

                }

                // データファイルをコミット
                foreach (HySXMLDataFile csXmlFile in csDataFileList)
                {
                    if (!csXmlFile.Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Commit(data) error");
                        return false;
                    }
                }
                // インデックスファイルをコミット
                foreach (string csIndexFilePath in csIndexFileList.Keys)
                {
                    if (!csIndexFileList[csIndexFilePath].Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Commit(index) error");
                        return false;
                    }
                }

                // ここで正常終了
                return true;
            }
            finally
            {
                foreach (string csIndexFilePath in csIndexFileList.Keys)
                {
                    csIndexFileList[csIndexFilePath].Unlock();
                }
                foreach (HySXMLDataFile csDataFile in csDataFileList)
                {
                    csDataFile.Unlock();
                }
            } 
        }

        /// <summary><para>method outline:</para>
        /// <para>登録済みデータの更新</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Update(csStockData) </para>
        /// </example>
        /// <param name="csStockData">更新対象のデータ</param>
        /// <returns>true:成功、false:失敗(データベースは更新されていない)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Update(HySStockData csStockData)
        {
            const string csMyMethodName = "Update";
            // データファイルパス、インデックスファイルパスを生成する
            string csIDString;
            string csDataFilePath;
            string csIndexFilePath;
            if (!CreateDataFilePathFromStockData(csStockData, out csIDString, out csDataFilePath, out csIndexFilePath))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateDataFilePathFromStockData() error");
                return false;
            }
            // インデックスファイルをロックする
            HySXMLDataFile csIndexFile = new HySXMLDataFile(csIndexFilePath);
            if (!csIndexFile.Lock())
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(index) error");
                return false;
            }
            try
            {
                // データファイルをロックする
                HySXMLDataFile csDataFile = new HySXMLDataFile(csDataFilePath);
                if (!csDataFile.Lock())
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(data) error");
                    return false;
                }
                try
                {
                    // 未登録チェック
                    if (csDataFile.GetRootNode() == null)
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "data not registered",
                            "RiverCode", ((HySGeoRiverCrossSectionData)csStockData.GetData()).GetRiverCode(),
                            "DistancePost", ((HySGeoRiverCrossSectionData)csStockData.GetData()).GetDistancePost(),
                            "AuxRiverCode", ((HySGeoRiverCrossSectionData)csStockData.GetData()).GetAuxRiverCode());
                        return false;
                    }

                    // データファイルを作成する
                    if (!CreateXMLDataFileFromObject(csIDString, (HySGeoRiverCrossSectionData)csStockData.GetData(), csDataFile))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateXMLDataFileFromObject() error");
                        return false;
                    }
                    // インデックスファイルから一旦削除する
                    DeleteCrossSectionFromIndex(csIDString, csIndexFile);

                    // インデックスファイルに追加する
                    AppendNewCrossSectionToIndex(csIDString, (HySGeoRiverCrossSectionData)csStockData.GetData(), csIndexFile);

                    // 変更を一時保存する
                    if (!csDataFile.Save())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "data file save error");
                        return false;
                    }
                    if (!csIndexFile.Save())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "index file save error");
                        return false;
                    }

                    // コミット
                    if (!csDataFile.Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "data file commit error");
                        return false;
                    }
                    if (!csIndexFile.Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "index file commit error (data-index unmatch)");
                        return false;
                    }
                    // ここで正常終了
                    return true;
                }
                finally
                {
                    csDataFile.Unlock();
                }
            }
            finally
            {
                csIndexFile.Unlock();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>1件削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Delete(csID) </para>
        /// </example>
        /// <param name="csID">削除対象データのID</param>
        /// <returns>true:成功、false:失敗(データベースは更新されていない)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Delete(HySIdentifier csID)
        {
            const string csMyMethodName = "Delete(HySIdentifier)";

            // 複数一括削除バージョンで実行する
            if (!Delete(new HySIdentifier[] { csID }))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "Delete(HySIdentifier[]) error",
                    "ID", csID);
                return false;
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>複数件削除(ID配列指定)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Delete(csIDs) </para>
        /// </example>
        /// <param name="csIDs">削除対象データのID配列</param>
        /// <returns>true:成功、false:失敗(データベースは更新されていない)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Delete(HySIdentifier[] csIDs)
        {
            const string csMyMethodName = "Delete(HySIdentifier[])";

            // IDをデータファイルパスに展開する
            List<string> csDataFilePathList = new List<string>();
            foreach (HySIdentifier csID in csIDs)
            {
                string csDataFilePath;
                if (!CreateDataFilePathFromID(csID.ToString(), out csDataFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateDataFilePathFromID() error",
                        "ID", csID);
                    return false;
                }
                if (csDataFilePathList.Contains(csDataFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "duplicate ID specified",
                        "ID", csID);
                    return false;
                }
                csDataFilePathList.Add(csDataFilePath);
            }

            List<HySXMLDataFile> csDataFileList = new List<HySXMLDataFile>();
            Dictionary<string, HySXMLDataFile> csIndexFileList = new Dictionary<string, HySXMLDataFile>();
            try
            {
                // 削除処理実行
                foreach (string csDataFilePath in csDataFilePathList)
                {
                    if (File.Exists(csDataFilePath))
                    {
                        // ファイルロック
                        HySXMLDataFile csDataFile = new HySXMLDataFile(csDataFilePath);
                        if (!csDataFile.Lock())
                        {
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(data) error",
                                "DataFilePath", csDataFilePath);
                            return false;
                        }
                        csDataFileList.Add(csDataFile);

                        // データファイルを削除する
                        if (!csDataFile.Delete())
                        {
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Delete() error");
                            return false;
                        }

                        // インデックスファイルからも削除する
                        string csID = Path.GetFileNameWithoutExtension(csDataFilePath);
                        string csIndexFilePath;
                        if (!CreateIndexFilePathFromID(csID, out csIndexFilePath))
                        {
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateIndexFilePathFromID() error");
                            return false;
                        }
                        HySXMLDataFile csIndexFile;
                        if (!csIndexFileList.ContainsKey(csIndexFilePath))
                        {
                            csIndexFile = new HySXMLDataFile(csIndexFilePath);
                            if (!csIndexFile.Lock())
                            {
                                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(index) error");
                                return false;
                            }
                            csIndexFileList.Add(csIndexFilePath, csIndexFile);
                        }
                        else
                        {
                            csIndexFile = csIndexFileList[csIndexFilePath];
                        }
                        HySXmlNode csIndexRoot = csIndexFile.GetRootNodeForUpdate();
                        if (csIndexRoot == null)
                        {
                            // インデックスファイルが更新できない
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "index file cannot update",
                                "IndexFilePath", csIndexFile.GetDataFilePath());
                            return false;
                        }
                        HySXmlNodeList csIndexItems = HySXMLDataFile.GetChildNodeList(csIndexRoot, NODE_NAME_CROSSSECTION);
                        long lItemCount = csIndexItems.GetCount();
                        for (long lIdx = 0; lIdx < lItemCount; ++lIdx)
                        {
                            HySXmlNode csIdxItem = csIndexItems.GetNode(lIdx);
                            if (csID == csIdxItem.GetAttribute(ATTR_NAME_ID))
                            {
                                csIndexRoot.RemoveChild(csIdxItem);
                                if (!csIndexFile.Save())
                                {
                                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Save(index) error");
                                    return false;
                                }
                                break;
                            }
                        }
                    }
                }
                if (csDataFileList.Count == 0)
                {
                    // 削除するデータがなければ終了
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "no data to delete");
                    return true;
                }

                // インデックスファイルをコミット
                foreach (string csIndexFilePath in csIndexFileList.Keys)
                {
                    if (!csIndexFileList[csIndexFilePath].Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Commit(index) error");
                        return false;
                    }
                }
                // データファイルをコミット
                foreach (HySXMLDataFile csXmlFile in csDataFileList)
                {
                    if (!csXmlFile.Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Commit(data) error");
                        return false;
                    }
                }

                // ここで正常終了
                return true;
            }
            finally
            {
                foreach (string csIndexFilePath in csIndexFileList.Keys)
                {
                    csIndexFileList[csIndexFilePath].Unlock();
                }
                foreach (HySXMLDataFile csDataFile in csDataFileList)
                {
                    csDataFile.Unlock();
                }
            } 
        }

        /// <summary><para>method outline:</para>
        /// <para>複数件一括削除(検索条件指定)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Delete(csQueryCtrlData) </para>
        /// </example>
        /// <param name="csQueryCtrlData">削除対象データの検索条件</param>
        /// <returns>true:成功、false:失敗(データベースは更新されていない)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool Delete(HySQueryCtlData csQueryCtrlData)
        {
            const string csMyMethodName = "Delete(HySQueryCtlData)";

            // 検索条件をデータファイルパスに展開する
            List<string> csDataFilePathList;
            if (!ExpandDataFilePathList(csQueryCtrlData, out csDataFilePathList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "ExpandDataFilePathList() error");
                return false;
            }

            List<HySXMLDataFile> csDataFileList = new List<HySXMLDataFile>();
            Dictionary<string, HySXMLDataFile> csIndexFileList = new Dictionary<string, HySXMLDataFile>();
            try
            {
                // データファイルごとに検索条件をチェックする
                foreach (string csDataFilePath in csDataFilePathList)
                {
                    // ファイルロック
                    HySXMLDataFile csDataFile = new HySXMLDataFile(csDataFilePath);
                    if (!csDataFile.Lock())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(data) error",
                            "DataFilePath", csDataFilePath);
                        return false;
                    }
                    csDataFileList.Add(csDataFile);

                    // 検索条件チェック
                    bool bIsMatch = false;
                    if (csQueryCtrlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.ID) == null &&
                        csQueryCtrlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.RIVER_CODE) == null &&
                        csQueryCtrlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.DISTANCE_POST) == null &&
                        csQueryCtrlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.AUX_RIVER_CODE) == null)
                    {
                        // 検索条件なし→一致とする
                        bIsMatch = true;
                    }
                    else
                    {
                        // 検索条件チェックを行う
                        HySXmlNode csCrossSectionNode = csDataFile.GetRootNode();
                        if (csCrossSectionNode == null)
                        {
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "xml file read error",
                                "DataFilePath", csDataFile.GetDataFilePath());
                            return false;
                        }
                        bIsMatch = IsMatchNode(csCrossSectionNode, csQueryCtrlData);
                    }
                    if (bIsMatch)
                    {
                        // データファイルを削除する
                        if (!csDataFile.Delete())
                        {
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Delete() error");
                            return false;
                        }

                        // インデックスファイルからも削除する
                        string csID = Path.GetFileNameWithoutExtension(csDataFilePath);
                        string csIndexFilePath;
                        if (!CreateIndexFilePathFromID(csID, out csIndexFilePath))
                        {
                            HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateIndexFilePathFromID() error");
                            return false;
                        }
                        HySXMLDataFile csIndexFile;
                        if (!csIndexFileList.ContainsKey(csIndexFilePath))
                        {
                            csIndexFile = new HySXMLDataFile(csIndexFilePath);
                            if (!csIndexFile.Lock())
                            {
                                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(index) error");
                                return false;
                            }
                            csIndexFileList.Add(csIndexFilePath, csIndexFile);
                        }
                        else
                        {
                            csIndexFile = csIndexFileList[csIndexFilePath];
                        }
                        HySXmlNode csIndexRoot = csIndexFile.GetRootNodeForUpdate();
                        HySXmlNodeList csIndexItems = HySXMLDataFile.GetChildNodeList(csIndexRoot, NODE_NAME_CROSSSECTION);
                        long lItemCount = csIndexItems.GetCount();
                        for (long lIdx = 0; lIdx < lItemCount; ++lIdx)
                        {
                            HySXmlNode csIdxItem = csIndexItems.GetNode(lIdx);
                            if (csID == csIdxItem.GetAttribute(ATTR_NAME_ID))
                            {
                                csIndexRoot.RemoveChild(csIdxItem);
                                if (!csIndexFile.Save())
                                {
                                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Save(index) error");
                                    return false;
                                }
                                break;
                            }
                        }
                    }
                    else
                    {
                        // 不一致のものはロック解放して処理対象から外す
                        csDataFile.Unlock();
                        csDataFileList.Remove(csDataFile);
                    }
                }
                if (csDataFileList.Count == 0)
                {
                    // 削除するデータがなければ終了
                    //HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "no data to delete");
                    return true;
                }

                // データファイルをコミット
                foreach (HySXMLDataFile csXmlFile in csDataFileList)
                {
                    if (!csXmlFile.Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Commit(data) error");
                        return false;
                    }
                }
                // インデックスファイルをコミット
                foreach (string csIndexFilePath in csIndexFileList.Keys)
                {
                    if (!csIndexFileList[csIndexFilePath].Commit())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Commit(index) error");
                        return false;
                    }
                }

                // ここで正常終了
                return true;
            }
            finally
            {
                foreach (string csIndexFilePath in csIndexFileList.Keys)
                {
                    csIndexFileList[csIndexFilePath].Unlock();
                }
                foreach (HySXMLDataFile csDataFile in csDataFileList)
                {
                    csDataFile.Unlock();
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>データ1件取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetData(csID, csStockData) </para>
        /// </example>
        /// <param name="csID">取得対象データのID</param>
        /// <param name="csStockData">取得したデータオブジェクト</param>
        /// <returns>true:成功、false:失敗(指定IDのデータは存在しない、その他エラー)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool GetData(HySIdentifier csID, out HySStockData csStockData)
        {
            const string csMyMethodName = "GetData";
            csStockData = null;

            // 一括取得バージョンで実行する
            HySStockDataList csStockDataList;
            if (!GetDataList(new HySIdentifier[] { csID }, out csStockDataList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "GetDataList() error");
                return false;
            }
            // データを詰め替える
            csStockData = (HySStockData)csStockDataList.GetFirstData();
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>データ複数件取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetDataList(csIDs, csStockDataList) </para>
        /// </example>
        /// <param name="csIDs">取得対象データのID配列</param>
        /// <param name="csStockDataList">取得したデータオブジェクトのリスト</param>
        /// <returns>true:成功、false:失敗(指定ID配列のデータの1つ以上が存在しない、その他エラー)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool GetDataList(HySIdentifier[] csIDs, out HySStockDataList csStockDataList)
        {
            const string csMyMethodName = "GetDataList";
            csStockDataList = null;

            // IDをデータファイルパスに展開する
            List<string> csDataFilePathList = new List<string>();
            foreach (HySIdentifier csID in csIDs)
            {
                string csDataFilePath;
                if (!CreateDataFilePathFromID(csID.ToString(), out csDataFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateDataFilePathFromID() error",
                        "ID", csID);
                    return false;
                }
                // ファイル存在チェック
                if (!File.Exists(csDataFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "data file not exists",
                        "ID", csID);
                    return false;
                }
                // ID重複チェック
                if (csDataFilePathList.Contains(csDataFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "duplicate ID specified",
                        "ID", csID);
                    return false;
                }
                csDataFilePathList.Add(csDataFilePath);
            }

            List<HySXMLDataFile> csDataFileList = new List<HySXMLDataFile>();
            Dictionary<string, HySXMLDataFile> csIndexFileList = new Dictionary<string, HySXMLDataFile>();
            try
            {
                // ファイルロック
                foreach (string csDataFilePath in csDataFilePathList)
                {
                    HySXMLDataFile csDataFile = new HySXMLDataFile(csDataFilePath);
                    if (!csDataFile.Lock())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock(data) error",
                            "DataFilePath", csDataFilePath);
                        return false;
                    }
                    csDataFileList.Add(csDataFile);
                }
                // データ取得処理実行
                HySStockDataList csStockDataListWork = new HySStockDataList();
                foreach (HySXMLDataFile csDataFile in csDataFileList)
                {
                    // XMLファイルをロードしてルートノードを取得する
                    HySXmlNode csCrossSectionNode = csDataFile.GetRootNode();
                    if (csCrossSectionNode == null)
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "XML file load error (root node is null)",
                            "DataFilePath", csDataFile.GetDataFilePath());
                        return false;
                    }
                    // XMLデータから断面形状データを作成する
                    HySGeoRiverCrossSectionData csCrossSectionData;
                    if (!CreateCrossSectionDataFromXML(csCrossSectionNode, out csCrossSectionData))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateCrossSectionDataFromXML() error");
                        return false;
                    }
                    // 出力データリストに追加する
                    HySStockData csStockData = new HySStockData(new HySObjectKind(), new HySID(""));
                    csStockData.SetID(new HySID(csCrossSectionData.GetID().ToString())); // IDを差し替える
                    csStockData.SetData(csCrossSectionData);
                    csStockDataListWork.AddLast(csStockData);
                }
                // ここで正常終了
                csStockDataList = csStockDataListWork;
                return true;
            }
            finally
            {
                foreach (HySXMLDataFile csDataFile in csDataFileList)
                {
                    csDataFile.Unlock();
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>一覧検索</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn  = SearchList(csQueryCtlData, out csStockDataList)</para>
        /// </example>
        /// <param name="csQueryCtlData">取得対象データの検索条件</param>
        /// <param name="csStockDataList">取得したデータオブジェクトのリスト</param>
        /// <returns>true:成功(0件を含む)、false:失敗(検索条件指定誤り、その他エラー)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override bool SearchList(HySQueryCtlData csQueryCtlData, out HySStockDataList csStockDataList)
        {
            const string csMyMethodName = "SearchList";
            csStockDataList = null;

            List<string> csFilePathList;
            List<HySXMLDataFile> csFileList = new List<HySXMLDataFile>();   // 後でロック解放するためのリスト
            try
            {
                /*
                 * 検索条件からファイルパスを展開する
                 */
                if (csQueryCtlData.GetIsMetaDataOnly())
                {
                    // インデックスファイルパスを展開する
                    if (!ExpandIndexFilePathList(csQueryCtlData, out csFilePathList))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "ExpandIndexFilePathList() error");
                        return false;
                    }
                }
                else
                {
                    // データファイルパスを展開する
                    if (!ExpandDataFilePathList(csQueryCtlData, out csFilePathList))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "ExpandDataFilePathList() error");
                        return false;
                    }
                }

                /*
                 * ロックをかける
                 */
                foreach (string csFilePath in csFilePathList)
                {
                    HySXMLDataFile csFile = new HySXMLDataFile(csFilePath);
                    if (!csFile.Lock())
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "HySXMLDataFile.Lock() error",
                            "FilePath", csFilePath);
                        return false;
                    }
                    csFileList.Add(csFile);
                }

                /*
                 * 検索実行＆データ収集
                 */
                HySStockDataList csStockDataListWork = new HySStockDataList();
                if (csQueryCtlData.GetIsMetaDataOnly())
                {
                    if (!CollectIndexFile(csFileList, csQueryCtlData, ref csStockDataListWork))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectIndexFile() error");
                        return false;
                    }
                }
                else
                {
                    if (!CollectDataFile(csFileList, csQueryCtlData, ref csStockDataListWork))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectDataFile() error");
                        return false;
                    }
                }

                // ここで正常終了
                csStockDataList = csStockDataListWork;
                return true;
            }
            finally
            {
                // ロック解放
                foreach (HySXMLDataFile csFile in csFileList)
                {
                    csFile.Unlock();
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>XMLデータファイルから横断面データオブジェクト生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CreateCrossSectionDataFromXML(csCrossSectionNode, out csGeoRiverCrossSectionData)</para>
        /// </example>
        /// <param name="csCrossSectionNode">crossSectionノード</param>
        /// <param name="csGeoRiverCrossSectionData">河川横断面データオブジェクト</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CreateCrossSectionDataFromXML(HySXmlNode csCrossSectionNode, out HySGeoRiverCrossSectionData csGeoRiverCrossSectionData)
        {
            const string csMyMethodName = "CreateCrossSectionDataFromXML";
            csGeoRiverCrossSectionData = null;

            // ノード名チェック(インデックスファイルとの共通処理のため)
            if (csCrossSectionNode.GetName() != NODE_NAME_CROSSSECTION)
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "invalid node name",
                    "name", csCrossSectionNode.GetName());
                return false;
            }

            // 子要素を取得
            HySXmlNode csShapeNode = null;
            HySXmlNode csAttributesNode = null;
            HySXmlNode csSectionAttributeNode = null;
            HySXmlNode csPointAttributeNode = null;
            HySXmlNode csLineAttributeNode = null;
            HySXmlNode csDistancePostNode = null;
            HySXmlNode[] csGeographicCoordinatesNode = new HySXmlNode[2];   // left, right
            csShapeNode = HySXMLDataFile.GetSingleChildNode(csCrossSectionNode, NODE_NAME_SHAPE);
            csAttributesNode = HySXMLDataFile.GetSingleChildNode(csCrossSectionNode, NODE_NAME_ATTRIBUTES);
            if (csAttributesNode != null)
            {
                csSectionAttributeNode = HySXMLDataFile.GetSingleChildNode(csAttributesNode, NODE_NAME_SECTIONATTRIBUTE);
                csPointAttributeNode = HySXMLDataFile.GetSingleChildNode(csAttributesNode, NODE_NAME_POINTATTRIBUTES);
                csLineAttributeNode = HySXMLDataFile.GetSingleChildNode(csAttributesNode, NODE_NAME_LINEATTRIBUTES);
            }
            csDistancePostNode = HySXMLDataFile.GetSingleChildNode(csCrossSectionNode, NODE_NAME_DISTANCEPOST);
            if (csDistancePostNode != null)
            {
                HySXmlNode csNode;
                if ((csNode = HySXMLDataFile.GetSingleChildNode(csDistancePostNode, NODE_NAME_LEFTDISTANCEPOST)) != null)
                {
                    csGeographicCoordinatesNode[0] = HySXMLDataFile.GetSingleChildNode(csNode, NODE_NAME_GEOGRAPHICCOORDINATES);
                }
                if ((csNode = HySXMLDataFile.GetSingleChildNode(csDistancePostNode, NODE_NAME_RIGHTDISTANCEPOST)) != null)
                {
                    csGeographicCoordinatesNode[1] = HySXMLDataFile.GetSingleChildNode(csNode, NODE_NAME_GEOGRAPHICCOORDINATES);
                }
            }

            /*
             * 断面要素 → HySGeoRiverCrossSectionData
             */
            string csRiverCodeString = csCrossSectionNode.GetAttribute(ATTR_NAME_RIVERCODE);
            string csDistancePostString = csCrossSectionNode.GetAttribute(NODE_NAME_DISTANCEPOST);
            string csAuxRiverCodeString = csCrossSectionNode.GetAttribute(ATTR_NAME_AUXRIVERCODE);
            string csIDString = csCrossSectionNode.GetAttribute(ATTR_NAME_ID);
            HySGeoRiverCrossSectionData csCrossSectionData = new HySGeoRiverCrossSectionData(new HySString(csIDString));
            csCrossSectionData.SetRiverCode(new HySString(csRiverCodeString));
            csCrossSectionData.SetAuxRiverCode(new HySString(csAuxRiverCodeString));
            csCrossSectionData.SetDistancePost(new HySString(csDistancePostString));

            /*
             * 断面要素/断面形状コンテナ要素 → HySGeoRiverCrossSectionShape
             * 断面要素/断面形状コンテナ要素/断面座標要素 → HySGeoRiverCrossSectionShapePoint
             */
            HySGeoRiverCrossSectionShape csShape = new HySGeoRiverCrossSectionShape();
            HySDataLinkedList csShapePointList = new HySDataLinkedList();
            csShape.SetShapePoints(csShapePointList);
            csCrossSectionData.SetSectionShape(csShape);
            int iPointNumber = 0;
            if (csShapeNode != null)
            {
                HySXmlNodeList csPointNodeList = HySXMLDataFile.GetChildNodeList(csShapeNode, NODE_NAME_POINT);
                if (csPointNodeList != null)
                {
                    long lNodeCount = csPointNodeList.GetCount();
                    for (long lIdx = 0; lIdx < lNodeCount; ++lIdx)
                    {
                        HySXmlNode csPointNode = csPointNodeList.GetNode(lIdx);
                        string csPointIDString = csPointNode.GetAttribute(ATTR_NAME_ID);    // 断面座標ID
                        string csXString = csPointNode.GetAttribute(ATTR_NAME_X);   // ｘ座標値
                        string csHString = csPointNode.GetAttribute(ATTR_NAME_H);   // ｈ座標値
                        double dXValue;
                        double dHValue;
                        if (!double.TryParse(csXString, out dXValue))
                        {
                            dXValue = 0.0;
                        }
                        if (!double.TryParse(csHString, out dHValue))
                        {
                            dHValue = 0.0;
                        }
                        HySGeoRiverCrossSectionShapePoint csPoint = new HySGeoRiverCrossSectionShapePoint();
                        csPoint.SetIDString(new HySString(csPointIDString));
                        csPoint.SetXValue(dXValue);
                        csPoint.SetHValue(dHValue);
                        csShapePointList.AddLast(csPoint);
                        ++iPointNumber;
                    }
                }
            }
            if (iPointNumber == 0)
            {
                // 点列が存在しない場合、インデックスファイルの最小ｘ座標値、最大ｘ座標値を設定する
                string csMinXCoordinateString = csCrossSectionNode.GetAttribute(ATTR_NAME_MINXCOORDINATE);  // インデックスファイルのみ存在
                string csMaxXCoordinateString = csCrossSectionNode.GetAttribute(ATTR_NAME_MAXXCOORDINATE);  // インデックスファイルのみ存在
                double dMinXCoordinate;
                double dMaxXCoordinate;
                if (double.TryParse(csMinXCoordinateString, out dMinXCoordinate))
                {
                    csCrossSectionData.SetMinXCoordinate(dMinXCoordinate);
                }
                if (double.TryParse(csMaxXCoordinateString, out dMaxXCoordinate))
                {
                    csCrossSectionData.SetMaxXCoordinate(dMaxXCoordinate);
                }
            }

            /*
             * 断面要素/属性コンテナ要素/断面属性要素/属性要素/属性項目要素 → HySGeoRiverCrossSectionAttribute
             */
            HySDataLinkedList csSectionAttributeList = new HySDataLinkedList();
            csCrossSectionData.SetSectionAttribute(csSectionAttributeList);
            if (csSectionAttributeNode != null)
            {
                HySXmlNode csAttributeNode = HySXMLDataFile.GetSingleChildNode(csSectionAttributeNode, NODE_NAME_ATTRIBUTE);
                if (csAttributeNode != null)
                {
                    HySXmlNodeList csItemNodeList = HySXMLDataFile.GetChildNodeList(csAttributeNode, NODE_NAME_ITEM);
                    if (csItemNodeList != null)
                    {
                        long lItemCount = csItemNodeList.GetCount();
                        for (long lItemIdx = 0; lItemIdx < lItemCount; ++lItemIdx)
                        {
                            HySXmlNode csItemNode = csItemNodeList.GetNode(lItemIdx);
                            string csItemIDString = csItemNode.GetAttribute(ATTR_NAME_ID);  // 断面属性定義ID
                            string csItemValueString = csItemNode.GetAttribute(ATTR_NAME_VALUE);    // 属性値
                            HySGeoRiverCrossSectionAttribute csAttribute = new HySGeoRiverCrossSectionAttribute();
                            csAttribute.SetIDString(new HySString(csItemIDString));
                            csAttribute.SetValue(new HySString(csItemValueString));
                            csSectionAttributeList.AddLast(csAttribute);
                        }
                    }
                }
            }
            /*
             * 断面要素/属性コンテナ要素/ポイント属性コンテナ要素/属性要素/属性項目要素 → HySGeoRiverCrossSectionAttribute
             */
            HySDataHashTable csPointAttributeTable = new HySDataHashTable();
            csCrossSectionData.SetPointAttributes(csPointAttributeTable);
            if (csPointAttributeNode != null)
            {
                CreatePointLineAttributeCommon(csPointAttributeTable, csPointAttributeNode);
            }
            /*
             * 断面要素/属性コンテナ要素/ライン属性コンテナ要素/属性要素/属性項目要素 → HySGeoRiverCrossSectionAttribute
             */
            HySDataHashTable csLineAttributeTable = new HySDataHashTable();
            csCrossSectionData.SetLinetAttributes(csLineAttributeTable);
            if (csLineAttributeNode != null)
            {
                CreatePointLineAttributeCommon(csLineAttributeTable, csLineAttributeNode);
            }

            /*
             * 断面要素/距離標要素 → HySGeoRiverDistancePost
             * 断面要素/距離標要素/左岸距離標要素/地理座標 → HySGeoRiverDistancePost
             * 断面要素/距離標要素/右岸距離標要素/地理座標 → HySGeoRiverDistancePost
             */
            if (csDistancePostNode != null)
            {
                HySGeoRiverDistancePost csGeoDistancePost = new HySGeoRiverDistancePost();
                csCrossSectionData.SetGeoRiverDistancePost(csGeoDistancePost);
                string csDistance = csDistancePostNode.GetAttribute(ATTR_NAME_DISTANCE);
                double dDistance;
                if (!double.TryParse(csDistance, out dDistance))
                {
                    dDistance = 0.0;
                }
                csGeoDistancePost.SetDistance(dDistance);

                for (int i = 0; i < csGeographicCoordinatesNode.Length; ++i)
                {
                    if (csGeographicCoordinatesNode[i] != null)
                    {
                        string csLatitudeString = csGeographicCoordinatesNode[i].GetAttribute(ATTR_NAME_LATITUDE);  // 緯度座標値
                        string csLongitudeString = csGeographicCoordinatesNode[i].GetAttribute(ATTR_NAME_LONGITUDE);    // 経度座標値
                        double dLatitude;
                        double dLongitude;
                        double dAltitude = 0.0; // 標高:常にゼロ
                        if (!double.TryParse(csLatitudeString, out dLatitude))
                        {
                            dLatitude = 0;
                        }
                        if (!double.TryParse(csLongitudeString, out dLongitude))
                        {
                            dLongitude = 0;
                        }
                        HySGeoPoint csGeoPoint = new HySGeoPoint(dLatitude, dLongitude, dAltitude);
                        if (i == 0)
                        {
                            csGeoDistancePost.SetLeftGeoPoint(csGeoPoint);
                        }
                        else
                        {
                            csGeoDistancePost.SetRightGeoPoint(csGeoPoint);
                        }
                    }
                }
            }

            // 出力設定
            csGeoRiverCrossSectionData = csCrossSectionData;
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>横断面データオブジェクトからXMLデータファイル作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CreateXMLDataFileFromObject(csIDString, csGeoRiverCrossSectionData, csDataFile)</para>
        /// </example>
        /// <param name="csIDString">断面ID</param>
        /// <param name="csGeoRiverCrossSectionData">河川横断面データ</param>
        /// <param name="csDataFile">XMLデータファイル</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CreateXMLDataFileFromObject(string csIDString, HySGeoRiverCrossSectionData csGeoRiverCrossSectionData, HySXMLDataFile csDataFile)
        {
            //const string csMyMethodName = "CreateXMLDataFileFromObject";
            // 断面要素
            HySXmlNode csCrossSectionNode = csDataFile.CreateRoot(NODE_NAME_CROSSSECTION);
            csCrossSectionNode.SetAttribute(ATTR_NAME_RIVERCODE, csGeoRiverCrossSectionData.GetRiverCode().ToString()); // 河川コード
            csCrossSectionNode.SetAttribute(ATTR_NAME_AUXRIVERCODE, csGeoRiverCrossSectionData.GetAuxRiverCode().ToString());   // 補助河川コード
            csCrossSectionNode.SetAttribute(ATTR_NAME_DISTANCEPOST, csGeoRiverCrossSectionData.GetDistancePost().ToString());   // 距離標
            csCrossSectionNode.SetAttribute(ATTR_NAME_ID, csIDString);    // 断面ID

            // 断面要素/断面形状コンテナ要素
            HySGeoRiverCrossSectionShape csShape = csGeoRiverCrossSectionData.GetSectionShape();
            if (csShape != null)
            {
                HySXmlNode csShapeNode = csDataFile.CreateElement(NODE_NAME_SHAPE);
                csCrossSectionNode.AppendChild(csShapeNode);

                // 断面要素/断面形状コンテナ要素/断面座標要素
                HySDataLinkedList csPointList = csShape.GetShapePoints();
                if (csPointList != null)
                {
                    long lPointCount = csPointList.GetCount();
                    csPointList.SetCursorFirst();
                    for (long lIdx = 0; lIdx < lPointCount; ++lIdx)
                    {
                        HySGeoRiverCrossSectionShapePoint csPoint = (HySGeoRiverCrossSectionShapePoint)csPointList.GetCursorData();
                        csPointList.MoveCursorNext();

                        HySXmlNode csPointNode = csDataFile.CreateElement(NODE_NAME_POINT);
                        csPointNode.SetAttribute(ATTR_NAME_ID, csPoint.GetIDString().ToString());   // 断面座標ID
                        csPointNode.SetAttribute(ATTR_NAME_H, csPoint.GetHValue().ToString());      // ｈ座標値
                        csPointNode.SetAttribute(ATTR_NAME_X, csPoint.GetXValue().ToString());      // ｘ座標値
                        csShapeNode.AppendChild(csPointNode);
                    }
                }
            }

            // 断面要素/属性コンテナ要素
            HySXmlNode csAttributesNode = csDataFile.CreateElement(NODE_NAME_ATTRIBUTES);
            csCrossSectionNode.AppendChild(csAttributesNode);

            // 断面要素/属性コンテナ要素/断面属性要素
            HySDataLinkedList csSectionAttributeList = csGeoRiverCrossSectionData.GetSectionAttribute();
            if (csSectionAttributeList != null)
            {
                HySXmlNode csSectionAttributeNode = csDataFile.CreateElement(NODE_NAME_SECTIONATTRIBUTE);
                csAttributesNode.AppendChild(csSectionAttributeNode);

                // 断面要素/属性コンテナ要素/断面属性要素/属性要素
                HySXmlNode csAttributeNode = csDataFile.CreateElement(NODE_NAME_ATTRIBUTE);
                csSectionAttributeNode.AppendChild(csAttributeNode);

                // 断面要素/属性コンテナ要素/断面属性要素/属性要素/属性項目要素
                long lAttrCount = csSectionAttributeList.GetCount();
                csSectionAttributeList.SetCursorFirst();
                for (long lIdx = 0; lIdx < lAttrCount; ++lIdx)
                {
                    HySGeoRiverCrossSectionAttribute csAttr = (HySGeoRiverCrossSectionAttribute)csSectionAttributeList.GetCursorData();
                    csSectionAttributeList.MoveCursorNext();

                    HySXmlNode csItemNode = csDataFile.CreateElement(NODE_NAME_ITEM);
                    csItemNode.SetAttribute(ATTR_NAME_ID, csAttr.GetIDString().ToString()); // 断面属性定義ID
                    csItemNode.SetAttribute(ATTR_NAME_VALUE, csAttr.GetValue().ToString()); // 属性値
                    csAttributeNode.AppendChild(csItemNode);
                }
            }

            // 断面要素/属性コンテナ要素/ポイント属性コンテナ要素
            HySDataHashTable csPointAttrTable = csGeoRiverCrossSectionData.GetPointAttributes();
            if (csPointAttrTable != null)
            {
                HySXmlNode csPointAttrNode = csDataFile.CreateElement(NODE_NAME_POINTATTRIBUTES);
                csAttributesNode.AppendChild(csPointAttrNode);

                CreateXMLDataPointLineAttributeCommon(csDataFile, csPointAttrNode, csPointAttrTable);
            }

            // 断面要素/属性コンテナ要素/ライン属性コンテナ要素
            HySDataHashTable csLineAttrTable = csGeoRiverCrossSectionData.GetLinetAttributes();
            if (csLineAttrTable != null)
            {
                HySXmlNode csLineAttrNode = csDataFile.CreateElement(NODE_NAME_LINEATTRIBUTES);
                csAttributesNode.AppendChild(csLineAttrNode);

                CreateXMLDataPointLineAttributeCommon(csDataFile, csLineAttrNode, csLineAttrTable);
            }

            // 断面要素/距離標要素
            HySGeoRiverDistancePost csGeoDistancePost = csGeoRiverCrossSectionData.GetGeoRiverDistancePost();
            if (csGeoDistancePost != null)
            {
                HySXmlNode csGeoDistancePostNode = csDataFile.CreateElement(NODE_NAME_DISTANCEPOST);
                csGeoDistancePostNode.SetAttribute(ATTR_NAME_DISTANCE, csGeoDistancePost.GetDistance().ToString()); // 距離
                csCrossSectionNode.AppendChild(csGeoDistancePostNode);

                // 断面要素/距離標要素/(左岸or右岸)距離標要素
                string[] csDistancePostNodeNames = new string[] { NODE_NAME_LEFTDISTANCEPOST, NODE_NAME_RIGHTDISTANCEPOST };
                HySGeoPoint[] csGeoPointList = new HySGeoPoint[csDistancePostNodeNames.Length];  // left, right
                csGeoPointList[0] = csGeoDistancePost.GetLeftGeoPoint();
                csGeoPointList[1] = csGeoDistancePost.GetRightGeoPoint();
                for (int i = 0; i < csGeoPointList.Length; ++i)
                {
                    HySGeoPoint csGeoPoint = csGeoPointList[i];
                    if (csGeoPoint != null)
                    {
                        HySXmlNode csLeftRightDistancePostNode = csDataFile.CreateElement(csDistancePostNodeNames[i]);
                        csGeoDistancePostNode.AppendChild(csLeftRightDistancePostNode);

                        // 断面要素/距離標要素/(左岸or右岸)距離標要素/地理座標
                        HySXmlNode csGeoPointNode = csDataFile.CreateElement(NODE_NAME_GEOGRAPHICCOORDINATES);
                        csGeoPointNode.SetAttribute(ATTR_NAME_LATITUDE, csGeoPoint.m_dLatitude.ToString());
                        csGeoPointNode.SetAttribute(ATTR_NAME_LONGITUDE, csGeoPoint.m_dLongitude.ToString());
                        csLeftRightDistancePostNode.AppendChild(csGeoPointNode);
                    }
                }
            }

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>ポイント属性、ライン属性XMLデータファイル生成共通処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>CreateXMLDataPointLineAttributeCommon(csDataFile, csBaseAttrNode, csAttributeTable)</para>
        /// </example>
        /// <param name="csDataFile">XMLデータファイル</param>
        /// <param name="csBaseAttrNode">基準属性ノード</param>
        /// <param name="csAttributeTable">属性テーブル</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private static void CreateXMLDataPointLineAttributeCommon(HySXMLDataFile csDataFile, HySXmlNode csBaseAttrNode, HySDataHashTable csAttributeTable)
        {
            // 断面要素/属性コンテナ要素/(ポイントorライン)属性コンテナ要素/属性要素
            HySDataLinkedList csPointIDList = csAttributeTable.GetKeyList();
            if (csPointIDList == null)
            {
                return;
            }
            long lPointCount = csPointIDList.GetCount();
            csPointIDList.SetCursorFirst();
            for (long lPointIdx = 0; lPointIdx < lPointCount; ++lPointIdx)
            {
                HySString csPointID = (HySString)csPointIDList.GetCursorData();
                csPointIDList.MoveCursorNext();

                HySXmlNode csAttributeNode = csDataFile.CreateElement(NODE_NAME_ATTRIBUTE);
                csAttributeNode.SetAttribute(ATTR_NAME_POINTID, csPointID.ToString());   // 対象断面座標ID
                csBaseAttrNode.AppendChild(csAttributeNode);

                // 断面要素/属性コンテナ要素/(ポイントorライン)属性コンテナ要素/属性要素/属性項目要素
                HySDataLinkedList csItemList = (HySDataLinkedList)csAttributeTable.GetObject(csPointID);
                long lItemCount = csItemList.GetCount();
                csItemList.SetCursorFirst();
                for (long lItemIdx = 0; lItemIdx < lItemCount; ++lItemIdx)
                {
                    HySGeoRiverCrossSectionAttribute csAttr = (HySGeoRiverCrossSectionAttribute)csItemList.GetCursorData();
                    csItemList.MoveCursorNext();

                    HySXmlNode csItemNode = csDataFile.CreateElement(NODE_NAME_ITEM);
                    csItemNode.SetAttribute(ATTR_NAME_ID, csAttr.GetIDString().ToString()); // ポイント属性定義ID
                    csItemNode.SetAttribute(ATTR_NAME_VALUE, csAttr.GetValue().ToString()); // 属性値
                    csAttributeNode.AppendChild(csItemNode);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>ポイント属性、ライン属性データオブジェクト生成共通処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>CreatePointLineAttributeCommon(csAttributeTable, csAttributeBaseNode)</para>
        /// </example>
        /// <param name="csAttributeTable">属性テーブル</param>
        /// <param name="csAttributeBaseNode">基準属性ノード。配下にattribute/itemの子を持つ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private static void CreatePointLineAttributeCommon(HySDataHashTable csAttributeTable, HySXmlNode csAttributeBaseNode)
        {
            HySXmlNodeList csAttributeNodeList = HySXMLDataFile.GetChildNodeList(csAttributeBaseNode, NODE_NAME_ATTRIBUTE);
            if (csAttributeNodeList != null)
            {
                long lAttributeCount = csAttributeNodeList.GetCount();
                for (long lAttributeIdx = 0; lAttributeIdx < lAttributeCount; ++lAttributeIdx)
                {
                    HySXmlNode csAttributeNode = csAttributeNodeList.GetNode(lAttributeIdx);
                    string csPointIDString = csAttributeNode.GetAttribute(ATTR_NAME_POINTID);
                    HySDataLinkedList csAttributeList = new HySDataLinkedList();
                    csAttributeTable.AddObject(new HySString(csPointIDString), csAttributeList);
                    HySXmlNodeList csItemNodeList = HySXMLDataFile.GetChildNodeList(csAttributeNode, NODE_NAME_ITEM);
                    if (csItemNodeList != null)
                    {
                        long lItemCount = csItemNodeList.GetCount();
                        for (long lItemIdx = 0; lItemIdx < lItemCount; ++lItemIdx)
                        {
                            HySXmlNode csItemNode = csItemNodeList.GetNode(lItemIdx);
                            string csItemIDString = csItemNode.GetAttribute(ATTR_NAME_ID);
                            string csItemValueString = csItemNode.GetAttribute(ATTR_NAME_VALUE);
                            HySGeoRiverCrossSectionAttribute csAttribute = new HySGeoRiverCrossSectionAttribute();
                            csAttribute.SetIDString(new HySString(csItemIDString));
                            csAttribute.SetValue(new HySString(csItemValueString));
                            csAttributeList.AddLast(csAttribute);
                        }
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>断面データをインデックスファイルから削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DeleteCrossSectionFromIndex(csIDString, csIndexFile)</para>
        /// </example>
        /// <param name="csIDString">断面ID</param>
        /// <param name="csIndexFile">インデックスファイル</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void DeleteCrossSectionFromIndex(string csIDString, HySXMLDataFile csIndexFile)
        {
            // ルート要素を取得する
            HySXmlNode csRootNode = csIndexFile.GetRootNodeForUpdate();
            if (csRootNode != null)
            {
                // 配下の断面形状データを断面IDで検索する
                HySXmlNodeList csIndexNodeList = HySXMLDataFile.GetChildNodeList(csRootNode, NODE_NAME_CROSSSECTION);
                if (csIndexNodeList != null)
                {
                    long lIndexCount = csIndexNodeList.GetCount();
                    for (long lIdx = 0; lIdx < lIndexCount; ++lIdx)
                    {
                        HySXmlNode csIndexNode = csIndexNodeList.GetNode(lIdx);
                        if (csIDString == csIndexNode.GetAttribute(ATTR_NAME_ID))
                        {
                            csRootNode.RemoveChild(csIndexNode);
                            break;
                        }
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>断面データをインデックスファイルに追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>AppendNewCrossSectionToIndex(csIDString, csGeoRiverCrossSectionData, csIndexFile)</para>
        /// </example>
        /// <param name="csIDString">断面ID</param>
        /// <param name="csGeoRiverCrossSectionData">河川横断面データ</param>
        /// <param name="csIndexFile">インデックスファイル</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void AppendNewCrossSectionToIndex(string csIDString, HySGeoRiverCrossSectionData csGeoRiverCrossSectionData, HySXMLDataFile csIndexFile)
        {
            //const string csMyMethodName = "AppendNewCrossSectionToIndex";

            // ルート要素を取得する
            HySXmlNode csRootNode = csIndexFile.GetRootNodeForUpdate();
            if (csRootNode == null)
            {
                // 新インデックスファイルの場合、新規作成する
                csRootNode = csIndexFile.CreateRoot(NODE_NAME_CROSSSECTIONS);
            }
            else
            {
                // 以前に発生した不整合で同じIDの断面要素が残っている可能性があるため、一旦削除処理を通す
                DeleteCrossSectionFromIndex(csIDString, csIndexFile);
            }

            // 断面要素
            HySXmlNode csCrossSectionNode = csIndexFile.CreateElement(NODE_NAME_CROSSSECTION);
            csCrossSectionNode.SetAttribute(ATTR_NAME_RIVERCODE, csGeoRiverCrossSectionData.GetRiverCode().ToString()); // 河川コード
            csCrossSectionNode.SetAttribute(ATTR_NAME_AUXRIVERCODE, csGeoRiverCrossSectionData.GetAuxRiverCode().ToString());   // 補助河川コード
            csCrossSectionNode.SetAttribute(ATTR_NAME_DISTANCEPOST, csGeoRiverCrossSectionData.GetDistancePost().ToString());   // 距離標
            csCrossSectionNode.SetAttribute(ATTR_NAME_ID, csIDString);    // 断面ID
            csCrossSectionNode.SetAttribute(ATTR_NAME_MINXCOORDINATE, csGeoRiverCrossSectionData.GetMinXCoordinate().ToString());   // 最小ｘ座標値
            csCrossSectionNode.SetAttribute(ATTR_NAME_MAXXCOORDINATE, csGeoRiverCrossSectionData.GetMaxXCoordinate().ToString());   // 最大ｘ座標値
            csRootNode.AppendChild(csCrossSectionNode);

            // 断面要素/距離標要素
            HySGeoRiverDistancePost csGeoDistancePost = csGeoRiverCrossSectionData.GetGeoRiverDistancePost();
            if (csGeoDistancePost != null)
            {
                HySXmlNode csGeoDistancePostNode = csIndexFile.CreateElement(NODE_NAME_DISTANCEPOST);
                csGeoDistancePostNode.SetAttribute(ATTR_NAME_DISTANCE, csGeoDistancePost.GetDistance().ToString()); // 距離
                csCrossSectionNode.AppendChild(csGeoDistancePostNode);

                // 断面要素/距離標要素/(左岸or右岸)距離標要素
                string[] csDistancePostNodeNames = new string[] { NODE_NAME_LEFTDISTANCEPOST, NODE_NAME_RIGHTDISTANCEPOST };
                HySGeoPoint[] csGeoPointList = new HySGeoPoint[csDistancePostNodeNames.Length];  // 左岸、右岸
                csGeoPointList[0] = csGeoDistancePost.GetLeftGeoPoint();
                csGeoPointList[1] = csGeoDistancePost.GetRightGeoPoint();
                for (int i = 0; i < csGeoPointList.Length; ++i)
                {
                    HySGeoPoint csGeoPoint = csGeoPointList[i];
                    if (csGeoPoint != null)
                    {
                        HySXmlNode csLeftRightDistancePostNode = csIndexFile.CreateElement(csDistancePostNodeNames[i]);
                        csGeoDistancePostNode.AppendChild(csLeftRightDistancePostNode);

                        // 断面要素/距離標要素/(左岸or右岸)距離標要素/地理座標
                        HySXmlNode csGeoPointNode = csIndexFile.CreateElement(NODE_NAME_GEOGRAPHICCOORDINATES);
                        csGeoPointNode.SetAttribute(ATTR_NAME_LATITUDE, csGeoPoint.m_dLatitude.ToString());
                        csGeoPointNode.SetAttribute(ATTR_NAME_LONGITUDE, csGeoPoint.m_dLongitude.ToString());
                        csLeftRightDistancePostNode.AppendChild(csGeoPointNode);
                    }
                }
            }

        }

        /// <summary><para>method outline:</para>
        /// <para>環境設定初期化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>InitialEnvInfo()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void InitialEnvInfo()
        {
            // 環境設定からDBフォルダを取得する
            string csDBPath = null;
            HySString csHySStringDBPath = HySEnvInf.GetEnvInf(ENVKEY_DBPATH);
            if (((object)csHySStringDBPath) != null)
            {
                csDBPath = csHySStringDBPath.ToString();
            }
            else
            {
                csDBPath = DEFAULT_DBPATH;  // fallback value
            }
            // 末尾にディレクトリ区切り文字を追加する
            if (csDBPath[csDBPath.Length - 1] != Path.DirectorySeparatorChar)
            {
                csDBPath += Path.DirectorySeparatorChar;
            }
            // DBディレクトリと連結してフルパスにする
            string csAppBasePath = HySEnvInf.GetDBDirectory().ToString();
            m_csDBPath = csAppBasePath + csDBPath;
        }

        /// <summary><para>method outline:</para>
        /// <para>DB登録データから格納ファイルパス作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CreateDataFilePathFromStockData(csStockData, out csIDString, out csDataFileFullPath, out csIndexFileFullPath)</para>
        /// </example>
        /// <param name="csStockData">DB登録データ</param>
        /// <param name="csIDString">断面ID</param>
        /// <param name="csDataFileFullPath">データファイルフルパス</param>
        /// <param name="csIndexFileFullPath">インデックスファイルフルパス</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CreateDataFilePathFromStockData(HySStockData csStockData, out string csIDString, out string csDataFileFullPath, out string csIndexFileFullPath)
        {
            const string csMyMethodName = "CreateDataFilePathFromStockData";
            csIDString = null;
            csDataFileFullPath = null;
            csIndexFileFullPath = null;

            // 保存データからID生成に必要なパラメータを取り出す
            HySGeoRiverCrossSectionData csGeoRiverCrossSectionData = (HySGeoRiverCrossSectionData)csStockData.GetData();
            string csRiverCodeString = null;
            string csAuxRiverCodeString = null;
            string csDistancePostString = null;
            HySString csHSString;
            if ((object)(csHSString = csGeoRiverCrossSectionData.GetRiverCode()) != null)
            {
                csRiverCodeString = csHSString.ToString();
            }
            if ((object)(csHSString = csGeoRiverCrossSectionData.GetDistancePost()) != null)
            {
                csDistancePostString = csHSString.ToString();
            }
            if ((object)(csHSString = csGeoRiverCrossSectionData.GetAuxRiverCode()) != null)
            {
                csAuxRiverCodeString = csHSString.ToString();
            }
            if (string.IsNullOrEmpty(csRiverCodeString) ||
                string.IsNullOrEmpty(csAuxRiverCodeString) ||
                string.IsNullOrEmpty(csDistancePostString))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "invalid attributes (null or empty)",
                    "RiverCode", csRiverCodeString,
                    "DistancePost", csDistancePostString,
                    "AuxRiverCode", csAuxRiverCodeString);
                return false;
            }
            // 断面IDを生成する
            csIDString = CreateCrossSectionID(csRiverCodeString, csDistancePostString, csAuxRiverCodeString);

            // 断面IDからファイルパスを作成する
            if (!CreateDataFilePathFromID(csIDString, out csDataFileFullPath))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateDataFilePathFromID() error");
                return false;
            }
            // 断面IDからインデックスファイルパスを作成する
            if (!CreateIndexFilePathFromID(csIDString, out csIndexFileFullPath))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateIndexFilePathFromID() error");
                return false;
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>断面IDからデータファイルパス作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CreateDataFilePathFromID(csIDString, out csDataFileFullPath)</para>
        /// </example>
        /// <param name="csIDString">断面ID文字列</param>
        /// <param name="csDataFileFullPath">データファイルパス（フルパス）</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CreateDataFilePathFromID(string csIDString, out string csDataFileFullPath)
        {
            csDataFileFullPath = null;
            const string csMyMethodName = "CreateDataFilePathFromID";
            // 断面IDを分解する
            string csRiverCode;
            string csAuxRiverCode;
            string csDistancePost;
            if (!SplitCrossSectionID(csIDString, out csRiverCode, out csDistancePost, out csAuxRiverCode))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "SplitCrossSectionID() error");
                return false;
            }
            // ファイルパスを作成する
            string csFolderPath = Path.Combine(m_csDBPath, csRiverCode);
            string csDataFileName = string.Format("{0}.xml", csIDString);
            csDataFileFullPath = Path.Combine(csFolderPath, csDataFileName);
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>断面IDからインデックスファイル名作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CreateIndexFilePathFromID(csIDString, out csIndexFileFullPath)</para>
        /// </example>
        /// <param name="csIDString">断面ID文字列</param>
        /// <param name="csIndexFileFullPath">インデックスファイルパス（フルパス）</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CreateIndexFilePathFromID(string csIDString, out string csIndexFileFullPath)
        {
            csIndexFileFullPath = null;
            const string csMyMethodName = "CreateIndexFilePathFromID";
            // 断面IDを分解する
            string csRiverCode;
            string csAuxRiverCode;
            string csDistancePost;
            if (!SplitCrossSectionID(csIDString, out csRiverCode, out csDistancePost, out csAuxRiverCode))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "SplitCrossSectionID() error");
                return false;
            }
            // ファイルパスを作成する
            string csFolderPath = Path.Combine(m_csDBPath, csRiverCode);
            string csIndexFileName = string.Format("{0}_index.xml", csRiverCode);
            csIndexFileFullPath = Path.Combine(csFolderPath, csIndexFileName);
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>検索条件をインデックスファイルパスに展開</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ExpandIndexFilePathList(csQueryCtlData, out csIndexFilePathList)</para>
        /// </example>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csIndexFilePathList">インデックスファイルパスリスト</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool ExpandIndexFilePathList(HySQueryCtlData csQueryCtlData, out List<string> csIndexFilePathList)
        {
            const string csMyMethodName = "ExpandIndexFilePathList";
            csIndexFilePathList = new List<string>();

            // IDをファイル名に変換する
            List<string> csIDList;
            if (!CollectHySIdentifierFileNamePattern(csQueryCtlData, HySGeoRiverCrossSectionQueryFieldNames.ID, out csIDList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectHySIdentifierFileNamePattern(ID) error");
                return false;
            }
            foreach (string csIDString in csIDList)
            {
                string csIndexFilePath;
                if (!CreateIndexFilePathFromID(csIDString, out csIndexFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateIndexFilePathFromID() error");
                    return false;
                }
                if (!csIndexFilePathList.Contains(csIndexFilePath))
                {
                    if (File.Exists(csIndexFilePath))
                    {
                        csIndexFilePathList.Add(csIndexFilePath);
                    }
                }
            }
            // 1件でも変換できた場合、ここで終了する
            if (csIndexFilePathList.Count > 0)
            {
                return true;
            }
            // 河川コードからファイル名を展開する
            List<string> csRiverCodeList;
            if (!CollectHySStringFileNamePattern(csQueryCtlData, HySGeoRiverCrossSectionQueryFieldNames.RIVER_CODE, out csRiverCodeList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectHySIdentifierFileNamePattern(RiverCode) error");
                return false;
            }
            // 河川コードが未指定または明示的にディレクトリ名にマッピングできない場合、
            // 全サブディレクトリを対象とする
            if (csRiverCodeList.Count == 0)
            {
                DirectoryInfo csDirInfo = new DirectoryInfo(m_csDBPath);
                foreach (DirectoryInfo csSubDirInfo in csDirInfo.GetDirectories())
                {
                    csRiverCodeList.Add(csSubDirInfo.Name);
                }
            }
            // サブディレクトリごとにファイル名一覧を取得する
            foreach (string csRiverCode in csRiverCodeList)
            {
                string csDataDir = Path.Combine(m_csDBPath, csRiverCode);
                string csIndexFilePath = Path.Combine(csDataDir, csRiverCode + "_index.xml");
                if (Directory.Exists(csDataDir) && File.Exists(csIndexFilePath))
                {
                    if (!csIndexFilePathList.Contains(csIndexFilePath))
                    {
                        csIndexFilePathList.Add(csIndexFilePath);
                    }
                }
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>検索条件をデータファイルパスに展開</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = ExpandDataFilePathList(csQueryCtlData, out csDataFilePathList)</para>
        /// </example>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csDataFilePathList">データファイルパスリスト</param>
        /// <returns>true:成功、false:失敗(非サポート検索条件)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool ExpandDataFilePathList(HySQueryCtlData csQueryCtlData, out List<string> csDataFilePathList)
        {
            const string csMyMethodName = "ExpandDataFilePathList";
            csDataFilePathList = new List<string>();

            // IDをファイル名に変換する
            List<string> csIDList;
            if (!CollectHySIdentifierFileNamePattern(csQueryCtlData, HySGeoRiverCrossSectionQueryFieldNames.ID, out csIDList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectHySIdentifierFileNamePattern(ID) error");
                return false;
            }
            foreach (string csIDString in csIDList)
            {
                string csDataFilePath;
                if (!CreateDataFilePathFromID(csIDString, out csDataFilePath))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateDataFilePathFromID() error");
                    return false;
                }
                if (!csDataFilePathList.Contains(csDataFilePath))
                {
                    if (File.Exists(csDataFilePath))
                    {
                        csDataFilePathList.Add(csDataFilePath);
                    }
                }
            }
            // 1件でも変換できた場合、ここで終了する
            if (csDataFilePathList.Count > 0)
            {
                return true;
            }

            // 河川コード、距離標、補助河川コードからファイル名を展開する
            List<string> csRiverCodeList;
            List<string> csDistancePostList;
            List<string> csAuxRiverCodeList;
            if (!CollectHySStringFileNamePattern(csQueryCtlData, HySGeoRiverCrossSectionQueryFieldNames.RIVER_CODE, out csRiverCodeList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectHySIdentifierFileNamePattern(RiverCode) error");
                return false;
            }
            if (!CollectHySStringFileNamePattern(csQueryCtlData, HySGeoRiverCrossSectionQueryFieldNames.DISTANCE_POST, out csDistancePostList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectHySIdentifierFileNamePattern(DistancePost) error");
                return false;
            }
            if (!CollectHySStringFileNamePattern(csQueryCtlData, HySGeoRiverCrossSectionQueryFieldNames.AUX_RIVER_CODE, out csAuxRiverCodeList))
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CollectHySIdentifierFileNamePattern(AuxRiverCode) error");
                return false;
            }
            // 河川コードが未指定または明示的にディレクトリ名にマッピングできない場合、
            // 全サブディレクトリを対象とする
            if (csRiverCodeList.Count == 0)
            {
                DirectoryInfo csDirInfo = new DirectoryInfo(m_csDBPath);
                foreach (DirectoryInfo csSubDirInfo in csDirInfo.GetDirectories())
                {
                    csRiverCodeList.Add(csSubDirInfo.Name);
                }
            }
            // 距離標、補助河川コードが未指定または明示的にファイル名にマッピングできない場合、
            // ワイルドカードを使用する
            if (csDistancePostList.Count == 0)
            {
                csDistancePostList.Add("*");
            }
            if (csAuxRiverCodeList.Count == 0)
            {
                csAuxRiverCodeList.Add("*");
            }
            // サブディレクトリごとにファイル名一覧を取得する
            foreach (string csRiverCode in csRiverCodeList)
            {
                string csDataDir = Path.Combine(m_csDBPath, csRiverCode);
                if (Directory.Exists(csDataDir))
                {
                    foreach (string csDistancePost in csDistancePostList)
                    {
                        foreach (string csAuxRiverCode in csAuxRiverCodeList)
                        {
                            string csDataFilePattern = string.Join(ID_DELIMITER.ToString(), new string[] { csRiverCode, csDistancePost, csAuxRiverCode });
                            csDataFilePattern += ".xml";
                            string[] csDataFiles = Directory.GetFiles(csDataDir, csDataFilePattern, SearchOption.TopDirectoryOnly);
                            foreach (string csDataFile in csDataFiles)
                            {
                                if (!csDataFilePathList.Contains(csDataFile))
                                {
                                    csDataFilePathList.Add(csDataFile);
                                }
                            }
                        }
                    }
                }
                
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>HySString型ファイル名展開パターン収集</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CollectHySStringFileNamePattern(csQueryCtlData, csFieldName, out csPatternList)</para>
        /// </example>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csFieldName">フィールド名</param>
        /// <param name="csPatternList">検索パターン文字列リスト</param>
        /// <returns>true:成功、false:失敗(非サポート組合せ)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CollectHySStringFileNamePattern(HySQueryCtlData csQueryCtlData, HySString csFieldName, out List<string> csPatternList)
        {
            const string csMyMethodName = "CollectHySStringPattern";
            csPatternList = new List<string>();

            HySQueryEntryData csEntry = csQueryCtlData.GetQueryEntryData(csFieldName);
            if (csEntry != null)
            {
                HySQueryCompareMode eCompareMode = csEntry.GetCompareMode();
                if (eCompareMode == HySQueryCompareMode.EQUALS)
                {
                    HySString csValue = (HySString)csEntry.GetValueToCompare();
                    string csStringValue = HySFileRecordIDUtility.CreateIDString(csValue.ToString());
                    csPatternList.Add(csStringValue);
                }
                else if (eCompareMode == HySQueryCompareMode.IN)
                {
                    HySString[] csValues = (HySString[])csEntry.GetValueToCompare();
                    foreach (HySString csValue in csValues)
                    {
                        string csStringValue = HySFileRecordIDUtility.CreateIDString(csValue.ToString());
                        csPatternList.Add(csStringValue);
                    }
                }
                else if (eCompareMode == HySQueryCompareMode.LIKE_BEGIN)
                {
                    // ファイル名に直接マッピングできないので何もしない
                }
                else if (eCompareMode == HySQueryCompareMode.LIKE_END)
                {
                    // ファイル名に直接マッピングできないので何もしない
                }
                else if (eCompareMode == HySQueryCompareMode.NOT)
                {
                    // ファイル名に直接マッピングできないので何もしない
                }
                else
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "unsupported query",
                        "FieldName", csFieldName,
                        "CompareMode", eCompareMode);
                    return false;
                }
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>HySIdentifier型ファイル名展開パターン収集</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CollectHySIdentifierFileNamePattern(csQueryCtlData, csFieldName, out csPatternList)</para>
        /// </example>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csFieldName">フィールド名</param>
        /// <param name="csPatternList">検索パターン文字列リスト</param>
        /// <returns>true:成功、false:失敗(非サポート組合せ)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CollectHySIdentifierFileNamePattern(HySQueryCtlData csQueryCtlData, HySString csFieldName, out List<string> csPatternList)
        {
            const string csMyMethodName = "CollectHySIdentifierPattern";
            csPatternList = new List<string>();

            HySQueryEntryData csEntry = csQueryCtlData.GetQueryEntryData(csFieldName);
            if (csEntry != null)
            {
                HySQueryCompareMode eCompareMode = csEntry.GetCompareMode();
                if (eCompareMode == HySQueryCompareMode.EQUALS)
                {
                    HySIdentifier csValue = (HySIdentifier)csEntry.GetValueToCompare();
                    string csStringValue = csValue.ToString();
                    csPatternList.Add(csStringValue);
                }
                else if (eCompareMode == HySQueryCompareMode.IN)
                {
                    HySIdentifier[] csValues = (HySIdentifier[])csEntry.GetValueToCompare();
                    foreach (HySIdentifier csValue in csValues)
                    {
                        string csStringValue = csValue.ToString();
                        csPatternList.Add(csStringValue);
                    }
                }
                else if (eCompareMode == HySQueryCompareMode.NOT)
                {
                    // ファイル名に直接マッピングできないので何もしない
                }
                else
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "unsupported query",
                        "FieldName", csFieldName,
                        "CompareMode", eCompareMode);
                    return false;
                }
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>断面ID作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string csID = CreateCrossSectionID(csRiverCodeString, csDistancePostString, csAuxRiverCodeString)</para>
        /// </example>
        /// <param name="csRiverCodeString">河川コード文字列</param>
        /// <param name="csDistancePostString">距離標文字列</param>
        /// <param name="csAuxRiverCodeString">補助河川コード文字列</param>
        /// <returns>断面ID</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private static string CreateCrossSectionID(string csRiverCodeString, string csDistancePostString, string csAuxRiverCodeString)
        {
            string csCrossSectionID = HySFileRecordIDUtility.CreateIDString(csRiverCodeString, csDistancePostString, csAuxRiverCodeString);
            return csCrossSectionID;
        }

        /// <summary><para>method outline:</para>
        /// <para>断面ID分解</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = SplitCrossSectionID(csIDString, out csRiverCode, out csDistancePost, csAuxRiverCode)</para>
        /// </example>
        /// <param name="csIDString">断面ID</param>
        /// <param name="csRiverCode">河川コード</param>
        /// <param name="csDistancePost">距離標</param>
        /// <param name="csAuxRiverCode">補助河川コード</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool SplitCrossSectionID(string csIDString, out string csRiverCode, out string csDistancePost, out string csAuxRiverCode)
        {
            csRiverCode = null;
            csDistancePost = null;
            csAuxRiverCode = null;
            const string csMyMethodName = "SplitCrossSectionID";
            string[] csCrossSectionIDParts = HySFileRecordIDUtility.SplitIDString(csIDString);
            if (csCrossSectionIDParts.Length != 3)
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "invalid ID",
                    "ID-string", csIDString);
                return false;
            }
            csRiverCode = csCrossSectionIDParts[0];
            csDistancePost = csCrossSectionIDParts[1];
            csAuxRiverCode = csCrossSectionIDParts[2];
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>インデックスファイル検索処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CollectIndexFile(csIndexFileList, csQueryCtlData, csStockDataList)</para>
        /// </example>
        /// <param name="csIndexFileList">インデックスファイルリスト</param>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csStockDataList">DB検索結果</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CollectIndexFile(List<HySXMLDataFile> csIndexFileList, HySQueryCtlData csQueryCtlData, ref HySStockDataList csStockDataList)
        {
            const string csMyMethodName = "CollectIndexFile";

            // インデックスファイルごとに繰り返し
            foreach (HySXMLDataFile csIndexFile in csIndexFileList)
            {
                // ルートノードを取得する
                HySXmlNode csRootNode = csIndexFile.GetRootNode();
                if (csRootNode == null)
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "xml file read error",
                        "IndexFilePath", csIndexFile.GetDataFilePath());
                    return false;
                }

                // 断面形状ノードリストを取得する
                HySXmlNodeList csCrossSectionNodeList = HySXMLDataFile.GetChildNodeList(csRootNode, NODE_NAME_CROSSSECTION);
                if (csCrossSectionNodeList == null)
                {
                    continue;   // 断面なし: 次のファイルへ
                }
                // 断面形状ノードごとに検索条件をチェックする
                long lNodeCount = csCrossSectionNodeList.GetCount();
                for (long lIdx = 0; lIdx < lNodeCount; ++lIdx)
                {
                    HySXmlNode csCrossSectionNode = csCrossSectionNodeList.GetNode(lIdx);
                    if (!IsMatchNode(csCrossSectionNode, csQueryCtlData))
                    {
                        continue;  // 検索条件不一致: 次の断面へ
                    }

                    // XMLデータから断面形状オブジェクトを作成する
                    HySGeoRiverCrossSectionData csCrossSectionData;
                    if (!CreateCrossSectionDataFromXML(csCrossSectionNode, out csCrossSectionData))
                    {
                        HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateCrossSectionDataFromXML(index) error");
                        return false;
                    }
                    // 作成したオブジェクトをDB検索結果データに追加する
                    HySStockData csStockData = new HySStockData(new HySObjectKind(), new HySID(""));
                    csStockData.SetData(csCrossSectionData);
                    csStockDataList.AddLast(csStockData);
                }
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>データファイル検索処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CollectDataFile(csDataFileList, csQueryCtlData, csStockDataList)</para>
        /// </example>
        /// <param name="csDataFileList">データファイルリスト</param>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <param name="csStockDataList">DB検索結果データ</param>
        /// <returns>true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CollectDataFile(List<HySXMLDataFile> csDataFileList, HySQueryCtlData csQueryCtlData, ref HySStockDataList csStockDataList)
        {
            const string csMyMethodName = "CollectDataFile";

            // データファイルごとに繰り返し
            foreach (HySXMLDataFile csDataFile in csDataFileList)
            {
                // XMLファイルをロードしてルートノードを取得する
                HySXmlNode csCrossSectionNode = csDataFile.GetRootNode();
                if (csCrossSectionNode == null)
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "xml file read error",
                        "DataFilePath", csDataFile.GetDataFilePath());
                    return false;
                }

                // 検索条件チェック
                if (!IsMatchNode(csCrossSectionNode, csQueryCtlData))
                {
                    continue;  // 不一致: 次のファイルへ
                }

                // XMLデータから断面形状オブジェクトを作成する
                HySGeoRiverCrossSectionData csCrossSectionData;
                if (!CreateCrossSectionDataFromXML(csCrossSectionNode, out csCrossSectionData))
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "CreateCrossSectionDataFromXML(data) error");
                    return false;
                }
                // 作成したオブジェクトをDB検索結果データに追加する
                HySStockData csStockData = new HySStockData(new HySObjectKind(), new HySID(""));
                csStockData.SetData(csCrossSectionData);
                csStockData.SetID(new HySID(csCrossSectionData.GetID().ToString()));    // IDを差し替える
                csStockDataList.AddLast(csStockData);
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>XMLノードの検索条件一致判定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsMatchNode(csCrossSectionNode, csQueryCtlData)</para>
        /// </example>
        /// <param name="csCrossSectionNode">断面形状ノード</param>
        /// <param name="csQueryCtlData">検索条件</param>
        /// <returns>true:一致、false:不一致</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>検索条件が指定されていない場合は「一致」となる</para>
        /// </remarks>
        private bool IsMatchNode(HySXmlNode csCrossSectionNode, HySQueryCtlData csQueryCtlData)
        {
            // 検索条件を取り出す
            HySQueryEntryData csQueryID = csQueryCtlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.ID);
            HySQueryEntryData csQueryRiverCode = csQueryCtlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.RIVER_CODE);
            HySQueryEntryData csQueryDistancePost = csQueryCtlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.DISTANCE_POST);
            HySQueryEntryData csQueryAuxRiverCode = csQueryCtlData.GetQueryEntryData(HySGeoRiverCrossSectionQueryFieldNames.AUX_RIVER_CODE);

            // 検索条件ごとに比較する
            if (csQueryID != null)
            {
                string csID = csCrossSectionNode.GetAttribute(ATTR_NAME_ID);
                if (!CompareHySIdentifier(csID, csQueryID))
                {
                    return false;
                }
            }
            if (csQueryRiverCode != null)
            {
                string csRiverCode = csCrossSectionNode.GetAttribute(ATTR_NAME_RIVERCODE);
                if (!CompareHySString(csRiverCode, csQueryRiverCode))
                {
                    return false;
                }
            }
            if (csQueryDistancePost != null)
            {
                string csDistancePost = csCrossSectionNode.GetAttribute(ATTR_NAME_DISTANCEPOST);
                if (!CompareHySString(csDistancePost, csQueryDistancePost))
                {
                    return false;
                }
            }
            if (csQueryAuxRiverCode != null)
            {
                string csAuxRiverCode = csCrossSectionNode.GetAttribute(ATTR_NAME_AUXRIVERCODE);
                if (!CompareHySString(csAuxRiverCode, csQueryAuxRiverCode))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>HySIdentifier型の検索条件判定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CompareHySIdentifier(csDBValue, csQueryCtlData)</para>
        /// </example>
        /// <param name="csDBValue">DBの値</param>
        /// <param name="csQueryEntry">検索条件エントリ</param>
        /// <returns>true:一致、false:不一致</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CompareHySIdentifier(string csDBValue, HySQueryEntryData csQueryEntry)
        {
            HySQueryCompareMode csCompareMode = csQueryEntry.GetCompareMode();
            if (csCompareMode == HySQueryCompareMode.EQUALS)
            {
                return csDBValue == ((HySIdentifier)csQueryEntry.GetValueToCompare()).ToString();
            }
            else if (csCompareMode == HySQueryCompareMode.IN)
            {
                HySIdentifier[] csValues = (HySIdentifier[])csQueryEntry.GetValueToCompare();
                foreach (HySIdentifier csValue in csValues)
                {
                    if (csDBValue == csValue.ToString())
                    {
                        return true;
                    }
                }
            }
            else if (csCompareMode == HySQueryCompareMode.NOT)
            {
                return csDBValue != ((HySIdentifier)csQueryEntry.GetValueToCompare()).ToString();
            }

            return false;
        }

        /// <summary><para>method outline:</para>
        /// <para>HySString型の検索条件判定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = CompareHySString(csDBValue, csQueryCtlData)</para>
        /// </example>
        /// <param name="csDBValue">DBの値</param>
        /// <param name="csQueryEntry">検索条件エントリ</param>
        /// <returns>true:一致、false:不一致</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool CompareHySString(string csDBValue, HySQueryEntryData csQueryEntry)
        {
            HySQueryCompareMode csCompareMode = csQueryEntry.GetCompareMode();
            if (csCompareMode == HySQueryCompareMode.EQUALS)
            {
                return csDBValue == ((HySString)csQueryEntry.GetValueToCompare()).ToString();
            }
            else if (csCompareMode == HySQueryCompareMode.IN)
            {
                HySString[] csValues = (HySString[])csQueryEntry.GetValueToCompare();
                foreach (HySString csValue in csValues)
                {
                    if (csDBValue == csValue.ToString())
                    {
                        return true;
                    }
                }
            }
            else if (csCompareMode == HySQueryCompareMode.LIKE_BEGIN)
            {
                return csDBValue.StartsWith(((HySString)csQueryEntry.GetValueToCompare()).ToString());
            }
            else if (csCompareMode == HySQueryCompareMode.LIKE_END)
            {
                return csDBValue.EndsWith(((HySString)csQueryEntry.GetValueToCompare()).ToString());
            }
            else if (csCompareMode == HySQueryCompareMode.NOT)
            {
                return csDBValue != ((HySString)csQueryEntry.GetValueToCompare()).ToString();
            }

            return false;
        }
    }
}

