﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSData;

namespace CommonMP.HYSSOP.CoreImpl.HSDBA.FileBase
{
    /// <summary><para>class outline:</para>
    /// <para>メタデータリスト</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>メタデータのリストを保持する</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/6/30][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class HySStockMetaDataList : HySDataRoot, IEnumerable<HySStockMetaData>, IEnumerable<HySID>, HySVersionManagement
    {
        /// <summary>メタデータテーブル</summary>
        protected Dictionary<HySID, HySStockMetaData> m_csMetaDataTable = null;

        /// <summary>自クラス名(ログ出力用)</summary>
        private const string m_csMyClassName = "HySStockMetaDataList";

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySStockMetaDataList csStockMetaDataList = new HySStockMetaDataList()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySStockMetaDataList()
        {
            m_csMetaDataTable = new Dictionary<HySID, HySStockMetaData>();
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>キーチェック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = Contains(csID)</para>
        /// </example>
        /// <param name="csID">ユニークID</param>
        /// <returns>true:在り,false:無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool Contains(HySID csID)
        {
            return m_csMetaDataTable.ContainsKey(csID);
        }

        /// <summary><para>method outline:</para>
        /// <para>メタデータ追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>AddData(csID, csStockMetaData)</para>
        /// </example>
        /// <param name="csID">ユニークID</param>
        /// <param name="csStockMetaData">メタデータ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void AddData(HySID csID, HySStockMetaData csStockMetaData)
        {
            m_csMetaDataTable.Add(csID, csStockMetaData);
        }

        /// <summary><para>method outline:</para>
        /// <para>メタデータ取得(キー指定)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySStockMetaData csStockMetaData = GetData(csID)</para>
        /// </example>
        /// <param name="csID">ユニークID</param>
        /// <returns>メタデータ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySStockMetaData GetData(HySID csID)
        {
            return m_csMetaDataTable[csID];
        }

        /// <summary>
        /// 登録件数を返す
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> int iCnt = GetCount() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>登録件数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public int GetCount()
        {
            return m_csMetaDataTable.Count;
        }

        /// <summary>
        /// インデクサ
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySStockMetaData csData = csStockMetaData[csID] </para>
        /// <para> csStockMetaData[csID] = csData </para>
        /// </example>
        /// <param name="csID">ユニークID</param>
        /// <returns>メタデータ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySStockMetaData this[HySID csID]
        {
            get { return m_csMetaDataTable[csID]; }
            set { m_csMetaDataTable[csID] = value; }
        }
	
        /// <summary><para>method outline:</para>
        /// <para>メタデータ削除(キー指定)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = RemoveData(csID)</para>
        /// </example>
        /// <param name="csID">ユニークID</param>
        /// <returns>true:正常,false: 異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public Boolean RemoveData(HySID csID)
        {
            if (m_csMetaDataTable.ContainsKey(csID))
            {
                m_csMetaDataTable.Remove(csID);
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, "RemoveData", "ID not exists",
                    "ID", csID);
            }

            return true;
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================

        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            // メタデータオブジェクトのデシリアライズ整合性回復を行う
            foreach (HySID csID in m_csMetaDataTable.Keys)
            {
                HySStockMetaData csMetaData = m_csMetaDataTable[csID];
                csMetaData.PostDeserialize();
            }

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }


        #region IEnumerable<HySStockMetaData> メンバ

        /// <summary>
        /// メタデータの列挙子を取得する
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IEnumerator(HySStockMetaData) = GetEnumerator() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>メタデータの列挙子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public IEnumerator<HySStockMetaData> GetEnumerator()
        {
            return m_csMetaDataTable.Values.GetEnumerator();
        }

        #endregion

        #region IEnumerable メンバ

        /// <summary>
        /// IDとメタデータの列挙子を取得する
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IEnumerator = IEnumerable.GetEnumerator() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>IDとメタデータの列挙子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return m_csMetaDataTable.GetEnumerator();
        }

        #endregion

        #region IEnumerable<HySID> メンバ

        /// <summary>
        /// IDの列挙子を取得する
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IEnumerator(HySID) = GetEnumerator() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>IDの列挙子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        IEnumerator<HySID> IEnumerable<HySID>.GetEnumerator()
        {
            return m_csMetaDataTable.Keys.GetEnumerator();
        }

        #endregion
    }
}
