﻿using System;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSDBA
{
    /// <summary>
    /// フィールド比較オブジェクト
    /// </summary>
    /// <remarks>
    /// <para>1個のHySStockDataと1個の検索条件フィールドとの一致・不一致を調べる</para>
    /// <para>複数項目間の関連を調べる場合、本クラスを適切にオーバーライドすること。</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/07/03][新規作成]</para>
    /// </remarks>
    public class HySDBAFieldComparer
    {
        /// <summary>
        /// フィールド比較デリゲート
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = csIsMatchDBField( csDBVal, csValueToCompare ) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public delegate bool IsMatchDBField(object csDBVal, object csValueToCompare);

        /// <summary>
        /// HySStockDataから値を取得するデリゲート
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> object obj = csGetFieldValue( csStockData ) </para>
        /// </example>
        /// <param name="csStockData">DB登録データ</param>
        /// <returns>比較に必要な値</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public delegate object GetFieldValue(HySStockData csStockData);

        /// <summary>
        /// フィールドの値を比較するデリゲート
        /// </summary>
        protected IsMatchDBField m_csIsMatch = null;

        /// <summary>
        /// フィールドの値を取得するデリゲート
        /// </summary>
        protected GetFieldValue m_csGetFieldValue = null;

        /// <summary>
        /// 検索条件エントリ
        /// </summary>
        protected HySQueryEntryData m_csQueryEntryData = null;

        /// <summary>
        /// 自クラス名(ログ出力用)
        /// </summary>
        private const string m_csMyClassName = "HySDBAFieldComparer";

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDBAFieldComparer csNewObj = new HySDBAFieldComparer() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>サブクラスでオーバーライドする場合のみ使用可能。</para>
        /// </remarks>
        protected HySDBAFieldComparer()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDBAFieldComparer csNewObj = new HySDBAFieldComparer(csQueryEntryData, delegate(HySStockData csStockData)) </para>
        /// </example>
        /// <param name="csQueryEntryData">検索条件エントリ</param>
        /// <param name="csGetFieldValue">フィールドの値を取得するデリゲート</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySDBAFieldComparer(HySQueryEntryData csQueryEntryData, GetFieldValue csGetFieldValue)
        {
            const string csMyMethodName = "HySDBAFieldComparer";

            m_csGetFieldValue = csGetFieldValue;
            m_csQueryEntryData = csQueryEntryData;

            // データ型と比較タイプにより、比較デリゲートをプリセットしておく
            try
            {
                Type csDataType = csQueryEntryData.GetDataType();
                HySQueryCompareMode csCompareMode = csQueryEntryData.GetCompareMode();
                if (csDataType == typeof(HySID))
                {
                    if (csCompareMode == HySQueryCompareMode.EQUALS)
                    {
                        m_csIsMatch = HySIDEquals;
                    }
                    else if (csCompareMode == HySQueryCompareMode.IN)
                    {
                        m_csIsMatch = HySIDIn;
                    }
                    else if (csCompareMode == HySQueryCompareMode.NOT)
                    {
                        m_csIsMatch = HySIDNot;
                    }
                }
                else if (csDataType == typeof(HySString))
                {
                    if (csCompareMode == HySQueryCompareMode.EQUALS)
                    {
                        m_csIsMatch = HySStringEquals;
                    }
                    else if (csCompareMode == HySQueryCompareMode.LIKE_BEGIN)
                    {
                        m_csIsMatch = HySStringLikeBegin;
                    }
                    else if (csCompareMode == HySQueryCompareMode.LIKE_END)
                    {
                        m_csIsMatch = HySStringLikeEnd;
                    }
                    else if (csCompareMode == HySQueryCompareMode.IN)
                    {
                        m_csIsMatch = HySStringIn;
                    }
                    else if (csCompareMode == HySQueryCompareMode.NOT)
                    {
                        m_csIsMatch = HySStringNot;
                    }
                }
                else if (csDataType == typeof(HySTime))
                {
                    if (csCompareMode == HySQueryCompareMode.EQUALS)
                    {
                        m_csIsMatch = HySTimeEquals;
                    }
                    else if (csCompareMode == HySQueryCompareMode.BETWEEN)
                    {
                        m_csIsMatch = HySTimeBetween;
                    }
                    else if (csCompareMode == HySQueryCompareMode.IN)
                    {
                        m_csIsMatch = HySTimeIn;
                    }
                    else if (csCompareMode == HySQueryCompareMode.NOT)
                    {
                        m_csIsMatch = HySTimeNot;
                    }
                }
                else if (csDataType == typeof(HySObjectKind))
                {
                    if (csCompareMode == HySQueryCompareMode.EQUALS)
                    {
                        m_csIsMatch = HySObjectKindEquals;
                    }
                    else if (csCompareMode == HySQueryCompareMode.IN)
                    {
                        m_csIsMatch = HySObjectKindIn;
                    }
                    else if (csCompareMode == HySQueryCompareMode.NOT)
                    {
                        m_csIsMatch = HySObjectKindNot;
                    }
                }
                else if (csDataType == typeof(HySStockData.PeservedPeriod))
                {
                    if (csCompareMode == HySQueryCompareMode.EQUALS)
                    {
                        m_csIsMatch = PeservedPeriodEquals;
                    }
                    else if (csCompareMode == HySQueryCompareMode.NOT)
                    {
                        m_csIsMatch = PeservedPeriodNot;
                    }
                }
                else if (csDataType == typeof(HySLong))
                {
                    if (csCompareMode == HySQueryCompareMode.EQUALS)
                    {
                        m_csIsMatch = HySLongEquals;
                    }
                    else if (csCompareMode == HySQueryCompareMode.BETWEEN)
                    {
                        m_csIsMatch = HySLongBetween;
                    }
                    else if (csCompareMode == HySQueryCompareMode.IN)
                    {
                        m_csIsMatch = HySLongIn;
                    }
                    else if (csCompareMode == HySQueryCompareMode.NOT)
                    {
                        m_csIsMatch = HySLongNot;
                    }
                }
                if (m_csIsMatch == null)
                {
                    HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "unsupported datatype and comparemode",
                        "DataType", csDataType.GetType().Name,
                        "CompareMode", csCompareMode);
                }
            }
            catch (Exception ex)
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "initialize failed",
                    "exception", ex);
                // 初期化失敗
                m_csIsMatch = null;
                m_csQueryEntryData = null;
                m_csGetFieldValue = null;
            }
        }

        /// <summary>
        /// 検索条件を評価する
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Evaluate(csStockData) </para>
        /// </example>
        /// <param name="csStockData">DB登録データ</param>
        /// <returns>true:検索条件に合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool Evaluate(HySStockData csStockData)
        {
            if (m_csIsMatch != null)
            {
                return m_csIsMatch(m_csGetFieldValue(csStockData), m_csQueryEntryData.GetValueToCompare());
            }
            else
            {
                return false;   // 初期化に失敗している場合
            }
        }

        #region HySID比較

        /// <summary>
        ///  フィールド比較メソッド：HySID -- EQUALS
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySIDEquals(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySIDEquals(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySIDEquals";

            if (csDBVal is HySID && csValueToCompare is HySID)
            {
                return csDBVal.Equals(csValueToCompare);
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal==null? csDBVal: csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare==null? null: csValueToCompare.GetType().Name);
                return false;
            }
        }

        /// <summary>
        ///  フィールド比較メソッド：HySID -- IN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySIDIn(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySIDIn(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySIDIn";
            if (csDBVal is HySID && csValueToCompare is object[])
            {
                foreach (object csID in (object[])csValueToCompare)
                {
                    if (csID is HySID && ((HySID)csDBVal).Equals((HySID)csID))
                    {
                        return true;
                    }
                }
                return false;
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }

        /// <summary>
        ///  フィールド比較メソッド：HySID -- NOT
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySIDNot(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySIDNot(object csDBVal, object csValueToCompare)
        {
            return !HySIDEquals(csDBVal, csValueToCompare);
        }
        #endregion

        #region HySString比較
        /// <summary>
        ///  フィールド比較メソッド：HySString -- EQUALS
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySStringEquals(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySStringEquals(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySStringEquals";
            if (csDBVal is HySString && csValueToCompare is HySString)
            {
                return ((HySString)csDBVal).Equals(csValueToCompare);
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }

        /// <summary>
        ///  フィールド比較メソッド：HySString -- LIKE_BEGIN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySStringLikeBegin(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySStringLikeBegin(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySStringLikeBegin";
            if (csDBVal is HySString && csValueToCompare is HySString)
            {
                return ((HySString)csDBVal).ToString().StartsWith(((HySString)csValueToCompare).ToString());
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }

        /// <summary>
        ///  フィールド比較メソッド：HySString -- LIKE_END
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySStringLikeEnd(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySStringLikeEnd(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySStringLikeEnd";
            if (csDBVal is HySString && csValueToCompare is HySString)
            {
                return ((HySString)csDBVal).ToString().EndsWith(((HySString)csValueToCompare).ToString());
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySString -- IN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySStringIn(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySStringIn(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySStringIn";
            if (csDBVal is HySString)
            {
                foreach (object csHStr in (object[])csValueToCompare)
                {
                    if (csHStr is HySString && ((HySString)csDBVal).Equals(csHStr))
                    {
                        return true;
                    }
                }
                return false;
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySString -- NOT
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySStringNot(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySStringNot(object csDBVal, object csValueToCompare)
        {
            return !HySStringEquals(csDBVal, csValueToCompare);
        }
        #endregion

        #region HySTime比較

        /// <summary>
        ///  フィールド比較メソッド：HySTime -- EQUALS
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySTimeEquals(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySTimeEquals(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySTimeEquals";
            if (csDBVal is HySTime && csValueToCompare is HySTime)
            {
                return ((HySTime)csDBVal).Equals((HySTime)csValueToCompare);
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySTime -- BETWEEN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySTimeBetween(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySTimeBetween(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySTimeBetween";
            if (csDBVal is HySTime && csValueToCompare is HySTime[])
            {
                HySTime csDBHySTimeValue = (HySTime)csDBVal;
                HySTime csFromTime = ((HySTime[])csValueToCompare)[0];
                HySTime csToTime = ((HySTime[])csValueToCompare)[1];
                bool bFrom = true;
                bool bTo = true;
                // 開始時刻が指定されている場合、それより後もしくは一致していなければならない
                if (csFromTime != null)
                {
                    bFrom = (csDBHySTimeValue.After(csFromTime) || csDBHySTimeValue.Equals(csFromTime));
                }
                // 終了時刻が指定されている場合、それより前もしくは一致していなければならない
                if (csToTime != null)
                {
                    bTo = (csDBHySTimeValue.Before(csToTime) || csDBHySTimeValue.Equals(csToTime));
                }

                return bFrom && bTo;
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySTime -- IN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySTimeIn(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySTimeIn(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySTimeIn";
            if (csDBVal is HySTime && csValueToCompare is object[])
            {
                foreach (object csHTime in (object[])csValueToCompare)
                {
                    if (csHTime is HySTime && ((HySTime)csDBVal).Equals((HySTime)csHTime))
                    {
                        return true;
                    }
                }
                return false;
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySTime -- NOT
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySTimeNot(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySTimeNot(object csDBVal, object csValueToCompare)
        {
            return !HySTimeEquals(csDBVal, csValueToCompare);
        }
        #endregion

        #region int比較

        /// <summary>
        ///  フィールド比較メソッド：int -- EQUALS
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.IntEquals(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool IntEquals(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "IntEquals";
            if (csDBVal is int && csValueToCompare is int)
            {
                return (int)csDBVal == (int)csValueToCompare;
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：int -- BETWEEN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.IntBetween(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool IntBetween(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "IntBetween";
            if (csDBVal is int && csValueToCompare is int[])
            {
                int iDBValue = (int)csDBVal;
                int iFromValue = ((int[])csValueToCompare)[0];
                int iToValue = ((int[])csValueToCompare)[1];
                if (iFromValue <= iDBValue && iDBValue <= iToValue)
                {
                    return true;
                }
                else
                {
                    return false;   // 不一致
                }
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：int -- IN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.IntIn(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool IntIn(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "IntIn";
            if (csDBVal is int && csValueToCompare is object[])
            {
                foreach (object iValue in (object[])csValueToCompare)
                {
                    if (iValue is int && (int)csDBVal == (int)iValue)
                    {
                        return true;
                    }
                }
                return false;
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：int -- NOT
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.IntNot(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool IntNot(object csDBVal, object csValueToCompare)
        {
            return !IntEquals(csDBVal, csValueToCompare);
        }
        #endregion

        #region HySObjectKind比較
        /// <summary>
        ///  フィールド比較メソッド：HySObjectKind -- IN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySObjectKindEquals(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySObjectKindEquals(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySObjectKindEquals";
            if (csDBVal is HySObjectKind && csValueToCompare is HySObjectKind)
            {
                return ((HySObjectKind)csDBVal).Equals((HySObjectKind)csValueToCompare);
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySObjectKind -- IN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySObjectKindIn(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySObjectKindIn(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySObjectKindIn";
            if (csDBVal is HySObjectKind && csValueToCompare is object[])
            {
                foreach (object csObjectKind in (object[])csValueToCompare)
                {
                    if (csObjectKind is HySObjectKind && ((HySObjectKind)csDBVal).Equals((HySObjectKind)csObjectKind))
                    {
                        return true;
                    }
                }
                return false;
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySObjectKind -- NOT
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySObjectKindNot(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySObjectKindNot(object csDBVal, object csValueToCompare)
        {
            return !HySObjectKindEquals(csDBVal, csValueToCompare);
        }
        #endregion

        #region HySStockData.PeservedPeriod比較

        /// <summary>
        ///  フィールド比較メソッド：HySStockData.PeservedPeriod -- EQUALS
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.PeservedPeriodEquals(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool PeservedPeriodEquals(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "PeservedPeriodEquals";
            if (csDBVal is HySStockData.PeservedPeriod && csValueToCompare is HySStockData.PeservedPeriod)
            {
                return (HySStockData.PeservedPeriod)csDBVal == (HySStockData.PeservedPeriod)csValueToCompare;
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySStockData.PeservedPeriod -- NOT
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.PeservedPeriodNot(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool PeservedPeriodNot(object csDBVal, object csValueToCompare)
        {
            return !PeservedPeriodEquals(csDBVal, csValueToCompare);
        }

        #endregion

        #region HySLong比較

        /// <summary>
        ///  フィールド比較メソッド：HySLong -- EQUALS
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySLongEquals(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySLongEquals(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySLongEquals";
            if (csDBVal is HySLong && csValueToCompare is HySLong)
            {
                return (((HySLong)csDBVal).m_lValue == ((HySLong)csValueToCompare).m_lValue);
            }
            else
            {
                HySDBALog.WriteOnline(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySLong -- BETWEEN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySLongBetween(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySLongBetween(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySLongBetween";
            if (csDBVal is HySLong && csValueToCompare is HySLong[])
            {
                HySLong csDBValue = (HySLong)csDBVal;
                HySLong csFromValue = ((HySLong[])csValueToCompare)[0];
                HySLong csToValue = ((HySLong[])csValueToCompare)[1];
                if (csFromValue.m_lValue <= csDBValue.m_lValue && csDBValue.m_lValue <= csToValue.m_lValue)
                {
                    return true;
                }
                else
                {
                    return false;   // 不一致
                }
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }
        /// <summary>
        ///  フィールド比較メソッド：HySLong -- IN
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySLongIn(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySLongIn(object csDBVal, object csValueToCompare)
        {
            const string csMyMethodName = "HySLongIn";
            if (csDBVal is HySLong && csValueToCompare is HySLong[])
            {
                HySLong csDBValue = (HySLong)csDBVal;
                foreach (HySLong csValue in (HySLong[])csValueToCompare)
                {
                    if (csValue != null && csDBValue.m_lValue == csValue.m_lValue)
                    {
                        return true;
                    }
                }
                return false;
            }
            else
            {
                HySDBALog.WriteDebug(m_csMyClassName, csMyMethodName, "wrong parameter type",
                    "DBValue", csDBVal == null ? csDBVal : csDBVal.GetType().Name,
                    "ValueToCompare", csValueToCompare == null ? null : csValueToCompare.GetType().Name);
                return false;
            }
        }        /// <summary>
        ///  フィールド比較メソッド：HySLong -- NOT
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = HySDBAFieldComparer.HySLongNot(csDBVal, csValueToCompare) </para>
        /// </example>
        /// <param name="csDBVal">DBの値</param>
        /// <param name="csValueToCompare">比較する値</param>
        /// <returns>true:合致する, false:合致しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public static bool HySLongNot(object csDBVal, object csValueToCompare)
        {
            return !HySLongEquals(csDBVal, csValueToCompare);
        }

        #endregion
    }
}
