// <summary>ソースコード：CSVファイル取り扱い用共通データクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSData
{    
    /// <summary><para>class outline:</para>
    /// <para>ＣＳＶデータ列</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>無し</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/05/01][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class HySCSVFileData : HySDataRoot, HySVersionManagement
    {
        /// <summary>データの種別</summary>
        protected DATA_KIND m_lDataKind = DATA_KIND.TIME_SERIES_DATA;
        /// <summary>データの種別定義</summary>
        public enum DATA_KIND
        {
            /// <summary>時系列データ</summary>
            TIME_SERIES_DATA = 1,
            /// <summary>一般線データ</summary>
            COMMON_LINE_DATA        
        }
        
        /// <summary>系列データの数</summary>
        protected long m_lSeriesNum = 1;
        //// <summary>系列データの最大数</summary>
        //protected static readonly int MAX_SERIES = 8;

        /// <summary>時刻リスト</summary>
        protected HySLinkedListTime m_csTimeList = null;
        /// <summary>基本データリスト</summary>
        protected HySLinkedListDouble m_csBaseDataList = null;
        /// <summary>データリスト</summary>
        protected HySLinkedListDouble[] m_csDataList = null;

        ///// <summary>凡例文字列:(CSV書き出し時に必要)</summary>
        //protected HySString[] m_csLegendLabel = null;

        /// <summary>読み込みファイル名称</summary>
        protected HySString m_csFileName = null;
        /// <summary>最後に追加した時刻</summary>
        protected HySTime m_csLastTime = HySTime.DEFAULT_TIME.Clone();

                
        // ユーザーに使用させない為
        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> 使用禁止 </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySLineGraphDataPrototype 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>使用禁止</para>
        /// </remarks>
        protected HySCSVFileData()
        {

            m_lDataKind = DATA_KIND.TIME_SERIES_DATA;
            m_lSeriesNum = 1;
            Init();
            //バージョン情報設定
            SetNowVersion();
        }
        
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGraphData csLineGraphData = new HySLineGraphDataPrototype(lDataKind, lSeriesNum) </para>
        /// </example>
        /// <param name="lDataKind">
        /// グラフの種別定義
        /// DATA_KIND.TIME_SERIES_DATA の時は、横軸は時間となる
        /// DATA_KIND.COMMON_LINE_DATA の時は、横軸はdouble形式の情報となる
        /// </param>
        /// <param name="lSeriesNum">時系列の数</param>
        /// <returns> HySLineGraphDataPrototype 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySCSVFileData(DATA_KIND lDataKind, long lSeriesNum)
        {
            m_lDataKind = lDataKind;
            m_lSeriesNum = lSeriesNum;
            Init();
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>初期化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Init() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>コンストラクターから呼ばれる</para>
        /// </remarks>
        protected virtual void Init()
        {
            if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
            {   // 時系列の場合
                m_csTimeList = new HySLinkedListTime();
                //m_csRawLabel = new HySString("時間");
            }
            else if (m_lDataKind == DATA_KIND.COMMON_LINE_DATA)
            {   // 一般データの場合
                m_csBaseDataList = new HySLinkedListDouble();
                //m_csRawLabel = new HySString("単位");
            }

            m_csDataList = new HySLinkedListDouble[m_lSeriesNum];
            //m_csLegendLabel = new HySString[m_lSeriesNum];
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp] = new HySLinkedListDouble();
                //m_csLegendLabel[iLp] = new HySString();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>データの種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DATA_KIND lGraphKind = GetGraphKind() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>DATA_KIND　グラフの種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual DATA_KIND GetDataind()
        {
            return m_lDataKind;
        }

        //=================================
        // データ操作を行うメソッド群
        //=================================

        /// <summary><para>method outline:</para>
        /// <para>データの系列数を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDtNum = GetSeriesNum() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long　データの系列数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetSeriesNum()
        {
            return m_lSeriesNum;
        }

        /// <summary><para>method outline:</para>
        /// <para>データ数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lNum = GetCount() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long　データ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetCount()
        {
            long lRtn = 0;
            if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
            {
                lRtn = m_csTimeList.GetCount();
            }
            else
            {
                lRtn = m_csBaseDataList.GetCount();
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>内部の情報を全てクリアする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ClearData( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ClearData()
        {
            if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
            {
                m_csTimeList.Clear();
                m_csLastTime = HySTime.DEFAULT_TIME.Clone();
            }
            else
            {
                m_csBaseDataList.Clear();
            }
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp].Clear();
            }
        }

        //=================================
        // 時系列データ操作を行うメソッド群
        //=================================

        /// <summary><para>method outline:</para>
        /// <para>時系列最後のデータを削除する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveLastData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void RemoveLastData()
        {
            if (m_lDataKind != DATA_KIND.TIME_SERIES_DATA)
            {
                return;
            }
            // 戻り値を設定
            this.SetCursorLast();
            if (this.GetCount() == 0)
            {
                m_csLastTime = HySTime.DEFAULT_TIME.Clone();
            }
            else if (this.GetCount() == 1)
            {
                m_csLastTime = HySTime.DEFAULT_TIME.Clone();
                m_csTimeList.Clear();
                for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
                {
                    m_csDataList[iLp].Clear();
                }
            }
            else
            {
                this.MoveCursorPrev();
                // 最終値の設定
                m_csLastTime = m_csTimeList.GetCursorData();
                // 最後のデータの取り外し
                m_csTimeList.RemoveLastData();
                for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
                {
                    m_csDataList[iLp].RemoveLastData();
                }
            }
            return;
        }

        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻と同一位置に持ってくる
        /// 同一時刻がない場合には、指定時刻に最も近い直後の時刻のレコード位置に持ってくる
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lRtn = SetCursor(csTime) </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long　カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>戻り値=0の場合には，指定時刻よりも後の情報が無い </para>
        /// </remarks>
        public virtual long SetCursor(HySTime csTime)
        {
            long lRtn = 0;
            if (m_csLastTime.Before(csTime) == false)
            {   // 最終情報は指定時刻よりも後
                HySTime csWorkTime;
                long lDtNum = this.GetCount();

                // リストの最後からたどってゆく
                this.SetCursorLast();
                for (long lLp = 0; lLp < lDtNum; lLp++)
                {
                    lRtn += 1;
                    csWorkTime = m_csTimeList.GetCursorData();
                    if (csWorkTime.Before(csTime) == true)
                    {   // 記録時刻が、指定時刻以前
                        this.MoveCursorNext();
                        lRtn -= 1;
                        break;
                    }
                    else if (csWorkTime.Equals(csTime) == true)
                    {
                        break;
                    }
                    this.MoveCursorPrev();
                }
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻の直後に持ってくる
        /// （指定時刻に最も近い直後の時刻のレコード位置に持ってくる）
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lRtn = SetCursorPlus(csTime) </para>
        /// csTime = 前回読み出した時刻
        /// long lReadDataNum = SetCursorPlus(csTime);
        /// lReadDataNum は今回読み出すべきデータ数
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long　=カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>戻り値=0の場合には，指定時刻よりも後の情報が無い </para>
        /// </remarks>
        public virtual long SetCursorPlus(HySTime csTime)
        {
            long lRtn = 0;
            HySTime csWorkTime;
            long lDtNum = this.GetCount();

            // リストの最後からたどってゆく
            this.SetCursorLast();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                csWorkTime = m_csTimeList.GetCursorData();

                if (csWorkTime.After(csTime) == false)
                {   // 指定時刻が、記録位置よりも後でない場合（記録位置が同じか、より前になった場合）
                    this.MoveCursorNext();
                    break;
                }
                else
                {
                    lRtn += 1;
                    this.MoveCursorPrev();
                }
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>データ追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = AddData(csTime,lDataDimNum,dData) </para>
        /// </example>
        /// <param name="csTime">時刻（横軸となる）</param>
        /// <param name="lDataDimNum">データ列数</param>
        /// <param name="dData">データ値（配列）</param>
        /// <returns>bool　true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 本データ生成時に 種別を DATA_KIND.TIME_SERIES_DATA として生成したデータの場合に使用する。
        /// 時間経過に従って追加すること。（内部で順序の整合性はチェックしない）
        /// </para>
        /// </remarks>
        public virtual bool AddData(HySTime csTime, long lDataDimNum, double[] dData)
        {
            if (m_lDataKind != DATA_KIND.TIME_SERIES_DATA ||
                m_lSeriesNum > lDataDimNum)
            {   // 型が異なってるか　データ配列数が異なる
                return false;
            }
            m_csLastTime = csTime.Clone();
            m_csTimeList.AddLast(m_csLastTime);
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp].AddLast(dData[iLp]);
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置の時間を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetCursorTimeSeriesData(ref csTime,ref dData,ref lDimNum) </para>
        /// </example>
        /// <param name="csTime">時刻（横軸となる）</param>
        /// <param name="dData">データ値（配列）</param>
        /// <param name="lDimNum">データ列数</param>
        /// <returns>bool　true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool GetCursorTimeSeriesData(ref HySTime csTime, ref double[] dData, long lDimNum)
        {
            if (m_lDataKind != DATA_KIND.TIME_SERIES_DATA)
            {
                return false;
            }
            csTime = m_csTimeList.GetCursorData();
            long lDataNum = (lDimNum < m_lSeriesNum) ? lDimNum : m_lSeriesNum;
            for (int iLp = 0; iLp < lDataNum; iLp++)
            {
                dData[iLp] = m_csDataList[iLp].GetCursorData();
            }
            return true;
        }

        //=================================
        // 時系列ではないデータ操作を行うメソッド群
        //=================================

        /// <summary><para>method outline:</para>
        /// <para>データ追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = AddData(dBaseData,lDataDimNum,dData) </para>
        /// </example>
        /// <param name="dBaseData">横軸データ</param>
        /// <param name="lDataDimNum">データ列数</param>
        /// <param name="dData">データ値（配列）</param>
        /// <returns>bool　true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>本データ生成時に 種別を DATA_KIND.COMMON_LINE_DATA として生成したデータの場合に使用する</para>
        /// </remarks>
        public virtual bool AddData(double dBaseData, long lDataDimNum, double[] dData)
        {
            if (m_lDataKind != DATA_KIND.COMMON_LINE_DATA ||
                m_lSeriesNum > lDataDimNum)
            {   // 型が異なってるか　データ配列数が異なる
                return false;
            }
            m_csBaseDataList.AddLast(dBaseData);
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp].AddLast(dData[iLp]);
            }
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置の時間を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = GetCursorCommonSeriesData(ref dBaseData,ref dData,lDimNum) </para>
        /// </example>
        /// <param name="dBaseData">横軸データ</param>
        /// <param name="dData">データ列数</param>
        /// <param name="lDimNum">データ値（配列）</param>
        /// <returns>bool　true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool GetCursorCommonSeriesData(ref double dBaseData, ref double[] dData, long lDimNum)
        {
            if (m_lDataKind != DATA_KIND.COMMON_LINE_DATA)
            {
                return false;
            }
            dBaseData = m_csBaseDataList.GetCursorData();
            long lDataNum = (lDimNum < m_lSeriesNum) ? lDimNum : m_lSeriesNum;
            for (int iLp = 0; iLp < lDataNum; iLp++)
            {
                dData[iLp] = m_csDataList[iLp].GetCursorData();
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定時系列データ（リンクリスト）を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySLinkedListDouble csDtList = GetTimeSeriesData(lDimNum) </para>
        /// </example>
        /// <param name="lDimNum">データ列数</param>
        /// <returns>HySLinkedListDouble　指定時系列データ（リンクリスト）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySLinkedListDouble GetTimeSeriesData(long lDimNum)
        {
            if (lDimNum >= 0 && lDimNum < m_lSeriesNum)
            {
                return m_csDataList[lDimNum];
            }
            else
            {
                return null;
            }
        }

        //=================================
        // 汎用カーソル操作
        //=================================

        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最初に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorFirst() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCursorFirst()
        {
            if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
            {
                m_csTimeList.SetCursorFirst();
            }
            else
            {
                m_csBaseDataList.SetCursorFirst();
            }
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp].SetCursorFirst();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最後に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorLast() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCursorLast()
        {
            if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
            {
                m_csTimeList.SetCursorLast();
            }
            else
            {
                m_csBaseDataList.SetCursorLast();
            }
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp].SetCursorLast();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を次の位置に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorNext() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void MoveCursorNext()
        {
            if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
            {
                m_csTimeList.MoveCursorNext();
            }
            else
            {
                m_csBaseDataList.MoveCursorNext();
            }
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp].MoveCursorNext();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を一つ前に戻す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorPrev() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void MoveCursorPrev()
        {
            if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
            {
                m_csTimeList.MoveCursorPrev();
            }
            else
            {
                m_csBaseDataList.MoveCursorPrev();
            }
            for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
            {
                m_csDataList[iLp].MoveCursorPrev();
            }
        }

        //=================================
        // 表示体裁を整える為のメソッド群
        //=================================
        ///// <summary><para>method outline:</para>
        ///// <para>凡例表示ラベル設定</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> SetLegend(lSeriesNo, sLegendName) </para>
        ///// </example>
        ///// <param name="lSeriesNo">凡例番号（１～凡例数）０相対番号</param>
        ///// <param name="sLegendName">凡例名称</param>
        ///// <returns>無し</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>何れ消す：使用禁止</para>
        ///// </remarks>
        //public virtual void SetLegend(long lSeriesNo, string sLegendName)
        //{
        //    this.SetLegend(lSeriesNo, new HySString(sLegendName));
        //}
        ///// <summary><para>method outline:</para>
        ///// <para>凡例表示ラベル設定</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> SetLegend(lSeriesNo, csLegendName) </para>
        ///// </example>
        ///// <param name="lSeriesNo">凡例番号（１～凡例数）０相対番号</param>
        ///// <param name="csLegendName">凡例名称</param>
        ///// <returns>無し</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>何れ消す：使用禁止</para>
        ///// </remarks>
        //public virtual void SetLegend(long lSeriesNo, HySString csLegendName)
        //{
        //    //if (lSeriesNo >= 1 && lSeriesNo <= m_lSeriesNum)
        //    //{
        //    //    m_csLegendLabel[lSeriesNo - 1] = csLegendName;
        //    //}
        //    if (lSeriesNo >= 0 && lSeriesNo < m_lSeriesNum)
        //    {
        //        m_csLegendLabel[lSeriesNo] = csLegendName;
        //    }
        //}
        ///// <summary><para>method outline:</para>
        ///// <para>凡例表示ラベル取得</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> HySString csLegendLabel = GetLegend(lSeriesNo) </para>
        ///// </example>
        ///// <param name="lSeriesNo">凡例番号（１～凡例数）</param>
        ///// <returns>HySString  凡例名称</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>何れ消す：使用禁止</para>
        ///// </remarks>
        //public virtual HySString GetLegend(long lSeriesNo)
        //{
        //    if (lSeriesNo >= 1 && lSeriesNo <= m_lSeriesNum)
        //    {
        //        return m_csLegendLabel[lSeriesNo - 1];
        //    }
        //    else
        //    {
        //        return null;
        //    }
        //}

        //=================================
        // ファイル関係
        //=================================

        /// <summary><para>method outline:</para>
        /// <para>入力ファイル名称設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetFileName(sFileName) </para>
        /// </example>
        /// <param name="sFileName">ファイル名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetFileName(string sFileName)
        {
            m_csFileName = new HySString(sFileName);
        }
        /// <summary><para>method outline:</para>
        /// <para>入力ファイル名称設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetFileName(csFileName) </para>
        /// </example>
        /// <param name="csFileName">ファイル名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetFileName(HySString csFileName)
        {
            m_csFileName = csFileName;
        }
        /// <summary><para>method outline:</para>
        /// <para>入力ファイル名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csFileName = GetFileName() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString　ファイル名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySString GetFileName()
        {
            return m_csFileName;
        }

        /// <summary><para>method outline:</para>
        /// <para>CSV形式ファイルを入力する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lRtn = ReadCSVFile( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long  0:=正常, -1:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>EXCEL等使用する場合に必要</para>
        /// </remarks>
        public virtual long ReadCSVFile()
        {
            long lRtn = -1;
            if ((object)m_csFileName != null)
            {
                HySFile csFile = new HySFile(m_csFileName);
                lRtn = this.ReadCSVFile(csFile);
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>CSV形式ファイルを入力する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lRtn = ReadCSVFile(csFile) </para>
        /// </example>
        /// <param name="csFile">入力ファイル</param>
        /// <returns>long  0:=正常, -1:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>EXCEL等使用する場合に必要</para>
        /// </remarks>
        public virtual long ReadCSVFile(HySFile csFile)
        {
            long lRtn = -1;
            HySString sOut = new HySString();
            string[] sInWk;

            if (csFile.Exist() == true)
            {  // 既存ファイルが有れば
                if (csFile.Open(HySFile.OPEN_MODE.OPEN, HySFile.READ_WRITE_MODE.READ, HySFile.DIRECTORY_MODE.NOT_MK_DIR) == 0)
                {  // 既存CSVファイルを正常に開ければ
                    csFile.ReadText(ref sOut);
                    sInWk = sOut.ToString().Split(',');
                    if (sInWk.LongLength >= 2)
                    {
                        HySString csDataName = new HySString(sInWk[0]);  // クラス名称取得
                        HySString csVerInf = new HySString(sInWk[1]);    // バージョン情報取得
                        if (csDataName.Equal(this.GetClassName()) == true || csDataName.Equal(new HySString("HySCSVFileDim1Data")))//[2010.12.16]HySCSVFileDim1Data型の追記
                        {   // クラスが一致すれば
                            lRtn = this.ReadCSV(csVerInf, csFile);       // CSV情報展開
                        }
                    }
                    csFile.Close();
                }
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>CSV形式ファイルを入力する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lRtn = ReadCSVFile(csVerInf, csFile) </para>
        /// </example>
        /// <param name="csVerInf">バージョン情報</param>
        /// <param name="csFile">入力ファイル</param>
        /// <returns>long  0:=正常, -1:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>EXCEL等使用する場合に必要</para>
        /// </remarks>
        protected virtual long ReadCSV(HySString csVerInf, HySFile csFile)
        {
            // 使用フィールド初期化
            long lDataNum = 0;
            HySString sOut = new HySString();
            long lGyo = 0;
            string[] sInWk;
            double dBaseData;
            double[] dData;
            HySTime csTime;
            long csCnt = 0;
            long lRtn = -1;

            // ＣＳＶファイルを読み込み＆展開
            while (csFile.ReadText(ref sOut) != 0)
            {
                // ','区切りで展開する
                sInWk = sOut.ToString().Split(',');
                csCnt = sInWk.LongLength;
                if (lGyo == 0) // 先頭行展開
                {
                    // デフォルト定義
                    m_lDataKind = DATA_KIND.COMMON_LINE_DATA;
                    // データ種別検索
                    for (long lP = 0; lP < csCnt; lP++)
                    {
                        string csWk = sInWk[lP].Trim();
                        // ver1.6 出力ファイル標準フォーマットの英語化
                        //if (csWk.CompareTo("時系列") == 0)
                        if ((csWk.CompareTo("TimeSeries") == 0) || csWk.CompareTo("時系列") == 0)
                        {
                            m_lDataKind = DATA_KIND.TIME_SERIES_DATA;
                            break;
                        }
                    }
                }
                else if (lGyo == 1) // 凡例表示ラベル行展開
                {
                    // 系列数設定
                    m_lSeriesNum = (csCnt <= 0) ? 0 : csCnt - 1;
                    // フィールド生成
                    this.Init();
                    // 凡例表示ラベル設定
                    //for (long lP = 1; lP <= m_lSeriesNum; lP++)
                    //{
                    //    this.SetLegend(lP, new HySString(sInWk[lP]));
                    //}
                }
                else // 演算結果データ行展開
                {
                    lDataNum = (csCnt <= 0) ? 0 : csCnt - 1;
                    dData = new double[lDataNum];
                    // 実データをデータ配列数分設定
                    for (long lP = 0; lP < lDataNum; lP++)
                    {
                        try
                        {
                            dData[lP] = double.Parse(sInWk[lP + 1]);
                        }
                        catch
                        {
                            HySLog.LogOut(HySLog.DEBUG, "HySCSVFileData::ReadCSV", "catch Exception CreateInstance(Data[" + (lP + 1) + "]=" + sInWk[lP + 1] + ")");
                            return lRtn;
                        }
                    }
                    // データは時系列か？
                    if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
                    {
                        // 時刻データ設定
                        try
                        {
                            //csTime = new HySTime(0);
                            //DateTime csDTBase = DateTime.Parse("1/1/1970 0:0:0");
                            //DateTime csDT = DateTime.Parse(sInWk[0]);
                            //TimeSpan csTS = csDT - csDTBase;
                            //csTime.SetTime((long)csTS.TotalSeconds);
                            csTime = HySCalendar.CreateTime(sInWk[0]);
                            this.AddData(csTime, lDataNum, dData);
                        }
                        catch
                        {
                            HySLog.LogOut(HySLog.DEBUG, "HySCSVFileData::ReadCSV", "catch Exception CreateInstance(Time[" + sInWk[0] + ")");
                            return lRtn;
                        }
                    }
                    else
                    {
                        try
                        {
                            // 単位データ設定
                            dBaseData = double.Parse(sInWk[0]);
                            this.AddData(dBaseData, lDataNum, dData);
                        }
                        catch
                        {
                            HySLog.LogOut(HySLog.DEBUG, "HySCSVFileData::ReadCSV", "catch Exception CreateInstance(BaseData=" + sInWk[0] + ")");
                            return lRtn;
                        }
                    }
                }
                lGyo++; // 行数更新
            }
            lRtn = 0;
            return lRtn; // 正常リターン
        }


        ///// <summary><para>method outline:</para>
        ///// <para>CSV形式ファイルを出力する</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> long lRtn = WriteCSVFile(csFile) </para>
        ///// </example>
        ///// <param name="csFile">出力ファイル</param>
        ///// <returns>long  0:=正常, -1:=異常</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>EXCEL等使用する場合に必要</para>
        ///// </remarks>
        //public virtual long WriteCSVFile()
        //{
        //    long lRtn = -1;
        //    if ((object)m_csFileName != null)
        //    {
        //        HySFile csFile = new HySFile(m_csFileName);
        //        lRtn = this.WriteCSVFile(csFile);
        //    }
        //    return lRtn;
        //}
        ///// <summary><para>method outline:</para>
        ///// <para>CSV形式ファイルを出力する</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> long lRtn = WriteCSVFile(csFile) </para>
        ///// </example>
        ///// <param name="csFile">出力ファイル</param>
        ///// <returns>long  0:=正常, -1:=異常</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>EXCEL等使用する場合に必要</para>
        ///// </remarks>
        //public virtual long WriteCSVFile(HySFile csFile)
        //{
        //    long lRtn = -1;

        //    if (csFile.Open(HySFile.OPEN_MODE.CREATE_NEW, HySFile.READ_WRITE_MODE.WRITE, HySFile.DIRECTORY_MODE.MK_DIR) == 0)
        //    {  // CSVファイルが生成できたら
        //        HySString csDataName = this.GetClassName(); // クラス名称取得
        //        HySString csVerInf = new HySString(this.GetNowVersion());      // バージョン情報取得
        //        HySString sOut = csDataName + new HySString(",") + csVerInf;
        //        csFile.WriteText(sOut);                     // クラス名称+バージョン情報出力
        //        lRtn = this.WriteCSV(csVerInf, csFile);
        //        csFile.Close();
        //    }
        //    return lRtn;
        //}
        ///// <summary><para>method outline:</para>
        ///// <para>CSV形式ファイルを出力する</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> long lRtn = WriteCSVFile(csVerInf, csFile) </para>
        ///// </example>
        ///// <param name="csVerInf">バージョン情報</param>
        ///// <param name="csFile">出力ファイル</param>
        ///// <returns>long  0:=正常, -1:=異常</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>EXCEL等使用する場合に必要</para>
        ///// </remarks>
        //protected virtual long WriteCSV(HySString csVerInf, HySFile csFile)
        //{
        //    // 使用フィールド初期化
        //    long lDataNum = 0;
        //    HySString sOut = null;

        //    // データは時系列か？（データ種別によりタイトルヘッダ出力）
        //    if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
        //    {
        //        csFile.WriteText(new HySString("データ区分,時系列"));
        //        sOut = new HySString("時刻");
        //    }
        //    else
        //    {
        //        csFile.WriteText(new HySString("データ区分,一般（汎用）"));
        //        sOut = new HySString("単位");
        //    }
        //    // 凡例表示ラベル出力
        //    for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
        //    {
        //        sOut += new HySString(",") + m_csLegendLabel[iLp];
        //    }
        //    csFile.WriteText(sOut); // リスト出力
        //    // 演算結果データ出力
        //    lDataNum = this.GetCount();
        //    this.SetCursorFirst(); // カーソルを先頭に移動する
        //    for (long lLp = 0; lLp < lDataNum; lLp++) // データ件数分繰り返す
        //    {
        //        // データは時系列か？
        //        if (m_lDataKind == DATA_KIND.TIME_SERIES_DATA)
        //        {
        //            // 時刻データ設定
        //            sOut = HySCalendar.GetString(m_csTimeList.GetCursorData(), HySCalendar.FORMAT.lSW_YEAR);
        //        }
        //        else
        //        {
        //            // 単位データ設定
        //            sOut = new HySString(m_csBaseDataList.GetCursorData().ToString());
        //        }
        //        // 実データをデータ配列数分設定
        //        for (int iLp = 0; iLp < m_lSeriesNum; iLp++)
        //        {
        //            sOut += new HySString("," + m_csDataList[iLp].GetCursorData().ToString());
        //        }
        //        csFile.WriteText(sOut); // リスト出力
        //        sOut = null;            // 解放
        //        this.MoveCursorNext();  // カーソルを次に移動する
        //    }
        //    return 0; // 正常リターン
        //}

        /// <summary><para>method outline:</para>
        /// <para>クラス名を返す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csClassName = GetClassName() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString  クラス名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual HySString GetClassName()
        {
            return new HySString("HySCSVFileData");
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
    }
}
