﻿// <summary>ソースコード：時系列グラフデータ</summary>
// <author>CommonMP</author>
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>時系列グラフデータクラス</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>本クラスの派生クラスは全て[Serializable]であること</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class HySGraphLineTimeSeries : HySTimeSeriesBase, HySVersionManagement
    {
        /// <summary>時刻情報リスト</summary>
        protected HySLinkedListTime m_csTimeList = new HySLinkedListTime();
        /// <summary>時系列１次元データ</summary>
        new protected HySLinkedListDouble m_csDataList = new HySLinkedListDouble();

        /// <summary>保持データ種別</summary>
        //public HySObjectKind m_csDataKind = null;
        protected HySObjectKind m_csDataKind = null;

        /// <summary>地点識別子 </summary>
        protected HySID m_csID = null;
        /// <summary>地点位置 </summary>
        protected HySGeoPoint m_csPosition = null;


        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> 使用禁止 </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySLineTimeSeries 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>使用禁止</para>
        /// </remarks>
        protected HySGraphLineTimeSeries()
        {
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGraphLineTimeSeries csData = new HySGraphLineTimeSeries(csDataKind); </para>
        /// </example>
        /// <param name="csDataKind">データの種別</param>
        /// <returns> HySGraphLineTimeSeries 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySGraphLineTimeSeries(HySObjectKind csDataKind)
        {
            m_csDataKind = csDataKind.Clone();
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>データの種別設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataMeaning(csDataKind) </para>
        /// </example>
        /// <param name="csDataKind">データの意味</param> 
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDataMeaning(HySObjectKind csDataKind)
        {
            m_csDataKind = csDataKind.Clone();
        }
        /// <summary><para>method outline:</para>
        /// <para>データの種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csDtMeaning = GetDataMeaning() </para>
        /// </example>
        /// <param name="">無し</param> 
        /// <returns>HySObjectKind データの意味</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySObjectKind GetDataMeaning()
        {
            return m_csDataKind;
        }

        /// <summary><para>method outline:</para>
        /// <para>地点識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetID( csID ) </para>
        /// </example>
        /// <param name="csID">地点識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetID(HySID csID)
        {
            m_csID = (HySID)csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>地点識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csIdentifier = GetID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID 地点識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetID()
        {
            return m_csID;
        }

        /// <summary><para>method outline:</para>
        /// <para>地点位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetPosition( csPosition ) </para>
        /// </example>
        /// <param name="csPosition">地点位置座標</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetPosition(HySGeoPoint csPosition)
        {
            m_csPosition = csPosition;
        }
        /// <summary><para>method outline:</para>
        /// <para>地点位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGeoPoint csPosition = GetPosition( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySGeoPoint 地点位置</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySGeoPoint GetPosition()
        {
            return m_csPosition;
        }

        /// <summary><para>method outline:</para>
        /// <para>全てのデータを削除する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Clear() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void Clear()
        {
            m_csLastTime = HySTime.DEFAULT_TIME.Clone();
            m_csDataList.Clear();
            m_csTimeList.Clear();
        }

        /// <summary><para>method outline:</para>
        /// <para>保持されている情報の最も古い時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime dTime = GetFirstTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime 最古時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySTime GetFirstTime()
        {
            return m_csTimeList.GetFirstData();
        }

        /// <summary><para>method outline:</para>
        /// <para>時系列データの数を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDtNum = GetCount() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 時系列データの数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override long GetCount()
        {
            return m_csTimeList.GetCount();
        }

        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻と同一位置に持ってくる
        /// 同一時刻がない場合には、指定時刻に最も近い直後の時刻のレコード位置に持ってくる(SetCursorPlus()と同じ動作)
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDtNum = SetCursor(csTime) </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long  =カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>戻り値=0の場合には，指定時刻よりも後の情報が無い </para>
        /// </remarks>
        public override long SetCursor(HySTime csTime)
        {
            long lRtn = 0;

            if (m_csLastTime.Before(csTime) == false)
            {   // 最終情報は指定時刻よりも後
                long lDtNum = m_csTimeList.GetCount();

                // リストの最後からたどってゆく
                m_csDataList.SetCursorLast();
                m_csTimeList.SetCursorLast();
                for (long lLp = 0; lLp < lDtNum; lLp++)
                {
                    lRtn += 1;
                    if (m_csTimeList.GetCursorData().Before(csTime) == true)
                    {   // 記録時刻が、指定時刻以前
                        m_csDataList.MoveCursorNext();
                        m_csTimeList.MoveCursorNext();
                        lRtn -= 1;
                        break;
                    }
                    else if (m_csTimeList.GetCursorData().Equals(csTime) == true)
                    {
                        break;
                    }
                    m_csDataList.MoveCursorPrev();
                    m_csTimeList.MoveCursorPrev();
                }
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻の直後に持ってくる
        /// （指定時刻に最も近い直後の時刻のレコード位置に持ってくる）
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lDtNum = SetCursorPlus(csTime);/ </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long  =カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>戻り値=0の場合には，指定時刻よりも後の情報が無い </para>
        /// </remarks>
        public override long SetCursorPlus(HySTime csTime)
        {
            long lRtn = 0;
            long lDtNum = m_csDataList.GetCount();

            // リストの最後からたどってゆく
            m_csDataList.SetCursorLast();
            m_csTimeList.SetCursorLast();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                if (m_csTimeList.GetCursorData().After(csTime) == false)
                {   // 指定時刻が、記録位置よりも後でない場合（記録位置が同じか、より前になった場合）
                    m_csDataList.MoveCursorNext();
                    m_csTimeList.MoveCursorNext();
                    break;
                }
                else
                {
                    lRtn += 1;
                    m_csDataList.MoveCursorPrev();
                    m_csTimeList.MoveCursorPrev();
                }
            }

            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻の位置の直前に持ってくる
        /// （指定時刻と同一時刻が存在すればその位置に持ってくる）
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lDtNum = SetCursorMinus(csTime);/ </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long  =カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 戻り値＜０の場合には　指定時刻以前の情報が無い
        /// </para>
        /// </remarks>
        public override long SetCursorMinus(HySTime csTime)
        {
            long lRtn = -1;
            long lDtNum = m_csDataList.GetCount();
            long lDtCnt = -1;

            // リストの最初からたどってゆく
            m_csDataList.SetCursorFirst();
            m_csTimeList.SetCursorFirst();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                if (m_csTimeList.GetCursorData().After(csTime) == true)
                {   // 指定時刻よりも記録位置が後になった場合（
                    m_csDataList.MoveCursorPrev();
                    m_csTimeList.MoveCursorPrev();
                    break;
                }
                else
                {
                    lDtCnt +=1;
                    m_csDataList.MoveCursorNext();
                    m_csTimeList.MoveCursorNext();
                }
            }
            if (lDtCnt >= 0)
            {
                lRtn = lDtNum - lDtCnt;
            }

            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最初に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorFirst() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetCursorFirst()
        {
            m_csTimeList.SetCursorFirst();
            m_csDataList.SetCursorFirst();
        }  
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最後に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorLast() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetCursorLast()
        {
            m_csTimeList.SetCursorLast();
            m_csDataList.SetCursorLast();
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を次の位置に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorNext() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void MoveCursorNext()
        {
            m_csTimeList.MoveCursorNext();
            m_csDataList.MoveCursorNext();
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を一つ前に戻す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorPrev() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void MoveCursorPrev()
        {
            m_csTimeList.MoveCursorPrev();
            m_csDataList.MoveCursorPrev();
        }

        /// <summary><para>method outline:</para>
        /// <para>カーソル位置の時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime =  GetCursorTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime 時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySTime GetCursorTime()
        {
            return m_csTimeList.GetCursorData();
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置のデータを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double dData = GetCursorData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double カーソル位置のデータ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new double GetCursorData()
        {
            return m_csDataList.GetCursorData();
        }

        /// <summary><para>method outline:</para>
        /// <para>最新のデータを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double dData = GetLastData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> double 最新データ </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new double GetLastData()
        {
            this.SetCursorLast();
            return m_csDataList.GetLastData();
        }

        /// <summary><para>method outline:</para>
        /// <para>指定時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = GetLastTime(lOrderNum) </para>
        /// </example>
        /// <param name="lOrderNum">順序：０相対</param>
        /// <returns>HySTime 時刻（データが無い場合にはnull）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override HySTime GetTime(long lOrderNum)
        {
            return m_csTimeList.GetDatat(lOrderNum);
        }
        /// <summary><para>method outline:</para>
        /// <para>最新のデータを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTimeRecordIF csDt = GetData(lNum) </para>
        /// </example>
        /// <param name="lNum">リスト上の位置（０相対）</param>
        /// <returns> double 最新データ </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new double GetData(long lNum)
        {
            return m_csDataList.GetData(lNum);
        }

        /// <summary><para>method outline:</para>
        /// <para>時系列最後のデータを削除する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveLastData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new double RemoveLastData()
        {
            double dRtn = -9.9E-32;
            if (m_csDataList.GetCount() == 0)
            {
                m_csLastTime = HySTime.DEFAULT_TIME;
            }
            else if (m_csDataList.GetCount() == 1)
            {
                dRtn = m_csDataList.RemoveLastData();
                m_csLastTime = HySTime.DEFAULT_TIME;
            }
            else
            {
                dRtn = m_csDataList.RemoveLastData();
                m_csLastTime = this.GetLastTime();
            }
            return dRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>所定時刻以降の情報を破棄</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveAfterData(csTime) </para>
        /// </example>
        /// <param name="csTime">時刻</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で与えた時刻よりも未来のレコードがあればそれらを削除する</para>
        /// </remarks>
        public override void RemoveAfterData(HySTime csTime)
        {
            long lDtNum = this.GetCount();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                if( this.GetLastTime().After(csTime) )
                {   // 指定時刻よりも後の情報がある（最新データは指定時刻よりも未来）
                    this.RemoveLastData(); // 最新時刻を削除する
                }
                else
                {
                    break;
                }
            }
            m_csTimeList.GetLastData();
            HySTime csLastTime = m_csTimeList.GetLastData();
            if (csLastTime != null)
            {
                m_csLastTime = csLastTime.Clone();
            }
            else
            {
                m_csLastTime = HySTime.DEFAULT_TIME.Clone();
            }

        }
        /// <summary><para>method outline:</para>
        /// <para>所定時刻以前の情報を破棄</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveBeforeData(csTime) </para>
        /// </example>
        /// <param name="csTime">時刻</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で与えた時刻よりも過去のレコードがあればそれらを削除する</para>
        /// </remarks>
        public override void RemoveBeforeData(HySTime csTime)
        {
            long lDtNum = this.GetCount();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                this.SetCursorFirst();
                HySTime csFstTime = m_csTimeList.GetCursorData();
                if (csFstTime != null)
                {
                    if (csFstTime.Before(csTime) == true)
                    {
                        m_csDataList.RemoveFirstData();
                        m_csTimeList.RemoveFirstData();
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    break;
                }
            }

        }

        /// <summary><para>method outline:</para>
        /// <para>情報を追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddData(csAddRecord) </para>
        /// </example>
        /// <param name="csAddRecord">追加データ</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で与えた時刻と同一又は未来のレコードがあればそれらを削除して最後に追加する</para>
        /// </remarks>
        public override void AddData(HySTimeRecordIF csAddRecord)
        {
            // Do Nothing
        }

        /// <summary><para>method outline:</para>
        /// <para>情報を追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddData(csTime,dData) </para>
        /// </example>
        /// <param name="csTime">時刻（横軸となる）</param>
        /// <param name="dData">データ値（配列）</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool AddData(HySTime csTime, double dData)
        {
            m_csLastTime = csTime.Clone();
            m_csTimeList.AddLast(m_csLastTime);
            m_csDataList.AddLast(dData);
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>新しいレコードを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTimeRecordIF csNewRecord = CreateNewRecord( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTimeRecordIF 新レコード</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>時系列として保持すべき情報を新規に生成する</para>
        /// </remarks>
        public override HySTimeRecordIF CreateNewRecord()
        { return null; }
            
        /// <summary><para>method outline:</para>
        /// <para>引数で与えられたデータを自分にコピーする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CopyData(csOrgData) </para>
        /// </example>
        /// <param name="csOrgData">コピー元情報</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void CopyData(HySTimeSeriesBase csOrgData)
        {
            HySTime csTime = HySTime.DEFAULT_TIME.Clone();
            double dData = 0;

            if (csOrgData is HySGraphLineTimeSeries)
            {  // 地点識別子が有る場合
                this.SetID(((HySGraphLineTimeSeries)csOrgData).GetID());
                this.SetPosition(((HySGraphLineTimeSeries)csOrgData).GetPosition());
                long lDataNum = csOrgData.GetCount();
                for (long lP = 0; lP < lDataNum; lP++)
                {  // データ件数分繰り返す
                    csTime.SetTime(csOrgData.GetTime(lP).GetTime());
                    dData = ((HySGraphLineTimeSeries)csOrgData).GetData(lP);
                    this.AddData(csTime, dData);
                }
            }
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public new string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public new void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public override void PostDeserialize()
        {
            base.PostDeserialize();

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /*
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
        */
    }
}
