﻿// <summary>ソースコード：河道縦断面グラフデータクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;


namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>河道縦断面グラフデータ</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// <para>remarks:</para>
    /// <para>河道縦断面グラフデータ（HySGraphLineLaneKiloをハッシュテーブル管理したもの）</para>
    /// <para>Remarks</para>
    /// <para>無し</para>
    /// </remarks>
    [Serializable]
    public class HySRiverbedData : HySTimeRecordIF, HySVersionManagement
    {
        /// <summary> 河川を識別する為のＩＤ </summary>
        protected HySID m_csRiverObjID = null;

        /*
        //============　下記３つのデータ対が全て有効であるとは限らない　================
        /// <summary>流量データ用河口からの距離 </summary>
        protected HySLinkedListDouble m_csHydroKiloDList = new HySLinkedListDouble();
        /// <summary>流量データリスト(m3/s) </summary>
        protected HySLinkedListDouble m_csHydroDataList = new HySLinkedListDouble();

        /// <summary>水位データ用河口からの距離 </summary>
        protected HySLinkedListDouble m_csWaterLevelKiloList = new HySLinkedListDouble();
        /// <summary>水位データリスト (m)</summary>
        protected HySLinkedListDouble m_csWaterLevelDataList = new HySLinkedListDouble();
        */

        /// <summary>データ管理テーブル(HySGraphLineLaneKilo)が管理される</summary>
        protected HySDataHashTable m_csDataTable = null;

        /// <summary>河床標高値用河口からの距離 </summary>
        protected HySLinkedListDouble m_csAltitudeKiloList = new HySLinkedListDouble();
        /// <summary>河床標高値リスト (m)</summary>
        protected HySLinkedListDouble m_csAltitudeDataList = new HySLinkedListDouble();

        //============　時刻情報　================
        /// <summary>時刻</summary>
        protected HySTime m_csTime = HySTime.DEFAULT_TIME.Clone();

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySRiverbedData csData = new HySRiverbedData(); </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySRiverbedData 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySRiverbedData()
        {
            m_csDataTable = new HySDataHashTable(); // ハッシュテーブル生成
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>河川を識別する為のＩＤ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetRiverID( csID ) </para>
        /// </example>
        /// <param name="csID">河川を識別する為のＩＤ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetRiverID(HySID csID)
        {
            m_csRiverObjID = (HySID)csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>河川を識別する為のＩＤ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csIdentifier = GetRiverID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID 河川を識別する為のＩＤ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetRiverID()
        {
            return m_csRiverObjID;
        }

        /// <summary><para>method outline:</para>
        /// <para>河道縦断グラフ作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGraphLineLaneKilo csGraph = CreateGraph(csGraphID,csDataKind) </para>
        /// </example>
        /// <param name="csGraphID">グラフを識別する識別子</param>
        /// <param name="csDataKind">データ種別</param>
        /// <returns>HySGraphLineLaneKilo 生成された河道縦断グラフ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySGraphLineLaneKilo CreateGraph(HySID csGraphID, HySObjectKind csDataKind)
        {
            HySGraphLineLaneKilo csGraphLineLaneKilo = new HySGraphLineLaneKilo(csDataKind); // インスタンス生成
            HySString csKeyID = new HySString(csGraphID.GetString().ToString());
            m_csDataTable.AddObject(csKeyID, csGraphLineLaneKilo);
            return csGraphLineLaneKilo;
        }
        /// <summary><para>method outline:</para>
        /// <para>データのクリア</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveGraph(csGraphID) </para>
        /// </example>
        /// <param name="csGraphID">グラフを識別する識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void RemoveGraph(HySID csGraphID)
        {
            HySString csKeyID = new HySString(csGraphID.GetString().ToString());
            m_csDataTable.DeleteData(csKeyID);
        }
        /// <summary><para>method outline:</para>
        /// <para>河道縦断グラフ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGraphLineLaneKilo csGraph = GetGraph(csGraphID) </para>
        /// </example>
        /// <param name="csGraphID">グラフを識別する識別子</param>
        /// <returns>HySGraphLineLaneKilo 河道縦断グラフ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySGraphLineLaneKilo GetGraph(HySID csGraphID)
        {
            HySString csKeyID = new HySString(csGraphID.GetString().ToString());
            return (HySGraphLineLaneKilo)m_csDataTable.GetObject(csKeyID);
        }
        /// <summary><para>method outline:</para>
        /// <para>河道縦断グラフ数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lGrphNum = GetGraphNum( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 河道縦断グラフ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetGraphNum()
        {
            HySDataLinkedList csList = m_csDataTable.GetKeyList(); // キーリスト一覧取得
            long lGraphNum = csList.GetCount(); // 一覧表件数取得
            csList = null;
            return lGraphNum;
        }
        /// <summary><para>method outline:</para>
        /// <para>グラフ識別子配列取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID[] csGraphIDList = GetGraphIDList( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID[] グラフＩＤ配列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID[] GetGraphIDList()
        {
            HySID[] csID = null;

            HySDataLinkedList csList = m_csDataTable.GetKeyList(); // キーリスト一覧取得
            long lGraphNum = csList.GetCount(); // 一覧表件数取得
            if (lGraphNum > 0)
            {  // 登録データがある場合
                csID = new HySID[lGraphNum];
                for (int iLp = 0; iLp < lGraphNum; iLp++)
                {  // 登録データ件数分繰り返す
                    HySString csGraphID = csList.GetData((long)iLp) as HySString;
                    if (csGraphID is HySString)
                    {  // データが有る場合
                        csID[iLp] = new HySID(new HySString(csGraphID.ToString())); // 時系列グラフＩＤ設定
                    }
                }
                csList = null;
            }
            return csID;
        }

        /// <summary><para>method outline:</para>
        /// <para>河床標高値設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetAltitudeData(dKilo,dAlt) </para>
        /// </example>
        /// <param name="dKilo">河口からの距離</param>
        /// <param name="dAlt">河床標高値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetAltitudeData(double dKilo, double dAlt)
        {
            double dDt1 = 0.0;
            double dDt2 = 0.0;
            HySLinkedListDouble dKiloBf = new HySLinkedListDouble();
            HySLinkedListDouble dAltBf = new HySLinkedListDouble();

            // 河口からの距離の設定
            dDt1 = m_csAltitudeKiloList.GetLastData();
            if (dDt1 < dKilo)
            {  // 最終データが前データの場合
                m_csAltitudeKiloList.AddLast(dKilo); // 河口からの距離追加
                m_csAltitudeDataList.AddLast(dAlt);  // 河床標高値追加
            }
            else
            {  // 最終データが後データの場合
                long lNum1 = m_csAltitudeKiloList.GetCount();
                for (long lP1 = 0; lP1 < lNum1; lP1++)
                {  // データ件数分繰り返す
                    dDt1 = m_csAltitudeKiloList.GetLastData();
                    dDt2 = m_csAltitudeDataList.GetLastData();
                    if (dDt1 == dKilo)
                    {  // データが一致の場合
                        m_csAltitudeKiloList.RemoveLastData(); // 一致データ削除
                        m_csAltitudeDataList.RemoveLastData(); // 一致データ削除
                        m_csAltitudeKiloList.AddLast(dKilo);   // 河口からの距離追加
                        m_csAltitudeDataList.AddLast(dAlt);    // 河床標高値追加
                        break;
                    }
                    else
                    {  // データが不一致の場合
                        if (dDt1 < dKilo)
                        {  // 最終データが前データの場合
                            m_csAltitudeKiloList.AddLast(dKilo);   // 河口からの距離追加
                            m_csAltitudeDataList.AddLast(dAlt);    // 河床標高値追加
                            long lNum2 = dKiloBf.GetCount();       // 退避件数取得
                            dKiloBf.SetCursorLast();
                            dAltBf.SetCursorLast();
                            for (long lP2 = 0; lP2 < lNum2; lP2++)
                            {  // 退避件数分繰り返す
                                dDt1 = m_csAltitudeKiloList.GetCursorData();
                                dDt2 = m_csAltitudeDataList.GetCursorData();
                                m_csAltitudeKiloList.AddLast(dDt1); // 河口からの距離戻し
                                m_csAltitudeDataList.AddLast(dDt2); // 河床標高値戻し
                                m_csAltitudeKiloList.MoveCursorPrev(); // 前へ
                                m_csAltitudeDataList.MoveCursorPrev(); // 前へ
                            }
                            break;
                        }
                        else
                        {  // 最終データが後データの場合
                            dKiloBf.AddLast(dDt1); // 河口からの距離退避
                            dAltBf.AddLast(dDt2);  // 河床標高値退避
                            m_csAltitudeKiloList.RemoveLastData(); // 一致データ削除
                            m_csAltitudeDataList.RemoveLastData(); // 一致データ削除
                        }
                    }
                }
                dKiloBf = null;
                dAltBf = null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>河床標高データ数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lNum = GetAltitudeDataNum( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 河床標高データ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetAltitudeDataNum()
        {
            return m_csAltitudeDataList.GetCount();
        }
        /// <summary><para>method outline:</para>
        /// <para>河床標高値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetAltitudeData(lOrderNo, ref dKilo, ref dAlt) </para>
        /// </example>
        /// <param name="lOrderNo">データ並び位置</param>
        /// <param name="dKilo">河口からの距離</param>
        /// <param name="dAlt">河床標高値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void GetAltitudeData(long lOrderNo, ref double dKilo, ref double dAlt)
        {
            long lP;
            long lDtNum;

            // 初期化
            lDtNum = m_csAltitudeKiloList.GetCount();
            if (lOrderNo <= 0 || lDtNum < lOrderNo)
            {  // データ並び位置が範囲外の場合
                dKilo = -1.0E32;
                dAlt = -1.0E32;
            }
            else
            {  // データ並び位置が範囲内の場合
                m_csAltitudeKiloList.SetCursorFirst();
                for (lP = 0; lP < lOrderNo; lP++)
                {  // データ件数分繰り返す
                    dKilo = m_csAltitudeKiloList.GetCursorData(); // データ取得
                    m_csAltitudeKiloList.MoveCursorNext(); // 次へ
                }
                lDtNum = m_csAltitudeDataList.GetCount();
                m_csAltitudeDataList.SetCursorFirst();
                for (lP = 0; lP < lOrderNo; lP++)
                {  // データ件数分繰り返す
                    dAlt = m_csAltitudeDataList.GetCursorData(); // データ取得
                    m_csAltitudeDataList.MoveCursorNext(); // 次へ
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>河口からの距離データ配列取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double[] dData = GetAltKiloDataDim( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] 河口からの距離データ配列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>参照のみ</para>
        /// </remarks>
        public virtual double[] GetAltKiloDataDim()
        {
            long lDtNum = m_csAltitudeKiloList.GetCount();
            double[] dAltKiloDataDim = new double[lDtNum];
            m_csAltitudeKiloList.SetCursorFirst(); // カーソルを先頭に設定
            for (long lP = 0; lP < lDtNum; lP++)
            {  // データ件数分繰り返す
                dAltKiloDataDim[(int)lP] = m_csAltitudeKiloList.GetCursorData(); // データ取得
                m_csAltitudeKiloList.MoveCursorNext(); // 次へ
            }
            return dAltKiloDataDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>標高データ配列取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double[] dData = GetAltitudeDataDim( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] データ配列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>参照のみ</para>
        /// </remarks>
        public virtual double[] GetAltitudeDataDim()
        {
            long lDtNum = m_csAltitudeDataList.GetCount();
            double[] dAltitudeDataDim = new double[lDtNum];
            m_csAltitudeDataList.SetCursorFirst(); // カーソルを先頭に設定
            for (long lP = 0; lP < lDtNum; lP++)
            {  // データ件数分繰り返す
                dAltitudeDataDim[(int)lP] = m_csAltitudeDataList.GetCursorData(); // データ取得
                m_csAltitudeDataList.MoveCursorNext(); // 次へ
            }
            return dAltitudeDataDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>河床標高値データのクリア</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ClearAltitudeData( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ClearAltitudeData()
        {
            m_csAltitudeKiloList.Clear();
            m_csAltitudeDataList.Clear();
        }

        /*
/// <summary><para>method outline:</para>
/// <para>流量設定</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> SetHydroData(dKilo,dQ) </para>
/// </example>
/// <param name="dKilo">河口からの距離</param>
/// <param name="dQ">流量</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void SetHydroData(double dKilo, double dQ)
{
    // 同じ地点に情報がセットされていなければ　追加する。
    // 同じ地点の情報が既にセットされていれば、後で付け加えた情報で上書きさせる

    //AddHydroData() でないことに注意
}
/// <summary><para>method outline:</para>
/// <para>流量取得</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> GetHydroData( ref dKilo, ref dQ) </para>
/// </example>
/// <param name="dKilo">河口からの距離</param>
/// <param name="dQ">流量</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void GetHydroData( ref double dKilo, ref double dQ)
{

}
/// <summary><para>method outline:</para>
/// <para>水位設定</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> SetWaterLevelData(dKilo,dH) </para>
/// </example>
/// <param name="dKilo">河口からの距離</param>
/// <param name="dH">水位</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void SetWaterLevelData(double dKilo, double dH)
{
    // 同じ地点に情報がセットされていなければ　追加する。
    // 同じ地点の情報が既にセットされていれば、後で付け加えた情報で上書きさせる

    //AddWaterLevelData() でないことに注意
}
/// <summary><para>method outline:</para>
/// <para>水位取得</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> GetWaterLevelData(ref dKilo, ref dH) </para>
/// </example>
/// <param name="dKilo">河口からの距離</param>
/// <param name="dH">水位</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void GetWaterLevelData( ref double dKilo, ref double dH)
{

}
/// <summary><para>method outline:</para>
/// <para>河床標高値設定</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> SetAltitudeData(dKilo,dAlt) </para>
/// </example>
/// <param name="dKilo">河口からの距離</param>
/// <param name="dAlt">河床標高値</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void SetAltitudeData(double dKilo, double dAlt)
{
    // 同じ地点に情報がセットされていなければ　追加する。
    // 同じ地点の情報が既にセットされていれば、後で付け加えた情報で上書きさせる

    //AddAltitudeData() でないことに注意
}
/// <summary><para>method outline:</para>
/// <para>河床標高値取得</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> GetAltitudeData(ref dKilo, ref dAlt) </para>
/// </example>
/// <param name="dKilo">河口からの距離</param>
/// <param name="dAlt">河床標高値</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void GetAltitudeData( ref double dKilo, ref double dAlt)
{

}
/// <summary><para>method outline:</para>
/// <para>流量データのクリア</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> ClearHydroData( ) </para>
/// </example>
/// <param name="">無し</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void ClearHydroData()
{

}
/// <summary><para>method outline:</para>
/// <para>水位データのクリア</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> ClearWaterLevelData( ) </para>
/// </example>
/// <param name="">無し</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void ClearWaterLevelData()
{

}
/// <summary><para>method outline:</para>
/// <para>河床標高値データのクリア</para>
/// </summary>
/// <example><para>usage:</para>
/// <para> ClearAltitudeData( ) </para>
/// </example>
/// <param name="">無し</param>
/// <returns>無し</returns>
/// <exception cref="">無し</exception>
/// <remarks><para>remarks:</para>
/// <para>無し</para>
/// </remarks>
public void ClearAltitudeData()
{

}
*/
        /// <summary><para>method outline:</para>
        /// <para>時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTime(csTime) </para>
        /// </example>
        /// <param name="csTime">時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetTime(HySTime csTime)
        {
            m_csTime.SetTime(csTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>時刻取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = GetTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime 時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTime GetLastTime()
        {
            return m_csTime;
        }
        /// <summary><para>method outline:</para>
        /// <para>日付時刻比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = EqualTime(csTime) </para>
        /// </example>
        /// <param name="csTime">比較用日付管理クラスオブジェクト(HySTime)</param>
        /// <returns>bool  =HySTime の現在時刻が 引数：Date と同じ場合は true、そうでない場合は false
        /// </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool EqualTime(HySTime csTime)
        {
            bool bRtn = false;
            if (m_csTime != null)
            {
                bRtn = m_csTime.Equals(csTime);
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>指定時刻よりも後か</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = After(csTime) </para>
        /// </example>
        /// <param name="csTime">
        /// 比較用日付管理クラスオブジェクト(HySTime)
        /// </param>
        /// <returns>bool  =HySTime の現在時刻が 引数：Date の時刻よりあとの場合は true、そうでない場合は false </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool After(HySTime csTime)
        {
            bool bRtn = false;
            if (m_csTime != null)
            {
                bRtn = m_csTime.After(csTime);
            }
            return bRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>指定時刻よりも前か</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Before(csTime) </para>
        /// </example>
        /// <param name="csTime">比較用日付管理クラスオブジェクト(HySTime)</param>
        /// <returns>
        /// bool　判定結果
        /// </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>bool  =HySTime の現在時刻が 引数：Date の時刻より前の場合は true、そうでない場合は false</para>
        /// </remarks>
        public virtual bool Before(HySTime csTime)
        {
            bool bRtn = false;
            if (m_csTime != null)
            {
                bRtn = m_csTime.Before(csTime);
            }
            return bRtn;
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
    }
}
