﻿// <summary>ソースコード：河道縦断面グラフ情報　時系列 データクラス<</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>河道縦断面グラフ情報時系列 データクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// <para>remarks:</para>
    /// <para>河道縦断面グラフデータ（HySGraphLineLaneKiloを時系列管理したもの）</para>
    /// <para>Remarks</para>
    /// <para>無し</para>
    /// </remarks>
    [Serializable]
    public class HySRiverbedSerialData : HySTimeSeriesBase, HySVersionManagement
    {
        /// <summary> 河川を識別する為のＩＤ </summary>
        protected HySID m_csRiverObjID = null;
        /// <summary>保持データ種別</summary>
        protected HySObjectKind m_csDataKind = null;

        /// <summary>河床標高値用河口からの距離 </summary>
        protected HySLinkedListDouble m_csAltitudeKiloList = new HySLinkedListDouble();
        /// <summary>河床標高値リスト (m)</summary>
        protected HySLinkedListDouble m_csAltitudeDataList = new HySLinkedListDouble();
 
        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> 使用禁止 </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySRiverbedSerialData 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>使用禁止</para>
        /// </remarks>
        protected HySRiverbedSerialData()
        {
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySRiverbedSerialData csData = new HySRiverbedSerialData(csDataKind); </para>
        /// </example>
        /// <param name="csDataKind">データの種別</param>
        /// <returns> HySRiverbedSerialData 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySRiverbedSerialData(HySObjectKind csDataKind)
        {
            m_csDataKind = csDataKind.Clone();
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>河道データ列作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGraphLineLaneKilo csData = CreateTimeSerialKilo(csDataKind,csTime) </para>
        /// </example>
        /// <param name="csDataKind">データ種別</param>
        /// <param name="csTime">時刻</param>
        /// <returns>HySGraphLineLaneKilo 生成された河道データ列インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySGraphLineLaneKilo CreateTimeSerialKilo(HySObjectKind csDataKind, HySTime csTime)
        {
            HySGraphLineLaneKilo csGraphLineLaneKilo = new HySGraphLineLaneKilo(csDataKind); // インスタンス生成
            if (csTime is HySTime)
            {  // 時刻が有効な場合は時刻を設定する
                csGraphLineLaneKilo.SetTime(csTime.Clone());
                this.SetLastTime(csTime.Clone());
            }
            m_csDataList.AddLast(csGraphLineLaneKilo);   // 河道データ列インスタンス記憶
            return csGraphLineLaneKilo;
        }
        /// <summary><para>method outline:</para>
        /// <para>河道データ列取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGraphLineLaneKilo csData = GetTimeSerialKilo(csTime) </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>HySGraphLineLaneKilo 河道データ列インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定時刻の河道データ列が無い場合はNULLが返る</para>
        /// </remarks>
        public virtual HySGraphLineLaneKilo GetTimeSerialKilo(HySTime csTime)
        {
            this.SetCursorMinus(csTime); // 指定時刻位置にカーソル移動
            HySGraphLineLaneKilo csGraphLineLaneKilo = (HySGraphLineLaneKilo)m_csDataList.GetCursorData();
            if (csTime.Equals(csGraphLineLaneKilo.GetLastTime()) == true)
            {  // 時刻が一致した場合
                return csGraphLineLaneKilo;
            }
            else
            {  // 時刻が不一致の場合
                return null;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>河川を識別する為のＩＤ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetRiverID( csID ) </para>
        /// </example>
        /// <param name="csID">河川を識別する為のＩＤ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetRiverID(HySID csID)
        {
            m_csRiverObjID = (HySID)csID;
        }
        /// <summary><para>method outline:</para>
        /// <para>河川を識別する為のＩＤ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csIdentifier = GetRiverID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID 河川を識別する為のＩＤ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetRiverID()
        {
            return m_csRiverObjID;
        }
        /// <summary><para>method outline:</para>
        /// <para>データの種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csDtMeaning = GetDataMeaning() </para>
        /// </example>
        /// <param name="">無し</param> 
        /// <returns>HySObjectKind データの意味</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySObjectKind GetDataMeaning()
        {
            return m_csDataKind;
        }

        /// <summary><para>method outline:</para>
        /// <para>河床標高値設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetAltitudeData(dKilo,dAlt) </para>
        /// </example>
        /// <param name="dKilo">河口からの距離</param>
        /// <param name="dAlt">河床標高値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetAltitudeData(double dKilo, double dAlt)
        {
            double dDt1 = 0.0;
            double dDt2 = 0.0;
            HySLinkedListDouble dKiloBf = new HySLinkedListDouble();
            HySLinkedListDouble dAltBf = new HySLinkedListDouble();

            // 河口からの距離の設定
            dDt1 = m_csAltitudeKiloList.GetLastData();
            if (dDt1 < dKilo)
            {  // 最終データが前データの場合
                m_csAltitudeKiloList.AddLast(dKilo); // 河口からの距離追加
                m_csAltitudeDataList.AddLast(dAlt);  // 河床標高値追加
            }
            else
            {  // 最終データが後データの場合
                long lNum1 = m_csAltitudeKiloList.GetCount();
                for (long lP1 = 0; lP1 < lNum1; lP1++)
                {  // データ件数分繰り返す
                    dDt1 = m_csAltitudeKiloList.GetLastData();
                    dDt2 = m_csAltitudeDataList.GetLastData();
                    if (dDt1 == dKilo)
                    {  // データが一致の場合
                        m_csAltitudeKiloList.RemoveLastData(); // 一致データ削除
                        m_csAltitudeDataList.RemoveLastData(); // 一致データ削除
                        m_csAltitudeKiloList.AddLast(dKilo);   // 河口からの距離追加
                        m_csAltitudeDataList.AddLast(dAlt);    // 河床標高値追加
                        break;
                    }
                    else
                    {  // データが不一致の場合
                        if (dDt1 < dKilo)
                        {  // 最終データが前データの場合
                            m_csAltitudeKiloList.AddLast(dKilo);   // 河口からの距離追加
                            m_csAltitudeDataList.AddLast(dAlt);    // 河床標高値追加
                            long lNum2 = dKiloBf.GetCount();       // 退避件数取得
                            dKiloBf.SetCursorLast();
                            dAltBf.SetCursorLast();
                            for (long lP2 = 0; lP2 < lNum2; lP2++)
                            {  // 退避件数分繰り返す
                                dDt1 = m_csAltitudeKiloList.GetCursorData();
                                dDt2 = m_csAltitudeDataList.GetCursorData();
                                m_csAltitudeKiloList.AddLast(dDt1); // 河口からの距離戻し
                                m_csAltitudeDataList.AddLast(dDt2); // 河床標高値戻し
                                m_csAltitudeKiloList.MoveCursorPrev(); // 前へ
                                m_csAltitudeDataList.MoveCursorPrev(); // 前へ
                            }
                            break;
                        }
                        else
                        {  // 最終データが後データの場合
                            dKiloBf.AddLast(dDt1); // 河口からの距離退避
                            dAltBf.AddLast(dDt2);  // 河床標高値退避
                            m_csAltitudeKiloList.RemoveLastData(); // 一致データ削除
                            m_csAltitudeDataList.RemoveLastData(); // 一致データ削除
                        }
                    }
                }
            }
            dKiloBf = null;
            dAltBf = null;
        }
        /// <summary><para>method outline:</para>
        /// <para>河床標高データ数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lNum = GetAltitudeDataNum( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 河床標高データ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetAltitudeDataNum()
        {
            return m_csAltitudeDataList.GetCount();
        }
        /// <summary><para>method outline:</para>
        /// <para>河床標高値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetAltitudeData(lOrderNo, ref dKilo, ref dAlt) </para>
        /// </example>
        /// <param name="lOrderNo">データ並び位置</param>
        /// <param name="dKilo">河口からの距離</param>
        /// <param name="dAlt">河床標高値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void GetAltitudeData(long lOrderNo, ref double dKilo, ref double dAlt)
        {
            long lP;
            long lDtNum;

            // 初期化
            lDtNum = m_csAltitudeKiloList.GetCount();
            if (lOrderNo <= 0 || lDtNum < lOrderNo)
            {  // データ並び位置が範囲外の場合
                dKilo = -1.0E32;
                dAlt = -1.0E32;
            }
            else
            {  // データ並び位置が範囲内の場合
                m_csAltitudeKiloList.SetCursorFirst();
                for (lP = 0; lP < lOrderNo; lP++)
                {  // データ件数分繰り返す
                    dKilo = m_csAltitudeKiloList.GetCursorData(); // データ取得
                    m_csAltitudeKiloList.MoveCursorNext(); // 次へ
                }
                lDtNum = m_csAltitudeDataList.GetCount();
                m_csAltitudeDataList.SetCursorFirst();
                for (lP = 0; lP < lOrderNo; lP++)
                {  // データ件数分繰り返す
                    dAlt = m_csAltitudeDataList.GetCursorData(); // データ取得
                    m_csAltitudeDataList.MoveCursorNext(); // 次へ
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>河口からの距離データ配列取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double[] dData = GetKiloDataDim( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] 河口からの距離データ配列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>参照のみ</para>
        /// </remarks>
        public virtual double[] GetAltKiloDataDim()
        {
            long lDtNum = m_csAltitudeKiloList.GetCount();
            double[] dAltKiloDataDim = new double[lDtNum];
            m_csAltitudeKiloList.SetCursorFirst(); // カーソルを先頭に設定
            for (long lP = 0; lP < lDtNum; lP++)
            {  // データ件数分繰り返す
                dAltKiloDataDim[(int)lP] = m_csAltitudeKiloList.GetCursorData(); // データ取得
                m_csAltitudeKiloList.MoveCursorNext(); // 次へ
            }
            return dAltKiloDataDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>標高データ配列取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double[] dData = GetDataDim( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>double[] データ配列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>参照のみ</para>
        /// </remarks>
        public virtual double[] GetAltitudeDataDim()
        {
            long lDtNum = m_csAltitudeDataList.GetCount();
            double[] dAltitudeDataDim = new double[lDtNum];
            m_csAltitudeDataList.SetCursorFirst(); // カーソルを先頭に設定
            for (long lP = 0; lP < lDtNum; lP++)
            {  // データ件数分繰り返す
                dAltitudeDataDim[(int)lP] = m_csAltitudeDataList.GetCursorData(); // データ取得
                m_csAltitudeDataList.MoveCursorNext(); // 次へ
            }
            return dAltitudeDataDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>河床標高値データのクリア</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ClearAltitudeData( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ClearAltitudeData()
        {
            m_csAltitudeKiloList.Clear();
            m_csAltitudeDataList.Clear();
        }

        /// <summary><para>method outline:</para>
        /// <para>新しいレコードを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTimeRecordIF csNewRecord = CreateNewRecord( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTimeRecordIF 新レコード</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>時系列として保持すべき情報を新規に生成する</para>
        /// </remarks>
        public override HySTimeRecordIF CreateNewRecord()
        {
            return null;
        }
        /// <summary><para>method outline:</para>
        /// <para>引数で与えられたデータを自分にコピーする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CopyData(csOrgTimeSeriesData) </para>
        /// </example>
        /// <param name="csOrgTimeSeriesData">コピー元情報</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void CopyData(HySTimeSeriesBase csOrgTimeSeriesData)
        {
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public new string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public new void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public override void PostDeserialize()
        {
            base.PostDeserialize();

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /*
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
        */
    }
}
