// <summary>ソースコード：セル内データをインデックス情報（０相対）から取得する</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;


namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>セル内データをインデックス情報（０相対）にて取得する</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>本クラスの派生クラスは全て[Serializable]であること</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/08/01][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class HySCellDataGetter : HySDataRoot, HySVersionManagement
    {
        /// <summary>インデックス数 </summary>
        protected long m_lIdxNum = 0;
        /// <summary>セル内の変数を示す（セル接続）インデックス値 </summary>
        protected long[] m_lIdx = null;

        /// <summary> 不定値データ </summary>
        protected HySCellData m_csUndefefValCell = null;

                
        /// <summary><para>method outline:</para>
        /// <para>デストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>無し</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メモリーリーク防止の為　内部変数は必ず null にセットする</para>
        /// </remarks>
        ~HySCellDataGetter()
        {
            m_lIdx = null;
            m_csUndefefValCell = null;
            m_sDecodedVersion = null;
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> 使用禁止 </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySDataCharacteristicInCell 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>使用禁止</para>
        /// </remarks>
        protected HySCellDataGetter()
        {
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellDataGetter csData = new HySCellDataGetter(lDim); </para>
        /// </example>
        /// <param name="lDim">データ配列数</param>
        /// <returns> HySCellDataGetter 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySCellDataGetter(long lDim)
        {
            m_lIdxNum = lDim;
            m_lIdx = new long[m_lIdxNum];
            for (long lLp = 0; lLp < m_lIdxNum; lLp++)
            {
                m_lIdx[lLp] = lLp; // NOT_SETTING;
                //  デフォルト値としては、入れ替えなしにそのまま取得できるものとする
            }
            m_csUndefefValCell = new HySCellData(m_lIdxNum + 1);
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>コピーコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellDataGetter csData = new HySCellDataGetter(csOrgData); </para>
        /// </example>
        /// <param name="csOrgData">オリジナルインスタンス</param>
        /// <returns> HySCellDataGetter 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySCellDataGetter(HySCellDataGetter csOrgData)
        {
            m_lIdxNum = csOrgData.m_lIdxNum;
            m_lIdx = new long[m_lIdxNum];
            for (long lLp = 0; lLp < m_lIdxNum; lLp++)
            {
                m_lIdx[lLp] = csOrgData.m_lIdx[lLp];
            }
            m_csUndefefValCell = csOrgData.m_csUndefefValCell.Clone();
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>セル内の変数を示すインデックス値設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetIdx(lIdx,lVal); </para>
        /// </example>
        /// <param name="lIdx">データ位置</param>
        /// <param name="lVal">インデックス値</param>
        /// <returns> なし</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetIdx(long lIdx, long lVal)
        {
            if (lIdx >= 0 && lIdx < m_lIdxNum)
            {
                m_lIdx[lIdx] = lVal;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>セル内の変数を示すインデックス値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lIdxVal = GetIdx(lIdx); </para>
        /// </example>
        /// <param name="lIdx">データ位置</param>
        /// <returns> long インデックス値</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetIdx(long lIdx)
        {
            if (lIdx >= 0 && lIdx < m_lIdxNum)
            {
                return m_lIdx[lIdx];
            }
            else
            {
                return NOT_SETTING;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>セル内の変数を示すインデックス値取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long[] lIdxVal = GetIdx(); </para>
        /// </example>
        /// <param name="">なし</param>
        /// <returns> long[] インデックス値配列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long[] GetIdx()
        {
            return m_lIdx;
        }
        /// <summary><para>method outline:</para>
        /// <para>インデックス数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lIdxNum = GetIdxNum(); </para>
        /// </example>
        /// <param name="">なし</param>
        /// <returns> long インデックス数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetIdxNum()
        {
            return m_lIdxNum;
        }

        /// <summary><para>method outline:</para>
        /// <para>不定値データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetIndefiniteValue(lDataIdx, dVal); </para>
        /// </example>
        /// <param name="lDataIdx">データ配列位置</param>
        /// <param name="dVal">データ値</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetIndefiniteValue(long lDataIdx, double dVal)
        {
            if (lDataIdx >= 0 && lDataIdx < m_lIdxNum)
            {
                m_csUndefefValCell.m_dData[lDataIdx + 1] = dVal;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>不定値データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetIndefiniteValue(); </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 不定値データインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySCellData GetIndefiniteValue()
        {
            return m_csUndefefValCell;
        }

        /// <summary><para>method outline:</para>
        /// <para>セル内データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> double Data(csCell, lDataIdx); </para>
        /// </example>
        /// <param name="csCell">対象セル</param>
        /// <param name="lDataIdx">データインデックス</param>
        /// <returns> double データ値</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// シミュレーション演算中に使用する
        /// 速度アップの為内部でのデータ整合性チェック等は行っていない。
        /// 使用者側の責任において使用すること。
        /// </para>
        /// </remarks>
        public virtual double Data(HySCellData csCell, long lDataIdx)
        {
            if (csCell == null)
            {
                return 0.0;
            }
            if (m_lIdx[lDataIdx] < 0)
            {
                return m_csUndefefValCell.m_dData[-(m_lIdx[lDataIdx])];
            }
            else
            {
                return csCell.m_dData[m_lIdx[lDataIdx]];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>セルと識別子の対応管理情報をXMLへ書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySXmlNode csXmlNode = XmlOUT(csXmlWriter) </para>
        /// </example>
        /// <param name="csXmlWriter">XML書き出しクラス</param>
        /// <returns>HySXmlNode  XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySXmlNode XmlOUT(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtn = HySXmlNode.CreateElement(HySDefine.xml_CELL_DATA_GETTER, csXmlWriter);

            HySXmlNode csLclNode;
            for (int iLp = 0; iLp < m_lIdxNum; iLp++)
            {
                csLclNode = HySXmlNode.CreateElement(HySDefine.xml_INDEX, csXmlWriter);

                csLclNode.SetAttribute(HySDefine.xml_INDEX_NO, iLp.ToString());
                csLclNode.SetAttribute(HySDefine.xml_INDEX_VALUE, m_lIdx[iLp].ToString());

                csRtn.AppendChild(csLclNode);   // AppendChildは最後に行う
            }
            if (m_csUndefefValCell != null)
            {
                csLclNode = m_csUndefefValCell.XmlOUT(ref csXmlWriter);
                csRtn.AppendChild(csLclNode);   // AppendChildは最後に行う
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>XMLからセルと識別子の対応管理情報を読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = XmlIN(csPrptyXml) </para>
        /// </example>
        /// <param name="csPrptyXml">入力XML</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>HySDefine.xml_CELL_INDEXタグの選択は外部で行っておく</para>
        /// </remarks>
        public virtual bool XmlIN(HySXmlNode csPrptyXml)
        {
            bool bRtn = true;
            if (csPrptyXml == null)
            { return false; }

            m_lIdx = null;

            HySXmlNode csXml = null;
            HySXmlNodeList csXmlVariableList = csPrptyXml.GetChildNodeListByTagName(HySDefine.xml_INDEX);
            long lValCnt = csXmlVariableList.GetCount();

            m_lIdxNum = lValCnt;
            m_lIdx = new long[m_lIdxNum];

            string sOrderAttr; long lOrder = 0;
            string sValueAttr; long lVal = 0;

            for (long lLoop = 0; lLoop < m_lIdxNum; lLoop++)
            {
                csXml = csXmlVariableList.GetNode(lLoop);
                // 順序
                sOrderAttr = csXml.GetAttribute(HySDefine.xml_INDEX_NO);
                lOrder = Convert.ToInt32(sOrderAttr);
                // セル接続インデックス値
                sValueAttr = csXml.GetAttribute(HySDefine.xml_INDEX_VALUE);
                lVal = Convert.ToInt32(sValueAttr);

                m_lIdx[lOrder] = lVal;
            }

            csXmlVariableList = csPrptyXml.GetChildNodeListByTagName(HySDefine.xml_CELL_DATA);
            lValCnt = csXmlVariableList.GetCount();
            if (lValCnt > 0)
            {
                m_csUndefefValCell = new HySCellData(1);
                csXml = csXmlVariableList.GetNode(0);
                bRtn = m_csUndefefValCell.XmlIN(csXml);
            }

            return bRtn;
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }

        /// <summary>未設定</summary>
        public static readonly long NOT_SETTING = -900000;
    }
}
