﻿// <summary>ソースコード：１セル内の配列データの種別</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;


namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>１セル内の各データの種別を示す</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>本クラスの派生クラスは全て[Serializable]であること</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    [Serializable]
    public partial class HySDataCharacteristicInCell : HySDataRoot, HySVersionManagement
    {
        /// <summary>配列数</summary>
        public long m_lDim = 0;
        /// <summary>保持データ</summary>
        public HySObjectKind[] m_csDataKind = null;
        /// <summary>単位（表示用）</summary>
        public String[] m_csUnit = null;
        /// <summary>ラベル（表示用）</summary>
        public String[] m_csLabel = null;


        // ---------------------------------------------------------------
        // ↓↓　情報は少ないと思われるので　態々ハッシュテーブル化しない  ↓↓
        //// <summary> データ種別格納テーブル </summary>
        //protected HySDataHashTable m_csKeyTbl = new HySDataHashTable();
        //// <summary>配列位置テーブル</summary>
        //protected Hashtable m_csItemTbl = new Hashtable();
        // ↑↑　情報は少ないと思われるので　態々ハッシュテーブル化しない  ↑↑
        // ---------------------------------------------------------------
         

        /// <summary><para>method outline:</para>
        /// <para>デストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>無し</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メモリーリーク防止の為　内部変数は必ず null にセットする</para>
        /// </remarks>
        ~HySDataCharacteristicInCell()
        {
            try
            {
                for (long lLp = 0; lLp < m_lDim; lLp++)
                {
                    m_csDataKind[lLp] = null;
                    m_csUnit[lLp] = null;
                    m_csLabel[lLp] = null;
                }
                m_lDim = 0;
                m_csDataKind = null;
                m_csUnit = null;
                m_csLabel = null;
            }
            catch
            {
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> 使用禁止 </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySDataCharacteristicInCell 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>使用禁止</para>
        /// </remarks>
        protected HySDataCharacteristicInCell()
        {
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataCharacteristicInCell csData = new HySDataCharacteristicInCell(lDim); </para>
        /// </example>
        /// <param name="lDim">データ配列数</param>
        /// <returns> HySDataCharacteristicInCell 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySDataCharacteristicInCell(long lDim) 
        {
            m_lDim = lDim;
            m_csDataKind = new HySObjectKind[m_lDim];
            m_csUnit = new String[m_lDim];
            m_csLabel = new String[m_lDim];
            for (long lLp = 0; lLp < m_lDim; lLp++)
            {
                m_csDataKind[lLp] = new HySObjectKind();
                //m_csDataKind[lLp] = null;
                m_csUnit[lLp] = "";
                m_csLabel[lLp] = "";
            }
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>セル内変数の数を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lNum = GetDataNumber( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>セル内の変数の数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetDataNumber()
        { return m_lDim; }

        /// <summary><para>method outline:</para>
        /// <para>セル内変数の物理的意味を設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataKind( lItemNo, csLabel, lDataKind, csUnit ) </para>
        /// </example>
        /// <param name="lItemNo">データ位置（０相対）</param>
        /// <param name="csLabel">ラベル（表示用）</param>
        /// <param name="lDataKind">データの種別：物理的意味</param>
        /// <param name="csUnit">単位（表示用）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDataKind(long lItemNo, String csLabel, HySDataCharacteristicInCell.DATA_KIND lDataKind, String csUnit)
        {
            HySObjectKind csDataKind = HySDataKindDefine.ANY_VALUE;
            switch (lDataKind)
            {
                case HySDataCharacteristicInCell.DATA_KIND.ANY_VALUE:
                    csDataKind = HySDataKindDefine.ANY_VALUE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.ANY_VECTOR_LAT:
                    csDataKind = HySDataKindDefine.ANY_VECTOR_LAT;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.ANY_VECTOR_LON:
                    csDataKind = HySDataKindDefine.ANY_VECTOR_LON;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.ANY_VECTOR_ALT:
                    csDataKind = HySDataKindDefine.ANY_VECTOR_ALT;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.QUANTITY_OF_WATER_FLOW:
                    csDataKind = HySDataKindDefine.QUANTITY_OF_WATER_FLOW;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.WATER_LEVEL:
                    csDataKind = HySDataKindDefine.WATER_LEVEL;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.WATER_VOLUME:
                    csDataKind = HySDataKindDefine.WATER_VOLUME;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.SCALOR_VELOCITY:
                    csDataKind = HySDataKindDefine.SCALOR_VELOCITY;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VELOCITY_HORIZONTAL_ANGLE:
                    csDataKind = HySDataKindDefine.VELOCITY_HORIZONTAL_ANGLE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VELOCITY_VERTICAL_ANGLE:
                    csDataKind = HySDataKindDefine.VELOCITY_VERTICAL_ANGLE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VECTOR_L_VELOCITY:
                    csDataKind = HySDataKindDefine.VECTOR_L_VELOCITY;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VECTOR_W_VELOCITY:
                    csDataKind = HySDataKindDefine.VECTOR_W_VELOCITY;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VECTOR_D_VELOCITY:
                    csDataKind = HySDataKindDefine.VECTOR_D_VELOCITY;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.SCALOR_PRESSURE:
                    csDataKind = HySDataKindDefine.SCALOR_PRESSURE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VECTOR_L_PRESSURE:
                    csDataKind = HySDataKindDefine.VECTOR_L_PRESSURE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VECTOR_W_PRESSURE:
                    csDataKind = HySDataKindDefine.VECTOR_W_PRESSURE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.VECTOR_D_PRESSURE:
                    csDataKind = HySDataKindDefine.VECTOR_D_PRESSURE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.SCALOR_RAIN_FALL:
                    csDataKind = HySDataKindDefine.SCALOR_RAIN_FALL;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.FLOODING_WATER_DEPTH:
                    csDataKind = HySDataKindDefine.FLOODING_WATER_DEPTH;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.TEMPERATURE:
                    csDataKind = HySDataKindDefine.TEMPERATURE;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.CONCENTRATION:
                    csDataKind = HySDataKindDefine.CONCENTRATION;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.DENSITY:
                    csDataKind = HySDataKindDefine.DENSITY;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.MASS:
                    csDataKind = HySDataKindDefine.MASS;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.ENERGY:
                    csDataKind = HySDataKindDefine.ENERGY;
                    break;
// Ver1.5 追加 >>
                case HySDataCharacteristicInCell.DATA_KIND.MAX_WATER_LEVEL:
                    csDataKind = HySDataKindDefine.MAX_WATER_LEVEL;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.MAX_W_VELOCITY:
                    csDataKind = HySDataKindDefine.MAX_W_VELOCITY;
                    break;
                case HySDataCharacteristicInCell.DATA_KIND.MAX_L_VELOCITY:
                    csDataKind = HySDataKindDefine.MAX_L_VELOCITY;
                    break;

// Ver1.5 追加 <<
                default:
                    csDataKind = HySDataKindDefine.ANY_VALUE;
                    break;
            }
            this.SetDataKind( lItemNo,  csLabel,  csDataKind,  csUnit);
        }

        /// <summary><para>method outline:</para>
        /// <para>セル内変数の物理的意味を設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataKind( lItemNo, csLabel, csDataKind, csUnit ) </para>
        /// </example>
        /// <param name="lItemNo">データ位置（０相対）</param>
        /// <param name="csLabel">ラベル（表示用）</param>
        /// <param name="csDataKind">データの種別：物理的意味</param>
        /// <param name="csUnit">単位（表示用）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDataKind(long lItemNo, String csLabel, HySObjectKind csDataKind, String csUnit)
        {
            if (lItemNo >= 0 && lItemNo < m_lDim)
            {
                m_csDataKind[lItemNo] = csDataKind;
                m_csUnit[lItemNo] = csUnit;
                m_csLabel[lItemNo] = csLabel;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>セル内変数の物理的意味を設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataKind( lItemNo, csDataKind ) </para>
        /// </example>
        /// <param name="lItemNo">データ位置（０相対）</param>
        /// <param name="csDataKind">データの種別：物理的意味</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDataKind(long lItemNo, HySObjectKind csDataKind)
        {
            if (lItemNo >= 0 && lItemNo < m_lDim)
            {
                m_csDataKind[lItemNo] = csDataKind;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>指定した物理的意味を持つセル内変数位置を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lItemNo = GetDatNo( csDataKind ) </para>
        /// </example>
        /// <param name="csDataKind">データの種別：物理的意味</param>
        /// <returns>セル内変数位置
        /// 　　　　　該当なければ　-1 が返る</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>この処理は時間がかかるので、演算中には使用しない方が良い</para>
        /// </remarks>
        public virtual long GetDatNo(HySObjectKind csDataKind)
        {
            long lRtn = -1;
            for (long lLp = 0; lLp < m_lDim; lLp++)
            {
                if (m_csDataKind[lLp] != null)
                {
                    if (m_csDataKind[lLp].Equals(csDataKind) == true)
                    {
                        lRtn = lLp;
                        break;
                    }
                }
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>セル内変数の指定位置の物理的意味取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csDataKind = GetDataKind( lItemNo ) </para>
        /// </example>
        /// <param name="lItemNo">セル内変数位置</param>
        /// <returns>データの種別：物理的意味
        ///          該当なければＮＵＬＬが返る</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySObjectKind GetDataKind(long lItemNo)
        {
            if (lItemNo >= 0 && lItemNo < m_lDim)
            {
                return m_csDataKind[lItemNo];
            }
            else
            {
                return null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>セル内変数の指定位置の物理的意味取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csDataKind = GetDataKind( lItemNo ) </para>
        /// </example>
        /// <param name="lItemNo">セル内変数位置（０相対）</param>
        /// <param name="csLabel">ラベル（表示用）</param>
        /// <param name="csUnit">単位（表示用）</param>
        /// <returns>データの種別：物理的意味
        ///          該当なければＮＵＬＬが返る</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySObjectKind GetDataKind(long lItemNo, ref String csLabel, ref String csUnit)
        {
            if (lItemNo >= 0 && lItemNo < m_lDim)
            {
                csLabel = m_csLabel[lItemNo];
                csUnit = m_csUnit[lItemNo];
                return m_csDataKind[lItemNo];
            }
            else
            {
                return null;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>内部情報クリア</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Clear() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void Clear()
        {
            m_lDim = 0;
            m_csDataKind = null;
        }
        
        /// <summary><para>method outline:</para>
        /// <para>自己複製</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataCharacteristicInCell csData = Clone( ); </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySDataCharacteristicInCell 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySDataCharacteristicInCell Clone()
        {
            HySDataCharacteristicInCell csRtn = new HySDataCharacteristicInCell();
            csRtn.CopyData(this);
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>引数で与えられたデータを自分にコピーする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CopyData(csOrgData) </para>
        /// </example>
        /// <param name="csOrgData">コピー元情報</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void CopyData(HySDataCharacteristicInCell csOrgData)
        {
            this.m_lDim = csOrgData.m_lDim;
            m_csDataKind = new HySObjectKind[m_lDim];
            m_csUnit = new String[m_lDim];
            m_csLabel = new String[m_lDim];

            for (long lLp = 0; lLp < m_lDim; lLp++)
            {
                m_csDataKind[lLp] = csOrgData.m_csDataKind[lLp];
                m_csUnit[lLp] = csOrgData.m_csUnit[lLp];
                m_csLabel[lLp] = csOrgData.m_csLabel[lLp];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>単独データ種別情報生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataCharacteristicInCell csCell = CreateSingleDataCharacteristic( lItemNo ) </para>
        /// </example>
        /// <param name="lItemNo">セル内変数位置（０相対）</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたインデックスの情報のみを持つセル内部データ種別情報を生成する</para>
        /// </remarks>
        public virtual HySDataCharacteristicInCell CreateSingleDataCharacteristic(long lItemNo )
        {
            if (lItemNo >= 0 && lItemNo < m_lDim)
            {
                HySDataCharacteristicInCell csRtnData = new HySDataCharacteristicInCell(1);
                csRtnData.SetDataKind(0, m_csLabel[lItemNo], m_csDataKind[lItemNo], m_csUnit[lItemNo]);
                return csRtnData;
            }
            else
            {
                return null;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>情報をXMLへ書き出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySXmlNode csXmlNode = XmlOUT(csXmlWriter) </para>
        /// </example>
        /// <param name="csXmlWriter">XML書き出しクラス</param>
        /// <returns>HySXmlNode  XMLノードクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySXmlNode XmlOUT(ref HySXmlWriter csXmlWriter)
        {
            HySXmlNode csRtn = HySXmlNode.CreateElement(HySDefine.xml_DATA_CHARACTERISTIC_IN_CELL, csXmlWriter);

            HySXmlNode csLclNode;
            for (int iLp = 0; iLp < m_lDim; iLp++)
            {
                csLclNode = HySXmlNode.CreateElement(HySDefine.xml_DATA_CHARACTERISTIC, csXmlWriter);

                csLclNode.SetAttribute(HySDefine.xml_DATA_LABEL, m_csLabel[iLp].ToString());
                csLclNode.SetAttribute(HySDefine.xml_DATA_TYPE, m_csDataKind[iLp].ToString());
                csLclNode.SetAttribute(HySDefine.xml_DATA_UNIT, m_csUnit[iLp].ToString());

                csRtn.AppendChild(csLclNode);   // AppendChildは最後に行う
            }

            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>XMLから情報を読み出す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = XmlIN(csPrptyXml) </para>
        /// </example>
        /// <param name="csPrptyXml">入力XML</param>
        /// <returns>bool  =true:正常、=false:型式不適等</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>HySDefine.xml_DATA_CHARACTERISTIC_IN_CELLタグの選択は外部で行っておく</para>
        /// </remarks>
        public virtual bool XmlIN(HySXmlNode csPrptyXml)
        {
            if (csPrptyXml == null)
            { return false; }

            this.Clear();

            HySXmlNode csXml = null;
            HySXmlNodeList csXmlVariableList = csPrptyXml.GetChildNodeListByTagName(HySDefine.xml_DATA_CHARACTERISTIC);
            long lValCnt = csXmlVariableList.GetCount();

            m_lDim = lValCnt;
            m_csDataKind = new HySObjectKind[m_lDim];
            m_csUnit = new String[m_lDim];
            m_csLabel = new String[m_lDim];

            string sLabelAttr;
            string sDataKindAttr;
            string sUnitAttr;
            
            for (long lLoop = 0; lLoop < lValCnt; lLoop++)
            {
                csXml = csXmlVariableList.GetNode(lLoop);
                // ラベル
                sLabelAttr = csXml.GetAttribute(HySDefine.xml_DATA_LABEL);
                // データタイプ
                sDataKindAttr = csXml.GetAttribute(HySDefine.xml_DATA_TYPE);
                // 単位
                sUnitAttr = csXml.GetAttribute(HySDefine.xml_DATA_UNIT);

                SetDataKind(lLoop, sLabelAttr, new HySObjectKind(sDataKindAttr), sUnitAttr);
            }
            return true;
        }


        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }

/*
        /// <summary>
        /// データの種別定義 HySDataKindDefine内に定義されている項目を、enum化
        /// HySDataKindDefine内の定義に追加が発生した場合には、この enum 内も合わせて追加する必要がある。
        /// </summary>
        //[NonSerialized]
        public enum DATA_KIND
        {
            /// <summary>任意値</summary>
            ANY_VALUE = 0,
            /// <summary>任意値＿緯度方向　北向きが正、南向きが負</summary>
            ANY_VECTOR_LAT,
            /// <summary>任意値＿経度方向　東向きが正、西向きが負</summary>
            ANY_VECTOR_LON,
            /// <summary>任意値＿高さ方向　上向きが正、下向きが負</summary>
            ANY_VECTOR_ALT,
            /// <summary>流量：ｍ３／ｓｅｃ　quantity of flow</summary>
            QUANTITY_OF_WATER_FLOW,
            /// <summary>水位：　ｍ　water level</summary>
            WATER_LEVEL,
            /// <summary>水量：　ｍ３　water volume</summary>
            WATER_VOLUME,
            /// <summary>スカラー：流速：　ｍ／Ｓｅｃ　water velocity</summary>
            SCALOR_VELOCITY,
            /// <summary>流速:水平方向： 北をゼロ、時計回りに正、－π～＋πまで　radian　</summary>
            VELOCITY_HORIZONTAL_ANGLE,
            /// <summary>流速:垂直方向： 水平をゼロ、上向きに正、－π／２～＋π／２まで　radian　</summary>
            VELOCITY_VERTICAL_ANGLE,
            /// <summary>ベクトル：ライン方向流速：　ｍ／Ｓｅｃ　water velocity</summary>
            VECTOR_L_VELOCITY,
            /// <summary>ベクトル：横方向流速：　ｍ／Ｓｅｃ　water velocity</summary>
            VECTOR_W_VELOCITY,
            /// <summary>ベクトル：深さ方流速：　ｍ／Ｓｅｃ　water velocity</summary>
            VECTOR_D_VELOCITY,
            /// <summary>スカラー：圧力：　ｍ／Ｓｅｃ　water velocity</summary>
            SCALOR_PRESSURE,
            /// <summary>ベクトル：ライン方向圧力：　ｍ／Ｓｅｃ　water velocity</summary>
            VECTOR_L_PRESSURE,
            /// <summary>ベクトル：横方向圧力：　ｍ／Ｓｅｃ　water velocity</summary>
            VECTOR_W_PRESSURE,
            /// <summary>ベクトル：深さ方圧力：　ｍ／Ｓｅｃ　water velocity</summary>
            VECTOR_D_PRESSURE,
            /// <summary>スカラー：単位時間当りの雨量：　ｍｍ／ＨＯＵＲ　Rain</summary>
            SCALOR_RAIN_FALL,
            /// <summary>スカラー：浸水深：　ｍ　flooding depth</summary>
            FLOODING_WATER_DEPTH,
            /// <summary>スカラー：温度：　℃　</summary>
            TEMPERATURE,
            /// <summary>スカラー：濃度（質量濃度）：　mg/Kg =ppm </summary>
            CONCENTRATION,
            /// <summary>スカラー：密度：　Kg/m3 </summary>
            DENSITY,
            /// <summary>スカラー：質量：　Kg </summary>
            MASS,
            /// <summary>スカラー：エネルギー：　J </summary>
            ENERGY,
// Ver1.5 追加 >>
            /// <summary>スカラー：最大水位：　ｍ </summary>
            MAX_WATER_LEVEL,
            /// <summary>スカラー：最大流速（横断方向）：　ｍ／Ｓｅｃ </summary>
            MAX_W_VELOCITY,
            /// <summary>スカラー：最大流速（下流方向）：　ｍ／Ｓｅｃ </summary>
            MAX_L_VELOCITY
// Ver1.5 追加 <<
        }
*/
    }
}
