﻿// <summary>ソースコード：>ＨＹＳＳＯＰデータリンクリストクラス</summary>
// <author>CommonMP</author>

using System;
using System.Threading;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>データリンクリストクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    [Serializable]
    public class HySDataLinkedList : HySDataRoot, HySVersionManagement
    {
        /// <summary>リンクリスト件数</summary>
        protected long m_lNRecord = 0;

        /// <summary>先頭ノード</summary>
        HySDataLinkedListNode m_csHeadNode = null;        // 先頭データ
        /// <summary>最終ノード</summary>
        HySDataLinkedListNode m_csTailNode = null;        // 最終データ
        /// <summary>カーソル位置ノード</summary>
        HySDataLinkedListNode m_csCursorNode = null;   // カーソル位置

        /// <summary>処理中ミューテックス</summary>
        [NonSerialized]
        private HySMutex m_csBusy = null;           // 処理中排他ミューテックス

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataLinkedList csDataLinkedList = new HySDataLinkedList() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataLinkedList  生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySDataLinkedList()
        {
            m_lNRecord = 0;
            m_csBusy = new HySMutex();
            m_csHeadNode = null;
            m_csTailNode = null;
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>デストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>無し</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メモリーリーク防止の為　内部変数は必ず null にセットする</para>
        /// </remarks>
        ~HySDataLinkedList()
        {
            try
            {
                for (long lLp = 0; lLp < m_lNRecord; lLp++)
                {
                    this.RemoveLastData();
                }
                m_lNRecord = 0;
                m_csHeadNode = null;
                m_csTailNode = null;
                m_sDecodedVersion = null;
                m_csBusy = null;
            }
            catch
            {
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>レコード数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lCount = GetCount() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long　レコード数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetCount()
        {
            return m_lNRecord;
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト追加(リストの最初に追加される)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lCount = AddFirst(csData) </para>
        /// </example>
        /// <param name="csData">オブジェクト</param>
        /// <returns>long  データの数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long AddFirst(HySDataRoot csData)
        {
            this.Lock();
            HySDataLinkedListNode csAddNode = new HySDataLinkedListNode(csData);
            m_csCursorNode = csAddNode;
            if (m_csHeadNode == null)
            {
                m_csTailNode = csAddNode;
                m_csHeadNode = csAddNode;
            }
            else
            {
                m_csHeadNode.AddPrev(csAddNode);
                m_csHeadNode = csAddNode;
            }

            m_lNRecord += 1;
            this.Unlock();
            return m_lNRecord;
        }
        /// <summary><para>method outline:</para>
        /// <para>オブジェクト追加(リストの最後に追加される)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lCount = AddLast(csData) </para>
        /// </example>
        /// <param name="csData">オブジェクト</param>
        /// <returns>long  データの数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long AddLast(HySDataRoot csData)
        {
            this.Lock();
            HySDataLinkedListNode csAddNode = new HySDataLinkedListNode(csData);
            m_csCursorNode = csAddNode;
            if (m_csTailNode == null)
            {
                m_csTailNode = csAddNode;
                m_csHeadNode = csAddNode;
            }
            else
            {
                // AddNext()の高速版を呼び出すように変更する
                m_csTailNode.AddNextQuick(csAddNode);
                m_csTailNode = csAddNode;
            }

            m_lNRecord += 1;
            this.Unlock();
            return m_lNRecord;
        }

        /// <summary><para>method outline:</para>
        /// <para>最初のデータを取り外す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csDataRoot = RemoveFirstData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataRoot  取り外したデータ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySDataRoot RemoveFirstData()
        {
            HySDataRoot csRtn = null;
            this.Lock();
            m_csCursorNode = m_csTailNode;
            if (m_csHeadNode != null)
            {
                csRtn = m_csHeadNode.GetData();
                m_csCursorNode = m_csHeadNode.GetNext();
                m_csHeadNode.SetPrev(null);
                m_csHeadNode.SetNext(null);
                m_lNRecord--;
            }
            m_csHeadNode = m_csCursorNode;
            if (m_csHeadNode != null)
            {
                m_csHeadNode.SetPrev(null);
            }
            this.Unlock();
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>最後のデータを取り外す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csLastData = RemoveLastData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataRoot  取り外したデータ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySDataRoot RemoveLastData()
        {
            HySDataRoot csRtn = null;
            this.Lock();
            m_csCursorNode = m_csHeadNode;
            if (m_csTailNode != null)
            {
                csRtn = m_csTailNode.GetData();
                m_csCursorNode = m_csTailNode.GetPrev();
                m_csTailNode.SetPrev(null);
                m_csTailNode.SetNext(null);
                m_lNRecord--;
            }
            m_csTailNode = m_csCursorNode;
            if (m_csTailNode != null)
            {
                m_csTailNode.SetNext(null);
            }
            this.Unlock();
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>指定データ削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csData = RemoveData(lOrderNum) </para>
        /// </example>
        /// <param name="lOrderNum">指定番号：０相対</param>
        /// <returns>HySDataRoot　データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>先頭から指定数後のデータをリストから取り外す</para>
        /// <para>戻り値は　取り外したデータ</para>
        /// </remarks>
        public HySDataRoot RemoveData(long lOrderNum)
        {
            if (m_csHeadNode == null)
            {
                return null;
            }
            else
            {
                HySDataRoot csRtn = null;

                this.Lock();
                HySDataLinkedListNode csNode = m_csHeadNode.GetAfterNode(lOrderNum);
                if (csNode != null)
                {
                    csRtn = csNode.GetData();
                    m_csHeadNode = csNode.Remove();
                    if (m_csHeadNode != null)
                    {
                        m_csTailNode = m_csHeadNode.GetTail();
                    }
                    else
                    {
                        m_csTailNode = null;
                    }
                    m_lNRecord -= 1;
                }
                this.Unlock();
                m_csCursorNode = m_csHeadNode;
                return csRtn;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>指定データ削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csData = RemoveData(csData) </para>
        /// </example>
        /// <param name="csData">指定データ</param>
        /// <returns>HySDataRoot　データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定のデータをリストから取り外す</para>
        /// <para>戻り値は　取り外したデータ</para>
        /// </remarks>
        public HySDataRoot RemoveData(HySDataRoot csData)
        {
            if (m_csHeadNode == null)
            {
                return null;
            }
            else
            {
                HySDataRoot csRtn = null;

                this.Lock();
                HySDataLinkedListNode csNode = m_csHeadNode;
                if (csNode.EqualData(csData))
                {
                    csRtn = csNode.GetData();
                    m_csHeadNode = csNode.Remove();
                    if (m_csHeadNode != null)
                    {
                        m_csTailNode = m_csHeadNode.GetTail();
                    }
                    else
                    {
                        m_csTailNode = null;
                    }
                    m_lNRecord -= 1;
                }
                else
                {
                    for (int iLp = 1; iLp < m_lNRecord; iLp++)
                    {
                        csNode = csNode.GetNext();
                        if (csNode == null)
                        {
                            break;
                        }
                        if (csNode.EqualData(csData))
                        {
                            csRtn = csNode.GetData();
                            m_csHeadNode = csNode.Remove();
                            if (m_csHeadNode != null)
                            {
                                m_csTailNode = m_csHeadNode.GetTail();
                            }
                            else
                            {
                                m_csTailNode = null;
                            }
                            m_lNRecord -= 1;
                            break;
                        }
                    }
                }
                this.Unlock();
                m_csCursorNode = m_csHeadNode;
                return csRtn;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最初に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorFirst() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCursorFirst()
        {
            m_csCursorNode = m_csHeadNode;
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最後に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorLast() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCursorLast()
        {
            m_csCursorNode = m_csTailNode;
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を次の位置に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorNext() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void MoveCursorNext()
        {
            if (m_csCursorNode != null)
            {
                if (m_csCursorNode.GetNext() != null)
                {
                    m_csCursorNode = m_csCursorNode.GetNext();
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を一つ前に戻す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorPrev() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void MoveCursorPrev()
        {
            if (m_csCursorNode != null)
            {
                if (m_csCursorNode.GetPrev() != null)
                {
                    m_csCursorNode = m_csCursorNode.GetPrev();
                }
            }

        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置のデータを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csData = GetCursorData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>
        /// HySDataRoot  データ
        /// カーソル位置不定の場合には　null が返る
        /// </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySDataRoot GetCursorData()
        {
            HySDataRoot csRtn = null;
            if (m_csCursorNode != null)
            {
                csRtn = m_csCursorNode.GetData();
            }
            return csRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csData = GetData(lOrderNum) </para>
        /// </example>
        /// <param name="lOrderNum">指定番号：０相対</param>
        /// <returns>HySDataRoot　データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>先頭から指定数後のデータを取得する</para>
        /// </remarks>
        public HySDataRoot GetData(long lOrderNum)
        {
            if (m_csHeadNode == null)
            {
                return null;
            }
            else
            {
                HySDataRoot csRtn = null;

                HySDataLinkedListNode csNode = m_csHeadNode.GetAfterNode(lOrderNum);
                if (csNode != null)
                {
                    csRtn = csNode.GetData();
                }

                return csRtn;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>先頭データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csData = GetFirstData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataRoot　データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>先頭から指定数後のデータを取得する</para>
        /// </remarks>
        public HySDataRoot GetFirstData()
        {
            HySDataRoot csRtn = null;
            if (m_csHeadNode != null)
            {
                csRtn = m_csHeadNode.GetData();
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>最終データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csData = GetLastData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataRoot　データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>先頭から指定数後のデータを取得する</para>
        /// </remarks>
        public HySDataRoot GetLastData()
        {
            HySDataRoot csRtn = null;
            if (m_csTailNode != null)
            {
                csRtn = m_csTailNode.GetData();
            }
            return csRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>全データ取り外し</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Clear() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void Clear( )
        {
            long lRecordCount = m_lNRecord;
            for (int iLp = 0; iLp < lRecordCount; iLp++)
            {
                this.RemoveData(0); // 先頭を外す
            }
            m_csCursorNode = null;
        }

        /// <summary><para>method outline:</para>
        /// <para>ミューテックスロック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Lock() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void Lock()
        {
            m_csBusy.Wait();
        }
        /// <summary><para>method outline:</para>
        /// <para>ミューテックスアンロック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Unlock() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void Unlock()
        {
            m_csBusy.Release();
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            m_csBusy = new HySMutex();  // <-- これが重要

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> OnDeserialization(csObj) </para>
        /// </example>
        /// <param name="csObj">コールバックを開始したオブジェクト。このパラメータの機能は、現在実装されていません。</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            //m_csBusy = new HySMutex();
            PostDeserialize();
        }

        ///// <summary><para>method outline:</para>
        ///// <para>復元後の整合性確保処理</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> PostDeserialize( ) </para>
        ///// </example>
        ///// <param name="">無し</param>
        ///// <returns>無し</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        ///// バージョンによって変数等追加が発生した場合にオーバーライドして実装する。
        ///// </para>
        ///// </remarks>
        //public virtual void PostDeserialize()
        //{
        //    m_csBusy = new HySMutex();
        //}
    }
}
