﻿// <summary>ソースコード：時系列データ基底クラス（抽象クラス）</summary>
// <author>CommonMP</author>
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>時系列データ基底クラス（抽象クラス）</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>本クラスの派生クラスは全て[Serializable]であること</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>
    /// [CommonMP][ver 1.0.0][2008/10/01][新規作成]
    /// [CommonMP][ver 1.4.0][2013/12/10][修正]
    /// </para>
    /// </remarks>
    [Serializable]
    public abstract class HySTimeSeriesBase : HySDataRoot, HySVersionManagement
    {
        /// <summary>時系列データの保持最大数を指定する</summary>
        protected long m_lMaxDataNum = 8192;

        /// <summary>時系列１次元データ</summary>
        protected HySDataLinkedList m_csDataList = new HySDataLinkedList();

        /// <summary>最後に追加した時刻</summary>
        protected HySTime m_csLastTime = HySTime.DEFAULT_TIME.Clone();

        /// <summary>処理中ミューテックス(Serialize 対象外)</summary>
        [NonSerialized]
        protected HySMutex m_csBusy = new HySMutex();
        //★★★★★★★★★★★★★★★★
        // Ver1.4で追加(一時ファイル作成)
        //★★★★★★★★★★★★★★★★
        /// <summary>ファイル出力リスト数管理ハッシュ</summary>
        protected Hashtable m_csFileOutDataTbl = new Hashtable();

        /// <summary><para>method outline:</para>
        /// <para>デストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>無し</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メモリーリーク防止の為　内部変数は必ず null にセットする</para>
        /// </remarks>
        ~HySTimeSeriesBase()
        {
            try
            {
                m_sDecodedVersion = null;
                if (m_csDataList != null)
                {
                    m_csDataList.Clear();
                }
                m_csDataList = null;
                m_csLastTime = null;
                m_csBusy = null;
            }
            catch
            {
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySTimeSeriesBase csBase = new HySTimeSeriesBase()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTimeSeriesBase 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySTimeSeriesBase()
        {
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>全てのデータを削除する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Clear() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void Clear()
        {
            m_csLastTime = HySTime.DEFAULT_TIME.Clone();
            m_csDataList.Clear();
            //★★★★★★★★★★★★★★★★
            // Ver1.4で追加(一時ファイル作成)
            //★★★★★★★★★★★★★★★★
            if (m_csFileOutDataTbl.Count != 0)
            {
                // 此処で出力ファイルを削除
                ICollection FileName = m_csFileOutDataTbl.Values;
                foreach (string sFileName in FileName)
                {
                    using (HySFile csFile = new HySFile(sFileName))
                    {
                        if (csFile.Exist() == true)
                        {
                            // ファイルを削除
                            csFile.Delete();
                        }
                    }
                }
                m_csFileOutDataTbl.Clear();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>最大保持レコード数設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetMaxHoldDataNum(lMaxDtNum) </para>
        /// </example>
        /// <param name="lMaxDtNum">最大保持数</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetMaxHoldDataNum(long lMaxDtNum)
        { m_lMaxDataNum = lMaxDtNum; }
        /// <summary><para>method outline:</para>
        /// <para>最大保持レコード数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetMaxHoldDataNum() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 最大保持数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetMaxHoldDataNum()
        { return m_lMaxDataNum; }

        /// <summary><para>method outline:</para>
        /// <para>保持されている情報の最も古い時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime dTime = GetLastTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>最古時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTime GetFirstTime()
        {
            this.SetCursorFirst();
            HySTimeRecordIF csFstRcd = this.GetCursorData();
            if (csFstRcd != null)
            {
                return csFstRcd.GetLastTime();
            }
            else
            {
                return m_csLastTime;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>最新の時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime dTime = GetLastTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>最新時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTime GetLastTime()
        {
            return m_csLastTime;
        }
        /// <summary><para>method outline:</para>
        /// <para>最後のデータの時間比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bWork = Before(csTime) </para>
        /// </example>
        /// <param name="csTime">比較用時刻</param>
        /// <returns>
        /// bool  最後に追加した時刻が 引数：csTime の時刻より前の場合は true、そうでない場合（同一時刻含む）は false
        /// </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool Before(HySTime csTime)
        {
            return m_csLastTime.Before(csTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>指定時刻よりも後か</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = After(csTime) </para>
        /// </example>
        /// <param name="csTime">
        /// 比較用日付管理クラスオブジェクト(HySTime)
        /// </param>
        /// <returns>bool  =HySTime の現在時刻が 引数：Date の時刻よりあとの場合は true、そうでない場合は false </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public bool After(HySTime csTime)
        {
            return m_csLastTime.After(csTime);
        }
        /// <summary><para>method outline:</para>
        /// <para>最後のデータの時間比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bWork = EqualTime(csTime) </para>
        /// </example>
        /// <param name="csTime">比較用時刻</param>
        /// <returns>
        /// bool  最後に追加した時刻が 引数：csTime と同じならばは true、そうでない場合は false
        /// </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool EqualTime(HySTime csTime)
        {
            return m_csLastTime.Equals(csTime);
        }

        /// <summary><para>method outline:</para>
        /// <para>最新の時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLastTime(csLastTime) </para>
        /// </example>
        /// <param name="csLastTime">時刻</param>
        /// <returns>最新時刻（データが無い場合にはnull）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void SetLastTime(HySTime csLastTime)
        {
            m_csLastTime.SetTime(csLastTime);
        }

        /// <summary><para>method outline:</para>
        /// <para>時系列データの数を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDtNum = GetCount() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>時系列データの数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetCount()
        {
            return m_csDataList.GetCount();
        }

        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻と同一位置に持ってくる
        /// 同一時刻がない場合には、指定時刻に最も近い直後の時刻のレコード位置に持ってくる(SetCursorPlus()と同じ動作)
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDtNum = SetCursor(csTime) </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long  =カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>戻り値=0の場合には，指定時刻よりも後の情報が無い </para>
        /// </remarks>
        public virtual long SetCursor(HySTime csTime)
        {
            long lRtn = 0;
            if (m_csLastTime.Before(csTime) == false)
            {   // 最終情報は指定時刻よりも後
                HySTimeRecordIF csWorkData;
                long lDtNum = m_csDataList.GetCount();

                // リストの最後からたどってゆく
                m_csDataList.SetCursorLast();
                for (long lLp = 0; lLp < lDtNum; lLp++)
                {
                    lRtn += 1;
                    csWorkData = (HySTimeRecordIF)m_csDataList.GetCursorData();
                    if (csWorkData.Before(csTime) == true)
                    {   // 記録時刻が、指定時刻以前
                        m_csDataList.MoveCursorNext();
                        lRtn -= 1;
                        break;
                    }
                    else if (csWorkData.EqualTime(csTime) == true)
                    {
                        break;
                    }
                    m_csDataList.MoveCursorPrev();
                }
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻の直後に持ってくる
        /// （指定時刻に最も近い直後の時刻のレコード位置に持ってくる）
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lDtNum = SetCursorPlus(csTime);/ </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long  =カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>戻り値=0の場合には，指定時刻よりも後の情報が無い </para>
        /// </remarks>
        public virtual long SetCursorPlus(HySTime csTime)
        {
            long lRtn = 0;
            HySTimeRecordIF csWorkData;
            long lDtNum = m_csDataList.GetCount();

            // リストの最後からたどってゆく
            m_csDataList.SetCursorLast();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                csWorkData = (HySTimeRecordIF)m_csDataList.GetCursorData();

                if (csWorkData.After(csTime) == false)
                {   // 指定時刻が、記録位置よりも後でない場合（記録位置が同じか、より前になった場合）
                    m_csDataList.MoveCursorNext();
                    break;
                }
                else
                {
                    lRtn += 1;
                    m_csDataList.MoveCursorPrev();
                }
            }
            return lRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>
        /// カーソル位置を指定時刻の位置の直前に持ってくる
        /// （指定時刻と同一時刻が存在すればその位置に持ってくる）
        /// </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>long lDtNum = SetCursorMinus(csTime);/ </para>
        /// </example>
        /// <param name="csTime">指定時刻</param>
        /// <returns>long  =カーソル位置から最後までのデータ数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 戻り値＜０の場合には　指定時刻以前の情報が無い
        /// </para>
        /// </remarks>
        public virtual long SetCursorMinus(HySTime csTime)
        {
            long lRtn = 0;
            HySTimeRecordIF csWorkData;
            long lDtNum = m_csDataList.GetCount();

            // リストの最後からたどってゆく(実際の計算では、最後からたどる方が、早くループを抜け出すとが多いため)
            bool bBreakFlg = false;
            m_csDataList.SetCursorLast();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                csWorkData = (HySTimeRecordIF)m_csDataList.GetCursorData();

                if (csWorkData.After(csTime) == false)
                {   // 指定時刻が、記録位置よりも後でない場合（記録位置が同じか、より前になった場合）
                    lRtn += 1;
                    bBreakFlg = true;
                    break;
                }
                else
                {
                    lRtn += 1;
                    m_csDataList.MoveCursorPrev();
                }
            }
            if (bBreakFlg == false)
            {   // Breakが無かったという事は、全データが指定時刻よりも後であるという事
                lRtn = -1;
            }
            /****** CommonMPモデル内で使用するときには、最初からたどって行くよりも、最後からたどった方が、早くヒットする可能性が高いため、
             * リストを最初からたどっていう処理は使用しない。
            // リストの最初からたどってゆく
            lRtn = -1;
            long lDtCnt = -1;
            m_csDataList.SetCursorFirst();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                csWorkData = (HySTimeRecordIF)m_csDataList.GetCursorData();

                if (csWorkData.After(csTime) == true)
                {   // 指定時刻よりも記録位置が後になった場合（
                    m_csDataList.MoveCursorPrev();
                    break;
                }
                else
                {
                    lDtCnt +=1;
                    m_csDataList.MoveCursorNext();
                }
            }
            if (lDtCnt >= 0)
            {
                lRtn = lDtNum - lDtCnt;
            }
            *****/
            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最初に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorFirst() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCursorFirst()
        {
            m_csDataList.SetCursorFirst();
        }  
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を最後に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCursorFirst() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCursorLast()
        {
            m_csDataList.SetCursorLast();
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を次の位置に持ってくる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorNext() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void MoveCursorNext()
        {
            m_csDataList.MoveCursorNext();
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置を一つ前に戻す</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MoveCursorPrev() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void MoveCursorPrev()
        {
            m_csDataList.MoveCursorPrev();
        }

        /// <summary><para>method outline:</para>
        /// <para>カーソル位置の時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime =  GetCursorTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime 時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTime GetCursorTime()
        {
            return ((HySTimeRecordIF)m_csDataList.GetCursorData()).GetLastTime();
        }
        /// <summary><para>method outline:</para>
        /// <para>カーソル位置のデータを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McTimeRecordRoot = McTimeRecordRoot = GetCursorData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTimeRecordIF GetCursorData()
        {
            return (HySTimeRecordIF)m_csDataList.GetCursorData();
        }

        /// <summary><para>method outline:</para>
        /// <para>最新のデータを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTimeRecordIF csDt = GetLastData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 最新データ </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTimeRecordIF GetLastData()
        {
            this.SetCursorLast();
            return this.GetCursorData();
        }

        /// <summary><para>method outline:</para>
        /// <para>指定時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime dtime = GetLastTime(lOrderNum) </para>
        /// </example>
        /// <param name="lOrderNum">順序：０相対</param>
        /// <returns>HySTime 時刻（データが無い場合にはnull）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTime GetTime(long lOrderNum)
        {
            return ((HySTimeRecordIF)m_csDataList.GetData(lOrderNum)).GetLastTime();
        }
        /// <summary><para>method outline:</para>
        /// <para>最新のデータを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTimeRecordIF csDt = GetData(lNum) </para>
        /// </example>
        /// <param name="lNum">リスト上の位置（０相対）</param>
        /// <returns> 最新データ </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTimeRecordIF GetData(long lNum)
        {
            return (HySTimeRecordIF)m_csDataList.GetData(lNum);
        }

        /// <summary><para>method outline:</para>
        /// <para>時系列最後のデータを削除する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveLastData() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTimeRecordIF RemoveLastData()
        {
            HySTimeRecordIF csRtn = null;
            if (m_csDataList.GetCount() == 0)
            {
                m_csLastTime = HySTime.DEFAULT_TIME;
            }
            else if (m_csDataList.GetCount() == 1)
            {
                csRtn = m_csDataList.RemoveLastData() as HySTimeRecordIF;
                m_csLastTime = HySTime.DEFAULT_TIME;
            }
            else
            {
                csRtn = m_csDataList.RemoveLastData() as HySTimeRecordIF;
                m_csLastTime = this.GetLastData().GetLastTime();
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>所定時刻以降の情報を破棄</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveAfterData(csTime) </para>
        /// </example>
        /// <param name="csTime">時刻</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で与えた時刻よりも未来のレコードがあればそれらを削除する</para>
        /// </remarks>
        public virtual void RemoveAfterData(HySTime csTime)
        {
            long lDtNum = this.GetCount();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                if( this.GetLastTime().After(csTime) )
                {   // 指定時刻よりも後の情報がある（最新データは指定時刻よりも未来）
                    this.RemoveLastData(); // 最新時刻を削除する
                }
                else
                {
                    break;
                }
            }
            HySTimeRecordIF csLastRcd = this.GetLastData();
            if (csLastRcd != null)
            {
                m_csLastTime = csLastRcd.GetLastTime().Clone();
            }
            else
            {
                m_csLastTime = HySTime.DEFAULT_TIME.Clone();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>所定時刻以前の情報を破棄</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> RemoveBeforeData(csTime) </para>
        /// </example>
        /// <param name="csTime">時刻</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で与えた時刻よりも過去のレコードがあればそれらを削除する</para>
        /// </remarks>
        public virtual void RemoveBeforeData(HySTime csTime)
        {
            long lDtNum = this.GetCount();
            for (long lLp = 0; lLp < lDtNum; lLp++)
            {
                this.SetCursorFirst();
                HySTimeRecordIF csFstRcd = this.GetCursorData();
                if (csFstRcd != null)
                {
                    if (csFstRcd.GetLastTime().Before(csTime) == true)
                    {
                        this.m_csDataList.RemoveFirstData();
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    break;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>情報を追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddData(csAddRecord) </para>
        /// </example>
        /// <param name="csAddRecord">追加データ</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で与えた時刻と同一又は未来のレコードがあればそれらを削除して最後に追加する</para>
        /// </remarks>
        public virtual void AddData(HySTimeRecordIF csAddRecord)
        {
            HySTime csAddTime = csAddRecord.GetLastTime();
            if (this.Before(csAddTime) == false)
            {   // 同一または、未来情報有り
                this.RemoveAfterData(csAddTime);
                if (this.EqualTime(csAddTime) == true)
                {   // 同一情報有り
                    this.m_csDataList.RemoveLastData();
                }
            }
            this.m_csDataList.AddLast(csAddRecord);
            this.m_csLastTime = csAddTime.Clone();

            if (m_lMaxDataNum < this.GetCount())
            {
                this.m_csDataList.RemoveFirstData();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>時系列情報を追加</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddData(csTimeSeriesBase) </para>
        /// </example>
        /// <param name="csTimeSeriesBase">追加する時系列データ</param>
        /// <returns> 無し </returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で与えた時刻と同一又は未来のレコードがあればそれらを削除して追加する</para>
        /// </remarks>
        public virtual void AddTimeSeriesData(HySTimeSeriesBase csTimeSeriesBase)
        {
            // 引数で受け取った時系列情報のレコードを一件ずつ取得して自身に追加する
            csTimeSeriesBase.SetCursorFirst();
            for (long lTimeRecordIdx = 0; lTimeRecordIdx < csTimeSeriesBase.GetCount(); ++lTimeRecordIdx)
            {
                HySTimeRecordIF csTimeRecord = csTimeSeriesBase.GetCursorData();
                csTimeSeriesBase.MoveCursorNext();

                AddData(csTimeRecord);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>新しいレコードを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTimeRecordIF csNewRecord = CreateNewRecord( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>GetNewRecord 新レコード</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>時系列として保持すべき情報を新規に生成する</para>
        /// </remarks>
        public abstract HySTimeRecordIF CreateNewRecord();
            
        /// <summary><para>method outline:</para>
        /// <para>引数で与えられたデータを自分にコピーする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = CopyData(csOrgData) </para>
        /// </example>
        /// <param name="csOrgData">コピー元情報</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract void CopyData(HySTimeSeriesBase csOrgData);

        //=================================
        // マルチスレッドから扱う場合の排他メソッド群
        //=================================

        /// <summary><para>method outline:</para>
        /// <para>処理排他ロック</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Lock() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>異なるマルチスレッドからコールされる可能性がある場合に使用する（デッドロックに気を付けること）</para>
        /// </remarks>
        public void Lock()
        {
            m_csBusy.Wait();
        }
        /// <summary><para>method outline:</para>
        /// <para>処理排他解除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Unlock() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>異なるマルチスレッドからコールされる可能性がある場合に使用する（デッドロックに気を付けること）</para>
        /// </remarks>
        public void Unlock()
        {
            m_csBusy.Release();
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            m_csBusy = new HySMutex();

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
    }
}
