﻿// <summary>ソースコード：ＨＹＳＳＯＰ地理メッシュ情報 データ 基底抽象化クラス<</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
//using CommonMP.HYSSOP.CoreImpl.HSData.CommonData;

namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>地理メッシュ情報 データ 基底抽象化クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// <para>remarks:</para>
    /// <para>地理空間上に　メッシュ状の情報を保持できる（但し直交座標）</para>
    /// </remarks>
    [Serializable]
    public abstract class HySGeoMeshSerial : HySTimeSeriesBase, HySVersionManagement
    {
        //============　境界情報　================
        /// <summary>南西頂点緯度（度単位）</summary>
        protected double m_dSWLatitude;
        /// <summary>南西頂点経度（度単位）</summary>
        protected double m_dSWLongitude;
        /// <summary>北東頂点緯度（度単位）</summary>
        protected double m_dNELatitude;
        /// <summary>北東頂点経度（度単位）</summary>
        protected double m_dNELongitude;

        /// <summary>標高上限（m単位）</summary>
        protected double m_dUpperAltitude = 0.0;
        /// <summary>標高下限（m単位）</summary>
        protected double m_dLowerAltitude = 0.0;

        //============　保持データの種別　================
        /// <summary>メッシュデータの種別</summary>
        protected HySGeoMesh.MESH_DATA_KIND m_lMeshKind = HySGeoMesh.MESH_DATA_KIND.DOUBLE_DATA_HOLD;

        //============　保持データ　================
        /// <summary>緯度方向分割数</summary>
        protected long m_lLatDim = 1;
        /// <summary>経度方向分割数</summary>
        protected long m_lLonDim = 1;
        /// <summary>高度方向分割数</summary>
        protected long m_lAltDim = 1;

        //============　パターン情報　================
        /// <summary>メッシュセルパターン数</summary>
        protected long m_lPatternNum = 0;
        /// <summary>メッシュセル表示パターン情報</summary>
        protected HySGeoMeshDispPattern[] m_csPatternInf = null;
        //============　データの意味　================
        /// <summary> セル内データの意味を示す </summary>
        protected HySDataCharacteristicInCell m_csCellDataKind = null;
        /// <summary>セル識別子管理情報 </summary>
        protected HySCellIDManageData m_csCellIDMngData = new HySCellIDManageData();

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGeoMeshSerial csGeoMesh = new HySGeoMeshSerial( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected HySGeoMeshSerial()
        {
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGeoMeshSerial csGeoMesh = new HySGeoMeshSerial(lKind, lLonDim, lLatDim, lAltDim, lCellDataDim) </para>
        /// </example>
        /// <param name="lKind">保持するデータの形</param>
        /// <param name="lLatDim">緯度方向のメッシュ分割数</param>
        /// <param name="lLonDim">経度方向のメッシュ分割数</param>
        /// <param name="lAltDim">高度方向のメッシュ分割数</param>
        /// <param name="lCellDataDim">１メッシュ内のデータ配列数</param>
        /// <returns> HySGeoMeshSerial 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para> 抽象化クラスであるため、派生側で　:base(lKind, lLonDim, lLatDim, lAltDim, lCellDataDim) として使用する </para>
        /// </remarks>
        public HySGeoMeshSerial(HySGeoMesh.MESH_DATA_KIND lKind, long lLonDim, long lLatDim, long lAltDim, long lCellDataDim)
        {
            m_lMeshKind = lKind;
            m_lLatDim = lLatDim;
            m_lLonDim = lLonDim;
            m_lAltDim = lAltDim;

            m_csCellDataKind = new HySDataCharacteristicInCell(lCellDataDim);
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGeoMeshSerial csGeoMesh = new HySGeoMeshSerial(csMeshData) </para>
        /// </example>
        /// <param name="csMeshData">メッシュ情報クラス</param>
        /// <returns> HySGeoMeshSerial 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para> 抽象化クラスであるため、派生側で　:base(csMeshData) として使用する </para>
        /// </remarks>
        public HySGeoMeshSerial(HySGeoMesh csMeshData)
        {
            CopyCommonInfo(csMeshData);
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>メッシュデータ種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MESH_DATA_KIND lKind = GetDataKind() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>MESH_DATA_KIND メッシュデータ種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySGeoMesh.MESH_DATA_KIND GetDataMeshKind()
        {
            return m_lMeshKind;
        }
        /// <summary><para>method outline:</para>
        /// <para>緯度方向のメッシュ分割数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDim = GetLatDimension() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 緯度方向のメッシュ分割数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetLatDimension()
        {
            return m_lLatDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>経度方向のメッシュ分割数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDim = GetLonDimension() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 経度方向のメッシュ分割数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetLonDimension()
        {
            return m_lLonDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>高度方向のメッシュ分割数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lDim = GetAltDimension() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 高度方向のメッシュ分割数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetAltDimension()
        {
            return m_lAltDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>緯度方向のメッシュ分割数設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLatDimension(lLatDim) </para>
        /// </example>
        /// <param name="lLatDim">緯度方向のメッシュ分割数設定</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetLatDimension(long lLatDim)
        {
            m_lLatDim = lLatDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>経度方向のメッシュ分割数設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLonDimension(lLonDim) </para>
        /// </example>
        /// <param name="lLonDim">経度方向のメッシュ分割数</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetLonDimension(long lLonDim)
        {
            m_lLonDim = lLonDim;
        }
        /// <summary><para>method outline:</para>
        /// <para>高度方向のメッシュ分割数設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetAltDimension(lAltDim) </para>
        /// </example>
        /// <param name="lAltDim">高度方向のメッシュ分割数</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetAltDimension(long lAltDim)
        {
            m_lAltDim = lAltDim;
        }

        /// <summary><para>method outline:</para>
        /// <para>地理的位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLocation(dSWLat, dSWLon, dNELat, dNELon) </para>
        /// </example>
        /// <param name="dSWLat">南西頂点緯度（度単位）</param>
        /// <param name="dSWLon">南西頂点経度（度単位）</param>
        /// <param name="dNELat">北東頂点緯度（度単位）</param>
        /// <param name="dNELon">北東頂点経度（度単位）</param> 
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetLocation(double dSWLat, double dSWLon, double dNELat, double dNELon)
        {
            m_dSWLatitude = dSWLat;
            m_dSWLongitude = dSWLon;
            m_dNELatitude = dNELat;
            m_dNELongitude = dNELon;
        }
        /// <summary><para>method outline:</para>
        /// <para>地理的位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetLocation(ref dSWLat,ref dSWLon,ref dNELat,ref dNELon) </para>
        /// </example>
        /// <param name="dSWLat">南西頂点緯度（度単位）</param>
        /// <param name="dSWLon">南西頂点経度（度単位）</param>
        /// <param name="dNELat">北東頂点緯度（度単位）</param>
        /// <param name="dNELon">北東頂点経度（度単位）</param> 
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void GetLocation(ref double dSWLat, ref double dSWLon, ref double dNELat, ref double dNELon)
        {
            dSWLat = m_dSWLatitude;
            dSWLon = m_dSWLongitude;
            dNELat = m_dNELatitude;
            dNELon = m_dNELongitude;
        }
        /// <summary><para>method outline:</para>
        /// <para>高度位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetAltitude(dUpAlt, dLwAlt) </para>
        /// </example>
        /// <param name="dUpAlt">上限高度（ｍ単位）</param>
        /// <param name="dLwAlt">下限高度（ｍ単位）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetAltitude(double dUpAlt, double dLwAlt)
        {
            m_dUpperAltitude = dUpAlt;
            m_dLowerAltitude = dLwAlt;
        }
        /// <summary><para>method outline:</para>
        /// <para>高度位置取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HGetAltitude(ref dUpAlt, ref dLwAlt) </para>
        /// </example>
        /// <param name="dUpAlt">上限高度（ｍ単位）</param>
        /// <param name="dLwAlt">下限高度（ｍ単位）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void GetAltitude(ref double dUpAlt, ref double dLwAlt)
        {
            dUpAlt = m_dUpperAltitude;
            dLwAlt = m_dLowerAltitude;
        }

        /// <summary><para>method outline:</para>
        /// <para>表示パターン数設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetPatternNumber(lPatternNum) </para>
        /// </example>
        /// <param name="lPatternNum">表示パターン数</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetPatternNumber(long lPatternNum )
        {
            m_lPatternNum = lPatternNum;
            m_csPatternInf = new HySGeoMeshDispPattern[lPatternNum];
            for (long lLp = 0; lLp < lPatternNum; lLp++)
            {
                m_csPatternInf[lLp] = new HySGeoMeshDispPattern("");
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>表示パターン数取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lPtn = GetPatternNumber() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long 表示パターン数</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long GetPatternNumber()
        {
            return m_lPatternNum;
        }
        /// <summary><para>method outline:</para>
        /// <para>パターン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetPatternNumber(lOrderNo, csPatternInf) </para>
        /// </example>
        /// <param name="lOrderNo">パターン位置(０相対)</param>
        /// <param name="csPatternInf">パターン情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetPattern(long lOrderNo, HySGeoMeshDispPattern csPatternInf)
        {
            if (lOrderNo >= 0 && lOrderNo < m_lPatternNum)
            {
                m_csPatternInf[lOrderNo].DataCopy(csPatternInf);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>表示パターン取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGeoMeshDispPattern csPtnInf = GetPattern(lOrderNo) </para>
        /// </example>
        /// <param name="lOrderNo">パターン位置（０相対）</param>
        /// <returns>HySGeoMeshDispPattern 表示パターン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySGeoMeshDispPattern GetPattern(long lOrderNo)
        {
            HySGeoMeshDispPattern csRtn = null;
            if (lOrderNo >= 0 && lOrderNo < m_lPatternNum)
            {
                csRtn = m_csPatternInf[lOrderNo];
            }
            return csRtn;
        }

        ///// <summary><para>method outline:</para>
        ///// <para>引数で与えられたデータを自分にコピーする</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> bool bRtn = CopyData(csOrgTimeSeriesData) </para>
        ///// </example>
        ///// <param name="csOrgTimeSeriesData">コピー元情報</param>
        ///// <returns> 無し</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>無し</para>
        ///// </remarks>
        //public override void CopyData(HySTimeSeriesBase csOrgTimeSeriesData)
        //{
        //    HySGeoMeshSerial csOrgData = csOrgTimeSeriesData as HySGeoMeshSerial;
        //    if (csOrgData != null)
        //    {
        //        m_lMaxDataNum = csOrgData.m_lMaxDataNum;
        //        m_csLastTime = csOrgData.m_csLastTime.Clone();
        //        m_dSWLatitude = csOrgData.m_dSWLatitude;
        //        m_dSWLongitude = csOrgData.m_dSWLongitude;
        //        m_dNELatitude = csOrgData.m_dNELatitude;
        //        m_dNELongitude = csOrgData.m_dNELongitude;
        //        m_dUpperAltitude = csOrgData.m_dUpperAltitude;
        //        m_dLowerAltitude = csOrgData.m_dLowerAltitude;
        //        m_lMeshKind = csOrgData.m_lMeshKind;

        //        m_lLatDim = csOrgData.m_lLatDim;        
        //        m_lLonDim = csOrgData.m_lLonDim;
        //        m_lAltDim = csOrgData.m_lAltDim;

        //        m_lPatternNum = csOrgData.m_lPatternNum;
        //        m_csPatternInf = new HySGeoMeshDispPattern[m_lPatternNum];
        //        for( long lLp=0;lLp<m_lPatternNum;lLp++)
        //        {
        //            m_csPatternInf[lLp] = new HySGeoMeshDispPattern("");
        //            m_csPatternInf[lLp].DataCopy(csOrgData.m_csPatternInf[lLp]);
        //        }
        //        m_csCellDataKind = csOrgData.m_csCellDataKind.Clone();
        //    }
        //}

        /// <summary><para>method outline:</para>
        /// <para>共通情報をコピーする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CopyCommonInfo(csOrgData) </para>
        /// </example>
        /// <param name="csOrgData">コピー元情報</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void CopyCommonInfo(HySGeoMeshSerial csOrgData)
        {
            m_lMaxDataNum = csOrgData.m_lMaxDataNum;
            m_csLastTime = csOrgData.m_csLastTime.Clone();
            m_dSWLatitude = csOrgData.m_dSWLatitude;
            m_dSWLongitude = csOrgData.m_dSWLongitude;
            m_dNELatitude = csOrgData.m_dNELatitude;
            m_dNELongitude = csOrgData.m_dNELongitude;
            m_dUpperAltitude = csOrgData.m_dUpperAltitude;
            m_dLowerAltitude = csOrgData.m_dLowerAltitude;
            m_lMeshKind = csOrgData.m_lMeshKind;

            m_lLatDim = csOrgData.m_lLatDim;
            m_lLonDim = csOrgData.m_lLonDim;
            m_lAltDim = csOrgData.m_lAltDim;

            m_lPatternNum = csOrgData.m_lPatternNum;
            m_csPatternInf = new HySGeoMeshDispPattern[m_lPatternNum];
            for (long lLp = 0; lLp < m_lPatternNum; lLp++)
            {
                m_csPatternInf[lLp] = new HySGeoMeshDispPattern("");
                m_csPatternInf[lLp].DataCopy(csOrgData.m_csPatternInf[lLp]);
            }
            m_csCellDataKind = csOrgData.m_csCellDataKind.Clone();
            m_csCellIDMngData.CopyData(csOrgData.GetCellIDMngData());
        }
        /// <summary><para>method outline:</para>
        /// <para>共通情報をコピーする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CopyCommonInfo(csMeshData) </para>
        /// </example>
        /// <param name="csMeshData">コピー元地理メッシュ情報</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>引数で受け取った地理メッシュ情報を自身にコピーする</para>
        /// </remarks>
        protected void CopyCommonInfo(HySGeoMesh csMeshData)
        {
            // 座標情報
            csMeshData.GetLocation(ref m_dSWLatitude, ref m_dSWLongitude, ref m_dNELatitude, ref m_dNELongitude);

            // 標高情報
            csMeshData.GetAltitude(ref m_dUpperAltitude, ref m_dLowerAltitude);

            // メッシュ種別
            m_lMeshKind = csMeshData.GetDataMeshKind();

            // メッシュ分割数
            m_lLatDim = csMeshData.GetLatDimension();
            m_lLonDim = csMeshData.GetLonDimension();
            m_lAltDim = csMeshData.GetAltDimension();

            // パターン情報
            m_lPatternNum = csMeshData.GetPatternNumber();
            m_csPatternInf = new HySGeoMeshDispPattern[m_lPatternNum];
            for (long lLp = 0; lLp < m_lPatternNum; lLp++)
            {
                m_csPatternInf[lLp] = new HySGeoMeshDispPattern("");
                m_csPatternInf[lLp].DataCopy(csMeshData.GetPattern(lLp));
            }

            // セル情報
            m_csCellDataKind = csMeshData.GetCellDataCharacteristic().Clone();
            m_csCellIDMngData.CopyData(csMeshData.GetCellIDMngData());
        }

        /// <summary><para>method outline:</para>
        /// <para>セル内データ種別設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCellDataCharacteristic(csCellChara) </para>
        /// </example>
        /// <param name="csCellChara">１セル内の各データの種別</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetCellDataCharacteristic(HySDataCharacteristicInCell csCellChara)
        { m_csCellDataKind = csCellChara; }
        /// <summary><para>method outline:</para>
        /// <para>セル内データ種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataCharacteristicInCell csKind = GetCellDataCharacteristic();</para>
        /// <para> csKind.SetDataKind( lItemNo,  csLabel,  csDataKind,  csUnit) 等の処理</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataCharacteristicInCell セル内データ種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySDataCharacteristicInCell GetCellDataCharacteristic()
        { return m_csCellDataKind; }
        /*
        /// <summary><para>method outline:</para>
        /// <para>系列データの種別設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataMeaning(lDimItemNo, csMeaningKind) </para>
        /// </example>
        /// <param name="lDimItemNo">配列の順番（０相対）</param> 
        /// <param name="csMeaning">データの意味</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetDataMeaning(long lDimItemNo, HySObjectKind csMeaning)
        {
            if (lDimItemNo >= 0 && lDimItemNo < m_csCellDataKind.m_lDim)
            {
                m_csCellDataKind.m_csDataKind[lDimItemNo] = csMeaning.Clone();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>系列データの種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csDtMeaning = GetDataMeaning(lDimItemNo) </para>
        /// </example>
        /// <param name="lDimItemNo">配列の順番（０相対）</param> 
        /// <returns>HySObjectKind データの意味</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySObjectKind GetDataMeaning(long lDimItemNo)
        {
            HySObjectKind csRtn = null;
            if (lDimItemNo >= 0 && lDimItemNo < m_csCellDataKind.m_lDim)
            {
                csRtn = m_csCellDataKind.m_csDataKind[lDimItemNo];
            }

            return csRtn;
        }
        */

        /// <summary><para>method outline:</para>
        /// <para>セル識別管理データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCellIDMngData(csCellIDMngData) </para>
        /// </example>
        /// <param name="csCellIDMngData">セル識別管理データ</param> 
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void SetCellIDMngData(HySCellIDManageData csCellIDMngData)
        {
            m_csCellIDMngData = csCellIDMngData;
        }
        /// <summary><para>method outline:</para>
        /// <para>セル識別管理データ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCellIDManageData csCellIDMngDt = GetCellIDMngData( ) </para>
        /// </example>
        /// <param name=""></param> 
        /// <returns>HySCellIDManageData セル識別管理データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySCellIDManageData GetCellIDMngData()
        {
            return m_csCellIDMngData;
        }

        /// <summary><para>method outline:</para>
        /// <para>単独データ時系列メッシュ情報生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGeoMeshSerial csMeshSerial = CreateSingleDataMeshSerial(lDataIndex) </para>
        /// </example>
        /// <param name="lDataIndex">セル内データIndex(0相対)</param> 
        /// <returns>HySGeoMeshSerial 単独データ時系列メッシュ情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたインデックスのセル内部データのみを持つ時系列メッシュ情報データを生成する</para>
        /// </remarks>
        public HySGeoMeshSerial CreateSingleDataMeshSerial(long lDataIndex)
        {
            HySGeoMeshSerial csRtnData = null;

            // データがパターン保持のみの場合は作成不可能
            // Indexが正しい場合のみ生成
            if ((m_lMeshKind != HySGeoMesh.MESH_DATA_KIND.PATTARN_DATA_HOLD) &&
                (lDataIndex >= 0) && (lDataIndex < m_csCellDataKind.m_lDim))
            {
                // 自身の共通情報所持インスタンス生成
                csRtnData = CreateInstance();
                csRtnData.CopyCommonInfo(this);

                // セルデータ種別情報設定
                if (m_csCellDataKind != null)
                {
                    csRtnData.m_csCellDataKind = m_csCellDataKind.CreateSingleDataCharacteristic(lDataIndex);
                }

                // 時系列情報の設定
                long lDataNum = GetCount();
                SetCursorFirst();
                // 元のレコードを単独データにして設定する
                for (long lLp = 0; lLp < lDataNum; lLp++)
                {
                    HySGeoMesh csRecordData = GetCursorData() as HySGeoMesh;
                    csRtnData.AddData(csRecordData.CreateSingleDataMesh(lDataIndex));
                    MoveCursorNext();
                }
            }

            return csRtnData;
        }

        /// <summary><para>method outline:</para>
        /// <para>インスタンス生成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGeoMeshSerial csMeshSerial = CreateInstance() </para>
        /// </example>
        /// <param name="">無し</param> 
        /// <returns>HySGeoMeshSerial 生成されたクラスのインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>自分自身の同一の型のインスタンスを生成する</para>
        /// </remarks>
        protected abstract HySGeoMeshSerial CreateInstance();

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public new string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public new void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public override void PostDeserialize()
        {
            base.PostDeserialize();

            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        break;
                }
                SetNowVersion();
            }
        }
    }
}