﻿// <summary>ソースコード：>データ保存クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Xml;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.Interface.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSData
{
    /// <summary><para>class outline:</para>
    /// <para>データ保存クラス</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>無し</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/04/22][新規作成]</para>
    /// <para>Remarks:</para>
    /// <para>計算結果等をDBへ保存する場合には、本データに格納した後DBに登録する</para>
    /// </remarks>
    [Serializable]
    public class HySStockData : HySDataRoot, HySVersionManagement
    {
        /// <summary>登録データ本体</summary>
        protected HySDataRoot m_csStockData = null;

        /// <summary>データ識別子（必須）：一貫ユニークキー自動的に　生成される</summary>
        protected HySID m_csDataID = null;
        /// <summary>データの種別（必須）：何の情報か　検索キーとなる </summary>
        protected HySObjectKind m_csDataKind = null;

        //=============================================================================
        // シミュレーター種別とシミュレーターＩＤで　シミュレーターはユニークに決定される
        //=============================================================================
        /// <summary>シミュレーター種別（必須） 検索キーとなる</summary>
        protected HySObjectKind m_csSimKind = null;
        /// <summary>シミュレータープロジェクトＩＤ（必須） 検索キーとなる</summary>
        protected HySID m_csSimPrjctID = null;
        //=============================================================================

        /// <summary>データ地点ＩＤ（必須）：検索キーとなる</summary>
        protected HySID m_csPointID = null;

        /// <summary>登録名称：検索キーとなり得る</summary>
        protected HySString m_csSaveName = null;

        /// <summary>その他検索時にキーとなる情報の数</summary>
        protected long m_lKeyNumber = 0;
        /// <summary>その他検索時にキーとなる情報を管理するハッシュテーブル </summary>
        protected HySDataHashTable m_csKeyInfo = new HySDataHashTable();

        /// <summary>データ登録日時：閲覧のみ</summary>
        protected HySTime  m_csRegisertTime = null;
        /// <summary>備考：閲覧のみ</summary>
        protected HySString m_csRemark = null;

        /// <summary>保存期限定義</summary>
        public enum PeservedPeriod
        {
            /// <summary> 永久保存 </summary>
            Eternity = 1,
            //// <summary> 一定期間 </summary>
            //Limited,
            /// <summary> 一時的（Ｍａｘ１日） </summary>
            Temporary
        };

        /// <summary> 保存期限区分</summary>
        protected PeservedPeriod m_csPreservedPeriod = PeservedPeriod.Temporary;

        /// <summary>保存期限区分変更日</summary>
        /// <remarks>version 1.01で追加</remarks>
        protected HySTime m_csPreservedPeriodModifyTime = null;

        /// <summary> ロックオブジェクト(初期化時のユニークID生成に使用する)</summary>
        private static object m_csSync = new object();

        /// <summary> 最後に生成したID時刻</summary>
        private static DateTime m_csLastIDTime = DateTime.Now;
        /// <summary> 最後に生成したID枝番</summary>
        private static int m_csLastIDSuffix = 0;

        /// <summary><para>method outline:</para>
        /// <para>デフォルトコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySStockData csStockData = new HySStockData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySStockData  生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>使用禁止</para>
        /// </remarks>
        protected HySStockData()
        {
            this.Init();
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySStockData csData = new HySStockData(csSimKind,csSimID)</para>
        /// </example>
        /// <param name="csSimKind">ｼﾐｭﾚｰｼｮﾝ種別</param>
        /// <param name="csSimID">ｼﾐｭﾚｰｼｮﾝＩＤ</param>
        /// <returns>HySStockData  生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySStockData(HySObjectKind csSimKind, HySID csSimID) //, HySDataRoot csResistData)
        {
            this.Init();
            m_csSimKind = csSimKind.Clone();
            m_csSimPrjctID = csSimID.Clone();
            //バージョン情報設定
            SetNowVersion();

            //m_csSaveName = csSaveName;
            //m_csResistData = csResistData;
        }
        /*
        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySStockData csData = new HySStockData(sSaveName)</para>
        /// </example>
        /// <param name="sSaveName">名称</param>
        /// <returns>HySDepositData  生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySStockData(string sSaveName)
        {
            this.Init();
            m_csSaveName = new HySString(sSaveName);
            //m_csResistData = csResistData;
        }
        */

        /// <summary><para>method outline:</para>
        /// <para>初期化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Init() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void Init()
        {
            lock (m_csSync)
            {
                /*
                 * 高速化対応:
                 * システム時刻の解像力は十数ミリ秒であり、IDの時刻が変わるまで待つと
                 * 連続生成時に1秒間に数十個という不明瞭な制約が発生する。
                 * そのため、時刻＋枝番でIDを生成する。
                 */
                String csIDString;
                DateTime csIDTime = DateTime.Now;
                // 時刻が変わった場合、ID生成用時刻を再設定し、枝番をリセットする
                TimeSpan dt = csIDTime.Subtract(m_csLastIDTime);
                TimeSpan minDt = new TimeSpan(0, 0, 0, 0, 1);   // 最小時間差=1ms
                if (dt > minDt)
                {
                    m_csLastIDTime = csIDTime;
                    m_csLastIDSuffix = 0;
                }
                else
                {
                    ++m_csLastIDSuffix;   // 枝番を加算
                }
                // ID文字列を生成({年/月/日 時:分:秒.ミリ秒}-{枝番})
                csIDString = string.Format("{0:yyyy/MM/dd HH:mm:ss.fff}-{1:D6}", m_csLastIDTime, m_csLastIDSuffix);

                // オブジェクトのIDを設定する
                m_csDataID = new HySID(csIDString);

                /*
                 * 以下、旧ロジック
                 */
                ////String csIDString;
                ////while (true)
                ////{
                ////    csIDString = System.DateTime.Now.Millisecond.ToString() + System.DateTime.Now.ToString();
                ////    //String csIDString = HySCalendar.CreateNowTime().ToString();
                ////    if (csIDString != m_csLastIDString)
                ////    {
                ////        break;
                ////    }
                ////    System.Threading.Thread.Sleep(1);   // 1ms待ってリトライ
                ////}
                ////m_csLastIDString = csIDString;
                ////m_csDataID = new HySID(csIDString);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>識別子設定(DBAが使用します)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetID(csID) </para>
        /// </example>
        /// <param name="csID">識別子</param>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアル化オブジェクトではない形式(CSVなど)のデータベースからオブジェクトを復元する際に利用します。他の用途では利用しないでください。</para>
        /// </remarks>
        public virtual void SetID(HySID csID)
        {
            m_csDataID = csID;
        }

        /// <summary><para>method outline:</para>
        /// <para>識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csID = GetID() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID  識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetID()
        {
            return m_csDataID;
        }
        /// <summary><para>method outline:</para>
        /// <para>識別子比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = Equal(csID) </para>
        /// </example>
        /// <param name="csID">比較する種別識別子</param>
        /// <returns>Boolean　true:=一致, false:=不一致</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean Equal(HySID csID)
        {
            return m_csDataID.Equals(csID);
        }

        /// <summary><para>method outline:</para>
        /// <para>インスタンス複製</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csDataRoot = Clone() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataRoot データ複製インスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>複製に失敗した場合はnullが返却される</para>
        /// </remarks>
        public virtual HySDataRoot Clone()
        {
            HySStockData csStockData = new HySStockData(this.GetSimKind(), this.GetSimID());
            if (m_csPointID is HySID)
            {  // 地点ＩＤが有効な場合
                csStockData.SetPointID(m_csPointID);
            }
            if (m_csDataKind is HySObjectKind)
            {  // データ種別が有効な場合
                csStockData.SetDataKind(m_csDataKind);
            }
            if (m_csSaveName is HySString)
            {  // 名称が有効な場合
                csStockData.SetSaveName(m_csSaveName);
            }
            if (m_csRegisertTime is HySTime)
            {  // 登録日付が有効な場合
                csStockData.SetRegisertTime(m_csRegisertTime);
            }
            if (m_csPreservedPeriodModifyTime is HySTime)
            {  // 保存期限区分変更日が有効な場合
                csStockData.SetPreservedPeriodModifyTime(m_csPreservedPeriodModifyTime);
            }
            if (m_lKeyNumber > 0)
            {  // ユニーク検索キーが有る場合
                csStockData.SetKeyInfo(m_csKeyInfo);
            }
            if (m_csRemark is HySString)
            {  // 備考が有効な場合
                csStockData.SetRemark(m_csRemark);
            }
            if (m_csStockData is HySDataRoot)
            {  // データ本体が有効な場合
                csStockData.SetData(m_csStockData);
            }
            return csStockData;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csSimKind = GetSimKind( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySObjectKind シミュレーター種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySObjectKind GetSimKind()
        {
            return m_csSimKind;
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーターＩＤ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csSimID = GetSimID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID シミュレーターＩＤ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetSimID()
        {
            return m_csSimPrjctID;
        }

        /// <summary><para>method outline:</para>
        /// <para>データの種別設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataKind(csDataKind) </para>
        /// </example>
        /// <param name="csDataKind">データの種別</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDataKind(HySObjectKind csDataKind)
        {
            m_csDataKind = csDataKind.Clone();
        }
        /// <summary><para>method outline:</para>
        /// <para>データの種別取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csDataKind = GetDataKind( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySObjectKind データの種別</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySObjectKind GetDataKind()
        {
            return m_csDataKind;
        }

        /// <summary><para>method outline:</para>
        /// <para>データ地点ＩＤ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetPointID(csPointID) </para>
        /// </example>
        /// <param name="csPointID">データ地点ＩＤ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetPointID(HySID csPointID)
        {
            m_csPointID = csPointID.Clone();
        }
        /// <summary><para>method outline:</para>
        /// <para>データ地点ＩＤ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySID csSimID = GetPointID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySID データ地点ＩＤ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySID GetPointID()
        {
            return m_csPointID;
        }

        /// <summary><para>method outline:</para>
        /// <para>登録名称設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSaveName(csSaveName) </para>
        /// </example>
        /// <param name="csSaveName">登録名称</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSaveName(HySString csSaveName)
        {
            m_csSaveName = csSaveName.Clone();
        }
        /// <summary><para>method outline:</para>
        /// <para>登録名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csSaveName = GetSaveName( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString 登録名称</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySString GetSaveName()
        {
            return m_csSaveName;
        }

        /// <summary><para>method outline:</para>
        /// <para>データ登録日時設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetRegisertTime(csRegisertTime) </para>
        /// </example>
        /// <param name="csRegisertTime">データ登録日時</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetRegisertTime(HySTime csRegisertTime)
        {
            m_csRegisertTime = csRegisertTime.Clone();
        }
        /// <summary><para>method outline:</para>
        /// <para>データ登録日時取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csRegisertTime = GetRegisertTime( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime データ登録日時</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTime GetRegisertTime()
        {
            return m_csRegisertTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>保存期限区分変更日設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetPreservedPeriodModifyTime(csPreservedPeriodModifyTime) </para>
        /// </example>
        /// <param name="csPreservedPeriodModifyTime">保存期限区分変更日</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetPreservedPeriodModifyTime(HySTime csPreservedPeriodModifyTime)
        {
            m_csPreservedPeriodModifyTime = csPreservedPeriodModifyTime.Clone();
        }
        /// <summary><para>method outline:</para>
        /// <para>保存期限区分変更日取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csPreservedPeriodModifyTime = GetPreservedPeriodModifyTime( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime 保存期限区分変更日</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySTime GetPreservedPeriodModifyTime()
        {
            return m_csPreservedPeriodModifyTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>その他検索時にキーとなる情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetKeyInfo(csKeyInfo) </para>
        /// </example>
        /// <param name="csKeyInfo">その他検索時にキーとなる情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetKeyInfo(HySDataHashTable csKeyInfo)
        {
            // 全キー情報取得
            HySDataLinkedList csKeyList = csKeyInfo.GetKeyList();
            if (csKeyList is HySDataLinkedList)
            {  // 取得キー情報が有効な場合
                if (csKeyList.GetCount() > 0)
                {  // その他検索時にキーとなる情報が有る場合
                    csKeyList.SetCursorFirst();
                    for (int iLp = 0; iLp < csKeyList.GetCount(); iLp++)
                    {  // キー情報件数分繰り返す
                        HySString csPkey = (HySString)csKeyList.GetCursorData();
                        m_csKeyInfo.AddObject(csPkey, csKeyInfo.GetObject(csPkey));
                        m_lKeyNumber++;
                        csKeyList.MoveCursorNext();
                    }
                }
                csKeyList = null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>その他検索時にキーとなる情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataHashTable csKeyInfo = GetKeyInfo( ref lKeyNumber ) </para>
        /// </example>
        /// <param name="lKeyNumber">その他検索時にキーとなる情報の数</param>
        /// <returns>HySDataHashTable その他検索時にキーとなる情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySDataHashTable GetKeyInfo(ref long lKeyNumber)
        {
            lKeyNumber = m_lKeyNumber;
            return m_csKeyInfo;
        }

        /// <summary><para>method outline:</para>
        /// <para>備考設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetRemark(csRemark) </para>
        /// </example>
        /// <param name="csRemark">備考</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetRemark(HySString csRemark)
        {
            m_csRemark = csRemark.Clone();
        }
        /// <summary><para>method outline:</para>
        /// <para>備考取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csRemark = GetRemark( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySString 備考</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySString GetRemark()
        {
            return m_csRemark;
        }

        /// <summary><para>method outline:</para>
        /// <para>登録データ本体設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetData(csStockData) </para>
        /// </example>
        /// <param name="csStockData">登録データ本体</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>データ構造体を意識しない為、登録データ本体は複製を引き渡すこと</para>
        /// </remarks>
        public virtual void SetData(HySDataRoot csStockData)
        {
            m_csStockData = csStockData;
            //
            // データの種別を意識すれば複製可能
            //
            //try
            //{
            //    if (csStockData is HySHQData)
            //    {
            //        m_csStockData = ((HySHQData)csStockData).Clon();
            //    }
            //    else if (csStockData is xxxxxxx)
            //    {
            //    }
            //}
            //catch
            //{  // Clonメソッドが無い場合
            //    m_csStockData = csStockData;
            //}
        }
        /// <summary><para>method outline:</para>
        /// <para>登録データ本体取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csStockData = GetData( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataRoot 登録データ本体</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySDataRoot GetData()
        {
            return m_csStockData;
        }

        /// <summary><para>method outline:</para>
        /// <para>保存期限区分設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetPeservedPeriod(csPeservedPeriod) </para>
        /// </example>
        /// <param name="csPeservedPeriod">保存期限</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetPeservedPeriod(PeservedPeriod csPeservedPeriod)
        {
            m_csPreservedPeriod = csPeservedPeriod;
        }
        /// <summary><para>method outline:</para>
        /// <para>保存期限区分取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PeservedPeriod csPeservedPeriod = GetPeservedPeriod() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>PeservedPeriod 保存期限</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual PeservedPeriod GetPeservedPeriod()
        {
            return m_csPreservedPeriod;
        }

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================

        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.01";
        }

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                //旧バージョンから順にバージョンを上げて行く
                switch (GetDecodedVersion())
                {
                    case "1.00":
                        // データ登録日時が設定されている場合、保存期限区分変更日に同じ値を設定する
                        if (m_csRegisertTime != null)
                        {
                            m_csPreservedPeriodModifyTime = m_csRegisertTime.Clone();
                        }
                        break;
                }
                SetNowVersion();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
    }
}
