﻿// <summary>ソースコード：演算を行うクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.Interface.HSController;
using CommonMP.HYSSOP.Interface.HSSimulator;
using CommonMP.HYSSOP.Interface.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSController;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSSimulator
{
    /// <summary><para>class outline:</para>
    /// <para>シミュレーション演算抽象化クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    public abstract class HySSimulatorBase : HySSimulatorRoot
    {
        /// <summary>シミュレーター種別識別子</summary>
        protected HySObjectKind m_csKindID;
        /// <summary>演算インスタンス識別ＩＤ</summary>
        protected HySID m_csID;
        /// <summary>ファクトリクラス</summary>
        protected HySSimulatorFactory m_csSimulatorFactory;
        /// <summary>演算スレッド制御クラス</summary>
        protected HySSimThreadController m_csThreadCtl;


        /// <summary>シミュレーション開始時刻</summary>
        protected HySTime m_csStartTime;
        /// <summary>シミュレーション目標時刻</summary>
        protected HySTime m_csGoalTime;
        /// <summary>シミュレーション計算時刻</summary>
        protected HySTime m_csSimuTime;
        /// <summary>刻み時間</summary>
        protected HySTime m_csDltTime;

        /// <summary>データコンテナクラス</summary>
        protected HySSimulationDataContainer m_csDataContainer;

        /// <summary>計算状態</summary>
        protected STATUS m_eCalStatus = HySSimulatorBase.STATUS.DEFAULT_STATE;

        /// <summary>計算状態定義</summary>
        public enum STATUS
        { 
            /// <summary>デフォルト</summary>
            DEFAULT_STATE = -1,
            /// <summary>内部モデル生成済み状態</summary>
            MODEL_CONSTRUCTION_STATE ,
            /// <summary>初期化完了状態</summary>
            INITIAL_STATE,
            /// <summary>計算中状態</summary>
            CALCULATION_STATE,
            /// <summary>中断状態</summary>
            PAUSE_STATE,
            //// <summary>再開段階状態</summary>
            //RESTART_STATE,
            /// <summary>異常状態</summary>
            ABEND_STATE,
            /// <summary>計算完了状態</summary>
            END_STATE
        };


        // ====================================
        // 立ち上がり、終了
        // ====================================
        /// <summary><para>method outline:</para>
        /// <para>立ち上がり処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = Initialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>インスタンス生成後に呼ばれる処理</para>
        /// </remarks>
        public virtual Boolean Initialize()
        {
            // 計算状態を『デフォルト状態』に設定する【★重要】
            m_eCalStatus = HySSimulatorBase.STATUS.DEFAULT_STATE;
            return true;
        }
        /// <summary><para>method outline:</para>
        /// <para>終了処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySSimulator.Terminate( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean Terminate()
        {
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>内部演算モデル作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = ModelConstruction(csEvent) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract Boolean ModelConstruction(HySSysEvent csEvent);

        // ====================================
        // 演算実行
        // ====================================
        /// <summary><para>method outline:</para>
        /// <para>シミュレーションの初期化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdInitializeSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract Boolean CmdInitializeSimulation(HySSysEvent csEvent);

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の開始</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdStartSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract Boolean CmdStartSimulation(HySSysEvent csEvent);
 
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の中断</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdPauseSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract Boolean CmdPauseSimulation(HySSysEvent csEvent);

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の再開</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdRestartSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract Boolean CmdRestartSimulation(HySSysEvent csEvent);

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の強制終了</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmdStopSimulation( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract Boolean CmdStopSimulation(HySSysEvent csEvent);

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算の進捗を取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lPrg = GetSimulationProgress( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>進捗（０～１００％）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long GetSimulationProgress(HySSysEvent csEvent)
        {
            long lRtn = 0;

            double dSpan = 0.0;
            double dPass = 0.0;

            if (m_csGoalTime.GetGeologicalYear() == 0)
            {   // 一般年代ならば
                dSpan = m_csGoalTime.GetTime() - m_csStartTime.GetTime();
                dPass = m_csSimuTime.GetTime() - m_csStartTime.GetTime();
            }
            else
            {   // 地質年代ならば
                dSpan = m_csGoalTime.GetGeologicalYear() - m_csStartTime.GetGeologicalYear();
                dPass = m_csSimuTime.GetGeologicalYear() - m_csStartTime.GetGeologicalYear();
            }
            if (dSpan != 0.0)
            {
                lRtn = (long)(dPass / dSpan * 100.0);
            }
            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>任意イベント</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CmdArbitraryEvent( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント（必要ならば使用する）</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public abstract Boolean CmdArbitraryEvent(HySSysEvent csEvent);

        /// <summary><para>method outline:</para>
        /// <para>計算実行の終了</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = TerminateCalculation( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>具体的には計算を行っていたスレッドが終了する</para>
        /// </remarks>
        public virtual Boolean TerminateCalculation()
        {
            // 計算状態を『終了状態』に設定する【★重要】
            m_eCalStatus = HySSimulatorBase.STATUS.END_STATE;
            return true;
        }

        /* ★★フレームワーク側で行うため、本メソッドは不要★★
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター内にシミュレーションモデルを構築完了したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeModelConstruction( bSuccessFlg ) </para>
        /// </example>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual void NoticeModelConstruction(bool bSuccessFlg)
        {
            if (bSuccessFlg == true)
            {
                HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_OK_MODEL_CONSTRUCTION, m_csDataContainer);
                this.NoticeArbitraryEvent(csEvent);
            }
            else
            {
                HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_NG_MODEL_CONSTRUCTION, m_csDataContainer);
                this.NoticeArbitraryEvent(csEvent);
            }
        }
        */

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション初期化完了したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = NoticeSimulatorDelete(bSuccessFlg) </para>
        /// </example>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>必要ならば発生クラス側でオーバーライドする</para>
        /// </remarks>
        protected virtual void NoticeInitializeSimulation(bool bSuccessFlg)
        {
            HySEventObject csEvent = null;
            if (bSuccessFlg == true)
            {
                csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_INITIALIZE_SIMULATION, m_csDataContainer);
            }
            else
            {
                csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_CANNOT_INITIALIZE_SIMULATION, m_csDataContainer);
            }

            this.NoticeArbitraryEvent(csEvent);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算を開始したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeStartSimulation( bSuccessFlg ) </para>
        /// </example>
        /// <param name="bSuccessFlg">成功／失敗フラグ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticeStartSimulation(bool bSuccessFlg)
        {
            if (bSuccessFlg == true)
            {
                HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_START_SIMULATION, m_csDataContainer);
                this.NoticeArbitraryEvent(csEvent);
            }
            else
            {
                HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_CANNOT_START_SIMULATION, m_csDataContainer);
                this.NoticeArbitraryEvent(csEvent);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算を中断したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticePauseSimulation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticePauseSimulation()
        {
            HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_PAUSE_SIMULATION, m_csDataContainer);
            this.NoticeArbitraryEvent(csEvent);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算を強制終了したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeStopSimulation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticeStopSimulation()
        {
            HySEventObject csEvent = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_STOP_SIMULATION, m_csDataContainer);
            this.NoticeArbitraryEvent(csEvent);
        }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算完了したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeCompleteCalculation(  ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        /// 
        protected virtual void NoticeCompleteCalculation()
        {
            m_csThreadCtl.NoticeCompleteCalculation();
        } 
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算が異常終了したという報告</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeCalculationAbend( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void NoticeCalculationAbend()
        {
            m_csThreadCtl.NoticeCalculationAbend();
        }
        /// <summary><para>method outline:</para>
        /// <para>任意イベントを送る</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeArbitraryEvent( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送るイベント</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>自分からイベントを送信する時に使用する</para>
        /// </remarks>
        protected virtual void NoticeArbitraryEvent(HySSysEvent csEvent)
        {
            ((HySEventObject)csEvent).SetToSimKind(m_csKindID);
            m_csThreadCtl.PutEvent(csEvent);
        }


        /// <summary><para>method outline:</para>
        /// <para>シミュレーター間データ交換用コンテナに演算結果を出力する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = MediateDataOut( csDataContainer ) </para>
        /// </example>
        /// <param name="csDataContainer">シミュレーター間データ交換用コンテナ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>将来用：必要ならば派生クラスでオーバーライドする</para>
        /// </remarks>
        public virtual Boolean OutputMediateData(HySContainer csDataContainer)
        { return true; }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター間データ交換用コンテナから他のシミュレーターの演算結果を取り入れる</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = InputMediateData( csDataContainer ) </para>
        /// </example>
        /// <param name="csDataContainer">シミュレーター間データ交換用コンテナ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>将来用：必要ならば派生クラスでオーバーライドする</para>
        /// </remarks>
        public virtual Boolean InputMediateData(HySContainer csDataContainer)
        { return true; }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算内部情報を全て 引数で与えられたデータクラスに出力する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = SerializeAllInfo( csData ) </para>
        /// </example>
        /// <param name="csData">保存先データ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 計算中断した状態で内容を全て引数で与えられたデータクラス(csData)に保存しておく必要がある場合等に使用する。
        /// その必要が無い場合には何もしない。
        /// 上記処理が必要なクラスはオーバーライドする。
        /// </para>
        /// </remarks>
        public virtual Boolean SerializeAllInfo(HySDataRoot csData)
        { return true; }
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算内部情報を全て 引数で与えられたデータクラスから復元する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = StructureAllInfo( csData ) </para>
        /// </example>
        /// <param name="csData">保存データ</param>
        /// <returns>true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>SerializeAllInfo()により保存された計算中断した状態の情報を　復元する必要がある場合等に使用する
        /// 必要クラスはオーバーライドする
        /// </para>
        /// </remarks>
        public virtual Boolean StructureAllInfo(HySDataRoot csData)
        { return true; }

        // ====================================
        // 各種設定
        // ====================================
         
        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimKind(csSimKindID)</para>
        /// </example>
        /// <param name="csSimKindID">シミュレーター種別識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimKind(HySKind csSimKindID)
        {
            m_csKindID = (HySObjectKind)csSimKindID;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySKind csKind = GetSimKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>シミュレーター種別識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySKind GetSimKind()
        {
            return m_csKindID;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bVar = EqualSimKind(csSimKindID)</para>
        /// </example>
        /// <param name="csSimKindID">シミュレーター種別識別子</param>
        /// <returns>
        /// true  : 同じ, false : 異なる</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool EqualSimKind(HySKind csSimKindID)
        {
            return m_csKindID.Equals(csSimKindID);
        }

        /// <summary><para>method outline:</para>
        /// <para>識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetID( csID ) </para>
        /// </example>
        /// <param name="csID">識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetID(HySIdentifier csID)
        {
            m_csID = (HySID)csID;
        }

        /// <summary><para>method outline:</para>
        /// <para>識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySIdentifier csID = GetID( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySIdentifier GetID()
        {
            return m_csID;
        }

        /// <summary><para>method outline:</para>
        /// <para>識別子による同一比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> bool bRtn = Equal( sID ) </para>
        /// </example>
        /// <param name="sID">比較する種別識別子</param>
        /// <returns>  true :＝ , false:≠</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual bool Equal(HySIdentifier sID)
        {
            return m_csID.Equals(sID);
        }

        // ＝＝　シミュレーション時間関係管理　＝＝
        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算初期時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuStartTime( csTm )</para>
        /// </example>
        /// <param name="csTm">時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuStartTime(HySDate csTm)
        {
            m_csStartTime = (HySTime)csTm;
        }

        /// <summary><para>method outline:</para>
        /// <para>目標時間設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuGoalTime( csTm )</para>
        /// </example>
        /// <param name="csTm">時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuGoalTime(HySDate csTm)
        {
            m_csGoalTime = (HySTime)csTm;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーション計算時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuTIme( csTm )</para>
        /// </example>
        /// <param name="csTm">時刻</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuTime(HySDate csTm)
        {
            m_csSimuTime = (HySTime)csTm;
        }
        /// <summary><para>method outline:</para>
        /// <para>演算時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySDate csSimuTime = GetSimuTime()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>演算時刻</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySDate GetSimuTime()
        {
            return m_csSimuTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算目標時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SetSimuDeltaTime(csDltTime)</para>
        /// </example>
        /// <param name="csDltTime">演算刻み時間</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimuDeltaTime(HySDate csDltTime)
        {
            m_csDltTime = (HySTime)csDltTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーターファクトリクラス設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimulatorFactory(csSimulatorFactory)</para>
        /// </example>
        /// <param name="csSimulatorFactory">シミュレーターファクトリクラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimulatorFactory(HySSimulatorFactory csSimulatorFactory)
        {
            m_csSimulatorFactory = csSimulatorFactory;
        }
   
        /// <summary><para>method outline:</para>
        /// <para>コンテナ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetContainer( csContainer ) </para>
        /// </example>
        /// <param name="csContainer">コンテナクラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetContainer(HySSimulationContainer csContainer)
        {
            m_csDataContainer = (HySSimulationDataContainer)csContainer;
            this.SetID(m_csDataContainer.GetID());
        }

        /// <summary><para>method outline:</para>
        /// <para>コンテナ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySContainer csCntainer = GetContainer() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>コンテナインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySSimulationContainer GetContainer()
        {
            return m_csDataContainer;
        }

        /// <summary><para>method outline:</para>
        /// <para>演算スレッド制御クラス設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimThreadCtl( csThrdCtl ) </para>
        /// </example>
        /// <param name="csThrdCtl">演算スレッド制御クラス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimThreadCtl(HySSimThreadCtl csThrdCtl)
        {
            m_csThreadCtl = (HySSimThreadController)csThrdCtl;
        }

    }
}
