﻿// <summary>ソースコード：>ＨＹＳＳＯＰ暦操作クラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface;
using CommonMP.HYSSOP.CoreImpl.HSData;

namespace CommonMP.HYSSOP.CoreImpl.HSTools
{
    /// <summary><para>class outline:</para>
    /// <para>暦操作クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    public class HySCalendar : HySRoot
    {
        /// <summary>出力フォーマット区分定義</summary>
        public enum FORMAT
        {
            /// <summary>フォーマット区分（年月日付き）yyyy/MM/dd HH:mm:ss</summary>
            lSW_YEAR = 0,              //  年月日付き
            /// <summary>フォーマット区分（月日付き） MM/dd HH:mm:ss</summary>
            lSW_MONTH,             //　月日付き
            /// <summary>フォーマット区分（日時付き）dd HH:mm:ss</summary>
            lSW_DAY,               //　日時付き
            /// <summary>フォーマット区分（時間以下のみ）HH:mm:ss</summary>
            lSW_HOUR,              //  時間以下のみ
            /// <summary>フォーマット区分（ミリ秒以下の単位付き（年月はつかない）dd HH:mm:ss.xxxxxxx</summary>
            lSW_WITH_SUB_SECOND,   // ミリ秒以下の単位付き（年月はつかない）
            /// <summary>フォーマット区分（地質年（２０００年をゼロとして））yyyyyy </summary>
            lSW_GEOLOGICAL,        // 地質年（２０００年をゼロとして）
            /// <summary>フォーマット区分（地域設定に依存）</summary>
            lSW_LOCAL_YEAR,        // 地域設定に依存
        }

        /// <summary> 基準時刻 </summary>
        protected static readonly DateTime ms_csDTBase = DateTime.Parse("1/1/1970 0:0:0");

        /// <summary><para>method outline:</para>
        /// <para>指定時刻の時間クラスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = CreateTime(lYear, lMonth, lDate, lHour, lMinute, lSecond) </para>
        /// </example>
        /// <param name="lYear">年（西暦）但し、１９７０年以後を指定</param>
        /// <param name="lMonth">月（１～１２）</param>
        /// <param name="lDate">日（１～３１）</param>
        /// <param name="lHour">時間（０～２３）</param>
        /// <param name="lMinute">分（０～５９）</param>
        /// <param name="lSecond">秒（０～５９）</param>
        /// <returns>HySTime　時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static HySTime CreateTime(long lYear, long lMonth, long lDate, long lHour, long lMinute, long lSecond)
        {

            HySTime csRtnTime = new HySTime(0);
            try
            {
                DateTime csDT = new DateTime((int)lYear, (int)lMonth, (int)lDate, (int)lHour, (int)lMinute, (int)lSecond, 0);
                TimeSpan csTS = csDT - ms_csDTBase;
                csRtnTime.SetTime((long)csTS.TotalSeconds);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "CreateTime", csExp.Message);
            }
            return csRtnTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定時刻の時間クラスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = CreateTime(lYear, lMonth, lDate, lHour, lMinute, lSecond, dSubSecond) </para>
        /// </example>
        /// <param name="lYear">年（西暦）但し、１９７０年以後を指定</param>
        /// <param name="lMonth">月（１～１２）</param>
        /// <param name="lDate">日（１～３１）</param>
        /// <param name="lHour">時間（０～２３）</param>
        /// <param name="lMinute">分（０～５９）</param>
        /// <param name="lSecond">秒（０～５９）</param>
        /// <param name="dSubSecond">秒(1秒以下を小数点で設定　＜１)</param>
        /// <returns>HySTime　時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static HySTime CreateTime(long lYear, long lMonth, long lDate, long lHour, long lMinute, long lSecond, double dSubSecond)
        {

            HySTime csRtnTime = new HySTime(0);
            try
            {
                DateTime csDT = new DateTime((int)lYear, (int)lMonth, (int)lDate, (int)lHour, (int)lMinute, (int)lSecond, 0);
                TimeSpan csTS = csDT - ms_csDTBase;
                csRtnTime.SetTime((long)csTS.TotalSeconds);
                csRtnTime.SetSubSecondTime(dSubSecond);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "CreateTime", csExp.Message);
            }
            return csRtnTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定時刻の時間クラスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = CreateTime(sTimeString) </para>
        /// </example>
        /// <param name="sTimeString">時間文字列 例：　2008/12/03 22:04:07 </param>
        /// <returns>HySTime　時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static HySTime CreateTime(string sTimeString)
        {
            HySTime csRtnTime = null;
            try
            {
                DateTime csDT = DateTime.Parse(sTimeString);
                TimeSpan csTS = csDT - ms_csDTBase;
                csRtnTime = new HySTime(csTS.TotalSeconds);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "CreateTime", csExp.Message);
            }
            return csRtnTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>現在時刻の時間クラスを生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = CreateNowTime() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySTime　時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>インスタンス生成時のＰＣ上のシステム時間が設定される</para>
        /// </remarks> 
        public static HySTime CreateNowTime()
        {
            HySTime csRtnTime = null;
            try
            {
                DateTime csDT = DateTime.Now;
                TimeSpan csTS = csDT - ms_csDTBase;

                csRtnTime = new HySTime(csTS.TotalSeconds);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "CreateNowTime", csExp.Message);
            }

            return csRtnTime;
        }

        /// <summary><para>method outline:</para>
        /// <para>時間クラスに指定時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = SetTime(ref csTimeObj, lYear, lMonth, lDate, lHour, lMinute, lSecond) </para>
        /// </example>
        /// <param name="csTimeObj">時刻を設定したい時間クラス</param>
        /// <param name="lYear">年（西暦）但し、１９７０年以後を指定</param>
        /// <param name="lMonth">月（１～１２）</param>
        /// <param name="lDate">日（１～３１）</param>
        /// <param name="lHour">時間（０～２３）</param>
        /// <param name="lMinute">分（０～５９）</param>
        /// <param name="lSecond">秒（０～５９）</param>
        /// <returns>HySTime　設定された時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static HySTime SetTime(ref HySTime csTimeObj, long lYear, long lMonth, long lDate, long lHour, long lMinute, long lSecond)
        {
            try
            {
                DateTime csDT = new DateTime((int)lYear, (int)lMonth, (int)lDate, (int)lHour, (int)lMinute, (int)lSecond, 0);
                TimeSpan csTS = csDT - ms_csDTBase;
                csTimeObj.SetTime((long)csTS.TotalSeconds);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "SetTime", csExp.Message);
            }
            return csTimeObj;
        }

        /// <summary><para>method outline:</para>
        /// <para>時間クラスに指定時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTime(ref csTimeObj, sTimeString) </para>
        /// </example>
        /// <param name="csTimeObj">時刻を設定したい時間クラス</param>
        /// <param name="sTimeString">時間文字列 例：　2008/12/03 22:04:07 </param>
        /// <returns>HySTime　設定された時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static HySTime SetTime(ref HySTime csTimeObj, string sTimeString)
        {
            try
            {
                DateTime csDT = DateTime.Parse(sTimeString);
                TimeSpan csTS = csDT - ms_csDTBase;
                csTimeObj.SetTime((long)csTS.TotalSeconds);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "SetTime", csExp.Message);
            }

            return csTimeObj;
        }

        /// <summary><para>method outline:</para>
        /// <para>時間クラスに１秒以下の時刻を追加設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = SetSubSecond(ref csTimeObj, dSubSecond) </para>
        /// </example>
        /// <param name="csTimeObj">時刻を設定したい時間クラス</param>
        /// <param name="dSubSecond">秒(1秒以下を小数点で設定　＜１)</param>
        /// <returns>HySTime　設定された時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static HySTime SetSubSecond(ref HySTime csTimeObj, double dSubSecond)
        {
            csTimeObj.SetSubSecondTime(dSubSecond);

            return csTimeObj;
        }

        /// <summary><para>method outline:</para>
        /// <para>時間クラスに地質学的年を追加設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = SetGeologicalTime(ref csTimeObj, dYear) </para>
        /// </example>
        /// <param name="csTimeObj">時刻を設定したい時間クラス</param>
        /// <param name="dYear">２０００年を０としての地質学的経年を示す。負は、２０００年よりも過去</param>
        /// <returns>HySTime　設定された時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>この値を設定したときには、通常の時刻は無効となる</para>
        /// </remarks> 
        public static HySTime SetGeologicalTime(ref HySTime csTimeObj, double dYear)
        {
            csTimeObj.SetGeologicalYear(dYear);

            return csTimeObj;
        }

        /// <summary><para>method outline:</para>
        /// <para>現在時刻を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csTime = SetNow(ref csTimeObj) </para>
        /// </example>
        /// <param name="csTimeObj">時刻を設定したい時間クラス</param>
        /// <returns>HySTime　設定された時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>PC上のシステム時刻が設定される</para>
        /// </remarks> 
        public static HySTime SetNow(ref HySTime csTimeObj)
        {
            DateTime csDT = DateTime.Now;
            TimeSpan csTS = csDT - ms_csDTBase;
            csTimeObj.SetTime(csTS.TotalSeconds);
            return csTimeObj;
        }

        /// <summary><para>method outline:</para>
        /// <para>文字列化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> String csString = ToString(csTimeObj, lSW) </para>
        /// </example>
        /// <param name="csTimeObj">時間クラス</param>
        /// <param name="lSW">フォーマット区分</param>
        /// <returns>String　文字列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static String ToString(HySTime csTimeObj, FORMAT lSW)
        {
            HySString csRtn = HySCalendar.GetString(csTimeObj, lSW);
            return csRtn.ToString();
        }
        /// <summary><para>method outline:</para>
        /// <para>文字列化</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySString csString = GetString(csTimeObj, lSW) </para>
        /// </example>
        /// <param name="csTimeObj">時間クラス</param>
        /// <param name="lSW">フォーマット区分</param>
        /// <returns>HySString　文字列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static HySString GetString(HySTime csTimeObj, FORMAT lSW)
        {
            HySString csRtnString = new HySString();
            if ( (object)csTimeObj == null)
            {
                return csRtnString;
            }

            if (lSW == FORMAT.lSW_YEAR)
            {   //  年月日付き
                // （書式：2010/03/05 12:34:45）
                DateTime csDT = DateTime.Parse("1/1/1970 0:0:0");
                TimeSpan csTS = TimeSpan.FromMilliseconds(csTimeObj.GetTime() * 1000);
                csDT = csDT.Add(csTS);
                csRtnString.SetChar(csDT.ToString("yyyy/MM/dd HH:mm:ss"));
            }
            else if (lSW == FORMAT.lSW_MONTH)
            {   //  月日付き
                // （書式：03/05 12:34:45）
                DateTime csDT = DateTime.Parse("1/1/1970 0:0:0");
                TimeSpan csTS = TimeSpan.FromMilliseconds(csTimeObj.GetTime() * 1000);
                csDT = csDT.Add(csTS);
                csRtnString.SetChar(csDT.ToString("MM/dd HH:mm:ss"));
            }
            else if (lSW == FORMAT.lSW_DAY)
            {   //  日時付き
                // （書式：05 12:34:45）
                DateTime csDT = DateTime.Parse("1/1/1970 0:0:0");
                TimeSpan csTS = TimeSpan.FromMilliseconds(csTimeObj.GetTime() * 1000);
                csDT = csDT.Add(csTS);
                csRtnString.SetChar(csDT.ToString("dd HH:mm:ss"));
            }
            else if (lSW == FORMAT.lSW_HOUR)
            {   //  時間以下のみ
                // （書式：12:34:45）
                DateTime csDT = DateTime.Parse("1/1/1970 0:0:0");
                TimeSpan csTS = TimeSpan.FromMilliseconds(csTimeObj.GetTime() * 1000);
                csDT = csDT.Add(csTS);
                csRtnString.SetChar(csDT.ToString("HH:mm:ss"));
            }
            else if (lSW == FORMAT.lSW_WITH_SUB_SECOND)
            {   //  ミリ秒以下の単位付き（年月はつかない）
                // （書式：05 12:34:45.345）
                DateTime csDT = DateTime.Parse("1/1/1970 0:0:0");
                TimeSpan csTS = TimeSpan.FromMilliseconds(csTimeObj.GetTime() * 1000);
                csDT = csDT.Add(csTS);
                String sWork1 = csDT.ToString("dd HH:mm:ss");
                String sWork2 = Convert.ToString(csTimeObj.GetTime());
                csRtnString.SetChar(sWork1 + sWork2.Substring(sWork2.LastIndexOf('.'), 4));
            }
            else if (lSW == FORMAT.lSW_GEOLOGICAL)
            {   //  地質年（２０００年をゼロとして）
                // 地質年が　-10000 <  < +10000 ならば、　年の単位で出力　例：BC　3340年  , AD 2023年 
                // それ以上過去/未来ならば、２０００年を原点として　有効桁３桁で表す　例：21万5000年後、1億3400万年前
                double dWork = (double)((long)csTimeObj.GetGeologicalYear());
                if ((dWork > -10000.0) && (dWork < 10000.0))
                {
                    if (dWork >= -2000.0)
                    {
                        dWork += 2000.0;
                        csRtnString.SetChar("AD " + dWork.ToString() + "年");
                    }
                    else
                    {
                        dWork = dWork * -1.0 - 2000.0;
                        csRtnString.SetChar("BC " + dWork.ToString() + "年");
                    }
                }
                else
                {
                    string sWork = Convert.ToString(csTimeObj.GetGeologicalYear());
                    if (sWork.LastIndexOf('.') > -1)
                    {
                        sWork = sWork.Remove(sWork.LastIndexOf('.'));
                    }
                    string sWork2 = "";
                    string sEditWork = "";

                    if (sWork.Substring(0, 1) == "-")
                    {
                        sWork = sWork.Remove(0, 1);
                    }
                    if (sWork.Length > 8)
                    {
                        sWork2 = sWork.Substring(0, sWork.Length - 8);
                        sWork = sWork.Remove(0, sWork.Length - 8);
                        if (sWork2.Length >= 3)
                        {
                            sEditWork = sWork2.Substring(0, 3);
                            for (int iCount = 0; iCount < sWork2.Length - 3; iCount++)
                            {
                                sEditWork += "0";
                            }
                            sEditWork += "億";
                        }
                        else
                        {
                            sEditWork += sWork2;
                            sEditWork += "億";
                            if (sWork2.Length == 2)
                            {
                                if (sWork.Substring(0, 1) != "0")
                                {
                                    sEditWork += sWork.Substring(0, 1);
                                    sEditWork += "000";
                                    sEditWork += "万";
                                }
                            }
                            else
                            {
                                if (sWork.Substring(0, 2) != "00")
                                {
                                    sEditWork += sWork.Substring(0, 2);
                                    sEditWork += "00";
                                    sEditWork += "万";
                                }
                            }
                        }
                    }
                    else if (sWork.Length > 4)
                    {
                        sWork2 = sWork.Substring(0, sWork.Length - 4);
                        sWork = sWork.Remove(0, sWork.Length - 4);
                        if (sWork2.Length >= 3)
                        {
                            sEditWork += sWork2.Substring(0, 3);
                            for (int iCount = 0; iCount < sWork2.Length - 3; iCount++)
                            {
                                sEditWork += "0";
                            }
                            sEditWork += "万";
                        }
                        else
                        {
                            sEditWork += sWork2;
                            sEditWork += "万";
                            if (sWork2.Length == 2)
                            {
                                if (sWork.Substring(0, 1) != "0")
                                {
                                    sEditWork += sWork.Substring(0, 1);
                                    sEditWork += "000";
                                }
                            }
                            else
                            {
                                if (sWork.Substring(0, 2) != "00")
                                {
                                    sEditWork += sWork.Substring(0, 2);
                                    sEditWork += "00";
                                }
                            }
                        }
                    }
                    else
                    {
                        sEditWork += sWork;
                    }

                    if (dWork >= 0)
                    {
                        sEditWork += "年後";
                    }
                    else
                    {
                        sEditWork += "年前";
                    }
                    csRtnString.SetChar(sEditWork);
                }
            }
            else if (lSW == FORMAT.lSW_LOCAL_YEAR)
            {   //  年月日付き
                // （書式：2010/03/05 12:34:45）
                DateTime csDT = DateTime.Parse("1/1/1970 0:0:0");
                TimeSpan csTS = TimeSpan.FromMilliseconds(csTimeObj.GetTime() * 1000);
                csDT = csDT.Add(csTS);
                csRtnString.SetChar(csDT.ToString());
            } 
            return csRtnString;
        }

        /// <summary><para>method outline:</para>
        /// <para>現在時刻を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetNowTime(ref lYear, ref lMonth, ref lDay, ref lHour, ref lMin, ref lSec) </para>
        /// </example>
        /// <param name="lYear">年（西暦）</param>
        /// <param name="lMonth">月（１～１２）</param>
        /// <param name="lDay">日（１～３１）</param>
        /// <param name="lHour">時間（０～２３）</param>
        /// <param name="lMin">分（０～５９）</param>
        /// <param name="lSec">秒（０～５９）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static void GetNowTime(ref long lYear, ref long lMonth, ref long lDay,ref long lHour, ref long lMin, ref long lSec)
        {
            DateTime csDT = DateTime.Now;
            lYear = (long)csDT.Year;
            lMonth = (long)csDT.Month;
            lDay = (long)csDT.Day;
            lHour = (long)csDT.Hour;
            lMin = (long)csDT.Minute;
            lSec = (long)csDT.Second;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定時刻を分解して各要素を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetTime(csTimeObj,ref lYear, ref lMonth, ref lDay, ref lHour, ref lMin, ref lSec) </para>
        /// </example>
        /// <param name="csTimeObj">指定時刻</param>
        /// <param name="lYear">年（西暦）</param>
        /// <param name="lMonth">月（１～１２）</param>
        /// <param name="lDay">日（１～３１）</param>
        /// <param name="lHour">時間（０～２３）</param>
        /// <param name="lMin">分（０～５９）</param>
        /// <param name="lSec">秒（０～５９）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks> 
        public static void GetTime(HySTime csTimeObj, ref long lYear, ref long lMonth, ref long lDay, ref long lHour, ref long lMin, ref long lSec)
        {
            DateTime csDT = DateTime.Parse("1/1/1970 0:0:0");
            TimeSpan csTS = TimeSpan.FromMilliseconds(csTimeObj.GetTime() * 1000);
            csDT = csDT.Add(csTS);

            lYear = (long)csDT.Year;
            lMonth = (long)csDT.Month;
            lDay = (long)csDT.Day;
            lHour = (long)csDT.Hour;
            lMin = (long)csDT.Minute;
            lSec = (long)csDT.Second;
        }
        /// <summary><para>method outline:</para>
        /// <para>時間形式変換　DateTime形式→HySTime形式</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySTime csExTime = Exchange(csTime) </para>
        /// </example>
        /// <param name="csTime">DateTime形式の時間 </param>
        /// <returns>HySTime　変換済み時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>Java等へのコード移行時の利便性を考えて.net 固有の時間形式は通常使用しないことを奨励する</para>
        /// </remarks> 
        public static HySTime Exchange(DateTime csTime)
        {
            HySTime csRtnTime = new HySTime(0);
            try
            {
                TimeSpan csTS = csTime - ms_csDTBase;
                csRtnTime.SetTime((long)csTS.TotalSeconds);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "Exchange", csExp.Message);
            }
            return csRtnTime;
        }
        /// <summary><para>method outline:</para>
        /// <para>時間形式変換　HySTime形式→DateTime形式</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DateTime csExTime = Exchange(csTime) </para>
        /// </example>
        /// <param name="csTime">HySTime形式の時間 </param>
        /// <returns>DateTime　変換済み時間クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>Java等へのコード移行時の利便性を考えて.net 固有の時間形式は通常使用しないことを奨励する</para>
        /// </remarks> 
        public static DateTime Exchange( HySTime csTime)
        {
            DateTime csRtnTime = new DateTime();
            try
            {
                long lYear = 0; long lMonth = 0; long lDay = 0; long lHour = 0; long lMin = 0; long lSec = 0;
                HySCalendar.GetTime(csTime, ref  lYear, ref  lMonth, ref  lDay, ref  lHour, ref  lMin, ref  lSec);
                long lmilSec = (long)(1000.0*(csTime.GetTime() - (long)csTime.GetTime()));
                csRtnTime = new DateTime((int)lYear, (int)lMonth, (int)lDay, (int)lHour, (int)lMin, (int)lSec, (int)lmilSec);
            }
            catch (Exception csExp)
            {
                HySLog.LogOut(HySLog.TRIAL_RUN, "HySCalendar", "Exchange", csExp.Message);
            }
            return csRtnTime;
        }
    }
}
