// <summary>ソースコード：内挿を行わない（直近データ取得）処理クラス</summary>
// <author>CommonMP</author>
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;
//using CommonMP.HYSSOP.CoreImpl.HSData.CommonData;

namespace CommonMP.HYSSOP.CoreImpl.HSTools
{
    /// <summary><para>class outline:</para>
    /// <para>内挿処理を行わない（直近データ取得）クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/08/01][新規作成]</para>
    /// <para>remarks:</para>
    /// <para>
    /// 指定時刻に一番近い過去（一致時刻あるばあいはその時刻）のデータを取得する
    /// 指定時刻より過去の情報が無い場合にはNULLを返す
    /// </para>
    /// </remarks>
    [Serializable]
    public class HySIdleInterpolator : HySInterpolatorIF, HySVersionManagement
    {
        /// <summary> 内挿処理を行う時系列データ</summary>
        protected HySTimeSeriesBase m_csSeriesData = null;
        /// <summary> セル内の内挿処理対象データ</summary>
        protected HySCellDataGetter m_csDataIdx = null;

        /// <summary> 取得したい時刻 </summary>
        protected HySTime m_csCurrentTime = HySTime.DEFAULT_TIME.Clone();


        /// <summary>動作データ</summary>
        protected HySD1CellArrayData m_csCurrentD1Data = null;
        /// <summary>動作データ</summary>
        protected HySD2CellArrayData m_csCurrentD2Data = null;
        /// <summary>動作データ</summary>
        protected HySD3CellArrayData m_csCurrentD3Data = null;

        /// <summary> セルアレイの次元 </summary>
        protected long m_lArraylDimention = 0;

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>HySIdleInterpolator csObj = new HySIdleInterpolator()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySIdleInterpolator()
        {
            //バージョン情報設定
            SetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿処理対象伝送データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTimeSeriesData(csTimeSeriesData) </para>
        /// </example>
        /// <param name="csTimeSeriesData">対象時系列データ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetTimeSeriesData(HySTimeSeriesBase csTimeSeriesData)
        {
            m_csSeriesData = csTimeSeriesData;

            HySTimeSeriesD1CellArrayData csD1 = csTimeSeriesData as HySTimeSeriesD1CellArrayData;
            HySTimeSeriesD2CellArrayData csD2 = csTimeSeriesData as HySTimeSeriesD2CellArrayData;
            HySTimeSeriesD3CellArrayData csD3 = csTimeSeriesData as HySTimeSeriesD3CellArrayData;
            HySGeoDim1SerialData csGeoD1 = csTimeSeriesData as HySGeoDim1SerialData;
            HySGeoDim2MeshSerialData csGeoD2 = csTimeSeriesData as HySGeoDim2MeshSerialData;
            HySGeoDim3MeshSerialData csGeoD3 = csTimeSeriesData as HySGeoDim3MeshSerialData;
            if (csD1 != null) { m_lArraylDimention = 1; }
            else if (csD2 != null) { m_lArraylDimention = 2; }
            else if (csD3 != null) { m_lArraylDimention = 3; }
            else if (csGeoD1 != null) { m_lArraylDimention = 1; }
            else if (csGeoD2 != null) { m_lArraylDimention = 2; }
            else if (csGeoD3 != null) { m_lArraylDimention = 3; }


            m_csCurrentD1Data = null;
            m_csCurrentD2Data = null;
            m_csCurrentD3Data = null;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿処理対象データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTimeRecordData(csTimeRecordData) </para>
        /// </example>
        /// <param name="csTimeRecordData">対象データ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 瞬時値データ(HySTimeRecordIF)の場合でも　外部からは、
        /// 内挿したかの様に振舞わせたい場合に使用する</para>
        /// </remarks>
        public virtual void SetTimeRecordData(HySTimeRecordIF csTimeRecordData)
        {
            m_csCurrentD1Data = csTimeRecordData as HySD1CellArrayData;
            if (m_csCurrentD1Data != null) { m_lArraylDimention = 1; }
            if (csTimeRecordData is HySGeoDim1Data)
            {
                m_csCurrentD1Data = ((HySGeoDim1Data)csTimeRecordData).GetCellArray() as HySD1CellArrayData;
                m_lArraylDimention = 1;
            } 
            m_csCurrentD2Data = csTimeRecordData as HySD2CellArrayData;
            if (m_csCurrentD2Data != null) { m_lArraylDimention = 2; }
            if (csTimeRecordData is HySGeoDim2MeshData)
            {
                m_csCurrentD2Data = ((HySGeoDim2MeshData)csTimeRecordData).GetCellArray() as HySD2CellArrayData;
                m_lArraylDimention = 2;
            } 
            m_csCurrentD3Data = csTimeRecordData as HySD3CellArrayData;
            if (m_csCurrentD3Data != null) { m_lArraylDimention = 3; }
            if (csTimeRecordData is HySGeoDim3MeshData)
            {
                m_csCurrentD3Data = ((HySGeoDim3MeshData)csTimeRecordData).GetCellArray() as HySD3CellArrayData;
                m_lArraylDimention = 3;
            }

        }
        /// <summary><para>method outline:</para>
        /// <para>内挿処理を行うべきセル内データ配列のインデックス設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataIndex(csIdxDt) </para>
        /// </example>
        /// <param name="csIdxDt">内挿処理を行うべきセル内データ配列のインデックス設定クラス</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetDataIndex(HySCellDataGetter csIdxDt)
        {
            if (csIdxDt != null)
            {
                m_csDataIdx = csIdxDt;
            }
            else
            {
                m_csDataIdx = new HySCellDataGetter(0);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCurrentTime(csTime) </para>
        /// </example>
        /// <param name="csTime">内挿すべき時刻</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>設定した時刻に内挿する</para>
        /// </remarks>
        public virtual void SetCurrentTime(HySTime csTime)
        {
            m_csCurrentTime.SetTime(csTime);
            GetCurrentRecord();
        }

        //==========================================
        //  個別セルについて内挿処理を行う場合
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCell(bWithoutDataCheck) </para>
        /// </example>
        /// <param name="bWithoutDataCheck">=true:データチェック無し</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// bWithoutDataCheck=true の場合には、既にデータチェックが他の場所で行われている場合に使用する
        /// （データ整合性チェックによる演算速度減少を防ぐため）
        /// </para>
        /// </remarks>
        public virtual HySCellData GetCell(bool bWithoutDataCheck)
        {
            if (m_csCurrentD1Data != null)
            {
                if (bWithoutDataCheck == true)
                {
                    return m_csCurrentD1Data.GetCellData()[0];
                }
                else
                {
                    if (m_csCurrentD1Data.GetDimension1() > 0)
                    {
                        return m_csCurrentD1Data.GetCellData()[0];
                    }
                    else
                    {
                        return null;
                    }
                }
            }
            else
            {
                return null;
            }

        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCell(lDataIdx,bWithoutDataCheck) </para>
        /// </example>
        /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
        /// <param name="bWithoutDataCheck">=true:データチェック無し</param>
        /// <returns> double 内挿された指定インデックス情報　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// bWithoutDataCheck=true の場合には、既にデータチェックが他の場所で行われている場合に使用する
        /// （データ整合性チェックによる演算速度減少を防ぐため）
        /// </para>
        /// </remarks>
        public virtual double GetData(long lDataIdx, bool bWithoutDataCheck)
        {
            HySCellData csCell = this.GetCell(bWithoutDataCheck);
            if (csCell == null)
            {
                return HySDefine.dINVALID_VALUE;
            }
            else
            {
                if (bWithoutDataCheck == true)
                {
                    return csCell.m_dData[lDataIdx];
                }
                else
                {
                    if (lDataIdx >= 0 && lDataIdx < csCell.m_lDim)
                    {
                        return csCell.m_dData[lDataIdx];
                    }
                    else
                    {
                        return HySDefine.dINVALID_VALUE;
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1(lIdx1) </para>
        /// </example>
        /// <param name="lIdx1">第１配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySCellData GetCell(long lIdx1)
        {
            if (m_csCurrentD1Data != null)
            {
                return m_csCurrentD1Data.GetCell(lIdx1);
            }
            else
            {
                return null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1(lIdx1,lIdx2) </para>
        /// </example>
        /// <param name="lIdx1">第１配列のセル位置</param>
        /// <param name="lIdx2">第２配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySCellData GetCell(long lIdx1, long lIdx2)
        {
            if (m_csCurrentD2Data != null)
            {
                return m_csCurrentD2Data.GetCell(lIdx1,lIdx2);
            }
            else
            {
                return null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1(lIdx1,lIdx2,lIdx3) </para>
        /// </example>
        /// <param name="lIdx1">第１配列のセル位置</param>
        /// <param name="lIdx2">第２配列のセル位置</param>
        /// <param name="lIdx3">第３配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySCellData GetCell(long lIdx1, long lIdx2, long lIdx3)
        {
            if (m_csCurrentD2Data != null)
            {
                return m_csCurrentD3Data.GetCell(lIdx1, lIdx2, lIdx3);
            }
            else
            {
                return null;
            }
        }

        //==========================================
        //  セルの配列全体について内挿処理を行う場合
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellArrayD1() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySD1CellArrayData 内挿された情報を格納したセル配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySD1CellArrayData GetCellArrayD1()
        {
            return m_csCurrentD1Data;
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[] 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySCellData[] GetCellD1()
        {
            if (m_csCurrentD1Data != null)
            {
                return m_csCurrentD1Data.GetCellData();
            }
            else
            {
                return null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDataD1(lDataIdx) </para>
        /// </example>
        /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
        /// <returns> double[] 内挿された指定インデックス情報　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public virtual double[] GetDataD1(long lDataIdx)
        {
            double[] csRtnDt = null;
            HySCellData[] csCell = this.GetCellD1();
            if (csCell != null)
            {
                csRtnDt = new double[m_csCurrentD1Data.GetDimension1()];
                if (lDataIdx >= 0 && lDataIdx < m_csCurrentD1Data.GetDataDimentionInCell())
                {
                    for (long lLp = 0; lLp < m_csCurrentD1Data.GetDimension1(); lLp++)
                    {
                        csRtnDt[lLp] = csCell[lLp].m_dData[lDataIdx];
                    }
                }
            }
            return csRtnDt;
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellArrayD2() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySD2CellArrayData 内挿された情報を格納したセル配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD2CellArrayData,HySGeoDim2MeshSerialDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySD2CellArrayData GetCellArrayD2()
        {
            return m_csCurrentD2Data;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD2() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[] 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD2CellArrayData,HySGeoDim2MeshSerialDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySCellData[,] GetCellD2()
        {
            if (m_csCurrentD2Data != null)
            {
                return m_csCurrentD2Data.GetCellData();
            }
            else
            {
                return null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDataD2(lDataIdx) </para>
        /// </example>
        /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
        /// <returns> double[,]内挿された指定インデックス情報　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD2CellArrayData,HySGeoDim2MeshSerialDataの時に使用
        /// </para>
        /// </remarks>
        public virtual double[,] GetDataD2(long lDataIdx)
        {
            double[,] csRtnDt = null;

            HySCellData[,] csCell = this.GetCellD2();
            if (csCell != null)
            {
                csRtnDt = new double[m_csCurrentD2Data.GetDimension1(),m_csCurrentD2Data.GetDimension2()];
                if (lDataIdx >= 0 && lDataIdx < m_csCurrentD2Data.GetDataDimentionInCell())
                {
                    for (long lLp = 0; lLp < m_csCurrentD2Data.GetDimension1(); lLp++)
                    {
                        for (long lLp2 = 0; lLp2 < m_csCurrentD2Data.GetDimension2(); lLp2++)
                        {
                            csRtnDt[lLp,lLp2] = csCell[lLp,lLp2].m_dData[lDataIdx];
                        }
                    }
                }
            }
            return csRtnDt;
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellArrayD3() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySD3CellArrayData 内挿された情報を格納したセル配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD3CellArrayData,HySGeoDim3MeshSerialDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySD3CellArrayData GetCellArrayD3()
        {
            return m_csCurrentD3Data;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD3() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[] 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD3CellArrayData,HySGeoDim3MeshSerialDataの時に使用
        /// </para>
        /// </remarks>
        public virtual HySCellData[, ,] GetCellD3()
        {
            if (m_csCurrentD3Data != null)
            {
                return m_csCurrentD3Data.GetCellData();
            }
            else
            {
                return null;
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDataD3(lDataIdx) </para>
        /// </example>
        /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
        /// <returns> double[,,] 内挿された指定インデックス情報　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD3CellArrayData,HySGeoDim3MeshSerialDataの時に使用
        /// </para>
        /// </remarks>
        public virtual double[, ,] GetDataD3(long lDataIdx)
        {
            double[,,] csRtnDt = null;
            HySCellData[,,] csCell = this.GetCellD3();
            if (csCell != null)
            {
                csRtnDt = new double[m_csCurrentD3Data.GetDimension1(), m_csCurrentD3Data.GetDimension2(),m_csCurrentD3Data.GetDimension3()];
                if (lDataIdx >= 0 && lDataIdx < m_csCurrentD3Data.GetDataDimentionInCell())
                {
                    for (long lLp = 0; lLp < m_csCurrentD3Data.GetDimension1(); lLp++)
                    {
                        for (long lLp2 = 0; lLp2 < m_csCurrentD3Data.GetDimension2(); lLp2++)
                        {
                            for (long lLp3 = 0; lLp3 < m_csCurrentD3Data.GetDimension3(); lLp3++)
                            {
                                csRtnDt[lLp, lLp2,lLp3] = csCell[lLp, lLp2,lLp3].m_dData[lDataIdx];
                            }
                        }
                    }
                }
            }
            return csRtnDt;
        }

        /// <summary><para>method outline:</para>
        /// <para>対象レコード取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCurrentRecord() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 無し　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>対象時刻に一致または直前のレコードを取得する</para>
        /// </remarks>
        protected virtual void GetCurrentRecord()
        {

            if (m_csSeriesData != null)
            {
                m_csCurrentD1Data = null;
                m_csCurrentD2Data = null;
                m_csCurrentD3Data = null;
                long lRtn = m_csSeriesData.SetCursorMinus(this.m_csCurrentTime);   // カーソル位置を指定時刻直前（一致含む）に設定
                if( lRtn > 0 )
                {   // 一致または、直前のデータあり
                    HySTimeRecordIF csRcd = m_csSeriesData.GetCursorData();
                    m_csCurrentD1Data = csRcd as HySD1CellArrayData;
                    HySGeoDim1Data csGeo1D = csRcd as HySGeoDim1Data;
                    if (csGeo1D != null)
                    {
                        m_csCurrentD1Data = csGeo1D.GetCellArray() as HySD1CellArrayData;
                    }
                    m_csCurrentD2Data = csRcd as HySD2CellArrayData;
                    HySGeoDim2MeshData csGeo2D = csRcd as HySGeoDim2MeshData;
                    if (csGeo2D != null)
                    {
                        m_csCurrentD2Data = csGeo2D.GetCellArray() as HySD2CellArrayData;
                    }
                    m_csCurrentD3Data = csRcd as HySD3CellArrayData;
                    HySGeoDim3MeshData csGeo3D = csRcd as HySGeoDim3MeshData;
                    if (csGeo3D != null)
                    {
                        m_csCurrentD3Data = csGeo3D.GetCellArray() as HySD3CellArrayData;
                    }
                }
            } //end of if(m_csSeriesData)
        }
     

        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public virtual void PostDeserialize()
        {
            if (GetDecodedVersion() != GetNowVersion())
            {
                if (GetDecodedVersion() == null)
                {
                    // 旧バージョンからの移行処理を入れる
                }
                else
                {
                    //旧バージョンから順にバージョンを上げて行く
                    switch (GetDecodedVersion())
                    {
                        case "1.00":
                            {
                                // 旧バージョンからの移行処理を入れる
                                break;
                            }
                        //case "2.00": << 以後バージョンアップに伴って処理追加
                        //{
                        //    break;
                        //}
                    }                    
                }
                SetNowVersion();
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
        
    }
}
