// <summary>ソースコード：線形内挿処理クラス</summary>
// <author>CommonMP</author>
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.Serialization;

using CommonMP.HYSSOP.Interface;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;

namespace CommonMP.HYSSOP.CoreImpl.HSTools
{
    /// <summary><para>class outline:</para>
    /// <para>線形内挿処理クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/08/01][新規作成]</para>
    /// <para>remarks:</para>
    /// <para>
    /// 時系列情報の線形内挿処理を行う為のツールクラス
    ///   HySInterpolatorIF csInp = new HySLinearInterpolator();
    ///   csInp.SetTimeSeriesData(csTimeSeriesData);
    ///   csInp.SetDataIndex( csCellDataGetter );
    ///   csInp.SetCurrentTime(csTime);
    ///   csInp.GetCellArrayD1() or .GetCellArrayD2() or .GetCellArrayD3();
    ///     or
    ///   csInp.GetInterpolatedCell(lIdx1) or GetInterpolatedCell(lIdx1,lIdx2) or GetInterpolatedCell(lIdx1,lIdx2,lIdx3);
    /// </para>
    /// </remarks>
    [Serializable]
    public class HySLinearInterpolator : HySIdleInterpolator, HySVersionManagement
    {
        //// <summary> 内挿処理を行う時系列データ</summary>
        //protected HySTimeSeriesBase m_csSeriesData = null;
        //// <summary> セル内の内挿処理対象データ</summary>
        //protected HySCellDataGetter m_csDataIdx = null;

        //// <summary> 取得したい時刻 </summary>
        //protected HySTime m_csCurrentTime = new HySTime(0);
        /// <summary> 前時刻 </summary>
        protected HySTime m_csPrevTime = HySTime.DEFAULT_TIME.Clone();
        /// <summary> 後時刻 </summary>
        protected HySTime m_csAfterTime = HySTime.DEFAULT_TIME.Clone();

        /// <summary>動作データ:一度生成すると何回も使いまわす</summary>
        protected HySD1CellArrayData m_csWorkD1Data = null;
        /// <summary>動作データ:一度生成すると何回も使いまわす</summary>
        protected HySD2CellArrayData m_csWorkD2Data = null;
        /// <summary>動作データ:一度生成すると何回も使いまわす</summary>
        protected HySD3CellArrayData m_csWorkD3Data = null;

        /// <summary>動作データ</summary>
        protected HySD1CellArrayData m_csD1Prev = null;
        /// <summary>動作データ</summary>
        protected HySD1CellArrayData m_csD1After = null;
        /// <summary>動作データ</summary>
        protected HySD2CellArrayData m_csD2Prev = null;
        /// <summary>動作データ</summary>
        protected HySD2CellArrayData m_csD2After = null;
        /// <summary>動作データ</summary>
        protected HySD3CellArrayData m_csD3Prev = null;
        /// <summary>動作データ</summary>
        protected HySD3CellArrayData m_csD3After = null;


        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example>
        /// <para>usage:</para>
        /// <para>HySLinearInterpolator csObj = new HySLinearInterpolator()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks>
        /// <para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySLinearInterpolator()
        {
            //バージョン情報設定
            SetNowVersion();
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿処理対象伝送データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTimeSeriesData(csTimeSeriesData) </para>
        /// </example>
        /// <param name="csTimeSeriesData">対象時系列データ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetTimeSeriesData(HySTimeSeriesBase csTimeSeriesData)
        {
            base.SetTimeSeriesData(csTimeSeriesData);

            //if (m_lArraylDimention == 1)
            //{
            //    HySTimeSeriesD1CellArrayData csD1 = csTimeSeriesData as HySTimeSeriesD1CellArrayData;
            //    csD1.GetDataDimentionInCell();
            //    csD1.GetDimension1();
            //    //m_csWorkD1Data = new HySD1CellArrayData(,csD1.GetDataDimentionInCell());
            //}
            m_csWorkD1Data = null;
            m_csWorkD2Data = null;
            m_csWorkD3Data = null;

            m_csD1Prev = null;
            m_csD1After = null;
            m_csD2Prev = null;
            m_csD2After = null;
            m_csD3Prev = null;
            m_csD3After = null;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿処理対象データ設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetTimeRecordData(csTimeRecordData) </para>
        /// </example>
        /// <param name="csTimeRecordData">対象データ</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 瞬時値データ(HySTimeRecordIF)の場合でも　外部からは、
        /// 内挿したかの様に振舞わせたい場合に使用する</para>
        /// </remarks>
        public override void SetTimeRecordData(HySTimeRecordIF csTimeRecordData)
        {
            base.SetTimeRecordData(csTimeRecordData);

            m_csWorkD1Data = csTimeRecordData as HySD1CellArrayData;
            if (csTimeRecordData is HySGeoDim1Data)
            {
                m_csCurrentD1Data = ((HySGeoDim1Data)csTimeRecordData).GetCellArray() as HySD1CellArrayData;
                m_csWorkD1Data = ((HySGeoDim1Data)csTimeRecordData).GetCellArray() as HySD1CellArrayData;
                m_lArraylDimention = 1;
            }
            m_csWorkD2Data = csTimeRecordData as HySD2CellArrayData;
            if (csTimeRecordData is HySGeoDim2MeshData)
            {
                m_csCurrentD2Data = ((HySGeoDim2MeshData)csTimeRecordData).GetCellArray() as HySD2CellArrayData;
                m_csWorkD2Data = ((HySGeoDim2MeshData)csTimeRecordData).GetCellArray() as HySD2CellArrayData;
                m_lArraylDimention = 2;
            }
            m_csWorkD3Data = csTimeRecordData as HySD3CellArrayData;
            if (csTimeRecordData is HySGeoDim3MeshData)
            {
                m_csCurrentD3Data = ((HySGeoDim3MeshData)csTimeRecordData).GetCellArray() as HySD3CellArrayData;
                m_csWorkD3Data = ((HySGeoDim3MeshData)csTimeRecordData).GetCellArray() as HySD3CellArrayData;
                m_lArraylDimention = 3;
            }
            m_csD1Prev = m_csWorkD1Data;
            m_csD1After = m_csWorkD1Data;
            m_csD2Prev = m_csWorkD2Data;
            m_csD2After = m_csWorkD2Data;
            m_csD3Prev = m_csWorkD3Data;
            m_csD3After = m_csWorkD3Data;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿処理を行うべきセル内データ配列のインデックス設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetDataIndex(csIdxDt) </para>
        /// </example>
        /// <param name="csIdxDt">内挿処理を行うべきセル内データ配列のインデックス設定クラス</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public override void SetDataIndex(HySCellDataGetter csIdxDt)
        {
            if (csIdxDt != null)
            {
                m_csDataIdx = csIdxDt;
            }
            else
            {
                m_csDataIdx = new HySCellDataGetter(0);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿時刻設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCurrentTime(csTime) </para>
        /// </example>
        /// <param name="csTime">内挿すべき時刻</param>
        /// <returns> 無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>設定した時刻に内挿する</para>
        /// </remarks>
        public override void SetCurrentTime(HySTime csTime)
        {
            m_csCurrentTime.SetTime(csTime);
            this.GetCurrentRecord();
            
            if (m_lArraylDimention == 1 && m_csWorkD1Data == null)
            {
                HySTimeSeriesD1CellArrayData csD1 = m_csSeriesData as HySTimeSeriesD1CellArrayData;
                if (csD1 != null)
                {
                    m_csWorkD1Data = new HySD1CellArrayData(csD1.GetDimension1(), csD1.GetDataDimentionInCell());
                }
                else
                {
                    HySGeoDim1SerialData csGeoD1 = m_csSeriesData as HySGeoDim1SerialData;
                    if (csGeoD1 != null)
                    {
                        m_csWorkD1Data = new HySD1CellArrayData(csGeoD1.GetCellDimension(), csGeoD1.GetDataDimentionInCell());
                    }                
                }
            }
            else if (m_lArraylDimention == 2 && m_csWorkD2Data == null)
            {
                HySTimeSeriesD2CellArrayData csD2 = m_csSeriesData as HySTimeSeriesD2CellArrayData;
                if (csD2 != null)
                {
                    m_csWorkD2Data = new HySD2CellArrayData(csD2.GetDimension1(), csD2.GetDimension2(), csD2.GetDataDimentionInCell());
                }
                else
                {
                    HySGeoDim2MeshSerialData csGoeD2 = m_csSeriesData as HySGeoDim2MeshSerialData;
                    if (csGoeD2 != null)
                    {
                        m_csWorkD2Data = new HySD2CellArrayData(csGoeD2.GetLonDimension(), csGoeD2.GetLatDimension(), csGoeD2.GetDataDimentionInCell());
                    }
                }
            }
            else if (m_lArraylDimention == 3 && m_csWorkD3Data == null)
            {
                HySTimeSeriesD3CellArrayData csD3 = m_csSeriesData as HySTimeSeriesD3CellArrayData;
                if (csD3 != null)
                {
                     m_csWorkD3Data = new HySD3CellArrayData(csD3.GetDimension1(),csD3.GetDimension2(),csD3.GetDimension3(),csD3.GetDataDimentionInCell());
                }
                else
                {
                    HySGeoDim3MeshSerialData csGeoD3 = m_csSeriesData as HySGeoDim3MeshSerialData;
                    if (csGeoD3 != null)
                    {
                        m_csWorkD3Data = new HySD3CellArrayData(csGeoD3.GetLonDimension(), csGeoD3.GetLatDimension(), csGeoD3.GetAltDimension(), csGeoD3.GetDataDimentionInCell());
                    }
                }
           }

        }

        //==========================================
        //  個別セルについて内挿処理を行う場合
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCell(bWithoutDataCheck) </para>
        /// </example>
        /// <param name="bWithoutDataCheck">=true:データチェック無し</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// bWithoutDataCheck=true の場合には、既にデータチェックが他の場所で行われている場合に使用する
        /// （データ整合性チェックによる演算速度減少を防ぐため）
        /// </para>
        /// </remarks>
        public override HySCellData GetCell(bool bWithoutDataCheck)
        {
            return GetCell(0);
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCell(lDataIdx,bWithoutDataCheck) </para>
        /// </example>
        /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
        /// <param name="bWithoutDataCheck">=true:データチェック無し</param>
        /// <returns> double 内挿された指定インデックス情報　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// bWithoutDataCheck=true の場合には、既にデータチェックが他の場所で行われている場合に使用する
        /// （データ整合性チェックによる演算速度減少を防ぐため）
        /// </para>
        /// </remarks>
        public override double GetData(long lDataIdx, bool bWithoutDataCheck)
        {
            HySCellData csCell = this.GetCell(bWithoutDataCheck);
            if (csCell == null)
            {
                return HySDefine.dINVALID_VALUE;
            }
            else
            {
                if (bWithoutDataCheck == true)
                {
                    return csCell.m_dData[lDataIdx];
                }
                else
                {
                    if (lDataIdx >= 0 && lDataIdx < csCell.m_lDim)
                    {
                        return csCell.m_dData[lDataIdx];
                    }
                    else
                    {
                        return HySDefine.dINVALID_VALUE;
                    }
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1(lIdx1) </para>
        /// </example>
        /// <param name="lIdx1">第１配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public override HySCellData GetCell(long lIdx1)
        {
            HySCellData csRtn = null;
            double dPrvTime = m_csPrevTime.GetTime();
            double dCurrntTime = m_csCurrentTime.GetTime();
            double dAfterTime = m_csAfterTime.GetTime();

            // 内挿計算開始
            double dDltT = dAfterTime - dPrvTime;
            if (m_csD1Prev == null && m_csD1After != null)
            {
                m_csD1Prev = m_csD1After;
                dPrvTime = m_csAfterTime.GetTime();
                dDltT = 0;
            }
            if (m_csD1Prev != null)
            {   // 前データ有り
                if (dDltT > 0.0 && m_csD1After != null)
                {   // 内挿計算可能ならば
                    m_csCurrentD1Data = m_csWorkD1Data;
                    HySCellData csCrtCell = m_csCurrentD1Data.GetCell(lIdx1);
                    HySCellData csPrvCell = m_csD1Prev.GetCell(lIdx1);
                    HySCellData csAftCell = m_csD1After.GetCell(lIdx1);
                    long lDataIdx = -1; // データインデックス
                    for (long lLp = 0; lLp < m_csDataIdx.GetIdxNum(); lLp++)
                    {
                        lDataIdx = m_csDataIdx.GetIdx(lLp);
                        if (lDataIdx < 0) { continue; }

                        // 本クラス内では、データの入れ替えは行わない（但し、速度アップのため、後でデータ入れ替え後に使用しないデータについては、補間を行わない）
                        //csCrtCell.m_dData[lLp] = CalInterValue(
                        //                        csPrvCell.m_dData[lLp],
                        //                        csAftCell.m_dData[lLp],
                        //                        dPrvTime,
                        //                        dCurrntTime,
                        //                        dAfterTime);
                        csCrtCell.m_dData[lDataIdx] = CalInterValue(
                                                csPrvCell.m_dData[lDataIdx],
                                                csAftCell.m_dData[lDataIdx],
                                                dPrvTime,
                                                dCurrntTime,
                                                dAfterTime);
                    } //end of for(lLp)
                    csRtn = csCrtCell;
                }
                else
                {
                    csRtn = m_csD1Prev.GetCell(lIdx1);
                }
            }
            else
            {   // レコードゼロの場合
                if (m_csWorkD1Data != null)
                {
                    csRtn = m_csWorkD1Data.GetCell(lIdx1);
                }
                else
                {
                    // Do Nothing
                    //if (m_csD1After != null)
                    //{
                    //    csRtn = m_csD1After.GetCell(lIdx1);
                    //}
                    //m_csWorkD1Data = new HySD1CellArrayData();
                }
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1(lIdx1,lIdx2) </para>
        /// </example>
        /// <param name="lIdx1">第１配列のセル位置</param>
        /// <param name="lIdx2">第２配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public override HySCellData GetCell(long lIdx1, long lIdx2)
        {
            HySCellData csRtn = null;
            double dPrvTime = m_csPrevTime.GetTime();
            double dCurrntTime = m_csCurrentTime.GetTime();
            double dAfterTime = m_csAfterTime.GetTime();

            // 内挿計算開始
            double dDltT = dAfterTime - dPrvTime;
            if (m_csD2Prev == null && m_csD2After != null)
            {
                m_csD2Prev = m_csD2After;
                dPrvTime = m_csAfterTime.GetTime();
                dDltT = 0;
            }
            if (m_csD2Prev != null)
            {   // 前データ有り
                if (dDltT > 0.0 && m_csD2After != null)
                {   // 内挿計算可能ならば
                    m_csCurrentD2Data = m_csWorkD2Data;
                    HySCellData csCrtCell = m_csCurrentD2Data.GetCell(lIdx1, lIdx2);
                    HySCellData csPrvCell = m_csD2Prev.GetCell(lIdx1, lIdx2);
                    HySCellData csAftCell = m_csD2After.GetCell(lIdx1, lIdx2);
                    long lDataIdx = -1; // データインデックス
                    for (long lLp = 0; lLp < m_csDataIdx.GetIdxNum(); lLp++)
                    {
                        lDataIdx = m_csDataIdx.GetIdx(lLp);
                        if (lDataIdx < 0) { continue; }

                            // 本クラス内では、データの入れ替えは行わない（但し、速度アップのため、後でデータ入れ替え後に使用しないデータについては、補間を行わない）
                            //csCrtCell.m_dData[lLp] = CalInterValue(
                            //                        csPrvCell.m_dData[lLp],
                            //                        csAftCell.m_dData[lLp],
                            //                        dPrvTime,
                            //                        dCurrntTime,
                            //                        dAfterTime);
                            csCrtCell.m_dData[lDataIdx] = CalInterValue(
                                                    csPrvCell.m_dData[lDataIdx],
                                                    csAftCell.m_dData[lDataIdx],
                                                    dPrvTime,
                                                    dCurrntTime,
                                                    dAfterTime);

                    } //end of for(lLp)
                    csRtn = csCrtCell;
                }
                else
                {
                    csRtn = m_csD2Prev.GetCell(lIdx1, lIdx2);
                }
            }
            else
            {   // レコードゼロの場合
                if (m_csWorkD2Data != null)
                {
                    csRtn = m_csWorkD2Data.GetCell(lIdx1, lIdx2);
                }
                else
                {
                    // Do Nothing
                    //if (m_csD2After != null)
                    //{
                    //    csRtn = m_csD2After.GetCell(lIdx1, lIdx2);
                    //}
                }
            }
            return csRtn;
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1(lIdx1,lIdx2,lIdx3) </para>
        /// </example>
        /// <param name="lIdx1">第１配列のセル位置</param>
        /// <param name="lIdx2">第２配列のセル位置</param>
        /// <param name="lIdx3">第３配列のセル位置</param>
        /// <returns> HySCellData 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public override HySCellData GetCell(long lIdx1, long lIdx2, long lIdx3)
        {
            HySCellData csRtn = null;
            double dPrvTime = m_csPrevTime.GetTime();
            double dCurrntTime = m_csCurrentTime.GetTime();
            double dAfterTime = m_csAfterTime.GetTime();

            // 内挿計算開始
            double dDltT = dAfterTime - dPrvTime;
            if (m_csD3Prev == null && m_csD3After != null)
            {
                m_csD3Prev = m_csD3After;
                dPrvTime = m_csAfterTime.GetTime();
                dDltT = 0;
            }
            if (m_csD3Prev != null)
            {   // 前データ有り
                if (dDltT > 0.0 && m_csD3After != null)
                {   // 内挿計算可能ならば
                    m_csCurrentD3Data = m_csWorkD3Data;
                    HySCellData csCrtCell = m_csCurrentD3Data.GetCell(lIdx1, lIdx2, lIdx3);
                    HySCellData csPrvCell = m_csD3Prev.GetCell(lIdx1, lIdx2, lIdx3);
                    HySCellData csAftCell = m_csD3After.GetCell(lIdx1, lIdx2, lIdx3);
                    long lDataIdx = -1; // データインデックス
                    for (long lLp = 0; lLp < m_csDataIdx.GetIdxNum(); lLp++)
                    {
                        lDataIdx = m_csDataIdx.GetIdx(lLp);
                        if (lDataIdx < 0) { continue; }

                        // 本クラス内では、データの入れ替えは行わない（但し、速度アップのため、後でデータ入れ替え後に使用しないデータについては、補間を行わない）
                        //csCrtCell.m_dData[lLp] = CalInterValue(
                        //                        csPrvCell.m_dData[lLp],
                        //                        csAftCell.m_dData[lLp],
                        //                        dPrvTime,
                        //                        dCurrntTime,
                        //                        dAfterTime);
                        csCrtCell.m_dData[lDataIdx] = CalInterValue(
                                                csPrvCell.m_dData[lDataIdx],
                                                csAftCell.m_dData[lDataIdx],
                                                dPrvTime,
                                                dCurrntTime,
                                                dAfterTime);

                    } //end of for(lLp)
                    csRtn = csCrtCell;
                }
                else
                {
                    csRtn = m_csD3Prev.GetCell(lIdx1, lIdx2, lIdx3);
                }
            }
            else
            {   // レコードゼロの場合
                if (m_csWorkD3Data != null)
                {
                    csRtn = m_csWorkD3Data.GetCell(lIdx1, lIdx2, lIdx3);
                }
                else
                {
                    // Do Nothing
                    //if (m_csD3After != null)
                    //{
                    //    csRtn = m_csD3After.GetCell(lIdx1, lIdx2, lIdx3);
                    //}
                }
            }
            return csRtn;
        }

        //==========================================
        //  セルの配列全体について内挿処理を行う場合
        //==========================================

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellArrayD1() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySD1CellArrayData 内挿された情報を格納したセル配列　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public override HySD1CellArrayData GetCellArrayD1()
        {
            this.CalD1AllInterValue();
            return m_csCurrentD1Data;
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCellD1() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> HySCellData[] 内挿された情報を格納したセル　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public override HySCellData[] GetCellD1()
        {
            this.CalD1AllInterValue();
            if(m_csCurrentD1Data == null) { return null; }
            return m_csCurrentD1Data.GetCellData();
        }
        /// <summary><para>method outline:</para>
        /// <para>内挿結果取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDataD1(lDataIdx) </para>
        /// </example>
        /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
        /// <returns> double[] 内挿された指定インデックス情報　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>
        /// 内挿対象データが HySTimeSeriesD1CellArrayDataの時に使用
        /// </para>
        /// </remarks>
        public override double[] GetDataD1(long lDataIdx)
        {
            this.CalD1AllInterValue();
            return base.GetDataD1(lDataIdx);
        }
       
       /// <summary><para>method outline:</para>
       /// <para>内挿結果取得</para>
       /// </summary>
       /// <example><para>usage:</para>
       /// <para> GetCellArrayD2() </para>
       /// </example>
       /// <param name="">無し</param>
       /// <returns> HySD2CellArrayData 内挿された情報を格納したセル配列　</returns>
       /// <exception cref="">無し</exception>
       /// <remarks><para>remarks:</para>
       /// <para>
       /// 内挿対象データが HySTimeSeriesD2CellArrayData,HySGeoDim2MeshSerialDataの時に使用
       /// </para>
       /// </remarks>
       public override HySD2CellArrayData GetCellArrayD2()
       {
           this.CalD2AllInterValue();
           return m_csCurrentD2Data;
       }
       /// <summary><para>method outline:</para>
       /// <para>内挿結果取得</para>
       /// </summary>
       /// <example><para>usage:</para>
       /// <para> GetCellD2() </para>
       /// </example>
       /// <param name="">無し</param>
       /// <returns> HySCellData[] 内挿された情報を格納したセル　</returns>
       /// <exception cref="">無し</exception>
       /// <remarks><para>remarks:</para>
       /// <para>
       /// 内挿対象データが HySTimeSeriesD2CellArrayData,HySGeoDim2MeshSerialDataの時に使用
       /// </para>
       /// </remarks>
       public override HySCellData[,] GetCellD2()
       {
           this.CalD2AllInterValue();
           return m_csCurrentD2Data.GetCellData();
       }
       /// <summary><para>method outline:</para>
       /// <para>内挿結果取得</para>
       /// </summary>
       /// <example><para>usage:</para>
       /// <para> GetDataD2(lDataIdx) </para>
       /// </example>
       /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
       /// <returns> double[,]内挿された指定インデックス情報　</returns>
       /// <exception cref="">無し</exception>
       /// <remarks><para>remarks:</para>
       /// <para>
       /// 内挿対象データが HySTimeSeriesD2CellArrayData,HySGeoDim2MeshSerialDataの時に使用
       /// </para>
       /// </remarks>
       public override double[,] GetDataD2(long lDataIdx)
       {
           this.CalD2AllInterValue();
           return GetDataD2(lDataIdx);
       }

       /// <summary><para>method outline:</para>
       /// <para>内挿結果取得</para>
       /// </summary>
       /// <example><para>usage:</para>
       /// <para> GetCellArrayD3() </para>
       /// </example>
       /// <param name="">無し</param>
       /// <returns> HySD3CellArrayData 内挿された情報を格納したセル配列　</returns>
       /// <exception cref="">無し</exception>
       /// <remarks><para>remarks:</para>
       /// <para>
       /// 内挿対象データが HySTimeSeriesD3CellArrayData,HySGeoDim3MeshSerialDataの時に使用
       /// </para>
       /// </remarks>
       public override HySD3CellArrayData GetCellArrayD3()
       {
           this.CalD3AllInterValue();
           return m_csCurrentD3Data;
       }
       /// <summary><para>method outline:</para>
       /// <para>内挿結果取得</para>
       /// </summary>
       /// <example><para>usage:</para>
       /// <para> GetCellD3() </para>
       /// </example>
       /// <param name="">無し</param>
       /// <returns> HySCellData[] 内挿された情報を格納したセル　</returns>
       /// <exception cref="">無し</exception>
       /// <remarks><para>remarks:</para>
       /// <para>
       /// 内挿対象データが HySTimeSeriesD3CellArrayData,HySGeoDim3MeshSerialDataの時に使用
       /// </para>
       /// </remarks>
       public override HySCellData[, ,] GetCellD3()
       {
           this.CalD3AllInterValue();
           return m_csCurrentD3Data.GetCellData();
       }
       /// <summary><para>method outline:</para>
       /// <para>内挿結果取得</para>
       /// </summary>
       /// <example><para>usage:</para>
       /// <para> GetDataD3(lDataIdx) </para>
       /// </example>
       /// <param name="lDataIdx">セル内のデータインデックス（０相対）</param>
       /// <returns> double[,,] 内挿された指定インデックス情報　</returns>
       /// <exception cref="">無し</exception>
       /// <remarks><para>remarks:</para>
       /// <para>
       /// 内挿対象データが HySTimeSeriesD3CellArrayData,HySGeoDim3MeshSerialDataの時に使用
       /// </para>
       /// </remarks>
       public override double[, ,] GetDataD3(long lDataIdx)
       {
           this.CalD3AllInterValue();
           return GetDataD3(lDataIdx);
       }


        /// <summary><para>method outline:</para>
        /// <para>計算レコード取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetCurrentRecord() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 無し　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>対象時刻の前後のレコードを取得する</para>
        /// </remarks>
        protected override void GetCurrentRecord()
        {
            m_csCurrentD1Data = null;
            m_csCurrentD2Data = null;
            m_csCurrentD3Data = null;

            m_csD1Prev = null;
            m_csD1After = null;
            m_csD2Prev = null;
            m_csD2After = null;
            m_csD3Prev = null;
            m_csD3After = null;

            if (m_csDataIdx == null)
            {
                return;
            }

            if (m_csSeriesData != null)
            {
                if (m_csSeriesData.GetLastTime().Before(this.m_csCurrentTime) == true)
                {   // 伝送データの最終時刻がが現在値よりも前ならば
                    // 内挿処理は出来ない
                    base.GetCurrentRecord();

                    if (m_csCurrentD1Data != null)
                    {
                        m_csD1Prev = m_csCurrentD1Data;
                    }
                    if (m_csCurrentD2Data != null)
                    {
                        m_csD2Prev = m_csCurrentD2Data;
                    }
                    if (m_csCurrentD3Data != null)
                    {
                        m_csD3Prev = m_csCurrentD3Data;
                    }
                    return;
                }

                m_csPrevTime.SetTime(HySTime.DEFAULT_TIME);
                m_csAfterTime.SetTime(HySTime.DEFAULT_TIME);



                long lRtn = m_csSeriesData.SetCursorMinus(this.m_csCurrentTime);   // カーソル位置を指定時刻直前（一致含む）に設定
                if (lRtn > 0)
                {   // 一致または、直前のデータあり
                    HySTimeRecordIF csRcd = m_csSeriesData.GetCursorData();
                    m_csPrevTime.SetTime(csRcd.GetLastTime());

                    m_csD1Prev = csRcd as HySD1CellArrayData;
                    if (m_csWorkD1Data == null && m_csD1Prev != null)
                    {
                        m_csWorkD1Data = new HySD1CellArrayData(m_csD1Prev);
                        m_csDataIdx = new HySCellDataGetter(m_csD1Prev.GetDataDimentionInCell());
                    }

                    HySGeoDim1Data csGeo1D = csRcd as HySGeoDim1Data;
                    if (csGeo1D != null)
                    {
                        m_csD1Prev = csGeo1D.GetCellArray() as HySD1CellArrayData;
                        if (m_csWorkD1Data == null && m_csD1Prev != null)
                        {
                            m_csWorkD1Data = new HySD1CellArrayData(m_csD1Prev);
                            m_csDataIdx = new HySCellDataGetter(m_csD1Prev.GetDataDimentionInCell());
                        }
                    }

                    m_csD2Prev = csRcd as HySD2CellArrayData;
                    if (m_csWorkD2Data == null && m_csD2Prev != null)
                    {
                        m_csWorkD2Data = new HySD2CellArrayData(m_csD2Prev);
                        m_csDataIdx = new HySCellDataGetter(m_csD2Prev.GetDataDimentionInCell());
                    }

                    HySGeoDim2MeshData csGeo2D = csRcd as HySGeoDim2MeshData;
                    if (csGeo2D != null)
                    {
                        m_csD2Prev = csGeo2D.GetCellArray() as HySD2CellArrayData;
                        if (m_csWorkD2Data == null && m_csD2Prev != null)
                        {
                            m_csWorkD2Data = new HySD2CellArrayData(m_csD2Prev);
                            m_csDataIdx = new HySCellDataGetter(m_csD2Prev.GetDataDimentionInCell());
                        }
                    }

                    m_csD3Prev = csRcd as HySD3CellArrayData;
                    if (m_csWorkD3Data == null && m_csD3Prev != null)
                    {
                        m_csWorkD3Data = new HySD3CellArrayData(m_csD3Prev);
                        m_csDataIdx = new HySCellDataGetter(m_csD3Prev.GetDataDimentionInCell());
                    }

                    HySGeoDim3MeshData csGeo3D = csRcd as HySGeoDim3MeshData;
                    if (csGeo3D != null)
                    {
                        m_csD3Prev = csGeo3D.GetCellArray() as HySD3CellArrayData;
                        if (m_csWorkD3Data == null && m_csD3Prev != null)
                        {
                            m_csWorkD3Data = new HySD3CellArrayData(m_csD3Prev);
                            m_csDataIdx = new HySCellDataGetter(m_csD3Prev.GetDataDimentionInCell());
                        }
                    }
                }

                lRtn = m_csSeriesData.SetCursor(this.m_csCurrentTime);   // カーソル位置を指定時刻直後（一致含む）に設定
                if (lRtn > 0)
                {   // 一致または、直後のデータあり
                    HySTimeRecordIF csRcd = m_csSeriesData.GetCursorData();
                    m_csAfterTime.SetTime(csRcd.GetLastTime());
                    m_csD1After = csRcd as HySD1CellArrayData;
                    m_csD2After = csRcd as HySD2CellArrayData;
                    HySGeoDim1Data csGeo1D = csRcd as HySGeoDim1Data;
                    if (csGeo1D != null)
                    {
                        m_csD1After = csGeo1D.GetCellArray() as HySD1CellArrayData;
                    }
                    HySGeoDim2MeshData csGeo2D = csRcd as HySGeoDim2MeshData;
                    if (csGeo2D != null)
                    {
                        m_csD2After = csGeo2D.GetCellArray() as HySD2CellArrayData;
                    }
                    m_csD3After = csRcd as HySD3CellArrayData;
                    HySGeoDim3MeshData csGeo3D = csRcd as HySGeoDim3MeshData;
                    if (csGeo3D != null)
                    {
                        m_csD3After = csGeo3D.GetCellArray() as HySD3CellArrayData;
                    }
                }
                /*
                // 内挿計算用時刻
                double dPrvTime = 0.0;
                double dCurrntTime = m_csCurrentTime.GetTime();
                double dAfterTime = 0.0;
                // 内挿計算開始
                double dDltT = dAfterTime - dPrvTime;
                if (m_csD1Prev != null)
                {   // 前データ有り

                    //m_csCurrentD1Data = new HySD1CellArrayData(m_csD1Prev);
                    if (dDltT > 0.0 && m_csD1After != null)
                    {   // 内挿計算可能ならば
                        m_csCurrentD1Data = m_csWorkD1Data;
                        CalD1InterValue(dPrvTime, dCurrntTime, dAfterTime); // １次元セル内挿計算
                    }
                    else
                    {
                        m_csCurrentD1Data = m_csD1Prev;
                    }
                }
                else if (m_csD2Prev != null)
                {   // 前データ有り

                    //m_csCurrentD2Data = new HySD2CellArrayData(m_csD2Prev);
                    if (dDltT > 0.0 && m_csD2After != null)
                    {   // 内挿計算可能ならば
                        m_csCurrentD2Data = m_csWorkD2Data;
                        CalD2InterValue(dPrvTime, dCurrntTime, dAfterTime); // ２次元セル内挿計算
                    }
                    else
                    {
                        m_csCurrentD2Data = m_csD2Prev;
                    }
                }
                else if (m_csD3Prev != null)
                {   // 前データ有り

                    //m_csCurrentD3Data = new HySD3CellArrayData(m_csD3Prev);
                    if (dDltT > 0.0 && m_csD3After != null)
                    {   // 内挿計算可能ならば
                        m_csCurrentD3Data = m_csWorkD3Data;
                        CalD3InterValue(dPrvTime, dCurrntTime, dAfterTime); // ３次元セル内挿計算
                    }
                    else
                    {
                        m_csCurrentD3Data = m_csD3Prev;
                    }
                }
                */
            }
            else
            {
                m_csCurrentD1Data = this.m_csWorkD1Data;
                m_csCurrentD2Data = this.m_csWorkD2Data;
                m_csCurrentD3Data = this.m_csWorkD3Data;
                m_csD1Prev = this.m_csWorkD1Data;
                m_csD1After = this.m_csWorkD1Data;
                m_csD2Prev = this.m_csWorkD2Data;
                m_csD2After = this.m_csWorkD2Data;
                m_csD3Prev = this.m_csWorkD3Data;
                m_csD3After = this.m_csWorkD3Data;

            }

        }

             
        /// <summary><para>method outline:</para>
        /// <para>全１次元セル内挿計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalD1AllInterValue( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 無し　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void CalD1AllInterValue()
        {
            // 内挿計算用時刻
            double dPrvTime = m_csPrevTime.GetTime();
            double dCurrntTime = m_csCurrentTime.GetTime();
            double dAfterTime = m_csAfterTime.GetTime();
            // 内挿計算開始
            double dDltT = dAfterTime - dPrvTime;
            if (m_csD1Prev != null)
            {   // 前データ有り
                if (dDltT > 0.0 && m_csD1After != null)
                {   // 内挿計算可能ならば
                    m_csCurrentD1Data = m_csWorkD1Data;
                    CalD1InterValue(dPrvTime, dCurrntTime, dAfterTime); // １次元セル内挿計算
                }
                else
                {
                    m_csCurrentD1Data = m_csD1Prev;
                }
            }
            else
            {   // 万が一過去のデータが無ければ
                if (m_csD1After != null)
                {    // 過去は無くても未来がある場合には未来を返す
                    m_csCurrentD1Data = m_csD1After;
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>１次元セル内挿計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalD1InterValue(dPrvTime,dCurrntTime,dAfterTime) </para>
        /// </example>
        /// <param name="dPrvTime">直前時刻</param>
        /// <param name="dCurrntTime">指定時刻</param>
        /// <param name="dAfterTime">直後時刻</param>
        /// <returns> double　計算された内挿値　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void CalD1InterValue(
            double dPrvTime,
            double dCurrntTime,
            double dAfterTime)
        {
            HySCellData[] csCrtCell = m_csCurrentD1Data.GetCellData();
            HySCellData[] csPrvCell = m_csD1Prev.GetCellData();
            HySCellData[] csAftCell = m_csD1After.GetCellData();

            long lDataIdx = -1; // データインデックス
            for (long lLp = 0; lLp < m_csDataIdx.GetIdxNum(); lLp++)
            {
                lDataIdx = m_csDataIdx.GetIdx(lLp);
                if (lDataIdx < 0) { continue; }
                for (long lCll = 0; lCll < m_csCurrentD1Data.GetDimension1(); lCll++)
                {
                    // 本クラス内では、データの入れ替えは行わない（但し、速度アップのため、後でデータ入れ替え後に使用しないデータについては、補間を行わない）
                    //csCrtCell[lCll].m_dData[lLp] = CalInterValue(
                    //                        csPrvCell[lCll].m_dData[lLp],
                    //                        csAftCell[lCll].m_dData[lLp],
                    //                        dPrvTime,
                    //                        dCurrntTime,
                    //                        dAfterTime);
                    csCrtCell[lCll].m_dData[lDataIdx] = CalInterValue(
                                            csPrvCell[lCll].m_dData[lDataIdx],
                                            csAftCell[lCll].m_dData[lDataIdx],
                                            dPrvTime,
                                            dCurrntTime,
                                            dAfterTime);

                }// end of for(セル)
            } //end of for(DataIndex)
        }

        /// <summary><para>method outline:</para>
        /// <para>全２次元セル内挿計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalD2AllInterValue( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 無し　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void CalD2AllInterValue()
        {
            // 内挿計算用時刻
            double dPrvTime = m_csPrevTime.GetTime();
            double dCurrntTime = m_csCurrentTime.GetTime();
            double dAfterTime = m_csAfterTime.GetTime();
            // 内挿計算開始
            double dDltT = dAfterTime - dPrvTime;
                            
            if (m_csD2Prev != null)
            {   // 前データ有り

                //m_csCurrentD2Data = new HySD2CellArrayData(m_csD2Prev);
                if (dDltT > 0.0 && m_csD2After != null)
                {   // 内挿計算可能ならば
                    m_csCurrentD2Data = m_csWorkD2Data;
                    CalD2InterValue(dPrvTime, dCurrntTime, dAfterTime); // ２次元セル内挿計算
                }
                else
                {
                    m_csCurrentD2Data = m_csD2Prev;
                }
            }
            else
            {   // 万が一過去のデータが無ければ
                if (m_csD2After != null)
                {    // 過去は無くても未来がある場合には未来を返す
                    m_csCurrentD2Data = m_csD2After;
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>２次元セル内挿計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalD2InterValue(dPrvTime,dCurrntTime,dAfterTime) </para>
        /// </example>
        /// <param name="dPrvTime">直前時刻</param>
        /// <param name="dCurrntTime">指定時刻</param>
        /// <param name="dAfterTime">直後時刻</param>
        /// <returns> double　計算された内挿値　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void CalD2InterValue(
            double dPrvTime,
            double dCurrntTime,
            double dAfterTime)
        {
            HySCellData[,] csCrtCell = m_csCurrentD2Data.GetCellData();
            HySCellData[,] csPrvCell = m_csD2Prev.GetCellData();
            HySCellData[,] csAftCell = m_csD2After.GetCellData();

            long lDataIdx = -1; // データインデックス
            for (long lLp = 0; lLp < m_csDataIdx.GetIdxNum(); lLp++)
            {
                lDataIdx = m_csDataIdx.GetIdx(lLp);
                if (lDataIdx < 0) { continue; }
                for (long lCll = 0; lCll < m_csCurrentD2Data.GetDimension1(); lCll++)
                {
                    for(long lCll2=0;lCll2<m_csCurrentD2Data.GetDimension2();lCll2++)
                    {
                        // 本クラス内では、データの入れ替えは行わない（但し、速度アップのため、後でデータ入れ替え後に使用しないデータについては、補間を行わない）
                        //csCrtCell[lCll, lCll2].m_dData[lLp] = CalInterValue(
                        //                        csPrvCell[lCll, lCll2].m_dData[lLp],
                        //                        csAftCell[lCll, lCll2].m_dData[lLp],
                        //                        dPrvTime,
                        //                        dCurrntTime,
                        //                        dAfterTime);
                        csCrtCell[lCll, lCll2].m_dData[lDataIdx] = CalInterValue(
                                                csPrvCell[lCll, lCll2].m_dData[lDataIdx],
                                                csAftCell[lCll, lCll2].m_dData[lDataIdx],
                                                dPrvTime,
                                                dCurrntTime,
                                                dAfterTime);

                    }
                }// end of for(セル)
            } //end of for(DataIndex)
        }

        /// <summary><para>method outline:</para>
        /// <para>全３次元セル内挿計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalD3AllInterValue( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns> 無し　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void CalD3AllInterValue()
        {
            // 内挿計算用時刻
            double dPrvTime = m_csPrevTime.GetTime();
            double dCurrntTime = m_csCurrentTime.GetTime();
            double dAfterTime = m_csAfterTime.GetTime();
            // 内挿計算開始
            double dDltT = dAfterTime - dPrvTime;
                
            if (m_csD3Prev != null)
            {   // 前データ有り
                if (dDltT > 0.0 && m_csD3After != null)
                {   // 内挿計算可能ならば
                    m_csCurrentD3Data = m_csWorkD3Data;
                    CalD3InterValue(dPrvTime, dCurrntTime, dAfterTime); // ３次元セル内挿計算
                }
                else
                {
                    m_csCurrentD3Data = m_csD3Prev;
                }
            }
            else
            {   // 万が一過去のデータが無ければ
                if (m_csD3After != null)
                {    // 過去は無くても未来がある場合には未来を返す
                    m_csCurrentD3Data = m_csD3After;
                }
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>３次元セル内挿計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalD3InterValue(dPrvTime,dCurrntTime,dAfterTime) </para>
        /// </example>
        /// <param name="dPrvTime">直前時刻</param>
        /// <param name="dCurrntTime">指定時刻</param>
        /// <param name="dAfterTime">直後時刻</param>
        /// <returns> double　計算された内挿値　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual void CalD3InterValue(
            double dPrvTime,
            double dCurrntTime,
            double dAfterTime)
        {
            HySCellData[,,] csCrtCell = m_csCurrentD3Data.GetCellData();
            HySCellData[,,] csPrvCell = m_csD3Prev.GetCellData();
            HySCellData[,,] csAftCell = m_csD3After.GetCellData();

            long lDataIdx = -1; // データインデックス
            for (long lLp = 0; lLp < m_csDataIdx.GetIdxNum(); lLp++)
            {
                lDataIdx = m_csDataIdx.GetIdx(lLp);
                if (lDataIdx < 0) { continue; }
                for (long lCll = 0; lCll < m_csCurrentD3Data.GetDimension1(); lCll++)
                {
                    for (long lCll2 = 0; lCll2 < m_csCurrentD3Data.GetDimension2(); lCll2++)
                    {
                        for (long lCll3 = 0; lCll3 < m_csCurrentD3Data.GetDimension3(); lCll3++)
                        {
                            // 本クラス内では、データの入れ替えは行わない（但し、速度アップのため、後でデータ入れ替え後に使用しないデータについては、補間を行わない）
                            //csCrtCell[lCll, lCll2, lCll3].m_dData[lLp] = CalInterValue(
                            //                        csPrvCell[lCll, lCll2, lCll3].m_dData[lLp],
                            //                        csAftCell[lCll, lCll2, lCll3].m_dData[lLp],
                            //                        dPrvTime,
                            //                        dCurrntTime,
                            //                        dAfterTime);
                            csCrtCell[lCll, lCll2, lCll3].m_dData[lDataIdx] = CalInterValue(
                                                    csPrvCell[lCll, lCll2, lCll3].m_dData[lDataIdx],
                                                    csAftCell[lCll, lCll2, lCll3].m_dData[lDataIdx],
                                                    dPrvTime,
                                                    dCurrntTime,
                                                    dAfterTime);

                        }
                    }
                }// end of for(セル)
            } //end of for(DataIndex)
        }

        /// <summary><para>method outline:</para>
        /// <para>内挿計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CalInterValue(dDPrev,dDAfter,dPrvTime,dCurrntTime,dAfterTime) </para>
        /// </example>
        /// <param name="dDPrev">前値</param>
        /// <param name="dDAfter">後値</param>
        /// <param name="dPrvTime">直前時刻</param>
        /// <param name="dCurrntTime">指定時刻</param>
        /// <param name="dAfterTime">直後時刻</param>
        /// <returns> double　計算された内挿値　</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected virtual double CalInterValue( 
            double dDPrev,
            double dDAfter,
            double dPrvTime,
            double dCurrntTime,
            double dAfterTime)
        {
            //return dDPrev + (dDAfter - dDPrev) * (m_csCurrentTime.GetTime() / (m_csAfterTime.GetTime() - m_csPrevTime.GetTime()));
            return dDPrev + (dDAfter - dDPrev) * ((dCurrntTime - dPrvTime)/(dAfterTime - dPrvTime));
        }


        // ================================================
        //   バージョンアップ時の下位互換性を保つ為の処理
        // ================================================
        /// <summary>保存バージョン情報</summary>
        private string m_sDecodedVersion;
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sDecodedVersion = GetDecodedVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 保存バージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>メンバー変数に設定されているバージョン情報を返す</para>
        /// </remarks>
        public new string GetDecodedVersion()
        {
            return m_sDecodedVersion;
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> string sVersion = GetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>string 現在のバージョン情報</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public new string GetNowVersion()
        {
            return "1.00";
        }
        /// <summary><para>method outline:</para>
        /// <para>バージョン情報設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetNowVersion()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>現在のバージョン情報をメンバー変数に設定する</para>
        /// </remarks>
        public new void SetNowVersion()
        {
            m_sDecodedVersion = GetNowVersion();
        }
        /// <summary><para>method outline:</para>
        /// <para>復元後の整合性確保処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PostDeserialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>シリアライズされた旧バージョン情報を復元する際、過去の整合性をとる時に使用する
        /// バージョンによって変数等追加がない場合には、実装の必要がない
        /// </para>
        /// </remarks>
        public override void PostDeserialize()
        {
            base.PostDeserialize();

            if (GetDecodedVersion() != GetNowVersion())
            {
                if (GetDecodedVersion() == null)
                {
                    // 旧バージョンからの移行処理を入れる
                }
                else
                {
                    //旧バージョンから順にバージョンを上げて行く
                    switch (GetDecodedVersion())
                    {
                        case "1.00":
                            {
                                // 旧バージョンからの移行処理を入れる
                                break;
                            }
                        //case "2.00": << 以後バージョンアップに伴って処理追加
                        //{
                        //    break;
                        //}
                    }                    
                }
                SetNowVersion();
            }
        }
        /*
        /// <summary><para>method outline:</para>
        /// <para>逆シリアル化時追加処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> IDeserializationCallback.OnDeserialization(csObj)</para>
        /// </example>
        /// <param name="csObj">ダミー</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>予約処理</para>
        /// </remarks>
        void IDeserializationCallback.OnDeserialization(Object csObj)
        {
            PostDeserialize();
        }
        */

    }
}
