﻿// <summary>ソースコード：ファイルユーティリティクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

namespace CommonMP.HYSSOP.CoreImpl.HSTools
{
    /// <summary><para>class outline:</para>
    /// <para>ライブラリ出力CUI　ファイルユーティリティクラス</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>無し</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/12/22][新規作成]</para>
    /// </remarks>
    public class HySFileUtility
    {
        /// <summary><para>method outline:</para>
        /// <para>ファイル有無確認</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool result = ExistFile(fileFullPath); </para>
        /// </example>
        /// <param name="fileFullPath">ファイルフルパス</param>
        /// <returns>true: 存在する false: 存在しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public bool ExistFile(
            HySString fileFullPath
            )
        {
            bool result = System.IO.File.Exists(fileFullPath.ToString());
            return result;
        }

        /// <summary><para>method outline:</para>
        /// <para>フォルダ有無確認</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool result = ExistDirectory(folderFullPath); </para>
        /// </example>
        /// <param name="folderFullPath">フォルダフルパス</param>
        /// <returns>true: 存在する false: 存在しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        static public bool ExistDirectory(
            HySString folderFullPath
            )
        {
            bool result = System.IO.Directory.Exists(folderFullPath.ToString());
            return result;
        }
        
        /// <summary><para>method outline:</para>
        /// <para>指定フォルダ以下のファイル取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HysDataLinkedList dataList = GetFiles(folderFullPath); </para>
        /// </example>
        /// <param name="folderFullPath">フォルダフルパス</param>
        /// <returns>指定フォルダ以下のファイルおよび空フォルダリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>フォルダフルパスが不正な場合はnullを返却する</para>
        /// </remarks>
        static public HySDataLinkedList GetFiles(
            HySString folderFullPath
            )
        {
            HySDataLinkedList dataList = new HySDataLinkedList();
            try
            {
                string[] entries = System.IO.Directory.GetFileSystemEntries(
                    folderFullPath.ToString()
                );
                foreach(string entry in entries)
                {
                    if (true == System.IO.Directory.Exists(entry))
                    {

                        HySDataLinkedList subEnties = HySFileUtility.GetFiles(
                                                    new HySString(entry)
                                                    );

                        if (0 < subEnties.GetCount())
                        {

                            for (int i = 0; i < subEnties.GetCount(); i++ )
                            {
                                dataList.AddLast(subEnties.GetData(i));
                            }
                        }
                        else
                        {
                            // 空フォルダの場合は、空フォルダを出力引数に登録
                            dataList.AddLast(new HySString(entry));
                        }

                    }
                    else
                    {
                        dataList.AddLast(new HySString(entry));
                    }
                }

            }
            catch (UnauthorizedAccessException)
            {
                return null;
            }
            catch (ArgumentException)
            {
                return null;
            }
            catch (System.IO.PathTooLongException)
            {
                return null;
            }
            catch (System.IO.DirectoryNotFoundException)
            {
                return null;
            }
            catch (System.IO.IOException)
            {
                return null;
            }
            return dataList;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定ファイル削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool result = Delete(fileFullPath); </para>
        /// </example>
        /// <param name="fileFullPath">ファイルフルパス</param>
        /// <returns>true: 正常終了 false: 異常終了</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ファイルフルパスが不正な場合はfalseを返却する</para>
        /// </remarks>
        static public bool Delete(
            HySString fileFullPath
            )
        {
            try
            {
                System.IO.File.Delete(fileFullPath.ToString());
            }
            catch (ArgumentException)
            {
                return false;
            }
            catch (System.IO.DirectoryNotFoundException)
            {
                return false;
            }
            catch (System.IO.IOException)
            {
                return false;
            }
            catch (NotSupportedException)
            {
                return false;
            }
            catch (UnauthorizedAccessException)
            {
                return false;
            }
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>使用可能なファイルパスかチェックする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool result = EnablFillFullPath(fileFullPath); </para>
        /// </example>
        /// <param name="fileFullPath">ファイルフルパス</param>
        /// <returns>true: 使用可能 false:  使用不可</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ファイルパスとして無効な場合はfalseを返却する</para>
        /// </remarks>
        static public bool EnableFileFullPath(
            HySString fileFullPath)
        {
            try
            {
                // ディレクトリ名を取得して、戻り値をチェックする
                string directoryName = System.IO.Path.GetDirectoryName(fileFullPath.ToString());
                if (directoryName == null || directoryName == string.Empty)
                {
                    return false;
                }
                // ファイル名を取得して、戻り値をチェックする
                string fileName = System.IO.Path.GetFileName(fileFullPath.ToString());
                if (fileName == null || fileName == string.Empty)
                {
                    return false;
                }
                // パス名、ファイル名として無効な文字(*,?,:)がないかチェックする
                if (
                    (fileFullPath.ToString().Contains("*")) ||
                    (fileFullPath.ToString().Contains("?")) 
                    )
                {
                    return false;
                }
                string rootDirName = System.IO.Path.GetPathRoot(fileFullPath.ToString());

                if ((null != rootDirName) && (string.Empty != rootDirName))
                {
                    string dirName = fileFullPath.ToString().Remove(0,rootDirName.Length);
                    if (dirName.Contains(":"))
                    {
                        return false;
                    }
                }
                else
                {
                    if (fileFullPath.ToString().Contains(":"))
                    {
                        return false;
                    }
                }

            }
            catch (ArgumentException)
            {
                return false;
            }
            catch (System.IO.PathTooLongException)
            {
                return false;
            }

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定ファイル拡張子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySString extension = GetExtension(fileFullPath); </para>
        /// </example>
        /// <param name="fileFullPath">ファイルフルパス</param>
        /// <returns>拡張子文字列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ファイルフルパスが不正な場合はnullを返却する</para>
        /// </remarks>
        static public HySString GetExtension(
            HySString fileFullPath
            )
        {
            try
            {
                // 指定されたパスがファイルパスか判定する
                bool result = ExistFile(fileFullPath);
                if (result == false)
                {
                    return null;
                }
                HySString extension = new HySString(System.IO.Path.GetExtension
                    (fileFullPath.ToString()
                    ));
                return extension;
            }
            catch (ArgumentException)
            {
                return null;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>ファイル名称取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>HySString extension = GetFileName(fileFullPath); </para>
        /// </example>
        /// <param name="fileFullPath">ファイルフルパス</param>
        /// <returns>ファイル名称（拡張子を含む）</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ファイルフルパスが不正な場合はnullを返却する</para>
        /// </remarks>
        static public HySString GetFileName(
            HySString fileFullPath
            )
        {
            try
            {
                // 指定されたパスがファイルパスか判定する
                bool result = ExistFile(fileFullPath);
                if (result == false)
                {
                    return null;
                }
                HySString extension = new HySString(System.IO.Path.GetFileName(
                    fileFullPath.ToString()
                    ));
                return extension;
            }
            catch (ArgumentException)
            {
                return null;
            }
        }

        /// <summary><para>method outline</para>
        /// <para>ファイルパス体裁整理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>FormatPath()</para>
        /// </example>
        /// <param name="sPath">パス名</param>
        /// <returns>体裁整理後のパス名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>パスに半角スペースを含む場合は"(ダブルクォーテーション)で囲む</para>
        /// </remarks>
        static public string FormatPath(string sPath)
        {
            string formatedPath = sPath;
            if (sPath.Contains(" "))
            {   // 半角スペースを含む場合
                formatedPath = "\"" + sPath + "\"";
            }
            return formatedPath;
        }

    }
}
