// <summary>ソースコード：ＫＭＬ作成クラス<</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Globalization;

using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;

//namespace CommonMP.HYSSOP.CoreImpl.HSViewer.HSScreens
namespace CommonMP.HYSSOP.CoreImpl.HSTools
{
    /// <summary><para>class outline:</para>
    /// <para>ＫＭＬ作成クラス</para>
    /// </summary>
    /// <remarks><para>remarks:</para>
    /// <para>無し</para>
    /// </remarks>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/09/01][新規作成]</para>
    /// </remarks>
    public class HySCreateKmlFile
    {
        /// <summary>ＫＭＬ出力先パス</summary>
        private HySString m_csOutPath = null;

        /// <summary>シンボルテーブルを管理するハッシュテーブル</summary>
        private Hashtable m_csSymTable = null;

        /// <summary>メッシュ色情報を管理するハッシュテーブル</summary>
        private Hashtable m_csColTable = null;

        /// <summary>メッシュ色情報を管理するキーテーブル</summary>
        private List<double> m_lColKey = null;

        /// <summary>スプライン補間算出係数</summary>
        private double[] a, b, c, d;


        /// <summary>南西頂点緯度（度単位）</summary>
        protected double m_dSWLatitude = 90.0;

        /// <summary>南西頂点経度（度単位）</summary>
        protected double m_dSWLongitude = 180.0;

        /// <summary>北東頂点緯度（度単位）</summary>
        protected double m_dNELatitude = 0.0;

        /// <summary>北東頂点経度（度単位）</summary>
        protected double m_dNELongitude = 0.0;

        /// <summary>緯度方向分割数</summary>
        protected long m_lLatDim = 0;

        /// <summary>経度方向分割数</summary>
        protected long m_lLonDim = 0;


        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySCreateKmlFile csKml = new HySCreateKmlFile( )</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySCreateKmlFile()
        {
            // シンボルテーブル読み込み
            ReadSymbolTable();

            // メッシュ色情報テーブル読み込み
            ReadMeshColorTable();
        }

        /// <summary><para>method outline:</para>
        /// <para>作図データからＫＭＬファイル作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlFileFromFigureData(csFigData, csOutPath)</para>
        /// </example>
        /// <param name="csFigData">作図データ</param>
        /// <param name="csOutPath">出力パス（フルパス）</param>
        /// <returns>0 = 正常, -1 = 異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public long CreateKmlFileFromFigureData(HySGeoMapFigureData csFigData, HySString csOutPath)
        {
            long            lCnt = 0;
            HySGeometry     csGeo = null;

            string          wkKml = "";     // ＫＭＬファイル作成文字列格納


            // 図形オブジェクト数
            lCnt = csFigData.GetCount();

            if (lCnt == 0)
            {
                return -1;
            }


            // ＫＭＬ出力先パス
            m_csOutPath = csOutPath;

            // テーブルから読み込み
            if ((object)m_csOutPath == null || m_csOutPath.GetLength() < 1)
            {
                m_csOutPath = HySEnvInf.GetEnvInf(new HySString("KMLFILE_OUT_PATH"));
            }

            if ((object)m_csOutPath == null || m_csOutPath.GetLength() < 1)
            {
                m_csOutPath = new HySString("c:\\kml.kml");
            }

            File.Delete(m_csOutPath.ToString());


            // ＫＭＬヘッダ部作成処理
            CreateKmlHeader(ref wkKml);


            // カーソル位置を最初に持ってくる
            csFigData.SetCursorFirst();

            for (int i = 0; i < lCnt; i++)
            {
                // カーソル位置のデータを取得する
                csGeo = csFigData.GetCursorData();

                // ＫＭＬファイル作成
                CreateKmlData(csGeo, ref wkKml);

                // ＫＭＬファイル出力処理
                KmlFileOut(wkKml);
                wkKml = "";

                // カーソル位置を次の位置に持ってくる
                csFigData.MoveCursorNext();
            }

            wkKml += "</Document>\n</kml>\n";

            // ＫＭＬファイル出力処理
            KmlFileOut(wkKml);

            return 0;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬファイル作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlData(enumObj)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">図形部格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void CreateKmlData(HySGeometry csGeo, ref string stKml)
        {
            string          wkKml = "";     // ＫＭＬファイル作成文字列格納


            // 図形種別判定
            if ((csGeo is HySGeometryEllipse) ||
                (csGeo is HySGeometryPolygon) ||
                (csGeo is HySGeometryRectangle))
            {
                // ＫＭＬ閉図形部作成処理
                CreateKmlPolygon(csGeo, ref wkKml);
            }

            if ((csGeo is HySGeometryLine) ||
                (csGeo is HySGeometryPolyLine) ||
                (csGeo is HySGeometrySpline))
            {
                // ＫＭＬ線図形部作成処理
                CreateKmlPolyLine(csGeo, ref wkKml);
            }

            if (csGeo is HySGeometryText)
            {
                // ＫＭＬ文字列図形部作成処理
                CreateKmlText(csGeo, ref wkKml);
            }

            if (csGeo is HySGeometrySymbol)
            {
                // ＫＭＬシンボル図形部作成処理
                CreateKmlSymbol(csGeo, ref wkKml);
            }

            stKml += wkKml;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬヘッダ部作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlHeader(ref string stKml)</para>
        /// </example>
        /// <param name="stKml">ヘッダ部格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void CreateKmlHeader(ref string stKml)
        {
            string str = "";

            // ＫＭＬヘッダ部分
            str += "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
            str += "<kml xmlns=\"http://earth.google.com/kml/2.2\">\n";
            str += "<Document>\n";
            str += "    <name>Figure.kml</name>\n";

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬ線図形部作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlPolyLine(ref string stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">線図形部格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void CreateKmlPolyLine(HySGeometry csGeo, ref string stKml)
        {
            string      str = "";
            string      wkKml = "";

            string      fig = "";
            string      color = "";
            string      width = "";

            HySID       csID = null;


            // 個別図形識別子
            csID = csGeo.GetID();

            if (csID == null)
            {
                csID = new HySID("");
            }

            // 線色情報取得
            ColorToString(csGeo, 0, ref color);

            // 線幅を取得
            width = csGeo.GetLineWidth().ToString();

            // 図形種別判定
            if (csGeo is HySGeometryLine)
            {
                // ＧＩＳ線分クラス
                fig = "Nm_LINE";
            }
            else if (csGeo is HySGeometryPolyLine)
            {
                // ＧＩＳ折れ線クラス
                fig = "Nm_POLYLINE";
            }
            else
            {
                // ＧＩＳ曲線クラス
                fig = "Nm_SPLINE";
            }


            // ＫＭＬ線図形（直線、折れ線、曲線）
            str += "    <Placemark id = ";
            str += "\"";
            str += (csID.GetString()).ToString();
            str += "\">\n";
            str += "     <name>";
            str += fig;
            str += "</name>\n";
            str += "     <Style>\n";
            str += "      <LineStyle>\n";
            str += "       <color>";
            str += color;
            str += "</color>\n";
            str += "       <width>";
            str += width;
            str += "</width>\n";
            str += "      </LineStyle>\n";
            str += "     </Style>\n";
            str += "     <LineString>\n";

            // オブジェクト高度モード設定処理
            SetAltitudeMode(csGeo, ref wkKml);

            str += wkKml;            
            str += "      <coordinates>\n";

            wkKml = "";

            // 線図形オブジェクト点列作成処理
            PolyLinePointVec(csGeo, ref wkKml);

            str += wkKml;
            str += "      </coordinates>\n";
            str += "     </LineString>\n";
            str += "    </Placemark>\n";

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬ閉図形部作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlPolygon(ref string stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">閉図形部格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void CreateKmlPolygon(HySGeometry csGeo, ref string stKml)
        {
            string      str = "";
            string      wkKml = "";

            string      fig = "";
            string      color1 = "";
            string      color2 = "";
            string      width = "";
            string      fill = "";

            HySID       csID = null;


            // 個別図形識別子
            csID = csGeo.GetID();

            if (csID == null)
            {
                csID = new HySID("");
            }

            // 線色情報取得
            ColorToString(csGeo, 0, ref color1);

            // 面色情報取得
            ColorToString(csGeo, 1, ref color2);

            // 線幅を取得
            width = csGeo.GetLineWidth().ToString();

            // 塗り潰しパターン番号
            fill = ((long)csGeo.GetFillStyle()).ToString();

            // 図形種別判定
            if (csGeo is HySGeometryPolygon)
            {
                // ＧＩＳ多角形クラス
                fig = "Nm_POLYGON";
            }
            else if (csGeo is HySGeometryRectangle)
            {
                // ＧＩＳ矩形クラス
                fig = "Nm_RECT";
            }
            else
            {
                // ＧＩＳ楕円クラス
                fig = "Nm_ELLIPSE";
            }


            // ＫＭＬ閉図形（矩形、多角形、楕円）
            str += "    <Placemark id = ";
            str += "\"";
            str += (csID.GetString()).ToString();
            str += "\">\n";
            str += "     <name>";
            str += fig;
            str += "</name>\n";
            str += "     <Style>\n";
            str += "      <LineStyle>\n";
            str += "       <color>";
            str += color1;
            str += "</color>\n";
            str += "       <width>";
            str += width;
            str += "</width>\n";
            str += "      </LineStyle>\n";
            str += "      <PolyStyle>\n";
            str += "       <color>";
            str += color2;
            str += "</color>\n";
            str += "       <fill>";
            str += fill;
            str += "</fill>\n";
            str += "       <outline>1</outline>\n";
            str += "      </PolyStyle>\n";
            str += "     </Style>\n";
            str += "     <Polygon>\n";

            // オブジェクト高度モード設定処理
            SetAltitudeMode(csGeo, ref wkKml);

            str += wkKml;            
            str += "      <outerBoundaryIs>\n";
            str += "      <LinearRing>\n";
            str += "       <coordinates>\n";

            // 面図形オブジェクト点列作成処理
            PolygonPointVec(csGeo, ref wkKml);

            str += wkKml;
            str += "       </coordinates>\n";
            str += "      </LinearRing>\n";
            str += "      </outerBoundaryIs>\n";
            str += "     </Polygon>\n";
            str += "    </Placemark>\n";

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬ文字列図形部作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlPolyLine(ref string stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">文字列図形部格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void CreateKmlText(HySGeometry csGeo, ref string stKml)
        {
            HySGeoPoint     csGeoPt;
            double          dx = 0.0;
            double          dy = 0.0;
            double          dz = 0.0;
            string          color = "";
            HySString       csString;

            string          str = "";
            string          wk = "";

            HySID           csID = null;
            HySGeometryText csText = null;


            // 個別図形識別子
            csID = csGeo.GetID();

            if (csID == null)
            {
                csID = new HySID("");
            }

            // ＧＩＳ文字クラス
            csText = (HySGeometryText)csGeo;

            // 文字色情報取得
            ColorToString(csGeo, 0, ref color);

            // 位置座標取得
            csGeoPt = csText.GetPosition();
            dx = csGeoPt.m_dLongitude;
            dy = csGeoPt.m_dLatitude;
            dz = csGeoPt.m_dAltitude;

            wk  = "        ";
            wk += dx.ToString();
            wk += ", ";
            wk += dy.ToString();
            wk += ", ";
            wk += dz.ToString();
            wk += "\n";

            // 文字列取得
            csString = csText.GetString();
            csString += "\n";


            // ＫＭＬ文字列図形
            str += "    <Placemark id = ";
            str += "\"";
            str += (csID.GetString()).ToString();
            str += "\">\n";
            str += "     <name>Nm_TEXT</name>\n";
            str += "     <Style>\n";
            str += "      <LabelStyle>\n";
            str += "       <color>";
            str += color;
            str += "</color>\n";
            str += "       <scale>1</scale>\n";
            str += "      </LabelStyle>\n";
            str += "      <IconStyle>\n";
            str += "       <scale>0</scale>\n";
            str += "      </IconStyle>\n";
            str += "     </Style>\n";
            str += "     <name>\n";
            str += csString.ToString();
            str += "     </name>\n";
            str += "     <Point>\n";
            str += "      <coordinates>\n";
            str += wk;
            str += "      </coordinates>\n";
            str += "     </Point>\n";
            str += "    </Placemark>\n";

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬシンボル図形部作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlPolyLine(ref string stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">シンボル図形部格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void CreateKmlSymbol(HySGeometry csGeo, ref string stKml)
        {
            HySGeoPoint     csGeoPt;
            double          dx = 0.0;
            double          dy = 0.0;
            double          dz = 0.0;
            string          color = "";
            HySObjectKind   csKind;

            string          str = "";
            string          wk = "";

            HySID               csID = null;
            HySGeometrySymbol   csSymbol = null;
            HySString           csSymPath = null;


            // 個別図形識別子
            csID = csGeo.GetID();

            if (csID == null)
            {
                csID = new HySID("");
            }

            // ＧＩＳシンボルクラス
            csSymbol = (HySGeometrySymbol)csGeo;

            // シンボル色情報取得
            ColorToString(csGeo, 2, ref color);

            // 位置座標取得
            csGeoPt = csSymbol.GetPosition();
            dx = csGeoPt.m_dLongitude;
            dy = csGeoPt.m_dLatitude;
            dz = csGeoPt.m_dAltitude;

            wk  = "        ";
            wk += dx.ToString();
            wk += ", ";
            wk += dy.ToString();
            wk += ", ";
            wk += dz.ToString();
            wk += "\n";

            // シンボル種別取得
            csKind = csSymbol.GetSymbolKind();

            if (m_csSymTable.Count > 0)
            {
                string stPath = "";

                stPath = (string)m_csSymTable[csKind.ToString()];
                csSymPath = HySEnvInf.GetHomeDirectory() + "\\" + stPath;
            }

            if ((object)csSymPath == null)
            {
                csSymPath = new HySString("c:\\kml.png");
            }

            // ＫＭＬシンボル図形
            str += "    <Placemark id = ";
            str += "\"";
            str += (csID.GetString()).ToString();
            str += "\">\n";
            str += "     <name>Nm_SYMBOL</name>\n";
            str += "     <Style>\n";
            str += "      <LabelStyle>\n";
            str += "       <scale>1</scale>\n";
            str += "      </LabelStyle>\n";
            str += "       <IconStyle>\n";
            str += "       <color>";
            str += color;
            str += "</color>\n";
            str += "        <colorMode>normal</colorMode>\n";
            str += "        <scale>1</scale>\n";
            str += "        <Icon>\n";
            str += "         <href>\n";
            str += "          ";
            str += csSymPath.ToString();
            str += "\n";
            str += "         </href>\n";
            str += "        </Icon>\n";
            str += "       </IconStyle>\n";
            str += "     </Style>\n";
            str += "     <Point>\n";
            str += "      <coordinates>\n";
            str += wk;
            str += "      </coordinates>\n";
            str += "     </Point>\n";
            str += "    </Placemark>\n";

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬファイル出力処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> KmlFileOut(string stKml)</para>
        /// </example>
        /// <param name="stKml">ＫＭＬ文字列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void KmlFileOut(string stKml)
        {
            File.AppendAllText(m_csOutPath.ToString(), stKml);
        }

        /// <summary><para>method outline:</para>
        /// <para>シンボルテーブル読み込み処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ReadSymbolTable()</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReadSymbolTable()
        {
            HySString   csTableFile = null;
            HySString   csFileName = null;  // シンボルテーブルファイル名称
            HySFile     csFile = null;

            HySString   csReadStr;          // 読み取りデータ
            int         lKeyPos = 0;        // キーの位置
            int         lValue = 0;         // 値の位置
            int         lEqualPos = 0;      // イコール文字位置

            string      stKey = "";         // キー
            string      stVal = "";         // 値


            m_csSymTable = new Hashtable();

            // 管理テーブルのクリア
            m_csSymTable.Clear();

            // テーブルから読み込み
            csTableFile = HySEnvInf.GetEnvInf(new HySString("SYMBOL_TABLE_FILE"));

            if ((object)csTableFile == null)
            {
                return;
            }

            csFileName = HySEnvInf.GetHomeDirectory() + "\\conf\\" + csTableFile;

            csFile = new HySFile(csFileName);

            // ファイルオープン失敗
            if (csFile.Open(HySFile.OPEN_MODE.OPEN, HySFile.READ_WRITE_MODE.READ,
                            HySFile.DIRECTORY_MODE.NOT_MK_DIR) != 0)
            {
                return;
            }

            csReadStr = new HySString("");

            // ファイルの最後まで読み込む
            while (csFile.ReadText(ref csReadStr) != 0)
            {
                // コメント文、空行読み飛ばし
                if (csReadStr.ToString().IndexOf("#") == 0 ||
                    csReadStr.ToString() == String.Empty)
                {
                    continue;
                }

                // イコール文字の位置
                lEqualPos = csReadStr.IndexOf("=");

                lKeyPos = lEqualPos - 1;
                lValue  = lEqualPos + 2;

                
                // キー取得
                stKey = csReadStr.ToString().Substring(0, lKeyPos);

                // 値取得
                stVal = (HySEnvInf.GetHomeDirectory()).ToString() + "\\" + csReadStr.ToString().Substring(lValue);

                // 取得確認
                if (stKey.Length > 0 && stVal.Length > 0)
                {
                    m_csSymTable.Add(stKey, stVal);
                }
            }

            csFile.Close();
            csFile = null;
        }

        /// <summary><para>method outline:</para>
        /// <para>メッシュ色情報テーブル読み込み</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ReadMeshColorTable()</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ReadMeshColorTable()
        {
            HySString   csTableFile = null;
            HySString   csFileName = null;  // メッシュ色情報テーブルファイル名称
            HySFile     csFile = null;

            HySString   csReadStr;          // 読み取りデータ
            int         lKeyPos = 0;        // キーの位置
            int         lValue = 0;         // 値の位置
            int         lEqualPos = 0;      // イコール文字位置

            string      stKey = "";         // キー
            string      stVal = "";         // 値


            m_csColTable = new Hashtable();
            m_lColKey    = new List<double>();

            // 管理テーブルのクリア
            m_csColTable.Clear();
            m_lColKey.Clear();

            // テーブルから読み込み
            csTableFile = HySEnvInf.GetEnvInf(new HySString("MESH_RANGE"));

            if ((object)csTableFile == null)
            {
                return;
            }

            csFileName = HySEnvInf.GetHomeDirectory() + "\\conf\\" + csTableFile;

            csFile = new HySFile(csFileName);

            // ファイルオープン失敗
            if (csFile.Open(HySFile.OPEN_MODE.OPEN, HySFile.READ_WRITE_MODE.READ,
                            HySFile.DIRECTORY_MODE.NOT_MK_DIR) != 0)
            {
                return;
            }

            csReadStr = new HySString("");

            // ファイルの最後まで読み込む
            while (csFile.ReadText(ref csReadStr) != 0)
            {
                // コメント文、空行読み飛ばし
                if (csReadStr.ToString().IndexOf("#") == 0 ||
                    csReadStr.ToString() == String.Empty)
                {
                    continue;
                }

                // イコール文字の位置
                lEqualPos = csReadStr.IndexOf("=");

                lKeyPos = lEqualPos - 1;
                lValue  = lEqualPos + 2;

                
                // キー取得
                stKey = csReadStr.ToString().Substring(0, lKeyPos);

                // 値取得
                stVal = csReadStr.ToString().Substring(lValue);

                // 取得確認
                if (stKey.Length > 0 && stVal.Length > 0)
                {
                    m_csColTable.Add(stKey, stVal);
                    m_lColKey.Add(Double.Parse(stKey));
                }
            }

            csFile.Close();
            csFile = null;
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト高度モード設定処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetAltitudeMode(csGeo, ref string stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">文字列格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetAltitudeMode(HySGeometry csGeo, ref string stKml)
        {
            string              str = "";
            string              wkKml = "";

            long                lExtrudeMode = 0;
            long                lTessellateMode = 0;
            HySGeometry.AltMode eAltMode = HySGeometry.AltMode.ABSOLUTE;


            // 地面からの突き出しモード取得
            if (csGeo.GetExtrudeMode() == true)
            {
                lExtrudeMode = 1;
            }

            // 地面に沿った表示を行うモード取得
            if (csGeo.GetTessellateMode() == true)
            {
                lTessellateMode = 1;
            }

            // 高度設定モード取得
            eAltMode = csGeo.GetAltMode();

            switch (eAltMode)
            {
                // 地面に配置
                case HySGeometry.AltMode.CLAMP_TO_GND:

                    wkKml = "clampToGround";
                    break;

                // 地形に沿って配置
                case HySGeometry.AltMode.RELATIVE_GND:

                    wkKml = "relativeToGround";
                    break;

                // 海抜
                default:

                    wkKml = "absolute";
                    break;
            }

            str += "      <extrude>";
            str += lExtrudeMode.ToString();
            str += "</extrude>\n";
            str += "      <tessellate>";
            str += lTessellateMode.ToString();
            str += "</tessellate>\n";
            str += "      <altitudeMode>";
            str += wkKml;
            str += "</altitudeMode>\n";

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>線図形オブジェクト点列作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PolyLinePointVec(csGeo, ref string stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">点列格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void PolyLinePointVec(HySGeometry csGeo, ref string stKml)
        {
            HySGeometryLine     csLine = null;
            HySGeoPoint         csPtStart = null;
            HySGeoPoint         csPtEnd = null;

            HySGeometryPolyLine csPolyLine = null;
            HySDataLinkedList   csPtList = null;
            HySDataRoot         csDtRoot = null;
            HySGeoPoint         csGeoPt = null;

            HySGeometrySpline   csSpline = null;

            long                lCnt = 0;
            string              str = "";


            // 図形種別判定
            if (csGeo is HySGeometryLine)
            {
                // ＧＩＳ線分クラス
                csLine = (HySGeometryLine)csGeo;

                csPtStart = csLine.GetStartPoint();
                csPtEnd = csLine.GetEndPoint();

                // 始点
                str += "        ";
                str += csPtStart.m_dLongitude.ToString();
                str += ", ";
                str += csPtStart.m_dLatitude.ToString();
                str += ", ";
                str += csPtStart.m_dAltitude.ToString();
                str += "\n";

                // 終点
                str += "        ";
                str += csPtEnd.m_dLongitude.ToString();
                str += ", ";
                str += csPtEnd.m_dLatitude.ToString();
                str += ", ";
                str += csPtEnd.m_dAltitude.ToString();
                str += "\n";
            }
            else if (csGeo is HySGeometryPolyLine)
            {
                // ＧＩＳ折れ線クラス
                csPolyLine = (HySGeometryPolyLine)csGeo;

                // 頂点リストの取得
                csPtList = csPolyLine.GetPointList();
                lCnt = csPtList.GetCount();

                // 座標点列
                for (int i = 0; i < lCnt; i++)
                {
                    // 指定データ取得
                    csDtRoot = csPtList.GetData(i);
                    csGeoPt = (HySGeoPoint)csDtRoot;

                    str += "        ";
                    str += csGeoPt.m_dLongitude.ToString();
                    str += ", ";
                    str += csGeoPt.m_dLatitude.ToString();
                    str += ", ";
                    str += csGeoPt.m_dAltitude.ToString();
                    str += "\n";
                }
            }
            else
            {
                // ＧＩＳ曲線クラス
                csSpline = (HySGeometrySpline)csGeo;

                // 曲線オブジェクト点列作成処理
                SplinePointVec(csSpline, ref str);
            }

            stKml = str;

        }

        /// <summary><para>method outline:</para>
        /// <para>面図形オブジェクト点列作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PolygonPointVec(csGeo, ref string stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="stKml">点列格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void PolygonPointVec(HySGeometry csGeo, ref string stKml)
        {
            HySGeometryPolygon      csPolygon = null;
            HySDataLinkedList       csPtList = null;
            HySDataRoot             csDtRoot = null;
            HySGeoPoint             csGeoPt = null;

            HySGeometryRectangle    csRect = null;
            double                  dLat1 = 0.0;    // 南西緯度
            double                  dLon1 = 0.0;    // 南西経度
            double                  dLat2 = 0.0;    // 北東緯度
            double                  dLon2 = 0.0;    // 北東経度
            double                  dAlt = 0.0;     // 標高 [m]

            HySGeometryEllipse      csEllipse = null;

            long                    lCnt = 0;
            string                  str = "";


            // 図形種別判定
            if (csGeo is HySGeometryPolygon)
            {
                // ＧＩＳ多角形クラス
                csPolygon = (HySGeometryPolygon)csGeo;

                // 頂点リストの取得
                csPtList = csPolygon.GetPointList();
                lCnt = csPtList.GetCount();

                // 座標点列
                for (int i = 0; i < lCnt; i++)
                {
                    // 指定データ取得
                    csDtRoot = csPtList.GetData(i);
                    csGeoPt = (HySGeoPoint)csDtRoot;

                    str += "        ";
                    str += csGeoPt.m_dLongitude.ToString();
                    str += ",";
                    str += csGeoPt.m_dLatitude.ToString();
                    str += ",";
                    str += csGeoPt.m_dAltitude.ToString();
                    str += "\n";
                }

                // 指定データ取得（閉じる）
                csDtRoot = csPtList.GetData(0);
                csGeoPt = (HySGeoPoint)csDtRoot;

                str += "        ";
                str += csGeoPt.m_dLongitude.ToString();
                str += ",";
                str += csGeoPt.m_dLatitude.ToString();
                str += ",";
                str += csGeoPt.m_dAltitude.ToString();
                str += "\n";
            }
            else if (csGeo is HySGeometryRectangle)
            {
                // ＧＩＳ矩形クラス
                csRect = (HySGeometryRectangle)csGeo;

                // ２点座標取得
                dLat1 = csRect.GetSWLatitude();     // 南西緯度
                dLon1 = csRect.GetSWLongitude();    // 南西経度
                dLat2 = csRect.GetNELatitude();     // 北東緯度
                dLon2 = csRect.GetNELongitude();    // 北東経度
                dAlt  = csRect.GetAltitude();

                // 南西
                str += "        ";
                str += dLon1.ToString();
                str += ",";
                str += dLat1.ToString();
                str += ",";
                str += dAlt.ToString();
                str += "\n";

                // 南東
                str += "        ";
                str += dLon2.ToString();
                str += ",";
                str += dLat1.ToString();
                str += ",";
                str += dAlt.ToString();
                str += "\n";

                // 北東
                str += "        ";
                str += dLon2.ToString();
                str += ",";
                str += dLat2.ToString();
                str += ",";
                str += dAlt.ToString();
                str += "\n";

                // 北西
                str += "        ";
                str += dLon1.ToString();
                str += ",";
                str += dLat2.ToString();
                str += ",";
                str += dAlt.ToString();
                str += "\n";

                // 南西（閉じる）
                str += "        ";
                str += dLon1.ToString();
                str += ",";
                str += dLat1.ToString();
                str += ",";
                str += dAlt.ToString();
                str += "\n";
            }
            else
            {
                // ＧＩＳ楕円クラス
                csEllipse = (HySGeometryEllipse)csGeo;

                // 楕円オブジェクト点列作成処理
                EllipsePointVec(csEllipse, ref str);
            }

            stKml = str;

        }

        /// <summary><para>method outline:</para>
        /// <para>曲線オブジェクト点列作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SplinePointVec(csSpline, ref string stKml)</para>
        /// </example>
        /// <param name="csSpline">ＧＩＳ曲線クラス</param>
        /// <param name="stKml">点列格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SplinePointVec(HySGeometrySpline csSpline, ref string stKml)
        {
            // 参考ＵＲＬ
            // http://www5d.biglobe.ne.jp/~stssk/maze/spline.html
            // 構成点ごとの高度対応

            long        lCnt = 0;
            int         i = 0;
            int         num = 0;

            double[]    dx = new double[256];
            double[]    dy = new double[256];
            double[]    dz = new double[256];

            double[]    dxx;
            double[]    dyy;
            double[]    dzz;

            HySDataLinkedList   csPtList;
            HySGeoPoint         csGeoPt;
            HySDataRoot         csDtRoot;


            // 頂点リストの取得
            csPtList = csSpline.GetPointList();
            lCnt = csPtList.GetCount();
            num = (int)lCnt;

            // 連続的な補間データ取得
            for (i = 0; i < lCnt; i++)
            {
                // 指定データ取得
                csDtRoot = csPtList.GetData(i);
                csGeoPt = (HySGeoPoint)csDtRoot;

                dx[i] = csGeoPt.m_dLongitude;
                dy[i] = csGeoPt.m_dLatitude;
                dz[i] = csGeoPt.m_dAltitude;
            }

            a = new double[num + 1];
            b = new double[num + 1];
            c = new double[num + 1];
            d = new double[num + 1];

            dxx = new double[(lCnt - 1) * 100 + 1];
            dyy = new double[(lCnt - 1) * 100 + 1];
            dzz = new double[(lCnt - 1) * 100 + 1];


            double  t = 0.0;
            double  m = 0.0;


            m = (double)(num - 1);

            init(dx, num);
            for (t = 0.0, i = 0; t <= m; t += 0.01)
            {
                dxx[i] = culc(t, num-1);
                i++;
            }

            init(dy, num);
            for (t = 0.0, i = 0; t <= m; t += 0.01)
            {
                dyy[i] = culc(t, num-1);
                i++;
            }

            init(dz, num);
            for (t = 0.0, i = 0; t <= m; t += 0.01)
            {
                dzz[i] = culc(t, num-1);
                i++;
            }


            string str = "";

            i = 0;
            for (t = 0.0; t <= m; t += 0.01)
            {
                str += "        ";
                str += dxx[i].ToString();
                str += ", ";
                str += dyy[i].ToString();
                str += ", ";
                str += dzz[i].ToString();
                str += "\n";
                i++;
            }

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>スプラインデータ初期化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> init(sp, spNum)</para>
        /// </example>
        /// <param name="sp">構成点</param>
        /// <param name="spNum">構成点の数</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        void init(double[] sp, int spNum)
        {
            double      tmp = 0.0;
            double[]    w;
            int         i = 0;
            int         num = 0;


            num = spNum - 1;
            w = new double[num + 1];

            // ３次多項式の０次係数(a)を設定
            for (i = 0; i <= num; i++)
            {
                a[i] = sp[i];
            }

            // ３次多項式の２次係数(c)を計算
            // 連立方程式を解く。
            // 但し、一般解法でなくスプライン計算にチューニングした方法
            c[0] = c[num] = 0.0;
            for (i = 1; i < num; i++)
            {
                c[i] = 3.0 * (a[i - 1] - 2.0 * a[i] + a[i + 1]);
            }

            // 左下を消す
            w[0] = 0.0;
            for (i = 1; i < num; i++)
            {
                tmp = 4.0 - w[i - 1];
                c[i] = (c[i] - c[i - 1]) / tmp;
                w[i] = 1.0 / tmp;
            }

            // 右上を消す
            for (i = num - 1; i > 0; i--)
            {
                c[i] = c[i] - c[i + 1] * w[i];
            }

            // ３次多項式の１次係数(b)と３次係数(b)を計算
            b[num] = d[num] = 0.0;
            for (i = 0; i < num; i++)
            {
                d[i] = (c[i + 1] - c[i]) / 3.0;
                b[i] =  a[i + 1] - a[i] - c[i] - d[i];
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>媒介変数(0～num-1の実数）に対する値を計算</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> culc(t, num)</para>
        /// </example>
        /// <param name="t">媒介変数</param>
        /// <param name="num">構成点の数</param>
        /// <returns>座標値</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        double culc(double t, int num)
        {
            int     j = 0;
            double  dt = 0.0;


            // 小数点以下切捨て
            j = (int)Math.Floor(t);

            // 丸め誤差を考慮
            if (j < 0)
            {
                j = 0;
            }
            else if (j >= num)
            {
                j = num - 1;
            }

            dt = t - (double)j;

            return a[j] + ( b[j] + (c[j] + d[j] * dt) * dt ) * dt;
        }
/*
        /// <summary><para>method outline:</para>
        /// <para>曲線オブジェクト点列作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SplinePointVec(csSpline, ref string stKml)</para>
        /// </example>
        /// <param name="csSpline">ＧＩＳ曲線クラス</param>
        /// <param name="stKml">点列格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SplinePointVec2(HySGeometrySpline csSpline, ref string stKml)
        {
            // 参考ＵＲＬ
            // http://mikilab.doshisha.ac.jp/dia/research/person/kouta/report/beat/math/1209/rep1209.htm

            double              dxx = 0.0;  // 点列Ｘ
            double              dfx = 0.0;  // 点列Ｙ

            double              dq = 0.0;
            double              dp = 0.0;
            double              ds = 0.0;

            long                lCnt = 0;
            double[]            dx = new double[256];
            double[]            df = new double[256];
            double[]            dd = new double[256];

            int                 i = 0;
            int                 j = 0;
            string              str = "";

            HySDataLinkedList   csPtList;
            HySGeoPoint         csGeoPt;
            HySDataRoot         csDtRoot;


            // 頂点リストの取得
            csPtList = csSpline.GetPointList();
            lCnt = csPtList.GetCount();

            // 連続的な補間データ取得
            for (i = 0; i < lCnt; i++)
            {
                // 指定データ取得
                csDtRoot = csPtList.GetData(i);
                csGeoPt = (HySGeoPoint)csDtRoot;

                dx[i] = csGeoPt.m_dLongitude;
                df[i] = csGeoPt.m_dLatitude;
            }

            for (i = 0; i < lCnt; i++)
            {
                dq = 1.0;
                for (j = 0; j < lCnt; j++)
                {
                    if (i != j)
                    {
                        dq *= dx[i] - dx[j];
                    }
                }
                dd[i] = dq;
            }

            // 標高 [m]
            csDtRoot = csPtList.GetData(0);
            csGeoPt = (HySGeoPoint)csDtRoot;

            // １００点列を出力する
            for (dxx = dx[0]; dxx < dx[lCnt - 1]; dxx += (dx[lCnt - 1] - dx[0]) / 100.0)
            {
                dp = 1.0;
                ds = 0.0;

                for (i = 0; i < lCnt; i++)
                {
                    if (dxx == dx[i])
                    {
                        dp = df[i];
                        ds = 1.0;
                        break;
                    }
                    dp *= dxx - dx[i];
                    ds += df[i] / dd[i] / (dxx - dx[i]);
                }

                dfx = dp * ds;

                str += "        ";
                str += dxx.ToString();
                str += ", ";
                str += dfx.ToString();
                str += ", ";
                str += csGeoPt.m_dAltitude.ToString(); 
                str += "\n";
            }

            stKml = str;
        }
*/
        /// <summary><para>method outline:</para>
        /// <para>楕円オブジェクト点列作成処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> EllipsePointVec(csEllipse, ref string stKml)</para>
        /// </example>
        /// <param name="csEllipse">ＧＩＳ楕円クラス</param>
        /// <param name="stKml">点列格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void EllipsePointVec(HySGeometryEllipse csEllipse, ref string stKml)
        {
            double  dLat1 = 0.0;    // 南西緯度
            double  dLon1 = 0.0;    // 南西経度
            double  dLat2 = 0.0;    // 北東緯度
            double  dLon2 = 0.0;    // 北東経度
            double  dAlt = 0.0;     // 標高 [m]

            double  dRadX = 0.0;    // 半径Ｘ
            double  dRadY = 0.0;    // 半径Ｙ

            double  dx = 0.0;       // 円周点列Ｘ
            double  dy = 0.0;       // 円周点列Ｙ

            double  dRad = 0.0;     // 角度
            string  str = "";


            // ２点座標取得
            dLat1 = csEllipse.GetSWLatitude();
            dLon1 = csEllipse.GetSWLongitude();
            dLat2 = csEllipse.GetNELatitude();
            dLon2 = csEllipse.GetNELongitude();
            dAlt  = csEllipse.GetAltitude();


            // 長径、短径
            dRadX = (dLon2 - dLon1) / 2;
            dRadY = (dLat2 - dLat1) / 2;


            // 分割数分ループする
            for (int i = 0; i <= 360; i++)
            {
                // 角度の増加
                dRad = dRad + (2.0 * Math.PI / 360.0);

                // 座標列 = 中心点 + 点列
                dx = (dLon1 + dRadX) + (dRadX * Math.Cos(dRad));
                dy = (dLat1 + dRadY) + (dRadY * Math.Sin(dRad));

                str += "        ";
                str += dx.ToString();
                str += ", ";
                str += dy.ToString();
                str += ", ";
                str += dAlt.ToString();
                str += "\n";
            }

            stKml = str;
        }

        /// <summary><para>method outline:</para>
        /// <para>色情報取得処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ColorToString(csGeo, sColType, stKml)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="sColType">線、面、シンボル色</param>
        /// <param name="stKml">文字列格納用</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void ColorToString(HySGeometry csGeo, short sColType, ref string stKml)
        {
            short   sR = 0;
            short   sG = 0;
            short   sB = 0;
            short   sA = 0;

            string  wk = "";


            if (sColType == 0)
            {
                // 線色情報取得
                csGeo.GetLineColor(ref sR, ref sG, ref sB);

                // 線透明度取得
                sA = csGeo.GetLineTransparency();

                wk += sA.ToString("X2");
                wk += sB.ToString("X2");
                wk += sG.ToString("X2");
                wk += sR.ToString("X2");
            }
            else if (sColType == 1)
            {
                // 面色情報取得
                csGeo.GetPolyColor(ref sR, ref sG, ref sB);

                // 面透明度取得
                sA = csGeo.GetPolyTransparency();

                wk += sA.ToString("X2");
                wk += sB.ToString("X2");
                wk += sG.ToString("X2");
                wk += sR.ToString("X2");
            }
            else
            {
                // シンボル色情報取得
                csGeo.GetSymbolColor(ref sR, ref sG, ref sB);

                // シンボル透明度取得
                sA = csGeo.GetSymbolTransparency();

                wk += sA.ToString("X2");
                wk += sB.ToString("X2");
                wk += sG.ToString("X2");
                wk += sR.ToString("X2");
            }

            stKml = wk;
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬファイルから作図クラス作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateFigureDataFromKmlFile(csKmlFile)</para>
        /// </example>
        /// <param name="csKmlFile">ＫＭＬファイル</param>
        /// <returns>null != 正常, null = 異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySGeoMapFigureData CreateFigureDataFromKmlFile(HySString csKmlFile)
        {
            HySGeoMapFigureData csFigData = null;
            HySGeometry         csGeo = null;

            long                lRtn = 0;
            int                 start = 0;
            int                 end = 0;
            HySFile             csFile = null;

            HySString           csFigStr = null;
            string              allData = "";
            string              oneData = "";

            bool                isFlag = false;


            // ファイルクラス
            csFile = new HySFile(csKmlFile);

            csFigStr = new HySString("");

            // 幾何図形のリスト
            csFigData = new HySGeoMapFigureData();


            // ファイルオープン処理
            lRtn = csFile.Open(HySFile.OPEN_MODE.OPEN, HySFile.READ_WRITE_MODE.READ,
                               HySFile.DIRECTORY_MODE.MK_DIR);
            if (lRtn != 0)
            {
                return null;
            }

            // ファイルからテキストデータを読み込む
            while (csFile.ReadText(ref csFigStr) != 0)
            {
                allData += csFigStr.ToString();

                if (isFlag == true)
                {
                    allData += "\n";
                }

                if (csFigStr.IndexOf("<coordinates>") > -1)
                {
                    isFlag = true;
                }
                if (csFigStr.IndexOf("</coordinates>") > -1)
                {
                    isFlag = false;
                }
            }

            csFile.Close();


            lRtn = allData.Length;
            while (lRtn != 0)
            {
                start = allData.IndexOf("<Placemark");
                end   = allData.IndexOf("</Placemark>");

                if (start == -1)
                {
                    // データ終了
                    break;
                }

                // １データを取得する
                oneData = allData.Substring(start, (end + 12) - start);

                // 作図データから作図クラス作成
                csGeo = ConvertToClass(oneData);

                if (csGeo == null)
                {
                    continue;
                }

                // データクラス追加
                csFigData.AddLast(csGeo);

                // １データを削除する
                allData = allData.Remove(0, end + 12);

                lRtn = allData.Length;
            }

            return csFigData;
        }

        /// <summary><para>method outline:</para>
        /// <para>作図データから作図クラス作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ConvertToClass(oneData)</para>
        /// </example>
        /// <param name="oneData">作図データ</param>
        /// <returns>null != 正常, null = 異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private HySGeometry ConvertToClass(string oneData)
        {
            HySGeometry         csGeo = null;
            int                 start = 0;
            int                 end = 0;
            string              FigType = "";


            start = oneData.IndexOf("<name>");
            end = oneData.IndexOf("</name>");

            // 形状情報を取得する
            FigType = oneData.Substring((start + 6), end - (start + 6));

            if (FigType == "Nm_LINE")
            {
                csGeo = new HySGeometryLine();
                PolyLineToGeoClass(csGeo, oneData);
            }
            else if (FigType == "Nm_POLYLINE")
            {
                csGeo = new HySGeometryPolyLine();
                PolyLineToGeoClass(csGeo, oneData);
            }
            else if (FigType == "Nm_SPLINE")
            {
                csGeo = new HySGeometrySpline();
                PolyLineToGeoClass(csGeo, oneData);
            }
            else if (FigType == "Nm_POLYGON")
            {
                csGeo = new HySGeometryPolygon();
                PolygonToGeoClass(csGeo, oneData);
            }
            else if (FigType == "Nm_RECT")
            {
                csGeo = new HySGeometryRectangle();
                PolygonToGeoClass(csGeo, oneData);
            }
            else if (FigType == "Nm_ELLIPSE")
            {
                csGeo = new HySGeometryEllipse();
                PolygonToGeoClass(csGeo, oneData);
            }
            else if (FigType == "Nm_TEXT")
            {
                csGeo = new HySGeometryText();
                TextToGeoClass(csGeo, oneData);
            }
            else if (FigType == "Nm_SYMBOL")
            {
                csGeo = new HySGeometrySymbol();
                SymbolToGeoClass(csGeo, oneData);
            }
            else
            {
                return null;
            }

            return csGeo;
        }

        /// <summary><para>method outline:</para>
        /// <para>図形基底抽象化クラス化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PolyLineToGeoClass(csGeo, oneData)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="oneData">作図データ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void PolyLineToGeoClass(HySGeometry csGeo, string oneData)
        {
            int     start = 0;
            int     end = 0;
            string  data = "";
            short   sR = 0;
            short   sG = 0;
            short   sB = 0;
            short   sA = 0;
            

            // データ識別子を取得する
            start = oneData.IndexOf("<Placemark id = ");
            end = oneData.IndexOf("\">");

            data = oneData.Substring((start + 17), end - (start + 17));
            csGeo.SetID(new HySID(data));


            // 線色を取得する
            start = oneData.IndexOf("<color>");
            end = oneData.IndexOf("</color>");

            data = oneData.Substring((start + 7), end - (start + 7));
            StringToColor(data, ref sR, ref sG, ref sB, ref sA);
            csGeo.SetLineColor(sR, sG, sB);
            csGeo.SetLineTransparency(sA);


            // 線幅を取得する
            start = oneData.IndexOf("<width>");
            end = oneData.IndexOf("</width>");

            data = oneData.Substring((start + 7), end - (start + 7));
            csGeo.SetLineWidth(short.Parse(data));


            // 高度モードを取得する
            GetAltitudeMode(ref csGeo, oneData);


            // 座標点列を取得する
            start = oneData.IndexOf("<coordinates>");
            end = oneData.IndexOf("</coordinates>");
            data = oneData.Substring((start + 13), end - (start + 13));

            // 座標点列取得処理
            SetCoordinates(csGeo, data);
        }

        /// <summary><para>method outline:</para>
        /// <para>図形基底抽象化クラス化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> PolygonToGeoClass(csGeo, oneData)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="oneData">作図データ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void PolygonToGeoClass(HySGeometry csGeo, string oneData)
        {
            int     start = 0;
            int     end = 0;
            string  data = "";
            short   sR = 0;
            short   sG = 0;
            short   sB = 0;
            short   sA = 0;
            HySGeometry.FillStyle   eFill = HySGeometry.FillStyle.FILL;


            // 図形基底抽象化クラス化処理
            PolyLineToGeoClass(csGeo, oneData);


            // 線色を取得する
            start = oneData.IndexOf("<color>");
            end = oneData.IndexOf("</color>");

            // 面色を取得する
            start = oneData.IndexOf("<color>", end);
            end = oneData.IndexOf("</color>", start);

            data = oneData.Substring((start + 7), end - (start + 7));
            StringToColor(data, ref sR, ref sG, ref sB, ref sA);
            csGeo.SetPolyColor(sR, sG, sB);
            csGeo.SetPolyTransparency(sA);


            // 塗り潰しパターンを取得する
            start = oneData.IndexOf("<fill>");
            end = oneData.IndexOf("</fill>");

            data = oneData.Substring((start + 6), end - (start + 6));
            if (data == "2")
            {
                eFill = HySGeometry.FillStyle.BLANK;
            }
            csGeo.SetFillStyle(eFill);
        
        }

        /// <summary><para>method outline:</para>
        /// <para>図形基底抽象化クラス化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> TextToGeoClass(csGeo, oneData)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="oneData">作図データ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void TextToGeoClass(HySGeometry csGeo, string oneData)
        {
            int         start = 0;
            int         end = 0;
            string      data = "";
            short       sR = 0;
            short       sG = 0;
            short       sB = 0;
            short       sA = 0;
            HySString   csString = null;


            // データ識別子を取得する
            start = oneData.IndexOf("<Placemark id = ");
            end = oneData.IndexOf("\">");

            data = oneData.Substring((start + 17), end - (start + 17));
            csGeo.SetID(new HySID(data));


            // 文字色を取得する
            start = oneData.IndexOf("<color>");
            end = oneData.IndexOf("</color>");

            data = oneData.Substring((start + 7), end - (start + 7));
            StringToColor(data, ref sR, ref sG, ref sB, ref sA);
            csGeo.SetLineColor(sR, sG, sB);
            csGeo.SetLineTransparency(sA);


            // 形状情報を取得する
            start = oneData.IndexOf("<name>");
            end = oneData.IndexOf("</name>");

            // 文字列情報を取得する
            start = oneData.IndexOf("<name>", end);
            end = oneData.IndexOf("</name>", start);

            data = oneData.Substring((start + 6), end - (start + 6));
            csString = new HySString(data);
            ((HySGeometryText)csGeo).SetString(csString);


            // 座標点列を取得する
            start = oneData.IndexOf("<coordinates>");
            end = oneData.IndexOf("</coordinates>");
            data = oneData.Substring((start + 13), end - (start + 13));


            // 座標点列取得処理
            SetCoordinates(csGeo, data);
        }

        /// <summary><para>method outline:</para>
        /// <para>図形基底抽象化クラス化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SymbolToGeoClass(csGeo, oneData)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="oneData">作図データ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SymbolToGeoClass(HySGeometry csGeo, string oneData)
        {
            int             start = 0;
            int             end = 0;
            string          data = "";
            HySObjectKind   csSymbolKind = null;
            short           sR = 0;
            short           sG = 0;
            short           sB = 0;
            short           sA = 0;


            // データ識別子を取得する
            start = oneData.IndexOf("<Placemark id = ");
            end = oneData.IndexOf("\">");

            data = oneData.Substring((start + 17), end - (start + 17));
            csGeo.SetID(new HySID(data));


            // シンボル色を取得する
            start = oneData.IndexOf("<color>");
            end = oneData.IndexOf("</color>");

            data = oneData.Substring((start + 7), end - (start + 7));
            StringToColor(data, ref sR, ref sG, ref sB, ref sA);
            ((HySGeometrySymbol)csGeo).SetSymbolColor(sR, sG, sB);
            ((HySGeometrySymbol)csGeo).SetSymbolTransparency(sA);


            // シンボル種別を取得する
            start = oneData.IndexOf("<href>");
            end = oneData.IndexOf("</href>");

            data = oneData.Substring((start + 6), end - (start + 6));
            data = data.Replace(" ", "");

            if (m_csSymTable.ContainsValue(data) == true)
            {
                foreach (string key in m_csSymTable.Keys)
                {
                    if (data == (string)m_csSymTable[key])
                    {
                        csSymbolKind = new HySObjectKind(key);
                        break;
                    }
                }
            }
            else
            {
                csSymbolKind = new HySObjectKind();
            }

            ((HySGeometrySymbol)csGeo).SetSymbolKind(csSymbolKind);


            // 座標点列を取得する
            start = oneData.IndexOf("<coordinates>");
            end = oneData.IndexOf("</coordinates>");
            data = oneData.Substring((start + 13), end - (start + 13));


            // 座標点列取得処理
            SetCoordinates(csGeo, data);
        }

        /// <summary><para>method outline:</para>
        /// <para>文字列色情報書式化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> StringToColor(sColor, sR, sG, sB, sA)</para>
        /// </example>
        /// <param name="sColor">文字列型色情報</param>
        /// <param name="sR">参照型Ｒ</param>
        /// <param name="sG">参照型Ｇ</param>
        /// <param name="sB">参照型Ｂ</param>
        /// <param name="sA">参照型Ａ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void StringToColor(string sColor, ref short sR, ref short sG, ref short sB, ref short sA)
        {
            string  data = "";

            data = sColor.Substring(0, 2);
            sA = (short)Convert.ToInt32(data, 16);

            data = sColor.Substring(2, 2);
            sR = (short)Convert.ToInt32(data, 16);

            data = sColor.Substring(4, 2);
            sG = (short)Convert.ToInt32(data, 16);

            data = sColor.Substring(6, 2);
            sB = (short)Convert.ToInt32(data, 16);
        }

        /// <summary><para>method outline:</para>
        /// <para>オブジェクト高度モード取得処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetAltitudeMode(ref csGeo, oneData)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="oneData">文字列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void GetAltitudeMode(ref HySGeometry csGeo, string oneData)
        {
            int     start = 0;
            int     end = 0;
            string  data = "";
            bool    bMode = false;

            HySGeometry.AltMode eAltMode = HySGeometry.AltMode.ABSOLUTE;


            // 地面からの突き出しモードを取得する
            start = oneData.IndexOf("<extrude>");
            end = oneData.IndexOf("</extrude>");

            if (start == -1)
            {
                return;
            }

            data = oneData.Substring((start + 9), end - (start + 9));
            if (data == "1")
            {
                bMode = true;
            }
            csGeo.SetExtrudeMode(bMode);
            bMode = false;


            // 地面に沿った表示を行うモード取得
            start = oneData.IndexOf("<tessellate>");
            end = oneData.IndexOf("</tessellate>");

            if (start == -1)
            {
                return;
            }

            data = oneData.Substring((start + 12), end - (start + 12));
            if (data == "1")
            {
                bMode = true;
            }
            csGeo.SetTessellateMode(bMode);


            // 高度設定モード取得
            start = oneData.IndexOf("<altitudeMode>");
            end = oneData.IndexOf("</altitudeMode>");

            if (start == -1)
            {
                return;
            }

            data = oneData.Substring((start + 14), end - (start + 14));

            if (data == "clampToGround")
            {
                eAltMode = HySGeometry.AltMode.CLAMP_TO_GND;
            }
            if (data == "relativeToGround")
            {
                eAltMode = HySGeometry.AltMode.RELATIVE_GND;
            }

            csGeo.SetAltMode(eAltMode);
        }
        /// <summary><para>method outline:</para>
        /// <para>座標点列取得処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetCoordinates(csGeo, oneData)</para>
        /// </example>
        /// <param name="csGeo">図形基底抽象化クラス</param>
        /// <param name="oneData">文字列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetCoordinates(HySGeometry csGeo, string oneData)
        {
            int         start = 0;
            int         end = 0;
            long        loop = 0;
            string      data = "";
            char[]      sep = new char[] {','};
            string[]    buf = null;

            double      dLon = 0.0;
            double      dLat = 0.0;
            double      dAlt = 0.0;
            double      dSWLat = 90.0;
            double      dSWLon = 180.0;
            double      dNELat = 0.0;
            double      dNELon = 0.0;

            HySGeoPoint         csPoint = null;
            HySDataLinkedList   csPtList = null;
            HySDataRoot         csDtRoot = null;


            // データリンクリストクラス
            csPtList = new HySDataLinkedList();

            csDtRoot = new HySGeoPoint(0.0, 0.0, 0.0);


            end = oneData.IndexOf("\n");

            while (end > -1)
            {
                data = oneData.Substring(start, end);
                buf = data.Split(sep);

                dLon = double.Parse(buf[0]);
                dLat = double.Parse(buf[1]);
                dAlt = double.Parse(buf[2]);

                // 地図上位置情報 データクラス
                csPoint = new HySGeoPoint(dLat, dLon, dAlt);

                // 直線
                if (csGeo is HySGeometryLine)
                {
                    if (loop == 1)
                    {
                        // 終点
                        ((HySGeometryLine)csGeo).SetEndPoint(csPoint);
                        break;
                    }

                    // 始点
                    ((HySGeometryLine)csGeo).SetStartPoint(csPoint);
                }

                // 文字列
                if (csGeo is HySGeometryText)
                {
                    ((HySGeometryText)csGeo).SetPosition(csPoint);
                    break;
                }
                
                // シンボル
                if (csGeo is HySGeometrySymbol)
                {
                    ((HySGeometrySymbol)csGeo).SetPosition(csPoint);
                    break;
                }

                // 折れ線
                if (csGeo is HySGeometryPolyLine)
                {
                    ((HySGeometryPolyLine)csGeo).AddPoint(csPoint);
                }

                // 多角形
                if (csGeo is HySGeometryPolygon)
                {
                    ((HySGeometryPolygon)csGeo).AddPoint(csPoint);
                }

                // 曲線
                if (csGeo is HySGeometrySpline)
                {
                    ((HySGeometrySpline)csGeo).AddPoint(csPoint);
                }

                // 矩形
                if (csGeo is HySGeometryRectangle)
                {
                    if (loop == 0)
                    {
                        // 南西頂点緯度設定
                        ((HySGeometryRectangle)csGeo).SetSWLatitude(dLat);
                        // 南西頂点経度設定
                        ((HySGeometryRectangle)csGeo).SetSWLongitude(dLon);
                        // 標高設定
                        ((HySGeometryRectangle)csGeo).SetAltitude(dAlt);
                    }
                    if (loop == 2)
                    {
                        // 北東頂点緯度設定
                        ((HySGeometryRectangle)csGeo).SetNELatitude(dLat);
                        // 北東頂点経度設定
                        ((HySGeometryRectangle)csGeo).SetNELongitude(dLon);

                        break;
                    }
                }

                // 楕円
                if (csGeo is HySGeometryEllipse)
                {
                    if (dLat < dSWLat)
                    {
                        // 南西緯度
                        dSWLat = dLat;
                        ((HySGeometryEllipse)csGeo).SetSWLatitude(dSWLat);
                    }
                    if (dLon < dSWLon)
                    {
                        // 南西経度
                        dSWLon = dLon;
                        ((HySGeometryEllipse)csGeo).SetSWLongitude(dSWLon);
                    }
                    if (dLat > dNELat)
                    {
                        // 北東緯度
                        dNELat = dLat;
                        ((HySGeometryEllipse)csGeo).SetNELatitude(dNELat);
                    }
                    if (dLon > dNELon)
                    {
                        // 北東経度
                        dNELon = dLon;
                        ((HySGeometryEllipse)csGeo).SetNELongitude(dNELon);
                    }
                }

                oneData = oneData.Remove(0, end + 1);
                end = oneData.IndexOf("\n");

                loop++;
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>２次元メッシュクラスからＫＭＬファイル作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlFileFromDim2Mesh(csKmlFile)</para>
        /// </example>
        /// <param name="csDim2Mesh">２次元地理メッシュ情報</param>
        /// <param name="outPath">出力ファイル名（フルパス）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void CreateKmlFileFromDim2Mesh(HySGeoDim2MeshSerialData csDim2Mesh, HySString outPath)
        {
            double  dSWLat = 0.0;
            double  dSWLon = 0.0;
            double  dNELat = 0.0;
            double  dNELon = 0.0;

            double  dUpAlt = 0.0;
            double  dLwAlt = 0.0;

            long    lLatDim = 0;
            long    lLonDim = 0;
            long    lAltDim = 0;

            double  dx = 0.0;
            double  dy = 0.0;
            double  dLat = 0.0;
            double  dLon = 0.0;

            short                   sR = 255;
            double                  dw = 0.0;
            double[]                dData = null;
            string                  stData = "";
            long                    lCnt = 0;
            int                     num = 0;
            HySTime                 csTime = null;
            string                  sTime = "";
            HySString               outPath2 = null;

            HySGeoMapFigureData     csFigData = null;
            HySGeometryRectangle    csRect = null;
            HySGeoDim2MeshData      csDim2Data = null;
            HySCellData[,]          csCellData = null;


            // 地理的位置取得
            csDim2Mesh.GetLocation(ref dSWLat, ref dSWLon, ref dNELat, ref dNELon);

            // 高度位置取得
            csDim2Mesh.GetAltitude(ref dUpAlt, ref dLwAlt);

            // 緯度方向のメッシュ分割数取得
            lLatDim = csDim2Mesh.GetLatDimension();

            // 経度方向のメッシュ分割数取得
            lLonDim = csDim2Mesh.GetLonDimension();

            // 高度方向のメッシュ分割数取得
            lAltDim = csDim2Mesh.GetAltDimension();


            // 緯度方向の１区画算出（度単位）
            dy = (dNELat - dSWLat) / (double)lLatDim;

            // 経度方向の１区画算出（度単位）
            dx = (dNELon - dSWLon) / (double)lLonDim;


            // 時系列データの数を取得する
            lCnt = csDim2Mesh.GetCount();

            // カーソル位置を最初に持ってくる
            csDim2Mesh.SetCursorFirst();

            outPath2 = new HySString();

            // 時系列データの数だけＫＭＬを作成する
            for (int p = 0; p < lCnt; p++)
            {
                outPath2.SetString(outPath);

                // ２次元地理メッシュ情報 データクラス
                csDim2Data = ((HySGeoDim2MeshData)csDim2Mesh.GetCursorData());

                // メッシュ情報インスタンス取得
                csCellData = csDim2Data.GetMeshData();


                // 自由図形表示データクラス
                csFigData = new HySGeoMapFigureData();

                dLat = dSWLat;
                dLon = dSWLon;
                for (int i = 0; i < lLatDim; i++)
                {
                    for (int j = 0; j < lLonDim; j++)
                    {
                        // ＧＩＳ矩形クラス
                        csRect = new HySGeometryRectangle();

                        // ２点座標設定
                        csRect.SetSWLatitude(dLat);                 // 南西緯度
                        csRect.SetSWLongitude(dLon);                // 南西経度
                        csRect.SetNELatitude(dLat + dy);            // 北東緯度
                        csRect.SetNELongitude(dLon + dx);           // 北東経度

                        // メッシュ保持データ
                      //dData = csCellData[lLonDim, lLatDim].m_dData;
                        dData = csCellData[j, i].m_dData;

                        for (int k = 0; k < m_csColTable.Count; k++)
                        {
                            // キーテーブル
                            dw = m_lColKey[k];

                            if (dData[0] < 0)
                            {
                                sR = 0;
                                break;
                            }

                            if (dw >= dData[0])
                            {
                                // 色情報取得
                                stData = dw.ToString("0.0");
                                stData = (string)m_csColTable[stData];

                                if (stData == null)
                                {
                                    break;
                                }

                                sR = short.Parse(stData);
                                break;
                            }
                        }

                        ((HySGeometry)csRect).SetLineColor(0, 0, 0);
                        ((HySGeometry)csRect).SetPolyColor(255, 0, 0);
                        ((HySGeometry)csRect).SetPolyTransparency(sR);
                        ((HySGeometry)csRect).SetFillStyle(HySGeometry.FillStyle.FILL);
                        ((HySGeometry)csRect).SetLineWidth(1);
                        csFigData.AddLast((HySGeometry)csRect);

                        csRect = null;

                        // 次の座標
                        dLon = dLon + dx;
                    }

                    // 次の座標
                    dLat = dLat + dy;
                    dLon = dSWLon;
                }

                //num = outPath2.IndexOf(".kml");
                num = outPath2.ToString().LastIndexOf(".");

                csTime = csDim2Data.GetLastTime();

                sTime = "_";
                sTime += HySCalendar.ToString(csTime, HySCalendar.FORMAT.lSW_DAY);
                sTime = sTime.Replace(':', '-');

                //outPath2 = outPath2.Insert(num, sTime);
                if (num >= 0)
                {  // 識別子が有る場合
                    outPath2 = outPath2.Insert(num, sTime);
                }
                else
                {  // 識別子が無い場合
                    outPath2 = outPath2.Insert(outPath2.GetLength(), sTime);
                }

                // 作図データからＫＭＬファイル作成
                CreateKmlFileFromFigureData(csFigData, outPath2);

                // カーソル位置を次の位置に持ってくる
                csDim2Mesh.MoveCursorNext();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬファイルから２次元メッシュクラス作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateDim2MeshFromKmlFile(csKmlFile)</para>
        /// </example>
        /// <param name="csKmlFile">ＫＭＬファイル（フルパス）</param>
        /// <returns>null != 正常, null = 異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>将来用：コーディング</para>
        /// </remarks>
        public HySGeoDim2MeshSerialData CreateDim2MeshFromKmlFile(HySString csKmlFile)
        {
            HySGeoDim2MeshSerialData    csDim2Mesh = null;
            HySGeoDim2MeshData          csDim2Data = null;
            HySCellData[,]              csCellData = null;

            long            lRtn = 0;
            bool            isFlag = false;
            bool            isFind = false;
            bool            bFind = false;
            HySFile         csFile = null;
            HySString       csFigStr = null;
            string          oneData = "";

            int             start = 0;
            short           sR = 255;

            double          dU = 0.0;
            double          dD = 0.0;
            string          stU = null;
            string          stD = null;
            short           sU = 0;
            short           sD = 0;

            // ファイルクラス
            csFile = new HySFile(csKmlFile);
            csFigStr = new HySString("");


            // ファイルオープン処理
            lRtn = csFile.Open(HySFile.OPEN_MODE.OPEN, HySFile.READ_WRITE_MODE.READ,
                               HySFile.DIRECTORY_MODE.MK_DIR);
            if (lRtn != 0)
            {
                return null;
            }


            // ファイルからテキストデータを読み込む
            while (csFile.ReadText(ref csFigStr) != 0)
            {
                if (csFigStr.IndexOf("</coordinates>") > -1)
                {
                    isFlag = false;

                    // 地理的位置設定
                    SetLocation(oneData);
                    oneData = "";
                }

                if (isFlag == true)
                {
                    oneData += csFigStr.ToString() + "\n";
                }

                if (csFigStr.IndexOf("<coordinates>") > -1)
                {
                    isFlag = true;
                }


                if (isFind == true)
                {
                    oneData = csFigStr.ToString();

                    // 透過率を取得する
                    start = oneData.IndexOf(">");
                    oneData = oneData.Substring(start + 1, 2);
                    sR = short.Parse(oneData, NumberStyles.HexNumber);

                    for (int k = 0; k < m_csColTable.Count - 1; k++)
                    {
                        // キーテーブル
                        dD = m_lColKey[k];

                        // 色情報取得
                        stD = dD.ToString("0.0");
                        stD = (string)m_csColTable[stD];


                        // キーテーブル
                        dU = m_lColKey[k + 1];

                        // 色情報取得
                        stU = dU.ToString("0.0");
                        stU = (string)m_csColTable[stU];

                        if (stD == null || stU == null)
                        {
                            break;
                        }

                        // 上下限値
                        sD = short.Parse(stD);
                        sU = short.Parse(stU);

                        if ((sD <= sR) && (sR < sU))
                        {
                            bFind = true;
                            break;
                        }
                    }

                    if (bFind == false)
                    {
                        dD = dU;
                    }

                    isFind = false;
                    oneData = "";
                }

                if (csFigStr.IndexOf("<PolyStyle>") > -1)
                {
                    isFind = true;
                }
            }

            csFile.Close();


            // ２次元地理メッシュ情報時系列 データクラス
            csDim2Mesh = new HySGeoDim2MeshSerialData(csKmlFile);

            // 地理的位置設定
            csDim2Mesh.SetLocation(m_dSWLatitude, m_dSWLongitude,
                                   m_dNELatitude, m_dNELongitude);

            // 緯度方向のメッシュ分割数設定
            csDim2Mesh.SetLatDimension(m_lLatDim - 1);

            // 経度方向のメッシュ分割数設定
            csDim2Mesh.SetLonDimension(m_lLonDim - 1);


            // ２次元地理メッシュ情報 データクラス
            csDim2Data = ((HySGeoDim2MeshData)csDim2Mesh.CreateNewRecord());

            // メッシュ情報インスタンス取得
            csCellData = csDim2Data.GetMeshData();

            // メッシュ保持データ
            csCellData[m_lLonDim, m_lLatDim].m_dData[0] = dD;

            return csDim2Mesh;
        }

        /// <summary><para>method outline:</para>
        /// <para>地理的位置設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLocation(dSWLat, dSWLon, dNELat, dNELon) </para>
        /// </example>
        /// <param name="oneData">位置情報文字列</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetLocation(string oneData)
        {
            int         start = 0;
            int         end = 0;
            double      dLon = 0.0;
            double      dLat = 0.0;
            double      dAlt = 0.0;

            string      data = "";
            char[]      sep = new char[] {','};
            string[]    buf = null;


            end = oneData.IndexOf("\n");


            while (end > -1)
            {
                data = oneData.Substring(start, end);
                buf = data.Split(sep);

                dLon = double.Parse(buf[0]);
                dLat = double.Parse(buf[1]);
                dAlt = double.Parse(buf[2]);

                if (dLat < m_dSWLatitude)
                {
                    // 南西緯度
                    m_dSWLatitude = dLat;
                }
                if (dLon < m_dSWLongitude)
                {
                    // 南西経度
                    m_dSWLongitude = dLon;
                }
                if (dLat > m_dNELatitude)
                {
                    // 北東緯度
                    m_dNELatitude = dLat;
                    m_lLatDim++;
                }
                if (dLon > m_dNELongitude)
                {
                    // 北東経度
                    m_dNELongitude = dLon;
                    m_lLonDim++;
                }

                oneData = oneData.Remove(0, end + 1);
                end = oneData.IndexOf("\n");
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>３次元メッシュクラスからＫＭＬファイル作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateKmlFileFromDim3Mesh(csDim3Mesh,outPath)</para>
        /// </example>
        /// <param name="csDim3Mesh">３次元地理メッシュ情報</param>
        /// <param name="outPath">出力ファイル名（フルパス）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void CreateKmlFileFromDim3Mesh(HySGeoDim3MeshSerialData csDim3Mesh, HySString outPath)
        {
        }

        /// <summary><para>method outline:</para>
        /// <para>ＫＭＬファイルから３次元メッシュクラス作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateDim3MeshDataClass(csKmlFile)</para>
        /// </example>
        /// <param name="csKmlFile">ＫＭＬファイル</param>
        /// <returns>null != 正常, null = 異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySGeoDim3MeshSerialData CreateDim3MeshFromKmlFile(HySString csKmlFile)
        {
            HySGeoDim3MeshSerialData csDim3Mesh = null;


            csDim3Mesh = new HySGeoDim3MeshSerialData(csKmlFile);


            return csDim3Mesh;
        }
    }
}
