﻿// <summary>ソースコード：>ＨＹＳＳＯＰキュー制御クラス</summary>
// <author>CommonMP</author>

using System;
using System.Threading;
using System.Collections.Generic;
using System.Text;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.CoreImpl.HSData;

namespace CommonMP.HYSSOP.CoreImpl.HSTools
{
    /// <summary><para>class outline:</para>
    /// <para>キュー制御クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2008/10/01][新規作成]</para>
    /// </remarks>
    public class HySQueue
    {
        /// <summary> オブジェクト名称 </summary>
        protected HySString m_csName;

        /// <summary> キュー最大数 </summary>
        protected long m_lMaxQueue = 0;                 // 最大キュー数

        /// <summary> 書き込みポインタ </summary>
        protected long m_lWritePnt = 0;                   // 書き込み位置
        /// <summary> 読み込みポインタ </summary>
        protected long m_lReadPnt = 0;                    // 読み出し位置

        /// <summary> 同時書き込み排他ミューテックス </summary>
        private HySMutex m_csPushPull = null;           // 同時書き込み排他ミューテックス
        /// <summary> エンプティ監視セマフォ </summary>
        HySSemaphore m_csEmpty = null;                   // Empty_監視セマフォ
        /// <summary> フル監視セマフォ </summary>
        HySSemaphore m_csFull = null;                    // Full__監視セマフォ

        /// <summary> キュー格納データ </summary>
        private HySQueueNode[] m_csNode;                // データ

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySQueue csQueue = new HySQueue(lMaxLength) </para>
        /// </example>
        /// <param name="lMaxLength">キュー最大登録数</param>
        /// <returns>HySQueue  生成したインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySQueue(long lMaxLength)
        {
            m_csName = new HySString();
            m_lMaxQueue = lMaxLength;

            m_csPushPull = new HySMutex();

            m_csEmpty = new HySSemaphore( 0, (int)m_lMaxQueue); // 初期ではエントリをブロックする(リリース待ち)

            m_csFull = new HySSemaphore( (int)m_lMaxQueue, (int)m_lMaxQueue); // 初期では最大キュー数分だけエントリ可

            m_csNode = new HySQueueNode[m_lMaxQueue];
            for (long lLoop = 0; lLoop < m_lMaxQueue; lLoop++)
            {
                m_csNode[lLoop] = new HySQueueNode();
            }
        }

        //---将来拡張用----------------------------------------------------------------------------------------------------
        ///// <summary><para>method outline:</para>
        ///// <para>コンストラクタ</para>
        ///// </summary>
        ///// <example><para>usage:</para>
        ///// <para> HySQueue csHySQueue = new HySQueue(csName, lMaxLength) </para>
        ///// </example>
        ///// <param name="csName">キー名称</param>
        ///// <param name="lMaxLength">キュー最大登録数</param>
        ///// <returns>無し</returns>
        ///// <exception cref="">無し</exception>
        ///// <remarks><para>remarks:</para>
        ///// <para>注意：他のHySQueueインスタンスとcsNameがかぶらないようにすること</para>
        ///// <para>　　　（システム全体でcsNameがユニークになること）</para>
        ///// </remarks>
        //public HySQueue(HySString csName, long lMaxLength)
        //{
        //    m_csName = csName;
        //    m_lMaxQueue = lMaxLength;

        //    m_csPushPull = new HySMutex(csName.GetString() + "_mtx");

        //    m_csEmpty = new HySSemaphore(csName.GetString() + "_epty", 0, m_lMaxQueue); // 初期ではエントリをブロックする(リリース待ち)

        //    m_csFull = new HySSemaphore(csName.GetString() + "_full", m_lMaxQueue, m_lMaxQueue); // 初期では最大キュー数分だけエントリ可

        //    m_csNode = new HySQueueNode[m_lMaxQueue];
        //    for (long lLoop = 0; lLoop < m_lMaxQueue; lLoop++)
        //    {
        //        m_csNode[lLoop] = new HySQueueNode();
        //    }
        //}
        //---将来拡張用----------------------------------------------------------------------------------------------------

        /// <summary><para>method outline:</para>
        /// <para>データの積み込み</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bWork = PushObject(csData) </para>
        /// </example>
        /// <param name="csData">データ</param>
        /// <returns>Boolean　true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>データQueue積み込み Queue 満杯ならば、読み出されて空きが出来るまで待つ</para>
        /// </remarks>
        public Boolean PushObject(HySDataRoot csData)
        {
            m_csFull.Wait();

            this.MutexLock();
            m_csNode[m_lWritePnt].SetObject(csData);
            m_lWritePnt += 1;
            if (m_lWritePnt >= m_lMaxQueue)
            {
                m_lWritePnt = 0;
            }
            this.MutexUnlock();

            m_csEmpty.Release();

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>データの取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySDataRoot csDataRoot = PullObject() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDataRoot　データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>データ読み出し Queue 空ならば、積み込まれるまで待つ</para>
        /// </remarks>
        public HySDataRoot PullObject()
        {
            HySDataRoot csRtn = null;
            m_csEmpty.Wait();

            this.MutexLock();
            csRtn = m_csNode[m_lReadPnt].GetObject();
            
            m_csNode[m_lReadPnt].SetObject(null);  //取り外してしまった後にはNULLを入れておく

            m_lReadPnt += 1;
            if (m_lReadPnt >= m_lMaxQueue)
            {
                m_lReadPnt = 0;
            }
            this.MutexUnlock();

            m_csFull.Release();

            return csRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>キュー番号の積み込み</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bWork = PushNumber(lNumber) </para>
        /// </example>
        /// <param name="lNumber">キューの番号</param>
        /// <returns>Boolean　true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>数（long）のQueue積み込み Queue 満杯ならば、読み出されて空きが出来るまで待つ</para>
        /// </remarks>
        public Boolean PushNumber(long lNumber)
        {
            m_csFull.Wait();

            this.MutexLock();
            m_csNode[m_lWritePnt].SetNumber(lNumber);
            m_lWritePnt += 1;
            if (m_lWritePnt >= m_lMaxQueue)
            {
                m_lWritePnt = 0;
            }
            this.MutexUnlock();

            m_csEmpty.Release();

            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>キュー番号取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lNum = PullNumber() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>long　キュー番号</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>数（long）の読み出し Queue 空ならば、積み込まれるまで待つ</para>
        /// </remarks>
        public long PullNumber()
        {
            long lRtn = -1;

            m_csEmpty.Wait();

            this.MutexLock();
            lRtn = m_csNode[m_lReadPnt].GetNumber();
            m_lReadPnt += 1;
            if (m_lReadPnt >= m_lMaxQueue)
            {
                m_lReadPnt = 0;
            }
            this.MutexUnlock();

            m_csFull.Release();

            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>排他制御設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MutexLock() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void MutexLock()
        {
            m_csPushPull.Wait(); // 同時書き込み排他ミューテックス
        }

        /// <summary><para>method outline:</para>
        /// <para>排他制御解除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> MutexUnlock() </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        protected void MutexUnlock()
        {
            m_csPushPull.Release(); // 同時書き込み排他ミューテックス
        }
    }
}
