﻿// <summary>ソースコード：評価用表示関連ビューアジョブクラス</summary>
// <author>CommonMP</author>

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.CompilerServices;

using CommonMP.HYSSOP.Interface.HSData;
using CommonMP.HYSSOP.Interface.HSViewer;
using CommonMP.HYSSOP.CoreImpl;
using CommonMP.HYSSOP.CoreImpl.HSData;
using CommonMP.HYSSOP.CoreImpl.HSTools;
using CommonMP.HYSSOP.CoreImpl.HSBusiProc;


namespace CommonMP.HYSSOP.OptionImpl.HSViewer.DotNetViewer
{
    /// <summary><para>class outline:</para>
    /// <para>評価用表示関連ビューア共通表示処理クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2009/05/01][新規作成]</para>
    /// </remarks>
    internal class HySDispViewerJob : HySViewerJob
    {
        /// <summary>業務動作中フラグ</summary>
        protected bool m_bWorkFlg = false;

        /// <summary>シミュレータ識別子</summary>
        protected HySKind m_csSimKindID;

        /// <summary>ビューア</summary>
        protected HySViewer m_csViewer;

        /// <summary>スクリーンファクトリ</summary>
        protected HySScreenFactory m_csScreenFactory;

        /// <summary>フォームファクトリ</summary>
        protected HySViewFormFactory m_csViewFormFactory;

        /// <summary>スクリーン識別管理ハッシュテーブル</summary>
        protected Hashtable m_csScreenTbl = null;

        /// <summary>GIS表示制御用ビューア共通表示処理クラス</summary>
        protected HySGISCtlViewerSubJob m_csGISCtlSubJob = new HySGISCtlViewerSubJob();

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySViewerJob csJob = new HySDispViewerJob( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySDispViewerJob 生成されたインスタンス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public HySDispViewerJob()
        {
            m_csScreenTbl = new Hashtable();    // スクリーン管理用ハッシュテーブル
            m_csGISCtlSubJob.SetOwnerJob(this);
        }

        /// <summary><para>method outline:</para>
        /// <para>スクリーン取り込み</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> LoadScreen( csPrc, csScreenKind ) </para>
        /// </example>
        /// <param name="csPrc">スクリーン</param>
        /// <param name="csScreenKind">画面種別識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void LoadScreen(HySScreen csPrc, HySObjectKind csScreenKind)
        {
            // スクリーンにビューア設定
            csPrc.SetViewer(m_csViewer);
            // （種別識別子＋データ識別子＋画面種別識別子）でユニークキー作成
            HySIdentifier csDataID = csPrc.GetID();
            HySKind csSimKind = csPrc.GetSimKind();
            HySString csPrcID = (HySString)csDataID.GetString();
            HySString csPrcKind = (HySString)csSimKind.GetString();
            HySString csScKind = new HySString("-") + (HySString)csScreenKind.GetString();
            HySString csPrcKey = csPrcKind + csPrcID + csScKind;
            m_csScreenTbl[csPrcKey.ToString()] = csPrc;
        }

        /// <summary><para>method outline:</para>
        /// <para>スクリーンを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetScreen( csScreen, csScreenKind ) </para>
        /// </example>
        /// <param name="csScreen">スクリーン</param>
        /// <param name="csScreenKind">画面種別識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetScreen(HySScreen csScreen, HySObjectKind csScreenKind)
        {
            this.LoadScreen(csScreen, csScreenKind);
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetSimKind(csSimKindID)</para>
        /// </example>
        /// <param name="csSimKindID">シミュレーター種別識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetSimKind(HySKind csSimKindID)
        {
            m_csSimKindID = csSimKindID;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySKind csKind = GetSimKind()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySKind シミュレーター種別識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySKind GetSimKind()
        {
            return m_csSimKindID;
        }

        /// <summary><para>method outline:</para>
        /// <para>シミュレーター種別識別子比較</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = EqualSimKind(csSimKindID)</para>
        /// </example>
        /// <param name="csSimKindID">シミュレーター種別識別子</param>
        /// <returns>Boolean true:同じ、false:異なる</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean EqualSimKind(HySKind csSimKindID)
        {
            return m_csSimKindID.Equals(csSimKindID);
        }

        /// <summary><para>method outline:</para>
        /// <para>ビューアを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetViewr( csViewr ) </para>
        /// </example>
        /// <param name="csViewr">ビューア</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetViewer(HySViewer csViewr)
        {
            m_csViewer = csViewr;
            m_csGISCtlSubJob.SetViewer(csViewr);
        }

        /// <summary><para>method outline:</para>
        /// <para>スクリーンファクトリを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetScreenFactory( csScreenFactory ) </para>
        /// </example>
        /// <param name="csScreenFactory">スクリーンファクトリ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetScreenFactory(HySScreenFactory csScreenFactory)
        {
            m_csScreenFactory = csScreenFactory;
        }

        /// <summary><para>method outline:</para>
        /// <para>フォームファクトリを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetViewFormFactory( csViewrFormFactory ) </para>
        /// </example>
        /// <param name="csViewFormFactory">フォームファクトリ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void SetViewFormFactory(HySViewFormFactory csViewFormFactory)
        {
            m_csViewFormFactory = csViewFormFactory;
        }

        /// <summary><para>method outline:</para>
        /// <para>指定画面表示</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = CmndShowScreen( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">イベント</param>
        /// <returns>Boolean true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean CmndShowScreen(HySSysEvent csEvent)
        {
            Boolean bRtn = false;
            HySEventObject csEventObj = null;

            // イベント情報展開
            long lEventSubNo = ((HySEventObject)csEvent).GetSubEventNo();
            HySID csID = ((HySEventObject)csEvent).GetSuppID();
            HySScreen csScreen = null;
            HySObjectKind csSimKind = (HySObjectKind)((HySEventObject)csEvent).GetToSimKind();
            // 指定種別の画面スクリーンをファクトリから生成
            HySObjectKind csScreenKind = this.GetScreenKind(((HySEventObject)csEvent).GetData()); // 画面識別子取得
            csScreen = this.GetScreen(csSimKind, csID, csScreenKind);
            if (csScreen == null)
            {  // スクリーンが未生成の場合
                csScreen = m_csScreenFactory.CreateScreen(csSimKind, csID, csScreenKind);
                if (csScreen != null)
                {  // スクリーン生成が正常の場合
                    if (csScreen is HySScreen)
                    { // 画面フォームファクトリからフォーム取得の場合
                        if (m_csViewFormFactory != null)
                        { // 画面フォームファクトリが有効な場合
                            HySObjectKind csFormKind = this.GetFormKind(((HySEventObject)csEvent).GetData()); // Form識別子取得
                            HySViewForm csViewFrom = m_csViewFormFactory.CreateViewForm(csSimKind, csID, csFormKind);
                            if (csViewFrom != null)
                            {  // 対応画面フォーム正常生成の場合
                                csScreen.SetViewer(m_csViewer); // ビューア設定
                                csScreen.SetViewerJob(m_csViewer.GetViewerJob(HySDefine.COMMON_DISP_BUSIPROCEDURE.GetString())); // ビューアジョブ設定
                                ((HySScreen)csScreen).SetViewForm(csViewFrom); // フォーム設定
                            }
                        }
                    }
                    csScreen.Initialize(); // 画面初期化
                    this.SetScreen(csScreen, csScreenKind); // スクリーン設定
                    csScreen.SetDispEventData(csEvent); // 特殊イベント処理
                    csScreen.ShowScreen(); // スクリーン可視化
                    // 画面表示完了報告を生成
                    csEventObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_DISPLY_OK);
                    bRtn = true;
                }
                else
                {  // スクリーン生成が異常の場合
                    // 画面表示異常報告を生成
                    csEventObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_DISPLY_NG);
                    bRtn = false;
                }
            }
            else
            {  // 既存スクリーンが有る場合
                //csScreen.SetDispEventData(csEvent); // 特殊イベント処理
                csScreen.ShowScreen(); // スクリーン可視化
                csEventObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.NOTICE_DISPLY_OK); // 画面表示完了報告
                bRtn = true;
            }
            // 業務に画面表示結果をイベント報告
            csEventObj.SetToSimKind(csSimKind);
            csEventObj.SetSuppID(csID);
            csEventObj.SetSubEventNo(lEventSubNo);
            csEventObj.SetData(csScreenKind);
            m_csViewer.PutEvent(csEventObj);
            return bRtn;
        }

        /// <summary><para>method outline</para>
        /// <para>スクリーン取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySScreen csScreen = GetScreen(csSimKind,csID,csScreenKind) </para>
        /// </example>
        /// <param name="csSimKind">業務種別識別子</param>
        /// <param name="csID">データ識別子</param>
        /// <param name="csScreenKind">画面種別識別子</param>
        /// <returns>HySScreen スクリーンクラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual HySScreen GetScreen(HySKind csSimKind, HySIdentifier csID, HySKind csScreenKind)
        {
            HySString csPrcID = (HySString)csID.GetString();
            HySString csPrcKind = (HySString)csSimKind.GetString();
            HySString csScKind = new HySString("-") + (HySString)csScreenKind.GetString();
            HySString csPrcKey = csPrcKind + csPrcID + csScKind;
            HySScreen csScreen = (HySScreen)m_csScreenTbl[csPrcKey.ToString()];
            return csScreen;
        }

        /// <summary><para>method outline</para>
        /// <para>スクリーン削除</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DeleteScreen(csSimKind,csID,csScreenKind) </para>
        /// </example>
        /// <param name="csSimKind">業務種別識別子</param>
        /// <param name="csID">データ識別子</param>
        /// <param name="csScreenKind">画面種別識別子</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void DeleteScreen(HySKind csSimKind, HySIdentifier csID, HySKind csScreenKind)
        {
            HySString csPrcID = (HySString)csID.GetString();
            HySString csPrcKind = (HySString)csSimKind.GetString();
            HySString csScKind = new HySString("-") + (HySString)csScreenKind.GetString();
            HySString csPrcKey = csPrcKind + csPrcID + csScKind;
            m_csScreenTbl.Remove(csPrcKey.ToString());
        }

        /// <summary><para>method outline:</para>
        /// <para>メニューコールバック処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> long lRtn = MenuCallBack( lCallBackNo,bCheckOnOff ) </para>
        /// </example>
        /// <param name="lCallBackNo">メニューコールバック番号</param>
        /// <param name="bCheckOnOff">チェックオンオフフラグ</param>
        /// <returns>long =0：正常、≠0：異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual long MenuCallBack(long lCallBackNo, Boolean bCheckOnOff)
        {
            long lRtn = 0;
            HySEventObject csEventObj;

            if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_LINEGRAPH ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_HYDROHYETO ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_RIVERBED ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_HQ ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_EXCEL ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_EDIT_CROSS_SECTION_DATA)
            { 
                // ●計算結果がDBに登録されている場合、その情報をグラフ表示する
                // 画面表示（基本画面、ハイドロ／ハイエトグラフ画面、河川水位縦断図画面表示、Ｈ－Ｑ曲線図画面表示）の場合
                csEventObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.CMND_DISP_SCREEN);
                if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_LINEGRAPH)
                {  // 基本画面の場合
                    csEventObj.SetData(HySDefine.DISP_FORM_LINE_GRAPH);
                }
                else if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_HYDROHYETO)
                {  // ハイドロ／ハイエトグラフ画面の場合
                    csEventObj.SetData(HySDefine.DISP_FORM_HYDRO_HYETO_GRAPH);
                }
                else if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_RIVERBED)
                {  // 河川水位縦断図画面の場合
                    csEventObj.SetData(HySDefine.DISP_FORM_RIVERBED_SHAPE);
                }
                else if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_HQ)
                {  // Ｈ－Ｑ曲線図画面の場合
                    csEventObj.SetData(HySDefine.DISP_FORM_HQ_GRAPH);
                }
                else if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_REPORT_EXCEL)
                {  // ＥＸＣＥＬ画面の場合
                    csEventObj.SetData(HySDefine.DISP_FORM_EXCEL);
                }
                csEventObj.SetToSimKind(HySDefine.COMMON_DISP_BUSIPROCEDURE);
                m_csViewer.PutEvent(csEventObj);
            }
            //else if (
            //    lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_FILE_LINEGRAPH 
            //    )
            //{   // 計算結果がファイルに保存されている場合、その情報をグラフに表示する。
            //    // ファイルベースなので、そのまま表示を行う。
            //}
            else if (
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_BASE ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_BASE_V1_0_0 ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_MESH_DB ||
                lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_EDIT_GIS_CROSSSECTION
                )
            {

                // ●計算結果がDBに登録されている場合、その結果を地図に表示する
                // 先ず背景地図を表示させる（２重起動は　HySDispBusiProcedureが制御する）
                HySEventObject csLclEventObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.CMND_DISP_SCREEN);
                csLclEventObj.SetData(HySDefine.DISP_FORM_GIS_BASEMAP);
                csLclEventObj.SetToSimKind(HySDefine.COMMON_DISP_BUSIPROCEDURE);
                m_csViewer.PutEvent(csLclEventObj);

                // 背景地図と対で横断面データ編集画面用GISスクリーン生成を行う
                csEventObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.CMND_DISP_SCREEN);
                csEventObj.SetToSimKind(HySDefine.COMMON_DISP_BUSIPROCEDURE);
                csEventObj.SetData(HySDefine.DISP_FORM_RIVER_CROSSSECTION);
                m_csViewer.PutEvent(csEventObj);

                if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_BASE)
                {
                    csEventObj = new HySEventObject(HySSysEvent.OBJID_VIEWER, HySEventObject.CMND_DISP_SCREEN);
                    csEventObj.SetToSimKind(HySDefine.COMMON_DISP_BUSIPROCEDURE);
                    csEventObj.SetData(HySDotNetViewerDefine.GIS_RESULT_CTL_SCREEN);    // 計算結果ファイル表示選択画面
                    csEventObj.SetSuppID(new HySID(HySDotNetViewerDefine.GIS_RESULT_CTL_SCREEN.ToString())); // 画面は一つの為、IDは固定
                    m_csViewer.PutEvent(csEventObj);
                }
                else
                {
                    if (lCallBackNo != HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_BASE_V1_0_0)
                    {
                        csEventObj = new HySEventObject(HySSysEvent.OBJID_BUSIPROCEDURE, HySEventObject.CMND_DISP_SCREEN);
                        csEventObj.SetToSimKind(HySDefine.COMMON_DISP_BUSIPROCEDURE);
                        if (lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_MESH_DB)
                        {  // メッシュの場合
                            csEventObj.SetData(HySDefine.DISP_FORM_GIS2D_MESH);
                        }
                        m_csViewer.PutEvent(csEventObj);
                    }
                }
            }
            //else if (
            //    lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_FILE_2MESH
            //    // || lCallBackNo == HySDotNetViewerDefine.SYSTEM_TOOL_DISP_GIS_FILE_3MESH
            //        )
            //{   // 計算結果（２次元メッシュ、３次元メッシュ）がDBに登録されている場合、その情報と地図に表示する
            //    // 先ず、表示するファイルを選択する画面を選び、そこからGISへ表示を依頼する
            //}
            else
            {
                lRtn = -1;
            }
            return lRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>初期化処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = Initialize( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>Boolean true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean Initialize()
        {
            m_bWorkFlg = true;
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>終了処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = Terminate( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>Boolean true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean Terminate()
        {
            m_bWorkFlg = false;
            HySString csPrcKey = null;
            HySScreen csScreen = null;
            // 全スクリーンの終了処理を実行
            foreach (string name in m_csScreenTbl.Keys)
            {  // ハッシュテーブル件数繰り返す
                csPrcKey = new HySString(name);
                csScreen = (HySScreen)m_csScreenTbl[csPrcKey.ToString()];
                if (csScreen != null)
                {  // スクリーンが有効の場合
                    csScreen.Terminate(); // 終了処理実施
                    csScreen = null;
                }
            }
            m_csScreenTbl.Clear(); // ハッシュテーブルクリア
            return true;
        }

        /// <summary><para>method outline:</para>
        /// <para>システム終了準備完了判別</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = ExitOK( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>Boolean true:システムExit準備OK　、false:システムExit準備NG</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean ExitOK()
        {
            if (m_bWorkFlg == true)
            {
                return false;
            }
            else
            {
                return true;
            }
        }
       
        /// <summary><para>method outline:</para>
        /// <para>イベントを受け取った時に動作するメソッド</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Boolean bRtn = EventCallback( csEvent ) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント</param>
        /// <returns>Boolean true:=正常, false:=異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual Boolean EventCallback(HySSysEvent csEvent)
        {
            Boolean bRtn = true;
            HySObjectKind csScreenKind = null;
            //String csMethod = "HySDispViewerJob.EventCallback";

            // イベント情報展開
            HySEventObject csHySEvent = (HySEventObject)csEvent;
            long lEventNo = csHySEvent.GetEventNo();
            HySID csID = csHySEvent.GetSuppID();
            HySObjectKind csSimKind = (HySObjectKind)csHySEvent.GetToSimKind();
            long lSubEventNo = csHySEvent.GetSubEventNo();

            // 該当イベント毎処理
            if (lEventNo == HySEventObject.CMND_DISP_SCREEN)
            {  // 画面表示指示の場合
                bRtn = CmndShowScreen(csEvent); // スクリーン生成と画面表示
            }
            else if (lEventNo == HySEventObject.CMND_DISPLY_COMMON)
            {  // 画面関連一般指示の場合
                if (lSubEventNo == HySDispDefine.NOTICE_ACTIVE_EVENT ||
                    lSubEventNo == HySDispDefine.NOTICE_NON_ACTIVE_EVENT ||
                    lSubEventNo == HySDispDefine.NOTICE_MODEL_CLOSE)
                {  // モデル管理画面からのアクティブ／非アクティブ報告の場合
                    ModelActiveStateNotification(csEvent);
                }
                else if (lSubEventNo == HySEventObject.NOTICE_STOP_SIMULATION ||
                         lSubEventNo == HySEventObject.NOTICE_COMPLETE_SIMULATION ||
                         lSubEventNo == HySEventObject.NOTICE_SIMULATION_ABEND)
                {  // シミュレーション計算終了報告の場合
                    ModelCalculationEndNotification(csEvent);
                }
                else
                {  // モデル管理画面からのアクティブ／非アクティブ報告以外の場合
                    csScreenKind = this.GetScreenKind(((HySEventObject)csEvent).GetData()); // 画面識別子取得
                    if (csScreenKind is HySObjectKind)
                    {  // 画面識別子が有効な場合
                        HySScreen csScreen = this.GetScreen(csSimKind, csID, csScreenKind); // スクリーンを取得
                        if (csScreen != null)
                        {  // スクリーン取得が正常の場合
                            if (lSubEventNo == HySDispDefine.NOTICE_FROM_DISPLAY)
                            {  // 指定フォーム表示の場合
                                if (m_csViewFormFactory != null)
                                { // 画面フォームファクトリが有効な場合
                                    HySObjectKind csFormKind = this.GetFormKind(((HySEventObject)csEvent).GetData()); // Form識別子取得
                                    if (csFormKind is HySObjectKind)
                                    {  // 画面フォームが有効な場合
                                        HySViewForm csViewFrom = m_csViewFormFactory.CreateViewForm(csSimKind, csID, csFormKind);
                                        if (csViewFrom != null)
                                        {  // 対応画面フォーム正常生成の場合
                                            ((HySScreen)csScreen).SetViewForm(csViewFrom); // フォーム設定
                                        }
                                    }
                                }
                            }
                            csScreen.EventCallback(csEvent); // イベントをそのまま渡す
                        }
                    }
                }
            }
            else if (lEventNo == HySEventObject.NOTICE_DISPLY_OK ||
                     lEventNo == HySEventObject.NOTICE_DISPLY_NG)
            {  // 画面関連一般報告(正常報告)／画面関連一般報告(異常報告)
                csScreenKind = GetScreenKind(((HySEventObject)csEvent).GetData());
                if (csScreenKind is HySObjectKind)
                {  // 画面識別子が有効な場合
                    HySScreen csScreen = this.GetScreen(csSimKind, csID, csScreenKind); // スクリーンを取得
                    if (csScreen != null)
                    {  // スクリーン取得が正常の場合
                        csScreen.EventCallback(csEvent); // イベントをそのまま渡す
                    }
                }
            }
            else if (lEventNo == HySEventObject.CMND_CLOSE_SCREEN)
            {  // 画面終了の場合
                csScreenKind = GetScreenKind(((HySEventObject)csEvent).GetData());
                if (csScreenKind is HySObjectKind)
                {  // 画面識別子が有効な場合
                    HySScreen csScreen = this.GetScreen(csSimKind, csID, csScreenKind); // スクリーンを取得
                    if (csScreen != null)
                    {  // スクリーン取得が正常の場合
                        this.DeleteScreen(csSimKind, csID, csScreenKind); // スクリーン削除
                        // スクリーン終了イベント通知
                        csHySEvent.SetEventNo(HySEventObject.NOTICE_DISPLY_OK);
                        csHySEvent.SetTo(HySSysEvent.OBJID_BUSIPROCEDURE);
                        m_csViewer.PutEvent(csHySEvent);
                    }
                }
            }
            else if (lEventNo == HySEventObject.NOTICE_DB_OK || lEventNo == HySEventObject.NOTICE_DB_NG)
            {   // DB関連一般報告(正常報告)／DB関連一般報告(異常報告)の場合

                // 各スクリーンへイベントを委譲
                foreach (HySScreen csScreen in m_csScreenTbl.Values)
                {
                    csScreen.EventCallback(csEvent);
                }
                bRtn = true;
            }
            else if (lEventNo == HySEventObject.CMND_GIS_COMMON)
            {  // GIS関連一般指示の場合
            }
            else if (lEventNo == HySEventObject.NOTICE_GIS_OK)
            {  // GIS関連一般報告(正常報告)の場合
            }
            else if (lEventNo == HySEventObject.NOTICE_GIS_NG)
            {  // GIS関連一般報告(異常報告)の場合
            }
            //else
            //{  // その他
            //    HySLog.LogOut(HySLog.ONLINE, csMethod, "処理対象外イベント: イベント番号: " + lEventNo.ToString());
            //}
            return bRtn;
        }

        /// <summary><para>method outline:</para>
        /// <para>全スクリーン初期配置表示</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> InitiateAllScreen( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void InitiateAllScreen()
        {
            HySString csPrcKey = null;
            HySScreen csScreen = null;
            // 全スクリーンの終了処理を実行
            foreach (string name in m_csScreenTbl.Keys)
            {  // ハッシュテーブル件数繰り返す
                csPrcKey = new HySString(name);
                csScreen = (HySScreen)m_csScreenTbl[csPrcKey.ToString()];
                if (csScreen != null)
                {  // スクリーンが有効の場合
                    csScreen.InitiateScreen();
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>全スクリーンにモデル管理アクティブ状態通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ModelActiveStateNotification(csEvent) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ModelActiveStateNotification(HySSysEvent csEvent)
        {
            HySString csPrcKey = null;
            HySScreen csScreen = null;
            // 全スクリーン分実行する
            foreach (string name in m_csScreenTbl.Keys)
            {  // ハッシュテーブル件数繰り返す
                csPrcKey = new HySString(name);
                csScreen = (HySScreen)m_csScreenTbl[csPrcKey.ToString()];
                if (csScreen != null)
                {  // スクリーンが有効の場合
                    csScreen.EventCallback(csEvent);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>全スクリーンに演算終了状態通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> ModelCalculationEndNotification(csEvent) </para>
        /// </example>
        /// <param name="csEvent">送られたイベント</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void ModelCalculationEndNotification(HySSysEvent csEvent)
        {
            HySString csPrcKey = null;
            HySScreen csScreen = null;
            // 全スクリーン分実行する
            foreach (string name in m_csScreenTbl.Keys)
            {  // ハッシュテーブル件数繰り返す
                csPrcKey = new HySString(name);
                csScreen = (HySScreen)m_csScreenTbl[csPrcKey.ToString()];
                if (csScreen != null)
                {  // スクリーンが有効の場合
                    csScreen.EventCallback(csEvent);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>画面識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csScreenKind = GetScreenKind(csScreenGrKind) </para>
        /// </example>
        /// <param name="csScreenGrKind">画面識別子グループ情報</param>
        /// <returns>HySObjectKind 画面識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private HySObjectKind GetScreenKind(HySDataRoot csScreenGrKind)
        {
            HySObjectKind csKind = null;

            if (csScreenGrKind is HySDataLinkedList)
            {  // 画面グループ指定の場合
                HySDataLinkedList csArrayList = csScreenGrKind as HySDataLinkedList;
                if (csArrayList.GetCount() >= 1)
                {  // 画面識別のみの場合
                    csKind = csArrayList.GetData(0) as HySObjectKind;
                }
            }
            else
            {  // 画面識別の場合
                csKind = csScreenGrKind as HySObjectKind;
            }
            return csKind;
        }
        /// <summary><para>method outline:</para>
        /// <para>Ｆｏｒｍ識別子取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySObjectKind csFormKind = GetFormKind(csScreenGrKind) </para>
        /// </example>
        /// <param name="csScreenGrKind">画面識別子グループ情報</param>
        /// <returns>HySObjectKind Form識別子</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private HySObjectKind GetFormKind(HySDataRoot csScreenGrKind)
        {
            HySObjectKind csKind = null;

            if (csScreenGrKind is HySDataLinkedList)
            {  // 画面グループ指定の場合
                HySDataLinkedList csArrayList = csScreenGrKind as HySDataLinkedList;
                if (csArrayList.GetCount() == 1)
                {  // 画面識別のみの場合
                    csKind = csArrayList.GetData(0) as HySObjectKind;
                }
                else if (csArrayList.GetCount() == 2)
                {  // 画面識別のみの場合
                    csKind = csArrayList.GetData(1) as HySObjectKind;
                }
            }
            else
            {  // 画面識別の場合
                csKind = csScreenGrKind as HySObjectKind;
            }
            return csKind;
        }

        /// <summary><para>method outline:</para>
        /// <para>Ctrl+Zキーイベント通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeUndoEvent( ) </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeUndoEvent(long lCallBackNo)
        {
            HySScreen csScreeen = null;
            csScreeen = this.GetFocusOnScreen();
            if (csScreeen != null)
            {
                // フォーカスオンのスクリーンがある場合のみ、通知する
                csScreeen.NoticeUndoEvent();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>Ctrl+Yキーイベント通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeRedoEvent( ) </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeRedoEvent(long lCallBackNo)
        {
            HySScreen csScreeen = null;
            csScreeen = this.GetFocusOnScreen();
            if (csScreeen != null)
            {
                // フォーカスオンのスクリーンがある場合のみ、通知する
                csScreeen.NoticeRedoEvent();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>Ctrl+Xキーイベント通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeCutEvent( ) </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeCutEvent(long lCallBackNo)
        {
            HySScreen csScreeen = null;
            csScreeen = this.GetFocusOnScreen();
            if (csScreeen != null)
            {
                // フォーカスオンのスクリーンがある場合のみ、通知する
                csScreeen.NoticeCutEvent();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>Ctrl+Cキーイベント通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeCopyEvent( ) </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeCopyEvent(long lCallBackNo)
        {
            HySScreen csScreeen = null;
            csScreeen = this.GetFocusOnScreen();
            if (csScreeen != null)
            {
                // フォーカスオンのスクリーンがある場合のみ、通知する
                csScreeen.NoticeCopyEvent();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>Ctrl+Vキーイベント通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticePasteEvent( ) </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticePasteEvent(long lCallBackNo)
        {
            HySScreen csScreeen = null;
            csScreeen = this.GetFocusOnScreen();
            if (csScreeen != null)
            {
                // フォーカスオンのスクリーンがある場合のみ、通知する
                csScreeen.NoticePasteEvent();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>Delキーイベント通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeDeleteEvent( ) </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeDeleteEvent(long lCallBackNo)
        {
            HySScreen csScreeen = null;
            csScreeen = this.GetFocusOnScreen();
            if (csScreeen != null)
            {
                // フォーカスオンのスクリーンがある場合のみ、通知する
                csScreeen.NoticeDeleteEvent();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>Ctrl+Aキーイベント通知</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> NoticeAllSelectEvent( ) </para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public virtual void NoticeAllSelectEvent(long lCallBackNo)
        {
            HySScreen csScreeen = null;
            csScreeen = this.GetFocusOnScreen();
            if (csScreeen != null)
            {
                // フォーカスオンのスクリーンがある場合のみ、通知する
                csScreeen.NoticeAllSelectEvent();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>フォーカスオンスクリーン取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySScreen csScreen = GetFocusOnScreen( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>アクティブスクリーン</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual public HySScreen GetFocusOnScreen()
        {
            HySString csPrcKey = null;
            HySScreen csScreen = null;
            HySScreen csScreenResult = null;
            // 全スクリーン分実行する
            foreach (string name in m_csScreenTbl.Keys)
            {  // ハッシュテーブル件数繰り返す
                csPrcKey = new HySString(name);
                csScreen = (HySScreen)m_csScreenTbl[csPrcKey.ToString()];
                if (csScreen != null)
                {  // スクリーンが有効の場合
                    if (((HySScreen)csScreen).IsFocusOn() == true)
                    {
                        csScreenResult = csScreen;
                        break;
                    }
                }
            }
            return csScreenResult;
        }

        /// <summary><para>method outline:</para>
        /// <para>GIS表示制御用サブジョブ取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> HySGISCtlViewerSubJob csJob = GetGISCtlJob( ) </para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>HySGISCtlViewerSubJob GIS表示制御用サブジョブ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        virtual internal HySGISCtlViewerSubJob GetGISCtlJob()
        { return m_csGISCtlSubJob; }
    }
}
