using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace CommonMP.HYMCO.OptionImpl.McGraphViewForDevelop
{
    /// <summary><para>class outline:</para>
    /// <para>グラフ表示モジュール　凡例ユーザーコントロール</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2010/06/04][新規作成]</para>
    /// </remarks>
    public partial class LegendList : UserControl
    {
        /// <summary> ライン描画 </summary>
        private const string DRAW_LINE = "L";
        /// <summary> ポイント描画 </summary>
        private const string DRAW_POINT = "P";


        /// <summary><para>event outline:</para>
        /// <para>凡例の色・線種変更時のイベント </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>legendList1.LegendStyleChanged += new EventHandler(legendList1_LegendStyleChanged);</para>
        /// <para>private void legendList1_LegendStyleChanged(object sender, EventArgs e){ }</para>
        /// </example>
        public event System.EventHandler LegendStyleChanged;


        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>LegendList legendList1 = new LegendList();</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public LegendList()
        {
            InitializeComponent();

            //一覧の初期設定
            InitList();

            //線種リスト設定
            cboLineStyle.Items.Add("Solid");
            cboLineStyle.Items.Add("Dot");
            cboLineStyle.Items.Add("Dash");
            cboLineStyle.Items.Add("DashDot");
            cboLineStyle.SelectedIndex = 0;

            //太さリスト設定
            cboLineSize.Items.Add(1.5F);
            cboLineSize.Items.Add(2.0F);
            cboLineSize.Items.Add(3.0F);
            cboLineSize.Items.Add(4.0F);
            cboLineSize.SelectedIndex = 0;

            //設定フォームを表示/非表示
            pnlSetting.Visible = false;
        }

        /// <summary><para>method outline:</para>
        /// <para>一覧の初期設定</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>InitList();</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void InitList()
        {
            //----------------------------------------
            // 一覧の設定
            //----------------------------------------
            dgvList.RowHeadersVisible = false;
            dgvList.ColumnHeadersVisible = false;
            dgvList.AllowUserToResizeColumns = false;       //ﾕｰｻﾞｰが列幅変更できないにｾｯﾄ
            dgvList.AllowUserToResizeRows = false;          //ﾕｰｻﾞｰが行高変更できないにｾｯﾄ
            dgvList.ColumnHeadersDefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleCenter;
            dgvList.ColumnHeadersHeightSizeMode = DataGridViewColumnHeadersHeightSizeMode.DisableResizing;
            dgvList.ReadOnly = false;

            DataGridViewImageColumn colI = new DataGridViewImageColumn();
            colI.Name = "LegendImage";
            colI.ReadOnly = true;
            colI.Width = 40;
            dgvList.Columns.Add(colI);

            DataGridViewTextBoxColumn colG = new DataGridViewTextBoxColumn();
            colG.Name = "Label";
            colG.HeaderText = Properties.HysAddinHydroHyetoGraphResources.LABEL;
            colG.Width = 60;
            colG.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleLeft;
            dgvList.Columns.Add(colG);

            DataGridViewTextBoxColumn colF = new DataGridViewTextBoxColumn();
            colF.Name = "FileName";
            colF.HeaderText = Properties.HysAddinHydroHyetoGraphResources.FILE_NAME;
            colF.Width = 80;
            colF.ReadOnly = true;
            colF.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleLeft;
            dgvList.Columns.Add(colF);

            DataGridViewButtonColumn colC = new DataGridViewButtonColumn();
            colC.Name = "LegendSet";
            colC.HeaderText = Properties.HysAddinHydroHyetoGraphResources.SETTING;
            colC.Text = Properties.HysAddinHydroHyetoGraphResources.MSG_INFO_CHANGE_COLOR_AND_LINETYPE; //色と線種を変更します。
            colC.Width = 20;
            colC.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
            colC.UseColumnTextForButtonValue = true;
            dgvList.Columns.Add(colC);

            //データ格納用非表示列（分類, 色,   線種,   太さ,       大きさ,   ライン・ポイント描画, ID）
            string[] sNames = { "Group", "Color", "Line", "LineSize", "MarkSize", "DrawLineMark", "ID"};
            for (int i = 0; i < sNames.Length; i++)
            {
                DataGridViewTextBoxColumn colData1 = new DataGridViewTextBoxColumn();
                colData1.Name = sNames[i];
                colData1.HeaderText = sNames[i];
                colData1.Visible = false;
                dgvList.Columns.Add(colData1);
            }
            
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧クリック時処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> dgvList_CellClick( sender, e )</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">Event</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void dgvList_CellClick(object sender, DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex < 0 || e.RowIndex < 0)
                return; //ヘッダーは処理なし

            DataGridView dgv = (DataGridView)sender;
            if (dgv.Columns[e.ColumnIndex].Name == "LegendSet")
            {   //設定ボタンクリック時

                //選択行を格納
                pnlSetting.Tag = e.RowIndex;
                //設定フォームを表示
                SwitchSettingForm(0);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>キャンセルボタンクリック時処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> btnCancel_Click( sender, e )</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">Event</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void btnCancel_Click(object sender, EventArgs e)
        {
            //設定フォームを非表示
            SwitchSettingForm(-1);
        }

        /// <summary><para>method outline:</para>
        /// <para>OKボタンクリック時処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> btnOK_Click( sender, e )</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">Event</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void btnOK_Click(object sender, EventArgs e)
        {
            //設定フォームを非表示
            if (!SwitchSettingForm(1))
                return;

            //設定変更したイベントをあげる
            this.LegendStyleChanged(this, new EventArgs());
        }

        /// <summary><para>method outline:</para>
        /// <para>色選択処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> btnColor_Click( sender, e )</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">Event</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void btnColor_Click(object sender, EventArgs e)
        {
            ColorDialog cd = new ColorDialog();
            if (cd.ShowDialog() == DialogResult.OK)
            {   //選択された場合

                //選択された色を表示
                lblColor.BackColor = cd.Color;

                //線種 再描画
                cboLineStyle.Refresh();
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>線種コンボボックス描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> cboLineStyle_DrawItem( sender, e )</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">Event</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cboLineStyle_DrawItem(object sender, DrawItemEventArgs e)
        {
            //コンボボックス
            ComboBox cbo = (ComboBox)sender;

            int idx = e.Index;
            if (idx >= 0)
            {
                Graphics g = e.Graphics;

                //背景を白色にする
                g.FillRectangle(Brushes.White, e.Bounds.X, e.Bounds.Y, e.Bounds.Size.Width, cbo.ItemHeight);

                //項目
                string lineStyle = cbo.Items[idx].ToString();

                using (SolidBrush brs = new SolidBrush(lblColor.BackColor))
                {
                    //線種を描画
                    using (Pen pen1 = new Pen(brs, 4))
                    {
                        //線種指定
                        pen1.DashStyle = GetDashStyle(lineStyle);

                        g.DrawLine(pen1, e.Bounds.X + 3, e.Bounds.Y + 6, e.Bounds.X + cbo.Width - 8, e.Bounds.Y + 6);
                    }
                }

            }

            //フォーカスを示す四角形を描画
            e.DrawFocusRectangle();
        }

        /// <summary><para>method outline:</para>
        /// <para>太さコンボボックス描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> cboLineSize_DrawItem( sender, e )</para>
        /// </example>
        /// <param name="sender">object</param>
        /// <param name="e">Event</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cboLineSize_DrawItem(object sender, DrawItemEventArgs e)
        {
            //コンボボックス
            ComboBox cbo = (ComboBox)sender;

            int idx = e.Index;
            if (idx >= 0)
            {
                Graphics g = e.Graphics;

                //背景を白色にする
                g.FillRectangle(Brushes.White, e.Bounds.X, e.Bounds.Y, e.Bounds.Size.Width, cbo.ItemHeight);

                //項目
                float lineSize = (float)cbo.Items[idx];

                using (SolidBrush brs = new SolidBrush(Color.Black))
                {
                    //太さを描画
                    using (Pen pen1 = new Pen(brs, lineSize))
                    {
                        g.DrawLine(pen1, e.Bounds.X + 3, e.Bounds.Y + 6, e.Bounds.X + cbo.Width - 8, e.Bounds.Y + 6);
                    }
                }

            }

            //フォーカスを示す四角形を描画
            e.DrawFocusRectangle();
        }

        /// <summary><para>method outline:</para>
        /// <para> 設定フォームの表示/非表示 </para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SwitchSettingForm( iMode )</para>
        /// </example>
        /// <param name="iMode">0=表示/1=反映して閉じる/-1=ｷｬﾝｾﾙして閉じる</param>
        /// <returns>true/false</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool SwitchSettingForm(int iMode)
        {
            int iRow = (int)pnlSetting.Tag;
            string sLineStyle = "Solid";
            float fLineSize = 1.5F;
            float fMarkSize = 4.0F;
            string sLineMark = "";

            if (iMode == 0)
            {   //一覧の値を設定フォームに反映
                //色
                lblColor.BackColor = (Color)dgvList["Color", iRow].Value;
                
                if (dgvList["Group", iRow].Value.ToString() == Properties.HysAddinHydroHyetoGraphResources.PRECIPITATION)
                {   //雨量の場合、線種は選択不可
                    grpLine.Enabled = false;
                }
                else
                {
                    grpLine.Enabled = true;

                    //一覧から現在の設定値取得
                    sLineStyle = dgvList["Line", iRow].Value.ToString();
                    fLineSize = (float)dgvList["LineSize", iRow].Value;
                    fMarkSize = (float)dgvList["MarkSize", iRow].Value;
                    sLineMark = dgvList["DrawLineMark", iRow].Value.ToString();
                }

                //線種
                cboLineStyle.SelectedItem = sLineStyle;
                //太さ
                cboLineSize.SelectedItem = fLineSize;
                //大きさ
                numMarkSize.Value = (decimal)fMarkSize;
                //ライン・ポイント描画
                chkLine.Checked = sLineMark.Contains(DRAW_LINE);
                chkMarker.Checked = sLineMark.Contains(DRAW_POINT);

            }
            else if (iMode == 1)
            {   //設定フォームの値を一覧に反映
                if (grpLine.Enabled)
                {
                    sLineStyle = cboLineStyle.SelectedItem.ToString();
                    fLineSize = (float)cboLineSize.SelectedItem;
                    fMarkSize = (float)numMarkSize.Value;
                    if (chkLine.Checked) sLineMark += DRAW_LINE;
                    if (chkMarker.Checked) sLineMark += DRAW_POINT;
                    if (sLineMark.Length == 0)
                    {
                        //ラインかポイントのどちらかを選択して下さい。
                        MessageBox.Show(Properties.HysAddinHydroHyetoGraphResources.MSG_ERROR_CHOOSE_LINE_OR_POINT,
                            Properties.HysAddinHydroHyetoGraphResources.CAPTION_INPUT_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                        return false;
                    }
                }
                else
                {   //雨量の場合、凡例は棒(ライン)
                    sLineMark = DRAW_LINE;
                }

                //凡例画像表示
                SetLegendImage(iRow, lblColor.BackColor, sLineStyle, fLineSize, fMarkSize, sLineMark);
            }

            //設定フォームを表示/非表示
            pnlSetting.Visible = (iMode == 0 ? true : false);
            //一覧を操作可/不可
            dgvList.Enabled = (!pnlSetting.Visible);

            return true;
        }

        
        /// <summary><para>method outline:</para>
        /// <para>画像作成</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateImage( bmp, lineColor, lineStyle, iStartX, drawLine )</para>
        /// </example>
        /// <param name="bmp">Bitmap</param>
        /// <param name="lineColor">色</param>
        /// <param name="lineStyle">線種</param>
        /// <param name="iStartX">描画開始X</param>
        /// <param name="drawLine">true=ラインを描画/false=マークを描画</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void CreateImage(Bitmap bmp, Color lineColor, string lineStyle, int iStartX, bool drawLine)
        {
            Graphics g = null;
            try
            {
                //Graphicsオブジェクトを取得
                g = Graphics.FromImage(bmp);
                //背景を白色にする
                g.FillRectangle(Brushes.White, 0, 0, bmp.Width, bmp.Height);

                float penSize = (drawLine ? 4 : 10);
                float y = (bmp.Height - penSize) / 2 + 1;

                //描画
                using (SolidBrush brs = new SolidBrush(lineColor))
                {
                    if (drawLine)
                    {
                        using (Pen pen1 = new Pen(brs, penSize))
                        {
                            //線種指定
                            pen1.DashStyle = GetDashStyle(lineStyle);

                            g.DrawLine(pen1, iStartX, y, bmp.Width - iStartX, y);
                        }
                    }
                    else
                    {   //マーク描画
                        float x = (bmp.Width / 2) - penSize / 2;
                        g.FillEllipse(brs, x, y, penSize, penSize);
                    }
                }

            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine(ex.Message);
            }
            finally
            {
                if (g != null)
                {
                    g.Dispose();
                    g = null;
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧の凡例画像を設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> SetLegendImage( iRow, lineColor, lineStyle, lineSize, markSize, lineMark )</para>
        /// </example>
        /// <param name="iRow">行インデックス</param>
        /// <param name="lineColor">色</param>
        /// <param name="lineStyle">線種</param>
        /// <param name="lineSize">太さ</param>
        /// <param name="markSize">大きさ</param>
        /// <param name="lineMark">ライン・ポイント描画</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void SetLegendImage(int iRow, Color lineColor, string lineStyle, float lineSize, float markSize, string lineMark)
        {
            //イメージを変更する
            Bitmap bmp = null;
            if (dgvList["LegendImage", iRow].Value != null)
                bmp = (Bitmap)dgvList["LegendImage", iRow].Value;
            else
                bmp = new Bitmap(dgvList.Columns["LegendImage"].Width - 2, 18);

            //画像作成
            CreateImage(bmp, lineColor, lineStyle, 3, lineMark.Contains(DRAW_LINE));

            //凡例画像を表示
            dgvList["LegendImage", iRow].Value = bmp;
            dgvList["Color", iRow].Value = lineColor;
            dgvList["Line", iRow].Value = lineStyle;
            dgvList["LineSize", iRow].Value = lineSize;
            dgvList["MarkSize", iRow].Value = markSize;
            dgvList["DrawLineMark", iRow].Value = lineMark;
        }

        /// <summary><para>method outline:</para>
        /// <para>デフォルト色取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> GetDefaultColor( sGroup )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <returns>色</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private Color GetDefaultColor(string sGroup)
        {
            if (sGroup == Properties.HysAddinHydroHyetoGraphResources.PRECIPITATION)     //雨量
                return Color.DeepSkyBlue;
            else if (sGroup == Properties.HysAddinHydroHyetoGraphResources.WATER_LEVEL)  //水位
                return Color.Indigo;
            else if (sGroup == Properties.HysAddinHydroHyetoGraphResources.DISCHARGE)    //流量
                return Color.DarkGreen;

            return Color.White;
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧にセットするIDを作成（このID順にソートされる）</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> CreateID( sGroup, sFileName )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <param name="sFileName">ファイル名</param>
        /// <returns>ID[分類ID_ファイル名]</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private string CreateID(string sGroup, string sFileName)
        {
            string ID = null;
            if (sGroup == Properties.HysAddinHydroHyetoGraphResources.PRECIPITATION)     //雨量
                ID = "0_" + sFileName;
            else if (sGroup == Properties.HysAddinHydroHyetoGraphResources.WATER_LEVEL)  //水位
                ID = "1_" + sFileName;
            else if (sGroup == Properties.HysAddinHydroHyetoGraphResources.DISCHARGE)    //流量
                ID = "2_" + sFileName;
            else
                ID = "9_" + sFileName;
            return ID;
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧から一致する行インデックスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> int iRow = SearchRow( sGroup, sFileName )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <param name="sFileName">ファイル名</param>
        /// <returns>行インデックス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private int SearchRow(string sGroup, string sFileName)
        {
            string ID = CreateID(sGroup, sFileName);
            for (int i = 0; i < dgvList.RowCount; i++)
            {
                if (dgvList["ID", i].Value.ToString() == ID)
                {   //見つかった
                    return i;
                }
            }

            //見つからない
            return -1;
        }
        

        //============================================================
        // publicメソッド
        //============================================================

        /// <summary><para>method outline:</para>
        /// <para>線種文字列からDashStyleを取得</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DashStyle style = GetDashStyle( lineStyle )</para>
        /// </example>
        /// <param name="lineStyle">線種</param>
        /// <returns>DashStyle</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public System.Drawing.Drawing2D.DashStyle GetDashStyle(string lineStyle)
        {
            System.Drawing.Drawing2D.DashStyle dStyle = System.Drawing.Drawing2D.DashStyle.Solid;
            if (lineStyle == "Solid")
                dStyle = System.Drawing.Drawing2D.DashStyle.Solid;      //実線
            else if (lineStyle == "Dot")
                dStyle = System.Drawing.Drawing2D.DashStyle.Dot;        //点線
            else if (lineStyle == "Dash")
                dStyle = System.Drawing.Drawing2D.DashStyle.Dash;       //Dash
            else if (lineStyle == "DashDot")
                dStyle = System.Drawing.Drawing2D.DashStyle.DashDot;    //DashDot

            return dStyle;
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧から分類に一致する行を削除する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DelGroupList( sGroup )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DelGroupList(string sGroup)
        {
            //分類ID取得
            string ID_Head = CreateID(sGroup, "");
            for (int i = dgvList.RowCount - 1; i >= 0; i--) //削除するので後ろからループ
            {
                if (dgvList["ID", i].Value.ToString().StartsWith(ID_Head))
                {   //分類が一致する場合、削除
                    dgvList.Rows.RemoveAt(i);
                }
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧から削除する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DelList( sGroup, sFileName )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <param name="sFileName">ファイル名</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DelList(string sGroup, string sFileName)
        {
            //行検索
            int iRow = SearchRow(sGroup, sFileName);
            if (iRow >= 0)
            {   //行削除
                dgvList.Rows.RemoveAt(iRow);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧に追加する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> AddList( sGroup, sFileName )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <param name="sFileName">ファイル名</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void AddList(string sGroup, string sFileName)
        {
            //行検索
            int iRow = SearchRow(sGroup, sFileName);
            if (iRow < 0)
            {   //一致行がない場合、追加
                dgvList.Rows.Add();
                iRow = dgvList.RowCount - 1;

                dgvList["ID", iRow].Value = CreateID(sGroup, sFileName);
                dgvList["Group", iRow].Value = sGroup;
                dgvList["Label", iRow].Value = sGroup;
                dgvList["FileName", iRow].Value = sFileName;

                //凡例画像表示
                SetLegendImage(iRow, GetDefaultColor(sGroup), "Solid", 1.5F, 4.0F, DRAW_LINE + DRAW_POINT);
                

                //並び替え
                dgvList.Sort(dgvList.Columns["ID"], ListSortDirection.Ascending);
            }
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例一覧から一致する色と線種を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Color cl = GetLegend( sGroup, sFileName, out sLineType, out fLineSize, out fMarkSize, out bDrawLine, out bDrawPoint )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <param name="sFileName">ファイル名</param>
        /// <param name="sLineType">[out]線種</param>
        /// <param name="fLineSize">[out]太さ</param>
        /// <param name="fMarkSize">[out]大きさ</param>
        /// <param name="bDrawLine"> [out]ライン描画</param>
        /// <param name="bDrawPoint">[out]ポイント描画</param>
        /// <returns>色</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public Color GetLegend(string sGroup, string sFileName, 
                    out string sLineType, out float fLineSize, out float fMarkSize, out bool bDrawLine, out bool bDrawPoint)
        {
            Color lineColor = GetDefaultColor(sGroup);
            sLineType = "Solid";
            fLineSize = 1.5F;
            fMarkSize = 4.0F;
            bDrawLine = false;
            bDrawPoint = false;

            //行検索
            int iRow = SearchRow(sGroup, sFileName);
            if (iRow >= 0)
            {
                lineColor = (Color)dgvList["Color", iRow].Value;
                sLineType = dgvList["Line", iRow].Value.ToString();
                fLineSize = (float)dgvList["LineSize", iRow].Value;
                fMarkSize = (float)dgvList["MarkSize", iRow].Value;
                string sLineMark = dgvList["DrawLineMark", iRow].Value.ToString();
                bDrawLine = sLineMark.Contains(DRAW_LINE);
                bDrawPoint = sLineMark.Contains(DRAW_POINT);
            }

            return lineColor;
        }
        /// <summary><para>method outline:</para>
        /// <para>凡例一覧から一致する色を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> Color cl = GetLegend( sGroup, sFileName )</para>
        /// </example>
        /// <param name="sGroup">分類（雨量/水位/流量）</param>
        /// <param name="sFileName">ファイル名</param>
        /// <returns>色</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public Color GetLegend(string sGroup, string sFileName)
        {
            string sLineType;
            float fLineSize;
            float fMarkSize;
            bool bDrawLine;
            bool bDrawPoint;
            return GetLegend(sGroup, sFileName, out sLineType, out fLineSize, out fMarkSize, out bDrawLine, out bDrawPoint);
        }

        /// <summary><para>method outline:</para>
        /// <para>凡例描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para> DrawLegend( ref bmp, bDrawFileName )</para>
        /// </example>
        /// <param name="bmp">Bitmap</param>
        /// <param name="bDrawFileName">true=ファイル名も付ける</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawLegend(ref Bitmap bmp, bool bDrawFileName)
        {
            Graphics g = null;
            try
            {
                int iLegendWidth = 140;
                int iLegendHeight = 10 + (18 * dgvList.RowCount);
                if (bDrawFileName)
                {
                    iLegendWidth += 130;
                }

                bmp = new Bitmap(iLegendWidth + 10, iLegendHeight + 10);

                //Graphicsオブジェクトを取得
                g = Graphics.FromImage(bmp);
                //背景を白色にする
                g.FillRectangle(Brushes.White, 0, 0, bmp.Width, bmp.Height);

                int x = 2;
                int y = 8;

                //描画
                for (int iRow = 0; iRow < dgvList.RowCount; iRow++)
                {
                    //一覧から現在の設定値取得
                    string sLabel = dgvList["Label", iRow].Value.ToString();
                    string sFileName = dgvList["FileName", iRow].Value.ToString();
                    Color lineColor = (Color)dgvList["Color", iRow].Value;
                    string sLineStyle = dgvList["Line", iRow].Value.ToString();
                    float fLineSize = (float)dgvList["LineSize", iRow].Value;
                    float fMarkSize = (float)dgvList["MarkSize", iRow].Value;
                    string sLineMark = dgvList["DrawLineMark", iRow].Value.ToString();
                    bool drawLine = sLineMark.Contains(DRAW_LINE);

                    int penSize = (drawLine ? 8 : 10);

                    if (bDrawFileName)
                    {   //ファイル名も付ける場合
                        sLabel += "(" + sFileName + ")";
                    }

                    using (SolidBrush brs = new SolidBrush(lineColor))
                    {
                        if (drawLine)
                        {
                            using (Pen pen1 = new Pen(brs, penSize))
                            {
                                //線種指定
                                pen1.DashStyle = GetDashStyle(sLineStyle);

                                g.DrawLine(pen1, x + 5, y + 8, x + 25, y + 8);
                            }
                        }
                        else
                        {   //マーク描画
                            g.FillEllipse(brs, x + 15, y + 8, penSize, penSize);
                        }
                    }

                    //ラベル
                    using (SolidBrush bs = new SolidBrush(Color.Black))
                    {
                        using (Font ft = new Font(Properties.HysAddinHydroHyetoGraphResources.DEFAULT_FONT_NAME,
                            float.Parse(Properties.HysAddinHydroHyetoGraphResources.DEFAULT_FONT_SIZE)))
                        {
                            RectangleF rect = new RectangleF(x + 30, y, iLegendWidth - 30, 16);
                            StringFormat sformat = new StringFormat();
                            sformat.Alignment = StringAlignment.Near;
                            sformat.LineAlignment = StringAlignment.Near;

                            g.DrawString(sLabel, ft, bs, rect, sformat);
                        }
                    }

                    y += 18;
                }


                //外枠
                using (Pen pen1 = new Pen(Color.Black, 1))
                {   //外枠
                    g.DrawRectangle(pen1, x, 2, iLegendWidth, iLegendHeight);
                }

            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine(ex.Message);
            }
            finally
            {
                if (g != null)
                {
                    g.Dispose();
                    g = null;
                }
            }
        }


    }
}
