﻿//using System;
//using System.Collections.Generic;
//using System.Text;
using System;
using System.Data;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;

namespace CommonMP.HYMCO.OptionImpl.McGraphViewForDevelop
{
    #region enumの定義
    /**********************************************************************
		 * enumの定義
	**********************************************************************/
    /// <summary>出力グラフィックの種類です。</summary>
    /// <remarks></remarks>
    public enum enumGraphKind
    {
        /// <summary>日表</summary>
        IsTime = 1,
        /// <summary>月表</summary>
        IsDay = 2
    };
    /// <summary>出力グラフィックのX軸目盛の種類です。</summary>
    /// <remarks></remarks>
    public enum enumXKind
    {
        /// <summary>数値</summary>
        IsNumeric = 0,
        /// <summary>日表もしくは月表</summary>
        IsTimeOrDay = 1
    };
    /// <summary>出力グラフィックの凡例表示の有無です。</summary>
    /// <remarks></remarks>
    public enum enumHasLegend
    {
        /// <summary>表示</summary>
        HasTrue = 1,
        /// <summary>非表示</summary>
        HasFalse = 0
    };
    /// <summary>出力グラフィックのプロットの種類です。</summary>
    /// <remarks></remarks>
    public enum enumPlotKind
    {
        /// <summary>なし</summary>
        Nothing = 0,
        /// <summary>折れ線グラフ</summary>
        Line = 1,
        /// <summary>棒グラフ</summary>
        Bar = 2,
        /// <summary>散布図</summary>
        Scatter = 3
    };
    /// <summary>出力グラフィックのプロットの型です。</summary>
    /// <remarks></remarks>
    public enum enumPlotForm
    {
        /// <summary>なし</summary>
        Nothing = 0,
        /// <summary>塗り潰しなしの円</summary>
        Circle = 1,
        /// <summary>塗り潰しありの円</summary>
        FillCircle = 2,
        /// <summary>塗り潰しなしの四角形</summary>
        Rectangle = 3,
        /// <summary>塗り潰しありの四角形</summary>
        FillRectangle = 4,
        /// <summary>塗り潰しなしの三角形</summary>
        Triangle = 5,
        /// <summary>塗り潰しありの三角形</summary>
        FillTriangle = 6,
        /// <summary>塗り潰しなしの菱形</summary>
        Rhombus = 7,
        /// <summary>塗り潰しありの菱形</summary>
        FillRhombus = 8
    };
    /// <summary>グラフィックの基本色です。</summary>
    /// <remarks>プロットの種類・型を指定してプロットする際にランダムな色を取得する場合に使用します。</remarks>
    public enum enumBaseColor
    {
        /// <summary>赤</summary>
        red = 1,
        /// <summary>緑</summary>
        green = 2,
        /// <summary>青</summary>
        blue = 3
    };
    #endregion

    /// <summary><para>class outline:</para>
    /// <para>グラフィック作成用の共通クラスです。</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[CommonMP][ver 1.0.0][2010/02/04][新規作成]</para>
    /// </remarks>
    /// <remarks>
    /// WindowsアプリケーションもしくはWebアプリケーション上のグラフィック作成を支援します。
    /// </remarks>
    /// <example>
    /// [C#]
    /// <code>
    ///      clsGraph a_grp = new clsGraph();
    ///      a_grp.ImageWidth = 880;	// 出力グラフ画像の横サイズ（pixel）
    ///      a_grp.ImageHeight = 400;	// 出力グラフ画像の縦サイズ（pixel）
    ///      a_grp.GraphKind = obcCommon_lib.enumGraphKind.IsDay;	// グラフ種別（1：日表、2：月表）
    ///      a_grp.OutImgaeFileName =
    ///          m_clsCommon.GetTempPathFromINI() + m_clsCommon.TEMP_GRAPH_OUT + DateTime.Now.ToString("yyyyMMddhhmmss") + ".png";	// 出力画像ファイル名（PNGのみサポート）
    ///      a_grp.InvalidValue = -1F;	// 欠測値
    ///      a_grp.GraphMaxY = a_AccessSumMax;	// グラフY軸の最大値
    ///      a_grp.GraphMinY = 0.0F;	// グラフY軸の最小値
    ///      a_grp.GraphSepY = 5.0F;	// グラフY軸のセパレート値
    ///      a_grp.GraphMaxX = a_pfMax;		// グラフX軸の最大値
    ///      a_grp.GraphMinX = 1F;		// グラフX軸の最小値
    ///      a_grp.GraphSepX = 1F;
    ///      a_grp.GraphXKind = obcCommon_lib.enumXKind.IsTimeOrDay;	// X軸の種別（0：数値、1：時刻or日付）
    ///      a_grp.GraphHasLegend = obcCommon_lib.enumHasLegend.HasTrue;	// 凡例の表示・非表示
    ///      a_grp.GraphYReverse = false;	// Y軸の反転
    ///      a_grp.GraphHasMeasureX = false;   // X軸の目盛表示
    ///      a_grp.GraphMeasureXFormat = "0"; // X軸の目盛書式
    ///      a_grp.GraphMeasureXKind = false; // X軸の目盛線
    ///      a_grp.GraphHasMeasureY = true;   // Y軸の目盛表示
    ///      a_grp.GraphMeasureYFormat = "0";    // a_sFormat; // Y軸の目盛書式
    ///      a_grp.GraphMeasureYKind = true; // Y軸の目盛線
    ///
    ///      a_grp.InitGraph(a_userInfo);	// グラフの初期化
    ///
    ///      a_grp.SetGraphItems(m_analyzedItems);	// X,Y値のプロット
    ///      a_grp.MakeGraph();	// グラフの作成
    ///
    ///      this.pictureBox_1_2.Image = System.Drawing.Image.FromFile(a_grp.OutImgaeFileName);
    /// </code>
    /// </example>
    public class McGraphViewImageLib
    {
        #region 定数の定義
        /**********************************************************************
		 * 定数の定義
		**********************************************************************/
        /// <summary> カラーリスト </summary>
        private int[] m_safeColor = {
            0x000000,0x333333,0x666666,
            0x999999,0xcccccc,0xffffff,

        };
        /// <summary>上部余白</summary>
        public float m_sep_top = 40;
        /// <summary>左部余白</summary>
        public float m_sep_left = 70;
        /// <summary>下部余白</summary>
        public float m_sep_bottom = 50;
        /// <summary>右部余白</summary>
        public float m_sep_right = 40;
		//protected Font m_font1 = new Font("Tahoma",10);
        /// <summary>出力グラフで扱うフォント（"ＭＳ ゴシック" 9pt）</summary>
        public Font m_font1 = new Font(Properties.HysAddinHydroHyetoGraphResources.DEFAULT_FONT_NAME,
            float.Parse(Properties.HysAddinHydroHyetoGraphResources.DEFAULT_FONT_SIZE));
        /// <summary>X軸の文字のセパレータ値</summary>
        public float m_sep_string_x = 4;
        /// <summary>Y軸の文字のセパレータ値</summary>
        public float m_sep_string_y = 4;
        /// <summary>グラフ枠の色（Black 0.5）</summary>
        public Pen m_pen1 = new Pen(Color.Black, 0.5f);
        /// <summary>グラフ内におけるXY軸の目盛線の色（Black 0.5）</summary>
        public Pen m_pen2 = new Pen(Color.LightGray, 0.5f);
        /// <summary>凡例のサイズ</summary>
        public float m_size_Legend = 8;
        #endregion

        #region 変数の定義
        /**********************************************************************
		 * 変数の定義
		**********************************************************************/
        /// <summary> ビットマップオブジェクト </summary>
		public Bitmap m_bm = null;
        /// <summary> Graphicsオブジェクト </summary>
        public Graphics m_g = null;
        ///// <summary> 初期化状態 </summary>
        //private bool m_InitOK = false;
		//protected float m_sep_x = 0;
        /// <summary> Y軸セパレート値 </summary>
		public float m_sep_y_graph = 0;
        /// <summary> Y軸セパレート数 </summary>
		protected int m_num_y_graph = 0;
        /// <summary> X軸セパレータ値 </summary>
        protected float m_sep_x_graph = 0;
        /// <summary> X軸セパレータ数 </summary>
        protected int m_num_x_graph = 0;
        /// <summary> グラフの種類：enumGraphKind </summary>
        private enumGraphKind m_GraphKind = enumGraphKind.IsTime;	// グラフの種類（1：日表、2：月表）
		//protected int m_GraphXSum = 0;	// グラフの横軸数
        /// <summary> 描画領域の幅 </summary>
        private float m_Width = 800;		// 描画領域の幅
        /// <summary> 描画領域の高さ </summary>
		public float m_Height = 600;		// 描画領域の高さ
        /// <summary> グラフ出力ファイル名 </summary>
        private string m_OutImageFileName = "";	// 出力ファイル名
        /// <summary> 欠測値 </summary>
        private float m_InvalidValue = 9999.9F;	// 欠測値
        /// <summary> Y軸最大値 </summary>
        private float m_max_y_val = 60.0F;	// 描画Y軸最大値
        /// <summary> Y軸最小値 </summary>
		public float m_min_y_val = -10.0F;	// 描画Y軸最小値
        /// <summary> Y軸セパレート値 </summary>
        private float m_sep_y_val = 10.0F;	// 区切り
        /// <summary> X軸最大値 </summary>
        private float m_max_x_val = 0.0F;	// 描画X軸最大値
        /// <summary> X軸最小値 </summary>
        private float m_min_x_val = 0.0F;	// 描画X軸最小値
        /// <summary> X軸セパレート値 </summary>
		public float m_sep_x_val = 0;	// 区切り
        /// <summary> X軸の種別：enumXKind </summary>
        private enumXKind m_x_kind = enumXKind.IsNumeric;	// X軸の種別（0：数値、1：時刻or日付）
        /// <summary> 凡例の表示・非表示：enumHasLegend </summary>
        private enumHasLegend m_HasLegend = enumHasLegend.HasTrue;	// 凡例の表示・非表示（true：表示、false：非表示）
        /// <summary> Y軸の反転（true：反転、false：通常） </summary>
        private bool m_YReverse = false;	// Y軸の反転（true：反転、false：通常）
        /// <summary> X軸目盛の表示 （true：目盛線あり、false：目盛線なし）</summary>
        private bool m_has_measure_x = true;    // X軸目盛の表示
        /// <summary> Y軸目盛の表示 （true：目盛線あり、false：目盛線なし）</summary>
        private bool m_has_measure_y = true;    // Y軸目盛の表示
        /// <summary> X軸目盛の種別 （true：別目盛線あり、false：別目盛線なし）</summary>
        private bool m_measure_x_kind = false;    // X軸目盛の種別（true：別目盛線あり、false：別目盛線なし）
        /// <summary> Y軸目盛の種別 （true：別目盛線あり、false：別目盛線なし）</summary>
        private bool m_measure_y_kind = false;    // Y軸目盛の種別（true：別目盛線あり、false：別目盛線なし）
        /// <summary> X軸目盛の書式 </summary>
        private string m_measure_x_format = "0";    // X軸の目盛書式
        /// <summary> X軸目盛の書式 </summary>
        private string m_measure_y_format = "0";    // Y軸の目盛書式
        /// <summary> 背景画像ファイル名 </summary>
        private string m_InImageFileName = "";	// 取り込みファイル名
        /// <summary> 背景画像のX値 </summary>
        private float m_InImageX = 0;   // 取り込み位置X
        /// <summary> 背景画像のY値 </summary>
        private float m_InImageY = 0;   // 取り込み位置Y
        /// <summary> 描画開始日時 </summary>
        private DateTime m_StartTime;   // 描画開始日時[2010.02.07]

        /// <summary>X値：一つ前</summary>
        protected float m_PrevX = McGraphViewDefine.g_InvalidValue2; //[2010.06.02]
        /// <summary>Y値：一つ前</summary>
        protected float m_PrevY = McGraphViewDefine.g_InvalidValue2; //[2010.06.02]

        #endregion

        #region プロパティの定義
        /**********************************************************************
		 * プロパティの定義
		**********************************************************************/
        /// <summary>グラフの種類</summary>
        public enumGraphKind GraphKind	// グラフの種類
		{
			set
			{
				m_GraphKind = value;
			}
			get
			{
				return m_GraphKind;
			}
		}
//		public int GraphXSum	// グラフの横軸数
//		{
//			set
//			{
//				m_GraphXSum = value;
//			}
//			get
//			{
//				return m_GraphXSum;
//			}
//		}
        /// <summary>グラフの横幅</summary>
        public int ImageWidth	// 描画領域の幅
		{
			set
			{
				m_Width = (float)value;
			}
			get
			{
				return (int)m_Width;
			}
		}
        /// <summary>グラフの縦幅</summary>
        public int ImageHeight	// 描画領域の高さ
		{
			set
			{
				m_Height = (float)value;
			}
			get
			{
				return (int)m_Height;
			}
		}
        /// <summary>グラフの出力ファイル名</summary>
        public string OutImgaeFileName	// 出力ファイル名
		{
			set
			{
				m_OutImageFileName = value;
			}
			get
			{
				return m_OutImageFileName;
			}
		}
        /// <summary>欠測値</summary>
        public float InvalidValue	// 欠測値
		{
			set
			{
				m_InvalidValue = value;
			}
			get
			{
				return m_InvalidValue;
			}
		}
        /// <summary>Y軸の最大値</summary>
        public float GraphMaxY	// 描画Y最大値
		{
			set
			{
				m_max_y_val = value;
			}
			get
			{
				return m_max_y_val;
			}
		}
        /// <summary>Y軸の最小値</summary>
        public float GraphMinY	// 描画Y最小値
		{
			set
			{
				m_min_y_val = value;
			}
			get
			{
				return m_min_y_val;
			}
		}
        /// <summary>Y軸の区切り値</summary>
        public float GraphSepY	// Yの区切り
		{
			set
			{
				m_sep_y_val = value;
			}
			get
			{
				return m_sep_y_val;
			}
		}
        /// <summary>X軸の最大値</summary>
        public float GraphMaxX	// 描画X最大値
		{
			set
			{
				m_max_x_val = value;
			}
			get
			{
				return m_max_x_val;
			}
		}
        /// <summary>X軸の最小値</summary>
        public float GraphMinX	// 描画X最小値
		{
			set
			{
				m_min_x_val = value;
			}
			get
			{
				return m_min_x_val;
			}
		}
        /// <summary>X軸の区切り値</summary>
        public float GraphSepX	// Xの区切り
		{
			set
			{
				m_sep_x_val = value;
			}
			get
			{
				return m_sep_x_val;
			}
		}
        /// <summary>X軸の種別</summary>
        public enumXKind GraphXKind	// X軸の種別
		{
			set
			{
				m_x_kind = value;
			}
			get
			{
				return m_x_kind;
			}
		}
        /// <summary>凡例の表示有無</summary>
        public enumHasLegend GraphHasLegend	// 凡例の表示・非表示
		{
			set
			{
				m_HasLegend = value;
			}
			get
			{
				return m_HasLegend;
			}
		}
        /// <summary>Y軸の反転有無</summary>
        public bool GraphYReverse	// Y軸の反転有無
		{
			set
			{
				m_YReverse = value;
			}
			get
			{
				return m_YReverse;
			}
		}
        /// <summary>X軸の目盛表示有無</summary>
        public bool GraphHasMeasureX	// X軸の目盛表示
        {
            set
            {
                m_has_measure_x = value;
            }
            get
            {
                return m_has_measure_x;
            }
        }
        /// <summary>Y軸の目盛表示有無</summary>
        public bool GraphHasMeasureY	// Y軸の目盛表示
        {
            set
            {
                m_has_measure_y = value;
            }
            get
            {
                return m_has_measure_y;
            }
        }
        /// <summary>X軸の目盛種別</summary>
        public bool GraphMeasureXKind	// X軸の目盛の種別
        {
            set
            {
                m_measure_x_kind = value;
            }
            get
            {
                return m_measure_x_kind;
            }
        }
        /// <summary>Y軸の目盛種別</summary>
        public bool GraphMeasureYKind	// Y軸の目盛の種別
        {
            set
            {
                m_measure_y_kind = value;
            }
            get
            {
                return m_measure_y_kind;
            }
        }
        /// <summary>X軸の目盛書式</summary>
        public string GraphMeasureXFormat	// X軸の目盛の書式
        {
            set
            {
                m_measure_x_format = value;
            }
            get
            {
                return m_measure_x_format;
            }
        }
        /// <summary>Y軸の目盛書式</summary>
        public string GraphMeasureYFormat	// Y軸の目盛の書式
        {
            set
            {
                m_measure_y_format = value;
            }
            get
            {
                return m_measure_y_format;
            }
        }
        /// <summary>取込み画像のファイル名</summary>
        public string InImageFileName	// 取り込みファイル名
        {
            set
            {
                m_InImageFileName = value;
            }
            get
            {
                return m_InImageFileName;
            }
        }
        /// <summary>取込み画像の描画位置X</summary>
        public float InImageX	// 取り込み位置X
        {
            set
            {
                m_InImageX = value;
            }
            get
            {
                return m_InImageX;
            }
        }
        /// <summary>取込み画像の描画位置Y</summary>
        public float InImageY	// 取り込み位置Y
        {
            set
            {
                m_InImageY = value;
            }
            get
            {
                return m_InImageY;
            }
        }
        /// <summary>描画開始日時</summary>
        public DateTime StartTime	// 描画開始日時
        {
            set
            {
                m_StartTime = value;
            }
            get
            {
                return m_StartTime;
            }
        }
        #endregion

        #region obcImagesLib
        /// <summary><para>method outline</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>McGraphViewImageLib csImageLib = new McGraphViewImageLib();</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public McGraphViewImageLib()
		{
			// 
			// TODO: コンストラクタ ロジックをここに追加してください。
			//
        }
        #endregion

        #region ~obcImagesLib
        /// <summary><para>method outline</para>
        /// <para>本クラスのデストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para></para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        ~McGraphViewImageLib()
        {
            // 
            // TODO: デストラクタ ロジックをここに追加してください。
            //
        }
        #endregion

        #region InitializeGraph
        /// <summary><para>method outline</para>
        /// <para>グラフの初期化を行います。</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>InitializeGraph()</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたプロパティ値を元に、出力するグラフの背景を作成します。</para>
        /// <para>本メソッドを呼び出す前に、必ず以下のプロパティ値を設定して下さい。</para>
        /// <para>    ImageWidth</para>
        /// <para>    ImageHeight</para>
        /// <para>    GraphKind</para>
        /// <para>    OutImgaeFileName</para>
        /// <para>    InvalidValue</para>
        /// <para>    GraphMaxY</para>
        /// <para>    GraphMinY</para>
        /// <para>    GraphSepY</para>
        /// <para>    GraphMaxX</para>
        /// <para>    GraphMinX</para>
        /// <para>    GraphSepX</para>
        /// <para>    GraphXKind</para>
        /// <para>    GraphHasLegend</para>
        /// <para>    GraphYReverse</para>
        /// <para>    GraphHasMeasureX</para>
        /// <para>    GraphMeasureXFormat</para>
        /// <para>    GraphMeasureXKind</para>
        /// <para>    GraphHasMeasureY</para>
        /// <para>    GraphMeasureYFormat</para>
        /// <para>    GraphMeasureYKind</para>
        /// </remarks>
        public void InitializeGraph()
		{
			//			if (m_GraphKind == 0)
			//				return;
			if (m_Width == 0)
				return;
			if (m_Height == 0)
				return;
            /*
			if (m_OutImageFileName.Trim().Length == 0)
				return;
            */

            if (m_bm == null)
            {
                m_bm = new Bitmap((int)m_Width, (int)m_Height);
                m_g = Graphics.FromImage(m_bm);
                m_g.Clear(Color.White);
            }

			// 横軸の区切りを計算
			//m_sep_x = (m_Width - (m_sep_left + m_sep_right)) / m_GraphXSum;
			/*
			switch (m_x_kind)
			{
				case enumXKind.IsNumeric:
					break;
				case enumXKind.IsTimeOrDay:
					m_sep_right = 60;
					break;
			}
			*/

            InitializeSeparate();   //[2010.02.07]
            /* [2010.02.07]
			float a_iTmp = (m_Width - (m_sep_left + m_sep_right));
			m_sep_x_graph = a_iTmp / (m_max_x_val - m_min_x_val);
			int a_iTmp2 = (int)((m_max_x_val - m_min_x_val)/ m_sep_x_val);
			if ((a_iTmp2 * m_sep_x_val) > (m_max_x_val - m_min_x_val))
			{
				a_iTmp2++;
			}
			m_num_x_graph = a_iTmp2;
			switch (m_x_kind)
			{
				case enumXKind.IsNumeric:
					break;
				case enumXKind.IsTimeOrDay:
					m_num_x_graph++;
					// 再計算する。
					if ((m_num_x_graph * m_sep_x_graph) > a_iTmp)
					{
						m_sep_x_graph = a_iTmp / m_num_x_graph;
					}
					break;
			}

			// 縦軸の区切りを計算
			a_iTmp = (m_Height - (m_sep_bottom + m_sep_top));
			m_sep_y_graph = a_iTmp / (m_max_y_val - m_min_y_val);
			a_iTmp2 = (int)((m_max_y_val - m_min_y_val)/ m_sep_y_val);
			if ((a_iTmp2 * m_sep_y_val) > (m_max_y_val - m_min_y_val))
				a_iTmp2++;
			m_num_y_graph = a_iTmp2;
             */ 

			DrawGraphBackGround();	// グラフの背景を描画
            //_drawGraphBackGround();	// グラフの背景を描画

            //m_InitOK = true;
        }
        #endregion

        #region
        //[2010.02.07]
        /// <summary><para>method outline</para>
        /// <para>グラフセパレート値算出処理</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>InitializeSeparate()</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>グラフのX軸・Y軸それぞれのセパレート値を算出します。</para>
        /// </remarks>
        public void InitializeSeparate()
        {
            float a_iTmp = (m_Width - (m_sep_left + m_sep_right));
            m_sep_x_graph = a_iTmp / (m_max_x_val - m_min_x_val);
            int a_iTmp2 = (int)((m_max_x_val - m_min_x_val) / m_sep_x_val);
            if ((a_iTmp2 * m_sep_x_val) > (m_max_x_val - m_min_x_val))
            {
                a_iTmp2++;
            }
            m_num_x_graph = a_iTmp2;
            switch (m_x_kind)
            {
                case enumXKind.IsNumeric:
                    break;
                case enumXKind.IsTimeOrDay:
                    //[2010.02.17]
                    /*
                    m_num_x_graph++;
                    // 再計算する。
                    if ((m_num_x_graph * m_sep_x_graph) > a_iTmp)
                    {
                        m_sep_x_graph = a_iTmp / m_num_x_graph;
                    }
                     */ 
                    break;
            }

            // 縦軸の区切りを計算
            a_iTmp = (m_Height - (m_sep_bottom + m_sep_top));
            m_sep_y_graph = a_iTmp / (m_max_y_val - m_min_y_val);
            a_iTmp2 = (int)((m_max_y_val - m_min_y_val) / m_sep_y_val);
            if ((a_iTmp2 * m_sep_y_val) > (m_max_y_val - m_min_y_val))
                a_iTmp2++;
            m_num_y_graph = a_iTmp2;
        }
        #endregion

        #region 消去　グラフの背景に画像を取り込む　InitializeInImage
        /*
        /// <summary>
        /// グラフの背景に画像を取り込みます。
        /// </summary>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// 指定されたプロパティ値を元に、出力するグラフの背景に画像を取り込みます。
        /// <para>本メソッドを呼び出す前に、必ず以下のプロパティ値を設定して下さい。</para>
        /// <para>    ImageWidth</para>
        /// <para>    ImageHeight</para>
        /// <para>    GraphKind</para>
        /// <para>    OutImgaeFileName</para>
        /// <para>    InvalidValue</para>
        /// <para>    GraphMaxY</para>
        /// <para>    GraphMinY</para>
        /// <para>    GraphSepY</para>
        /// <para>    GraphMaxX</para>
        /// <para>    GraphMinX</para>
        /// <para>    GraphSepX</para>
        /// <para>    GraphXKind</para>
        /// <para>    GraphHasLegend</para>
        /// <para>    GraphYReverse</para>
        /// <para>    GraphHasMeasureX</para>
        /// <para>    GraphMeasureXFormat</para>
        /// <para>    GraphMeasureXKind</para>
        /// <para>    GraphHasMeasureY</para>
        /// <para>    GraphMeasureYFormat</para>
        /// <para>    GraphMeasureYKind</para>
        /// </remarks>
        public void InitializeInImage()
        {
            //			if (m_GraphKind == 0)
            //				return;
            if (m_Width == 0)
                return;
            if (m_Height == 0)
                return;
            if (m_OutImageFileName.Trim().Length == 0)
                return;
            if (m_InImageFileName.Trim().Length == 0)
                return;

            if (m_bm == null)
            {
                m_bm = new Bitmap((int)m_Width, (int)m_Height);
                m_g = Graphics.FromImage(m_bm);
                m_g.Clear(Color.White);
            }

            if (File.Exists(m_InImageFileName.Trim()) == true)
            {
                Image a_img = Image.FromFile(m_InImageFileName.Trim());
                Point a_point = new Point((int)m_InImageX,(int)m_InImageY);
                m_g.DrawImageUnscaled(
                    a_img,
                    a_point
                    );
            }

            m_InitOK = true;
        }
        */
        #endregion

        #region 消去　MakeImageFile
        /*
        /// <summary>
        /// 最終的な描画内容を元に、グラフ画像を作成します。
        /// </summary>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void MakeImageFile()
		{
			if (m_InitOK == false)
				return;

            m_bm.Save(m_OutImageFileName.Trim(),ImageFormat.Png);
			m_g.Dispose();
			m_g = null;
			m_bm.Dispose();
			m_bm = null;
        }
        */
        #endregion

        #region 円（塗りつぶし）　_plotFillEllipse
        //*********************************************************************
		// @(f)
		// 機能      : 円（塗り潰し）の描画
		//
		// 引数      : ARG1 - Brushオブジェクト
		//             ARG2 - プロットの中心X
		//             ARG3 - プロットの中心Y
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /// <summary><para>method outline</para>
        /// <para>円（塗り潰し）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotFillEllipse(b, x, y)</para>
        /// </example>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotFillEllipse(
			Brush b,
			float x,
			float y
			)
		{
            //[2010.02.21]
            float a_size_Legend = m_size_Legend;
            m_g.FillEllipse(
                b,
                x - (a_size_Legend / 2),
                y - (a_size_Legend / 2),
                a_size_Legend,
                a_size_Legend
                );
            /*
			m_g.FillEllipse(
				b,
				x - (m_size_Legend / 2),
				y - (m_size_Legend / 2),
				m_size_Legend,
				m_size_Legend
				);
             */ 
        }
        #endregion

        #region 円（塗りつぶし無し）_plotEllipse
        //*********************************************************************
        // @(f)
        // 機能      : 円（塗り潰しなし）の描画
        //
        // 引数      : ARG1 - Penオブジェクト
        //             ARG2 - プロットの中心X
        //             ARG3 - プロットの中心Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>円（塗り潰しなし）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotEllipse(p, x, y)</para>
        /// </example>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotEllipse(
            Pen p,
            float x,
            float y
            )
        {
            m_g.FillEllipse(
                Brushes.White,
                x - (m_size_Legend / 2),
                y - (m_size_Legend / 2),
                m_size_Legend,
                m_size_Legend
                );
            m_g.DrawEllipse(
                p,
                x - (m_size_Legend / 2),
                y - (m_size_Legend / 2),
                m_size_Legend,
                m_size_Legend
                );
        }
        #endregion

        #region 四角形（塗りつぶし）_plotFillRectangle
        //*********************************************************************
		// @(f)
		// 機能      : 四角形（塗り潰し）の描画
		//
		// 引数      : ARG1 - Brushオブジェクト
		//             ARG2 - プロットの中心X
		//             ARG3 - プロットの中心Y
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>四角形（塗り潰し）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotFillRectangle(p, x, y)</para>
        /// </example>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotFillRectangle(
			Brush b,
			float x,
			float y
			)
		{
			m_g.FillRectangle(
				b,
				x - (m_size_Legend / 2),
				y - (m_size_Legend / 2),
				m_size_Legend,
				m_size_Legend
				);
        }
        #endregion

        #region 四角形（塗り潰しなし）_plotRectangle
        //*********************************************************************
        // @(f)
        // 機能      : 四角形（塗り潰しなし）の描画
        //
        // 引数      : ARG1 - Penオブジェクト
        //             ARG2 - プロットの中心X
        //             ARG3 - プロットの中心Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>四角形（塗り潰しなし）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotRectangle(p, x, y)</para>
        /// </example>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotRectangle(
            Pen p,
            float x,
            float y
            )
        {
            m_g.FillRectangle(
                Brushes.White,
                x - (m_size_Legend / 2),
                y - (m_size_Legend / 2),
                m_size_Legend,
                m_size_Legend
                );
            m_g.DrawRectangle(
                p,
                x - (m_size_Legend / 2),
                y - (m_size_Legend / 2),
                m_size_Legend,
                m_size_Legend
                );
        }
        #endregion

        #region 三角形（塗り潰し）_plotFillTriangle
        //*********************************************************************
		// @(f)
		// 機能      : 三角形（塗り潰し）の描画
		//
		// 引数      : ARG1 - Brushオブジェクト
		//             ARG2 - プロットの中心X
		//             ARG3 - プロットの中心Y
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>三角形（塗り潰し）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotFillTriangle(p, x, y)</para>
        /// </example>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotFillTriangle(
			Brush b,
			float x,
			float y
			)
		{
			PointF[] a_pf2 = new PointF[3];
			a_pf2[0] = new PointF(x,y - (m_size_Legend / 1.8F));
			a_pf2[1] = new PointF(x - (m_size_Legend / 1.8F),y + (m_size_Legend / 1.8F));
			a_pf2[2] = new PointF(x + (m_size_Legend / 1.8F),y + (m_size_Legend / 1.8F));
			m_g.FillPolygon(
				b,
				a_pf2,
				System.Drawing.Drawing2D.FillMode.Winding
				);
        }
        #endregion

        #region 三角形（塗り潰しなし）_plotTriangle
        //*********************************************************************
        // @(f)
        // 機能      : 三角形（塗り潰しなし）の描画
        //
        // 引数      : ARG1 - Penオブジェクト
        //             ARG2 - プロットの中心X
        //             ARG3 - プロットの中心Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>三角形（塗り潰しなし）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotTriangle(p, x, y)</para>
        /// </example>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotTriangle(
            Pen p,
            float x,
            float y
            )
        {
            PointF[] a_pf2 = new PointF[3];
            a_pf2[0] = new PointF(x, y - (m_size_Legend / 1.8F));
            a_pf2[1] = new PointF(x - (m_size_Legend / 1.8F), y + (m_size_Legend / 1.8F));
            a_pf2[2] = new PointF(x + (m_size_Legend / 1.8F), y + (m_size_Legend / 1.8F));
            m_g.FillPolygon(
                Brushes.White,
                a_pf2,
                System.Drawing.Drawing2D.FillMode.Winding
                );
            m_g.DrawPolygon(
                p,
                a_pf2
                );
        }
        #endregion

        #region ひし形（塗り潰し）_plotFillRhombus
        //*********************************************************************
        // @(f)
        // 機能      : ひし形（塗り潰し）の描画
        //
        // 引数      : ARG1 - Brushオブジェクト
        //             ARG2 - プロットの中心X
        //             ARG3 - プロットの中心Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>ひし形（塗り潰し）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotFillRhombus(p, x, y)</para>
        /// </example>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotFillRhombus(
            Brush b,
            float x,
            float y
            )
        {
            PointF[] a_pf2 = new PointF[4];
            a_pf2[0] = new PointF(x - (m_size_Legend / 1.8F), y);
            a_pf2[1] = new PointF(x, y - (m_size_Legend / 1.8F));
            a_pf2[2] = new PointF(x + (m_size_Legend / 1.8F), y);
            a_pf2[3] = new PointF(x, y + (m_size_Legend / 1.8F));
            m_g.FillPolygon(
                b,
                a_pf2,
                System.Drawing.Drawing2D.FillMode.Winding
                );
        }
        #endregion

        #region ひし形（塗り潰しなし）_plotRhombus
        //*********************************************************************
        // @(f)
        // 機能      : ひし形（塗り潰しなし）の描画
        //
        // 引数      : ARG1 - Penオブジェクト
        //             ARG2 - プロットの中心X
        //             ARG3 - プロットの中心Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>ひし形（塗り潰しなし）の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotRhombus(p, x, y)</para>
        /// </example>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="x">プロットの中心X</param>
        /// <param name="y">プロットの中心Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotRhombus(
            Pen p,
            float x,
            float y
            )
        {
            PointF[] a_pf2 = new PointF[4];
            a_pf2[0] = new PointF(x - (m_size_Legend / 1.8F), y);
            a_pf2[1] = new PointF(x, y - (m_size_Legend / 1.8F));
            a_pf2[2] = new PointF(x + (m_size_Legend / 1.8F), y);
            a_pf2[3] = new PointF(x, y + (m_size_Legend / 1.8F));
            m_g.FillPolygon(
                Brushes.White,
                a_pf2,
                System.Drawing.Drawing2D.FillMode.Winding
                );
            m_g.DrawPolygon(
                p,
                a_pf2
                );
        }
        #endregion

        #region 棒グラフの描画 _plotFillBarGraph
        //*********************************************************************
		// @(f)
		// 機能      : 棒グラフの描画
		//
		// 引数      : ARG1 - Penオブジェクト
		//             ARG2 - Brushオブジェクト
		//             ARG3 - プロットの左上X
		//             ARG4 - プロットの左上Y
		//             ARG5 - 幅
		//             ARG6 - 高さ
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /// <summary><para>method outline</para>
        /// <para>棒グラフの描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotFillBarGraph((b, x, y, w, h)</para>
        /// </example>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="x">プロットの左上X</param>
        /// <param name="y">プロットの左上Y</param>
        /// <param name="w">幅</param>
        /// <param name="h">高さ</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
		private void _plotFillBarGraph(
			Pen p,
			Brush b,
			float x,
			float y,
			float w,
			float h
			)
		{
			m_g.FillRectangle(
				b,
				x,
				y,
				w,
				h
				);
			m_g.DrawRectangle(
				p,
				x,
				y,
				w,
				h
				);
        }
        #endregion

        #region 消去　凡例の描画（引数６）DrawLegend
        //*********************************************************************
		// @(f)
		// 機能      : 凡例の描画
		//
		// 引数      : ARG1 - Penオブジェクト
		//             ARG2 - Brushオブジェクト
		//             ARG3 - 凡例文字
		//             ARG4 - 凡例表示順序
        //             ARG5 - 凡例グラフ種別
		//             ARG6 - 凡例種別
		//                    1：○
		//                    2：□
		//                    3：△
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /*/// <summary>
        /// 凡例を描画します。
        /// </summary>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="s">凡例の文字列</param>
        /// <param name="i">凡例の描画順序</param>
        /// <param name="pk">凡例のプロット種別：enumPlotKind</param>
        /// <param name="pf">凡例のプロット型：enumPlotForm</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawLegend(
			Pen p,
			Brush b,
			string s,
			int i,
            enumPlotKind pk,
			enumPlotForm pf
			)
		{
			if (m_HasLegend == enumHasLegend.HasFalse)
				return;

            float a_x = m_sep_left + m_sep_x_graph + (m_sep_x_graph * 4 * (float)(i - 1)) + (m_sep_x_graph / 2);
            float a_y = (m_sep_string_y * 2);
            float a_moveX = 0;

            if (pk == enumPlotKind.Line)
            {
                m_g.DrawLine(
                    p,
                    m_sep_left + m_sep_x_graph + (m_sep_x_graph * 4 * (float)(i - 1)),
                    m_sep_string_y * 2,
                    m_sep_left + (m_sep_x_graph * 2) + (m_sep_x_graph * 4 * (float)(i - 1)),
                    m_sep_string_y * 2
                    );
            }
            else
            {
                a_moveX = -(m_sep_string_x * 4);
            }

            _plotXYPoint(
                p,
                b,
                a_x,
                a_y,
                pf
                );

			m_g.DrawString(
				s,
				m_font1,
				Brushes.Black,
				new PointF(m_sep_left + (m_sep_x_graph * 2) + (m_sep_x_graph * 4 * (float)(i - 1)) + m_sep_string_x + a_moveX,m_sep_string_y)
				);
        }*/
        #endregion

        #region 消去　凡例の描画（引数７）DrawLegend
        //*********************************************************************
        // @(f)
        // 機能      : 凡例の描画
        //
        // 引数      : ARG1 - Penオブジェクト
        //             ARG2 - Brushオブジェクト
        //             ARG3 - 凡例文字
        //             ARG4 - X
        //             ARG5 - Y
        //             ARG5 - 凡例グラフ種別
        //             ARG6 - 凡例種別
        //                    1：○
        //                    2：□
        //                    3：△
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /*/// <summary>
        /// 凡例を描画します。
        /// </summary>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="s">凡例の文字列</param>
        /// <param name="x">凡例の描画位置X</param>
        /// <param name="y">凡例の描画位置Y</param>
        /// <param name="pk">凡例のプロット種別：enumPlotKind</param>
        /// <param name="pf">凡例のプロット型：enumPlotForm</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawLegend(
            Pen p,
            Brush b,
            string s,
            float x,
            float y,
            enumPlotKind pk,
            enumPlotForm pf
            )
        {
            if (m_HasLegend == enumHasLegend.HasFalse)
                return;

            float a_x = x;  // m_sep_left + m_sep_x_graph + (m_sep_x_graph * 4 * (float)(i - 1)) + (m_sep_x_graph / 2);
            float a_y = y;  // (m_sep_string_y * 2);
            float a_moveX = 0;

            if (pk == enumPlotKind.Line)
            {
                m_g.DrawLine(
                    p,
                    a_x - (m_sep_x_graph / 2),   //m_sep_left + m_sep_x_graph + (m_sep_x_graph * 4 * (float)(i - 1)),
                    m_sep_string_y * 2,
                    a_x + (m_sep_x_graph / 2),    //m_sep_left + (m_sep_x_graph * 2) + (m_sep_x_graph * 4 * (float)(i - 1)),
                    m_sep_string_y * 2
                    );
                a_moveX = (m_sep_string_x * 4);
            }
            else
            {
                a_moveX = -(m_sep_string_x * 4);
            }

            _plotXYPoint(
                p,
                b,
                a_x,
                a_y,
                pf
                );

            m_g.DrawString(
                s,
                m_font1,
                Brushes.Black,
                new PointF(a_x + a_moveX, m_sep_string_y)
                );
        }*/
        #endregion

        #region グラフY軸目盛の描画　_drawYMeasureString
        //*********************************************************************
        // @(f)
        // 機能      : グラフY軸目盛の描画
        //
        // 引数      : ARG1 - 文字
        //             ARG2 - X（元々よりずらす差分値）
        //             ARG3 - Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      : 引数Xについての仕様は、_drawXMeasureStringに合わせるべき？
        //
        //*********************************************************************
        /// <summary><para>method outline</para>
        /// <para>グラフY軸目盛の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_drawYMeasureString(s, x, y)</para>
        /// </example>
        /// <param name="s">文字</param>
        /// <param name="x">X座標（元々よりずらす差分値）</param>
        /// <param name="y">Y座標</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _drawYMeasureString(
            string s,
            float x,
            float y
            )
        {
            /* 元に作成したソース
            s.PadLeft(6);
            int a_len = s.Length;
            float a_baseX = m_sep_left - (m_sep_string_x * 8);
            float a_x = 0;
            if ((a_baseX + (a_len * (m_sep_string_x * 2))) >= m_sep_left)
                a_x = a_baseX - ((a_baseX + (a_len * (m_sep_string_x * 2))) - m_sep_left);
            else
                a_x = a_baseX;
            m_g.DrawString(
                s,
                m_font1,
                Brushes.Black,
                new PointF(a_x + x, y)
                );
             */

            //[2010.02.21]
            int a_baseLen = s.Length;
            //int a_baseLen = Convert.ToString(m_max_y_val).Length;
            int a_len = s.Length;
            //[2010.02.21]
            float a_baseX = Convert.ToSingle((a_baseLen * 6.6F));
            //float a_baseX = Convert.ToSingle(m_sep_left - (a_baseLen * 6.6F));
            //float a_baseX = m_sep_left - (m_sep_string_x * 5);
            float a_x = 0;

            //[2010.02.17]
            /*
            if ((a_baseX + (a_len * (m_sep_string_x * 2))) >= m_sep_left)
                a_x = a_baseX - ((a_baseX + (a_len * (m_sep_string_x * 2))) - m_sep_left);
            else
                a_x = a_baseX;
             */

            a_x = m_sep_left - a_baseX - m_font1.Size/2;
            //a_x = a_baseX;

            for (int a_iCnt = 1; a_iCnt <= (a_baseLen - a_len); a_iCnt++)
            {
                a_x += (m_sep_string_x * 1.5F);
            }
            m_g.DrawString(
                s,
                m_font1,
                Brushes.Black,
                new PointF(a_x + x, y)
                );
        }
        #endregion

        #region グラフY軸目盛の描画（右軸）DrawYMeasureString2
        //*********************************************************************
        // @(f)
        // 機能      : グラフY軸目盛の描画
        //
        // 引数      : ARG1 - 文字
        //             ARG2 - X（元々よりずらす差分値）
        //             ARG3 - Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      : 引数Xについての仕様は、_drawXMeasureStringに合わせるべき？
        //
        //*********************************************************************
        /// <summary><para>method outline</para>
        /// <para>グラフY軸目盛の描画2</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_drawYMeasureString2(s, x, y)</para>
        /// </example>
        /// <param name="s">文字</param>
        /// <param name="x">X座標（元々よりずらす差分値）</param>
        /// <param name="y">Y座標</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public void DrawYMeasureString2(
            string s,
            float x,
            float y
            )
        {
            /* 元に作成したソース
            s.PadLeft(6);
            int a_len = s.Length;
            float a_baseX = m_sep_left - (m_sep_string_x * 8);
            float a_x = 0;
            if ((a_baseX + (a_len * (m_sep_string_x * 2))) >= m_sep_left)
                a_x = a_baseX - ((a_baseX + (a_len * (m_sep_string_x * 2))) - m_sep_left);
            else
                a_x = a_baseX;
            m_g.DrawString(
                s,
                m_font1,
                Brushes.Black,
                new PointF(a_x + x, y)
                );
             */

            //[2010.02.21]
            int a_baseLen = s.Length;
            //int a_baseLen = Convert.ToString(m_max_y_val).Length;
            int a_len = s.Length;
            //[2010.02.21]
            float a_baseX = Convert.ToSingle((a_baseLen * 6.6F));
            //float a_baseX = Convert.ToSingle(m_sep_left - (a_baseLen * 6.6F));
            //float a_baseX = m_sep_left - (m_sep_string_x * 5);
            float a_x = 0;

            //[2010.02.17]
            /*
            if ((a_baseX + (a_len * (m_sep_string_x * 2))) >= m_sep_left)
                a_x = a_baseX - ((a_baseX + (a_len * (m_sep_string_x * 2))) - m_sep_left);
            else
                a_x = a_baseX;
             */

            a_x = m_sep_right/2 - a_baseX;
            //a_x = a_baseX;

            for (int a_iCnt = 1; a_iCnt <= (a_baseLen - a_len); a_iCnt++)
            {
                a_x += (m_sep_string_x * 1.5F);
            }

            m_g.DrawString(
                s,
                m_font1,
                Brushes.Black,
                new PointF(x, y)
                );
        }
        #endregion

        #region 消去 Y軸の目盛値の描画 DrawYUnitString
        //*********************************************************************
        // @(f)
        // 機能      : グラフY軸単位の描画
        //
        // 引数      : ARG1 - 単位文字
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /*/// <summary>
        /// Y軸の目盛値を描画します。
        /// </summary>
        /// <param name="s">目盛値</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawYUnitString(
			string s
			)
		{

			float a_fyUnit = 0;
			if (m_YReverse == false)
				a_fyUnit = m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val))) - (m_sep_string_y * 6);
			else
				a_fyUnit = m_Height - m_sep_bottom + (m_sep_string_y * 4);
			m_g.DrawString(
				s,
				m_font1,
				Brushes.Black,
                new PointF(m_sep_left - (m_sep_string_x * 12), a_fyUnit)
				);
        }*/
        #endregion

        #region 消去　グラフY軸タイトルの描画　DrawYTitleStringVertical
        //*********************************************************************
		// @(f)
		// 機能      : グラフY軸タイトルの描画
		//
		// 引数      : ARG1 - タイトル文字
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /*/// <summary>
        /// Y軸のタイトルを縦書き描画します。
        /// </summary>
        /// <param name="s">タイトル</param>
        /// <param name="h_unit">単位</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawYTitleStringVertical(
			string s,
            string h_unit
			)
		{
            int a_iCnt = 0;
            int a_len = s.Length;
            float a_baseY = m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val)) / 2);
            float a_y = a_baseY - (((float)a_len / 2F) * (m_sep_string_y * 2))- 16;
            //float a_y = a_baseY - (((float)a_len / 2F) * (m_sep_string_y * 2));

            for (a_iCnt = 1; a_iCnt <= a_len; a_iCnt++)
            {
                m_g.DrawString(
                    s.Substring(a_iCnt - 1, 1),
                    m_font1,
                    Brushes.Black,
                    new PointF(m_sep_string_x, a_y + ((m_sep_string_y * 3.3F) * (a_iCnt - 1)))
                    );
            }
            m_g.DrawString(
                h_unit,
                m_font1,
                Brushes.Black,
                new PointF(m_sep_string_x - 4, a_y + ((m_sep_string_y * 3.3F) * (a_iCnt - 1)) + 4)
                );
        }*/
        #endregion

        #region 消去　グラフY軸タイトルの描画（右軸）DrawYTitleStringVertical2
        //*********************************************************************
        // @(f)
        // 機能      : グラフY軸タイトルの描画
        //
        // 引数      : ARG1 - タイトル文字
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /*/// <summary>
        /// Y軸のタイトルを縦書き描画します。
        /// </summary>
        /// <param name="s">タイトル</param>
        /// <param name="h_unit">単位</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawYTitleStringVertical2(
            string s,
            string h_unit
            )
        {
            int a_iCnt = 0;
            int a_len = s.Length;
            float a_baseY = m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val)) / 2);
            float a_y = a_baseY - (((float)a_len / 2F) * (m_sep_string_y * 2)) - 16;
            //float a_y = a_baseY - (((float)a_len / 2F) * (m_sep_string_y * 2));

            for (a_iCnt = 1; a_iCnt <= a_len; a_iCnt++)
            {
                m_g.DrawString(
                    s.Substring(a_iCnt - 1, 1),
                    m_font1,
                    Brushes.Black,
                    new PointF(this.m_Width - (m_font1.Size * 2), a_y + ((m_sep_string_y * 3.3F) * (a_iCnt - 1)))
                    );
            }
            m_g.DrawString(
                h_unit,
                m_font1,
                Brushes.Black,
                new PointF(this.m_Width - (m_font1.Size * 4 + 4) , a_y + ((m_sep_string_y * 3.3F) * (a_iCnt - 1)) + 4)
                );
        }*/
        #endregion

        #region グラフX軸目盛の描画 _drawXMeasureString
        //*********************************************************************
        // @(f)
        // 機能      : グラフX軸目盛の描画
        //
        // 引数      : ARG1 - 文字
        //             ARG2 - X
        //             ARG3 - Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>グラフX軸目盛の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_drawXMeasureString(s, x, y)</para>
        /// </example>
        /// <param name="s">文字</param>
        /// <param name="x">描画位置X</param>
        /// <param name="y">描画位置Y</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _drawXMeasureString(
            string s,
            float x,
            float y
            )
        {
            int a_len = s.Length;
            float a_center = (a_len * m_sep_string_x) / 2;
            float a_x = x;

            if (m_x_kind == enumXKind.IsNumeric)
            {
                a_x = x - a_center;
            }
            else
            {
            }

            m_g.DrawString(
                s,
                m_font1,
                Brushes.Black,
                new PointF(a_x, y)
                );
        }
        #endregion

        #region 消去　グラフX軸目盛の描画（縦書き）DrawXMeasureStringVertical
        //*********************************************************************
        // @(f)
        // 機能      : グラフX軸目盛の描画（縦書き）
        //
        // 引数      : ARG1 - 文字
        //             ARG2 - X
        //             ARG3 - Y
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /*/// <summary>
        /// X軸の目盛値を縦書き描画します。
        /// </summary>
        /// <param name="s">目盛値</param>
        /// <param name="x">描画位置X</param>
        /// <param name="y">描画位置Y</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawXMeasureStringVertical(
            string s,
            float x,
            float y
            )
        {
            int a_len = s.Length;
            float a_center = (a_len * m_sep_string_x) / 2;
            float a_x = x;  // a_baseX

            if (m_x_kind == enumXKind.IsNumeric)
            {
                a_x = x - a_center;
                m_g.DrawString(
                    s,
                    m_font1,
                    Brushes.Black,
                    new PointF(a_x, y)
                    );
            }
            else
            {
                a_x = x - (m_sep_string_x * 2);
                for (int a_iCnt = 1; a_iCnt <= a_len; a_iCnt++)
                    m_g.DrawString(
                        s.Substring(a_iCnt - 1, 1),
                        m_font1,
                        Brushes.Black,
                        new PointF(a_x, y + ((m_sep_string_y * 3.3F) * (a_iCnt - 1)))
                        );
            }

        }*/
        #endregion

        #region 消去　グラフX軸単位の描画　DrawXUnitString
        //*********************************************************************
        // @(f)
        // 機能      : グラフX軸単位の描画
        //
        // 引数      : ARG1 - 単位文字
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /*/// <summary>
        /// X軸の目盛値を描画します。
        /// </summary>
        /// <param name="s">目盛値</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawXUnitString(
			string s
			)
		{
			float a_fxUnit = 0;
			if (m_YReverse == false)
				a_fxUnit = m_Height - m_sep_bottom + (m_sep_string_y * 7);
			else
				a_fxUnit = m_sep_top - (m_sep_string_y * 6);
			m_g.DrawString(
				s,
				m_font1,
				Brushes.Black,
				new PointF(m_sep_left + ((m_sep_x_val * m_num_x_graph) * m_sep_x_graph) + (m_sep_string_x),a_fxUnit)
				);
        }*/
        #endregion

        #region 消去　グラフX軸タイトルの描画　DrawXTitleString
        //*********************************************************************
		// @(f)
		// 機能      : グラフX軸タイトルの描画
		//
		// 引数      : ARG1 - タイトル文字
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /*/// <summary>
        /// X軸のタイトルを描画します。
        /// </summary>
        /// <param name="s">タイトル</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void DrawXTitleString(
			string s
			)
		{
			float a_fxUnit = 0;
			if (m_YReverse == false)
				a_fxUnit = m_Height - m_sep_bottom + (m_sep_string_y * 7);
			else
				a_fxUnit = m_sep_top - (m_sep_string_y * 9);
			m_g.DrawString(
				s,
				m_font1,
				Brushes.Black,
				new PointF(
                m_Width - (m_sep_left + ((m_sep_x_val * m_num_x_graph) * m_sep_x_graph) / 2),
				a_fxUnit)
				);
        }*/
        #endregion

        #region グラフ値プロット描画　PlotGraphValue
        //*********************************************************************
		// @(f)
		// 機能      : グラフ値プロットの描画
		//
		// 引数      : ARG1 - Penオブジェクト
		//             ARG2 - Brushオブジェクト
		//             ARG3 - X値
		//             ARG4 - Y値
		//             ARG5 - 一つ前のX値
		//             ARG6 - 一つ前のY値
		//             ARG7 - 表示インデックス（1オリジン）
		//             ARG9 - グラフの種類
		//                    1：折れ線
		//                    2：棒
		//                    3：散布
		//             ARG8 - プロット種別
		//                    1：○
		//                    2：□
		//                    3：△
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /// <summary><para>method outline</para>
        /// <para>グラフ上にXY値のプロットを描画します。</para>
        /// <para>指定するXY値はXY軸上で表現される値です。</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>PlotGraphValue(p, b, xv, yv, pxv, pyv, gk, pk)</para>
        /// </example>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="xv">プロットするX値</param>
        /// <param name="yv">プロットするY値</param>
        /// <param name="pxv">一つ前のプロットX値</param>
        /// <param name="pyv">一つ前のプロットY値</param>
        /// <param name="gk">プロット種類：enumPlotKind</param>
        /// <param name="pk">プロット型：enumPlotForm</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>X軸の値は、プロパティ値のGraphMinX～GraphMaxXの範囲内である必要があります。</para>
        /// <para>Y軸の値は、プロパティ値のGraphMinY～GraphMaxYの範囲内である必要があります。</para>
        /// </remarks>
        public void PlotGraphValue(
			Pen p,
			Brush b,
			float xv,
			float yv,
			float pxv,
			ref float pyv,
			//int i,
            enumPlotKind gk,
            enumPlotForm pk
			)
		{
			float a_fTmp = yv;
			float a_fxTmp = 0;
			float a_fxTmp2 = 0;
			float a_fxTmp3 = 0;
			float a_fyTmp1 = 0;
			float a_fyTmp2 = 0;

            //float a_fTmp2 = 0;  //[2010.06.01]

			switch (m_x_kind)
			{
				case enumXKind.IsNumeric:
					a_fxTmp = 0;
					break;
				case enumXKind.IsTimeOrDay:
                    //[2010.02.21]数値表示と同じとする。
					a_fxTmp = 0;
                    //a_fxTmp = (m_sep_x_graph / 2);
                    break;
			}

			a_fxTmp2 = m_sep_left + (m_sep_x_graph * (xv - m_min_x_val));

            //[2010.06.02]↓
            if (pyv > McGraphViewDefine.g_InvalidValue2)
            {
                m_PrevX = pxv;
            }
            //[2010.06.02]↑

            a_fxTmp3 = m_sep_left + (m_sep_x_graph * (m_PrevX - m_min_x_val));  //[2010.06.02]
            //a_fxTmp3 = m_sep_left + (m_sep_x_graph * (pxv - m_min_x_val));    //[2010.06.02]

            if (gk == enumPlotKind.Line)
            {
                //[2010.02.18]
                if (pyv > m_InvalidValue)  //(pyv != m_InvalidValue)
                {
                    //[2010.02.18]
                    if (a_fTmp > m_InvalidValue)   //(a_fTmp != m_InvalidValue)
                    {
                        //[2010.06.02]↓
                        if (pyv != McGraphViewDefine.g_InvalidValue2)
                        {
                            m_PrevY = pyv;
                        }

                        if ((m_PrevY != McGraphViewDefine.g_InvalidValue2) &&
                            (a_fTmp != McGraphViewDefine.g_InvalidValue2))
                        {
                            if (m_YReverse == false)
                            {
                                a_fyTmp1 = m_Height - (m_sep_bottom + ((m_PrevY - m_min_y_val) * m_sep_y_graph));   //[2010.06.02]
                                //a_fyTmp1 = m_Height - (m_sep_bottom + ((pyv - m_min_y_val) * m_sep_y_graph));     //[2010.06.02]
                                a_fyTmp2 = m_Height - (m_sep_bottom + ((a_fTmp - m_min_y_val) * m_sep_y_graph));
                            }
                            else
                            {
                                a_fyTmp1 = m_sep_top + ((m_PrevY - m_min_y_val) * m_sep_y_graph);   //[2010.06.02]
                                //a_fyTmp1 = m_sep_top + ((pyv - m_min_y_val) * m_sep_y_graph);     //[2010.06.02]
                                a_fyTmp2 = m_sep_top + ((a_fTmp - m_min_y_val) * m_sep_y_graph);
                            }

                            m_g.DrawLine(
                                p,
                                a_fxTmp3 + a_fxTmp,
                                a_fyTmp1,
                                a_fxTmp2 + a_fxTmp,
                                a_fyTmp2
                                );

                            _plotXYPoint(
                                p,
                                b,
                                a_fxTmp3 + a_fxTmp,
                                a_fyTmp1,
                                pk
                                );
                        }
                        //[2010.06.02]↑
                    }
                }
                else
                {   //[2010.06.02]
                    m_PrevY = McGraphViewDefine.g_InvalidValue2;
                }
            }

            if (m_YReverse == false)
				a_fyTmp1 = m_Height - (m_sep_bottom + ((a_fTmp - m_min_y_val) * m_sep_y_graph));
			else
				a_fyTmp1 = m_sep_top + ((a_fTmp - m_min_y_val) * m_sep_y_graph);

            //[2010.02.18]
            if (a_fTmp > m_InvalidValue)   //(a_fTmp != m_InvalidValue)
			{
				if (gk != enumPlotKind.Bar)
				{
                    if (a_fTmp != McGraphViewDefine.g_InvalidValue2)
                    {   //[2010.06.01]
                        _plotXYPoint(
                            p,
                            b,
                            a_fxTmp2 + a_fxTmp,
                            a_fyTmp1,
                            pk
                            );
                    }
				}
				else
				{
					switch (m_x_kind)
					{
						case enumXKind.IsNumeric:
							a_fxTmp2 = m_sep_left + (m_sep_x_graph * (xv - m_min_x_val)) - (m_sep_x_graph / 2);
							break;
						case enumXKind.IsTimeOrDay:
							break;
					}
					if (m_YReverse == false)
					{
						a_fyTmp1 = m_Height - (m_sep_bottom + ((a_fTmp - m_min_y_val) * m_sep_y_graph));
						a_fyTmp2 = (a_fTmp - m_min_y_val) * m_sep_y_graph;
					}
					else
					{
						a_fyTmp1 = m_sep_top;
						a_fyTmp2 = (a_fTmp - m_min_y_val) * m_sep_y_graph;
					}
					_plotFillBarGraph(
						p,
						b,
						a_fxTmp2,
						a_fyTmp1,
						m_sep_x_graph,
						a_fyTmp2
						);
				}
			}

			pyv = a_fTmp;
        }
        #endregion

        #region 消去　単純プロットの描画　PlotDirectValue
        //*********************************************************************
		// @(f)
		// 機能      : 単純プロットの描画
		//
		// 引数      : ARG1 - Penオブジェクト
		//             ARG2 - Brushオブジェクト
		//             ARG3 - X値
		//             ARG4 - Y値
		//             ARG5 - 一つ前のX値
		//             ARG6 - 一つ前のY値
		//             ARG7 - グラフの種類
		//                    1：折れ線
		//                    2：棒
		//                    3：散布
		//             ARG8 - プロット種別
		//                    1：○
		//                    2：□
		//                    3：△
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /*/// <summary>
        /// グラフ上にXY値のプロットを描画します。
        /// <para>指定するXY値は、グラフ内での座標値です。</para>
        /// </summary>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="xv">プロットするX値</param>
        /// <param name="yv">プロットするY値</param>
        /// <param name="pxv">一つ前のプロットX値</param>
        /// <param name="pyv">一つ前のプロットY値</param>
        /// <param name="gk">プロット種類：enumPlotKind</param>
        /// <param name="pk">プロット型：enumPlotForm</param>
        /// <returns>
        /// なし
        /// </returns>
        /// <remarks>
        /// </remarks>
        public void PlotDirectValue(
            Pen p,
            Brush b,
            float xv,
            float yv,
            float pxv,
            ref float pyv,
            enumPlotKind gk,
            enumPlotForm pk
            )
        {
            float a_fTmp = yv;

            if (gk == enumPlotKind.Line)
            {
                //[2010.02.18]
                if (pyv > m_InvalidValue)  //(pyv != m_InvalidValue)
                {
                    //[2010.02.18]
                    if (a_fTmp > m_InvalidValue)   //(a_fTmp != m_InvalidValue)
                    {
                        m_g.DrawLine(
                            p,
                            pxv,
                            pyv,
                            xv,
                            yv
                            );

                        _plotXYPoint(
                            p,
                            b,
                            pxv,
                            pyv,
                            pk
                            );
                    }
                }
            }

            //[2010.02.18]
            if (a_fTmp > m_InvalidValue)   //(a_fTmp != m_InvalidValue)
            {
                if (gk != enumPlotKind.Bar)
                {
                    _plotXYPoint(
                        p,
                        b,
                        xv,
                        yv,
                        pk
                        );
                }
                else
                {
                    // 棒グラフは未サポート。
                    //_plotFillBarGraph(
                    //    p,
                    //    b,
                    //    xv,
                    //    yv,
                    //    m_sep_x_graph,
                    //    a_fyTmp2
                    //    );
                }
            }

            pyv = a_fTmp;
        }*/
        #endregion

        #region XY値のプロット _plotXYPoint
        //*********************************************************************
        // @(f)
        // 機能      : XY値のプロット
        //
        // 引数      : ARG1 - Penオブジェクト
        //             ARG2 - Brushオブジェクト
        //             ARG3 - X値
        //             ARG4 - Y値
        //             ARG5 - プロット種別
        //                    1：○
        //                    2：□
        //                    3：△
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>XY値のプロット</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_plotXYPoint(p, b, x, y, pk)</para>
        /// </example>
        /// <param name="p">Penオブジェクト</param>
        /// <param name="b">Brushオブジェクト</param>
        /// <param name="x">X値</param>
        /// <param name="y">Y値</param>
        /// <param name="pk">プロット種別：enumPlotForm</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _plotXYPoint(
            Pen p,
            Brush b,
            float x,
            float y,
            enumPlotForm pk
            )
        {
            switch (pk)
            {
                case enumPlotForm.Circle:
                    _plotEllipse(
                        p,
                        x,
                        y
                        );
                    break;
                case enumPlotForm.FillCircle:
                    _plotFillEllipse(
                        b,
                        x,
                        y
                        );
                    break;
                case enumPlotForm.Rectangle:
                    _plotRectangle(
                        p,
                        x,
                        y
                        );
                    break;
                case enumPlotForm.FillRectangle:
                    _plotFillRectangle(
                        b,
                        x,
                        y
                        );
                    break;
                case enumPlotForm.Triangle:
                    _plotTriangle(
                        p,
                        x,
                        y
                        );
                    break;
                case enumPlotForm.FillTriangle:
                    _plotFillTriangle(
                        b,
                        x,
                        y
                        );
                    break;
                case enumPlotForm.Rhombus:
                    _plotRhombus(
                        p,
                        x,
                        y
                        );
                    break;
                case enumPlotForm.FillRhombus:
                    _plotFillRhombus(
                        b,
                        x,
                        y
                        );
                    break;
            }
        }
        #endregion

        #region X軸別目盛線の描画　_drawXMeasureLine
        //*********************************************************************
        // @(f)
        // 機能      : X軸の別目盛線の描画
        //
        // 引数      : ARG1 - 目盛表示のY軸開始値
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>X軸の別目盛線の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_drawXMeasureLine(mY)</para>
        /// </example>
        /// <param name="mY">目盛表示のY軸開始値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _drawXMeasureLine(
            ref float mY
            )
        {
            float a_fy = 0;
            float a_fdiv = 0;
            if (m_YReverse == false)
            {
                a_fy = m_Height - m_sep_bottom + m_sep_string_y;
                a_fdiv = m_sep_string_y;
            }
            else
            {
                a_fy = m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val))) - (m_sep_string_y);
                a_fdiv = - m_sep_string_y;
            }

            // 全体の横線
			m_g.DrawLine(
				m_pen1,
				m_sep_left,
				a_fy,
                m_sep_left + ((m_sep_x_val * m_num_x_graph) * m_sep_x_graph),
				a_fy
				);

            for (int a_iCnt = 0; a_iCnt <= (m_num_x_graph); a_iCnt++)
            {
                m_g.DrawLine(
                    m_pen1,
                    m_sep_left + ((m_sep_x_val * a_iCnt) * m_sep_x_graph),
                    a_fy,
                    m_sep_left + ((m_sep_x_val * a_iCnt) * m_sep_x_graph),
                    a_fy + a_fdiv
                    );
            }

            if (m_YReverse == false)
            {
                mY += (m_sep_string_y * 2);
            }
            else
            {
                mY -= (m_sep_string_y * 1.5F);
            }
        }
        #endregion

        #region Y軸の別目盛り線の描画　_drawYMeasureLine
        //*********************************************************************
        // @(f)
        // 機能      : Y軸の別目盛線の描画
        //
        // 引数      : ARG1 - 目盛表示のX軸開始値
        //
        // 返り値    : なし
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /// <summary><para>method outline:</para>
        /// <para>Y軸の別目盛線の描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>_drawYMeasureLine(mX)</para>
        /// </example>
        /// <param name="mX">目盛表示のX軸開始値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void _drawYMeasureLine(
            ref float mX
            )
        {
            // 全体の横線
            m_g.DrawLine(
                m_pen1,
                m_sep_left - m_sep_string_x,
                m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val))),
                m_sep_left - m_sep_string_x,
                m_Height - m_sep_bottom
                );

            for (int a_iCnt = 0; a_iCnt <= (m_num_y_graph); a_iCnt++)
            {
                m_g.DrawLine(
                    m_pen1,
                    m_sep_left - m_sep_string_x,
                    m_Height - (m_sep_bottom + (a_iCnt * (m_sep_y_graph * m_sep_y_val))),
                    m_sep_left - (m_sep_string_x * 2.2F),
                    m_Height - (m_sep_bottom + (a_iCnt * (m_sep_y_graph * m_sep_y_val)))
                    );
            }

            mX -= m_sep_string_x;
        }
        #endregion

        #region グラフの背景描画　_drawGraphBackGround
        //*********************************************************************
		// @(f)
		// 機能      : グラフの背景描画
		//
		// 引数      : なし
		//
		// 返り値    : なし
		//
		// 機能説明  :
		//
		// 備考      :
		//
		//*********************************************************************
        /// <summary><para>method outline</para>
        /// <para>グラフの背景描画</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DrawGraphBackGround()</para>
        /// </example>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
		public void DrawGraphBackGround()
		{
			int a_iCnt = 0;
			float a_fTmp = 0;
			float a_fxUnit = 0;
			float a_fyUnit = 0;
			string a_strTmp = "";
//			string a_strTmp2 = "";

			if (m_YReverse == false)
				a_fxUnit = m_Height - m_sep_bottom + m_sep_string_y;
			else
				a_fxUnit = m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val))) - (m_sep_string_y * 4);

            //-----------------------------------------------------------------
            // 別目盛線の描画
            //-----------------------------------------------------------------
            if ((m_has_measure_x == true) && (m_measure_x_kind == true))
                _drawXMeasureLine(ref a_fxUnit);

			//-----------------------------------------------------------------
			// 縦軸の描画
			//-----------------------------------------------------------------
			for (a_iCnt = 1; a_iCnt <= (m_num_x_graph - 1); a_iCnt++)
			{
				m_pen2.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;
                /*[2010.03.07]
				m_g.DrawLine(
					m_pen2,
					//(m_sep_left + (m_sep_x_graph * a_iCnt)),
                    m_sep_left + ((m_sep_x_val * a_iCnt) * m_sep_x_graph),
                    m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val))),
					//m_sep_left + (m_sep_x_graph * a_iCnt),
                    m_sep_left + ((m_sep_x_val * a_iCnt) * m_sep_x_graph),
                    m_Height - m_sep_bottom
					);
                 */ 
                if (m_has_measure_x == true)
                {
                    // タイトル文字の描画
                    switch (m_x_kind)
                    {
                        case enumXKind.IsNumeric:
                            a_strTmp = (m_min_x_val + (m_sep_x_val * (a_iCnt - 1))).ToString(m_measure_x_format);
                            //a_fTmp = m_sep_left + (m_sep_x_graph * (a_iCnt - 1)) - (m_sep_string_x * 2);
                            a_fTmp = m_sep_left + ((m_sep_x_val * (a_iCnt - 1)) * m_sep_x_graph) - (m_sep_string_x);
                            break;
                        case enumXKind.IsTimeOrDay:
                            //[2010.02.07]
                            a_strTmp = m_StartTime.AddMinutes( (m_min_x_val + (m_sep_x_val * (a_iCnt - 1))) * 10 ).ToString("HH");
                            //a_strTmp = (m_min_x_val + (m_sep_x_val * (a_iCnt - 1))).ToString(m_measure_x_format);
                            //a_fTmp = m_sep_left + (m_sep_x_graph * (a_iCnt - 1)) + m_sep_string_x;
                            a_fTmp = m_sep_left + ((m_sep_x_val * (a_iCnt - 1)) * m_sep_x_graph) + (m_sep_string_x * 2);
                            break;
                    }
                    //m_g.DrawString(
                    //    a_strTmp,
                    //    m_font1,
                    //    Brushes.Black,
                    //    new PointF(a_fTmp, a_fxUnit)
                    //    );
                    _drawXMeasureString(
                        a_strTmp,
                        a_fTmp,
                        a_fxUnit
                        );
                }
			}

            if (m_has_measure_x == true)
            {
                // タイトル文字の描画
                switch (m_x_kind)
                {
                    case enumXKind.IsNumeric:
                        a_strTmp = (m_min_x_val + (m_sep_x_val * (a_iCnt - 1))).ToString(m_measure_x_format);
                        //a_fTmp = m_sep_left + (m_sep_x_graph * (a_iCnt - 1)) - (m_sep_string_x * 2);
                        a_fTmp = m_sep_left + ((m_sep_x_val * (a_iCnt - 1)) * m_sep_x_graph) - (m_sep_string_x);
                        break;
                    case enumXKind.IsTimeOrDay:
                        //[2010.02.07]
                        a_strTmp = m_StartTime.AddMinutes((m_min_x_val + (m_sep_x_val * (a_iCnt - 1))) * 10).ToString("HH");
                        //a_strTmp = (m_min_x_val + (m_sep_x_val * (a_iCnt - 1))).ToString(m_measure_x_format);
                        //a_fTmp = m_sep_left + (m_sep_x_graph * (a_iCnt - 1)) + m_sep_string_x;
                        a_fTmp = m_sep_left + ((m_sep_x_val * (a_iCnt - 1)) * m_sep_x_graph) + (m_sep_string_x * 2);
                        break;
                }
                //m_g.DrawString(
                //    a_strTmp,
                //    m_font1,
                //    Brushes.Black,
                //    new PointF(a_fTmp, a_fxUnit)
                //    );
                _drawXMeasureString(
                    a_strTmp,
                    a_fTmp,
                    a_fxUnit
                    );
                switch (m_x_kind)
                {
                    case enumXKind.IsNumeric:
                        a_strTmp = (m_min_x_val + (m_sep_x_val * (a_iCnt))).ToString(m_measure_x_format);
                        //a_fTmp = m_sep_left + (m_sep_x_graph * (a_iCnt)) - (m_sep_string_x * 2);
                        a_fTmp = m_sep_left + ((m_sep_x_val * (a_iCnt)) * m_sep_x_graph) - (m_sep_string_x);
                        //m_g.DrawString(
                        //    a_strTmp,
                        //    m_font1,
                        //    Brushes.Black,
                        //    new PointF(a_fTmp, a_fxUnit)
                        //    );
                        _drawXMeasureString(
                            a_strTmp,
                            a_fTmp,
                            a_fxUnit
                            );
                        break;
                    case enumXKind.IsTimeOrDay:
                        break;
                }
            }

			// 外枠の描画
			m_g.DrawRectangle(
				m_pen1,
				m_sep_left,
				m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val))),
				//m_sep_x_graph * m_num_x_graph,
                (m_sep_x_val * m_num_x_graph) * m_sep_x_graph,
                m_num_y_graph * (m_sep_y_graph * m_sep_y_val)
				);

            //-----------------------------------------------------------------
            // 別目盛線の描画
            //-----------------------------------------------------------------
            float a_moveYUnit = 0;
            if ((m_has_measure_y == true) && (m_measure_y_kind == true))
                _drawYMeasureLine(ref a_moveYUnit);
            
            //-----------------------------------------------------------------
			// 横線の描画
			//-----------------------------------------------------------------
			for (a_iCnt = 1; a_iCnt < m_num_y_graph; a_iCnt++)
			{
				m_pen2.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;
				m_g.DrawLine(
					m_pen2,
					m_sep_left,
					m_Height - (m_sep_bottom + (a_iCnt * (m_sep_y_graph * m_sep_y_val))),
					m_sep_left + ((m_sep_x_val * m_num_x_graph) * m_sep_x_graph),
					m_Height - (m_sep_bottom + (a_iCnt * (m_sep_y_graph * m_sep_y_val)))
					);

                if (m_has_measure_y == true)
                {
                    // Y軸目盛の描画
                    if (m_YReverse == false)
                        a_fyUnit = (m_min_y_val + a_iCnt * m_sep_y_val);
                    else
                        a_fyUnit = (m_max_y_val - a_iCnt * m_sep_y_val);
                    _drawYMeasureString(
                        a_fyUnit.ToString(m_measure_y_format),
                        a_moveYUnit,
                        m_Height - (m_sep_bottom + (a_iCnt * (m_sep_y_graph * m_sep_y_val))) - m_sep_string_y
                        );
                }
			}

            if (m_has_measure_y == true)
            {
                // Y軸目盛の描画
                if (m_YReverse == false)
                    a_fyUnit = (m_min_y_val + a_iCnt * m_sep_y_val);
                else
                    a_fyUnit = (m_max_y_val - a_iCnt * m_sep_y_val);
                _drawYMeasureString(
                    a_fyUnit.ToString(m_measure_y_format),
                    a_moveYUnit,
                    m_Height - (m_sep_bottom + (a_iCnt * (m_sep_y_graph * m_sep_y_val))) - m_sep_string_y
                    );
                if (m_YReverse == false)
                    a_fyUnit = m_min_y_val;
                else
                    a_fyUnit = m_max_y_val;
                _drawYMeasureString(
                    a_fyUnit.ToString(m_measure_y_format),
                    a_moveYUnit,
                    m_Height - m_sep_bottom - m_sep_string_y
                    );
            }

            //[2010.03.07] 外枠の描画
            m_g.DrawRectangle(
                m_pen1,
                m_sep_left,
                m_Height - (m_sep_bottom + (m_num_y_graph * (m_sep_y_graph * m_sep_y_val))),
                //m_sep_x_graph * m_num_x_graph,
                (m_sep_x_val * m_num_x_graph) * m_sep_x_graph,
                m_num_y_graph * (m_sep_y_graph * m_sep_y_val)
                );

        }
        #endregion

        #region 消去　プロット型のランダム選択　GetNextPlotForm
        //*********************************************************************
        // @(f)
        // 機能      : プロット型のランダム選択
        //
        // 引数      : ARG1 - 現在のプロット型
        //
        // 返り値    : 次のプロット型
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /*/// <summary>
        /// ランダムにプロット型を描画した場合等に、次に描画するプロット型を取得します。
        /// </summary>
        /// <param name="nowpf">現在のプロット型</param>
        /// <returns>
        /// 次のプロット型
        /// </returns>
        /// <remarks>
        /// </remarks>
        public enumPlotForm GetNextPlotForm(
            enumPlotForm nowpf
            )
        {
            enumPlotForm a_eRet =enumPlotForm.Nothing;

            switch (nowpf)
            {
                case enumPlotForm.Nothing:
                    a_eRet = enumPlotForm.FillCircle;
                    break;
                case enumPlotForm.Circle:
                    a_eRet = enumPlotForm.Rectangle;
                    break;
                case enumPlotForm.FillCircle:
                    a_eRet = enumPlotForm.FillRectangle;
                    break;
                case enumPlotForm.FillRectangle:
                    a_eRet = enumPlotForm.FillRhombus;
                    break;
                case enumPlotForm.FillRhombus:
                    a_eRet = enumPlotForm.FillTriangle;
                    break;
                case enumPlotForm.FillTriangle:
                    a_eRet = enumPlotForm.Circle;
                    break;
                case enumPlotForm.Rectangle:
                    a_eRet = enumPlotForm.Rhombus;
                    break;
                case enumPlotForm.Rhombus:
                    a_eRet = enumPlotForm.Triangle;
                    break;
                case enumPlotForm.Triangle:
                    a_eRet = enumPlotForm.FillCircle;
                    break;
            }

            return a_eRet;
        }*/
        #endregion

        #region 消去　色のランダム選択　SelectRandamColor
        //*********************************************************************
        // @(f)
        // 機能      : 色のランダム選択
        //
        // 引数      : ARG1 - 基本の原色（1：red、2：green、3：bule）
        //             ARG2 - インデックス
        //
        // 返り値    : RGB
        //
        // 機能説明  :
        //
        // 備考      :
        //
        //*********************************************************************
        /*/// <summary>
        /// ランダムに色を選択して描画したい場合等に、次の色を取得します。
        /// </summary>
        /// <param name="ebc">起点の色(RかGかB）</param>
        /// <param name="idx">インデックス</param>
        /// <returns>
        /// 次の色（R,G,B）
        /// </returns>
        /// <remarks>
        /// </remarks>
        public int[] SelectRandamColor(
            enumBaseColor ebc,
            int idx
            )
        {
            // 一時的な実現コード
            int a_plus = 51;   // Web Safe Colorは51ずつずれる？
            int[] a_RGB_R = { 255, 0, 0 };
            int[] a_RGB_G = { 0, 255, 0 };
            int[] a_RGB_B = { 0, 0, 255 };
            int[] a_rgbT = null;

            //-------------------------------------------------------------
            // 色の選択　→　共通化？
            //-------------------------------------------------------------
            switch (ebc)
            {
                case enumBaseColor.red:
                    a_RGB_R[2] += (a_plus * idx);
                    if (a_RGB_R[2] > 255)
                        a_RGB_R[2] = 0;
                    a_rgbT = a_RGB_R;
                    break;
                case enumBaseColor.green:
                    a_RGB_G[0] += (a_plus * idx);
                    if (a_RGB_G[0] > 255)
                        a_RGB_G[0] = 0;
                    a_rgbT = a_RGB_G;
                    break;
                case enumBaseColor.blue:
                    a_RGB_G[1] += (a_plus * idx);
                    if (a_RGB_G[1] > 255)
                        a_RGB_G[1] = 0;
                    a_rgbT = a_RGB_B;
                    break;
            }

            //Random a_random = new Random((int)ebc * idx);
            //a_rgbT = new int[3];
            //a_rgbT[0] = a_random.Next(256);
            //a_rgbT[1] = a_random.Next(256);
            //a_rgbT[2] = a_random.Next(256);
            

            int[,] a_webSafeColor = {
                {0,0,0},{51,51,51},{102,102,102},
                {153,153,153},{204,204,204},{255,255,255},
                {255,255,204},{255,255,153},{255,25,102},
                {255,255,51},{255,255,0},{204,204,0},
                {255,20,102},{255,204,0},{255,204,51},
                {204,153,0},{204,153,51},{153,102,0},
                {255,153,0},{255,153,53},{204,153,102},
                {204,102,0},{153,102,51},{102,51,0},
                {255,204,153},{255,153,102},{255,102,0},
                {204,102,51},{153,51,0},{153,0,0},
                {255,102,51},{204,51,0},{255,51,0},
                {255,0,0},{204,0,0},{102,0,0},
                {255,204,204},{204,153,153},{255,153,153},
                {204,102,102},{255,102,102},{255,51,51},
                {255,0,51},{204,0,51},{204,51,51},
                {153,51,51},{153,0,51},{51,0,0},
                {255,102,153},{255,51,102},{255,0,102},
                {204,51,102},{153,102,102},{102,51,51},
                {255,153,204},{255,51,153},{255,0,153},
                {204,0,102},{153,51,102},{102,0,51},
                {255,102,204},{255,0,204},{255,51,204},
                {204,102,153},{204,0,153},{153,0,102},
                {255,204,255},{255,153,255},{255,102,255},
                {255,51,255},{255,0,255},{204,51,153},
                {204,153,204},{204,102,204},{204,0,204},
                {204,51,204},{153,0,153},{153,51,153},
                {204,102,255},{204,51,255},{204,0,255},
                {153,0,204},{153,102,153},{102,0,102},
                {204,153,255},{153,51,255},{153,0,255},
                {153,51,204},{102,0,153},{102,51,102},
                {153,102,204},{153,102,255},{102,0,204},
                {102,51,204},{102,51,153},{51,0,51},
                {204,204,255},{153,153,255},{102,51,255},
                {102,0,255},{51,0,153},{51,0,102},
                {153,102,204},{153,102,255},{102,0,204},
                {51,51,255},{51,0,255},{51,0,204},
                {51,51,204},{0,0,204},{0,0,102},
                {102,153,255},{102,102,255},{0,0,255},
                {0,0,204},{0,51,204},{0,0,51},
                {0,102,255},{0,102,204},{51,102,204},
                {0,51,255},{0,51,153},{0,51,102},
                {153,153,204},{102,153,204},{51,153,255},
                {0,153,255},{51,102,153},{0,102,153},
                {102,204,255},{51,204,255},{0,204,255},
                {51,153,204},{0,153,204},{153,204,204},
                {204,255,255},{102,204,204},{102,153,153},
                {51,153,153},{0,102,102},{51,102,102},
                {153,255,255},{102,255,255},{51,255,255},
                {0,255,255},{0,204,204},{153,255,204},
                {102,255,204},{51,255,204},{0,255,204},
                {51,204,204},{0,153,153},{102,204,153},
                {51,204,153},{0,204,153},{0,153,102},
                {51,153,102},{0,102,51},{0,51,51},
                {102,255,153},{51,255,153},{0,255,153},
                {51,204,102},{0,204,102},{0,153,51},
                {153,255,153},{102,255,102},{51,255,102},
                {0,255,102},{51,153,51},{0,102,0},
                {204,255,204},{153,204,153},{102,153,102},
                {102,204,102},{51,102,51},{0,51,0},
                {51,255,51},{0,255,51},{0,255,0},
                {0,204,0},{51,204,51},{0,204,0},
                {102,255,0},{102,255,51},{51,255,0},
                {51,204,0},{51,153,0},{0,153,0},
                {204,255,153},{153,255,102},{102,204,0},
                {102,204,51},{102,153,51},{51,102,0},
                {153,255,0},{153,255,51},{153,204,102},
                {153,204,0},{153,204,51},{102,153,0},
                {204,255,102},{204,255,0},{204,255,51},
                {204,204,153},{102,102,51},{51,51,0},
                {204,204,102},{204,204,51},{153,153,102},
                {153,153,51},{153,153,0},{102,102,0}
            };

            int a_colorSum = a_webSafeColor.Length;

            int a_index_start_red = 12;
            int a_index_base_red = 42;  //33;
            int a_index_end_red = 77;
            int a_plus_red = 2;

            int a_index_start_blue = 78;
            int a_index_base_blue = 113; // 113;
            int a_index_end_blue = 140;
            int a_plus_blue = 4;

            int a_index_start_green = 141;
            int a_index_base_green = 180;   // 179;
            int a_index_end_green = 212;
            int a_plus_green = 5;

            int a_iTmp1 = 0;
            int a_iTmp2 = 0;

            switch (ebc)
            {
                case enumBaseColor.red:
                    a_iTmp1 = (a_index_end_red - a_index_base_red) / a_plus_red;
                    if (idx <= a_iTmp1)
                        a_iTmp2 = a_index_base_red + (idx * a_plus_red);
                    else
                        a_iTmp2 = a_index_start_red + ((idx - a_iTmp1 - 1) * a_plus_red);
                    break;
                case enumBaseColor.green:
                    a_iTmp1 = (a_index_end_green - a_index_base_green) / a_plus_green;
                    if (idx <= a_iTmp1)
                        a_iTmp2 = a_index_base_green + (idx * a_plus_green);
                    else
                        a_iTmp2 = a_index_start_green + ((idx - a_iTmp1 - 1) * a_plus_green);
                    break;
                case enumBaseColor.blue:
                    a_iTmp1 = (a_index_end_blue - a_index_base_blue) / a_plus_blue;
                    if (idx <= a_iTmp1)
                        a_iTmp2 = a_index_base_blue + (idx * a_plus_blue);
                    else
                        a_iTmp2 = a_index_start_blue + ((idx - a_iTmp1 - 1) * a_plus_blue);
                    break;
            }

            int[] a_rgbT = new int[3];
            a_rgbT[0] = a_webSafeColor[a_iTmp2, 0];
            a_rgbT[1] = a_webSafeColor[a_iTmp2, 1];
            a_rgbT[2] = a_webSafeColor[a_iTmp2, 2];

            return a_rgbT;
        }*/
        #endregion
    }
}
