﻿/// <summary>
/// 共通ユーティリティ
/// </summary>
/// <create>2010/01/27</create>
/// <modifier></modifier>
/// <modify></modify>
/// <modification></modification>
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;

namespace lsor
{
    /// <summary><para>class outline:</para>
    /// <para>共通ユーティリティクラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[ver 1.0.0][2010/02/02][新規作成]</para>
    /// </remarks>
    class Util
    {
        #region 定数定義

        /// <summary>ヘッダ列数</summary>
        private const int HEADER_COLUMNS = 4;

        #region グラフ表示設定

        /// <summary>スケール</summary>
        public enum SCALE
        {
            None = 0,
            Auto,
            LeftY,
            RightY
        };

        /// <summary>線種</summary>
        public enum LINESTYLE
        {
            None = 0,
            Solid,
            Dot,
            Dash,
            DashDot,
            DashDotDot
        };

        /// <summary>太さ</summary>
        public enum LINEWIDTH
        {
            None = 0,
            Thin,
            Middle,
            Thick
        };

        /// <summary>シンボル形状</summary>
        public enum SYMBOLSTYLE
        {
            None = 0,
            Cross,
            Cross2,
            Circle,
            CircleF,
            Triangle,
            TriangleF,
            Triangle2,
            Triangle2F,
            Square,
            SquareF,
            Square2,
            Square2F
        };

        /// <summary>シンボルサイズ</summary>
        public enum SYMBOLSIZE
        {
            None = 0,
            Small,
            Medium,
            Large
        };

        #endregion

        #endregion

        #region パブリックメソッド

        #region ファイルインデックス取得

        /// <summary><para>method outline:</para>
        /// <para>ファイルインデックスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iFileIndex = GetFileIndex(inputFileList, fileID)</para>
        /// </example>
        /// <param name="inputFileList">ファイルリスト</param>
        /// <param name="fileId">ファイルID</param>
        /// <returns>インデックス番号</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>"結果出力2"などの文字列を受け取り、該当するリストインデックスを返却する</para>
        /// </remarks>
        public static int GetFileIndex(List<InputFile> inputFileList, string fileId)
        {

            for (int i = 0; i < inputFileList.Count; i++)
            {
                if (fileId == inputFileList[i].FileId)
                {
                    return i;
                }
            }

            return -1;
        }

        #endregion

        #region 項目インデックス取得

        /// <summary><para>method outline:</para>
        /// <para>項目インデックスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iColumnIndex = GetColumnIndex(inputFile, columnNm)</para>
        /// </example>
        /// <param name="inputFile">入力ファイル</param>
        /// <param name="columnNm">項目名</param>
        /// <returns>項目リストインデックス番号</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>項目名を受け取り、該当する項目リストインデックス番号を返却する</para>
        /// </remarks>
        public static int GetColumnIndex(InputFile inputFile, string columnNm)
        {

            for (int i = 0; i < inputFile.ColumnList.Count; i++)
            {
                if (String.Compare(columnNm, inputFile.ColumnList[i].ColumnNm, true) == 0)
                {
                    return i;
                }
            }

            return -1;
        }

        #endregion

        #region ファイルIDの取得

        /// <summary><para>method outline:</para>
        /// <para>ファイルIDを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string strFileId = ConvertFileId(commonData, fileIdx)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileIdx">ファイルIDインデックス</param>
        /// <returns>ファイルID</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>0, 1, 2, ... を渡すと "結果入力", "結果出力1", "結果出力2", ... を返却する</para>
        /// </remarks>
        public static string ConvertFileId(CommonData commonData, int fileIdx)
        {
            string fileId = "";

            try
            {
                fileId = commonData.FileList[fileIdx].FileId;
            }
            catch
            {
            }
            return fileId;

        }

        #endregion

        #region ファイルIDインデックスの取得

        /// <summary><para>method outline:</para>
        /// <para>ファイルIDインデックスを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string strFileId = ConvertFileId(commonData, fileId)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileId">ファイルID</param>
        /// <returns>ファイルIDインデックス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>"結果入力", "結果出力1", "結果出力2", ... を渡すと 0, 1, 2, ... を返却する</para>
        /// </remarks>
        public static int ConvertFileId(CommonData commonData, string fileId)
        {
            int fileIdx = -1;

            for (int i = 0; i < commonData.FileList.Count; i++)
            {
                if (fileId == commonData.FileList[i].FileId)
                {
                    fileIdx = i;
                }
            }

            return fileIdx;

        }

        #endregion

        #region ファイル名の取得

        /// <summary><para>method outline:</para>
        /// <para>ファイル名を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string strFileName = GetFileName(commonData, fileId)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileIdx">ファイルIDインデックス</param>
        /// <returns>ファイル名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>0, 1, 2, ... を渡すと該当するファイル名を返却する</para>
        /// </remarks>
        public static string GetFileName(CommonData commonData, int fileIdx)
        {
            string fileNm = "";

            try
            {
                fileNm = commonData.FileList[fileIdx].FileNm;
            }
            catch
            {
            }
            return fileNm;
        }

        #endregion

        #region ファイル名の取得2

        /// <summary><para>method outline:</para>
        /// <para>ファイル名を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string strFileName = GetFileName(commonData, fileId)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileId">ファイルID</param>
        /// <returns>ファイル名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>"結果入力", "結果出力1", "結果出力2", ... を渡すと該当するファイル名を返却する</para>
        /// </remarks>
        public static string GetFileName(CommonData commonData, string fileId)
        {
            return (GetFileName(commonData, ConvertFileId(commonData, fileId)));
        }

        #endregion

        #region データファイルの存在チェック

        /// <summary><para>method outline:</para>
        /// <para>データファイルの存在チェックをする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = IsExistFileId(commonData, fileId)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileId">ファイルID</param>
        /// <returns>Boolean true:存在する、false:存在しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>0, 1, 2, ... を渡すと該当するデータファイルがあるかどうかをチェックする</para>
        /// </remarks>
        public static Boolean IsExistFileId(CommonData commonData, int fileId)
        {
            if (fileId < 0 || fileId > CommonData.MAX_FILE_ID)
            {
                return (false);
            }
            else if (GetFileName(commonData, fileId) == null || GetFileName(commonData, fileId).Length <= 0)
            {
                return (false);
            }
            else
            {
                return (true);
            }
        }

        #endregion

        #region データファイルの存在チェック2

        /// <summary><para>method outline:</para>
        /// <para>データファイルの存在チェックをする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = IsExistFileId(commonData, fileId)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileId">ファイルID</param>
        /// <returns>Boolean true:存在する、false:存在しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>"結果入力", "結果出力1", "結果出力2", ... を渡すと該当するデータファイルがあるかどうかをチェックする</para>
        /// </remarks>
        public static Boolean IsExistFileId(CommonData commonData, string fileId)
        {
            return (IsExistFileId(commonData, ConvertFileId(commonData, fileId)));
        }

        #endregion

        #region 項目名の存在チェック

        /// <summary><para>method outline:</para>
        /// <para>項目名の存在チェックをする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>Boolean bRtn = IsExistItem(commonData, fileId, itemName)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileId">ファイルID</param>
        /// <param name="itemName">項目名</param>
        /// <returns>Boolean true:存在する、false:存在しない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定したファイルIDと項目名のペアが存在するかをチェックする</para>
        /// </remarks>
        public static Boolean IsExistItem(CommonData commonData, string fileId, string itemName)
        {
            try
            {
                for (int idx = 0; idx < commonData.FileList.Count; idx++)
                {
                    if (commonData.FileList[idx].FileId.Equals(fileId))
                    {
                        for (int col = 0; col < commonData.FileList[idx].ColumnList.Count; col++)
                        {
                            if (String.Compare(commonData.FileList[idx].ColumnList[col].ColumnNm, itemName, true) == 0)
                            {
                                return (true);
                            }
                        }
                        return (false);
                    }
                }
            }
            catch
            {
            }

            return (false);
        }

        #endregion

        #region 河川名リストの取得

        /// <summary><para>method outline:</para>
        /// <para>河川名リストを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string[] strRiverNames = GetRiverNameList(commonData)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <returns>河川名リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>河川名の string[] を返却する</para>
        /// </remarks>
        public static string[] GetRiverNameList(CommonData commonData)
        {
            string[] river = new string[commonData.RiverList.Count];
            if (commonData.RiverList.Count == 0)
            {
                return null;   // 1個もない場合
            }
            for (int i = 0; i < commonData.RiverList.Count; i++)
            {
                river[i] = commonData.RiverList[i].RiverNm;
            }
            return river;
        }

        #endregion

        #region ファイルIDリストの取得

        /// <summary><para>method outline:</para>
        /// <para>ファイルIDリストを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string[] strFileIds = GetFileIdList(commonData)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <returns>ファイルIDリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>有効なファイルIDの string[] を返却する</para>
        /// </remarks>
        public static string[] GetFileIdList(CommonData commonData)
        {
            int cnt = 0;

            // 件数取得
            for (int i = 0; i < commonData.FileList.Count; i++)
            {
                if (commonData.FileList[i].FileNm != null && commonData.FileList[i].FileNm != "")
                {
                    cnt++;
                }
            }

            if (cnt == 0)
            {
                return null;
            }

            string[] fileId = new string[cnt];

            int j = 0;

            for (int i = 0; i < commonData.FileList.Count; i++)
            {
                if (commonData.FileList[i].FileNm != null && commonData.FileList[i].FileNm != "")
                {
                    fileId[j] = commonData.FileList[i].FileId;
                    j++;
                }
            }

            return fileId;
        }

        #endregion

        #region 項目名リストの取得

        /// <summary><para>method outline:</para>
        /// <para>項目名リストを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string[] strItemNames = GetItemNameList(commonData, fileId)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileId">ファイルID</param>
        /// <returns>項目名リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ファイルIDを渡すと項目名の string[] を返却する</para>
        /// </remarks>
        public static string[] GetItemNameList(CommonData commonData, string fileId)
        {
            return (GetItemNameList(commonData, ConvertFileId(commonData, fileId)));
        }

        #endregion

        #region 累加距離リストの取得

        /// <summary><para>method outline:</para>
        /// <para>累加距離リストを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double?[] dDistanceData = GetDistanceData(commonData, river)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="river">河川名</param>
        /// <returns>累加距離リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>河川名を渡すと累加距離の配列を返却する</para>
        /// </remarks>
        public static double?[] GetDistanceData(CommonData commonData, string river)
        {
            string[] str = GetData(commonData, river, ConvertFileId(commonData, 0), CommonData.KEY_SUM_DISTANCE);
            double?[] data = new double?[str.Length];
            for (int i = 0; i < str.Length; i++)
            {
                try
                {
                    data[i] = Double.Parse(str[i]);
                }
                catch
                {
                    data[i] = null;
                }
            }

            return (data);
        }

        #endregion

        #region 断面Noリストの取得

        /// <summary><para>method outline:</para>
        /// <para>断面Noリストを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string[] strSectionNos = GetSectionNo(commonData, river)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="river">河川名</param>
        /// <returns>断面Noリスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>河川名を渡すと断面Noの配列を返却する</para>
        /// </remarks>
        public static string[] GetSectionNo(CommonData commonData, string river)
        {
            return (GetData(commonData, river, ConvertFileId(commonData, 0), CommonData.KEY_SLICE_NO));
        }

        #endregion

        #region 項目データの取得(double?[]形式)

        /// <summary><para>method outline:</para>
        /// <para>項目データを取得する(double?[]形式)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>double?[] dCsvDatas = GetCsvData(commonData, river, fileId, itemName)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="river">河川名</param>
        /// <param name="fileId">ファイルID</param>
        /// <param name="itemName">項目名</param>
        /// <returns>項目データ(double?[]形式)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>河川名,ファイルID,項目名を渡すと値の配列を返却する</para>
        /// </remarks>
        public static double?[] GetCsvData(CommonData commonData, string river, string fileId, string itemName)
        {
            string[] csv = GetData(commonData, river, fileId, itemName);
            double?[] data = null;
            if (csv != null && csv.Length > 0)
            {
                data = new double?[csv.Length];
                for (int n = 0; n < csv.Length; n++)
                {
                    try
                    {
                        data[n] = Double.Parse(csv[n]);
                    }
                    catch
                    {
                        data[n] = null;
                    }
                }
            }

            return (data);
        }

        #endregion

        #region 項目データの取得(string[]形式)

        /// <summary><para>method outline:</para>
        /// <para>項目データを取得する(string[]形式)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string[] strDatas = GetCsvData(commonData, river, fileId, itemName)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="river">河川名</param>
        /// <param name="fileId">ファイルID</param>
        /// <param name="itemName">項目名</param>
        /// <returns>項目データ(string[]形式)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>河川名,ファイルID,項目名を渡すと値の配列を返却する(累加距離でソート)</para>
        /// </remarks>
        public static string[] GetData(CommonData commonData, string river, string fileId, string itemName)
        {
            string[] data = null;
            SortedList<double, string> list = new SortedList<double, string>();

            try
            {
                for (int idx = 0; idx < commonData.FileList.Count; idx++)
                {
                    if (commonData.FileList[idx].FileId.Equals(fileId))
                    {
                        int col_distance = Util.GetColumnIndex(commonData.FileList[idx], CommonData.KEY_SUM_DISTANCE);

                        for (int col = 0; col < commonData.FileList[idx].ColumnList.Count; col++)
                        {
                            if (String.Compare(commonData.FileList[idx].ColumnList[col].ColumnNm, itemName, true) == 0)
                            {
                                for (int row = 0; row < commonData.FileList[idx].CsvData.Count; row++)
                                {
                                    if (((string[])commonData.FileList[idx].CsvData[row])[0].Equals(river))
                                    {
                                        double distance = 0.0;
                                        try
                                        {
                                            distance = Double.Parse(((string[])commonData.FileList[idx].CsvData[row])[col_distance]);
                                        }
                                        catch
                                        {
                                            distance = 0.0;
                                        }

                                        string val = ((string[])commonData.FileList[idx].CsvData[row])[col];
                                        list.Add(distance, val);    // 累加距離が重複していると、ArgumentExceptionが発生
                                    }
                                }
                                if (list.Count > 0)
                                {
                                    data = new string[list.Count];
                                    int r = 0;
                                    foreach (string s in list.Values)
                                    {
                                        data[r++] = s;
                                    }
                                }
                                break;
                            }
                        }
                        break;
                    }
                }
            }
            catch (ArgumentException)
            {
                // 累加距離の重複による例外を捕捉
                // メッセージ:区間距離が0mになっているデータが存在します。
                throw new Exception(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_SECTION_LENGTH_ZERO);
            }
            catch
            {
            }

            return (data);
        }

        /// <summary><para>method outline:</para>
        /// <para>項目データを取得する(string[]形式)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string[] strDatas = GetCsvData(commonData, fileId, itemName)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileId">ファイルID</param>
        /// <param name="itemName">項目名</param>
        /// <returns>項目データ(string[]形式)</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>ファイルID,項目名を渡すと値の配列を返却する(結果入力ファイルと同順)</para>
        /// </remarks>
        public static string[] GetData(CommonData commonData, string fileId, string itemName)
        {
            string[] data = null;

            try
            {
                Hashtable hash = new Hashtable();
                int idx = GetFileIndex(commonData.FileList, fileId);
                int col = GetColumnIndex(commonData.FileList[idx], itemName);
                for (int row = 0; row < commonData.FileList[idx].CsvData.Count; row++)
                {
                    string key =
                        ((string[])commonData.FileList[idx].CsvData[row])[0] + "," +
                        ((string[])commonData.FileList[idx].CsvData[row])[1] + "," +
                        ((string[])commonData.FileList[idx].CsvData[row])[2] + "," +
                        ((string[])commonData.FileList[idx].CsvData[row])[3];
                    string val = ((string[])commonData.FileList[idx].CsvData[row])[col];
                    hash.Add(key, val);
                }

                data = new string[commonData.FileList[0].CsvData.Count];
                for (int row = 0; row < commonData.FileList[0].CsvData.Count; row++)
                {
                    string key =
                        ((string[])commonData.FileList[0].CsvData[row])[0] + "," +
                        ((string[])commonData.FileList[0].CsvData[row])[1] + "," +
                        ((string[])commonData.FileList[0].CsvData[row])[2] + "," +
                        ((string[])commonData.FileList[0].CsvData[row])[3];
                    try
                    {
                        data[row] = (string)hash[key];
                    }
                    catch
                    {
                        data[row] = null;
                    }
                }
            }
            catch
            {
            }

            return (data);
        }

        #endregion

        #region 項目名の取得

        /// <summary><para>method outline:</para>
        /// <para>項目名を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string[] strDatas = GetItemNameList(commonData, fileIdx)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="fileIdx">ファイルIDインデックス</param>
        /// <returns>項目名</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>0, 1, 2, ... を渡すと項目名の string[] を返却する</para>
        /// </remarks>
        public static string[] GetItemNameList(CommonData commonData, int fileIdx)
        {
            // 件数を取得
            int cnt = commonData.FileList[fileIdx].ColumnList.Count - HEADER_COLUMNS;
            if (cnt <= 0)
            {
                return null;
            }

            int idx = 0;
            // 先頭の4項目を含めるかどうかだけど、とりあえず無しの方向で
            string[] itemName = new string[cnt];
            for (int i = HEADER_COLUMNS; i < commonData.FileList[fileIdx].ColumnList.Count; i++)
            {
                itemName[idx] = commonData.FileList[fileIdx].ColumnList[i].ColumnNm;
                idx++;

            }
            return itemName;
        }

        #endregion

        #region 河川データの取得

        /// <summary><para>method outline:</para>
        /// <para>河川データを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>RiverData csRiverData = GetRiverData(commonData, riverName)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <param name="riverName">河川名</param>
        /// <returns>河川データ</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>河川名を渡すと該当するRiverDataを返却する</para>
        /// </remarks>
        public static RiverData GetRiverData(CommonData commonData, string riverName)
        {
            if (commonData != null && commonData.RiverList != null)
            {
                foreach (RiverData river in commonData.RiverList)
                {
                    if (river.RiverNm != null && river.RiverNm.Equals(riverName))
                    {
                        return (river);
                    }
                }
            }
            return (null);
        }

        #endregion

        #region null->0変換

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトをintに変換する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iObj = NullToZero(obj)</para>
        /// </example>
        /// <param name="obj">変換元オブジェクト</param>
        /// <returns>intに変換したオブジェクト。nullの場合は0</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたオブジェクトをintに変換。nullの場合は0を返却する</para>
        /// </remarks>
        public static int NullToZero(object obj)
        {
            try
            {
                return (int.Parse(obj.ToString()));
            }
            catch
            {
                return (0);
            }
        }

        #endregion

        #region null->空文字変換

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトをstringに変換する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string strObj = NullToBlank(obj)</para>
        /// </example>
        /// <param name="obj">変換元オブジェクト</param>
        /// <returns>stringに変換したオブジェクト。nullの場合は空文字</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたオブジェクトをstringに変換。nullの場合は空文字を返却する</para>
        /// </remarks>
        public static string NullToBlank(object obj)
        {
            if (obj == null)
            {
                return ("");
            }
            else
            {
                return (obj.ToString());
            }
        }

        #endregion

        #region Boolチェック

        /// <summary><para>method outline:</para>
        /// <para>オブジェクトのBoolチェックをする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsTrue(obj)</para>
        /// </example>
        /// <param name="obj">変換元オブジェクト</param>
        /// <returns>bool true:正常、false:異常</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたオブジェクトがtrueかどうかをチェックする</para>
        /// </remarks>
        public static bool IsTrue(object obj)
        {
            if (NullToZero(obj) == 0)
            {
                return (false);
            }
            else
            {
                return (true);
            }
        }

        #endregion

        #region 文字列のブランクチェック

        /// <summary><para>method outline:</para>
        /// <para>文字列のブランクチェックをする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsBlank(s)</para>
        /// </example>
        /// <param name="s">変換元文字列</param>
        /// <returns>bool true:文字列、false:nullか空文字</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたstringがnullか空文字かどうかをチェックする</para>
        /// </remarks>
        public static bool IsBlank(string s)
        {
            if (s == null || s.Trim().Length <= 0)
            {
                return (true);
            }
            else
            {
                return (false);
            }
        }

        #endregion

        #region 文字列のdouble変換

        /// <summary><para>method outline:</para>
        /// <para>文字列をdoubleに変換する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string str = ToDoubleString(s)</para>
        /// </example>
        /// <param name="s">変換元文字列</param>
        /// <returns>変換後文字列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたstringがdoubleを表す場合はそのまま返却する。そうでない場合は空文字を返却する</para>
        /// </remarks>
        public static string ToDoubleString(string s)
        {
            if (s == null || s.Length <= 0)
            {
                return ("");
            }
            try
            {
                double d = double.Parse(s);
                return (d.ToString());
            }
            catch
            {
                return ("");
            }
        }

        #endregion

        #region 文字列のdoubleチェック

        /// <summary><para>method outline:</para>
        /// <para>文字列がdoubleを表す文字かチェックする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsDouble(s)</para>
        /// </example>
        /// <param name="s">文字列</param>
        /// <returns>bool true:doubleを表す、false:doubleを表さない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたstringがdoubleを表す文字かどうかをチェックする</para>
        /// </remarks>
        public static bool IsDouble(string s)
        {
            try
            {
                double d = double.Parse(s);
                return (true);
            }
            catch
            {
                return (false);
            }
        }

        #endregion

        #region 文字列のintegerチェック

        /// <summary><para>method outline:</para>
        /// <para>文字列がintを表す文字かチェックする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = IsInteger(s)</para>
        /// </example>
        /// <param name="s">文字列</param>
        /// <returns>bool true:intを表す、false:intを表さない</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>指定されたstringがintを表す文字かどうかをチェックする</para>
        /// </remarks>
        public static bool IsInteger(string s)
        {
            try
            {
                int i = int.Parse(s);
                return (true);
            }
            catch
            {
                return (false);
            }
        }

        #endregion

        #endregion
    }
}
