﻿/// <summary>
/// データ系列画面
/// </summary>
/// <create>2010/01/26</create>
/// <modifier></modifier>
/// <modify></modify>
/// <modification></modification>
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace lsor
{
    /// <summary><para>class outline:</para>
    /// <para>データ系列画面出力制御クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[ver 1.0.0][2010/02/02][新規作成]</para>
    /// </remarks>
    public partial class GraphProp : Form
    {
        #region プロパティ

        /// <summary><para>method outline:</para>
        /// <para>現在選択中の河川名を設定・取得する</para>
        /// </summary>
        /// <value>現在選択中の河川名</value>
        /// <example><para>usage:</para>
        /// <para>string strRiver = dlgGraphProp.CurrentRiver()</para>
        /// </example>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string CurrentRiver { get; set; }

        #endregion

        #region プライベート変数

        /// <summary>データ系列の最大数</summary>
        private const int MAX_ROWS = 10;

        /// <summary>プロジェクト共通データ</summary>
        private CommonData _commonData;

        /// <summary>データ系列設定のコンボボックスの情報</summary>
        private Dictionary<string, Dictionary<int, string>> dicProp;

        #endregion

        #region コンストラクタ

        /// <summary><para>method outline:</para>
        /// <para>標準のコンストラクタ。未使用</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GraphProp csGraphProp = new GraphProp()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>データ系列画面出力制御クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public GraphProp()
        {
            InitializeComponent();
        }

        /// <summary><para>method outline:</para>
        /// <para>データ系列追加ダイアログのコンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>GraphProp csGraphProp = new GraphProp(commonData)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <returns>データ系列画面出力制御クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public GraphProp(CommonData commonData)
        {
            _commonData = commonData;
            if (_commonData == null)
            {
                _commonData = new CommonData();
            }

            InitializeComponent();
        }

        #endregion

        #region プライベートメソッド

        #region コマンドボタンの使用可不可を更新

        /// <summary><para>method outline:</para>
        /// <para>コマンドボタンの使用可不可を更新する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>UpdateMenu()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void UpdateMenu()
        {
            // 追加ボタン --- 10行になるまでは使用可
            cmdAdd.Enabled = (Util.IsBlank(CurrentRiver) || dgvProp.Rows.Count >= MAX_ROWS) ? false : true;
            // 削除ボタン --- 0行になるまでは使用可
            cmdDel.Enabled = (Util.IsBlank(CurrentRiver) || dgvProp.Rows.Count <= 0) ? false : true;
            // 上へボタン --- 2行以上で使用可
            cmdUp.Enabled = (Util.IsBlank(CurrentRiver) || dgvProp.Rows.Count <= 1) ? false : true;
            // 下へボタン --- 2行以上で使用可
            cmdDown.Enabled = (Util.IsBlank(CurrentRiver) || dgvProp.Rows.Count <= 1) ? false : true;
        }

        #endregion

        #region コンボボックスの項目設定

        /// <summary><para>method outline:</para>
        /// <para>DataGridViewのコンボボックス(スケールとか線種とか)に選択肢を追加する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>AddCombo(key, list)</para>
        /// </example>
        /// <param name="key">dgvのカラム名</param>
        /// <param name="list">選択肢リスト</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void AddCombo(string key, Dictionary<int, string> list)
        {
            if (dicProp == null)
            {
                dicProp = new Dictionary<string, Dictionary<int, string>>();
            }

            DataGridViewComboBoxColumn cbo = (DataGridViewComboBoxColumn)dgvProp.Columns[key];
            if (list != null && list.Count > 0)
            {
                Dictionary<int, string> clone = new Dictionary<int, string>();
                foreach (int k in list.Keys)
                {
                    clone[k] = list[k];
                }
                dicProp.Add(key, clone);

                foreach (string v in list.Values)
                {
                    cbo.Items.Add(v);
                }
            }
        }

        #endregion

        #region 色選択セルの更新

        /// <summary><para>method outline:</para>
        /// <para>色選択セルの更新をする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setColorCell(columnName, row, color)</para>
        /// </example>
        /// <param name="columnName">カラム名</param>
        /// <param name="row">行番号</param>
        /// <param name="color">Color構造体</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setColorCell(string columnName, int row, Color color)
        {
            int col = dgvProp.Columns[columnName].Index;
            if (col >= 0)
            {
                setColorCell(col, row, color);
            }
        }
        /// <summary><para>method outline:</para>
        /// <para>色選択セルの更新をする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setColorCell(columnName, row, color)</para>
        /// </example>
        /// <param name="columnName">カラム名</param>
        /// <param name="row">行番号</param>
        /// <param name="color">色のバイナリ表記</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setColorCell(string columnName, int row, int color)
        {
            setColorCell(columnName, row, System.Drawing.Color.FromArgb(color));
        }
        /// <summary><para>method outline:</para>
        /// <para>色選択セルの更新をする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setColorCell(col, row, color)</para>
        /// </example>
        /// <param name="col">カラム番号</param>
        /// <param name="row">行番号</param>
        /// <param name="color">色のバイナリ表記</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setColorCell(int col, int row, int color)
        {
            setColorCell(col, row, System.Drawing.Color.FromArgb(color));
        }
        /// <summary><para>method outline:</para>
        /// <para>色選択セルの更新をする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setColorCell(col, row, color)</para>
        /// </example>
        /// <param name="col">カラム番号</param>
        /// <param name="row">行番号</param>
        /// <param name="color">Color構造体</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setColorCell(int col, int row, Color color)
        {
            dgvProp[col, row].Style.ForeColor = color;
            dgvProp[col, row].Style.SelectionForeColor = color;
            dgvProp[col, row].Value = "Color";
        }

        #endregion

        #region コンボボックスセルの更新

        /// <summary><para>method outline:</para>
        /// <para>コンボボックスセルの更新をする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setComboCell(columnName, row, value)</para>
        /// </example>
        /// <param name="columnName">カラム名</param>
        /// <param name="row">行番号</param>
        /// <param name="value">コンボボックス更新値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setComboCell(string columnName, int row, int value)
        {
            int col = dgvProp.Columns[columnName].Index;
            if (col >= 0)
            {
                dgvProp[col, row].Value = GetComboText(columnName, value);
            }
        }

        #endregion

        #region チェックボックスセルの更新

        /// <summary><para>method outline:</para>
        /// <para>チェックボックスセルの更新をする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setChkBoxCell(columnName, row, chk)</para>
        /// </example>
        /// <param name="columnName">カラム名</param>
        /// <param name="row">行番号</param>
        /// <param name="chk">チェックボックス更新値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setChkBoxCell(string columnName, int row, bool chk)
        {
            int col = dgvProp.Columns[columnName].Index;
            if (col >= 0)
            {
                dgvProp[col, row].Value = chk ? "1" : "0";
            }
        }

        #endregion

        #region テキストボックスセルの更新

        /// <summary><para>method outline:</para>
        /// <para>テキストボックスセルの更新をする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setTextBoxCell(columnName, row, str)</para>
        /// </example>
        /// <param name="columnName">カラム名</param>
        /// <param name="row">行番号</param>
        /// <param name="str">テキストボックスセル更新値</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setTextBoxCell(string columnName, int row, string str)
        {
            int col = dgvProp.Columns[columnName].Index;
            if (col >= 0)
            {
                dgvProp[col, row].Value = Util.NullToBlank(str);
            }
        }

        #endregion

        #region 行の追加

        /// <summary><para>method outline:</para>
        /// <para>DataGridViewの末尾に行を追加する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>AddRow(fileId, itemName)</para>
        /// </example>
        /// <param name="fileId">ファイルID</param>
        /// <param name="itemName">項目名</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void AddRow(string fileId, string itemName)
        {
            int row = dgvProp.Rows.Add();

            setTextBoxCell("colFileID", row, fileId);
            setTextBoxCell("colFileName", row, Util.GetFileName(_commonData, fileId));
            setTextBoxCell("colItemName", row, itemName);
            setComboCell("colScale", row, (int)Util.SCALE.Auto);
            setComboCell("colLineStyle", row, (int)Util.LINESTYLE.Solid);
            setColorCell("colLineColor", row, Color.Black);
            setComboCell("colLineWidth", row, (int)Util.LINEWIDTH.Thin);
            setComboCell("colSymbolStyle", row, (int)Util.SYMBOLSTYLE.Cross);
            setColorCell("colSymbolColor", row, Color.Black);
            setComboCell("colSymbolSize", row, (int)Util.SYMBOLSIZE.Small);
        }

        #endregion

        #region コンボボックスの表示文字列取得

        /// <summary><para>method outline:</para>
        /// <para>コンボボックスから表記文字列を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string strComboText = GetComboText(key, v)</para>
        /// </example>
        /// <param name="key">dgvカラム名</param>
        /// <param name="v">設定値</param>
        /// <returns>表記文字列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private string GetComboText(string key, int v)
        {
            if (key == null || !dicProp.ContainsKey(key) || !dicProp[key].ContainsKey(v))
            {
                return (null);
            }
            else
            {
                return (dicProp[key][v]);
            }
        }

        #endregion

        #region コンボボックスの設定値取得

        /// <summary><para>method outline:</para>
        /// <para>コンボボックスから設定値を取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>int iComboValue = GetComboValue(key, t)</para>
        /// </example>
        /// <param name="key">dgvカラム名</param>
        /// <param name="t">表記文字列</param>
        /// <returns>設定値</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private int GetComboValue(string key, string t)
        {
            if (key != null && dicProp.ContainsKey(key) && dicProp[key].ContainsValue(t))
            {
                foreach (KeyValuePair<int, string> pair in dicProp[key])
                {
                    if (pair.Value.Equals(t))
                    {
                        return (pair.Key);
                    }
                }
            }
            return (0);
        }

        #endregion

        #region データ系列の保存

        /// <summary><para>method outline:</para>
        /// <para>共通データにデータ系列を保存する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>SaveProp()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool SaveProp()
        {
            // 河川データの取得
            RiverData rd = Util.GetRiverData(_commonData, this.CurrentRiver);
            if (rd == null)
            {
                // メッセージ:河川データが見つかりません。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_FIND_RIVER_DATA);
                return (false);
            }

            // 一旦、データ系列をクリア
            rd.GraphPropList.Clear();

            // 行数分だけループしながらデータ系列を保存
            for (int row = 0; row < dgvProp.RowCount; row++)
            {
                GraphPropData data = new GraphPropData();
                data.FileId = Util.NullToBlank(dgvProp[dgvProp.Columns["colFileID"].Index, row].Value);
                data.ColumnNm = Util.NullToBlank(dgvProp[dgvProp.Columns["colItemName"].Index, row].Value);
                data.NotesNm = Util.NullToBlank(dgvProp[dgvProp.Columns["colLegendName"].Index, row].Value);
                data.ScaleType = GetComboValue("colScale", (string)dgvProp[dgvProp.Columns["colScale"].Index, row].Value);
                data.LineDisp = Util.IsTrue(dgvProp[dgvProp.Columns["colLineDisp"].Index, row].Value);
                data.LineType = GetComboValue("colLineStyle", (string)dgvProp[dgvProp.Columns["colLineStyle"].Index, row].Value);
                data.LineColor = dgvProp[dgvProp.Columns["colLineColor"].Index, row].Style.ForeColor.ToArgb();
                data.LineSize = GetComboValue("colLineWidth", (string)dgvProp[dgvProp.Columns["colLineWidth"].Index, row].Value);
                data.SymbolDisp = Util.IsTrue(dgvProp[dgvProp.Columns["colSymbolDisp"].Index, row].Value);
                data.SymbolType = GetComboValue("colSymbolStyle", (string)dgvProp[dgvProp.Columns["colSymbolStyle"].Index, row].Value);
                data.SymbolColor = dgvProp[dgvProp.Columns["colSymbolColor"].Index, row].Style.ForeColor.ToArgb();
                data.SymbolSize = GetComboValue("colSymbolSize", (string)dgvProp[dgvProp.Columns["colSymbolSize"].Index, row].Value);

                rd.GraphPropList.Add(data);
            }

            return (true);
        }

        #endregion

        #region データ系列の読み込み

        /// <summary><para>method outline:</para>
        /// <para>共通データからデータ系列を読み込む</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>LoadProp()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void LoadProp()
        {
            // 河川データの取得
            RiverData rd = Util.GetRiverData(_commonData, this.CurrentRiver);

            // データグリッドのクリア
            dgvProp.Rows.Clear();
            if (rd.GraphPropList != null)
            {
                // 取得したデータ系列をデータグリッドに一行ずつ追加
                foreach (GraphPropData data in rd.GraphPropList)
                {
                    // 念のため、データ系列のファイルID/項目名が存在するかチェック
                    if (Util.IsExistItem(_commonData, data.FileId, data.ColumnNm))
                    {
                        int row = dgvProp.Rows.Add();
                        setTextBoxCell("colFileID", row, data.FileId);
                        setTextBoxCell("colFileName", row, Util.GetFileName(_commonData, data.FileId));
                        setTextBoxCell("colItemName", row, data.ColumnNm);
                        setTextBoxCell("colLegendName", row, data.NotesNm);
                        setComboCell("colScale", row, data.ScaleType);
                        setChkBoxCell("colLineDisp", row, data.LineDisp);
                        setComboCell("colLineStyle", row, data.LineType);
                        setColorCell("colLineColor", row, data.LineColor);
                        setComboCell("colLineWidth", row, data.LineSize);
                        setChkBoxCell("colSymbolDisp", row, data.SymbolDisp);
                        setComboCell("colSymbolStyle", row, data.SymbolType);
                        setColorCell("colSymbolColor", row, data.SymbolColor);
                        setComboCell("colSymbolSize", row, data.SymbolSize);
                    }
                }
            }
        }

        #endregion

        #region セルクリック時の処理(カラーダイアログ開く処理)

        /// <summary><para>method outline:</para>
        /// <para>色選択用のセルがクリックまたはEnterキー入力された場合、色選択ダイアログを表示する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>clickEvent(col, row)</para>
        /// </example>
        /// <param name="col">列番号</param>
        /// <param name="row">行番号</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void clickEvent(int col, int row)
        {
            if (col >= 0 && col < dgvProp.ColumnCount)
            {
                // 線色かシンボル色のセルの場合
                if (dgvProp.Columns[col].Name == "colLineColor"
                 || dgvProp.Columns[col].Name == "colSymbolColor")
                {
                    if (row >= 0 && row < dgvProp.RowCount)
                    {
                        // 色選択ダイアログのオープン
                        dlgColor.Color = dgvProp[col, row].Style.ForeColor;
                        if (dlgColor.ShowDialog(this) == DialogResult.OK)
                        {
                            // 選択した色をデータグリッドに保存
                            setColorCell(col, row, dlgColor.Color);
                        }
                    }
                }
            }
        }

        #endregion

        #endregion

        #region イベントメソッド

        #region フォームロード時

        /// <summary><para>method outline:</para>
        /// <para>フォームロード時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void GraphProp_Load(object sender, EventArgs e)
        {
            // 河川のチェック
            string river = this.CurrentRiver;
            if (river == null || river.Length <= 0)
            {
                // メッセージ:河川名を選択してください。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_SELECT_RIVER_NAME,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_DATA_SERIES);
                this.Close();
                return;
            }

            this.Width = 750;
            this.Height = 400;

            // スケールの設定
            Dictionary<int, string> list = new Dictionary<int, string>();
            list.Add((int)Util.SCALE.Auto, Properties.HySAddinLsor2VieweEXEResources.AUTOMATIC); // 自動
            list.Add((int)Util.SCALE.LeftY, Properties.HySAddinLsor2VieweEXEResources.Y_AXIS_LEFT);  // Y軸-左
            list.Add((int)Util.SCALE.RightY, Properties.HySAddinLsor2VieweEXEResources.Y_AXIS_RIGHT); // Y軸-右
            AddCombo("colScale", list);

            // 線種の設定
            list.Clear();
            list.Add((int)Util.LINESTYLE.Solid, Properties.HySAddinLsor2VieweEXEResources.SOLID_LINE);    // 実線
            list.Add((int)Util.LINESTYLE.Dot, Properties.HySAddinLsor2VieweEXEResources.DOTTED_LINE);  // 点線
            list.Add((int)Util.LINESTYLE.Dash, Properties.HySAddinLsor2VieweEXEResources.DASHED_LINE); // 破線
            list.Add((int)Util.LINESTYLE.DashDot, Properties.HySAddinLsor2VieweEXEResources.CHAIN_LINE);  // 一点鎖線
            list.Add((int)Util.LINESTYLE.DashDotDot, Properties.HySAddinLsor2VieweEXEResources.TWO_DOT_CHAIN_LINE);   // 二点鎖線
            AddCombo("colLineStyle", list);

            // 線太さの設定
            list.Clear();
            list.Add((int)Util.LINEWIDTH.Thin, Properties.HySAddinLsor2VieweEXEResources.THIN);   // 細
            list.Add((int)Util.LINEWIDTH.Middle, Properties.HySAddinLsor2VieweEXEResources.MEDIUM); // 中
            list.Add((int)Util.LINEWIDTH.Thick, Properties.HySAddinLsor2VieweEXEResources.THICK);  // 太
            AddCombo("colLineWidth", list);

            // シンボル形状の設定
            list.Clear();
            list.Add((int)Util.SYMBOLSTYLE.Cross, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_CROSS);    // ＋
            list.Add((int)Util.SYMBOLSTYLE.Cross2, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_CROSS2);   // ×
            list.Add((int)Util.SYMBOLSTYLE.Circle, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_CIRCLE);   // ○
            list.Add((int)Util.SYMBOLSTYLE.CircleF, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_CIRCLEF);  // ●
            list.Add((int)Util.SYMBOLSTYLE.Triangle, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_TRIANGLE); // △
            list.Add((int)Util.SYMBOLSTYLE.TriangleF, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_TRIANGLEF);    // ▲
            list.Add((int)Util.SYMBOLSTYLE.Triangle2, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_TRIANGLE2);    // ▽
            list.Add((int)Util.SYMBOLSTYLE.Triangle2F, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_TRIANGLE2F);   // ▼
            list.Add((int)Util.SYMBOLSTYLE.Square, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_SQUARE);   // □
            list.Add((int)Util.SYMBOLSTYLE.SquareF, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_SQUAREF);  // ■
            list.Add((int)Util.SYMBOLSTYLE.Square2, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_SQUARE2);  // ◇
            list.Add((int)Util.SYMBOLSTYLE.Square2F, Properties.HySAddinLsor2VieweEXEResources.SYMBOL_SQUARE2F); // ◆ 
            AddCombo("colSymbolStyle", list);

            // シンボルサイズの設定
            list.Clear();
            list.Add((int)Util.SYMBOLSIZE.Small, Properties.HySAddinLsor2VieweEXEResources.SMALL); // 小
            list.Add((int)Util.SYMBOLSIZE.Medium, Properties.HySAddinLsor2VieweEXEResources.MEDIUM);    // 中
            list.Add((int)Util.SYMBOLSIZE.Large, Properties.HySAddinLsor2VieweEXEResources.LARGE); // 大
            AddCombo("colSymbolSize", list);

            // 共通データからデータ系列を読み込み
            LoadProp();

            // 適用ボタンにフォーカス移す
            cmdShowGraph.Focus();

            // ボタン表示の更新
            UpdateMenu();
        }

        #endregion

        #region [適用]ボタン押下時

        /// <summary><para>method outline:</para>
        /// <para>「適用」ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdShowGraph_Click(object sender, EventArgs e)
        {
            this.Close();   // 入力値のチェックはFormClosingで処理
        }

        #endregion

        #region [取消]ボタン押下時

        /// <summary><para>method outline:</para>
        /// <para>「取消」ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdCancel_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        #endregion

        #region [追加]ボタン押下時

        /// <summary><para>method outline:</para>
        /// <para>「追加」ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdAdd_Click(object sender, EventArgs e)
        {
            // データ系列追加ダイアログの表示
            GraphPropAdd f = new GraphPropAdd(_commonData);
            f.ShowDialog(this);
            if (f.FileId.Length > 0 && f.ItemName.Length > 0)
            {
                for (int row = 0; row < dgvProp.Rows.Count; row++)
                {
                    // 重複チェック
                    if (f.FileId.Equals(dgvProp[dgvProp.Columns["colFileID"].Index, row].Value)
                     && f.ItemName.Equals(dgvProp[dgvProp.Columns["colItemName"].Index, row].Value))
                    {
                        // メッセージ:すでに同じ項目が設定されています。
                        MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_SAME_ITEM_SET,
                            Properties.HySAddinLsor2VieweEXEResources.CAPTION_DATA_SERIES);
                        return;
                    }
                }
                // データグリッドに追加
                AddRow(f.FileId, f.ItemName);
            }
            f.Dispose();

            // ボタンの更新
            UpdateMenu();
        }

        #endregion

        #region [削除]ボタン押下時

        /// <summary><para>method outline:</para>
        /// <para>「削除」ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdDel_Click(object sender, EventArgs e)
        {
            if (dgvProp.RowCount > 0)
            {
                // 行の削除
                dgvProp.Rows.Remove(dgvProp.CurrentRow);
            }

            // ボタンの更新
            UpdateMenu();
        }

        #endregion

        #region [上へ]ボタン押下時

        /// <summary><para>method outline:</para>
        /// <para>「上へ」ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdUp_Click(object sender, EventArgs e)
        {
            if (dgvProp.CurrentCell != null)
            {
                int r = dgvProp.CurrentCell.RowIndex;
                if (dgvProp.RowCount > 1 && r > 0)
                {
                    // 1行上と行を入れ替え
                    DataGridViewRow row = dgvProp.Rows[r - 1];
                    dgvProp.Rows.RemoveAt(r - 1);
                    dgvProp.Rows.Insert(r, row);
                }
            }

            // ボタンの更新
            UpdateMenu();
        }

        #endregion

        #region [下へ]ボタン押下時

        /// <summary><para>method outline:</para>
        /// <para>「下へ」ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdDown_Click(object sender, EventArgs e)
        {
            if (dgvProp.CurrentCell != null)
            {
                int r = dgvProp.CurrentCell.RowIndex;
                if (dgvProp.RowCount > 1 && r + 1 < dgvProp.RowCount)
                {
                    // 1行下と行を入れ替え
                    DataGridViewRow row = dgvProp.Rows[r + 1];
                    dgvProp.Rows.RemoveAt(r + 1);
                    dgvProp.Rows.Insert(r, row);
                }
            }

            // ボタンの更新
            UpdateMenu();
        }

        #endregion

        #region データグリッドフォーカスイン時(自動プルダウン処理)

        /// <summary><para>method outline:</para>
        /// <para>DataGridViewのセルにフォーカスが移った時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void dgvProp_CellEnter(object sender, DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= 0 && e.ColumnIndex < dgvProp.ColumnCount
             && e.RowIndex >= 0 && e.RowIndex < dgvProp.RowCount)
            {
                // セルがコンボボックスタイプの場合
                if (dgvProp[e.ColumnIndex, e.RowIndex] is DataGridViewComboBoxCell)
                {
                    // F4キーを送ってコンボを展開
                    SendKeys.Send("{F4}");
                }
            }
        }

        #endregion

        #region データグリッドクリック時(カラーダイアログ開く処理)

        /// <summary><para>method outline:</para>
        /// <para>DataGridViewのセルがクリックされた時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void dgvProp_CellClick(object sender, DataGridViewCellEventArgs e)
        {
            clickEvent(e.ColumnIndex, e.RowIndex);
        }

        #endregion

        #region キー押下時(カラーダイアログ開く処理)

        /// <summary><para>method outline:</para>
        /// <para>キーが押下された時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void dgvProp_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                clickEvent(dgvProp.CurrentCell.ColumnIndex, dgvProp.CurrentCell.RowIndex);
                e.SuppressKeyPress = true;
            }
        }

        #endregion

        #region フォームクローズ直前時

        /// <summary><para>method outline:</para>
        /// <para>フォームクローズ直前の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void GraphProp_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (this.DialogResult == DialogResult.OK)
            {
                // 設定の保存
                if (SaveProp() == false)
                {
                    e.Cancel = true;
                }
            }
        }

        #endregion

        #endregion
    }
}
