﻿/// <summary>
/// データ編集画面
/// </summary>
/// <create>2010/01/26</create>
/// <modifier></modifier>
/// <modify></modify>
/// <modification></modification>
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Xml.Serialization;
using System.IO;

using Microsoft.VisualBasic.FileIO;

namespace lsor
{
    /// <summary><para>class outline:</para>
    /// <para>データ編集画面出力制御クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[ver 1.0.0][2010/02/02][新規作成]</para>
    /// </remarks>
    public partial class DataEdit : Form
    {
        #region プライベート変数

        /// <summary>プロジェクト共通データ</summary>
        private CommonData _commonData = new CommonData();
        /// <summary>グラフ画面</summary>
        private GraphMain _graphMain;
        /// <summary>ファイル名ラベルのコントロール配列</summary>
        private Label[] _lblFileNm = new Label[MAX_TABPAGE_NUM];
        /// <summary>タブページのコントロール配列</summary>
        private TabPage[] _tabPage = new TabPage[MAX_TABPAGE_NUM];
        /// <summary>データグリッドのコントロール配列</summary>
        private DataGridView[] _dgvData = new DataGridView[MAX_TABPAGE_NUM];
        /// <summary>データ表示コントロール</summary>
        private DispData[] _dispData = new DispData[MAX_TABPAGE_NUM];
        /// <summary>タブページ表示管理</summary>
        private TabPageManager _tabPageManager;
        /// <summary>イベント抑止フラグ</summary>
        private bool _eventCancelFlg = false;
        /// <summary>データ保存ストリーム</summary>
        private StreamWriter _sw;
        /// <summary>プロジェクト保存ストリーム</summary>
        private FileStream _fs;
        /// <summary>文字エンコーディング</summary>
        private Encoding _enc = Encoding.Default;
        // プロファイル用
        //        private System.Diagnostics.Stopwatch _tt = new System.Diagnostics.Stopwatch();

        /// <summary>列挿入用ワーク(カラム名)</summary>
        private string _colAddColumnNm;
        /// <summary>列挿入用ワーク(データ種別)</summary>
        private string _colAddDataType;
        /// <summary>列挿入用ワーク(単位種別)</summary>
        private string _colAddUnitType;
        /// <summary>列挿入用ワーク(モード)</summary>
        private string _colAddMode;

        /// <summary>ヘッダ列数</summary>
        private const int HEADER_COLUMNS = 4;
        /// <summary>データを表示するタブページ数</summary>
        private const int MAX_TABPAGE_NUM = 51;

        #endregion

        #region コンストラクタ

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>DataEdit csDataEdit = new DataEdit()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>データ編集画面出力制御クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public DataEdit()
        {
            InitializeComponent();
        }
        #endregion

        #region プライベートメソッド

        #region フォームデータ初期化

        /// <summary><para>method outline:</para>
        /// <para>フォームデータを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>clearFormData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void clearFormData()
        {
            // グリッドイベント抑止
            this._eventCancelFlg = true;
            for (int i = 0; i < 51; i++)
            {
                this._lblFileNm[i].Text = "";
                this._dgvData[i].Rows.Clear();
                this._dgvData[i].Columns.Clear();
                this._dgvData[i].ClipboardCopyMode = DataGridViewClipboardCopyMode.EnableWithoutHeaderText;
                this._dgvData[i].ContextMenuStrip = this.contextMenuStrip;
            }
            // グリッドイベント再開
            this._eventCancelFlg = false;
        }
        #endregion

        #region フォームデータ設定

        /// <summary><para>method outline:</para>
        /// <para>フォームデータを設定する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setFormData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setFormData()
        {
            // グリッドイベント抑止
            this._eventCancelFlg = true;

            // 事前にタブの表示を切り替えておく（高速化対応）
            for (int i = 0; i < this._commonData.FileList.Count; i++)
            {
                if (this._commonData.FileList[i].FileNm != null)
                {
                    // 該当タブを表示
                    this._tabPageManager.ChangeTabPageVisible(i, true);
                }
                else
                {
                    // 該当タブを非表示
                    this._tabPageManager.ChangeTabPageVisible(i, false);
                }
            }

            // CommonDataより、フォーム上の各コントロールへデータをセットする
            for (int i = 0; i < this._commonData.FileList.Count; i++)
            {
                if (this._commonData.FileList[i].FileNm != null)
                {
                    // タイムアウト回避
                    Application.DoEvents();
                    // 未保存フラグがtrueであればタブ名称に"*"を付与する
                    if (this._commonData.FileList[i].UnsaveFlg)
                    {
                        this._tabPage[i].Text = this._tabPage[i].Text.Replace("*", "") + "*";
                    }
                    else
                    {
                        this._tabPage[i].Text = this._tabPage[i].Text.Replace("*", "");
                    }
                    // ファイル名をセット
                    this._lblFileNm[i].Text = this._commonData.FileList[i].FileNm;
                    // グリッド列を生成
                    DataGridViewTextBoxColumn column;
                    for (int j = 0; j < this._commonData.FileList[i].ColumnList.Count; j++)
                    {
                        column = new DataGridViewTextBoxColumn();
                        column.Name = "column" + j.ToString();
                        column.HeaderText = this._commonData.FileList[i].ColumnList[j].ColumnNm + "\r\n"
                            + this._commonData.FileList[i].ColumnList[j].DataType + "\r\n" + this._commonData.FileList[i].ColumnList[j].UnitType;
                        column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
                        column.Resizable = DataGridViewTriState.True;

                        // キー項目専用の処理
                        if (j < HEADER_COLUMNS)
                        {
                            column.DefaultCellStyle.BackColor = Color.LightGray;
                            column.ReadOnly = true;
                        }
                        // 数値項目専用の処理
                        if (this._commonData.FileList[i].ColumnList[j].DataType == "double"
                            || this._commonData.FileList[i].ColumnList[j].DataType == "int")
                        {
                            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
                        }
                        column.SortMode = DataGridViewColumnSortMode.NotSortable;

                        this._dgvData[i].Columns.Add(column);
                    }
                    // グリッドデータをセット
                    if (this._commonData.FileList[i].CsvData.Count > 0)
                    {
                        this._dgvData[i].SuspendLayout();
                        this._dgvData[i].Rows.Add(this._commonData.FileList[i].CsvData.Count);
                        for (int j = 0; j < this._commonData.FileList[i].CsvData.Count; j++)
                        {
                            for (int k = 0; k < this._commonData.FileList[i].ColumnList.Count; k++)
                            {
                                this._dgvData[i][k, j].Value = ((string[])this._commonData.FileList[i].CsvData[j])[k];
                                //                                this._dgvData[i].AutoResizeColumn(k);
                            }
                        }
                        // エラーセルの強調表示
                        checkData(i);

                        this._dgvData[i].AutoResizeColumns();
                        this._dgvData[i].ResumeLayout();
                    }
                    // 該当タブを表示
                    //                    this._tabPageManager.ChangeTabPageVisible(i, true);
                }
                else
                {
                    // 不要タブを隠蔽
                    //                    this._tabPageManager.ChangeTabPageVisible(i, false);
                }
            }

            // メニューボタンを使用可
            mnuFOverwrite.Enabled = true;
            mnuFAppend.Enabled = true;
            mnuEdit.Enabled = true;
            contextMenuStrip.Enabled = true;
            mnuDisp.Enabled = true;
            mnuTool.Enabled = true;
            mnuFOverwriteData.Enabled = true;

            // グリッドのイベント再開
            this._eventCancelFlg = false;

        }
        #endregion

        #region グリッド再描画

        /// <summary><para>method outline:</para>
        /// <para>グリッドを再描画する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>resetGridData(fileIdx)</para>
        /// </example>
        /// <param name="fileIdx">ファイルIDインデックス</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void resetGridData(int fileIdx)
        {
            this._eventCancelFlg = true;

            this._dgvData[fileIdx].SuspendLayout();

            // 画面再描画
            this._dgvData[fileIdx].Rows.Clear();
            this._dgvData[fileIdx].Columns.Clear();
            // グリッド列を生成
            DataGridViewTextBoxColumn column;
            for (int j = 0; j < this._commonData.FileList[fileIdx].ColumnList.Count; j++)
            {
                column = new DataGridViewTextBoxColumn();
                column.Name = "column" + j.ToString();
                //                column.HeaderText = this._commonData.FileList[fileIdx].ColumnList[j].ColumnNm;
                column.HeaderText = this._commonData.FileList[fileIdx].ColumnList[j].ColumnNm + "\r\n"
                    + this._commonData.FileList[fileIdx].ColumnList[j].DataType + "\r\n" + this._commonData.FileList[fileIdx].ColumnList[j].UnitType;
                column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
                column.Resizable = DataGridViewTriState.True;

                if (j < HEADER_COLUMNS)
                {
                    column.DefaultCellStyle.BackColor = Color.LightGray;
                    column.ReadOnly = true;
                }
                if (this._commonData.FileList[fileIdx].ColumnList[j].DataType == "double"
                    || this._commonData.FileList[fileIdx].ColumnList[j].DataType == "int")
                {
                    column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
                }
                column.SortMode = DataGridViewColumnSortMode.NotSortable;

                this._dgvData[fileIdx].Columns.Add(column);
            }
            this._dgvData[fileIdx].Rows.Add(this._commonData.FileList[fileIdx].CsvData.Count);
            // グリッドデータをセット
            for (int j = 0; j < this._commonData.FileList[fileIdx].CsvData.Count; j++)
            {
                for (int k = 0; k < this._commonData.FileList[fileIdx].ColumnList.Count; k++)
                {
                    this._dgvData[fileIdx][k, j].Value = ((string[])this._commonData.FileList[fileIdx].CsvData[j])[k];
                    //                    this._dgvData[fileIdx].AutoResizeColumn(k);
                }
            }
            this._dgvData[fileIdx].AutoResizeColumns();
            this._eventCancelFlg = false;

            // エラーセルの強調表示
            checkData(fileIdx);

            this._dgvData[fileIdx].ResumeLayout();

        }
        #endregion

        #region グラフ再描画

        /// <summary><para>method outline:</para>
        /// <para>グラフを再描画する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>resetGraph()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void resetGraph()
        {
            if (this._graphMain != null && !this._graphMain.IsDisposed)
            {
                // グラフ再描画メソッド呼び出し
                this._graphMain.SetCommonData(this._commonData);
                this._graphMain.UpdateGraph(true);
            }
        }
        #endregion

        #region ダブルクォート加工

        /// <summary><para>method outline:</para>
        /// <para>必要に応じて文字列をダブルクォートで囲む</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>resetGraph()</para>
        /// </example>
        /// <param name="dt">加工を行う文字列</param>
        /// <returns>加工を行った文字列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private string addDoubleQuote(string dt)
        {
            if (dt == null)
            {
                return dt;
            }

            // "で囲む必要があるか
            if (dt.IndexOf('"') > -1 ||
                dt.IndexOf(',') > -1 ||
                dt.IndexOf('\r') > -1 ||
                dt.IndexOf('\n') > -1 ||
                dt.StartsWith(" ") ||
                dt.StartsWith("\t") ||
                dt.EndsWith(" ") ||
                dt.EndsWith("\t"))
            {
                if (dt.IndexOf('"') > -1)
                {
                    dt = dt.Replace("\"", "\"\"");
                }
                return "\"" + dt + "\"";
            }
            else
            {
                return dt;
            }
        }
        #endregion

        #region データファイル保存

        /// <summary><para>method outline:</para>
        /// <para>データファイルを保存する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = saveDataFile(fileNm)</para>
        /// </example>
        /// <param name="fileNm">ファイル名</param>
        /// <returns>bool true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool saveDataFile(string fileNm)
        {

            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
                this.tabMain.SelectedTab.Text.Replace("*", ""));

            // 最終列のインデックスを取得
            int lastColIndex = this._commonData.FileList[fileIdx].ColumnList.Count - 1;

            bool ret = false;

            try
            {
                this._sw = new StreamWriter(fileNm, false, this._enc);

                // 項目名
                for (int i = 0; i < this._commonData.FileList[fileIdx].ColumnList.Count; i++)
                {
                    string dt = this._commonData.FileList[fileIdx].ColumnList[i].ColumnNm;
                    // データを書き込む
                    dt = this.addDoubleQuote(dt);
                    this._sw.Write(dt);
                    // カンマを書き込む
                    if (lastColIndex > i)
                    {
                        this._sw.Write(',');
                    }
                }
                // 改行する
                this._sw.Write("\r\n");

                // データ型
                for (int i = 0; i < this._commonData.FileList[fileIdx].ColumnList.Count; i++)
                {
                    string dt = this._commonData.FileList[fileIdx].ColumnList[i].DataType;
                    // データを書き込む
                    dt = this.addDoubleQuote(dt);
                    this._sw.Write(dt);
                    // カンマを書き込む
                    if (lastColIndex > i)
                    {
                        this._sw.Write(',');
                    }
                }
                // 改行する
                this._sw.Write("\r\n");

                // 単位
                for (int i = 0; i < this._commonData.FileList[fileIdx].ColumnList.Count; i++)
                {
                    string dt = this._commonData.FileList[fileIdx].ColumnList[i].UnitType;
                    // データを書き込む
                    dt = this.addDoubleQuote(dt);
                    this._sw.Write(dt);
                    // カンマを書き込む
                    if (lastColIndex > i)
                    {
                        this._sw.Write(',');
                    }
                }
                // 改行する
                this._sw.Write("\r\n");

                // データ
                // CSVデータ加工
                for (int i = 0; i < this._commonData.FileList[fileIdx].CsvData.Count; i++)
                {
                    for (int j = 0; j < this._commonData.FileList[fileIdx].ColumnList.Count; j++)
                    {
                        string dt = ((string[])this._commonData.FileList[fileIdx].CsvData[i])[j];

                        // データを書き込む
                        dt = this.addDoubleQuote(dt);
                        this._sw.Write(dt);
                        // カンマを書き込む
                        if (lastColIndex > j)
                        {
                            this._sw.Write(',');
                        }
                    }
                    // 改行する
                    this._sw.Write("\r\n");
                }
                ret = true;
                this._commonData.FileList[fileIdx].UnsaveFlg = false;
                this._tabPage[fileIdx].Text = this._tabPage[fileIdx].Text.Replace("*", "");
                // メッセージ:データファイルを保存しました。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_INFO_DATA_FILE_SAVED,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_DATA_FILE,
                    MessageBoxButtons.OK, MessageBoxIcon.Information);

            }
            catch
            {
                // メッセージ:データファイルの保存に失敗しました。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_FAILED_TO_SAVE_DATA_FILE,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_DATA_FILE,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                if (this._sw != null)
                {
                    this._sw.Close();
                }
            }
            return ret;
        }
        #endregion

        #region クリップボードのデータをグリッドへペースト

        /// <summary><para>method outline:</para>
        /// <para>クリップボードのデータをグリッドへペーストする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>pasteData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void pasteData()
        {
            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
                this.tabMain.SelectedTab.Text.Replace("*", ""));

            if (this._dgvData[fileIdx].CurrentCell == null)
            {
                return;
            }
            int insRowIdx = this._dgvData[fileIdx].CurrentCell.RowIndex;

            // クリップボードの内容を取得して、行で分ける
            string pasteText = Clipboard.GetText();
            if (String.IsNullOrEmpty(pasteText))
            {
                return;
            }
            pasteText = pasteText.Replace("\r\n", "\n");
            pasteText = pasteText.Replace("\r", "\n");
            pasteText.TrimEnd(new char[] { '\n' });
            string[] lines = pasteText.Split('\n');

            foreach (String line in lines)
            {
                string[] vals = line.Split('\t');   // タブで分割
                try
                {
                    DataGridViewRow row = this._dgvData[fileIdx].Rows[insRowIdx];

                    int j = 0;
                    for (int i = this._dgvData[fileIdx].CurrentCell.ColumnIndex; i < row.Cells.Count; i++)
                    {
                        try
                        {
                            if (!String.IsNullOrEmpty(vals[j]))
                            {
                                // キー項目は貼り付けしない
                                if (i >= HEADER_COLUMNS)
                                {
                                    row.Cells[i].Value = vals[j];
                                }
                            }

                            if (j == vals.Length - 1)
                            {
                                break;
                            }
                            j++;
                        }
                        catch
                        {
                        }
                    }

                    // 次の行へ
                    insRowIdx++;
                }
                catch
                {
                }
            }
            // エラーセルの強調表示
            checkData(-1);

        }
        #endregion

        #region 選択中の列リストの取得

        /// <summary><para>method outline:</para>
        /// <para>選択中の列リストを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜int＞ ColumnList = getSelectedColumns(dgv)</para>
        /// </example>
        /// <param name="dgv">DataGridViewクラス</param>
        /// <returns>選択中の列リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private List<int> getSelectedColumns(DataGridView dgv)
        {
            List<int> cols = new List<int>();

            foreach (DataGridViewCell cell in dgv.SelectedCells)
            {
                if (!cols.Contains(cell.ColumnIndex))
                {
                    cols.Add(cell.ColumnIndex);
                }
            }
            cols.Sort();

            return (cols);
        }
        #endregion

        #region 選択中の行リストの取得

        /// <summary><para>method outline:</para>
        /// <para>選択中の行リストを取得する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>List＜int＞ ColumnList = getSelectedRows(dgv)</para>
        /// </example>
        /// <param name="dgv">DataGridViewクラス</param>
        /// <returns>選択中の行リスト</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private List<int> getSelectedRows(DataGridView dgv)
        {
            List<int> rows = new List<int>();

            foreach (DataGridViewCell cell in dgv.SelectedCells)
            {
                if (!rows.Contains(cell.RowIndex))
                {
                    rows.Add(cell.RowIndex);
                }
            }
            rows.Sort();

            return (rows);
        }
        #endregion

        #region セルの背景色設定(セル指定)

        /// <summary><para>method outline:</para>
        /// <para>セルの背景色設定をする(セル指定)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = checkData(fileIdx, columnIdx, rowIdx)</para>
        /// </example>
        /// <param name="fileIdx">ファイルIDインデックス</param>
        /// <param name="columnIdx">列番号</param>
        /// <param name="rowIdx">行番号</param>
        /// <returns>bool true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool checkData(int fileIdx, int columnIdx, int rowIdx)
        {
            bool ret = false;

            if (fileIdx < 0)
            {
                fileIdx = Util.GetFileIndex(this._commonData.FileList,
                    this.tabMain.SelectedTab.Text.Replace("*", ""));
            }
            if (fileIdx < 0 || fileIdx >= this._commonData.FileList.Count)
            {
                return (ret);
            }

            if (columnIdx < 0 || rowIdx < 0)
            {
                if (columnIdx < 0 && rowIdx < 0)
                {
                    checkData(fileIdx);
                }
                else if (columnIdx < 0)
                {
                    for (int col = HEADER_COLUMNS; col < this._dgvData[fileIdx].ColumnCount; col++)
                    {
                        checkData(fileIdx, col, rowIdx);
                    }
                }
                else if (rowIdx < 0)
                {
                    for (int row = 0; row < this._dgvData[fileIdx].RowCount; row++)
                    {
                        checkData(fileIdx, columnIdx, row);
                    }
                }
            }

            if (columnIdx >= HEADER_COLUMNS
                && columnIdx >= 0 && columnIdx < this._dgvData[fileIdx].ColumnCount
                && rowIdx >= 0 && rowIdx < this._dgvData[fileIdx].RowCount)
            {
                DataGridViewCell cell = this._dgvData[fileIdx][columnIdx, rowIdx];

                if (cell != null)
                {
                    if (cell.Value != null && !Util.IsBlank(cell.Value.ToString()))
                    {
                        if (this._commonData.FileList[fileIdx].ColumnList[columnIdx].DataType == "double"
                            && !Util.IsDouble(cell.Value.ToString()))
                        {
                            ret = true;
                        }
                        else if (this._commonData.FileList[fileIdx].ColumnList[columnIdx].DataType == "int"
                            && !Util.IsInteger(cell.Value.ToString()))
                        {
                            ret = true;
                        }
                    }
                    if (ret)
                    {
                        if (cell.Style.BackColor != Color.Pink)
                        {
                            cell.Style.BackColor = Color.Pink;
                        }
                    }
                    else
                    {
                        if (cell.Style.BackColor != _dgvData[fileIdx].DefaultCellStyle.BackColor)
                        {
                            cell.Style.BackColor = _dgvData[fileIdx].DefaultCellStyle.BackColor;
                        }
                    }
                }
            }

            return (ret);
        }
        #endregion

        #region セルの背景色設定(ファイル指定)

        /// <summary><para>method outline:</para>
        /// <para>セルの背景色設定をする(ファイル指定)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>bool bRtn = checkData(fileIdx)</para>
        /// </example>
        /// <param name="fileIdx">ファイルIDインデックス</param>
        /// <returns>bool true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool checkData(int fileIdx)
        {
            bool ret = false;

            if (fileIdx < 0)
            {
                // カレントのDataGridViewを対象とする
                fileIdx = Util.GetFileIndex(this._commonData.FileList,
                    this.tabMain.SelectedTab.Text.Replace("*", ""));
            }

            if (!Util.IsBlank(this._commonData.FileList[fileIdx].FileNm)
                && this._commonData.FileList[fileIdx].CsvData != null)
            {
                for (int row = 0; row < this._commonData.FileList[fileIdx].CsvData.Count; row++)
                {
                    for (int col = HEADER_COLUMNS; col < this._commonData.FileList[fileIdx].ColumnList.Count; col++)
                    {
                        if (checkData(fileIdx, col, row))
                        {
                            ret = true;
                        }
                    }
                }
            }

            return (ret);
        }
        #endregion

        #region セルの背景色設定(全DataGridView)

        /// <summary><para>method outline:</para>
        /// <para>セルの背景色設定をする(全DataGridView)</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>checkData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>bool true:成功、false:失敗</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private bool checkData()
        {
            bool ret = false;

            for (int fileIdx = 0; fileIdx < this._commonData.FileList.Count; fileIdx++)
            {
                if (checkData(fileIdx))
                {
                    ret = true;
                }
            }

            return (ret);
        }
        #endregion

        #endregion

        #region イベントハンドラ

        #region フォーム_Load

        /// <summary><para>method outline:</para>
        /// <para>フォームロード時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void DataEdit_Load(object sender, EventArgs e)
        {

            // === コントロール初期化 ===

            #region タブページ生成
            for (int i = 0; i < MAX_TABPAGE_NUM; i++)
            {
                // データ表示コントロール配列生成
                this._dispData[i] = new DispData();

                // ファイル名コントロール配列と関連付け
                this._lblFileNm[i] = _dispData[i].GetFileNameLabel();

                // データグリッドコントロール配列と関連付け
                this._dgvData[i] = this._dispData[i].GetDataGridView();

                // イベントハンドラに関連付け
                this._dgvData[i].CellEndEdit += new DataGridViewCellEventHandler(this.dgvData_CellEndEdit);
                this._dgvData[i].CellValueChanged += new DataGridViewCellEventHandler(this.dgvData_CellValueChanged);
                this._dgvData[i].KeyDown += new System.Windows.Forms.KeyEventHandler(this.dgvData_KeyDown);

                // タブページ配列生成
                this._tabPage[i] = new TabPage();
                if (i == 0)
                {
                    this._tabPage[i].Text = Properties.HySAddinLsor2VieweEXEResources.CALCULATION_INPUT;    // 計算入力
                }
                else
                {
                    this._tabPage[i].Text = string.Format(Properties.HySAddinLsor2VieweEXEResources.CALCULATION_RESULT, i);    // 計算結果
                }

                this._tabPage[i].Controls.Add(_dispData[i]);    // データ表示コントロールをタブページに追加 
                this.tabMain.TabPages.Add(this._tabPage[i]);    // 生成したタブページをタブに追加
            }
            #endregion

            this._tabPageManager = new TabPageManager(this.tabMain);

            // フォームデータ初期化
            this.clearFormData();

            // メニューバー
            mnuFOverwrite.Enabled = false;
            mnuFAppend.Enabled = false;
            mnuEdit.Enabled = false;
            contextMenuStrip.Enabled = false;
            mnuDisp.Enabled = false;
            mnuTool.Enabled = false;
            mnuFOverwriteProject.Enabled = false;
            mnuFOverwriteData.Enabled = false;

            // 結果出力タブの隠蔽
            for (int i = 1; i < 51; i++)
            {
                this._tabPageManager.ChangeTabPageVisible(i, false);
            }

            this.Width = 750;
            this.Height = 550;

            for (int i = 0; i < _dgvData[0].ColumnCount; i++)
            {
                if (i < 4)
                {
                    _dgvData[0].Columns[i].DefaultCellStyle.BackColor = Color.LightGray;
                    _dgvData[0].Columns[i].ReadOnly = true;
                    _dgvData[0].Columns[i].Frozen = false;
                }
                if (i >= 3)
                {
                    _dgvData[0].Columns[i].DefaultCellStyle.Alignment = DataGridViewContentAlignment.BottomRight;
                }
            }

            if (this._graphMain != null)
            {
                this._graphMain.Dispose();
            }
        }
        #endregion

        #region データファイルを開く_Click

        /// <summary><para>method outline:</para>
        /// <para>データファイルを開くボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFOpenData_Click(object sender, EventArgs e)
        {
            DataFileOpen f = new DataFileOpen(this._commonData);

            if (f.ShowDialog() == DialogResult.OK)
            {
                Cursor.Current = Cursors.WaitCursor;
                // フォームデータ初期化
                this.clearFormData();
                // フォームデータ設定
                this.setFormData();

                this.resetGraph();

                Cursor.Current = Cursors.Default;
            }
            f.Dispose();

        }
        #endregion

        #region プロジェクトファイルを開く_Click

        /// <summary><para>method outline:</para>
        /// <para>プロジェクトファイルを開くボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFOpenProject_Click(object sender, EventArgs e)
        {
            bool unsaveFlg = false;
            for (int i = 0; i < this._commonData.FileList.Count; i++)
            {
                if (this._commonData.FileList[i].UnsaveFlg == true)
                {
                    unsaveFlg = true;
                }
            }
            if (unsaveFlg)
            {
                DialogResult result;
                // メッセージ:保存していないデータは破棄されます。よろしいですか？
                result = MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_WARNING_UNSAVED_DATA_DISCARDED,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_OPEN_PROJECT_FILE,
                        MessageBoxButtons.OKCancel, MessageBoxIcon.Exclamation);
                if (result != DialogResult.OK)
                {
                    return;
                }
            }

            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                Cursor.Current = Cursors.WaitCursor;
                try
                {
                    // デシリアライズ
                    XmlSerializer deserializer = new XmlSerializer(typeof(CommonData));
                    this._fs = new FileStream(openFileDialog.FileName, FileMode.Open);
                    this._commonData = (CommonData)deserializer.Deserialize(this._fs);

                    // プロジェクトファイル名をセット
                    this._commonData.ProjectFileNm = openFileDialog.FileName;

                    // フォームデータ初期化
                    this.clearFormData();
                    // フォームデータ設定
                    this.setFormData();

                    // 上書保存使用可
                    mnuFOverwriteProject.Enabled = true;

                    this.resetGraph();

                }
                catch
                {
                    // エラーメッセージを表示
                    // メッセージ:プロジェクトファイルの読込みに失敗しました。
                    MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_FAILED_TO_READ_PROJECT_FILE,
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_OPEN_PROJECT_FILE,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);

                }
                finally
                {
                    if (this._fs != null)
                    {
                        this._fs.Close();
                    }
                }
                Cursor.Current = Cursors.Default;
            }
            openFileDialog.Dispose();
        }
        #endregion

        #region 終了_Click

        /// <summary><para>method outline:</para>
        /// <para>終了ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFEnd_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        #endregion

        #region フォームを閉じるときの処理

        /// <summary><para>method outline:</para>
        /// <para>フォームクローズ時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void DataEdit_FormClosing(object sender, FormClosingEventArgs e)
        {
            bool unsaveFlg = false;
            for (int i = 0; i < this._commonData.FileList.Count; i++)
            {
                if (this._commonData.FileList[i].UnsaveFlg == true)
                {
                    unsaveFlg = true;
                }
            }
            if (unsaveFlg)
            {
                DialogResult result;
                // メッセージ:保存してないデータを保存しますか？
                result = MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_INFO_SAVE_UNSAVED_DATA,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_EXIT,
                        MessageBoxButtons.YesNo, MessageBoxIcon.Exclamation);
                if (result == DialogResult.Yes)
                {
                    Boolean success = true;
                    string projectFileNm = this._commonData.ProjectFileNm;
                    if (projectFileNm != null && projectFileNm != "")
                    {
                        // 上書き保存
                        this._fs = new FileStream(projectFileNm, FileMode.Truncate);
                    }
                    else
                    {
                        // 名前を付けて保存
                        saveFileDialog.Filter = Properties.HySAddinLsor2VieweEXEResources.FILE_DLG_FILTER_LSOR;
                        if (saveFileDialog.ShowDialog() == DialogResult.OK)
                        {
                            projectFileNm = saveFileDialog.FileName;
                            File.Delete(projectFileNm);   // 例外は発生しない
                            this._fs = new FileStream(projectFileNm, FileMode.OpenOrCreate);
                        }
                        else
                        {
                            success = false;
                        }
                        saveFileDialog.Dispose();
                    }
                    if (success)
                    {
                        try
                        {
                            // シリアライズ
                            XmlSerializer serializer = new XmlSerializer(typeof(CommonData));
                            serializer.Serialize(this._fs, this._commonData);
                            // 上書保存使用可
                            mnuFOverwriteProject.Enabled = true;
                            // メッセージ:プロジェクトファイルを保存しました。
                            MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_INFO_PROJECT_FILE_SAVED,
                                Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_PROJECT_FILE,
                                MessageBoxButtons.OK, MessageBoxIcon.Information);
                        }
                        catch
                        {
                            // メッセージ:プロジェクトファイルの保存に失敗しました。
                            MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_FAILED_TO_SAVE_PROJECT_FILE,
                                Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_PROJECT_FILE,
                                MessageBoxButtons.OK, MessageBoxIcon.Error);
                            success = false;
                        }
                        finally
                        {
                            if (this._fs != null)
                            {
                                this._fs.Close();
                            }
                        }
                    }
                    if (success == false)
                    {
                        // 保存失敗の場合は終了しない
                        e.Cancel = true;
                    }
                }
            }

        }

        #endregion

        #region 行の挿入_Click

        /// <summary><para>method outline:</para>
        /// <para>行の挿入ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuERowInsert_Click(object sender, EventArgs e)
        {
            DataRowAdd f = new DataRowAdd(this._commonData);
            if (f.ShowDialog() == DialogResult.OK)
            {
                Cursor.Current = Cursors.WaitCursor;

                // 河川名の列インデックスを取得
                int idxRiverNm = Util.GetColumnIndex(this._commonData.FileList[0], CommonData.KEY_RIVER_NM);
                if (idxRiverNm < 0)
                {
                    // メッセージ:キー項目に「{0}」が見つかりません。
                    MessageBox.Show(string.Format(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_FOUND_IN_KEY_FIELD, CommonData.KEY_RIVER_NM),
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_ADDITION_ROW,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    f.Dispose();
                    return;
                }

                // TopoIDの列インデックスを取得
                int idxTopoId = Util.GetColumnIndex(this._commonData.FileList[0], CommonData.KEY_TOPO_ID);
                if (idxTopoId < 0)
                {
                    // メッセージ:キー項目に「{0}」が見つかりません。
                    MessageBox.Show(string.Format(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_FOUND_IN_KEY_FIELD, CommonData.KEY_TOPO_ID),
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_ADDITION_ROW,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    f.Dispose();
                    return;
                }

                // 累加距離の列インデックスを取得
                int idxSumDistance = Util.GetColumnIndex(this._commonData.FileList[0], CommonData.KEY_SUM_DISTANCE);
                if (idxSumDistance < 0)
                {
                    // メッセージ:キー項目に「{0}」が見つかりません。
                    MessageBox.Show(string.Format(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_FOUND_IN_KEY_FIELD, CommonData.KEY_SUM_DISTANCE),
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_ADDITION_ROW,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    f.Dispose();
                    return;
                }

                // 断面Noの列インデックスを取得
                int idxSliceNo = Util.GetColumnIndex(this._commonData.FileList[0], CommonData.KEY_SLICE_NO);
                if (idxSliceNo < 0)
                {
                    // メッセージ:キー項目に「{0}」が見つかりません。
                    MessageBox.Show(string.Format(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_FOUND_IN_KEY_FIELD, CommonData.KEY_SLICE_NO),
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_ADDITION_ROW,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    f.Dispose();
                    return;
                }

                string topoId = "";
                for (int i = 0; i < this._commonData.RiverList.Count; i++)
                {
                    if (f.RiverNm == this._commonData.RiverList[i].RiverNm)
                    {
                        topoId = this._commonData.RiverList[i].TopoId;
                    }
                }

                // CSVデータ加工
                for (int i = 0; i < this._commonData.FileList.Count; i++)
                {
                    if (this._commonData.FileList[i].FileNm != null)
                    {
                        this._commonData.FileList[i].UnsaveFlg = true;

                        ArrayList newCsvData = new ArrayList();
                        ArrayList orgCsvData = this._commonData.FileList[i].CsvData;
                        bool insFlg = false;    // 挿入済みフラグ
                        bool rvrFlg = false;    // 河川名マッチフラグ

                        // 全行ループ
                        for (int j = 0; j < orgCsvData.Count + 1; j++)
                        {
                            newCsvData.Add(new string[this._commonData.FileList[i].ColumnList.Count]); // 列を用意

                            if (insFlg)
                            {
                                // 既に挿入済みなので後は1行ずらしてコピーするだけ
                                newCsvData[j] = orgCsvData[j - 1];
                            }
                            else if (
                                (rvrFlg && j == orgCsvData.Count)
                                ||
                                (rvrFlg && f.RiverNm != ((string[])orgCsvData[j])[idxRiverNm])
                                ||
                                (f.RiverNm == ((string[])orgCsvData[j])[idxRiverNm]
                                && double.Parse(f.SumDistance) < double.Parse(((string[])orgCsvData[j])[idxSumDistance])))
                            {
                                // 河川名が等しく累加距離が次の行より小さいので新規行の挿入
                                for (int k = 0; k < this._commonData.FileList[i].ColumnList.Count; k++)
                                {
                                    if (k == idxRiverNm)
                                    {
                                        ((string[])newCsvData[j])[k] = f.RiverNm;
                                    }
                                    else if (k == idxTopoId)
                                    {
                                        ((string[])newCsvData[j])[k] = topoId;
                                    }
                                    else if (k == idxSliceNo)
                                    {
                                        ((string[])newCsvData[j])[k] = f.SliceNo;
                                    }
                                    else if (k == idxSumDistance)
                                    {
                                        ((string[])newCsvData[j])[k] = f.SumDistance;
                                    }
                                    else
                                    {
                                        ((string[])newCsvData[j])[k] = "";
                                    }
                                }
                                insFlg = true;
                            }
                            else
                            {
                                // 該当行より前の行(河川名,累加距離で判断)なのでそのままコピーする
                                newCsvData[j] = orgCsvData[j];
                                if (f.RiverNm == ((string[])orgCsvData[j])[idxRiverNm])
                                {
                                    rvrFlg = true;
                                }
                            }
                        }
                        this._commonData.FileList[i].CsvData = newCsvData;
                    }
                }

                // フォームデータ初期化
                this.clearFormData();
                // フォームデータ設定
                this.setFormData();

                this.mnuERowDelete.Enabled = true;
                this.cmnuERowDelete.Enabled = true;

                Cursor.Current = Cursors.Default;
            }
            f.Dispose();
        }
        #endregion

        #region 行の削除_Click

        /// <summary><para>method outline:</para>
        /// <para>行の削除ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuERowDelete_Click(object sender, EventArgs e)
        {
            // 削除確認ダイアログを表示
            DialogResult result;

            // メッセージ:削除してもよろしいですか？
            result = MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_WARNING_CONFIRM_DELETE,
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_DELETION_ROW,
                        MessageBoxButtons.OKCancel, MessageBoxIcon.Question);

            if (result == DialogResult.OK)
            {
                Cursor.Current = Cursors.WaitCursor;
                // ファイルインデックス取得
                int fileIdx = Util.GetFileIndex(this._commonData.FileList,
                this.tabMain.SelectedTab.Text.Replace("*", ""));

                // 削除場所のインデックスを取得
                List<int> rowIdxs = this.getSelectedRows(this._dgvData[fileIdx]);

                // 削除場所のキー作成
                ArrayList keyList = new ArrayList();
                for (int i = 0; i < rowIdxs.Count; i++)
                {
                    string key =
                        this._dgvData[fileIdx][0, rowIdxs[i]].Value + "," +
                        this._dgvData[fileIdx][1, rowIdxs[i]].Value + "," +
                        this._dgvData[fileIdx][2, rowIdxs[i]].Value + "," +
                        this._dgvData[fileIdx][3, rowIdxs[i]].Value;
                    keyList.Add(key);
                }

                // CSVデータ加工
                for (int i = 0; i < this._commonData.FileList.Count; i++)
                {
                    if (this._commonData.FileList[i].FileNm != null)
                    {
                        this._commonData.FileList[i].UnsaveFlg = true;
                        ArrayList csvData = new ArrayList();
                        // 全行ループ
                        for (int j = 0; j < this._commonData.FileList[i].CsvData.Count; j++)
                        {
                            string key =
                                ((string[])this._commonData.FileList[i].CsvData[j])[0] + "," +
                                ((string[])this._commonData.FileList[i].CsvData[j])[1] + "," +
                                ((string[])this._commonData.FileList[i].CsvData[j])[2] + "," +
                                ((string[])this._commonData.FileList[i].CsvData[j])[3];
                            if (keyList.IndexOf(key) < 0)
                            {
                                csvData.Add(this._commonData.FileList[i].CsvData[j]);
                            }
                        }
                        this._commonData.FileList[i].CsvData = csvData;
                    }
                }
                if (this._commonData.FileList[0].CsvData.Count == 0)
                {
                    this.mnuERowDelete.Enabled = false;
                    this.cmnuERowDelete.Enabled = false;
                }

                // フォームデータ初期化
                this.clearFormData();
                // フォームデータ設定
                this.setFormData();

                Cursor.Current = Cursors.Default;
            }
        }
        #endregion

        #region 列の挿入_Click

        /// <summary><para>method outline:</para>
        /// <para>列の挿入ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuEColInsert_Click(object sender, EventArgs e)
        {
            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
            this.tabMain.SelectedTab.Text.Replace("*", ""));

            DataColAdd f = new DataColAdd(this._commonData, fileIdx);

            f.ColumnNm = this._colAddColumnNm;
            f.DataType = this._colAddDataType;
            f.UnitType = this._colAddUnitType;
            f.Mode = this._colAddMode;

            if (f.ShowDialog() == DialogResult.OK)
            {


                ColumnData columnData = new ColumnData();

                columnData.ColumnNm = f.ColumnNm;
                columnData.DataType = f.DataType;
                columnData.UnitType = f.UnitType;

                ArrayList csvData = new ArrayList();

                if (f.Mode == "ins")
                {
                    // 列の挿入

                    // 挿入場所のインデックスを取得
                    int colIdx = this._dgvData[fileIdx].CurrentCell.ColumnIndex;

                    // キー項目であれば挿入不可            
                    if (colIdx < HEADER_COLUMNS)
                    {
                        this._colAddColumnNm = f.ColumnNm;
                        this._colAddDataType = f.DataType;
                        this._colAddUnitType = f.UnitType;
                        this._colAddMode = f.Mode;

                        // メッセージ:キー項目には挿入できません。
                        MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_INSERT_KEY_FIELD,
                            Properties.HySAddinLsor2VieweEXEResources.CAPTION_INSERT_COLUMN,
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                        f.Dispose();
                        this.mnuEColInsert_Click(sender, e);

                        return;
                    }


                    // 列の挿入
                    this._commonData.FileList[fileIdx].ColumnList.Insert(colIdx, columnData);

                    // CSVデータ加工
                    for (int i = 0; i < this._commonData.FileList[fileIdx].CsvData.Count; i++)
                    {
                        csvData.Add(new string[this._commonData.FileList[fileIdx].ColumnList.Count]); // 列を用意
                        bool insFlg = false;
                        for (int j = 0; j < this._commonData.FileList[fileIdx].ColumnList.Count; j++)
                        {
                            if (j == colIdx)
                            {
                                ((string[])csvData[i])[j] = "";
                                insFlg = true;
                            }
                            else
                            {
                                if (insFlg)
                                {
                                    ((string[])csvData[i])[j] = ((string[])this._commonData.FileList[fileIdx].CsvData[i])[j - 1];
                                }
                                else
                                {
                                    ((string[])csvData[i])[j] = ((string[])this._commonData.FileList[fileIdx].CsvData[i])[j];
                                }
                            }
                        }
                    }
                }
                else
                {
                    // 列の追加
                    this._commonData.FileList[fileIdx].ColumnList.Add(columnData);

                    // CSVデータ加工
                    for (int i = 0; i < this._commonData.FileList[fileIdx].CsvData.Count; i++)
                    {
                        csvData.Add(new string[this._commonData.FileList[fileIdx].ColumnList.Count]); // 列を用意
                        for (int j = 0; j < ((string[])this._commonData.FileList[fileIdx].CsvData[i]).Length; j++)
                        {
                            ((string[])csvData[i])[j] = ((string[])this._commonData.FileList[fileIdx].CsvData[i])[j];
                        }
                        ((string[])csvData[i])[this._commonData.FileList[fileIdx].ColumnList.Count - 1] = "";
                    }
                }
                this._colAddColumnNm = "";
                this._colAddDataType = "";
                this._colAddUnitType = "";
                this._colAddMode = "";

                this._commonData.FileList[fileIdx].CsvData = csvData;
                this._commonData.FileList[fileIdx].UnsaveFlg = true;
                this._tabPage[fileIdx].Text = this._tabPage[fileIdx].Text.Replace("*", "") + "*";

                this.resetGridData(fileIdx);

                //                this.resetGraph();

            }
            f.Dispose();
        }
        #endregion

        #region 列の削除_Click

        /// <summary><para>method outline:</para>
        /// <para>列の削除ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuEColDelete_Click(object sender, EventArgs e)
        {

            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
            this.tabMain.SelectedTab.Text.Replace("*", ""));

            // 削除場所のインデックスを取得
            List<int> colIdxs = this.getSelectedColumns(this._dgvData[fileIdx]);
            if (colIdxs == null || colIdxs.Count <= 0)
            {
                return;
            }

            // キー項目を含んでいれば削除不可(行ヘッダ選択された場合も除外できる)            
            if (colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_RIVER_NM))
                || colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_TOPO_ID))
                || colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_SLICE_NO))
                || colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_SUM_DISTANCE)))
            {
                // メッセージ:キー項目は削除できません。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_DELETE_KEY_FIELD,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_DELETION_COLUMN,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            // 削除確認ダイアログを表示
            DialogResult result;

            // メッセージ:削除してもよろしいですか？
            result = MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_WARNING_CONFIRM_DELETE,
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_DELETION_COLUMN,
                        MessageBoxButtons.OKCancel, MessageBoxIcon.Question);

            if (result == DialogResult.OK)
            {

                // CSVデータ加工
                ArrayList csvData = new ArrayList();

                int colCnt = this._commonData.FileList[fileIdx].ColumnList.Count;
                for (int i = 0; i < this._commonData.FileList[fileIdx].CsvData.Count; i++)
                {
                    csvData.Add(new string[colCnt - colIdxs.Count]);
                    int target = 0;
                    for (int j = 0; j < colCnt; j++)
                    {
                        if (!colIdxs.Contains(j))
                        {
                            ((string[])csvData[i])[target++] = ((string[])this._commonData.FileList[fileIdx].CsvData[i])[j];
                        }
                        else if (i == 0)
                        {
                            // 項目リストから削除
                            this._commonData.FileList[fileIdx].ColumnList.RemoveAt(target);
                        }
                    }
                }

                this._commonData.FileList[fileIdx].CsvData = csvData;
                this._commonData.FileList[fileIdx].UnsaveFlg = true;
                this._tabPage[fileIdx].Text = this._tabPage[fileIdx].Text.Replace("*", "") + "*";

                this.resetGridData(fileIdx);

                //                this.resetGraph();
            }

        }
        #endregion

        #region 列の変更_Click

        /// <summary><para>method outline:</para>
        /// <para>列の変更ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuEColChange_Click(object sender, EventArgs e)
        {
            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
            this.tabMain.SelectedTab.Text.Replace("*", ""));

            List<int> colIdxs = this.getSelectedColumns(_dgvData[fileIdx]);
            if (colIdxs.Count <= 0)
            {
                // メッセージ:セルを選択してください。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_CHOOSE_CELLS,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_CHANGE_COLUMN,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            else if (colIdxs.Count > 1)
            {
                // メッセージ:複数の列を同時に変更することはできません。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_CHANGE_MULTIPLE_COLUMNS_SIMULTANEOUSLY,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_CHANGE_COLUMN,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            if (colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_RIVER_NM))
                || colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_TOPO_ID))
                || colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_SLICE_NO))
                || colIdxs.Contains(Util.GetColumnIndex(this._commonData.FileList[fileIdx], CommonData.KEY_SUM_DISTANCE)))
            {
                // メッセージ:キー項目は変更できません。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_CHANGE_KEY_FIELD,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_CHANGE_COLUMN,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            DataColChg f = new DataColChg(this._commonData, fileIdx, colIdxs[0]);

            f.ColumnNm = this._commonData.FileList[fileIdx].ColumnList[colIdxs[0]].ColumnNm;
            f.DataType = this._commonData.FileList[fileIdx].ColumnList[colIdxs[0]].DataType;
            f.UnitType = this._commonData.FileList[fileIdx].ColumnList[colIdxs[0]].UnitType;

            if (f.ShowDialog() == DialogResult.OK)
            {
                this._commonData.FileList[fileIdx].ColumnList[colIdxs[0]].ColumnNm = f.ColumnNm;
                this._commonData.FileList[fileIdx].ColumnList[colIdxs[0]].DataType = f.DataType;
                this._commonData.FileList[fileIdx].ColumnList[colIdxs[0]].UnitType = f.UnitType;

                this._commonData.FileList[fileIdx].UnsaveFlg = true;
                this._tabPage[fileIdx].Text = this._tabPage[fileIdx].Text.Replace("*", "") + "*";

                this.resetGridData(fileIdx);

                //                this.resetGraph();

            }
            f.Dispose();

            // エラーセルの強調表示
            checkData(fileIdx, colIdxs[0], -1);
        }
        #endregion

        #region グラフウィンドウ_Click

        /// <summary><para>method outline:</para>
        /// <para>グラフウィンドウボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuDGraph_Click(object sender, EventArgs e)
        {
            if (this._graphMain == null || this._graphMain.IsDisposed)
            {
                this._graphMain = new GraphMain(this._commonData);
                this._graphMain.Show();
            }
            else if (this._graphMain.Visible == false)
            {
                this._graphMain.Show();
            }
            else
            {
                this._graphMain.Activate();
            }
        }
        #endregion

        #region データファイルの上書き保存_Click

        /// <summary><para>method outline:</para>
        /// <para>データファイルの上書き保存ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFOverwriteData_Click(object sender, EventArgs e)
        {
            // 名前を付けて保存をしてない場合は以下のメッセージを表示
            // 名前を付けて保存をしている場合は黙ってそのファイルに上書き
            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
            this.tabMain.SelectedTab.Text.Replace("*", ""));

            if (this._commonData.FileList[fileIdx].NewFileNm == "")
            {

                // 上書確認ダイアログを表示
                DialogResult result;

                // メッセージ:元のファイルに上書きしてよろしいですか？
                result = MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_WARNING_OVERWRITE_ORIGINAL_FILE,
                            Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_DATA_FILE,
                            MessageBoxButtons.OKCancel, MessageBoxIcon.Question);

                if (result == DialogResult.OK)
                {
                    // 保存
                    bool ret = this.saveDataFile(this._commonData.FileList[fileIdx].FileNm);
                }
            }
            else
            {
                bool ret = this.saveDataFile(this._commonData.FileList[fileIdx].NewFileNm);
            }
        }
        #endregion

        #region プロジェクトファイルの上書き保存_Click

        /// <summary><para>method outline:</para>
        /// <para>プロジェクトファイルの上書き保存ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFOverwriteProject_Click(object sender, EventArgs e)
        {
            try
            {
                // シリアライズ
                XmlSerializer serializer = new XmlSerializer(typeof(CommonData));
                this._fs = new FileStream(this._commonData.ProjectFileNm, FileMode.Truncate);
                serializer.Serialize(this._fs, this._commonData);
                // メッセージ:プロジェクトファイルを保存しました。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_INFO_PROJECT_FILE_SAVED,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_PROJECT_FILE,
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
            catch
            {
                // メッセージ:プロジェクトファイルの保存に失敗しました。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_FAILED_TO_SAVE_PROJECT_FILE,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_PROJECT_FILE,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                if (this._fs != null)
                {
                    this._fs.Close();
                }
            }
        }
        #endregion

        #region データファイルの名前を付けて保存_Click

        /// <summary><para>method outline:</para>
        /// <para>データファイルの名前を付けて保存ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFAppendData_Click(object sender, EventArgs e)
        {
            saveFileDialog.Filter = Properties.HySAddinLsor2VieweEXEResources.FILE_DLG_FILTER_CSV_ALL;
            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                // 保存
                bool ret = this.saveDataFile(saveFileDialog.FileName);
                if (ret)
                {
                    // ファイルインデックス取得
                    int fileIdx = Util.GetFileIndex(this._commonData.FileList,
                    this.tabMain.SelectedTab.Text.Replace("*", ""));

                    this._commonData.FileList[fileIdx].NewFileNm = saveFileDialog.FileName;
                }
            }
            saveFileDialog.Dispose();
        }
        #endregion

        #region プロジェクトファイルの名前を付けて保存_Click

        /// <summary><para>method outline:</para>
        /// <para>プロジェクトファイルの名前を付けて保存ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFAppendProject_Click(object sender, EventArgs e)
        {
            saveFileDialog.Filter = Properties.HySAddinLsor2VieweEXEResources.FILE_DLG_FILTER_LSOR;
            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                try
                {
                    // シリアライズ
                    XmlSerializer serializer = new XmlSerializer(typeof(CommonData));
                    File.Delete(saveFileDialog.FileName);   // 例外は発生しない
                    this._fs = new FileStream(saveFileDialog.FileName, FileMode.OpenOrCreate);
                    serializer.Serialize(this._fs, this._commonData);
                    this._commonData.ProjectFileNm = saveFileDialog.FileName;
                    // 上書保存使用可
                    mnuFOverwriteProject.Enabled = true;
                    // メッセージ:プロジェクトファイルを保存しました。
                    MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_INFO_PROJECT_FILE_SAVED,
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_PROJECT_FILE,
                        MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
                catch
                {
                    // メッセージ:プロジェクトファイルの保存に失敗しました。
                    MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_FAILED_TO_SAVE_PROJECT_FILE,
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_PROJECT_FILE,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                finally
                {
                    if (this._fs != null)
                    {
                        this._fs.Close();
                    }
                }
            }
            saveFileDialog.Dispose();
        }
        #endregion

        #region バージョン情報_Click

        /// <summary><para>method outline:</para>
        /// <para>バージョン情報ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuFVersion_Click(object sender, EventArgs e)
        {
            Version f = new Version();
            f.ShowDialog(this);
            f.Dispose();
        }
        #endregion

        #region グリッド_CellEndEdit

        /// <summary><para>method outline:</para>
        /// <para>選択セルの編集モードが停止した場合の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void dgvData_CellEndEdit(object sender, DataGridViewCellEventArgs e)
        {
            // CellValueChanged へ移動
        }
        #endregion

        #region グリッド_CellValueChanged

        /// <summary><para>method outline:</para>
        /// <para>選択セルの値が変更された場合の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void dgvData_CellValueChanged(object sender, DataGridViewCellEventArgs e)
        {
            if (this._eventCancelFlg)
            {
                return;
            }

            DataGridView dgvData = (DataGridView)sender;

            // ファイルインデックス取得
            int idx = Util.GetFileIndex(this._commonData.FileList,
            this.tabMain.SelectedTab.Text.Replace("*", ""));

            // _commonData書き換え
            if (dgvData[e.ColumnIndex, e.RowIndex].Value != null)
            {
                ((string[])this._commonData.FileList[idx].CsvData[e.RowIndex])[e.ColumnIndex]
                = dgvData[e.ColumnIndex, e.RowIndex].Value.ToString();
            }
            else
            {
                ((string[])this._commonData.FileList[idx].CsvData[e.RowIndex])[e.ColumnIndex] = "";
            }
            /*
                        // _commonData書き換え
                        if (dgvData.CurrentCell.Value != null) {
                            ((string[])this._commonData.FileList[idx].CsvData[e.RowIndex])[e.ColumnIndex]
                            = dgvData.CurrentCell.Value.ToString();
                        } else {
                            ((string[])this._commonData.FileList[idx].CsvData[e.RowIndex])[e.ColumnIndex] = "";
                        }
            */
            this._commonData.FileList[idx].UnsaveFlg = true;
            this._tabPage[idx].Text = this._tabPage[idx].Text.Replace("*", "") + "*";

            // エラーセルの強調表示
            checkData(idx, e.ColumnIndex, e.RowIndex);

            //            this.resetGraph();

        }
        #endregion

        #region 流下能力算定_Click

        /// <summary><para>method outline:</para>
        /// <para>流下能力算定ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuTRyuka_Click(object sender, EventArgs e)
        {
            // 流下能力算定画面
            FlowAbility flowAbility = new FlowAbility(this._commonData);
            flowAbility.ShowDialog(this);

            if (flowAbility.ResultFileNm != "")
            {
                // データファイルに追加する処理

                int fileIdx = 0;
                // 同名ファイルを検索
                bool hit = false;
                for (int i = 0; i < this._commonData.FileList.Count; i++)
                {
                    if (flowAbility.ResultFileNm == this._commonData.FileList[i].FileNm)
                    {
                        hit = true;
                        fileIdx = i;
                        break;
                    }
                }

                if (!hit)
                {
                    // 空きインデックスを検索
                    for (int i = 0; i < this._commonData.FileList.Count; i++)
                    {
                        if (this._commonData.FileList[i].FileNm == null || this._commonData.FileList[i].FileNm == "")
                        {
                            fileIdx = i;
                            break;
                        }
                    }
                }
                // メモリにデータをセット
                this._commonData.FileList[fileIdx].FileNm = flowAbility.ResultFileNm;

                // CSV読込み
                TextFieldParser parser = new TextFieldParser(this._commonData.FileList[fileIdx].FileNm, this._enc);

                using (parser)
                {
                    parser.TextFieldType = FieldType.Delimited;
                    parser.SetDelimiters(",");

                    int rowCnt = 0;

                    ArrayList csvData = new ArrayList();

                    List<ColumnData> columnList = new List<ColumnData>();
                    int columnCnt = 0;
                    ColumnData columnData = new ColumnData();

                    while (!parser.EndOfData)
                    {

                        string[] csvRow = parser.ReadFields();  // 1行リード

                        if (rowCnt == 0)
                        { // 項目名
                            int i = 0;
                            foreach (string field in csvRow)
                            {
                                columnData = new ColumnData();
                                columnData.ColumnNm = field;
                                columnList.Add(columnData);
                                i++;
                            }
                            columnCnt = i;
                        }
                        else if (rowCnt == 1)
                        {   // データ型
                            int i = 0;
                            foreach (string field in csvRow)
                            {
                                columnList[i].DataType = field;
                                i++;
                            }
                        }
                        else if (rowCnt == 2)
                        {   // 単位
                            int i = 0;
                            foreach (string field in csvRow)
                            {
                                columnList[i].UnitType = field;
                                i++;
                            }
                            this._commonData.FileList[fileIdx].ColumnList = columnList;
                        }
                        else
                        {    // データ
                            int i = 0;
                            csvData.Add(new string[columnCnt]); // 列を用意
                            foreach (string field in csvRow)
                            {
                                ((string[])csvData[rowCnt - 3])[i] = field;
                                i++;
                            }

                        }

                        rowCnt++;
                    }
                    this._commonData.FileList[fileIdx].CsvData = csvData;
                }

                // タブを表示
                this._tabPageManager.ChangeTabPageVisible(fileIdx, true);
                // 表示データをセット
                this._lblFileNm[fileIdx].Text = this._commonData.FileList[fileIdx].FileNm;
                this.resetGridData(fileIdx);

            }

            flowAbility.Dispose();
        }
        #endregion

        #region コピー_Click

        /// <summary><para>method outline:</para>
        /// <para>コピーボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuECopy_Click(object sender, EventArgs e)
        {

            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
                this.tabMain.SelectedTab.Text.Replace("*", ""));

            try
            {
                Clipboard.SetDataObject(this._dgvData[fileIdx].GetClipboardContent().GetText());
            }
            catch
            {
            }

        }
        #endregion

        #region すべて選択_Click

        /// <summary><para>method outline:</para>
        /// <para>すべて選択ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuEAllSelect_Click(object sender, EventArgs e)
        {
            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
                this.tabMain.SelectedTab.Text.Replace("*", ""));

            this._dgvData[fileIdx].SelectAll();
            // 一応クリップボードにコピー
            try
            {
                Clipboard.SetDataObject(this._dgvData[fileIdx].GetClipboardContent().GetText());
            }
            catch
            {
            }
        }
        #endregion

        #region 貼り付け_Click

        /// <summary><para>method outline:</para>
        /// <para>貼り付けボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void mnuEPaste_Click(object sender, EventArgs e)
        {
            // ファイルインデックス取得
            int fileIdx = Util.GetFileIndex(this._commonData.FileList,
            this.tabMain.SelectedTab.Text.Replace("*", ""));

            // 貼り付け場所のインデックスを取得
            int colIdx = this._dgvData[fileIdx].CurrentCell.ColumnIndex;

            // キー項目であれば貼り付け不可            
            if (colIdx < HEADER_COLUMNS)
            {
                // メッセージ:キー項目には貼り付けできません。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_NOT_PASTE_KEY_FIELD,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_PASTE,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            this.pasteData();
        }
        #endregion

        #region グリッド_KeyDown

        /// <summary><para>method outline:</para>
        /// <para>コントロールにフォーカスがあるときにキーが押されたときの処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void dgvData_KeyDown(object sender, KeyEventArgs e)
        {
            if ((e.Modifiers & Keys.Control) == Keys.Control && e.KeyCode == Keys.V)
            {
                this.pasteData();
                return;
            }
            if (e.KeyCode == Keys.Delete)
            {
                DataGridView dgvData = (DataGridView)sender;
                // ファイルインデックス取得
                int idx = Util.GetFileIndex(this._commonData.FileList,
                this.tabMain.SelectedTab.Text.Replace("*", ""));

                foreach (DataGridViewCell cell in dgvData.SelectedCells)
                {

                    if (cell.ColumnIndex >= HEADER_COLUMNS)
                    {
                        cell.Value = "";
                        // エラーセルの強調表示
                        checkData(idx, cell.ColumnIndex, cell.RowIndex);
                    }
                }
                return;
            }
        }
        #endregion

        #region 右クリックメニューの処理

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログの行の挿入処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuERowInsert_Click(object sender, EventArgs e)
        {
            this.mnuERowInsert_Click(sender, e);
        }

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログの行の削除処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuERowDelete_Click(object sender, EventArgs e)
        {
            this.mnuERowDelete_Click(sender, e);
        }

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログの列の挿入処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuEColInsert_Click(object sender, EventArgs e)
        {
            this.mnuEColInsert_Click(sender, e);
        }

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログの列の削除処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuEColDelete_Click(object sender, EventArgs e)
        {
            this.mnuEColDelete_Click(sender, e);
        }

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログの列の変更処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuEColChange_Click(object sender, EventArgs e)
        {
            this.mnuEColChange_Click(sender, e);
        }

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログのコピー処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuECopy_Click(object sender, EventArgs e)
        {
            this.mnuECopy_Click(sender, e);
        }

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログの貼り付け処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuEPaste_Click(object sender, EventArgs e)
        {
            this.mnuEPaste_Click(sender, e);
        }

        /// <summary><para>method outline:</para>
        /// <para>右クリック時の選択ダイアログのすべて選択処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmnuEAllSelect_Click(object sender, EventArgs e)
        {
            this.mnuEAllSelect_Click(sender, e);
        }

        #endregion

        #endregion
    }
}
