﻿/// <summary>
/// 流下能力算定画面
/// </summary>
/// <create>2010/02/02</create>
/// <modifier></modifier>
/// <modify></modify>
/// <modification></modification>
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;

namespace lsor
{
    /// <summary><para>class outline:</para>
    /// <para>流下能力算定画面出力制御クラス</para>
    /// </summary>
    /// <remarks><para>history:</para>
    /// <para>[ver 1.0.0][2010/02/02][新規作成]</para>
    /// </remarks>
    public partial class FlowAbility : Form
    {
        #region プライベート変数

        /// <summary>プロジェクト共通データ</summary>
        private CommonData _commonData = new CommonData();

        /// <summary>流下能力算定グリッド現在選択行</summary>
        private int _sourceCurrentRow = 0;

        /// <summary>評価水位グリッド現在選択行</summary>
        private int _targetCurrentRow = 0;

        /// <summary>流下能力算出用クラス</summary>
        private FlowCalculator _flowCalculator;

        /// <summary>データ保存ストリーム</summary>
        private StreamWriter _sw;

        /// <summary>文字エンコーディング</summary>
        private Encoding _enc = Encoding.Default;

        #endregion

        #region プロパティ

        /// <summary><para>method outline:</para>
        /// <para>計算結果のファイル名を設定・取得する</para>
        /// </summary>
        /// <value>計算結果のファイル名</value>
        /// <example><para>usage:</para>
        /// <para>string strFileNm = frmFlowAbility.ResultFileNm()</para>
        /// </example>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public string ResultFileNm { get; set; }

        #endregion

        #region コンストラクタ

        /// <summary><para>method outline:</para>
        /// <para>コンストラクタ</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>FlowAbility csCommonData = new FlowAbility(commonData)</para>
        /// </example>
        /// <param name="commonData">共通データ</param>
        /// <returns>流下能力算定画面出力制御クラス</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        public FlowAbility(CommonData commonData)
        {
            this._commonData = commonData;
            this.ResultFileNm = "";
            InitializeComponent();
        }
        #endregion

        #region プライベートメソッド

        #region 計算結果グリッドのキー列生成

        /// <summary><para>method outline:</para>
        /// <para>計算結果グリッドのキー列を生成する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>createResultKeyCols()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void createResultKeyCols()
        {
            dgvResult.Rows.Clear();
            dgvResult.Columns.Clear();

            // グリッド列を生成
            DataGridViewTextBoxColumn column;

            // 河川名
            column = new DataGridViewTextBoxColumn();
            column.Name = "colResultRiverNm";
            column.HeaderText = CommonData.KEY_RIVER_NM;
            //            column.HeaderText = CommonData.KEY_RIVER_NM + "\r\nchar\r\nnd";
            column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
            column.Resizable = DataGridViewTriState.True;
            column.ReadOnly = true;
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleLeft;
            column.SortMode = DataGridViewColumnSortMode.NotSortable;
            dgvResult.Columns.Add(column);

            // topoID
            column = new DataGridViewTextBoxColumn();
            column.Name = "colResultTopoId";
            column.HeaderText = CommonData.KEY_TOPO_ID;
            //            column.HeaderText = CommonData.KEY_TOPO_ID + "\r\nchar\r\nnd";
            column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
            column.Resizable = DataGridViewTriState.True;
            column.ReadOnly = true;
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleLeft;
            column.SortMode = DataGridViewColumnSortMode.NotSortable;
            dgvResult.Columns.Add(column);

            // 断面No
            column = new DataGridViewTextBoxColumn();
            column.Name = "colResultSliceNo";
            column.HeaderText = CommonData.KEY_SLICE_NO;
            //            column.HeaderText = CommonData.KEY_SLICE_NO + "\r\nchar\r\nnd";
            column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
            column.Resizable = DataGridViewTriState.True;
            column.ReadOnly = true;
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleLeft;
            column.SortMode = DataGridViewColumnSortMode.NotSortable;
            dgvResult.Columns.Add(column);

            // 累加距離
            column = new DataGridViewTextBoxColumn();
            column.Name = "colResultSumDistance";
            column.HeaderText = CommonData.KEY_SUM_DISTANCE;
            //            column.HeaderText = CommonData.KEY_SUM_DISTANCE + "\r\ndouble\r\nnd";
            column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
            column.Resizable = DataGridViewTriState.True;
            column.ReadOnly = true;
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
            column.SortMode = DataGridViewColumnSortMode.NotSortable;
            dgvResult.Columns.Add(column);

            // a
            column = new DataGridViewTextBoxColumn();
            column.Name = "colResultA";
            column.HeaderText = "a";
            //            column.HeaderText = "a\r\ndouble\r\nnd";
            column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
            column.Resizable = DataGridViewTriState.True;
            column.ReadOnly = true;
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
            column.SortMode = DataGridViewColumnSortMode.NotSortable;
            dgvResult.Columns.Add(column);

            // b
            column = new DataGridViewTextBoxColumn();
            column.Name = "colResultB";
            column.HeaderText = "b";
            //            column.HeaderText = "b\r\ndouble\r\nnd";
            column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
            column.Resizable = DataGridViewTriState.True;
            column.ReadOnly = true;
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
            column.SortMode = DataGridViewColumnSortMode.NotSortable;
            dgvResult.Columns.Add(column);

            // r
            column = new DataGridViewTextBoxColumn();
            column.Name = "colResultR";
            column.HeaderText = "r";
            //            column.HeaderText = "r\r\ndouble\r\nnd";
            column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
            column.Resizable = DataGridViewTriState.True;
            column.ReadOnly = true;
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
            column.SortMode = DataGridViewColumnSortMode.NotSortable;
            dgvResult.Columns.Add(column);

        }

        #endregion

        #region 計算タブ初期化

        /// <summary><para>method outline:</para>
        /// <para>計算タブを初期化する</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>initResult()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void initResult()
        {
            dgvResult.Rows.Clear();
            dgvResult.Columns.Clear();
            cmdOutput.Enabled = false;

            // 計算結果グリッドのキー列生成
            this.createResultKeyCols();

            // 流下能力算定用データと評価水位データが揃っていれば計算実施ボタンを使用可
            if (this._commonData.FlowData.FlowSourceList.Count < 2)
            {
                cmdCalc.Enabled = false;
            }
            else
            {
                cmdCalc.Enabled = true;
            }
        }

        #endregion

        #region 流下能力算定用データをグリッドへセット

        /// <summary><para>method outline:</para>
        /// <para>流下能力算定用データをグリッドへセットする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setSourceData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setSourceData()
        {
            dgvSource.Rows.Clear();

            if (this._commonData.FlowData.FlowSourceList.Count > 0)
            {

                for (int row = 0; row < this._commonData.FlowData.FlowSourceList.Count; row++)
                {
                    dgvSource.Rows.Add();
                    dgvSource[0, row].Value = this._commonData.FlowData.FlowSourceList[row].FileId;
                    dgvSource[1, row].Value = this._commonData.FileList[Util.ConvertFileId(this._commonData, this._commonData.FlowData.FlowSourceList[row].FileId)].FileNm;
                    dgvSource[2, row].Value = this._commonData.FlowData.FlowSourceList[row].ColumnNmQ;
                    dgvSource[3, row].Value = this._commonData.FlowData.FlowSourceList[row].ColumnNmH;
                }
                dgvSource.AutoResizeColumn(0);
                dgvSource.AutoResizeColumn(1);
                dgvSource.AutoResizeColumn(2);
                dgvSource.AutoResizeColumn(3);

                cmdSourceDel.Enabled = true;
                cmdSourceUpper.Enabled = true;
                cmdSourceLower.Enabled = true;
            }
            else
            {
                cmdSourceDel.Enabled = false;
                cmdSourceUpper.Enabled = false;
                cmdSourceLower.Enabled = false;
            }
        }

        #endregion

        #region 評価水位データをグリッドへセット

        /// <summary><para>method outline:</para>
        /// <para>評価水位データをグリッドへセットする</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>setTargetData()</para>
        /// </example>
        /// <param name="">無し</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void setTargetData()
        {

            dgvTarget.Rows.Clear();

            if (this._commonData.FlowData.FlowTargetList.Count > 0)
            {
                for (int row = 0; row < this._commonData.FlowData.FlowTargetList.Count; row++)
                {
                    dgvTarget.Rows.Add();
                    dgvTarget[0, row].Value = this._commonData.FlowData.FlowTargetList[row].FileId;
                    dgvTarget[1, row].Value = this._commonData.FileList[Util.ConvertFileId(this._commonData, this._commonData.FlowData.FlowTargetList[row].FileId)].FileNm;
                    dgvTarget[2, row].Value = this._commonData.FlowData.FlowTargetList[row].ColumnNmH;
                }
                dgvTarget.AutoResizeColumn(0);
                dgvTarget.AutoResizeColumn(1);
                dgvTarget.AutoResizeColumn(2);

                cmdTargetDel.Enabled = true;
                cmdTargetUpper.Enabled = true;
                cmdTargetLower.Enabled = true;
            }
            else
            {
                cmdTargetDel.Enabled = false;
                cmdTargetUpper.Enabled = false;
                cmdTargetLower.Enabled = false;
            }
        }

        #endregion

        #region ダブルクォート加工

        /// <summary><para>method outline:</para>
        /// <para>必要に応じて文字列をダブルクォートで囲む</para>
        /// </summary>
        /// <example><para>usage:</para>
        /// <para>string sData = addDoubleQuote(dt)</para>
        /// </example>
        /// <param name="dt">加工を行う文字列</param>
        /// <returns>加工を行った文字列</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private string addDoubleQuote(string dt)
        {
            // "で囲む必要があるか
            if (dt.IndexOf('"') > -1 ||
                dt.IndexOf(',') > -1 ||
                dt.IndexOf('\r') > -1 ||
                dt.IndexOf('\n') > -1 ||
                dt.StartsWith(" ") ||
                dt.StartsWith("\t") ||
                dt.EndsWith(" ") ||
                dt.EndsWith("\t"))
            {
                if (dt.IndexOf('"') > -1)
                {
                    dt = dt.Replace("\"", "\"\"");
                }
                return "\"" + dt + "\"";
            }
            else
            {
                return dt;
            }
        }

        #endregion

        #endregion

        #region イベントハンドラ

        #region フォーム_Load

        /// <summary><para>method outline:</para>
        /// <para>フォームロード時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void FlowAbility_Load(object sender, EventArgs e)
        {
            this.Width = 750;
            this.Height = 350;

            // コントロール初期化
            dgvSource.Rows.Clear();
            dgvTarget.Rows.Clear();

            cmdSourceDel.Enabled = false;
            cmdTargetDel.Enabled = false;

            // ファイルと項目の存在チェックを行い、存在しなければメモリから削除
            for (int i = 0; i < this._commonData.FlowData.FlowSourceList.Count; i++)
            {
                if (!Util.IsExistFileId(this._commonData, this._commonData.FlowData.FlowSourceList[i].FileId))
                {
                    this._commonData.FlowData.FlowSourceList.RemoveAt(i);
                    i = -1;
                    continue;
                }
                else if (Util.GetColumnIndex(this._commonData.FileList[Util.ConvertFileId(this._commonData, this._commonData.FlowData.FlowSourceList[i].FileId)], this._commonData.FlowData.FlowSourceList[i].ColumnNmQ) == -1)
                {
                    this._commonData.FlowData.FlowSourceList.RemoveAt(i);
                    i = -1;
                    continue;
                }
                else if (Util.GetColumnIndex(this._commonData.FileList[Util.ConvertFileId(this._commonData, this._commonData.FlowData.FlowSourceList[i].FileId)], this._commonData.FlowData.FlowSourceList[i].ColumnNmH) == -1)
                {
                    this._commonData.FlowData.FlowSourceList.RemoveAt(i);
                    i = -1;
                    continue;
                }
            }

            for (int i = 0; i < this._commonData.FlowData.FlowTargetList.Count; i++)
            {
                if (!Util.IsExistFileId(this._commonData, this._commonData.FlowData.FlowTargetList[i].FileId))
                {
                    this._commonData.FlowData.FlowTargetList.RemoveAt(i);
                    i = -1;
                    continue;
                }
                else if (Util.GetColumnIndex(this._commonData.FileList[Util.ConvertFileId(this._commonData, this._commonData.FlowData.FlowTargetList[i].FileId)], this._commonData.FlowData.FlowTargetList[i].ColumnNmH) == -1)
                {
                    this._commonData.FlowData.FlowTargetList.RemoveAt(i);
                    i = -1;
                    continue;
                }
            }

            // 流下能力算定用データをセット
            this.setSourceData();

            // 評価水位データをセット
            this.setTargetData();

            // 計算タブ初期化
            this.initResult();

        }

        #endregion

        #region 流下能力算定用データ_追加ボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>流下能力算定用データの追加ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdSourceAdd_Click(object sender, EventArgs e)
        {
            FlowSourceAdd f = new FlowSourceAdd(this._commonData);
            f.ShowDialog(this);
            if (f.FileId.Length > 0 && f.ItemNameQ.Length > 0 && f.ItemNameH.Length > 0)
            {
                FlowFileData flowFileData = new FlowFileData();
                flowFileData.FileId = f.FileId;
                flowFileData.ColumnNmH = f.ItemNameH;
                flowFileData.ColumnNmQ = f.ItemNameQ;
                this._commonData.FlowData.FlowSourceList.Add(flowFileData);

                // 流下能力算定用データをセット
                this.setSourceData();

                dgvSource[0, dgvSource.Rows.Count - 1].Selected = true;

                // 計算タブ初期化
                this.initResult();
            }
            f.Dispose();
        }

        #endregion

        #region 流下能力算定用データ_削除ボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>流下能力算定用データの削除ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdSourceDel_Click(object sender, EventArgs e)
        {

            this._commonData.FlowData.FlowSourceList.RemoveAt(dgvSource.CurrentRow.Index);

            // 流下能力算定用データをセット
            this.setSourceData();

            // 計算タブ初期化
            this.initResult();
        }

        #endregion

        #region 評価水位データ_追加ボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>評価水位データの追加ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdTargetAdd_Click(object sender, EventArgs e)
        {
            FlowTargetAdd f = new FlowTargetAdd(this._commonData);
            f.ShowDialog(this);
            if (f.FileId.Length > 0 && f.ItemName.Length > 0)
            {
                FlowFileData flowFileData = new FlowFileData();
                flowFileData.FileId = f.FileId;
                flowFileData.ColumnNmH = f.ItemName;
                this._commonData.FlowData.FlowTargetList.Add(flowFileData);

                // 評価水位データをセット
                this.setTargetData();

                dgvTarget[0, dgvTarget.Rows.Count - 1].Selected = true;

                // 計算タブ初期化
                this.initResult();
            }
            f.Dispose();
        }

        #endregion

        #region 評価水位データ_削除ボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>評価水位データの削除ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdTargetDel_Click(object sender, EventArgs e)
        {

            this._commonData.FlowData.FlowTargetList.RemoveAt(dgvTarget.CurrentRow.Index);

            // 評価水位データをセット
            this.setTargetData();

            // 計算タブ初期化
            this.initResult();
        }

        #endregion

        #region 閉じるボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>閉じるボタンをクリック時処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdClose_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        #endregion

        #region 計算タブ_Enter

        /// <summary><para>method outline:</para>
        /// <para>計算タブが選択されたときの処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void tabPage2_Enter(object sender, EventArgs e)
        {
            // 計算ボタンが使用不可の場合はその理由を表示
            if (!cmdCalc.Enabled)
            {

                if (this._commonData.FlowData.FlowSourceList.Count < 2)
                {
                    // メッセージ:流下能力算定用データは２つ以上のファイルが必要です。
                    lblMessage.Text = Properties.HySAddinLsor2VieweEXEResources.MSG_WARNING_DISCHARGE_CAPACITY_REQUIRE_AT_LEAST_TWO_FILES;
                }

            }
            else
            {
                lblMessage.Text = "";
            }
        }

        #endregion

        #region 計算タブ_Leave

        /// <summary><para>method outline:</para>
        /// <para>計算タブが未選択になったときの処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void tabPage2_Leave(object sender, EventArgs e)
        {
            lblMessage.Text = "";
        }

        #endregion

        #region 計算実施ボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>計算実施ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdCalc_Click(object sender, EventArgs e)
        {

            // 計算タブ初期化
            this.initResult();

            // 流下能力算出用クラス
            this._flowCalculator = new FlowCalculator(this._commonData);

            DataGridViewTextBoxColumn column;

            for (int i = 0; i < this._commonData.FlowData.FlowTargetList.Count; i++)
            {
                // dgvResultにカラム追加
                column = new DataGridViewTextBoxColumn();
                column.Name = "colH" + (i + 1).ToString();
                column.HeaderText = "H" + (i + 1).ToString();
                column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
                column.Resizable = DataGridViewTriState.True;
                column.ReadOnly = true;
                column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
                column.SortMode = DataGridViewColumnSortMode.NotSortable;
                dgvResult.Columns.Add(column);

                column = new DataGridViewTextBoxColumn();
                column.Name = "colQ" + (i + 1).ToString();
                column.HeaderText = "Q" + (i + 1).ToString();
                column.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
                column.Resizable = DataGridViewTriState.True;
                column.ReadOnly = true;
                column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
                column.SortMode = DataGridViewColumnSortMode.NotSortable;
                dgvResult.Columns.Add(column);
            }

            // abr計算実行
            if (this._flowCalculator.ExecCalcABR())
            {
                try
                {
                    // dgvResultにデータをセット
                    dgvResult.Rows.Add(this._flowCalculator.KeyList.Count);
                    for (int i = 0; i < this._flowCalculator.KeyList.Count; i++)
                    {
                        dgvResult[0, i].Value = this._flowCalculator.KeyList[i].RiverNm;
                        dgvResult[1, i].Value = this._flowCalculator.KeyList[i].TopoId;
                        dgvResult[2, i].Value = this._flowCalculator.KeyList[i].SliceNo;
                        dgvResult[3, i].Value = this._flowCalculator.KeyList[i].SumDistance;
                        dgvResult[4, i].Value = String.Format("{0:#,##0.00000}", this._flowCalculator.KeyList[i].resultA);
                        dgvResult[5, i].Value = String.Format("{0:#,##0.00000}", this._flowCalculator.KeyList[i].resultB);
                        dgvResult[6, i].Value = String.Format("{0:#,##0.00000}", this._flowCalculator.KeyList[i].resultR);
                    }

                    for (int j = 0; j < this._commonData.FlowData.FlowTargetList.Count; j++)
                    {
                        string fileId = this._commonData.FlowData.FlowTargetList[j].FileId;
                        string itemName = this._commonData.FlowData.FlowTargetList[j].ColumnNmH;
                        string[] csvData = Util.GetData(this._commonData, fileId, itemName);
                        for (int i = 0; i < this._flowCalculator.KeyList.Count; i++)
                        {
                            try
                            {
                                dgvResult[j * 2 + 7, i].Value = csvData[i];

                                // Q = a(H + b)^2
                                double H = double.Parse(csvData[i]);
                                double a = this._flowCalculator.KeyList[i].resultA;
                                double b = this._flowCalculator.KeyList[i].resultB;
                                dgvResult[j * 2 + 8, i].Value = String.Format("{0:#,##0.000}",
                                    Math.Round(a * (H + b) * (H + b), 5));
                            }
                            catch
                            {
                            }
                        }

                    }

                    dgvResult.AutoResizeColumns();

                    // ファイル出力ボタン使用可
                    cmdOutput.Enabled = true;
                }
                catch
                {
                    dgvResult.Rows.Clear();
                    // エラーメッセージ表示
                    // メッセージ:流下能力算定ができませんでした。
                    MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_DISCHARGE_CAPACITY_NOT_CALCULATED,
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_DISCHARGE_CAPACITY_CALCULATE_ERROR,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            else
            {
                // エラーメッセージ表示
                // メッセージ:流下能力算定ができませんでした。
                MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_DISCHARGE_CAPACITY_NOT_CALCULATED,
                    Properties.HySAddinLsor2VieweEXEResources.CAPTION_DISCHARGE_CAPACITY_CALCULATE_ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);

            }

        }

        #endregion

        #region 計算結果のファイル出力ボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>計算結果のファイル出力ボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdOutput_Click(object sender, EventArgs e)
        {
            saveFileDialog.Filter = Properties.HySAddinLsor2VieweEXEResources.FILE_DLG_FILTER_CSV_ALL;
            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                try
                {
                    // 保存処理
                    this._sw = new StreamWriter(saveFileDialog.FileName, false, this._enc);

                    int lastColIndex = dgvResult.Columns.Count - 1;

                    // 項目名
                    for (int i = 0; i < dgvResult.Columns.Count; i++)
                    {
                        string dt = dgvResult.Columns[i].HeaderText.ToString();
                        // データを書き込む
                        dt = this.addDoubleQuote(dt);
                        this._sw.Write(dt);
                        // カンマを書き込む
                        if (lastColIndex > i)
                        {
                            this._sw.Write(',');
                        }
                    }
                    // 改行する
                    this._sw.Write("\r\n");

                    // データ型
                    for (int i = 0; i < dgvResult.Columns.Count; i++)
                    {
                        string dt = "";
                        switch (i)
                        {
                            case 0:
                            case 1:
                            case 2:
                                dt = "char";
                                break;
                            default:
                                dt = "double";
                                break;
                        }
                        // データを書き込む
                        dt = this.addDoubleQuote(dt);
                        this._sw.Write(dt);
                        // カンマを書き込む
                        if (lastColIndex > i)
                        {
                            this._sw.Write(',');
                        }
                    }
                    // 改行する
                    this._sw.Write("\r\n");

                    // 単位
                    for (int i = 0; i < dgvResult.Columns.Count; i++)
                    {
                        string dt = "";
                        switch (i)
                        {
                            case 0:
                            case 1:
                            case 2:
                            case 4:
                            case 5:
                            case 6:
                                dt = "nd";
                                break;
                            case 3:
                                dt = "m";
                                break;
                            default:
                                if (dgvResult.Columns[i].HeaderText.ToString().IndexOf("H") >= 0)
                                {
                                    dt = "m";
                                }
                                else if (dgvResult.Columns[i].HeaderText.ToString().IndexOf("Q") >= 0)
                                {
                                    dt = "m3/s";
                                }
                                else
                                {
                                    dt = "nd";
                                }
                                break;
                        }
                        // データを書き込む
                        dt = this.addDoubleQuote(dt);
                        this._sw.Write(dt);
                        // カンマを書き込む
                        if (lastColIndex > i)
                        {
                            this._sw.Write(',');
                        }
                    }
                    // 改行する
                    this._sw.Write("\r\n");

                    // データ
                    // CSVデータ加工
                    for (int i = 0; i < dgvResult.Rows.Count; i++)
                    {
                        for (int j = 0; j < dgvResult.Columns.Count; j++)
                        {
                            string dt = dgvResult[j, i].Value.ToString();
                            // データを書き込む
                            dt = this.addDoubleQuote(dt);
                            this._sw.Write(dt);
                            // カンマを書き込む
                            if (lastColIndex > j)
                            {
                                this._sw.Write(',');
                            }
                        }
                        // 改行する
                        this._sw.Write("\r\n");
                    }

                    DialogResult result;
                    // メッセージ:保存したファイルをデータファイルへ追加しますか？
                    result = MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_INFO_ADD_SAVED_FILE_TO_DATA_FILES,
                            Properties.HySAddinLsor2VieweEXEResources.CAPTION_CONFIRM,
                            MessageBoxButtons.YesNo, MessageBoxIcon.Information);
                    if (result == DialogResult.Yes)
                    {
                        this.ResultFileNm = saveFileDialog.FileName;
                        this.Close();
                    }
                    else
                    {
                        this.ResultFileNm = "";
                    }

                }
                catch
                {
                    // メッセージ:データファイルの保存に失敗しました。
                    MessageBox.Show(Properties.HySAddinLsor2VieweEXEResources.MSG_ERROR_FAILED_TO_SAVE_DATA_FILE,
                        Properties.HySAddinLsor2VieweEXEResources.CAPTION_SAVE_DATA_FILE,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    this._flowCalculator.SaveFileNm = "";
                }
                finally
                {
                    if (this._sw != null)
                    {
                        this._sw.Close();
                    }
                }
            }
            saveFileDialog.Dispose();
        }

        #endregion

        #region 流下能力算定用データ　上へボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>流下能力算定用データの上へボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdSourceUpper_Click(object sender, EventArgs e)
        {
            // 現在選択行を保存
            this._sourceCurrentRow = dgvSource.CurrentCell.RowIndex;
            if (this._sourceCurrentRow == 0)
            {
                return;
            }
            // バックアップ
            FlowFileData flowFileData = this._commonData.FlowData.FlowSourceList[this._sourceCurrentRow];
            // 削除
            this._commonData.FlowData.FlowSourceList.RemoveAt(this._sourceCurrentRow);
            // 挿入
            this._commonData.FlowData.FlowSourceList.Insert(this._sourceCurrentRow - 1, flowFileData);

            this.setSourceData();
            dgvSource[0, this._sourceCurrentRow - 1].Selected = true;
        }

        #endregion

        #region 流下能力算定用データ　下へボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>流下能力算定用データの下へボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdSourceLower_Click(object sender, EventArgs e)
        {
            // 現在選択行を保存
            this._sourceCurrentRow = dgvSource.CurrentCell.RowIndex;
            if (this._sourceCurrentRow == dgvSource.Rows.Count - 1)
            {
                return;
            }
            // バックアップ
            FlowFileData flowFileData = this._commonData.FlowData.FlowSourceList[this._sourceCurrentRow];
            // 削除
            this._commonData.FlowData.FlowSourceList.RemoveAt(this._sourceCurrentRow);
            if (this._sourceCurrentRow >= this._commonData.FlowData.FlowSourceList.Count)
            {
                // 追加
                this._commonData.FlowData.FlowSourceList.Add(flowFileData);
            }
            else
            {
                // 挿入
                this._commonData.FlowData.FlowSourceList.Insert(this._sourceCurrentRow + 1, flowFileData);
            }

            this.setSourceData();
            dgvSource[0, this._sourceCurrentRow + 1].Selected = true;
        }

        #endregion

        #region 評価水位データ　上へボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>評価水位データの上へボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdTargetUpper_Click(object sender, EventArgs e)
        {
            // 現在選択行を保存
            this._targetCurrentRow = dgvTarget.CurrentCell.RowIndex;
            if (this._targetCurrentRow == 0)
            {
                return;
            }

            // バックアップ
            FlowFileData flowFileData = this._commonData.FlowData.FlowTargetList[this._targetCurrentRow];
            // 削除
            this._commonData.FlowData.FlowTargetList.RemoveAt(this._targetCurrentRow);
            // 挿入
            this._commonData.FlowData.FlowTargetList.Insert(this._targetCurrentRow - 1, flowFileData);

            this.setTargetData();
            dgvTarget[0, this._targetCurrentRow - 1].Selected = true;
        }

        #endregion

        #region 評価水位データ　下へボタン_Click

        /// <summary><para>method outline:</para>
        /// <para>評価水位データの下へボタンクリック時の処理</para>
        /// </summary>
        /// <param name="sender">発生元オブジェクト</param>
        /// <param name="e">イベントの追加情報</param>
        /// <returns>無し</returns>
        /// <exception cref="">無し</exception>
        /// <remarks><para>remarks:</para>
        /// <para>無し</para>
        /// </remarks>
        private void cmdTargetLower_Click(object sender, EventArgs e)
        {
            // 現在選択行を保存
            this._targetCurrentRow = dgvTarget.CurrentCell.RowIndex;
            if (this._targetCurrentRow == dgvTarget.Rows.Count - 1)
            {
                return;
            }
            // バックアップ
            FlowFileData flowFileData = this._commonData.FlowData.FlowTargetList[this._targetCurrentRow];
            // 削除
            this._commonData.FlowData.FlowTargetList.RemoveAt(this._targetCurrentRow);
            if (this._targetCurrentRow >= this._commonData.FlowData.FlowTargetList.Count)
            {
                // 追加
                this._commonData.FlowData.FlowTargetList.Add(flowFileData);
            }
            else
            {
                // 挿入
                this._commonData.FlowData.FlowTargetList.Insert(this._targetCurrentRow + 1, flowFileData);
            }

            this.setTargetData();
            dgvTarget[0, this._targetCurrentRow + 1].Selected = true;
        }

        #endregion

        #endregion
    }
}
